"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MirrorRepository = void 0;
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const constructs_1 = require("constructs");
const context_1 = require("../util/context");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const customNodejsFunction_1 = require("./customNodejsFunction");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const path = require("path");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const triggers_1 = require("aws-cdk-lib/triggers");
class MirrorRepository extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.codeCommitRepository = this.createCodeCommitRepository();
        const { triggerMirrorFunction, triggerMirrorFunctionUrl, } = this.createRepositoryMirroring(props.repoTokenParam, props.repository, this.codeCommitRepository);
        const webhook = this.createWebhook(props.repoTokenParam, props.repository, triggerMirrorFunctionUrl);
        this.triggerInitialMirror(triggerMirrorFunction, [webhook]);
    }
    createCodeCommitRepository() {
        return new aws_codecommit_1.Repository(this, 'Repository', {
            repositoryName: context_1.getProjectName(this),
        });
    }
    createRepositoryMirroring(repoTokenParam, repository, codeCommit) {
        const sourceRepositoryDomain = repository.host === 'github' ? 'github.com' : 'bitbucket.org';
        const mirrorProject = new aws_codebuild_1.Project(this, 'RepositoryMirrorProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-mirrorRepository`,
            timeout: aws_cdk_lib_1.Duration.minutes(20),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_6_0,
                computeType: aws_codebuild_1.ComputeType.SMALL,
            },
            environmentVariables: {
                REPO_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PARAMETER_STORE,
                    value: repoTokenParam.parameterName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'pip install git-remote-codecommit',
                        ],
                    },
                    build: {
                        commands: [
                            `git clone --mirror https://x-token-auth:$REPO_TOKEN@${sourceRepositoryDomain}/${repository.name}.git repository`,
                            'cd repository',
                            `git push --mirror ${codeCommit.repositoryCloneUrlGrc}`,
                        ],
                    },
                },
            }),
        });
        codeCommit.grantPullPush(mirrorProject);
        repoTokenParam.grantRead(mirrorProject);
        const webhookSecret = aws_cdk_lib_1.Fn.select(2, aws_cdk_lib_1.Fn.split('/', aws_cdk_lib_1.Stack.of(this).stackId));
        const triggerMirrorFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'TriggerMirrorFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'mirrorRepository')),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: {
                CODEBUILD_PROJECT_NAME: mirrorProject.projectName,
                SECRET: webhookSecret,
            },
        });
        triggerMirrorFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codebuild:StartBuild'],
            resources: [mirrorProject.projectArn],
        }));
        const triggerMirrorFunctionUrl = triggerMirrorFunction.addFunctionUrl({
            authType: aws_lambda_1.FunctionUrlAuthType.NONE,
        });
        return {
            triggerMirrorFunction,
            triggerMirrorFunctionUrl: `${triggerMirrorFunctionUrl.url}?secret=${webhookSecret}`,
        };
    }
    createWebhook(repoTokenParam, repository, webhookUrl) {
        const setupWebhooksFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'SetupWebhooksFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'setupWebhooks')),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
        });
        repoTokenParam.grantRead(setupWebhooksFunction);
        const provider = new custom_resources_1.Provider(this, 'WebhookProvider', {
            onEventHandler: setupWebhooksFunction,
            logRetention: aws_logs_1.RetentionDays.ONE_MONTH,
        });
        return new aws_cdk_lib_1.CustomResource(this, 'Webhook', {
            serviceToken: provider.serviceToken,
            properties: {
                StackName: aws_cdk_lib_1.Stack.of(this).stackName,
                RepositoryHost: repository.host,
                RepositoryName: repository.name,
                RepositoryTokenParamName: repoTokenParam.parameterName,
                WebhookUrl: webhookUrl,
            },
        });
    }
    triggerInitialMirror(triggerMirrorRepoFunction, executeAfter) {
        new triggers_1.Trigger(this, 'TriggerInitialMirror', {
            handler: triggerMirrorRepoFunction,
            executeAfter,
            executeOnHandlerChange: false,
        });
    }
}
exports.MirrorRepository = MirrorRepository;
//# sourceMappingURL=data:application/json;base64,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