"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// CDK parameters
const STACK_NAME = 'aws-cdk-dynamodb';
// DynamoDB table parameters
const TABLE = 'Table';
const TABLE_WITH_GLOBAL_AND_LOCAL_SECONDARY_INDEX = 'TableWithGlobalAndLocalSecondaryIndex';
const TABLE_WITH_GLOBAL_SECONDARY_INDEX = 'TableWithGlobalSecondaryIndex';
const TABLE_WITH_LOCAL_SECONDARY_INDEX = 'TableWithLocalSecondaryIndex';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_TEST_CASE_1 = 'GSI-PartitionKeyOnly';
const GSI_TEST_CASE_2 = 'GSI-PartitionAndSortKeyWithReadAndWriteCapacity';
const GSI_TEST_CASE_3 = 'GSI-ProjectionTypeKeysOnly';
const GSI_TEST_CASE_4 = 'GSI-ProjectionTypeInclude';
const GSI_TEST_CASE_5 = 'GSI-InverseTableKeySchema';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.NUMBER };
const GSI_NON_KEY = [];
for (let i = 0; i < 10; i++) { // 'A' to 'J'
    GSI_NON_KEY.push(String.fromCharCode(65 + i));
}
// DynamoDB local secondary index parameters
const LSI_TEST_CASE_1 = 'LSI-PartitionAndSortKey';
const LSI_TEST_CASE_2 = 'LSI-PartitionAndTableSortKey';
const LSI_TEST_CASE_3 = 'LSI-ProjectionTypeKeysOnly';
const LSI_TEST_CASE_4 = 'LSI-ProjectionTypeInclude';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = [];
for (let i = 0; i < 10; i++) { // 'K' to 'T'
    LSI_NON_KEY.push(String.fromCharCode(75 + i));
}
const app = new core_1.App();
const stack = new core_1.Stack(app, STACK_NAME);
const table = new lib_1.Table(stack, TABLE, {
    partitionKey: TABLE_PARTITION_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
});
const tableWithGlobalAndLocalSecondaryIndex = new lib_1.Table(stack, TABLE_WITH_GLOBAL_AND_LOCAL_SECONDARY_INDEX, {
    pointInTimeRecovery: true,
    encryption: lib_1.TableEncryption.AWS_MANAGED,
    stream: lib_1.StreamViewType.KEYS_ONLY,
    timeToLiveAttribute: 'timeToLive',
    partitionKey: TABLE_PARTITION_KEY,
    sortKey: TABLE_SORT_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
core_1.Tags.of(tableWithGlobalAndLocalSecondaryIndex).add('Environment', 'Production');
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_1,
    partitionKey: GSI_PARTITION_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_2,
    partitionKey: GSI_PARTITION_KEY,
    sortKey: GSI_SORT_KEY,
    readCapacity: 10,
    writeCapacity: 10,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_3,
    partitionKey: GSI_PARTITION_KEY,
    sortKey: GSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.KEYS_ONLY,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_4,
    partitionKey: GSI_PARTITION_KEY,
    sortKey: GSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.INCLUDE,
    nonKeyAttributes: GSI_NON_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_5,
    partitionKey: TABLE_SORT_KEY,
    sortKey: TABLE_PARTITION_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_2,
    sortKey: LSI_SORT_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_1,
    sortKey: TABLE_SORT_KEY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_3,
    sortKey: LSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.KEYS_ONLY,
});
tableWithGlobalAndLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_4,
    sortKey: LSI_SORT_KEY,
    projectionType: lib_1.ProjectionType.INCLUDE,
    nonKeyAttributes: LSI_NON_KEY,
});
const encryptionKey = new kms.Key(stack, 'Key', {
    enableKeyRotation: true,
});
const tableWithGlobalSecondaryIndex = new lib_1.Table(stack, TABLE_WITH_GLOBAL_SECONDARY_INDEX, {
    partitionKey: TABLE_PARTITION_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryptionKey,
});
tableWithGlobalSecondaryIndex.addGlobalSecondaryIndex({
    indexName: GSI_TEST_CASE_1,
    partitionKey: GSI_PARTITION_KEY,
});
const tableWithLocalSecondaryIndex = new lib_1.Table(stack, TABLE_WITH_LOCAL_SECONDARY_INDEX, {
    partitionKey: TABLE_PARTITION_KEY,
    sortKey: TABLE_SORT_KEY,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
    encryption: lib_1.TableEncryption.DEFAULT,
});
tableWithLocalSecondaryIndex.addLocalSecondaryIndex({
    indexName: LSI_TEST_CASE_1,
    sortKey: LSI_SORT_KEY,
});
const role = new iam.Role(stack, 'Role', {
    assumedBy: new iam.ServicePrincipal('sqs.amazonaws.com'),
});
table.grantReadData(role);
tableWithGlobalAndLocalSecondaryIndex.grantReadData(role);
app.synth();
//# sourceMappingURL=data:application/json;base64,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