from django.db import models
from django.contrib.postgres.fields import ArrayField
from django.utils.translation import gettext_lazy as lazy

from taggit.managers import TaggableManager
from mkite_db.orm.repr import _named_repr
from mkite_db.orm.base.models import DbEntry, ChemNode, Elements


class SpaceGroups(models.IntegerChoices):
    S1 = 1, lazy("P1")
    S2 = 2, lazy("P-1")
    S3 = 3, lazy("P2")
    S4 = 4, lazy("P21")
    S5 = 5, lazy("C2")
    S6 = 6, lazy("Pm")
    S7 = 7, lazy("Pc")
    S8 = 8, lazy("Cm")
    S9 = 9, lazy("Cc")
    S10 = 10, lazy("P2/m")
    S11 = 11, lazy("P21/m")
    S12 = 12, lazy("C2/m")
    S13 = 13, lazy("P2/c")
    S14 = 14, lazy("P21/c")
    S15 = 15, lazy("C2/c")
    S16 = 16, lazy("P222")
    S17 = 17, lazy("P2221")
    S18 = 18, lazy("P21212")
    S19 = 19, lazy("P212121")
    S20 = 20, lazy("C2221")
    S21 = 21, lazy("C222")
    S22 = 22, lazy("F222")
    S23 = 23, lazy("I222")
    S24 = 24, lazy("I212121")
    S25 = 25, lazy("Pmm2")
    S26 = 26, lazy("Pmc21")
    S27 = 27, lazy("Pcc2")
    S28 = 28, lazy("Pma2")
    S29 = 29, lazy("Pca21")
    S30 = 30, lazy("Pnc2")
    S31 = 31, lazy("Pmn21")
    S32 = 32, lazy("Pba2")
    S33 = 33, lazy("Pna21")
    S34 = 34, lazy("Pnn2")
    S35 = 35, lazy("Cmm2")
    S36 = 36, lazy("Cmc21")
    S37 = 37, lazy("Ccc2")
    S38 = 38, lazy("Amm2")
    S39 = 39, lazy("Aem2")
    S40 = 40, lazy("Ama2")
    S41 = 41, lazy("Aea2")
    S42 = 42, lazy("Fmm2")
    S43 = 43, lazy("Fdd2")
    S44 = 44, lazy("Imm2")
    S45 = 45, lazy("Iba2")
    S46 = 46, lazy("Ima2")
    S47 = 47, lazy("Pmmm")
    S48 = 48, lazy("Pnnn")
    S49 = 49, lazy("Pccm")
    S50 = 50, lazy("Pban")
    S51 = 51, lazy("Pmma")
    S52 = 52, lazy("Pnna")
    S53 = 53, lazy("Pmna")
    S54 = 54, lazy("Pcca")
    S55 = 55, lazy("Pbam")
    S56 = 56, lazy("Pccn")
    S57 = 57, lazy("Pbcm")
    S58 = 58, lazy("Pnnm")
    S59 = 59, lazy("Pmmn")
    S60 = 60, lazy("Pbcn")
    S61 = 61, lazy("Pbca")
    S62 = 62, lazy("Pnma")
    S63 = 63, lazy("Cmcm")
    S64 = 64, lazy("Cmce")
    S65 = 65, lazy("Cmmm")
    S66 = 66, lazy("Cccm")
    S67 = 67, lazy("Cmme")
    S68 = 68, lazy("Ccce")
    S69 = 69, lazy("Fmmm")
    S70 = 70, lazy("Fddd")
    S71 = 71, lazy("Immm")
    S72 = 72, lazy("Ibam")
    S73 = 73, lazy("Ibca")
    S74 = 74, lazy("Imma")
    S75 = 75, lazy("P4")
    S76 = 76, lazy("P41")
    S77 = 77, lazy("P42")
    S78 = 78, lazy("P43")
    S79 = 79, lazy("I4")
    S80 = 80, lazy("I41")
    S81 = 81, lazy("P-4")
    S82 = 82, lazy("I-4")
    S83 = 83, lazy("P4/m")
    S84 = 84, lazy("P42/m")
    S85 = 85, lazy("P4/n")
    S86 = 86, lazy("P42/n")
    S87 = 87, lazy("I4/m")
    S88 = 88, lazy("I41/a")
    S89 = 89, lazy("P422")
    S90 = 90, lazy("P4212")
    S91 = 91, lazy("P4122")
    S92 = 92, lazy("P41212")
    S93 = 93, lazy("P4222")
    S94 = 94, lazy("P42212")
    S95 = 95, lazy("P4322")
    S96 = 96, lazy("P43212")
    S97 = 97, lazy("I422")
    S98 = 98, lazy("I4122")
    S99 = 99, lazy("P4mm")
    S100 = 100, lazy("P4bm")
    S101 = 101, lazy("P42cm")
    S102 = 102, lazy("P42nm")
    S103 = 103, lazy("P4cc")
    S104 = 104, lazy("P4nc")
    S105 = 105, lazy("P42mc")
    S106 = 106, lazy("P42bc")
    S107 = 107, lazy("I4mm")
    S108 = 108, lazy("I4cm")
    S109 = 109, lazy("I41md")
    S110 = 110, lazy("I41cd")
    S111 = 111, lazy("P-42m")
    S112 = 112, lazy("P-42c")
    S113 = 113, lazy("P-421m")
    S114 = 114, lazy("P-421c")
    S115 = 115, lazy("P-4m2")
    S116 = 116, lazy("P-4c2")
    S117 = 117, lazy("P-4b2")
    S118 = 118, lazy("P-4n2")
    S119 = 119, lazy("I-4m2")
    S120 = 120, lazy("I-4c2")
    S121 = 121, lazy("I-42m")
    S122 = 122, lazy("I-42d")
    S123 = 123, lazy("P4/mmm")
    S124 = 124, lazy("P4/mcc")
    S125 = 125, lazy("P4/nbm")
    S126 = 126, lazy("P4/nnc")
    S127 = 127, lazy("P4/mbm")
    S128 = 128, lazy("P4/mnc")
    S129 = 129, lazy("P4/nmm")
    S130 = 130, lazy("P4/ncc")
    S131 = 131, lazy("P42/mmc")
    S132 = 132, lazy("P42/mcm")
    S133 = 133, lazy("P42/nbc")
    S134 = 134, lazy("P42/nnm")
    S135 = 135, lazy("P42/mbc")
    S136 = 136, lazy("P42/mnm")
    S137 = 137, lazy("P42/nmc")
    S138 = 138, lazy("P42/ncm")
    S139 = 139, lazy("I4/mmm")
    S140 = 140, lazy("I4/mcm")
    S141 = 141, lazy("I41/amd")
    S142 = 142, lazy("I41/acd")
    S143 = 143, lazy("P3")
    S144 = 144, lazy("P31")
    S145 = 145, lazy("P32")
    S146 = 146, lazy("R3")
    S147 = 147, lazy("P-3")
    S148 = 148, lazy("R-3")
    S149 = 149, lazy("P312")
    S150 = 150, lazy("P321")
    S151 = 151, lazy("P3112")
    S152 = 152, lazy("P3121")
    S153 = 153, lazy("P3212")
    S154 = 154, lazy("P3221")
    S155 = 155, lazy("R32")
    S156 = 156, lazy("P3m1")
    S157 = 157, lazy("P31m")
    S158 = 158, lazy("P3c1")
    S159 = 159, lazy("P31c")
    S160 = 160, lazy("R3m")
    S161 = 161, lazy("R3c")
    S162 = 162, lazy("P-31m")
    S163 = 163, lazy("P-31c")
    S164 = 164, lazy("P-3m1")
    S165 = 165, lazy("P-3c1")
    S166 = 166, lazy("R-3m")
    S167 = 167, lazy("R-3c")
    S168 = 168, lazy("P6")
    S169 = 169, lazy("P61")
    S170 = 170, lazy("P65")
    S171 = 171, lazy("P62")
    S172 = 172, lazy("P64")
    S173 = 173, lazy("P63")
    S174 = 174, lazy("P-6")
    S175 = 175, lazy("P6/m")
    S176 = 176, lazy("P63/m")
    S177 = 177, lazy("P622")
    S178 = 178, lazy("P6122")
    S179 = 179, lazy("P6522")
    S180 = 180, lazy("P6222")
    S181 = 181, lazy("P6422")
    S182 = 182, lazy("P6322")
    S183 = 183, lazy("P6mm")
    S184 = 184, lazy("P6cc")
    S185 = 185, lazy("P63cm")
    S186 = 186, lazy("P63mc")
    S187 = 187, lazy("P-6m2")
    S188 = 188, lazy("P-6c2")
    S189 = 189, lazy("P-62m")
    S190 = 190, lazy("P-62c")
    S191 = 191, lazy("P6/mmm")
    S192 = 192, lazy("P6/mcc")
    S193 = 193, lazy("P63/mcm")
    S194 = 194, lazy("P63/mmc")
    S195 = 195, lazy("P23")
    S196 = 196, lazy("F23")
    S197 = 197, lazy("I23")
    S198 = 198, lazy("P213")
    S199 = 199, lazy("I213")
    S200 = 200, lazy("Pm-3")
    S201 = 201, lazy("Pn-3")
    S202 = 202, lazy("Fm-3")
    S203 = 203, lazy("Fd-3")
    S204 = 204, lazy("Im-3")
    S205 = 205, lazy("Pa-3")
    S206 = 206, lazy("Ia-3")
    S207 = 207, lazy("P432")
    S208 = 208, lazy("P4232")
    S209 = 209, lazy("F432")
    S210 = 210, lazy("F4132")
    S211 = 211, lazy("I432")
    S212 = 212, lazy("P4332")
    S213 = 213, lazy("P4132")
    S214 = 214, lazy("I4132")
    S215 = 215, lazy("P-43m")
    S216 = 216, lazy("F-43m")
    S217 = 217, lazy("I-43m")
    S218 = 218, lazy("P-43n")
    S219 = 219, lazy("F-43c")
    S220 = 220, lazy("I-43d")
    S221 = 221, lazy("Pm-3m")
    S222 = 222, lazy("Pn-3n")
    S223 = 223, lazy("Pm-3n")
    S224 = 224, lazy("Pn-3m")
    S225 = 225, lazy("Fm-3m")
    S226 = 226, lazy("Fm-3c")
    S227 = 227, lazy("Fd-3m")
    S228 = 228, lazy("Fd-3c")
    S229 = 229, lazy("Im-3m")
    S230 = 230, lazy("Ia-3d")


class Crystal(ChemNode):
    spacegroup = models.PositiveSmallIntegerField(
        null=False,
        choices=SpaceGroups.choices,
    )

    species = ArrayField(
        models.CharField(
            max_length=2,
            null=False,
            choices=Elements.choices,
        )
    )

    coords = ArrayField(ArrayField(models.FloatField(), size=3))

    lattice = ArrayField(ArrayField(models.FloatField(), size=3), size=3)

    siteprops = models.JSONField(default=dict)

    attributes = models.JSONField(default=dict)

    tags = TaggableManager()

    def as_info(self):
        from mkite_core.models import CrystalInfo

        return CrystalInfo.from_crystal(self)

    @property
    def formula(self):
        if "formula" in self.attributes:
            return self.attributes["formula"]
        return None

    def __repr__(self):
        spgrp = SpaceGroups(self.spacegroup).label
        formula = str(self.formula)
        return f"<{self.__class__.__name__}: {formula}, {spgrp} ({self.id})>"
