from pathlib import Path
from unittest.mock import patch

import pytest

from jupyterlab_git_newland.git import Git

from .testutils import maybe_future


@pytest.mark.asyncio
async def test_single_file_log():
    with patch("jupyterlab_git.git.execute") as mock_execute:
        # Given
        process_output = [
            "8b080cb6cdd526e91199fc003b10e1ba8010f1c4",
            "Awesome Developer",
            "4 months ago",
            "Set _commitAndPush_ to false by default",
            "1\t\t1\tREADME.md",
            "",
            "4fe9688ba74b2cf023af3d76fd737c9fe31548fe",
            "Awesome Developer",
            "4 months ago",
            "Fixes #762 (#914)",
            "1\t\t1\tREADME.md",
            "",
            "4444de0ee37e26d4a7849812f845370b6c7da994",
            "Awesome Developer",
            "4 months ago",
            "Add new setting to README",
            "1\t\t0\tREADME.md",
            "",
            "a216011e407d58bd7ef5c6c1005903a72758016d",
            "Awesome Developer",
            "5 months ago",
            "restore preview gif",
            "2\t\t0\tREADME.md",
            "",
            "2777209f76c70b91ff0ffdd4d878b779758ab355",
            "Awesome Developer",
            "5 months ago",
            "fill out details in troubleshooting",
            "1\t\t1\tREADME.md",
            "",
            "ab35dafe2afb9de3da69590f94028f9582b52862",
            "Awesome Developer",
            "5 months ago",
            "Update README.md:Git panel is not visible (#891)",
            "4\t\t1\tREADME.md",
            "",
            "966671c3e0d4ec62bb4ab03c3a363fbba2ef2666",
            "Awesome Developer",
            "6 months ago",
            "Add some notes for JupyterLab <3 (#865)",
            "13\t\t1\tREADME.md",
            "",
            "c03c1aae4d638cf7ff533ac4065148fa4fce88e0",
            "Awesome Developer",
            "6 months ago",
            "Fix #859",
            "2\t\t2\tREADME.md",
            "",
            "a72d4b761e0701c226b39a822ac42b80a91ad9df",
            "Awesome Developer",
            "7 months ago",
            "Update to Jupyterlab 3.0 (#818)",
            "47\t4\t3\tREADME.md",
            "",
            "a72b9852ed5af0f9368acb6737557960515a6fdb",
            "Awesome Developer",
            "8 months ago",
            "Update README.md",
            "3\t\t1\tREADME.md",
            "",
            "120b04e1a3641eb5ae0a042e3934bd48eb6469df",
            "Awesome Developer",
            "9 months ago",
            "Reduce polling (#812)",
            "1\t\t0\tREADME.md",
            "",
            "7b7f0e53a7a25c0726bbf6f27912918185193b2d",
            "Awesome Developer",
            "10 months ago",
            "Hide overflow on focus (#792)",
            "1\t\t1\tREADME.md",
            "",
            "4efb0a776cd22756b80cae7ec2c1cda5f701ab93",
            "Awesome Developer",
            "10 months ago",
            "Add conda installation instructions",
            "12\t\t5\tREADME.md",
            "",
            "706fb7a8d5cc8f98bcaf61374a8f4ca77f92f056",
            "Awesome Developer",
            "10 months ago",
            "Clarify post_init directions (#781)",
            "6\t\t0\tREADME.md",
            "",
            "70f26818d434e086ca5d5aead8548bdeb0fd6564",
            "Awesome Developer",
            "11 months ago",
            "add black to pre-commit",
            "1\t\t0\tREADME.md",
            "",
            "a046b66415c6afabcbdf6e624e2a367523ee2e80",
            "Awesome Developer",
            "12 months ago",
            "Switch to GitHub action for testing (#738)",
            "1\t\t1\tREADME.md",
            "",
            "6eb994e7fee4a6bc482f641f1265a6fa031112d4",
            "Awesome Developer",
            "12 months ago",
            "Restore server_root endpoint (#733)",
            "19\t\t1\tREADME.md",
            "",
            "5056ca14dd90472b4a3f621d1bc2edd0c30a9a12",
            "Awesome Developer",
            "1 year ago",
            "Check git version and compatibility front/backend version (#658)",
            "1\t\t1\tREADME.md",
            "",
            "ab00273e351d04d1f4be0c89d3943c083bee0b73",
            "Awesome Developer",
            "1 year ago",
            "docs: update README.md [skip ci]",
            "1\t\t1\tREADME.md",
            "",
            "6ffe9fec322ac4f6c126b4d46f145b003c52195d",
            "Awesome Developer",
            "1 year ago",
            "docs: update README.md [skip ci]",
            "12\t1\t3\tREADME.md",
            "",
            "dafc4006d5d3df5b07d4ac0ef045a73ea41577da",
            "Awesome Developer",
            "1 year ago",
            'Merge PR #630 "Provide UI feedback during Git command execution"',
            "4\t\t1\tREADME.md",
            "",
            "2f9ad78074bd8219200587020236ac77daa761be",
            "Awesome Developer",
            "1 year, 1 month ago",
            "link -> install",
            "2\t\t2\tREADME.md",
            "",
            "bbcc5d8e6b7f5a8abc71b0d473845a55fbbaad42",
            "Awesome Developer",
            "1 year, 2 months ago",
            "add doubleClickDiff setting to readme",
            "1\t\t0\tREADME.md",
            "",
            "8e79eae1277f8a9b8a07123717dbd5cc8ba5f83d",
            "Awesome Developer",
            "1 year, 2 months ago",
            "Switch sponsorship icon (#579)",
            "11\t\t7\tREADME.md",
            "",
            "ec66c24fb7391116ea0d91d5b7a275cf57ff0fe7",
            "Awesome Developer",
            "1 year, 2 months ago",
            "Fix #551 (#649)",
            "2\t\t2\tREADME.md",
            "",
        ]

        mock_execute.return_value = maybe_future((0, "\n".join(process_output), ""))

        expected_response = {
            "code": 0,
            "commits": [
                {
                    "commit": "8b080cb6cdd526e91199fc003b10e1ba8010f1c4",
                    "author": "Awesome Developer",
                    "date": "4 months ago",
                    "commit_msg": "Set _commitAndPush_ to false by default",
                    "pre_commit": "4fe9688ba74b2cf023af3d76fd737c9fe31548fe",
                    "is_binary": False,
                },
                {
                    "commit": "4fe9688ba74b2cf023af3d76fd737c9fe31548fe",
                    "author": "Awesome Developer",
                    "date": "4 months ago",
                    "commit_msg": "Fixes #762 (#914)",
                    "pre_commit": "4444de0ee37e26d4a7849812f845370b6c7da994",
                    "is_binary": False,
                },
                {
                    "commit": "4444de0ee37e26d4a7849812f845370b6c7da994",
                    "author": "Awesome Developer",
                    "date": "4 months ago",
                    "commit_msg": "Add new setting to README",
                    "pre_commit": "a216011e407d58bd7ef5c6c1005903a72758016d",
                    "is_binary": False,
                },
                {
                    "commit": "a216011e407d58bd7ef5c6c1005903a72758016d",
                    "author": "Awesome Developer",
                    "date": "5 months ago",
                    "commit_msg": "restore preview gif",
                    "pre_commit": "2777209f76c70b91ff0ffdd4d878b779758ab355",
                    "is_binary": False,
                },
                {
                    "commit": "2777209f76c70b91ff0ffdd4d878b779758ab355",
                    "author": "Awesome Developer",
                    "date": "5 months ago",
                    "commit_msg": "fill out details in troubleshooting",
                    "pre_commit": "ab35dafe2afb9de3da69590f94028f9582b52862",
                    "is_binary": False,
                },
                {
                    "commit": "ab35dafe2afb9de3da69590f94028f9582b52862",
                    "author": "Awesome Developer",
                    "date": "5 months ago",
                    "commit_msg": "Update README.md:Git panel is not visible (#891)",
                    "pre_commit": "966671c3e0d4ec62bb4ab03c3a363fbba2ef2666",
                    "is_binary": False,
                },
                {
                    "commit": "966671c3e0d4ec62bb4ab03c3a363fbba2ef2666",
                    "author": "Awesome Developer",
                    "date": "6 months ago",
                    "commit_msg": "Add some notes for JupyterLab <3 (#865)",
                    "pre_commit": "c03c1aae4d638cf7ff533ac4065148fa4fce88e0",
                    "is_binary": False,
                },
                {
                    "commit": "c03c1aae4d638cf7ff533ac4065148fa4fce88e0",
                    "author": "Awesome Developer",
                    "date": "6 months ago",
                    "commit_msg": "Fix #859",
                    "pre_commit": "a72d4b761e0701c226b39a822ac42b80a91ad9df",
                    "is_binary": False,
                },
                {
                    "commit": "a72d4b761e0701c226b39a822ac42b80a91ad9df",
                    "author": "Awesome Developer",
                    "date": "7 months ago",
                    "commit_msg": "Update to Jupyterlab 3.0 (#818)",
                    "pre_commit": "a72b9852ed5af0f9368acb6737557960515a6fdb",
                    "is_binary": False,
                },
                {
                    "commit": "a72b9852ed5af0f9368acb6737557960515a6fdb",
                    "author": "Awesome Developer",
                    "date": "8 months ago",
                    "commit_msg": "Update README.md",
                    "pre_commit": "120b04e1a3641eb5ae0a042e3934bd48eb6469df",
                    "is_binary": False,
                },
                {
                    "commit": "120b04e1a3641eb5ae0a042e3934bd48eb6469df",
                    "author": "Awesome Developer",
                    "date": "9 months ago",
                    "commit_msg": "Reduce polling (#812)",
                    "pre_commit": "7b7f0e53a7a25c0726bbf6f27912918185193b2d",
                    "is_binary": False,
                },
                {
                    "commit": "7b7f0e53a7a25c0726bbf6f27912918185193b2d",
                    "author": "Awesome Developer",
                    "date": "10 months ago",
                    "commit_msg": "Hide overflow on focus (#792)",
                    "pre_commit": "4efb0a776cd22756b80cae7ec2c1cda5f701ab93",
                    "is_binary": False,
                },
                {
                    "commit": "4efb0a776cd22756b80cae7ec2c1cda5f701ab93",
                    "author": "Awesome Developer",
                    "date": "10 months ago",
                    "commit_msg": "Add conda installation instructions",
                    "pre_commit": "706fb7a8d5cc8f98bcaf61374a8f4ca77f92f056",
                    "is_binary": False,
                },
                {
                    "commit": "706fb7a8d5cc8f98bcaf61374a8f4ca77f92f056",
                    "author": "Awesome Developer",
                    "date": "10 months ago",
                    "commit_msg": "Clarify post_init directions (#781)",
                    "pre_commit": "70f26818d434e086ca5d5aead8548bdeb0fd6564",
                    "is_binary": False,
                },
                {
                    "commit": "70f26818d434e086ca5d5aead8548bdeb0fd6564",
                    "author": "Awesome Developer",
                    "date": "11 months ago",
                    "commit_msg": "add black to pre-commit",
                    "pre_commit": "a046b66415c6afabcbdf6e624e2a367523ee2e80",
                    "is_binary": False,
                },
                {
                    "commit": "a046b66415c6afabcbdf6e624e2a367523ee2e80",
                    "author": "Awesome Developer",
                    "date": "12 months ago",
                    "commit_msg": "Switch to GitHub action for testing (#738)",
                    "pre_commit": "6eb994e7fee4a6bc482f641f1265a6fa031112d4",
                    "is_binary": False,
                },
                {
                    "commit": "6eb994e7fee4a6bc482f641f1265a6fa031112d4",
                    "author": "Awesome Developer",
                    "date": "12 months ago",
                    "commit_msg": "Restore server_root endpoint (#733)",
                    "pre_commit": "5056ca14dd90472b4a3f621d1bc2edd0c30a9a12",
                    "is_binary": False,
                },
                {
                    "commit": "5056ca14dd90472b4a3f621d1bc2edd0c30a9a12",
                    "author": "Awesome Developer",
                    "date": "1 year ago",
                    "commit_msg": "Check git version and compatibility front/backend version (#658)",
                    "pre_commit": "ab00273e351d04d1f4be0c89d3943c083bee0b73",
                    "is_binary": False,
                },
                {
                    "commit": "ab00273e351d04d1f4be0c89d3943c083bee0b73",
                    "author": "Awesome Developer",
                    "date": "1 year ago",
                    "commit_msg": "docs: update README.md [skip ci]",
                    "pre_commit": "6ffe9fec322ac4f6c126b4d46f145b003c52195d",
                    "is_binary": False,
                },
                {
                    "commit": "6ffe9fec322ac4f6c126b4d46f145b003c52195d",
                    "author": "Awesome Developer",
                    "date": "1 year ago",
                    "commit_msg": "docs: update README.md [skip ci]",
                    "pre_commit": "dafc4006d5d3df5b07d4ac0ef045a73ea41577da",
                    "is_binary": False,
                },
                {
                    "commit": "dafc4006d5d3df5b07d4ac0ef045a73ea41577da",
                    "author": "Awesome Developer",
                    "date": "1 year ago",
                    "commit_msg": 'Merge PR #630 "Provide UI feedback during Git command execution"',
                    "pre_commit": "2f9ad78074bd8219200587020236ac77daa761be",
                    "is_binary": False,
                },
                {
                    "commit": "2f9ad78074bd8219200587020236ac77daa761be",
                    "author": "Awesome Developer",
                    "date": "1 year, 1 month ago",
                    "commit_msg": "link -> install",
                    "pre_commit": "bbcc5d8e6b7f5a8abc71b0d473845a55fbbaad42",
                    "is_binary": False,
                },
                {
                    "commit": "bbcc5d8e6b7f5a8abc71b0d473845a55fbbaad42",
                    "author": "Awesome Developer",
                    "date": "1 year, 2 months ago",
                    "commit_msg": "add doubleClickDiff setting to readme",
                    "pre_commit": "8e79eae1277f8a9b8a07123717dbd5cc8ba5f83d",
                    "is_binary": False,
                },
                {
                    "commit": "8e79eae1277f8a9b8a07123717dbd5cc8ba5f83d",
                    "author": "Awesome Developer",
                    "date": "1 year, 2 months ago",
                    "commit_msg": "Switch sponsorship icon (#579)",
                    "pre_commit": "ec66c24fb7391116ea0d91d5b7a275cf57ff0fe7",
                    "is_binary": False,
                },
                {
                    "commit": "ec66c24fb7391116ea0d91d5b7a275cf57ff0fe7",
                    "author": "Awesome Developer",
                    "date": "1 year, 2 months ago",
                    "commit_msg": "Fix #551 (#649)",
                    "pre_commit": "",
                    "is_binary": False,
                },
            ],
        }

        # When
        actual_response = await Git().log(
            path=str(Path("/bin/test_curr_path")),
            history_count=25,
            follow_path="README.md",
        )

        # Then
        mock_execute.assert_called_once_with(
            [
                "git",
                "log",
                "--pretty=format:%H%n%an%n%ar%n%s",
                "-25",
                "--numstat",
                "--follow",
                "--",
                "README.md",
            ],
            cwd=str(Path("/bin") / "test_curr_path"),
        )

        assert expected_response == actual_response
