from proto_topy.entities import ProtoCollection, ProtoModule, CompilationFailed
import pytest
import os
import sys
from pathlib import Path
from distutils.spawn import find_executable

protoc_path = Path(find_executable("protoc") or os.environ.get('PROTOC'))


def test_add_proto():
    proto = ProtoModule(file_path=Path("test1.proto"), source="")
    modules = ProtoCollection(compiler_path=protoc_path)
    modules.add_proto(proto)
    assert Path("test1.proto") in modules.modules


def test_add_proto2():
    modules = ProtoCollection(protoc_path, *(ProtoModule(file_path=Path("test2.proto"), source=""),
                                             ProtoModule(file_path=Path("test3.proto"), source="")), )
    assert Path("test2.proto") in modules.modules
    assert Path("test3.proto") in modules.modules


def test_bad_protoc():
    with pytest.raises(FileNotFoundError):
        modules = ProtoCollection(Path('dummy'))
        modules.compile()


def test_no_protoc():
    with pytest.raises(TypeError):
        ProtoCollection()


def test_compile_wrong_proto():
    modules = ProtoCollection(protoc_path, ProtoModule(file_path=Path("test4.proto"), source=""))
    with pytest.raises(CompilationFailed):
        modules.compile()


def test_compile_minimal_proto():
    from google.protobuf.timestamp_pb2 import Timestamp
    proto = ProtoModule(file_path=Path("test5.proto"), source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message Test5 {
        google.protobuf.Timestamp created = 1;
    }
    """)
    modules = ProtoCollection(protoc_path, proto)
    modules.compile()
    sys.modules["test5"] = proto.py
    atest5 = proto.py.Test5()
    assert isinstance(atest5.created, Timestamp)
    del sys.modules["test5"]


def test_compile_minimal_proto_in_a_package():
    from google.protobuf.timestamp_pb2 import Timestamp
    proto = ProtoModule(file_path=Path("p1/p2/p3/thing.proto"), source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message Thing {
        google.protobuf.Timestamp created = 1;
    }
    """)
    modules = ProtoCollection(protoc_path, proto)
    modules.compile()
    assert "\n".join(proto.py_source.split("\n")[:4]) == '''# -*- coding: utf-8 -*-
# Generated by the protocol buffer compiler.  DO NOT EDIT!
# source: p1/p2/p3/thing.proto
"""Generated protocol buffer code."""'''
    sys.modules["thing"] = proto.py
    athing = proto.py.Thing()
    assert isinstance(athing.created, Timestamp)


def test_compile_missing_dependency():
    proto = ProtoModule(file_path=Path("test.proto"), source="""
    syntax = "proto3";
    import "other.proto";
    """)
    modules = ProtoCollection(protoc_path, proto)
    with pytest.raises(CompilationFailed, match=r'other.proto: File not found.*'):
        modules.compile()


def test_compile_ununsed_dependency():
    proto = ProtoModule(file_path=Path("test.proto"), source="""
    syntax = "proto3";
    import "other.proto";
    """)
    other_proto = ProtoModule(file_path=Path("other.proto"), source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message OtherThing {
        google.protobuf.Timestamp created = 1;
    }
    """)
    modules = ProtoCollection(protoc_path, proto, other_proto)
    modules.compile()


def test_compile_simple_dependency():
    from google.protobuf.timestamp_pb2 import Timestamp
    proto = ProtoModule(file_path=Path("p3/p4/test6.proto"), source="""
    syntax = "proto3";
    import "p1/p2/other2.proto";
    message Test6 {
        OtherThing2 foo = 1;
    };
    """)
    other_proto = ProtoModule(file_path=Path("p1/p2/other2.proto"), source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message OtherThing2 {
        google.protobuf.Timestamp created = 1;
    }
    """)
    modules = ProtoCollection(protoc_path, proto, other_proto)
    modules.compile()
    sys.modules.update({proto.name: proto.py for proto in modules.modules.values()})
    atest6 = modules.modules[Path("p3/p4/test6.proto")].py.Test6()
    assert isinstance(atest6.foo.created, Timestamp)
    for proto in modules.modules.values():
        del sys.modules[proto.name]
