"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: props.tableName ?? cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: props.sortStyle ?? this.getDefaultSortStyle(props.tableColumns),
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.62.1-alpha.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
//# sourceMappingURL=data:application/json;base64,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