
import os
import re
import sys
import json
import logging
import subprocess

try:
    from pathlib import Path  # noqa, py3
except ImportError:
    from pathlib2 import Path  # noqa, py2

from . import Revision, Container, ContainerError


_log = logging.getLogger("rezup.util")


def resolve_environ(revision, requests_or_rxt):
    """Resolve package requests with Rez from container

    Open a subprocess and call rez-python that is located from container to
    resolve the request and returns the context environment.

    Args:
        revision: A revision instance that is ready to use
        requests_or_rxt: List of strings or list of PackageRequest objects,
            or, a resolved context RXT file.

    Returns:
        dict: The environment dict generated by the resolved context.

    Raises:
        ContainerError: when no valid revision to use.
        subprocess.CalledProcessError

    """
    if not revision.is_ready():
        raise ContainerError("Revision is not ready to be used.")

    env = os.environ.copy()
    env.update(revision.recipe_env() or {})

    ext = ".exe" if sys.platform == "win32" else ""
    rez_python = None

    for bin_dir in revision.production_bin_dirs():
        _exec = str(bin_dir / ("rez-python" + ext))
        if os.access(_exec, os.X_OK):
            rez_python = _exec
            break

    if rez_python is None:
        raise ContainerError("rez-python not found in revision: %s"
                             % revision.path())

    if isinstance(requests_or_rxt, list):
        requests_or_rxt = " ".join([str(r) for r in requests_or_rxt])

    action_py = os.path.join(os.path.dirname(__file__), "_actions.py")

    args = [
        rez_python,
        "-B",  # just to be safe, no .pyc ('bad magic number in .pyc' error)
        action_py,
        _message_wrap,
        "action_resolve",  # resolve and return serialized context env
        str(requests_or_rxt),
    ]
    try:
        out = subprocess.check_output(
            args,
            env=env,
            stderr=subprocess.STDOUT,
            universal_newlines=True,
        )
    except subprocess.CalledProcessError as e:
        _log.error(e.output)
        raise

    return json.loads(polish(out))


def get_revision(container=None, create=False, fallback=True):
    """Returns a revision instance from container

    Args:
        container: Container name, use default container if name not given.
        create: Create local revision if not exists, default False.
        fallback: If True, accept earlier revision when no timestamp matched
            found in local.

    Returns:
        Revision: An instance of Revision

    Raises:
        ContainerError

    """
    name = container or Container.DEFAULT_NAME
    container = Container(name)
    revision = container.get_latest_revision()

    if revision is None:
        if container.is_remote() or not create:
            raise ContainerError("No valid revision in container %r: %s"
                                 % (container.name(), container.path()))
        else:
            revision = container.new_revision()

    revision = revision.pull(check_out=create, fallback=fallback)
    if revision is None:
        raise ContainerError("No matched revision in local container.")

    return revision


_message_wrap = "::rezup.msg.start::%s::rezup.msg.end::"
_polish_regex = re.compile(_message_wrap % "(.*)")


def polish(message):
    match = _polish_regex.search(message)
    if match:
        return match.group(1)
