"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Check for ApiGateway params
// --------------------------------------------------------------
test('Test for default Params construct props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    const construct = new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params', props);
    // Assertion
    expect(construct.apiGateway).not.toBeNull();
    expect(construct.apiGatewayCloudWatchRole).not.toBeNull();
    expect(construct.apiGatewayLogGroup).not.toBeNull();
    expect(construct.apiGatewayRole).not.toBeNull();
});
// --------------------------------------------------------------
// Check for Default IAM Role
// --------------------------------------------------------------
test('Test for default IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "awsapigatewayiotpolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Request Validator
// --------------------------------------------------------------
test('Test for default Params request validator', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-request-validator', props);
    // Assertion
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RequestValidator", {
        ValidateRequestBody: false,
        ValidateRequestParameters: true,
    });
});
// --------------------------------------------------------------
// Check for Integ Props and Method Props
// --------------------------------------------------------------
test('Test for default Params Integ Props and Method Props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-integpros-methodprops', props);
    // Assertion for {topic-level-7} to ensure all Integration Request Params, Integration Responses,
    // Method Request Params and Method Reponses are intact
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "5\\d{2}",
                    StatusCode: "500"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    StatusCode: "403"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.path.topic-level-1": "method.request.path.topic-level-1",
                "integration.request.path.topic-level-2": "method.request.path.topic-level-2",
                "integration.request.path.topic-level-3": "method.request.path.topic-level-3",
                "integration.request.path.topic-level-4": "method.request.path.topic-level-4",
                "integration.request.path.topic-level-5": "method.request.path.topic-level-5",
                "integration.request.path.topic-level-6": "method.request.path.topic-level-6",
                "integration.request.path.topic-level-7": "method.request.path.topic-level-7"
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:${props.iotEndpoint}.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}/{topic-level-4}/{topic-level-5}/{topic-level-6}/{topic-level-7}`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200"
            },
            {
                StatusCode: "500"
            },
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true,
            "method.request.path.topic-level-2": true,
            "method.request.path.topic-level-3": true,
            "method.request.path.topic-level-4": true,
            "method.request.path.topic-level-5": true,
            "method.request.path.topic-level-6": true,
            "method.request.path.topic-level-7": true
        }
    });
});
// --------------------------------------------------------------
// Check for valid IoT Endpoint
// --------------------------------------------------------------
test('Test for valid iot enpoint', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: ' '
    };
    const app = () => {
        new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-no-endpoint', props);
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Check for binaryMediaTypes
// --------------------------------------------------------------
test('Test for Binary Media types', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-binaryMediaTypes', {
        iotEndpoint: 'a1234567890123-ats'
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        BinaryMediaTypes: [
            "application/octet-stream",
        ],
    });
});
// --------------------------------------------------------------
// Check for Api Name and Desc
// --------------------------------------------------------------
test('Test for Api Name and Desc', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        restApiName: 'RestApi-Regional',
        description: 'Description for the Regional Rest Api'
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-name-desc', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        Name: 'RestApi-Regional',
        Description: 'Description for the Regional Rest Api'
    });
});
// --------------------------------------------------------------
// Check for Overriden IAM Role
// --------------------------------------------------------------
test('Test for overriden IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const policyJSON = {
        Version: "2012-10-17",
        Statement: [
            {
                Action: [
                    "iot:UpdateThingShadow"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/mything1`,
                Effect: "Allow"
            },
            {
                Action: [
                    "iot:Publish"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/topic-abc`,
                Effect: "Allow"
            }
        ]
    };
    const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
    const iamRoleProps = {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        path: '/',
        inlinePolicies: { testPolicy: policyDocument }
    };
    // Create a policy that overrides the default policy that gets created with the construct
    const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayExecutionRole,
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-overriden-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/mything1"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/topic-abc"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "testPolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Api Key Source
// --------------------------------------------------------------
test('Test for APi Key Source', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        apiKeySourceType: api.ApiKeySourceType.AUTHORIZER,
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key-source', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        ApiKeySourceType: "AUTHORIZER"
    });
});
// --------------------------------------------------------------
// Check for Api Key Creation
// --------------------------------------------------------------
test('Test for Api Key Creation', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayCreateApiKey: true
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key', props);
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Properties: {
            ApiKeyRequired: true
        },
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W59"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------
// Test deployment for ApiGateway endPointCongiurationOverride
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-auth-none', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.REGIONAL]
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        EndpointConfiguration: {
            Types: ["REGIONAL"]
        }
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway AuthorizationType to NONE
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
});
// -----------------------------------------------------------------
// Test deployment for fully qualified iotEndpoint name
// -----------------------------------------------------------------
test('Test for handling fully qualified iotEndpoint', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats.iot.ap-south-1.amazonaws.com',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Integration: {
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        ":a1234567890123-ats.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}"
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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