import jinja2
import pluggy
import pytest
import pathlib

from render_engine import Page

@pytest.fixture()
def page_from_file(tmp_path: pathlib.Path):
    d = tmp_path / "test_page.md"
    content = """---
title: Test Page
custom: "test"
---

# Test Page
This is a test page
"""
    d.write_text(content)

    class CustomPage(Page):
        content_path = d

    return CustomPage()


def test_page_attrs_from_file(page_from_file: Page):
    """Tests that expected page attrsibutes are set from the file"""
    assert page_from_file._title == "Test Page"


def test_page_custom_attrs_from_file(page_from_file: Page):
    """Tests that unique page attrsibutes are set from the file"""
    assert page_from_file.custom == "test"


def test_page_from_template(tmp_path: pathlib.Path):
    """Tests that page attributes are set from a template"""

    class CustomPage(Page):
        template = "test.html"
        title = "Test Page"
        template = "test.html"

    environment = jinja2.Environment(
        loader=jinja2.DictLoader({"test.html": "{{ title }}"})
    )

    page = CustomPage()
    assert page._render_content(engine=environment) == "Test Page"



def test_page_content_renders_jinja():
    """Tests that page content is rendered with jinja"""

    class CustomPage(Page):
        content = "Test Page"

    page = CustomPage()
    assert page.content == "Test Page"
    assert page._content == "Test Page"