"use strict";
// Copyright (c) Martin Renou
// Distributed under the terms of the Modified BSD License.
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const THREE = __importStar(require("three"));
const base_1 = require("@jupyter-widgets/base");
const version_1 = require("./version");
// Import the CSS
require("../css/gany.css");
const ganyjs_1 = require("ganyjs");
function deserialize_float32array(data, manager) {
    return new Float32Array(data.data.buffer);
}
function deserialize_uint32array(data, manager) {
    return new Uint32Array(data.data.buffer);
}
function deserialize_component_array(value, manager) {
    if (typeof value == 'string') {
        return base_1.unpack_models(value, manager);
    }
    else {
        return deserialize_float32array(value, manager);
    }
}
class _GanyWidgetModel extends base_1.WidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_module: _GanyWidgetModel.model_module, _model_module_version: _GanyWidgetModel.model_module_version });
    }
}
_GanyWidgetModel.model_module = version_1.MODULE_NAME;
_GanyWidgetModel.model_module_version = version_1.MODULE_VERSION;
class _GanyDOMWidgetModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_module: _GanyDOMWidgetModel.model_module, _model_module_version: _GanyDOMWidgetModel.model_module_version, _view_module: _GanyDOMWidgetModel.view_module, _view_module_version: _GanyDOMWidgetModel.view_module_version });
    }
}
_GanyDOMWidgetModel.model_module = version_1.MODULE_NAME;
_GanyDOMWidgetModel.model_module_version = version_1.MODULE_VERSION;
_GanyDOMWidgetModel.view_module = version_1.MODULE_NAME;
_GanyDOMWidgetModel.view_module_version = version_1.MODULE_VERSION;
class ComponentModel extends _GanyWidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: ComponentModel.model_name, name: '', array: [] });
    }
    get array() {
        const array = this.get('array');
        if (array.hasOwnProperty('name') && array.name == 'NDArrayModel') {
            return array.getNDArray().data;
        }
        else {
            return array;
        }
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.component = new ganyjs_1.Component(this.get('name'), this.array);
        this.on('change:array', () => { this.component.array = this.array; });
    }
}
ComponentModel.serializers = Object.assign({}, _GanyWidgetModel.serializers, { array: { deserialize: deserialize_component_array } });
ComponentModel.model_name = 'ComponentModel';
exports.ComponentModel = ComponentModel;
class DataModel extends _GanyWidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: DataModel.model_name, name: '', components: [] });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        const components = this.get('components').map((componentModel) => {
            return componentModel.component;
        });
        this.data = new ganyjs_1.Data(this.get('name'), components);
    }
}
DataModel.serializers = Object.assign({}, _GanyWidgetModel.serializers, { components: { deserialize: base_1.unpack_models } });
DataModel.model_name = 'DataModel';
exports.DataModel = DataModel;
class BlockModel extends _GanyWidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: BlockModel.model_name, vertices: [], data: [], environment_meshes: [], default_color: '#6395b0' });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.block = this.createBlock();
        this.block.defaultColor = this.defaultColor;
        this.initEventListeners();
    }
    get vertices() {
        return this.get('vertices');
    }
    get data() {
        return this.get('data').map((dataModel) => dataModel.data);
        ;
    }
    get environmentMeshes() {
        return this.get('environment_meshes').map((model) => model.obj);
    }
    get defaultColor() {
        return this.get('default_color');
    }
    initEventListeners() {
        this.on('change:default_color', () => { this.block.defaultColor = this.defaultColor; });
    }
}
BlockModel.serializers = Object.assign({}, _GanyWidgetModel.serializers, { vertices: { deserialize: deserialize_float32array }, data: { deserialize: base_1.unpack_models }, environment_meshes: { deserialize: base_1.unpack_models } });
BlockModel.model_name = 'BlockModel';
class PolyMeshModel extends BlockModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: PolyMeshModel.model_name, triangle_indices: [] });
    }
    createBlock() {
        return new ganyjs_1.PolyMesh(this.vertices, this.triangleIndices, this.data, { environmentMeshes: this.environmentMeshes });
    }
    get triangleIndices() {
        return this.get('triangle_indices');
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:vertices', () => { this.block.vertices = this.vertices; });
        this.on('change:triangle_indices', () => { this.block.triangleIndices = this.triangleIndices; });
    }
}
PolyMeshModel.serializers = Object.assign({}, BlockModel.serializers, { triangle_indices: { deserialize: deserialize_uint32array } });
PolyMeshModel.model_name = 'PolyMeshModel';
exports.PolyMeshModel = PolyMeshModel;
class TetraMeshModel extends PolyMeshModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: TetraMeshModel.model_name, tetrahedron_indices: [] });
    }
    createBlock() {
        return new ganyjs_1.TetraMesh(this.vertices, this.triangleIndices, this.tetrahedronIndices, this.data, { environmentMeshes: this.environmentMeshes });
    }
    get tetrahedronIndices() {
        return this.get('tetrahedron_indices');
    }
}
TetraMeshModel.serializers = Object.assign({}, PolyMeshModel.serializers, { tetrahedron_indices: { deserialize: deserialize_uint32array } });
TetraMeshModel.model_name = 'TetraMeshModel';
exports.TetraMeshModel = TetraMeshModel;
class PointCloudModel extends BlockModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: PointCloudModel.model_name });
    }
    createBlock() {
        return new ganyjs_1.PointCloud(this.vertices, this.data, { environmentMeshes: this.environmentMeshes });
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:vertices', () => { this.block.vertices = this.vertices; });
    }
}
PointCloudModel.model_name = 'PointCloudModel';
exports.PointCloudModel = PointCloudModel;
class EffectModel extends BlockModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: EffectModel.model_name, parent: null, input: null });
    }
    get parent() {
        return this.get('parent');
    }
    get input() {
        return this.get('input');
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:input', this.updateInput.bind(this));
    }
    updateInput() {
        if (this.block.inputDimension != 0) {
            this.block.setInput(this.input);
        }
    }
}
EffectModel.serializers = Object.assign({}, BlockModel.serializers, { parent: { deserialize: base_1.unpack_models } });
EffectModel.model_name = 'EffectModel';
class WarpModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: WarpModel.model_name });
    }
    get input() {
        return this.get('input');
    }
    get offset() {
        const offset = this.get('offset');
        if (typeof offset == 'number') {
            return new THREE.Vector3(offset, offset, offset);
        }
        else {
            return new THREE.Vector3(offset[0], offset[1], offset[2]);
        }
    }
    get factor() {
        const factor = this.get('factor');
        if (typeof factor == 'number') {
            return new THREE.Vector3(factor, factor, factor);
        }
        else {
            return new THREE.Vector3(factor[0], factor[1], factor[2]);
            ;
        }
    }
    createBlock() {
        return new ganyjs_1.Warp(this.parent.block, this.input, this.factor, this.offset);
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:factor', () => { this.block.factor = this.factor; });
        this.on('change:offset', () => { this.block.offset = this.offset; });
    }
}
WarpModel.model_name = 'WarpModel';
exports.WarpModel = WarpModel;
class AlphaModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: AlphaModel.model_name });
    }
    get input() {
        const input = this.get('input');
        return typeof input == 'string' ? input : [input];
    }
    createBlock() {
        return new ganyjs_1.Alpha(this.parent.block, this.input);
    }
}
AlphaModel.model_name = 'AlphaModel';
exports.AlphaModel = AlphaModel;
class RGBModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: RGBModel.model_name });
    }
    get input() {
        return this.get('input');
    }
    createBlock() {
        return new ganyjs_1.RGB(this.parent.block, this.input);
    }
}
RGBModel.model_name = 'RGBModel';
exports.RGBModel = RGBModel;
class IsoColorModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: IsoColorModel.model_name, min: 0., max: 0. });
    }
    get min() {
        return this.get('min');
    }
    get max() {
        return this.get('max');
    }
    get input() {
        const input = this.get('input');
        return typeof input == 'string' ? input : [input];
    }
    createBlock() {
        return new ganyjs_1.IsoColor(this.parent.block, this.input, this.min, this.max);
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:min', () => { this.block.min = this.min; });
        this.on('change:max', () => { this.block.max = this.max; });
    }
}
IsoColorModel.model_name = 'IsoColorModel';
exports.IsoColorModel = IsoColorModel;
class IsoSurfaceModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: IsoSurfaceModel.model_name, value: 0., dynamic: false });
    }
    get value() {
        return this.get('value');
    }
    get dynamic() {
        return this.get('dynamic');
    }
    get input() {
        const input = this.get('input');
        return typeof input == 'string' ? input : [input];
    }
    createBlock() {
        return new ganyjs_1.IsoSurface(this.parent.block, this.input, { value: this.value, dynamic: this.dynamic });
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:value', () => { this.block.value = this.value; });
    }
}
IsoSurfaceModel.model_name = 'IsoSurfaceModel';
exports.IsoSurfaceModel = IsoSurfaceModel;
class ThresholdModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: ThresholdModel.model_name, min: 0., max: 0., dynamic: false });
    }
    get min() {
        return this.get('min');
    }
    get max() {
        return this.get('max');
    }
    get dynamic() {
        return this.get('dynamic');
    }
    get input() {
        const input = this.get('input');
        return typeof input == 'string' ? input : [input];
    }
    createBlock() {
        return new ganyjs_1.Threshold(this.parent.block, this.input, { min: this.min, max: this.max, dynamic: this.dynamic });
    }
    initEventListeners() {
        super.initEventListeners();
        this.on('change:min', () => { this.block.min = this.min; });
        this.on('change:max', () => { this.block.max = this.max; });
    }
}
ThresholdModel.model_name = 'ThresholdModel';
exports.ThresholdModel = ThresholdModel;
class UnderWaterModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: UnderWaterModel.model_name });
    }
    get input() {
        const input = this.get('input');
        return typeof input == 'string' ? input : [input];
    }
    createBlock() {
        return new ganyjs_1.UnderWater(this.parent.block, this.input);
    }
}
UnderWaterModel.model_name = 'UnderWaterModel';
exports.UnderWaterModel = UnderWaterModel;
class WaterModel extends EffectModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: WaterModel.model_name, under_water_blocks: [] });
    }
    get underWaterBlocks() {
        return this.get('under_water_blocks').map((underWaterBlockWidget) => underWaterBlockWidget.block);
    }
    createBlock() {
        return new ganyjs_1.Water(this.parent.block, { underWaterBlocks: this.underWaterBlocks, causticsEnabled: false });
    }
}
WaterModel.model_name = 'WaterModel';
WaterModel.serializers = Object.assign({}, EffectModel.serializers, { under_water_blocks: { deserialize: base_1.unpack_models } });
exports.WaterModel = WaterModel;
class SceneModel extends _GanyDOMWidgetModel {
    defaults() {
        return Object.assign({}, super.defaults(), { _model_name: SceneModel.model_name, _view_name: SceneModel.view_name, background_color: '#fff', background_opacity: 0., children: [] });
    }
    initialize(attributes, options) {
        super.initialize(attributes, options);
        this.scene = new ganyjs_1.Scene();
        this.updateChildren();
        this.on('change:children', this.updateChildren.bind(this));
    }
    get backgroundColor() {
        return this.get('background_color');
    }
    get backgroundOpacity() {
        return this.get('background_opacity');
    }
    updateChildren() {
        // TODO: Remove old children
        const blocks = this.get('children').map((child) => child.block);
        if (blocks.length == 0) {
            return;
        }
        const boundingSphereRadius = Math.max(...blocks.map((block) => block.boundingSphere.radius));
        const scale = new THREE.Vector3(1 / boundingSphereRadius, 1 / boundingSphereRadius, 1 / boundingSphereRadius);
        const position = blocks[0].boundingSphere.center;
        for (const block of blocks) {
            block.scale = scale;
            block.position = new THREE.Vector3(-position.x, -position.y, -position.z);
            this.scene.addBlock(block);
        }
    }
}
SceneModel.serializers = Object.assign({}, _GanyDOMWidgetModel.serializers, { children: { deserialize: base_1.unpack_models } });
SceneModel.model_name = 'SceneModel';
SceneModel.view_name = 'SceneView';
exports.SceneModel = SceneModel;
class SceneView extends base_1.DOMWidgetView {
    render() {
        this.el.classList.add('gany-scene');
        this.renderer = new ganyjs_1.Renderer(this.el, this.model.scene);
        this.displayed.then(() => {
            this.renderer.initialize();
            this.renderer.backgroundColor = this.model.backgroundColor;
            this.renderer.backgroundOpacity = this.model.backgroundOpacity;
            this.initEventListeners();
        });
    }
    initEventListeners() {
        window.addEventListener('resize', this.resize.bind(this), false);
        this.model.on('change:background_color', () => { this.renderer.backgroundColor = this.model.backgroundColor; });
        this.model.on('change:background_opacity', () => { this.renderer.backgroundOpacity = this.model.backgroundOpacity; });
    }
    processPhosphorMessage(msg) {
        super.processPhosphorMessage(msg);
        switch (msg.type) {
            case 'resize':
            case 'after-show':
                this.resize();
                break;
        }
    }
    remove() {
        this.renderer.dispose();
        return super.remove();
    }
    resize() {
        this.renderer.resize();
    }
}
exports.SceneView = SceneView;
//# sourceMappingURL=widget.js.map