"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const stepfunction = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack) {
    const sourceOutput = new codepipeline.Artifact();
    const startState = new stepfunction.Pass(stack, 'StartState');
    const simpleStateMachine = new stepfunction.StateMachine(stack, 'SimpleStateMachine', {
        definition: startState,
    });
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline');
    const sourceStage = pipeline.addStage({
        stageName: 'Source',
        actions: [
            new cpactions.S3SourceAction({
                actionName: 'Source',
                bucket: new s3.Bucket(stack, 'MyBucket'),
                bucketKey: 'some/path/to',
                output: sourceOutput,
                trigger: cpactions.S3Trigger.POLL,
            }),
        ],
    });
    pipeline.addStage({
        stageName: 'Invoke',
        actions: [
            new cpactions.StepFunctionInvokeAction({
                actionName: 'Invoke',
                stateMachine: simpleStateMachine,
                stateMachineInput: cpactions.StateMachineInput.literal({ IsHelloWorldExample: true }),
            }),
        ],
    });
    return sourceStage;
}
module.exports = {
    'StepFunctions Invoke Action': {
        'Verify stepfunction configuration properties are set to specific values'(test) {
            const stack = new core_1.Stack();
            // when
            minimalPipeline(stack);
            // then
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    //  Must have a source stage
                    {
                        Actions: [
                            {
                                ActionTypeId: {
                                    Category: 'Source',
                                    Owner: 'AWS',
                                    Provider: 'S3',
                                    Version: '1',
                                },
                                Configuration: {
                                    S3Bucket: {
                                        Ref: 'MyBucketF68F3FF0',
                                    },
                                    S3ObjectKey: 'some/path/to',
                                },
                            },
                        ],
                    },
                    // Must have stepfunction invoke action configuration
                    {
                        Actions: [
                            {
                                ActionTypeId: {
                                    Category: 'Invoke',
                                    Owner: 'AWS',
                                    Provider: 'StepFunctions',
                                    Version: '1',
                                },
                                Configuration: {
                                    StateMachineArn: {
                                        Ref: 'SimpleStateMachineE8E2CF40',
                                    },
                                    InputType: 'Literal',
                                    // JSON Stringified input when the input type is Literal
                                    Input: '{\"IsHelloWorldExample\":true}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'Allows the pipeline to invoke this stepfunction'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: ['states:StartExecution', 'states:DescribeStateMachine'],
                            Resource: {
                                Ref: 'SimpleStateMachineE8E2CF40',
                            },
                            Effect: 'Allow',
                        },
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role'));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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