"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
/**
 * Make a Token that renders to given region if used in a different stack, otherwise undefined
 */
function regionIfDifferentFromStack(region) {
    return cdk.Token.asString(new StackDependentToken(region, stack => stack.region));
}
exports.regionIfDifferentFromStack = regionIfDifferentFromStack;
/**
 * Make a Token that renders to given account if used in a different stack, otherwise undefined
 */
function accountIfDifferentFromStack(account) {
    return cdk.Token.asString(new StackDependentToken(account, stack => stack.account));
}
exports.accountIfDifferentFromStack = accountIfDifferentFromStack;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(originalValue, fn) {
        this.originalValue = originalValue;
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        const stackValue = this.fn(cdk.Stack.of(context.scope));
        // Don't render if the values are definitely the same. If the stack
        // is unresolved we don't know, better output the value.
        if (!cdk.Token.isUnresolved(stackValue) && stackValue === this.originalValue) {
            return undefined;
        }
        return this.originalValue;
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return this.originalValue;
    }
}
//# sourceMappingURL=data:application/json;base64,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