"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Test = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const synthetics_canned_metrics_generated_1 = require("./synthetics-canned-metrics.generated");
const aws_synthetics_1 = require("aws-cdk-lib/aws-synthetics");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CustomTestOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.custom);
            }
            throw error;
        }
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
}
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Test", version: "2.72.0-alpha.0" };
exports.Test = Test;
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CanaryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Canary);
            }
            throw error;
        }
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = props.artifactsBucketLocation?.bucket ?? new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            enforceSSL: true,
        });
        this.role = props.role ?? this.createDefaultRole(props);
        if (props.vpc) {
            // Security Groups are created and/or appended in `createVpcConfig`.
            this._connections = new ec2.Connections({});
        }
        const resource = new aws_synthetics_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject(props.artifactsBucketLocation?.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: props.startAfterCreation ?? true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: props.failureRetentionPeriod?.toDays(),
            successRetentionPeriod: props.successRetentionPeriod?.toDays(),
            code: this.createCode(props),
            runConfig: this.createRunConfig(props),
            vpcConfig: this.createVpcConfig(props),
            deleteLambdaResourcesOnCanaryDeletion: props.enableAutoDeleteLambdas,
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Canary
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Canaries have security groups to manage. Supply the "vpc" parameter when creating the Canary.');
        }
        return this._connections;
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return new aws_cloudwatch_1.Metric({
            ...synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationMaximum({ CanaryName: this.canaryName }),
            ...{ statistic: 'Average' },
            ...options,
        }).attachTo(this);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(props) {
        const prefix = props.artifactsBucketLocation?.prefix;
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.bucketArn],
                    actions: ['s3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [this.logGroupArn()],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        const managedPolicies = [];
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
            managedPolicies,
        });
    }
    logGroupArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: '/aws/lambda/cwsyn-*',
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler, props.runtime.family),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: codeConfig.s3Location?.bucketName,
            s3Key: codeConfig.s3Location?.objectKey,
            s3ObjectVersion: codeConfig.s3Location?.objectVersion,
        };
    }
    createRunConfig(props) {
        if (!props.environmentVariables) {
            return undefined;
        }
        return {
            environmentVariables: props.environmentVariables,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        return {
            durationInSeconds: String(`${props.timeToLive?.toSeconds() ?? 0}`),
            expression: props.schedule?.expressionString ?? 'rate(5 minutes)',
        };
    }
    createVpcConfig(props) {
        if (!props.vpc) {
            if (props.vpcSubnets != null || props.securityGroups != null) {
                throw new Error("You must provide the 'vpc' prop when using VPC-related properties.");
            }
            return undefined;
        }
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnetIds.length < 1) {
            throw new Error('No matching subnets found in the VPC.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Canary ' + cdk.Names.uniqueId(this),
            });
            securityGroups = [securityGroup];
        }
        this._connections.addSecurityGroup(...securityGroups);
        return {
            vpcId: props.vpc.vpcId,
            subnetIds,
            securityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
_b = JSII_RTTI_SYMBOL_1;
Canary[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.Canary", version: "2.72.0-alpha.0" };
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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