from typing import List, Any, Dict

from plotagain.iddict import IDDict
from plotagain.utils import is_primitive, insert_next_available


class PyplotCall:
    """
    Represent a single call to some function contained in the pyplot module. Stores the call's function name, positioned
    args and keyword kwargs. Used to store and save the arguments as pickles and generate a line of code to re-create
    the call

    Attributes
    ----------
    fn_name
        The name of the called function
    call_args
        A list of the positioned arguments passed into the function
    call_kwargs
        A dict of the keyword arguments passed into the function
    recall_args
        A list containing the names of the variable in the autogenerated script corresponding to each element in the
        original call. Variable names are inferred from the locals() and globals() dicts in the scope of the plotting
        code. Any object not contained in either dict is assigned the variable name 'unnamed_arg', with an integer
        post-fix if required,  in the autogenerated script. If the argument is not in locals() or globals() and is a
        primitive type, see utils.is_primitive, then no variable is created for this object in the autogenerated script
        and repr(arg) is used directly
    recall_kwargs
        Same as recall_args except for the keyword arguments for the function call. Instead of unnamed_arg post-fixed
        with an integer, the key of the kwarg is used with a post-fixed integer if required
    """
    fn_name: str
    call_args: List[Any]
    call_kwargs: Dict[str, Any]

    recall_args: List[Any]
    recall_kwargs: Dict[str, str]

    def __init__(
        self,
        fn_name,
        call_args,
        call_kwargs,
    ):
        """
        Parameters
        ----------
        fn_name
            The name of the matplotlib.pyplot function called
        call_args
            A list of the positioned arguments passed into the function
        call_kwargs
            A dict of the keyword arguments passed into the function
        """
        self.fn_name = fn_name
        self.call_args = call_args
        self.call_kwargs = call_kwargs

        self.recall_args = []
        self.recall_kwargs = {}

    def find_or_name_obj(
        self,
        outer_scope: IDDict,
        used_variables: Dict[str, Any],
        obj: Any,
        default_name: str = 'unnamed_arg'
    ) -> str:
        """
        Assigns a variable name for obj in the autogenerated script. If obj is found as a value in outer_scope, the key
        is used as its variable name. Otherwise, default_name, with an integer postfix is used. If the object is a
        primitive type (see utils.is_primitive) then the literal value of repr(obj) is returned instead

        Parameters
        ----------
        outer_scope
            Union of the locals() and globals() dict in the scope of the plotting code. Used to infer what variable
            names to use
        used_variables
            A dict of all named variables in the autogenerated script
        obj
            An object to name. If the object is in outer_scope, the object is named the key of the obj in outer_scope.
            If obj is not in outer_scope, default_name is used with an integer post-fix, if required.
        default_name
            The default name to use if obj is not found in outer_scope. Defaults to 'unnamed_arg'

        Returns
        -------
        str
            The name or literal value of the object for use in the auto-generated script
        """
        reversed_named_variables = outer_scope.reverse
        reversed_named_variables.update(used_variables, reverse=True)

        if obj not in reversed_named_variables and is_primitive(obj):
            name = repr(obj)
        else:
            if obj not in reversed_named_variables:
                name = insert_next_available(used_variables, default_name, obj)
            else:
                name = reversed_named_variables[obj]
            reversed_named_variables[obj] = name
            used_variables[name] = obj

        return name

    def find_or_name_args(self, outer_scope: IDDict, used_variables: Dict[str, Any]) -> None:
        """
        Assigns a variable name to each positioned and keyword argument in the original call for use in the
        autogenerated script. See PyplotCall.find_or_name_obj

        Parameters
        ----------
        outer_scope
            Union of the locals() and globals() dict in the scope of the plotting code. Used to infer what variable
            names to use
        used_variables
            A mapping from the variable name to the objects passed into any other matplotlib.pyplot call
        """
        self.find_or_name_positional_args(outer_scope, used_variables)
        self.find_or_name_kwargs(outer_scope, used_variables)

    def find_or_name_positional_args(self, outer_scope: IDDict, used_variables: Dict[str, Any]) -> None:
        """
        Assigns a variable name to each positioned argument in the original call for use in the autogenerated script.
        Any argument not found in outer_scope is assigned the name unnamed_arg, with an integer post-fix if required.
        The object is added to used_variables

        Parameters
        ----------
        outer_scope
            Union of the locals() and globals() dict in the scope of the plotting code. Used to infer what variable
            names to use
        used_variables
            A mapping from the variable name to the objects passed into any other matplotlib.pyplot call
        """
        for arg in self.call_args:
            recall_arg = self.find_or_name_obj(outer_scope, used_variables, arg)
            self.recall_args.append(recall_arg)

    def find_or_name_kwargs(self, outer_scope: IDDict, used_variables: Dict[str, Any]) -> None:
        """
        Assigns a variable name to each keyword argument in the original call for use in the autogenerated script. Any
        argument not found in outer_scope is assigned the name of the key of the argument, with an integer post-fix if
        required. The object is added to used_variables

        Parameters
        ----------
        outer_scope
            Union of the locals() and globals() dict in the scope of the plotting code. Used to infer what variable
            names to use
        used_variables
            A mapping from the variable name to the objects passed into any other matplotlib.pyplot call
        """
        for key, value in self.call_kwargs.items():
            recall_arg = self.find_or_name_obj(outer_scope, used_variables, value, default_name=key)
            self.recall_kwargs[key] = recall_arg

    def render_recall(self) -> str:
        """
        Constructs the line of code which re-creates this pyplot call using the variable names in the autogenerated
        script

        Returns
        -------
        str
            A line of code used to recreate the function call in the autogenerated script
        """
        all_args = self.recall_args + [f"{k}={v}" for k, v in self.recall_kwargs.items()]
        call_str = self.fn_name + '(' + ", ".join(all_args) + ')'
        return f'plt.{call_str}'
