# -*- coding: utf-8 -*-
#------------------------------------------------------------------------------
# file: $Id$
# auth: metagriffin <mg.bitbucket@metagriffin.net>
# date: 2015/10/15
# copy: (C) Copyright 2015-EOT metagriffin -- see LICENSE.txt
#------------------------------------------------------------------------------
# This software is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This software is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see http://www.gnu.org/licenses/.
#------------------------------------------------------------------------------

'''better archive fingerprint file composition'''

import os.path
import time
import uuid
import mimetypes
import shlex
import re
import json
import xml.etree.ElementTree as ET
try:
  import yaml
except ImportError:
  yaml = None

from mercurial.i18n import _
from mercurial import commands, extensions, scmutil, cmdutil, util, encoding
from mercurial.node import hex
from mercurial import archival

try:
  from mercurial import registrar
  libcommand = registrar.command
except ImportError:
  libcommand = cmdutil.command

#------------------------------------------------------------------------------

testedwith              = '2.8.2'
buglink                 = 'https://bitbucket.org/metagriffin/hg-fingerprint/issues'
DEFAULT_FP_PATH         = '.hg_archival.txt'
DEFAULT_FP_MAGIC        = 'repo: '
cmdtable                = {}
command                 = libcommand(cmdtable)

#------------------------------------------------------------------------------
@command(
  '^fingerprint|fp',
  [
    ('f', 'format',  '',   _('output format (text, json, yaml, xml)'), _('FORMAT')),
    ('d', 'no-date', None, _('exclude current timestamp from fingerprint')),
    ('r', 'rev',     '',   _('revision to create fingerprint for'), _('REV')),
  ],
  _('hg fingerprint [OPTIONS]...'))
def fingerprint_command(ui, repo, **opts):
  '''Generates a repository fingerprint file and adds options to'''\
  ''' control the fingerprint file generated by the :hg:`archive` command.'''
  node  = scmutil.revsingle(repo, opts.get('rev', '')).node()
  ndate = opts.get('no_date')
  date  = ndate is False \
    or ( ndate is None and repo.ui.configbool('fingerprint', 'timestamp', True) )
  fmt   = opts.get('format')
  if not fmt:
    fmt = shlex.split(repo.ui.config('fingerprint', 'format', 'auto').lower())[0]
  meta = getMetadata(ui, repo, node)
  fmt  = resolveFormat(ui, repo, '<stdout>', fmt)
  data = renderMetadata(ui, repo, meta, fmt, date)
  ui.write(data)

#------------------------------------------------------------------------------
def md2_text_plain(repo, meta):
  ret = ''
  for key, val in meta.items():
    if key not in ('tags', 'latesttags'):
      ret += '{key}: {value}\n'.format(key=key, value=val)
    else:
      for item in val:
        ret += '{key}: {value}\n'.format(key=key[:-1], value=item)
  return ret
md2_text = md2_text_plain

#------------------------------------------------------------------------------
def md2_application_json(repo, meta):
  return json.dumps(meta) + '\n'
md2_json = md2_application_json

#------------------------------------------------------------------------------
def md2_application_yaml(repo, meta):
  if not yaml:
    raise util.Abort(
      _('fingerprint format "yaml" requires installation of a YAML'
        ' serializer (try "pip install PyYAML")'))
  return yaml.dump(meta)
md2_yaml = md2_application_yaml

#------------------------------------------------------------------------------
def md2_application_xml(repo, meta):
  root = ET.Element('fingerprint')
  for key, val in meta.items():
    if key not in ('tags', 'latesttags'):
      node = ET.SubElement(root, key)
      node.text = str(val)
    else:
      for item in val:
        node = ET.SubElement(root, key[:-1])
        node.text = item
  return ET.tostring(root, 'UTF-8') + '\n'
md2_xml = md2_application_xml

#------------------------------------------------------------------------------
badformatchars_cre = re.compile('[^a-zA-Z0-9_]')
def renderMetadata(ui, repo, meta, fmt, date):
  if date:
    meta['timestamp'] = time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime())
  if not fmt or fmt == 'auto':
    fmt = 'plain/text'
  try:
    func = eval('md2_' + badformatchars_cre.sub('_', fmt).lower())
  except NameError:
    raise util.Abort(_('unknown fingerprint format "{format}"').format(format=fmt))
  return func(repo, meta)

#------------------------------------------------------------------------------
def getMetadata(ui, repo, node):
  ctx   = repo[node]
  meta  = dict(
    repo   = repo[0].hex(),
    node   = hex(node),
    branch = encoding.fromlocal(ctx.branch()),
  )
  tags = [str(t) for t in ctx.tags() if repo.tagtype(t) == 'global']
  if tags:
    meta['tags'] = tags
  else:
    repo.ui.pushbuffer()
    opts = {
      'template' : '{latesttag}\0{latesttagdistance}',
      'style'    : '',
      'patch'    : None,
      'git'      : None,
    }
    cmdutil.show_changeset(repo.ui, repo, opts).show(ctx)
    ltags, dist = repo.ui.popbuffer().split('\0')
    meta.update(
      latesttags        = [str(t) for t in ltags.split(':')],
      latesttagdistance = dist,
    )
    try:
      # todo: is `latesttagdistance` *always* an int?...
      meta['latesttagdistance'] = int(meta['latesttagdistance'])
    except: pass
  return meta

#------------------------------------------------------------------------------
def resolveFormat(ui, repo, filename, fmt):
  if fmt == 'auto':
    fmt = mimetypes.guess_type(filename)[0]
    if fmt is None and filename.endswith('.yaml'):
      fmt = 'application/yaml'
  return fmt

#------------------------------------------------------------------------------
class ProxyArchiver(object):
  def __init__(self, repo, node, prefix, ctor):
    self.repo    = repo
    self.node    = node
    self.prefix  = prefix
    self.fppath  = os.path.normpath(os.path.join(prefix, DEFAULT_FP_PATH))
    self.formats = shlex.split(repo.ui.config('fingerprint', 'format', 'auto').lower())
    self.ctor    = ctor
    self.proxy   = None
  def __call__(self, *args, **kw):
    if self.proxy:
      raise ValueError('archiver cannot be double-constructed')
    self.proxy = self.ctor(*args, **kw)
    return self
  def addfile(self, name, mode, link, data, *args, **kw):
    if not self.proxy:
      raise ValueError('archiver has not been constructed')
    if not (
        ( name == DEFAULT_FP_PATH or os.path.normpath(name) == self.fppath )
        and data.startswith(DEFAULT_FP_MAGIC)
      ):
      return self.proxy.addfile(name, mode, link, data, *args, **kw)
    date = self.repo.ui.configbool('fingerprint', 'timestamp', True)
    meta = getMetadata(self.repo.ui, self.repo, self.node)
    for idx, name in enumerate(
        shlex.split(self.repo.ui.config('fingerprint', 'path', DEFAULT_FP_PATH))):
      name = os.path.join(self.prefix, name)
      fmt  = self.formats[idx if idx < len(self.formats) else -1]
      fmt  = resolveFormat(self.repo.ui, self.repo, name, fmt)
      data = renderMetadata(self.repo.ui, self.repo, meta, fmt, date)
      self.proxy.addfile(name, mode, link, data, *args, **kw)
  def done(self, *args, **kw):
    if not self.proxy:
      raise ValueError('archiver has not been constructed')
    self.proxy.done(*args, **kw)

#------------------------------------------------------------------------------
class ProxyArchiverRegistry(object):
  def __init__(self, proxy):
    self.proxy  = proxy
    self.repo   = None
    self.node   = None
    self.prefix = None
  def __getitem__(self, archiver):
    if not self.repo:
      raise TypeError('`setRepo()` not called... call monkeypatch_archival!')
    return ProxyArchiver(self.repo, self.node, self.prefix, self.proxy[archiver])
  def __contains__(self, archiver):
    return archiver in self.proxy
  def setRepo(self, repo, node, prefix):
    self.repo   = repo
    self.node   = node
    self.prefix = prefix

#------------------------------------------------------------------------------
def proxyArchive(repo, dest, node, kind, *args, **kw):
  prefix = ''
  if 'prefix' in kw:
    prefix = kw.get('prefix', '')
  else:
    if len(args) >= 3:
      prefix = args[2] or ''
  if kind != 'files':
    prefix = archival.tidyprefix(dest, kind, prefix)
  archival.archivers.setRepo(repo, node, prefix)
  return archival._fingerprint_archive(repo, dest, node, kind, *args, **kw)

#------------------------------------------------------------------------------
def monkeypatch_archival(ui):
  archival._fingerprint_archivers = archival.archivers
  archival.archivers = ProxyArchiverRegistry(archival.archivers)
  # todo: use `extensions.wrapcommand` instead... problem is, i
  #       can't get it to work, and the docs are non-existent.
  archival._fingerprint_archive = archival.archive
  archival.archive = proxyArchive

#------------------------------------------------------------------------------
def uisetup(ui):
  if not hasattr(archival, '_fingerprint_archivers'):
    monkeypatch_archival(ui)

#------------------------------------------------------------------------------
def extsetup(ui):
  try:
    keyword = extensions.find('keyword')
    keyword.restricted += ' fingerprint'
    try:
      keyword.recordextensions += ' fingerprint'
      keyword.recordcommands += ' fingerprint'
    except AttributeError:
      pass
  except KeyError:
    pass

#------------------------------------------------------------------------------
# end of $Id$
# $ChangeLog$
#------------------------------------------------------------------------------
