# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vban_cmd']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'vban-cmd',
    'version': '1.0.8',
    'description': 'Python interface for the VBAN RT Packet Service (Sendtext)',
    'long_description': '[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://github.com/onyx-and-iris/vban-cmd-python/blob/dev/LICENSE)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Imports: isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n![Tests Status](./tests/basic.svg?dummy=8484744)\n![Tests Status](./tests/banana.svg?dummy=8484744)\n![Tests Status](./tests/potato.svg?dummy=8484744)\n\n# VBAN CMD\n\nThis package offers a Python interface for the Voicemeeter RT Packet Service as well as Voicemeeter VBAN-TEXT.\n\nThis allows a user to get (rt packets) and set (vban-text) parameters over a local network. Consider the Streamer View app over VBAN, for example.\n\nIt may be used standalone or to extend the [Voicemeeter Remote Python API](https://github.com/onyx-and-iris/voicemeeter-api-python)\n\nFor sending audio across a network with VBAN you will need to look elsewhere.\n\nFor an outline of past/future changes refer to: [CHANGELOG](CHANGELOG.md)\n\n## Tested against\n\n-   Basic 1.0.8.2\n-   Banana 2.0.6.2\n-   Potato 3.0.2.2\n\n## Requirements\n\n-   [Voicemeeter](https://voicemeeter.com/)\n-   Python 3.11 or greater\n\n## Installation\n\n### `Pip`\n\nInstall vban-cmd package from your console\n\n`pip install vban-cmd`\n\n## `Use`\n\nSimplest use case, use a context manager to request a VbanCmd class of a kind.\n\nLogin and logout are handled for you in this scenario.\n\n#### `__main__.py`\n\n```python\nimport vban_cmd\n\n\nclass ManyThings:\n    def __init__(self, vban):\n        self.vban = vban\n\n    def things(self):\n        self.vban.strip[0].label = "podmic"\n        self.vban.strip[0].mute = True\n        print(\n            f"strip 0 ({self.vban.strip[0].label}) has been set to {self.vban.strip[0].mute}"\n        )\n\n    def other_things(self):\n        info = (\n            f"bus 3 gain has been set to {self.vban.bus[3].gain}",\n            f"bus 4 eq has been set to {self.vban.bus[4].eq}",\n        )\n        self.vban.bus[3].gain = -6.3\n        self.vban.bus[4].eq = True\n        print("\\n".join(info))\n\n\ndef main():\n    with vban_cmd.api(kind_id, **opts) as vban:\n        do = ManyThings(vban)\n        do.things()\n        do.other_things()\n\n        # set many parameters at once\n        vban.apply(\n            {\n                "strip-2": {"A1": True, "B1": True, "gain": -6.0},\n                "bus-2": {"mute": True},\n            }\n        )\n\n\nif __name__ == "__main__":\n    kind_id = "banana"\n    opts = {\n        "ip": "<ip address>",\n        "streamname": "Command1",\n        "port": 6980,\n    }\n\n    main()\n```\n\nOtherwise you must remember to call `vban.login()`, `vban.logout()` at the start/end of your code.\n\n## `kind_id`\n\nPass the kind of Voicemeeter as an argument. kind_id may be:\n\n-   `basic`\n-   `banana`\n-   `potato`\n\n## `Available commands`\n\n### Channels (strip/bus)\n\nThe following properties exist for audio channels.\n\n-   `mono`: boolean\n-   `solo`: boolean\n-   `mute`: boolean\n-   `label`: string\n-   `gain`: float, -60 to 12\n-   `A1 - A5`, `B1 - B3`: boolean\n-   `comp`: float, from 0.0 to 10.0\n-   `gate`: float, from 0.0 to 10.0\n-   `limit`: int, from -40 to 12\n\nexample:\n\n```python\nvban.strip[3].gain = 3.7\nprint(strip[0].label)\n\nvban.bus[4].mono = true\n```\n\n### Command\n\nCertain \'special\' commands are defined by the API as performing actions rather than setting values. The following methods are available:\n\n-   `show()` : Bring Voiceemeter GUI to the front\n-   `shutdown()` : Shuts down the GUI\n-   `restart()` : Restart the audio engine\n\nThe following properties are write only and accept boolean values.\n\n-   `showvbanchat`: boolean\n-   `lock`: boolean\n\nexample:\n\n```python\nvban.command.restart()\nvban.command.showvbanchat = true\n```\n\n### Multiple parameters\n\n-   `apply`\n    Set many strip/bus parameters at once, for example:\n\n```python\nvban.apply(\n    {\n        "strip-2": {"A1": True, "B1": True, "gain": -6.0},\n        "bus-2": {"mute": True},\n    }\n)\n```\n\nOr for each class you may do:\n\n```python\nvban.strip[0].apply(mute: true, gain: 3.2, A1: true)\nvban.vban.outstream[0].apply(on: true, name: \'streamname\', bit: 24)\n```\n\n## Config Files\n\n`vban.apply_config(<configname>)`\n\nYou may load config files in TOML format.\nThree example profiles have been included with the package. Remember to save\ncurrent settings before loading a profile. To set one you may do:\n\n```python\nimport vban_cmd\nwith vban_cmd.api(\'banana\') as vban:\n    vban.apply_config(\'example\')\n```\n\nwill load a config file at configs/banana/example.toml for Voicemeeter Banana.\n\n## `Base Module`\n\n### VbanCmd class\n\n#### `vban.pdirty`\n\nTrue iff a parameter has been changed.\n\n#### `vban.ldirty`\n\nTrue iff a level value has been changed.\n\n#### `vban.sendtext(script)`\n\nSends a script block as a string request RT Packet, for example:\n\n```python\nvban.sendtext("Strip[0].Mute=1;Bus[0].Mono=1")\n```\n\n#### `vban.public_packet`\n\nReturns a Voicemeeter rt data packet object. Designed to be used internally by the interface but available for parsing through this read only property object. States not guaranteed to be current (requires use of dirty parameters to confirm).\n\n### `Errors`\n\n-   `errors.VMCMDErrors`: Base VMCMD error class.\n\n### `Tests`\n\nFirst make sure you installed the [development dependencies](https://github.com/onyx-and-iris/vban-cmd-python#installation)\n\nThen from tests directory:\n\n`pytest -v`\n\n## Resources\n\n-   [Voicemeeter VBAN TEXT](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=19)\n\n-   [Voicemeeter RT Packet Service](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=27)\n',
    'author': 'onyx-and-iris',
    'author_email': 'code@onyxandiris.online',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/onyx-and-iris/vban-cmd-python',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.11,<4.0',
}


setup(**setup_kwargs)
