"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManager({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.getSecretValue({ SecretId: event.ResourceProperties.githubTokenSecret });
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log('Create new ActionSecret with props ' + JSON.stringify(props));
    await createOrUpdateRepoSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log(`Update ActionSecret ${props.repositorySecretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateRepoSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit) => {
    console.log('Delete ActionSecret ' + event.ResourceProperties.repositorySecretName);
    await deleteRepoSecret(event, octokit);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const createOrUpdateRepoSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const secretId = event.ResourceProperties.sourceSecretArn;
    const secretToEncrypt = await smClient.getSecretValue({ SecretId: secretId });
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const secretResponse = await octokit.request('PUT /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteRepoSecret = async (event, octokit) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const buildPhysicalResourceId = async (event, octokit) => {
    const { repositorySecretName: secret, repositoryOwner, repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    return [secret, owner, repo].join('/');
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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