"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManager({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.getSecretValue({ SecretId: event.ResourceProperties.githubTokenSecret });
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    console.log('Create new ActionEnvironmentSecret with props ' + JSON.stringify(event.ResourceProperties));
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log(`Update ActionEnvironmentSecret ${props.repositorySecretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit) => {
    console.log('Delete ActionEnvironmentSecret ' + event.ResourceProperties.repositorySecretName);
    await deleteEnvironmentSecret(event, octokit);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const createOrUpdateEnvironmentSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name, environment: environment_name, sourceSecretArn: secretId, } = event.ResourceProperties;
    const secretToEncrypt = await smClient.getSecretValue({ SecretId: secretId });
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const repository_id = await getRepositoryId(event, octokit, owner);
    const secretResponse = await octokit.request('PUT /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteEnvironmentSecret = async (event, octokit) => {
    const { environment: environment_name, repositorySecretName: secret_name, repositoryOwner } = event.ResourceProperties;
    const repository_id = await getRepositoryId(event, octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const getRepositoryId = async (event, octokit, repositoryOwner) => {
    const { repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}', {
        owner,
        repo,
    });
    return data.id;
};
const buildPhysicalResourceId = async (event, octokit) => {
    const { environment, repositorySecretName: secret, repositoryOwner, repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    return [environment, secret, owner, repo].join('/');
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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