"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const statement = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An EC2 Key Pair.
 *
 * @stability stable
 */
class KeyPair extends constructs_1.Construct {
    /**
     * Defines a new EC2 Key Pair.
     *
     * The private key will be stored in AWS Secrets Manager
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.privateKeyArn = '';
        /**
         * ARN of the public key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.publicKeyArn = '';
        /**
         * The public key.
         *
         * Only filled, when `exposePublicKey = true`
         *
         * @stability stable
         */
        this.publicKeyValue = '';
        /**
         * Name of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairName = '';
        /**
         * ID of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairID = '';
        /**
         * @stability stable
         */
        this.prefix = '';
        if (props.removeKeySecretsAfterDays &&
            (props.removeKeySecretsAfterDays < 0 ||
                (props.removeKeySecretsAfterDays > 0 &&
                    props.removeKeySecretsAfterDays < 7) ||
                props.removeKeySecretsAfterDays > 30)) {
            cdk.Annotations.of(this).addError(`Parameter removeKeySecretsAfterDays must be 0 or between 7 and 30. Got ${props.removeKeySecretsAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        if (this.prefix.length + cleanID.length > 62)
            // Cloudformation limits names to 63 characters.
            cdk.Annotations.of(this).addError(`Cloudformation limits names to 63 characters.
         Prefix ${this.prefix} is too long to be used as a prefix for your roleName. Define parameter resourcePrefix?:`);
        this.lambda = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag(createdByTag, ID);
        const kmsPrivate = props.kmsPrivateKey || props.kms;
        const kmsPublic = props.kmsPublicKey || props.kms;
        const key = new cdk.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            serviceToken: this.lambda.functionArn,
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KmsPrivate: (kmsPrivate === null || kmsPrivate === void 0 ? void 0 : kmsPrivate.keyArn) || 'alias/aws/secretsmanager',
                KmsPublic: (kmsPublic === null || kmsPublic === void 0 ? void 0 : kmsPublic.keyArn) || 'alias/aws/secretsmanager',
                StorePublicKey: props.storePublicKey || false,
                ExposePublicKey: props.exposePublicKey || false,
                RemoveKeySecretsAfterDays: props.removeKeySecretsAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-ssh-key/',
                StackName: stack,
                Tags: cdk.Lazy.any({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPrivateKey !== 'undefined') {
            props.kmsPrivateKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPrivateKey);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPublicKey !== 'undefined') {
            props.kmsPublicKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPublicKey);
            key.node.addDependency(this.lambda.role);
        }
        this.privateKeyArn = key.getAttString('PrivateKeyARN');
        this.publicKeyArn = key.getAttString('PublicKeyARN');
        this.publicKeyValue = key.getAttString('PublicKeyValue');
        this.keyPairName = key.getAttString('KeyPairName');
        this.keyPairID = key.getAttString('KeyPairID');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2() // generally allow to inspect key pairs
                    .allow()
                    .toDescribeKeyPairs(),
                new statement.Ec2() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateKeyPair()
                    .toCreateTags()
                    .onKeyPair('*', undefined, undefined, stack.partition)
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Ec2() // allow delete/update, only if createdByTag is set
                    .allow()
                    .toDeleteKeyPair()
                    .toCreateTags()
                    .toDeleteTags()
                    .onKeyPair('*', undefined, undefined, stack.partition)
                    .ifResourceTag(createdByTag, ID),
                new statement.Secretsmanager() // generally allow to list secrets. we need this to check if a secret exists before attempting to delete it
                    .allow()
                    .toListSecrets(),
                new statement.Secretsmanager() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateSecret()
                    .toTagResource()
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Secretsmanager() // allow delete/update, only if createdByTag is set
                    .allow()
                    .allMatchingActions('/^(Describe|Delete|Put|Update)/')
                    .toGetSecretValue()
                    .toGetResourcePolicy()
                    .toRestoreSecret()
                    .toListSecretVersionIds()
                    .toUntagResource()
                    .ifResourceTag(createdByTag, ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPrivateKey(grantee) {
        return this.grantRead(this.privateKeyArn, grantee);
    }
    /**
     * Grants read access to the public key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPublicKey(grantee) {
        return this.grantRead(this.publicKeyArn, grantee);
    }
    grantRead(arn, grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
_a = JSII_RTTI_SYMBOL_1;
KeyPair[_a] = { fqn: "cdk-ec2-key-pair.KeyPair", version: "3.0.1" };
//# sourceMappingURL=data:application/json;base64,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