import io
import os
from typing import Dict, List

from setuptools import find_packages, setup

HERE = os.path.abspath(os.path.dirname(__file__))


def load_readme() -> str:
    with io.open(os.path.join(HERE, "README.rst"), "rt", encoding="utf8") as f:
        readme = f.read()
    # Replace img src for publication on pypi
    return readme.replace(
        "./docs/img/", "https://github.com/lektorium-tutor/tutor/raw/master/docs/img/"
    )


def load_about() -> Dict[str, str]:
    about: Dict[str, str] = {}
    with io.open(
            os.path.join(HERE, "lekt", "__about__.py"), "rt", encoding="utf-8"
    ) as f:
        exec(f.read(), about)  # pylint: disable=exec-used
    return about


def load_requirements(filename: str) -> List[str]:
    with io.open(
            os.path.join(HERE, "requirements", filename), "rt", encoding="utf-8"
    ) as f:
        return [line.strip() for line in f if is_requirement(line)]


def is_requirement(line: str) -> bool:
    return not (line.strip() == "" or line.startswith("#"))


ABOUT = load_about()

setup(
    name="lekt",
    version=ABOUT["__package_version__"],
    url="https://docs.tutor.overhang.io/",
    project_urls={
        "Documentation": "https://docs.tutor.overhang.io/",
        "Code": "https://github.com/lektorium-tutor/lekt",
        "Issue tracker": "https://github.com/lektorium-tutor/lekt/issues",
        "Community": "https://discuss.openedx.org/tag/tutor",
    },
    license="AGPLv3",
    author="Overhang.io",
    author_email="contact@overhang.io",
    description="The Docker-based Open edX distribution designed for peace of mind",
    long_description=load_readme(),
    long_description_content_type="text/x-rst",
    packages=find_packages(exclude=["tests*"]),
    include_package_data=True,
    python_requires=">=3.6",
    install_requires=load_requirements("base.in"),
    extras_require={
        "full": load_requirements("plugins.txt"),
    },
    entry_points={"console_scripts": ["lekt=lekt.commands.cli:main"]},
    classifiers=[
        "Development Status :: 5 - Production/Stable",
        "Intended Audience :: Developers",
        "License :: OSI Approved :: GNU Affero General Public License v3",
        "Operating System :: OS Independent",
        "Programming Language :: Python",
        "Programming Language :: Python :: 3.6",
        "Programming Language :: Python :: 3.7",
        "Programming Language :: Python :: 3.8",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
    ],
    test_suite="tests",
)
