# Copyright (c) 2024 XX Xiao

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files(the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

r""" 
This module contains the QuantumWrapper class, which provides an interface to convert 
quantum circuits produced by MindQuantum and Qiskit into OpenQASM 2.0 string representation.
"""

from typing import Type,Literal

class QuantumCircuitWrapper:
    r"""
    Export qiskit or mindquantum circuit to an OpenQASM 2 program in a string.
    """
    def __init__(self,circuit: Type, circuit_source: Literal['qiskit', 'mindquantum']):
        r"""
        Initialize the circuit wrapper.

        Args:
            circuit (class): Qiskit circuit or mindquantum circuit.
            circuit_source (str): Clarify the circuit source.
        """
        if circuit_source not in ['qiskit','mindquantum']:
            print('Currently, only quantum circuits generated by Qiskit and MindQuantum are supported！')
        self.read_circuit = circuit
        self.circuit_source = circuit_source
        self.to_openqasm2 = self.circuit_to_openqasm2()
        
    def circuit_to_openqasm2(self) -> str:
        r"""
        Export to an OpenQASM 2 program in a string.

        Returns:
            str: An OpenQASM 2 string representing the circuit.
        """
        if self.circuit_source == 'qiskit':
            return self.read_circuit_from_qiskit()
        elif self.circuit_source == 'mindquantum':
            return self.read_circuit_from_mindquantum()
        
    def read_circuit_from_qiskit(self) -> str:
        r"""
        Export qiskit circuit to an OpenQASM 2 program in a string.

        Returns:
            str: An OpenQASM 2 string representing the circuit.
        """
        from qiskit import qasm2
        cir_str = qasm2.dumps(self.read_circuit)
        return cir_str
    
    def read_circuit_from_mindquantum(self) -> str:
        r"""
        Export mindquantum circuit to an OpenQASM 2 program in a string.

        Returns:
            str: An OpenQASM 2 string representing the circuit.
        """
        from mindquantum.io import OpenQASM
        qasm2 = OpenQASM()
        cir_str = qasm2.to_string(self.read_circuit)
        return cir_str
        
    # def draw(self):
    #     from qiskit import QuantumCircuit
    #     cir = QuantumCircuit.from_qasm_str(self.to_openqasm2)
    #     return cir.draw()