"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days
 * specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 *
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or
 * has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to
 * a SNS topic. Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFuYWdlZC1ydWxlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm1hbmFnZWQtcnVsZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBd0M7QUFFeEMsd0NBQWlFO0FBQ2pFLGlDQUFnRDtBQWNoRDs7Ozs7OztHQU9HO0FBQ0gsTUFBYSxpQkFBa0IsU0FBUSxrQkFBVztJQUNoRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQWdDLEVBQUU7UUFDMUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixHQUFHLEtBQUs7WUFDUixVQUFVLEVBQUUscUJBQXFCO1lBQ2pDLGVBQWUsRUFBRTtnQkFDZixHQUFHLEtBQUssQ0FBQyxNQUFNO29CQUNiLENBQUMsQ0FBQzt3QkFDQSxlQUFlLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUU7cUJBQ3ZDO29CQUNELENBQUMsQ0FBQyxFQUFFO2FBQ1A7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUFkRCw4Q0FjQztBQXdCRDs7Ozs7OztHQU9HO0FBQ0gsTUFBYSxzQ0FBdUMsU0FBUSxrQkFBVztJQUdyRSxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXFELEVBQUU7UUFDL0YsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixHQUFHLEtBQUs7WUFDUixVQUFVLEVBQUUsNENBQTRDO1lBQ3hELGVBQWUsRUFBRTtnQkFDZixxQkFBcUIsRUFBRSxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7YUFDOUU7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZUFBZSxDQUFDLDRCQUE0QixFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUU1RyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDbkQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELGVBQWUsRUFBRTtnQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLGdCQUFnQixDQUFDO2FBQzdEO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBckJELHdGQXFCQztBQWNEOzs7Ozs7O0dBT0c7QUFDSCxNQUFhLG9DQUFxQyxTQUFRLGtCQUFXO0lBQ25FLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBbUQsRUFBRTtRQUM3RixJQUFJLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0NBQW9DLENBQUMsQ0FBQztTQUN2RDtRQUVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsR0FBRyxLQUFLO1lBQ1IsVUFBVSxFQUFFLHlDQUF5QztZQUNyRCxlQUFlLEVBQUUsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FDbEQsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLEdBQUcsTUFBTSxFQUFFLENBQUMsV0FBVyxHQUFHLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsRUFDL0UsRUFBRSxDQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQ3JELENBQUM7Q0FDRjtBQWpCRCxvRkFpQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBzbnMgZnJvbSAnQGF3cy1jZGsvYXdzLXNucyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBMYXp5LCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgTWFuYWdlZFJ1bGUsIFJ1bGVQcm9wcyB9IGZyb20gJy4vcnVsZSc7XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQWNjZXNzS2V5c1JvdGF0ZWRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBY2Nlc3NLZXlzUm90YXRlZFByb3BzIGV4dGVuZHMgUnVsZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBkYXlzIHdpdGhpbiB3aGljaCB0aGUgYWNjZXNzIGtleXMgbXVzdCBiZSByb3RhdGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDkwKVxuICAgKi9cbiAgcmVhZG9ubHkgbWF4QWdlPzogRHVyYXRpb247XG59XG5cbi8qKlxuICogQ2hlY2tzIHdoZXRoZXIgdGhlIGFjdGl2ZSBhY2Nlc3Mga2V5cyBhcmUgcm90YXRlZCB3aXRoaW4gdGhlIG51bWJlciBvZiBkYXlzXG4gKiBzcGVjaWZpZWQgaW4gYG1heEFnZWAuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29uZmlnL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hY2Nlc3Mta2V5cy1yb3RhdGVkLmh0bWxcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpDb25maWc6OkNvbmZpZ1J1bGVcbiAqL1xuZXhwb3J0IGNsYXNzIEFjY2Vzc0tleXNSb3RhdGVkIGV4dGVuZHMgTWFuYWdlZFJ1bGUge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQWNjZXNzS2V5c1JvdGF0ZWRQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIGlkZW50aWZpZXI6ICdBQ0NFU1NfS0VZU19ST1RBVEVEJyxcbiAgICAgIGlucHV0UGFyYW1ldGVyczoge1xuICAgICAgICAuLi5wcm9wcy5tYXhBZ2VcbiAgICAgICAgICA/IHtcbiAgICAgICAgICAgIG1heEFjY2Vzc0tleUFnZTogcHJvcHMubWF4QWdlLnRvRGF5cygpLFxuICAgICAgICAgIH1cbiAgICAgICAgICA6IHt9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciBhIENsb3VkRm9ybWF0aW9uU3RhY2tEcmlmdERldGVjdGlvbkNoZWNrXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2xvdWRGb3JtYXRpb25TdGFja0RyaWZ0RGV0ZWN0aW9uQ2hlY2tQcm9wcyBleHRlbmRzIFJ1bGVQcm9wcyB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGNoZWNrIG9ubHkgdGhlIHN0YWNrIHdoZXJlIHRoaXMgcnVsZSBpcyBkZXBsb3llZC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG93blN0YWNrT25seT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSB0byB1c2UgZm9yIHRoaXMgcnVsZS4gSXQgbXVzdCBoYXZlIHBlcm1pc3Npb25zIHRvIGRldGVjdCBkcmlmdFxuICAgKiBmb3IgQVdTIENsb3VkRm9ybWF0aW9uIHN0YWNrcy4gRW5zdXJlIHRvIGF0dGFjaCBgY29uZmlnLmFtYXpvbmF3cy5jb21gIHRydXN0ZWRcbiAgICogcGVybWlzc2lvbnMgYW5kIGBSZWFkT25seUFjY2Vzc2AgcG9saWN5IHBlcm1pc3Npb25zLiBGb3Igc3BlY2lmaWMgcG9saWN5IHBlcm1pc3Npb25zLFxuICAgKiByZWZlciB0byBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS91c2luZy1jZm4tc3RhY2stZHJpZnQuaHRtbC5cbiAgICpcbiAgICogQGRlZmF1bHQgYSByb2xlIHdpbGwgYmUgY3JlYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbn1cblxuLyoqXG4gKiBDaGVja3Mgd2hldGhlciB5b3VyIENsb3VkRm9ybWF0aW9uIHN0YWNrcycgYWN0dWFsIGNvbmZpZ3VyYXRpb24gZGlmZmVycywgb3JcbiAqIGhhcyBkcmlmdGVkLCBmcm9tIGl0cyBleHBlY3RlZCBjb25maWd1cmF0aW9uLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbmZpZy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY2xvdWRmb3JtYXRpb24tc3RhY2stZHJpZnQtZGV0ZWN0aW9uLWNoZWNrLmh0bWxcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpDb25maWc6OkNvbmZpZ1J1bGVcbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRm9ybWF0aW9uU3RhY2tEcmlmdERldGVjdGlvbkNoZWNrIGV4dGVuZHMgTWFuYWdlZFJ1bGUge1xuICBwcml2YXRlIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2xvdWRGb3JtYXRpb25TdGFja0RyaWZ0RGV0ZWN0aW9uQ2hlY2tQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIGlkZW50aWZpZXI6ICdDTE9VREZPUk1BVElPTl9TVEFDS19EUklGVF9ERVRFQ1RJT05fQ0hFQ0snLFxuICAgICAgaW5wdXRQYXJhbWV0ZXJzOiB7XG4gICAgICAgIGNsb3VkZm9ybWF0aW9uUm9sZUFybjogTGF6eS5zdHJpbmdWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucm9sZS5yb2xlQXJuIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMuc2NvcGVUb1Jlc291cmNlKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjaycsIHByb3BzLm93blN0YWNrT25seSA/IFN0YWNrLm9mKHRoaXMpLnN0YWNrSWQgOiB1bmRlZmluZWQpO1xuXG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnY29uZmlnLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ1JlYWRPbmx5QWNjZXNzJyksXG4gICAgICBdLFxuICAgIH0pO1xuICB9XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQ2xvdWRGb3JtYXRpb25TdGFja05vdGlmaWNhdGlvbkNoZWNrLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkRm9ybWF0aW9uU3RhY2tOb3RpZmljYXRpb25DaGVja1Byb3BzIGV4dGVuZHMgUnVsZVByb3BzIHtcbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBhbGxvd2VkIHRvcGljcy4gQXQgbW9zdCA1IHRvcGljcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyB0b3BpY3MuXG4gICAqL1xuICByZWFkb25seSB0b3BpY3M/OiBzbnMuSVRvcGljW107XG59XG5cbi8qKlxuICogQ2hlY2tzIHdoZXRoZXIgeW91ciBDbG91ZEZvcm1hdGlvbiBzdGFja3MgYXJlIHNlbmRpbmcgZXZlbnQgbm90aWZpY2F0aW9ucyB0b1xuICogYSBTTlMgdG9waWMuIE9wdGlvbmFsbHkgY2hlY2tzIHdoZXRoZXIgc3BlY2lmaWVkIFNOUyB0b3BpY3MgYXJlIHVzZWQuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29uZmlnL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jbG91ZGZvcm1hdGlvbi1zdGFjay1ub3RpZmljYXRpb24tY2hlY2suaHRtbFxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkNvbmZpZzo6Q29uZmlnUnVsZVxuICovXG5leHBvcnQgY2xhc3MgQ2xvdWRGb3JtYXRpb25TdGFja05vdGlmaWNhdGlvbkNoZWNrIGV4dGVuZHMgTWFuYWdlZFJ1bGUge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2xvdWRGb3JtYXRpb25TdGFja05vdGlmaWNhdGlvbkNoZWNrUHJvcHMgPSB7fSkge1xuICAgIGlmIChwcm9wcy50b3BpY3MgJiYgcHJvcHMudG9waWNzLmxlbmd0aCA+IDUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbW9zdCA1IHRvcGljcyBjYW4gYmUgc3BlY2lmaWVkLicpO1xuICAgIH1cblxuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgLi4ucHJvcHMsXG4gICAgICBpZGVudGlmaWVyOiAnQ0xPVURGT1JNQVRJT05fU1RBQ0tfTk9USUZJQ0FUSU9OX0NIRUNLJyxcbiAgICAgIGlucHV0UGFyYW1ldGVyczogcHJvcHMudG9waWNzICYmIHByb3BzLnRvcGljcy5yZWR1Y2UoXG4gICAgICAgIChwYXJhbXMsIHRvcGljLCBpZHgpID0+ICh7IC4uLnBhcmFtcywgW2BzbnNUb3BpYyR7aWR4ICsgMX1gXTogdG9waWMudG9waWNBcm4gfSksXG4gICAgICAgIHt9LFxuICAgICAgKSxcbiAgICB9KTtcblxuICAgIHRoaXMuc2NvcGVUb1Jlc291cmNlKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjaycpO1xuICB9XG59XG4iXX0=