#!/usr/bin/env python3
"""

A Bookmark JSON viewer using pure python

PyBookmarkJSONViewer used the PyJSONViewer by Atsushi Sakai (@Atsushi_twi) as a
reference. The result is distinctly different with many additional features and
modifications to handle a specific bookmark file type and search.

JSON input bookmark file generated by scripts/bookmarks_merge.py process

ref: https://docs.python.org/3/library/tkinter.html
ref: shared scroll list: https://stackoverflow.com/questions/32038701/python-tkinter-making-two-text-widgets-scrolling-synchronize
ref: error with windows do to missing reference: https://stackoverflow.com/questions/35166821/valueerror-attempted-relative-import-beyond-top-level-package
ref: image error due to missing master reference: https://stackoverflow.com/questions/20251161/tkinter-tclerror-image-pyimage3-doesnt-exist
ref: listbox: https://www.tutorialspoint.com/python/tk_listbox.htm
ref: text: https://www.tutorialspoint.com/python/tk_text.htm

@author: Crumbs
"""


import argparse
import datetime
import glob
import os
import re
import time
import tkinter as tk
import tkinter.ttk as ttk
import webbrowser
from tkinter import messagebox, filedialog, Tk, simpledialog
from urllib.parse import urlparse
import yaml

# run imports from top of bookmarks_merge.py
import sys
import pybookmark.bookmarks_parse as bp


def get_version(project_dir):
    try:
        version = open(project_dir + "/VERSION", "r").readline()
    except FileNotFoundError:
        version = "unknown"
    return version



# === Config ===
PADX = 3
PADY = 3
MAX_N_SHOW_ITEM = 300
MAX_HISTORY = 10
FILETYPES = [("JSON files", "*.json"), ("All Files", "*.*")]
HISTORY_FILE_PATH = os.path.join(os.path.expanduser('~'),
                                 ".pybookmarkjsonviewer_history")
PROJECT_DIR = os.path.dirname(os.path.abspath(__file__))
VERSION = get_version(PROJECT_DIR)


def addr_newest(file_use, file_path=None):
    """ given file_use use find the newest json file in the same path with
        the same starting name
        
        some_name.<anything>.json
        
        which means it returns the newest some_name.*.json file
        meant to match the latest timestamped book where anthing is the time
        
    Args:
        file_use (str): reference name for file
            if file_path is not specified it must be an absolute path
        file_path (str): file_path to search, if None pulls from file_use
            default=None
    Returns:
        (str): newest file that matches the file_use pattern
            if no matching files are found it returns None
    """
    file_use_base = os.path.basename(file_use).split('.')[0]
    if file_path is None:
        file_path = os.path.dirname(file_use)
    files = glob.glob(
        os.path.join(file_path, f'{file_use_base}*.json'),
        recursive=True)
    # return the newest file by ctime in the set of files
    if len(files) > 0:
        return max(files, key = os.path.getctime)
    else:
        return None


def field_to_list(field):
    """give a field, presumably str, convert to list dropping '' and None values
    
    Args:
        field (*): value to convert, expect string but can put anything
            into a list
    Returns:
        either input as list or if input = '' or None return []
        field if type(field) is list
        []  if input is '' or None
        [field] otherwise
    """
    if type(field) is list:
        fieldaslist = []
        for fieldx in field:
            if fieldx is None:
                continue
            if type(fieldx) is str and fieldx == '':
                continue
            fieldaslist.append(fieldx)
        return fieldaslist
    
    if field is None:
        fieldaslist = []
    elif type(field) is str:
        if len(field) == 0:
            fieldaslist = []
        else:
            fieldaslist = [field]
    else:
        fieldaslist = [field]
    return fieldaslist


def get_time_str(format_str='%Y%m%d.%H%M%S'):
    """ return current time as formatted string """
    return time.strftime(format_str)
    

# @staticmethod
def is_url(text):
    """ check input text is url or not

    Args:
        text (str): input text
    Return:
        (boolean) url or not
    """
    parsed = urlparse(text)
    return all([parsed.scheme, parsed.netloc, parsed.path])


class ScrollText(tk.Frame):
    """ define a scrolling text box
        | text box | scrollbar |
    Args:
        master = tk.Tk() or other Tk parent
        content_text (str): optional, if defined sets widget text during creation
    """
    def __init__(self, master, **kwargs):
        tk.Frame.__init__(self, master) # no need for super; ToDo width config

        # Creating the widgets
        
        self.text_box = tk.Text(self, **kwargs)
        ysb = ttk.Scrollbar(
            self, orient=tk.VERTICAL, command=self.text_box.yview)
        ysb.pack(side=tk.RIGHT)  # back before expanding text_box
        self.text_box.configure(yscrollcommand=ysb.set)
        self.text_box.pack(side=tk.LEFT, fill=tk.X, expand=True)  # QQQ: does this need to match ScrolledText method?
        
    def add_text(self, text):
        """ method to append newline to the ScrollText text """
        self.text_box.insert(tk.END, '\n' + text)
        
    def set_text(self, text):
        """ method to set the ScrollText text by overwriting existing value """
        # self.text_box.destroy()  # QQQ should this be delete(0, tk.END) ?
        self.text_box.delete(0, tk.END)  # do not use destroy it drops the element from packing
        self.text_box.insert(tk.END, text)
    
    def get_text(self):
        """ method to get ScrollText text value """
        return self.text_box.get(0, tk.END) # gets from 0 index to end


class ScrollEntryLabel(tk.Frame):
    """ define a labelled scrolling text entry box
            | text label | text entry box | scrollbar |
    Args:
        master = tk.Tk() or other Tk parent
        label_text (str) = text used to label the object
    QQQ mod with https://stackoverflow.com/questions/12160331/python-tkinter-scroll-bar-with-user-generated-entry-fields
    """
    def __init__(self, master, label_text, **kwargs):
        tk.Frame.__init__(self, master, width=50) # no need for super; ToDo width config

        # Creating the widgets
        
        self.sel_label = tk.Label(
            self,
            text=label_text)
        self.sel_label.pack(side=tk.LEFT)
        
        # need text entry box
        self.entryText = tk.StringVar()
        self.sel_text = tk.Entry(self, textvariable=self.entryText)
        self.sel_text.pack(side=tk.LEFT, fill=tk.X, expand=True)

    def get_text(self):
        return self.entryText.get()
        
    def get_text_as_type(self):
        """ return the entryText value as type (well string or list) """            
        text = self.get_text()
        # print(f'get_text_as_type:::{text}:::')  # DDD hidden debug line
        if len(text) == 0:
            return text
        if text[0] == '(' and text[-1] == ')':
            # by default tk prints lists as {} for reasons unknown
            #        this happens when self.entryText.set(list_to_print)')
            #    when printed in program show as ('element1','element2')
            #   example text:  text = "('x','y',)"
            # type is list need to convert
            # remove list identifier () and trailing , and then 
            #   the lead and trailing single quote for first and last element
            # then replace space between elements so split works, space from editing
            text = text.strip('(').strip(')').strip(',').strip("'").replace("', '","','")
            text_list = text.split("','")  # break on quoted comma between elements
            return text_list
        elif text[0] == '[' and text[-1] == ']':
            # tk prints lists as [] if use formated string to set value
            #        this happens when self.entryText.set(f'{list_to_print}')
            #    when printed in program show as ['element1','element2']
            #    does not seem to have trailing , like above () wrapped version
            #   example text:  text = "['x','y']"
            # type is list need to convert
            # remove list identifier [] any trailing , and then 
            #   the lead and trailing single quote for first and last element
            # then replace space between elements so split works, space from editing
            text = text.strip('[').strip(']').strip(',').strip("'").replace("', '","','")
            text_list = text.split("','")  # break on quoted comma between elements
            return text_list
        else:
            return text
            

class ScrolledTextPair(tk.Frame):
    """Two ListBox widgets and a Scrollbar in a Frame with labels above them
    
    | URL set | Label |
    | Listbox | Listbox | linked scrollbar
    
    
    ref: https://stackoverflow.com/questions/32038701/python-tkinter-making-two-text-widgets-scrolling-synchronize
    
    modified to use tk import and Listbox vs text
    """

    def __init__(self, master, **kwargs):
        tk.Frame.__init__(self, master) # no need for super

        # Different default width
        if 'width' not in kwargs:
            kwargs['width'] = 30        # TODO make width pass in config

        # Creating the widgets for the two listboxes and scrollbar
        self.left_frame = tk.Frame(self)
        self.left_label = tk.Label(
            self.left_frame,
            text='URL set')
        self.left_label.pack(side=tk.TOP)
        self.left = tk.Listbox(self.left_frame, **kwargs)
        # self.update_left([])  # caused failure unclear why, because of destroy?
        self.left.pack(side=tk.BOTTOM, fill=tk.BOTH, expand=True)
                
        self.right_frame = tk.Frame(self)
        self.right_label = tk.Label(
            self.right_frame,
            text='Label')
        self.right_label.pack(side=tk.TOP)        
        self.right = tk.Listbox(self.right_frame, **kwargs)
        self.right.pack(side=tk.BOTTOM, fill=tk.BOTH, expand=True)
        
        # pack both frames and scrollbar
        #   because the scrollbar is not in the same frame as the list boxes
        #   add a vertical spacer above scrollbar to offset it down
        #   so expanded scrollbar size matches the adjacent listbox size
        # may be easier to use grid packing
        self.left_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.right_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        text_spacer = tk.Label(self, text=' ', pady=2) # hidden, moves scrollbar down
        text_spacer.pack(side=tk.TOP)
        self.scrollbar = tk.Scrollbar(self)
        self.scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        # Changing the settings to make the scrolling work
        self.scrollbar['command'] = self.on_scrollbar
        self.left['yscrollcommand'] = self.on_textscroll
        self.right['yscrollcommand'] = self.on_textscroll
        
        # handle arrow-key interaction with lists to update selection
        self.left.bind('<Down>', self.OnEntryUpDown)
        self.left.bind('<Up>', self.OnEntryUpDown)
        self.right.bind('<Down>', self.OnEntryUpDown)
        self.right.bind('<Up>', self.OnEntryUpDown)

    def on_scrollbar(self, *args):
        '''Scrolls both text widgets when the scrollbar is moved'''
        self.left.yview(*args)
        self.right.yview(*args)

    def on_textscroll(self, *args):
        '''Moves the scrollbar and scrolls text widgets when the mousewheel
        is moved on a text widget'''
        self.scrollbar.set(*args)
        self.on_scrollbar('moveto', args[0])
        
    def add(self, left_list, right_list):
        """ add content to the left and right listbox values w/o removing
        existing content
        
        Args:
            left_list (list): list of items to put in left listbox
            right_list (list): list of items to put in left listbox
        Returns:
            no object returned, function updates listbox widget objects
        """
        if len(left_list) != len(right_list):
            print('Warning: ScrolledTextPair: unequal left & right lists')
        
        self.update_left(left_list, clear=False)
        self.update_right(right_list, clear=False)
        
    def update(self, left_list, right_list, sort_side = 0):
        """ update the left and right listbox values
        note may be cheaper to do this function outside to avoid double-loop
        
        Args:
            left_list (list): list of items to put in left listbox
            right_list (list): list of items to put in left listbox
            sort_side (int): integer flag to control whether to and which list
                to sort by. sort is by one list but both are reordered to 
                maintain the link between them
                    0 = no sort, technically any value not 1 or 2 is no sort
                    1 = sort by left
                    2 = sort by right
                default = 0.
        Returns:
            no object returned, function updates listbox widget objects
        """
        if len(left_list) != len(right_list):
            print('Warning: ScrolledTextPair: unequal left & right lists')
            
        if sort_side == 1:
            # first argument to zip is what sort by so sort by left_list
            # second argument is just reordered to match the sorted list
            # sort ref: https://stackoverflow.com/questions/7851077/how-to-return-index-of-a-sorted-list
            #   see Trenton McKinney and Shawn Chin response
            combined = zip(left_list, right_list)
            zipped_sorted = sorted(combined, key=lambda x: x[0])
            left_list, right_list = map(list, zip(*zipped_sorted))
        elif sort_side == 2:
            # first argument to zip is what sort by so sort by right_list
            # second argument is just reordered to match the sorted list
            combined = zip(right_list, left_list)
            zipped_sorted = sorted(combined, key=lambda x: x[0])
            right_list, left_list = map(list, zip(*zipped_sorted))
            
        self.update_left(left_list)
        self.update_right(right_list)
        
    def update_left(self, left_list, clear=True):
        # clear the left list before updating
        if clear:
            self.left.delete(0, tk.END)   # using destroy() removes packing?
        for left_text in left_list:
            if type(left_text) is str:
                left_text = left_text.replace("\n", "")
            self.left.insert(tk.END, left_text)

    def update_right(self, right_list, clear=True):
        # clear the right list before updating
        # self.right.destroy()  # - or - # self.right.delete(0, tk.END)
        if clear:
            self.right.delete(0, tk.END)
        for right_text in right_list:
            if type(right_text) is str:
                right_text = right_text.replace("\n", "")
            self.right.insert(tk.END, right_text)
    
    def OnEntryUpDown(self, event):
        """ move the cursor selection by keyboard arrow keys 
        ref: https://stackoverflow.com/questions/29484287/tkinter-listbox-that-scrolls-with-arrow-keys
        """
        selection = event.widget.curselection()[0]
        
        if event.keysym == 'Up':
            selection += -1
    
        if event.keysym == 'Down':
            selection += 1
    
        if 0 <= selection < event.widget.size():
            event.widget.selection_clear(0, tk.END)
            event.widget.select_set(selection)


class SearchLogic(tk.Frame):
    """ define a double check box, text entry box and label of form:
        | and box | or box | text entry box | text label |
    Args:
        master = tk.Tk() or other Tk parent
        label_text (str) = text used to label the object
        on_value (int) = check box on value
        off_value (int) = check box off value
    """
    def __init__(self, master, label_text, on_value, off_value, **kwargs):
        tk.Frame.__init__(self, master, width=50) # no need for super; ToDo width config

        # Creating the widgets
        
        # need variable for checkboxes to write to, necessary to get value
        # initialize to off_value to avoid checking for off_value and
        # not matching prior to cycling the checkbox
        self.var_chk_and = tk.IntVar(value=off_value)
        self.var_chk_or = tk.IntVar(value=off_value)
        self.var_chk_not = tk.IntVar(value=off_value)
        
        # need checkboxes for 'and' 'or' and not integers
        self.search_chk_and = tk.Checkbutton(
            self,
            text='',
            onvalue=on_value,
            offvalue=off_value,
            variable=self.var_chk_and
            )
        self.search_chk_and.pack(side=tk.LEFT, padx=2)

        self.search_chk_or = tk.Checkbutton(
            self,
            text='',
            onvalue=on_value,
            offvalue=off_value,
            variable=self.var_chk_or
            )
        self.search_chk_or.pack(side=tk.LEFT, padx=2)
        
        self.search_chk_not = tk.Checkbutton(
            self,
            text='',
            onvalue=on_value,
            offvalue=off_value,
            variable=self.var_chk_not
            )
        self.search_chk_not.pack(side=tk.LEFT, padx=2)
        
        # need a row label; shown after entry box but packed before due to expand
        self.search_chk_label = tk.Label(
            self,
            text=label_text)
        self.search_chk_label.pack(side=tk.RIGHT, padx=2)
        
        # need text entry box
        self.entryText = tk.StringVar()
        self.search_chk_box = tk.Entry(self, textvariable=self.entryText)
        self.search_chk_box.pack(side=tk.LEFT, fill=tk.X, expand=True)
        # self.search_chk_box.bind('<Enter>', self.find_bookmark) # bind in object not class
        
    def check_box(self, box_name):
        """ given a box_name of AND, OR or NOT check the corresponding box """
        box_name = box_name.lower()
        if box_name == 'and':
            self.search_chk_and.select()
        if box_name == 'or':
            self.search_chk_or.select()
        if box_name == 'not':
            self.search_chk_not.select()     
    
    def set_text(self, text):
        """ set the text of the text entry box """
        self.entryText.set(text)

        
def test_ScrolledTextPair():
    # code for how to run; example need to incorporate below
    root = tk.Tk()
    x = SearchLogic(root, label_text='test cow', on_value=6, off_value=-4)
    x.pack(fill=tk.X, expand=True)
    t = ScrolledTextPair(root, bg='white', fg='black')
    t.pack(fill=tk.BOTH, expand=True)
    for i in range(50):
        t.left.insert(tk.END,"foo %s\n" % i)
        t.right.insert(tk.END,"bar %s\n" % i)

    t.update(list(range(5)), list(range(6)))
    x.search_chk_box.insert(0, 'lazy brown cow')

    root.title("Text scrolling example")
    root.mainloop()

    
def test_BookmarkGUI():
    addrStruct = bp.readAddressStruct(os.path.join(PROJECT_DIR, 'data', 'addr.json'))
    root= tk.Tk()
    root.minsize(500, 500)  # width x height
    app = BookmarkGUI(root, bookmarks_data=addrStruct)
    root.title('Testing Bookmark GUI')
    root.mainloop()
        
    
class BookmarkGUI(tk.Frame):
    ''' primary bookmark class application to build GUI
        this is the new GUI version
        
        note it creates two files in the output_dir
            a timestamped json file see save_structure() upon exit
                f'{self.output_base}.{get_time_str()}.json'
            a changes file see write_log() if self.update_action is True
                f'{self.output_base}.changes.log'
    '''
    
    def __init__(self, master, bookmarks_data=None, output_dir=None, output_base='addr', config=None, **kwargs):
        tk.Frame.__init__(self, master, width=50) # no need for super
        
        self.output_base = output_base  # builds into addr.YYYYMMDD.MMSS.json
        self.output_dir = output_dir
        self.update_action = False  # boolean tracks if change occurred, if yes save updates on exit
        
        self.addrStruct = {}  # address bookmarks data dictionary
        # searchAddressStruct: copied from bookmarks_parse definition
        #    [-1] = search URL addresses themselves
        #     [0] = label
        #     [1] = age
        #     [2] = tags
        #     [3] = location
        #     [4] = description
        #     [5] = file location
        self.field_list = ['URL',
                      'label',
                      'age',
                      'tags',
                      'location',
                      'description',
                      'file location',
                      ]
        self.addrStruct_url_list = []  # used with search so removing from list is efficient
        
        self.root = master
        
        self.create_widgets(config=config)
        
        if bookmarks_data is not None:
            if type(bookmarks_data) is str:
                # assumes file path listing to read in the bookmark data
                self.addrStruct = bp.readAddressStruct(bookmarks_data)
            elif type(bookmarks_data) is dict:
                # assumes bookmarks_data is the addrStruct data
                self.addrStruct = bookmarks_data
        
        self.view_url_update()  # populate the lists for viewing after defined
    
    def create_widgets(self, config):
        """ function used to define the widgets layout
        Args:
            config (dict): dictionary of configuration values; presumably read
                in from a yaml configuration file
                expects certain fields, example search, others are ignored
        """

        # - top frame: top pack
        #   ScrolledTextPair
        self.bookmark_lists = ScrolledTextPair(self.root, bg='white', fg='black', selectmode=tk.SINGLE)
        # don't bind to Button-1 press use <<ListboxSelect>> instead
        #   using Button-1 was causing weird behavior
        #       not responding to single click or move of cursor in lists
        #       must click then click to get selected; annoying
        self.bookmark_lists.left.bind('<<ListboxSelect>>', self.click_url)  # YYY do not put () on the function here or it will fail to bind
        self.bookmark_lists.right.bind('<<ListboxSelect>>', self.click_label)
        self.bookmark_lists.left.bind('<Return>', self.click_url)
        self.bookmark_lists.right.bind('<Return>', self.click_label)
        self.bookmark_lists.left.bind('<Double-1>', self.click_double_url)
        self.bookmark_lists.right.bind('<Double-1>', self.click_double_label)
        self.bookmark_lists.left.bind('<Button-3>', self.click_right_url)
        self.bookmark_lists.right.bind('<Button-3>', self.click_right_label)
        
        # - bottom frame        
        input_frame = ttk.Frame(self.root)
        
        # - pack the frames inside the window
        # organize top frame and bottom pack but pack bottom first to allow
        # the top to expand
        input_frame.pack(side=tk.BOTTOM, fill=tk.X)
        self.bookmark_lists.pack(side=tk.TOP, fill=tk.BOTH, expand=True, padx=PADX, pady=PADY)
                
        # - frame 2 left pack
        #   tabbed search or new
        #   search tab:
        #       search button + search entry box
        #       rows of SearchLogic chkbox sets
        #   new tab: 
        #       update button
        #       entry fields
        # create the tab control
        tabControl = ttk.Notebook(input_frame)
        tabControl.pack(side=tk.LEFT, 
                        expand=True, fill=tk.X,
                        padx=PADX, pady=PADY, ipadx=1, ipady=1)
        # create the tabs        
        tab1 = ttk.Frame(tabControl)
        tab2 = ttk.Frame(tabControl)
        tab3 = ttk.Frame(tabControl)
        tab4 = ttk.Frame(tabControl)
        tabControl.add(tab1, text='Search')
        tabControl.add(tab2, text='Add New Bookmark')
        tabControl.add(tab3, text='Directions')
        tabControl.add(tab4, text='Console Log')
        
        # the search tab content
        search_top_frame = tk.Frame(tab1)
        search_top_frame.pack(side=tk.TOP, expand=True)
        search_better_label = tk.Label(
            search_top_frame,
            text="Bookmark Search:")
        search_better_label.pack(side=tk.LEFT, padx=10)
        self.search_clear = tk.Button(search_top_frame, 
                                      text='Reset Search', 
                                      command=self.view_search_reset)
        self.search_clear.pack(side=tk.RIGHT, ipadx=PADX, padx=10)
        self.var_chk_ignorecase = tk.IntVar(value=0)
        self.search_chk_ignorecase = tk.Checkbutton(
            search_top_frame,
            text=' Ignore Case',
            onvalue=1,
            offvalue=0,
            variable=self.var_chk_ignorecase
            )
        self.search_chk_ignorecase.pack(side=tk.RIGHT, ipadx=PADX, padx=10)
        
        self.var_chk_phrase = tk.IntVar(value=0)
        self.search_chk_phrase = tk.Checkbutton(
            search_top_frame,
            text=' Phrase Search',
            onvalue=1,
            offvalue=0,
            variable=self.var_chk_phrase
            )
        self.search_chk_phrase.pack(side=tk.RIGHT, ipadx=PADX, padx=10)
        
        self.search_box_text = tk.StringVar()
        self.search_box_better = tk.Entry(tab1, textvariable=self.search_box_text)
        self.search_box_better.config({"background": "yellow"})
        self.search_box_better.bind('<Return>', self.click_search)
        self.search_box_better.pack(side=tk.TOP, fill='x', padx=PADX)

        # insert search tab content and|or|not buttons in same order as addrJson elements
        # need checkboxes for searching specific elements easily per bp.searchAddressStruct() definition
        # first generate header-column labels for the SearchLogic controls
        search_chk_list_label_frame = tk.Frame(tab1)  # use to pack text
        search_chk_list_label_frame.pack(side=tk.TOP, fill=tk.X, expand=True)
        search_chk_list_label_and = tk.Label(
            search_chk_list_label_frame,
            text='And')
        search_chk_list_label_and.grid(row=0, column=0, padx=2, sticky=tk.NW)
        search_chk_list_label_or = tk.Label(
            search_chk_list_label_frame,
            text='Or')
        search_chk_list_label_or.grid(row=0, column=2, padx=2, sticky=tk.NW)
        search_chk_list_label_not = tk.Label(
            search_chk_list_label_frame,
            text='Not')
        search_chk_list_label_not.grid(row=0, column=3, padx=2, sticky=tk.NW)
        search_chk_list_label_desc = tk.Label(
            search_chk_list_label_frame,
            text='Checkboxes modify search; URL only by default')
        search_chk_list_label_desc.grid(row=0, column=4, padx=2, sticky=tk.NW)
        self.search_chk_list = []
        for labeli, label in enumerate(self.field_list):
            x = SearchLogic(tab1,
                            label_text=label,
                            on_value=(labeli - 1),
                            off_value=-(labeli + 99))
            x.search_chk_box.bind('<Return>', self.click_search)
            x.pack(side=tk.TOP, expand=True, fill=tk.X)
            self.search_chk_list.append(x)
                
        # - set search checkbox defaults
        if config is not None and 'search' in config:
            # set which search logic boxes are set via external configuration
            for logic_name in config['search']:
                box_row = -1
                logic_name_lc = logic_name.lower()
                if logic_name_lc == 'url':
                    box_row = 0
                if logic_name_lc == 'label':
                    box_row = 1
                if logic_name_lc == 'tags':
                    box_row = 3
                if logic_name_lc == 'description':
                    box_row = 5
                if logic_name_lc == 'file':
                    box_row = 6
                if logic_name_lc == 'ignorecase':
                    if config['search'][logic_name] is True:
                        self.search_chk_ignorecase.select()
                    continue
                if logic_name_lc == 'phrase':
                    if config['search'][logic_name] is True:
                        self.search_chk_phrase.select()
                    continue
                if box_row != -1:
                    # found logic                             
                    for boxname in config['search'][logic_name]:
                        # print(f'{logic_name}::{boxname}') # DDD
                        if boxname in ['AND', 'OR', 'NOT']:
                            self.search_chk_list[box_row].check_box(boxname)
                        else:
                            self.search_chk_list[box_row].set_text(boxname)
        else:
            # default logic boxes
            self.search_chk_list[0].search_chk_or.select()  # URL
            self.search_chk_list[1].search_chk_or.select()  # label
            self.search_chk_list[3].search_chk_or.select()  # tags
            self.search_chk_list[5].search_chk_or.select()  # description
        
        # add the new tab content
        self.new_url = ScrollEntryLabel(tab2, label_text='URL')
        self.new_url.pack(side=tk.TOP, expand=True, fill=tk.X, padx=PADX)
        self.new_label = ScrollEntryLabel(tab2, label_text='Label')
        self.new_label.pack(side=tk.TOP, expand=True, fill=tk.X, padx=PADX)
        self.new_tags = ScrollEntryLabel(tab2, label_text='Tags')
        self.new_tags.pack(side=tk.TOP, expand=True, fill=tk.X, padx=PADX)
        self.new_location = ScrollEntryLabel(tab2, label_text='Location')
        self.new_location.pack(side=tk.TOP, expand=True, fill=tk.X, padx=PADX)
        self.new_description = ScrollEntryLabel(tab2, label_text='Description')
        self.new_description.pack(side=tk.TOP, expand=True, fill=tk.X, padx=PADX)
        self.new_button = tk.Button(tab2,
                                    text='Add Bookmark',
                                    command=self.add_bookmark)
        self.new_button.pack(side=tk.TOP)
        
        # add the direction tab content
        direction_text = 'Scrollbar of list boxes above is linked\n\n' \
            + '## Search ##\n' \
            + 'Searching in search tab will restrict list boxes\n' \
            + '  search is done OR then AND then NOT\n' \
            + '  or = inclusive search ie it sums find lists\n' \
            + '  and = exclusive search ie it reduces, must match all conditions\n' \
            + '  not = excludes any URL that matches NOT condition\n' \
            + '  note: age based search is POSIX time integer must use ># or <# syntax\n' \
            + '  note: must click <ENTER> in search box to trigger search\n' \
            + 'Clicking Reset Search returns to the original list\n' \
            + 'Check Ignore Case to search without matching case\n\n' \
            + '## New or Edit ##\n' \
            + 'Add new bookmarks in the Add tab\n' \
            + '\tAdding checks existing URLs and shows match in edit pane\n' \
            + 'To update the found bookmark edit, then click Update\n\n' \
            + '## Mouse Actions ##\n' \
            + 'First left click into listbox to activate mouse actions then:\n' \
            + '\tdouble-click on url or label: opens url in browser\n' \
            + '\tsingle-click on url or label: opens in edit tab\n' \
            + '\tright-click on url or label: copies the selected item to clipboard'
        directions = tk.Label(tab3, text=direction_text, justify=tk.LEFT)
        directions.pack(side=tk.TOP)
        
        # add the console log
        self.console_log = ScrollText(tab4)
        self.console_log.pack(side=tk.TOP, expand=True, fill=tk.BOTH, padx=PADX, pady=PADY)
        
        # - frame 3 right pack
        #   edit check box
        #   new check box
        #   rows of ScrollText for each Search logic box # (QQQ just define in row with SearchLogic chkboxes to hold together?)
        edit_frame = tk.Frame(input_frame)
        edit_frame.pack(side=tk.RIGHT, expand=True, fill=tk.X, padx=PADX, pady=PADY)
        edit_label = tk.Label(edit_frame, text='Currently Selected\n(or Found)\nBookmark')
        edit_label.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_url = ScrollEntryLabel(edit_frame, label_text='URL')
        self.edit_url.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_label = ScrollEntryLabel(edit_frame, label_text='Label')
        self.edit_label.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_age = ScrollEntryLabel(edit_frame, label_text='Age: (edit ignored)')
        self.edit_age.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_tags = ScrollEntryLabel(edit_frame, label_text='Tags')
        self.edit_tags.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_location = ScrollEntryLabel(edit_frame, label_text='Location')
        self.edit_location.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_description = ScrollEntryLabel(edit_frame, label_text='Description')
        self.edit_description.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_button = tk.Button(edit_frame,
                                     text='Update Bookmark',
                                     command=self.edit_bookmark
                                     )
        self.edit_button.pack(side=tk.TOP, expand=True, fill=tk.X)
        self.edit_rmv_button = tk.Button(edit_frame,
                                     text='Remove Bookmark',
                                     command=self.remove_bookmark,
                                     bg='red'
                                     )
        self.edit_rmv_button.pack(side=tk.TOP, expand=True, fill=tk.X)
        
    def click_label(self, event=None):
        """ get the clicked label and update the selected to edit boxes
        
        Args:
            event: event argument that is passed by the click event
        
        """
        labeli = self.bookmark_lists.right.curselection()
        if len(labeli) > 0:
            labeli = int(labeli[0])
            label = self.bookmark_lists.right.get(labeli)
        else:
            return
        
        # reverse lookup by index the url in the other list
        url = self.bookmark_lists.left.get(labeli)
        age = self.addrStruct[url][1]
        tags = self.addrStruct[url][2]
        location = self.addrStruct[url][3]
        description = self.addrStruct[url][4]
        
        edit_update_list = [url, label, age, tags, location, description]
        self.view_edit_update(edit_update_list)
        
    def click_double_label(self, event=None):
        """
        Callback function when label is double clicked. open in browser

        :param event: event arg (not used)
        """
        labeli = self.bookmark_lists.right.curselection()
        if len(labeli) > 0:
            labeli = int(labeli[0])
            # reverse lookup by index the url in the other list
            url = self.bookmark_lists.left.get(labeli)
            self.open_url(url)
                
    def click_right_label(self, event=None):
        """
        Callback function when label is right clicked. copy the text

        :param event: event arg (not used)
        """
        labeli = self.bookmark_lists.right.curselection()
        if len(labeli) > 0:
            labeli = int(labeli[0])
            # reverse lookup by index the url in the other list
            label = self.bookmark_lists.right.get(labeli)
            self.clipboard_clear()
            self.clipboard_append(label)
    
    def click_url(self, event=None):
        """ get the clicked url and update the selected to edit boxes via call
            to view_edit_update_by_url()
                
        Args:
            event: event argument that is passed by the click event
        
        """
        urli = self.bookmark_lists.left.curselection()
        if len(urli) > 0:
            urli = int(urli[0])
            url = self.bookmark_lists.left.get(urli)
            self.view_edit_update_by_url(url)
        
    def click_double_url(self, event=None):
        """
        Callback function when url is double clicked. open in browser

        :param event: event arg (not used)
        """
        urli = self.bookmark_lists.left.curselection()
        if len(urli) > 0:
            urli = int(urli[0])
            url = self.bookmark_lists.left.get(urli)
            self.open_url(url)

    def click_right_url(self, event=None):
        """
        Callback function when url is right clicked. copy url as text

        :param event: event arg (not used)
        """
        urli = self.bookmark_lists.left.curselection()
        if len(urli) > 0:
            urli = int(urli[0])
            url = self.bookmark_lists.left.get(urli)
            self.clipboard_clear()
            self.clipboard_append(url)
        
    def click_search(self, event=None):
        """ using the defined search parameters reduce url/label to matches 
        
            note default bp.searchAddressStructWrapper() is or behavior
            want to search inclusive (or) then search exclusive (and) using
            specific bp.searchAddressStruct() where both return a list of urls
            but must account for shared vs specific search pattern
        Args:
            event: event argument that is passed by the click event
        """
        pattern_shared = self.search_box_text.get().strip() # shared search pattern
        urls_found = []
        # print(f'Searched for: {pattern_shared}') # DDD
                
        # need to apply each and|or term checkbox value
        # first apply OR using pattern_shared when a specific pattern is not 
        #   defined, then apply AND
        
        # QQQ what happens if pattern_shared is blank?
        
        # - handle inclusive search by or
        element_or = []
        pattern_or = []
        search_or = False   # set True if any or search requested
        specific_or_bool = False  # set to True if any specific or pattern set
        
        for labeli, label in enumerate(self.field_list):
            elementi = labeli - 1
            if self.search_chk_list[labeli].var_chk_not.get() == elementi:
                # do not add to OR search if NOT also set
                continue
            # print(f'click_search: OR: {self.search_chk_list[labeli].var_chk_or.get()} at {elementi}')  # DDD
            if self.search_chk_list[labeli].var_chk_or.get() == elementi:
                # indicates to include this element in OR search
                # decide if shared or specific pattern
                pattern_specific = self.search_chk_list[labeli].entryText.get().strip()
                # print(f'search_or input:{elementi}:{pattern_specific}:') # DDD
                if len(pattern_specific) > 0:
                    element_or.append(elementi)
                    pattern_or.append(pattern_specific)
                    search_or = True
                    specific_or_bool = True
                else:
                    if len(pattern_shared) > 0:
                        # print(f'\tadd pattern_shared: {pattern_shared}') # DDD
                        element_or.append(elementi)
                        pattern_or.append(pattern_shared)
                        search_or = True
        if search_or:    # could test len(element_or) > 0 but using search_or
            # any OR search set; therefore search using OR
            if not specific_or_bool:
                # no specific search, reduce the pattern from list to string
                if len(pattern_shared) > 0:
                    pattern_or = pattern_shared
                else:
                    print('click_search is it possible to get here?')
                    search_or = False   # no specific OR shared pattern defined
            if search_or:
                # print(f'SEarch OR1 {type(pattern_or)} string {pattern_or}')
                # YYY oh this breaks because the code can't handle unequal pattern and element lists!
                #   if want to do phrase addition need to handle as separate searches that build up like and and not
                # pattern_or = self.phrase_adder(pattern_or)
                # print(f'SEarch OR2 {type(pattern_or)} string {pattern_or}')
                urls_found = bp.searchAddressStructWrapper(
                    self.addrStruct, 
                    pattern_or, 
                    element_or,
                    ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
                # print(f'click_search: OR: {pattern_or}::::{element_or} found {len(urls_found)}') # DDD
                
                if self.var_chk_phrase.get() == 0:
                    # means to do a phrase based search
                    # must do it based on each element so requires 2 loops
                    # one to break into sub-patterns, then one to loop on them
                    # searching the specific element 
                    if type(pattern_or) is str:
                        # print('String to LIST') # DDD
                        pattern_or = [pattern_or]
                    if type(pattern_or) is list and len(pattern_or) != len(element_or):
                        # should only be 1 force it to 1
                        # print('LIST multiplied') # DDD
                        pattern_or = [pattern_or[0]] * len(element_or)

                    # print(f'Print pattern_or: {pattern_or}: {type(pattern_or)}::{element_or}::') # DDD
                    for patterni, patternis in enumerate(pattern_or):
                        if re.search(' ', patternis) is not None:
                            # print(f'search_or call adder: {patternis} at {patterni}') # DDD
                            patternis_list = self.phrase_adder(patternis)
                            if type(patternis_list) is not list:
                                continue
                            # print(f'search_or past list check') # DDD
                            for patternisi in patternis_list:
                                # print(f'search or3: {patternisi}::{patterni}') # DDD
                                urls_found = urls_found + bp.searchAddressStruct(
                                    self.addrStruct,
                                    patternisi,
                                    element_or[patterni],
                                    ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
                    urls_found = list(set(urls_found))
            
        # - handle exclusive search by and, does not use arrays because exclusive
        if not search_or:
            # test again because search_or can be reset inside search_or test
            urls_found = list(self.addrStruct.keys())
            # print(f'click_search: AND: starts with {len(urls_found)}') # DDD
        search_and_any = False
        for labeli, label in enumerate(self.field_list):
            elementi = labeli - 1
            if self.search_chk_list[labeli].var_chk_not.get() == elementi:
                # do not add to AND search if NOT also set
                continue
            # print(f'click_search: AND: {self.search_chk_list[labeli].var_chk_and.get()} at {elementi}') # DDD
            if self.search_chk_list[labeli].var_chk_and.get() == elementi:
                # indicates to include this element in AND search
                # decide if shared or specific pattern and pass a reduced
                #   addrStruct set based on the current list of urls_found
                # use dictionary comprehension to reduce passed addrStruct
                #   ref: https://stackoverflow.com/questions/3420122/filter-dict-to-contain-only-certain-keys
                search_and = False   # set True if also have a search pattern
                pattern_specific = self.search_chk_list[labeli].entryText.get().strip()
                if len(pattern_specific) > 0:
                    pattern_and = pattern_specific
                    search_and = True
                else:
                    if len(pattern_shared) > 0:
                        pattern_and = pattern_shared
                        search_and = True
                if search_and:
                    # search AND set and pattern defined therefore search
                    pattern_andL = self.phrase_adder(pattern_and)
                    if type(pattern_andL) is list and len(pattern_andL) > 1:
                        for patterni in pattern_andL:
                            urls_found = bp.searchAddressStruct(
                                {urlkey: self.addrStruct[urlkey] for urlkey in urls_found},
                                patterni,
                                elementi,
                                ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
                    else:
                        urls_found = bp.searchAddressStruct(
                            {urlkey: self.addrStruct[urlkey] for urlkey in urls_found},
                            pattern_and,
                            elementi,
                            ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
                    search_and_any = True
                    # print(f'click_search: AND: {pattern_and}::::{elementi} found {len(urls_found)}') # DDD
            if len(urls_found) == 0:
                # stop searching if no urls are left to search
                # indicates search pattern is too exclusive
                break
        if not search_or and not search_and_any:
            # no search has occurred yet, if pattern_shared defined search -1
            # search_or only set if or search check box was set
            # len(urls_found) != full set urls is a any search_and proxy
            pattern_sharedL = self.phrase_adder(pattern_shared)
            if type(pattern_sharedL) is list and len(pattern_sharedL) > 1:
                for patterni in pattern_sharedL:
                    urls_found = bp.searchAddressStruct(
                        {urlkey: self.addrStruct[urlkey] for urlkey in urls_found},
                        patterni,
                        -1,
                        ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
            else:
                urls_found = bp.searchAddressStruct(
                    self.addrStruct,
                    pattern_shared,
                    -1,
                    ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
        
        # - handle exclusive search by NOT, does not use arrays because exclusive
        for labeli, label in enumerate(self.field_list):
            elementi = labeli - 1
            # print(f'click_search: NOT: {self.search_chk_list[labeli].var_chk_not.get()} at {elementi}') # DDD
            if self.search_chk_list[labeli].var_chk_not.get() == elementi:
                # indicates to include this element in NOT search
                # decide if shared or specific pattern and pass a reduced
                #   addrStruct set based on the current list of urls_found
                # use dictionary comprehension to reduce passed addrStruct
                #   ref: https://stackoverflow.com/questions/3420122/filter-dict-to-contain-only-certain-keys
                search_not = False   # set True if also have a search pattern
                pattern_specific = self.search_chk_list[labeli].entryText.get().strip()
                if len(pattern_specific) > 0:
                    pattern_not = pattern_specific
                    search_not = True
                else:
                    if len(pattern_shared) > 0:
                        pattern_not = pattern_shared
                        search_not = True
                if search_not:
                    # search NOT set and pattern defined therefore search
                    urls_to_drop = bp.searchAddressStruct(
                        {urlkey: self.addrStruct[urlkey] for urlkey in urls_found},
                        pattern_not,
                        elementi,
                        ignore_case=True if self.var_chk_ignorecase.get() == 1 else False)
                    # print(f'drop these:\t{urls_to_drop}')
                    urls_found = [x for x in urls_found if x not in urls_to_drop]
                    # print(f'click_search: NOT: {pattern_not}::::{elementi} dropped {len(urls_to_drop)}, found {len(urls_found)}') # DDD
            if len(urls_found) == 0:
                # stop searching if no urls are left to search
                # indicates search pattern is too exclusive
                break
        
        # apply the list of urls_found to update the data shown
        # print(f'search found {len(urls_found)} vs total {len(list(self.addrStruct.keys()))}') # DDD
        self.view_url_update(urls_found)
        
    def phrase_adder(self, phrases):
        """ Given a list break (or not) the phrases and add them to the list
            of phrases if the var_check_phrase flag is not set
        Args:
            phrases (list): list of strings that may or may not be phrases
        Returns:
            (list): list of strings with phrases broken apart (if flag set)
                if input is string and no additions are made, ie only 1 item
                then the output is a string
        """
        
        if self.var_chk_phrase.get() == 0:
            # not set to phrase search so break apart all the phrases on spaces
            # this will lazy duplicate ie the phrase is also searched
            pattern_build = []
            if type(phrases) is str:
                phrases = phrases.strip()
                if len(phrases) == 0:
                    return phrases
                phrases = [phrases]
                switchback = True
            else:
                switchback = False
            for patterni in phrases:
                # print(f'phrase_adder: {patterni}') # DDD
                patterni = re.sub(' +', ' ', patterni)  # YYY could compile
                pattern_build = pattern_build + patterni.split(' ')
            #if type(phrases) is list:
            # print(f'phrases: {phrases}\npattern_build: {pattern_build}') # DDD
            phrases = list(set(phrases + pattern_build))
            if switchback and len(phrases) == 1:
                # change it back to single input element because there is only 1
                phrases = phrases[0]
        return phrases

    def exit_cleanup(self):
        """ Before exiting run these commands to save the bookmark set """
        self.save_structure()
        self.root.destroy()  # actually exit

    def view_edit_update(self, edit_update_list):
        """ given an edit_update_list of values change the edit_* fields
            note need to print formatted strings to properly format lists
        
        Args:
            edit_update_list (list): a list that defines each edit field in
                turn to update: url, label, age, tags, location, description
                
        """
        assert len(edit_update_list) == 6
        # self.console_log.add_text(f'view_edit_update:{edit_update_list}') # DDD
        self.edit_url.entryText.set(edit_update_list[0]) # set the text

        addr_lab = edit_update_list[1]
        addr_lab = field_to_list(addr_lab)
        # print(f'view_edit_update: push: {type(addr_lab)}::::{addr_lab}::::')  # DDD
        self.edit_label.entryText.set(f'{addr_lab}') # set the text
        
        # print(f'view_edit_update: {type(edit_update_list[2])}::{edit_update_list[2]} for {edit_update_list}') # DDD
        age_str = str(datetime.datetime.fromtimestamp(int(edit_update_list[2])))
        self.edit_age.entryText.set(f'{age_str}') # set the text
        
        addr_tag = edit_update_list[3]
        addr_tag = field_to_list(addr_tag)
        self.edit_tags.entryText.set(f'{addr_tag}') # set the text
                
        addr_loc = edit_update_list[4]
        addr_loc = field_to_list(addr_loc)        
        self.edit_location.sel_text.delete(0, tk.END)  # clear the text
        self.edit_location.sel_text.insert(tk.END, f'{addr_loc}') # set the text
        
        addr_desc = edit_update_list[5]
        addr_desc = field_to_list(addr_desc)
        self.edit_description.sel_text.delete(0, tk.END)  # clear the text
        self.edit_description.sel_text.insert(tk.END, f'{addr_desc}') # set the text
        
    def view_edit_update_by_url(self, url):
        """ given a url update the selected to edit boxes
                
        Args:
            url (str): addr url as a string
        """
        # reverse lookup by index the label in the other list
        if 0: # DDD
            print(f'view_edit_update_by_url::{url}')
            if url in self.addrStruct:
                print('found url')
            else:
                print('URL not found')
        if url not in self.addrStruct: # this seems very unlikely
            return
        label = self.addrStruct[url][0]
        age = self.addrStruct[url][1]
        tags = self.addrStruct[url][2]
        location = self.addrStruct[url][3]
        description = self.addrStruct[url][4]

        edit_update_list = [url, label, age, tags, location, description]
        self.view_edit_update(edit_update_list)

    def view_url_update(self, url_list=None):
        """ given a urls or just use addrStruct to update the bookmark_lists 
        paired listbox. the called function clears the lists first.
        
        Args:
            url_list (list): list of urls to put in left listbox
                if None (default) generate both url and labels from addrStruct
        """
        if url_list is None:
            # generate the url_list from addrStruct keys ie all urls
            url_list = list(self.addrStruct.keys())
        self.addrStruct_url_list = url_list  # so current list matches view
        label_list = []
        # generate the label_list from addrStruct for the urls specified
        for addr in url_list:
            label_list.append(self.addrStruct[addr][0][0]) # the url name label
        self.bookmark_lists.update(url_list, label_list, sort_side=2)
    
    def view_search_reset(self):
        """ reset the search ie clear it, resets URL view to unfiltered """
        self.view_url_update()
        
    def add_bookmark(self):
        """ pull new bookmark information to add bookmark by:
            check for existance
            call bp methods to add it
        """
        addr_url = self.new_url.entryText.get()
        if len(addr_url) == 0:
            self.console_log.add_text('url add needs a url')
            return
        if addr_url in self.addrStruct.keys():
            # old url select it into the edit/select screen
            self.console_log.add_text(f'url exists: {addr_url} at {time.time()}')
            self.view_edit_update_by_url(addr_url)
        else:
            # new url, pull content from self.new_* definitions
            addr_lab = self.new_label.get_text_as_type()
            addr_lab = field_to_list(addr_lab)
            addr_tag = self.new_tags.get_text_as_type()
            addr_tag = field_to_list(addr_tag)
            addr_loc = self.new_location.get_text_as_type()
            addr_loc = field_to_list(addr_loc)
            addr_desc = self.new_description.get_text_as_type()
            addr_desc = field_to_list(addr_desc)
            
            # set the content in the addrStruct
            self.addrStruct[addr_url] = [
                addr_lab,    # label list
                str(int(time.time())),      # age in posix time as string
                addr_tag,    # tags list
                addr_loc,    # location list
                addr_desc,   # description list
                []           # file location list
                ]
            
            # update the GUI log and list views
            self.console_log.add_text(f'url added: {addr_url} at {time.time()}')
            self.bookmark_lists.add([addr_url], addr_lab)
            self.update_action = True # set change occurred
            self.write_log(change_type='Add', change_url=addr_url)
        
    def edit_bookmark(self):
        """ commit edited bookmark information associated with URL in the
        selected bookmark information in self.edit_* GUI elements
        overwriting existing definition by:
            check for existance
            call bp methods to overwrite it
        confirmed that reading bookmark into GUI then writing back is not 
            mangling types or content
        """
        addr_url = self.edit_url.get_text_as_type()
        if addr_url not in self.addrStruct.keys():
            self.console_log.add_text('url add not allowed via edit dialog. Add via update and remove existing url here.')
            return
        
        # update the addrStruct with the edited bookmark information
        addr_age = self.addrStruct[addr_url][1]
        # print(f'edit_bookmark: pull: age: {type(addr_age)}::::{addr_age}::::') # DDD
        addr_file = self.addrStruct[addr_url][5]
        addr_lab = self.edit_label.get_text_as_type()
        # print(f'edit_bookmark: pull: label: {type(addr_lab)}::::{addr_lab}::::') # DDD
        addr_lab = field_to_list(addr_lab)
        addr_tag = self.edit_tags.get_text_as_type()
        addr_tag = field_to_list(addr_tag)
        addr_loc = self.edit_location.get_text_as_type()
        addr_loc = field_to_list(addr_loc)
        addr_desc = self.edit_description.get_text_as_type()
        addr_desc = field_to_list(addr_desc)

        self.addrStruct[addr_url] = [
            addr_lab,    # label list
            addr_age,    # age in posix time as string
            addr_tag,    # tags list
            addr_loc,    # location list
            addr_desc,   # description list
            addr_file    # file location list
            ]
        self.console_log.add_text(f'url updated: {addr_url} at {time.time()}')
        self.update_action = True # set change tracker to true
        self.write_log(change_type='Edit', change_url=addr_url)
        
        # update the lists shown to users
        #   trigger a list update for list view impact
        # getting list from listbox
        #   ref: https://www.tutorialspoint.com/python/tk_listbox.htm
        # note it may be faster to pull 1 at a time up to total n
        urls_shown_n = self.bookmark_lists.left.size()
        urls_shown = self.bookmark_lists.left.get(0, urls_shown_n)
        # print(f'edit_bookmark replace found {urls_shown_n} and {len(urls_shown)}') # DDD
        url_update = -1
        for i, url in enumerate(urls_shown):
            if url == addr_url:
                # means to update the url at i by remove and replace existing
                url_update = i
                self.bookmark_lists.left.delete(url_update)
                self.bookmark_lists.right.delete(url_update)
                if type(addr_lab) is list:
                    # handle addr_lab so it doesn't look stupid in list view
                    if len(addr_lab) > 1:
                        # convert to non-list string
                        self.console_log.add_text(f'edit_bookmark {addr_url} has {len(addr_lab)} label list length')
                        # could join: ("','\t').join(['a','b','c'])
                        # choose to dump as string to be a clear list
                        addr_lab = f'{addr_lab}' # .strip('[').strip(']')
                    else:
                        addr_lab = addr_lab[0]        
                self.bookmark_lists.left.insert(url_update, addr_url)
                self.bookmark_lists.right.insert(url_update, addr_lab)
                break
        if url_update == -1:
            self.console_log.add_text('How was url being edited not found? Could happen if select, search removes selected from shown set, then edit selected')
            
        # TODO list view impact needs to handle list of values ie formatted text input
        
    def remove_bookmark(self):
        """ remove the bookmark associated with the URL in the selected
        bookmark information in self.edit_* GUI elements. 
            check for existance
            remove from addrStruct
            remove from the current addr url list
            remove from bookmark_lists ScrolledTextPair
        """
        addr_url = self.edit_url.entryText.get()
    
        confirm_int = simpledialog.askinteger(
            title='URL Remove Confirmation',
            prompt=f'Enter an even number to confirm delete of url {addr_url}')
        # if confirm_int is None or confirm_int % 2 != 0:
        #     print(f'URL not removed; good thing I asked. {confirm_int}')
        # else:
        #     print(f'you confirmed action as: {confirm_int}')
        if not(confirm_int is None or confirm_int % 2 != 0):
            # you confirmed the action
            if addr_url in self.addrStruct.keys():
                del self.addrStruct[addr_url]
                url_list = [url for url in self.addrStruct_url_list if url != addr_url]
                self.view_url_update(url_list)  # updates the GUI
                self.console_log.add_text(f'url removed: {addr_url} at {time.time()}')        
                self.update_action = True # set change tracker to true
                self.write_log(change_type='Drop', change_url=addr_url)
        
    def save_structure(self):
        """ save the current structure with a timestamped file if change occurred """
        if not self.update_action: # boolean tracks if change occurred
            return
        if not os.path.exists(self.output_dir):
            # print('Defined output_path does not exist, create it.')
            os.makedirs(self.output_dir)
        path_save = os.path.join(self.output_dir,
                                 f'{self.output_base}.{get_time_str()}.json')
        print(f'Exiting: Save the state to {path_save}')  
        bp.writeAddressStruct(self.addrStruct, path_save)
        
    def set_output_filename(self, event=None):
        """ change base filename to save data to. GUI calls
        
        Args:
            event: event arg (not used)
        """
        file_name = simpledialog.askstring(
            title='Define base output filename used as <base>.YYYYMMDD.MMSS.json',
            prompt='Enter base string to build output filename from:')
        if self.set_output_filename_act(file_name) == 0:
            self.console_log.add_text(f'Set output base filename to {self.output_base}')

    def set_output_filename_act(self, file_name):
        """ change base filename to save data to
        Args:
            file_name (str)
        Returns:
            0 success
            1 not changed
        """
        if type(file_name) is str and len(file_name) > 0:
            self.output_base = file_name
            return 0
        else:
            return 1        
    
    def get_output_filename(self):
        return self.output_base

    def set_output_path(self, event=None):
        """
        Args:
            event: event arg (not used)
        """
        file_path = filedialog.askdirectory(
            title='Select directory to save bookmark addrStruct to',
            initialdir=self.output_dir)
        if file_path is None or type(file_path) is not str:
            # handle case when return without a value from the dialog
            return
        if self.set_output_path_act(file_path) == 0:
            self.console_log.add_text(f'Set output path to {self.output_dir}')

    def set_output_path_act(self, file_path):
        if os.path.exists(file_path):
            self.output_dir = file_path
            return 0
        else:
            return 1

    def get_output_path(self):
        return self.output_dir

    @staticmethod
    def show_info_window():
        msg = """
        pure python Bookmark JSON viewer 
        by Crumbs
        Ver.""" + VERSION + """\n
        """
        messagebox.showinfo("About", msg)
        
    def show_disabled(self, event=None):
        """
        Args:
            event: event arg
        """
        msg = f"this functionality is disabled: {event}"
        messagebox.showinfo("Disabled Action", msg)
        
    def open_github_page(self):
        self.open_url("https://github.com/Crumbs350/pybookmark")
        
    def open_url(self, url):
        """ use webbrowser package to open a url in default webbrowser """
        if is_url(url):
            webbrowser.open(url)
        else:
            self.console_log.add_text(f'Error: this is not url: {url}')
            
    def write_log(self, change_type, change_url):
        """ write the log file that tracks changes
        Args:
            change_type (str): type of change, printed value. any allowed but
                the expected values are 'Add', 'Edit', 'Drop'
            change_url (str): url that was changed
        """
        if not os.path.exists(self.output_dir):
            os.makedirs(self.output_dir)
        path_save = os.path.join(self.output_dir,
                                 f'{self.output_base}.changes.log')
        with open(path_save, 'a') as fHan:
            fHan.write(f'{change_type}, {change_url}\n')


def view_data(json_file=None, json_data=None, initial_dir=None, output_dir=None,
              width=800, height=800, posx=None, posy=None, config=None):
    """ View_data defines the tk GUI, if no arguments given opens blank viewer
    
        powered by the BookmarkGUI class
    
    Args:
        json_file (str): path to a json file, ignores json_data if defined
            file will be opened and viewed. default = None
            preferred method
        json_data (dict): dictionary generated from json data content
            alternative to json_file for defining content to view
            this is less tested and some functionality may not be configurable
            default = None
        initial_dir (str): path to start in. default = None
        output_dir (str): path to save files to; uses input json_file if
            not defined, then uses PROJECT_DIR if even json_file isn't defined
            but don't do that, just define it.
        width (int): Tk GUI window width, default = 500
        height (int): Tk GUI window width, default = 500
        posx (int): Tk GUI window x offset, default = None
        posy (int): Tk GUI window y offset, default = None
        config (dict): configuration dictionary
    Returns:
        None        
    """
    # - create the basic GUI window
    root: Tk = tk.Tk() # per https://stackoverflow.com/questions/20251161/tkinter-tclerror-image-pyimage3-doesnt-exist
    root.title('Bookmark GUI')     # window title
    root.minsize(500, 500)  # width x height
    window_geometry = f'{width}x{height}'
    if posx is not None:
        if posy is not None:
            window_geometry = f'{window_geometry}+{posx}+{posy}'
        else:
            window_geometry = f'{window_geometry}+{posx}+{0}'
    root.geometry(window_geometry)          # window size and position
    root.resizable(True, True)
    root.tk.call('wm',
                 'iconphoto',
                 root._w,
                 tk.PhotoImage(file=os.path.join(PROJECT_DIR, 'icon.png'), master=root))
    menubar = tk.Menu(root)

    # - handle input data definition: get the data into the 'action' class
    addrStruct = None  # starting state
    if json_file:
        addrStruct = bp.readAddressStruct(json_file)
        if output_dir is None:
            output_dir = os.path.dirname(json_file)
        output_base = os.path.basename(json_file).split('.')[0]
    if json_data:
        addrStruct = json_data
        output_base = 'addr'
    assert addrStruct is not None
    if output_dir is None:
        # occurs if pass json_data not json_file and no output_dir
        output_dir = PROJECT_DIR
    
    # clean the addrStruct of invalid values
    emptyContentDropSet = ['', 'None', None]
    addrStruct = bp.cleanAddressStruct(addrStruct, emptyContentDropSet)

    # - define the GUI
    app = BookmarkGUI(root, bookmarks_data=addrStruct, output_dir=output_dir, output_base=output_base, config=config)
    
    # - define GUI menus (note accelerator just shows, bind_all defines action)
    # YYY note purposely did NOT:
    #  1. implement ability to open any random addrStruct file via GUI
    #     if do MUST put/update state of output path in BookmarkGUI class
    #  2. implement ability import a bookmark file into struct via GUI
    #     prefer to do manually via command line for now
    #  3. implement ability to export content via GUI
    #     prefer to do manually via command line for now
    file_menu = tk.Menu(menubar, tearoff=0)
    file_menu.add_command(label="Open File (disabled)",
                          accelerator='Ctrl+O',
                          command=app.show_disabled)
    file_menu.add_command(label="Save File",
                          accelerator='Ctrl+S',
                          command=app.save_structure)
    file_menu.add_command(label="Set Base File Name",
                          command=app.set_output_filename)
    file_menu.add_command(label="Set Output Path",
                          command=app.set_output_path)
    menubar.add_cascade(label="File", menu=file_menu)

    help_menu = tk.Menu(menubar, tearoff=0)
    help_menu.add_command(label="About", command=app.show_info_window)
    help_menu.add_command(label="Show GitHub page",
                          command=app.open_github_page)
    #help_menu.add_command(label="Show release note",
    #                      command=app.open_release_note)
    menubar.add_cascade(label="Help", menu=help_menu)

    root.columnconfigure(0, weight=1)
    root.rowconfigure(0, weight=1)

    root.config(menu=menubar)
    root.bind_all("<Control-o>", lambda e: app.show_disabled(event=e))
    root.bind_all("<Control-s>", lambda e: app.show_disabled(event=e))
    root.protocol("WM_DELETE_WINDOW", app.exit_cleanup)  # https://stackoverflow.com/questions/111155/how-do-i-handle-the-window-close-event-in-tkinter
    root.mainloop()


def main():
    """ Main function used to define how to handle the arguments passed and
        run the application. Can also use YAML config file
        
    Args:
        -f --file (str): JSON file path for AddrStruct
            required even if yaml specified and it is ignored
        -o --dir (str): where to save output content to during shutdown
            updated files are saved with time stamp
            default = path of the JSON file
        -n --newest (str): finds newest file of the same name as file in path
            but with timestamp
        -w --width (int): change GUI width
        -h --height (int): change GUI height
        -y --config (str): yaml config file
            can also load as current path pybookmark_viewer.yaml w/o any arguments
        
    """

    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--file', 
                        type=str,
                        help='AddrStruct JSON file path')
    parser.add_argument('-o', '--dir', type=str,
                        required=False,
                        help='output directory')
    parser.add_argument('-n', '--newest', action='store_true',
                        default=False,
                        required=False,
                        help='Find newest matching JSON file')
    parser.add_argument('-w', '--width',
                        type=int,
                        default=500,
                        required=False,
                        help='window width')
    parser.add_argument('-h', '--height',
                        type=int,
                        default=500,
                        required=False,
                        help='window height')
    parser.add_argument('-y', '--config',
                        type=str,
                        default=None,
                        required=False,
                        help='yaml config file path other arguments overwritten/ignored')
    args = parser.parse_args()

    
    use_yaml = False
    if not len(sys.argv) > 1:
        # no passed arguments use yaml config in the current path
        use_yaml = True
        config_f = os.path.join(os.path.abspath(os.path.curdir), 'pybookmark_viewer.yaml')
    else:
        # test if user asked for a specific yaml file to be used
        if args.config:
            use_yaml = True
            config_f = args.config

    # set viewer geometry
    height = width = 800
    if args.width:
        width = args.width
    if args.height:
        height = args.height

    # set file and directory paths
    file_use = args.file
    if args.newest:
        file_use = addr_newest(file_use)
    if args.dir:
        output_dir = args.dir
    else:
        output_dir = os.path.abspath(os.path.dirname(file_use))

    if use_yaml:
        # use yaml instead of passed arguments, yaml assumed in current path
        assert os.path.exists(config_f)
        with open(config_f, 'r') as fHan:
            config_args = yaml.safe_load(fHan)
        # file (str)
        # load_newest (boolean)
        # output_dir (str)
        # width (int)
        # height (int)
        if 'file' in config_args:
            file_use = config_args['file']
            
        if not os.path.exists(file_use):
            # test for file_use is in the same path as the yaml config file
            # this allows lazy -y definition and finding all files w/o
            # changing to the data path prior to running the python script
            file_use2 = os.path.join(
                os.path.dirname(os.path.abspath(config_f)),
                os.path.basename(file_use))
            if os.path.exists(file_use2):
                print(f'User specified {file_use} but found {file_use2} to use.')
                file_use = file_use2
            else:
                print(f'User specified {file_use} not found even adjacent to yaml file')
    
        if 'width' in config_args:
            width = config_args['width']
        if 'height' in config_args:
            height = config_args['height']

        if 'output_dir' in config_args:
            output_dir = config_args['output_dir']
        else:
            output_dir = os.path.abspath(os.path.dirname(file_use))
            
        if config_args['load_newest']:
            file_use = addr_newest(file_use)
    else:
        config_args = None

    view_data(json_file = file_use,
              initial_dir = os.path.dirname(file_use),
              output_dir = output_dir, 
              width = width, 
              height = height,
              config = config_args)


def test_view_data():
    addrStruct = bp.readAddressStruct(os.path.join(PROJECT_DIR, 'data', 'addr.json'))
    view_data(json_data=addrStruct)


if __name__ == '__main__':
    main()
    # test_BookmarkGUI()
    # test_view_data()
