"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const eks = require("../lib");
const util_1 = require("./util");
/* eslint-disable max-len */
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'create nodegroup correctly'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        new eks.Nodegroup(stack, 'Nodegroup', { cluster });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            ClusterName: {
                Ref: 'Cluster9EE0221C',
            },
            NodeRole: {
                'Fn::GetAtt': [
                    'NodegroupNodeGroupRole038A128B',
                    'Arn',
                ],
            },
            Subnets: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
            ForceUpdateEnabled: true,
            ScalingConfig: {
                DesiredSize: 2,
                MaxSize: 2,
                MinSize: 1,
            },
        }));
        test.done();
    },
    'aws-auth will be updated'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        new eks.Nodegroup(stack, 'Nodegroup', { cluster });
        // THEN
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'NodegroupNodeGroupRole038A128B',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
            ClusterName: {
                Ref: 'Cluster9EE0221C',
            },
            RoleArn: {
                'Fn::GetAtt': [
                    'ClusterCreationRole360249B6',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'create nodegroup correctly with security groups provided'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            remoteAccess: {
                sshKeyName: 'foo',
                sourceSecurityGroups: [new ec2.SecurityGroup(stack, 'SG', { vpc })],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            RemoteAccess: {
                Ec2SshKey: 'foo',
                SourceSecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'SGADB53937',
                            'GroupId',
                        ],
                    },
                ],
            },
        }));
        test.done();
    },
    'create nodegroup with forceUpdate disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        new eks.Nodegroup(stack, 'Nodegroup', { cluster, forceUpdate: false });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            ForceUpdateEnabled: false,
        }));
        test.done();
    },
    'create nodegroups with kubectlEnabled is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.LegacyCluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 2,
            version: CLUSTER_VERSION,
        });
        // add a extra nodegroup
        cluster.addNodegroup('extra-ng');
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EKS::Nodegroup', 2));
        test.done();
    },
    'create nodegroup with instanceType provided'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            instanceType: new ec2.InstanceType('m5.large'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            InstanceTypes: [
                'm5.large',
            ],
        }));
        test.done();
    },
    'remoteAccess without security group provided'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            remoteAccess: {
                sshKeyName: 'foo',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            RemoteAccess: {
                Ec2SshKey: 'foo',
            },
        }));
        test.done();
    },
    'import nodegroup correctly'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        // const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        const ng = new eks.Nodegroup(stack1, 'Nodegroup', { cluster });
        const imported = eks.Nodegroup.fromNodegroupName(stack2, 'ImportedNg', ng.nodegroupName);
        new cdk.CfnOutput(stack2, 'NodegroupName', { value: imported.nodegroupName });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                NodegroupName: {
                    Value: {
                        'Fn::ImportValue': 'Stack:ExportsOutputRefNodegroup62B4B2C1EF8AB7C1',
                    },
                },
            },
        });
        test.done();
    },
    'addNodegroup correctly'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addNodegroupCapacity('ng');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            ClusterName: {
                Ref: 'Cluster9EE0221C',
            },
            NodeRole: {
                'Fn::GetAtt': [
                    'ClusterNodegroupngNodeGroupRoleDA0D35DA',
                    'Arn',
                ],
            },
            Subnets: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
            ForceUpdateEnabled: true,
            ScalingConfig: {
                DesiredSize: 2,
                MaxSize: 2,
                MinSize: 1,
            },
        }));
        test.done();
    },
    'throws when desiredSize > maxSize'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // THEN
        test.throws(() => cluster.addNodegroupCapacity('ng', { desiredSize: 3, maxSize: 2 }), /Desired capacity 3 can't be greater than max size 2/);
        test.done();
    },
    'throws when desiredSize < minSize'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // THEN
        test.throws(() => cluster.addNodegroupCapacity('ng', { desiredSize: 2, minSize: 3 }), /Minimum capacity 3 can't be greater than desired size 2/);
        test.done();
    },
    'create nodegroup correctly with launch template'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        const userData = ec2.UserData.forLinux();
        userData.addCommands('set -o xtrace', `/etc/eks/bootstrap.sh ${cluster.clusterName}`);
        const lt = new ec2.CfnLaunchTemplate(stack, 'LaunchTemplate', {
            launchTemplateData: {
                imageId: new eks.EksOptimizedImage().getImage(stack).imageId,
                instanceType: new ec2.InstanceType('t3.small').toString(),
                userData: cdk.Fn.base64(userData.render()),
            },
        });
        cluster.addNodegroupCapacity('ng-lt', {
            launchTemplateSpec: {
                id: lt.ref,
                version: lt.attrDefaultVersionNumber,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            LaunchTemplate: {
                Id: {
                    Ref: 'LaunchTemplate',
                },
                Version: {
                    'Fn::GetAtt': [
                        'LaunchTemplate',
                        'DefaultVersionNumber',
                    ],
                },
            },
        }));
        test.done();
    },
    'throws when both diskSize and launch template specified'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        const userData = ec2.UserData.forLinux();
        userData.addCommands('set -o xtrace', `/etc/eks/bootstrap.sh ${cluster.clusterName}`);
        const lt = new ec2.CfnLaunchTemplate(stack, 'LaunchTemplate', {
            launchTemplateData: {
                imageId: new eks.EksOptimizedImage().getImage(stack).imageId,
                instanceType: new ec2.InstanceType('t3.small').toString(),
                userData: cdk.Fn.base64(userData.render()),
            },
        });
        // THEN
        test.throws(() => cluster.addNodegroupCapacity('ng-lt', {
            diskSize: 100,
            launchTemplateSpec: {
                id: lt.ref,
                version: lt.attrDefaultVersionNumber,
            },
        }), /diskSize must be specified within the launch template/);
        test.done();
    },
    'throws when both instanceType and launch template specified'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: true,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        const userData = ec2.UserData.forLinux();
        userData.addCommands('set -o xtrace', `/etc/eks/bootstrap.sh ${cluster.clusterName}`);
        const lt = new ec2.CfnLaunchTemplate(stack, 'LaunchTemplate', {
            launchTemplateData: {
                imageId: new eks.EksOptimizedImage().getImage(stack).imageId,
                instanceType: new ec2.InstanceType('t3.small').toString(),
                userData: cdk.Fn.base64(userData.render()),
            },
        });
        // THEN
        test.throws(() => cluster.addNodegroupCapacity('ng-lt', {
            instanceType: new ec2.InstanceType('c5.large'),
            launchTemplateSpec: {
                id: lt.ref,
                version: lt.attrDefaultVersionNumber,
            },
        }), /Instance types must be specified within the launch template/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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