"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const pinger_1 = require("./pinger/pinger");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        const secretsEncryptionKey = new kms.Key(this, 'SecretsKey');
        // just need one nat gateway to simplify the test
        this.vpc = new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // create the cluster with a default nodegroup capacity
        this.cluster = new eks.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            mastersRole,
            defaultCapacity: 2,
            version: eks.KubernetesVersion.V1_17,
            secretsEncryptionKey,
        });
        this.assertFargateProfile();
        this.assertCapacityX86();
        this.assertCapacityArm();
        this.assertBottlerocket();
        this.assertSpotCapacity();
        this.assertInferenceInstances();
        this.assertNodeGroupX86();
        this.assertNodeGroupArm();
        this.assertNodeGroupCustomAmi();
        this.assertSimpleManifest();
        this.assertSimpleHelmChart();
        this.assertCreateNamespace();
        this.assertServiceAccount();
        this.assertServiceLoadBalancerAddress();
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: this.cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: this.cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: this.cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterSecurityGroupId', { value: this.cluster.clusterSecurityGroupId });
        new core_1.CfnOutput(this, 'ClusterEncryptionConfigKeyArn', { value: this.cluster.clusterEncryptionConfigKeyArn });
        new core_1.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
    }
    assertServiceAccount() {
        // add a service account connected to a IAM role
        this.cluster.addServiceAccount('MyServiceAccount');
    }
    assertCreateNamespace() {
        // deploy an nginx ingress in a namespace
        const nginxNamespace = this.cluster.addManifest('nginx-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'nginx',
            },
        });
        const nginxIngress = this.cluster.addHelmChart('nginx-ingress', {
            chart: 'nginx-ingress',
            repository: 'https://helm.nginx.com/stable',
            namespace: 'nginx',
            wait: true,
            createNamespace: false,
            timeout: core_1.Duration.minutes(15),
        });
        // make sure namespace is deployed before the chart
        nginxIngress.node.addDependency(nginxNamespace);
    }
    assertSimpleHelmChart() {
        // deploy the Kubernetes dashboard through a helm chart
        this.cluster.addHelmChart('dashboard', {
            chart: 'kubernetes-dashboard',
            repository: 'https://kubernetes.github.io/dashboard/',
        });
    }
    assertSimpleManifest() {
        // apply a kubernetes manifest
        this.cluster.addManifest('HelloApp', ...hello.resources);
    }
    assertNodeGroupX86() {
        // add a extra nodegroup
        this.cluster.addNodegroupCapacity('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: this.cluster.defaultCapacity ? this.cluster.defaultCapacity.role : undefined,
        });
    }
    assertNodeGroupCustomAmi() {
        var _a, _b;
        // add a extra nodegroup
        const userData = ec2.UserData.forLinux();
        userData.addCommands('set -o xtrace', `/etc/eks/bootstrap.sh ${this.cluster.clusterName}`);
        const lt = new ec2.CfnLaunchTemplate(this, 'LaunchTemplate', {
            launchTemplateData: {
                imageId: new eks.EksOptimizedImage().getImage(this).imageId,
                instanceType: new ec2.InstanceType('t3.small').toString(),
                userData: core_1.Fn.base64(userData.render()),
            },
        });
        this.cluster.addNodegroupCapacity('extra-ng2', {
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: ((_a = this.cluster.defaultNodegroup) === null || _a === void 0 ? void 0 : _a.role) || ((_b = this.cluster.defaultCapacity) === null || _b === void 0 ? void 0 : _b.role),
            launchTemplateSpec: {
                id: lt.ref,
                version: lt.attrDefaultVersionNumber,
            },
        });
    }
    assertNodeGroupArm() {
        // add a extra nodegroup
        this.cluster.addNodegroupCapacity('extra-ng-arm', {
            instanceType: new ec2.InstanceType('m6g.medium'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: this.cluster.defaultCapacity ? this.cluster.defaultCapacity.role : undefined,
        });
    }
    assertInferenceInstances() {
        // inference instances
        this.cluster.addAutoScalingGroupCapacity('InferenceInstances', {
            instanceType: new ec2.InstanceType('inf1.2xlarge'),
            minCapacity: 1,
        });
    }
    assertSpotCapacity() {
        // spot instances (up to 10)
        this.cluster.addAutoScalingGroupCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
    }
    assertBottlerocket() {
        // add bottlerocket nodes
        this.cluster.addAutoScalingGroupCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
    }
    assertCapacityX86() {
        // add some x86_64 capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addAutoScalingGroupCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
    }
    assertCapacityArm() {
        // add some arm64 capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addAutoScalingGroupCapacity('NodesArm', {
            instanceType: new ec2.InstanceType('m6g.medium'),
            minCapacity: 1,
        });
    }
    assertFargateProfile() {
        // fargate profile for resources in the "default" namespace
        this.cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
    }
    assertServiceLoadBalancerAddress() {
        const serviceName = 'webservice';
        const labels = { app: 'simple-web' };
        const containerPort = 80;
        const servicePort = 9000;
        const pingerSecurityGroup = new ec2.SecurityGroup(this, 'WebServiceSecurityGroup', {
            vpc: this.vpc,
        });
        pingerSecurityGroup.addIngressRule(pingerSecurityGroup, ec2.Port.tcp(servicePort), `allow http ${servicePort} access from myself`);
        this.cluster.addManifest('simple-web-pod', {
            kind: 'Pod',
            apiVersion: 'v1',
            metadata: { name: 'webpod', labels: labels },
            spec: {
                containers: [{
                        name: 'simplewebcontainer',
                        image: 'nginx',
                        ports: [{ containerPort: containerPort }],
                    }],
            },
        });
        this.cluster.addManifest('simple-web-service', {
            kind: 'Service',
            apiVersion: 'v1',
            metadata: {
                name: serviceName,
                annotations: {
                    // this is furtile soil for cdk8s-plus! :)
                    'service.beta.kubernetes.io/aws-load-balancer-internal': 'true',
                    'service.beta.kubernetes.io/aws-load-balancer-extra-security-groups': pingerSecurityGroup.securityGroupId,
                },
            },
            spec: {
                type: 'LoadBalancer',
                ports: [{ port: servicePort, targetPort: containerPort }],
                selector: labels,
            },
        });
        const loadBalancerAddress = this.cluster.getServiceLoadBalancerAddress(serviceName);
        // create a resource that hits the load balancer to make sure
        // everything is wired properly.
        const pinger = new pinger_1.Pinger(this, 'ServicePinger', {
            url: `http://${loadBalancerAddress}:${servicePort}`,
            securityGroup: pingerSecurityGroup,
            vpc: this.vpc,
        });
        // this should display a proper nginx response
        // <title>Welcome to nginx!</title>...
        new core_1.CfnOutput(this, 'Response', {
            value: pinger.response,
        });
    }
}
// this test uses both the bottlerocket image and the inf1 instance, which are only supported in these
// regions. see https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-eks#bottlerocket
// and https://aws.amazon.com/about-aws/whats-new/2019/12/introducing-amazon-ec2-inf1-instances-high-performance-and-the-lowest-cost-machine-learning-inference-in-the-cloud/
const supportedRegions = [
    'us-east-1',
    'us-west-2',
];
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
const stack = new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
if (process.env.CDK_INTEG_ACCOUNT !== '12345678') {
    // only validate if we are about to actually deploy.
    // TODO: better way to determine this, right now the 'CDK_INTEG_ACCOUNT' seems like the only way.
    if (core_1.Token.isUnresolved(stack.region)) {
        throw new Error(`region (${stack.region}) cannot be a token and must be configured to one of: ${supportedRegions}`);
    }
    if (!supportedRegions.includes(stack.region)) {
        throw new Error(`region (${stack.region}) must be configured to one of: ${supportedRegions}`);
    }
}
app.synth();
//# sourceMappingURL=data:application/json;base64,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