import requests
from typing import Dict

from pyoti import __version__
from pyoti.classes import FileHash
from pyoti.exceptions import MalwareHashRegistryError


class MalwareHashRegistry(FileHash):
    """MalwareHashRegistry Malicious File Hashes

    Team Cymru aggregates results of over 30 AV tools, including their own analysis,
    to improve detection rates of malicious files.
    """
    def __init__(self, api_key: str, api_url: str = "https://hash.cymru.com/v2"):
        """
        :param api_key: MalwareHashRegistry API key
        :param api_url: MalwareHashRegistry API URL
        """
        FileHash.__init__(self, api_key=api_key, api_url=api_url)

    def _api_get(self) -> requests.models.Response:
        """GET request to API for single hash lookup"""
        credentials = self.api_key.split(":")

        headers = {"User-Agent": f"PyOTI {__version__}"}

        response = requests.request(
            "GET",
            url=f"{self.api_url}/{self.file_hash}",
            auth=(credentials[0], credentials[1]),
            headers=headers
        )

        return response

    def check_hash(self) -> Dict:
        """Checks File Hash reputation

        :return: request response dict
        """
        response = self._api_get()
        r = response.json()
        if r.get('error'):
            raise MalwareHashRegistryError(r.get('msg'))
        else:
            return r
