import copy
from dataclasses import dataclass
from typing import Any, Dict, List, Optional, Union

import yaml


@dataclass
class EvalResult:
    """
    Flattened representation of individual evaluation results found in model-index.
    """

    # Required

    # The task identifier
    # Example: automatic-speech-recognition
    task_type: str

    # The dataset identifier
    # Example: common_voice. Use dataset id from https://hf.co/datasets
    dataset_type: str

    # A pretty name for the dataset.
    # Example: Common Voice (French)
    dataset_name: str

    # The metric identifier
    # Example: wer. Use metric id from https://hf.co/metrics
    metric_type: str

    # Value of the metric.
    # Example: 20.0 or "20.0 ± 1.2"
    metric_value: Any

    # Optional

    # A pretty name for the task.
    # Example: Speech Recognition
    task_name: Optional[str] = None

    # The name of the dataset configuration used in `load_dataset()`.
    # Example: fr in `load_dataset("common_voice", "fr")`.
    # See the `datasets` docs for more info:
    # https://huggingface.co/docs/datasets/package_reference/loading_methods#datasets.load_dataset.name
    dataset_config: Optional[str] = None

    # The split used in `load_dataset()`.
    # Example: test
    dataset_split: Optional[str] = None

    # The revision (AKA Git Sha) of the dataset used in `load_dataset()`.
    # Example: 5503434ddd753f426f4b38109466949a1217c2bb
    dataset_revision: Optional[str] = None

    # The arguments passed during `Metric.compute()`.
    # Example for `bleu`: max_order: 4
    dataset_args: Optional[Dict[str, Any]] = None

    # A pretty name for the metric.
    # Example: Test WER
    metric_name: Optional[str] = None

    # The name of the metric configuration used in `load_metric()`.
    # Example: bleurt-large-512 in `load_metric("bleurt", "bleurt-large-512")`.
    # See the `datasets` docs for more info: https://huggingface.co/docs/datasets/v2.1.0/en/loading#load-configurations
    metric_config: Optional[str] = None

    # The arguments passed during `Metric.compute()`.
    # Example for `bleu`: max_order: 4
    metric_args: Optional[Dict[str, Any]] = None

    # If true, indicates that evaluation was generated by Hugging Face (vs. self-reported).
    verified: Optional[bool] = None


@dataclass
class CardData:
    def __init__(
        self,
        language: Optional[Union[str, List[str]]] = None,
        license: Optional[str] = None,
        library_name: Optional[str] = None,
        tags: Optional[List[str]] = None,
        datasets: Optional[Union[str, List[str]]] = None,
        metrics: Optional[Union[str, List[str]]] = None,
        eval_results: Optional[List[EvalResult]] = None,
        model_name: Optional[str] = None,
        **kwargs,
    ):
        """Model Card Metadata that is used by Hugging Face Hub when included at the top of your README.md

        Args:
            language (Union[str, List[str]], *optional*):
                Language of model's training data or metadata. Example: `'en'`. Defaults to `None`.
            license (str, *optional*):
                License of this model. Example: apache-2.0 or any license from
                https://hf.co/docs/hub/model-repos#list-of-license-identifiers. Defaults to None.
            library_name (str, *optional*):
                Name of library used by this model. Example: keras or any library from
                https://github.com/huggingface/hub-docs/blob/main/js/src/lib/interfaces/Libraries.ts.
                Defaults to None.
            tags (List[str], *optional*):
                List of tags to add to your model that can be used when filtering on the Hugging
                Face Hub. Defaults to None.
            datasets (Union[str, List[str]], *optional*):
                Dataset or list of datasets that were used to train this model. Should be a dataset ID
                found on https://hf.co/datasets. Defaults to None.
            metrics (Union[str, List[str]], *optional*):
                List of metrics used to evaluate this model. Should be a metric name that can be found
                at https://hf.co/metrics. Example: 'accuracy'. Defaults to None.
            eval_results (Union[List[EvalResult], EvalResult], *optional*):
                List of `modelcards.EvalResult` that define evaluation results of the model. If provided,
                `model_name` kwarg must be provided. Defaults to `None`.
            model_name (str, *optional*):
                A name for this model. Required if you provide `eval_results`. It is used along with
                `eval_results` to construct the `model-index` within the card's metadata. The name
                you supply here is what will be used on PapersWithCode's leaderboards. Defaults to None.
            kwargs (dict, *optional*):
                Additional metadata that will be added to the model card. Defaults to None.

        Example:
            >>> from modelcards.card_data import CardData
            >>> card_data = CardData(
            ...     language="en",
            ...     license="mit",
            ...     library_name="timm",
            ...     tags=['image-classification', 'resnet'],
            ... )
            >>> card_data.to_dict()
            {'language': 'en', 'license': 'mit', 'library_name': 'timm', 'tags': ['image-classification', 'resnet']}

        """

        self.language = language
        self.license = license
        self.library_name = library_name
        self.tags = tags
        self.datasets = datasets
        self.metrics = metrics
        self.eval_results = eval_results
        self.model_name = model_name
        self.__dict__.update(kwargs)

        if self.eval_results:
            if type(self.eval_results) == EvalResult:
                self.eval_results = [self.eval_results]
            if self.model_name is None:
                raise ValueError("`eval_results` requires `model_name` to be set.")

    def to_dict(self):
        """Converts CardData to a dict. It also formats the internal eval_results to
        be compatible with the model-index format.

        Returns:
            dict: CardData represented as a dictionary ready to be dumped to a YAML
            block for inclusion in a README.md file.
        """

        data_dict = copy.deepcopy(self.__dict__)
        if self.eval_results is not None:
            data_dict["model-index"] = eval_results_to_model_index(
                self.model_name, self.eval_results
            )
            del data_dict["eval_results"], data_dict["model_name"]

        return _remove_none(data_dict)

    def to_yaml(self):
        """Dumps CardData to a YAML block for inclusion in a README.md file."""
        return yaml.dump(self.to_dict(), sort_keys=False).strip()

    def __repr__(self):
        return self.to_yaml()


def model_index_to_eval_results(model_index: List[Dict[str, Any]]):
    """Takes in a model index and returns a list of `modelcards.EvalResult` objects.

    A detailed spec of the model index can be found here:
    https://github.com/huggingface/hub-docs/blob/main/modelcard.md

    Args:
        model_index (List[Dict[str, Any]]):
        A model index data structure, likely coming from a README.md file on the
        Hugging Face Hub.

    Returns:
        - model_name (str):
            The name of the model as found in the model index. This is used as the
            identifier for the model on leaderboards like PapersWithCode.
        - eval_results (List[EvalResult]):
            A list of `modelcards.EvalResult` objects containing the metrics
            reported in the provided model_index.

    Example:
        >>> from modelcards.card_data import model_index_to_eval_results
        >>> # Define a minimal model index
        >>> model_index = [
        ...     {
        ...         "name": "my-cool-model",
        ...         "results": [
        ...         {
        ...             "task": {
        ...             "type": "image-classification"
        ...             },
        ...             "dataset": {
        ...             "type": "beans",
        ...             "name": "Beans"
        ...             },
        ...             "metrics": [
        ...             {
        ...                 "type": "accuracy",
        ...                 "value": 0.9
        ...             }
        ...             ]
        ...         }
        ...         ]
        ...     }
        ... ]
        >>> model_name, eval_results = model_index_to_eval_results(model_index)
        >>> model_name
        'my-cool-model'
        >>> eval_results[0].task_type
        'image-classification'
        >>> eval_results[0].metric_type
        'accuracy'
    """

    eval_results = []
    for elem in model_index:
        name = elem["name"]
        results = elem["results"]
        for result in results:
            task_type = result["task"]["type"]
            task_name = result["task"].get("name")
            dataset_type = result["dataset"]["type"]
            dataset_name = result["dataset"]["name"]
            dataset_config = result["dataset"].get("config")
            dataset_split = result["dataset"].get("split")
            dataset_revision = result["dataset"].get("revision")
            dataset_args = result["dataset"].get("args")

            for metric in result["metrics"]:
                metric_type = metric["type"]
                metric_value = metric["value"]
                metric_name = metric.get("name")
                metric_args = metric.get("args")
                verified = metric.get("verified")

                eval_result = EvalResult(
                    task_type=task_type,  # Required
                    dataset_type=dataset_type,  # Required
                    dataset_name=dataset_name,  # Required
                    metric_type=metric_type,  # Required
                    metric_value=metric_value,  # Required
                    task_name=task_name,
                    dataset_config=dataset_config,
                    dataset_split=dataset_split,
                    dataset_revision=dataset_revision,
                    dataset_args=dataset_args,
                    metric_name=metric_name,
                    metric_args=metric_args,
                    verified=verified,
                )
                eval_results.append(eval_result)
    return name, eval_results


def _remove_none(obj):
    """
    Recursively remove `None` values from a dict. Borrowed from: https://stackoverflow.com/a/20558778
    """
    if isinstance(obj, (list, tuple, set)):
        return type(obj)(_remove_none(x) for x in obj if x is not None)
    elif isinstance(obj, dict):
        return type(obj)(
            (_remove_none(k), _remove_none(v))
            for k, v in obj.items()
            if k is not None and v is not None
        )
    else:
        return obj


def eval_results_to_model_index(model_name: str, eval_results: List[EvalResult]):
    """Takes in given model name and list of `modelcards.EvalResult` and returns a
    valid model-index that will be compatible with the format expected by the
    Hugging Face Hub.

    Args:
        model_name (str):
            Name of the model (ex. "my-cool-model"). This is used as the identifier
            for the model on leaderboards like PapersWithCode.
        eval_results (List[EvalResult]):
            List of `modelcards.EvalResult` objects containing the metrics to be
            reported in the model-index.

    Returns:
        model_index (List[Dict[str, Any]]): The eval_results converted to a model-index.

    Example:
        >>> from modelcards.card_data import eval_results_to_model_index, EvalResult
        >>> # Define minimal eval_results
        >>> eval_results = [
        ...     EvalResult(
        ...         task_type="image-classification",  # Required
        ...         dataset_type="beans",  # Required
        ...         dataset_name="Beans",  # Required
        ...         metric_type="accuracy",  # Required
        ...         metric_value=0.9,  # Required
        ...     )
        ... ]
        >>> eval_results_to_model_index("my-cool-model", eval_results)
        [{'name': 'my-cool-model', 'results': [{'task': {'type': 'image-classification'}, 'dataset': {'type': 'beans', 'name': 'Beans'}, 'metrics': [{'type': 'accuracy', 'value': 0.9}]}]}]

    """

    # Metrics are reported on a unique task-and-dataset basis.
    # Here, we make a map of those pairs and the associated EvalResults.
    task_and_ds_types_map = dict()
    for eval_result in eval_results:
        task_and_ds_pair = (eval_result.task_type, eval_result.dataset_type)
        if task_and_ds_pair in task_and_ds_types_map:
            task_and_ds_types_map[task_and_ds_pair].append(eval_result)
        else:
            task_and_ds_types_map[task_and_ds_pair] = [eval_result]

    # Use the map from above to generate the model index data.
    model_index_data = []
    for (task_type, dataset_type), results in task_and_ds_types_map.items():
        data = {
            "task": {
                "type": task_type,
                "name": results[0].task_name,
            },
            "dataset": {
                "type": dataset_type,
                "name": results[0].dataset_name,
                "config": results[0].dataset_config,
                "split": results[0].dataset_split,
                "revision": results[0].dataset_revision,
                "args": results[0].dataset_args,
            },
            "metrics": [
                {
                    "name": result.metric_name,
                    "type": result.metric_type,
                    "value": result.metric_value,
                    "args": result.metric_args,
                    "verified": result.verified,
                }
                for result in results
            ],
        }
        model_index_data.append(data)

    # TODO - Check if there cases where this list is longer than one?
    # Finally, the model index itself is list of dicts.
    model_index = [
        {
            "name": model_name,
            "results": model_index_data,
        }
    ]
    return _remove_none(model_index)
