"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? true;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNagApp", version: "0.4.1" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            console.info(`Unable to add Nag Suppression for path: ${path} as it does not exist.`);
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            return `${stack.stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNag", version: "0.4.1" };
//# sourceMappingURL=data:application/json;base64,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