from .context import gpytoolbox as gpy
from .context import numpy as np
from .context import unittest
import numpy as np
import scipy as sp


class TestMassmatrix(unittest.TestCase):
    def test_polyline_integral(self):
        # This is a cube, centered at the origin, with side length 1
        # v,f = gpy.read_mesh("test/unit_tests_data/cube.obj")
        #
        # Let's make up a simple polyline
        v = np.array([[0],[0.2],[0.5],[0.98],[1.0]])
        fun_integral_five = 0*v + 5
        fun_integral_other = v + 5
        # integral is 1^3/2 + 5*1
        M = gpy.massmatrix(v)
        self.assertTrue(np.isclose(np.sum(M @ fun_integral_five),5.))
        self.assertTrue(np.isclose(np.sum(M @ fun_integral_other),5.5))
        

    def test_single_triangle_2d(self):
        v = np.array([[0.0,0.0],[1.0,0.0],[0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)

        M_b = gpy.massmatrix(v,f, type='barycentric')
        M_b_gt = np.array([[0.5,0.0,0.0],[0.0,0.5,0.0],[0.0,0.0,0.5]])/3.
        self.assertTrue(np.isclose(M_b.toarray(), M_b_gt).all())

        M_v = gpy.massmatrix(v,f, type='voronoi')
        M_v_gt = np.array([[0.25,0.0,0.0],[0.0,0.125,0.0],[0.0,0.0,0.125]])
        self.assertTrue(np.isclose(M_v.toarray(), M_v_gt).all())

        M_d = gpy.massmatrix(v,f)
        self.assertTrue(np.isclose(M_d.toarray(), M_v_gt).all())

        M_f = gpy.massmatrix(v,f, type='full')
        M_f_gt = np.array([[2.,1.,1.],[1.,2.,1.],[1.,1.,2.]])/24.
        self.assertTrue(np.isclose(M_f.toarray(), M_f_gt).all())
    
    def test_single_triangle_3d(self):
        v = np.array([[0.0,0.0,0.0],[1.0,0.0,0.0],[0.0,0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)

        M_b = gpy.massmatrix(v,f, type='barycentric')
        M_b_gt = np.array([[0.5,0.0,0.0],[0.0,0.5,0.0],[0.0,0.0,0.5]])/3.
        self.assertTrue(np.isclose(M_b.toarray(), M_b_gt).all())

        M_v = gpy.massmatrix(v,f, type='voronoi')
        M_v_gt = np.array([[0.25,0.0,0.0],[0.0,0.125,0.0],[0.0,0.0,0.125]])
        self.assertTrue(np.isclose(M_v.toarray(), M_v_gt).all())

        M_d = gpy.massmatrix(v,f)
        self.assertTrue(np.isclose(M_d.toarray(), M_v_gt).all())

        M_f = gpy.massmatrix(v,f, type='full')
        M_f_gt = np.array([[2.,1.,1.],[1.,2.,1.],[1.,1.,2.]])/24.
        self.assertTrue(np.isclose(M_f.toarray(), M_f_gt).all())

    def test_bunny_oded(self):
        v,f = gpy.read_mesh("test/unit_tests_data/bunny_oded.obj")

        M_b = gpy.massmatrix(v,f, type='barycentric')
        M_v = gpy.massmatrix(v,f, type='voronoi')
        M_d = gpy.massmatrix(v,f)
        M_f = gpy.massmatrix(v,f, type='full')

        data_b_gt = np.array([0.00153451, 0.0009645 , 0.0018503 , 0.00157772, 0.00144574,
       0.00175107, 0.00183772, 0.00140891, 0.00180358, 0.00187107,
       0.0017068 , 0.00190684, 0.00185875, 0.00139906, 0.00180097,
       0.00161579, 0.00187627, 0.00175416, 0.00190659, 0.00185798,
       0.00162403, 0.00176723, 0.0015973 , 0.00188687, 0.00176269,
       0.00171252, 0.0019662 , 0.00202314, 0.00204631, 0.00143825,
       0.00150313, 0.00181675, 0.00151351, 0.00187736, 0.00182981,
       0.00176393, 0.00188891, 0.00175673, 0.00172703, 0.00172471,
       0.00153756, 0.0018542 , 0.00140207, 0.00182927, 0.00179065,
       0.001722  , 0.00168449, 0.00187487, 0.00198196, 0.00211273,
       0.00206485, 0.0016255 , 0.00158783, 0.00157249, 0.00181474,
       0.00152658, 0.0021413 , 0.00171799, 0.00187285, 0.00174894,
       0.00183469, 0.00187654, 0.0013075 , 0.00177957, 0.00192308,
       0.0019922 , 0.00105021, 0.00184931, 0.00148944, 0.00180057,
       0.00177104, 0.00180268, 0.0016882 , 0.00165846, 0.00164535,
       0.00175372, 0.00197934, 0.00208346, 0.00185563, 0.0018384 ,
       0.00171735, 0.00166872, 0.00164011, 0.00180149, 0.00162818,
       0.00124878, 0.00185947, 0.00165724, 0.00176625, 0.00171786,
       0.00232732, 0.00197812, 0.00178686, 0.00197615, 0.00107264,
       0.00150589, 0.00159068, 0.00183655, 0.00181839, 0.00168803,
       0.0016364 , 0.00177485, 0.0017458 , 0.0017662 , 0.0017844 ,
       0.00170337, 0.00171745, 0.00131696, 0.00128242, 0.00160229,
       0.00201453, 0.00201066, 0.0017908 , 0.0017971 , 0.00182749,
       0.00174264, 0.00170578, 0.00168982, 0.00181907, 0.00177699,
       0.00168118, 0.00157814, 0.00167825, 0.00161839, 0.0016836 ,
       0.00169901, 0.00173183, 0.00198673, 0.00183162, 0.00174913,
       0.00199291, 0.00182136, 0.00185573, 0.00174864, 0.00174964,
       0.001772  , 0.00178924, 0.00177468, 0.00170073, 0.0016093 ,
       0.00173857, 0.0016718 , 0.00170749, 0.00167972, 0.00170998,
       0.00175247, 0.00179799, 0.00118701, 0.00189246, 0.00166873,
       0.00192828, 0.00188908, 0.0017572 , 0.00181125, 0.0017813 ,
       0.00178227, 0.00175344, 0.00173493, 0.0017205 , 0.00173734,
       0.0017457 , 0.00170177, 0.00166316, 0.00161614, 0.00163989,
       0.00162986, 0.00170667, 0.00168411, 0.00175016, 0.00183977,
       0.00178255, 0.0017545 , 0.00175811, 0.00179504, 0.00174217,
       0.00158423, 0.00164647, 0.00171199, 0.0016999 , 0.00176242,
       0.00173168, 0.00177498, 0.00177079, 0.00169148, 0.00167545,
       0.00172333, 0.00158411, 0.00159798, 0.00163433, 0.00165764,
       0.00173667, 0.00183823, 0.00185686, 0.0016238 , 0.00168318,
       0.00170521, 0.00177381, 0.00185636, 0.00165937, 0.00180196,
       0.00179491, 0.00178165, 0.00177118, 0.00176489, 0.00175164,
       0.00172357, 0.00170698, 0.00179512, 0.00170043, 0.00170333,
       0.00164553, 0.0016805 , 0.00166212, 0.00170061, 0.00171107,
       0.00172253, 0.00179479, 0.00169044, 0.00163338, 0.00162488,
       0.00171794, 0.00176741, 0.00169987, 0.00169917, 0.00156155,
       0.00145312, 0.00167256, 0.00173957, 0.00168941, 0.00172201,
       0.0017753 , 0.00169371, 0.00179097, 0.00178685, 0.00169289,
       0.00170883, 0.00170964, 0.00160558, 0.00152725, 0.00144207,
       0.00155159, 0.00171453, 0.0017934 , 0.00191069, 0.00161483,
       0.00160351, 0.00127694, 0.00175721, 0.00176681, 0.00183121,
       0.00152409, 0.00180475, 0.00180137, 0.00179449, 0.00178279,
       0.00177759, 0.00177113, 0.00174215, 0.00169683, 0.00168841,
       0.00182277, 0.0016817 , 0.00170742, 0.00170478, 0.001752  ,
       0.00175251, 0.00179765, 0.0017163 , 0.00170023, 0.00185433,
       0.00181751, 0.00161414, 0.00145485, 0.00133766, 0.00158081,
       0.00172525, 0.00174297, 0.001654  , 0.00166909, 0.00155601,
       0.00163669, 0.00147095, 0.00171444, 0.00176284, 0.00167575,
       0.00158992, 0.0017103 , 0.00181607, 0.00161966, 0.00182054,
       0.00177174, 0.00168728, 0.00169521, 0.00165705, 0.001672  ,
       0.0015761 , 0.00150563, 0.00108014, 0.00148859, 0.00173918,
       0.00185711, 0.00098736, 0.00163549, 0.00166592, 0.00173922,
       0.0017935 , 0.00174444, 0.00154338, 0.00180762, 0.00180089,
       0.00179999, 0.00178983, 0.00177925, 0.00179002, 0.00175164,
       0.00168221, 0.00166586, 0.00164202, 0.00184972, 0.00176969,
       0.00168317, 0.00173036, 0.00175543, 0.00176397, 0.00173952,
       0.00213161, 0.00169951, 0.00164493, 0.00183499, 0.00137604,
       0.00168474, 0.00141675, 0.00159765, 0.00093428, 0.00131384,
       0.00159234, 0.00164471, 0.00167835, 0.0016337 , 0.00158712,
       0.00166627, 0.0017894 , 0.00169895, 0.00167398, 0.00175469,
       0.00178873, 0.00163888, 0.00108132, 0.0014491 , 0.00207445,
       0.0018911 , 0.00160479, 0.0018279 , 0.00173852, 0.00168835,
       0.00165201, 0.00147917, 0.00169102, 0.00167138, 0.00163827,
       0.0014608 , 0.00146407, 0.00177953, 0.00191199, 0.00149876,
       0.00170301, 0.00174024, 0.00162219, 0.00156239, 0.00180479,
       0.00179671, 0.00178646, 0.00180367, 0.0017692 , 0.00174529,
       0.00180511, 0.00168197, 0.00151448, 0.00159627, 0.00158542,
       0.00180428, 0.00181719, 0.0016789 , 0.0017356 , 0.00163325,
       0.00153376, 0.00175412, 0.00176273, 0.00165163, 0.00168283,
       0.00167915, 0.00161382, 0.00144504, 0.00164973, 0.00132267,
       0.00153695, 0.00163473, 0.00150523, 0.00144008, 0.0014384 ,
       0.00169674, 0.00160029, 0.00146981, 0.00161139, 0.00169329,
       0.00138793, 0.00174278, 0.00173419, 0.00177518, 0.00180001,
       0.00159092, 0.00157694, 0.00169445, 0.0019367 , 0.00183861,
       0.00168174, 0.00174601, 0.00168083, 0.00168306, 0.00165825,
       0.00122817, 0.00123808, 0.00173873, 0.0016552 , 0.00176998,
       0.00156921, 0.0014352 , 0.00175275, 0.00177123, 0.00154738,
       0.00166153, 0.00142126, 0.00101552, 0.00145455, 0.00179748,
       0.00179122, 0.00178046, 0.00176726, 0.00179958, 0.00174369,
       0.0017033 , 0.00180122, 0.00155385, 0.0010312 , 0.00192286,
       0.00125388, 0.00199048, 0.00168285, 0.00174913, 0.00180222,
       0.0017088 , 0.00176563, 0.00158513, 0.00154926, 0.00181888,
       0.00180424, 0.00159889, 0.00158992, 0.00170591, 0.00182912,
       0.00162235, 0.00141202, 0.00168645, 0.00154968, 0.00159568,
       0.00158832, 0.0015651 , 0.00112627, 0.00179915, 0.00166535,
       0.00202105, 0.00166982, 0.00153917, 0.00172514, 0.0015982 ,
       0.00133644, 0.00175067, 0.00176875, 0.00178053, 0.0017961 ,
       0.00165607, 0.00186613, 0.00193873, 0.00177447, 0.00184885,
       0.00170188, 0.00224334, 0.00164134, 0.00157018, 0.00166508,
       0.00179441, 0.00178109, 0.00191254, 0.00159452, 0.00166479,
       0.0014927 , 0.00166695, 0.00171071, 0.0015022 , 0.00163958,
       0.00095557, 0.00184297, 0.00177724, 0.00177931, 0.00177612,
       0.00176859, 0.00175214, 0.0017845 , 0.00171864, 0.0016965 ,
       0.00178716, 0.00155338, 0.00177903, 0.00187454, 0.00164628,
       0.00170512, 0.00215899, 0.00167427, 0.00172841, 0.00182237,
       0.00163835, 0.00163596, 0.00192142, 0.00189376, 0.00163975,
       0.00161804, 0.00179216, 0.00182093, 0.00169788, 0.00143272,
       0.00179137, 0.00157174, 0.00148765, 0.00154256, 0.00169878,
       0.00168291, 0.00166358, 0.001711  , 0.00155977, 0.0017335 ,
       0.00167437, 0.00154492, 0.00176121, 0.00178394, 0.00178301,
       0.00179158, 0.00180417, 0.00178307, 0.00189484, 0.00200109,
       0.00162859, 0.0016902 , 0.00179923, 0.00164719, 0.00127105,
       0.00166206, 0.00162243, 0.0017139 , 0.00170887, 0.00181751,
       0.00204363, 0.00180257, 0.0015681 , 0.00157161, 0.00173562,
       0.00180997, 0.00146977, 0.00167298, 0.00150092, 0.00180714,
       0.00174381, 0.00175535, 0.00174199, 0.00176021, 0.00176059,
       0.00174182, 0.00175315, 0.00172176, 0.0017461 , 0.00176903,
       0.00176722, 0.00175642, 0.00184317, 0.00186289, 0.00172885,
       0.00167369, 0.00174814, 0.00133272, 0.0017037 , 0.00176813,
       0.00180834, 0.00192278, 0.00166063, 0.00164017, 0.0018974 ,
       0.00185406, 0.00170436, 0.00165269, 0.0020258 , 0.00169898,
       0.00181273, 0.00145384, 0.00223222, 0.00152279, 0.00154201,
       0.00168573, 0.00164342, 0.00161226, 0.00168293, 0.00171373,
       0.00172925, 0.00163137, 0.00173876, 0.00172306, 0.00164843,
       0.00176063, 0.00178342, 0.00178318, 0.00178885, 0.00176887,
       0.0015879 , 0.00188154, 0.00181453, 0.00141403, 0.0016489 ,
       0.00168025, 0.00182576, 0.00222135, 0.00198972, 0.00156795,
       0.00155611, 0.00160793, 0.00208561, 0.00180957, 0.00189887,
       0.00165438, 0.00164187, 0.00231551, 0.00195116, 0.00145412,
       0.0012271 , 0.00177759, 0.0019291 , 0.00168745, 0.00169801,
       0.00171261, 0.00169851, 0.00173284, 0.00175426, 0.00175411,
       0.00173662, 0.00175781, 0.0017765 , 0.00173554, 0.00173451,
       0.00172497, 0.00157507, 0.00181738, 0.0018651 , 0.00164657,
       0.001726  , 0.00187019, 0.00175997, 0.00170117, 0.00181633,
       0.00197768, 0.00204606, 0.00167214, 0.00166223, 0.00176204,
       0.00162226, 0.00176554, 0.00169075, 0.0013971 , 0.00167167,
       0.00191934, 0.00147965, 0.0015695 , 0.00214108, 0.00121385,
       0.0017162 , 0.00172461, 0.00172763, 0.00171605, 0.0016973 ,
       0.00173458, 0.00171471, 0.00172738, 0.00176937, 0.00176875,
       0.00179684, 0.00181148, 0.00175704, 0.00142519, 0.00177955,
       0.00166573, 0.00109463, 0.00143688, 0.00183902, 0.00188997,
       0.00185769, 0.00201258, 0.00187484, 0.00154911, 0.00103426,
       0.00119514, 0.00157555, 0.00200115, 0.00205991, 0.00199277,
       0.00183873, 0.00187633, 0.00176611, 0.00207772, 0.00196536,
       0.00140213, 0.00193363, 0.00163324, 0.00163377, 0.00163909,
       0.00167028, 0.00166176, 0.00171562, 0.00174928, 0.00176019,
       0.00172794, 0.0017878 , 0.00172274, 0.00169959, 0.00174208,
       0.00167965, 0.00109824, 0.00170592, 0.00190415, 0.00165781,
       0.00176117, 0.0016928 , 0.00161145, 0.00172379, 0.00173719,
       0.00170657, 0.00186777, 0.00232402, 0.00196126, 0.00167924,
       0.00168348, 0.00161978, 0.0016128 , 0.00172667, 0.00180053,
       0.00148815, 0.00153583, 0.00170815, 0.00188684, 0.0019818 ,
       0.00159614, 0.00155876, 0.00175258, 0.00164739, 0.00173549,
       0.00172331, 0.00173464, 0.00170364, 0.00169381, 0.00171224,
       0.00174167, 0.0016952 , 0.00168791, 0.00179169, 0.00170766,
       0.00182151, 0.0018025 , 0.00175617, 0.00143583, 0.00178975,
       0.00199138, 0.00202953, 0.00183022, 0.00186643, 0.00221783,
       0.00119204, 0.00185125, 0.0018689 , 0.00156004, 0.00167515,
       0.00195352, 0.00212845, 0.00208381, 0.00198921, 0.00190987,
       0.0020095 , 0.00199012, 0.00183781, 0.00192242, 0.00167835,
       0.00159338, 0.00156701, 0.00157496, 0.00157585, 0.00161911,
       0.00159043, 0.00168616, 0.00174262, 0.00171263, 0.00175803,
       0.00174383, 0.00172186, 0.00169234, 0.00174758, 0.00170134,
       0.00158682, 0.00191826, 0.00165651, 0.00179953, 0.00172408,
       0.0017512 , 0.00170179, 0.00171009, 0.0017032 , 0.00169296,
       0.0016965 , 0.00167878, 0.00168893, 0.0016628 , 0.00169212,
       0.0017077 , 0.00169126, 0.00171886, 0.00174531, 0.00179745,
       0.00162427, 0.00175827, 0.00186633, 0.00173866, 0.00177394,
       0.00172173, 0.00187043, 0.00172817, 0.00174588, 0.00173461,
       0.00168869, 0.00162842, 0.00171244, 0.00171187, 0.00191506,
       0.00163856, 0.0016498 , 0.00184546, 0.00169844, 0.00179295,
       0.00172779, 0.00174301, 0.00161331, 0.00182729, 0.00187095,
       0.0014359 , 0.00181341, 0.00183093, 0.00163932, 0.00184453,
       0.00190888, 0.00154875, 0.00126052, 0.00172886, 0.00191209,
       0.00243694, 0.00217271, 0.00188123, 0.00180359, 0.00222848,
       0.00179656, 0.00191178, 0.00158028, 0.00178811, 0.00156174,
       0.00162531, 0.00122685, 0.00148387, 0.00146184, 0.00155816,
       0.00165163, 0.00167249, 0.00170241, 0.00170262, 0.00175172,
       0.00175751, 0.00174693, 0.00175798, 0.00172945, 0.00171415,
       0.00172684, 0.00185697, 0.00197468, 0.00179668, 0.0016551 ,
       0.00171525, 0.00169263, 0.00172159, 0.00177148, 0.00161284,
       0.00170388, 0.00169919, 0.00170158, 0.00166159, 0.00169968,
       0.00158616, 0.00156838, 0.00165853, 0.00174686, 0.00170283,
       0.00170989, 0.00178299, 0.00166411, 0.00172263, 0.0017447 ,
       0.00176203, 0.00181491, 0.00172332, 0.00168692, 0.00161264,
       0.00186691, 0.00173171, 0.00168124, 0.0017652 , 0.00187865,
       0.0016064 , 0.00127593, 0.00224497, 0.00156286, 0.00193582,
       0.0015223 , 0.00168378, 0.0017913 , 0.00172817, 0.00166224,
       0.00162946, 0.00168365, 0.00121426, 0.00195491, 0.00185067,
       0.0017921 , 0.00183803, 0.00182063, 0.00189062, 0.0018459 ,
       0.00177067, 0.00163877, 0.00183815, 0.00297007, 0.00157949,
       0.00181334, 0.00193369, 0.00206619, 0.00177195, 0.00121344,
       0.00183703, 0.00160284, 0.00171875, 0.0014629 , 0.00172658,
       0.00218012, 0.00124049, 0.00132219, 0.00155452, 0.00186239,
       0.00170994, 0.00167027, 0.00179069, 0.00176108, 0.00177673,
       0.0017837 , 0.00176864, 0.00171223, 0.00170475, 0.00210273,
       0.00189222, 0.00181823, 0.00166092, 0.00167518, 0.00165427,
       0.00166035, 0.00163006, 0.00174298, 0.00178498, 0.00159287,
       0.00170829, 0.00170511, 0.00169753, 0.00170405, 0.00163721,
       0.00168765, 0.00159076, 0.00168159, 0.001817  , 0.00182277,
       0.00170968, 0.0018063 , 0.00169245, 0.00177521, 0.00166282,
       0.00174414, 0.00168247, 0.00184464, 0.00116788, 0.00190581,
       0.00183106, 0.00174754, 0.00169088, 0.00162833, 0.00188608,
       0.00214232, 0.00164351, 0.00188921, 0.00119056, 0.0017918 ,
       0.00166428, 0.00172254, 0.0017406 , 0.00127687, 0.00167616,
       0.00160469, 0.00185406, 0.00180762, 0.001749  , 0.00176866,
       0.00173361, 0.00173887, 0.00170496, 0.00163453, 0.00177713,
       0.00175533, 0.00164799, 0.00227869, 0.00190547, 0.00221875,
       0.00165407, 0.00167043, 0.00187425, 0.00184286, 0.00182594,
       0.00171935, 0.00181824, 0.00156784, 0.0012079 , 0.00183753,
       0.00192442, 0.00082375, 0.00132097, 0.0017759 , 0.00169702,
       0.00166257, 0.00179552, 0.00179683, 0.00175772, 0.00179134,
       0.00178473, 0.00173992, 0.00169069, 0.00168084, 0.00167776,
       0.00178325, 0.00158346, 0.00164749, 0.00167031, 0.00167166,
       0.00161787, 0.00168202, 0.00163343, 0.00183438, 0.00178003,
       0.00156597, 0.00171818, 0.00172785, 0.00171878, 0.00169639,
       0.0017216 , 0.00160737, 0.00170706, 0.00170094, 0.00189553,
       0.00185385, 0.00166528, 0.00181657, 0.00193557, 0.00197576,
       0.00168062, 0.00174584, 0.00159704, 0.00216939, 0.00161765,
       0.00207318, 0.00179109, 0.00169042, 0.00162668, 0.00161083,
       0.00188994, 0.0017441 , 0.00177846, 0.00179647, 0.00190285,
       0.00172266, 0.00224107, 0.0017986 , 0.00152953, 0.0016662 ,
       0.0016913 , 0.00153181, 0.00163005, 0.00164174, 0.00147918,
       0.00165857, 0.00165211, 0.00171712, 0.00176473, 0.0016294 ,
       0.00164992, 0.00166829, 0.00152115, 0.0017655 , 0.00174124,
       0.00173409, 0.00181196, 0.0016712 , 0.0018455 , 0.00166402,
       0.00149453, 0.00169033, 0.00122052, 0.00182092, 0.00181128,
       0.00172684, 0.00166692, 0.00166035, 0.00180716, 0.00179743,
       0.00180222, 0.00175564, 0.00179706, 0.00175657, 0.00169048,
       0.00168459, 0.0016145 , 0.00155173, 0.00172791, 0.00140092,
       0.00165156, 0.00164021, 0.00175457, 0.00163268, 0.00126045,
       0.0022205 , 0.0016837 , 0.00186497, 0.00177822, 0.00153771,
       0.00174304, 0.00176628, 0.00180291, 0.00175299, 0.00168017,
       0.00171959, 0.00168821, 0.00172833, 0.0019321 , 0.00185998,
       0.00162861, 0.00179128, 0.00200869, 0.00233427, 0.00134322,
       0.0017437 , 0.00169628, 0.00124829, 0.00160879, 0.00154377,
       0.00122663, 0.00173372, 0.00175121, 0.00169141, 0.00160714,
       0.00157696, 0.00176911, 0.00173777, 0.00180494, 0.0017548 ,
       0.00217659, 0.00187376, 0.00179692, 0.00165862, 0.0015545 ,
       0.00165238, 0.00153735, 0.00167184, 0.00161441, 0.00173176,
       0.00114548, 0.00170948, 0.00167963, 0.00176239, 0.00183203,
       0.00165611, 0.00147519, 0.00165758, 0.00170006, 0.00168958,
       0.00166161, 0.00170008, 0.00162923, 0.00155008, 0.00173047,
       0.00123235, 0.00165389, 0.00175669, 0.00179827, 0.00175863,
       0.00167251, 0.00166184, 0.00172099, 0.00181934, 0.00181168,
       0.00179808, 0.00180362, 0.00176806, 0.00177476, 0.0017113 ,
       0.00159379, 0.00171096, 0.00123003, 0.00160892, 0.00170706,
       0.00201497, 0.00162978, 0.00208082, 0.00183922, 0.00174813,
       0.00170338, 0.00197122, 0.0017084 , 0.00182569, 0.00175465,
       0.00154853, 0.00175971, 0.0017889 , 0.00188663, 0.00175905,
       0.00172668, 0.00163541, 0.00171559, 0.0019009 , 0.0017538 ,
       0.00187101, 0.00163383, 0.00189457, 0.00157694, 0.00182144,
       0.00172497, 0.0017956 , 0.00154861, 0.00171542, 0.00166075,
       0.00213429, 0.00115327, 0.0019758 , 0.00166132, 0.00168691,
       0.00158325, 0.00148109, 0.00170262, 0.00187797, 0.00172947,
       0.00167165, 0.00155098, 0.00149705, 0.00148071, 0.00165582,
       0.00165148, 0.00162881, 0.00185599, 0.00173327, 0.00179242,
       0.00173744, 0.00170069, 0.0017647 , 0.00195666, 0.00194029,
       0.00182121, 0.00193685, 0.001753  , 0.00178974, 0.00169432,
       0.00164127, 0.00227673, 0.00161739, 0.0015955 , 0.00170585,
       0.00182953, 0.00175581, 0.00178574, 0.0017785 , 0.00168875,
       0.00154711, 0.00164565, 0.00187868, 0.00181921, 0.00181767,
       0.00181269, 0.00181409, 0.00178407, 0.00179975, 0.00173863,
       0.00167404, 0.00161533, 0.00170066, 0.00167653, 0.00186218,
       0.00165147, 0.00179958, 0.00180231, 0.0018581 , 0.00174299,
       0.00182708, 0.00183588, 0.00171874, 0.00174954, 0.00171429,
       0.00161654, 0.00176593, 0.00178886, 0.00187751, 0.0017707 ,
       0.00159467, 0.00178537, 0.00187371, 0.00181815, 0.00190525,
       0.00190604, 0.00167112, 0.00239334, 0.00180454, 0.00183668,
       0.00164841, 0.00183096, 0.00180991, 0.00158104, 0.00122505,
       0.00158866, 0.00164386, 0.00167924, 0.00147498, 0.00169081,
       0.00170444, 0.0016981 , 0.00164683, 0.00145618, 0.00145027,
       0.00105243, 0.00152793, 0.00172668, 0.00183074, 0.00161334,
       0.00176806, 0.00184569, 0.00177938, 0.00181711, 0.00178764,
       0.00175373, 0.00168122, 0.00177703, 0.00197485, 0.00208952,
       0.00204816, 0.00212105, 0.00184992, 0.00183002, 0.00169369,
       0.00165931, 0.00191702, 0.00166667, 0.00170648, 0.00156202,
       0.00168474, 0.00161108, 0.00180286, 0.00179449, 0.00175281,
       0.00159243, 0.00114082, 0.00208508, 0.00187402, 0.00183959,
       0.00181865, 0.00181004, 0.00179664, 0.00183805, 0.0017499 ,
       0.00171454, 0.00169054, 0.00149463, 0.00159649, 0.00199217,
       0.0017404 , 0.00171234, 0.00167546, 0.00181398, 0.00177358,
       0.00186081, 0.00176471, 0.00171693, 0.00178266, 0.00176882,
       0.00167959, 0.00170498, 0.00179801, 0.00183209, 0.00184048,
       0.00176037, 0.00158194, 0.0018782 , 0.00180524, 0.00195327,
       0.00180086, 0.001728  , 0.00185513, 0.00187388, 0.0017238 ,
       0.00184408, 0.00182515, 0.00184656, 0.00232291, 0.00179265,
       0.0016272 , 0.00144077, 0.00158335, 0.00149444, 0.00164027,
       0.00165855, 0.00164531, 0.00166296, 0.00167384, 0.00178729,
       0.00164224, 0.00151247, 0.00118098, 0.00162346, 0.00173584,
       0.00171776, 0.00142042, 0.00180917, 0.00179048, 0.00183261,
       0.00187601, 0.00181933, 0.00178681, 0.00161512, 0.0017769 ,
       0.00157866, 0.00204422, 0.00208178, 0.00205729, 0.00183922,
       0.00169656, 0.00168241, 0.00167442, 0.00190559, 0.00183456,
       0.00172534, 0.00172896, 0.00153284, 0.00168923, 0.00162545,
       0.00211359, 0.00182074, 0.00180602, 0.00176226, 0.00164399,
       0.00186051, 0.0017905 , 0.00183901, 0.0018276 , 0.00180332,
       0.001791  , 0.00176303, 0.00172544, 0.00171308, 0.0010813 ,
       0.00154239, 0.00175311, 0.00176234, 0.00166556, 0.00165897,
       0.00164531, 0.00182086, 0.00178487, 0.00188749, 0.00174368,
       0.00168329, 0.0016312 , 0.00178417, 0.00186859, 0.00186997,
       0.00186787, 0.0018404 , 0.00178166, 0.00176222, 0.00159875,
       0.00181495, 0.00170449, 0.00177852, 0.00186694, 0.00169406,
       0.00167418, 0.00158201, 0.00159397, 0.00192996, 0.00192479,
       0.00180269, 0.00185163, 0.00176896, 0.00163717, 0.00138799,
       0.00161357, 0.00158031, 0.00164824, 0.00161981, 0.00156196,
       0.0016664 , 0.00164402, 0.00163394, 0.00185717, 0.0016146 ,
       0.00165917, 0.00185558, 0.0017171 , 0.00169648, 0.00151684,
       0.00180259, 0.00163585, 0.00174723, 0.00184205, 0.00183892,
       0.00180405, 0.00176542, 0.00162557, 0.00179656, 0.00109988,
       0.00235694, 0.00244575, 0.00185077, 0.00180271, 0.0014033 ,
       0.0019481 , 0.00170187, 0.00173408, 0.00172266, 0.0024035 ,
       0.00186204, 0.00173423, 0.00175472, 0.00176351, 0.00154632,
       0.00108913, 0.00154179, 0.0018423 , 0.00174165, 0.00182497,
       0.00182901, 0.00181408, 0.00179588, 0.00182101, 0.0017459 ,
       0.00181961, 0.00180359, 0.00178072, 0.00175661, 0.00172962,
       0.00171091, 0.00160508, 0.00168141, 0.00170512, 0.00175135,
       0.00166197, 0.00165778, 0.00167244, 0.00170737, 0.00182467,
       0.00177851, 0.00195972, 0.00167044, 0.00173738, 0.00159182,
       0.00178079, 0.00228554, 0.00187492, 0.00181304, 0.00178311,
       0.00165845, 0.00175547, 0.00156896, 0.00171286, 0.00171865,
       0.00097084, 0.00210936, 0.00125407, 0.00171659, 0.00199765,
       0.00165164, 0.00197373, 0.0017806 , 0.00186283, 0.00180031,
       0.00185825, 0.00142728, 0.00170028, 0.0014679 , 0.00167011,
       0.00166324, 0.00160479, 0.00140066, 0.00169648, 0.00164571,
       0.00165983, 0.00164824, 0.00182107, 0.00169111, 0.00183299,
       0.0016978 , 0.00168124, 0.00163135, 0.00182612, 0.00177136,
       0.00181876, 0.00182877, 0.00180942, 0.00177086, 0.00173743,
       0.00176648, 0.00169398, 0.00191854, 0.00199194, 0.00119158,
       0.00178539, 0.00173109, 0.00180473, 0.0018017 , 0.00174661,
       0.00172751, 0.00195036, 0.00180862, 0.00171477, 0.00179939,
       0.00172205, 0.00177555, 0.00170671, 0.00142066, 0.00170723,
       0.00175133, 0.00182238, 0.00183897, 0.00177749, 0.00176953,
       0.00178769, 0.00173328, 0.00179361, 0.00177407, 0.00174797,
       0.00171996, 0.00169586, 0.00168881, 0.00155391, 0.00164087,
       0.00175159, 0.00170491, 0.00163914, 0.00168253, 0.00170558,
       0.00161639, 0.00212509, 0.00183403, 0.00175448, 0.00203682,
       0.0016709 , 0.00158304, 0.00174311, 0.00174355, 0.00130531,
       0.00184996, 0.00189433, 0.00188385, 0.00156735, 0.00144937,
       0.0017316 , 0.00159529, 0.00182056, 0.0017049 , 0.00169322,
       0.00180517, 0.00186495, 0.00192127, 0.00196389, 0.00188318,
       0.00170767, 0.00173014, 0.00182566, 0.00187898, 0.00185409,
       0.00182608, 0.00174402, 0.00142693, 0.00167767, 0.00169732,
       0.00173184, 0.00150389, 0.00093561, 0.00165658, 0.00158992,
       0.00166826, 0.00170931, 0.00178067, 0.00170012, 0.0018118 ,
       0.00168071, 0.00163164, 0.00180389, 0.00180802, 0.00181884,
       0.00184287, 0.00184183, 0.00178081, 0.00187872, 0.0021176 ,
       0.00172263, 0.00201618, 0.00172568, 0.00184724, 0.00180232,
       0.00187318, 0.00194789, 0.00203046, 0.00174019, 0.00171066,
       0.00176162, 0.00171754, 0.00165107, 0.00183913, 0.00182065,
       0.00166652, 0.00162321, 0.00156101, 0.00198934, 0.00161951,
       0.00176193, 0.00141353, 0.00168306, 0.00172059, 0.00180527,
       0.0017324 , 0.00176419, 0.00171902, 0.0017424 , 0.0017169 ,
       0.00176634, 0.00174129, 0.00170852, 0.00167362, 0.0017536 ,
       0.00120349, 0.00164975, 0.00190651, 0.00163477, 0.00159823,
       0.00167889, 0.00185395, 0.00179011, 0.00124653, 0.00188876,
       0.00175931, 0.00179639, 0.00180659, 0.00162778, 0.00170748,
       0.00173664, 0.00170751, 0.00161765, 0.00181095, 0.00175155,
       0.00200066, 0.00201898, 0.00121891, 0.0016036 , 0.00120343,
       0.00181085, 0.00208382, 0.00170898, 0.0018356 , 0.00181354,
       0.00194437, 0.00178196, 0.00177161, 0.00175986, 0.0018434 ,
       0.0016524 , 0.00167771, 0.00172291, 0.00163216, 0.00154921,
       0.00163331, 0.00169157, 0.00171724, 0.00167356, 0.00176513,
       0.00167459, 0.00159532, 0.00181738, 0.0018717 , 0.00183085,
       0.00184674, 0.00184542, 0.00175843, 0.00172307, 0.00176606,
       0.00168627, 0.00171976, 0.00160278, 0.00200828, 0.00162467,
       0.00192208, 0.00184381, 0.00195875, 0.00150351, 0.00199641,
       0.00173196, 0.00164592, 0.00147665, 0.00180495, 0.00183265,
       0.00179996, 0.00161858, 0.00156169, 0.00155742, 0.00087797,
       0.00176934, 0.00139461, 0.00163354, 0.00165417, 0.00168561,
       0.00168859, 0.00166684, 0.0017145 , 0.00167967, 0.00170987,
       0.00169031, 0.00172924, 0.00169365, 0.00164819, 0.00211102,
       0.00211955, 0.00207531, 0.00159273, 0.00166614, 0.00160512,
       0.00185396, 0.00233243, 0.00170692, 0.00165552, 0.00169491,
       0.0017427 , 0.00171868, 0.00166632, 0.00168089, 0.00172002,
       0.00169968, 0.0016938 , 0.00164295, 0.00174394, 0.00159403,
       0.00164313, 0.00171803, 0.00171346, 0.00145021, 0.00195351,
       0.00173448, 0.00178492, 0.00172951, 0.00190543, 0.00158584,
       0.00177252, 0.00172501, 0.00163898, 0.00155391, 0.00136303,
       0.00143723, 0.00160959, 0.0015732 , 0.00166523, 0.00167344,
       0.00162796, 0.00167516, 0.00167954, 0.00159257, 0.001838  ,
       0.00194613, 0.00181802, 0.00182435, 0.00178945, 0.00171967,
       0.00165166, 0.00171303, 0.00159569, 0.00165519, 0.00170917,
       0.00161129, 0.00176441, 0.00169887, 0.00180099, 0.00192025,
       0.00171157, 0.00174744, 0.00163125, 0.00188781, 0.00195403,
       0.00172092, 0.00156104, 0.00106298, 0.00183289, 0.00178132,
       0.00182655, 0.00180655, 0.00163395, 0.0015353 , 0.0016743 ,
       0.001721  , 0.00145299, 0.00162246, 0.00161354, 0.0016594 ,
       0.00165892, 0.00163387, 0.00163902, 0.00167669, 0.00165744,
       0.00167862, 0.0016717 , 0.00163464, 0.00189915, 0.00192103,
       0.0020098 , 0.00167674, 0.0016589 , 0.00169981, 0.00162677,
       0.00177431, 0.00169578, 0.00170164, 0.00172447, 0.00170965,
       0.00169919, 0.00167006, 0.00166663, 0.00168598, 0.00167386,
       0.00165972, 0.00167634, 0.00163204, 0.00159906, 0.00157004,
       0.00156013, 0.00165295, 0.00192072, 0.00173759, 0.00198835,
       0.00179301, 0.00120498, 0.00191985, 0.00113386, 0.0017963 ,
       0.00171303, 0.00133677, 0.00094976, 0.00144992, 0.00165076,
       0.00176178, 0.00152166, 0.00169753, 0.00159089, 0.00169283,
       0.00167052, 0.00160426, 0.00181406, 0.00163208, 0.00179212,
       0.00179141, 0.00175621, 0.00173122, 0.00172264, 0.00165848,
       0.00175768, 0.00167803, 0.00162275, 0.00169645, 0.001674  ,
       0.0017117 , 0.00162945, 0.00173215, 0.00213905, 0.00153677,
       0.00189615, 0.00160278, 0.00188212, 0.00191036, 0.00174851,
       0.0016109 , 0.00181509, 0.00184204, 0.00176117, 0.00182306,
       0.00180306, 0.00164986, 0.00162583, 0.0016869 , 0.00192775,
       0.00163541, 0.00159098, 0.00164094, 0.00164747, 0.00162293,
       0.00161345, 0.00161089, 0.0016627 , 0.00162061, 0.00166896,
       0.00192115, 0.00170088, 0.00110656, 0.00168614, 0.00162704,
       0.00166326, 0.00155766, 0.00163412, 0.00155147, 0.00162495,
       0.00177652, 0.00172741, 0.00178988, 0.00174589, 0.00165057,
       0.00164716, 0.00167042, 0.00166635, 0.00161527, 0.00156108,
       0.00166672, 0.00128714, 0.00160608, 0.00159192, 0.00173995,
       0.00162408, 0.00184696, 0.00208125, 0.00181232, 0.00225344,
       0.00180045, 0.00172889, 0.00140891, 0.00160889, 0.00179333,
       0.00162111, 0.00173161, 0.0015123 , 0.00157866, 0.0016812 ,
       0.00167232, 0.00160424, 0.00172511, 0.00163981, 0.00154897,
       0.00174435, 0.0017522 , 0.00174667, 0.0017233 , 0.0016019 ,
       0.00169751, 0.00145599, 0.00135186, 0.00179525, 0.00168118,
       0.00179032, 0.00162534, 0.00211256, 0.00173088, 0.0019356 ,
       0.00205576, 0.00189522, 0.00190125, 0.00185575, 0.00181205,
       0.00177756, 0.00182844, 0.00187417, 0.0017965 , 0.00176241,
       0.00161657, 0.00163504, 0.00111867, 0.00169181, 0.00162954,
       0.00163504, 0.00164473, 0.00165965, 0.00163352, 0.00161914,
       0.00167919, 0.00170588, 0.00181268, 0.00185967, 0.00188968,
       0.00163406, 0.00163794, 0.00153629, 0.00167974, 0.00151654,
       0.00165671, 0.00157635, 0.00164835, 0.00174428, 0.00183906,
       0.00167362, 0.00183452, 0.00173907, 0.00159762, 0.00161777,
       0.001778  , 0.00166495, 0.00120193, 0.00161425, 0.00168329,
       0.00160277, 0.0017089 , 0.00169894, 0.00182681, 0.00124794,
       0.00178337, 0.00177024, 0.00152312, 0.00185942, 0.00167218,
       0.00175792, 0.001729  , 0.00162958, 0.00169113, 0.00157443,
       0.00167468, 0.00171134, 0.00117624, 0.00167061, 0.00179454,
       0.0016509 , 0.00172217, 0.00174575, 0.00174122, 0.00170178,
       0.00159671, 0.00167114, 0.0016541 , 0.00166299, 0.00165703,
       0.00178325, 0.00186254, 0.00156777, 0.00220617, 0.00157493,
       0.00179833, 0.0017416 , 0.00171704, 0.00179861, 0.00218856,
       0.00175271, 0.00172873, 0.00164319, 0.0018216 , 0.00147318,
       0.00169152, 0.00156093, 0.00170857, 0.00172297, 0.0017    ,
       0.00169795, 0.00171987, 0.00174346, 0.00194787, 0.00207486,
       0.00182138, 0.00165711, 0.00132377, 0.00164163, 0.00173959,
       0.00160049, 0.00153058, 0.00175771, 0.00154952, 0.00155323,
       0.00163066, 0.001827  , 0.00170395, 0.00202979, 0.0017778 ,
       0.00178158, 0.00166942, 0.00154451, 0.0017462 , 0.00188283,
       0.00202457, 0.00185208, 0.00165498, 0.00157791, 0.00164534,
       0.00181444, 0.0016628 , 0.00167464, 0.0018037 , 0.00174605,
       0.00171226, 0.00179091, 0.00199505, 0.0011326 , 0.00169185,
       0.00173269, 0.0017818 , 0.00171607, 0.00194271, 0.0017191 ,
       0.00173535, 0.00171343, 0.00167628, 0.00157979, 0.00166581,
       0.00259484, 0.00163003, 0.00174617, 0.00180378, 0.00152462,
       0.00186582, 0.0016667 , 0.00178869, 0.00174466, 0.00163708,
       0.00126458, 0.00175808, 0.00170312, 0.00175137, 0.00172732,
       0.00177649, 0.00208926, 0.00165447, 0.00157692, 0.00179485,
       0.00185022, 0.00185671, 0.00176864, 0.00176409, 0.00176769,
       0.00236184, 0.00178543, 0.0016655 , 0.00145192, 0.00164606,
       0.00103542, 0.00166766, 0.0018114 , 0.00197502, 0.00157081,
       0.00158173, 0.0015537 , 0.00157529, 0.0016462 , 0.00163766,
       0.00175297, 0.00168005, 0.00152154, 0.00175757, 0.00160201,
       0.00165569, 0.00171189, 0.00188118, 0.00181348, 0.00173169,
       0.00171456, 0.00159891, 0.00164408, 0.0016801 , 0.00175116,
       0.00165382, 0.00122574, 0.00175917, 0.00171891, 0.00177926,
       0.00153195, 0.00171237, 0.00174259, 0.00156365, 0.00168229,
       0.00152386, 0.00165729, 0.00168305, 0.00165179, 0.00165066,
       0.00160526, 0.00172473, 0.00110332, 0.00148055, 0.00171847,
       0.00174954, 0.00167265, 0.00177095, 0.00177278, 0.00177188,
       0.001719  , 0.00163494, 0.00176821, 0.00165216, 0.0015926 ,
       0.00176461, 0.00170956, 0.00180538, 0.00162819, 0.0017086 ,
       0.0018983 , 0.00183901, 0.0020065 , 0.00179836, 0.00178569,
       0.00177806, 0.00210741, 0.0010032 , 0.00179292, 0.00124126,
       0.00164801, 0.00170308, 0.00157756, 0.00175502, 0.001269  ,
       0.00168635, 0.0017269 , 0.00238626, 0.00158868, 0.00159423,
       0.00159092, 0.00162736, 0.00167343, 0.00153541, 0.00170637,
       0.00168742, 0.00166969, 0.00155809, 0.00147753, 0.0017145 ,
       0.00161549, 0.0015157 , 0.00183731, 0.00179373, 0.00177932,
       0.00166807, 0.0016563 , 0.0016507 , 0.0017152 , 0.00178252,
       0.00172034, 0.00170869, 0.00171833, 0.00172011, 0.00170627,
       0.00172304, 0.00176429, 0.00150197, 0.00174717, 0.00148242,
       0.0016169 , 0.00163638, 0.00159239, 0.00155792, 0.00163114,
       0.00164344, 0.00184953, 0.00179016, 0.00185454, 0.00177215,
       0.00180098, 0.00178625, 0.00177131, 0.00176696, 0.00179551,
       0.00169904, 0.00174695, 0.00167397, 0.00174787, 0.00152253,
       0.00162931, 0.00167925, 0.00188294, 0.00194771, 0.00184043,
       0.0018048 , 0.00179098, 0.00178983, 0.00179556, 0.0013114 ,
       0.0016402 , 0.00174375, 0.00175026, 0.00178468, 0.00181566,
       0.0013247 , 0.0017202 , 0.00156389, 0.0016889 , 0.0015457 ,
       0.00185792, 0.00164724, 0.00204583, 0.00159447, 0.00159435,
       0.00160208, 0.0016663 , 0.00169148, 0.00175368, 0.00167418,
       0.00164476, 0.00181287, 0.00156484, 0.00104941, 0.00177609,
       0.00175584, 0.00152535, 0.00166819, 0.00174918, 0.00180756,
       0.00148447, 0.0017259 , 0.0017503 , 0.00172764, 0.00174744,
       0.00178152, 0.00158843, 0.00185057, 0.00149947, 0.00163729,
       0.00176294, 0.00193319, 0.00113652, 0.00150211, 0.00167821,
       0.00179493, 0.00156224, 0.00160742, 0.00178534, 0.00151899,
       0.00100313, 0.00179967, 0.00183808, 0.00182796, 0.00179266,
       0.0017966 , 0.00179447, 0.00173321, 0.00175111, 0.00177341,
       0.00169761, 0.00162224, 0.00186343, 0.0018658 , 0.00183347,
       0.00181738, 0.00179642, 0.00180431, 0.00176078, 0.00145476,
       0.00187046, 0.00177057, 0.00167815, 0.00182551, 0.00147011,
       0.00176986, 0.00153393, 0.0015249 , 0.00157006, 0.00153769,
       0.00154386, 0.00168256, 0.00159657, 0.00143805, 0.0017068 ,
       0.00161142, 0.00161165, 0.00164649, 0.00169844, 0.00181654,
       0.00173457, 0.0017503 , 0.00187417, 0.00159669, 0.00176744,
       0.00166555, 0.00149042, 0.00180583, 0.00185799, 0.00179253,
       0.00179309, 0.00186454, 0.00175953, 0.00177377, 0.0016615 ,
       0.00186682, 0.00161208, 0.00160794, 0.00169245, 0.00180061,
       0.00151239, 0.00162836, 0.00172259, 0.0017111 , 0.00141397,
       0.00161899, 0.00142856, 0.00143296, 0.00161662, 0.00189847,
       0.00168797, 0.00176703, 0.00181545, 0.00182099, 0.00180387,
       0.00179261, 0.00182233, 0.00178888, 0.0018005 , 0.0017884 ,
       0.00171091, 0.00163843, 0.00183129, 0.00182221, 0.0018384 ,
       0.00181144, 0.00179825, 0.00181502, 0.00151297, 0.00188957,
       0.00111542, 0.00152668, 0.00177361, 0.00164474, 0.00173391,
       0.00176534, 0.00155363, 0.00155083, 0.00164479, 0.00159268,
       0.00182645, 0.00140187, 0.00162348, 0.00161914, 0.00155969,
       0.00157779, 0.00141165, 0.001745  , 0.00170727, 0.00165609,
       0.00159876, 0.00156005, 0.00166955, 0.00184029, 0.00177869,
       0.00178319, 0.00190776, 0.00171582, 0.00215731, 0.00166049,
       0.00179965, 0.00187997, 0.00171015, 0.00177646, 0.00153214,
       0.00179717, 0.00167271, 0.00172604, 0.00178882, 0.00173477,
       0.00092858, 0.00162627, 0.00167379, 0.00171013, 0.00169657,
       0.00139159, 0.00149268, 0.00164106, 0.00137806, 0.00153774,
       0.00166083, 0.00174392, 0.00169871, 0.00177401, 0.0017862 ,
       0.00180135, 0.00192261, 0.00205151, 0.00181764, 0.00184829,
       0.00177903, 0.00172512, 0.00183949, 0.00178784, 0.00183243,
       0.001788  , 0.00180162, 0.00182586, 0.00160116, 0.00186755,
       0.0015818 , 0.00174706, 0.00183044, 0.00151296, 0.00156742,
       0.00173134, 0.00158476, 0.00154357, 0.00144734, 0.00184578,
       0.00162159, 0.00145493, 0.00153515, 0.00139649, 0.00168735,
       0.00160981, 0.00168981, 0.00161631, 0.00170831, 0.00193385,
       0.00175655, 0.0017858 , 0.00170617, 0.00164537, 0.00145838,
       0.00145774, 0.00170844, 0.00186255, 0.00179255, 0.00125377,
       0.00162415, 0.001823  , 0.0016875 , 0.00171344, 0.00181773,
       0.00161496, 0.00182153, 0.00171906, 0.00177385, 0.00184348,
       0.00189055, 0.00174938, 0.00166851, 0.00167006, 0.00169164,
       0.00115705, 0.00170655, 0.00143866, 0.00148635, 0.00187375,
       0.00102206, 0.00145244, 0.00160485, 0.00170241, 0.00176557,
       0.00176106, 0.00176295, 0.00181055, 0.00207491, 0.00189133,
       0.00243826, 0.001415  , 0.0018078 , 0.00192961, 0.00183496,
       0.00189036, 0.00175019, 0.001799  , 0.00184476, 0.00185132,
       0.0018588 , 0.00175344, 0.00181391, 0.00189835, 0.00165701,
       0.00172005, 0.00162313, 0.00157094, 0.00137374, 0.00096937,
       0.00204368, 0.00166609, 0.00158029, 0.00153372, 0.00149851,
       0.00162645, 0.00148105, 0.0017251 , 0.00171648, 0.00185089,
       0.00182719, 0.00162549, 0.00186146, 0.00190986, 0.0018345 ,
       0.00178729, 0.00174609, 0.00179129, 0.00167697, 0.00151228,
       0.00109287, 0.00152744, 0.00175642, 0.00163981, 0.00176641,
       0.0018281 , 0.00178591, 0.0018783 , 0.00202453, 0.00186163,
       0.00175628, 0.00178238, 0.00182975, 0.00185412, 0.0018479 ,
       0.00180384, 0.0016996 , 0.00165593, 0.00166874, 0.00188192,
       0.00177013, 0.00156206, 0.00163203, 0.0018408 , 0.001452  ,
       0.00160688, 0.00168707, 0.00174767, 0.00176874, 0.00176928,
       0.00190565, 0.00203628, 0.00185159, 0.00181555, 0.00193554,
       0.00221374, 0.00169005, 0.00170265, 0.00179249, 0.0018891 ,
       0.0018829 , 0.00182466, 0.0018682 , 0.0018299 , 0.00162726,
       0.00170186, 0.00167411, 0.00164718, 0.00153536, 0.00129954,
       0.00168902, 0.00200514, 0.00171242, 0.00173005, 0.00166488,
       0.0015701 , 0.00159648, 0.00171337, 0.00165062, 0.00181639,
       0.00194423, 0.0019744 , 0.00193174, 0.00198332, 0.00183896,
       0.00191472, 0.00173159, 0.00176748, 0.00161892, 0.00176676,
       0.00175394, 0.00181595, 0.00173769, 0.00153924, 0.0016947 ,
       0.00175306, 0.002001  , 0.00169064, 0.00189409, 0.00181905,
       0.0018424 , 0.00182179, 0.00181067, 0.00178934, 0.00176478,
       0.00110203, 0.00162448, 0.00160791, 0.00162781, 0.0014255 ,
       0.00155144, 0.00155101, 0.0018005 , 0.00174184, 0.00179912,
       0.00180271, 0.00156028, 0.00164795, 0.00169998, 0.00175111,
       0.00176248, 0.00173468, 0.00180579, 0.00173721, 0.00189216,
       0.00123989, 0.001587  , 0.00169102, 0.00180304, 0.00183076,
       0.00182722, 0.0019403 , 0.00173728, 0.00148543, 0.00166373,
       0.00168458, 0.00170568, 0.00164508, 0.00182302, 0.00156914,
       0.00155949, 0.00149067, 0.00174156, 0.00173925, 0.00163421,
       0.00167604, 0.0018002 , 0.00185132, 0.00190334, 0.00200595,
       0.00204901, 0.00207185, 0.00206582, 0.00204385, 0.00136018,
       0.00190296, 0.00161691, 0.00167667, 0.00168531, 0.00160422,
       0.00206221, 0.00178325, 0.00167824, 0.00184407, 0.00197329,
       0.00190925, 0.00193319, 0.00182056, 0.00178113, 0.00172839,
       0.00155081, 0.00154056, 0.00152611, 0.00151519, 0.00158388,
       0.00161438, 0.00149344, 0.00147435, 0.00150025, 0.00178629,
       0.00169948, 0.00115508, 0.00169537, 0.00176241, 0.00180751,
       0.00176932, 0.00167995, 0.00171908, 0.00164625, 0.00179223,
       0.00157624, 0.00162982, 0.00162884, 0.00185031, 0.00217883,
       0.00172807, 0.00170673, 0.00170511, 0.00166216, 0.00172852,
       0.00164521, 0.00171158, 0.00157485, 0.00161438, 0.00155944,
       0.00157012, 0.00175935, 0.00178341, 0.00191729, 0.00167703,
       0.00173909, 0.00184375, 0.00200376, 0.00192984, 0.00201126,
       0.00191936, 0.00207428, 0.00212526, 0.00110241, 0.00146682,
       0.00143053, 0.00210979, 0.00166529, 0.00153743, 0.00115387,
       0.00173138, 0.0015615 , 0.00180305, 0.00165304, 0.00189598,
       0.00158424, 0.00189235, 0.00172411, 0.00120412, 0.00211832,
       0.00189295, 0.00117639, 0.0015198 , 0.00157865, 0.00176231,
       0.00174563, 0.00161664, 0.00152301, 0.00151793, 0.00182244,
       0.00133056, 0.00145082, 0.00169761, 0.00188594, 0.00195407,
       0.00173456, 0.0016131 , 0.00163092, 0.00208198, 0.00111945,
       0.00115267, 0.00165193, 0.00170296, 0.00172232, 0.00169299,
       0.0017383 , 0.00210168, 0.00178705, 0.00169089, 0.0016437 ,
       0.00161833, 0.00156655, 0.00173205, 0.00177402, 0.00192202,
       0.00178254, 0.00203826, 0.00171258, 0.001782  , 0.00184921,
       0.00209508, 0.00241096, 0.00158911, 0.00200503, 0.00218522,
       0.00143536, 0.00163875, 0.00160496, 0.00112702, 0.00141648,
       0.00152872, 0.00171077, 0.00150747, 0.00149828, 0.00160461,
       0.00156053, 0.00108947, 0.00180823, 0.00185716, 0.00183758,
       0.00174649, 0.00187143, 0.00184926, 0.00168961, 0.00160301,
       0.0017148 , 0.00149132, 0.00191287, 0.00173311, 0.0022036 ,
       0.00172374, 0.00166235, 0.00154676, 0.00145958, 0.00160736,
       0.00186899, 0.00173157, 0.00172744, 0.00171345, 0.00172038,
       0.00169541, 0.00168767, 0.00158175, 0.001776  , 0.00171438,
       0.00171906, 0.00164674, 0.00179563, 0.00172457, 0.0018725 ,
       0.00202886, 0.00173312, 0.00172524, 0.00179482, 0.00187815,
       0.00204875, 0.00195718, 0.00157021, 0.00190557, 0.00217378,
       0.00161631, 0.00176713, 0.00135151, 0.00162538, 0.00156807,
       0.0015785 , 0.00181752, 0.00178557, 0.00175674, 0.00174999,
       0.00166949, 0.00188737, 0.00159589, 0.00196511, 0.00190551,
       0.00175335, 0.00168418, 0.0018359 , 0.00202227, 0.0015637 ,
       0.00154977, 0.00161409, 0.0017353 , 0.00131638, 0.0016269 ,
       0.00157372, 0.00176689, 0.00171275, 0.0016958 , 0.00163168,
       0.00150994, 0.00154396, 0.00147805, 0.00182622, 0.00176868,
       0.00171126, 0.00178309, 0.00188049, 0.00199133, 0.00156244,
       0.00178275, 0.00197002, 0.00166588, 0.00171443, 0.00178813,
       0.0018394 , 0.00144311, 0.00185703, 0.00195364, 0.00161719,
       0.00187442, 0.00208767, 0.00168929, 0.00163143, 0.00162856,
       0.00175164, 0.00166269, 0.00174083, 0.00181252, 0.00164919,
       0.00160917, 0.00168513, 0.00206234, 0.00146809, 0.00198764,
       0.00198073, 0.00182645, 0.00174776, 0.00166596, 0.00189406,
       0.0016116 , 0.0016831 , 0.00172995, 0.0016271 , 0.00160208,
       0.00186248, 0.00179077, 0.00172881, 0.00190039, 0.00179922,
       0.00209196, 0.00175732, 0.00187923, 0.00169291, 0.0016148 ,
       0.00169419, 0.0018524 , 0.00224741, 0.00185519, 0.00171241,
       0.00192591, 0.00184713, 0.00166324, 0.00199158, 0.00146741,
       0.0014582 , 0.00163135, 0.00201472, 0.0020716 , 0.00186727,
       0.00175353, 0.00198643, 0.00162923, 0.00101862, 0.00184809,
       0.00179966, 0.00172617, 0.00187257, 0.0020694 , 0.00176317,
       0.00178013, 0.00179214, 0.00163571, 0.00161057, 0.00171277,
       0.00179388, 0.00194725, 0.00169632, 0.00181497, 0.0019003 ,
       0.00181397, 0.00187076, 0.00168404, 0.00151121, 0.0014248 ,
       0.00177206, 0.00146096, 0.00209018, 0.00209254, 0.00185154,
       0.00173225, 0.00151792, 0.00161729, 0.00179887, 0.00178072,
       0.00171843, 0.00184613, 0.00202695, 0.00183399, 0.00171445,
       0.00169434, 0.00161664, 0.00166901, 0.00169624, 0.00175029,
       0.00179837, 0.00167354, 0.00176695, 0.00188487, 0.00177966,
       0.00173628, 0.00161921, 0.00166038, 0.00162347, 0.0016552 ,
       0.00187148, 0.00215169, 0.00206234, 0.00183163, 0.00170455,
       0.00159842, 0.00175572, 0.00172876, 0.00169765, 0.00181329,
       0.00203137, 0.00189096, 0.00164004, 0.00160417, 0.00170745,
       0.00179183, 0.00165522, 0.00122051, 0.001705  , 0.00170583,
       0.00192971, 0.00176621, 0.00134985, 0.00124121, 0.00175814,
       0.00175172, 0.00176326, 0.00195557, 0.00211372, 0.00196449,
       0.0017623 , 0.00164639, 0.00171258, 0.00173155, 0.00167879,
       0.00166187, 0.00173024, 0.00201999, 0.00196042, 0.00159506,
       0.0018884 , 0.00205752, 0.00196086, 0.00155503, 0.00138344,
       0.00176102, 0.00203688, 0.00182936, 0.0018156 , 0.00180716,
       0.00193389, 0.00196773, 0.00172639, 0.00154687, 0.0016723 ,
       0.00202901, 0.00170386, 0.0016603 , 0.00164795, 0.00165112,
       0.00194783, 0.00202241, 0.00166504, 0.0014415 , 0.00189472,
       0.001842  , 0.00182097, 0.00190418, 0.0018327 , 0.0011972 ,
       0.00159521, 0.00176878, 0.00170898, 0.00165112, 0.00159641,
       0.00187932, 0.00169564, 0.00205602, 0.00180748, 0.0019274 ,
       0.0018362 , 0.00183692, 0.00191141, 0.00177829, 0.0018823 ,
       0.00155733, 0.00172152, 0.00179132, 0.00156914, 0.00178811,
       0.0020467 , 0.00175279, 0.00190513, 0.00191521, 0.00179344,
       0.00182043, 0.00193473, 0.00196536, 0.00192751, 0.00159173,
       0.00172712, 0.0015659 , 0.00190818, 0.00144171, 0.0019182 ,
       0.00184632, 0.00172237, 0.00183415, 0.00134076, 0.00137708,
       0.00150031, 0.00167123, 0.0014268 , 0.00177046, 0.00172696,
       0.00155265])
        indices_b_gt = np.array([   0,    1,    2,    3,    4,    5,    6,    7,    8,    9,   10,
         11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
         22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
         33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,
         44,   45,   46,   47,   48,   49,   50,   51,   52,   53,   54,
         55,   56,   57,   58,   59,   60,   61,   62,   63,   64,   65,
         66,   67,   68,   69,   70,   71,   72,   73,   74,   75,   76,
         77,   78,   79,   80,   81,   82,   83,   84,   85,   86,   87,
         88,   89,   90,   91,   92,   93,   94,   95,   96,   97,   98,
         99,  100,  101,  102,  103,  104,  105,  106,  107,  108,  109,
        110,  111,  112,  113,  114,  115,  116,  117,  118,  119,  120,
        121,  122,  123,  124,  125,  126,  127,  128,  129,  130,  131,
        132,  133,  134,  135,  136,  137,  138,  139,  140,  141,  142,
        143,  144,  145,  146,  147,  148,  149,  150,  151,  152,  153,
        154,  155,  156,  157,  158,  159,  160,  161,  162,  163,  164,
        165,  166,  167,  168,  169,  170,  171,  172,  173,  174,  175,
        176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,
        187,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,
        198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,
        209,  210,  211,  212,  213,  214,  215,  216,  217,  218,  219,
        220,  221,  222,  223,  224,  225,  226,  227,  228,  229,  230,
        231,  232,  233,  234,  235,  236,  237,  238,  239,  240,  241,
        242,  243,  244,  245,  246,  247,  248,  249,  250,  251,  252,
        253,  254,  255,  256,  257,  258,  259,  260,  261,  262,  263,
        264,  265,  266,  267,  268,  269,  270,  271,  272,  273,  274,
        275,  276,  277,  278,  279,  280,  281,  282,  283,  284,  285,
        286,  287,  288,  289,  290,  291,  292,  293,  294,  295,  296,
        297,  298,  299,  300,  301,  302,  303,  304,  305,  306,  307,
        308,  309,  310,  311,  312,  313,  314,  315,  316,  317,  318,
        319,  320,  321,  322,  323,  324,  325,  326,  327,  328,  329,
        330,  331,  332,  333,  334,  335,  336,  337,  338,  339,  340,
        341,  342,  343,  344,  345,  346,  347,  348,  349,  350,  351,
        352,  353,  354,  355,  356,  357,  358,  359,  360,  361,  362,
        363,  364,  365,  366,  367,  368,  369,  370,  371,  372,  373,
        374,  375,  376,  377,  378,  379,  380,  381,  382,  383,  384,
        385,  386,  387,  388,  389,  390,  391,  392,  393,  394,  395,
        396,  397,  398,  399,  400,  401,  402,  403,  404,  405,  406,
        407,  408,  409,  410,  411,  412,  413,  414,  415,  416,  417,
        418,  419,  420,  421,  422,  423,  424,  425,  426,  427,  428,
        429,  430,  431,  432,  433,  434,  435,  436,  437,  438,  439,
        440,  441,  442,  443,  444,  445,  446,  447,  448,  449,  450,
        451,  452,  453,  454,  455,  456,  457,  458,  459,  460,  461,
        462,  463,  464,  465,  466,  467,  468,  469,  470,  471,  472,
        473,  474,  475,  476,  477,  478,  479,  480,  481,  482,  483,
        484,  485,  486,  487,  488,  489,  490,  491,  492,  493,  494,
        495,  496,  497,  498,  499,  500,  501,  502,  503,  504,  505,
        506,  507,  508,  509,  510,  511,  512,  513,  514,  515,  516,
        517,  518,  519,  520,  521,  522,  523,  524,  525,  526,  527,
        528,  529,  530,  531,  532,  533,  534,  535,  536,  537,  538,
        539,  540,  541,  542,  543,  544,  545,  546,  547,  548,  549,
        550,  551,  552,  553,  554,  555,  556,  557,  558,  559,  560,
        561,  562,  563,  564,  565,  566,  567,  568,  569,  570,  571,
        572,  573,  574,  575,  576,  577,  578,  579,  580,  581,  582,
        583,  584,  585,  586,  587,  588,  589,  590,  591,  592,  593,
        594,  595,  596,  597,  598,  599,  600,  601,  602,  603,  604,
        605,  606,  607,  608,  609,  610,  611,  612,  613,  614,  615,
        616,  617,  618,  619,  620,  621,  622,  623,  624,  625,  626,
        627,  628,  629,  630,  631,  632,  633,  634,  635,  636,  637,
        638,  639,  640,  641,  642,  643,  644,  645,  646,  647,  648,
        649,  650,  651,  652,  653,  654,  655,  656,  657,  658,  659,
        660,  661,  662,  663,  664,  665,  666,  667,  668,  669,  670,
        671,  672,  673,  674,  675,  676,  677,  678,  679,  680,  681,
        682,  683,  684,  685,  686,  687,  688,  689,  690,  691,  692,
        693,  694,  695,  696,  697,  698,  699,  700,  701,  702,  703,
        704,  705,  706,  707,  708,  709,  710,  711,  712,  713,  714,
        715,  716,  717,  718,  719,  720,  721,  722,  723,  724,  725,
        726,  727,  728,  729,  730,  731,  732,  733,  734,  735,  736,
        737,  738,  739,  740,  741,  742,  743,  744,  745,  746,  747,
        748,  749,  750,  751,  752,  753,  754,  755,  756,  757,  758,
        759,  760,  761,  762,  763,  764,  765,  766,  767,  768,  769,
        770,  771,  772,  773,  774,  775,  776,  777,  778,  779,  780,
        781,  782,  783,  784,  785,  786,  787,  788,  789,  790,  791,
        792,  793,  794,  795,  796,  797,  798,  799,  800,  801,  802,
        803,  804,  805,  806,  807,  808,  809,  810,  811,  812,  813,
        814,  815,  816,  817,  818,  819,  820,  821,  822,  823,  824,
        825,  826,  827,  828,  829,  830,  831,  832,  833,  834,  835,
        836,  837,  838,  839,  840,  841,  842,  843,  844,  845,  846,
        847,  848,  849,  850,  851,  852,  853,  854,  855,  856,  857,
        858,  859,  860,  861,  862,  863,  864,  865,  866,  867,  868,
        869,  870,  871,  872,  873,  874,  875,  876,  877,  878,  879,
        880,  881,  882,  883,  884,  885,  886,  887,  888,  889,  890,
        891,  892,  893,  894,  895,  896,  897,  898,  899,  900,  901,
        902,  903,  904,  905,  906,  907,  908,  909,  910,  911,  912,
        913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,
        924,  925,  926,  927,  928,  929,  930,  931,  932,  933,  934,
        935,  936,  937,  938,  939,  940,  941,  942,  943,  944,  945,
        946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,
        957,  958,  959,  960,  961,  962,  963,  964,  965,  966,  967,
        968,  969,  970,  971,  972,  973,  974,  975,  976,  977,  978,
        979,  980,  981,  982,  983,  984,  985,  986,  987,  988,  989,
        990,  991,  992,  993,  994,  995,  996,  997,  998,  999, 1000,
       1001, 1002, 1003, 1004, 1005, 1006, 1007, 1008, 1009, 1010, 1011,
       1012, 1013, 1014, 1015, 1016, 1017, 1018, 1019, 1020, 1021, 1022,
       1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1031, 1032, 1033,
       1034, 1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044,
       1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055,
       1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066,
       1067, 1068, 1069, 1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077,
       1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088,
       1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099,
       1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107, 1108, 1109, 1110,
       1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1120, 1121,
       1122, 1123, 1124, 1125, 1126, 1127, 1128, 1129, 1130, 1131, 1132,
       1133, 1134, 1135, 1136, 1137, 1138, 1139, 1140, 1141, 1142, 1143,
       1144, 1145, 1146, 1147, 1148, 1149, 1150, 1151, 1152, 1153, 1154,
       1155, 1156, 1157, 1158, 1159, 1160, 1161, 1162, 1163, 1164, 1165,
       1166, 1167, 1168, 1169, 1170, 1171, 1172, 1173, 1174, 1175, 1176,
       1177, 1178, 1179, 1180, 1181, 1182, 1183, 1184, 1185, 1186, 1187,
       1188, 1189, 1190, 1191, 1192, 1193, 1194, 1195, 1196, 1197, 1198,
       1199, 1200, 1201, 1202, 1203, 1204, 1205, 1206, 1207, 1208, 1209,
       1210, 1211, 1212, 1213, 1214, 1215, 1216, 1217, 1218, 1219, 1220,
       1221, 1222, 1223, 1224, 1225, 1226, 1227, 1228, 1229, 1230, 1231,
       1232, 1233, 1234, 1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242,
       1243, 1244, 1245, 1246, 1247, 1248, 1249, 1250, 1251, 1252, 1253,
       1254, 1255, 1256, 1257, 1258, 1259, 1260, 1261, 1262, 1263, 1264,
       1265, 1266, 1267, 1268, 1269, 1270, 1271, 1272, 1273, 1274, 1275,
       1276, 1277, 1278, 1279, 1280, 1281, 1282, 1283, 1284, 1285, 1286,
       1287, 1288, 1289, 1290, 1291, 1292, 1293, 1294, 1295, 1296, 1297,
       1298, 1299, 1300, 1301, 1302, 1303, 1304, 1305, 1306, 1307, 1308,
       1309, 1310, 1311, 1312, 1313, 1314, 1315, 1316, 1317, 1318, 1319,
       1320, 1321, 1322, 1323, 1324, 1325, 1326, 1327, 1328, 1329, 1330,
       1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341,
       1342, 1343, 1344, 1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352,
       1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363,
       1364, 1365, 1366, 1367, 1368, 1369, 1370, 1371, 1372, 1373, 1374,
       1375, 1376, 1377, 1378, 1379, 1380, 1381, 1382, 1383, 1384, 1385,
       1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396,
       1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407,
       1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 1416, 1417, 1418,
       1419, 1420, 1421, 1422, 1423, 1424, 1425, 1426, 1427, 1428, 1429,
       1430, 1431, 1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440,
       1441, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451,
       1452, 1453, 1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462,
       1463, 1464, 1465, 1466, 1467, 1468, 1469, 1470, 1471, 1472, 1473,
       1474, 1475, 1476, 1477, 1478, 1479, 1480, 1481, 1482, 1483, 1484,
       1485, 1486, 1487, 1488, 1489, 1490, 1491, 1492, 1493, 1494, 1495,
       1496, 1497, 1498, 1499, 1500, 1501, 1502, 1503, 1504, 1505, 1506,
       1507, 1508, 1509, 1510, 1511, 1512, 1513, 1514, 1515, 1516, 1517,
       1518, 1519, 1520, 1521, 1522, 1523, 1524, 1525, 1526, 1527, 1528,
       1529, 1530, 1531, 1532, 1533, 1534, 1535, 1536, 1537, 1538, 1539,
       1540, 1541, 1542, 1543, 1544, 1545, 1546, 1547, 1548, 1549, 1550,
       1551, 1552, 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560, 1561,
       1562, 1563, 1564, 1565, 1566, 1567, 1568, 1569, 1570, 1571, 1572,
       1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580, 1581, 1582, 1583,
       1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593, 1594,
       1595, 1596, 1597, 1598, 1599, 1600, 1601, 1602, 1603, 1604, 1605,
       1606, 1607, 1608, 1609, 1610, 1611, 1612, 1613, 1614, 1615, 1616,
       1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626, 1627,
       1628, 1629, 1630, 1631, 1632, 1633, 1634, 1635, 1636, 1637, 1638,
       1639, 1640, 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648, 1649,
       1650, 1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660,
       1661, 1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671,
       1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682,
       1683, 1684, 1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693,
       1694, 1695, 1696, 1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704,
       1705, 1706, 1707, 1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715,
       1716, 1717, 1718, 1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726,
       1727, 1728, 1729, 1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737,
       1738, 1739, 1740, 1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748,
       1749, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759,
       1760, 1761, 1762, 1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770,
       1771, 1772, 1773, 1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781,
       1782, 1783, 1784, 1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792,
       1793, 1794, 1795, 1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803,
       1804, 1805, 1806, 1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814,
       1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825,
       1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836,
       1837, 1838, 1839, 1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847,
       1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858,
       1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869,
       1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880,
       1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891,
       1892, 1893, 1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902,
       1903, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913,
       1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924,
       1925, 1926, 1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935,
       1936, 1937, 1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946,
       1947, 1948, 1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957,
       1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968,
       1969, 1970, 1971, 1972, 1973, 1974, 1975, 1976, 1977, 1978, 1979,
       1980, 1981, 1982, 1983, 1984, 1985, 1986, 1987, 1988, 1989, 1990,
       1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001,
       2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012,
       2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023,
       2024, 2025, 2026, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034,
       2035, 2036, 2037, 2038, 2039, 2040, 2041, 2042, 2043, 2044, 2045,
       2046, 2047, 2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056,
       2057, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067,
       2068, 2069, 2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078,
       2079, 2080, 2081, 2082, 2083, 2084, 2085, 2086, 2087, 2088, 2089,
       2090, 2091, 2092, 2093, 2094, 2095, 2096, 2097, 2098, 2099, 2100,
       2101, 2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111,
       2112, 2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122,
       2123, 2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133,
       2134, 2135, 2136, 2137, 2138, 2139, 2140, 2141, 2142, 2143, 2144,
       2145, 2146, 2147, 2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155,
       2156, 2157, 2158, 2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166,
       2167, 2168, 2169, 2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177,
       2178, 2179, 2180, 2181, 2182, 2183, 2184, 2185, 2186, 2187, 2188,
       2189, 2190, 2191, 2192, 2193, 2194, 2195, 2196, 2197, 2198, 2199,
       2200, 2201, 2202, 2203, 2204, 2205, 2206, 2207, 2208, 2209, 2210,
       2211, 2212, 2213, 2214, 2215, 2216, 2217, 2218, 2219, 2220, 2221,
       2222, 2223, 2224, 2225, 2226, 2227, 2228, 2229, 2230, 2231, 2232,
       2233, 2234, 2235, 2236, 2237, 2238, 2239, 2240, 2241, 2242, 2243,
       2244, 2245, 2246, 2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254,
       2255, 2256, 2257, 2258, 2259, 2260, 2261, 2262, 2263, 2264, 2265,
       2266, 2267, 2268, 2269, 2270, 2271, 2272, 2273, 2274, 2275, 2276,
       2277, 2278, 2279, 2280, 2281, 2282, 2283, 2284, 2285, 2286, 2287,
       2288, 2289, 2290, 2291, 2292, 2293, 2294, 2295, 2296, 2297, 2298,
       2299, 2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307, 2308, 2309,
       2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320,
       2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331,
       2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342,
       2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353,
       2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364,
       2365, 2366, 2367, 2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375,
       2376, 2377, 2378, 2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386,
       2387, 2388, 2389, 2390, 2391, 2392, 2393, 2394, 2395, 2396, 2397,
       2398, 2399, 2400, 2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408,
       2409, 2410, 2411, 2412, 2413, 2414, 2415, 2416, 2417, 2418, 2419,
       2420, 2421, 2422, 2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430,
       2431, 2432, 2433, 2434, 2435, 2436, 2437, 2438, 2439, 2440, 2441,
       2442, 2443, 2444, 2445, 2446, 2447, 2448, 2449, 2450, 2451, 2452,
       2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463,
       2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2473, 2474,
       2475, 2476, 2477, 2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485,
       2486, 2487, 2488, 2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496,
       2497, 2498, 2499, 2500, 2501, 2502, 2503, 2504, 2505, 2506, 2507,
       2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515, 2516, 2517, 2518,
       2519, 2520, 2521, 2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529,
       2530, 2531, 2532, 2533, 2534, 2535, 2536, 2537, 2538, 2539, 2540,
       2541, 2542, 2543, 2544, 2545, 2546, 2547, 2548, 2549, 2550, 2551,
       2552, 2553, 2554, 2555, 2556, 2557, 2558, 2559, 2560, 2561, 2562,
       2563, 2564, 2565, 2566, 2567, 2568, 2569, 2570, 2571, 2572, 2573,
       2574, 2575, 2576, 2577, 2578, 2579, 2580, 2581, 2582, 2583, 2584,
       2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595,
       2596, 2597, 2598, 2599, 2600, 2601, 2602, 2603, 2604, 2605, 2606,
       2607, 2608, 2609, 2610, 2611, 2612, 2613, 2614, 2615, 2616, 2617,
       2618, 2619, 2620, 2621, 2622, 2623, 2624, 2625, 2626, 2627, 2628,
       2629, 2630, 2631, 2632, 2633, 2634, 2635, 2636, 2637, 2638, 2639,
       2640, 2641, 2642, 2643, 2644, 2645, 2646, 2647, 2648, 2649, 2650,
       2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658, 2659, 2660, 2661,
       2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2672,
       2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680, 2681, 2682, 2683,
       2684, 2685, 2686, 2687, 2688, 2689, 2690, 2691, 2692, 2693, 2694,
       2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702, 2703, 2704, 2705,
       2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716,
       2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727,
       2728, 2729, 2730, 2731, 2732, 2733, 2734, 2735, 2736, 2737, 2738,
       2739, 2740, 2741, 2742, 2743, 2744, 2745, 2746, 2747, 2748, 2749,
       2750, 2751, 2752, 2753, 2754, 2755, 2756, 2757, 2758, 2759, 2760,
       2761, 2762, 2763, 2764, 2765, 2766, 2767, 2768, 2769, 2770, 2771,
       2772, 2773, 2774, 2775, 2776, 2777, 2778, 2779, 2780, 2781, 2782,
       2783, 2784, 2785, 2786, 2787, 2788, 2789, 2790, 2791, 2792, 2793,
       2794, 2795, 2796, 2797, 2798, 2799, 2800, 2801, 2802, 2803, 2804,
       2805, 2806, 2807, 2808, 2809, 2810, 2811, 2812, 2813, 2814, 2815,
       2816, 2817, 2818, 2819, 2820, 2821, 2822, 2823, 2824, 2825, 2826,
       2827, 2828, 2829, 2830, 2831, 2832, 2833, 2834, 2835, 2836, 2837,
       2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847, 2848,
       2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2857, 2858, 2859,
       2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867, 2868, 2869, 2870,
       2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878, 2879, 2880, 2881,
       2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889, 2890, 2891, 2892,
       2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900, 2901, 2902, 2903,
       2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913, 2914,
       2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925,
       2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935, 2936,
       2937, 2938, 2939, 2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947,
       2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956, 2957, 2958,
       2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967, 2968, 2969,
       2970, 2971, 2972, 2973, 2974, 2975, 2976, 2977, 2978, 2979, 2980,
       2981, 2982, 2983, 2984, 2985, 2986, 2987, 2988, 2989, 2990, 2991,
       2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3001, 3002,
       3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010, 3011, 3012, 3013,
       3014, 3015, 3016, 3017, 3018, 3019, 3020, 3021, 3022, 3023, 3024,
       3025, 3026, 3027, 3028, 3029, 3030, 3031, 3032, 3033, 3034, 3035,
       3036, 3037, 3038, 3039, 3040, 3041, 3042, 3043, 3044, 3045, 3046,
       3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055, 3056, 3057,
       3058, 3059, 3060, 3061, 3062, 3063, 3064, 3065, 3066, 3067, 3068,
       3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076, 3077, 3078, 3079,
       3080, 3081, 3082, 3083, 3084, 3085, 3086, 3087, 3088, 3089, 3090,
       3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099, 3100, 3101,
       3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110, 3111, 3112,
       3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121, 3122, 3123,
       3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131, 3132, 3133, 3134,
       3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142, 3143, 3144, 3145,
       3146, 3147, 3148, 3149, 3150, 3151, 3152, 3153, 3154, 3155, 3156,
       3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164, 3165, 3166, 3167,
       3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175, 3176, 3177, 3178,
       3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 3189,
       3190, 3191, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3199, 3200,
       3201, 3202, 3203, 3204, 3205, 3206, 3207, 3208, 3209, 3210, 3211,
       3212, 3213, 3214, 3215, 3216, 3217, 3218, 3219, 3220, 3221, 3222,
       3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233,
       3234, 3235, 3236, 3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244,
       3245, 3246, 3247, 3248, 3249, 3250, 3251, 3252, 3253, 3254, 3255,
       3256, 3257, 3258, 3259, 3260, 3261, 3262, 3263, 3264, 3265, 3266,
       3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274, 3275, 3276, 3277,
       3278, 3279, 3280, 3281, 3282, 3283, 3284, 3285, 3286, 3287, 3288,
       3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296, 3297, 3298, 3299,
       3300], dtype=np.int32)
        indptr_b_gt = np.array([   0,    1,    2,    3,    4,    5,    6,    7,    8,    9,   10,
         11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
         22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
         33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,
         44,   45,   46,   47,   48,   49,   50,   51,   52,   53,   54,
         55,   56,   57,   58,   59,   60,   61,   62,   63,   64,   65,
         66,   67,   68,   69,   70,   71,   72,   73,   74,   75,   76,
         77,   78,   79,   80,   81,   82,   83,   84,   85,   86,   87,
         88,   89,   90,   91,   92,   93,   94,   95,   96,   97,   98,
         99,  100,  101,  102,  103,  104,  105,  106,  107,  108,  109,
        110,  111,  112,  113,  114,  115,  116,  117,  118,  119,  120,
        121,  122,  123,  124,  125,  126,  127,  128,  129,  130,  131,
        132,  133,  134,  135,  136,  137,  138,  139,  140,  141,  142,
        143,  144,  145,  146,  147,  148,  149,  150,  151,  152,  153,
        154,  155,  156,  157,  158,  159,  160,  161,  162,  163,  164,
        165,  166,  167,  168,  169,  170,  171,  172,  173,  174,  175,
        176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,
        187,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,
        198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,
        209,  210,  211,  212,  213,  214,  215,  216,  217,  218,  219,
        220,  221,  222,  223,  224,  225,  226,  227,  228,  229,  230,
        231,  232,  233,  234,  235,  236,  237,  238,  239,  240,  241,
        242,  243,  244,  245,  246,  247,  248,  249,  250,  251,  252,
        253,  254,  255,  256,  257,  258,  259,  260,  261,  262,  263,
        264,  265,  266,  267,  268,  269,  270,  271,  272,  273,  274,
        275,  276,  277,  278,  279,  280,  281,  282,  283,  284,  285,
        286,  287,  288,  289,  290,  291,  292,  293,  294,  295,  296,
        297,  298,  299,  300,  301,  302,  303,  304,  305,  306,  307,
        308,  309,  310,  311,  312,  313,  314,  315,  316,  317,  318,
        319,  320,  321,  322,  323,  324,  325,  326,  327,  328,  329,
        330,  331,  332,  333,  334,  335,  336,  337,  338,  339,  340,
        341,  342,  343,  344,  345,  346,  347,  348,  349,  350,  351,
        352,  353,  354,  355,  356,  357,  358,  359,  360,  361,  362,
        363,  364,  365,  366,  367,  368,  369,  370,  371,  372,  373,
        374,  375,  376,  377,  378,  379,  380,  381,  382,  383,  384,
        385,  386,  387,  388,  389,  390,  391,  392,  393,  394,  395,
        396,  397,  398,  399,  400,  401,  402,  403,  404,  405,  406,
        407,  408,  409,  410,  411,  412,  413,  414,  415,  416,  417,
        418,  419,  420,  421,  422,  423,  424,  425,  426,  427,  428,
        429,  430,  431,  432,  433,  434,  435,  436,  437,  438,  439,
        440,  441,  442,  443,  444,  445,  446,  447,  448,  449,  450,
        451,  452,  453,  454,  455,  456,  457,  458,  459,  460,  461,
        462,  463,  464,  465,  466,  467,  468,  469,  470,  471,  472,
        473,  474,  475,  476,  477,  478,  479,  480,  481,  482,  483,
        484,  485,  486,  487,  488,  489,  490,  491,  492,  493,  494,
        495,  496,  497,  498,  499,  500,  501,  502,  503,  504,  505,
        506,  507,  508,  509,  510,  511,  512,  513,  514,  515,  516,
        517,  518,  519,  520,  521,  522,  523,  524,  525,  526,  527,
        528,  529,  530,  531,  532,  533,  534,  535,  536,  537,  538,
        539,  540,  541,  542,  543,  544,  545,  546,  547,  548,  549,
        550,  551,  552,  553,  554,  555,  556,  557,  558,  559,  560,
        561,  562,  563,  564,  565,  566,  567,  568,  569,  570,  571,
        572,  573,  574,  575,  576,  577,  578,  579,  580,  581,  582,
        583,  584,  585,  586,  587,  588,  589,  590,  591,  592,  593,
        594,  595,  596,  597,  598,  599,  600,  601,  602,  603,  604,
        605,  606,  607,  608,  609,  610,  611,  612,  613,  614,  615,
        616,  617,  618,  619,  620,  621,  622,  623,  624,  625,  626,
        627,  628,  629,  630,  631,  632,  633,  634,  635,  636,  637,
        638,  639,  640,  641,  642,  643,  644,  645,  646,  647,  648,
        649,  650,  651,  652,  653,  654,  655,  656,  657,  658,  659,
        660,  661,  662,  663,  664,  665,  666,  667,  668,  669,  670,
        671,  672,  673,  674,  675,  676,  677,  678,  679,  680,  681,
        682,  683,  684,  685,  686,  687,  688,  689,  690,  691,  692,
        693,  694,  695,  696,  697,  698,  699,  700,  701,  702,  703,
        704,  705,  706,  707,  708,  709,  710,  711,  712,  713,  714,
        715,  716,  717,  718,  719,  720,  721,  722,  723,  724,  725,
        726,  727,  728,  729,  730,  731,  732,  733,  734,  735,  736,
        737,  738,  739,  740,  741,  742,  743,  744,  745,  746,  747,
        748,  749,  750,  751,  752,  753,  754,  755,  756,  757,  758,
        759,  760,  761,  762,  763,  764,  765,  766,  767,  768,  769,
        770,  771,  772,  773,  774,  775,  776,  777,  778,  779,  780,
        781,  782,  783,  784,  785,  786,  787,  788,  789,  790,  791,
        792,  793,  794,  795,  796,  797,  798,  799,  800,  801,  802,
        803,  804,  805,  806,  807,  808,  809,  810,  811,  812,  813,
        814,  815,  816,  817,  818,  819,  820,  821,  822,  823,  824,
        825,  826,  827,  828,  829,  830,  831,  832,  833,  834,  835,
        836,  837,  838,  839,  840,  841,  842,  843,  844,  845,  846,
        847,  848,  849,  850,  851,  852,  853,  854,  855,  856,  857,
        858,  859,  860,  861,  862,  863,  864,  865,  866,  867,  868,
        869,  870,  871,  872,  873,  874,  875,  876,  877,  878,  879,
        880,  881,  882,  883,  884,  885,  886,  887,  888,  889,  890,
        891,  892,  893,  894,  895,  896,  897,  898,  899,  900,  901,
        902,  903,  904,  905,  906,  907,  908,  909,  910,  911,  912,
        913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,
        924,  925,  926,  927,  928,  929,  930,  931,  932,  933,  934,
        935,  936,  937,  938,  939,  940,  941,  942,  943,  944,  945,
        946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,
        957,  958,  959,  960,  961,  962,  963,  964,  965,  966,  967,
        968,  969,  970,  971,  972,  973,  974,  975,  976,  977,  978,
        979,  980,  981,  982,  983,  984,  985,  986,  987,  988,  989,
        990,  991,  992,  993,  994,  995,  996,  997,  998,  999, 1000,
       1001, 1002, 1003, 1004, 1005, 1006, 1007, 1008, 1009, 1010, 1011,
       1012, 1013, 1014, 1015, 1016, 1017, 1018, 1019, 1020, 1021, 1022,
       1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1031, 1032, 1033,
       1034, 1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044,
       1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055,
       1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066,
       1067, 1068, 1069, 1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077,
       1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088,
       1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099,
       1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107, 1108, 1109, 1110,
       1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1120, 1121,
       1122, 1123, 1124, 1125, 1126, 1127, 1128, 1129, 1130, 1131, 1132,
       1133, 1134, 1135, 1136, 1137, 1138, 1139, 1140, 1141, 1142, 1143,
       1144, 1145, 1146, 1147, 1148, 1149, 1150, 1151, 1152, 1153, 1154,
       1155, 1156, 1157, 1158, 1159, 1160, 1161, 1162, 1163, 1164, 1165,
       1166, 1167, 1168, 1169, 1170, 1171, 1172, 1173, 1174, 1175, 1176,
       1177, 1178, 1179, 1180, 1181, 1182, 1183, 1184, 1185, 1186, 1187,
       1188, 1189, 1190, 1191, 1192, 1193, 1194, 1195, 1196, 1197, 1198,
       1199, 1200, 1201, 1202, 1203, 1204, 1205, 1206, 1207, 1208, 1209,
       1210, 1211, 1212, 1213, 1214, 1215, 1216, 1217, 1218, 1219, 1220,
       1221, 1222, 1223, 1224, 1225, 1226, 1227, 1228, 1229, 1230, 1231,
       1232, 1233, 1234, 1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242,
       1243, 1244, 1245, 1246, 1247, 1248, 1249, 1250, 1251, 1252, 1253,
       1254, 1255, 1256, 1257, 1258, 1259, 1260, 1261, 1262, 1263, 1264,
       1265, 1266, 1267, 1268, 1269, 1270, 1271, 1272, 1273, 1274, 1275,
       1276, 1277, 1278, 1279, 1280, 1281, 1282, 1283, 1284, 1285, 1286,
       1287, 1288, 1289, 1290, 1291, 1292, 1293, 1294, 1295, 1296, 1297,
       1298, 1299, 1300, 1301, 1302, 1303, 1304, 1305, 1306, 1307, 1308,
       1309, 1310, 1311, 1312, 1313, 1314, 1315, 1316, 1317, 1318, 1319,
       1320, 1321, 1322, 1323, 1324, 1325, 1326, 1327, 1328, 1329, 1330,
       1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341,
       1342, 1343, 1344, 1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352,
       1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363,
       1364, 1365, 1366, 1367, 1368, 1369, 1370, 1371, 1372, 1373, 1374,
       1375, 1376, 1377, 1378, 1379, 1380, 1381, 1382, 1383, 1384, 1385,
       1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396,
       1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407,
       1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 1416, 1417, 1418,
       1419, 1420, 1421, 1422, 1423, 1424, 1425, 1426, 1427, 1428, 1429,
       1430, 1431, 1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440,
       1441, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451,
       1452, 1453, 1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462,
       1463, 1464, 1465, 1466, 1467, 1468, 1469, 1470, 1471, 1472, 1473,
       1474, 1475, 1476, 1477, 1478, 1479, 1480, 1481, 1482, 1483, 1484,
       1485, 1486, 1487, 1488, 1489, 1490, 1491, 1492, 1493, 1494, 1495,
       1496, 1497, 1498, 1499, 1500, 1501, 1502, 1503, 1504, 1505, 1506,
       1507, 1508, 1509, 1510, 1511, 1512, 1513, 1514, 1515, 1516, 1517,
       1518, 1519, 1520, 1521, 1522, 1523, 1524, 1525, 1526, 1527, 1528,
       1529, 1530, 1531, 1532, 1533, 1534, 1535, 1536, 1537, 1538, 1539,
       1540, 1541, 1542, 1543, 1544, 1545, 1546, 1547, 1548, 1549, 1550,
       1551, 1552, 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560, 1561,
       1562, 1563, 1564, 1565, 1566, 1567, 1568, 1569, 1570, 1571, 1572,
       1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580, 1581, 1582, 1583,
       1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593, 1594,
       1595, 1596, 1597, 1598, 1599, 1600, 1601, 1602, 1603, 1604, 1605,
       1606, 1607, 1608, 1609, 1610, 1611, 1612, 1613, 1614, 1615, 1616,
       1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626, 1627,
       1628, 1629, 1630, 1631, 1632, 1633, 1634, 1635, 1636, 1637, 1638,
       1639, 1640, 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648, 1649,
       1650, 1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660,
       1661, 1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671,
       1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682,
       1683, 1684, 1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693,
       1694, 1695, 1696, 1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704,
       1705, 1706, 1707, 1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715,
       1716, 1717, 1718, 1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726,
       1727, 1728, 1729, 1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737,
       1738, 1739, 1740, 1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748,
       1749, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759,
       1760, 1761, 1762, 1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770,
       1771, 1772, 1773, 1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781,
       1782, 1783, 1784, 1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792,
       1793, 1794, 1795, 1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803,
       1804, 1805, 1806, 1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814,
       1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825,
       1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836,
       1837, 1838, 1839, 1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847,
       1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858,
       1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869,
       1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880,
       1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891,
       1892, 1893, 1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902,
       1903, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913,
       1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924,
       1925, 1926, 1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935,
       1936, 1937, 1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946,
       1947, 1948, 1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957,
       1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968,
       1969, 1970, 1971, 1972, 1973, 1974, 1975, 1976, 1977, 1978, 1979,
       1980, 1981, 1982, 1983, 1984, 1985, 1986, 1987, 1988, 1989, 1990,
       1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001,
       2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012,
       2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023,
       2024, 2025, 2026, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034,
       2035, 2036, 2037, 2038, 2039, 2040, 2041, 2042, 2043, 2044, 2045,
       2046, 2047, 2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056,
       2057, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067,
       2068, 2069, 2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078,
       2079, 2080, 2081, 2082, 2083, 2084, 2085, 2086, 2087, 2088, 2089,
       2090, 2091, 2092, 2093, 2094, 2095, 2096, 2097, 2098, 2099, 2100,
       2101, 2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111,
       2112, 2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122,
       2123, 2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133,
       2134, 2135, 2136, 2137, 2138, 2139, 2140, 2141, 2142, 2143, 2144,
       2145, 2146, 2147, 2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155,
       2156, 2157, 2158, 2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166,
       2167, 2168, 2169, 2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177,
       2178, 2179, 2180, 2181, 2182, 2183, 2184, 2185, 2186, 2187, 2188,
       2189, 2190, 2191, 2192, 2193, 2194, 2195, 2196, 2197, 2198, 2199,
       2200, 2201, 2202, 2203, 2204, 2205, 2206, 2207, 2208, 2209, 2210,
       2211, 2212, 2213, 2214, 2215, 2216, 2217, 2218, 2219, 2220, 2221,
       2222, 2223, 2224, 2225, 2226, 2227, 2228, 2229, 2230, 2231, 2232,
       2233, 2234, 2235, 2236, 2237, 2238, 2239, 2240, 2241, 2242, 2243,
       2244, 2245, 2246, 2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254,
       2255, 2256, 2257, 2258, 2259, 2260, 2261, 2262, 2263, 2264, 2265,
       2266, 2267, 2268, 2269, 2270, 2271, 2272, 2273, 2274, 2275, 2276,
       2277, 2278, 2279, 2280, 2281, 2282, 2283, 2284, 2285, 2286, 2287,
       2288, 2289, 2290, 2291, 2292, 2293, 2294, 2295, 2296, 2297, 2298,
       2299, 2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307, 2308, 2309,
       2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320,
       2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331,
       2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342,
       2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353,
       2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364,
       2365, 2366, 2367, 2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375,
       2376, 2377, 2378, 2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386,
       2387, 2388, 2389, 2390, 2391, 2392, 2393, 2394, 2395, 2396, 2397,
       2398, 2399, 2400, 2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408,
       2409, 2410, 2411, 2412, 2413, 2414, 2415, 2416, 2417, 2418, 2419,
       2420, 2421, 2422, 2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430,
       2431, 2432, 2433, 2434, 2435, 2436, 2437, 2438, 2439, 2440, 2441,
       2442, 2443, 2444, 2445, 2446, 2447, 2448, 2449, 2450, 2451, 2452,
       2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463,
       2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2473, 2474,
       2475, 2476, 2477, 2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485,
       2486, 2487, 2488, 2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496,
       2497, 2498, 2499, 2500, 2501, 2502, 2503, 2504, 2505, 2506, 2507,
       2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515, 2516, 2517, 2518,
       2519, 2520, 2521, 2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529,
       2530, 2531, 2532, 2533, 2534, 2535, 2536, 2537, 2538, 2539, 2540,
       2541, 2542, 2543, 2544, 2545, 2546, 2547, 2548, 2549, 2550, 2551,
       2552, 2553, 2554, 2555, 2556, 2557, 2558, 2559, 2560, 2561, 2562,
       2563, 2564, 2565, 2566, 2567, 2568, 2569, 2570, 2571, 2572, 2573,
       2574, 2575, 2576, 2577, 2578, 2579, 2580, 2581, 2582, 2583, 2584,
       2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595,
       2596, 2597, 2598, 2599, 2600, 2601, 2602, 2603, 2604, 2605, 2606,
       2607, 2608, 2609, 2610, 2611, 2612, 2613, 2614, 2615, 2616, 2617,
       2618, 2619, 2620, 2621, 2622, 2623, 2624, 2625, 2626, 2627, 2628,
       2629, 2630, 2631, 2632, 2633, 2634, 2635, 2636, 2637, 2638, 2639,
       2640, 2641, 2642, 2643, 2644, 2645, 2646, 2647, 2648, 2649, 2650,
       2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658, 2659, 2660, 2661,
       2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2672,
       2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680, 2681, 2682, 2683,
       2684, 2685, 2686, 2687, 2688, 2689, 2690, 2691, 2692, 2693, 2694,
       2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702, 2703, 2704, 2705,
       2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716,
       2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727,
       2728, 2729, 2730, 2731, 2732, 2733, 2734, 2735, 2736, 2737, 2738,
       2739, 2740, 2741, 2742, 2743, 2744, 2745, 2746, 2747, 2748, 2749,
       2750, 2751, 2752, 2753, 2754, 2755, 2756, 2757, 2758, 2759, 2760,
       2761, 2762, 2763, 2764, 2765, 2766, 2767, 2768, 2769, 2770, 2771,
       2772, 2773, 2774, 2775, 2776, 2777, 2778, 2779, 2780, 2781, 2782,
       2783, 2784, 2785, 2786, 2787, 2788, 2789, 2790, 2791, 2792, 2793,
       2794, 2795, 2796, 2797, 2798, 2799, 2800, 2801, 2802, 2803, 2804,
       2805, 2806, 2807, 2808, 2809, 2810, 2811, 2812, 2813, 2814, 2815,
       2816, 2817, 2818, 2819, 2820, 2821, 2822, 2823, 2824, 2825, 2826,
       2827, 2828, 2829, 2830, 2831, 2832, 2833, 2834, 2835, 2836, 2837,
       2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847, 2848,
       2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2857, 2858, 2859,
       2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867, 2868, 2869, 2870,
       2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878, 2879, 2880, 2881,
       2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889, 2890, 2891, 2892,
       2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900, 2901, 2902, 2903,
       2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913, 2914,
       2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925,
       2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935, 2936,
       2937, 2938, 2939, 2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947,
       2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956, 2957, 2958,
       2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967, 2968, 2969,
       2970, 2971, 2972, 2973, 2974, 2975, 2976, 2977, 2978, 2979, 2980,
       2981, 2982, 2983, 2984, 2985, 2986, 2987, 2988, 2989, 2990, 2991,
       2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3001, 3002,
       3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010, 3011, 3012, 3013,
       3014, 3015, 3016, 3017, 3018, 3019, 3020, 3021, 3022, 3023, 3024,
       3025, 3026, 3027, 3028, 3029, 3030, 3031, 3032, 3033, 3034, 3035,
       3036, 3037, 3038, 3039, 3040, 3041, 3042, 3043, 3044, 3045, 3046,
       3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055, 3056, 3057,
       3058, 3059, 3060, 3061, 3062, 3063, 3064, 3065, 3066, 3067, 3068,
       3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076, 3077, 3078, 3079,
       3080, 3081, 3082, 3083, 3084, 3085, 3086, 3087, 3088, 3089, 3090,
       3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099, 3100, 3101,
       3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110, 3111, 3112,
       3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121, 3122, 3123,
       3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131, 3132, 3133, 3134,
       3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142, 3143, 3144, 3145,
       3146, 3147, 3148, 3149, 3150, 3151, 3152, 3153, 3154, 3155, 3156,
       3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164, 3165, 3166, 3167,
       3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175, 3176, 3177, 3178,
       3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 3189,
       3190, 3191, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3199, 3200,
       3201, 3202, 3203, 3204, 3205, 3206, 3207, 3208, 3209, 3210, 3211,
       3212, 3213, 3214, 3215, 3216, 3217, 3218, 3219, 3220, 3221, 3222,
       3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233,
       3234, 3235, 3236, 3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244,
       3245, 3246, 3247, 3248, 3249, 3250, 3251, 3252, 3253, 3254, 3255,
       3256, 3257, 3258, 3259, 3260, 3261, 3262, 3263, 3264, 3265, 3266,
       3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274, 3275, 3276, 3277,
       3278, 3279, 3280, 3281, 3282, 3283, 3284, 3285, 3286, 3287, 3288,
       3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296, 3297, 3298, 3299,
       3300, 3301], dtype=np.int32)
        data_v_gt = np.array([0.00151038, 0.00112114, 0.00173692, 0.00160491, 0.00142958,
       0.00176806, 0.00182612, 0.00144052, 0.0017937 , 0.00184814,
       0.00172271, 0.00189734, 0.00186129, 0.00141699, 0.00180315,
       0.00161264, 0.00185074, 0.00175612, 0.00190281, 0.00185712,
       0.00162403, 0.00176552, 0.00157568, 0.00187786, 0.00176565,
       0.00171791, 0.00195411, 0.00200628, 0.0020402 , 0.00143864,
       0.00150056, 0.00181413, 0.00151696, 0.00186226, 0.00184446,
       0.00177113, 0.00188314, 0.00175922, 0.00173213, 0.00174472,
       0.00151565, 0.00185367, 0.001413  , 0.00182556, 0.00179308,
       0.00172757, 0.0016864 , 0.0018636 , 0.00196818, 0.00209116,
       0.002047  , 0.00163023, 0.00158592, 0.00157499, 0.00181872,
       0.00153333, 0.00194793, 0.00172147, 0.00193312, 0.00176286,
       0.00181018, 0.00186532, 0.00153401, 0.00174656, 0.00184644,
       0.00184984, 0.00121311, 0.00185523, 0.00151276, 0.00179976,
       0.00177359, 0.00180806, 0.00169787, 0.00166358, 0.00169219,
       0.00174134, 0.00196833, 0.00205651, 0.00185522, 0.00183749,
       0.00171944, 0.00166851, 0.00163966, 0.00180197, 0.00162784,
       0.00141158, 0.00185176, 0.00165818, 0.00176949, 0.00172495,
       0.00211973, 0.00196779, 0.0017796 , 0.00196428, 0.00124471,
       0.00147123, 0.00155485, 0.00184039, 0.00180475, 0.00168576,
       0.00162899, 0.00177409, 0.00174391, 0.00176312, 0.00176071,
       0.00169776, 0.00171948, 0.00121323, 0.00141771, 0.00162423,
       0.00198693, 0.00196332, 0.00178493, 0.00179817, 0.00183364,
       0.00174356, 0.00170644, 0.00169111, 0.00181942, 0.00177119,
       0.00168233, 0.00157858, 0.00168001, 0.00162297, 0.00168238,
       0.00170331, 0.00173334, 0.00198752, 0.00182522, 0.00174786,
       0.00199831, 0.00181869, 0.00176707, 0.00175394, 0.00174516,
       0.00177372, 0.00179361, 0.00177818, 0.00169621, 0.00160855,
       0.00173799, 0.00167907, 0.00169666, 0.00169574, 0.00171234,
       0.00173568, 0.00180074, 0.00134015, 0.00179575, 0.00166924,
       0.00190123, 0.00188465, 0.00175669, 0.0018083 , 0.0017811 ,
       0.00178002, 0.00175194, 0.00173487, 0.00171762, 0.00173853,
       0.00175685, 0.00170218, 0.00166214, 0.00162058, 0.00164048,
       0.00163157, 0.00170407, 0.00168613, 0.00174942, 0.0018423 ,
       0.0017796 , 0.00174901, 0.00175421, 0.00179361, 0.00174126,
       0.00161738, 0.00164153, 0.00171566, 0.00170445, 0.00176312,
       0.00173186, 0.00177494, 0.0017713 , 0.00169354, 0.00167239,
       0.00172135, 0.00159095, 0.00159831, 0.00161442, 0.00166336,
       0.00173213, 0.00181121, 0.00184997, 0.00160827, 0.00167133,
       0.00168123, 0.00177138, 0.00182976, 0.00169068, 0.00179988,
       0.00179195, 0.00178042, 0.00177288, 0.00176448, 0.00175186,
       0.00172303, 0.00170328, 0.00177523, 0.00169928, 0.00169991,
       0.00165058, 0.00168399, 0.00166426, 0.0016956 , 0.00170834,
       0.00172423, 0.00184271, 0.00168935, 0.00163982, 0.001623  ,
       0.00170829, 0.00175534, 0.00170082, 0.00169448, 0.00156414,
       0.00147269, 0.00166864, 0.00174027, 0.00168568, 0.00172228,
       0.001774  , 0.00169   , 0.0017915 , 0.00179726, 0.00169146,
       0.00170929, 0.00170742, 0.00161249, 0.00153091, 0.00144496,
       0.00154106, 0.00171287, 0.00179297, 0.00182406, 0.00155585,
       0.00158653, 0.00144214, 0.0017622 , 0.00173011, 0.00181462,
       0.00157946, 0.0018052 , 0.00179872, 0.00179508, 0.00178308,
       0.0017781 , 0.00176996, 0.00173763, 0.0016968 , 0.00168454,
       0.00182878, 0.00168153, 0.00170542, 0.00170892, 0.00174311,
       0.001743  , 0.0018044 , 0.00172016, 0.00169957, 0.00186888,
       0.00177483, 0.00161055, 0.00145686, 0.00134146, 0.00157614,
       0.00172382, 0.00174107, 0.0016558 , 0.00166554, 0.00156238,
       0.0016379 , 0.0015122 , 0.00171383, 0.0017629 , 0.00167814,
       0.00158375, 0.00170951, 0.00182003, 0.00162108, 0.00181058,
       0.00177115, 0.00168567, 0.00168946, 0.00165386, 0.00166386,
       0.00159452, 0.00151017, 0.00119699, 0.00149703, 0.0017328 ,
       0.0018448 , 0.00112876, 0.00165234, 0.00165544, 0.00171919,
       0.00175928, 0.00174125, 0.001547  , 0.00180283, 0.00179847,
       0.00179936, 0.00178969, 0.00177383, 0.00178933, 0.00174688,
       0.00168367, 0.00165116, 0.00164106, 0.00186864, 0.00176961,
       0.00168209, 0.00172982, 0.00175035, 0.00176562, 0.00174111,
       0.00193373, 0.00170211, 0.00164201, 0.0018206 , 0.00131955,
       0.00167926, 0.00141783, 0.0015962 , 0.00106887, 0.00129564,
       0.00158921, 0.00162074, 0.00168499, 0.0016356 , 0.00157988,
       0.00166612, 0.00167898, 0.00170032, 0.00168034, 0.00175076,
       0.00178728, 0.00164084, 0.00117338, 0.00150117, 0.00194977,
       0.00189449, 0.00160586, 0.00180846, 0.00173949, 0.00168972,
       0.00165622, 0.00151188, 0.00169163, 0.00167157, 0.00163978,
       0.00146333, 0.00146667, 0.00178039, 0.00185186, 0.00153062,
       0.00168853, 0.00173174, 0.00163071, 0.00154884, 0.00180245,
       0.00179535, 0.0017844 , 0.00180282, 0.00176859, 0.00174506,
       0.00180366, 0.001682  , 0.00154404, 0.0015603 , 0.00158625,
       0.00180147, 0.00181552, 0.00168494, 0.00173763, 0.00163302,
       0.00152302, 0.00175588, 0.00176099, 0.00163451, 0.00168583,
       0.00168345, 0.00161329, 0.00143847, 0.00164976, 0.00133623,
       0.00152867, 0.00163098, 0.00150044, 0.00146595, 0.00138352,
       0.00171312, 0.00159301, 0.00141333, 0.00160952, 0.00169525,
       0.00135636, 0.00174231, 0.00173384, 0.00177322, 0.00179577,
       0.00160169, 0.00155509, 0.00169374, 0.00194605, 0.00185258,
       0.00167893, 0.00174209, 0.00168542, 0.00168466, 0.00165164,
       0.00144223, 0.00138013, 0.00174263, 0.00165603, 0.0017723 ,
       0.00157285, 0.001434  , 0.00175558, 0.00179197, 0.00155355,
       0.00165686, 0.00146524, 0.0011023 , 0.00146026, 0.00179751,
       0.0017912 , 0.00178034, 0.0017676 , 0.00179925, 0.00174215,
       0.00170037, 0.00180046, 0.00158382, 0.00117277, 0.0017845 ,
       0.00149119, 0.0019809 , 0.00167651, 0.00175266, 0.00183486,
       0.00170419, 0.00176214, 0.00158203, 0.00154721, 0.00181863,
       0.00180575, 0.00160277, 0.00159925, 0.00172617, 0.00184475,
       0.00162263, 0.0014124 , 0.00168676, 0.00154159, 0.00157885,
       0.00158769, 0.0015503 , 0.00120815, 0.0017102 , 0.00169737,
       0.00187968, 0.00167241, 0.00153863, 0.0017181 , 0.00159638,
       0.00138459, 0.00175005, 0.00177162, 0.00177911, 0.00179582,
       0.0016201 , 0.00186446, 0.00191332, 0.00176476, 0.00184129,
       0.00171187, 0.00201582, 0.00164321, 0.00157193, 0.00167119,
       0.00179475, 0.00178338, 0.00191761, 0.00159741, 0.00168113,
       0.00149185, 0.00166876, 0.00171037, 0.00150467, 0.00163548,
       0.00111339, 0.001731  , 0.00177668, 0.00177822, 0.00177374,
       0.0017673 , 0.00175309, 0.00178382, 0.00172027, 0.00169401,
       0.00178493, 0.00151542, 0.00178038, 0.00187411, 0.00164485,
       0.00170401, 0.00191723, 0.00167584, 0.00172966, 0.00181868,
       0.0016283 , 0.00164477, 0.00192095, 0.00189466, 0.00164085,
       0.00162012, 0.00179501, 0.00183228, 0.00169913, 0.00143424,
       0.00179565, 0.00156742, 0.00149531, 0.00155802, 0.0016888 ,
       0.00169408, 0.00166341, 0.00171022, 0.00156363, 0.00173228,
       0.00167335, 0.00154896, 0.00176201, 0.00177443, 0.00178163,
       0.00178036, 0.00179992, 0.00177692, 0.00189127, 0.0020179 ,
       0.00163137, 0.00168958, 0.00179982, 0.0016471 , 0.00142331,
       0.00164504, 0.00163292, 0.00172678, 0.00171398, 0.00179531,
       0.00204734, 0.00179519, 0.00159151, 0.00156982, 0.00173778,
       0.0018107 , 0.00147248, 0.00166127, 0.00140121, 0.00181511,
       0.00174355, 0.00175256, 0.00174104, 0.00175353, 0.00176034,
       0.00174014, 0.00175216, 0.00172473, 0.00175241, 0.00177007,
       0.00177225, 0.00174127, 0.00185159, 0.00185854, 0.00171616,
       0.00167479, 0.00174241, 0.00156333, 0.00170248, 0.00176223,
       0.00179594, 0.00192178, 0.00165974, 0.00164667, 0.00190148,
       0.00187446, 0.00170516, 0.00165348, 0.00182744, 0.00169768,
       0.00181515, 0.00145449, 0.0020665 , 0.00157021, 0.00153508,
       0.00170362, 0.00164145, 0.0016491 , 0.00168276, 0.00170938,
       0.00172109, 0.00163463, 0.00174129, 0.00171995, 0.00165166,
       0.0017581 , 0.00178222, 0.00178401, 0.00179045, 0.00176829,
       0.00159753, 0.00188162, 0.00180487, 0.00147494, 0.00164675,
       0.00167162, 0.00182493, 0.00205388, 0.00199688, 0.00157268,
       0.00151371, 0.00160066, 0.00208526, 0.00183017, 0.00190727,
       0.00163965, 0.00164073, 0.00218953, 0.00194725, 0.00145512,
       0.00137885, 0.00177869, 0.00196061, 0.00168747, 0.00169761,
       0.00171251, 0.00169894, 0.00173147, 0.001753  , 0.00175036,
       0.00173648, 0.00175426, 0.00177507, 0.00173785, 0.00174177,
       0.00172817, 0.00157436, 0.00182059, 0.00187288, 0.00164811,
       0.00172199, 0.00172755, 0.00176894, 0.00170317, 0.00182198,
       0.0019835 , 0.00205979, 0.00167477, 0.00166307, 0.00179148,
       0.00160448, 0.00176547, 0.0016912 , 0.00138561, 0.00166647,
       0.0019209 , 0.00148048, 0.00152142, 0.00196829, 0.00136659,
       0.00170202, 0.00171936, 0.00173085, 0.00171865, 0.00170108,
       0.00173423, 0.00170425, 0.00172947, 0.00176486, 0.00176627,
       0.00179725, 0.00180999, 0.00175762, 0.00144951, 0.00178558,
       0.00166357, 0.00116131, 0.00139892, 0.00183855, 0.00188152,
       0.00186604, 0.00203128, 0.00187511, 0.00154001, 0.00120389,
       0.00139441, 0.00158599, 0.00204782, 0.00205489, 0.00200906,
       0.00186243, 0.00195654, 0.00176273, 0.0021076 , 0.00196836,
       0.00141359, 0.00191779, 0.00163351, 0.00163242, 0.0016378 ,
       0.00166755, 0.0016626 , 0.00171315, 0.00174194, 0.00175681,
       0.00172789, 0.00178603, 0.00172489, 0.00170401, 0.00174443,
       0.00168592, 0.00122984, 0.00168833, 0.00190599, 0.00163201,
       0.00178064, 0.001697  , 0.00157675, 0.00173012, 0.00173894,
       0.00170524, 0.00185542, 0.00210964, 0.00189706, 0.00167955,
       0.00168377, 0.00163948, 0.00163703, 0.00172681, 0.00180016,
       0.00149675, 0.00151517, 0.00169577, 0.00188652, 0.00196509,
       0.00160535, 0.00156384, 0.00174158, 0.00164575, 0.00173005,
       0.0017276 , 0.00172093, 0.00170703, 0.0016936 , 0.00171494,
       0.00173109, 0.00169866, 0.00168947, 0.00179293, 0.00170792,
       0.00181558, 0.00179808, 0.00176046, 0.00141357, 0.00177774,
       0.00183969, 0.00194322, 0.00184551, 0.00187884, 0.00207237,
       0.00131841, 0.00185597, 0.00186987, 0.00158712, 0.00167451,
       0.00195408, 0.00212344, 0.00207014, 0.00197286, 0.0019054 ,
       0.00202643, 0.00199103, 0.00183304, 0.00192114, 0.00167586,
       0.00159054, 0.00156848, 0.00157375, 0.00157328, 0.00161767,
       0.00159695, 0.00168666, 0.00174738, 0.00170731, 0.00175673,
       0.00174479, 0.00172401, 0.00169399, 0.00174544, 0.00170316,
       0.00163264, 0.0019164 , 0.00166543, 0.0017666 , 0.00173903,
       0.00173708, 0.00169533, 0.00171013, 0.00170335, 0.00169312,
       0.00169659, 0.00168333, 0.00168428, 0.00165187, 0.0016924 ,
       0.00170762, 0.00169062, 0.00171137, 0.00174501, 0.00179722,
       0.00162675, 0.00175668, 0.00186696, 0.00173845, 0.00178299,
       0.00170685, 0.00185263, 0.00172573, 0.00173779, 0.00173875,
       0.00169143, 0.00163268, 0.00169612, 0.001713  , 0.00190183,
       0.00163134, 0.00165442, 0.00184584, 0.0017004 , 0.00179183,
       0.00172748, 0.00172458, 0.00161848, 0.001832  , 0.00187168,
       0.00158058, 0.00180208, 0.00183236, 0.00159213, 0.00184703,
       0.00190979, 0.0015545 , 0.00145698, 0.00172536, 0.00190591,
       0.00237308, 0.00211049, 0.00186876, 0.00179561, 0.00224917,
       0.00175561, 0.00191213, 0.00158922, 0.0018042 , 0.0015593 ,
       0.00159423, 0.00145448, 0.00148514, 0.00146579, 0.00155805,
       0.001623  , 0.00167204, 0.00169893, 0.00170078, 0.00175372,
       0.00176022, 0.00174633, 0.00175055, 0.00173104, 0.00171517,
       0.00171383, 0.00184919, 0.00196404, 0.00177009, 0.00165416,
       0.00172071, 0.00169294, 0.00172398, 0.00176102, 0.00162047,
       0.00170388, 0.00169922, 0.00170167, 0.00166163, 0.00169979,
       0.0015874 , 0.00156587, 0.00165238, 0.00175003, 0.0017032 ,
       0.00170936, 0.00178068, 0.00165156, 0.0017222 , 0.00174765,
       0.0017596 , 0.00181676, 0.00172491, 0.00170253, 0.00158102,
       0.00185878, 0.00173281, 0.00168086, 0.00176745, 0.00186563,
       0.00160958, 0.00151891, 0.0020227 , 0.00154548, 0.00194058,
       0.00155166, 0.00167849, 0.00179249, 0.00173575, 0.0016645 ,
       0.00162973, 0.00168316, 0.00141259, 0.00195024, 0.0018389 ,
       0.00177395, 0.00182084, 0.0018177 , 0.00187262, 0.00184664,
       0.00176718, 0.00164075, 0.00183652, 0.00263525, 0.00187251,
       0.00177335, 0.00193725, 0.00209009, 0.00173831, 0.00136507,
       0.00180142, 0.00161729, 0.00174356, 0.00146203, 0.00172771,
       0.00198233, 0.00125661, 0.00132762, 0.0015463 , 0.00185925,
       0.00170866, 0.0016686 , 0.00177724, 0.00175977, 0.00178194,
       0.00177849, 0.00176109, 0.00171254, 0.00170485, 0.00192197,
       0.00189056, 0.00181383, 0.00165671, 0.00167214, 0.00165844,
       0.00166057, 0.00162956, 0.00174405, 0.00177656, 0.00159339,
       0.00170827, 0.00170514, 0.0016974 , 0.00170411, 0.00163794,
       0.0016871 , 0.00159273, 0.00167705, 0.00181658, 0.0018235 ,
       0.00171795, 0.00181881, 0.00167774, 0.00176612, 0.001661  ,
       0.00174971, 0.00168133, 0.00184977, 0.00134677, 0.00175964,
       0.00184958, 0.00175   , 0.00168247, 0.00162883, 0.00189458,
       0.00198519, 0.00162371, 0.00190216, 0.00133246, 0.00179653,
       0.00166472, 0.00171257, 0.0017236 , 0.00150778, 0.00166532,
       0.0016067 , 0.00183288, 0.00179304, 0.00174278, 0.00176574,
       0.00173411, 0.00173836, 0.0017099 , 0.00163976, 0.0017631 ,
       0.00176753, 0.00181141, 0.00205751, 0.00192128, 0.00212418,
       0.00163055, 0.00160196, 0.00186909, 0.00184181, 0.00190932,
       0.00172341, 0.00181229, 0.00156889, 0.00118913, 0.00183877,
       0.00192889, 0.00094002, 0.00132324, 0.00177209, 0.00169613,
       0.0016626 , 0.00178981, 0.00179891, 0.00175288, 0.00179145,
       0.00178399, 0.0017402 , 0.00168172, 0.00168012, 0.00170962,
       0.00179304, 0.00158934, 0.00164744, 0.00166804, 0.00167003,
       0.00161919, 0.00167253, 0.00164258, 0.00183116, 0.00178357,
       0.00156779, 0.00171828, 0.00172781, 0.00171894, 0.00169516,
       0.0017231 , 0.00161165, 0.00171453, 0.00169849, 0.00189192,
       0.00185688, 0.00165833, 0.00183326, 0.00189661, 0.00198562,
       0.00166381, 0.00174535, 0.00160512, 0.00197914, 0.00159389,
       0.0018748 , 0.00179373, 0.00169368, 0.00162761, 0.0016152 ,
       0.0018906 , 0.00174464, 0.00177841, 0.00179389, 0.00188819,
       0.001722  , 0.00202072, 0.00178714, 0.00153672, 0.00168033,
       0.00169815, 0.00153661, 0.00163428, 0.00166662, 0.001485  ,
       0.0016617 , 0.00165342, 0.00171755, 0.00178673, 0.00164063,
       0.00163811, 0.00180772, 0.00155103, 0.00178027, 0.00174725,
       0.00170934, 0.00179597, 0.00167681, 0.00184148, 0.00164823,
       0.00149957, 0.00169073, 0.00123504, 0.00182197, 0.0018103 ,
       0.00172602, 0.00166615, 0.00165814, 0.00180555, 0.00179456,
       0.0018035 , 0.00175442, 0.0017881 , 0.00175679, 0.00168915,
       0.00168314, 0.00161646, 0.0015596 , 0.00173053, 0.00141575,
       0.00165264, 0.00162796, 0.00175554, 0.00164342, 0.00143868,
       0.00202332, 0.00167891, 0.00186498, 0.00177882, 0.00153868,
       0.00174321, 0.00176638, 0.00179995, 0.00175265, 0.00168681,
       0.00171169, 0.00168706, 0.00172551, 0.00193076, 0.00186429,
       0.00163103, 0.00180348, 0.00201974, 0.00214567, 0.00153293,
       0.00174264, 0.00170665, 0.00143423, 0.00161945, 0.00152074,
       0.00144747, 0.00167488, 0.00175666, 0.00169963, 0.0016092 ,
       0.00157966, 0.00176591, 0.00173882, 0.00180783, 0.0017558 ,
       0.00196601, 0.00188413, 0.00179709, 0.00165631, 0.00155886,
       0.00164405, 0.00157103, 0.00165929, 0.00159906, 0.00172714,
       0.00128166, 0.00172161, 0.00167294, 0.00176317, 0.00182921,
       0.00166363, 0.00155636, 0.00166357, 0.00170108, 0.00169571,
       0.00166129, 0.00173177, 0.00163176, 0.00154279, 0.00173037,
       0.00139417, 0.00165462, 0.00175637, 0.00179645, 0.00175326,
       0.0016706 , 0.00165778, 0.00172053, 0.00181961, 0.00181132,
       0.00179171, 0.00180221, 0.00176317, 0.00177361, 0.00171113,
       0.0016043 , 0.00170821, 0.00142093, 0.00161014, 0.00170808,
       0.00188222, 0.00163592, 0.00190085, 0.00183624, 0.00173762,
       0.00171069, 0.00200335, 0.00170966, 0.00182901, 0.00175523,
       0.00154881, 0.0017599 , 0.00178908, 0.00188748, 0.00175423,
       0.00173422, 0.00164127, 0.00172687, 0.00189585, 0.0017566 ,
       0.00187203, 0.00163843, 0.00188826, 0.00178951, 0.00182937,
       0.00172646, 0.00179963, 0.00156056, 0.00173886, 0.00166729,
       0.00203738, 0.00135448, 0.00179989, 0.00165181, 0.00168816,
       0.00158712, 0.00148453, 0.00170446, 0.00194924, 0.00173803,
       0.00167522, 0.00159447, 0.00148239, 0.0015006 , 0.00165324,
       0.00165374, 0.00165617, 0.00185485, 0.00174533, 0.00174919,
       0.00175409, 0.00169769, 0.00176689, 0.00193299, 0.0018923 ,
       0.00182183, 0.001945  , 0.0017634 , 0.00178685, 0.0017009 ,
       0.00165379, 0.00212156, 0.00162268, 0.00160594, 0.00170667,
       0.00175912, 0.0017543 , 0.00178439, 0.00177444, 0.0016895 ,
       0.00155953, 0.00161504, 0.0018718 , 0.00181575, 0.00181595,
       0.0018101 , 0.00181129, 0.00178331, 0.00179586, 0.0017373 ,
       0.00166904, 0.0015874 , 0.00170126, 0.00167701, 0.00185145,
       0.00166696, 0.00180755, 0.00180578, 0.00185251, 0.00174449,
       0.00182414, 0.00183206, 0.00172415, 0.00175345, 0.00171527,
       0.00161708, 0.00176868, 0.0017889 , 0.00187743, 0.00175853,
       0.00159582, 0.00177553, 0.00187264, 0.00180969, 0.0019061 ,
       0.00189923, 0.00167306, 0.00217105, 0.00180599, 0.00183706,
       0.00165753, 0.00188408, 0.00181079, 0.00155082, 0.00134833,
       0.00158484, 0.00163825, 0.00167848, 0.00148297, 0.00168353,
       0.00170637, 0.00168959, 0.00164462, 0.00146521, 0.00147056,
       0.00120597, 0.00152819, 0.00172384, 0.00183451, 0.00161216,
       0.00177002, 0.00185926, 0.00179588, 0.00181692, 0.00180017,
       0.00175416, 0.00168356, 0.00177796, 0.00195228, 0.00195068,
       0.00200303, 0.00211588, 0.00184533, 0.0018269 , 0.0016992 ,
       0.00166251, 0.00194062, 0.00166986, 0.00170873, 0.00159847,
       0.00168137, 0.0015821 , 0.00180066, 0.00179262, 0.00175112,
       0.00160046, 0.00132477, 0.00192337, 0.00188504, 0.00183838,
       0.00181861, 0.0018068 , 0.00179186, 0.00183408, 0.00174896,
       0.00171345, 0.00168504, 0.00147094, 0.001598  , 0.00186045,
       0.00172976, 0.0017168 , 0.00167539, 0.00181135, 0.00177368,
       0.00185612, 0.00176351, 0.00172017, 0.00178229, 0.00176503,
       0.00167929, 0.00170713, 0.00179527, 0.00180356, 0.00183899,
       0.00174934, 0.0015831 , 0.00188435, 0.00180801, 0.00194724,
       0.00180779, 0.00172924, 0.00187106, 0.00189603, 0.00173415,
       0.00184563, 0.0018554 , 0.00185506, 0.00215399, 0.0017958 ,
       0.00162843, 0.00140587, 0.00158192, 0.00149725, 0.00163981,
       0.00166158, 0.00164605, 0.00166381, 0.00167857, 0.00177758,
       0.00162415, 0.00151239, 0.00139867, 0.00162214, 0.00173963,
       0.00171961, 0.00145113, 0.00181267, 0.00180067, 0.00184695,
       0.0018353 , 0.00182025, 0.00177989, 0.00161796, 0.00177938,
       0.00159602, 0.00204406, 0.00201527, 0.00200032, 0.00181711,
       0.00172072, 0.00169825, 0.00167811, 0.00188775, 0.00187245,
       0.00172578, 0.00173014, 0.00153518, 0.0016909 , 0.00162526,
       0.00192467, 0.00181878, 0.00180415, 0.00175933, 0.00161699,
       0.00185258, 0.00179247, 0.00183652, 0.00182773, 0.00180199,
       0.00178741, 0.00176116, 0.00172558, 0.0017131 , 0.00122021,
       0.00155196, 0.00175855, 0.00176451, 0.00167508, 0.00166357,
       0.00164538, 0.00181797, 0.00178282, 0.00188907, 0.0017439 ,
       0.00168364, 0.001632  , 0.00177907, 0.0018968 , 0.00186405,
       0.00184545, 0.00181937, 0.0018045 , 0.00175151, 0.00160277,
       0.00181587, 0.00164887, 0.00178477, 0.00176825, 0.0016854 ,
       0.00168188, 0.00155169, 0.00160969, 0.00192499, 0.00190553,
       0.00185504, 0.00188647, 0.00177172, 0.00163639, 0.00139321,
       0.00161166, 0.00153639, 0.00164949, 0.00162046, 0.00156438,
       0.00167637, 0.00164544, 0.00164297, 0.00187715, 0.00160928,
       0.00166061, 0.00185728, 0.00171738, 0.00169738, 0.00150672,
       0.00180649, 0.00163764, 0.00175629, 0.00185432, 0.00183574,
       0.00179894, 0.00176813, 0.00162864, 0.00179715, 0.00128786,
       0.00238987, 0.00214079, 0.0018457 , 0.00178585, 0.00153744,
       0.00194779, 0.00170601, 0.00173128, 0.00171682, 0.00221016,
       0.00189137, 0.00175025, 0.00175099, 0.00175959, 0.00155693,
       0.00125478, 0.00154657, 0.00183429, 0.00175368, 0.0018236 ,
       0.00182731, 0.00181141, 0.00178977, 0.00181618, 0.00174552,
       0.00181922, 0.00180318, 0.00177972, 0.00175576, 0.0017298 ,
       0.00171092, 0.00160771, 0.00165236, 0.00171373, 0.00175228,
       0.00166582, 0.00166453, 0.0016744 , 0.00170705, 0.00182424,
       0.0017835 , 0.00197098, 0.00166972, 0.00173381, 0.00159317,
       0.00178144, 0.00213037, 0.00187293, 0.00182347, 0.00179631,
       0.0016658 , 0.00176347, 0.00158789, 0.00171463, 0.00178821,
       0.00106525, 0.00191212, 0.00146023, 0.00172014, 0.00183729,
       0.00161691, 0.00197463, 0.00176044, 0.00189   , 0.001813  ,
       0.00185783, 0.00150015, 0.00169076, 0.00149311, 0.00167379,
       0.00166192, 0.00160202, 0.00142571, 0.00168434, 0.00164331,
       0.00165829, 0.00165885, 0.00182846, 0.00169131, 0.00183179,
       0.00169728, 0.00168163, 0.00163496, 0.00178411, 0.00177567,
       0.00183148, 0.00183495, 0.00180975, 0.00177147, 0.00173892,
       0.00175687, 0.00166612, 0.00199102, 0.00200059, 0.00130613,
       0.00177727, 0.00171762, 0.00178848, 0.0018051 , 0.00174296,
       0.00172217, 0.00193268, 0.00182788, 0.00172921, 0.00180856,
       0.00172206, 0.00172423, 0.00166254, 0.00144175, 0.00170063,
       0.00174852, 0.00182118, 0.00183769, 0.00177688, 0.0017667 ,
       0.0017857 , 0.00173666, 0.00179317, 0.00177383, 0.00174692,
       0.0017193 , 0.00169705, 0.00169075, 0.001589  , 0.0016374 ,
       0.00175401, 0.00170025, 0.00163471, 0.0016801 , 0.00171004,
       0.00160135, 0.0019499 , 0.00183575, 0.00175434, 0.0020548 ,
       0.00167214, 0.00158195, 0.00174339, 0.00174493, 0.00149927,
       0.0018554 , 0.00188211, 0.00188562, 0.00154771, 0.00146666,
       0.00175712, 0.00160231, 0.00182009, 0.00170522, 0.00166995,
       0.00181116, 0.001866  , 0.00191056, 0.00197218, 0.00188779,
       0.00168608, 0.0017158 , 0.00182243, 0.00188709, 0.00184431,
       0.00182572, 0.00173062, 0.00149246, 0.00164934, 0.00168902,
       0.00172081, 0.0014781 , 0.00106079, 0.00167532, 0.00159229,
       0.00166208, 0.00169459, 0.00178376, 0.00170215, 0.00181074,
       0.00168149, 0.00163921, 0.00181041, 0.00181014, 0.00181489,
       0.00183348, 0.00182536, 0.00177397, 0.00189993, 0.00199586,
       0.00171957, 0.00200549, 0.00171725, 0.00184893, 0.00185318,
       0.00190135, 0.00193297, 0.00202954, 0.00172801, 0.00170512,
       0.00176012, 0.00171581, 0.00164335, 0.00183857, 0.00181805,
       0.00167837, 0.00163367, 0.00157254, 0.00183065, 0.00162356,
       0.00176493, 0.00138732, 0.00168158, 0.00171638, 0.00180492,
       0.0017373 , 0.00175936, 0.00171988, 0.00174197, 0.0017142 ,
       0.00176627, 0.00174057, 0.00170841, 0.00167607, 0.00175423,
       0.00132074, 0.00163281, 0.00189801, 0.00163282, 0.00160955,
       0.001683  , 0.00183559, 0.00179029, 0.00141064, 0.00189912,
       0.00175907, 0.00179779, 0.00182461, 0.0016298 , 0.0017047 ,
       0.00173428, 0.00170908, 0.00163546, 0.00179574, 0.00188103,
       0.00190141, 0.00185901, 0.00135986, 0.00160536, 0.00139257,
       0.00183371, 0.00196951, 0.00170782, 0.00182694, 0.00181253,
       0.00194486, 0.0017728 , 0.00177043, 0.00176244, 0.00183009,
       0.00165264, 0.00167981, 0.0015913 , 0.00162649, 0.00155772,
       0.001634  , 0.00169409, 0.00171191, 0.00167517, 0.00176494,
       0.00167409, 0.00159986, 0.00181929, 0.00186732, 0.00182305,
       0.00184333, 0.00183194, 0.00175785, 0.00172706, 0.00177137,
       0.00170731, 0.00171888, 0.00163227, 0.0019802 , 0.00162906,
       0.00192237, 0.00183733, 0.00192607, 0.00155462, 0.00197491,
       0.00173042, 0.00164788, 0.00147864, 0.00180378, 0.00182782,
       0.00179873, 0.00162346, 0.00153297, 0.0015516 , 0.00104397,
       0.00176506, 0.00138028, 0.00163713, 0.00165291, 0.00168065,
       0.00169071, 0.00166858, 0.00171577, 0.0016812 , 0.00171066,
       0.00169443, 0.00172952, 0.00169429, 0.00164976, 0.00199205,
       0.0021404 , 0.00207602, 0.00159224, 0.00166878, 0.0016133 ,
       0.0018562 , 0.00218157, 0.0017336 , 0.0016642 , 0.00169572,
       0.00174282, 0.0017217 , 0.0016684 , 0.00167767, 0.00171439,
       0.00169678, 0.00169531, 0.00164189, 0.00173673, 0.00159293,
       0.00162888, 0.00170837, 0.00168083, 0.00147544, 0.00195681,
       0.00174687, 0.00181762, 0.00171655, 0.00190158, 0.00154117,
       0.0017719 , 0.00172241, 0.00163834, 0.00155543, 0.00136761,
       0.00143857, 0.00160955, 0.00158591, 0.00166374, 0.0016747 ,
       0.00162758, 0.00167656, 0.00167886, 0.00159488, 0.00183968,
       0.00193196, 0.00182157, 0.00182895, 0.00179279, 0.00171698,
       0.00166032, 0.00172935, 0.00160285, 0.00165888, 0.00170909,
       0.00161338, 0.0017657 , 0.00169624, 0.00179931, 0.00188953,
       0.00163814, 0.00175179, 0.00160995, 0.00187292, 0.00193262,
       0.00171988, 0.00155721, 0.00122127, 0.00182731, 0.00178029,
       0.00182624, 0.00180817, 0.00164276, 0.00156712, 0.00163187,
       0.00173164, 0.00146015, 0.00162504, 0.00160766, 0.0016524 ,
       0.00166057, 0.00163451, 0.00163983, 0.00167597, 0.00165769,
       0.00167884, 0.00167206, 0.00163536, 0.00193685, 0.00185861,
       0.00191051, 0.00167594, 0.00165695, 0.00170689, 0.0016294 ,
       0.00179369, 0.00169782, 0.00170365, 0.00172236, 0.0017015 ,
       0.00170039, 0.00167033, 0.00166779, 0.00167559, 0.00166775,
       0.00165744, 0.00167989, 0.00162421, 0.00160243, 0.00159608,
       0.00155432, 0.00166969, 0.0019256 , 0.0017232 , 0.0018822 ,
       0.00181039, 0.00135222, 0.00195345, 0.00128699, 0.00179333,
       0.0017125 , 0.0013284 , 0.00107987, 0.00145562, 0.00164764,
       0.00176523, 0.00152538, 0.0017009 , 0.0015896 , 0.00169062,
       0.00167212, 0.00160605, 0.00180939, 0.00161086, 0.0017907 ,
       0.00177505, 0.00175527, 0.00171513, 0.0017387 , 0.00166002,
       0.00176725, 0.00167949, 0.00162906, 0.00169694, 0.00167204,
       0.00170862, 0.00162692, 0.00173506, 0.00214966, 0.00154238,
       0.00194599, 0.00158671, 0.00186095, 0.00190515, 0.00174308,
       0.00163117, 0.00180838, 0.00182614, 0.00175867, 0.00182835,
       0.00180017, 0.00164647, 0.00162758, 0.00173709, 0.00180091,
       0.0016295 , 0.00158562, 0.00162548, 0.00164812, 0.00162453,
       0.00161517, 0.00161196, 0.0016631 , 0.00162236, 0.00166994,
       0.00182227, 0.00177367, 0.00127513, 0.00168523, 0.0016283 ,
       0.00166199, 0.00156545, 0.00163898, 0.00156305, 0.00162593,
       0.00176521, 0.00172581, 0.00178838, 0.00174115, 0.00165083,
       0.00164648, 0.00167249, 0.00166246, 0.00161065, 0.00157775,
       0.00166741, 0.00144336, 0.00155527, 0.00160464, 0.00177765,
       0.001626  , 0.00185148, 0.00202526, 0.00178722, 0.00209636,
       0.00179893, 0.0017297 , 0.00141438, 0.00160735, 0.00178993,
       0.00162666, 0.0017268 , 0.00151418, 0.00157377, 0.00168125,
       0.00167682, 0.00160719, 0.00172435, 0.00161472, 0.00157036,
       0.00174419, 0.00175108, 0.00174727, 0.00172321, 0.00160131,
       0.00169471, 0.00142659, 0.00141025, 0.00174159, 0.00163859,
       0.0017578 , 0.00162606, 0.00199605, 0.00173955, 0.00192849,
       0.0019266 , 0.00188944, 0.00191065, 0.00184789, 0.00181232,
       0.00177231, 0.00183351, 0.00185828, 0.00179504, 0.00176227,
       0.00165066, 0.00164199, 0.00121062, 0.00167525, 0.0016293 ,
       0.0016226 , 0.0016485 , 0.00166102, 0.001638  , 0.00162118,
       0.00167906, 0.00170401, 0.00182423, 0.00188364, 0.00182976,
       0.00162621, 0.00163809, 0.00153952, 0.0016784 , 0.0015243 ,
       0.00166183, 0.00157842, 0.00164685, 0.00174083, 0.00183834,
       0.00168752, 0.00183504, 0.00174288, 0.00159824, 0.00161692,
       0.00178041, 0.00165311, 0.00134282, 0.0015735 , 0.00168253,
       0.00160828, 0.00171489, 0.00171138, 0.00184667, 0.00144626,
       0.00183462, 0.00177065, 0.00149514, 0.0018702 , 0.00165631,
       0.00176048, 0.00172124, 0.0016404 , 0.00170618, 0.00155163,
       0.00167565, 0.00170697, 0.0012386 , 0.0016813 , 0.00179337,
       0.00165031, 0.00172104, 0.00174423, 0.00174206, 0.00170287,
       0.00159889, 0.00167327, 0.00175656, 0.00165716, 0.00169084,
       0.00177441, 0.00185478, 0.0015704 , 0.00204068, 0.00159743,
       0.00179971, 0.00174043, 0.00169787, 0.00178632, 0.00197809,
       0.00175149, 0.00173005, 0.00164486, 0.0018213 , 0.00151064,
       0.00168884, 0.00155982, 0.00171515, 0.00173503, 0.001705  ,
       0.00170434, 0.00172347, 0.00174135, 0.00194064, 0.00208343,
       0.00181193, 0.00165596, 0.00135207, 0.00164385, 0.00174023,
       0.00160053, 0.00154804, 0.00175463, 0.00155364, 0.00155753,
       0.00163136, 0.00182729, 0.00170418, 0.00187579, 0.00179785,
       0.00180429, 0.00167909, 0.0015486 , 0.00174341, 0.00188405,
       0.00186345, 0.00185305, 0.00165849, 0.00158497, 0.00165077,
       0.00183702, 0.00165111, 0.00167111, 0.00176586, 0.00174603,
       0.00171217, 0.00181776, 0.00183851, 0.00130121, 0.0016968 ,
       0.00173507, 0.00178709, 0.00170882, 0.00171671, 0.0017176 ,
       0.00173136, 0.00171215, 0.00167755, 0.00158294, 0.00166777,
       0.00235629, 0.00163494, 0.00175024, 0.00181002, 0.00153167,
       0.00186643, 0.00168636, 0.00178712, 0.00174499, 0.00164017,
       0.00149632, 0.00175821, 0.00170549, 0.00175229, 0.00171992,
       0.00181606, 0.00188681, 0.00165687, 0.00158522, 0.00179611,
       0.00184079, 0.00184994, 0.00177244, 0.00176939, 0.00176474,
       0.00230138, 0.00176421, 0.00163665, 0.00150082, 0.00164239,
       0.00116036, 0.00166972, 0.00181759, 0.00197208, 0.0015926 ,
       0.00158316, 0.0015601 , 0.00157897, 0.00164861, 0.00161577,
       0.00175063, 0.0016797 , 0.00147802, 0.0017414 , 0.00161654,
       0.0016593 , 0.00171144, 0.00188674, 0.00180835, 0.00174532,
       0.00171745, 0.00160455, 0.00165196, 0.00168377, 0.0017537 ,
       0.00162771, 0.00144325, 0.00175956, 0.00171944, 0.00178417,
       0.00154173, 0.00171337, 0.00174567, 0.00156206, 0.00168497,
       0.00154873, 0.00165622, 0.00168017, 0.00164577, 0.00163317,
       0.00160989, 0.00172431, 0.0012683 , 0.00172373, 0.0017163 ,
       0.00175552, 0.00168204, 0.00179132, 0.00177093, 0.00176923,
       0.00172058, 0.00163811, 0.00176976, 0.00165585, 0.00159636,
       0.00175835, 0.00171082, 0.00181416, 0.00161514, 0.00174399,
       0.00189192, 0.00183638, 0.00200246, 0.0017974 , 0.00178579,
       0.00178299, 0.00200451, 0.00112783, 0.00176052, 0.00139578,
       0.00164895, 0.0016955 , 0.00158085, 0.00174389, 0.0012631 ,
       0.00168693, 0.00172993, 0.0022272 , 0.00159353, 0.0016    ,
       0.00160725, 0.00164228, 0.00167351, 0.00153067, 0.00170303,
       0.0016888 , 0.00167566, 0.00155568, 0.00149318, 0.00171758,
       0.0016198 , 0.00148056, 0.001832  , 0.00179208, 0.00178118,
       0.00168109, 0.00165873, 0.00166097, 0.00171708, 0.00179169,
       0.00172534, 0.0017042 , 0.00171865, 0.00172021, 0.00171127,
       0.00172279, 0.00176265, 0.00150903, 0.00175614, 0.00147946,
       0.00162118, 0.00163419, 0.00158593, 0.00154885, 0.00163107,
       0.00162708, 0.0017079 , 0.00178537, 0.00173874, 0.00176834,
       0.00179922, 0.00177919, 0.00176788, 0.00177073, 0.0017863 ,
       0.00169979, 0.00174206, 0.00167405, 0.00175163, 0.00149142,
       0.00163009, 0.00168262, 0.00187678, 0.00194273, 0.00183439,
       0.0018053 , 0.0017874 , 0.00179302, 0.00177045, 0.00133992,
       0.00163351, 0.00175334, 0.0017385 , 0.00176029, 0.00179322,
       0.00132359, 0.00171668, 0.00157069, 0.00172338, 0.00155885,
       0.00175998, 0.00169792, 0.00189468, 0.00160164, 0.00159736,
       0.00161366, 0.00166524, 0.00169267, 0.00175777, 0.00166943,
       0.00163987, 0.00181789, 0.00157875, 0.00120634, 0.00177652,
       0.00176395, 0.00150058, 0.00167245, 0.00175205, 0.00178056,
       0.00150659, 0.00172771, 0.00175163, 0.00173365, 0.00174898,
       0.00177347, 0.00159313, 0.00184377, 0.00150772, 0.00163764,
       0.00176337, 0.0017781 , 0.00127704, 0.00151833, 0.00167796,
       0.00182049, 0.00159352, 0.00162867, 0.00179083, 0.00148615,
       0.00114499, 0.00180172, 0.00184192, 0.00182959, 0.00179412,
       0.00179901, 0.00179663, 0.00173121, 0.0017468 , 0.00177339,
       0.00169461, 0.00162085, 0.00185875, 0.00186657, 0.00183488,
       0.00181229, 0.00179392, 0.00180833, 0.00176765, 0.00146162,
       0.00187591, 0.00173434, 0.00166975, 0.0018024 , 0.00147153,
       0.00175995, 0.00153621, 0.00152631, 0.00158103, 0.00155455,
       0.00159752, 0.0016937 , 0.00161662, 0.00145255, 0.00169127,
       0.00162233, 0.00160872, 0.0016435 , 0.00168917, 0.00181049,
       0.00173288, 0.00172333, 0.00184836, 0.00156379, 0.00176754,
       0.00165591, 0.00151092, 0.00181503, 0.00185642, 0.00179055,
       0.0017971 , 0.00187158, 0.00175674, 0.0017637 , 0.00167792,
       0.00186962, 0.0016396 , 0.00161067, 0.00168649, 0.00167728,
       0.00148382, 0.00162712, 0.0017238 , 0.00171283, 0.00144794,
       0.0016174 , 0.00145828, 0.00145602, 0.00160981, 0.00187327,
       0.0016926 , 0.00176313, 0.00180238, 0.0018133 , 0.00180319,
       0.00179023, 0.00181946, 0.00178487, 0.00179945, 0.00178711,
       0.00170894, 0.00163661, 0.00182866, 0.00182912, 0.00183541,
       0.00181009, 0.00179706, 0.00181874, 0.00150899, 0.00188305,
       0.00122483, 0.0015479 , 0.0017591 , 0.00164635, 0.00174763,
       0.00176642, 0.00155492, 0.00155236, 0.0016572 , 0.00159544,
       0.00173045, 0.00141241, 0.00162152, 0.00161882, 0.00158008,
       0.00157286, 0.00145384, 0.00174519, 0.00170782, 0.00165451,
       0.00158592, 0.0015591 , 0.00166148, 0.00183105, 0.00175944,
       0.00177886, 0.0018911 , 0.00171416, 0.00196096, 0.0016282 ,
       0.00180051, 0.00188244, 0.00171279, 0.00177385, 0.00155226,
       0.00180335, 0.00169705, 0.00172303, 0.00178293, 0.00173867,
       0.00101393, 0.00162161, 0.00166921, 0.0017075 , 0.00170327,
       0.00140088, 0.00149411, 0.00166121, 0.00139501, 0.00154127,
       0.00166182, 0.00174239, 0.00169362, 0.00177444, 0.00178003,
       0.00179912, 0.00191738, 0.00204336, 0.00181537, 0.00181927,
       0.00177833, 0.00172277, 0.00183677, 0.00180416, 0.00182976,
       0.00178418, 0.00179736, 0.00183111, 0.00162503, 0.00185682,
       0.00161249, 0.00174459, 0.00180593, 0.00146983, 0.00159848,
       0.00173149, 0.00158527, 0.00154508, 0.00142499, 0.00166977,
       0.00162371, 0.00147994, 0.00152754, 0.00140381, 0.00168553,
       0.00161583, 0.00170727, 0.00161727, 0.00172042, 0.00180047,
       0.00175074, 0.00178463, 0.00170734, 0.0016446 , 0.00145234,
       0.00145688, 0.0017044 , 0.00185707, 0.00179175, 0.00147741,
       0.00163337, 0.0018268 , 0.00165219, 0.00170946, 0.0018162 ,
       0.00161606, 0.00182464, 0.00173153, 0.00177255, 0.00183168,
       0.00188256, 0.00175019, 0.00166266, 0.00166703, 0.00169118,
       0.00132788, 0.00172024, 0.00143957, 0.00149198, 0.00186541,
       0.00117023, 0.00146697, 0.00160428, 0.00170212, 0.00176337,
       0.00176254, 0.00176326, 0.0018056 , 0.00208511, 0.00188691,
       0.00220398, 0.00164686, 0.00181256, 0.00193367, 0.00184403,
       0.00188599, 0.00174955, 0.00179382, 0.00184632, 0.0018729 ,
       0.00185299, 0.00174521, 0.00181592, 0.00188587, 0.00163176,
       0.00171586, 0.00162324, 0.00157351, 0.00138802, 0.00112665,
       0.00193523, 0.00166835, 0.00160427, 0.00154211, 0.00149935,
       0.00162351, 0.00148572, 0.00172846, 0.00172635, 0.00185887,
       0.00182996, 0.00160891, 0.00186514, 0.00192061, 0.0017109 ,
       0.00178361, 0.00174498, 0.00178957, 0.0016748 , 0.00150736,
       0.00125053, 0.00153248, 0.00175418, 0.00164216, 0.0017715 ,
       0.00181761, 0.00178655, 0.00187818, 0.00187876, 0.00186494,
       0.00176161, 0.0017823 , 0.00181924, 0.00184965, 0.0018429 ,
       0.00180466, 0.00170298, 0.00165691, 0.00166932, 0.00178031,
       0.0017748 , 0.00156212, 0.00165708, 0.00183918, 0.00144219,
       0.00160765, 0.00168709, 0.00174524, 0.00176747, 0.00176492,
       0.00190671, 0.00204396, 0.0018348 , 0.00181052, 0.00194585,
       0.00201294, 0.00167546, 0.00170089, 0.00179487, 0.00188959,
       0.00186878, 0.00182191, 0.0018723 , 0.00182831, 0.00167899,
       0.00170262, 0.00167293, 0.00164475, 0.00154567, 0.00134497,
       0.00168855, 0.00192626, 0.00171251, 0.00175019, 0.00166825,
       0.00157016, 0.00159817, 0.00171615, 0.00164938, 0.00182032,
       0.00195819, 0.00196377, 0.00192613, 0.00198398, 0.00170629,
       0.00191522, 0.00176668, 0.00177499, 0.00162983, 0.00176529,
       0.00174995, 0.00180725, 0.00174515, 0.00154354, 0.00169508,
       0.00175335, 0.00199255, 0.00165812, 0.00188994, 0.00182059,
       0.00184612, 0.00181787, 0.00180541, 0.00178678, 0.00176388,
       0.00124385, 0.00163545, 0.00160588, 0.00162575, 0.00145506,
       0.00156112, 0.00159422, 0.00179742, 0.00174339, 0.00178047,
       0.00180274, 0.00160547, 0.00164911, 0.00170055, 0.00175148,
       0.00176155, 0.00173291, 0.00179935, 0.00173639, 0.00189847,
       0.00146772, 0.00159098, 0.0016921 , 0.00180208, 0.00183252,
       0.00181994, 0.00194185, 0.00173666, 0.00148852, 0.00166427,
       0.00168255, 0.00170468, 0.00165357, 0.00174061, 0.00157491,
       0.00157282, 0.00150107, 0.00173945, 0.00173526, 0.00163326,
       0.0016764 , 0.00180106, 0.00184851, 0.00190519, 0.00200224,
       0.00204809, 0.00206755, 0.00194462, 0.00204147, 0.00133456,
       0.00190057, 0.00162733, 0.00168074, 0.00168345, 0.00157722,
       0.00188128, 0.00178571, 0.00166071, 0.00184142, 0.00183229,
       0.00190086, 0.00193872, 0.00182182, 0.00177888, 0.00172908,
       0.00159801, 0.0015103 , 0.00153205, 0.00152619, 0.0015838 ,
       0.00165146, 0.00149259, 0.00147658, 0.00151252, 0.00178737,
       0.00165169, 0.00127456, 0.00169569, 0.00176071, 0.0018076 ,
       0.00176786, 0.00167764, 0.00171829, 0.00164661, 0.0018183 ,
       0.00158099, 0.00162891, 0.00164383, 0.00184266, 0.00201796,
       0.00171939, 0.00171366, 0.00170086, 0.00166647, 0.00172813,
       0.00166571, 0.00176712, 0.00157798, 0.00161895, 0.00156174,
       0.00158578, 0.00175442, 0.0017856 , 0.00191598, 0.00167779,
       0.00173659, 0.00184195, 0.00199729, 0.00193016, 0.00194587,
       0.00190801, 0.00209159, 0.00211689, 0.00119603, 0.0014919 ,
       0.00144973, 0.00197377, 0.00166615, 0.00154795, 0.00133983,
       0.00173954, 0.00155059, 0.0017884 , 0.00165348, 0.00188298,
       0.00157957, 0.00175981, 0.00172433, 0.00128917, 0.00198551,
       0.00189273, 0.00135384, 0.00149838, 0.00157354, 0.00176159,
       0.00175029, 0.00162033, 0.00152507, 0.00152195, 0.00182814,
       0.0015468 , 0.00146045, 0.00170065, 0.00189855, 0.00195729,
       0.00173168, 0.00160158, 0.00162831, 0.00191367, 0.00127866,
       0.00133399, 0.0016864 , 0.00170431, 0.00172387, 0.00169923,
       0.0017397 , 0.00196392, 0.00180663, 0.00169077, 0.00164461,
       0.00161935, 0.0015658 , 0.00173699, 0.0017754 , 0.00191804,
       0.00177896, 0.00202842, 0.00170979, 0.00177832, 0.00184853,
       0.0021082 , 0.00223004, 0.00159217, 0.00199701, 0.00217133,
       0.0014493 , 0.00162428, 0.00159851, 0.00122027, 0.00149051,
       0.0015277 , 0.00170257, 0.00147404, 0.00149975, 0.00157302,
       0.0015818 , 0.00122559, 0.00181752, 0.00186656, 0.00184481,
       0.00175805, 0.00186591, 0.00184751, 0.00169094, 0.00160275,
       0.00171725, 0.00148694, 0.00184792, 0.00173693, 0.00202832,
       0.00168378, 0.00165678, 0.00154306, 0.00146419, 0.00160583,
       0.00185731, 0.00173562, 0.00173121, 0.00171449, 0.00170707,
       0.00169566, 0.00169206, 0.00158788, 0.00176865, 0.00171152,
       0.0017177 , 0.00164659, 0.00179171, 0.00172968, 0.00186985,
       0.00202054, 0.00173391, 0.00172525, 0.00179548, 0.00188294,
       0.00205301, 0.00196588, 0.00157032, 0.00189203, 0.00216878,
       0.0016177 , 0.00175893, 0.0013166 , 0.00161479, 0.00155849,
       0.00159365, 0.0018155 , 0.00179271, 0.00174678, 0.00175264,
       0.00170448, 0.00188689, 0.00159749, 0.00197424, 0.00190924,
       0.00175161, 0.00168314, 0.0018391 , 0.00193919, 0.0015636 ,
       0.00157251, 0.0016423 , 0.00173871, 0.00148399, 0.00162209,
       0.00156895, 0.00176572, 0.0017203 , 0.00170062, 0.00163453,
       0.00151904, 0.00155712, 0.001483  , 0.00182225, 0.0017657 ,
       0.00170839, 0.00178194, 0.00187712, 0.00198439, 0.00157245,
       0.001783  , 0.00197082, 0.00166639, 0.00171507, 0.00179077,
       0.00184073, 0.00165962, 0.00185549, 0.00195983, 0.00161812,
       0.00185966, 0.00208987, 0.00169198, 0.00164146, 0.00162196,
       0.00173093, 0.00164297, 0.00175449, 0.00175546, 0.00165095,
       0.00162051, 0.00168385, 0.00189293, 0.00147108, 0.00199817,
       0.00198129, 0.00182165, 0.00174408, 0.0016756 , 0.00186665,
       0.00161121, 0.00170415, 0.0017018 , 0.00162806, 0.00160024,
       0.00185902, 0.00178792, 0.00172651, 0.00189862, 0.00179448,
       0.00208538, 0.00174339, 0.00188052, 0.00169416, 0.00161655,
       0.00169694, 0.00185411, 0.00204397, 0.00185679, 0.00171443,
       0.00192344, 0.0018315 , 0.00166332, 0.00199423, 0.0014718 ,
       0.0014381 , 0.00159337, 0.00202189, 0.00206286, 0.00186814,
       0.00175223, 0.00192868, 0.00162897, 0.00119033, 0.0018465 ,
       0.00179809, 0.00172511, 0.00187165, 0.00207625, 0.00176142,
       0.0017788 , 0.00179552, 0.00163833, 0.00161355, 0.00171751,
       0.00180274, 0.0019246 , 0.00169912, 0.00182035, 0.00190037,
       0.00179893, 0.00187507, 0.00168528, 0.00150922, 0.00141668,
       0.00178188, 0.00148098, 0.00205317, 0.00210442, 0.00185756,
       0.00173447, 0.00165004, 0.00161887, 0.00180006, 0.00177959,
       0.00171513, 0.00184142, 0.00202347, 0.00184128, 0.00171635,
       0.00170019, 0.00161802, 0.00167286, 0.0017018 , 0.00177504,
       0.00173388, 0.00167762, 0.00176274, 0.00182412, 0.0017154 ,
       0.00174533, 0.00164355, 0.00166013, 0.00163392, 0.00165826,
       0.00187724, 0.00209453, 0.00205007, 0.00183612, 0.001709  ,
       0.00159921, 0.00175877, 0.00173287, 0.00169748, 0.00180884,
       0.00202898, 0.001895  , 0.00164306, 0.00160747, 0.00170677,
       0.0018001 , 0.00164636, 0.00140559, 0.00162892, 0.00171224,
       0.00193895, 0.00164072, 0.00152959, 0.00142045, 0.00175926,
       0.00175715, 0.0017658 , 0.00195884, 0.00208634, 0.0019506 ,
       0.00176637, 0.0016482 , 0.0017189 , 0.00173551, 0.00168064,
       0.00166488, 0.00173356, 0.0020162 , 0.00195946, 0.0015952 ,
       0.00189439, 0.00204581, 0.00195562, 0.00153808, 0.00140371,
       0.00176094, 0.00190468, 0.00183113, 0.00181748, 0.00180836,
       0.00193371, 0.00195802, 0.0016997 , 0.00153156, 0.00167321,
       0.00194138, 0.00170847, 0.00166351, 0.00165402, 0.00165614,
       0.00196766, 0.00201191, 0.0016674 , 0.00148491, 0.00189244,
       0.00184399, 0.00182242, 0.00190466, 0.00181414, 0.00134525,
       0.00157439, 0.00178604, 0.00171549, 0.00165463, 0.00159848,
       0.00188478, 0.00170146, 0.00203671, 0.00180205, 0.00192244,
       0.00184037, 0.00184024, 0.00191472, 0.00175131, 0.00184865,
       0.0016329 , 0.00172979, 0.00178998, 0.00156186, 0.00177548,
       0.00202151, 0.00173627, 0.00178212, 0.001914  , 0.00179811,
       0.00182262, 0.0019456 , 0.00194897, 0.00188658, 0.00160006,
       0.00167932, 0.00153373, 0.00188414, 0.00150616, 0.00192347,
       0.00186209, 0.00172563, 0.00182982, 0.00137834, 0.00137944,
       0.00154682, 0.00169304, 0.00145494, 0.00176103, 0.00175355,
       0.00153439])
        indices_v_gt = np.array([   0,    1,    2,    3,    4,    5,    6,    7,    8,    9,   10,
         11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
         22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
         33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,
         44,   45,   46,   47,   48,   49,   50,   51,   52,   53,   54,
         55,   56,   57,   58,   59,   60,   61,   62,   63,   64,   65,
         66,   67,   68,   69,   70,   71,   72,   73,   74,   75,   76,
         77,   78,   79,   80,   81,   82,   83,   84,   85,   86,   87,
         88,   89,   90,   91,   92,   93,   94,   95,   96,   97,   98,
         99,  100,  101,  102,  103,  104,  105,  106,  107,  108,  109,
        110,  111,  112,  113,  114,  115,  116,  117,  118,  119,  120,
        121,  122,  123,  124,  125,  126,  127,  128,  129,  130,  131,
        132,  133,  134,  135,  136,  137,  138,  139,  140,  141,  142,
        143,  144,  145,  146,  147,  148,  149,  150,  151,  152,  153,
        154,  155,  156,  157,  158,  159,  160,  161,  162,  163,  164,
        165,  166,  167,  168,  169,  170,  171,  172,  173,  174,  175,
        176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,
        187,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,
        198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,
        209,  210,  211,  212,  213,  214,  215,  216,  217,  218,  219,
        220,  221,  222,  223,  224,  225,  226,  227,  228,  229,  230,
        231,  232,  233,  234,  235,  236,  237,  238,  239,  240,  241,
        242,  243,  244,  245,  246,  247,  248,  249,  250,  251,  252,
        253,  254,  255,  256,  257,  258,  259,  260,  261,  262,  263,
        264,  265,  266,  267,  268,  269,  270,  271,  272,  273,  274,
        275,  276,  277,  278,  279,  280,  281,  282,  283,  284,  285,
        286,  287,  288,  289,  290,  291,  292,  293,  294,  295,  296,
        297,  298,  299,  300,  301,  302,  303,  304,  305,  306,  307,
        308,  309,  310,  311,  312,  313,  314,  315,  316,  317,  318,
        319,  320,  321,  322,  323,  324,  325,  326,  327,  328,  329,
        330,  331,  332,  333,  334,  335,  336,  337,  338,  339,  340,
        341,  342,  343,  344,  345,  346,  347,  348,  349,  350,  351,
        352,  353,  354,  355,  356,  357,  358,  359,  360,  361,  362,
        363,  364,  365,  366,  367,  368,  369,  370,  371,  372,  373,
        374,  375,  376,  377,  378,  379,  380,  381,  382,  383,  384,
        385,  386,  387,  388,  389,  390,  391,  392,  393,  394,  395,
        396,  397,  398,  399,  400,  401,  402,  403,  404,  405,  406,
        407,  408,  409,  410,  411,  412,  413,  414,  415,  416,  417,
        418,  419,  420,  421,  422,  423,  424,  425,  426,  427,  428,
        429,  430,  431,  432,  433,  434,  435,  436,  437,  438,  439,
        440,  441,  442,  443,  444,  445,  446,  447,  448,  449,  450,
        451,  452,  453,  454,  455,  456,  457,  458,  459,  460,  461,
        462,  463,  464,  465,  466,  467,  468,  469,  470,  471,  472,
        473,  474,  475,  476,  477,  478,  479,  480,  481,  482,  483,
        484,  485,  486,  487,  488,  489,  490,  491,  492,  493,  494,
        495,  496,  497,  498,  499,  500,  501,  502,  503,  504,  505,
        506,  507,  508,  509,  510,  511,  512,  513,  514,  515,  516,
        517,  518,  519,  520,  521,  522,  523,  524,  525,  526,  527,
        528,  529,  530,  531,  532,  533,  534,  535,  536,  537,  538,
        539,  540,  541,  542,  543,  544,  545,  546,  547,  548,  549,
        550,  551,  552,  553,  554,  555,  556,  557,  558,  559,  560,
        561,  562,  563,  564,  565,  566,  567,  568,  569,  570,  571,
        572,  573,  574,  575,  576,  577,  578,  579,  580,  581,  582,
        583,  584,  585,  586,  587,  588,  589,  590,  591,  592,  593,
        594,  595,  596,  597,  598,  599,  600,  601,  602,  603,  604,
        605,  606,  607,  608,  609,  610,  611,  612,  613,  614,  615,
        616,  617,  618,  619,  620,  621,  622,  623,  624,  625,  626,
        627,  628,  629,  630,  631,  632,  633,  634,  635,  636,  637,
        638,  639,  640,  641,  642,  643,  644,  645,  646,  647,  648,
        649,  650,  651,  652,  653,  654,  655,  656,  657,  658,  659,
        660,  661,  662,  663,  664,  665,  666,  667,  668,  669,  670,
        671,  672,  673,  674,  675,  676,  677,  678,  679,  680,  681,
        682,  683,  684,  685,  686,  687,  688,  689,  690,  691,  692,
        693,  694,  695,  696,  697,  698,  699,  700,  701,  702,  703,
        704,  705,  706,  707,  708,  709,  710,  711,  712,  713,  714,
        715,  716,  717,  718,  719,  720,  721,  722,  723,  724,  725,
        726,  727,  728,  729,  730,  731,  732,  733,  734,  735,  736,
        737,  738,  739,  740,  741,  742,  743,  744,  745,  746,  747,
        748,  749,  750,  751,  752,  753,  754,  755,  756,  757,  758,
        759,  760,  761,  762,  763,  764,  765,  766,  767,  768,  769,
        770,  771,  772,  773,  774,  775,  776,  777,  778,  779,  780,
        781,  782,  783,  784,  785,  786,  787,  788,  789,  790,  791,
        792,  793,  794,  795,  796,  797,  798,  799,  800,  801,  802,
        803,  804,  805,  806,  807,  808,  809,  810,  811,  812,  813,
        814,  815,  816,  817,  818,  819,  820,  821,  822,  823,  824,
        825,  826,  827,  828,  829,  830,  831,  832,  833,  834,  835,
        836,  837,  838,  839,  840,  841,  842,  843,  844,  845,  846,
        847,  848,  849,  850,  851,  852,  853,  854,  855,  856,  857,
        858,  859,  860,  861,  862,  863,  864,  865,  866,  867,  868,
        869,  870,  871,  872,  873,  874,  875,  876,  877,  878,  879,
        880,  881,  882,  883,  884,  885,  886,  887,  888,  889,  890,
        891,  892,  893,  894,  895,  896,  897,  898,  899,  900,  901,
        902,  903,  904,  905,  906,  907,  908,  909,  910,  911,  912,
        913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,
        924,  925,  926,  927,  928,  929,  930,  931,  932,  933,  934,
        935,  936,  937,  938,  939,  940,  941,  942,  943,  944,  945,
        946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,
        957,  958,  959,  960,  961,  962,  963,  964,  965,  966,  967,
        968,  969,  970,  971,  972,  973,  974,  975,  976,  977,  978,
        979,  980,  981,  982,  983,  984,  985,  986,  987,  988,  989,
        990,  991,  992,  993,  994,  995,  996,  997,  998,  999, 1000,
       1001, 1002, 1003, 1004, 1005, 1006, 1007, 1008, 1009, 1010, 1011,
       1012, 1013, 1014, 1015, 1016, 1017, 1018, 1019, 1020, 1021, 1022,
       1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1031, 1032, 1033,
       1034, 1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044,
       1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055,
       1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066,
       1067, 1068, 1069, 1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077,
       1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088,
       1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099,
       1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107, 1108, 1109, 1110,
       1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1120, 1121,
       1122, 1123, 1124, 1125, 1126, 1127, 1128, 1129, 1130, 1131, 1132,
       1133, 1134, 1135, 1136, 1137, 1138, 1139, 1140, 1141, 1142, 1143,
       1144, 1145, 1146, 1147, 1148, 1149, 1150, 1151, 1152, 1153, 1154,
       1155, 1156, 1157, 1158, 1159, 1160, 1161, 1162, 1163, 1164, 1165,
       1166, 1167, 1168, 1169, 1170, 1171, 1172, 1173, 1174, 1175, 1176,
       1177, 1178, 1179, 1180, 1181, 1182, 1183, 1184, 1185, 1186, 1187,
       1188, 1189, 1190, 1191, 1192, 1193, 1194, 1195, 1196, 1197, 1198,
       1199, 1200, 1201, 1202, 1203, 1204, 1205, 1206, 1207, 1208, 1209,
       1210, 1211, 1212, 1213, 1214, 1215, 1216, 1217, 1218, 1219, 1220,
       1221, 1222, 1223, 1224, 1225, 1226, 1227, 1228, 1229, 1230, 1231,
       1232, 1233, 1234, 1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242,
       1243, 1244, 1245, 1246, 1247, 1248, 1249, 1250, 1251, 1252, 1253,
       1254, 1255, 1256, 1257, 1258, 1259, 1260, 1261, 1262, 1263, 1264,
       1265, 1266, 1267, 1268, 1269, 1270, 1271, 1272, 1273, 1274, 1275,
       1276, 1277, 1278, 1279, 1280, 1281, 1282, 1283, 1284, 1285, 1286,
       1287, 1288, 1289, 1290, 1291, 1292, 1293, 1294, 1295, 1296, 1297,
       1298, 1299, 1300, 1301, 1302, 1303, 1304, 1305, 1306, 1307, 1308,
       1309, 1310, 1311, 1312, 1313, 1314, 1315, 1316, 1317, 1318, 1319,
       1320, 1321, 1322, 1323, 1324, 1325, 1326, 1327, 1328, 1329, 1330,
       1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341,
       1342, 1343, 1344, 1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352,
       1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363,
       1364, 1365, 1366, 1367, 1368, 1369, 1370, 1371, 1372, 1373, 1374,
       1375, 1376, 1377, 1378, 1379, 1380, 1381, 1382, 1383, 1384, 1385,
       1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396,
       1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407,
       1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 1416, 1417, 1418,
       1419, 1420, 1421, 1422, 1423, 1424, 1425, 1426, 1427, 1428, 1429,
       1430, 1431, 1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440,
       1441, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451,
       1452, 1453, 1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462,
       1463, 1464, 1465, 1466, 1467, 1468, 1469, 1470, 1471, 1472, 1473,
       1474, 1475, 1476, 1477, 1478, 1479, 1480, 1481, 1482, 1483, 1484,
       1485, 1486, 1487, 1488, 1489, 1490, 1491, 1492, 1493, 1494, 1495,
       1496, 1497, 1498, 1499, 1500, 1501, 1502, 1503, 1504, 1505, 1506,
       1507, 1508, 1509, 1510, 1511, 1512, 1513, 1514, 1515, 1516, 1517,
       1518, 1519, 1520, 1521, 1522, 1523, 1524, 1525, 1526, 1527, 1528,
       1529, 1530, 1531, 1532, 1533, 1534, 1535, 1536, 1537, 1538, 1539,
       1540, 1541, 1542, 1543, 1544, 1545, 1546, 1547, 1548, 1549, 1550,
       1551, 1552, 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560, 1561,
       1562, 1563, 1564, 1565, 1566, 1567, 1568, 1569, 1570, 1571, 1572,
       1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580, 1581, 1582, 1583,
       1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593, 1594,
       1595, 1596, 1597, 1598, 1599, 1600, 1601, 1602, 1603, 1604, 1605,
       1606, 1607, 1608, 1609, 1610, 1611, 1612, 1613, 1614, 1615, 1616,
       1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626, 1627,
       1628, 1629, 1630, 1631, 1632, 1633, 1634, 1635, 1636, 1637, 1638,
       1639, 1640, 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648, 1649,
       1650, 1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660,
       1661, 1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671,
       1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682,
       1683, 1684, 1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693,
       1694, 1695, 1696, 1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704,
       1705, 1706, 1707, 1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715,
       1716, 1717, 1718, 1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726,
       1727, 1728, 1729, 1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737,
       1738, 1739, 1740, 1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748,
       1749, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759,
       1760, 1761, 1762, 1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770,
       1771, 1772, 1773, 1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781,
       1782, 1783, 1784, 1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792,
       1793, 1794, 1795, 1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803,
       1804, 1805, 1806, 1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814,
       1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825,
       1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836,
       1837, 1838, 1839, 1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847,
       1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858,
       1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869,
       1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880,
       1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891,
       1892, 1893, 1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902,
       1903, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913,
       1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924,
       1925, 1926, 1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935,
       1936, 1937, 1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946,
       1947, 1948, 1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957,
       1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968,
       1969, 1970, 1971, 1972, 1973, 1974, 1975, 1976, 1977, 1978, 1979,
       1980, 1981, 1982, 1983, 1984, 1985, 1986, 1987, 1988, 1989, 1990,
       1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001,
       2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012,
       2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023,
       2024, 2025, 2026, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034,
       2035, 2036, 2037, 2038, 2039, 2040, 2041, 2042, 2043, 2044, 2045,
       2046, 2047, 2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056,
       2057, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067,
       2068, 2069, 2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078,
       2079, 2080, 2081, 2082, 2083, 2084, 2085, 2086, 2087, 2088, 2089,
       2090, 2091, 2092, 2093, 2094, 2095, 2096, 2097, 2098, 2099, 2100,
       2101, 2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111,
       2112, 2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122,
       2123, 2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133,
       2134, 2135, 2136, 2137, 2138, 2139, 2140, 2141, 2142, 2143, 2144,
       2145, 2146, 2147, 2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155,
       2156, 2157, 2158, 2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166,
       2167, 2168, 2169, 2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177,
       2178, 2179, 2180, 2181, 2182, 2183, 2184, 2185, 2186, 2187, 2188,
       2189, 2190, 2191, 2192, 2193, 2194, 2195, 2196, 2197, 2198, 2199,
       2200, 2201, 2202, 2203, 2204, 2205, 2206, 2207, 2208, 2209, 2210,
       2211, 2212, 2213, 2214, 2215, 2216, 2217, 2218, 2219, 2220, 2221,
       2222, 2223, 2224, 2225, 2226, 2227, 2228, 2229, 2230, 2231, 2232,
       2233, 2234, 2235, 2236, 2237, 2238, 2239, 2240, 2241, 2242, 2243,
       2244, 2245, 2246, 2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254,
       2255, 2256, 2257, 2258, 2259, 2260, 2261, 2262, 2263, 2264, 2265,
       2266, 2267, 2268, 2269, 2270, 2271, 2272, 2273, 2274, 2275, 2276,
       2277, 2278, 2279, 2280, 2281, 2282, 2283, 2284, 2285, 2286, 2287,
       2288, 2289, 2290, 2291, 2292, 2293, 2294, 2295, 2296, 2297, 2298,
       2299, 2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307, 2308, 2309,
       2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320,
       2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331,
       2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342,
       2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353,
       2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364,
       2365, 2366, 2367, 2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375,
       2376, 2377, 2378, 2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386,
       2387, 2388, 2389, 2390, 2391, 2392, 2393, 2394, 2395, 2396, 2397,
       2398, 2399, 2400, 2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408,
       2409, 2410, 2411, 2412, 2413, 2414, 2415, 2416, 2417, 2418, 2419,
       2420, 2421, 2422, 2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430,
       2431, 2432, 2433, 2434, 2435, 2436, 2437, 2438, 2439, 2440, 2441,
       2442, 2443, 2444, 2445, 2446, 2447, 2448, 2449, 2450, 2451, 2452,
       2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463,
       2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2473, 2474,
       2475, 2476, 2477, 2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485,
       2486, 2487, 2488, 2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496,
       2497, 2498, 2499, 2500, 2501, 2502, 2503, 2504, 2505, 2506, 2507,
       2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515, 2516, 2517, 2518,
       2519, 2520, 2521, 2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529,
       2530, 2531, 2532, 2533, 2534, 2535, 2536, 2537, 2538, 2539, 2540,
       2541, 2542, 2543, 2544, 2545, 2546, 2547, 2548, 2549, 2550, 2551,
       2552, 2553, 2554, 2555, 2556, 2557, 2558, 2559, 2560, 2561, 2562,
       2563, 2564, 2565, 2566, 2567, 2568, 2569, 2570, 2571, 2572, 2573,
       2574, 2575, 2576, 2577, 2578, 2579, 2580, 2581, 2582, 2583, 2584,
       2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595,
       2596, 2597, 2598, 2599, 2600, 2601, 2602, 2603, 2604, 2605, 2606,
       2607, 2608, 2609, 2610, 2611, 2612, 2613, 2614, 2615, 2616, 2617,
       2618, 2619, 2620, 2621, 2622, 2623, 2624, 2625, 2626, 2627, 2628,
       2629, 2630, 2631, 2632, 2633, 2634, 2635, 2636, 2637, 2638, 2639,
       2640, 2641, 2642, 2643, 2644, 2645, 2646, 2647, 2648, 2649, 2650,
       2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658, 2659, 2660, 2661,
       2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2672,
       2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680, 2681, 2682, 2683,
       2684, 2685, 2686, 2687, 2688, 2689, 2690, 2691, 2692, 2693, 2694,
       2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702, 2703, 2704, 2705,
       2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716,
       2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727,
       2728, 2729, 2730, 2731, 2732, 2733, 2734, 2735, 2736, 2737, 2738,
       2739, 2740, 2741, 2742, 2743, 2744, 2745, 2746, 2747, 2748, 2749,
       2750, 2751, 2752, 2753, 2754, 2755, 2756, 2757, 2758, 2759, 2760,
       2761, 2762, 2763, 2764, 2765, 2766, 2767, 2768, 2769, 2770, 2771,
       2772, 2773, 2774, 2775, 2776, 2777, 2778, 2779, 2780, 2781, 2782,
       2783, 2784, 2785, 2786, 2787, 2788, 2789, 2790, 2791, 2792, 2793,
       2794, 2795, 2796, 2797, 2798, 2799, 2800, 2801, 2802, 2803, 2804,
       2805, 2806, 2807, 2808, 2809, 2810, 2811, 2812, 2813, 2814, 2815,
       2816, 2817, 2818, 2819, 2820, 2821, 2822, 2823, 2824, 2825, 2826,
       2827, 2828, 2829, 2830, 2831, 2832, 2833, 2834, 2835, 2836, 2837,
       2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847, 2848,
       2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2857, 2858, 2859,
       2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867, 2868, 2869, 2870,
       2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878, 2879, 2880, 2881,
       2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889, 2890, 2891, 2892,
       2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900, 2901, 2902, 2903,
       2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913, 2914,
       2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925,
       2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935, 2936,
       2937, 2938, 2939, 2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947,
       2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956, 2957, 2958,
       2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967, 2968, 2969,
       2970, 2971, 2972, 2973, 2974, 2975, 2976, 2977, 2978, 2979, 2980,
       2981, 2982, 2983, 2984, 2985, 2986, 2987, 2988, 2989, 2990, 2991,
       2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3001, 3002,
       3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010, 3011, 3012, 3013,
       3014, 3015, 3016, 3017, 3018, 3019, 3020, 3021, 3022, 3023, 3024,
       3025, 3026, 3027, 3028, 3029, 3030, 3031, 3032, 3033, 3034, 3035,
       3036, 3037, 3038, 3039, 3040, 3041, 3042, 3043, 3044, 3045, 3046,
       3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055, 3056, 3057,
       3058, 3059, 3060, 3061, 3062, 3063, 3064, 3065, 3066, 3067, 3068,
       3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076, 3077, 3078, 3079,
       3080, 3081, 3082, 3083, 3084, 3085, 3086, 3087, 3088, 3089, 3090,
       3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099, 3100, 3101,
       3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110, 3111, 3112,
       3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121, 3122, 3123,
       3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131, 3132, 3133, 3134,
       3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142, 3143, 3144, 3145,
       3146, 3147, 3148, 3149, 3150, 3151, 3152, 3153, 3154, 3155, 3156,
       3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164, 3165, 3166, 3167,
       3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175, 3176, 3177, 3178,
       3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 3189,
       3190, 3191, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3199, 3200,
       3201, 3202, 3203, 3204, 3205, 3206, 3207, 3208, 3209, 3210, 3211,
       3212, 3213, 3214, 3215, 3216, 3217, 3218, 3219, 3220, 3221, 3222,
       3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233,
       3234, 3235, 3236, 3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244,
       3245, 3246, 3247, 3248, 3249, 3250, 3251, 3252, 3253, 3254, 3255,
       3256, 3257, 3258, 3259, 3260, 3261, 3262, 3263, 3264, 3265, 3266,
       3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274, 3275, 3276, 3277,
       3278, 3279, 3280, 3281, 3282, 3283, 3284, 3285, 3286, 3287, 3288,
       3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296, 3297, 3298, 3299,
       3300], dtype=np.int32)
        indptr_v_gt = np.array([   0,    1,    2,    3,    4,    5,    6,    7,    8,    9,   10,
         11,   12,   13,   14,   15,   16,   17,   18,   19,   20,   21,
         22,   23,   24,   25,   26,   27,   28,   29,   30,   31,   32,
         33,   34,   35,   36,   37,   38,   39,   40,   41,   42,   43,
         44,   45,   46,   47,   48,   49,   50,   51,   52,   53,   54,
         55,   56,   57,   58,   59,   60,   61,   62,   63,   64,   65,
         66,   67,   68,   69,   70,   71,   72,   73,   74,   75,   76,
         77,   78,   79,   80,   81,   82,   83,   84,   85,   86,   87,
         88,   89,   90,   91,   92,   93,   94,   95,   96,   97,   98,
         99,  100,  101,  102,  103,  104,  105,  106,  107,  108,  109,
        110,  111,  112,  113,  114,  115,  116,  117,  118,  119,  120,
        121,  122,  123,  124,  125,  126,  127,  128,  129,  130,  131,
        132,  133,  134,  135,  136,  137,  138,  139,  140,  141,  142,
        143,  144,  145,  146,  147,  148,  149,  150,  151,  152,  153,
        154,  155,  156,  157,  158,  159,  160,  161,  162,  163,  164,
        165,  166,  167,  168,  169,  170,  171,  172,  173,  174,  175,
        176,  177,  178,  179,  180,  181,  182,  183,  184,  185,  186,
        187,  188,  189,  190,  191,  192,  193,  194,  195,  196,  197,
        198,  199,  200,  201,  202,  203,  204,  205,  206,  207,  208,
        209,  210,  211,  212,  213,  214,  215,  216,  217,  218,  219,
        220,  221,  222,  223,  224,  225,  226,  227,  228,  229,  230,
        231,  232,  233,  234,  235,  236,  237,  238,  239,  240,  241,
        242,  243,  244,  245,  246,  247,  248,  249,  250,  251,  252,
        253,  254,  255,  256,  257,  258,  259,  260,  261,  262,  263,
        264,  265,  266,  267,  268,  269,  270,  271,  272,  273,  274,
        275,  276,  277,  278,  279,  280,  281,  282,  283,  284,  285,
        286,  287,  288,  289,  290,  291,  292,  293,  294,  295,  296,
        297,  298,  299,  300,  301,  302,  303,  304,  305,  306,  307,
        308,  309,  310,  311,  312,  313,  314,  315,  316,  317,  318,
        319,  320,  321,  322,  323,  324,  325,  326,  327,  328,  329,
        330,  331,  332,  333,  334,  335,  336,  337,  338,  339,  340,
        341,  342,  343,  344,  345,  346,  347,  348,  349,  350,  351,
        352,  353,  354,  355,  356,  357,  358,  359,  360,  361,  362,
        363,  364,  365,  366,  367,  368,  369,  370,  371,  372,  373,
        374,  375,  376,  377,  378,  379,  380,  381,  382,  383,  384,
        385,  386,  387,  388,  389,  390,  391,  392,  393,  394,  395,
        396,  397,  398,  399,  400,  401,  402,  403,  404,  405,  406,
        407,  408,  409,  410,  411,  412,  413,  414,  415,  416,  417,
        418,  419,  420,  421,  422,  423,  424,  425,  426,  427,  428,
        429,  430,  431,  432,  433,  434,  435,  436,  437,  438,  439,
        440,  441,  442,  443,  444,  445,  446,  447,  448,  449,  450,
        451,  452,  453,  454,  455,  456,  457,  458,  459,  460,  461,
        462,  463,  464,  465,  466,  467,  468,  469,  470,  471,  472,
        473,  474,  475,  476,  477,  478,  479,  480,  481,  482,  483,
        484,  485,  486,  487,  488,  489,  490,  491,  492,  493,  494,
        495,  496,  497,  498,  499,  500,  501,  502,  503,  504,  505,
        506,  507,  508,  509,  510,  511,  512,  513,  514,  515,  516,
        517,  518,  519,  520,  521,  522,  523,  524,  525,  526,  527,
        528,  529,  530,  531,  532,  533,  534,  535,  536,  537,  538,
        539,  540,  541,  542,  543,  544,  545,  546,  547,  548,  549,
        550,  551,  552,  553,  554,  555,  556,  557,  558,  559,  560,
        561,  562,  563,  564,  565,  566,  567,  568,  569,  570,  571,
        572,  573,  574,  575,  576,  577,  578,  579,  580,  581,  582,
        583,  584,  585,  586,  587,  588,  589,  590,  591,  592,  593,
        594,  595,  596,  597,  598,  599,  600,  601,  602,  603,  604,
        605,  606,  607,  608,  609,  610,  611,  612,  613,  614,  615,
        616,  617,  618,  619,  620,  621,  622,  623,  624,  625,  626,
        627,  628,  629,  630,  631,  632,  633,  634,  635,  636,  637,
        638,  639,  640,  641,  642,  643,  644,  645,  646,  647,  648,
        649,  650,  651,  652,  653,  654,  655,  656,  657,  658,  659,
        660,  661,  662,  663,  664,  665,  666,  667,  668,  669,  670,
        671,  672,  673,  674,  675,  676,  677,  678,  679,  680,  681,
        682,  683,  684,  685,  686,  687,  688,  689,  690,  691,  692,
        693,  694,  695,  696,  697,  698,  699,  700,  701,  702,  703,
        704,  705,  706,  707,  708,  709,  710,  711,  712,  713,  714,
        715,  716,  717,  718,  719,  720,  721,  722,  723,  724,  725,
        726,  727,  728,  729,  730,  731,  732,  733,  734,  735,  736,
        737,  738,  739,  740,  741,  742,  743,  744,  745,  746,  747,
        748,  749,  750,  751,  752,  753,  754,  755,  756,  757,  758,
        759,  760,  761,  762,  763,  764,  765,  766,  767,  768,  769,
        770,  771,  772,  773,  774,  775,  776,  777,  778,  779,  780,
        781,  782,  783,  784,  785,  786,  787,  788,  789,  790,  791,
        792,  793,  794,  795,  796,  797,  798,  799,  800,  801,  802,
        803,  804,  805,  806,  807,  808,  809,  810,  811,  812,  813,
        814,  815,  816,  817,  818,  819,  820,  821,  822,  823,  824,
        825,  826,  827,  828,  829,  830,  831,  832,  833,  834,  835,
        836,  837,  838,  839,  840,  841,  842,  843,  844,  845,  846,
        847,  848,  849,  850,  851,  852,  853,  854,  855,  856,  857,
        858,  859,  860,  861,  862,  863,  864,  865,  866,  867,  868,
        869,  870,  871,  872,  873,  874,  875,  876,  877,  878,  879,
        880,  881,  882,  883,  884,  885,  886,  887,  888,  889,  890,
        891,  892,  893,  894,  895,  896,  897,  898,  899,  900,  901,
        902,  903,  904,  905,  906,  907,  908,  909,  910,  911,  912,
        913,  914,  915,  916,  917,  918,  919,  920,  921,  922,  923,
        924,  925,  926,  927,  928,  929,  930,  931,  932,  933,  934,
        935,  936,  937,  938,  939,  940,  941,  942,  943,  944,  945,
        946,  947,  948,  949,  950,  951,  952,  953,  954,  955,  956,
        957,  958,  959,  960,  961,  962,  963,  964,  965,  966,  967,
        968,  969,  970,  971,  972,  973,  974,  975,  976,  977,  978,
        979,  980,  981,  982,  983,  984,  985,  986,  987,  988,  989,
        990,  991,  992,  993,  994,  995,  996,  997,  998,  999, 1000,
       1001, 1002, 1003, 1004, 1005, 1006, 1007, 1008, 1009, 1010, 1011,
       1012, 1013, 1014, 1015, 1016, 1017, 1018, 1019, 1020, 1021, 1022,
       1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1031, 1032, 1033,
       1034, 1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044,
       1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055,
       1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066,
       1067, 1068, 1069, 1070, 1071, 1072, 1073, 1074, 1075, 1076, 1077,
       1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088,
       1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099,
       1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107, 1108, 1109, 1110,
       1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1120, 1121,
       1122, 1123, 1124, 1125, 1126, 1127, 1128, 1129, 1130, 1131, 1132,
       1133, 1134, 1135, 1136, 1137, 1138, 1139, 1140, 1141, 1142, 1143,
       1144, 1145, 1146, 1147, 1148, 1149, 1150, 1151, 1152, 1153, 1154,
       1155, 1156, 1157, 1158, 1159, 1160, 1161, 1162, 1163, 1164, 1165,
       1166, 1167, 1168, 1169, 1170, 1171, 1172, 1173, 1174, 1175, 1176,
       1177, 1178, 1179, 1180, 1181, 1182, 1183, 1184, 1185, 1186, 1187,
       1188, 1189, 1190, 1191, 1192, 1193, 1194, 1195, 1196, 1197, 1198,
       1199, 1200, 1201, 1202, 1203, 1204, 1205, 1206, 1207, 1208, 1209,
       1210, 1211, 1212, 1213, 1214, 1215, 1216, 1217, 1218, 1219, 1220,
       1221, 1222, 1223, 1224, 1225, 1226, 1227, 1228, 1229, 1230, 1231,
       1232, 1233, 1234, 1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242,
       1243, 1244, 1245, 1246, 1247, 1248, 1249, 1250, 1251, 1252, 1253,
       1254, 1255, 1256, 1257, 1258, 1259, 1260, 1261, 1262, 1263, 1264,
       1265, 1266, 1267, 1268, 1269, 1270, 1271, 1272, 1273, 1274, 1275,
       1276, 1277, 1278, 1279, 1280, 1281, 1282, 1283, 1284, 1285, 1286,
       1287, 1288, 1289, 1290, 1291, 1292, 1293, 1294, 1295, 1296, 1297,
       1298, 1299, 1300, 1301, 1302, 1303, 1304, 1305, 1306, 1307, 1308,
       1309, 1310, 1311, 1312, 1313, 1314, 1315, 1316, 1317, 1318, 1319,
       1320, 1321, 1322, 1323, 1324, 1325, 1326, 1327, 1328, 1329, 1330,
       1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341,
       1342, 1343, 1344, 1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352,
       1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363,
       1364, 1365, 1366, 1367, 1368, 1369, 1370, 1371, 1372, 1373, 1374,
       1375, 1376, 1377, 1378, 1379, 1380, 1381, 1382, 1383, 1384, 1385,
       1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396,
       1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407,
       1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 1416, 1417, 1418,
       1419, 1420, 1421, 1422, 1423, 1424, 1425, 1426, 1427, 1428, 1429,
       1430, 1431, 1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440,
       1441, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451,
       1452, 1453, 1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462,
       1463, 1464, 1465, 1466, 1467, 1468, 1469, 1470, 1471, 1472, 1473,
       1474, 1475, 1476, 1477, 1478, 1479, 1480, 1481, 1482, 1483, 1484,
       1485, 1486, 1487, 1488, 1489, 1490, 1491, 1492, 1493, 1494, 1495,
       1496, 1497, 1498, 1499, 1500, 1501, 1502, 1503, 1504, 1505, 1506,
       1507, 1508, 1509, 1510, 1511, 1512, 1513, 1514, 1515, 1516, 1517,
       1518, 1519, 1520, 1521, 1522, 1523, 1524, 1525, 1526, 1527, 1528,
       1529, 1530, 1531, 1532, 1533, 1534, 1535, 1536, 1537, 1538, 1539,
       1540, 1541, 1542, 1543, 1544, 1545, 1546, 1547, 1548, 1549, 1550,
       1551, 1552, 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560, 1561,
       1562, 1563, 1564, 1565, 1566, 1567, 1568, 1569, 1570, 1571, 1572,
       1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580, 1581, 1582, 1583,
       1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593, 1594,
       1595, 1596, 1597, 1598, 1599, 1600, 1601, 1602, 1603, 1604, 1605,
       1606, 1607, 1608, 1609, 1610, 1611, 1612, 1613, 1614, 1615, 1616,
       1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626, 1627,
       1628, 1629, 1630, 1631, 1632, 1633, 1634, 1635, 1636, 1637, 1638,
       1639, 1640, 1641, 1642, 1643, 1644, 1645, 1646, 1647, 1648, 1649,
       1650, 1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660,
       1661, 1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671,
       1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682,
       1683, 1684, 1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693,
       1694, 1695, 1696, 1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704,
       1705, 1706, 1707, 1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715,
       1716, 1717, 1718, 1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726,
       1727, 1728, 1729, 1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737,
       1738, 1739, 1740, 1741, 1742, 1743, 1744, 1745, 1746, 1747, 1748,
       1749, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 1757, 1758, 1759,
       1760, 1761, 1762, 1763, 1764, 1765, 1766, 1767, 1768, 1769, 1770,
       1771, 1772, 1773, 1774, 1775, 1776, 1777, 1778, 1779, 1780, 1781,
       1782, 1783, 1784, 1785, 1786, 1787, 1788, 1789, 1790, 1791, 1792,
       1793, 1794, 1795, 1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803,
       1804, 1805, 1806, 1807, 1808, 1809, 1810, 1811, 1812, 1813, 1814,
       1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825,
       1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836,
       1837, 1838, 1839, 1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847,
       1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858,
       1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1867, 1868, 1869,
       1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880,
       1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891,
       1892, 1893, 1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902,
       1903, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913,
       1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924,
       1925, 1926, 1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935,
       1936, 1937, 1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946,
       1947, 1948, 1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957,
       1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968,
       1969, 1970, 1971, 1972, 1973, 1974, 1975, 1976, 1977, 1978, 1979,
       1980, 1981, 1982, 1983, 1984, 1985, 1986, 1987, 1988, 1989, 1990,
       1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001,
       2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012,
       2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023,
       2024, 2025, 2026, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034,
       2035, 2036, 2037, 2038, 2039, 2040, 2041, 2042, 2043, 2044, 2045,
       2046, 2047, 2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056,
       2057, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067,
       2068, 2069, 2070, 2071, 2072, 2073, 2074, 2075, 2076, 2077, 2078,
       2079, 2080, 2081, 2082, 2083, 2084, 2085, 2086, 2087, 2088, 2089,
       2090, 2091, 2092, 2093, 2094, 2095, 2096, 2097, 2098, 2099, 2100,
       2101, 2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2111,
       2112, 2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122,
       2123, 2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133,
       2134, 2135, 2136, 2137, 2138, 2139, 2140, 2141, 2142, 2143, 2144,
       2145, 2146, 2147, 2148, 2149, 2150, 2151, 2152, 2153, 2154, 2155,
       2156, 2157, 2158, 2159, 2160, 2161, 2162, 2163, 2164, 2165, 2166,
       2167, 2168, 2169, 2170, 2171, 2172, 2173, 2174, 2175, 2176, 2177,
       2178, 2179, 2180, 2181, 2182, 2183, 2184, 2185, 2186, 2187, 2188,
       2189, 2190, 2191, 2192, 2193, 2194, 2195, 2196, 2197, 2198, 2199,
       2200, 2201, 2202, 2203, 2204, 2205, 2206, 2207, 2208, 2209, 2210,
       2211, 2212, 2213, 2214, 2215, 2216, 2217, 2218, 2219, 2220, 2221,
       2222, 2223, 2224, 2225, 2226, 2227, 2228, 2229, 2230, 2231, 2232,
       2233, 2234, 2235, 2236, 2237, 2238, 2239, 2240, 2241, 2242, 2243,
       2244, 2245, 2246, 2247, 2248, 2249, 2250, 2251, 2252, 2253, 2254,
       2255, 2256, 2257, 2258, 2259, 2260, 2261, 2262, 2263, 2264, 2265,
       2266, 2267, 2268, 2269, 2270, 2271, 2272, 2273, 2274, 2275, 2276,
       2277, 2278, 2279, 2280, 2281, 2282, 2283, 2284, 2285, 2286, 2287,
       2288, 2289, 2290, 2291, 2292, 2293, 2294, 2295, 2296, 2297, 2298,
       2299, 2300, 2301, 2302, 2303, 2304, 2305, 2306, 2307, 2308, 2309,
       2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320,
       2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331,
       2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342,
       2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353,
       2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2362, 2363, 2364,
       2365, 2366, 2367, 2368, 2369, 2370, 2371, 2372, 2373, 2374, 2375,
       2376, 2377, 2378, 2379, 2380, 2381, 2382, 2383, 2384, 2385, 2386,
       2387, 2388, 2389, 2390, 2391, 2392, 2393, 2394, 2395, 2396, 2397,
       2398, 2399, 2400, 2401, 2402, 2403, 2404, 2405, 2406, 2407, 2408,
       2409, 2410, 2411, 2412, 2413, 2414, 2415, 2416, 2417, 2418, 2419,
       2420, 2421, 2422, 2423, 2424, 2425, 2426, 2427, 2428, 2429, 2430,
       2431, 2432, 2433, 2434, 2435, 2436, 2437, 2438, 2439, 2440, 2441,
       2442, 2443, 2444, 2445, 2446, 2447, 2448, 2449, 2450, 2451, 2452,
       2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463,
       2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2473, 2474,
       2475, 2476, 2477, 2478, 2479, 2480, 2481, 2482, 2483, 2484, 2485,
       2486, 2487, 2488, 2489, 2490, 2491, 2492, 2493, 2494, 2495, 2496,
       2497, 2498, 2499, 2500, 2501, 2502, 2503, 2504, 2505, 2506, 2507,
       2508, 2509, 2510, 2511, 2512, 2513, 2514, 2515, 2516, 2517, 2518,
       2519, 2520, 2521, 2522, 2523, 2524, 2525, 2526, 2527, 2528, 2529,
       2530, 2531, 2532, 2533, 2534, 2535, 2536, 2537, 2538, 2539, 2540,
       2541, 2542, 2543, 2544, 2545, 2546, 2547, 2548, 2549, 2550, 2551,
       2552, 2553, 2554, 2555, 2556, 2557, 2558, 2559, 2560, 2561, 2562,
       2563, 2564, 2565, 2566, 2567, 2568, 2569, 2570, 2571, 2572, 2573,
       2574, 2575, 2576, 2577, 2578, 2579, 2580, 2581, 2582, 2583, 2584,
       2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595,
       2596, 2597, 2598, 2599, 2600, 2601, 2602, 2603, 2604, 2605, 2606,
       2607, 2608, 2609, 2610, 2611, 2612, 2613, 2614, 2615, 2616, 2617,
       2618, 2619, 2620, 2621, 2622, 2623, 2624, 2625, 2626, 2627, 2628,
       2629, 2630, 2631, 2632, 2633, 2634, 2635, 2636, 2637, 2638, 2639,
       2640, 2641, 2642, 2643, 2644, 2645, 2646, 2647, 2648, 2649, 2650,
       2651, 2652, 2653, 2654, 2655, 2656, 2657, 2658, 2659, 2660, 2661,
       2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2672,
       2673, 2674, 2675, 2676, 2677, 2678, 2679, 2680, 2681, 2682, 2683,
       2684, 2685, 2686, 2687, 2688, 2689, 2690, 2691, 2692, 2693, 2694,
       2695, 2696, 2697, 2698, 2699, 2700, 2701, 2702, 2703, 2704, 2705,
       2706, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716,
       2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727,
       2728, 2729, 2730, 2731, 2732, 2733, 2734, 2735, 2736, 2737, 2738,
       2739, 2740, 2741, 2742, 2743, 2744, 2745, 2746, 2747, 2748, 2749,
       2750, 2751, 2752, 2753, 2754, 2755, 2756, 2757, 2758, 2759, 2760,
       2761, 2762, 2763, 2764, 2765, 2766, 2767, 2768, 2769, 2770, 2771,
       2772, 2773, 2774, 2775, 2776, 2777, 2778, 2779, 2780, 2781, 2782,
       2783, 2784, 2785, 2786, 2787, 2788, 2789, 2790, 2791, 2792, 2793,
       2794, 2795, 2796, 2797, 2798, 2799, 2800, 2801, 2802, 2803, 2804,
       2805, 2806, 2807, 2808, 2809, 2810, 2811, 2812, 2813, 2814, 2815,
       2816, 2817, 2818, 2819, 2820, 2821, 2822, 2823, 2824, 2825, 2826,
       2827, 2828, 2829, 2830, 2831, 2832, 2833, 2834, 2835, 2836, 2837,
       2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847, 2848,
       2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2857, 2858, 2859,
       2860, 2861, 2862, 2863, 2864, 2865, 2866, 2867, 2868, 2869, 2870,
       2871, 2872, 2873, 2874, 2875, 2876, 2877, 2878, 2879, 2880, 2881,
       2882, 2883, 2884, 2885, 2886, 2887, 2888, 2889, 2890, 2891, 2892,
       2893, 2894, 2895, 2896, 2897, 2898, 2899, 2900, 2901, 2902, 2903,
       2904, 2905, 2906, 2907, 2908, 2909, 2910, 2911, 2912, 2913, 2914,
       2915, 2916, 2917, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925,
       2926, 2927, 2928, 2929, 2930, 2931, 2932, 2933, 2934, 2935, 2936,
       2937, 2938, 2939, 2940, 2941, 2942, 2943, 2944, 2945, 2946, 2947,
       2948, 2949, 2950, 2951, 2952, 2953, 2954, 2955, 2956, 2957, 2958,
       2959, 2960, 2961, 2962, 2963, 2964, 2965, 2966, 2967, 2968, 2969,
       2970, 2971, 2972, 2973, 2974, 2975, 2976, 2977, 2978, 2979, 2980,
       2981, 2982, 2983, 2984, 2985, 2986, 2987, 2988, 2989, 2990, 2991,
       2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3001, 3002,
       3003, 3004, 3005, 3006, 3007, 3008, 3009, 3010, 3011, 3012, 3013,
       3014, 3015, 3016, 3017, 3018, 3019, 3020, 3021, 3022, 3023, 3024,
       3025, 3026, 3027, 3028, 3029, 3030, 3031, 3032, 3033, 3034, 3035,
       3036, 3037, 3038, 3039, 3040, 3041, 3042, 3043, 3044, 3045, 3046,
       3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055, 3056, 3057,
       3058, 3059, 3060, 3061, 3062, 3063, 3064, 3065, 3066, 3067, 3068,
       3069, 3070, 3071, 3072, 3073, 3074, 3075, 3076, 3077, 3078, 3079,
       3080, 3081, 3082, 3083, 3084, 3085, 3086, 3087, 3088, 3089, 3090,
       3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099, 3100, 3101,
       3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110, 3111, 3112,
       3113, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121, 3122, 3123,
       3124, 3125, 3126, 3127, 3128, 3129, 3130, 3131, 3132, 3133, 3134,
       3135, 3136, 3137, 3138, 3139, 3140, 3141, 3142, 3143, 3144, 3145,
       3146, 3147, 3148, 3149, 3150, 3151, 3152, 3153, 3154, 3155, 3156,
       3157, 3158, 3159, 3160, 3161, 3162, 3163, 3164, 3165, 3166, 3167,
       3168, 3169, 3170, 3171, 3172, 3173, 3174, 3175, 3176, 3177, 3178,
       3179, 3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 3189,
       3190, 3191, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3199, 3200,
       3201, 3202, 3203, 3204, 3205, 3206, 3207, 3208, 3209, 3210, 3211,
       3212, 3213, 3214, 3215, 3216, 3217, 3218, 3219, 3220, 3221, 3222,
       3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233,
       3234, 3235, 3236, 3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244,
       3245, 3246, 3247, 3248, 3249, 3250, 3251, 3252, 3253, 3254, 3255,
       3256, 3257, 3258, 3259, 3260, 3261, 3262, 3263, 3264, 3265, 3266,
       3267, 3268, 3269, 3270, 3271, 3272, 3273, 3274, 3275, 3276, 3277,
       3278, 3279, 3280, 3281, 3282, 3283, 3284, 3285, 3286, 3287, 3288,
       3289, 3290, 3291, 3292, 3293, 3294, 3295, 3296, 3297, 3298, 3299,
       3300, 3301], dtype=np.int32)
        data_f_gt = np.array([7.67257275e-04, 9.25985542e-05, 1.18086485e-04, 1.16932736e-04,
       1.45639159e-04, 1.45390924e-04, 1.48609416e-04, 9.25985542e-05,
       4.82251649e-04, 9.02503523e-05, 1.08373765e-04, 8.76269363e-05,
       1.03402041e-04, 1.18086485e-04, 9.02503523e-05, 9.25150926e-04,
       1.25335424e-04, 1.41425342e-04, 1.58728388e-04, 1.40635361e-04,
       1.50689574e-04, 1.16932736e-04, 1.08373765e-04, 7.88858946e-04,
       1.43346694e-04, 1.28698979e-04, 1.48797757e-04, 1.42709014e-04,
       8.76269363e-05, 1.25335424e-04, 7.22871216e-04, 9.34819551e-05,
       1.54263565e-04, 1.19545107e-04, 1.42618229e-04, 1.45639159e-04,
       1.41425342e-04, 8.75533642e-04, 1.52672394e-04, 1.37386826e-04,
       1.54740836e-04, 1.43669085e-04, 1.45390924e-04, 1.43346694e-04,
       9.18861735e-04, 1.56403241e-04, 1.55361712e-04, 1.59680932e-04,
       1.58678231e-04, 1.03402041e-04, 1.28698979e-04, 9.34819551e-05,
       7.04454070e-04, 1.42658513e-04, 1.06166481e-04, 1.30046100e-04,
       1.58728388e-04, 1.54263565e-04, 9.01788738e-04, 1.53728123e-04,
       1.45270706e-04, 1.46895275e-04, 1.42902681e-04, 1.48609416e-04,
       1.52672394e-04, 1.56403241e-04, 9.35534001e-04, 1.59811620e-04,
       1.59345964e-04, 1.58691365e-04, 1.40635361e-04, 1.37386826e-04,
       8.53400483e-04, 1.46554867e-04, 1.40155368e-04, 1.45909512e-04,
       1.42758549e-04, 1.48797757e-04, 1.55361712e-04, 9.53421086e-04,
       1.56038318e-04, 1.62925190e-04, 1.65310513e-04, 1.64987595e-04,
       1.42709014e-04, 1.42658513e-04, 1.56038318e-04, 9.29376339e-04,
       1.58843848e-04, 1.63135308e-04, 1.65991339e-04, 1.19545107e-04,
       1.06166481e-04, 6.99531309e-04, 1.27970027e-04, 1.10350279e-04,
       1.15629147e-04, 1.19870269e-04, 1.50689574e-04, 1.53728123e-04,
       1.46554867e-04, 9.00485752e-04, 1.49712786e-04, 1.49840516e-04,
       1.49959886e-04, 1.42618229e-04, 1.45270706e-04, 1.27970027e-04,
       8.07894100e-04, 1.37095296e-04, 1.24233525e-04, 1.30706317e-04,
       1.54740836e-04, 1.59811620e-04, 9.38134599e-04, 1.52584632e-04,
       1.59964351e-04, 1.56671048e-04, 1.54362113e-04, 1.43669085e-04,
       1.40155368e-04, 1.52584632e-04, 8.77077586e-04, 1.43105943e-04,
       1.51763765e-04, 1.45798794e-04, 1.59680932e-04, 1.62925190e-04,
       9.53294136e-04, 1.55968091e-04, 1.62415825e-04, 1.57753786e-04,
       1.54550311e-04, 1.58678231e-04, 1.59345964e-04, 1.55968091e-04,
       9.28988130e-04, 1.56127636e-04, 1.49688198e-04, 1.49180009e-04,
       1.30046100e-04, 1.58843848e-04, 1.10350279e-04, 8.12013163e-04,
       1.64057204e-04, 1.11903277e-04, 1.36812455e-04, 1.46895275e-04,
       1.49712786e-04, 8.83615903e-04, 1.43814047e-04, 1.51595416e-04,
       1.43317260e-04, 1.48281119e-04, 1.42902681e-04, 1.37095296e-04,
       1.43814047e-04, 7.98647691e-04, 1.25099324e-04, 1.31321840e-04,
       1.18414503e-04, 1.58691365e-04, 1.59964351e-04, 1.56127636e-04,
       9.43437271e-04, 1.59465382e-04, 1.53561888e-04, 1.55626649e-04,
       1.45909512e-04, 1.49840516e-04, 8.81345515e-04, 1.42614832e-04,
       1.49861714e-04, 1.48217409e-04, 1.44901531e-04, 1.42758549e-04,
       1.43105943e-04, 1.42614832e-04, 8.56258200e-04, 1.43759639e-04,
       1.41610912e-04, 1.42408325e-04, 1.65310513e-04, 1.62415825e-04,
       9.83097799e-04, 1.67924464e-04, 1.59883750e-04, 1.61208610e-04,
       1.66354636e-04, 1.64987595e-04, 1.63135308e-04, 1.67924464e-04,
       1.01157113e-03, 1.71261416e-04, 1.69536552e-04, 1.74725791e-04,
       1.65991339e-04, 1.64057204e-04, 1.71261416e-04, 1.02315530e-03,
       1.68904042e-04, 1.76682267e-04, 1.76259028e-04, 1.15629147e-04,
       1.11903277e-04, 7.19122533e-04, 1.22786749e-04, 1.18222954e-04,
       1.24871241e-04, 1.25709165e-04, 1.19870269e-04, 1.24233525e-04,
       1.22786749e-04, 7.51562574e-04, 1.29245843e-04, 1.26665175e-04,
       1.28761013e-04, 1.49959886e-04, 1.51595416e-04, 1.49861714e-04,
       9.08375164e-04, 1.53922689e-04, 1.50305006e-04, 1.52730451e-04,
       1.30706317e-04, 1.25099324e-04, 1.29245843e-04, 7.56752935e-04,
       1.20978994e-04, 1.26691156e-04, 1.24031300e-04, 1.56671048e-04,
       1.59465382e-04, 9.38679530e-04, 1.53828623e-04, 1.54791783e-04,
       1.57876934e-04, 1.56045760e-04, 1.54362113e-04, 1.51763765e-04,
       1.53828623e-04, 9.14906466e-04, 1.50082993e-04, 1.53008127e-04,
       1.51860845e-04, 1.45798794e-04, 1.43759639e-04, 1.50082993e-04,
       8.81964433e-04, 1.45738062e-04, 1.49445361e-04, 1.47139584e-04,
       1.57753786e-04, 1.59883750e-04, 9.44455221e-04, 1.52670301e-04,
       1.57786149e-04, 1.56687675e-04, 1.59673559e-04, 1.54550311e-04,
       1.49688198e-04, 1.52670301e-04, 8.78367462e-04, 1.41631875e-04,
       1.43001544e-04, 1.36825232e-04, 1.49180009e-04, 1.53561888e-04,
       1.41631875e-04, 8.63515997e-04, 1.50561059e-04, 1.32016930e-04,
       1.36564235e-04, 1.36812455e-04, 1.68904042e-04, 1.18222954e-04,
       8.62354881e-04, 1.68942404e-04, 1.25422581e-04, 1.44050444e-04,
       1.43317260e-04, 1.31321840e-04, 7.68778120e-04, 1.51668457e-04,
       1.10147122e-04, 1.30924678e-04, 1.01398763e-04, 1.48281119e-04,
       1.53922689e-04, 1.51668457e-04, 9.27097711e-04, 1.53366824e-04,
       1.61900913e-04, 1.57957710e-04, 1.18414503e-04, 1.20978994e-04,
       1.10147122e-04, 7.01036511e-04, 1.21004438e-04, 1.11099314e-04,
       1.19392140e-04, 1.55626649e-04, 1.54791783e-04, 1.50561059e-04,
       9.14637199e-04, 1.50726890e-04, 1.50965061e-04, 1.51965758e-04,
       1.48217409e-04, 1.50305006e-04, 8.95323163e-04, 1.47589563e-04,
       1.50195396e-04, 1.49248777e-04, 1.49767013e-04, 1.44901531e-04,
       1.41610912e-04, 1.47589563e-04, 8.61001363e-04, 1.38871115e-04,
       1.46728035e-04, 1.41300207e-04, 1.42408325e-04, 1.45738062e-04,
       1.38871115e-04, 8.42244685e-04, 1.42915370e-04, 1.35798648e-04,
       1.36513165e-04, 1.61208610e-04, 1.57786149e-04, 9.37432760e-04,
       1.58634689e-04, 1.58582732e-04, 1.48925037e-04, 1.52295541e-04,
       1.66354636e-04, 1.69536552e-04, 1.58634689e-04, 9.90979517e-04,
       1.71077136e-04, 1.58057987e-04, 1.67318518e-04, 1.74725791e-04,
       1.76682267e-04, 1.71077136e-04, 1.05636603e-03, 1.79918502e-04,
       1.73538722e-04, 1.80423612e-04, 1.76259028e-04, 1.68942404e-04,
       1.79918502e-04, 1.03242654e-03, 1.64105949e-04, 1.75848296e-04,
       1.67352366e-04, 1.24871241e-04, 1.25422581e-04, 8.12751479e-04,
       1.33447936e-04, 1.41218293e-04, 1.47505222e-04, 1.40286205e-04,
       1.25709165e-04, 1.26665175e-04, 1.33447936e-04, 7.93916008e-04,
       1.32412439e-04, 1.38836400e-04, 1.36844892e-04, 1.28761013e-04,
       1.26691156e-04, 1.32412439e-04, 7.86243937e-04, 1.29048562e-04,
       1.35312393e-04, 1.34018374e-04, 1.52730451e-04, 1.53366824e-04,
       1.50195396e-04, 9.07370062e-04, 1.52034418e-04, 1.48920162e-04,
       1.50122811e-04, 1.24031300e-04, 1.21004438e-04, 1.29048562e-04,
       7.63289967e-04, 1.24351778e-04, 1.33261905e-04, 1.31591983e-04,
       1.57876934e-04, 1.53008127e-04, 1.07065030e-03, 1.42206171e-04,
       1.63138145e-04, 1.22652785e-04, 1.79092206e-04, 1.52675933e-04,
       1.56045760e-04, 1.50726890e-04, 1.42206171e-04, 8.58994437e-04,
       1.48284635e-04, 1.25166823e-04, 1.36564158e-04, 1.51860845e-04,
       1.49445361e-04, 1.63138145e-04, 9.36427417e-04, 1.50926303e-04,
       1.65426561e-04, 1.55630203e-04, 1.47139584e-04, 1.42915370e-04,
       1.50926303e-04, 8.74472141e-04, 1.41313975e-04, 1.48782511e-04,
       1.43394398e-04, 1.56687675e-04, 1.43001544e-04, 9.17345855e-04,
       1.57055661e-04, 1.33141803e-04, 1.58615722e-04, 1.68843449e-04,
       1.59673559e-04, 1.58582732e-04, 1.57055661e-04, 9.38267567e-04,
       1.53312688e-04, 1.56147536e-04, 1.53495390e-04, 1.36825232e-04,
       1.32016930e-04, 1.33141803e-04, 6.53750628e-04, 1.24028350e-04,
       1.27738312e-04, 1.36564235e-04, 1.50965061e-04, 1.24028350e-04,
       8.89784166e-04, 1.51804462e-04, 1.56523387e-04, 1.69898672e-04,
       1.44050444e-04, 1.64105949e-04, 1.41218293e-04, 9.61539874e-04,
       1.48141722e-04, 1.58475560e-04, 9.42495697e-05, 1.11298336e-04,
       1.30924678e-04, 1.61900913e-04, 9.96100764e-04, 1.00890248e-04,
       1.58027561e-04, 1.29767102e-04, 1.55937587e-04, 1.58652675e-04,
       1.01398763e-04, 1.11099314e-04, 1.00890248e-04, 5.25102510e-04,
       1.10707024e-04, 1.01007161e-04, 1.57957710e-04, 1.52034418e-04,
       1.58027561e-04, 9.24655879e-04, 1.50500286e-04, 1.53869944e-04,
       1.52265960e-04, 1.19392140e-04, 1.24351778e-04, 1.10707024e-04,
       7.44720170e-04, 1.33554113e-04, 1.19413832e-04, 1.37301283e-04,
       1.51965758e-04, 1.48284635e-04, 1.51804462e-04, 9.00286417e-04,
       1.45885492e-04, 1.52291959e-04, 1.50054111e-04, 1.49248777e-04,
       1.46728035e-04, 8.85517727e-04, 1.49811194e-04, 1.45981926e-04,
       1.46219634e-04, 1.47528160e-04, 1.49767013e-04, 1.48920162e-04,
       1.49811194e-04, 9.01337867e-04, 1.50413071e-04, 1.50488849e-04,
       1.51937578e-04, 1.41300207e-04, 1.35798648e-04, 1.45981926e-04,
       8.44097718e-04, 1.34018634e-04, 1.46730019e-04, 1.40268285e-04,
       1.36513165e-04, 1.41313975e-04, 1.34018634e-04, 8.29227803e-04,
       1.41494919e-04, 1.36605818e-04, 1.39281293e-04, 1.48925037e-04,
       1.58057987e-04, 8.22674570e-04, 1.44576732e-04, 1.51867843e-04,
       1.10544404e-04, 1.08702566e-04, 1.52295541e-04, 1.53312688e-04,
       1.44576732e-04, 8.76862294e-04, 1.43368209e-04, 1.42767397e-04,
       1.40541727e-04, 1.67318518e-04, 1.73538722e-04, 1.51867843e-04,
       9.89670989e-04, 1.73692824e-04, 1.53824153e-04, 1.69428929e-04,
       1.80423612e-04, 1.75848296e-04, 1.73692824e-04, 1.04172831e-03,
       1.68650119e-04, 1.71790422e-04, 1.71323034e-04, 1.67352366e-04,
       1.48141722e-04, 1.68650119e-04, 9.27815152e-04, 1.32373750e-04,
       1.64181460e-04, 1.47115735e-04, 1.47505222e-04, 1.58475560e-04,
       9.19200492e-04, 1.45886484e-04, 1.61188749e-04, 1.50906275e-04,
       1.55238202e-04, 1.40286205e-04, 1.38836400e-04, 1.45886484e-04,
       8.58675008e-04, 1.40893430e-04, 1.48157869e-04, 1.44614620e-04,
       1.36844892e-04, 1.35312393e-04, 1.40893430e-04, 8.34360163e-04,
       1.37434581e-04, 1.42900609e-04, 1.40974259e-04, 1.34018374e-04,
       1.33261905e-04, 1.37434581e-04, 8.20053882e-04, 1.36062463e-04,
       1.39946105e-04, 1.39330455e-04, 1.50122811e-04, 1.50500286e-04,
       1.50413071e-04, 9.00742594e-04, 1.50303696e-04, 1.49944789e-04,
       1.49457939e-04, 1.31591983e-04, 1.33554113e-04, 1.36062463e-04,
       8.14091397e-04, 1.36792871e-04, 1.38660845e-04, 1.37429122e-04,
       1.22652785e-04, 1.25166823e-04, 6.24389484e-04, 1.29438650e-04,
       1.21121953e-04, 1.26009272e-04, 1.79092206e-04, 1.65426561e-04,
       9.29736860e-04, 1.49461791e-04, 1.56069748e-04, 1.43372121e-04,
       1.36314432e-04, 1.36564158e-04, 1.45885492e-04, 1.29438650e-04,
       8.28617993e-04, 1.44677552e-04, 1.33067287e-04, 1.38984855e-04,
       1.55630203e-04, 1.48782511e-04, 1.49461791e-04, 8.83126502e-04,
       1.43308197e-04, 1.42624852e-04, 1.43318949e-04, 1.43394398e-04,
       1.41494919e-04, 1.43308197e-04, 8.58932157e-04, 1.42847852e-04,
       1.43223829e-04, 1.44662963e-04, 1.58615722e-04, 1.27738312e-04,
       1.56523387e-04, 1.16365976e-03, 1.80421899e-04, 1.85650551e-04,
       1.75847029e-04, 1.78862864e-04, 1.68843449e-04, 1.56147536e-04,
       1.80421899e-04, 9.89058904e-04, 1.56528714e-04, 1.69157289e-04,
       1.57960017e-04, 1.53495390e-04, 1.43368209e-04, 1.56528714e-04,
       8.93432433e-04, 1.42867295e-04, 1.50353531e-04, 1.46819293e-04,
       1.69898672e-04, 1.52291959e-04, 1.85650551e-04, 9.88075952e-04,
       1.53050589e-04, 1.71088715e-04, 1.56095467e-04, 9.42495697e-05,
       1.32373750e-04, 5.36322306e-04, 6.46609762e-05, 1.39352817e-04,
       1.05685192e-04, 1.11298336e-04, 1.61188749e-04, 6.46609762e-05,
       7.52946020e-04, 1.56211800e-04, 1.08962874e-04, 1.50623285e-04,
       1.29767102e-04, 1.01007161e-04, 1.19413832e-04, 7.95339527e-04,
       1.43332458e-04, 1.53330145e-04, 1.48488830e-04, 1.55937587e-04,
       1.53869944e-04, 9.18274387e-04, 1.53228884e-04, 1.54271780e-04,
       1.49970722e-04, 1.50995470e-04, 1.52265960e-04, 1.50303696e-04,
       1.53228884e-04, 9.09194475e-04, 1.50731954e-04, 1.51599323e-04,
       1.51064657e-04, 1.37301283e-04, 1.36792871e-04, 1.43332458e-04,
       8.44016695e-04, 1.39044244e-04, 1.45662819e-04, 1.41883018e-04,
       1.52675933e-04, 1.21121953e-04, 1.56069748e-04, 8.18198676e-04,
       1.25208718e-04, 1.31214686e-04, 1.31907636e-04, 1.50054111e-04,
       1.44677552e-04, 1.53050589e-04, 8.87422789e-04, 1.43702213e-04,
       1.49955070e-04, 1.45983254e-04, 1.46219634e-04, 1.46730019e-04,
       8.72899699e-04, 1.45584312e-04, 1.46293076e-04, 1.43937140e-04,
       1.44135519e-04, 1.47528160e-04, 1.50488849e-04, 1.45584312e-04,
       8.83102350e-04, 1.50821833e-04, 1.43201181e-04, 1.45478013e-04,
       1.51937578e-04, 1.49944789e-04, 1.50821833e-04, 8.92197952e-04,
       1.48369137e-04, 1.45792261e-04, 1.45332353e-04, 1.40268285e-04,
       1.36605818e-04, 1.46293076e-04, 8.51683268e-04, 1.39515311e-04,
       1.46058038e-04, 1.42942741e-04, 1.39281293e-04, 1.42847852e-04,
       1.39515311e-04, 8.58727141e-04, 1.46091442e-04, 1.43990836e-04,
       1.47000407e-04, 1.10544404e-04, 1.42767397e-04, 6.58480966e-04,
       8.46477021e-05, 1.21836236e-04, 1.01825384e-04, 9.68598434e-05,
       1.08702566e-04, 1.53824153e-04, 8.46477021e-05, 6.41208699e-04,
       1.65801919e-04, 1.28232359e-04, 1.40541727e-04, 1.42867295e-04,
       1.21836236e-04, 8.01147451e-04, 1.41343177e-04, 1.18688822e-04,
       1.35870195e-04, 1.69428929e-04, 1.71790422e-04, 1.65801919e-04,
       1.00726456e-03, 1.73548740e-04, 1.60654609e-04, 1.66039936e-04,
       1.71323034e-04, 1.64181460e-04, 1.73548740e-04, 1.00532752e-03,
       1.60820183e-04, 1.70520541e-04, 1.64933559e-04, 1.47115735e-04,
       1.39352817e-04, 1.60820183e-04, 8.95400611e-04, 1.45001380e-04,
       1.55583191e-04, 1.47527305e-04, 1.50906275e-04, 1.48157869e-04,
       8.98550780e-04, 1.52402039e-04, 1.47120774e-04, 1.51248341e-04,
       1.48715482e-04, 1.55238202e-04, 1.56211800e-04, 1.52402039e-04,
       9.13742503e-04, 1.48015751e-04, 1.53617298e-04, 1.48257412e-04,
       1.44614620e-04, 1.42900609e-04, 1.47120774e-04, 8.71319920e-04,
       1.43169823e-04, 1.47875517e-04, 1.45638578e-04, 1.40974259e-04,
       1.39946105e-04, 1.43169823e-04, 8.52889797e-04, 1.41154652e-04,
       1.44315988e-04, 1.43328971e-04, 1.39330455e-04, 1.38660845e-04,
       1.41154652e-04, 8.44910982e-04, 1.39949258e-04, 1.43175778e-04,
       1.42639995e-04, 1.58652675e-04, 1.53330145e-04, 1.54271780e-04,
       9.09535574e-04, 1.47910851e-04, 1.48204260e-04, 1.47165863e-04,
       1.49457939e-04, 1.50731954e-04, 1.48369137e-04, 8.88495218e-04,
       1.49151881e-04, 1.45637788e-04, 1.45146518e-04, 1.37429122e-04,
       1.39044244e-04, 1.39949258e-04, 8.40590028e-04, 1.41146129e-04,
       1.41719763e-04, 1.41301511e-04, 1.26009272e-04, 1.33067287e-04,
       1.25208718e-04, 7.89070800e-04, 1.35833143e-04, 1.32692985e-04,
       1.36259395e-04, 1.43372121e-04, 1.42624852e-04, 8.39124239e-04,
       1.38566929e-04, 1.40882372e-04, 1.35307627e-04, 1.38370339e-04,
       1.36314432e-04, 1.31214686e-04, 1.38566929e-04, 8.09193658e-04,
       1.32611784e-04, 1.35670613e-04, 1.34815213e-04, 1.38984855e-04,
       1.43702213e-04, 1.35833143e-04, 8.41800655e-04, 1.43120976e-04,
       1.38794496e-04, 1.41364971e-04, 1.43318949e-04, 1.43223829e-04,
       1.40882372e-04, 8.49502905e-04, 1.41835562e-04, 1.39596942e-04,
       1.40645252e-04, 1.44662963e-04, 1.46091442e-04, 1.41835562e-04,
       8.65913050e-04, 1.46007815e-04, 1.42285747e-04, 1.45029522e-04,
       1.75847029e-04, 1.69157289e-04, 9.93365050e-04, 1.58416760e-04,
       1.71053879e-04, 1.62418737e-04, 1.56471357e-04, 1.57960017e-04,
       1.50353531e-04, 1.58416760e-04, 9.15807919e-04, 1.48725575e-04,
       1.51218368e-04, 1.49133669e-04, 1.46819293e-04, 1.41343177e-04,
       1.48725575e-04, 8.74563268e-04, 1.42484815e-04, 1.47977526e-04,
       1.47212882e-04, 1.78862864e-04, 1.71088715e-04, 1.71053879e-04,
       9.96456998e-04, 1.58109271e-04, 1.61256363e-04, 1.56085905e-04,
       1.56095467e-04, 1.49955070e-04, 1.58109271e-04, 9.10679465e-04,
       1.47565140e-04, 1.51679125e-04, 1.47275391e-04, 1.05685192e-04,
       1.08962874e-04, 1.45001380e-04, 9.27864720e-04, 1.47375796e-04,
       1.44095946e-04, 1.36996419e-04, 1.39747114e-04, 1.50623285e-04,
       1.48015751e-04, 1.47375796e-04, 8.74320699e-04, 1.46511025e-04,
       1.40026040e-04, 1.41768802e-04, 1.48488830e-04, 1.45662819e-04,
       1.47910851e-04, 8.74817627e-04, 1.43447137e-04, 1.45472847e-04,
       1.43835143e-04, 1.49970722e-04, 1.48204260e-04, 8.86002284e-04,
       1.48681884e-04, 1.46176789e-04, 1.46853631e-04, 1.46114998e-04,
       1.50995470e-04, 1.51599323e-04, 1.48681884e-04, 8.94617804e-04,
       1.48897967e-04, 1.47415465e-04, 1.47027695e-04, 1.51064657e-04,
       1.49151881e-04, 1.48897967e-04, 8.87337593e-04, 1.46026809e-04,
       1.46577330e-04, 1.45618948e-04, 1.41883018e-04, 1.41146129e-04,
       1.43447137e-04, 8.50363072e-04, 1.41241502e-04, 1.42057016e-04,
       1.40588270e-04, 1.31907636e-04, 1.32692985e-04, 1.32611784e-04,
       8.04649103e-04, 1.36090698e-04, 1.34326218e-04, 1.37019783e-04,
       1.45983254e-04, 1.43120976e-04, 1.47565140e-04, 8.69284152e-04,
       1.43196213e-04, 1.45462610e-04, 1.43955959e-04, 1.43937140e-04,
       1.43201181e-04, 8.35899482e-04, 1.40973253e-04, 1.38556937e-04,
       1.33775307e-04, 1.35455665e-04, 1.44135519e-04, 1.46058038e-04,
       1.40973253e-04, 8.53747377e-04, 1.45615793e-04, 1.37122377e-04,
       1.39842397e-04, 1.45478013e-04, 1.45792261e-04, 1.38556937e-04,
       8.39861410e-04, 1.38929838e-04, 1.35522854e-04, 1.35581507e-04,
       1.45332353e-04, 1.45637788e-04, 1.38929838e-04, 8.54987738e-04,
       1.43093497e-04, 1.39068019e-04, 1.42926243e-04, 1.42942741e-04,
       1.43990836e-04, 1.45615793e-04, 8.76233012e-04, 1.48531108e-04,
       1.46642658e-04, 1.48509878e-04, 1.47000407e-04, 1.46007815e-04,
       1.48531108e-04, 8.98992882e-04, 1.48128141e-04, 1.54367893e-04,
       1.54957518e-04, 1.01825384e-04, 1.18688822e-04, 5.93504799e-04,
       1.13969870e-04, 1.31717124e-04, 1.27303598e-04, 9.68598434e-05,
       1.28232359e-04, 1.60654609e-04, 1.13969870e-04, 9.46230048e-04,
       1.57362742e-04, 1.36759944e-04, 1.52390680e-04, 1.35870195e-04,
       1.42484815e-04, 1.31717124e-04, 8.34363044e-04, 1.44761091e-04,
       1.36550236e-04, 1.42979583e-04, 1.66039936e-04, 1.70520541e-04,
       1.57362742e-04, 9.64142057e-04, 1.61516255e-04, 1.54514838e-04,
       1.54187746e-04, 1.64933559e-04, 1.55583191e-04, 1.61516255e-04,
       9.44540047e-04, 1.53142386e-04, 1.54194078e-04, 1.55170578e-04,
       1.47527305e-04, 1.44095946e-04, 1.53142386e-04, 8.78602179e-04,
       1.38456425e-04, 1.53317360e-04, 1.42062757e-04, 1.51248341e-04,
       1.53617298e-04, 9.05625549e-04, 1.48822928e-04, 1.52184984e-04,
       1.50372549e-04, 1.49379449e-04, 1.48715482e-04, 1.47875517e-04,
       1.48822928e-04, 8.90649452e-04, 1.47563485e-04, 1.49045759e-04,
       1.48626282e-04, 1.48257412e-04, 1.46511025e-04, 1.52184984e-04,
       8.91132793e-04, 1.46119206e-04, 1.51189778e-04, 1.46870388e-04,
       1.45638578e-04, 1.44315988e-04, 1.47563485e-04, 8.76718122e-04,
       1.44733721e-04, 1.47986763e-04, 1.46479588e-04, 1.43328971e-04,
       1.43175778e-04, 1.44733721e-04, 8.67466271e-04, 1.44609332e-04,
       1.45794833e-04, 1.45823637e-04, 1.42639995e-04, 1.41719763e-04,
       1.44609332e-04, 8.60252332e-04, 1.42059336e-04, 1.45426836e-04,
       1.43797071e-04, 1.47165863e-04, 1.45472847e-04, 1.46176789e-04,
       8.68668506e-04, 1.42966030e-04, 1.44202360e-04, 1.42684617e-04,
       1.45146518e-04, 1.46026809e-04, 1.43093497e-04, 8.72850480e-04,
       1.46720924e-04, 1.44557798e-04, 1.47304934e-04, 1.41301511e-04,
       1.41241502e-04, 1.42059336e-04, 8.50884045e-04, 1.41502535e-04,
       1.42637976e-04, 1.42141185e-04, 1.36259395e-04, 1.38794496e-04,
       1.36090698e-04, 8.31582398e-04, 1.40584048e-04, 1.38947756e-04,
       1.40906005e-04, 1.35307627e-04, 1.35670613e-04, 8.08071581e-04,
       1.35127733e-04, 1.34795804e-04, 1.33932360e-04, 1.33237445e-04,
       1.38370339e-04, 1.39596942e-04, 1.35127733e-04, 8.19944443e-04,
       1.37972994e-04, 1.33628888e-04, 1.35247546e-04, 1.34815213e-04,
       1.34326218e-04, 1.34795804e-04, 8.14931375e-04, 1.37784913e-04,
       1.34865561e-04, 1.38343665e-04, 1.41364971e-04, 1.43196213e-04,
       1.40584048e-04, 8.53334317e-04, 1.43326597e-04, 1.41975590e-04,
       1.42886898e-04, 1.40645252e-04, 1.42285747e-04, 1.37972994e-04,
       8.42056643e-04, 1.42501677e-04, 1.37881393e-04, 1.40769581e-04,
       1.45029522e-04, 1.48128141e-04, 1.42501677e-04, 8.75077838e-04,
       1.48684070e-04, 1.43825328e-04, 1.46909102e-04, 1.62418737e-04,
       1.61256363e-04, 9.19883647e-04, 1.52412361e-04, 1.53032738e-04,
       1.46273099e-04, 1.44490348e-04, 1.56471357e-04, 1.51218368e-04,
       1.52412361e-04, 8.91272759e-04, 1.45865188e-04, 1.43299835e-04,
       1.42005651e-04, 1.49133669e-04, 1.47977526e-04, 1.45865188e-04,
       8.77250139e-04, 1.46405137e-04, 1.43086264e-04, 1.44782356e-04,
       1.47212882e-04, 1.44761091e-04, 1.46405137e-04, 8.79055022e-04,
       1.45455101e-04, 1.46859528e-04, 1.48361283e-04, 1.56085905e-04,
       1.51679125e-04, 1.53032738e-04, 8.97518486e-04, 1.46345901e-04,
       1.46327437e-04, 1.44047379e-04, 1.47275391e-04, 1.45462610e-04,
       1.46345901e-04, 8.71083254e-04, 1.44426263e-04, 1.43839973e-04,
       1.43733116e-04, 1.36996419e-04, 1.38456425e-04, 7.92117201e-04,
       1.32923061e-04, 1.31643034e-04, 1.27419148e-04, 1.24679115e-04,
       1.39747114e-04, 1.40026040e-04, 1.32923061e-04, 8.23237108e-04,
       1.37677129e-04, 1.34194311e-04, 1.38669453e-04, 1.41768802e-04,
       1.46119206e-04, 1.37677129e-04, 8.55992891e-04, 1.45283209e-04,
       1.40944434e-04, 1.44200110e-04, 1.43835143e-04, 1.42057016e-04,
       1.42966030e-04, 8.49948814e-04, 1.39638142e-04, 1.41501122e-04,
       1.39951360e-04, 1.46853631e-04, 1.47415465e-04, 8.81210259e-04,
       1.46013044e-04, 1.48160579e-04, 1.45590920e-04, 1.47176621e-04,
       1.46114998e-04, 1.44202360e-04, 1.46013044e-04, 8.65842259e-04,
       1.42598446e-04, 1.44207686e-04, 1.42705726e-04, 1.47027695e-04,
       1.46577330e-04, 1.48160579e-04, 8.87489626e-04, 1.47719609e-04,
       1.48997509e-04, 1.49006904e-04, 1.45618948e-04, 1.46720924e-04,
       1.47719609e-04, 8.85394623e-04, 1.49165788e-04, 1.47912576e-04,
       1.48256779e-04, 1.40588270e-04, 1.41502535e-04, 1.39638142e-04,
       8.45738935e-04, 1.41645565e-04, 1.40635632e-04, 1.41728790e-04,
       1.37019783e-04, 1.38947756e-04, 1.37784913e-04, 8.37723691e-04,
       1.41295706e-04, 1.40546356e-04, 1.42129176e-04, 1.43955959e-04,
       1.43326597e-04, 1.44426263e-04, 8.61667441e-04, 1.43119041e-04,
       1.43758721e-04, 1.43080860e-04, 1.33775307e-04, 1.35522854e-04,
       7.92052784e-04, 1.31887348e-04, 1.33849288e-04, 1.28616190e-04,
       1.28401797e-04, 1.35455665e-04, 1.37122377e-04, 1.31887348e-04,
       7.98987936e-04, 1.33806485e-04, 1.30231819e-04, 1.30484242e-04,
       1.39842397e-04, 1.46642658e-04, 1.33806485e-04, 8.17167481e-04,
       1.42313692e-04, 1.26427652e-04, 1.28134598e-04, 1.35581507e-04,
       1.39068019e-04, 1.33849288e-04, 8.28820323e-04, 1.42482667e-04,
       1.36345988e-04, 1.41492854e-04, 1.42926243e-04, 1.44557798e-04,
       1.42482667e-04, 8.68336230e-04, 1.47670170e-04, 1.43571702e-04,
       1.47127650e-04, 1.48509878e-04, 1.54367893e-04, 1.42313692e-04,
       9.19116689e-04, 1.62721574e-04, 1.48326893e-04, 1.62876760e-04,
       1.54957518e-04, 1.48684070e-04, 1.62721574e-04, 9.28431950e-04,
       1.50687589e-04, 1.58570868e-04, 1.52810331e-04, 1.27303598e-04,
       1.36759944e-04, 1.36550236e-04, 8.11899692e-04, 1.41438418e-04,
       1.37207829e-04, 1.32639666e-04, 1.52390680e-04, 1.54514838e-04,
       1.41438418e-04, 8.41591344e-04, 1.40005762e-04, 1.28399230e-04,
       1.24842416e-04, 1.42979583e-04, 1.45455101e-04, 1.37207829e-04,
       8.52603553e-04, 1.49635893e-04, 1.33686300e-04, 1.43638847e-04,
       1.54187746e-04, 1.54194078e-04, 1.40005762e-04, 8.86903166e-04,
       1.54274097e-04, 1.34989740e-04, 1.49251742e-04, 1.55170578e-04,
       1.53317360e-04, 1.54274097e-04, 9.28180266e-04, 1.51267569e-04,
       1.57651986e-04, 1.56498676e-04, 1.42062757e-04, 1.31643034e-04,
       1.51267569e-04, 8.29687304e-04, 1.24053365e-04, 1.48727530e-04,
       1.31933049e-04, 1.50372549e-04, 1.51189778e-04, 9.00977846e-04,
       1.49834400e-04, 1.49593387e-04, 1.50522987e-04, 1.49464746e-04,
       1.49379449e-04, 1.49045759e-04, 1.49834400e-04, 8.97455283e-04,
       1.49279951e-04, 1.50068241e-04, 1.49847483e-04, 1.48626282e-04,
       1.47986763e-04, 1.49279951e-04, 8.90825783e-04, 1.47666216e-04,
       1.49120394e-04, 1.48146178e-04, 1.46870388e-04, 1.45283209e-04,
       1.49593387e-04, 8.85591496e-04, 1.46286803e-04, 1.49638557e-04,
       1.47919152e-04, 1.46479588e-04, 1.45794833e-04, 1.47666216e-04,
       8.82444592e-04, 1.46662146e-04, 1.48080562e-04, 1.47761248e-04,
       1.45823637e-04, 1.45426836e-04, 1.46662146e-04, 8.75821793e-04,
       1.45025640e-04, 1.47061620e-04, 1.45821915e-04, 1.43797071e-04,
       1.42637976e-04, 1.45025640e-04, 8.61783033e-04, 1.42200382e-04,
       1.44894063e-04, 1.43227901e-04, 1.42684617e-04, 1.41501122e-04,
       1.42598446e-04, 8.53490313e-04, 1.41344128e-04, 1.42716412e-04,
       1.42645589e-04, 1.47304934e-04, 1.49165788e-04, 1.47670170e-04,
       8.97557765e-04, 1.49611838e-04, 1.51862110e-04, 1.51942924e-04,
       1.42141185e-04, 1.41645565e-04, 1.42200382e-04, 8.50217151e-04,
       1.41600612e-04, 1.41366779e-04, 1.41262628e-04, 1.40906005e-04,
       1.41975590e-04, 1.41295706e-04, 8.51666461e-04, 1.42537362e-04,
       1.42389864e-04, 1.42561934e-04, 1.33932360e-04, 1.33628888e-04,
       8.22765887e-04, 1.38137764e-04, 1.36437948e-04, 1.39616292e-04,
       1.41012635e-04, 1.33237445e-04, 1.34865561e-04, 1.38137764e-04,
       8.40250243e-04, 1.41720538e-04, 1.45167139e-04, 1.47121797e-04,
       1.35247546e-04, 1.37881393e-04, 1.36437948e-04, 8.31061104e-04,
       1.40831780e-04, 1.39451225e-04, 1.41211211e-04, 1.38343665e-04,
       1.40546356e-04, 1.41720538e-04, 8.50305162e-04, 1.42243890e-04,
       1.44565026e-04, 1.42885687e-04, 1.42886898e-04, 1.43119041e-04,
       1.42537362e-04, 8.55535540e-04, 1.42604911e-04, 1.42275961e-04,
       1.42111368e-04, 1.40769581e-04, 1.43825328e-04, 1.40831780e-04,
       8.61266784e-04, 1.41469707e-04, 1.48394105e-04, 1.45976285e-04,
       1.46909102e-04, 1.50687589e-04, 1.41469707e-04, 8.97393884e-04,
       1.58185061e-04, 1.43602780e-04, 1.56539646e-04, 1.46273099e-04,
       1.46327437e-04, 8.45221272e-04, 1.38918278e-04, 1.42085792e-04,
       1.37364921e-04, 1.34251745e-04, 1.44490348e-04, 1.43299835e-04,
       1.38918278e-04, 8.16687749e-04, 1.33594984e-04, 1.30258542e-04,
       1.26125762e-04, 1.42005651e-04, 1.43086264e-04, 1.33594984e-04,
       8.12439039e-04, 1.39592059e-04, 1.24621810e-04, 1.29538271e-04,
       1.44782356e-04, 1.46859528e-04, 1.39592059e-04, 8.58968172e-04,
       1.46758440e-04, 1.37943290e-04, 1.43032498e-04, 1.48361283e-04,
       1.49635893e-04, 1.46758440e-04, 8.83702621e-04, 1.46209746e-04,
       1.47280281e-04, 1.45456977e-04, 1.44047379e-04, 1.43839973e-04,
       1.42085792e-04, 8.49932955e-04, 1.41340713e-04, 1.39578386e-04,
       1.39040713e-04, 1.43733116e-04, 1.43758721e-04, 1.41340713e-04,
       8.49585346e-04, 1.41838753e-04, 1.39220804e-04, 1.39693240e-04,
       1.27419148e-04, 1.34194311e-04, 7.80776077e-04, 1.22993037e-04,
       1.34769851e-04, 1.28199039e-04, 1.33200690e-04, 1.24679115e-04,
       1.24053365e-04, 1.22993037e-04, 7.26561310e-04, 1.14480664e-04,
       1.24120876e-04, 1.16234253e-04, 1.38669453e-04, 1.40944434e-04,
       1.34769851e-04, 8.36278738e-04, 1.42577758e-04, 1.36892158e-04,
       1.42425084e-04, 1.44200110e-04, 1.46286803e-04, 1.42577758e-04,
       8.69787340e-04, 1.46433110e-04, 1.44260451e-04, 1.46029109e-04,
       1.39951360e-04, 1.40635632e-04, 1.41344128e-04, 8.44707392e-04,
       1.40876010e-04, 1.41526326e-04, 1.40373937e-04, 1.45590920e-04,
       1.44207686e-04, 8.61003894e-04, 1.46851918e-04, 1.41398455e-04,
       1.39442343e-04, 1.43512572e-04, 1.47176621e-04, 1.48997509e-04,
       1.46851918e-04, 8.87649833e-04, 1.49337857e-04, 1.47310439e-04,
       1.47975490e-04, 1.42705726e-04, 1.42716412e-04, 1.41398455e-04,
       8.46855090e-04, 1.41218980e-04, 1.39502840e-04, 1.39312678e-04,
       1.49006904e-04, 1.47912576e-04, 1.49337857e-04, 8.95485759e-04,
       1.48575186e-04, 1.50160790e-04, 1.50492446e-04, 1.48256779e-04,
       1.49611838e-04, 1.48575186e-04, 8.93423872e-04, 1.48813047e-04,
       1.49642110e-04, 1.48524912e-04, 1.41728790e-04, 1.41600612e-04,
       1.40876010e-04, 8.46446267e-04, 1.40919982e-04, 1.40574362e-04,
       1.40746512e-04, 1.42129176e-04, 1.42389864e-04, 1.42243890e-04,
       8.54416508e-04, 1.42689950e-04, 1.42389128e-04, 1.42574500e-04,
       1.43080860e-04, 1.42604911e-04, 1.41838753e-04, 8.54818749e-04,
       1.42886628e-04, 1.41441886e-04, 1.42965711e-04, 1.28616190e-04,
       1.36345988e-04, 8.02789221e-04, 1.26125020e-04, 1.41659429e-04,
       1.38923603e-04, 1.31118991e-04, 1.28401797e-04, 1.30231819e-04,
       1.26125020e-04, 7.63624550e-04, 1.25297103e-04, 1.28113375e-04,
       1.25455437e-04, 1.30484242e-04, 1.26427652e-04, 1.25297103e-04,
       7.21032742e-04, 1.09994248e-04, 1.20037881e-04, 1.08791617e-04,
       1.28134598e-04, 1.48326893e-04, 1.09994248e-04, 7.75796751e-04,
       1.61379621e-04, 9.83841563e-05, 1.29577235e-04, 1.41492854e-04,
       1.43571702e-04, 1.41659429e-04, 8.57262877e-04, 1.44240060e-04,
       1.42898524e-04, 1.43400307e-04, 1.47127650e-04, 1.51862110e-04,
       1.44240060e-04, 8.96698619e-04, 1.54644609e-04, 1.46577051e-04,
       1.52247140e-04, 1.62876760e-04, 1.58570868e-04, 1.61379621e-04,
       9.55345785e-04, 1.21034080e-04, 1.48925518e-04, 8.96721887e-05,
       1.12886750e-04, 1.52810331e-04, 1.58185061e-04, 1.21034080e-04,
       8.07416254e-04, 1.59154542e-04, 9.17432548e-05, 1.24488986e-04,
       1.32639666e-04, 1.28399230e-04, 1.33686300e-04, 8.01754901e-04,
       1.20987436e-04, 1.47250349e-04, 1.38791920e-04, 1.24842416e-04,
       1.34989740e-04, 1.20987436e-04, 6.38472045e-04, 1.33917035e-04,
       1.23735418e-04, 1.43638847e-04, 1.46209746e-04, 1.47250349e-04,
       8.78602950e-04, 1.44468866e-04, 1.51193762e-04, 1.45841380e-04,
       1.49251742e-04, 1.57651986e-04, 1.33917035e-04, 8.83403773e-04,
       1.57109295e-04, 1.35340849e-04, 1.50132865e-04, 1.56498676e-04,
       1.48727530e-04, 1.57109295e-04, 9.15606830e-04, 1.45782190e-04,
       1.55522549e-04, 1.51966590e-04, 1.31933049e-04, 1.14480664e-04,
       1.45782190e-04, 7.62043075e-04, 1.08368789e-04, 1.40719700e-04,
       1.20758683e-04, 1.50522987e-04, 1.50068241e-04, 9.02372988e-04,
       1.50658055e-04, 1.49958036e-04, 1.50705471e-04, 1.50460197e-04,
       1.49464746e-04, 1.49638557e-04, 1.50658055e-04, 9.00685512e-04,
       1.49605111e-04, 1.51272900e-04, 1.50046144e-04, 1.49847483e-04,
       1.49120394e-04, 1.49958036e-04, 8.97243619e-04, 1.48558485e-04,
       1.50215842e-04, 1.49543379e-04, 1.48146178e-04, 1.48080562e-04,
       1.48558485e-04, 8.91395651e-04, 1.48564015e-04, 1.48987632e-04,
       1.49058779e-04, 1.47919152e-04, 1.46433110e-04, 1.49605111e-04,
       8.88795063e-04, 1.47392111e-04, 1.49086269e-04, 1.48359311e-04,
       1.47761248e-04, 1.47061620e-04, 1.48564015e-04, 8.85563039e-04,
       1.46447763e-04, 1.48572509e-04, 1.47155885e-04, 1.45821915e-04,
       1.44894063e-04, 1.46447763e-04, 8.71073589e-04, 1.43494223e-04,
       1.46220657e-04, 1.44194969e-04, 1.43227901e-04, 1.41366779e-04,
       1.43494223e-04, 8.48416354e-04, 1.39474703e-04, 1.41505573e-04,
       1.39347175e-04, 1.42645589e-04, 1.41526326e-04, 1.41218980e-04,
       8.44206804e-04, 1.39946243e-04, 1.39511571e-04, 1.39358096e-04,
       1.51942924e-04, 1.48813047e-04, 1.54644609e-04, 9.11383827e-04,
       1.47339498e-04, 1.56409491e-04, 1.52234257e-04, 1.41262628e-04,
       1.40919982e-04, 1.39474703e-04, 8.40850160e-04, 1.42000461e-04,
       1.37161991e-04, 1.40030396e-04, 1.42561934e-04, 1.42275961e-04,
       1.42689950e-04, 8.53711956e-04, 1.41567673e-04, 1.42726372e-04,
       1.41890067e-04, 1.39616292e-04, 1.39451225e-04, 8.52390134e-04,
       1.43981588e-04, 1.42191754e-04, 1.44387022e-04, 1.42762253e-04,
       1.41012635e-04, 1.45167139e-04, 1.43981588e-04, 8.75997838e-04,
       1.48548635e-04, 1.48437648e-04, 1.48850192e-04, 1.47121797e-04,
       1.44565026e-04, 1.48548635e-04, 8.76254493e-04, 1.43831337e-04,
       1.47174112e-04, 1.45013585e-04, 1.41211211e-04, 1.48394105e-04,
       1.42191754e-04, 8.98826861e-04, 1.53938506e-04, 1.54263714e-04,
       1.58827572e-04, 1.42885687e-04, 1.42389128e-04, 1.43831337e-04,
       8.58147704e-04, 1.42259641e-04, 1.43928524e-04, 1.42853387e-04,
       1.42111368e-04, 1.42886628e-04, 1.41567673e-04, 8.50114039e-04,
       1.41634704e-04, 1.41310948e-04, 1.40602719e-04, 1.45976285e-04,
       1.43602780e-04, 1.53938506e-04, 9.27164938e-04, 1.52677675e-04,
       1.64928509e-04, 1.66041183e-04, 1.56539646e-04, 1.59154542e-04,
       1.52677675e-04, 9.08753979e-04, 1.48916611e-04, 1.48920733e-04,
       1.42544773e-04, 1.37364921e-04, 1.39578386e-04, 8.07071552e-04,
       1.30712149e-04, 1.37111200e-04, 1.29059656e-04, 1.33245242e-04,
       1.34251745e-04, 1.30258542e-04, 1.30712149e-04, 7.27425866e-04,
       1.10243507e-04, 1.19217681e-04, 1.02742242e-04, 1.26125762e-04,
       1.24621810e-04, 1.10243507e-04, 6.68831747e-04, 1.22175183e-04,
       8.61149287e-05, 9.95505568e-05, 1.29538271e-04, 1.37943290e-04,
       1.22175183e-04, 7.90406136e-04, 1.39439077e-04, 1.26225720e-04,
       1.35084595e-04, 1.43032498e-04, 1.47280281e-04, 1.39439077e-04,
       8.62626728e-04, 1.47280784e-04, 1.41000081e-04, 1.44594005e-04,
       1.45456977e-04, 1.44468866e-04, 1.47280784e-04, 8.71486333e-04,
       1.46621196e-04, 1.43664993e-04, 1.43993516e-04, 1.39040713e-04,
       1.39220804e-04, 1.37111200e-04, 8.27001230e-04, 1.38136283e-04,
       1.36323619e-04, 1.37168611e-04, 1.39693240e-04, 1.41441886e-04,
       1.38136283e-04, 8.34542690e-04, 1.40419233e-04, 1.37158873e-04,
       1.37693176e-04, 1.28199039e-04, 1.24120876e-04, 7.78006905e-04,
       1.33498090e-04, 1.24869033e-04, 1.35935380e-04, 1.31384486e-04,
       1.33200690e-04, 1.36892158e-04, 1.33498090e-04, 8.18345162e-04,
       1.39740046e-04, 1.36231845e-04, 1.38782333e-04, 1.16234253e-04,
       1.08368789e-04, 1.24869033e-04, 7.35476535e-04, 1.18703346e-04,
       1.32800669e-04, 1.34500446e-04, 1.42425084e-04, 1.44260451e-04,
       1.39740046e-04, 8.57220583e-04, 1.45126819e-04, 1.41058389e-04,
       1.44609794e-04, 1.46029109e-04, 1.47392111e-04, 1.45126819e-04,
       8.81419885e-04, 1.48094259e-04, 1.46586574e-04, 1.48191013e-04,
       1.40373937e-04, 1.40574362e-04, 1.39946243e-04, 8.37874459e-04,
       1.40042006e-04, 1.38521287e-04, 1.38416625e-04, 1.39442343e-04,
       1.39502840e-04, 7.94960619e-04, 1.36989831e-04, 1.30046825e-04,
       1.27991141e-04, 1.20987640e-04, 1.43512572e-04, 1.47310439e-04,
       1.36989831e-04, 8.55150725e-04, 1.50303612e-04, 1.33759179e-04,
       1.43275093e-04, 1.47975490e-04, 1.50160790e-04, 1.50303612e-04,
       9.08033737e-04, 1.52175154e-04, 1.53866225e-04, 1.53552467e-04,
       1.39312678e-04, 1.39511571e-04, 1.30046825e-04, 8.09830825e-04,
       1.37419015e-04, 1.28183719e-04, 1.35357016e-04, 1.50492446e-04,
       1.49642110e-04, 1.52175154e-04, 9.10267702e-04, 1.49535200e-04,
       1.55106205e-04, 1.53316587e-04, 1.48524912e-04, 1.47339498e-04,
       1.49535200e-04, 8.85868041e-04, 1.46557550e-04, 1.47851558e-04,
       1.46059323e-04, 1.40746512e-04, 1.42000461e-04, 1.40042006e-04,
       8.43641799e-04, 1.41952706e-04, 1.39121682e-04, 1.39778432e-04,
       1.42574500e-04, 1.42726372e-04, 1.42259641e-04, 8.47604036e-04,
       1.41708962e-04, 1.39518556e-04, 1.38816005e-04, 1.42965711e-04,
       1.41634704e-04, 1.40419233e-04, 8.28527487e-04, 1.37440231e-04,
       1.33857801e-04, 1.32209807e-04, 1.38923603e-04, 1.42898524e-04,
       8.35999118e-04, 1.33635999e-04, 1.42169618e-04, 1.41465036e-04,
       1.36906338e-04, 1.31118991e-04, 1.28113375e-04, 1.33635999e-04,
       7.88051749e-04, 1.29298881e-04, 1.33608003e-04, 1.32276501e-04,
       1.25455437e-04, 1.20037881e-04, 1.29298881e-04, 7.52817066e-04,
       1.20599783e-04, 1.30353312e-04, 1.27071771e-04, 1.08791617e-04,
       9.83841563e-05, 1.20599783e-04, 5.40071940e-04, 9.97539633e-05,
       1.12542421e-04, 1.29577235e-04, 1.48925518e-04, 9.97539633e-05,
       7.44294400e-04, 1.31524370e-04, 1.11045595e-04, 1.23467719e-04,
       1.43400307e-04, 1.46577051e-04, 1.42169618e-04, 8.69590950e-04,
       1.47856280e-04, 1.43538888e-04, 1.46048807e-04, 1.52247140e-04,
       1.56409491e-04, 1.47856280e-04, 9.28555469e-04, 1.60883948e-04,
       1.51146647e-04, 1.60011964e-04, 8.96721887e-05, 9.17432548e-05,
       4.93680773e-04, 1.16037746e-04, 8.59669872e-05, 1.10260597e-04,
       1.24488986e-04, 1.48916611e-04, 1.16037746e-04, 8.17745690e-04,
       1.43842029e-04, 1.40541829e-04, 1.43918488e-04, 1.38791920e-04,
       1.23735418e-04, 1.51193762e-04, 1.35340849e-04, 8.32961301e-04,
       1.42347881e-04, 1.41551471e-04, 1.45841380e-04, 1.46621196e-04,
       1.42347881e-04, 8.69611527e-04, 1.44428260e-04, 1.44536123e-04,
       1.45836686e-04, 1.50132865e-04, 1.55522549e-04, 1.41551471e-04,
       1.44428260e-04, 8.96748982e-04, 1.53813366e-04, 1.51300471e-04,
       1.51966590e-04, 1.40719700e-04, 1.53813366e-04, 8.72220693e-04,
       1.38039179e-04, 1.46104577e-04, 1.41577280e-04, 1.20758683e-04,
       1.18703346e-04, 1.38039179e-04, 7.71687814e-04, 1.29468990e-04,
       1.35616234e-04, 1.29101382e-04, 1.50705471e-04, 1.51272900e-04,
       9.03812364e-04, 1.50374963e-04, 1.50935967e-04, 1.50258319e-04,
       1.50264745e-04, 1.50460197e-04, 1.50215842e-04, 1.50374963e-04,
       9.00444697e-04, 1.49900897e-04, 1.49861255e-04, 1.49631544e-04,
       1.50046144e-04, 1.49086269e-04, 1.50935967e-04, 8.99996080e-04,
       1.49378296e-04, 1.50573600e-04, 1.49975804e-04, 1.49543379e-04,
       1.48987632e-04, 1.49900897e-04, 8.94912919e-04, 1.48853346e-04,
       1.49059735e-04, 1.48567931e-04, 1.49058779e-04, 1.48572509e-04,
       1.48853346e-04, 8.89625588e-04, 1.47621401e-04, 1.48132614e-04,
       1.47386939e-04, 1.48359311e-04, 1.48094259e-04, 1.49378296e-04,
       8.95009496e-04, 1.49473273e-04, 1.49672164e-04, 1.50032193e-04,
       1.47155885e-04, 1.46220657e-04, 1.47621401e-04, 8.75820848e-04,
       1.44309290e-04, 1.46445249e-04, 1.44068366e-04, 1.44194969e-04,
       1.41505573e-04, 1.44309290e-04, 8.41103423e-04, 1.35109305e-04,
       1.41247437e-04, 1.34736848e-04, 1.39347175e-04, 1.37161991e-04,
       1.35109305e-04, 8.32931703e-04, 1.48460219e-04, 1.28658457e-04,
       1.44194555e-04, 1.39358096e-04, 1.38521287e-04, 1.37419015e-04,
       8.21008684e-04, 1.36728041e-04, 1.34418205e-04, 1.34564040e-04,
       1.52234257e-04, 1.46557550e-04, 1.60883948e-04, 9.24859123e-04,
       1.45510885e-04, 1.64684420e-04, 1.54988064e-04, 1.40030396e-04,
       1.41952706e-04, 1.48460219e-04, 8.84843026e-04, 1.44034992e-04,
       1.58356125e-04, 1.52008587e-04, 1.41890067e-04, 1.41310948e-04,
       1.41708962e-04, 8.41583459e-04, 1.38705757e-04, 1.40195906e-04,
       1.37771820e-04, 1.44387022e-04, 1.48437648e-04, 8.65179026e-04,
       1.40008826e-04, 1.47877528e-04, 1.44143038e-04, 1.40324963e-04,
       1.42762253e-04, 1.54263714e-04, 1.40008826e-04, 8.77714484e-04,
       1.62955353e-04, 1.33139636e-04, 1.44584702e-04, 1.48850192e-04,
       1.47174112e-04, 1.47877528e-04, 8.81985273e-04, 1.45589600e-04,
       1.46552845e-04, 1.45940997e-04, 1.45013585e-04, 1.43928524e-04,
       1.45589600e-04, 8.69758520e-04, 1.44138726e-04, 1.45726949e-04,
       1.45361136e-04, 1.58827572e-04, 1.64928509e-04, 1.62955353e-04,
       1.06580279e-03, 1.74138968e-04, 1.36157195e-04, 1.49586629e-04,
       1.19208562e-04, 1.42853387e-04, 1.39518556e-04, 1.44138726e-04,
       8.49753104e-04, 1.36470472e-04, 1.45552693e-04, 1.41219271e-04,
       1.40602719e-04, 1.37440231e-04, 1.38705757e-04, 8.22466701e-04,
       1.33075116e-04, 1.37555516e-04, 1.35087363e-04, 1.66041183e-04,
       1.48920733e-04, 1.74138968e-04, 9.17494792e-04, 1.40101220e-04,
       1.52604993e-04, 1.35687696e-04, 1.12886750e-04, 1.31524370e-04,
       8.59669872e-05, 6.88018920e-04, 1.28354122e-04, 1.02768588e-04,
       1.26518103e-04, 1.42544773e-04, 1.43842029e-04, 1.40101220e-04,
       8.42370112e-04, 1.42295915e-04, 1.36344368e-04, 1.37241807e-04,
       1.29059656e-04, 1.19217681e-04, 7.08374823e-04, 1.28289543e-04,
       1.05433998e-04, 1.06680188e-04, 1.19693758e-04, 1.33245242e-04,
       1.36323619e-04, 1.28289543e-04, 7.98823281e-04, 1.36035523e-04,
       1.30130876e-04, 1.34798479e-04, 1.02742242e-04, 8.61149287e-05,
       1.05433998e-04, 4.67137831e-04, 8.06936397e-05, 9.21530230e-05,
       9.95505568e-05, 1.26225720e-04, 8.06936397e-05, 6.56921835e-04,
       1.26468641e-04, 1.02441720e-04, 1.21541558e-04, 1.35084595e-04,
       1.41000081e-04, 1.26468641e-04, 7.96169459e-04, 1.36142421e-04,
       1.26857714e-04, 1.30616008e-04, 1.44594005e-04, 1.43664993e-04,
       1.36142421e-04, 8.22354641e-04, 1.38658245e-04, 1.27925070e-04,
       1.31369907e-04, 1.43993516e-04, 1.44536123e-04, 1.38658245e-04,
       8.39174799e-04, 1.43202727e-04, 1.32391156e-04, 1.36393031e-04,
       1.37168611e-04, 1.37158873e-04, 1.36035523e-04, 8.16851550e-04,
       1.35324081e-04, 1.35933083e-04, 1.35231379e-04, 1.37693176e-04,
       1.33857801e-04, 1.35324081e-04, 7.93562021e-04, 1.28343573e-04,
       1.30744262e-04, 1.27599128e-04, 1.35935380e-04, 1.36231845e-04,
       8.33133231e-04, 1.41298174e-04, 1.39600707e-04, 1.41030528e-04,
       1.39036597e-04, 1.31384486e-04, 1.32800669e-04, 1.41298174e-04,
       8.94699991e-04, 1.46752021e-04, 1.34764101e-04, 1.10870421e-04,
       9.68301194e-05, 1.38782333e-04, 1.41058389e-04, 1.39600707e-04,
       8.49475458e-04, 1.44219335e-04, 1.41736061e-04, 1.44078633e-04,
       1.34500446e-04, 1.29468990e-04, 1.46752021e-04, 8.36987801e-04,
       1.38689798e-04, 1.45303657e-04, 1.42272890e-04, 1.44609794e-04,
       1.46586574e-04, 1.44219335e-04, 8.77345775e-04, 1.47543017e-04,
       1.46520076e-04, 1.47866978e-04, 1.48191013e-04, 1.49473273e-04,
       1.47543017e-04, 8.94362875e-04, 1.50098748e-04, 1.48891677e-04,
       1.50165148e-04, 1.38416625e-04, 1.39121682e-04, 1.36728041e-04,
       8.19438089e-04, 1.38139389e-04, 1.33163030e-04, 1.33869323e-04,
       1.27991141e-04, 1.33759179e-04, 5.40658060e-04, 9.82967335e-05,
       1.04482726e-04, 7.61282798e-05, 1.20987640e-04, 1.28183719e-04,
       9.82967335e-05, 7.24552038e-04, 1.34793464e-04, 1.06494915e-04,
       1.35795567e-04, 1.43275093e-04, 1.53866225e-04, 1.04482726e-04,
       1.03722418e-03, 1.59020844e-04, 1.27845542e-04, 1.70690713e-04,
       1.78043034e-04, 1.53552467e-04, 1.55106205e-04, 1.59020844e-04,
       9.45550602e-04, 1.58103143e-04, 1.61119692e-04, 1.58648252e-04,
       1.35357016e-04, 1.34418205e-04, 1.34793464e-04, 8.02393302e-04,
       1.32593437e-04, 1.33246197e-04, 1.31984981e-04, 1.53316587e-04,
       1.47851558e-04, 1.58103143e-04, 9.13951407e-04, 1.45956917e-04,
       1.57702199e-04, 1.51021003e-04, 1.46059323e-04, 1.45510885e-04,
       1.45956917e-04, 8.69259715e-04, 1.44229165e-04, 1.44341370e-04,
       1.43162056e-04, 1.39778432e-04, 1.44034992e-04, 1.38139389e-04,
       8.44173466e-04, 1.45814161e-04, 1.36494140e-04, 1.39912351e-04,
       1.38816005e-04, 1.40195906e-04, 1.36470472e-04, 8.26002897e-04,
       1.38009341e-04, 1.36176102e-04, 1.36335070e-04, 1.32209807e-04,
       1.33075116e-04, 1.28343573e-04, 7.39583511e-04, 1.28599595e-04,
       1.08982353e-04, 1.08373067e-04, 1.41465036e-04, 1.43538888e-04,
       8.45508305e-04, 1.39019054e-04, 1.44058093e-04, 1.37231023e-04,
       1.40196210e-04, 1.36906338e-04, 1.33608003e-04, 1.39019054e-04,
       8.35688372e-04, 1.39023758e-04, 1.41914090e-04, 1.45217129e-04,
       1.32276501e-04, 1.30353312e-04, 1.39023758e-04, 8.19134254e-04,
       1.34593243e-04, 1.42697383e-04, 1.40190057e-04, 1.27071771e-04,
       1.12542421e-04, 1.11045595e-04, 1.34593243e-04, 7.30399856e-04,
       1.18064265e-04, 1.27082562e-04, 1.23467719e-04, 1.28354122e-04,
       1.18064265e-04, 7.32035957e-04, 1.23859481e-04, 1.18690779e-04,
       1.19599592e-04, 1.46048807e-04, 1.51146647e-04, 1.44058093e-04,
       8.89764025e-04, 1.55949459e-04, 1.42883747e-04, 1.49677272e-04,
       1.60011964e-04, 1.64684420e-04, 1.55949459e-04, 9.55992685e-04,
       1.62586267e-04, 1.55398112e-04, 1.57362464e-04, 1.10260597e-04,
       1.40541829e-04, 1.02768588e-04, 7.49379373e-04, 1.43621094e-04,
       1.20807995e-04, 1.31379269e-04, 1.43918488e-04, 1.42295915e-04,
       1.43621094e-04, 8.51507109e-04, 1.39610318e-04, 1.42224748e-04,
       1.39836545e-04, 1.45836686e-04, 1.51300471e-04, 1.46104577e-04,
       1.43202727e-04, 8.70119539e-04, 1.40556572e-04, 1.43118507e-04,
       1.41577280e-04, 1.35616234e-04, 1.40556572e-04, 8.11093700e-04,
       1.25091471e-04, 1.38878099e-04, 1.29374044e-04, 1.29101382e-04,
       1.38689798e-04, 1.25091471e-04, 7.81193967e-04, 1.46562738e-04,
       1.14932863e-04, 1.26815716e-04, 1.50258319e-04, 1.50573600e-04,
       9.02392688e-04, 1.50095367e-04, 1.50844371e-04, 1.50527377e-04,
       1.50093653e-04, 1.50264745e-04, 1.49861255e-04, 1.50095367e-04,
       8.98356854e-04, 1.49463297e-04, 1.49450385e-04, 1.49221805e-04,
       1.49631544e-04, 1.49059735e-04, 1.49463297e-04, 8.93230442e-04,
       1.48308291e-04, 1.48675386e-04, 1.48092189e-04, 1.49975804e-04,
       1.49672164e-04, 1.50844371e-04, 9.01833851e-04, 1.50438770e-04,
       1.50502352e-04, 1.50400390e-04, 1.48567931e-04, 1.48132614e-04,
       1.48308291e-04, 8.84600850e-04, 1.46414424e-04, 1.47318070e-04,
       1.45859520e-04, 1.47386939e-04, 1.46445249e-04, 1.46414424e-04,
       8.72644012e-04, 1.43884274e-04, 1.45050793e-04, 1.43462333e-04,
       1.50032193e-04, 1.50098748e-04, 1.50438770e-04, 9.02554610e-04,
       1.50851944e-04, 1.50393169e-04, 1.50739788e-04, 1.44068366e-04,
       1.41247437e-04, 1.43884274e-04, 8.40986590e-04, 1.35749755e-04,
       1.40675174e-04, 1.35361584e-04, 1.34736848e-04, 1.28658457e-04,
       1.35749755e-04, 7.57239568e-04, 1.12910413e-04, 1.30972524e-04,
       1.14211573e-04, 1.44194555e-04, 1.58356125e-04, 1.12910413e-04,
       7.98134776e-04, 1.58440395e-04, 9.64324389e-05, 1.27800851e-04,
       1.34564040e-04, 1.33163030e-04, 1.32593437e-04, 7.92708289e-04,
       1.31576912e-04, 1.30213192e-04, 1.30597677e-04, 1.54988064e-04,
       1.44229165e-04, 1.62586267e-04, 9.02139478e-04, 1.43221652e-04,
       1.52860023e-04, 1.44254306e-04, 1.52008587e-04, 1.45814161e-04,
       1.58440395e-04, 9.08597361e-04, 1.44308537e-04, 1.57981556e-04,
       1.50044125e-04, 1.37771820e-04, 1.37555516e-04, 1.38009341e-04,
       8.39448930e-04, 1.40934709e-04, 1.41017936e-04, 1.44159608e-04,
       1.44143038e-04, 1.46552845e-04, 8.67802048e-04, 1.41743925e-04,
       1.46513225e-04, 1.43244761e-04, 1.45604255e-04, 1.40324963e-04,
       1.33139636e-04, 1.41743925e-04, 8.16625815e-04, 1.28250224e-04,
       1.39737720e-04, 1.33429347e-04, 1.44584702e-04, 1.36157195e-04,
       1.28250224e-04, 7.66878994e-04, 1.11762184e-04, 1.27092611e-04,
       1.19032078e-04, 1.45940997e-04, 1.45726949e-04, 1.46513225e-04,
       8.77061328e-04, 1.46271055e-04, 1.46318613e-04, 1.46290490e-04,
       1.45361136e-04, 1.45552693e-04, 1.46271055e-04, 8.81366627e-04,
       1.46670738e-04, 1.48651439e-04, 1.48859565e-04, 1.49586629e-04,
       1.52604993e-04, 8.25815144e-04, 1.33653615e-04, 1.27980910e-04,
       1.32321669e-04, 1.29667328e-04, 1.41219271e-04, 1.36176102e-04,
       1.46670738e-04, 8.41413248e-04, 1.35006315e-04, 1.44481038e-04,
       1.37859784e-04, 1.35087363e-04, 1.28599595e-04, 1.40934709e-04,
       8.39575668e-04, 1.38871431e-04, 1.45829040e-04, 1.50253530e-04,
       1.35687696e-04, 1.36344368e-04, 1.33653615e-04, 8.06908362e-04,
       1.34759941e-04, 1.33006544e-04, 1.33456198e-04, 1.26518103e-04,
       1.23859481e-04, 1.20807995e-04, 7.22517521e-04, 1.16915482e-04,
       1.17825176e-04, 1.16591285e-04, 1.37241807e-04, 1.39610318e-04,
       1.34759941e-04, 8.24865668e-04, 1.40827258e-04, 1.34363769e-04,
       1.38062575e-04, 1.06680188e-04, 9.21530230e-05, 1.02441720e-04,
       6.61337463e-04, 1.17349531e-04, 1.21166177e-04, 1.21546823e-04,
       1.19693758e-04, 1.30130876e-04, 1.17349531e-04, 7.68476680e-04,
       1.35347953e-04, 1.29196629e-04, 1.36757933e-04, 1.34798479e-04,
       1.35933083e-04, 1.35347953e-04, 8.17365224e-04, 1.35858060e-04,
       1.38026073e-04, 1.37401576e-04, 1.21541558e-04, 1.26857714e-04,
       1.21166177e-04, 7.52614872e-04, 1.28812106e-04, 1.25953772e-04,
       1.28283545e-04, 1.30616008e-04, 1.27925070e-04, 1.28812106e-04,
       7.20040891e-04, 1.04338780e-04, 1.25065657e-04, 1.03283269e-04,
       1.31369907e-04, 1.32391156e-04, 1.04338780e-04, 7.19201638e-04,
       1.45147252e-04, 8.30836605e-05, 1.22870882e-04, 1.36393031e-04,
       1.43118507e-04, 1.38878099e-04, 1.45147252e-04, 8.48372146e-04,
       1.35920314e-04, 1.48914943e-04, 1.35231379e-04, 1.30744262e-04,
       1.35858060e-04, 8.00144072e-04, 1.26476400e-04, 1.38364257e-04,
       1.33469714e-04, 1.27599128e-04, 1.08982353e-04, 1.26476400e-04,
       7.34903776e-04, 9.48708130e-05, 1.44981947e-04, 1.31993135e-04,
       1.41030528e-04, 1.34764101e-04, 8.05696151e-04, 1.38269297e-04,
       1.25693635e-04, 1.36123913e-04, 1.29814678e-04, 1.39036597e-04,
       1.41736061e-04, 1.38269297e-04, 8.46643090e-04, 1.43694236e-04,
       1.40590712e-04, 1.43316187e-04, 1.10870421e-04, 1.45303657e-04,
       6.93966079e-04, 1.44343220e-04, 6.85596775e-05, 1.33119705e-04,
       9.17693988e-05, 1.44078633e-04, 1.46520076e-04, 1.43694236e-04,
       8.71390146e-04, 1.46805530e-04, 1.44810910e-04, 1.45480760e-04,
       1.42272890e-04, 1.46562738e-04, 1.44343220e-04, 8.67097302e-04,
       1.46387355e-04, 1.44888406e-04, 1.42642693e-04, 1.47866978e-04,
       1.48891677e-04, 1.46805530e-04, 8.87589007e-04, 1.49057694e-04,
       1.46869831e-04, 1.48097296e-04, 1.50165148e-04, 1.50851944e-04,
       1.49057694e-04, 9.00002822e-04, 1.50404518e-04, 1.49431745e-04,
       1.50091773e-04, 1.33869323e-04, 1.36494140e-04, 1.31576912e-04,
       7.95462037e-04, 1.37071578e-04, 1.26790118e-04, 1.29659967e-04,
       7.61282798e-05, 1.06494915e-04, 1.27845542e-04, 7.88468042e-04,
       1.48031221e-04, 1.70074520e-04, 1.59893563e-04, 1.35795567e-04,
       1.33246197e-04, 1.48031221e-04, 8.47227093e-04, 1.35222636e-04,
       1.52595344e-04, 1.42336128e-04, 1.70690713e-04, 1.61119692e-04,
       9.68351247e-04, 1.55290404e-04, 1.67351316e-04, 1.55704616e-04,
       1.58194506e-04, 1.58648252e-04, 1.57702199e-04, 1.55290404e-04,
       9.19302800e-04, 1.51245465e-04, 1.49757683e-04, 1.46658797e-04,
       1.31984981e-04, 1.30213192e-04, 1.35222636e-04, 8.40871491e-04,
       1.27826734e-04, 1.60624031e-04, 1.54999918e-04, 1.51021003e-04,
       1.44341370e-04, 1.51245465e-04, 8.73004438e-04, 1.40089482e-04,
       1.45391734e-04, 1.40915384e-04, 1.43162056e-04, 1.43221652e-04,
       1.40089482e-04, 8.40416142e-04, 1.41312238e-04, 1.35733778e-04,
       1.36896937e-04, 1.39912351e-04, 1.44308537e-04, 1.37071578e-04,
       8.41532004e-04, 1.42028670e-04, 1.38824981e-04, 1.39385888e-04,
       1.36335070e-04, 1.41017936e-04, 1.35006315e-04, 8.29124571e-04,
       1.43814427e-04, 1.34412788e-04, 1.38538034e-04, 1.19208562e-04,
       1.11762184e-04, 1.27980910e-04, 6.14085034e-04, 1.23630704e-04,
       1.31502673e-04, 1.08373067e-04, 1.38871431e-04, 9.48708130e-05,
       6.19040115e-04, 1.51291161e-04, 1.25633643e-04, 1.37231023e-04,
       1.41914090e-04, 8.69367241e-04, 1.35664048e-04, 1.50880277e-04,
       1.46572320e-04, 1.57105483e-04, 1.40196210e-04, 1.42883747e-04,
       1.35664048e-04, 8.27598915e-04, 1.41568053e-04, 1.32035193e-04,
       1.35251662e-04, 1.45217129e-04, 1.42697383e-04, 1.50880277e-04,
       8.84988625e-04, 1.44313900e-04, 1.52963284e-04, 1.48916651e-04,
       1.40190057e-04, 1.27082562e-04, 1.18690779e-04, 1.44313900e-04,
       7.84605363e-04, 1.20906220e-04, 1.33421846e-04, 1.19599592e-04,
       1.16915482e-04, 1.20906220e-04, 7.17602172e-04, 1.16367672e-04,
       1.22833822e-04, 1.20979384e-04, 1.49677272e-04, 1.55398112e-04,
       1.41568053e-04, 8.76375687e-04, 1.49226302e-04, 1.39284269e-04,
       1.41221678e-04, 1.57362464e-04, 1.52860023e-04, 1.49226302e-04,
       8.85613517e-04, 1.43170559e-04, 1.42273735e-04, 1.40720433e-04,
       1.31379269e-04, 1.42224748e-04, 1.17825176e-04, 7.73692433e-04,
       1.37157827e-04, 1.18309120e-04, 1.26796292e-04, 1.39836545e-04,
       1.40827258e-04, 1.37157827e-04, 8.30762644e-04, 1.42033110e-04,
       1.33511667e-04, 1.37396237e-04, 1.29374044e-04, 1.14932863e-04,
       1.35920314e-04, 7.10627736e-04, 9.44997150e-05, 1.31440109e-04,
       1.04460691e-04, 9.68301194e-05, 1.25693635e-04, 6.85596775e-05,
       5.07760534e-04, 1.22939858e-04, 9.37372446e-05, 1.26815716e-04,
       1.46387355e-04, 9.44997150e-05, 7.27276500e-04, 1.52884220e-04,
       8.39383138e-05, 1.22751180e-04, 1.50527377e-04, 1.50502352e-04,
       8.98738606e-04, 1.49694108e-04, 1.49894115e-04, 1.48947811e-04,
       1.49172843e-04, 1.50093653e-04, 1.49450385e-04, 1.49694108e-04,
       8.95612266e-04, 1.48846008e-04, 1.48866472e-04, 1.48661640e-04,
       1.49221805e-04, 1.48675386e-04, 1.48846008e-04, 8.90230101e-04,
       1.47884205e-04, 1.48009041e-04, 1.47593656e-04, 1.48092189e-04,
       1.47318070e-04, 1.47884205e-04, 8.83630925e-04, 1.46434577e-04,
       1.47288697e-04, 1.46613188e-04, 1.50400390e-04, 1.50393169e-04,
       1.49894115e-04, 8.99790988e-04, 1.50249391e-04, 1.49245712e-04,
       1.49608210e-04, 1.45859520e-04, 1.45050793e-04, 1.46434577e-04,
       8.71845692e-04, 1.44533851e-04, 1.45529475e-04, 1.44437477e-04,
       1.43462333e-04, 1.40675174e-04, 1.44533851e-04, 8.51651031e-04,
       1.39759632e-04, 1.42603550e-04, 1.40616490e-04, 1.50739788e-04,
       1.50404518e-04, 1.50249391e-04, 9.00607960e-04, 1.49819029e-04,
       1.49745164e-04, 1.49650071e-04, 1.35361584e-04, 1.30972524e-04,
       1.39759632e-04, 7.76925870e-04, 1.13819218e-04, 1.39282134e-04,
       1.17730779e-04, 1.14211573e-04, 9.64324389e-05, 1.13819218e-04,
       5.15597748e-04, 9.50918045e-05, 9.60427141e-05, 1.27800851e-04,
       1.57981556e-04, 9.50918045e-05, 9.61428130e-04, 1.52260490e-04,
       1.24900987e-04, 1.50286935e-04, 1.53105507e-04, 1.30597677e-04,
       1.26790118e-04, 1.27826734e-04, 6.26940661e-04, 1.19364515e-04,
       1.22361618e-04, 1.78043034e-04, 1.70074520e-04, 1.67351316e-04,
       9.95239091e-04, 1.60779706e-04, 1.58796806e-04, 1.60193709e-04,
       1.44254306e-04, 1.41312238e-04, 1.43170559e-04, 8.41425013e-04,
       1.36112692e-04, 1.40345508e-04, 1.36229709e-04, 1.50044125e-04,
       1.42028670e-04, 1.52260490e-04, 8.74564164e-04, 1.40752714e-04,
       1.46485243e-04, 1.42992921e-04, 1.44159608e-04, 1.45829040e-04,
       1.43814427e-04, 9.01109555e-04, 1.51595094e-04, 1.54800076e-04,
       1.60911311e-04, 1.43244761e-04, 1.39737720e-04, 8.54397908e-04,
       1.46211788e-04, 1.37849441e-04, 1.41249446e-04, 1.46104751e-04,
       1.45604255e-04, 1.46318613e-04, 1.46211788e-04, 8.82812816e-04,
       1.49548853e-04, 1.46253300e-04, 1.48876008e-04, 1.33429347e-04,
       1.27092611e-04, 1.37849441e-04, 7.92564943e-04, 1.26300150e-04,
       1.36552975e-04, 1.31340419e-04, 1.19032078e-04, 1.23630704e-04,
       1.26300150e-04, 7.74627938e-04, 1.34409552e-04, 1.33872338e-04,
       1.37383115e-04, 1.46290490e-04, 1.48651439e-04, 1.49548853e-04,
       9.09440859e-04, 1.53510296e-04, 1.54919643e-04, 1.56520136e-04,
       1.48859565e-04, 1.44481038e-04, 1.53510296e-04, 9.02122171e-04,
       1.46520692e-04, 1.55680828e-04, 1.53069752e-04, 1.32321669e-04,
       1.33006544e-04, 7.99443220e-04, 1.32506586e-04, 1.33973231e-04,
       1.34208480e-04, 1.33426711e-04, 1.29667328e-04, 1.31502673e-04,
       1.34409552e-04, 1.32506586e-04, 7.94962241e-04, 1.33471862e-04,
       1.33404240e-04, 1.37859784e-04, 1.34412788e-04, 1.46520692e-04,
       8.52957157e-04, 1.36191396e-04, 1.52427399e-04, 1.45545098e-04,
       1.50253530e-04, 1.51291161e-04, 1.51595094e-04, 9.14560798e-04,
       1.54709326e-04, 1.52317543e-04, 1.54394144e-04, 1.33456198e-04,
       1.34363769e-04, 1.33973231e-04, 8.11176912e-04, 1.35699652e-04,
       1.36432606e-04, 1.37251456e-04, 1.16591285e-04, 1.16367672e-04,
       1.18309120e-04, 7.06009429e-04, 1.17591245e-04, 1.18822184e-04,
       1.18327923e-04, 1.38062575e-04, 1.42033110e-04, 1.35699652e-04,
       8.43226988e-04, 1.44104729e-04, 1.39446190e-04, 1.43880732e-04,
       1.21546823e-04, 1.29196629e-04, 1.25953772e-04, 7.74841860e-04,
       1.34489743e-04, 1.31384363e-04, 1.32270529e-04, 1.36757933e-04,
       1.38026073e-04, 1.34489743e-04, 7.97839273e-04, 1.33050248e-04,
       1.29111455e-04, 1.26403821e-04, 1.37401576e-04, 1.38364257e-04,
       1.33050248e-04, 7.94162372e-04, 1.33758410e-04, 1.25921200e-04,
       1.25666680e-04, 1.28283545e-04, 1.25065657e-04, 1.31384363e-04,
       7.82552351e-04, 1.31504607e-04, 1.31488023e-04, 1.34826154e-04,
       1.03283269e-04, 8.30836605e-05, 1.31504607e-04, 5.63133813e-04,
       1.08770468e-04, 1.36491808e-04, 1.22870882e-04, 1.48914943e-04,
       1.31440109e-04, 1.08770468e-04, 8.99576526e-04, 1.11871644e-04,
       1.47209847e-04, 1.28498634e-04, 1.33469714e-04, 1.44981947e-04,
       1.33758410e-04, 8.32673185e-04, 1.56054606e-04, 1.26812273e-04,
       1.37596236e-04, 1.31993135e-04, 1.25633643e-04, 1.54709326e-04,
       1.56054606e-04, 1.01052616e-03, 1.52327711e-04, 1.45164419e-04,
       1.44643316e-04, 1.36123913e-04, 1.40590712e-04, 8.34911075e-04,
       1.34876275e-04, 1.42793347e-04, 1.38538278e-04, 1.41988550e-04,
       1.29814678e-04, 1.22939858e-04, 1.34876275e-04, 7.69585204e-04,
       1.19741097e-04, 1.35236827e-04, 1.26976469e-04, 1.43316187e-04,
       1.44810910e-04, 1.42793347e-04, 8.62571915e-04, 1.44504727e-04,
       1.43465044e-04, 1.43681701e-04, 1.33119705e-04, 1.44888406e-04,
       7.99100151e-04, 1.17717217e-04, 1.40449862e-04, 1.36944452e-04,
       1.25980509e-04, 9.17693988e-05, 9.37372446e-05, 1.19741097e-04,
       1.17717217e-04, 6.68219138e-04, 1.22655107e-04, 1.22599074e-04,
       1.45480760e-04, 1.46869831e-04, 1.44504727e-04, 8.75332868e-04,
       1.47227556e-04, 1.44958118e-04, 1.46291876e-04, 1.42642693e-04,
       1.52884220e-04, 1.40449862e-04, 8.84375478e-04, 1.56206812e-04,
       1.43338235e-04, 1.48853657e-04, 1.48097296e-04, 1.49431745e-04,
       1.47227556e-04, 8.90265092e-04, 1.49562800e-04, 1.47472449e-04,
       1.48473245e-04, 1.50091773e-04, 1.49819029e-04, 1.49562800e-04,
       8.98051848e-04, 1.49623034e-04, 1.49311117e-04, 1.49644095e-04,
       1.29659967e-04, 1.38824981e-04, 1.19364515e-04, 8.28034980e-04,
       1.44773938e-04, 1.39583585e-04, 1.55827994e-04, 1.59893563e-04,
       1.52595344e-04, 1.60779706e-04, 9.33064610e-04, 1.48800314e-04,
       1.57838428e-04, 1.53157255e-04, 1.42336128e-04, 1.60624031e-04,
       1.48800314e-04, 9.69362614e-04, 1.84984917e-04, 1.57360261e-04,
       1.75256962e-04, 1.55704616e-04, 1.49757683e-04, 8.87236700e-04,
       1.53298397e-04, 1.43731616e-04, 1.40562270e-04, 1.44182118e-04,
       1.58194506e-04, 1.58796806e-04, 1.53298397e-04, 9.24425967e-04,
       1.55827077e-04, 1.48191400e-04, 1.50117781e-04, 1.46658797e-04,
       1.45391734e-04, 1.43731616e-04, 8.50938878e-04, 1.40297465e-04,
       1.38513177e-04, 1.36346088e-04, 1.54999918e-04, 1.22361618e-04,
       1.44773938e-04, 1.84984917e-04, 1.12167058e-03, 1.71211469e-04,
       1.75770422e-04, 1.67568293e-04, 1.40915384e-04, 1.35733778e-04,
       1.40297465e-04, 8.20670341e-04, 1.33496871e-04, 1.35922915e-04,
       1.34303928e-04, 1.36896937e-04, 1.36112692e-04, 1.33496871e-04,
       7.85089211e-04, 1.31803198e-04, 1.23844471e-04, 1.22935043e-04,
       1.39385888e-04, 1.40752714e-04, 1.39583585e-04, 8.32539595e-04,
       1.39440768e-04, 1.37443142e-04, 1.35933499e-04, 1.38538034e-04,
       1.54800076e-04, 1.36191396e-04, 8.97206688e-04, 1.73974720e-04,
       1.36090590e-04, 1.57611873e-04, 1.46572320e-04, 1.32035193e-04,
       8.90544956e-04, 1.63795754e-04, 1.28531977e-04, 1.49441531e-04,
       1.70168181e-04, 1.57105483e-04, 1.52963284e-04, 1.63795754e-04,
       9.56270592e-04, 1.55186747e-04, 1.65843065e-04, 1.61376258e-04,
       1.35251662e-04, 1.39284269e-04, 1.28531977e-04, 7.97259330e-04,
       1.37352102e-04, 1.26025901e-04, 1.30813419e-04, 1.48916651e-04,
       1.33421846e-04, 1.22833822e-04, 1.55186747e-04, 8.32397271e-04,
       1.27590043e-04, 1.44448162e-04, 1.20979384e-04, 1.17591245e-04,
       1.27590043e-04, 7.46352375e-04, 1.18763019e-04, 1.33433784e-04,
       1.27994900e-04, 1.41221678e-04, 1.42273735e-04, 1.37352102e-04,
       8.33473963e-04, 1.41811458e-04, 1.33703846e-04, 1.37111145e-04,
       1.40720433e-04, 1.40345508e-04, 1.41811458e-04, 8.55355542e-04,
       1.42158028e-04, 1.44799309e-04, 1.45520805e-04, 1.26796292e-04,
       1.33511667e-04, 1.18822184e-04, 7.51100142e-04, 1.29800204e-04,
       1.18953575e-04, 1.23216220e-04, 1.37396237e-04, 1.44104729e-04,
       1.29800204e-04, 8.19788198e-04, 1.45818769e-04, 1.26679093e-04,
       1.35989166e-04, 1.04460691e-04, 8.39383138e-05, 1.11871644e-04,
       4.77784136e-04, 8.61639519e-05, 9.13495354e-05, 1.22751180e-04,
       1.56206812e-04, 8.61639519e-05, 9.21487230e-04, 1.52941938e-04,
       1.10064453e-04, 1.51390509e-04, 1.41968387e-04, 1.48947811e-04,
       1.48866472e-04, 8.88618953e-04, 1.48280318e-04, 1.48208184e-04,
       1.47153482e-04, 1.47162687e-04, 1.49172843e-04, 1.49245712e-04,
       1.48280318e-04, 8.89657118e-04, 1.48786168e-04, 1.46869548e-04,
       1.47302529e-04, 1.48661640e-04, 1.48009041e-04, 1.48208184e-04,
       8.88057599e-04, 1.47574983e-04, 1.47792177e-04, 1.47811575e-04,
       1.47593656e-04, 1.47288697e-04, 1.47574983e-04, 8.84293478e-04,
       1.47089647e-04, 1.47463435e-04, 1.47283060e-04, 1.46613188e-04,
       1.45529475e-04, 1.47089647e-04, 8.76068038e-04, 1.44720848e-04,
       1.46699983e-04, 1.45414897e-04, 1.49608210e-04, 1.49745164e-04,
       1.48786168e-04, 8.92250019e-04, 1.49219496e-04, 1.47297303e-04,
       1.47593678e-04, 1.44437477e-04, 1.42603550e-04, 1.44720848e-04,
       8.59317701e-04, 1.42107034e-04, 1.43114340e-04, 1.42334452e-04,
       1.40616490e-04, 1.39282134e-04, 1.42107034e-04, 8.48250953e-04,
       1.42056631e-04, 1.41452356e-04, 1.42736309e-04, 1.49650071e-04,
       1.49623034e-04, 1.49219496e-04, 8.93582395e-04, 1.49157479e-04,
       1.47983647e-04, 1.47948668e-04, 1.17730779e-04, 9.60427141e-05,
       1.24900987e-04, 1.42056631e-04, 7.76688421e-04, 1.50244866e-04,
       1.45712445e-04, 1.50286935e-04, 1.46485243e-04, 8.89514683e-04,
       1.45550906e-04, 1.50952430e-04, 1.48919500e-04, 1.47319668e-04,
       1.60193709e-04, 1.57838428e-04, 1.55827077e-04, 9.37269406e-04,
       1.55734045e-04, 1.52706949e-04, 1.54969199e-04, 1.36229709e-04,
       1.31803198e-04, 1.42158028e-04, 8.23139049e-04, 1.32154598e-04,
       1.43185217e-04, 1.37608299e-04, 1.42992921e-04, 1.39440768e-04,
       1.45550906e-04, 8.52557828e-04, 1.38312944e-04, 1.44973049e-04,
       1.41287239e-04, 1.60911311e-04, 1.52317543e-04, 1.73974720e-04,
       1.07949301e-03, 1.61171298e-04, 1.52509985e-04, 1.50862580e-04,
       1.27745575e-04, 1.41249446e-04, 1.36552975e-04, 8.37133880e-04,
       1.42639478e-04, 1.35359898e-04, 1.39374487e-04, 1.41957596e-04,
       1.46104751e-04, 1.46253300e-04, 1.42639478e-04, 8.64206254e-04,
       1.44935920e-04, 1.41062455e-04, 1.43210349e-04, 1.48876008e-04,
       1.54919643e-04, 1.44935920e-04, 9.11186021e-04, 1.58742174e-04,
       1.47974829e-04, 1.55737447e-04, 1.31340419e-04, 1.33872338e-04,
       1.35359898e-04, 8.19175274e-04, 1.39672525e-04, 1.38574693e-04,
       1.40355400e-04, 1.37383115e-04, 1.33471862e-04, 1.39672525e-04,
       8.17980734e-04, 1.35130553e-04, 1.36476699e-04, 1.35845980e-04,
       1.56520136e-04, 1.55680828e-04, 1.58742174e-04, 9.60709471e-04,
       1.61176836e-04, 1.62657763e-04, 1.65931734e-04, 1.53069752e-04,
       1.52427399e-04, 1.61176836e-04, 9.46879309e-04, 1.57952258e-04,
       1.62417645e-04, 1.59835420e-04, 1.34208480e-04, 1.36432606e-04,
       8.19875503e-04, 1.34617958e-04, 1.38636645e-04, 1.38887188e-04,
       1.37092627e-04, 1.33426711e-04, 1.33404240e-04, 1.35130553e-04,
       1.34617958e-04, 8.09019505e-04, 1.36487555e-04, 1.35952489e-04,
       1.45545098e-04, 1.36090590e-04, 1.57952258e-04, 8.96079970e-04,
       1.39723125e-04, 1.62771761e-04, 1.53997137e-04, 1.54394144e-04,
       1.52327711e-04, 1.61171298e-04, 9.10462993e-04, 1.45608283e-04,
       1.55229070e-04, 1.41732487e-04, 1.37251456e-04, 1.39446190e-04,
       1.38636645e-04, 8.48938889e-04, 1.44600582e-04, 1.42617406e-04,
       1.46386610e-04, 1.18327923e-04, 1.18763019e-04, 1.18953575e-04,
       7.16359745e-04, 1.19648261e-04, 1.20203689e-04, 1.20463278e-04,
       1.43880732e-04, 1.45818769e-04, 1.44600582e-04, 8.95686435e-04,
       1.52586082e-04, 1.51376404e-04, 1.57423866e-04, 1.32270529e-04,
       1.29111455e-04, 1.31488023e-04, 7.85871126e-04, 1.28578307e-04,
       1.32086727e-04, 1.32336085e-04, 1.26403821e-04, 1.25921200e-04,
       1.28578307e-04, 7.43822710e-04, 1.24953419e-04, 1.20554115e-04,
       1.17411849e-04, 1.25666680e-04, 1.26812273e-04, 1.24953419e-04,
       7.71279647e-04, 1.31302009e-04, 1.28671134e-04, 1.33874131e-04,
       1.34826154e-04, 1.36491808e-04, 1.47209847e-04, 1.32086727e-04,
       8.49387828e-04, 1.56115379e-04, 1.42657913e-04, 1.37596236e-04,
       1.45164419e-04, 1.31302009e-04, 8.41454390e-04, 1.44503563e-04,
       1.38627396e-04, 1.44260767e-04, 1.38538278e-04, 1.35236827e-04,
       8.31790190e-04, 1.41339103e-04, 1.35753202e-04, 1.39319164e-04,
       1.41603615e-04, 1.41988550e-04, 1.43465044e-04, 1.41339103e-04,
       8.55499421e-04, 1.42927637e-04, 1.42833524e-04, 1.42945564e-04,
       1.26976469e-04, 1.22655107e-04, 1.35753202e-04, 7.79882740e-04,
       1.26588669e-04, 1.36376232e-04, 1.31533061e-04, 1.43681701e-04,
       1.44958118e-04, 1.42927637e-04, 8.66750827e-04, 1.45617161e-04,
       1.44076551e-04, 1.45489659e-04, 1.36944452e-04, 1.43338235e-04,
       8.37187391e-04, 1.33066424e-04, 1.44182453e-04, 1.42189037e-04,
       1.37466791e-04, 1.25980509e-04, 1.22599074e-04, 1.26588669e-04,
       1.33066424e-04, 7.72461163e-04, 1.30565084e-04, 1.33661404e-04,
       1.46291876e-04, 1.47472449e-04, 1.45617161e-04, 8.80605902e-04,
       1.48342237e-04, 1.45668838e-04, 1.47213340e-04, 1.48853657e-04,
       1.52941938e-04, 1.44182453e-04, 8.91971939e-04, 1.51557827e-04,
       1.45574486e-04, 1.48861578e-04, 1.48473245e-04, 1.49311117e-04,
       1.48342237e-04, 8.91506250e-04, 1.49595511e-04, 1.47684369e-04,
       1.48099771e-04, 1.49644095e-04, 1.49157479e-04, 1.49595511e-04,
       8.95789385e-04, 1.48611725e-04, 1.49638872e-04, 1.49141702e-04,
       1.53105507e-04, 1.50244866e-04, 1.50952430e-04, 9.02082994e-04,
       1.48341793e-04, 1.49594197e-04, 1.49844201e-04, 1.55827994e-04,
       1.71211469e-04, 1.37443142e-04, 8.91535893e-04, 1.34201150e-04,
       1.55738802e-04, 1.37113335e-04, 1.53157255e-04, 1.57360261e-04,
       1.55734045e-04, 9.47422207e-04, 1.63945308e-04, 1.56608540e-04,
       1.60616797e-04, 1.75256962e-04, 1.75770422e-04, 1.63945308e-04,
       1.00054644e-03, 1.64035054e-04, 1.60981205e-04, 1.60557491e-04,
       1.40562270e-04, 1.38513177e-04, 8.14296835e-04, 1.37462099e-04,
       1.35133025e-04, 1.31453123e-04, 1.31173142e-04, 1.44182118e-04,
       1.48191400e-04, 1.37462099e-04, 8.45097995e-04, 1.46289696e-04,
       1.32077183e-04, 1.36895498e-04, 1.50117781e-04, 1.52706949e-04,
       1.46289696e-04, 8.99613716e-04, 1.53674238e-04, 1.46014839e-04,
       1.50810213e-04, 1.36346088e-04, 1.35922915e-04, 1.35133025e-04,
       8.23596580e-04, 1.35461618e-04, 1.39990584e-04, 1.40742350e-04,
       1.34303928e-04, 1.23844471e-04, 1.35461618e-04, 6.35527095e-04,
       1.15723257e-04, 1.26193823e-04, 1.22935043e-04, 1.32154598e-04,
       1.15723257e-04, 8.31027596e-04, 1.33375680e-04, 1.59203075e-04,
       1.67635944e-04, 1.35933499e-04, 1.38312944e-04, 1.34201150e-04,
       8.11217115e-04, 1.37810410e-04, 1.31864648e-04, 1.33094464e-04,
       1.57611873e-04, 1.52509985e-04, 1.39723125e-04, 8.56950102e-04,
       1.22384855e-04, 1.48478323e-04, 1.36241941e-04, 1.44643316e-04,
       1.45608283e-04, 1.44503563e-04, 8.54432694e-04, 1.41332023e-04,
       1.41241008e-04, 1.37104501e-04, 1.49441531e-04, 1.26025901e-04,
       9.08756168e-04, 1.75634041e-04, 1.28891068e-04, 1.76045486e-04,
       1.52718143e-04, 1.70168181e-04, 1.65843065e-04, 1.75634041e-04,
       1.02181462e-03, 1.66594806e-04, 1.74144321e-04, 1.69430202e-04,
       1.61376258e-04, 1.44448162e-04, 1.33433784e-04, 1.66594806e-04,
       9.01282859e-04, 1.39598462e-04, 1.55831388e-04, 1.30813419e-04,
       1.33703846e-04, 1.28891068e-04, 7.84049362e-04, 1.31580382e-04,
       1.29630880e-04, 1.29429768e-04, 1.27994900e-04, 1.19648261e-04,
       1.39598462e-04, 7.85804365e-04, 1.22227588e-04, 1.42679695e-04,
       1.33655460e-04, 1.37111145e-04, 1.44799309e-04, 1.31580382e-04,
       8.67811049e-04, 1.52686434e-04, 1.44107946e-04, 1.57525833e-04,
       1.45520805e-04, 1.43185217e-04, 1.52686434e-04, 9.04984880e-04,
       1.47116716e-04, 1.59854919e-04, 1.56620789e-04, 1.23216220e-04,
       1.26679093e-04, 1.20203689e-04, 7.34887319e-04, 1.24386607e-04,
       1.19840832e-04, 1.20560877e-04, 1.35989166e-04, 1.52586082e-04,
       1.24386607e-04, 8.36488855e-04, 1.57048989e-04, 1.25206273e-04,
       1.41271738e-04, 1.28498634e-04, 9.13495354e-05, 1.10064453e-04,
       1.56115379e-04, 7.50458987e-04, 1.36666407e-04, 1.27764579e-04,
       1.51390509e-04, 1.51557827e-04, 9.03568175e-04, 1.46118204e-04,
       1.51017429e-04, 1.49376150e-04, 1.54108057e-04, 1.47153482e-04,
       1.46869548e-04, 8.71903848e-04, 1.45907752e-04, 1.45383857e-04,
       1.43174624e-04, 1.43414585e-04, 1.47162687e-04, 1.47792177e-04,
       1.45907752e-04, 8.77674481e-04, 1.46932408e-04, 1.44742146e-04,
       1.45137311e-04, 1.47302529e-04, 1.47297303e-04, 1.45383857e-04,
       8.70995070e-04, 1.45451651e-04, 1.42743355e-04, 1.42816375e-04,
       1.47811575e-04, 1.47463435e-04, 1.46932408e-04, 8.80106373e-04,
       1.47010494e-04, 1.45231117e-04, 1.45657344e-04, 1.47283060e-04,
       1.46699983e-04, 1.47010494e-04, 8.80293388e-04, 1.46618085e-04,
       1.46245549e-04, 1.46436217e-04, 1.45414897e-04, 1.43114340e-04,
       1.46618085e-04, 8.70911710e-04, 1.43730075e-04, 1.46310884e-04,
       1.45723430e-04, 1.47593678e-04, 1.47983647e-04, 1.45451651e-04,
       8.76572872e-04, 1.46774797e-04, 1.43917961e-04, 1.44851138e-04,
       1.42334452e-04, 1.41452356e-04, 1.43730075e-04, 8.60879769e-04,
       1.43146262e-04, 1.44959171e-04, 1.45257453e-04, 1.42736309e-04,
       1.45712445e-04, 1.48341793e-04, 1.43146262e-04, 8.73052490e-04,
       1.47667539e-04, 1.45448143e-04, 1.47948668e-04, 1.48611725e-04,
       1.46774797e-04, 8.84515490e-04, 1.47906735e-04, 1.46402342e-04,
       1.46871223e-04, 1.48919500e-04, 1.49594197e-04, 8.83611161e-04,
       1.47733702e-04, 1.47124468e-04, 1.45761613e-04, 1.44477682e-04,
       1.47319668e-04, 1.44973049e-04, 1.47733702e-04, 8.78211696e-04,
       1.43859948e-04, 1.47926232e-04, 1.46399097e-04, 1.67568293e-04,
       1.55738802e-04, 1.64035054e-04, 9.21586840e-04, 1.40393122e-04,
       1.52832005e-04, 1.41019564e-04, 1.54969199e-04, 1.56608540e-04,
       1.53674238e-04, 9.31446081e-04, 1.57022646e-04, 1.53731196e-04,
       1.55440262e-04, 1.37608299e-04, 1.33375680e-04, 1.47116716e-04,
       8.64423920e-04, 1.42731266e-04, 1.51872396e-04, 1.51719564e-04,
       1.41287239e-04, 1.37810410e-04, 1.43859948e-04, 8.36847193e-04,
       1.34849517e-04, 1.42286841e-04, 1.36753238e-04, 1.50862580e-04,
       1.55229070e-04, 8.74070551e-04, 1.36664565e-04, 1.42401531e-04,
       1.45141641e-04, 1.43771165e-04, 1.27745575e-04, 1.22384855e-04,
       1.36664565e-04, 6.66362235e-04, 1.33924889e-04, 1.45642350e-04,
       1.39374487e-04, 1.38574693e-04, 8.51851190e-04, 1.47430403e-04,
       1.40269830e-04, 1.46281278e-04, 1.39920499e-04, 1.41957596e-04,
       1.41062455e-04, 1.47430403e-04, 8.84062836e-04, 1.48251634e-04,
       1.51822188e-04, 1.53538560e-04, 1.43210349e-04, 1.47974829e-04,
       1.48251634e-04, 9.04170200e-04, 1.53185099e-04, 1.55689653e-04,
       1.55858637e-04, 1.55737447e-04, 1.62657763e-04, 1.53185099e-04,
       9.61387714e-04, 1.61506598e-04, 1.63449813e-04, 1.64850996e-04,
       1.40355400e-04, 1.36476699e-04, 1.40269830e-04, 8.30315731e-04,
       1.36539073e-04, 1.38263392e-04, 1.38411337e-04, 1.35845980e-04,
       1.36487555e-04, 1.36539073e-04, 8.20082553e-04, 1.36851201e-04,
       1.37344095e-04, 1.37014649e-04, 1.65931734e-04, 1.62417645e-04,
       1.61506598e-04, 9.48698447e-04, 1.53233842e-04, 1.55183647e-04,
       1.50424980e-04, 1.59835420e-04, 1.62771761e-04, 1.53233842e-04,
       9.27032345e-04, 1.64281094e-04, 1.39399659e-04, 1.47510569e-04,
       1.38887188e-04, 1.42617406e-04, 8.52180998e-04, 1.38859214e-04,
       1.45732994e-04, 1.44613878e-04, 1.41470317e-04, 1.37092627e-04,
       1.35952489e-04, 1.36851201e-04, 1.38859214e-04, 8.26343229e-04,
       1.38359912e-04, 1.39227787e-04, 1.53997137e-04, 1.48478323e-04,
       1.64281094e-04, 1.01290009e-03, 1.56313855e-04, 1.46474898e-04,
       1.29529261e-04, 1.13825522e-04, 1.41732487e-04, 1.41332023e-04,
       1.42401531e-04, 8.49489280e-04, 1.39943426e-04, 1.43068727e-04,
       1.41011086e-04, 1.46386610e-04, 1.51376404e-04, 1.45732994e-04,
       9.06367258e-04, 1.57116188e-04, 1.49680831e-04, 1.56074232e-04,
       1.20463278e-04, 1.22227588e-04, 1.19840832e-04, 7.26918053e-04,
       1.23248813e-04, 1.19746936e-04, 1.21390606e-04, 1.57423866e-04,
       1.57048989e-04, 1.57116188e-04, 1.11611102e-03, 1.34444907e-04,
       1.69422849e-04, 1.54004790e-04, 1.86649432e-04, 1.32336085e-04,
       1.20554115e-04, 1.42657913e-04, 1.36666407e-04, 7.61397480e-04,
       1.11696248e-04, 1.17486712e-04, 1.17411849e-04, 1.28671134e-04,
       1.11696248e-04, 7.71003359e-04, 1.38908249e-04, 1.29181582e-04,
       1.45134298e-04, 1.33874131e-04, 1.38627396e-04, 1.38908249e-04,
       8.42862625e-04, 1.43729163e-04, 1.43828018e-04, 1.43895668e-04,
       1.44260767e-04, 1.41241008e-04, 1.43729163e-04, 8.21710692e-04,
       1.31261170e-04, 1.35333409e-04, 1.25885175e-04, 1.41968387e-04,
       1.27764579e-04, 1.46118204e-04, 1.17486712e-04, 1.29181582e-04,
       8.06128730e-04, 1.43609266e-04, 1.39319164e-04, 1.36376232e-04,
       8.41462540e-04, 1.42406578e-04, 1.38175794e-04, 1.41948460e-04,
       1.43236312e-04, 1.41603615e-04, 1.42833524e-04, 1.42406578e-04,
       8.56863042e-04, 1.43471055e-04, 1.43356851e-04, 1.43191419e-04,
       1.42945564e-04, 1.44076551e-04, 1.43471055e-04, 8.64625345e-04,
       1.44732855e-04, 1.44634253e-04, 1.44765067e-04, 1.31533061e-04,
       1.30565084e-04, 1.38175794e-04, 8.15686811e-04, 1.34514736e-04,
       1.41795609e-04, 1.39102528e-04, 1.45489659e-04, 1.45668838e-04,
       1.44732855e-04, 8.69380785e-04, 1.45742555e-04, 1.43458178e-04,
       1.44288699e-04, 1.42189037e-04, 1.45574486e-04, 8.61529506e-04,
       1.40658682e-04, 1.45869498e-04, 1.42706218e-04, 1.44531586e-04,
       1.37466791e-04, 1.33661404e-04, 1.34514736e-04, 1.40658682e-04,
       8.24214395e-04, 1.37787112e-04, 1.40125671e-04, 1.47213340e-04,
       1.47684369e-04, 1.45742555e-04, 8.80315323e-04, 1.47284991e-04,
       1.45659330e-04, 1.46730737e-04, 1.48861578e-04, 1.51017429e-04,
       1.45869498e-04, 8.91711476e-04, 1.51693034e-04, 1.45301125e-04,
       1.48968811e-04, 1.48099771e-04, 1.49638872e-04, 1.47284991e-04,
       8.91591685e-04, 1.49323307e-04, 1.48372764e-04, 1.48871980e-04,
       1.49141702e-04, 1.47906735e-04, 1.49323307e-04, 8.94423407e-04,
       1.48750256e-04, 1.49319745e-04, 1.49981661e-04, 1.49844201e-04,
       1.47667539e-04, 1.47124468e-04, 8.84434166e-04, 1.46779075e-04,
       1.45593807e-04, 1.47425077e-04, 1.37113335e-04, 1.31864648e-04,
       1.40393122e-04, 7.93948568e-04, 1.29276312e-04, 1.30584637e-04,
       1.24716514e-04, 1.60616797e-04, 1.60981205e-04, 1.57022646e-04,
       9.40767933e-04, 1.56603671e-04, 1.53163499e-04, 1.52380115e-04,
       1.60557491e-04, 1.52832005e-04, 1.56603671e-04, 9.07263146e-04,
       1.41381870e-04, 1.51692212e-04, 1.44195898e-04, 1.31453123e-04,
       1.32077183e-04, 7.07013803e-04, 1.26720482e-04, 1.26802528e-04,
       9.47092368e-05, 9.52512508e-05, 1.31173142e-04, 1.39990584e-04,
       1.26720482e-04, 8.24450698e-04, 1.44646690e-04, 1.36405517e-04,
       1.45514284e-04, 1.36895498e-04, 1.46014839e-04, 1.26802528e-04,
       8.40126912e-04, 1.46792265e-04, 1.36375692e-04, 1.47246089e-04,
       1.50810213e-04, 1.53731196e-04, 1.46792265e-04, 9.12878079e-04,
       1.55077067e-04, 1.50551760e-04, 1.55915579e-04, 1.40742350e-04,
       1.26193823e-04, 1.59203075e-04, 1.44646690e-04, 1.11067263e-03,
       2.00439850e-04, 1.55696655e-04, 1.83750185e-04, 1.67635944e-04,
       1.42731266e-04, 2.00439850e-04, 9.94859100e-04, 1.49635069e-04,
       1.80158537e-04, 1.54258434e-04, 1.33094464e-04, 1.34849517e-04,
       1.29276312e-04, 7.83972739e-04, 1.31650833e-04, 1.27241073e-04,
       1.27860541e-04, 1.36241941e-04, 1.33924889e-04, 1.56313855e-04,
       7.78054472e-04, 1.22518987e-04, 1.30266307e-04, 9.87884918e-05,
       1.37104501e-04, 1.39943426e-04, 1.31261170e-04, 8.03962874e-04,
       1.36696935e-04, 1.28180001e-04, 1.30776841e-04, 1.76045486e-04,
       1.74144321e-04, 1.04280520e-03, 1.72114902e-04, 1.72158781e-04,
       1.73198332e-04, 1.75143376e-04, 1.52718143e-04, 1.29630880e-04,
       1.72114902e-04, 9.04786522e-04, 1.34869404e-04, 1.64805714e-04,
       1.50647479e-04, 1.69430202e-04, 1.55831388e-04, 1.42679695e-04,
       1.72158781e-04, 9.49434967e-04, 1.46727315e-04, 1.62607587e-04,
       1.29429768e-04, 1.44107946e-04, 1.34869404e-04, 8.27188373e-04,
       1.47988743e-04, 1.36175676e-04, 1.34616836e-04, 1.33655460e-04,
       1.23248813e-04, 1.46727315e-04, 8.20934853e-04, 1.25825125e-04,
       1.50986841e-04, 1.40491299e-04, 1.57525833e-04, 1.59854919e-04,
       1.47988743e-04, 1.15775636e-03, 1.69265354e-04, 1.55618247e-04,
       1.85107081e-04, 1.82396184e-04, 1.56620789e-04, 1.51872396e-04,
       1.69265354e-04, 9.75581155e-04, 1.57581181e-04, 1.73588607e-04,
       1.66652828e-04, 1.20560877e-04, 1.25206273e-04, 1.19746936e-04,
       7.27059319e-04, 1.23914787e-04, 1.19053995e-04, 1.18576451e-04,
       1.41271738e-04, 1.34444907e-04, 1.23914787e-04, 6.13552128e-04,
       1.05657272e-04, 1.08263424e-04, 1.49376150e-04, 1.45134298e-04,
       1.43828018e-04, 1.43609266e-04, 8.88793991e-04, 1.56959711e-04,
       1.49886548e-04, 1.54108057e-04, 1.51693034e-04, 1.56959711e-04,
       9.64548536e-04, 1.74831635e-04, 1.53334576e-04, 1.73621523e-04,
       1.43174624e-04, 1.42743355e-04, 8.43723514e-04, 1.40881078e-04,
       1.40403542e-04, 1.38237324e-04, 1.38283591e-04, 1.43414585e-04,
       1.44742146e-04, 1.40881078e-04, 8.49007167e-04, 1.42555944e-04,
       1.38533055e-04, 1.38880359e-04, 1.45137311e-04, 1.45231117e-04,
       1.42555944e-04, 8.56306452e-04, 1.42802387e-04, 1.40213528e-04,
       1.40366165e-04, 1.42816375e-04, 1.43917961e-04, 1.40403542e-04,
       8.49253533e-04, 1.43336618e-04, 1.38473774e-04, 1.40305263e-04,
       1.45657344e-04, 1.46245549e-04, 1.42802387e-04, 8.66421487e-04,
       1.45758894e-04, 1.41794506e-04, 1.44162807e-04, 1.46436217e-04,
       1.46310884e-04, 1.45758894e-04, 8.77132017e-04, 1.46533658e-04,
       1.45596133e-04, 1.46496231e-04, 1.45723430e-04, 1.44959171e-04,
       1.46533658e-04, 8.77055249e-04, 1.46454329e-04, 1.46349865e-04,
       1.47034796e-04, 1.44851138e-04, 1.46402342e-04, 1.43336618e-04,
       8.68310390e-04, 1.46608664e-04, 1.42695394e-04, 1.44416235e-04,
       1.45257453e-04, 1.45448143e-04, 1.46779075e-04, 1.46454329e-04,
       8.78907267e-04, 1.47551161e-04, 1.47417105e-04, 1.46871223e-04,
       1.48750256e-04, 1.46608664e-04, 8.88252431e-04, 1.50106511e-04,
       1.47148482e-04, 1.48767296e-04, 1.45761613e-04, 1.47926232e-04,
       8.67771350e-04, 1.42952884e-04, 1.45191229e-04, 1.42932833e-04,
       1.43006559e-04, 1.44477682e-04, 1.45593807e-04, 1.42952884e-04,
       8.67257407e-04, 1.45533963e-04, 1.43617059e-04, 1.45082012e-04,
       1.46399097e-04, 1.42286841e-04, 1.45191229e-04, 8.62483540e-04,
       1.41834170e-04, 1.43008503e-04, 1.43763700e-04, 1.41019564e-04,
       1.30584637e-04, 1.41381870e-04, 7.87534315e-04, 1.22458536e-04,
       1.30289058e-04, 1.21800651e-04, 1.55440262e-04, 1.53163499e-04,
       1.55077067e-04, 9.08687641e-04, 1.48340121e-04, 1.50563437e-04,
       1.46103255e-04, 1.51719564e-04, 1.49635069e-04, 1.57581181e-04,
       9.32551690e-04, 1.53039158e-04, 1.61517122e-04, 1.59059595e-04,
       1.36753238e-04, 1.31650833e-04, 1.41834170e-04, 8.23282840e-04,
       1.32776025e-04, 1.42112157e-04, 1.38156417e-04, 1.45141641e-04,
       1.43068727e-04, 8.62999896e-04, 1.47442639e-04, 1.40828819e-04,
       1.45529488e-04, 1.40988581e-04, 1.43771165e-04, 1.45642350e-04,
       1.22518987e-04, 1.47442639e-04, 9.35094663e-04, 9.67743115e-05,
       1.52870158e-04, 1.26075052e-04, 1.46281278e-04, 1.51822188e-04,
       8.79987204e-04, 1.42304886e-04, 1.49148508e-04, 1.44563816e-04,
       1.45866528e-04, 1.39920499e-04, 1.38263392e-04, 1.42304886e-04,
       8.50584608e-04, 1.41039876e-04, 1.44331929e-04, 1.44724027e-04,
       1.53538560e-04, 1.55689653e-04, 1.49148508e-04, 9.08163388e-04,
       1.53870084e-04, 1.46673050e-04, 1.49243533e-04, 1.55858637e-04,
       1.63449813e-04, 1.53870084e-04, 9.88839336e-04, 1.72814860e-04,
       1.65746171e-04, 1.77099771e-04, 1.64850996e-04, 1.55183647e-04,
       1.72814860e-04, 1.02303035e-03, 1.62110224e-04, 1.84553957e-04,
       1.83516670e-04, 1.38411337e-04, 1.37344095e-04, 1.41039876e-04,
       8.36069069e-04, 1.38003112e-04, 1.41620086e-04, 1.39650564e-04,
       1.37014649e-04, 1.38359912e-04, 1.38003112e-04, 8.31116328e-04,
       1.39386146e-04, 1.39157369e-04, 1.39195141e-04, 1.50424980e-04,
       1.39399659e-04, 1.62110224e-04, 8.81022138e-04, 1.33410870e-04,
       1.56675218e-04, 1.39001185e-04, 1.47510569e-04, 1.46474898e-04,
       1.33410870e-04, 8.11128421e-04, 1.27679454e-04, 1.30374187e-04,
       1.25678442e-04, 1.44613878e-04, 1.49680831e-04, 8.82769953e-04,
       1.43117669e-04, 1.51672773e-04, 1.45098325e-04, 1.48586476e-04,
       1.41470317e-04, 1.39227787e-04, 1.39386146e-04, 1.43117669e-04,
       8.45375812e-04, 1.40342450e-04, 1.41831443e-04, 1.29529261e-04,
       1.30266307e-04, 6.98551957e-04, 1.10388125e-04, 1.02185925e-04,
       1.16823746e-04, 1.09358593e-04, 1.41011086e-04, 1.36696935e-04,
       1.40828819e-04, 8.35832902e-04, 1.36540955e-04, 1.40364410e-04,
       1.40390697e-04, 1.56074232e-04, 1.69422849e-04, 1.51672773e-04,
       9.59672413e-04, 1.71413051e-04, 1.52348923e-04, 1.58740585e-04,
       1.21390606e-04, 1.25825125e-04, 1.19053995e-04, 7.39825075e-04,
       1.28665722e-04, 1.19856209e-04, 1.25033417e-04, 1.54004790e-04,
       1.05657272e-04, 7.84751768e-04, 1.65279531e-04, 1.09015128e-04,
       1.21439080e-04, 1.29355966e-04, 1.43895668e-04, 1.35333409e-04,
       1.49886548e-04, 1.74831635e-04, 1.07053825e-03, 1.18487710e-04,
       1.94170027e-04, 1.53933255e-04, 1.25885175e-04, 1.28180001e-04,
       1.18487710e-04, 6.06923410e-04, 1.21697388e-04, 1.12673137e-04,
       1.41948460e-04, 1.41795609e-04, 8.58098536e-04, 1.43459132e-04,
       1.43150803e-04, 1.43950006e-04, 1.43794527e-04, 1.43236312e-04,
       1.43356851e-04, 1.43459132e-04, 8.62304630e-04, 1.43726877e-04,
       1.44189126e-04, 1.44336331e-04, 1.43191419e-04, 1.44634253e-04,
       1.43726877e-04, 8.63816342e-04, 1.43746106e-04, 1.44970646e-04,
       1.43547040e-04, 1.44765067e-04, 1.43458178e-04, 1.43746106e-04,
       8.58026175e-04, 1.42495393e-04, 1.41752628e-04, 1.41808803e-04,
       1.39102528e-04, 1.37787112e-04, 1.43150803e-04, 8.48648373e-04,
       1.42086581e-04, 1.43135078e-04, 1.43386272e-04, 1.44288699e-04,
       1.45659330e-04, 1.42495393e-04, 8.67289595e-04, 1.46502018e-04,
       1.42854081e-04, 1.45490074e-04, 1.42706218e-04, 1.40125671e-04,
       1.42086581e-04, 8.57352608e-04, 1.44379888e-04, 1.44170745e-04,
       1.43883505e-04, 1.44531586e-04, 1.45301125e-04, 1.44379888e-04,
       8.63689227e-04, 1.44231573e-04, 1.43081455e-04, 1.42163600e-04,
       1.46730737e-04, 1.48372764e-04, 1.46502018e-04, 8.84684837e-04,
       1.48607292e-04, 1.47004389e-04, 1.47467637e-04, 1.48968811e-04,
       1.53334576e-04, 1.44231573e-04, 8.84373128e-04, 1.49465178e-04,
       1.43752575e-04, 1.44620415e-04, 1.48871980e-04, 1.49319745e-04,
       1.48607292e-04, 8.98419207e-04, 1.50907279e-04, 1.49430344e-04,
       1.51282566e-04, 1.49981661e-04, 1.50106511e-04, 1.50907279e-04,
       9.05740546e-04, 1.50701682e-04, 1.52186930e-04, 1.51856483e-04,
       1.47425077e-04, 1.47551161e-04, 1.45533963e-04, 8.78518104e-04,
       1.46565341e-04, 1.45268634e-04, 1.46173928e-04, 1.24716514e-04,
       1.27241073e-04, 1.22458536e-04, 7.12596250e-04, 1.23976480e-04,
       1.07605131e-04, 1.06598516e-04, 1.52380115e-04, 1.51692212e-04,
       1.48340121e-04, 8.89775344e-04, 1.45592009e-04, 1.46915548e-04,
       1.44855339e-04, 1.44195898e-04, 1.30289058e-04, 1.45592009e-04,
       8.32863595e-04, 1.19494694e-04, 1.52741205e-04, 1.40550731e-04,
       9.47092368e-05, 1.36375692e-04, 5.47316204e-04, 6.62034056e-05,
       1.45464773e-04, 1.04563096e-04, 9.52512508e-05, 1.36405517e-04,
       6.62034056e-05, 7.18442228e-04, 1.44878096e-04, 1.19091767e-04,
       1.56612191e-04, 1.45514284e-04, 1.55696655e-04, 1.44878096e-04,
       9.19509296e-04, 1.66423413e-04, 1.47816952e-04, 1.59179897e-04,
       1.47246089e-04, 1.50551760e-04, 1.45464773e-04, 9.44987035e-04,
       1.67478785e-04, 1.57768643e-04, 1.76476985e-04, 1.55915579e-04,
       1.50563437e-04, 1.67478785e-04, 9.28846648e-04, 1.46373640e-04,
       1.62134105e-04, 1.46381101e-04, 1.83750185e-04, 1.80158537e-04,
       1.66423413e-04, 1.00628938e-03, 1.60164168e-04, 1.59230336e-04,
       1.56562740e-04, 1.54258434e-04, 1.53039158e-04, 1.60164168e-04,
       9.37421187e-04, 1.56980511e-04, 1.57471649e-04, 1.55507267e-04,
       1.27860541e-04, 1.32776025e-04, 1.23976480e-04, 7.74555895e-04,
       1.33186839e-04, 1.26230567e-04, 1.30525443e-04, 9.87884918e-05,
       9.67743115e-05, 1.10388125e-04, 5.17130510e-04, 9.97898397e-05,
       1.11389742e-04, 1.13825522e-04, 1.27679454e-04, 1.02185925e-04,
       5.97571277e-04, 1.32438346e-04, 1.21442029e-04, 1.30776841e-04,
       1.36540955e-04, 1.21697388e-04, 7.87774631e-04, 1.39746528e-04,
       1.23363947e-04, 1.35648973e-04, 1.86649432e-04, 1.71413051e-04,
       1.65279531e-04, 1.00057578e-03, 1.62613564e-04, 1.51024892e-04,
       1.63595306e-04, 1.73198332e-04, 1.64805714e-04, 1.02995555e-03,
       1.77590986e-04, 1.61101597e-04, 1.80677845e-04, 1.72581075e-04,
       1.75143376e-04, 1.62607587e-04, 1.50986841e-04, 1.77590986e-04,
       9.96385429e-04, 1.57994142e-04, 1.72062498e-04, 1.50647479e-04,
       1.36175676e-04, 1.61101597e-04, 9.19364914e-04, 1.43781779e-04,
       1.65253199e-04, 1.62405184e-04, 1.34616836e-04, 1.55618247e-04,
       1.43781779e-04, 9.38164131e-04, 1.65978563e-04, 1.68486666e-04,
       1.69682039e-04, 1.40491299e-04, 1.28665722e-04, 1.57994142e-04,
       8.83053212e-04, 1.34169333e-04, 1.66865974e-04, 1.54866742e-04,
       1.85107081e-04, 1.73588607e-04, 1.03885992e-03, 1.66740301e-04,
       1.78720171e-04, 1.67121179e-04, 1.67582576e-04, 1.66652828e-04,
       1.61517122e-04, 1.66740301e-04, 9.82681945e-04, 1.60133821e-04,
       1.64554324e-04, 1.63083549e-04, 1.18576451e-04, 1.08263424e-04,
       1.19856209e-04, 1.09015128e-04, 7.01065532e-04, 1.22941187e-04,
       1.22413132e-04, 1.73621523e-04, 1.94170027e-04, 1.49465178e-04,
       9.66813683e-04, 1.64590485e-04, 1.45194834e-04, 1.39771636e-04,
       1.38237324e-04, 1.38473774e-04, 8.16618944e-04, 1.35833410e-04,
       1.36763816e-04, 1.33308332e-04, 1.34002289e-04, 1.38283591e-04,
       1.38533055e-04, 1.35833410e-04, 8.16883607e-04, 1.35857039e-04,
       1.34301174e-04, 1.34075339e-04, 1.38880359e-04, 1.40213528e-04,
       1.35857039e-04, 8.19545737e-04, 1.37926336e-04, 1.32966173e-04,
       1.33702302e-04, 1.40366165e-04, 1.41794506e-04, 1.37926336e-04,
       8.35137512e-04, 1.41227581e-04, 1.35975010e-04, 1.37847915e-04,
       1.40305263e-04, 1.42695394e-04, 1.36763816e-04, 8.30879825e-04,
       1.41596908e-04, 1.33537742e-04, 1.35980702e-04, 1.44162807e-04,
       1.45596133e-04, 1.41227581e-04, 8.57808476e-04, 1.45062675e-04,
       1.39678755e-04, 1.42080524e-04, 1.46496231e-04, 1.46349865e-04,
       1.45062675e-04, 8.74638309e-04, 1.46553157e-04, 1.44269767e-04,
       1.45906614e-04, 1.47034796e-04, 1.47417105e-04, 1.46565341e-04,
       1.46553157e-04, 8.80092669e-04, 1.46076073e-04, 1.46446197e-04,
       1.44416235e-04, 1.47148482e-04, 1.41596908e-04, 8.63970262e-04,
       1.47297159e-04, 1.40271736e-04, 1.43239742e-04, 1.48767296e-04,
       1.50701682e-04, 1.47297159e-04, 8.93898153e-04, 1.50560471e-04,
       1.47621310e-04, 1.48950235e-04, 1.42932833e-04, 1.43617059e-04,
       8.61368401e-04, 1.42571074e-04, 1.44508042e-04, 1.44496068e-04,
       1.43243326e-04, 1.43006559e-04, 1.43008503e-04, 1.42571074e-04,
       8.49797408e-04, 1.40536426e-04, 1.41355719e-04, 1.39319128e-04,
       1.45082012e-04, 1.45268634e-04, 1.44508042e-04, 8.71040132e-04,
       1.45360240e-04, 1.45077813e-04, 1.45743390e-04, 1.43763700e-04,
       1.42112157e-04, 1.40536426e-04, 8.39825714e-04, 1.39672214e-04,
       1.36476944e-04, 1.37264274e-04, 1.21800651e-04, 1.07605131e-04,
       1.19494694e-04, 5.49117660e-04, 9.32449065e-05, 1.06972278e-04,
       1.46103255e-04, 1.46915548e-04, 1.46373640e-04, 8.52957963e-04,
       1.40734640e-04, 1.39641087e-04, 1.33189794e-04, 1.59059595e-04,
       1.56980511e-04, 1.60133821e-04, 9.52073277e-04, 1.56691251e-04,
       1.60285793e-04, 1.58922307e-04, 1.38156417e-04, 1.33186839e-04,
       1.39672214e-04, 8.28906481e-04, 1.34361977e-04, 1.41934847e-04,
       1.41594187e-04, 1.45529488e-04, 1.52870158e-04, 8.80583600e-04,
       1.38627742e-04, 1.54985378e-04, 1.48793900e-04, 1.39776934e-04,
       1.40988581e-04, 1.40364410e-04, 1.38627742e-04, 8.46399941e-04,
       1.41543820e-04, 1.40667569e-04, 1.44207818e-04, 1.26075052e-04,
       9.97898397e-05, 1.54985378e-04, 8.05723955e-04, 1.20864469e-04,
       1.55922456e-04, 1.48086760e-04, 1.44563816e-04, 1.44331929e-04,
       8.61895427e-04, 1.44341143e-04, 1.43357203e-04, 1.43026694e-04,
       1.42274641e-04, 1.45866528e-04, 1.46673050e-04, 1.44341143e-04,
       8.68594065e-04, 1.45602258e-04, 1.42828246e-04, 1.43282840e-04,
       1.44724027e-04, 1.41620086e-04, 1.43357203e-04, 8.53284041e-04,
       1.40466079e-04, 1.41451915e-04, 1.41664731e-04, 1.49243533e-04,
       1.65746171e-04, 1.45602258e-04, 9.33886743e-04, 1.76321791e-04,
       1.41378047e-04, 1.55594943e-04, 1.77099771e-04, 1.84553957e-04,
       1.76321791e-04, 1.16201173e-03, 1.92684604e-04, 1.49731559e-04,
       1.56545731e-04, 1.25074319e-04, 1.83516670e-04, 1.56675218e-04,
       1.92684604e-04, 9.80629742e-04, 1.42131567e-04, 1.64666635e-04,
       1.40955049e-04, 1.39650564e-04, 1.39157369e-04, 1.40466079e-04,
       8.39619267e-04, 1.39351479e-04, 1.40807591e-04, 1.40186185e-04,
       1.39195141e-04, 1.40342450e-04, 1.39351479e-04, 8.41739511e-04,
       1.41240203e-04, 1.40434412e-04, 1.41175827e-04, 1.39001185e-04,
       1.30374187e-04, 1.42131567e-04, 8.09890619e-04, 1.27300006e-04,
       1.38644118e-04, 1.32439556e-04, 1.25678442e-04, 1.32438346e-04,
       1.27300006e-04, 8.06402373e-04, 1.41023856e-04, 1.36498888e-04,
       1.43462835e-04, 1.45098325e-04, 1.41831443e-04, 1.41240203e-04,
       8.63334868e-04, 1.47138848e-04, 1.42697144e-04, 1.45328907e-04,
       1.48586476e-04, 1.52348923e-04, 1.47138848e-04, 9.00264986e-04,
       1.53353724e-04, 1.48192293e-04, 1.50644722e-04, 1.16823746e-04,
       1.11389742e-04, 1.20864469e-04, 7.44072856e-04, 1.22371242e-04,
       1.38275444e-04, 1.34348213e-04, 1.09358593e-04, 1.21442029e-04,
       1.41023856e-04, 1.22371242e-04, 7.67914508e-04, 1.40143983e-04,
       1.33574804e-04, 1.40390697e-04, 1.39746528e-04, 1.41543820e-04,
       8.54072586e-04, 1.42264743e-04, 1.44380853e-04, 1.45745945e-04,
       1.58740585e-04, 1.62613564e-04, 1.53353724e-04, 9.43418711e-04,
       1.60187067e-04, 1.52781703e-04, 1.55742067e-04, 1.82396184e-04,
       1.65978563e-04, 1.78720171e-04, 9.90901698e-04, 1.57573883e-04,
       1.55480781e-04, 1.50752115e-04, 1.25033417e-04, 1.34169333e-04,
       1.22941187e-04, 7.98067825e-04, 1.28961409e-04, 1.45039086e-04,
       1.41923392e-04, 1.21439080e-04, 1.22413132e-04, 1.28961409e-04,
       7.79379902e-04, 1.30098195e-04, 1.37178623e-04, 1.39289462e-04,
       1.29355966e-04, 1.51024892e-04, 1.30098195e-04, 8.76290286e-04,
       1.64914854e-04, 1.43874323e-04, 1.57022056e-04, 1.53933255e-04,
       1.12673137e-04, 1.23363947e-04, 1.64590485e-04, 8.23694922e-04,
       1.34583839e-04, 1.34550260e-04, 1.43950006e-04, 1.44189126e-04,
       8.67746162e-04, 1.44195034e-04, 1.44622517e-04, 1.45488921e-04,
       1.45300558e-04, 1.43794527e-04, 1.43135078e-04, 1.44195034e-04,
       8.61653312e-04, 1.42628245e-04, 1.44403884e-04, 1.43496543e-04,
       1.44336331e-04, 1.44970646e-04, 1.44622517e-04, 8.67321719e-04,
       1.44700116e-04, 1.44624412e-04, 1.44067696e-04, 1.43547040e-04,
       1.41752628e-04, 1.44700116e-04, 8.51819467e-04, 1.40329533e-04,
       1.42033160e-04, 1.39456990e-04, 1.41808803e-04, 1.42854081e-04,
       1.40329533e-04, 8.46902855e-04, 1.43295418e-04, 1.38347206e-04,
       1.40267814e-04, 1.43386272e-04, 1.44170745e-04, 1.42628245e-04,
       8.56119516e-04, 1.42910765e-04, 1.41762721e-04, 1.41260769e-04,
       1.45490074e-04, 1.47004389e-04, 1.43295418e-04, 8.70836270e-04,
       1.47762435e-04, 1.42165626e-04, 1.45118328e-04, 1.43883505e-04,
       1.43081455e-04, 1.42910765e-04, 8.47602086e-04, 1.39904967e-04,
       1.40012571e-04, 1.37808823e-04, 1.42163600e-04, 1.43752575e-04,
       1.39904967e-04, 8.43953944e-04, 1.41410610e-04, 1.38402763e-04,
       1.38319431e-04, 1.47467637e-04, 1.49430344e-04, 1.47762435e-04,
       8.95846911e-04, 1.51428275e-04, 1.49027544e-04, 1.50730677e-04,
       1.44620415e-04, 1.45194834e-04, 1.41410610e-04, 8.53829497e-04,
       1.43095627e-04, 1.39198706e-04, 1.40309305e-04, 1.51282566e-04,
       1.52186930e-04, 1.51428275e-04, 9.10755472e-04, 1.52232303e-04,
       1.51862867e-04, 1.51762532e-04, 1.51856483e-04, 1.50560471e-04,
       1.52232303e-04, 9.01247634e-04, 1.48981688e-04, 1.49785646e-04,
       1.47831043e-04, 1.46173928e-04, 1.46076073e-04, 1.45360240e-04,
       8.78083111e-04, 1.46283026e-04, 1.46584601e-04, 1.47605242e-04,
       1.06598516e-04, 1.26230567e-04, 9.32449065e-05, 7.17914291e-04,
       1.24935790e-04, 1.27422839e-04, 1.39481672e-04, 1.44855339e-04,
       1.52741205e-04, 1.40734640e-04, 8.94872745e-04, 1.59858443e-04,
       1.42722590e-04, 1.53960528e-04, 1.40550731e-04, 1.06972278e-04,
       1.24935790e-04, 1.59858443e-04, 9.95691762e-04, 1.55705578e-04,
       1.54130568e-04, 1.53538375e-04, 1.04563096e-04, 1.19091767e-04,
       1.57768643e-04, 1.01476406e-03, 1.59212852e-04, 1.72474244e-04,
       1.56916440e-04, 1.44737013e-04, 1.56612191e-04, 1.47816952e-04,
       1.59212852e-04, 9.15110358e-04, 1.54900552e-04, 1.46042435e-04,
       1.50525375e-04, 1.59179897e-04, 1.59230336e-04, 1.54900552e-04,
       9.33215548e-04, 1.52884397e-04, 1.54543480e-04, 1.52476886e-04,
       1.76476985e-04, 1.62134105e-04, 1.72474244e-04, 1.10891683e-03,
       1.15322127e-04, 1.72250158e-04, 1.33384514e-04, 1.76874701e-04,
       1.46381101e-04, 1.39641087e-04, 1.15322127e-04, 5.96018948e-04,
       1.08653006e-04, 8.60216267e-05, 1.56562740e-04, 1.57471649e-04,
       1.52884397e-04, 9.25625514e-04, 1.53424074e-04, 1.52825943e-04,
       1.52456711e-04, 1.55507267e-04, 1.56691251e-04, 1.53424074e-04,
       9.34448103e-04, 1.57354420e-04, 1.54362365e-04, 1.57108727e-04,
       1.30525443e-04, 1.34361977e-04, 1.27422839e-04, 7.80020156e-04,
       1.34183966e-04, 1.25300670e-04, 1.28225262e-04, 1.35648973e-04,
       1.42264743e-04, 1.34583839e-04, 8.37573793e-04, 1.38258388e-04,
       1.44879536e-04, 1.41938314e-04, 1.63595306e-04, 1.60187067e-04,
       1.64914854e-04, 9.76758880e-04, 1.59756690e-04, 1.65027444e-04,
       1.63277518e-04, 1.80677845e-04, 1.72062498e-04, 1.66865974e-04,
       1.06422415e-03, 1.83080461e-04, 1.76969116e-04, 1.84568256e-04,
       1.72581075e-04, 1.65253199e-04, 1.83080461e-04, 1.04190637e-03,
       1.65713107e-04, 1.82659003e-04, 1.72619524e-04, 1.62405184e-04,
       1.68486666e-04, 1.65713107e-04, 9.94605828e-04, 1.67822819e-04,
       1.67074910e-04, 1.63103141e-04, 1.69682039e-04, 1.57573883e-04,
       1.67822819e-04, 9.54936464e-04, 1.52872397e-04, 1.54913796e-04,
       1.52071529e-04, 1.54866742e-04, 1.45039086e-04, 1.76969116e-04,
       1.00474806e-03, 1.55834628e-04, 1.91672660e-04, 1.80365827e-04,
       1.67121179e-04, 1.64554324e-04, 9.95058082e-04, 1.67923000e-04,
       1.63906709e-04, 1.66501153e-04, 1.65051717e-04, 1.67582576e-04,
       1.55480781e-04, 1.67923000e-04, 9.18907006e-04, 1.41722235e-04,
       1.50148693e-04, 1.36049722e-04, 1.63083549e-04, 1.60285793e-04,
       1.63906709e-04, 9.61207548e-04, 1.57238828e-04, 1.60281397e-04,
       1.56411272e-04, 1.39771636e-04, 1.34550260e-04, 1.43095627e-04,
       1.38258388e-04, 8.39176799e-04, 1.41558375e-04, 1.41942513e-04,
       1.33308332e-04, 1.34301174e-04, 7.96689353e-04, 1.31634100e-04,
       1.32863892e-04, 1.32409402e-04, 1.32172453e-04, 1.34002289e-04,
       1.33537742e-04, 1.31634100e-04, 7.83503387e-04, 1.29952301e-04,
       1.27797104e-04, 1.26579851e-04, 1.34075339e-04, 1.32966173e-04,
       1.32863892e-04, 7.87482480e-04, 1.29483132e-04, 1.30182769e-04,
       1.27911175e-04, 1.33702302e-04, 1.35975010e-04, 1.29483132e-04,
       7.87923433e-04, 1.33182910e-04, 1.27076505e-04, 1.28503574e-04,
       1.37847915e-04, 1.39678755e-04, 1.33182910e-04, 8.09554406e-04,
       1.37755140e-04, 1.29174148e-04, 1.31915538e-04, 1.35980702e-04,
       1.40271736e-04, 1.29952301e-04, 7.95215499e-04, 1.38425050e-04,
       1.23201998e-04, 1.27383712e-04, 1.42080524e-04, 1.44269767e-04,
       1.37755140e-04, 8.43078751e-04, 1.43080858e-04, 1.36377993e-04,
       1.39514468e-04, 1.45906614e-04, 1.46446197e-04, 1.46283026e-04,
       1.43080858e-04, 8.71310830e-04, 1.46128359e-04, 1.43465775e-04,
       1.43239742e-04, 1.47621310e-04, 1.38425050e-04, 8.56316596e-04,
       1.46507191e-04, 1.38411366e-04, 1.42111938e-04, 1.48950235e-04,
       1.48981688e-04, 1.46507191e-04, 8.79017408e-04, 1.45819855e-04,
       1.44738614e-04, 1.44019825e-04, 1.44496068e-04, 1.45077813e-04,
       8.71915858e-04, 1.45158092e-04, 1.46111734e-04, 1.45722023e-04,
       1.45350127e-04, 1.43243326e-04, 1.41355719e-04, 1.45158092e-04,
       8.60929081e-04, 1.42098750e-04, 1.45122464e-04, 1.43950729e-04,
       1.39319128e-04, 1.36476944e-04, 1.42098750e-04, 8.46169426e-04,
       1.40078546e-04, 1.43687039e-04, 1.44509019e-04, 1.45743390e-04,
       1.46584601e-04, 1.46111734e-04, 8.73791006e-04, 1.46440277e-04,
       1.44711835e-04, 1.44199168e-04, 1.37264274e-04, 1.41934847e-04,
       1.40078546e-04, 8.50668168e-04, 1.42766099e-04, 1.45303711e-04,
       1.43320691e-04, 1.33189794e-04, 1.42722590e-04, 1.08653006e-04,
       7.93407850e-04, 1.48335278e-04, 1.15178853e-04, 1.45328329e-04,
       1.58922307e-04, 1.57354420e-04, 1.57238828e-04, 9.59132474e-04,
       1.61685640e-04, 1.58958290e-04, 1.64972990e-04, 1.41594187e-04,
       1.34183966e-04, 1.42766099e-04, 8.28254058e-04, 1.32861118e-04,
       1.39671723e-04, 1.37176964e-04, 1.48793900e-04, 1.55922456e-04,
       8.99765974e-04, 1.46034626e-04, 1.50769674e-04, 1.50319412e-04,
       1.47925904e-04, 1.39776934e-04, 1.40667569e-04, 1.46034626e-04,
       8.62038050e-04, 1.45993571e-04, 1.45248520e-04, 1.44316829e-04,
       1.44207818e-04, 1.44380853e-04, 1.45993571e-04, 8.75598390e-04,
       1.46542173e-04, 1.47049203e-04, 1.47424771e-04, 1.48086760e-04,
       1.38275444e-04, 1.50769674e-04, 8.50894142e-04, 1.35358331e-04,
       1.42001981e-04, 1.36401953e-04, 1.43026694e-04, 1.42828246e-04,
       8.55044809e-04, 1.42589204e-04, 1.42374874e-04, 1.42120836e-04,
       1.42104954e-04, 1.42274641e-04, 1.41451915e-04, 1.42589204e-04,
       8.51599952e-04, 1.41558806e-04, 1.41966529e-04, 1.41758857e-04,
       1.43282840e-04, 1.41378047e-04, 1.42374874e-04, 8.46480469e-04,
       1.38944150e-04, 1.41013244e-04, 1.39487313e-04, 1.41664731e-04,
       1.40807591e-04, 1.41558806e-04, 8.48250304e-04, 1.40877954e-04,
       1.41582467e-04, 1.41758755e-04, 1.55594943e-04, 1.49731559e-04,
       1.38944150e-04, 8.39388774e-04, 1.27040586e-04, 1.37058859e-04,
       1.31018678e-04, 1.56545731e-04, 1.64666635e-04, 8.44465156e-04,
       1.37432180e-04, 1.22653231e-04, 1.28254668e-04, 1.34912712e-04,
       1.40955049e-04, 1.38644118e-04, 1.37432180e-04, 8.31401604e-04,
       1.36807878e-04, 1.37937875e-04, 1.39624504e-04, 1.40186185e-04,
       1.40434412e-04, 1.40877954e-04, 8.46058000e-04, 1.41296694e-04,
       1.41546121e-04, 1.41716634e-04, 1.41175827e-04, 1.42697144e-04,
       1.41296694e-04, 8.53847674e-04, 1.43501607e-04, 1.42246402e-04,
       1.42929999e-04, 1.32439556e-04, 1.36498888e-04, 1.36807878e-04,
       8.45628340e-04, 1.47810525e-04, 1.42564090e-04, 1.49507404e-04,
       1.43462835e-04, 1.40143983e-04, 1.47810525e-04, 8.59430729e-04,
       1.38197881e-04, 1.48054648e-04, 1.41760857e-04, 1.45328907e-04,
       1.48192293e-04, 1.43501607e-04, 8.72656967e-04, 1.47849222e-04,
       1.43150354e-04, 1.44634583e-04, 1.50644722e-04, 1.52781703e-04,
       1.47849222e-04, 8.98726694e-04, 1.51356289e-04, 1.47362336e-04,
       1.48732421e-04, 1.34348213e-04, 1.33574804e-04, 1.35358331e-04,
       1.38197881e-04, 8.12135217e-04, 1.33521514e-04, 1.37134473e-04,
       1.45745945e-04, 1.44879536e-04, 1.46542173e-04, 8.79135290e-04,
       1.45620764e-04, 1.48200936e-04, 1.48145936e-04, 1.55742067e-04,
       1.59756690e-04, 1.51356289e-04, 9.33165112e-04, 1.60183513e-04,
       1.50656976e-04, 1.55469577e-04, 1.50752115e-04, 1.52872397e-04,
       1.41722235e-04, 8.69330072e-04, 1.47310080e-04, 1.36602841e-04,
       1.40070404e-04, 1.41923392e-04, 1.37178623e-04, 1.55834628e-04,
       8.86967696e-04, 1.41433490e-04, 1.60126965e-04, 1.50470596e-04,
       1.39289462e-04, 1.43874323e-04, 1.41433490e-04, 8.60866492e-04,
       1.47973930e-04, 1.43169854e-04, 1.45125433e-04, 1.57022056e-04,
       1.65027444e-04, 1.47973930e-04, 9.35214233e-04, 1.62393811e-04,
       1.48191250e-04, 1.54605742e-04, 1.45488921e-04, 1.44624412e-04,
       8.64086666e-04, 1.45773215e-04, 1.42181122e-04, 1.44373290e-04,
       1.41645705e-04, 1.45300558e-04, 1.44403884e-04, 1.45773215e-04,
       8.72942115e-04, 1.45223283e-04, 1.45947216e-04, 1.46293958e-04,
       1.43496543e-04, 1.41762721e-04, 1.45223283e-04, 8.67303434e-04,
       1.44063822e-04, 1.46091352e-04, 1.46665712e-04, 1.44067696e-04,
       1.42033160e-04, 1.42181122e-04, 8.44343082e-04, 1.38217343e-04,
       1.39886501e-04, 1.37957259e-04, 1.39456990e-04, 1.38347206e-04,
       1.38217343e-04, 8.14208219e-04, 1.32522548e-04, 1.35124572e-04,
       1.30539560e-04, 1.40267814e-04, 1.42165626e-04, 1.32522548e-04,
       8.56220588e-04, 1.60888703e-04, 1.26953777e-04, 1.53422120e-04,
       1.41260769e-04, 1.40012571e-04, 1.44063822e-04, 8.55933882e-04,
       1.41536434e-04, 1.45169739e-04, 1.43890548e-04, 1.45118328e-04,
       1.49027544e-04, 1.60888703e-04, 9.57531007e-04, 1.54381122e-04,
       1.79266053e-04, 1.68849257e-04, 1.37808823e-04, 1.38402763e-04,
       1.41536434e-04, 8.19278453e-04, 1.34353423e-04, 1.37476980e-04,
       1.29700030e-04, 1.38319431e-04, 1.39198706e-04, 1.34353423e-04,
       8.24900493e-04, 1.39597082e-04, 1.34533734e-04, 1.38898118e-04,
       1.50730677e-04, 1.51862867e-04, 1.54381122e-04, 9.22729655e-04,
       1.52470197e-04, 1.58163954e-04, 1.55120838e-04, 1.40309305e-04,
       1.41558375e-04, 1.39597082e-04, 8.49219908e-04, 1.43532103e-04,
       1.40768546e-04, 1.43454497e-04, 1.51762532e-04, 1.49785646e-04,
       1.52470197e-04, 8.96473414e-04, 1.46684502e-04, 1.49789673e-04,
       1.45980863e-04, 1.47831043e-04, 1.45819855e-04, 1.46684502e-04,
       8.63897399e-04, 1.42520885e-04, 1.41596771e-04, 1.39444343e-04,
       1.47605242e-04, 1.46128359e-04, 1.46440277e-04, 8.71507393e-04,
       1.43631205e-04, 1.44517249e-04, 1.43185060e-04, 1.39481672e-04,
       1.55705578e-04, 1.25300670e-04, 8.06655909e-04, 1.25839805e-04,
       1.38006478e-04, 1.22321707e-04, 1.53960528e-04, 1.54130568e-04,
       1.48335278e-04, 9.13643879e-04, 1.49899298e-04, 1.52962114e-04,
       1.54356094e-04, 1.56916440e-04, 1.72250158e-04, 9.35473690e-04,
       1.40011555e-04, 1.64918353e-04, 1.55475131e-04, 1.45902052e-04,
       1.44737013e-04, 1.46042435e-04, 1.40011555e-04, 7.17951526e-04,
       1.45359030e-04, 1.41801492e-04, 1.50525375e-04, 1.54543480e-04,
       1.45359030e-04, 9.06705250e-04, 1.52652419e-04, 1.50174831e-04,
       1.53450115e-04, 1.52476886e-04, 1.52825943e-04, 1.52652419e-04,
       9.15466290e-04, 1.51719377e-04, 1.53536882e-04, 1.52254783e-04,
       1.33384514e-04, 8.60216267e-05, 1.15178853e-04, 8.19658269e-04,
       1.53052171e-04, 1.70755337e-04, 1.61265768e-04, 1.52456711e-04,
       1.54362365e-04, 1.51719377e-04, 9.22266453e-04, 1.55377339e-04,
       1.53299177e-04, 1.55051485e-04, 1.57108727e-04, 1.61685640e-04,
       1.55377339e-04, 9.54440913e-04, 1.64454875e-04, 1.55656855e-04,
       1.60157477e-04, 1.28225262e-04, 1.32861118e-04, 1.25839805e-04,
       7.74375372e-04, 1.34353577e-04, 1.24608847e-04, 1.28486763e-04,
       1.25074319e-04, 1.27040586e-04, 1.22653231e-04, 6.30257829e-04,
       1.28891297e-04, 1.26598397e-04, 1.41938314e-04, 1.41942513e-04,
       1.45620764e-04, 1.43532103e-04, 8.64429930e-04, 1.46744548e-04,
       1.44651689e-04, 1.63277518e-04, 1.60183513e-04, 1.62393811e-04,
       9.56044213e-04, 1.56291785e-04, 1.58452803e-04, 1.55444783e-04,
       1.84568256e-04, 1.82659003e-04, 1.91672660e-04, 1.21846999e-03,
       2.07768101e-04, 2.16898637e-04, 2.34903332e-04, 1.72619524e-04,
       1.67074910e-04, 2.07768101e-04, 1.08635463e-03, 1.64462240e-04,
       2.06095552e-04, 1.68334305e-04, 1.63103141e-04, 1.54913796e-04,
       1.64462240e-04, 9.40616540e-04, 1.52038099e-04, 1.55167030e-04,
       1.50932234e-04, 1.52071529e-04, 1.47310080e-04, 1.52038099e-04,
       9.01795524e-04, 1.49776014e-04, 1.49050219e-04, 1.51549584e-04,
       1.80365827e-04, 1.60126965e-04, 2.16898637e-04, 1.11423777e-03,
       1.58176695e-04, 2.18576362e-04, 1.80093282e-04, 1.66501153e-04,
       1.50148693e-04, 8.98281647e-04, 1.70530851e-04, 1.30907172e-04,
       1.51732499e-04, 1.28461280e-04, 1.65051717e-04, 1.60281397e-04,
       1.70530851e-04, 9.55890837e-04, 1.50840784e-04, 1.62052917e-04,
       1.47133170e-04, 1.36049722e-04, 1.36602841e-04, 1.30907172e-04,
       7.90140516e-04, 1.34275322e-04, 1.24745214e-04, 1.27560244e-04,
       1.56411272e-04, 1.58958290e-04, 1.50840784e-04, 8.94056391e-04,
       1.48586866e-04, 1.42030057e-04, 1.37229121e-04, 1.32409402e-04,
       1.30182769e-04, 7.80872258e-04, 1.32144028e-04, 1.26679969e-04,
       1.28109332e-04, 1.31346758e-04, 1.32172453e-04, 1.27797104e-04,
       1.32144028e-04, 8.12652514e-04, 1.24816180e-04, 1.49337624e-04,
       1.46385125e-04, 1.26579851e-04, 1.23201998e-04, 1.24816180e-04,
       6.13426932e-04, 1.17264046e-04, 1.21564857e-04, 1.27911175e-04,
       1.27076505e-04, 1.26679969e-04, 7.41936719e-04, 1.22129072e-04,
       1.20928112e-04, 1.17211886e-04, 1.28503574e-04, 1.29174148e-04,
       1.22129072e-04, 7.30921977e-04, 1.24410959e-04, 1.12546456e-04,
       1.14157768e-04, 1.31915538e-04, 1.36377993e-04, 1.24410959e-04,
       7.79081938e-04, 1.35375227e-04, 1.22250205e-04, 1.28752017e-04,
       1.27383712e-04, 1.38411366e-04, 1.17264046e-04, 8.25813513e-04,
       1.38016513e-04, 1.47506532e-04, 1.57231345e-04, 1.39514468e-04,
       1.43465775e-04, 1.43631205e-04, 1.35375227e-04, 8.36244613e-04,
       1.39281305e-04, 1.34976633e-04, 1.42111938e-04, 1.44738614e-04,
       1.38016513e-04, 8.51205883e-04, 1.43018087e-04, 1.40472916e-04,
       1.42847814e-04, 1.44019825e-04, 1.42520885e-04, 1.43018087e-04,
       8.51309460e-04, 1.40342414e-04, 1.41292491e-04, 1.40115758e-04,
       1.45722023e-04, 1.44711835e-04, 8.75857754e-04, 1.47174002e-04,
       1.43803418e-04, 1.48403436e-04, 1.46043040e-04, 1.45350127e-04,
       1.45122464e-04, 1.47174002e-04, 8.78754861e-04, 1.45460304e-04,
       1.48567000e-04, 1.47080964e-04, 1.43950729e-04, 1.43687039e-04,
       1.45460304e-04, 8.73465320e-04, 1.46031088e-04, 1.46750843e-04,
       1.47585317e-04, 1.44509019e-04, 1.45303711e-04, 1.46031088e-04,
       8.78987954e-04, 1.46481813e-04, 1.48503145e-04, 1.48159178e-04,
       1.44199168e-04, 1.44517249e-04, 1.43803418e-04, 8.64725163e-04,
       1.43598808e-04, 1.44564606e-04, 1.44041914e-04, 1.43320691e-04,
       1.39671723e-04, 1.46481813e-04, 8.57075034e-04, 1.39779736e-04,
       1.45437089e-04, 1.42383981e-04, 1.53538375e-04, 1.38006478e-04,
       1.49899298e-04, 8.63420101e-04, 1.16987895e-04, 1.61183780e-04,
       1.43804275e-04, 1.45328329e-04, 1.52962114e-04, 1.53052171e-04,
       9.28482879e-04, 1.59032084e-04, 1.59881027e-04, 1.58227154e-04,
       1.76874701e-04, 1.64918353e-04, 1.70755337e-04, 9.87339925e-04,
       1.57635099e-04, 1.59160163e-04, 1.57996272e-04, 1.64972990e-04,
       1.64454875e-04, 1.48586866e-04, 8.98337990e-04, 1.50930674e-04,
       1.33265331e-04, 1.36127255e-04, 1.37176964e-04, 1.34353577e-04,
       1.39779736e-04, 8.27550501e-04, 1.37231936e-04, 1.39366350e-04,
       1.39641937e-04, 1.50319412e-04, 1.45248520e-04, 8.57625103e-04,
       1.44635004e-04, 1.40986382e-04, 1.37506757e-04, 1.38929028e-04,
       1.47925904e-04, 1.42001981e-04, 1.44635004e-04, 8.46315192e-04,
       1.36846190e-04, 1.38385502e-04, 1.36520611e-04, 1.44316829e-04,
       1.47049203e-04, 1.40986382e-04, 8.60794284e-04, 1.46094807e-04,
       1.39985505e-04, 1.42361556e-04, 1.47424771e-04, 1.48200936e-04,
       1.46094807e-04, 8.85738921e-04, 1.48661382e-04, 1.46783308e-04,
       1.48573717e-04, 1.36401953e-04, 1.33521514e-04, 1.36846190e-04,
       8.06418977e-04, 1.32963432e-04, 1.33844104e-04, 1.32841784e-04,
       1.42120836e-04, 1.41966529e-04, 8.51938868e-04, 1.42206422e-04,
       1.41685526e-04, 1.42163072e-04, 1.41796483e-04, 1.42104954e-04,
       1.41013244e-04, 1.42206422e-04, 8.49597040e-04, 1.40626731e-04,
       1.42066835e-04, 1.41578853e-04, 1.41758857e-04, 1.41582467e-04,
       1.41685526e-04, 8.50792190e-04, 1.41783216e-04, 1.41854023e-04,
       1.42128102e-04, 1.39487313e-04, 1.37058859e-04, 1.40626731e-04,
       8.30793430e-04, 1.36002285e-04, 1.39907117e-04, 1.37711125e-04,
       1.41758755e-04, 1.41546121e-04, 1.41783216e-04, 8.49838694e-04,
       1.41475058e-04, 1.41685535e-04, 1.41590010e-04, 1.31018678e-04,
       1.28891297e-04, 1.36002285e-04, 7.93078175e-04, 1.30817184e-04,
       1.34703225e-04, 1.31645506e-04, 1.28254668e-04, 1.26598397e-04,
       1.30817184e-04, 7.84191847e-04, 1.33555938e-04, 1.31941589e-04,
       1.33024072e-04, 1.34912712e-04, 1.37937875e-04, 1.33555938e-04,
       8.29266895e-04, 1.41097194e-04, 1.38623541e-04, 1.43139635e-04,
       1.39624504e-04, 1.42564090e-04, 1.41097194e-04, 8.73427540e-04,
       1.50326719e-04, 1.46762547e-04, 1.53052486e-04, 1.41716634e-04,
       1.42246402e-04, 1.41475058e-04, 8.51415673e-04, 1.42656948e-04,
       1.41334255e-04, 1.41986377e-04, 1.42929999e-04, 1.43150354e-04,
       1.42656948e-04, 8.54945117e-04, 1.41908498e-04, 1.42634061e-04,
       1.41665256e-04, 1.49507404e-04, 1.48054648e-04, 1.50326719e-04,
       8.91492578e-04, 1.45229896e-04, 1.51008989e-04, 1.47364921e-04,
       1.41760857e-04, 1.37134473e-04, 1.32963432e-04, 1.45229896e-04,
       8.32056607e-04, 1.33663935e-04, 1.41304014e-04, 1.44634583e-04,
       1.47362336e-04, 1.41908498e-04, 8.61316965e-04, 1.44799282e-04,
       1.41224618e-04, 1.41387649e-04, 1.48732421e-04, 1.50656976e-04,
       1.44799282e-04, 8.72352499e-04, 1.47027609e-04, 1.40416219e-04,
       1.40719991e-04, 1.48145936e-04, 1.46744548e-04, 1.48661382e-04,
       8.81017164e-04, 1.44448177e-04, 1.47914470e-04, 1.45102652e-04,
       1.55469577e-04, 1.56291785e-04, 1.47027609e-04, 9.07453853e-04,
       1.52856096e-04, 1.45401254e-04, 1.50407533e-04, 1.40070404e-04,
       1.49776014e-04, 1.34275322e-04, 8.61660623e-04, 1.52934713e-04,
       1.37825194e-04, 1.46778976e-04, 1.50470596e-04, 1.43169854e-04,
       1.58176695e-04, 8.43457788e-04, 1.22447475e-04, 1.48810822e-04,
       1.20382345e-04, 1.45125433e-04, 1.48191250e-04, 1.22447475e-04,
       8.06321709e-04, 1.56772612e-04, 1.01262810e-04, 1.32522129e-04,
       1.54605742e-04, 1.58452803e-04, 1.56772612e-04, 9.33457401e-04,
       1.53890895e-04, 1.58232063e-04, 1.51503285e-04, 1.44373290e-04,
       1.45947216e-04, 8.65856946e-04, 1.42072883e-04, 1.45514866e-04,
       1.44908374e-04, 1.43040317e-04, 1.41645705e-04, 1.39886501e-04,
       1.42072883e-04, 8.40621463e-04, 1.37589338e-04, 1.41075688e-04,
       1.38351347e-04, 1.46293958e-04, 1.46091352e-04, 1.45514866e-04,
       8.82601272e-04, 1.48394855e-04, 1.46611823e-04, 1.49694418e-04,
       1.46665712e-04, 1.45169739e-04, 1.48394855e-04, 9.39324118e-04,
       1.60398680e-04, 1.62597667e-04, 1.76097465e-04, 1.37957259e-04,
       1.35124572e-04, 1.37589338e-04, 8.03199185e-04, 1.29198895e-04,
       1.34443438e-04, 1.28885682e-04, 1.30539560e-04, 1.26953777e-04,
       1.29198895e-04, 6.37964462e-04, 1.26106381e-04, 1.25165848e-04,
       1.53422120e-04, 1.79266053e-04, 1.26106381e-04, 1.12248676e-03,
       1.70629840e-04, 1.50440121e-04, 1.68260383e-04, 1.74361860e-04,
       1.43890548e-04, 1.37476980e-04, 1.60398680e-04, 7.81428611e-04,
       1.11632091e-04, 1.35191666e-04, 9.28386452e-05, 1.68849257e-04,
       1.58163954e-04, 1.70629840e-04, 9.67907771e-04, 1.53855229e-04,
       1.61249400e-04, 1.55160092e-04, 1.29700030e-04, 1.34533734e-04,
       1.11632091e-04, 7.61149850e-04, 1.37297172e-04, 1.13577724e-04,
       1.34409100e-04, 1.38898118e-04, 1.40768546e-04, 1.37297172e-04,
       8.41890701e-04, 1.43645327e-04, 1.38401906e-04, 1.42879634e-04,
       1.55120838e-04, 1.49789673e-04, 1.53855229e-04, 8.95651303e-04,
       1.43777921e-04, 1.48926892e-04, 1.44180750e-04, 1.43454497e-04,
       1.44651689e-04, 1.44448177e-04, 1.43645327e-04, 8.64084296e-04,
       1.43745132e-04, 1.44139475e-04, 1.45980863e-04, 1.41596771e-04,
       1.43777921e-04, 8.31122325e-04, 1.35387313e-04, 1.34192987e-04,
       1.30186470e-04, 1.39444343e-04, 1.40342414e-04, 1.35387313e-04,
       8.14729154e-04, 1.37792962e-04, 1.30127273e-04, 1.31634851e-04,
       1.43185060e-04, 1.39281305e-04, 1.43598808e-04, 8.41826921e-04,
       1.36573890e-04, 1.41154511e-04, 1.38033348e-04, 1.22321707e-04,
       1.24608847e-04, 1.16987895e-04, 6.07130750e-04, 1.24033458e-04,
       1.19178842e-04, 1.54356094e-04, 1.61183780e-04, 1.59032084e-04,
       9.77456632e-04, 1.70942917e-04, 1.63429305e-04, 1.68512452e-04,
       1.55475131e-04, 1.57635099e-04, 9.25336288e-04, 1.52327855e-04,
       1.54787671e-04, 1.52405341e-04, 1.52705190e-04, 1.45902052e-04,
       1.41801492e-04, 1.50174831e-04, 1.52327855e-04, 8.96049524e-04,
       1.53895415e-04, 1.51947879e-04, 1.53450115e-04, 1.53536882e-04,
       1.53895415e-04, 9.19014769e-04, 1.53410703e-04, 1.52646567e-04,
       1.52075087e-04, 1.52254783e-04, 1.53299177e-04, 1.53410703e-04,
       9.10314443e-04, 1.53009665e-04, 1.49892774e-04, 1.48447342e-04,
       1.61265768e-04, 1.59881027e-04, 1.59160163e-04, 9.45310682e-04,
       1.55622925e-04, 1.55766648e-04, 1.53614151e-04, 1.55051485e-04,
       1.55656855e-04, 1.53009665e-04, 9.22950618e-04, 1.55260344e-04,
       1.51163480e-04, 1.52808789e-04, 1.60157477e-04, 1.50930674e-04,
       1.55260344e-04, 8.85332841e-04, 1.38571562e-04, 1.43937381e-04,
       1.36475402e-04, 1.28486763e-04, 1.37231936e-04, 1.24033458e-04,
       8.19387390e-04, 1.42481877e-04, 1.38725055e-04, 1.48428301e-04,
       1.55444783e-04, 1.52856096e-04, 1.53890895e-04, 9.19072667e-04,
       1.51545376e-04, 1.52546174e-04, 1.52789342e-04, 2.34903332e-04,
       2.06095552e-04, 2.18576362e-04, 1.48503500e-03, 1.73336651e-04,
       2.21376091e-04, 1.97454775e-04, 2.33292234e-04, 1.68334305e-04,
       1.55167030e-04, 1.73336651e-04, 7.89744180e-04, 1.41984435e-04,
       1.50921760e-04, 1.50932234e-04, 1.49050219e-04, 1.41984435e-04,
       9.06669096e-04, 1.65324454e-04, 1.37077860e-04, 1.62299894e-04,
       1.51549584e-04, 1.52934713e-04, 1.65324454e-04, 9.66844724e-04,
       1.55461790e-04, 1.76410989e-04, 1.65163195e-04, 1.80093282e-04,
       1.48810822e-04, 2.21376091e-04, 1.03309344e-03, 1.32424916e-04,
       2.04028524e-04, 1.46359809e-04, 1.51732499e-04, 1.62052917e-04,
       8.85976527e-04, 1.23174114e-04, 1.53572988e-04, 1.37682777e-04,
       1.57761233e-04, 1.28461280e-04, 1.24745214e-04, 1.23174114e-04,
       6.06721685e-04, 1.16836401e-04, 1.13504677e-04, 1.47133170e-04,
       1.42030057e-04, 1.53572988e-04, 9.18513281e-04, 1.50160356e-04,
       1.61963114e-04, 1.63653596e-04, 1.27560244e-04, 1.37825194e-04,
       1.16836401e-04, 8.01419181e-04, 1.46647261e-04, 1.26502086e-04,
       1.46047995e-04, 1.37229121e-04, 1.33265331e-04, 1.50160356e-04,
       8.59374932e-04, 1.38596739e-04, 1.53861606e-04, 1.46261778e-04,
       1.28109332e-04, 1.20928112e-04, 7.31448644e-04, 1.29582658e-04,
       1.13433510e-04, 1.24181480e-04, 1.15213552e-04, 1.31346758e-04,
       1.49337624e-04, 1.29582658e-04, 8.63288862e-04, 1.66226948e-04,
       1.34070726e-04, 1.52724149e-04, 1.46385125e-04, 1.21564857e-04,
       1.47506532e-04, 1.66226948e-04, 1.09006034e-03, 1.75905771e-04,
       1.64043986e-04, 1.68427123e-04, 1.17211886e-04, 1.12546456e-04,
       1.13433510e-04, 6.20242657e-04, 9.98349989e-05, 9.30744442e-05,
       8.41413631e-05, 1.14157768e-04, 1.22250205e-04, 9.98349989e-05,
       6.61095119e-04, 1.21405869e-04, 9.49839223e-05, 1.08462356e-04,
       1.28752017e-04, 1.34976633e-04, 1.36573890e-04, 1.21405869e-04,
       7.77259082e-04, 1.32247039e-04, 1.23303634e-04, 1.57231345e-04,
       1.40472916e-04, 1.75905771e-04, 9.31196381e-04, 1.43803561e-04,
       1.64563284e-04, 1.49219503e-04, 1.42847814e-04, 1.41292491e-04,
       1.43803561e-04, 8.54971242e-04, 1.40915679e-04, 1.43722128e-04,
       1.42389569e-04, 1.40115758e-04, 1.37792962e-04, 1.40915679e-04,
       8.35134109e-04, 1.37606931e-04, 1.39844366e-04, 1.38858413e-04,
       1.48403436e-04, 1.48567000e-04, 8.95343855e-04, 1.48743551e-04,
       1.48876287e-04, 1.50392204e-04, 1.50361377e-04, 1.46043040e-04,
       1.44564606e-04, 1.48743551e-04, 8.80539831e-04, 1.45211977e-04,
       1.49014898e-04, 1.46961758e-04, 1.47080964e-04, 1.46750843e-04,
       1.48876287e-04, 8.88367075e-04, 1.47982530e-04, 1.49120043e-04,
       1.48556407e-04, 1.47585317e-04, 1.48503145e-04, 1.47982530e-04,
       8.91852269e-04, 1.49679042e-04, 1.48661775e-04, 1.49440459e-04,
       1.48159178e-04, 1.45437089e-04, 1.49679042e-04, 8.84321673e-04,
       1.44960093e-04, 1.49041565e-04, 1.47044705e-04, 1.44041914e-04,
       1.41154511e-04, 1.45211977e-04, 8.56113774e-04, 1.38974491e-04,
       1.45040482e-04, 1.41690399e-04, 1.42383981e-04, 1.39366350e-04,
       1.44960093e-04, 8.52372800e-04, 1.39908459e-04, 1.43893960e-04,
       1.41859956e-04, 1.43804275e-04, 1.19178842e-04, 1.70942917e-04,
       1.42481877e-04, 1.05136324e-03, 1.59190970e-04, 1.60717743e-04,
       1.55046617e-04, 1.58227154e-04, 1.63429305e-04, 1.55622925e-04,
       9.46109705e-04, 1.64080472e-04, 1.50747226e-04, 1.54002622e-04,
       1.57996272e-04, 1.54787671e-04, 1.55766648e-04, 9.09114953e-04,
       1.49400119e-04, 1.47161085e-04, 1.44003157e-04, 1.36127255e-04,
       1.38571562e-04, 1.38596739e-04, 8.30460660e-04, 1.37081812e-04,
       1.42021263e-04, 1.38062029e-04, 1.39641937e-04, 1.38725055e-04,
       1.39908459e-04, 8.37587716e-04, 1.38900213e-04, 1.40251708e-04,
       1.40160344e-04, 1.37506757e-04, 1.38385502e-04, 8.27137425e-04,
       1.35912424e-04, 1.38717877e-04, 1.39270787e-04, 1.37344078e-04,
       1.38929028e-04, 1.39985505e-04, 1.35912424e-04, 8.30176558e-04,
       1.41113251e-04, 1.35046000e-04, 1.39190349e-04, 1.36520611e-04,
       1.33844104e-04, 1.38717877e-04, 8.15028089e-04, 1.32470789e-04,
       1.38522645e-04, 1.34952064e-04, 1.42361556e-04, 1.46783308e-04,
       1.41113251e-04, 8.71492426e-04, 1.49972936e-04, 1.43411721e-04,
       1.47849655e-04, 1.48573717e-04, 1.47914470e-04, 1.49972936e-04,
       8.92487959e-04, 1.47249189e-04, 1.50421073e-04, 1.48356574e-04,
       1.32841784e-04, 1.33663935e-04, 1.32470789e-04, 7.96434467e-04,
       1.34551343e-04, 1.30824106e-04, 1.32082510e-04, 1.42163072e-04,
       1.42066835e-04, 8.54145721e-04, 1.42044254e-04, 1.42387131e-04,
       1.42522657e-04, 1.42961771e-04, 1.41796483e-04, 1.41854023e-04,
       1.42044254e-04, 8.52554978e-04, 1.42072135e-04, 1.42408871e-04,
       1.42379212e-04, 1.41578853e-04, 1.39907117e-04, 1.42387131e-04,
       8.48765118e-04, 1.39880323e-04, 1.42923383e-04, 1.42088311e-04,
       1.42128102e-04, 1.41685535e-04, 1.42072135e-04, 8.52026387e-04,
       1.41466686e-04, 1.42418406e-04, 1.42255524e-04, 1.37711125e-04,
       1.34703225e-04, 1.39880323e-04, 8.18604028e-04, 1.33561761e-04,
       1.38029128e-04, 1.34718467e-04, 1.41590010e-04, 1.41334255e-04,
       1.41466686e-04, 8.43826861e-04, 1.39203659e-04, 1.41119761e-04,
       1.39112490e-04, 1.31645506e-04, 1.31941589e-04, 1.33561761e-04,
       7.95378809e-04, 1.33064620e-04, 1.32979278e-04, 1.32186055e-04,
       1.33024072e-04, 1.38623541e-04, 1.33064620e-04, 8.40793863e-04,
       1.46166413e-04, 1.41206448e-04, 1.48708770e-04, 1.43139635e-04,
       1.46762547e-04, 1.46166413e-04, 9.08498822e-04, 1.56482762e-04,
       1.54627014e-04, 1.61320451e-04, 1.53052486e-04, 1.51008989e-04,
       1.56482762e-04, 9.11385720e-04, 1.48313275e-04, 1.54327098e-04,
       1.48201109e-04, 1.41986377e-04, 1.42634061e-04, 1.39203659e-04,
       8.54841309e-04, 1.44216012e-04, 1.41218266e-04, 1.45582934e-04,
       1.41665256e-04, 1.41224618e-04, 1.44216012e-04, 9.03149159e-04,
       1.54184399e-04, 1.55724924e-04, 1.66133950e-04, 1.47364921e-04,
       1.41304014e-04, 1.34551343e-04, 1.48313275e-04, 8.46223805e-04,
       1.33494613e-04, 1.41195638e-04, 1.41387649e-04, 1.40416219e-04,
       1.54184399e-04, 8.87604980e-04, 1.42041787e-04, 1.60373055e-04,
       1.49201872e-04, 1.40719991e-04, 1.45401254e-04, 1.42041787e-04,
       8.31410290e-04, 1.37506996e-04, 1.37478157e-04, 1.28262105e-04,
       1.45102652e-04, 1.43745132e-04, 1.47249189e-04, 8.72069141e-04,
       1.44763187e-04, 1.46168731e-04, 1.45040249e-04, 1.50407533e-04,
       1.51545376e-04, 1.37506996e-04, 8.41235042e-04, 1.42610457e-04,
       1.27599531e-04, 1.31565149e-04, 1.46778976e-04, 1.55461790e-04,
       1.46647261e-04, 9.22320261e-04, 1.61450699e-04, 1.52930456e-04,
       1.59051080e-04, 1.20382345e-04, 1.01262810e-04, 1.32424916e-04,
       5.83938578e-04, 1.08864851e-04, 1.21003657e-04, 1.32522129e-04,
       1.58232063e-04, 1.08864851e-04, 9.52904284e-04, 1.46442837e-04,
       1.33509713e-04, 1.34127079e-04, 1.39205613e-04, 1.51503285e-04,
       1.52546174e-04, 1.46442837e-04, 9.15531226e-04, 1.52700585e-04,
       1.53561742e-04, 1.58776602e-04, 1.44908374e-04, 1.46611823e-04,
       8.73768187e-04, 1.43035042e-04, 1.48715599e-04, 1.47237229e-04,
       1.43260121e-04, 1.43040317e-04, 1.41075688e-04, 1.43035042e-04,
       8.45439302e-04, 1.38945825e-04, 1.40733509e-04, 1.38608921e-04,
       1.38351347e-04, 1.34443438e-04, 1.38945825e-04, 8.14163559e-04,
       1.33114088e-04, 1.35616344e-04, 1.33692516e-04, 1.49694418e-04,
       1.62597667e-04, 1.48715599e-04, 9.43039375e-04, 1.68125152e-04,
       1.53700117e-04, 1.60206422e-04, 1.76097465e-04, 1.35191666e-04,
       1.68125152e-04, 1.07115974e-03, 1.14519733e-04, 1.62007602e-04,
       1.51351973e-04, 1.63866145e-04, 1.28885682e-04, 1.25165848e-04,
       1.50440121e-04, 1.33114088e-04, 8.21754718e-04, 1.52597422e-04,
       1.31551556e-04, 1.68260383e-04, 1.61249400e-04, 9.44604104e-04,
       1.54079155e-04, 1.62876092e-04, 1.48176560e-04, 1.49962515e-04,
       9.28386452e-05, 1.13577724e-04, 1.14519733e-04, 5.95281824e-04,
       1.36405362e-04, 1.37940361e-04, 1.55160092e-04, 1.48926892e-04,
       1.54079155e-04, 8.95897706e-04, 1.44513190e-04, 1.48275571e-04,
       1.44942806e-04, 1.34409100e-04, 1.38401906e-04, 1.36405362e-04,
       8.32139077e-04, 1.41894684e-04, 1.39765754e-04, 1.41262271e-04,
       1.42879634e-04, 1.44139475e-04, 1.44763187e-04, 1.41894684e-04,
       8.61268878e-04, 1.44600280e-04, 1.42991618e-04, 1.44180750e-04,
       1.34192987e-04, 1.44513190e-04, 8.70298510e-04, 1.25821987e-04,
       1.65146518e-04, 1.56443078e-04, 1.30186470e-04, 1.30127273e-04,
       1.25821987e-04, 6.38433725e-04, 1.26727758e-04, 1.25570237e-04,
       1.31634851e-04, 1.37606931e-04, 1.26727758e-04, 8.38078407e-04,
       1.36970691e-04, 1.50433662e-04, 1.54704514e-04, 1.38033348e-04,
       1.32247039e-04, 1.38974491e-04, 8.02343868e-04, 1.25598459e-04,
       1.37540127e-04, 1.29950404e-04, 1.68512452e-04, 1.59190970e-04,
       1.64080472e-04, 9.27029020e-04, 1.48812937e-04, 1.46189121e-04,
       1.40243068e-04, 1.52405341e-04, 1.51947879e-04, 1.52646567e-04,
       9.03808617e-04, 1.49891795e-04, 1.50064635e-04, 1.46852401e-04,
       1.52705190e-04, 1.49400119e-04, 1.49891795e-04, 8.74499692e-04,
       1.39937188e-04, 1.44607469e-04, 1.37957932e-04, 1.52075087e-04,
       1.49892774e-04, 1.50064635e-04, 8.84329813e-04, 1.46100843e-04,
       1.43988976e-04, 1.42207498e-04, 1.48447342e-04, 1.51163480e-04,
       1.46100843e-04, 8.66804774e-04, 1.46210037e-04, 1.38745008e-04,
       1.36138064e-04, 1.53614151e-04, 1.50747226e-04, 1.47161085e-04,
       8.69435165e-04, 1.43903369e-04, 1.37200062e-04, 1.36809271e-04,
       1.52808789e-04, 1.43937381e-04, 1.46210037e-04, 8.52481869e-04,
       1.35282004e-04, 1.38150142e-04, 1.36093516e-04, 1.36475402e-04,
       1.37081812e-04, 1.35282004e-04, 8.17263722e-04, 1.35591754e-04,
       1.36564705e-04, 1.36268045e-04, 1.48428301e-04, 1.60717743e-04,
       1.38900213e-04, 8.88565592e-04, 1.53165317e-04, 1.42689178e-04,
       1.44664840e-04, 1.52789342e-04, 1.42610457e-04, 1.52700585e-04,
       8.77663318e-04, 1.37406032e-04, 1.48636285e-04, 1.43520617e-04,
       1.97454775e-04, 1.50921760e-04, 1.37077860e-04, 8.23993997e-04,
       1.48173814e-04, 1.90365789e-04, 1.62299894e-04, 1.76410989e-04,
       1.48173814e-04, 1.13934652e-03, 1.64340122e-04, 1.65638831e-04,
       1.56913989e-04, 1.65568880e-04, 1.65163195e-04, 1.61450699e-04,
       1.64340122e-04, 9.52733962e-04, 1.63226853e-04, 1.47976933e-04,
       1.50576160e-04, 2.33292234e-04, 2.04028524e-04, 1.90365789e-04,
       1.65638831e-04, 1.10937375e-03, 1.55755810e-04, 1.60292561e-04,
       1.46359809e-04, 1.21003657e-04, 1.33509713e-04, 1.55755810e-04,
       8.27037434e-04, 1.33649195e-04, 1.36759251e-04, 1.37682777e-04,
       1.13504677e-04, 1.26502086e-04, 8.35214996e-04, 1.59660491e-04,
       1.44442330e-04, 1.53422636e-04, 1.57761233e-04, 1.61963114e-04,
       1.59660491e-04, 9.37127408e-04, 1.57608649e-04, 1.53193822e-04,
       1.46940099e-04, 1.63653596e-04, 1.53861606e-04, 1.57608649e-04,
       9.21431194e-04, 1.48111560e-04, 1.48950442e-04, 1.49245342e-04,
       1.46047995e-04, 1.52930456e-04, 1.44442330e-04, 9.12971329e-04,
       1.63814975e-04, 1.46622695e-04, 1.59112878e-04, 1.46261778e-04,
       1.42021263e-04, 1.48111560e-04, 8.59673751e-04, 1.38206326e-04,
       1.45368772e-04, 1.39704052e-04, 1.74361860e-04, 1.52597422e-04,
       1.62876092e-04, 9.09119575e-04, 1.35170572e-04, 1.45027355e-04,
       1.39086274e-04, 1.24181480e-04, 1.34070726e-04, 7.83918090e-04,
       1.20555348e-04, 1.40705423e-04, 1.27072143e-04, 1.37332971e-04,
       1.15213552e-04, 9.30744442e-05, 1.20555348e-04, 6.03950890e-04,
       7.11837818e-05, 1.15578111e-04, 8.83456527e-05, 1.52724149e-04,
       1.64043986e-04, 1.40705423e-04, 9.18766381e-04, 1.62283275e-04,
       1.44375766e-04, 1.54633781e-04, 1.68427123e-04, 1.64563284e-04,
       1.62283275e-04, 9.62208952e-04, 1.54908939e-04, 1.57768415e-04,
       1.54257917e-04, 8.41413631e-05, 9.49839223e-05, 7.11837818e-05,
       4.11875001e-04, 8.85967603e-05, 7.29691737e-05, 1.08462356e-04,
       1.23303634e-04, 1.25598459e-04, 8.85967603e-05, 6.60483325e-04,
       1.18341268e-04, 9.61808469e-05, 1.49219503e-04, 1.43722128e-04,
       1.54908939e-04, 8.87951850e-04, 1.43505476e-04, 1.51250945e-04,
       1.45344859e-04, 1.42389569e-04, 1.39844366e-04, 1.43505476e-04,
       8.48510323e-04, 1.39500266e-04, 1.42365327e-04, 1.40905319e-04,
       1.38858413e-04, 1.36970691e-04, 1.39500266e-04, 8.31285249e-04,
       1.37780550e-04, 1.39003662e-04, 1.39171668e-04, 1.50392204e-04,
       1.49014898e-04, 8.97758691e-04, 1.50293768e-04, 1.48116168e-04,
       1.50370973e-04, 1.49570679e-04, 1.50361377e-04, 1.49120043e-04,
       1.50293768e-04, 8.98416794e-04, 1.49210613e-04, 1.49636407e-04,
       1.49794586e-04, 1.46961758e-04, 1.45040482e-04, 1.48116168e-04,
       8.78861578e-04, 1.44439420e-04, 1.48029611e-04, 1.46274138e-04,
       1.48556407e-04, 1.48661775e-04, 1.49210613e-04, 8.95671956e-04,
       1.49090646e-04, 1.50188925e-04, 1.49963590e-04, 1.49440459e-04,
       1.49041565e-04, 1.49090646e-04, 8.92363477e-04, 1.47632427e-04,
       1.49108853e-04, 1.48049528e-04, 1.47044705e-04, 1.43893960e-04,
       1.47632427e-04, 8.69960438e-04, 1.42099430e-04, 1.45836084e-04,
       1.43453832e-04, 1.41690399e-04, 1.37540127e-04, 1.44439420e-04,
       8.45343545e-04, 1.36460729e-04, 1.44520644e-04, 1.40692226e-04,
       1.41859956e-04, 1.40251708e-04, 1.42099430e-04, 8.40419127e-04,
       1.37253834e-04, 1.41095773e-04, 1.37858426e-04, 1.54002622e-04,
       1.48812937e-04, 1.43903369e-04, 8.38879167e-04, 1.33410258e-04,
       1.32026703e-04, 1.26723277e-04, 1.55046617e-04, 1.46189121e-04,
       1.53165317e-04, 8.91624256e-04, 1.43755339e-04, 1.47010172e-04,
       1.46457690e-04, 1.44003157e-04, 1.39937188e-04, 1.37200062e-04,
       7.91730351e-04, 1.24112036e-04, 1.27749982e-04, 1.18727926e-04,
       1.38062029e-04, 1.35591754e-04, 1.38206326e-04, 8.23743185e-04,
       1.36332457e-04, 1.37477106e-04, 1.38073513e-04, 1.40160344e-04,
       1.42689178e-04, 1.37253834e-04, 8.35157441e-04, 1.40868091e-04,
       1.36550286e-04, 1.37635708e-04, 1.39270787e-04, 1.38522645e-04,
       8.35828281e-04, 1.37630886e-04, 1.39968740e-04, 1.41760609e-04,
       1.38674614e-04, 1.37344078e-04, 1.35046000e-04, 1.37630886e-04,
       8.08936806e-04, 1.32441463e-04, 1.34682862e-04, 1.31791516e-04,
       1.39190349e-04, 1.43411721e-04, 1.32441463e-04, 8.41010672e-04,
       1.52775466e-04, 1.28539520e-04, 1.44652152e-04, 1.34952064e-04,
       1.30824106e-04, 1.39968740e-04, 8.16716526e-04, 1.29332899e-04,
       1.44073300e-04, 1.37565417e-04, 1.47849655e-04, 1.50421073e-04,
       1.52775466e-04, 9.17189652e-04, 1.50234868e-04, 1.60510303e-04,
       1.55398287e-04, 1.48356574e-04, 1.46168731e-04, 1.50234868e-04,
       8.90016070e-04, 1.46287237e-04, 1.50364223e-04, 1.48604436e-04,
       1.32082510e-04, 1.33494613e-04, 1.29332899e-04, 7.82986340e-04,
       1.31307302e-04, 1.28103358e-04, 1.28665658e-04, 1.42522657e-04,
       1.42408871e-04, 8.59090838e-04, 1.43217603e-04, 1.42835439e-04,
       1.43918945e-04, 1.44187323e-04, 1.42961771e-04, 1.42923383e-04,
       1.43217603e-04, 8.63926094e-04, 1.43815286e-04, 1.45185990e-04,
       1.45822061e-04, 1.42379212e-04, 1.42418406e-04, 1.42835439e-04,
       8.59391069e-04, 1.43244569e-04, 1.44071753e-04, 1.44441690e-04,
       1.42088311e-04, 1.38029128e-04, 1.43815286e-04, 8.48196955e-04,
       1.37513369e-04, 1.44496798e-04, 1.42254064e-04, 1.42255524e-04,
       1.41119761e-04, 1.43244569e-04, 8.60801358e-04, 1.41857740e-04,
       1.46287416e-04, 1.46036349e-04, 1.34718467e-04, 1.32979278e-04,
       1.37513369e-04, 8.03684607e-04, 1.32175289e-04, 1.34948548e-04,
       1.31349656e-04, 1.39112490e-04, 1.41218266e-04, 1.41857740e-04,
       8.53529383e-04, 1.44946178e-04, 1.42706023e-04, 1.43688686e-04,
       1.32186055e-04, 1.41206448e-04, 1.32175289e-04, 8.50468276e-04,
       1.51192660e-04, 1.41855423e-04, 1.51852402e-04, 1.48708770e-04,
       1.54627014e-04, 1.51192660e-04, 9.47764075e-04, 1.64444172e-04,
       1.60729094e-04, 1.68062363e-04, 1.61320451e-04, 1.54327098e-04,
       1.64444172e-04, 9.26923060e-04, 1.46979198e-04, 1.55161881e-04,
       1.44690259e-04, 1.48201109e-04, 1.41195638e-04, 1.31307302e-04,
       1.46979198e-04, 8.32638437e-04, 1.28144383e-04, 1.36810807e-04,
       1.45582934e-04, 1.55724924e-04, 1.44946178e-04, 9.08282663e-04,
       1.62114077e-04, 1.46444320e-04, 1.53470229e-04, 1.66133950e-04,
       1.60373055e-04, 1.62114077e-04, 9.67785773e-04, 1.60488102e-04,
       1.57270835e-04, 1.61405754e-04, 1.49201872e-04, 1.37478157e-04,
       1.60488102e-04, 9.87880695e-04, 1.65427556e-04, 1.79310920e-04,
       1.95974089e-04, 1.28262105e-04, 1.27599531e-04, 1.65427556e-04,
       8.40309470e-04, 1.26711937e-04, 1.65180694e-04, 1.27127648e-04,
       1.45040249e-04, 1.44600280e-04, 1.46287237e-04, 8.72920629e-04,
       1.44597480e-04, 1.46822585e-04, 1.45572797e-04, 1.31565149e-04,
       1.37406032e-04, 1.26711937e-04, 7.98522411e-04, 1.36794572e-04,
       1.30901485e-04, 1.35143237e-04, 1.59051080e-04, 1.63226853e-04,
       1.63814975e-04, 1.08469459e-03, 1.45729587e-04, 1.77934166e-04,
       1.23243243e-04, 1.51694684e-04, 1.34127079e-04, 1.53561742e-04,
       8.08826712e-04, 1.42958386e-04, 1.28428229e-04, 1.27327892e-04,
       1.22423385e-04, 1.58776602e-04, 1.48636285e-04, 1.42958386e-04,
       1.03659043e-03, 1.57187015e-04, 1.19109843e-04, 1.67136959e-04,
       1.42785338e-04, 1.47237229e-04, 1.53700117e-04, 8.95545081e-04,
       1.44359595e-04, 1.55653402e-04, 1.49712828e-04, 1.44881910e-04,
       1.43260121e-04, 1.40733509e-04, 1.44359595e-04, 8.45210704e-04,
       1.37618192e-04, 1.41727040e-04, 1.37512248e-04, 1.38608921e-04,
       1.35616344e-04, 1.37618192e-04, 8.13338845e-04, 1.33512904e-04,
       1.34547598e-04, 1.33434887e-04, 1.33692516e-04, 1.31551556e-04,
       1.35170572e-04, 1.33512904e-04, 8.05415131e-04, 1.37643106e-04,
       1.33844478e-04, 1.60206422e-04, 1.62007602e-04, 1.55653402e-04,
       9.44969797e-04, 1.59508829e-04, 1.52769647e-04, 1.54823895e-04,
       1.51351973e-04, 1.37940361e-04, 1.39765754e-04, 8.72051334e-04,
       1.43378854e-04, 1.54706452e-04, 1.44907939e-04, 1.48176560e-04,
       1.48275571e-04, 8.89232202e-04, 1.48991715e-04, 1.46799180e-04,
       1.49640361e-04, 1.47348815e-04, 1.49962515e-04, 1.45027355e-04,
       1.48991715e-04, 8.98232828e-04, 1.48786949e-04, 1.50366951e-04,
       1.55097345e-04, 1.44942806e-04, 1.65146518e-04, 1.46799180e-04,
       9.51426131e-04, 1.81598635e-04, 1.49171625e-04, 1.63767367e-04,
       1.41262271e-04, 1.42991618e-04, 1.44597480e-04, 1.43378854e-04,
       8.61331291e-04, 1.44295173e-04, 1.44805894e-04, 1.56443078e-04,
       1.25570237e-04, 1.50433662e-04, 1.81598635e-04, 1.12053377e-03,
       1.69789432e-04, 1.70766770e-04, 1.65931952e-04, 1.54704514e-04,
       1.37780550e-04, 1.69789432e-04, 8.99299291e-04, 1.39095708e-04,
       1.55849423e-04, 1.42079664e-04, 1.29950404e-04, 1.18341268e-04,
       1.36460729e-04, 7.64765379e-04, 1.17489318e-04, 1.34942968e-04,
       1.27580692e-04, 1.40243068e-04, 1.33410258e-04, 1.43755339e-04,
       8.33102183e-04, 1.32127580e-04, 1.44180444e-04, 1.39385495e-04,
       1.46852401e-04, 1.44607469e-04, 1.43988976e-04, 8.45649771e-04,
       1.37335222e-04, 1.38637263e-04, 1.34228441e-04, 1.37957932e-04,
       1.24112036e-04, 1.37335222e-04, 7.65904604e-04, 1.14489328e-04,
       1.30505041e-04, 1.21505045e-04, 1.42207498e-04, 1.38745008e-04,
       1.38637263e-04, 8.15023181e-04, 1.32047242e-04, 1.33256850e-04,
       1.30129319e-04, 1.36138064e-04, 1.38150142e-04, 1.32047242e-04,
       8.20870432e-04, 1.38356300e-04, 1.35940852e-04, 1.40237832e-04,
       1.36809271e-04, 1.32026703e-04, 1.27749982e-04, 7.39588773e-04,
       1.25229113e-04, 1.07756002e-04, 1.10017701e-04, 1.36093516e-04,
       1.36564705e-04, 1.38356300e-04, 8.29283551e-04, 1.37972826e-04,
       1.40575433e-04, 1.39720770e-04, 1.36268045e-04, 1.36332457e-04,
       1.37972826e-04, 8.26056230e-04, 1.38297342e-04, 1.38462728e-04,
       1.38722832e-04, 1.44664840e-04, 1.47010172e-04, 1.40868091e-04,
       8.58557812e-04, 1.46175955e-04, 1.38438111e-04, 1.41400643e-04,
       1.43520617e-04, 1.36794572e-04, 1.57187015e-04, 8.82366791e-04,
       1.41584531e-04, 1.56078249e-04, 1.47201809e-04, 1.56913989e-04,
       1.47976933e-04, 8.14698307e-04, 1.30246935e-04, 1.37209175e-04,
       1.22163045e-04, 1.20188229e-04, 1.50576160e-04, 1.45729587e-04,
       1.30246935e-04, 8.24961464e-04, 1.29729753e-04, 1.32174818e-04,
       1.36504210e-04, 1.65568880e-04, 1.60292561e-04, 1.33649195e-04,
       1.37209175e-04, 8.34143694e-04, 1.19570111e-04, 1.17853773e-04,
       1.39205613e-04, 1.36759251e-04, 1.28428229e-04, 1.19570111e-04,
       7.60573846e-04, 1.21511200e-04, 1.15099443e-04, 1.53422636e-04,
       1.53193822e-04, 1.46622695e-04, 8.82749168e-04, 1.45548210e-04,
       1.42403739e-04, 1.41558067e-04, 1.46940099e-04, 1.48950442e-04,
       1.45548210e-04, 8.70618214e-04, 1.46252079e-04, 1.42116929e-04,
       1.40810456e-04, 1.49245342e-04, 1.45368772e-04, 1.46252079e-04,
       8.67047019e-04, 1.42185723e-04, 1.42092444e-04, 1.41902659e-04,
       1.59112878e-04, 1.77934166e-04, 1.42403739e-04, 9.05980843e-04,
       1.54792618e-04, 1.39084925e-04, 1.32652516e-04, 1.39704052e-04,
       1.37477106e-04, 1.42185723e-04, 8.35601979e-04, 1.38100776e-04,
       1.39996161e-04, 1.38138160e-04, 1.63866145e-04, 1.59508829e-04,
       1.54706452e-04, 9.22749124e-04, 1.51453341e-04, 1.46055076e-04,
       1.47159281e-04, 1.39086274e-04, 1.37643106e-04, 1.48786949e-04,
       8.32007658e-04, 1.33433727e-04, 1.43483829e-04, 1.29573775e-04,
       1.27072143e-04, 1.15578111e-04, 7.47267034e-04, 1.33408972e-04,
       1.14671329e-04, 1.24646434e-04, 1.31890046e-04, 1.37332971e-04,
       1.44375766e-04, 1.33408972e-04, 8.45164059e-04, 1.46709291e-04,
       1.38539768e-04, 1.44797292e-04, 8.83456527e-05, 7.29691737e-05,
       9.61808469e-05, 1.17489318e-04, 1.14671329e-04, 6.10259641e-04,
       1.20603321e-04, 1.54633781e-04, 1.57768415e-04, 1.46709291e-04,
       9.10459168e-04, 1.52944506e-04, 1.47651297e-04, 1.50751879e-04,
       1.54257917e-04, 1.51250945e-04, 1.52944506e-04, 9.05642011e-04,
       1.47065950e-04, 1.51497139e-04, 1.48625554e-04, 1.45344859e-04,
       1.42365327e-04, 1.47065950e-04, 8.63421622e-04, 1.40292088e-04,
       1.46073864e-04, 1.42279535e-04, 1.40905319e-04, 1.39003662e-04,
       1.40292088e-04, 8.33461524e-04, 1.37713293e-04, 1.38112150e-04,
       1.37435012e-04, 1.39171668e-04, 1.39095708e-04, 1.37713293e-04,
       8.30176357e-04, 1.38889374e-04, 1.37027137e-04, 1.38279177e-04,
       1.50370973e-04, 1.49636407e-04, 9.03577655e-04, 1.50923581e-04,
       1.50041231e-04, 1.51376624e-04, 1.51228839e-04, 1.49570679e-04,
       1.48029611e-04, 1.50923581e-04, 8.98713516e-04, 1.47926090e-04,
       1.51859989e-04, 1.50403566e-04, 1.49794586e-04, 1.50188925e-04,
       1.50041231e-04, 9.01107689e-04, 1.50449038e-04, 1.50310220e-04,
       1.50323689e-04, 1.46274138e-04, 1.44520644e-04, 1.47926090e-04,
       8.77820776e-04, 1.43938228e-04, 1.48698022e-04, 1.46463654e-04,
       1.49963590e-04, 1.49108853e-04, 1.50449038e-04, 8.98529190e-04,
       1.48664665e-04, 1.50636341e-04, 1.49706704e-04, 1.48049528e-04,
       1.45836084e-04, 1.48664665e-04, 8.78286125e-04, 1.43730094e-04,
       1.47363441e-04, 1.44642313e-04, 1.43453832e-04, 1.41095773e-04,
       1.43730094e-04, 8.45240233e-04, 1.37341369e-04, 1.41824916e-04,
       1.37794250e-04, 1.40692226e-04, 1.34942968e-04, 1.43938228e-04,
       8.42295216e-04, 1.35993641e-04, 1.44461741e-04, 1.42266413e-04,
       1.37858426e-04, 1.36550286e-04, 1.37341369e-04, 8.07248714e-04,
       1.31394167e-04, 1.34371765e-04, 1.29732702e-04, 1.26723277e-04,
       1.32127580e-04, 1.25229113e-04, 7.75865030e-04, 1.33500618e-04,
       1.27708620e-04, 1.30575822e-04, 1.46457690e-04, 1.44180444e-04,
       1.46175955e-04, 8.63954469e-04, 1.41555679e-04, 1.43963866e-04,
       1.41620836e-04, 1.18727926e-04, 1.14489328e-04, 1.07756002e-04,
       7.00461312e-04, 1.35598272e-04, 9.59044579e-05, 1.27985325e-04,
       1.38073513e-04, 1.38297342e-04, 1.38100776e-04, 8.25778105e-04,
       1.38392755e-04, 1.36422785e-04, 1.36490935e-04, 1.37635708e-04,
       1.38438111e-04, 1.31394167e-04, 8.20103197e-04, 1.46387402e-04,
       1.26028489e-04, 1.40219321e-04, 1.41760609e-04, 1.44073300e-04,
       8.77286173e-04, 1.43101591e-04, 1.50344092e-04, 1.51468196e-04,
       1.46538385e-04, 1.38674614e-04, 1.34682862e-04, 1.43101591e-04,
       8.16341391e-04, 1.27353894e-04, 1.42142188e-04, 1.30386243e-04,
       1.31791516e-04, 1.28539520e-04, 1.27353894e-04, 6.30223819e-04,
       1.23103221e-04, 1.19435668e-04, 1.44652152e-04, 1.60510303e-04,
       1.23103221e-04, 1.11024945e-03, 1.60742223e-04, 1.55070313e-04,
       1.75659384e-04, 1.90511856e-04, 1.37565417e-04, 1.28103358e-04,
       1.50344092e-04, 8.41850509e-04, 1.29027326e-04, 1.54332511e-04,
       1.42477804e-04, 1.55398287e-04, 1.50364223e-04, 1.60742223e-04,
       9.32482697e-04, 1.49873727e-04, 1.60969335e-04, 1.55134902e-04,
       1.48604436e-04, 1.46822585e-04, 1.49873727e-04, 8.89109555e-04,
       1.46392319e-04, 1.49558023e-04, 1.47858466e-04, 1.28665658e-04,
       1.28144383e-04, 1.29027326e-04, 7.68856834e-04, 1.26959996e-04,
       1.28802762e-04, 1.27256708e-04, 1.43918945e-04, 1.44071753e-04,
       8.71518069e-04, 1.45377473e-04, 1.45129189e-04, 1.46309809e-04,
       1.46710901e-04, 1.44187323e-04, 1.45185990e-04, 1.45377473e-04,
       8.83141296e-04, 1.48372857e-04, 1.49010469e-04, 1.51007186e-04,
       1.45822061e-04, 1.44496798e-04, 1.48372857e-04, 9.01456138e-04,
       1.49135332e-04, 1.55770676e-04, 1.57858414e-04, 1.44441690e-04,
       1.46287416e-04, 1.45129189e-04, 8.76493753e-04, 1.48220589e-04,
       1.45584598e-04, 1.46830272e-04, 1.42254064e-04, 1.34948548e-04,
       1.49135332e-04, 8.40084426e-04, 1.32397475e-04, 1.45390675e-04,
       1.35958333e-04, 1.46036349e-04, 1.42706023e-04, 1.48220589e-04,
       8.59792764e-04, 1.40100485e-04, 1.43759548e-04, 1.38969770e-04,
       1.31349656e-04, 1.41855423e-04, 1.32397475e-04, 8.44107492e-04,
       1.51052927e-04, 1.39651163e-04, 1.47800848e-04, 1.43688686e-04,
       1.46444320e-04, 1.40100485e-04, 8.64164537e-04, 1.47708718e-04,
       1.40684865e-04, 1.45537463e-04, 1.51852402e-04, 1.60729094e-04,
       1.51052927e-04, 9.66049190e-04, 1.69382766e-04, 1.61789427e-04,
       1.71242574e-04, 1.68062363e-04, 1.55161881e-04, 1.69382766e-04,
       9.29988387e-04, 1.42099520e-04, 1.54832311e-04, 1.40449547e-04,
       1.44690259e-04, 1.36810807e-04, 1.26959996e-04, 1.42099520e-04,
       8.14303718e-04, 1.28241532e-04, 1.35501604e-04, 1.53470229e-04,
       1.57270835e-04, 1.47708718e-04, 8.95638656e-04, 1.49784845e-04,
       1.44564253e-04, 1.42839775e-04, 1.61405754e-04, 1.79310920e-04,
       1.49784845e-04, 1.00434391e-03, 1.81783325e-04, 1.58982874e-04,
       1.73076188e-04, 1.95974089e-04, 1.65180694e-04, 1.81783325e-04,
       1.16713649e-03, 1.35941672e-04, 1.59377833e-04, 1.63392688e-04,
       1.65486190e-04, 1.27127648e-04, 1.30901485e-04, 1.35941672e-04,
       6.71608436e-04, 1.37558533e-04, 1.40079098e-04, 1.45572797e-04,
       1.44295173e-04, 1.46392319e-04, 8.71850271e-04, 1.44259469e-04,
       1.46092869e-04, 1.45237644e-04, 1.35143237e-04, 1.41584531e-04,
       1.37558533e-04, 8.48142281e-04, 1.47313824e-04, 1.41614080e-04,
       1.44928077e-04, 1.23243243e-04, 1.29729753e-04, 6.24143053e-04,
       1.17614150e-04, 1.30846595e-04, 1.22709312e-04, 1.51694684e-04,
       1.54792618e-04, 1.17614150e-04, 8.04394333e-04, 1.30253931e-04,
       1.20248552e-04, 1.29790399e-04, 1.27327892e-04, 1.21511200e-04,
       7.71886779e-04, 1.25023357e-04, 1.22694372e-04, 1.35921125e-04,
       1.39408833e-04, 1.22423385e-04, 1.19109843e-04, 1.25023357e-04,
       6.13314687e-04, 1.22087728e-04, 1.24670375e-04, 1.67136959e-04,
       1.56078249e-04, 8.66861557e-04, 1.53383141e-04, 1.37641731e-04,
       1.12910679e-04, 1.39710799e-04, 1.49712828e-04, 1.52769647e-04,
       8.75605476e-04, 1.43822738e-04, 1.47863447e-04, 1.40226463e-04,
       1.41210352e-04, 1.44881910e-04, 1.41727040e-04, 1.43822738e-04,
       8.45705154e-04, 1.37925667e-04, 1.40045001e-04, 1.37302799e-04,
       1.37512248e-04, 1.34547598e-04, 1.37925667e-04, 8.03569229e-04,
       1.31136561e-04, 1.33135805e-04, 1.29311350e-04, 1.33434887e-04,
       1.33844478e-04, 1.33433727e-04, 1.31136561e-04, 7.88477698e-04,
       1.29257810e-04, 1.27370235e-04, 1.54823895e-04, 1.51453341e-04,
       1.47863447e-04, 8.84553791e-04, 1.44728324e-04, 1.42724676e-04,
       1.42960108e-04, 1.44907939e-04, 1.44805894e-04, 1.46055076e-04,
       1.44259469e-04, 8.68885071e-04, 1.45275128e-04, 1.43581566e-04,
       1.49640361e-04, 1.50366951e-04, 9.02469566e-04, 1.46952879e-04,
       1.53511457e-04, 1.53914954e-04, 1.48082965e-04, 1.47348815e-04,
       1.49171625e-04, 1.46952879e-04, 8.77397759e-04, 1.48640895e-04,
       1.42709169e-04, 1.42574376e-04, 1.55097345e-04, 1.43483829e-04,
       1.53511457e-04, 1.08829284e-03, 1.21213744e-04, 1.72895360e-04,
       1.52554275e-04, 1.89536832e-04, 1.63767367e-04, 1.70766770e-04,
       1.48640895e-04, 9.36879413e-04, 1.60432885e-04, 1.44239454e-04,
       1.49032041e-04, 1.65931952e-04, 1.55849423e-04, 1.60432885e-04,
       8.98461854e-04, 1.40414087e-04, 1.42884888e-04, 1.32948619e-04,
       1.42079664e-04, 1.38889374e-04, 1.40414087e-04, 8.29312493e-04,
       1.38634955e-04, 1.33941627e-04, 1.35352786e-04, 1.27580692e-04,
       1.24646434e-04, 1.20603321e-04, 1.35993641e-04, 7.77250365e-04,
       1.32028221e-04, 1.36398056e-04, 1.39385495e-04, 1.33500618e-04,
       1.41555679e-04, 8.26188526e-04, 1.34150139e-04, 1.39558629e-04,
       1.38037967e-04, 1.34228441e-04, 1.30505041e-04, 1.33256850e-04,
       7.68677253e-04, 1.23326053e-04, 1.26784133e-04, 1.20576734e-04,
       1.21505045e-04, 1.35598272e-04, 1.23326053e-04, 8.35919329e-04,
       1.66612829e-04, 1.29841791e-04, 1.59035339e-04, 1.30129319e-04,
       1.35940852e-04, 1.26784133e-04, 8.07207144e-04, 1.42037144e-04,
       1.31437109e-04, 1.40878588e-04, 1.40237832e-04, 1.40575433e-04,
       1.42037144e-04, 8.65880949e-04, 1.45820761e-04, 1.46881881e-04,
       1.50327898e-04, 1.10017701e-04, 1.27708620e-04, 9.59044579e-05,
       5.72739514e-04, 1.27445271e-04, 1.11663465e-04, 1.39720770e-04,
       1.38462728e-04, 1.45820761e-04, 8.54739008e-04, 1.39845951e-04,
       1.47802783e-04, 1.43086016e-04, 1.38722832e-04, 1.38392755e-04,
       1.39845951e-04, 8.39816100e-04, 1.39272469e-04, 1.41912749e-04,
       1.41669344e-04, 1.41400643e-04, 1.43963866e-04, 1.46387402e-04,
       8.81194715e-04, 1.45019146e-04, 1.54177568e-04, 1.50246090e-04,
       1.47201809e-04, 1.47313824e-04, 1.53383141e-04, 9.16017171e-04,
       1.50970059e-04, 1.59836718e-04, 1.57311621e-04, 1.22163045e-04,
       1.32174818e-04, 8.28055097e-04, 1.28248448e-04, 1.43300060e-04,
       1.48564443e-04, 1.53604282e-04, 1.20188229e-04, 1.17853773e-04,
       1.15099443e-04, 1.22694372e-04, 1.28248448e-04, 7.37593186e-04,
       1.33508920e-04, 1.36504210e-04, 1.30846595e-04, 1.43300060e-04,
       8.28791195e-04, 1.32753436e-04, 1.45137952e-04, 1.40248943e-04,
       1.41558067e-04, 1.42116929e-04, 1.39084925e-04, 8.50028412e-04,
       1.41149012e-04, 1.42307127e-04, 1.43812352e-04, 1.40810456e-04,
       1.42092444e-04, 1.41149012e-04, 8.44789196e-04, 1.39075862e-04,
       1.42508280e-04, 1.39153141e-04, 1.41902659e-04, 1.39996161e-04,
       1.39075862e-04, 8.30807301e-04, 1.36864016e-04, 1.36636976e-04,
       1.36331627e-04, 1.32652516e-04, 1.30253931e-04, 1.42307127e-04,
       8.50039504e-04, 1.43777768e-04, 1.48589468e-04, 1.52458695e-04,
       1.38138160e-04, 1.36422785e-04, 1.36864016e-04, 8.14616274e-04,
       1.34516754e-04, 1.34653224e-04, 1.34021336e-04, 1.42785338e-04,
       1.22087728e-04, 1.37641731e-04, 7.75039525e-04, 1.27871596e-04,
       1.16862828e-04, 1.27790304e-04, 1.47159281e-04, 1.44728324e-04,
       1.45275128e-04, 8.65233952e-04, 1.42794588e-04, 1.42663107e-04,
       1.42613524e-04, 1.29573775e-04, 1.29257810e-04, 1.21213744e-04,
       6.16175368e-04, 1.20899606e-04, 1.15230433e-04, 1.31890046e-04,
       1.38539768e-04, 1.32028221e-04, 8.26945583e-04, 1.43270456e-04,
       1.38312290e-04, 1.42904803e-04, 1.44797292e-04, 1.47651297e-04,
       1.43270456e-04, 8.78346212e-04, 1.48319383e-04, 1.46056431e-04,
       1.48251353e-04, 1.50751879e-04, 1.51497139e-04, 1.48319383e-04,
       8.99136819e-04, 1.49400620e-04, 1.49415911e-04, 1.49751888e-04,
       1.48625554e-04, 1.46073864e-04, 1.49400620e-04, 8.79315731e-04,
       1.43013227e-04, 1.48019085e-04, 1.44183382e-04, 1.42279535e-04,
       1.38112150e-04, 1.43013227e-04, 8.36254815e-04, 1.36102113e-04,
       1.39745760e-04, 1.37002031e-04, 1.37435012e-04, 1.37027137e-04,
       1.36102113e-04, 8.30918260e-04, 1.43475743e-04, 1.34548375e-04,
       1.42329880e-04, 1.38279177e-04, 1.38634955e-04, 1.43475743e-04,
       8.60497210e-04, 1.42619910e-04, 1.49349518e-04, 1.48137907e-04,
       1.51376624e-04, 1.51859989e-04, 9.09669714e-04, 1.51538438e-04,
       1.51908903e-04, 1.51436430e-04, 1.51549331e-04, 1.51228839e-04,
       1.50310220e-04, 1.51538438e-04, 9.05840579e-04, 1.50349542e-04,
       1.51341908e-04, 1.51071631e-04, 1.50403566e-04, 1.48698022e-04,
       1.51908903e-04, 8.99042240e-04, 1.47953748e-04, 1.51163806e-04,
       1.48914196e-04, 1.50323689e-04, 1.50636341e-04, 1.50349542e-04,
       9.01807813e-04, 1.49754277e-04, 1.50825941e-04, 1.49918024e-04,
       1.46463654e-04, 1.44461741e-04, 1.47953748e-04, 8.84028907e-04,
       1.47029082e-04, 1.48521717e-04, 1.49598964e-04, 1.49706704e-04,
       1.47363441e-04, 1.49754277e-04, 8.87379002e-04, 1.45693261e-04,
       1.48289535e-04, 1.46571783e-04, 1.44642313e-04, 1.41824916e-04,
       1.45693261e-04, 8.55647978e-04, 1.38662855e-04, 1.44518821e-04,
       1.40305812e-04, 1.37794250e-04, 1.34371765e-04, 1.38662855e-04,
       7.96894373e-04, 1.24181638e-04, 1.36471299e-04, 1.25412567e-04,
       1.42266413e-04, 1.36398056e-04, 1.38312290e-04, 1.47029082e-04,
       8.55479114e-04, 1.44312418e-04, 1.47160855e-04, 1.29732702e-04,
       1.26028489e-04, 1.24181638e-04, 6.15014265e-04, 1.20402967e-04,
       1.14668469e-04, 1.30575822e-04, 1.34150139e-04, 1.27445271e-04,
       8.04458772e-04, 1.36193538e-04, 1.35460025e-04, 1.40633976e-04,
       1.41620836e-04, 1.39558629e-04, 1.45019146e-04, 8.53530217e-04,
       1.39832975e-04, 1.45311299e-04, 1.42187333e-04, 1.27985325e-04,
       1.66612829e-04, 1.11663465e-04, 1.36193538e-04, 1.00748556e-03,
       1.63910096e-04, 1.49090682e-04, 1.52029629e-04, 1.36490935e-04,
       1.39272469e-04, 1.34516754e-04, 8.14890397e-04, 1.38430603e-04,
       1.32523660e-04, 1.33655975e-04, 1.40219321e-04, 1.54177568e-04,
       1.20402967e-04, 1.04040985e-03, 1.52070887e-04, 1.43124097e-04,
       1.59797299e-04, 1.70617712e-04, 1.51468196e-04, 1.54332511e-04,
       9.19609982e-04, 1.50686937e-04, 1.56527314e-04, 1.51809481e-04,
       1.54785543e-04, 1.46538385e-04, 1.42142188e-04, 1.50686937e-04,
       8.74066300e-04, 1.41252216e-04, 1.49242549e-04, 1.44204025e-04,
       1.30386243e-04, 1.19435668e-04, 1.55070313e-04, 1.41252216e-04,
       8.51689203e-04, 1.65156718e-04, 1.40388045e-04, 1.75659384e-04,
       1.60969335e-04, 9.85612221e-04, 1.57159724e-04, 1.74797325e-04,
       1.57039451e-04, 1.59987002e-04, 1.42477804e-04, 1.28802762e-04,
       1.56527314e-04, 8.54200197e-04, 1.27984252e-04, 1.56638043e-04,
       1.41770022e-04, 1.55134902e-04, 1.49558023e-04, 1.57159724e-04,
       9.12845666e-04, 1.47955687e-04, 1.53332244e-04, 1.49705085e-04,
       1.47858466e-04, 1.46092869e-04, 1.47955687e-04, 8.77327489e-04,
       1.44811439e-04, 1.45993840e-04, 1.44615189e-04, 1.27256708e-04,
       1.28241532e-04, 1.27984252e-04, 7.74264610e-04, 1.30643043e-04,
       1.29232554e-04, 1.30906521e-04, 1.46309809e-04, 1.45584598e-04,
       8.79856494e-04, 1.46965434e-04, 1.46473532e-04, 1.47378216e-04,
       1.47144906e-04, 1.46710901e-04, 1.49010469e-04, 1.46965434e-04,
       8.94449558e-04, 1.51584694e-04, 1.48929184e-04, 1.51248877e-04,
       1.51007186e-04, 1.55770676e-04, 1.51584694e-04, 9.43312606e-04,
       1.62598925e-04, 1.58050192e-04, 1.64300934e-04, 1.57858414e-04,
       1.45390675e-04, 1.62598925e-04, 8.79526147e-04, 1.33809677e-04,
       1.48094982e-04, 1.31773474e-04, 1.46830272e-04, 1.43759548e-04,
       1.46473532e-04, 8.63341309e-04, 1.40401781e-04, 1.44438601e-04,
       1.41437574e-04, 1.35958333e-04, 1.39651163e-04, 1.33809677e-04,
       8.17707267e-04, 1.39494963e-04, 1.33400338e-04, 1.35392793e-04,
       1.38969770e-04, 1.40684865e-04, 1.40401781e-04, 8.57795450e-04,
       1.43311767e-04, 1.46616188e-04, 1.47811079e-04, 1.47800848e-04,
       1.61789427e-04, 1.39494963e-04, 9.50452101e-04, 1.72659156e-04,
       1.54766046e-04, 1.73941660e-04, 1.45537463e-04, 1.44564253e-04,
       1.43311767e-04, 8.76900490e-04, 1.43833826e-04, 1.49078956e-04,
       1.50574225e-04, 1.71242574e-04, 1.54832311e-04, 1.72659156e-04,
       9.35506810e-04, 1.39513819e-04, 1.56997013e-04, 1.40261938e-04,
       1.40449547e-04, 1.35501604e-04, 1.30643043e-04, 1.39513819e-04,
       8.16916593e-04, 1.33442874e-04, 1.37365706e-04, 1.42839775e-04,
       1.58982874e-04, 1.43833826e-04, 9.47286465e-04, 1.69822374e-04,
       1.60981083e-04, 1.70826532e-04, 1.73076188e-04, 1.59377833e-04,
       1.69822374e-04, 7.88470312e-04, 1.38246487e-04, 1.47947429e-04,
       1.63392688e-04, 1.40079098e-04, 1.41614080e-04, 9.10720184e-04,
       1.46409837e-04, 1.68871156e-04, 1.50353324e-04, 1.45237644e-04,
       1.43581566e-04, 1.42794588e-04, 1.44811439e-04, 8.62482916e-04,
       1.42848454e-04, 1.43209227e-04, 1.44928077e-04, 1.50970059e-04,
       1.46409837e-04, 8.97800758e-04, 1.55069930e-04, 1.48902372e-04,
       1.51520482e-04, 1.22709312e-04, 1.20248552e-04, 1.32753436e-04,
       7.74306989e-04, 1.28002593e-04, 1.36441589e-04, 1.34151506e-04,
       1.29790399e-04, 1.43777768e-04, 1.28002593e-04, 8.57708307e-04,
       1.62599465e-04, 1.36464290e-04, 1.57073793e-04, 1.35921125e-04,
       1.24670375e-04, 1.27871596e-04, 8.30373035e-04, 1.53100728e-04,
       1.37415415e-04, 1.51393796e-04, 1.39408833e-04, 1.48564443e-04,
       1.33508920e-04, 1.53100728e-04, 1.06714323e-03, 1.66998131e-04,
       1.59289671e-04, 1.66272508e-04, 1.12910679e-04, 1.16862828e-04,
       5.76633330e-04, 1.14368383e-04, 1.19528305e-04, 1.12963135e-04,
       1.39710799e-04, 1.59836718e-04, 1.14368383e-04, 9.87899379e-04,
       1.49546070e-04, 1.31365074e-04, 1.44257196e-04, 1.48815138e-04,
       1.40226463e-04, 1.40045001e-04, 8.30661332e-04, 1.39089979e-04,
       1.37869203e-04, 1.36195687e-04, 1.37235001e-04, 1.41210352e-04,
       1.42724676e-04, 1.39089979e-04, 8.43452851e-04, 1.41150204e-04,
       1.39365869e-04, 1.39911771e-04, 1.37302799e-04, 1.33135805e-04,
       1.37869203e-04, 7.91625245e-04, 1.26612759e-04, 1.31897064e-04,
       1.24807615e-04, 1.29311350e-04, 1.27370235e-04, 1.20899606e-04,
       1.26612759e-04, 7.40546257e-04, 1.16290135e-04, 1.20062172e-04,
       1.42960108e-04, 1.42663107e-04, 1.41150204e-04, 8.51311115e-04,
       1.41523924e-04, 1.41171525e-04, 1.41842246e-04, 1.53914954e-04,
       1.72895360e-04, 9.38983176e-04, 1.48092635e-04, 1.69588406e-04,
       1.45988229e-04, 1.48503593e-04, 1.48082965e-04, 1.42709169e-04,
       1.48092635e-04, 8.64734010e-04, 1.41070878e-04, 1.43213163e-04,
       1.41565201e-04, 1.42574376e-04, 1.44239454e-04, 1.41070878e-04,
       8.35825259e-04, 1.41228315e-04, 1.34109939e-04, 1.32602298e-04,
       1.52554275e-04, 1.15230433e-04, 1.16290135e-04, 7.75490406e-04,
       1.54222454e-04, 1.18292316e-04, 1.18900794e-04, 1.49032041e-04,
       1.42884888e-04, 1.41228315e-04, 7.48523487e-04, 1.12641096e-04,
       1.12588606e-04, 9.01485407e-05, 1.32948619e-04, 1.33941627e-04,
       1.12641096e-04, 7.40354559e-04, 1.30571501e-04, 1.06657160e-04,
       1.23594557e-04, 1.35352786e-04, 1.42619910e-04, 1.30571501e-04,
       8.27910992e-04, 1.46401473e-04, 1.32201237e-04, 1.40764086e-04,
       1.38037967e-04, 1.35460025e-04, 1.39832975e-04, 8.25741378e-04,
       1.36085817e-04, 1.38746906e-04, 1.37577689e-04, 1.20576734e-04,
       1.29841791e-04, 1.31437109e-04, 8.14406431e-04, 1.46948403e-04,
       1.39678078e-04, 1.45924316e-04, 1.59035339e-04, 1.63910096e-04,
       1.46948403e-04, 9.27993835e-04, 1.54311688e-04, 1.50649890e-04,
       1.53138419e-04, 1.40878588e-04, 1.46881881e-04, 1.39678078e-04,
       8.66635002e-04, 1.50461885e-04, 1.41977029e-04, 1.46757541e-04,
       1.50327898e-04, 1.47802783e-04, 1.50461885e-04, 8.96209792e-04,
       1.48098605e-04, 1.49678393e-04, 1.49840228e-04, 1.43086016e-04,
       1.41912749e-04, 1.48098605e-04, 8.68720721e-04, 1.43200054e-04,
       1.48074291e-04, 1.44349007e-04, 1.41669344e-04, 1.38430603e-04,
       1.43200054e-04, 8.50343318e-04, 1.39347674e-04, 1.44154641e-04,
       1.43541002e-04, 1.50246090e-04, 1.45311299e-04, 1.52070887e-04,
       8.82350778e-04, 1.42878254e-04, 1.48051045e-04, 1.43793204e-04,
       1.90511856e-04, 1.65156718e-04, 1.74797325e-04, 9.78327903e-04,
       1.44502472e-04, 1.54149624e-04, 1.49209909e-04, 1.65486190e-04,
       1.38246487e-04, 1.68871156e-04, 9.70143013e-04, 1.66867154e-04,
       1.52718162e-04, 1.77953863e-04, 1.57311621e-04, 1.55069930e-04,
       1.49546070e-04, 9.10604051e-04, 1.57371878e-04, 1.40618527e-04,
       1.50686025e-04, 1.53604282e-04, 1.45137952e-04, 1.66998131e-04,
       9.68427101e-04, 1.50827006e-04, 1.79782262e-04, 1.72077467e-04,
       1.40248943e-04, 1.36441589e-04, 1.50827006e-04, 8.76498281e-04,
       1.39261659e-04, 1.58738539e-04, 1.50980546e-04, 1.43812352e-04,
       1.42508280e-04, 1.48589468e-04, 8.94868172e-04, 1.43617203e-04,
       1.60004531e-04, 1.56336338e-04, 1.39153141e-04, 1.36636976e-04,
       1.43617203e-04, 8.47157842e-04, 1.38014175e-04, 1.46411604e-04,
       1.43324742e-04, 1.36331627e-04, 1.34653224e-04, 1.38014175e-04,
       8.20635325e-04, 1.35663594e-04, 1.38322441e-04, 1.37650264e-04,
       1.52458695e-04, 1.62599465e-04, 1.60004531e-04, 1.13836647e-03,
       1.66940779e-04, 1.67156006e-04, 1.63223299e-04, 1.65983692e-04,
       1.34021336e-04, 1.32523660e-04, 1.35663594e-04, 8.08696333e-04,
       1.31971157e-04, 1.38355673e-04, 1.36160912e-04, 1.27790304e-04,
       1.37415415e-04, 1.19528305e-04, 7.97750258e-04, 1.44482838e-04,
       1.26601987e-04, 1.41931409e-04, 1.42613524e-04, 1.42848454e-04,
       1.41523924e-04, 8.52925416e-04, 1.42082316e-04, 1.41766868e-04,
       1.42090330e-04, 1.89536832e-04, 1.69588406e-04, 1.54222454e-04,
       9.14763753e-04, 1.42136110e-04, 1.25708934e-04, 1.33571017e-04,
       1.42904803e-04, 1.46056431e-04, 1.44312418e-04, 8.77903221e-04,
       1.47882254e-04, 1.48164553e-04, 1.48582761e-04, 1.48251353e-04,
       1.49415911e-04, 1.47882254e-04, 8.92872394e-04, 1.49747779e-04,
       1.48437065e-04, 1.49138032e-04, 1.49751888e-04, 1.48019085e-04,
       1.49747779e-04, 8.89249116e-04, 1.46080761e-04, 1.48791909e-04,
       1.46857694e-04, 1.44183382e-04, 1.39745760e-04, 1.46080761e-04,
       8.44377071e-04, 1.34163223e-04, 1.43841931e-04, 1.36362014e-04,
       1.37002031e-04, 1.34548375e-04, 1.34163223e-04, 7.73554030e-04,
       1.20719823e-04, 1.29055161e-04, 1.18065417e-04, 1.42329880e-04,
       1.49349518e-04, 1.20719823e-04, 8.22826141e-04, 1.60823332e-04,
       1.08259858e-04, 1.41343730e-04, 1.48137907e-04, 1.46401473e-04,
       1.60823332e-04, 9.39341138e-04, 1.49392760e-04, 1.72139902e-04,
       1.62445764e-04, 1.51436430e-04, 1.51163806e-04, 9.09603840e-04,
       1.51526518e-04, 1.50998070e-04, 1.52111596e-04, 1.52367420e-04,
       1.51549331e-04, 1.51341908e-04, 1.51526518e-04, 9.08836029e-04,
       1.51661836e-04, 1.51206848e-04, 1.51549588e-04, 1.51071631e-04,
       1.50825941e-04, 1.51661836e-04, 9.06342841e-04, 1.50402177e-04,
       1.51697613e-04, 1.50683644e-04, 1.48914196e-04, 1.48521717e-04,
       1.50998070e-04, 9.07044989e-04, 1.51846801e-04, 1.52761497e-04,
       1.54002708e-04, 1.49918024e-04, 1.48289535e-04, 1.50402177e-04,
       8.92033050e-04, 1.46702185e-04, 1.49396316e-04, 1.47324813e-04,
       1.49598964e-04, 1.47160855e-04, 1.48164553e-04, 1.51846801e-04,
       8.99873024e-04, 1.50928856e-04, 1.52172995e-04, 1.46571783e-04,
       1.44518821e-04, 1.46702185e-04, 8.69316729e-04, 1.43059451e-04,
       1.45027131e-04, 1.43437358e-04, 1.40305812e-04, 1.36471299e-04,
       1.43059451e-04, 8.37019427e-04, 1.36374850e-04, 1.41829052e-04,
       1.38978964e-04, 1.25412567e-04, 1.14668469e-04, 1.43124097e-04,
       1.36374850e-04, 8.07663293e-04, 1.52788328e-04, 1.35294982e-04,
       1.40633976e-04, 1.49090682e-04, 1.36085817e-04, 8.50327873e-04,
       1.46562711e-04, 1.37967250e-04, 1.39987438e-04, 1.42187333e-04,
       1.38746906e-04, 1.42878254e-04, 8.38267190e-04, 1.37518338e-04,
       1.39427924e-04, 1.37508435e-04, 1.52029629e-04, 1.54311688e-04,
       1.46562711e-04, 9.31091175e-04, 1.58843043e-04, 1.54672915e-04,
       1.64671188e-04, 1.33655975e-04, 1.39347674e-04, 1.31971157e-04,
       8.25735079e-04, 1.42304187e-04, 1.36907377e-04, 1.41548709e-04,
       1.59797299e-04, 1.48051045e-04, 8.99790743e-04, 1.58845135e-04,
       1.43905647e-04, 1.46192425e-04, 1.42999191e-04, 1.51809481e-04,
       1.49242549e-04, 9.01156886e-04, 1.53004815e-04, 1.47374368e-04,
       1.51394594e-04, 1.48331079e-04, 1.54785543e-04, 1.56638043e-04,
       1.53004815e-04, 9.29048896e-04, 1.56128131e-04, 1.53610774e-04,
       1.54881591e-04, 1.44204025e-04, 1.40388045e-04, 1.44502472e-04,
       1.47374368e-04, 8.71494833e-04, 1.47985003e-04, 1.47040920e-04,
       1.57039451e-04, 1.53332244e-04, 9.13537923e-04, 1.55394144e-04,
       1.49408281e-04, 1.50321230e-04, 1.48042574e-04, 1.59987002e-04,
       1.54149624e-04, 1.55394144e-04, 9.17941337e-04, 1.49204606e-04,
       1.49779061e-04, 1.49426901e-04, 1.41770022e-04, 1.29232554e-04,
       1.56128131e-04, 8.59369918e-04, 1.31738766e-04, 1.56176171e-04,
       1.44324273e-04, 1.49705085e-04, 1.45993840e-04, 1.49408281e-04,
       8.74770683e-04, 1.43478422e-04, 1.44201834e-04, 1.41983220e-04,
       1.44615189e-04, 1.43209227e-04, 1.42082316e-04, 1.43478422e-04,
       8.57146934e-04, 1.41885818e-04, 1.41875962e-04, 1.30906521e-04,
       1.33442874e-04, 1.31738766e-04, 8.08270202e-04, 1.36564189e-04,
       1.36664391e-04, 1.38953461e-04, 1.47378216e-04, 1.44438601e-04,
       8.82966016e-04, 1.47878794e-04, 1.44934591e-04, 1.49170201e-04,
       1.49165613e-04, 1.47144906e-04, 1.48929184e-04, 1.47878794e-04,
       8.94429939e-04, 1.50719346e-04, 1.49350718e-04, 1.50406991e-04,
       1.51248877e-04, 1.58050192e-04, 1.50719346e-04, 9.38753713e-04,
       1.63609273e-04, 1.54518707e-04, 1.60607319e-04, 1.64300934e-04,
       1.48094982e-04, 1.63609273e-04, 8.85349669e-04, 1.30678231e-04,
       1.47695670e-04, 1.30970580e-04, 1.31773474e-04, 1.33400338e-04,
       1.30678231e-04, 7.97336866e-04, 1.35201448e-04, 1.31693511e-04,
       1.34589864e-04, 1.41437574e-04, 1.46616188e-04, 1.44934591e-04,
       8.92685377e-04, 1.52507573e-04, 1.52397541e-04, 1.54791909e-04,
       1.35392793e-04, 1.54766046e-04, 1.35201448e-04, 9.36853918e-04,
       1.76293567e-04, 1.56740599e-04, 1.78459465e-04, 1.47811079e-04,
       1.49078956e-04, 1.52507573e-04, 9.09074688e-04, 1.55137398e-04,
       1.51588868e-04, 1.52950815e-04, 1.73941660e-04, 1.56997013e-04,
       1.76293567e-04, 9.52624902e-04, 1.41580790e-04, 1.60790001e-04,
       1.43021871e-04, 1.50574225e-04, 1.60981083e-04, 1.55137398e-04,
       9.53020086e-04, 1.71031645e-04, 1.54904174e-04, 1.60391562e-04,
       1.40261938e-04, 1.37365706e-04, 1.36564189e-04, 1.41580790e-04,
       8.35557515e-04, 1.38832262e-04, 1.40952631e-04, 1.70826532e-04,
       1.47947429e-04, 1.66867154e-04, 1.71031645e-04, 1.19667112e-03,
       1.96975992e-04, 1.65498088e-04, 1.77524281e-04, 1.50353324e-04,
       1.48902372e-04, 1.52718162e-04, 9.02271458e-04, 1.49432546e-04,
       1.51349859e-04, 1.49515195e-04, 1.51520482e-04, 1.57371878e-04,
       1.49432546e-04, 9.18340735e-04, 1.59591245e-04, 1.48058640e-04,
       1.52365943e-04, 1.34151506e-04, 1.36464290e-04, 1.39261659e-04,
       8.24203485e-04, 1.35968513e-04, 1.41981723e-04, 1.36375794e-04,
       1.57073793e-04, 1.66940779e-04, 1.35968513e-04, 9.15481180e-04,
       1.59417804e-04, 1.41248992e-04, 1.54831299e-04, 1.51393796e-04,
       1.59289671e-04, 1.44482838e-04, 9.04956980e-04, 1.55743091e-04,
       1.45341604e-04, 1.48705980e-04, 1.12963135e-04, 1.31365074e-04,
       1.26601987e-04, 7.90520423e-04, 1.39659905e-04, 1.42637171e-04,
       1.37293151e-04, 1.44257196e-04, 1.40618527e-04, 6.12525763e-04,
       1.15869123e-04, 1.19464308e-04, 9.23166090e-05, 1.36195687e-04,
       1.31897064e-04, 7.94331156e-04, 1.35076114e-04, 1.29233261e-04,
       1.31736631e-04, 1.30192400e-04, 1.37235001e-04, 1.39365869e-04,
       1.35076114e-04, 8.21932033e-04, 1.38978916e-04, 1.34752100e-04,
       1.36524034e-04, 1.39911771e-04, 1.41171525e-04, 1.38978916e-04,
       8.39619417e-04, 1.41310804e-04, 1.38587134e-04, 1.39659267e-04,
       1.24807615e-04, 1.20062172e-04, 1.18292316e-04, 1.29233261e-04,
       7.37490144e-04, 1.19449639e-04, 1.25645140e-04, 1.41842246e-04,
       1.41766868e-04, 1.41310804e-04, 8.45405536e-04, 1.41018039e-04,
       1.39842483e-04, 1.39625096e-04, 1.45988229e-04, 1.43213163e-04,
       8.52222303e-04, 1.43129783e-04, 1.39463809e-04, 1.40659113e-04,
       1.39768206e-04, 1.48503593e-04, 1.42136110e-04, 1.43129783e-04,
       8.49049045e-04, 1.35386318e-04, 1.40634611e-04, 1.39258629e-04,
       1.41565201e-04, 1.34109939e-04, 1.39463809e-04, 8.23415085e-04,
       1.28187538e-04, 1.41954804e-04, 1.38133795e-04, 1.32602298e-04,
       1.12588606e-04, 1.28187538e-04, 7.28089248e-04, 9.73563908e-05,
       1.34085935e-04, 1.23268480e-04, 1.18900794e-04, 1.25708934e-04,
       1.19449639e-04, 7.25134698e-04, 1.23422326e-04, 1.20244229e-04,
       1.17408776e-04, 9.01485407e-05, 1.06657160e-04, 9.73563908e-05,
       5.26213004e-04, 1.18869597e-04, 1.13181316e-04, 1.23594557e-04,
       1.32201237e-04, 1.18869597e-04, 7.63965063e-04, 1.35673184e-04,
       1.22714791e-04, 1.30911697e-04, 1.40764086e-04, 1.49392760e-04,
       1.35673184e-04, 8.63338816e-04, 1.52170976e-04, 1.38734346e-04,
       1.46603464e-04, 1.70617712e-04, 1.52788328e-04, 1.58845135e-04,
       9.15372317e-04, 1.40608273e-04, 1.46460173e-04, 1.46052695e-04,
       1.37577689e-04, 1.37967250e-04, 1.37518338e-04, 8.06671738e-04,
       1.33078992e-04, 1.32679188e-04, 1.27850282e-04, 1.45924316e-04,
       1.50649890e-04, 1.41977029e-04, 8.84030323e-04, 1.50983632e-04,
       1.45107214e-04, 1.49388243e-04, 1.53138419e-04, 1.58843043e-04,
       1.50983632e-04, 9.22842912e-04, 1.57045296e-04, 1.51237741e-04,
       1.51594781e-04, 1.46757541e-04, 1.49678393e-04, 1.45107214e-04,
       8.89691319e-04, 1.48142029e-04, 1.49946089e-04, 1.50060053e-04,
       1.49840228e-04, 1.48074291e-04, 1.48142029e-04, 9.08556785e-04,
       1.52403709e-04, 1.52034455e-04, 1.58062072e-04, 1.44349007e-04,
       1.44154641e-04, 1.52403709e-04, 8.93820892e-04, 1.46439341e-04,
       1.56122099e-04, 1.50352096e-04, 1.43541002e-04, 1.42304187e-04,
       1.46439341e-04, 8.76867341e-04, 1.45625483e-04, 1.49267186e-04,
       1.49690143e-04, 1.43793204e-04, 1.39427924e-04, 1.43905647e-04,
       8.40609087e-04, 1.36926825e-04, 1.39584515e-04, 1.36970973e-04,
       1.49209909e-04, 1.47985003e-04, 1.49204606e-04, 8.88514657e-04,
       1.46912588e-04, 1.48134832e-04, 1.47067719e-04, 1.77953863e-04,
       1.96975992e-04, 1.51349859e-04, 9.87427218e-04, 1.68373674e-04,
       1.47386073e-04, 1.45387758e-04, 1.50686025e-04, 1.59591245e-04,
       1.15869123e-04, 1.04476245e-03, 1.61003699e-04, 1.29928990e-04,
       1.63171915e-04, 1.64511456e-04, 1.66272508e-04, 1.79782262e-04,
       1.55743091e-04, 1.02408128e-03, 1.89230874e-04, 1.56537257e-04,
       1.76515286e-04, 1.72077467e-04, 1.58738539e-04, 1.89230874e-04,
       1.06052628e-03, 1.65519275e-04, 1.92666398e-04, 1.82293728e-04,
       1.50980546e-04, 1.41981723e-04, 1.65519275e-04, 9.24959992e-04,
       1.47895403e-04, 1.63604047e-04, 1.54978998e-04, 1.56336338e-04,
       1.46411604e-04, 1.67156006e-04, 9.15010181e-04, 1.45502715e-04,
       1.55666038e-04, 1.43937480e-04, 1.43324742e-04, 1.38322441e-04,
       1.45502715e-04, 8.46846965e-04, 1.39037932e-04, 1.41060808e-04,
       1.39598327e-04, 1.37650264e-04, 1.38355673e-04, 1.39037932e-04,
       8.29652985e-04, 1.38391779e-04, 1.38784450e-04, 1.37432887e-04,
       1.63223299e-04, 1.59417804e-04, 9.58510289e-04, 1.57805711e-04,
       1.66303587e-04, 1.49728258e-04, 1.62031629e-04, 1.36160912e-04,
       1.36907377e-04, 1.38391779e-04, 8.33335382e-04, 1.41395736e-04,
       1.39111043e-04, 1.41368535e-04, 1.41931409e-04, 1.45341604e-04,
       1.39659905e-04, 8.53240058e-04, 1.43775952e-04, 1.40912669e-04,
       1.41618520e-04, 1.48815138e-04, 1.42637171e-04, 1.19464308e-04,
       7.81010104e-04, 1.32304230e-04, 1.09385684e-04, 1.28403572e-04,
       1.42090330e-04, 1.41885818e-04, 1.41018039e-04, 8.42369469e-04,
       1.40170848e-04, 1.38923557e-04, 1.38280877e-04, 1.33571017e-04,
       1.35386318e-04, 1.23422326e-04, 8.05540791e-04, 1.53586142e-04,
       1.15613237e-04, 1.43961751e-04, 1.48582761e-04, 1.48437065e-04,
       1.50928856e-04, 9.01431791e-04, 1.50004721e-04, 1.52128412e-04,
       1.51349975e-04, 1.49138032e-04, 1.48791909e-04, 1.50004721e-04,
       8.97245898e-04, 1.49030167e-04, 1.50454750e-04, 1.49826319e-04,
       1.46857694e-04, 1.43841931e-04, 1.49030167e-04, 8.76406256e-04,
       1.42219389e-04, 1.49126045e-04, 1.45331030e-04, 1.36362014e-04,
       1.29055161e-04, 1.42219389e-04, 7.96215425e-04, 1.20685957e-04,
       1.41059742e-04, 1.26833162e-04, 1.18065417e-04, 1.08259858e-04,
       1.20685957e-04, 5.70407649e-04, 1.10895600e-04, 1.12500817e-04,
       1.41343730e-04, 1.72139902e-04, 1.10895600e-04, 1.04253858e-03,
       1.64374216e-04, 1.36452929e-04, 1.58038869e-04, 1.59293335e-04,
       1.62445764e-04, 1.52170976e-04, 1.64374216e-04, 9.37010545e-04,
       1.49197136e-04, 1.56862372e-04, 1.51960080e-04, 1.52111596e-04,
       1.52761497e-04, 9.19797471e-04, 1.52760770e-04, 1.54449561e-04,
       1.54376468e-04, 1.53337579e-04, 1.52367420e-04, 1.51206848e-04,
       1.52760770e-04, 9.09323239e-04, 1.50487756e-04, 1.51806444e-04,
       1.50694001e-04, 1.51549588e-04, 1.51697613e-04, 1.50487756e-04,
       9.05019706e-04, 1.50986568e-04, 1.49973697e-04, 1.50324485e-04,
       1.50683644e-04, 1.49396316e-04, 1.50986568e-04, 8.98321839e-04,
       1.48387544e-04, 1.50089731e-04, 1.48778035e-04, 1.54002708e-04,
       1.52172995e-04, 1.52128412e-04, 1.54449561e-04, 9.19023581e-04,
       1.52889235e-04, 1.53380671e-04, 1.47324813e-04, 1.45027131e-04,
       1.48387544e-04, 8.74948205e-04, 1.43972266e-04, 1.46177023e-04,
       1.44059428e-04, 1.43437358e-04, 1.41829052e-04, 1.43972266e-04,
       8.57270538e-04, 1.42579968e-04, 1.42617942e-04, 1.42833951e-04,
       1.38978964e-04, 1.35294982e-04, 1.40608273e-04, 1.42579968e-04,
       8.45272080e-04, 1.44761089e-04, 1.43048803e-04, 1.39987438e-04,
       1.54672915e-04, 1.33078992e-04, 7.47312786e-04, 1.26772293e-04,
       1.06896662e-04, 8.59044849e-05, 1.37508435e-04, 1.32679188e-04,
       1.36926825e-04, 7.98245221e-04, 1.28667447e-04, 1.32946729e-04,
       1.29516598e-04, 1.64671188e-04, 1.57045296e-04, 1.26772293e-04,
       9.96084934e-04, 1.52375393e-04, 1.04790190e-04, 1.52442953e-04,
       1.37987621e-04, 1.41548709e-04, 1.45625483e-04, 1.41395736e-04,
       8.70198559e-04, 1.47786195e-04, 1.45764376e-04, 1.48078061e-04,
       1.46192425e-04, 1.46460173e-04, 8.56169364e-04, 1.42211297e-04,
       1.42872743e-04, 1.39413212e-04, 1.39019513e-04, 1.42999191e-04,
       1.39584515e-04, 1.42211297e-04, 8.37731500e-04, 1.37301729e-04,
       1.38564830e-04, 1.37069938e-04, 1.51394594e-04, 1.53610774e-04,
       9.06991015e-04, 1.49255953e-04, 1.53444989e-04, 1.50628780e-04,
       1.48655924e-04, 1.48331079e-04, 1.47040920e-04, 1.46912588e-04,
       1.49255953e-04, 8.86792171e-04, 1.47099213e-04, 1.48152418e-04,
       1.54881591e-04, 1.56176171e-04, 1.53444989e-04, 9.30403038e-04,
       1.56358974e-04, 1.53960954e-04, 1.55580359e-04, 1.50321230e-04,
       1.49779061e-04, 8.82357429e-04, 1.46219130e-04, 1.48224797e-04,
       1.45180524e-04, 1.42632688e-04, 1.48042574e-04, 1.44201834e-04,
       1.46219130e-04, 8.58466138e-04, 1.40052596e-04, 1.41137899e-04,
       1.38812104e-04, 1.49426901e-04, 1.48134832e-04, 1.48224797e-04,
       8.91331593e-04, 1.48235774e-04, 1.48003121e-04, 1.49306168e-04,
       1.44324273e-04, 1.36664391e-04, 1.56358974e-04, 8.84410490e-04,
       1.41388728e-04, 1.56492243e-04, 1.49181880e-04, 1.41983220e-04,
       1.41875962e-04, 1.40170848e-04, 1.40052596e-04, 8.39793393e-04,
       1.37968138e-04, 1.37742629e-04, 1.38953461e-04, 1.38832262e-04,
       1.41388728e-04, 8.52491301e-04, 1.41881138e-04, 1.45411051e-04,
       1.46024661e-04, 1.49170201e-04, 1.49350718e-04, 8.99006706e-04,
       1.49702909e-04, 1.51674740e-04, 1.48658412e-04, 1.50449726e-04,
       1.49165613e-04, 1.52397541e-04, 1.49702909e-04, 9.16043016e-04,
       1.55741588e-04, 1.53114307e-04, 1.55921058e-04, 1.50406991e-04,
       1.54518707e-04, 1.51674740e-04, 9.20241060e-04, 1.57481460e-04,
       1.52232078e-04, 1.53927083e-04, 1.60607319e-04, 1.47695670e-04,
       1.57481460e-04, 8.80185922e-04, 1.33223138e-04, 1.46262504e-04,
       1.34915830e-04, 1.30970580e-04, 1.31693511e-04, 1.33223138e-04,
       7.90968802e-04, 1.32819451e-04, 1.31694370e-04, 1.30567752e-04,
       1.34589864e-04, 1.56740599e-04, 1.32819451e-04, 9.39102342e-04,
       1.80182910e-04, 1.54778396e-04, 1.79991120e-04, 1.54791909e-04,
       1.51588868e-04, 1.55741588e-04, 9.02621567e-04, 1.49747748e-04,
       1.46771127e-04, 1.43980328e-04, 1.78459465e-04, 1.60790001e-04,
       1.80182910e-04, 9.76634855e-04, 1.45285802e-04, 1.64572160e-04,
       1.47344516e-04, 1.52950815e-04, 1.54904174e-04, 1.49747748e-04,
       9.00429499e-04, 1.50814037e-04, 1.46449898e-04, 1.45562828e-04,
       1.43021871e-04, 1.40952631e-04, 1.41881138e-04, 1.45285802e-04,
       8.63998560e-04, 1.45760847e-04, 1.47096270e-04, 1.60391562e-04,
       1.65498088e-04, 1.50814037e-04, 9.27562579e-04, 1.57879343e-04,
       1.45510385e-04, 1.47469165e-04, 1.77524281e-04, 1.68373674e-04,
       1.57879343e-04, 9.36941411e-04, 1.42085469e-04, 1.48860956e-04,
       1.42217689e-04, 1.49515195e-04, 1.48058640e-04, 1.47386073e-04,
       8.61898339e-04, 1.46011532e-04, 1.35422443e-04, 1.35504456e-04,
       1.52365943e-04, 1.61003699e-04, 1.46011532e-04, 9.22039089e-04,
       1.63979057e-04, 1.44674545e-04, 1.54004314e-04, 1.36375794e-04,
       1.41248992e-04, 1.47895403e-04, 9.12576107e-04, 1.62688443e-04,
       1.57223817e-04, 1.67143657e-04, 1.65983692e-04, 1.55666038e-04,
       1.57805711e-04, 9.23278798e-04, 1.46060183e-04, 1.49595524e-04,
       1.48167650e-04, 1.54831299e-04, 1.66303587e-04, 1.62688443e-04,
       1.16145548e-03, 1.66428513e-04, 1.76342594e-04, 1.62072517e-04,
       1.72788523e-04, 1.48705980e-04, 1.56537257e-04, 1.43775952e-04,
       8.96322829e-04, 1.54307519e-04, 1.45147915e-04, 1.47848206e-04,
       1.37293151e-04, 1.40912669e-04, 1.32304230e-04, 8.13602173e-04,
       1.38876412e-04, 1.30631524e-04, 1.33584188e-04, 9.23166090e-05,
       1.29928990e-04, 1.09385684e-04, 7.20385225e-04, 1.25048551e-04,
       1.42827453e-04, 1.20877938e-04, 1.31736631e-04, 1.34752100e-04,
       7.91676216e-04, 1.28078454e-04, 1.34226586e-04, 1.33007553e-04,
       1.29874891e-04, 1.30192400e-04, 1.25645140e-04, 1.20244229e-04,
       1.28078454e-04, 7.47217947e-04, 1.19885379e-04, 1.23172344e-04,
       1.36524034e-04, 1.38587134e-04, 1.34226586e-04, 8.20134638e-04,
       1.37611958e-04, 1.35931327e-04, 1.37253599e-04, 1.39659267e-04,
       1.39842483e-04, 1.37611958e-04, 8.29273213e-04, 1.38015477e-04,
       1.36961862e-04, 1.37182166e-04, 1.39625096e-04, 1.38923557e-04,
       1.38015477e-04, 8.22653483e-04, 1.36468095e-04, 1.35233550e-04,
       1.34387708e-04, 1.40659113e-04, 1.40634611e-04, 8.31482033e-04,
       1.38699489e-04, 1.37835197e-04, 1.36406917e-04, 1.37246706e-04,
       1.39768206e-04, 1.41954804e-04, 1.38699489e-04, 8.36918925e-04,
       1.41318866e-04, 1.37372391e-04, 1.37805170e-04, 1.39258629e-04,
       1.53586142e-04, 1.37835197e-04, 8.93647237e-04, 1.69725869e-04,
       1.37839120e-04, 1.55402279e-04, 1.38133795e-04, 1.34085935e-04,
       1.41318866e-04, 8.21119446e-04, 1.35580878e-04, 1.36845051e-04,
       1.35154921e-04, 1.23268480e-04, 1.13181316e-04, 1.22714791e-04,
       1.35580878e-04, 7.56235874e-04, 1.29355744e-04, 1.32134665e-04,
       1.17408776e-04, 1.15613237e-04, 1.19885379e-04, 5.90492336e-04,
       1.18364915e-04, 1.19220029e-04, 1.30911697e-04, 1.38734346e-04,
       1.29355744e-04, 8.11727574e-04, 1.40857313e-04, 1.34094778e-04,
       1.37773698e-04, 1.46603464e-04, 1.49197136e-04, 1.40857313e-04,
       8.67921860e-04, 1.47210991e-04, 1.40146475e-04, 1.43906481e-04,
       1.46052695e-04, 1.44761089e-04, 1.42872743e-04, 8.58881389e-04,
       1.42650792e-04, 1.40737208e-04, 1.41806862e-04, 1.27850282e-04,
       1.06896662e-04, 1.28667447e-04, 7.10210892e-04, 1.02539021e-04,
       1.24716143e-04, 1.19541338e-04, 1.49388243e-04, 1.51237741e-04,
       1.49946089e-04, 9.04582708e-04, 1.50628325e-04, 1.52274786e-04,
       1.51107524e-04, 1.51594781e-04, 1.52375393e-04, 1.50628325e-04,
       8.95238361e-04, 1.49564129e-04, 1.46460270e-04, 1.44615463e-04,
       1.50060053e-04, 1.52034455e-04, 1.52274786e-04, 9.16306956e-04,
       1.56380050e-04, 1.51713375e-04, 1.53844237e-04, 1.58062072e-04,
       1.56122099e-04, 1.56380050e-04, 9.38006514e-04, 1.54444188e-04,
       1.56496998e-04, 1.56501108e-04, 1.50352096e-04, 1.49267186e-04,
       1.54444188e-04, 9.09662874e-04, 1.51493437e-04, 1.52985904e-04,
       1.51120063e-04, 1.49690143e-04, 1.47786195e-04, 1.51493437e-04,
       8.93406593e-04, 1.47437657e-04, 1.49575497e-04, 1.47423665e-04,
       1.36970973e-04, 1.32946729e-04, 1.37301729e-04, 8.07559204e-04,
       1.31441686e-04, 1.35366944e-04, 1.33531144e-04, 1.47067719e-04,
       1.47099213e-04, 1.48235774e-04, 8.88452387e-04, 1.47660966e-04,
       1.49497508e-04, 1.48891207e-04, 1.45387758e-04, 1.42085469e-04,
       1.35422443e-04, 7.89330163e-04, 1.21936655e-04, 1.27340668e-04,
       1.17157170e-04, 1.63171915e-04, 1.63979057e-04, 1.02210789e-03,
       1.72982323e-04, 1.69343592e-04, 1.74092565e-04, 1.78538438e-04,
       1.76515286e-04, 1.92666398e-04, 1.54307519e-04, 1.04088785e-03,
       1.90649446e-04, 1.53279193e-04, 1.73470009e-04, 1.82293728e-04,
       1.63604047e-04, 1.90649446e-04, 1.02864338e-03, 1.61868966e-04,
       1.70158995e-04, 1.60068196e-04, 1.54978998e-04, 1.57223817e-04,
       1.61868966e-04, 9.19607531e-04, 1.52223651e-04, 1.52601117e-04,
       1.40710983e-04, 1.43937480e-04, 1.41060808e-04, 1.46060183e-04,
       8.48278726e-04, 1.40017344e-04, 1.40184539e-04, 1.37018372e-04,
       1.39598327e-04, 1.38784450e-04, 1.40017344e-04, 8.41203034e-04,
       1.38259123e-04, 1.42744068e-04, 1.41799723e-04, 1.37432887e-04,
       1.39111043e-04, 1.38259123e-04, 8.37210634e-04, 1.40749981e-04,
       1.40422449e-04, 1.41235151e-04, 1.49728258e-04, 1.49595524e-04,
       9.52795466e-04, 1.48612986e-04, 1.64611880e-04, 1.78189223e-04,
       1.62057596e-04, 1.62031629e-04, 1.66428513e-04, 1.48612986e-04,
       9.17281879e-04, 1.50683529e-04, 1.45925781e-04, 1.43599441e-04,
       1.41368535e-04, 1.45764376e-04, 1.40749981e-04, 8.62670812e-04,
       1.46779732e-04, 1.43187139e-04, 1.44821048e-04, 1.41618520e-04,
       1.45147915e-04, 1.38876412e-04, 8.64481865e-04, 1.50156098e-04,
       1.40466314e-04, 1.48216605e-04, 1.28403572e-04, 1.30631524e-04,
       1.25048551e-04, 7.66419891e-04, 1.31705459e-04, 1.23100915e-04,
       1.27529871e-04, 1.64511456e-04, 1.42827453e-04, 1.72982323e-04,
       8.44615923e-04, 1.11982445e-04, 1.45814060e-04, 1.06498186e-04,
       1.38280877e-04, 1.37968138e-04, 1.36468095e-04, 8.12724453e-04,
       1.35106139e-04, 1.32975211e-04, 1.31925993e-04, 1.43961751e-04,
       1.69725869e-04, 1.18364915e-04, 1.05679704e-03, 1.40495345e-04,
       1.64887546e-04, 1.59559944e-04, 1.59801672e-04, 1.51349975e-04,
       1.50454750e-04, 1.52889235e-04, 9.10370623e-04, 1.51062082e-04,
       1.52773255e-04, 1.51841326e-04, 1.49826319e-04, 1.49126045e-04,
       1.51062082e-04, 9.03009094e-04, 1.49908164e-04, 1.51770065e-04,
       1.51316421e-04, 1.45331030e-04, 1.41059742e-04, 1.49908164e-04,
       8.81131704e-04, 1.44308310e-04, 1.50926512e-04, 1.49597947e-04,
       1.26833162e-04, 1.12500817e-04, 1.36452929e-04, 1.44308310e-04,
       8.21995011e-04, 1.54188379e-04, 1.47711414e-04, 1.58038869e-04,
       1.56862372e-04, 9.30253714e-04, 1.53664423e-04, 1.56292125e-04,
       1.51972359e-04, 1.53423565e-04, 1.51960080e-04, 1.47210991e-04,
       1.53664423e-04, 8.95247926e-04, 1.45412734e-04, 1.50251149e-04,
       1.46748549e-04, 1.54376468e-04, 1.53380671e-04, 1.52773255e-04,
       9.19505418e-04, 1.53928738e-04, 1.52443300e-04, 1.52602986e-04,
       1.53337579e-04, 1.51806444e-04, 1.53928738e-04, 9.13800364e-04,
       1.51113484e-04, 1.52449119e-04, 1.51164999e-04, 1.50694001e-04,
       1.49973697e-04, 1.51113484e-04, 9.01662224e-04, 1.49702484e-04,
       1.50434627e-04, 1.49743931e-04, 1.50324485e-04, 1.50089731e-04,
       1.49702484e-04, 8.95498389e-04, 1.48682084e-04, 1.48742626e-04,
       1.47956979e-04, 1.48778035e-04, 1.46177023e-04, 1.48682084e-04,
       8.81512872e-04, 1.45254658e-04, 1.46723743e-04, 1.45897329e-04,
       1.44059428e-04, 1.42617942e-04, 1.45254658e-04, 8.62718292e-04,
       1.42545642e-04, 1.44754075e-04, 1.43486546e-04, 1.42833951e-04,
       1.43048803e-04, 1.42650792e-04, 1.42545642e-04, 8.56537720e-04,
       1.42674415e-04, 1.42784118e-04, 8.59044849e-05, 1.04790190e-04,
       1.02539021e-04, 5.40648093e-04, 1.22637637e-04, 1.24776761e-04,
       1.29516598e-04, 1.24716143e-04, 1.31441686e-04, 7.71195139e-04,
       1.24365866e-04, 1.31715106e-04, 1.29439741e-04, 1.52442953e-04,
       1.49564129e-04, 8.76553077e-04, 1.46447063e-04, 1.43580777e-04,
       1.42252809e-04, 1.42265347e-04, 1.48078061e-04, 1.47437657e-04,
       1.46779732e-04, 8.81167679e-04, 1.45596019e-04, 1.46909760e-04,
       1.46366451e-04, 1.39413212e-04, 1.40737208e-04, 8.32778475e-04,
       1.37743738e-04, 1.39907914e-04, 1.37908292e-04, 1.37068112e-04,
       1.39019513e-04, 1.38564830e-04, 1.37743738e-04, 8.29484029e-04,
       1.37680291e-04, 1.38042210e-04, 1.38433447e-04, 1.37069938e-04,
       1.35366944e-04, 1.37680291e-04, 8.22652990e-04, 1.35791078e-04,
       1.38465479e-04, 1.38279260e-04, 1.50628780e-04, 1.53960954e-04,
       9.10430030e-04, 1.49013067e-04, 1.54967700e-04, 1.52240994e-04,
       1.49618535e-04, 1.48655924e-04, 1.48152418e-04, 1.47660966e-04,
       1.49013067e-04, 8.92434363e-04, 1.49051696e-04, 1.49900292e-04,
       1.55580359e-04, 1.56492243e-04, 1.54967700e-04, 9.43744783e-04,
       1.59127389e-04, 1.57164644e-04, 1.60412449e-04, 1.45180524e-04,
       1.48003121e-04, 8.71838015e-04, 1.42458076e-04, 1.48105751e-04,
       1.42632732e-04, 1.45457811e-04, 1.42632688e-04, 1.41137899e-04,
       1.42458076e-04, 8.41643250e-04, 1.38108267e-04, 1.40080670e-04,
       1.37225650e-04, 1.38812104e-04, 1.37742629e-04, 1.35106139e-04,
       1.38108267e-04, 8.15599676e-04, 1.31948943e-04, 1.33881595e-04,
       1.49306168e-04, 1.49497508e-04, 1.48105751e-04, 8.92086919e-04,
       1.49307646e-04, 1.47429645e-04, 1.48440200e-04, 1.49181880e-04,
       1.45411051e-04, 1.59127389e-04, 9.34295112e-04, 1.54606110e-04,
       1.63359566e-04, 1.62609116e-04, 1.46024661e-04, 1.45760847e-04,
       1.54606110e-04, 9.34985425e-04, 1.53125521e-04, 1.68342531e-04,
       1.67125755e-04, 1.48658412e-04, 1.53114307e-04, 9.33936534e-04,
       1.55884321e-04, 1.58344751e-04, 1.57969639e-04, 1.59965103e-04,
       1.50449726e-04, 1.52232078e-04, 1.55884321e-04, 9.20201203e-04,
       1.54690772e-04, 1.54960103e-04, 1.51984203e-04, 1.55921058e-04,
       1.46771127e-04, 1.58344751e-04, 8.90830915e-04, 1.39157928e-04,
       1.50336471e-04, 1.40299579e-04, 1.53927083e-04, 1.46262504e-04,
       1.54690772e-04, 8.81109422e-04, 1.37626317e-04, 1.49001311e-04,
       1.39601435e-04, 1.34915830e-04, 1.31694370e-04, 1.37626317e-04,
       7.99376809e-04, 1.30583937e-04, 1.34188638e-04, 1.30367718e-04,
       1.30567752e-04, 1.54778396e-04, 1.30583937e-04, 9.07475356e-04,
       1.76654772e-04, 1.46515154e-04, 1.68375346e-04, 1.79991120e-04,
       1.64572160e-04, 1.76654772e-04, 8.52245051e-04, 1.15865090e-04,
       1.30266315e-04, 8.48955930e-05, 1.43980328e-04, 1.46449898e-04,
       1.39157928e-04, 8.89258750e-04, 1.54550886e-04, 1.46098161e-04,
       1.59021549e-04, 1.47344516e-04, 1.47096270e-04, 1.53125521e-04,
       1.15865090e-04, 9.33469946e-04, 1.61842876e-04, 8.38606029e-05,
       1.24335069e-04, 1.45562828e-04, 1.45510385e-04, 1.54550886e-04,
       8.47031961e-04, 1.34381769e-04, 1.43571384e-04, 1.23454709e-04,
       1.47469165e-04, 1.48860956e-04, 1.34381769e-04, 8.37088793e-04,
       1.40907310e-04, 1.30167922e-04, 1.35301671e-04, 1.42217689e-04,
       1.21936655e-04, 1.40907310e-04, 7.91002943e-04, 1.04797678e-04,
       1.48486104e-04, 1.32657507e-04, 1.35504456e-04, 1.44674545e-04,
       1.27340668e-04, 7.96985639e-04, 1.44531122e-04, 1.18457241e-04,
       1.26477607e-04, 1.54004314e-04, 1.69343592e-04, 1.44531122e-04,
       9.64981512e-04, 1.77512290e-04, 1.50974152e-04, 1.68616042e-04,
       1.67143657e-04, 1.76342594e-04, 1.52223651e-04, 9.62395595e-04,
       1.45868700e-04, 1.68185440e-04, 1.52631553e-04, 1.48167650e-04,
       1.40184539e-04, 1.64611880e-04, 9.01345127e-04, 1.32790120e-04,
       1.69714793e-04, 1.45876145e-04, 1.62072517e-04, 1.50683529e-04,
       9.25815361e-04, 1.62389284e-04, 1.45186793e-04, 1.49834868e-04,
       1.55648371e-04, 1.47848206e-04, 1.53279193e-04, 1.50156098e-04,
       8.84478787e-04, 1.47628311e-04, 1.46762876e-04, 1.38804102e-04,
       1.33584188e-04, 1.40466314e-04, 1.31705459e-04, 8.18586352e-04,
       1.41887260e-04, 1.33821728e-04, 1.37121404e-04, 1.20877938e-04,
       1.23100915e-04, 1.11982445e-04, 6.93996402e-04, 1.18808267e-04,
       1.07311996e-04, 1.11914841e-04, 1.33007553e-04, 1.35931327e-04,
       8.06783412e-04, 1.31410546e-04, 1.37333900e-04, 1.33050252e-04,
       1.36049832e-04, 1.29874891e-04, 1.23172344e-04, 1.19220029e-04,
       1.40495345e-04, 1.31410546e-04, 7.90156472e-04, 1.45983316e-04,
       1.37253599e-04, 1.36961862e-04, 1.37333900e-04, 8.24117637e-04,
       1.36745935e-04, 1.38059284e-04, 1.37763056e-04, 1.37182166e-04,
       1.35233550e-04, 1.36745935e-04, 8.09904519e-04, 1.32184181e-04,
       1.35585913e-04, 1.32972774e-04, 1.34387708e-04, 1.32975211e-04,
       1.32184181e-04, 7.80977660e-04, 1.29367331e-04, 1.26733791e-04,
       1.25329438e-04, 1.36406917e-04, 1.37839120e-04, 8.33197586e-04,
       1.37235957e-04, 1.42879668e-04, 1.41523716e-04, 1.37312208e-04,
       1.37246706e-04, 1.37372391e-04, 1.37235957e-04, 8.22009094e-04,
       1.36959896e-04, 1.36797944e-04, 1.36396199e-04, 1.37805170e-04,
       1.36845051e-04, 1.36959896e-04, 8.16970135e-04, 1.35738011e-04,
       1.34941887e-04, 1.34680120e-04, 1.55402279e-04, 1.64887546e-04,
       1.42879668e-04, 9.28584889e-04, 1.61600818e-04, 1.47289347e-04,
       1.56525230e-04, 1.35154921e-04, 1.32134665e-04, 1.34094778e-04,
       1.35738011e-04, 8.07301517e-04, 1.35778083e-04, 1.34401060e-04,
       1.37773698e-04, 1.40146475e-04, 1.35778083e-04, 8.29585141e-04,
       1.41091716e-04, 1.35927157e-04, 1.38868013e-04, 1.59293335e-04,
       1.54188379e-04, 1.56292125e-04, 9.27788949e-04, 1.50868071e-04,
       1.53733068e-04, 1.53413970e-04, 1.43906481e-04, 1.45412734e-04,
       1.41091716e-04, 8.58552060e-04, 1.43935151e-04, 1.41434398e-04,
       1.42771580e-04, 1.41806862e-04, 1.42674415e-04, 1.39907914e-04,
       8.48239404e-04, 1.42332622e-04, 1.39980218e-04, 1.41537373e-04,
       1.19541338e-04, 1.22637637e-04, 1.24365866e-04, 7.58418742e-04,
       1.28950295e-04, 1.30717738e-04, 1.32205868e-04, 1.51107524e-04,
       1.46460270e-04, 1.51713375e-04, 9.01296547e-04, 1.46554384e-04,
       1.52986366e-04, 1.52474628e-04, 1.37987621e-04, 1.24776761e-04,
       1.46447063e-04, 1.28950295e-04, 8.17922796e-04, 1.42023482e-04,
       1.37737575e-04, 1.44615463e-04, 1.43580777e-04, 1.46554384e-04,
       8.73612787e-04, 1.42005695e-04, 1.50185235e-04, 1.46671233e-04,
       1.53844237e-04, 1.56496998e-04, 1.52986366e-04, 9.21025787e-04,
       1.52820148e-04, 1.53848508e-04, 1.51029530e-04, 1.56501108e-04,
       1.52985904e-04, 1.52820148e-04, 9.19460899e-04, 1.51913249e-04,
       1.51316092e-04, 1.53924397e-04, 1.51120063e-04, 1.49575497e-04,
       1.51913249e-04, 9.02024024e-04, 1.48294644e-04, 1.51597305e-04,
       1.49523266e-04, 1.47423665e-04, 1.45596019e-04, 1.48294644e-04,
       8.82711448e-04, 1.44783192e-04, 1.49148867e-04, 1.47465061e-04,
       1.33531144e-04, 1.31715106e-04, 1.35791078e-04, 8.12783242e-04,
       1.34054002e-04, 1.38806475e-04, 1.38885437e-04, 1.48891207e-04,
       1.49051696e-04, 1.49307646e-04, 8.98279561e-04, 1.49799582e-04,
       1.50448992e-04, 1.50780438e-04, 1.17157170e-04, 1.04797678e-04,
       1.18457241e-04, 5.49941328e-04, 1.03432349e-04, 1.06096890e-04,
       1.74092565e-04, 1.77512290e-04, 1.17846858e-03, 1.61950999e-04,
       1.78073719e-04, 1.57616980e-04, 1.67786393e-04, 1.61435634e-04,
       1.78538438e-04, 1.45814060e-04, 1.61950999e-04, 1.22287615e-03,
       1.07241863e-04, 1.63833750e-04, 1.14510593e-04, 1.89162420e-04,
       1.61824022e-04, 1.73470009e-04, 1.70158995e-04, 1.47628311e-04,
       9.25384810e-04, 1.51876047e-04, 1.37346349e-04, 1.44905099e-04,
       1.60068196e-04, 1.52601117e-04, 1.51876047e-04, 9.01356816e-04,
       1.37143805e-04, 1.53466407e-04, 1.46201244e-04, 1.40710983e-04,
       1.45868700e-04, 1.37143805e-04, 7.01648261e-04, 1.41920929e-04,
       1.36003844e-04, 1.72788523e-04, 1.68185440e-04, 1.62389284e-04,
       9.74050624e-04, 1.56746014e-04, 1.57490775e-04, 1.56450588e-04,
       1.37018372e-04, 1.42744068e-04, 1.32790120e-04, 8.50936559e-04,
       1.48506366e-04, 1.39943541e-04, 1.49934092e-04, 1.41799723e-04,
       1.40422449e-04, 1.48506366e-04, 8.67041363e-04, 1.43282710e-04,
       1.48438248e-04, 1.44591866e-04, 1.41235151e-04, 1.43187139e-04,
       1.43282710e-04, 8.61331558e-04, 1.44808242e-04, 1.44622386e-04,
       1.44195930e-04, 1.78189223e-04, 1.69714793e-04, 1.20174954e-03,
       1.72745737e-04, 1.66633225e-04, 1.77848646e-04, 1.65316419e-04,
       1.71301495e-04, 1.62057596e-04, 1.45925781e-04, 1.72745737e-04,
       9.31022148e-04, 1.42882587e-04, 1.60570891e-04, 1.46839555e-04,
       1.43599441e-04, 1.45186793e-04, 1.42882587e-04, 8.67114366e-04,
       1.43752575e-04, 1.46205167e-04, 1.45487803e-04, 1.44821048e-04,
       1.46909760e-04, 1.44808242e-04, 8.77361708e-04, 1.47295690e-04,
       1.46564116e-04, 1.46962852e-04, 1.48216605e-04, 1.46762876e-04,
       1.41887260e-04, 8.81755873e-04, 1.46735783e-04, 1.45925549e-04,
       1.52227800e-04, 1.27529871e-04, 1.33821728e-04, 1.18808267e-04,
       7.73162208e-04, 1.38580440e-04, 1.20470792e-04, 1.33951109e-04,
       1.06498186e-04, 1.07311996e-04, 1.07241863e-04, 5.44565419e-04,
       1.12136911e-04, 1.11376464e-04, 1.31925993e-04, 1.31948943e-04,
       1.29367331e-04, 7.70897288e-04, 1.29474863e-04, 1.24047788e-04,
       1.24132370e-04, 1.59559944e-04, 1.61600818e-04, 9.21148691e-04,
       1.51378958e-04, 1.56919777e-04, 1.44094625e-04, 1.47594570e-04,
       1.59801672e-04, 1.33050252e-04, 1.45983316e-04, 1.51378958e-04,
       8.70823932e-04, 1.38049692e-04, 1.42560042e-04, 1.51841326e-04,
       1.51770065e-04, 1.52443300e-04, 9.12486257e-04, 1.52579887e-04,
       1.51821916e-04, 1.52029764e-04, 1.51316421e-04, 1.50926512e-04,
       1.52579887e-04, 9.14504204e-04, 1.52480793e-04, 1.53454888e-04,
       1.53745702e-04, 1.49597947e-04, 1.47711414e-04, 1.50868071e-04,
       1.52480793e-04, 9.07040977e-04, 1.53328281e-04, 1.53054471e-04,
       1.51972359e-04, 1.50251149e-04, 8.97939028e-04, 1.51033938e-04,
       1.48262937e-04, 1.47684427e-04, 1.48734218e-04, 1.53423565e-04,
       1.53733068e-04, 1.51033938e-04, 9.10505634e-04, 1.52715725e-04,
       1.49113528e-04, 1.50485811e-04, 1.46748549e-04, 1.43935151e-04,
       1.48262937e-04, 8.72948665e-04, 1.42561699e-04, 1.47164084e-04,
       1.44276245e-04, 1.52602986e-04, 1.52449119e-04, 1.51821916e-04,
       9.09805843e-04, 1.51374126e-04, 1.50925810e-04, 1.50631887e-04,
       1.51164999e-04, 1.50434627e-04, 1.51374126e-04, 9.01792584e-04,
       1.49544126e-04, 1.50187166e-04, 1.49087539e-04, 1.49743931e-04,
       1.48742626e-04, 1.49544126e-04, 8.90359976e-04, 1.47315724e-04,
       1.48120333e-04, 1.46893236e-04, 1.47956979e-04, 1.46723743e-04,
       1.47315724e-04, 8.78305259e-04, 1.45166633e-04, 1.46029017e-04,
       1.45113162e-04, 1.45897329e-04, 1.44754075e-04, 1.45166633e-04,
       8.64808645e-04, 1.42837981e-04, 1.43669013e-04, 1.42483614e-04,
       1.43486546e-04, 1.42784118e-04, 1.42332622e-04, 1.42837981e-04,
       8.55454578e-04, 1.42105191e-04, 1.41908122e-04, 1.29439741e-04,
       1.30717738e-04, 1.34054002e-04, 8.02537969e-04, 1.33563850e-04,
       1.38265394e-04, 1.36497245e-04, 1.42252809e-04, 1.42023482e-04,
       8.40703235e-04, 1.42718927e-04, 1.40154640e-04, 1.37944169e-04,
       1.35609209e-04, 1.42265347e-04, 1.42005695e-04, 1.42718927e-04,
       8.52559104e-04, 1.44088176e-04, 1.39926030e-04, 1.41554930e-04,
       1.46366451e-04, 1.44783192e-04, 1.47295690e-04, 8.75677057e-04,
       1.45639414e-04, 1.45832663e-04, 1.45759646e-04, 1.37908292e-04,
       1.39980218e-04, 8.30985171e-04, 1.36625689e-04, 1.39583220e-04,
       1.38886678e-04, 1.38001074e-04, 1.37068112e-04, 1.38042210e-04,
       1.36625689e-04, 8.28892254e-04, 1.39730066e-04, 1.37647949e-04,
       1.39778229e-04, 1.38433447e-04, 1.38465479e-04, 1.39730066e-04,
       8.36221691e-04, 1.39282950e-04, 1.40394450e-04, 1.39915300e-04,
       1.38279260e-04, 1.38806475e-04, 1.39282950e-04, 8.53683287e-04,
       1.43354151e-04, 1.45208233e-04, 1.48752219e-04, 1.52240994e-04,
       1.57164644e-04, 9.12333367e-04, 1.46844365e-04, 1.56422792e-04,
       1.52157675e-04, 1.47502897e-04, 1.49618535e-04, 1.49900292e-04,
       1.49799582e-04, 1.46844365e-04, 8.89256624e-04, 1.47883656e-04,
       1.45210195e-04, 1.60412449e-04, 1.63359566e-04, 1.56422792e-04,
       9.79859351e-04, 1.71653480e-04, 1.57863747e-04, 1.70147317e-04,
       1.42632732e-04, 1.40080670e-04, 8.35219059e-04, 1.42036218e-04,
       1.35902260e-04, 1.39074537e-04, 1.35492641e-04, 1.45457811e-04,
       1.47429645e-04, 1.42036218e-04, 8.68691685e-04, 1.46844477e-04,
       1.42043555e-04, 1.44879979e-04, 1.37225650e-04, 1.33881595e-04,
       1.29474863e-04, 1.35902260e-04, 7.95910080e-04, 1.28171185e-04,
       1.31254527e-04, 1.48440200e-04, 1.50448992e-04, 1.46844477e-04,
       8.90393740e-04, 1.50159956e-04, 1.46596714e-04, 1.47903402e-04,
       1.62609116e-04, 1.68342531e-04, 1.71653480e-04, 1.14277025e-03,
       1.53599380e-04, 1.92871625e-04, 1.28217172e-04, 1.65476946e-04,
       1.67125755e-04, 1.61842876e-04, 1.53599380e-04, 9.37458349e-04,
       1.66497855e-04, 1.35105564e-04, 1.53286919e-04, 1.57969639e-04,
       1.50336471e-04, 9.06521840e-04, 1.55046084e-04, 1.45186310e-04,
       1.47878366e-04, 1.50104971e-04, 1.59965103e-04, 1.54960103e-04,
       1.55046084e-04, 8.91552775e-04, 1.41360932e-04, 1.44450352e-04,
       1.35770201e-04, 1.51984203e-04, 1.49001311e-04, 1.41360932e-04,
       8.29222654e-04, 1.32256494e-04, 1.30370631e-04, 1.24249084e-04,
       1.40299579e-04, 1.46098161e-04, 1.45186310e-04, 8.77735773e-04,
       1.51395706e-04, 1.47172601e-04, 1.47583416e-04, 1.39601435e-04,
       1.34188638e-04, 1.32256494e-04, 7.84480034e-04, 1.27573388e-04,
       1.25065193e-04, 1.25794886e-04, 1.30367718e-04, 1.46515154e-04,
       1.27573388e-04, 8.56429489e-04, 1.60944135e-04, 1.36902892e-04,
       1.54126202e-04, 1.68375346e-04, 1.30266315e-04, 1.60944135e-04,
       8.59325652e-04, 1.12078736e-04, 1.49208745e-04, 1.38452376e-04,
       8.48955930e-05, 8.38606029e-05, 1.12078736e-04, 4.85417705e-04,
       8.87002503e-05, 1.15882523e-04, 1.59021549e-04, 1.43571384e-04,
       1.51395706e-04, 1.05467876e-03, 1.22093787e-04, 1.57426563e-04,
       1.49848010e-04, 1.71321762e-04, 1.23454709e-04, 1.30167922e-04,
       1.22093787e-04, 6.27033113e-04, 1.29011318e-04, 1.22305377e-04,
       1.35301671e-04, 1.48486104e-04, 1.29011318e-04, 8.58297074e-04,
       1.61591041e-04, 1.32255825e-04, 1.51651115e-04, 1.32657507e-04,
       1.03432349e-04, 1.61591041e-04, 9.98822758e-04, 1.31294696e-04,
       1.55832640e-04, 1.59205488e-04, 1.54809038e-04, 1.26477607e-04,
       1.50974152e-04, 1.06096890e-04, 1.31294696e-04, 8.25822321e-04,
       1.55138857e-04, 1.55840119e-04, 1.68616042e-04, 1.78073719e-04,
       1.55138857e-04, 9.86865887e-04, 1.69706024e-04, 1.55110878e-04,
       1.60220367e-04, 1.52631553e-04, 1.41920929e-04, 1.56746014e-04,
       8.90300223e-04, 1.41201487e-04, 1.51317071e-04, 1.46483169e-04,
       1.45876145e-04, 1.39943541e-04, 1.66633225e-04, 9.31414006e-04,
       1.54665935e-04, 1.65164923e-04, 1.59130237e-04, 1.49834868e-04,
       1.43752575e-04, 9.00154789e-04, 1.53970959e-04, 1.45024389e-04,
       1.52353861e-04, 1.55218137e-04, 1.55648371e-04, 1.57490775e-04,
       1.53970959e-04, 9.29126283e-04, 1.55500988e-04, 1.53413783e-04,
       1.53101408e-04, 1.38804102e-04, 1.37346349e-04, 1.46735783e-04,
       7.13638888e-04, 1.42185029e-04, 1.48567625e-04, 1.37121404e-04,
       1.45925549e-04, 1.38580440e-04, 8.50140498e-04, 1.46743180e-04,
       1.41205665e-04, 1.40564260e-04, 1.11914841e-04, 1.20470792e-04,
       1.12136911e-04, 7.33950681e-04, 1.23679722e-04, 1.31380778e-04,
       1.34367638e-04, 1.36049832e-04, 1.38059284e-04, 1.38049692e-04,
       8.35054386e-04, 1.40866109e-04, 1.40611252e-04, 1.41418217e-04,
       1.37763056e-04, 1.35585913e-04, 1.40866109e-04, 8.31621399e-04,
       1.35239401e-04, 1.42808243e-04, 1.39358678e-04, 1.32972774e-04,
       1.26733791e-04, 1.35239401e-04, 8.02396093e-04, 1.21744754e-04,
       1.46480519e-04, 1.39224855e-04, 1.25329438e-04, 1.24047788e-04,
       1.21744754e-04, 7.00332451e-04, 1.20021490e-04, 1.04815298e-04,
       1.04373684e-04, 1.41523716e-04, 1.47289347e-04, 8.48239234e-04,
       1.37272704e-04, 1.49644653e-04, 1.39557565e-04, 1.32951248e-04,
       1.37312208e-04, 1.36797944e-04, 1.37272704e-04, 8.22854169e-04,
       1.38049543e-04, 1.36065333e-04, 1.37356436e-04, 1.36396199e-04,
       1.34941887e-04, 1.38049543e-04, 8.29915666e-04, 1.38989647e-04,
       1.39571987e-04, 1.41966402e-04, 1.34680120e-04, 1.34401060e-04,
       1.35927157e-04, 1.38989647e-04, 8.24118249e-04, 1.38668418e-04,
       1.41451848e-04, 1.56525230e-04, 1.56919777e-04, 1.49644653e-04,
       9.10533643e-04, 1.49947409e-04, 1.48794182e-04, 1.48702391e-04,
       1.38868013e-04, 1.41434398e-04, 1.38668418e-04, 8.45555671e-04,
       1.41501497e-04, 1.42408326e-04, 1.42675020e-04, 1.53413970e-04,
       1.53328281e-04, 1.52715725e-04, 9.16496021e-04, 1.53173839e-04,
       1.51660201e-04, 1.52204005e-04, 1.42771580e-04, 1.42561699e-04,
       1.41501497e-04, 8.48898503e-04, 1.40348558e-04, 1.41329113e-04,
       1.40386056e-04, 1.41537373e-04, 1.42105191e-04, 1.39583220e-04,
       8.40619096e-04, 1.40533027e-04, 1.38239148e-04, 1.38621137e-04,
       1.32205868e-04, 1.37737575e-04, 1.33563850e-04, 1.40154640e-04,
       8.15676658e-04, 1.35477821e-04, 1.36536904e-04, 1.52474628e-04,
       1.50185235e-04, 1.53848508e-04, 9.13060935e-04, 1.51289683e-04,
       1.52766156e-04, 1.52496724e-04, 1.46671233e-04, 1.44088176e-04,
       1.51289683e-04, 8.85678134e-04, 1.45752101e-04, 1.50415733e-04,
       1.47461208e-04, 1.51029530e-04, 1.51316092e-04, 1.52766156e-04,
       9.09381927e-04, 1.52878340e-04, 1.50783093e-04, 1.50608716e-04,
       1.53924397e-04, 1.51597305e-04, 1.52878340e-04, 9.14387289e-04,
       1.50046281e-04, 1.53222966e-04, 1.52718000e-04, 1.49523266e-04,
       1.49148867e-04, 1.50046281e-04, 9.04711172e-04, 1.49656331e-04,
       1.53175989e-04, 1.53160438e-04, 1.47465061e-04, 1.45639414e-04,
       1.49656331e-04, 8.85430470e-04, 1.44593714e-04, 1.50656460e-04,
       1.47419489e-04, 1.38885437e-04, 1.38265394e-04, 1.43354151e-04,
       8.68712764e-04, 1.43811568e-04, 1.51659377e-04, 1.52736837e-04,
       1.50780438e-04, 1.47883656e-04, 1.50159956e-04, 8.83242432e-04,
       1.44674792e-04, 1.46165987e-04, 1.43577604e-04, 1.24335069e-04,
       1.66497855e-04, 8.87002503e-05, 8.46991257e-04, 1.75948127e-04,
       1.23212433e-04, 1.68297524e-04, 1.57616980e-04, 1.63833750e-04,
       9.59269551e-04, 1.62732392e-04, 1.64309269e-04, 1.51491756e-04,
       1.59285403e-04, 1.67786393e-04, 1.69706024e-04, 9.95969891e-04,
       1.64039814e-04, 1.61641681e-04, 1.64239108e-04, 1.68556872e-04,
       1.14510593e-04, 1.11376464e-04, 1.23679722e-04, 5.95789867e-04,
       1.20567599e-04, 1.25655489e-04, 1.44905099e-04, 1.53466407e-04,
       1.42185029e-04, 8.92695324e-04, 1.52265703e-04, 1.49176861e-04,
       1.50696226e-04, 1.46201244e-04, 1.36003844e-04, 1.41201487e-04,
       1.52265703e-04, 8.65546308e-04, 1.44503607e-04, 1.45370422e-04,
       1.56450588e-04, 1.51317071e-04, 1.55500988e-04, 9.02364176e-04,
       1.46838603e-04, 1.47892897e-04, 1.44364029e-04, 1.49934092e-04,
       1.48438248e-04, 1.54665935e-04, 9.00850926e-04, 1.47806671e-04,
       1.52684701e-04, 1.47321280e-04, 1.44591866e-04, 1.44622386e-04,
       1.47806671e-04, 8.73306799e-04, 1.44577955e-04, 1.47483578e-04,
       1.44224343e-04, 1.44195930e-04, 1.46564116e-04, 1.44577955e-04,
       8.63753110e-04, 1.44622810e-04, 1.43057815e-04, 1.40734484e-04,
       1.77848646e-04, 1.65164923e-04, 9.75181333e-04, 1.54741038e-04,
       1.67650837e-04, 1.55000983e-04, 1.54774906e-04, 1.65316419e-04,
       1.60570891e-04, 9.04310668e-04, 1.48296791e-04, 1.51869284e-04,
       1.38542123e-04, 1.39715158e-04, 1.46839555e-04, 1.46205167e-04,
       1.48296791e-04, 8.57387313e-04, 1.40733960e-04, 1.41120141e-04,
       1.34191698e-04, 1.45487803e-04, 1.45024389e-04, 1.40733960e-04,
       8.99693376e-04, 1.55881097e-04, 1.48477788e-04, 1.64088338e-04,
       1.46962852e-04, 1.45832663e-04, 1.44622810e-04, 8.61023222e-04,
       1.41378618e-04, 1.42170142e-04, 1.40056138e-04, 1.52227800e-04,
       1.48567625e-04, 1.46743180e-04, 1.49176861e-04, 8.87775914e-04,
       1.42483296e-04, 1.48577151e-04, 1.33951109e-04, 1.41205665e-04,
       1.31380778e-04, 8.53356568e-04, 1.43658526e-04, 1.49068649e-04,
       1.54091841e-04, 1.24132370e-04, 1.28171185e-04, 1.20021490e-04,
       7.10331835e-04, 1.26077123e-04, 1.04956425e-04, 1.06973243e-04,
       1.44094625e-04, 1.42560042e-04, 1.40611252e-04, 8.53617067e-04,
       1.43011238e-04, 1.41237253e-04, 1.42102657e-04, 1.47594570e-04,
       1.49947409e-04, 1.43011238e-04, 8.75665154e-04, 1.48146669e-04,
       1.42091338e-04, 1.44873929e-04, 1.52029764e-04, 1.53454888e-04,
       1.50925810e-04, 9.11189839e-04, 1.53566068e-04, 1.49999087e-04,
       1.51214221e-04, 1.53745702e-04, 1.53054471e-04, 1.53173839e-04,
       1.53566068e-04, 9.19486933e-04, 1.53122927e-04, 1.52823925e-04,
       1.47684427e-04, 1.47164084e-04, 8.88744803e-04, 1.47765517e-04,
       1.48829915e-04, 1.47858059e-04, 1.49442800e-04, 1.48734218e-04,
       1.49113528e-04, 1.47765517e-04, 8.84764946e-04, 1.47793509e-04,
       1.45854745e-04, 1.45503429e-04, 1.50485811e-04, 1.51660201e-04,
       1.47793509e-04, 8.93844499e-04, 1.50247379e-04, 1.46189059e-04,
       1.47468539e-04, 1.44276245e-04, 1.40348558e-04, 1.48829915e-04,
       8.66639945e-04, 1.40052726e-04, 1.48991001e-04, 1.44141499e-04,
       1.50631887e-04, 1.50187166e-04, 1.49999087e-04, 8.96806574e-04,
       1.48765651e-04, 1.49005749e-04, 1.48217034e-04, 1.49087539e-04,
       1.48120333e-04, 1.48765651e-04, 8.87036293e-04, 1.46795721e-04,
       1.47634887e-04, 1.46632163e-04, 1.46893236e-04, 1.46029017e-04,
       1.46795721e-04, 8.73982684e-04, 1.44442954e-04, 1.45655151e-04,
       1.44166604e-04, 1.45113162e-04, 1.43669013e-04, 1.44442954e-04,
       8.59980858e-04, 1.42303727e-04, 1.42573540e-04, 1.41878462e-04,
       1.42483614e-04, 1.41908122e-04, 1.40533027e-04, 1.42303727e-04,
       8.47931525e-04, 1.39753913e-04, 1.40949121e-04, 1.36497245e-04,
       1.35477821e-04, 1.43811568e-04, 8.44404521e-04, 1.38265142e-04,
       1.47439874e-04, 1.42912872e-04, 1.37944169e-04, 1.39926030e-04,
       7.76954864e-04, 1.32879191e-04, 1.38559003e-04, 1.11974260e-04,
       1.15672211e-04, 1.35609209e-04, 1.36536904e-04, 1.38265142e-04,
       1.32879191e-04, 8.20437489e-04, 1.40802649e-04, 1.36344394e-04,
       1.41554930e-04, 1.45752101e-04, 1.38559003e-04, 8.75796324e-04,
       1.48083170e-04, 1.48260061e-04, 1.53587058e-04, 1.45759646e-04,
       1.44593714e-04, 1.41378618e-04, 8.52455153e-04, 1.44121951e-04,
       1.36345979e-04, 1.40255245e-04, 1.38886678e-04, 1.38239148e-04,
       8.19568652e-04, 1.39131042e-04, 1.34399625e-04, 1.32414136e-04,
       1.36498023e-04, 1.38001074e-04, 1.37647949e-04, 1.39131042e-04,
       8.41267363e-04, 1.41366789e-04, 1.41265819e-04, 1.43854690e-04,
       1.39778229e-04, 1.40394450e-04, 1.41366789e-04, 8.52790393e-04,
       1.41253593e-04, 1.45363375e-04, 1.44633958e-04, 1.39915300e-04,
       1.45208233e-04, 1.41253593e-04, 8.08197174e-04, 1.34839782e-04,
       1.29343504e-04, 1.17636760e-04, 1.48752219e-04, 1.51659377e-04,
       1.34839782e-04, 1.06254534e-03, 1.68963884e-04, 1.17988950e-04,
       1.86422427e-04, 1.53918699e-04, 1.52157675e-04, 1.57863747e-04,
       9.17015614e-04, 1.48521551e-04, 1.56926435e-04, 1.49423697e-04,
       1.52122510e-04, 1.47502897e-04, 1.45210195e-04, 1.44674792e-04,
       1.48521551e-04, 8.77242358e-04, 1.44889434e-04, 1.46443490e-04,
       1.70147317e-04, 1.92871625e-04, 1.56926435e-04, 1.01841211e-03,
       1.85759949e-04, 1.53298789e-04, 1.59407994e-04, 1.39074537e-04,
       1.42043555e-04, 8.35449442e-04, 1.35005781e-04, 1.42641721e-04,
       1.36008462e-04, 1.40675384e-04, 1.35492641e-04, 1.31254527e-04,
       1.26077123e-04, 1.35005781e-04, 7.91521116e-04, 1.29500250e-04,
       1.34190795e-04, 1.44879979e-04, 1.46596714e-04, 1.42641721e-04,
       8.71555800e-04, 1.46959573e-04, 1.43938348e-04, 1.46539465e-04,
       1.47903402e-04, 1.46165987e-04, 1.46959573e-04, 8.71772702e-04,
       1.43054026e-04, 1.44928923e-04, 1.42760791e-04, 1.28217172e-04,
       1.35105564e-04, 6.52654628e-04, 1.37426736e-04, 1.22988410e-04,
       1.28916746e-04, 1.53286919e-04, 1.75948127e-04, 1.37426736e-04,
       9.24978124e-04, 1.69632194e-04, 1.39569949e-04, 1.49114199e-04,
       1.47878366e-04, 1.47172601e-04, 9.47162872e-04, 1.54261977e-04,
       1.59783971e-04, 1.65919099e-04, 1.72146857e-04, 1.50104971e-04,
       1.44450352e-04, 1.54261977e-04, 9.41924103e-04, 1.58847798e-04,
       1.62009283e-04, 1.72249722e-04, 1.35770201e-04, 1.30370631e-04,
       1.58847798e-04, 7.83673987e-04, 1.11461880e-04, 1.44604913e-04,
       1.02618565e-04, 1.24249084e-04, 1.25065193e-04, 1.11461880e-04,
       7.24683273e-04, 1.26796126e-04, 1.11296426e-04, 1.25814563e-04,
       1.47583416e-04, 1.57426563e-04, 1.59783971e-04, 8.65798868e-04,
       1.42481122e-04, 1.42834896e-04, 1.15688900e-04, 1.25794886e-04,
       1.36902892e-04, 1.26796126e-04, 7.97643414e-04, 1.43912435e-04,
       1.29114387e-04, 1.35122689e-04, 1.54126202e-04, 1.49208745e-04,
       1.43912435e-04, 9.10278669e-04, 1.52328564e-04, 1.48684569e-04,
       1.62018155e-04, 1.38452376e-04, 1.15882523e-04, 1.23212433e-04,
       1.52328564e-04, 8.52450125e-04, 1.58013975e-04, 1.64560253e-04,
       1.49848010e-04, 1.22305377e-04, 1.32255825e-04, 8.46609343e-04,
       1.44401882e-04, 1.56597412e-04, 1.41200836e-04, 1.51651115e-04,
       1.55832640e-04, 1.44401882e-04, 9.02584888e-04, 1.53493349e-04,
       1.46147980e-04, 1.51057922e-04, 1.59205488e-04, 1.55840119e-04,
       1.55110878e-04, 9.32475429e-04, 1.54841235e-04, 1.55556360e-04,
       1.51921349e-04, 1.89162420e-04, 1.62732392e-04, 9.60633200e-04,
       1.68340779e-04, 1.49243429e-04, 1.44361465e-04, 1.46792716e-04,
       1.61435634e-04, 1.64309269e-04, 1.64039814e-04, 9.81943841e-04,
       1.63745504e-04, 1.65790783e-04, 1.62622838e-04, 1.60220367e-04,
       1.61641681e-04, 1.54841235e-04, 9.41591717e-04, 1.58904464e-04,
       1.51671028e-04, 1.54312942e-04, 1.61824022e-04, 1.20567599e-04,
       1.68340779e-04, 8.53836127e-04, 1.35303675e-04, 1.29790366e-04,
       1.38009686e-04, 1.46483169e-04, 1.44503607e-04, 1.46838603e-04,
       8.65067855e-04, 1.41621966e-04, 1.44428793e-04, 1.41191718e-04,
       1.59130237e-04, 1.52684701e-04, 1.54741038e-04, 9.12831013e-04,
       1.46503631e-04, 1.50781638e-04, 1.48989768e-04, 1.71301495e-04,
       1.67650837e-04, 1.51869284e-04, 9.39491267e-04, 1.53136065e-04,
       1.45308073e-04, 1.50225512e-04, 1.52353861e-04, 1.55881097e-04,
       9.27042993e-04, 1.53862172e-04, 1.52277872e-04, 1.58889764e-04,
       1.53778227e-04, 1.55218137e-04, 1.53413783e-04, 1.49243429e-04,
       1.53862172e-04, 9.13038769e-04, 1.52808970e-04, 1.48492277e-04,
       1.53101408e-04, 1.47892897e-04, 1.44361465e-04, 1.35303675e-04,
       1.52808970e-04, 8.72011085e-04, 1.38542670e-04, 1.40564260e-04,
       1.42483296e-04, 1.43658526e-04, 7.13466443e-04, 1.42729003e-04,
       1.44031358e-04, 1.34367638e-04, 1.25655489e-04, 1.49068649e-04,
       1.29790366e-04, 8.38834045e-04, 1.44692884e-04, 1.55259019e-04,
       1.41418217e-04, 1.42808243e-04, 1.41237253e-04, 8.48659042e-04,
       1.42776703e-04, 1.40134602e-04, 1.40284025e-04, 1.39358678e-04,
       1.46480519e-04, 1.42776703e-04, 8.65922095e-04, 1.53469364e-04,
       1.40133007e-04, 1.43703826e-04, 1.39224855e-04, 1.04815298e-04,
       1.53469364e-04, 7.51946839e-04, 8.67133843e-05, 1.50035180e-04,
       1.17688758e-04, 1.04373684e-04, 1.04956425e-04, 8.67133843e-05,
       4.67807263e-04, 8.72951634e-05, 8.44686076e-05, 1.39557565e-04,
       1.48794182e-04, 8.28290990e-04, 1.28412047e-04, 1.48390031e-04,
       1.36939266e-04, 1.26197898e-04, 1.32951248e-04, 1.36065333e-04,
       1.28412047e-04, 7.94958658e-04, 1.35886689e-04, 1.28641392e-04,
       1.33001949e-04, 1.37356436e-04, 1.39571987e-04, 1.35886689e-04,
       8.34132286e-04, 1.42533853e-04, 1.37175854e-04, 1.41607467e-04,
       1.41966402e-04, 1.41451848e-04, 1.42408326e-04, 1.42533853e-04,
       8.54653535e-04, 1.43341067e-04, 1.42952039e-04, 1.48702391e-04,
       1.48146669e-04, 1.48390031e-04, 8.90334831e-04, 1.46341905e-04,
       1.50123119e-04, 1.48630715e-04, 1.42675020e-04, 1.41329113e-04,
       1.43341067e-04, 8.50060511e-04, 1.40500707e-04, 1.41854528e-04,
       1.40360075e-04, 1.52204005e-04, 1.53122927e-04, 1.50247379e-04,
       9.05900395e-04, 1.52002815e-04, 1.48743377e-04, 1.49579891e-04,
       1.40386056e-04, 1.40052726e-04, 1.40500707e-04, 8.40356307e-04,
       1.40060906e-04, 1.39731192e-04, 1.39624721e-04, 1.38621137e-04,
       1.39753913e-04, 1.34399625e-04, 8.15819401e-04, 1.38191775e-04,
       1.31096788e-04, 1.33756162e-04, 1.52496724e-04, 1.50415733e-04,
       1.50783093e-04, 9.01946735e-04, 1.49176446e-04, 1.49300198e-04,
       1.49774541e-04, 1.47461208e-04, 1.48083170e-04, 1.49176446e-04,
       9.04007976e-04, 1.52882147e-04, 1.51660633e-04, 1.54744372e-04,
       1.50608716e-04, 1.53222966e-04, 1.49300198e-04, 9.09421059e-04,
       1.53972482e-04, 1.50129332e-04, 1.52187365e-04, 1.52718000e-04,
       1.53175989e-04, 1.53972482e-04, 9.21436779e-04, 1.54416963e-04,
       1.53583427e-04, 1.53569919e-04, 1.53160438e-04, 1.50656460e-04,
       1.54416963e-04, 9.20914074e-04, 1.52194430e-04, 1.55102168e-04,
       1.55383614e-04, 1.47419489e-04, 1.44121951e-04, 1.52194430e-04,
       8.90406064e-04, 1.44323068e-04, 1.53460474e-04, 1.48886651e-04,
       1.52736837e-04, 1.47439874e-04, 1.68963884e-04, 9.39359270e-04,
       1.49117281e-04, 1.67825517e-04, 1.53275877e-04, 1.65476946e-04,
       1.85759949e-04, 1.22988410e-04, 1.05879905e-03, 1.61418806e-04,
       1.28947301e-04, 1.52387271e-04, 1.41820366e-04, 1.43577604e-04,
       1.44889434e-04, 1.43054026e-04, 8.61315689e-04, 1.44484240e-04,
       1.42596001e-04, 1.42714384e-04, 1.68297524e-04, 1.69632194e-04,
       1.58013975e-04, 1.00809226e-03, 1.59651977e-04, 1.76096631e-04,
       1.76399962e-04, 1.71321762e-04, 1.42481122e-04, 1.56597412e-04,
       8.62842373e-04, 1.23800872e-04, 1.36298553e-04, 1.32342653e-04,
       1.54809038e-04, 1.53493349e-04, 1.55556360e-04, 9.23619037e-04,
       1.52934371e-04, 1.54066109e-04, 1.52759809e-04, 1.51491756e-04,
       1.46792716e-04, 1.52277872e-04, 1.48492277e-04, 9.01160828e-04,
       1.51509826e-04, 1.50596381e-04, 1.59285403e-04, 1.63745504e-04,
       1.51509826e-04, 9.36590061e-04, 1.58224191e-04, 1.50785436e-04,
       1.53039700e-04, 1.64239108e-04, 1.58904464e-04, 9.73942600e-04,
       1.70947313e-04, 1.52157810e-04, 1.70574382e-04, 1.57119523e-04,
       1.68556872e-04, 1.65790783e-04, 1.70947313e-04, 1.01522983e-03,
       1.67042668e-04, 1.72015375e-04, 1.70876819e-04, 1.50696226e-04,
       1.45370422e-04, 1.48577151e-04, 1.41621966e-04, 1.42729003e-04,
       8.70094390e-04, 1.41099621e-04, 1.44364029e-04, 1.38009686e-04,
       1.44428793e-04, 1.38542670e-04, 1.44692884e-04, 8.55328695e-04,
       1.45290633e-04, 1.47321280e-04, 1.47483578e-04, 1.46503631e-04,
       8.80809386e-04, 1.45899103e-04, 1.47184310e-04, 1.46417483e-04,
       1.44224343e-04, 1.43057815e-04, 1.45899103e-04, 8.58770079e-04,
       1.41880585e-04, 1.43280112e-04, 1.40428121e-04, 1.40734484e-04,
       1.42170142e-04, 1.41880585e-04, 8.25534575e-04, 1.35921810e-04,
       1.36110993e-04, 1.28716561e-04, 1.55000983e-04, 1.53136065e-04,
       9.19563533e-04, 1.54462095e-04, 1.51648769e-04, 1.52183606e-04,
       1.53132014e-04, 1.54774906e-04, 1.50781638e-04, 1.54462095e-04,
       9.10324336e-04, 1.48231247e-04, 1.52156014e-04, 1.49918435e-04,
       1.38542123e-04, 1.45308073e-04, 8.33261241e-04, 1.31665289e-04,
       1.43849143e-04, 1.39657259e-04, 1.34239354e-04, 1.39715158e-04,
       1.41120141e-04, 1.31665289e-04, 8.11604866e-04, 1.36645531e-04,
       1.29441744e-04, 1.33017004e-04, 1.34191698e-04, 1.48477788e-04,
       1.36645531e-04, 7.80504442e-04, 1.31020283e-04, 1.25040240e-04,
       1.05128902e-04, 1.64088338e-04, 1.58889764e-04, 1.31020283e-04,
       9.94668787e-04, 1.56079841e-04, 1.00229711e-04, 1.55285414e-04,
       1.29075436e-04, 1.40056138e-04, 1.36345979e-04, 1.35921810e-04,
       8.09755744e-04, 1.32416501e-04, 1.32405234e-04, 1.32610083e-04,
       1.54091841e-04, 1.41191718e-04, 1.44031358e-04, 1.55259019e-04,
       1.41099621e-04, 1.45290633e-04, 8.80964190e-04, 1.06973243e-04,
       1.29500250e-04, 8.72951634e-05, 7.06765670e-04, 1.32854959e-04,
       1.13554633e-04, 1.36587422e-04, 1.42102657e-04, 1.42091338e-04,
       1.40134602e-04, 8.41528001e-04, 1.40929966e-04, 1.37731378e-04,
       1.38538061e-04, 1.44873929e-04, 1.46341905e-04, 1.40929966e-04,
       8.60292882e-04, 1.44925628e-04, 1.40346883e-04, 1.42874570e-04,
       1.51214221e-04, 1.52823925e-04, 1.49005749e-04, 1.52002815e-04,
       9.02632548e-04, 1.48099237e-04, 1.49486600e-04, 1.47858059e-04,
       1.45854745e-04, 8.66198195e-04, 1.46935600e-04, 1.42627850e-04,
       1.42613189e-04, 1.40308753e-04, 1.49442800e-04, 1.48991001e-04,
       1.46935600e-04, 8.82095156e-04, 1.46940082e-04, 1.44664696e-04,
       1.45120977e-04, 1.45503429e-04, 1.46189059e-04, 1.42627850e-04,
       8.59508834e-04, 1.43600288e-04, 1.40650700e-04, 1.40937508e-04,
       1.47468539e-04, 1.48743377e-04, 1.43600288e-04, 8.71198000e-04,
       1.46585661e-04, 1.41544800e-04, 1.43255335e-04, 1.44141499e-04,
       1.40060906e-04, 1.46940082e-04, 8.58449563e-04, 1.40203138e-04,
       1.44880144e-04, 1.42223793e-04, 1.48217034e-04, 1.47634887e-04,
       1.48099237e-04, 8.83170822e-04, 1.46307261e-04, 1.47061117e-04,
       1.45851286e-04, 1.46632163e-04, 1.45655151e-04, 1.46307261e-04,
       8.70642588e-04, 1.43892558e-04, 1.44796573e-04, 1.43358882e-04,
       1.44166604e-04, 1.42573540e-04, 1.43892558e-04, 8.54257680e-04,
       1.41094341e-04, 1.41867894e-04, 1.40662742e-04, 1.41878462e-04,
       1.40949121e-04, 1.38191775e-04, 1.41094341e-04, 8.36810096e-04,
       1.36361586e-04, 1.38334811e-04, 1.42912872e-04, 1.40802649e-04,
       1.49117281e-04, 8.76799919e-04, 1.41234018e-04, 1.54253070e-04,
       1.48480030e-04, 1.11974260e-04, 1.36344394e-04, 1.41234018e-04,
       6.01745343e-04, 9.39961458e-05, 1.18196525e-04, 1.15672211e-04,
       1.48260061e-04, 9.39961458e-05, 8.24872593e-04, 1.58482021e-04,
       1.38282064e-04, 1.70180089e-04, 1.53587058e-04, 1.52882147e-04,
       1.58482021e-04, 9.53253570e-04, 1.56957753e-04, 1.66081973e-04,
       1.65262616e-04, 1.40255245e-04, 1.44323068e-04, 1.32416501e-04,
       8.17382568e-04, 1.43531492e-04, 1.24904547e-04, 1.31951715e-04,
       1.32414136e-04, 1.31096788e-04, 7.99115679e-04, 1.35255463e-04,
       1.31283971e-04, 1.35859733e-04, 1.33205589e-04, 1.36498023e-04,
       1.41265819e-04, 1.35255463e-04, 8.39446034e-04, 1.44778930e-04,
       1.38446064e-04, 1.43201735e-04, 1.43854690e-04, 1.45363375e-04,
       1.44778930e-04, 9.26975124e-04, 1.62164499e-04, 1.57468373e-04,
       1.73345257e-04, 1.44633958e-04, 1.29343504e-04, 1.62164499e-04,
       8.95055149e-04, 1.32799527e-04, 1.70094089e-04, 1.56019571e-04,
       1.17636760e-04, 1.17988950e-04, 1.32799527e-04, 6.23263769e-04,
       1.19194490e-04, 1.35644041e-04, 1.86422427e-04, 1.67825517e-04,
       9.44381489e-04, 1.68491112e-04, 1.45103900e-04, 1.35874116e-04,
       1.40664417e-04, 1.49423697e-04, 1.46443490e-04, 1.44484240e-04,
       8.79653804e-04, 1.49087193e-04, 1.44296218e-04, 1.45918965e-04,
       1.52122510e-04, 1.53298789e-04, 1.49087193e-04, 8.98196406e-04,
       1.49980909e-04, 1.46994784e-04, 1.46712220e-04, 1.59407994e-04,
       1.61418806e-04, 1.49980909e-04, 9.03295028e-04, 1.48221183e-04,
       1.44018337e-04, 1.40247799e-04, 1.36008462e-04, 1.34190795e-04,
       1.32854959e-04, 8.13890866e-04, 1.38389485e-04, 1.34365153e-04,
       1.38082012e-04, 1.40675384e-04, 1.43938348e-04, 1.38389485e-04,
       8.53739980e-04, 1.46087692e-04, 1.40106914e-04, 1.44542157e-04,
       1.46539465e-04, 1.44928923e-04, 1.46087692e-04, 8.68320185e-04,
       1.42948777e-04, 1.44671850e-04, 1.43143478e-04, 1.42760791e-04,
       1.42596001e-04, 1.42948777e-04, 8.53752560e-04, 1.41641401e-04,
       1.42474088e-04, 1.41331502e-04, 1.28916746e-04, 1.39569949e-04,
       1.28947301e-04, 8.08823129e-04, 1.43183779e-04, 1.32311039e-04,
       1.35894315e-04, 1.49114199e-04, 1.59651977e-04, 1.43183779e-04,
       9.05472715e-04, 1.60438463e-04, 1.43183695e-04, 1.49900601e-04,
       1.65919099e-04, 1.62009283e-04, 8.75775337e-04, 1.48652827e-04,
       1.43650421e-04, 1.27225558e-04, 1.28318148e-04, 1.72146857e-04,
       1.42834896e-04, 1.48652827e-04, 1.00033177e-03, 1.13582784e-04,
       1.37052517e-04, 1.35234450e-04, 1.50827433e-04, 1.72249722e-04,
       1.44604913e-04, 1.43650421e-04, 1.00948955e-03, 1.23076145e-04,
       1.33120468e-04, 1.45885193e-04, 1.46902691e-04, 1.02618565e-04,
       1.11296426e-04, 1.23076145e-04, 6.09454617e-04, 1.32521992e-04,
       1.39941489e-04, 1.25814563e-04, 1.29114387e-04, 1.32521992e-04,
       8.01797878e-04, 1.35667288e-04, 1.39417087e-04, 1.39262560e-04,
       1.15688900e-04, 1.23800872e-04, 1.13582784e-04, 6.01714611e-04,
       1.27317680e-04, 1.21324375e-04, 1.35122689e-04, 1.48684569e-04,
       1.35667288e-04, 9.05426079e-04, 1.68165913e-04, 1.49424437e-04,
       1.68361183e-04, 1.62018155e-04, 1.64560253e-04, 1.76096631e-04,
       1.68165913e-04, 1.04191245e-03, 1.88230060e-04, 1.82841440e-04,
       1.41200836e-04, 1.46147980e-04, 1.36298553e-04, 8.54490948e-04,
       1.46263440e-04, 1.39781198e-04, 1.44798942e-04, 1.51057922e-04,
       1.52934371e-04, 1.46263440e-04, 9.17798237e-04, 1.56037314e-04,
       1.51803883e-04, 1.59701307e-04, 1.51921349e-04, 1.51671028e-04,
       1.54066109e-04, 9.06768571e-04, 1.48875938e-04, 1.52586999e-04,
       1.47647149e-04, 1.62622838e-04, 1.58224191e-04, 1.67042668e-04,
       9.72187365e-04, 1.56859127e-04, 1.66611718e-04, 1.60826824e-04,
       1.54312942e-04, 1.52157810e-04, 1.48875938e-04, 8.90979630e-04,
       1.43497477e-04, 1.47679396e-04, 1.44456067e-04, 1.48989768e-04,
       1.47184310e-04, 1.48231247e-04, 8.85805760e-04, 1.46285385e-04,
       1.47627726e-04, 1.47487323e-04, 1.50225512e-04, 1.51648769e-04,
       1.43849143e-04, 8.79930658e-04, 1.48358793e-04, 1.41381023e-04,
       1.44467416e-04, 1.53778227e-04, 1.50596381e-04, 1.50785436e-04,
       1.56079841e-04, 9.21697801e-04, 1.54172679e-04, 1.56285238e-04,
       1.40284025e-04, 1.40133007e-04, 1.37731378e-04, 8.26199103e-04,
       1.36947160e-04, 1.35868367e-04, 1.35235167e-04, 1.43703826e-04,
       1.50035180e-04, 1.36947160e-04, 8.38854460e-04, 1.45477714e-04,
       1.30245690e-04, 1.32444890e-04, 1.17688758e-04, 8.44686076e-05,
       1.13554633e-04, 1.45477714e-04, 8.61452810e-04, 1.38409153e-04,
       1.32394671e-04, 1.29459273e-04, 1.36939266e-04, 1.50123119e-04,
       8.16082003e-04, 1.24110457e-04, 1.47695204e-04, 1.23406531e-04,
       1.33807426e-04, 1.26197898e-04, 1.28641392e-04, 1.24110457e-04,
       7.74604873e-04, 1.31325230e-04, 1.29779308e-04, 1.34550588e-04,
       1.33001949e-04, 1.37175854e-04, 1.31325230e-04, 8.16654387e-04,
       1.39977256e-04, 1.35347989e-04, 1.39826109e-04, 1.41607467e-04,
       1.42952039e-04, 1.41854528e-04, 1.39977256e-04, 8.45787356e-04,
       1.39964384e-04, 1.39431682e-04, 1.48630715e-04, 1.44925628e-04,
       1.47695204e-04, 8.58622158e-04, 1.41868710e-04, 1.38811654e-04,
       1.36690246e-04, 1.40360075e-04, 1.39731192e-04, 1.39964384e-04,
       8.36782258e-04, 1.38874291e-04, 1.39156762e-04, 1.38695553e-04,
       1.49579891e-04, 1.49486600e-04, 1.46585661e-04, 1.47061117e-04,
       8.82566623e-04, 1.44642304e-04, 1.45211050e-04, 1.39624721e-04,
       1.40203138e-04, 1.38874291e-04, 8.37297320e-04, 1.40331922e-04,
       1.38692017e-04, 1.39571230e-04, 1.33756162e-04, 1.36361586e-04,
       1.31283971e-04, 7.97661616e-04, 1.34980802e-04, 1.30093844e-04,
       1.31185252e-04, 1.49774541e-04, 1.51660633e-04, 1.50129332e-04,
       9.08687513e-04, 1.53162799e-04, 1.51406416e-04, 1.52553792e-04,
       1.54744372e-04, 1.56957753e-04, 1.53162799e-04, 9.35851440e-04,
       1.59096715e-04, 1.54084633e-04, 1.57805168e-04, 1.52187365e-04,
       1.53583427e-04, 1.51406416e-04, 9.15423801e-04, 1.53312629e-04,
       1.52211906e-04, 1.52722057e-04, 1.53569919e-04, 1.55102168e-04,
       1.53312629e-04, 9.23371459e-04, 1.55273147e-04, 1.52842664e-04,
       1.53270932e-04, 1.55383614e-04, 1.53460474e-04, 1.55273147e-04,
       9.22709163e-04, 1.51341592e-04, 1.54629375e-04, 1.52620960e-04,
       1.48886651e-04, 1.43531492e-04, 1.51341592e-04, 8.79214404e-04,
       1.42336645e-04, 1.48383906e-04, 1.44734118e-04, 1.53918699e-04,
       1.19194490e-04, 1.68491112e-04, 8.61535139e-04, 1.31218054e-04,
       1.45630816e-04, 1.43081967e-04, 1.53275877e-04, 1.54253070e-04,
       1.45103900e-04, 8.83028210e-04, 1.50431911e-04, 1.37806317e-04,
       1.42157135e-04, 1.52387271e-04, 1.48221183e-04, 8.43136214e-04,
       1.36951334e-04, 1.37892630e-04, 1.31288205e-04, 1.36395590e-04,
       1.42714384e-04, 1.44296218e-04, 1.41641401e-04, 8.59881593e-04,
       1.44630999e-04, 1.42595414e-04, 1.44003177e-04, 1.41820366e-04,
       1.32311039e-04, 1.36951334e-04, 8.01390397e-04, 1.29919106e-04,
       1.28955727e-04, 1.31432825e-04, 1.76399962e-04, 1.60438463e-04,
       1.88230060e-04, 1.00414205e-03, 1.54403452e-04, 1.71267610e-04,
       1.53402501e-04, 1.32342653e-04, 1.27317680e-04, 1.39781198e-04,
       8.12333090e-04, 1.32462531e-04, 1.41361361e-04, 1.39067666e-04,
       1.52759809e-04, 1.56037314e-04, 1.52586999e-04, 9.61039283e-04,
       1.65178679e-04, 1.62581153e-04, 1.71895329e-04, 1.53039700e-04,
       1.56859127e-04, 1.54172679e-04, 9.21907228e-04, 1.50965754e-04,
       1.56948159e-04, 1.49921808e-04, 1.70574382e-04, 1.72015375e-04,
       9.79375766e-04, 1.59098056e-04, 1.69061213e-04, 1.50052289e-04,
       1.58574452e-04, 1.57119523e-04, 1.43497477e-04, 1.59098056e-04,
       7.51755478e-04, 1.43255312e-04, 1.48785110e-04, 1.70876819e-04,
       1.66611718e-04, 1.69061213e-04, 9.98206065e-04, 1.64199560e-04,
       1.64026654e-04, 1.63430102e-04, 1.46417483e-04, 1.43280112e-04,
       1.46285385e-04, 8.65981991e-04, 1.40747929e-04, 1.45825584e-04,
       1.43425499e-04, 1.40428121e-04, 1.36110993e-04, 1.40747929e-04,
       8.22958827e-04, 1.31564626e-04, 1.39486666e-04, 1.34620492e-04,
       1.28716561e-04, 1.32405234e-04, 1.31564626e-04, 7.38323079e-04,
       1.17647814e-04, 1.22797165e-04, 1.05191679e-04, 1.52183606e-04,
       1.48358793e-04, 9.02477303e-04, 1.53707330e-04, 1.46350655e-04,
       1.52704391e-04, 1.49172529e-04, 1.53132014e-04, 1.52156014e-04,
       1.53707330e-04, 9.16326312e-04, 1.51355886e-04, 1.53675256e-04,
       1.52299812e-04, 1.49918435e-04, 1.47627726e-04, 1.51355886e-04,
       8.99982296e-04, 1.48444212e-04, 1.51628501e-04, 1.51007536e-04,
       1.39657259e-04, 1.41381023e-04, 8.09290983e-04, 1.33040202e-04,
       1.40391868e-04, 1.30224266e-04, 1.24596365e-04, 1.34239354e-04,
       1.29441744e-04, 1.33040202e-04, 7.80845344e-04, 1.30245488e-04,
       1.25937830e-04, 1.27940726e-04, 1.33017004e-04, 1.25040240e-04,
       1.30245488e-04, 7.78707780e-04, 1.19261745e-04, 1.34068162e-04,
       1.37075141e-04, 1.05128902e-04, 1.00229711e-04, 1.19261745e-04,
       4.38982912e-04, 1.14362554e-04, 1.55285414e-04, 1.56285238e-04,
       1.50965754e-04, 8.84669198e-04, 1.42069885e-04, 1.43979476e-04,
       1.36083431e-04, 1.32610083e-04, 1.24904547e-04, 1.17647814e-04,
       6.97305783e-04, 1.20435841e-04, 9.56069677e-05, 1.06100530e-04,
       1.36587422e-04, 1.34365153e-04, 1.38409153e-04, 8.16771045e-04,
       1.37232112e-04, 1.34565989e-04, 1.35611216e-04, 1.38538061e-04,
       1.40346883e-04, 1.35868367e-04, 8.27086907e-04, 1.40134328e-04,
       1.34871065e-04, 1.37328203e-04, 1.42874570e-04, 1.41868710e-04,
       1.40134328e-04, 8.42806631e-04, 1.38838591e-04, 1.39690155e-04,
       1.39400278e-04, 1.42613189e-04, 1.44664696e-04, 8.44294503e-04,
       1.38528697e-04, 1.42476678e-04, 1.38953858e-04, 1.37057384e-04,
       1.40308753e-04, 1.40650700e-04, 1.38528697e-04, 8.33420985e-04,
       1.39218720e-04, 1.37183020e-04, 1.37531095e-04, 1.45120977e-04,
       1.44880144e-04, 1.42476678e-04, 8.57250673e-04, 1.42524700e-04,
       1.40979660e-04, 1.41268514e-04, 1.40937508e-04, 1.41544800e-04,
       1.39218720e-04, 8.39836260e-04, 1.40913728e-04, 1.38066894e-04,
       1.39154610e-04, 1.43255335e-04, 1.44642304e-04, 1.40913728e-04,
       8.54934683e-04, 1.43567931e-04, 1.40644075e-04, 1.41911310e-04,
       1.42223793e-04, 1.40331922e-04, 1.42524700e-04, 8.45154082e-04,
       1.39815262e-04, 1.40537986e-04, 1.39720419e-04, 1.45851286e-04,
       1.44796573e-04, 1.45211050e-04, 1.43567931e-04, 8.64620148e-04,
       1.42890856e-04, 1.42302451e-04, 1.43358882e-04, 1.41867894e-04,
       1.42890856e-04, 8.46825615e-04, 1.39581877e-04, 1.40472048e-04,
       1.38654057e-04, 1.40662742e-04, 1.38334811e-04, 1.34980802e-04,
       1.39581877e-04, 8.24094158e-04, 1.34130391e-04, 1.36403534e-04,
       1.48480030e-04, 1.18196525e-04, 1.38282064e-04, 1.50431911e-04,
       1.05551090e-03, 1.82199870e-04, 1.49411155e-04, 1.68509342e-04,
       1.70180089e-04, 1.66081973e-04, 1.82199870e-04, 1.05977487e-03,
       1.80937538e-04, 1.78769810e-04, 1.81605593e-04, 1.65262616e-04,
       1.59096715e-04, 1.80937538e-04, 1.03765744e-03, 1.68860465e-04,
       1.84705636e-04, 1.78794465e-04, 1.31951715e-04, 1.42336645e-04,
       1.20435841e-04, 7.96362956e-04, 1.40318060e-04, 1.25911703e-04,
       1.35408992e-04, 1.35859733e-04, 1.38446064e-04, 8.33071921e-04,
       1.37075152e-04, 1.42056152e-04, 1.38620076e-04, 1.41014744e-04,
       1.33205589e-04, 1.30093844e-04, 1.37075152e-04, 8.02557833e-04,
       1.30763487e-04, 1.37309841e-04, 1.34109920e-04, 1.43201735e-04,
       1.57468373e-04, 1.42056152e-04, 9.26977534e-04, 1.70149484e-04,
       1.50137548e-04, 1.63964242e-04, 1.73345257e-04, 1.70094089e-04,
       1.70149484e-04, 1.16621251e-03, 1.56185910e-04, 1.68179366e-04,
       1.60362696e-04, 1.67895710e-04, 1.56019571e-04, 1.35644041e-04,
       1.31218054e-04, 1.56185910e-04, 8.53461129e-04, 1.34900613e-04,
       1.39492939e-04, 1.35874116e-04, 1.37806317e-04, 8.27760756e-04,
       1.36970659e-04, 1.38618396e-04, 1.39103401e-04, 1.39387867e-04,
       1.40664417e-04, 1.45630816e-04, 1.36970659e-04, 8.47454879e-04,
       1.42592820e-04, 1.40470202e-04, 1.41125964e-04, 1.45918965e-04,
       1.46994784e-04, 1.44630999e-04, 8.71348522e-04, 1.45635178e-04,
       1.44120119e-04, 1.44048479e-04, 1.46712220e-04, 1.44018337e-04,
       1.45635178e-04, 8.59339948e-04, 1.40280390e-04, 1.42677363e-04,
       1.40016460e-04, 1.40247799e-04, 1.37892630e-04, 1.40280390e-04,
       8.33160457e-04, 1.37363669e-04, 1.38968761e-04, 1.38407208e-04,
       1.38082012e-04, 1.40106914e-04, 1.37232112e-04, 8.40443693e-04,
       1.43586480e-04, 1.38553354e-04, 1.42882820e-04, 1.44542157e-04,
       1.44671850e-04, 1.43586480e-04, 8.60008858e-04, 1.42830060e-04,
       1.42632212e-04, 1.41746099e-04, 1.43143478e-04, 1.42474088e-04,
       1.42830060e-04, 8.49837875e-04, 1.41141883e-04, 1.40633577e-04,
       1.39614790e-04, 1.41331502e-04, 1.42595414e-04, 1.41141883e-04,
       8.46899638e-04, 1.41676363e-04, 1.40441953e-04, 1.39712522e-04,
       1.35894315e-04, 1.43183695e-04, 1.29919106e-04, 8.21476432e-04,
       1.41575680e-04, 1.33268221e-04, 1.37635415e-04, 1.49900601e-04,
       1.54403452e-04, 1.41575680e-04, 8.71970800e-04, 1.47880173e-04,
       1.38204626e-04, 1.40006268e-04, 1.27225558e-04, 1.33120468e-04,
       7.97012691e-04, 1.32189999e-04, 1.38219506e-04, 1.33061281e-04,
       1.33195879e-04, 1.28318148e-04, 1.37052517e-04, 1.32189999e-04,
       8.21562644e-04, 1.48056840e-04, 1.34406334e-04, 1.41538806e-04,
       1.35234450e-04, 1.21324375e-04, 1.32462531e-04, 8.59015314e-04,
       1.56136856e-04, 1.52046426e-04, 1.61810676e-04, 1.45885193e-04,
       1.39941489e-04, 1.39417087e-04, 8.56729548e-04, 1.38865195e-04,
       1.49558090e-04, 1.43062493e-04, 1.39262560e-04, 1.49424437e-04,
       1.38865195e-04, 7.25106988e-04, 1.53488090e-04, 1.44066707e-04,
       1.68361183e-04, 1.82841440e-04, 1.71267610e-04, 1.53488090e-04,
       9.76753058e-04, 1.52046999e-04, 1.48747736e-04, 1.44798942e-04,
       1.51803883e-04, 1.41361361e-04, 8.67237869e-04, 1.44072378e-04,
       1.44747614e-04, 1.40453691e-04, 1.59701307e-04, 1.65178679e-04,
       1.44072378e-04, 8.92457590e-04, 1.50813434e-04, 1.35714053e-04,
       1.36977737e-04, 1.47647149e-04, 1.47679396e-04, 1.62581153e-04,
       8.64752865e-04, 1.36075644e-04, 1.48653639e-04, 1.22115884e-04,
       1.60826824e-04, 1.56948159e-04, 1.64199560e-04, 9.52715938e-04,
       1.53517164e-04, 1.62013981e-04, 1.55210250e-04, 1.44456067e-04,
       1.43255312e-04, 1.36075644e-04, 7.92921615e-04, 1.31033284e-04,
       1.20971456e-04, 1.17129851e-04, 1.47487323e-04, 1.45825584e-04,
       1.48444212e-04, 8.86257771e-04, 1.46139883e-04, 1.49501679e-04,
       1.48859090e-04, 1.44467416e-04, 1.46350655e-04, 1.40391868e-04,
       8.62505672e-04, 1.46172603e-04, 1.40612817e-04, 1.44510313e-04,
       1.29075436e-04, 1.34068162e-04, 1.14362554e-04, 1.42069885e-04,
       8.19488527e-04, 1.53311825e-04, 1.46600666e-04, 1.35235167e-04,
       1.30245690e-04, 1.34871065e-04, 7.76953278e-04, 1.22613349e-04,
       1.30628123e-04, 1.23359884e-04, 1.32444890e-04, 1.32394671e-04,
       1.22613349e-04, 6.81514009e-04, 1.07231264e-04, 1.01080850e-04,
       8.57489841e-05, 1.29459273e-04, 1.34565989e-04, 1.07231264e-04,
       7.18614896e-04, 1.31983125e-04, 9.78650499e-05, 1.17510195e-04,
       1.23406531e-04, 1.29779308e-04, 8.04794497e-04, 1.26353110e-04,
       1.40688216e-04, 1.38302502e-04, 1.46264831e-04, 1.33807426e-04,
       1.38811654e-04, 1.26353110e-04, 7.86602052e-04, 1.34381251e-04,
       1.25112349e-04, 1.28136263e-04, 1.34550588e-04, 1.35347989e-04,
       1.40688216e-04, 8.32615789e-04, 1.39577807e-04, 1.42179499e-04,
       1.40271689e-04, 1.39826109e-04, 1.39431682e-04, 1.39156762e-04,
       1.39577807e-04, 8.36718428e-04, 1.39349725e-04, 1.39376342e-04,
       1.36690246e-04, 1.38838591e-04, 1.34381251e-04, 8.13981696e-04,
       1.36944029e-04, 1.33356573e-04, 1.33771006e-04, 1.38695553e-04,
       1.38692017e-04, 1.39349725e-04, 8.37577892e-04, 1.39702096e-04,
       1.40391298e-04, 1.40747205e-04, 1.39571230e-04, 1.39815262e-04,
       1.39702096e-04, 8.39772220e-04, 1.40025111e-04, 1.40289768e-04,
       1.40368753e-04, 1.31185252e-04, 1.34130391e-04, 1.30763487e-04,
       7.96286379e-04, 1.34685231e-04, 1.32272706e-04, 1.33249312e-04,
       1.52553792e-04, 1.54084633e-04, 1.52211906e-04, 9.19000607e-04,
       1.55681988e-04, 1.51264524e-04, 1.53203765e-04, 1.57805168e-04,
       1.68860465e-04, 1.55681988e-04, 9.73067112e-04, 1.75293670e-04,
       1.53434719e-04, 1.61991103e-04, 1.52722057e-04, 1.52842664e-04,
       1.51264524e-04, 9.09011938e-04, 1.50518453e-04, 1.51265460e-04,
       1.50398781e-04, 1.53270932e-04, 1.54629375e-04, 1.50518453e-04,
       9.12175452e-04, 1.52265837e-04, 1.50550957e-04, 1.50939898e-04,
       1.52620960e-04, 1.48383906e-04, 1.52265837e-04, 8.94722739e-04,
       1.45303913e-04, 1.49436496e-04, 1.46711627e-04, 1.44734118e-04,
       1.40318060e-04, 1.45303913e-04, 8.59834963e-04, 1.40882338e-04,
       1.44301025e-04, 1.44295508e-04, 1.43081967e-04, 1.34900613e-04,
       1.42592820e-04, 8.25830301e-04, 1.31164598e-04, 1.38668585e-04,
       1.35421717e-04, 1.42157135e-04, 1.49411155e-04, 1.38618396e-04,
       8.56514205e-04, 1.48274611e-04, 1.37825356e-04, 1.40227553e-04,
       1.31288205e-04, 1.28955727e-04, 7.97846449e-04, 1.37376334e-04,
       1.32554052e-04, 1.32591164e-04, 1.35080966e-04, 1.36395590e-04,
       1.37363669e-04, 1.37376334e-04, 8.27594976e-04, 1.39420340e-04,
       1.37981558e-04, 1.39057485e-04, 1.44003177e-04, 1.44120119e-04,
       1.41676363e-04, 8.54583555e-04, 1.42490552e-04, 1.40798048e-04,
       1.41495295e-04, 1.31432825e-04, 1.33268221e-04, 1.32554052e-04,
       8.05644359e-04, 1.34218920e-04, 1.37170435e-04, 1.36999906e-04,
       1.53402501e-04, 1.47880173e-04, 1.52046999e-04, 8.82205701e-04,
       1.40186115e-04, 1.47514234e-04, 1.41175678e-04, 1.46902691e-04,
       1.38219506e-04, 1.49558090e-04, 8.49434290e-04, 1.34998516e-04,
       1.42815938e-04, 1.36939549e-04, 1.50827433e-04, 1.48056840e-04,
       1.56136856e-04, 9.00494926e-04, 1.45833946e-04, 1.53586084e-04,
       1.46053768e-04, 1.39067666e-04, 1.52046426e-04, 1.44747614e-04,
       9.60123259e-04, 1.71688156e-04, 1.69305807e-04, 1.83267589e-04,
       1.71895329e-04, 1.50813434e-04, 1.48653639e-04, 8.55786625e-04,
       1.35789161e-04, 1.20208823e-04, 1.28426239e-04, 1.49921808e-04,
       1.43979476e-04, 1.53517164e-04, 8.73721581e-04, 1.36981202e-04,
       1.49957780e-04, 1.39364150e-04, 1.50052289e-04, 1.48785110e-04,
       1.31033284e-04, 8.15624553e-04, 1.49814445e-04, 1.09212721e-04,
       1.26726703e-04, 1.58574452e-04, 1.64026654e-04, 1.49814445e-04,
       9.43902884e-04, 1.62651548e-04, 1.50725442e-04, 1.58110343e-04,
       1.63430102e-04, 1.62013981e-04, 1.62651548e-04, 9.77017081e-04,
       1.62020415e-04, 1.63058024e-04, 1.63843012e-04, 1.43425499e-04,
       1.39486666e-04, 1.46139883e-04, 8.60458225e-04, 1.38533319e-04,
       1.48270294e-04, 1.44602564e-04, 1.34620492e-04, 1.22797165e-04,
       1.38533319e-04, 7.80521137e-04, 1.15352460e-04, 1.40287616e-04,
       1.28930084e-04, 1.05191679e-04, 9.56069677e-05, 1.15352460e-04,
       5.31491185e-04, 1.01266813e-04, 1.14073265e-04, 1.52704391e-04,
       1.53675256e-04, 9.16446722e-04, 1.51189571e-04, 1.53447348e-04,
       1.52071623e-04, 1.53358533e-04, 1.49172529e-04, 1.46172603e-04,
       1.51189571e-04, 8.90658045e-04, 1.45864750e-04, 1.50291744e-04,
       1.47966848e-04, 1.52299812e-04, 1.51628501e-04, 1.53447348e-04,
       9.13275303e-04, 1.50887950e-04, 1.53449890e-04, 1.51561803e-04,
       1.51007536e-04, 1.49501679e-04, 1.50887950e-04, 9.03273622e-04,
       1.50238569e-04, 1.50390706e-04, 1.51247182e-04, 1.30224266e-04,
       1.40612817e-04, 8.16977335e-04, 1.28987242e-04, 1.40266444e-04,
       1.37997958e-04, 1.38888608e-04, 1.24596365e-04, 1.25937830e-04,
       1.28987242e-04, 7.67652395e-04, 1.27234715e-04, 1.31995118e-04,
       1.28901125e-04, 1.27940726e-04, 1.37075141e-04, 1.53311825e-04,
       1.27234715e-04, 8.37149307e-04, 1.54264798e-04, 1.37322103e-04,
       1.36083431e-04, 1.46600666e-04, 1.36981202e-04, 1.54264798e-04,
       8.60499101e-04, 1.39901322e-04, 1.46667683e-04, 1.06100530e-04,
       1.25911703e-04, 1.01266813e-04, 7.26494606e-04, 1.35105068e-04,
       1.22041705e-04, 1.36068787e-04, 1.35611216e-04, 1.38553354e-04,
       1.31983125e-04, 8.11231261e-04, 1.39966150e-04, 1.30038264e-04,
       1.35079151e-04, 1.37328203e-04, 1.39690155e-04, 1.30628123e-04,
       8.06769180e-04, 1.39116341e-04, 1.26940047e-04, 1.33066312e-04,
       1.39400278e-04, 1.36944029e-04, 1.39116341e-04, 8.29699157e-04,
       1.36345014e-04, 1.39104286e-04, 1.38789209e-04, 1.38953858e-04,
       1.40979660e-04, 8.29462407e-04, 1.36234714e-04, 1.39999112e-04,
       1.37516830e-04, 1.35778234e-04, 1.37057384e-04, 1.37183020e-04,
       1.36234714e-04, 8.16933602e-04, 1.36101915e-04, 1.35307502e-04,
       1.35049067e-04, 1.37531095e-04, 1.38066894e-04, 1.36101915e-04,
       8.19509923e-04, 1.37032882e-04, 1.35190985e-04, 1.35586152e-04,
       1.41268514e-04, 1.40537986e-04, 1.39999112e-04, 8.38344262e-04,
       1.39269530e-04, 1.38634087e-04, 1.38635033e-04, 1.39154610e-04,
       1.40644075e-04, 1.37032882e-04, 8.28719473e-04, 1.39381964e-04,
       1.35823162e-04, 1.36682779e-04, 1.41911310e-04, 1.42302451e-04,
       1.40472048e-04, 1.39381964e-04, 8.39312069e-04, 1.37971620e-04,
       1.37272676e-04, 1.39720419e-04, 1.40025111e-04, 1.39269530e-04,
       8.35849608e-04, 1.39403363e-04, 1.38801023e-04, 1.38630162e-04,
       1.38654057e-04, 1.36403534e-04, 1.34685231e-04, 1.37971620e-04,
       8.17321068e-04, 1.34285380e-04, 1.35321246e-04, 1.68509342e-04,
       1.78769810e-04, 1.48274611e-04, 9.49576268e-04, 1.61460709e-04,
       1.44818084e-04, 1.47743713e-04, 1.81605593e-04, 1.84705636e-04,
       1.61460709e-04, 9.60513675e-04, 1.55781164e-04, 1.42870081e-04,
       1.34090492e-04, 1.78794465e-04, 1.75293670e-04, 1.55781164e-04,
       1.00490093e-03, 1.40821517e-04, 1.31390734e-04, 1.05478405e-04,
       1.17340972e-04, 1.35408992e-04, 1.40882338e-04, 1.35105068e-04,
       8.38367715e-04, 1.41973805e-04, 1.41801061e-04, 1.43196451e-04,
       1.38620076e-04, 1.37309841e-04, 8.29451113e-04, 1.38098159e-04,
       1.38363769e-04, 1.37741712e-04, 1.39317557e-04, 1.41014744e-04,
       1.50137548e-04, 1.38098159e-04, 8.49903285e-04, 1.46922824e-04,
       1.37014074e-04, 1.36715935e-04, 1.34109920e-04, 1.32272706e-04,
       1.38363769e-04, 8.13383837e-04, 1.33794156e-04, 1.38787843e-04,
       1.36055443e-04, 1.63964242e-04, 1.68179366e-04, 1.46922824e-04,
       8.87155881e-04, 1.47708654e-04, 1.31905045e-04, 1.28475750e-04,
       1.60362696e-04, 1.39492939e-04, 1.31164598e-04, 8.47892327e-04,
       1.51214970e-04, 1.33238254e-04, 1.32418870e-04, 1.39103401e-04,
       1.37825356e-04, 8.50817857e-04, 1.43053357e-04, 1.42075556e-04,
       1.44229971e-04, 1.44530215e-04, 1.39387867e-04, 1.40470202e-04,
       1.43053357e-04, 8.62233992e-04, 1.44529908e-04, 1.47199221e-04,
       1.47593437e-04, 1.41125964e-04, 1.38668585e-04, 1.44529908e-04,
       8.54826088e-04, 1.39944407e-04, 1.46342673e-04, 1.44214551e-04,
       1.44048479e-04, 1.42677363e-04, 1.42490552e-04, 8.49596994e-04,
       1.40256906e-04, 1.40493111e-04, 1.39630581e-04, 1.40016460e-04,
       1.38968761e-04, 1.40256906e-04, 8.35029956e-04, 1.38524748e-04,
       1.38973771e-04, 1.38289311e-04, 1.38407208e-04, 1.39420340e-04,
       1.38524748e-04, 8.33312516e-04, 1.39835505e-04, 1.38413544e-04,
       1.38711170e-04, 1.42882820e-04, 1.42632212e-04, 1.39966150e-04,
       8.42991278e-04, 1.40088961e-04, 1.38523858e-04, 1.38897277e-04,
       1.41746099e-04, 1.40633577e-04, 1.40088961e-04, 8.36930950e-04,
       1.38461059e-04, 1.38258317e-04, 1.37742937e-04, 1.39614790e-04,
       1.40441953e-04, 1.38461059e-04, 8.29860235e-04, 1.39600448e-04,
       1.35714879e-04, 1.36027105e-04, 1.39712522e-04, 1.40798048e-04,
       1.39600448e-04, 8.38172305e-04, 1.40720795e-04, 1.38652836e-04,
       1.38687657e-04, 1.37635415e-04, 1.38204626e-04, 1.34218920e-04,
       8.16022093e-04, 1.34172837e-04, 1.36202795e-04, 1.35587501e-04,
       1.40006268e-04, 1.40186115e-04, 1.34172837e-04, 7.99531932e-04,
       1.34139603e-04, 1.25620095e-04, 1.25407014e-04, 1.33061281e-04,
       1.34406334e-04, 7.85018478e-04, 1.29884908e-04, 1.33186721e-04,
       1.28217996e-04, 1.26261237e-04, 1.33195879e-04, 1.34998516e-04,
       1.29884908e-04, 7.80066556e-04, 1.30636470e-04, 1.25149854e-04,
       1.26200929e-04, 1.41538806e-04, 1.45833946e-04, 1.33186721e-04,
       8.26474406e-04, 1.40729643e-04, 1.30968754e-04, 1.34216536e-04,
       1.61810676e-04, 1.53586084e-04, 1.71688156e-04, 9.60361251e-04,
       1.50561781e-04, 1.67808169e-04, 1.54906386e-04, 1.43062493e-04,
       1.44066707e-04, 1.48747736e-04, 1.47514234e-04, 1.42815938e-04,
       8.68793758e-04, 1.42586649e-04, 1.40453691e-04, 1.35714053e-04,
       1.69305807e-04, 9.94174899e-04, 1.60502136e-04, 1.96131623e-04,
       1.92067588e-04, 1.36977737e-04, 1.35789161e-04, 1.60502136e-04,
       8.96506307e-04, 1.41798394e-04, 1.69477022e-04, 1.51961857e-04,
       1.22115884e-04, 1.20971456e-04, 1.20208823e-04, 6.02487613e-04,
       1.19442300e-04, 1.19749150e-04, 1.55210250e-04, 1.49957780e-04,
       1.62020415e-04, 9.59923167e-04, 1.59913340e-04, 1.64837994e-04,
       1.67983388e-04, 1.17129851e-04, 1.09212721e-04, 1.19442300e-04,
       5.66928288e-04, 1.14872651e-04, 1.06270766e-04, 1.48859090e-04,
       1.48270294e-04, 1.50238569e-04, 8.98150731e-04, 1.49103717e-04,
       1.51112559e-04, 1.50566502e-04, 1.44510313e-04, 1.45864750e-04,
       1.40266444e-04, 8.56513103e-04, 1.44527424e-04, 1.39218814e-04,
       1.42125358e-04, 1.23359884e-04, 1.01080850e-04, 1.26940047e-04,
       6.68384911e-04, 8.64995778e-05, 1.24332993e-04, 1.06171559e-04,
       8.57489841e-05, 9.78650499e-05, 8.64995778e-05, 4.74882088e-04,
       1.06081133e-04, 9.86873432e-05, 1.17510195e-04, 1.30038264e-04,
       1.06081133e-04, 7.24959108e-04, 1.30277085e-04, 1.14692273e-04,
       1.26360157e-04, 1.38302502e-04, 1.25112349e-04, 8.25379733e-04,
       1.52413900e-04, 1.22352807e-04, 1.52034557e-04, 1.35163617e-04,
       1.46264831e-04, 1.42179499e-04, 1.52413900e-04, 8.80887636e-04,
       1.43315958e-04, 1.50863029e-04, 1.45850419e-04, 1.28136263e-04,
       1.33356573e-04, 1.22352807e-04, 7.60829979e-04, 1.30371511e-04,
       1.21907216e-04, 1.24705609e-04, 1.40271689e-04, 1.39376342e-04,
       1.40391298e-04, 1.43315958e-04, 8.48765670e-04, 1.41690535e-04,
       1.43719848e-04, 1.33771006e-04, 1.36345014e-04, 1.30371511e-04,
       7.95443649e-04, 1.35032031e-04, 1.28918788e-04, 1.31005299e-04,
       1.40747205e-04, 1.40289768e-04, 1.41690535e-04, 8.46413268e-04,
       1.40447638e-04, 1.42011791e-04, 1.41226331e-04, 1.40368753e-04,
       1.39403363e-04, 1.40447638e-04, 8.35259422e-04, 1.38148245e-04,
       1.39112713e-04, 1.37778711e-04, 1.33249312e-04, 1.34285380e-04,
       1.33794156e-04, 8.02130747e-04, 1.33764152e-04, 1.34051910e-04,
       1.32985837e-04, 1.53203765e-04, 1.53434719e-04, 1.51265460e-04,
       9.07029122e-04, 1.50929635e-04, 1.49381161e-04, 1.48814383e-04,
       1.61991103e-04, 1.40821517e-04, 1.50929635e-04, 8.16040678e-04,
       1.02653690e-04, 1.43375546e-04, 1.16269188e-04, 1.50398781e-04,
       1.50550957e-04, 1.49381161e-04, 8.96061769e-04, 1.50326863e-04,
       1.47305241e-04, 1.48098766e-04, 1.50939898e-04, 1.49436496e-04,
       1.50326863e-04, 8.95704800e-04, 1.48851616e-04, 1.48060885e-04,
       1.48089041e-04, 1.46711627e-04, 1.44301025e-04, 1.48851616e-04,
       8.78103645e-04, 1.45698513e-04, 1.46641682e-04, 1.45899181e-04,
       1.44295508e-04, 1.41973805e-04, 1.45698513e-04, 8.65608703e-04,
       1.42917424e-04, 1.45591419e-04, 1.45132034e-04, 1.67895710e-04,
       1.47708654e-04, 1.51214970e-04, 8.61320213e-04, 1.30459728e-04,
       1.32304669e-04, 1.31736483e-04, 1.35421717e-04, 1.33238254e-04,
       1.39944407e-04, 8.29240217e-04, 1.35941592e-04, 1.43256800e-04,
       1.41437448e-04, 1.40227553e-04, 1.44818084e-04, 1.42075556e-04,
       8.78839182e-04, 1.56670146e-04, 1.42521892e-04, 1.52525951e-04,
       1.32591164e-04, 1.37170435e-04, 8.39014907e-04, 1.30746265e-04,
       1.44583147e-04, 1.42333143e-04, 1.51590754e-04, 1.35080966e-04,
       1.37981558e-04, 1.30746265e-04, 8.11374398e-04, 1.41551804e-04,
       1.29054429e-04, 1.36959376e-04, 1.39057485e-04, 1.39835505e-04,
       1.41551804e-04, 8.48222850e-04, 1.40708290e-04, 1.44345649e-04,
       1.42724116e-04, 1.41495295e-04, 1.40493111e-04, 1.40720795e-04,
       8.36999179e-04, 1.38519250e-04, 1.38485044e-04, 1.37285684e-04,
       1.36999906e-04, 1.36202795e-04, 1.44583147e-04, 8.55850497e-04,
       1.34786934e-04, 1.56138409e-04, 1.47139307e-04, 1.41175678e-04,
       1.36939549e-04, 1.34139603e-04, 1.30636470e-04, 1.42586649e-04,
       8.14725444e-04, 1.29247495e-04, 1.46053768e-04, 1.40729643e-04,
       1.50561781e-04, 8.66074210e-04, 1.38217819e-04, 1.48765518e-04,
       1.41745681e-04, 1.83267589e-04, 1.67808169e-04, 1.96131623e-04,
       1.06952603e-03, 1.62913517e-04, 1.88581910e-04, 1.70823224e-04,
       1.28426239e-04, 1.41798394e-04, 1.19749150e-04, 1.14872651e-04,
       7.68382912e-04, 1.22643912e-04, 1.40892567e-04, 1.39364150e-04,
       1.39901322e-04, 1.59913340e-04, 9.48075852e-04, 1.48300008e-04,
       1.86373768e-04, 1.74223263e-04, 1.26726703e-04, 1.50725442e-04,
       1.06270766e-04, 1.22643912e-04, 8.01392465e-04, 1.51325617e-04,
       1.43700025e-04, 1.58110343e-04, 1.63058024e-04, 1.51325617e-04,
       9.41058808e-04, 1.58054534e-04, 1.54364527e-04, 1.56145763e-04,
       1.63843012e-04, 1.64837994e-04, 1.58054534e-04, 9.55178190e-04,
       1.57246235e-04, 1.56499849e-04, 1.54696568e-04, 1.44602564e-04,
       1.40287616e-04, 1.49103717e-04, 8.74256750e-04, 1.41805856e-04,
       1.50719956e-04, 1.47737042e-04, 1.28930084e-04, 1.14073265e-04,
       1.22041705e-04, 1.41805856e-04, 8.05451876e-04, 1.46846817e-04,
       1.51754149e-04, 1.52071623e-04, 1.50291744e-04, 9.07544667e-04,
       1.53382833e-04, 1.49373560e-04, 1.52327151e-04, 1.50097757e-04,
       1.53358533e-04, 1.53449890e-04, 1.53382833e-04, 9.21018877e-04,
       1.53533751e-04, 1.53617154e-04, 1.53676716e-04, 1.47966848e-04,
       1.44527424e-04, 1.49373560e-04, 8.80587138e-04, 1.43370545e-04,
       1.48956176e-04, 1.46392585e-04, 1.51561803e-04, 1.50390706e-04,
       1.53533751e-04, 9.11529481e-04, 1.51278858e-04, 1.52924080e-04,
       1.51840284e-04, 1.51247182e-04, 1.51112559e-04, 1.51278858e-04,
       9.01527866e-04, 1.50149139e-04, 1.49367613e-04, 1.48372517e-04,
       1.37997958e-04, 1.31995118e-04, 8.24931664e-04, 1.34067307e-04,
       1.36102658e-04, 1.46403407e-04, 1.38365215e-04, 1.38888608e-04,
       1.39218814e-04, 1.34067307e-04, 8.12915389e-04, 1.40253997e-04,
       1.27315088e-04, 1.33171573e-04, 1.28901125e-04, 1.37322103e-04,
       1.46667683e-04, 1.48300008e-04, 1.36102658e-04, 8.43449539e-04,
       1.46155962e-04, 1.36068787e-04, 1.41801061e-04, 1.46846817e-04,
       9.63875772e-04, 1.39975177e-04, 1.65651660e-04, 1.07229625e-04,
       1.26302645e-04, 1.35079151e-04, 1.38523858e-04, 1.30277085e-04,
       8.17706660e-04, 1.39071982e-04, 1.34702197e-04, 1.40052387e-04,
       1.33066312e-04, 1.39104286e-04, 1.24332993e-04, 7.95489084e-04,
       1.39333380e-04, 1.25344850e-04, 1.34307263e-04, 1.38789209e-04,
       1.35032031e-04, 1.39333380e-04, 8.20469286e-04, 1.33895818e-04,
       1.37549616e-04, 1.35869232e-04, 1.37516830e-04, 1.38634087e-04,
       8.23734307e-04, 1.35854706e-04, 1.38048152e-04, 1.36302171e-04,
       1.37378360e-04, 1.35778234e-04, 1.35307502e-04, 1.35854706e-04,
       8.11463589e-04, 1.34608074e-04, 1.35345486e-04, 1.34569586e-04,
       1.35049067e-04, 1.35190985e-04, 1.34608074e-04, 8.06726959e-04,
       1.34173353e-04, 1.34141059e-04, 1.33564420e-04, 1.35586152e-04,
       1.35823162e-04, 1.34173353e-04, 8.05445702e-04, 1.34115234e-04,
       1.32726336e-04, 1.33021465e-04, 1.38635033e-04, 1.38801023e-04,
       1.38048152e-04, 8.31351950e-04, 1.38807316e-04, 1.38233626e-04,
       1.38826801e-04, 1.36682779e-04, 1.37272676e-04, 1.35321246e-04,
       1.33764152e-04, 1.34115234e-04, 8.10304124e-04, 1.33148037e-04,
       1.38630162e-04, 1.38148245e-04, 1.38807316e-04, 8.34480368e-04,
       1.38761768e-04, 1.39848254e-04, 1.40284623e-04, 1.47743713e-04,
       1.42870081e-04, 1.56670146e-04, 9.60577123e-04, 1.42797837e-04,
       1.18221434e-04, 1.45726432e-04, 1.06547481e-04, 1.34090492e-04,
       1.31390734e-04, 1.42797837e-04, 8.50439126e-04, 1.44628271e-04,
       1.46500799e-04, 1.51030992e-04, 1.05478405e-04, 1.02653690e-04,
       5.53278903e-04, 1.08742290e-04, 1.13862503e-04, 1.22542014e-04,
       1.43196451e-04, 1.42917424e-04, 1.39975177e-04, 8.43069573e-04,
       1.42051770e-04, 1.36286565e-04, 1.38642185e-04, 1.37741712e-04,
       1.37014074e-04, 8.13518131e-04, 1.38303143e-04, 1.33046950e-04,
       1.35970404e-04, 1.31441848e-04, 1.39317557e-04, 1.38787843e-04,
       1.38303143e-04, 8.31631123e-04, 1.38269147e-04, 1.38228858e-04,
       1.38724576e-04, 1.36715935e-04, 1.31905045e-04, 1.33046950e-04,
       7.78828103e-04, 1.25013939e-04, 1.27684177e-04, 1.24462057e-04,
       1.36055443e-04, 1.34051910e-04, 1.38269147e-04, 8.17061854e-04,
       1.34078859e-04, 1.38396625e-04, 1.36209871e-04, 1.28475750e-04,
       1.30459728e-04, 1.25013939e-04, 7.75734075e-04, 1.32744587e-04,
       1.26646700e-04, 1.32393371e-04, 1.32418870e-04, 1.32304669e-04,
       1.35941592e-04, 8.12476345e-04, 1.34341178e-04, 1.39478125e-04,
       1.37991912e-04, 1.44229971e-04, 1.47199221e-04, 8.88259763e-04,
       1.45267126e-04, 1.50814309e-04, 1.49085601e-04, 1.51663534e-04,
       1.44530215e-04, 1.42521892e-04, 1.45267126e-04, 8.63706521e-04,
       1.43165451e-04, 1.44157594e-04, 1.44064242e-04, 1.47593437e-04,
       1.46342673e-04, 1.50814309e-04, 8.94939944e-04, 1.46847337e-04,
       1.53029198e-04, 1.50312990e-04, 1.44214551e-04, 1.43256800e-04,
       1.46847337e-04, 8.72942799e-04, 1.43615530e-04, 1.48641318e-04,
       1.46367263e-04, 1.39630581e-04, 1.38973771e-04, 1.38519250e-04,
       8.25287364e-04, 1.37242584e-04, 1.35770517e-04, 1.35150661e-04,
       1.38289311e-04, 1.38413544e-04, 1.37242584e-04, 8.23580671e-04,
       1.37865142e-04, 1.35635882e-04, 1.36134207e-04, 1.38711170e-04,
       1.40708290e-04, 1.37865142e-04, 8.35210447e-04, 1.42331412e-04,
       1.36562641e-04, 1.39031791e-04, 1.38897277e-04, 1.38258317e-04,
       1.39071982e-04, 8.33176505e-04, 1.36238151e-04, 1.41452825e-04,
       1.39257953e-04, 1.37742937e-04, 1.35714879e-04, 1.36238151e-04,
       8.07636346e-04, 1.31590574e-04, 1.34484662e-04, 1.31865143e-04,
       1.36027105e-04, 1.38652836e-04, 1.31590574e-04, 7.80540527e-04,
       1.37548670e-04, 1.16694489e-04, 1.20026854e-04, 1.38687657e-04,
       1.38485044e-04, 1.37548670e-04, 8.33361194e-04, 1.36750447e-04,
       1.41242493e-04, 1.40646883e-04, 1.35587501e-04, 1.25620095e-04,
       1.34786934e-04, 6.43568615e-04, 1.20790008e-04, 1.26784076e-04,
       1.25407014e-04, 1.25149854e-04, 1.29247495e-04, 1.20790008e-04,
       8.03038520e-04, 1.51481757e-04, 1.50962393e-04, 1.28217996e-04,
       1.30968754e-04, 7.95959067e-04, 1.35142129e-04, 1.31344358e-04,
       1.31868651e-04, 1.38417179e-04, 1.26261237e-04, 1.26200929e-04,
       1.51481757e-04, 1.35142129e-04, 8.69974456e-04, 1.73644170e-04,
       1.57244234e-04, 1.34216536e-04, 1.38217819e-04, 1.31344358e-04,
       8.12040059e-04, 1.37738342e-04, 1.34065151e-04, 1.36457852e-04,
       1.54906386e-04, 1.48765518e-04, 1.62913517e-04, 9.23481685e-04,
       1.47153781e-04, 1.59680676e-04, 1.50061807e-04, 1.92067588e-04,
       1.69477022e-04, 1.88581910e-04, 1.04062488e-03, 1.57822651e-04,
       1.70422202e-04, 1.62253509e-04, 1.51961857e-04, 1.40892567e-04,
       1.43700025e-04, 1.54364527e-04, 1.57822651e-04, 9.06159491e-04,
       1.57417864e-04, 1.67983388e-04, 1.86373768e-04, 1.57246235e-04,
       1.12671915e-03, 1.61379974e-04, 1.65840429e-04, 1.29742334e-04,
       1.58153024e-04, 1.50566502e-04, 1.50719956e-04, 1.50149139e-04,
       9.00224869e-04, 1.52029725e-04, 1.47516208e-04, 1.49243340e-04,
       1.42125358e-04, 1.43370545e-04, 1.40253997e-04, 8.64447009e-04,
       1.48414690e-04, 1.41683456e-04, 1.48598962e-04, 1.06171559e-04,
       9.86873432e-05, 1.14692273e-04, 1.25344850e-04, 7.04455844e-04,
       1.28195728e-04, 1.31364090e-04, 1.26360157e-04, 1.34702197e-04,
       1.28195728e-04, 8.04446123e-04, 1.39900928e-04, 1.35961976e-04,
       1.39325136e-04, 1.52034557e-04, 1.50863029e-04, 8.96665195e-04,
       1.50318551e-04, 1.47524732e-04, 1.48773309e-04, 1.47151017e-04,
       1.35163617e-04, 1.21907216e-04, 1.50318551e-04, 8.10555229e-04,
       1.23587171e-04, 1.46526827e-04, 1.33051848e-04, 1.45850419e-04,
       1.43719848e-04, 1.42011791e-04, 1.47524732e-04, 8.65806149e-04,
       1.41658495e-04, 1.45040864e-04, 1.24705609e-04, 1.28918788e-04,
       1.23587171e-04, 7.56149479e-04, 1.28441170e-04, 1.24927960e-04,
       1.25568781e-04, 1.31005299e-04, 1.33895818e-04, 1.28441170e-04,
       7.89331646e-04, 1.35606948e-04, 1.28053576e-04, 1.32328835e-04,
       1.41226331e-04, 1.39112713e-04, 1.41658495e-04, 8.40599655e-04,
       1.38284632e-04, 1.40788865e-04, 1.39528620e-04, 1.37778711e-04,
       1.38761768e-04, 1.38284632e-04, 8.36161457e-04, 1.41143085e-04,
       1.39158933e-04, 1.41034329e-04, 1.32985837e-04, 1.32726336e-04,
       1.33148037e-04, 1.34078859e-04, 8.02120286e-04, 1.33833246e-04,
       1.35347970e-04, 1.48814383e-04, 1.43375546e-04, 1.47305241e-04,
       8.62553777e-04, 1.38204730e-04, 1.44257776e-04, 1.40596101e-04,
       1.17340972e-04, 1.44628271e-04, 1.08742290e-04, 8.19903929e-04,
       1.59834759e-04, 1.32776234e-04, 1.56581403e-04, 1.16269188e-04,
       1.13862503e-04, 1.38204730e-04, 7.74484025e-04, 1.35487456e-04,
       1.35485369e-04, 1.35174780e-04, 1.48098766e-04, 1.48060885e-04,
       1.44257776e-04, 8.72176427e-04, 1.45788840e-04, 1.42200608e-04,
       1.43769552e-04, 1.48089041e-04, 1.46641682e-04, 1.45788840e-04,
       8.76102203e-04, 1.45426882e-04, 1.44535178e-04, 1.45620579e-04,
       1.45899181e-04, 1.45591419e-04, 1.45426882e-04, 8.73336163e-04,
       1.45787371e-04, 1.44981529e-04, 1.45649780e-04, 1.45132034e-04,
       1.42051770e-04, 1.45787371e-04, 8.61649300e-04, 1.40995152e-04,
       1.44697464e-04, 1.42985509e-04, 1.31736483e-04, 1.32744587e-04,
       1.34341178e-04, 8.00949739e-04, 1.33301906e-04, 1.35436481e-04,
       1.33389104e-04, 1.41437448e-04, 1.39478125e-04, 1.43615530e-04,
       8.48757079e-04, 1.40022728e-04, 1.42918363e-04, 1.41284884e-04,
       1.52525951e-04, 1.18221434e-04, 1.43165451e-04, 7.27997204e-04,
       1.40301919e-04, 7.11707316e-05, 1.02611717e-04, 1.42333143e-04,
       1.29054429e-04, 6.75928360e-04, 1.45540719e-04, 1.27275172e-04,
       1.31724897e-04, 1.51590754e-04, 1.56138409e-04, 1.45540719e-04,
       8.97625654e-04, 1.55437723e-04, 1.41784351e-04, 1.47133699e-04,
       1.36959376e-04, 1.44345649e-04, 1.27275172e-04, 8.40590791e-04,
       1.46215712e-04, 1.37120307e-04, 1.48674574e-04, 1.42724116e-04,
       1.42331412e-04, 1.46215712e-04, 8.95160340e-04, 1.50226456e-04,
       1.54629598e-04, 1.59033046e-04, 1.37285684e-04, 1.35770517e-04,
       1.36750447e-04, 8.12671088e-04, 1.33548723e-04, 1.35501137e-04,
       1.33814580e-04, 1.47139307e-04, 1.26784076e-04, 1.50962393e-04,
       1.73644170e-04, 1.55437723e-04, 1.05627988e-03, 1.56333025e-04,
       1.45979183e-04, 1.41745681e-04, 1.37738342e-04, 1.47153781e-04,
       8.65439622e-04, 1.40527666e-04, 1.50446464e-04, 1.47827688e-04,
       1.70823224e-04, 1.59680676e-04, 1.70422202e-04, 9.67797641e-04,
       1.59887307e-04, 1.53795943e-04, 1.53188289e-04, 1.74223263e-04,
       1.46403407e-04, 1.46155962e-04, 1.61379974e-04, 1.02788081e-03,
       1.46838159e-04, 1.21152861e-04, 1.31727187e-04, 1.56145763e-04,
       1.56499849e-04, 1.62253509e-04, 1.57417864e-04, 1.59887307e-04,
       9.47610040e-04, 1.55405748e-04, 1.54696568e-04, 1.65840429e-04,
       1.53795943e-04, 1.55405748e-04, 9.50623697e-04, 1.54065671e-04,
       1.66819338e-04, 1.47737042e-04, 1.51754149e-04, 1.65651660e-04,
       1.52029725e-04, 9.27874431e-04, 1.50548513e-04, 1.60153341e-04,
       1.52327151e-04, 1.53617154e-04, 9.06025944e-04, 1.49794479e-04,
       1.51722645e-04, 1.49601339e-04, 1.48963177e-04, 1.50097757e-04,
       1.48956176e-04, 1.49794479e-04, 8.88779153e-04, 1.46470833e-04,
       1.47820987e-04, 1.45638922e-04, 1.53676716e-04, 1.52924080e-04,
       1.51722645e-04, 9.14218901e-04, 1.52074691e-04, 1.51358044e-04,
       1.52462725e-04, 1.46392585e-04, 1.48414690e-04, 1.46470833e-04,
       9.37082699e-04, 1.60953117e-04, 1.61195648e-04, 1.73655826e-04,
       1.51840284e-04, 1.49367613e-04, 1.52074691e-04, 8.98249047e-04,
       1.47193648e-04, 1.50090592e-04, 1.47682220e-04, 1.48372517e-04,
       1.47516208e-04, 1.47193648e-04, 8.81205378e-04, 1.46041484e-04,
       1.46188689e-04, 1.45892833e-04, 1.38365215e-04, 1.27315088e-04,
       1.46838159e-04, 8.08287358e-04, 1.25855053e-04, 1.39923411e-04,
       1.29990432e-04, 1.33171573e-04, 1.41683456e-04, 1.25855053e-04,
       8.17522263e-04, 1.45286537e-04, 1.30303022e-04, 1.41222623e-04,
       1.07229625e-04, 1.36286565e-04, 5.59335534e-04, 1.32693352e-04,
       7.99870507e-05, 1.03138941e-04, 1.40052387e-04, 1.41452825e-04,
       1.39900928e-04, 8.45905493e-04, 1.41289546e-04, 1.41610813e-04,
       1.41598993e-04, 1.34307263e-04, 1.37549616e-04, 1.31364090e-04,
       1.35961976e-04, 8.14769432e-04, 1.37115477e-04, 1.38471011e-04,
       1.35869232e-04, 1.35606948e-04, 1.37115477e-04, 8.17521717e-04,
       1.38116907e-04, 1.34774719e-04, 1.36038433e-04, 1.36302171e-04,
       1.35345486e-04, 8.22363722e-04, 1.37692708e-04, 1.36297762e-04,
       1.38581928e-04, 1.38143667e-04, 1.37378360e-04, 1.38233626e-04,
       1.37692708e-04, 8.29822615e-04, 1.38625271e-04, 1.38907676e-04,
       1.38984974e-04, 1.34569586e-04, 1.34141059e-04, 1.36297762e-04,
       8.16761810e-04, 1.35513969e-04, 1.38297350e-04, 1.37942084e-04,
       1.33564420e-04, 1.33021465e-04, 1.33833246e-04, 1.35513969e-04,
       8.09567932e-04, 1.36248532e-04, 1.37386300e-04, 1.38826801e-04,
       1.39848254e-04, 1.38625271e-04, 8.39596239e-04, 1.40824171e-04,
       1.40147147e-04, 1.41324595e-04, 1.40284623e-04, 1.41143085e-04,
       1.40824171e-04, 8.52940548e-04, 1.43306144e-04, 1.42879507e-04,
       1.44503017e-04, 1.45726432e-04, 1.46500799e-04, 9.06341760e-04,
       1.50804951e-04, 1.47897898e-04, 1.58772182e-04, 1.56639497e-04,
       1.51030992e-04, 1.59834759e-04, 1.50804951e-04, 9.29837152e-04,
       1.56922764e-04, 1.56964820e-04, 1.54278866e-04, 1.22542014e-04,
       1.32776234e-04, 1.35487456e-04, 9.44838812e-04, 1.45029116e-04,
       1.30205897e-04, 1.44193804e-04, 1.34604291e-04, 1.38642185e-04,
       1.40995152e-04, 1.32693352e-04, 8.17030805e-04, 1.40677897e-04,
       1.29195321e-04, 1.34826898e-04, 1.35970404e-04, 1.38228858e-04,
       8.18972296e-04, 1.32791038e-04, 1.39500525e-04, 1.38466251e-04,
       1.34015218e-04, 1.31441848e-04, 1.27684177e-04, 1.32791038e-04,
       7.68144095e-04, 1.22733982e-04, 1.29896217e-04, 1.23596832e-04,
       1.38724576e-04, 1.38396625e-04, 1.39500525e-04, 8.39868307e-04,
       1.39649838e-04, 1.41559739e-04, 1.42037003e-04, 1.24462057e-04,
       1.26646700e-04, 1.22733982e-04, 7.58267749e-04, 1.30884849e-04,
       1.23786968e-04, 1.29753192e-04, 1.36209871e-04, 1.35347970e-04,
       1.36248532e-04, 1.39649838e-04, 8.28354354e-04, 1.39179369e-04,
       1.41718774e-04, 1.32393371e-04, 1.33301906e-04, 1.30884849e-04,
       7.88172596e-04, 1.31755589e-04, 1.29937338e-04, 1.29899543e-04,
       1.37991912e-04, 1.35436481e-04, 1.40022728e-04, 8.24176762e-04,
       1.35692575e-04, 1.38403894e-04, 1.36629172e-04, 1.49085601e-04,
       1.44157594e-04, 8.72142209e-04, 1.49052486e-04, 1.38483562e-04,
       1.48501941e-04, 1.42861024e-04, 1.51663534e-04, 1.53029198e-04,
       1.49052486e-04, 9.19530691e-04, 1.53705021e-04, 1.54396790e-04,
       1.57683661e-04, 1.44064242e-04, 1.40301919e-04, 1.38483562e-04,
       8.36811773e-04, 1.36547372e-04, 1.37794272e-04, 1.39620406e-04,
       1.50312990e-04, 1.48641318e-04, 1.53705021e-04, 9.17262130e-04,
       1.49390753e-04, 1.58927323e-04, 1.56284726e-04, 1.46367263e-04,
       1.42918363e-04, 1.49390753e-04, 8.69536341e-04, 1.42250016e-04,
       1.46150891e-04, 1.42459054e-04, 1.35150661e-04, 1.35635882e-04,
       1.33548723e-04, 7.98809540e-04, 1.34258379e-04, 1.29995730e-04,
       1.30220164e-04, 1.36134207e-04, 1.36562641e-04, 1.34258379e-04,
       8.08883635e-04, 1.35952786e-04, 1.32354825e-04, 1.33620797e-04,
       1.39031791e-04, 1.50226456e-04, 1.35952786e-04, 8.88999849e-04,
       1.59005531e-04, 1.46462602e-04, 1.58320682e-04, 1.39257953e-04,
       1.34484662e-04, 1.41289546e-04, 8.32473627e-04, 1.31954044e-04,
       1.45024816e-04, 1.40462605e-04, 1.31865143e-04, 1.16694489e-04,
       1.31954044e-04, 6.00965376e-04, 1.03248545e-04, 1.17203156e-04,
       1.20026854e-04, 1.41242493e-04, 1.03248545e-04, 8.07125142e-04,
       1.43056742e-04, 1.40478975e-04, 1.59071533e-04, 1.40646883e-04,
       1.35501137e-04, 1.43056742e-04, 8.41646600e-04, 1.34265366e-04,
       1.45911051e-04, 1.42265421e-04, 1.31868651e-04, 1.34065151e-04,
       8.01384437e-04, 1.31029829e-04, 1.36276393e-04, 1.32547175e-04,
       1.35597239e-04, 1.38417179e-04, 1.57244234e-04, 1.56333025e-04,
       1.31029829e-04, 8.54448120e-04, 1.38949998e-04, 1.32473856e-04,
       1.36457852e-04, 1.40527666e-04, 1.36276393e-04, 8.49470199e-04,
       1.46695747e-04, 1.41581500e-04, 1.47931041e-04, 1.50061807e-04,
       1.50446464e-04, 1.53188289e-04, 1.54065671e-04, 9.13406183e-04,
       1.53068338e-04, 1.52575613e-04, 1.29742334e-04, 1.21152861e-04,
       6.23970533e-04, 1.29760464e-04, 1.20638254e-04, 1.22676620e-04,
       1.58153024e-04, 1.66819338e-04, 1.53068338e-04, 1.29760464e-04,
       8.91683889e-04, 1.49262143e-04, 1.34620582e-04, 1.49243340e-04,
       1.50548513e-04, 1.46041484e-04, 8.85121660e-04, 1.49891524e-04,
       1.43425966e-04, 1.45970833e-04, 1.26302645e-04, 1.60153341e-04,
       7.99870507e-05, 1.49891524e-04, 7.61559444e-04, 1.04585553e-04,
       1.40639330e-04, 1.48598962e-04, 1.60953117e-04, 1.45286537e-04,
       9.29710875e-04, 1.65458050e-04, 1.50798425e-04, 1.58615784e-04,
       1.39325136e-04, 1.41610813e-04, 1.38471011e-04, 1.38116907e-04,
       8.36089758e-04, 1.40602836e-04, 1.37963055e-04, 1.48773309e-04,
       1.46526827e-04, 8.78958919e-04, 1.46936403e-04, 1.44309330e-04,
       1.46016229e-04, 1.46396821e-04, 1.47151017e-04, 1.45040864e-04,
       1.40788865e-04, 1.46936403e-04, 8.64500247e-04, 1.40272856e-04,
       1.44310242e-04, 1.33051848e-04, 1.24927960e-04, 1.44309330e-04,
       8.14787616e-04, 1.29544551e-04, 1.44797410e-04, 1.38156518e-04,
       1.25568781e-04, 1.28053576e-04, 1.29544551e-04, 8.45562563e-04,
       1.48674034e-04, 1.48538466e-04, 1.65183154e-04, 1.32328835e-04,
       1.34774719e-04, 1.48674034e-04, 7.87214294e-04, 1.23627429e-04,
       1.37650883e-04, 1.10158394e-04, 1.39528620e-04, 1.39158933e-04,
       1.40272856e-04, 8.37338848e-04, 1.40222686e-04, 1.38918118e-04,
       1.39237635e-04, 1.41034329e-04, 1.43306144e-04, 1.40222686e-04,
       8.55667992e-04, 1.45260090e-04, 1.41539578e-04, 1.44305165e-04,
       1.06547481e-04, 7.11707316e-05, 1.47897898e-04, 5.88120713e-04,
       1.11140555e-04, 1.51364046e-04, 1.40596101e-04, 1.35485369e-04,
       1.42200608e-04, 8.35304576e-04, 1.34960531e-04, 1.42095656e-04,
       1.39966312e-04, 1.56581403e-04, 1.56922764e-04, 1.45029116e-04,
       8.97268524e-04, 1.50562235e-04, 1.41490623e-04, 1.46682382e-04,
       1.35174780e-04, 1.30205897e-04, 1.34960531e-04, 8.25451228e-04,
       1.21677781e-04, 1.55873053e-04, 1.47559186e-04, 1.43769552e-04,
       1.44535178e-04, 1.42095656e-04, 8.61083681e-04, 1.45157543e-04,
       1.41614745e-04, 1.43911006e-04, 1.45620579e-04, 1.44981529e-04,
       1.45157543e-04, 8.72875115e-04, 1.45625297e-04, 1.45191682e-04,
       1.46298485e-04, 1.45649780e-04, 1.44697464e-04, 1.45625297e-04,
       8.70609857e-04, 1.44154498e-04, 1.45500650e-04, 1.44982168e-04,
       1.42985509e-04, 1.40677897e-04, 1.44154498e-04, 8.50892137e-04,
       1.38881802e-04, 1.43578757e-04, 1.40613673e-04, 1.33389104e-04,
       1.31755589e-04, 1.35692575e-04, 7.98357479e-04, 1.31522747e-04,
       1.34266888e-04, 1.31730576e-04, 1.41284884e-04, 1.38403894e-04,
       1.42250016e-04, 8.35569146e-04, 1.37779355e-04, 1.38720334e-04,
       1.37130663e-04, 1.02611717e-04, 1.36547372e-04, 1.11140555e-04,
       8.27048726e-04, 1.59723317e-04, 1.51189329e-04, 1.65836435e-04,
       1.31724897e-04, 1.41784351e-04, 1.37120307e-04, 8.31496695e-04,
       1.39284081e-04, 1.44739370e-04, 1.36843690e-04, 1.47133699e-04,
       1.45979183e-04, 1.38949998e-04, 1.39284081e-04, 8.28514985e-04,
       1.28994228e-04, 1.28173795e-04, 1.48674574e-04, 1.54629598e-04,
       1.44739370e-04, 8.91625095e-04, 1.53864943e-04, 1.43273030e-04,
       1.46443580e-04, 1.59033046e-04, 1.59005531e-04, 1.53864943e-04,
       9.31272497e-04, 1.56690330e-04, 1.49912872e-04, 1.52765774e-04,
       1.33814580e-04, 1.29995730e-04, 1.34265366e-04, 7.83883386e-04,
       1.27644506e-04, 1.30482607e-04, 1.27680597e-04, 1.47827688e-04,
       1.46695747e-04, 1.52575613e-04, 1.49262143e-04, 1.10308360e-03,
       1.63716782e-04, 1.63751090e-04, 1.79254537e-04, 1.31727187e-04,
       1.39923411e-04, 1.20638254e-04, 7.87462977e-04, 1.38225791e-04,
       1.23778510e-04, 1.33169824e-04, 1.49601339e-04, 1.51358044e-04,
       8.99166703e-04, 1.47979958e-04, 1.52242753e-04, 1.50245349e-04,
       1.47739259e-04, 1.48963177e-04, 1.47820987e-04, 1.47979958e-04,
       8.70802418e-04, 1.43146982e-04, 1.43291050e-04, 1.39600264e-04,
       1.45638922e-04, 1.61195648e-04, 1.43146982e-04, 8.58521085e-04,
       1.51114947e-04, 1.32506674e-04, 1.24917912e-04, 1.52462725e-04,
       1.50090592e-04, 1.52242753e-04, 8.99305227e-04, 1.47629172e-04,
       1.49560716e-04, 1.47319268e-04, 1.73655826e-04, 1.65458050e-04,
       1.51114947e-04, 1.09428046e-03, 1.64121212e-04, 1.23287103e-04,
       1.69232447e-04, 1.47410877e-04, 1.47682220e-04, 1.46188689e-04,
       1.47629172e-04, 8.76357201e-04, 1.44551751e-04, 1.45944629e-04,
       1.44360740e-04, 1.45892833e-04, 1.43425966e-04, 1.44551751e-04,
       8.64362779e-04, 1.41549483e-04, 1.44739073e-04, 1.44203672e-04,
       1.29990432e-04, 1.30303022e-04, 1.38225791e-04, 8.21592849e-04,
       1.37242894e-04, 1.43563098e-04, 1.42267611e-04, 1.41222623e-04,
       1.50798425e-04, 1.37242894e-04, 9.10799703e-04, 1.66395335e-04,
       1.47781893e-04, 1.67358532e-04, 1.03138941e-04, 1.29195321e-04,
       1.04585553e-04, 7.36590215e-04, 1.34170124e-04, 1.30986043e-04,
       1.34514233e-04, 1.41598993e-04, 1.45024816e-04, 1.40602836e-04,
       8.45759262e-04, 1.34275921e-04, 1.47004716e-04, 1.37251979e-04,
       1.36038433e-04, 1.37963055e-04, 1.23627429e-04, 1.34275921e-04,
       7.80467474e-04, 1.19919383e-04, 1.28643253e-04, 1.38581928e-04,
       1.38907676e-04, 8.54284734e-04, 1.42295533e-04, 1.40627650e-04,
       1.45939158e-04, 1.47932790e-04, 1.38143667e-04, 1.38297350e-04,
       1.42295533e-04, 8.61483533e-04, 1.42594812e-04, 1.50003288e-04,
       1.50148884e-04, 1.38984974e-04, 1.40147147e-04, 1.40627650e-04,
       8.50001829e-04, 1.41659153e-04, 1.44356788e-04, 1.44226118e-04,
       1.37942084e-04, 1.37386300e-04, 1.39179369e-04, 1.42594812e-04,
       8.48972798e-04, 1.44505287e-04, 1.47364946e-04, 1.41324595e-04,
       1.42879507e-04, 1.41659153e-04, 8.59933622e-04, 1.44158718e-04,
       1.44483498e-04, 1.45428151e-04, 1.44503017e-04, 1.45260090e-04,
       1.44158718e-04, 8.71730441e-04, 1.45567013e-04, 1.45795190e-04,
       1.46446413e-04, 1.58772182e-04, 1.56964820e-04, 9.73936831e-04,
       1.69019141e-04, 1.56060800e-04, 1.72135433e-04, 1.60984454e-04,
       1.56639497e-04, 1.51364046e-04, 1.59723317e-04, 1.69019141e-04,
       1.03742819e-03, 1.98330905e-04, 2.02351279e-04, 1.54278866e-04,
       1.50562235e-04, 1.56060800e-04, 9.10690603e-04, 1.48180553e-04,
       1.52885883e-04, 1.48722266e-04, 1.44193804e-04, 1.41490623e-04,
       8.28555827e-04, 1.39277615e-04, 1.41605367e-04, 1.28478742e-04,
       1.33509675e-04, 1.34604291e-04, 1.21677781e-04, 1.39277615e-04,
       6.61884221e-04, 9.53511665e-05, 1.00986653e-04, 6.99867137e-05,
       1.34826898e-04, 1.38881802e-04, 1.34170124e-04, 8.20816307e-04,
       1.38885636e-04, 1.36695619e-04, 1.37356227e-04, 1.38466251e-04,
       1.41559739e-04, 8.69793180e-04, 1.41814543e-04, 1.46212516e-04,
       1.50217823e-04, 1.51522308e-04, 1.34015218e-04, 1.29896217e-04,
       1.41814543e-04, 8.00243060e-04, 1.26183146e-04, 1.39923165e-04,
       1.28410770e-04, 1.23596832e-04, 1.23786968e-04, 1.26183146e-04,
       7.65290833e-04, 1.28786844e-04, 1.30261740e-04, 1.32675302e-04,
       1.42037003e-04, 1.41718774e-04, 1.44505287e-04, 1.46212516e-04,
       8.78853131e-04, 1.51495276e-04, 1.52884275e-04, 1.29753192e-04,
       1.29937338e-04, 1.28786844e-04, 7.74762003e-04, 1.28194183e-04,
       1.29433626e-04, 1.28656819e-04, 1.29899543e-04, 1.31522747e-04,
       1.28194183e-04, 7.76615855e-04, 1.30475571e-04, 1.27932814e-04,
       1.28590997e-04, 1.36629172e-04, 1.34266888e-04, 1.37779355e-04,
       8.15329180e-04, 1.33364271e-04, 1.37671147e-04, 1.35618347e-04,
       1.48501941e-04, 1.54396790e-04, 9.13501416e-04, 1.46646404e-04,
       1.60004679e-04, 1.55707513e-04, 1.48244088e-04, 1.42861024e-04,
       1.37794272e-04, 1.46646404e-04, 8.51973388e-04, 1.38907391e-04,
       1.44218279e-04, 1.41546017e-04, 1.57683661e-04, 1.58927323e-04,
       1.60004679e-04, 1.01489687e-03, 1.63121805e-04, 1.33074589e-04,
       1.34949000e-04, 1.07135812e-04, 1.39620406e-04, 1.51189329e-04,
       1.38907391e-04, 8.88902330e-04, 1.65599827e-04, 1.39230932e-04,
       1.54354445e-04, 1.56284726e-04, 1.46150891e-04, 1.63121805e-04,
       8.90789020e-04, 1.42223826e-04, 1.46885958e-04, 1.36121814e-04,
       1.42459054e-04, 1.38720334e-04, 1.42223826e-04, 8.34710602e-04,
       1.36333024e-04, 1.38563223e-04, 1.36411140e-04, 1.30220164e-04,
       1.32354825e-04, 1.27644506e-04, 7.72255922e-04, 1.30476273e-04,
       1.25431524e-04, 1.26128630e-04, 1.33620797e-04, 1.46462602e-04,
       1.30476273e-04, 8.73098718e-04, 1.59305027e-04, 1.43623534e-04,
       1.59610484e-04, 1.58320682e-04, 1.56690330e-04, 1.59305027e-04,
       9.41415950e-04, 1.54435019e-04, 1.57952274e-04, 1.54712617e-04,
       1.40462605e-04, 1.17203156e-04, 1.40478975e-04, 1.47004716e-04,
       1.01228281e-03, 1.70388850e-04, 1.43455795e-04, 1.53288714e-04,
       1.59071533e-04, 1.45911051e-04, 1.70388850e-04, 9.26039558e-04,
       1.47318799e-04, 1.56629447e-04, 1.46719878e-04, 1.42265421e-04,
       1.30482607e-04, 1.47318799e-04, 8.27490718e-04, 1.29651259e-04,
       1.41828679e-04, 1.35943953e-04, 1.32547175e-04, 1.32473856e-04,
       1.28994228e-04, 7.88957347e-04, 1.31631963e-04, 1.30372855e-04,
       1.32937270e-04, 1.35597239e-04, 1.41581500e-04, 1.31631963e-04,
       8.22670125e-04, 1.41894263e-04, 1.33843561e-04, 1.38121599e-04,
       1.47931041e-04, 1.63716782e-04, 1.41894263e-04, 9.07222232e-04,
       1.63261697e-04, 1.42418378e-04, 1.48000071e-04, 1.22676620e-04,
       1.34620582e-04, 1.63751090e-04, 1.23778510e-04, 8.31400836e-04,
       1.57301326e-04, 1.29272708e-04, 1.45970833e-04, 1.40639330e-04,
       1.41549483e-04, 1.30986043e-04, 8.37319596e-04, 1.41702922e-04,
       1.36470985e-04, 1.58615784e-04, 1.64121212e-04, 1.66395335e-04,
       9.01850345e-04, 1.43590307e-04, 1.48719082e-04, 1.20408626e-04,
       1.46016229e-04, 1.44797410e-04, 8.73026045e-04, 1.44740799e-04,
       1.44887146e-04, 1.46974814e-04, 1.45609647e-04, 1.46396821e-04,
       1.44310242e-04, 1.38918118e-04, 1.44740799e-04, 8.56130986e-04,
       1.39014452e-04, 1.42750555e-04, 1.38156518e-04, 1.48538466e-04,
       1.44887146e-04, 8.95454070e-04, 1.58644113e-04, 1.50926404e-04,
       1.54301423e-04, 1.65183154e-04, 1.37650883e-04, 1.58644113e-04,
       9.97523383e-04, 1.07591523e-04, 1.53105525e-04, 1.26931615e-04,
       1.48416569e-04, 1.10158394e-04, 1.19919383e-04, 1.07591523e-04,
       5.66301017e-04, 1.18941988e-04, 1.09689729e-04, 1.39237635e-04,
       1.41539578e-04, 1.39014452e-04, 8.45924381e-04, 1.42846061e-04,
       1.40878495e-04, 1.42408161e-04, 1.44305165e-04, 1.45567013e-04,
       1.42846061e-04, 8.66342618e-04, 1.45864349e-04, 1.43001795e-04,
       1.44758236e-04, 1.39966312e-04, 1.55873053e-04, 1.41614745e-04,
       8.90901491e-04, 1.62842522e-04, 1.42641912e-04, 1.47962947e-04,
       1.46682382e-04, 1.48180553e-04, 1.41605367e-04, 8.58032688e-04,
       1.44455312e-04, 1.37878650e-04, 1.39230424e-04, 1.47559186e-04,
       9.53511665e-05, 1.62842522e-04, 9.71355836e-04, 9.57465893e-05,
       1.29145587e-04, 1.33551687e-04, 9.39325428e-05, 1.13226555e-04,
       1.43911006e-04, 1.45191682e-04, 1.42641912e-04, 8.59552007e-04,
       1.45040400e-04, 1.40824597e-04, 1.41942410e-04, 1.46298485e-04,
       1.45500650e-04, 1.45040400e-04, 8.67673523e-04, 1.44580865e-04,
       1.42957411e-04, 1.43295711e-04, 1.44982168e-04, 1.43578757e-04,
       1.44580865e-04, 8.56716954e-04, 1.41126020e-04, 1.42250289e-04,
       1.40198854e-04, 1.40613673e-04, 1.38885636e-04, 1.41126020e-04,
       8.38138983e-04, 1.38056836e-04, 1.40398982e-04, 1.39057836e-04,
       1.31730576e-04, 1.30475571e-04, 1.33364271e-04, 7.89895610e-04,
       1.29372961e-04, 1.33844268e-04, 1.31107964e-04, 1.37130663e-04,
       1.37671147e-04, 1.36333024e-04, 8.32903840e-04, 1.39298089e-04,
       1.40023168e-04, 1.42447749e-04, 1.65836435e-04, 1.98330905e-04,
       1.65599827e-04, 1.29742030e-03, 2.38504938e-04, 1.64727111e-04,
       1.59873794e-04, 2.04547288e-04, 1.36843690e-04, 1.28173795e-04,
       1.43273030e-04, 1.30372855e-04, 8.15012647e-04, 1.40289780e-04,
       1.36059498e-04, 1.46443580e-04, 1.49912872e-04, 1.40289780e-04,
       8.73084586e-04, 1.49043465e-04, 1.41055248e-04, 1.46339641e-04,
       1.52765774e-04, 1.54435019e-04, 1.49043465e-04, 9.01888408e-04,
       1.50751223e-04, 1.47427206e-04, 1.47465720e-04, 1.27680597e-04,
       1.25431524e-04, 1.29651259e-04, 7.62307797e-04, 1.25101681e-04,
       1.28371620e-04, 1.26071115e-04, 1.79254537e-04, 1.63261697e-04,
       1.57301326e-04, 9.32908883e-04, 1.37935456e-04, 1.49264449e-04,
       1.45891419e-04, 1.33169824e-04, 1.43563098e-04, 1.29272708e-04,
       1.37935456e-04, 8.33351278e-04, 1.45570359e-04, 1.43839833e-04,
       1.50245349e-04, 1.49560716e-04, 8.94343896e-04, 1.49945851e-04,
       1.47191780e-04, 1.47665380e-04, 1.49734819e-04, 1.47739259e-04,
       1.43291050e-04, 1.49945851e-04, 8.72328134e-04, 1.40959656e-04,
       1.47465153e-04, 1.42927166e-04, 1.39600264e-04, 1.32506674e-04,
       1.40959656e-04, 8.18539772e-04, 1.30425310e-04, 1.39244312e-04,
       1.35803556e-04, 1.24917912e-04, 1.23287103e-04, 1.30425310e-04,
       6.32290669e-04, 1.23767709e-04, 1.29892636e-04, 1.47319268e-04,
       1.45944629e-04, 1.47191780e-04, 8.79040391e-04, 1.45169960e-04,
       1.47030967e-04, 1.46383786e-04, 1.69232447e-04, 1.43590307e-04,
       8.51560710e-04, 1.52847408e-04, 1.24040053e-04, 1.32507855e-04,
       1.29342640e-04, 1.44360740e-04, 1.44739073e-04, 1.45169960e-04,
       8.75685125e-04, 1.46572527e-04, 1.46909296e-04, 1.47933529e-04,
       1.44203672e-04, 1.41702922e-04, 1.46572527e-04, 8.63660055e-04,
       1.40561848e-04, 1.47064506e-04, 1.43554579e-04, 1.42267611e-04,
       1.47781893e-04, 1.45570359e-04, 8.88245601e-04, 1.53470613e-04,
       1.48384577e-04, 1.50770548e-04, 1.67358532e-04, 1.48719082e-04,
       1.53470613e-04, 1.04462885e-03, 1.20583393e-04, 1.52971581e-04,
       1.42009638e-04, 1.59516015e-04, 1.34514233e-04, 1.36695619e-04,
       1.36470985e-04, 1.40561848e-04, 8.27234826e-04, 1.38541331e-04,
       1.40450808e-04, 1.37251979e-04, 1.28643253e-04, 1.43455795e-04,
       1.18941988e-04, 7.88457582e-04, 1.38034823e-04, 1.22129742e-04,
       1.45939158e-04, 1.44356788e-04, 8.97425237e-04, 1.51153739e-04,
       1.48106382e-04, 1.54667078e-04, 1.53202091e-04, 1.47932790e-04,
       1.50003288e-04, 1.51153739e-04, 9.25108425e-04, 1.57731073e-04,
       1.56890350e-04, 1.61397185e-04, 1.50148884e-04, 1.47364946e-04,
       1.51495276e-04, 1.57731073e-04, 9.28353536e-04, 1.59080749e-04,
       1.62532608e-04, 1.44226118e-04, 1.44483498e-04, 1.48106382e-04,
       8.84321227e-04, 1.46876569e-04, 1.51057927e-04, 1.49570733e-04,
       1.45428151e-04, 1.45795190e-04, 1.46876569e-04, 8.82044618e-04,
       1.47739886e-04, 1.47854272e-04, 1.48350549e-04, 1.46446413e-04,
       1.45864349e-04, 1.47739886e-04, 8.83843393e-04, 1.46609716e-04,
       1.48865567e-04, 1.48317462e-04, 1.72135433e-04, 2.02351279e-04,
       2.38504938e-04, 1.18091864e-03, 1.74706601e-04, 2.13401438e-04,
       1.79818948e-04, 1.60984454e-04, 1.52885883e-04, 1.74706601e-04,
       8.92712502e-04, 1.36450664e-04, 1.48921133e-04, 1.18763767e-04,
       1.48722266e-04, 1.44455312e-04, 1.36450664e-04, 8.32751874e-04,
       1.41705798e-04, 1.25947873e-04, 1.35469961e-04, 1.28478742e-04,
       1.00986653e-04, 7.25958114e-04, 1.31646745e-04, 1.00070554e-04,
       1.34429761e-04, 1.30345659e-04, 1.33509675e-04, 1.37878650e-04,
       1.31646745e-04, 8.23030345e-04, 1.37755320e-04, 1.40250178e-04,
       1.41989778e-04, 6.99867137e-05, 9.57465893e-05, 1.00070554e-04,
       5.17709054e-04, 1.25148675e-04, 1.26756521e-04, 1.37356227e-04,
       1.38056836e-04, 1.38541331e-04, 8.33832288e-04, 1.39754290e-04,
       1.39805627e-04, 1.40317976e-04, 1.50217823e-04, 1.39923165e-04,
       9.05700376e-04, 1.67254004e-04, 1.27343623e-04, 1.45673018e-04,
       1.75288742e-04, 1.51522308e-04, 1.52884275e-04, 1.59080749e-04,
       1.67254004e-04, 9.87508620e-04, 1.73616031e-04, 1.83151253e-04,
       1.28410770e-04, 1.30261740e-04, 1.27343623e-04, 7.85403010e-04,
       1.34776873e-04, 1.29513862e-04, 1.35096142e-04, 1.32675302e-04,
       1.29433626e-04, 1.34776873e-04, 7.90865745e-04, 1.30628642e-04,
       1.32128928e-04, 1.31222374e-04, 1.28656819e-04, 1.27932814e-04,
       1.30628642e-04, 7.76851547e-04, 1.28917420e-04, 1.30851535e-04,
       1.29864317e-04, 1.28590997e-04, 1.29372961e-04, 1.28917420e-04,
       7.87643032e-04, 1.32714690e-04, 1.32515828e-04, 1.35531135e-04,
       1.35618347e-04, 1.33844268e-04, 1.39298089e-04, 8.23101795e-04,
       1.34643342e-04, 1.41289209e-04, 1.38408541e-04, 1.55707513e-04,
       1.33074589e-04, 8.18830482e-04, 1.51760492e-04, 1.07494439e-04,
       1.24580160e-04, 1.46213289e-04, 1.48244088e-04, 1.44218279e-04,
       1.51760492e-04, 8.76486616e-04, 1.42441998e-04, 1.47557222e-04,
       1.42264537e-04, 1.41546017e-04, 1.39230932e-04, 1.42441998e-04,
       8.40024330e-04, 1.38436216e-04, 1.40029931e-04, 1.38339234e-04,
       1.34949000e-04, 1.46885958e-04, 7.60771124e-04, 1.30001522e-04,
       1.06308593e-04, 1.27191011e-04, 1.15435040e-04, 1.54354445e-04,
       1.64727111e-04, 1.38436216e-04, 8.78785202e-04, 1.47960309e-04,
       1.37077847e-04, 1.36229274e-04, 1.36121814e-04, 1.38563223e-04,
       1.30001522e-04, 8.01004669e-04, 1.35426156e-04, 1.28954365e-04,
       1.31937590e-04, 1.36411140e-04, 1.40023168e-04, 1.35426156e-04,
       8.27843193e-04, 1.41112766e-04, 1.36397690e-04, 1.38472273e-04,
       1.26128630e-04, 1.43623534e-04, 1.25101681e-04, 8.55942629e-04,
       1.60430835e-04, 1.41411849e-04, 1.59246099e-04, 1.59610484e-04,
       1.57952274e-04, 1.60430835e-04, 9.40587578e-04, 1.54204773e-04,
       1.56478532e-04, 1.51910680e-04, 1.54712617e-04, 1.50751223e-04,
       1.54204773e-04, 9.06742055e-04, 1.47801957e-04, 1.50856453e-04,
       1.48415031e-04, 1.53288714e-04, 1.56629447e-04, 1.38034823e-04,
       8.65843486e-04, 1.44603864e-04, 1.35029166e-04, 1.38257473e-04,
       1.46719878e-04, 1.41828679e-04, 1.44603864e-04, 8.57278135e-04,
       1.41559206e-04, 1.40359983e-04, 1.42206525e-04, 1.35943953e-04,
       1.28371620e-04, 1.41559206e-04, 7.99454462e-04, 1.26206853e-04,
       1.37576424e-04, 1.29796405e-04, 1.32937270e-04, 1.33843561e-04,
       1.36059498e-04, 1.41055248e-04, 8.22041723e-04, 1.37028344e-04,
       1.41117803e-04, 1.38121599e-04, 1.42418378e-04, 1.37028344e-04,
       8.40050155e-04, 1.42994474e-04, 1.38909005e-04, 1.40578356e-04,
       1.48000071e-04, 1.49264449e-04, 1.42994474e-04, 8.75581644e-04,
       1.47427140e-04, 1.42363611e-04, 1.45531899e-04, 1.47410877e-04,
       1.23767709e-04, 1.52847408e-04, 8.26909692e-04, 1.31025813e-04,
       1.35018156e-04, 1.36839728e-04, 1.20408626e-04, 1.24040053e-04,
       1.20583393e-04, 6.12872329e-04, 1.25418822e-04, 1.22421435e-04,
       1.46974814e-04, 1.50926404e-04, 8.79582985e-04, 1.44787423e-04,
       1.48373474e-04, 1.44077666e-04, 1.44443205e-04, 1.45609647e-04,
       1.42750555e-04, 1.40878495e-04, 1.44787423e-04, 8.59454252e-04,
       1.42189149e-04, 1.43238984e-04, 1.54301423e-04, 1.53105525e-04,
       1.48373474e-04, 8.89627552e-04, 1.47139082e-04, 1.43373271e-04,
       1.43334777e-04, 1.26931615e-04, 1.09689729e-04, 1.22129742e-04,
       1.35029166e-04, 7.65976852e-04, 1.38269532e-04, 1.33927068e-04,
       1.42408161e-04, 1.43001795e-04, 1.42189149e-04, 8.56183221e-04,
       1.43188534e-04, 1.42494915e-04, 1.42900667e-04, 1.44758236e-04,
       1.46609716e-04, 1.43188534e-04, 8.71295978e-04, 1.47277281e-04,
       1.43612473e-04, 1.45849739e-04, 1.47962947e-04, 1.29145587e-04,
       1.40824597e-04, 7.81826619e-04, 1.35252969e-04, 1.07697393e-04,
       1.20943125e-04, 1.39230424e-04, 1.41705798e-04, 1.37755320e-04,
       8.41145437e-04, 1.39146834e-04, 1.42195461e-04, 1.41111601e-04,
       1.33551687e-04, 1.25148675e-04, 7.61927543e-04, 1.25008650e-04,
       1.28579179e-04, 1.22403435e-04, 1.27235917e-04, 1.41942410e-04,
       1.42957411e-04, 1.35252969e-04, 8.28643675e-04, 1.41349373e-04,
       1.31030054e-04, 1.36111458e-04, 1.43295711e-04, 1.42250289e-04,
       1.41349373e-04, 8.41523257e-04, 1.39915655e-04, 1.37550262e-04,
       1.37161966e-04, 1.40198854e-04, 1.40398982e-04, 1.39915655e-04,
       8.25895057e-04, 1.37240560e-04, 1.35508114e-04, 1.32632892e-04,
       1.39057836e-04, 1.39754290e-04, 1.37240560e-04, 8.25330176e-04,
       1.38364059e-04, 1.35243194e-04, 1.35670239e-04, 1.31107964e-04,
       1.32714690e-04, 1.34643342e-04, 8.02628238e-04, 1.33029418e-04,
       1.37176738e-04, 1.33956087e-04, 1.42447749e-04, 1.41289209e-04,
       1.41112766e-04, 8.62364596e-04, 1.44209299e-04, 1.44525251e-04,
       1.48780323e-04, 1.07135812e-04, 1.07494439e-04, 1.06308593e-04,
       5.51658309e-04, 1.15953129e-04, 1.14766336e-04, 1.59873794e-04,
       1.47960309e-04, 7.40273651e-04, 1.51075706e-04, 1.38828991e-04,
       1.42534851e-04, 1.46339641e-04, 1.47427206e-04, 1.41117803e-04,
       1.38909005e-04, 8.59236748e-04, 1.44369728e-04, 1.41073365e-04,
       1.47465720e-04, 1.47801957e-04, 1.44369728e-04, 8.74767606e-04,
       1.47429056e-04, 1.42489027e-04, 1.45212118e-04, 1.26071115e-04,
       1.41411849e-04, 1.26206853e-04, 8.36326674e-04, 1.54350475e-04,
       1.37741747e-04, 1.50544634e-04, 1.45891419e-04, 1.43839833e-04,
       1.48384577e-04, 1.47427140e-04, 8.85474966e-04, 1.51470510e-04,
       1.48461487e-04, 1.47665380e-04, 1.47030967e-04, 8.86388486e-04,
       1.48653493e-04, 1.46914878e-04, 1.47509782e-04, 1.48613985e-04,
       1.49734819e-04, 1.47465153e-04, 1.48653493e-04, 8.85938747e-04,
       1.45877776e-04, 1.47356779e-04, 1.46850727e-04, 1.42927166e-04,
       1.39244312e-04, 1.45877776e-04, 8.59500622e-04, 1.39838137e-04,
       1.46985008e-04, 1.44628223e-04, 1.35803556e-04, 1.29892636e-04,
       1.31025813e-04, 1.39838137e-04, 8.17472470e-04, 1.39005462e-04,
       1.41906866e-04, 1.46383786e-04, 1.46909296e-04, 1.46914878e-04,
       8.84104746e-04, 1.47867484e-04, 1.47801103e-04, 1.48228200e-04,
       1.32507855e-04, 1.35018156e-04, 8.26077668e-04, 1.34696139e-04,
       1.40302819e-04, 1.40228161e-04, 1.43324539e-04, 1.29342640e-04,
       1.25418822e-04, 1.34696139e-04, 7.96301468e-04, 1.29850391e-04,
       1.38957703e-04, 1.38035773e-04, 1.47933529e-04, 1.47064506e-04,
       1.47867484e-04, 8.82306714e-04, 1.45225644e-04, 1.47994184e-04,
       1.46221367e-04, 1.43554579e-04, 1.40450808e-04, 1.39805627e-04,
       1.45225644e-04, 8.54778616e-04, 1.41712856e-04, 1.44029103e-04,
       1.50770548e-04, 1.52971581e-04, 1.51470510e-04, 9.02688493e-04,
       1.49384624e-04, 1.51189075e-04, 1.46902156e-04, 1.42009638e-04,
       1.22421435e-04, 1.29850391e-04, 8.14096404e-04, 1.47761507e-04,
       1.36865259e-04, 1.35188173e-04, 1.48416569e-04, 1.47139082e-04,
       1.38269532e-04, 8.54297517e-04, 1.36760170e-04, 1.41972019e-04,
       1.41740145e-04, 1.54667078e-04, 1.56890350e-04, 9.49151643e-04,
       1.55594819e-04, 1.62330837e-04, 1.57577906e-04, 1.62090652e-04,
       1.53202091e-04, 1.51057927e-04, 1.55594819e-04, 9.19505704e-04,
       1.51535608e-04, 1.55015153e-04, 1.53100106e-04, 1.61397185e-04,
       1.62532608e-04, 1.73616031e-04, 1.62330837e-04, 1.00324908e-03,
       1.76761097e-04, 1.66611326e-04, 1.49570733e-04, 1.47854272e-04,
       1.51535608e-04, 8.99179322e-04, 1.48761358e-04, 1.51257570e-04,
       1.50199781e-04, 1.48350549e-04, 1.48865567e-04, 1.48761358e-04,
       8.92844952e-04, 1.49105650e-04, 1.48966277e-04, 1.48795551e-04,
       1.48317462e-04, 1.47277281e-04, 1.49105650e-04, 8.89030499e-04,
       1.48048677e-04, 1.48149110e-04, 1.48132319e-04, 2.04547288e-04,
       2.13401438e-04, 1.51075706e-04, 1.05370333e-03, 1.67636398e-04,
       1.54667981e-04, 1.62374523e-04, 9.39325428e-05, 1.07697393e-04,
       5.01598219e-04, 8.20884824e-05, 1.16564245e-04, 1.01315555e-04,
       1.79818948e-04, 1.48921133e-04, 1.67636398e-04, 8.96461360e-04,
       1.17916199e-04, 1.50495533e-04, 1.31673149e-04, 1.18763767e-04,
       1.25947873e-04, 1.17916199e-04, 6.20632181e-04, 1.31234125e-04,
       1.26770217e-04, 1.35469961e-04, 1.39146834e-04, 1.31234125e-04,
       8.24003596e-04, 1.39650584e-04, 1.36881253e-04, 1.41620839e-04,
       1.34429761e-04, 1.40250178e-04, 8.51539944e-04, 1.37253174e-04,
       1.45310780e-04, 1.46029431e-04, 1.48266620e-04, 1.30345659e-04,
       1.26756521e-04, 1.25008650e-04, 1.37253174e-04, 7.88778592e-04,
       1.30379601e-04, 1.39034987e-04, 1.41989778e-04, 1.42195461e-04,
       1.45310780e-04, 8.77509729e-04, 1.47603038e-04, 1.49162049e-04,
       1.51248624e-04, 1.13226555e-04, 1.28579179e-04, 8.20884824e-05,
       6.34502139e-04, 1.15548514e-04, 8.84759998e-05, 1.06583408e-04,
       1.40317976e-04, 1.38364059e-04, 1.41712856e-04, 8.43175970e-04,
       1.38138935e-04, 1.43131074e-04, 1.41511071e-04, 1.45673018e-04,
       1.29513862e-04, 8.63452019e-04, 1.67205290e-04, 1.37081743e-04,
       1.35006857e-04, 1.48971248e-04, 1.75288742e-04, 1.83151253e-04,
       1.76761097e-04, 1.67205290e-04, 1.19313230e-03, 1.67369324e-04,
       1.61910501e-04, 1.61446092e-04, 1.35096142e-04, 1.32128928e-04,
       1.37081743e-04, 7.94338683e-04, 1.29478662e-04, 1.32594537e-04,
       1.27958670e-04, 1.31222374e-04, 1.30851535e-04, 1.29478662e-04,
       7.97116285e-04, 1.37227731e-04, 1.30108038e-04, 1.38227945e-04,
       1.29864317e-04, 1.32515828e-04, 1.37227731e-04, 7.95462054e-04,
       1.27924130e-04, 1.39942580e-04, 1.27987468e-04, 1.35531135e-04,
       1.33029418e-04, 1.27924130e-04, 8.13678482e-04, 1.38777438e-04,
       1.32530668e-04, 1.45885694e-04, 1.38408541e-04, 1.37176738e-04,
       1.44209299e-04, 8.36713243e-04, 1.37326644e-04, 1.42621437e-04,
       1.36970585e-04, 1.24580160e-04, 1.15953129e-04, 7.67706026e-04,
       1.38897444e-04, 1.24295705e-04, 1.29002440e-04, 1.34977148e-04,
       1.46213289e-04, 1.47557222e-04, 1.38897444e-04, 8.53186761e-04,
       1.42419088e-04, 1.37961004e-04, 1.40138715e-04, 1.42264537e-04,
       1.40029931e-04, 1.42419088e-04, 8.43707972e-04, 1.38701369e-04,
       1.40888080e-04, 1.39404967e-04, 1.38339234e-04, 1.37077847e-04,
       1.38701369e-04, 8.34845123e-04, 1.39037277e-04, 1.40046051e-04,
       1.41643345e-04, 1.27191011e-04, 1.28954365e-04, 7.79046187e-04,
       1.25236014e-04, 1.32085029e-04, 1.35332715e-04, 1.30247054e-04,
       1.15435040e-04, 1.14766336e-04, 1.24295705e-04, 1.25236014e-04,
       7.38763704e-04, 1.29379502e-04, 1.29651108e-04, 1.36229274e-04,
       1.38828991e-04, 1.39037277e-04, 8.57251305e-04, 1.46745135e-04,
       1.45651243e-04, 1.50759385e-04, 1.31937590e-04, 1.36397690e-04,
       1.32085029e-04, 8.07743952e-04, 1.36666750e-04, 1.35267636e-04,
       1.35389257e-04, 1.38472273e-04, 1.44525251e-04, 1.36666750e-04,
       7.57849632e-04, 1.23353729e-04, 1.17098814e-04, 9.77328153e-05,
       1.59246099e-04, 1.56478532e-04, 1.54350475e-04, 9.18656995e-04,
       1.49967258e-04, 1.50115140e-04, 1.48499490e-04, 1.51910680e-04,
       1.50856453e-04, 1.49967258e-04, 8.96863962e-04, 1.49013417e-04,
       1.47507884e-04, 1.47608269e-04, 1.48415031e-04, 1.47429056e-04,
       1.49013417e-04, 8.89660724e-04, 1.49677870e-04, 1.46737461e-04,
       1.48387888e-04, 1.38257473e-04, 1.40359983e-04, 1.33927068e-04,
       1.36760170e-04, 8.34032798e-04, 1.41998756e-04, 1.42729347e-04,
       1.42206525e-04, 1.37576424e-04, 1.41998756e-04, 8.28150760e-04,
       1.33364920e-04, 1.38503936e-04, 1.34500200e-04, 1.29796405e-04,
       1.37741747e-04, 1.33364920e-04, 8.25351995e-04, 1.44744556e-04,
       1.38135037e-04, 1.41569331e-04, 1.40578356e-04, 1.42363611e-04,
       1.41073365e-04, 1.42489027e-04, 8.57599056e-04, 1.45732145e-04,
       1.45362551e-04, 1.45531899e-04, 1.48461487e-04, 1.49384624e-04,
       1.45732145e-04, 8.91258719e-04, 1.50712837e-04, 1.51435727e-04,
       1.36839728e-04, 1.39005462e-04, 1.40302819e-04, 8.60169709e-04,
       1.46031135e-04, 1.47213991e-04, 1.50776574e-04, 1.59516015e-04,
       1.51189075e-04, 1.47761507e-04, 8.54346455e-04, 1.36702801e-04,
       1.30954411e-04, 1.28222645e-04, 1.44077666e-04, 1.43373271e-04,
       8.59165191e-04, 1.43938260e-04, 1.42387503e-04, 1.43117427e-04,
       1.42271065e-04, 1.44443205e-04, 1.43238984e-04, 1.42494915e-04,
       1.43938260e-04, 8.60054209e-04, 1.42849861e-04, 1.43088984e-04,
       1.43334777e-04, 1.41972019e-04, 1.42387503e-04, 8.53133970e-04,
       1.41092473e-04, 1.42139735e-04, 1.42207463e-04, 1.42900667e-04,
       1.43612473e-04, 1.42849861e-04, 8.61518366e-04, 1.44439301e-04,
       1.43419215e-04, 1.44296849e-04, 1.45849739e-04, 1.48048677e-04,
       1.44439301e-04, 8.82146138e-04, 1.49530287e-04, 1.45693043e-04,
       1.48585091e-04, 1.20943125e-04, 1.31030054e-04, 1.16564245e-04,
       7.50986587e-04, 1.31889778e-04, 1.22660391e-04, 1.27898994e-04,
       1.41111601e-04, 1.39650584e-04, 1.47603038e-04, 8.73583548e-04,
       1.43723078e-04, 1.51957096e-04, 1.49538152e-04, 1.22403435e-04,
       1.15548514e-04, 7.41207592e-04, 1.31459633e-04, 1.15362518e-04,
       1.23595648e-04, 1.32837843e-04, 1.27235917e-04, 1.30379601e-04,
       1.31459633e-04, 8.08451915e-04, 1.40861270e-04, 1.36128770e-04,
       1.42386724e-04, 1.36111458e-04, 1.37550262e-04, 1.31889778e-04,
       8.18188151e-04, 1.38720944e-04, 1.34261674e-04, 1.39654036e-04,
       1.37161966e-04, 1.35508114e-04, 1.38720944e-04, 7.96193096e-04,
       1.28374453e-04, 1.32560129e-04, 1.23867490e-04, 1.32632892e-04,
       1.35243194e-04, 1.28374453e-04, 7.78962142e-04, 1.31801268e-04,
       1.25046911e-04, 1.25863424e-04, 1.35670239e-04, 1.38138935e-04,
       1.31801268e-04, 8.15571841e-04, 1.40094812e-04, 1.32020869e-04,
       1.37845716e-04, 1.33956087e-04, 1.38777438e-04, 1.37326644e-04,
       8.21722102e-04, 1.45423101e-04, 1.31481863e-04, 1.34756969e-04,
       1.48780323e-04, 1.42621437e-04, 1.23353729e-04, 9.24763165e-04,
       1.40753485e-04, 9.67628295e-05, 1.48026058e-04, 1.24465303e-04,
       1.42534851e-04, 1.54667981e-04, 1.46745135e-04, 8.95082089e-04,
       1.56407766e-04, 1.48598428e-04, 1.46127928e-04, 1.45212118e-04,
       1.49677870e-04, 1.45362551e-04, 1.50712837e-04, 9.27270392e-04,
       1.23101467e-04, 1.22216327e-04, 9.09872217e-05, 1.50544634e-04,
       1.50115140e-04, 1.44744556e-04, 8.86073329e-04, 1.47688753e-04,
       1.44803278e-04, 1.48176968e-04, 1.47509782e-04, 1.47801103e-04,
       9.00490178e-04, 1.49963004e-04, 1.50949761e-04, 1.52480982e-04,
       1.51785546e-04, 1.48613985e-04, 1.47356779e-04, 1.49963004e-04,
       8.93123528e-04, 1.46996384e-04, 1.50951395e-04, 1.49241981e-04,
       1.46850727e-04, 1.46985008e-04, 1.46996384e-04, 8.85653656e-04,
       1.47693340e-04, 1.48282761e-04, 1.48845437e-04, 1.44628223e-04,
       1.41906866e-04, 1.46031135e-04, 1.47693340e-04, 8.83480880e-04,
       1.52140234e-04, 1.51081083e-04, 1.48228200e-04, 1.47994184e-04,
       1.50949761e-04, 8.97755734e-04, 1.48341003e-04, 1.52308664e-04,
       1.49933922e-04, 1.40228161e-04, 1.38957703e-04, 8.49517733e-04,
       1.43320352e-04, 1.40962172e-04, 1.42480812e-04, 1.43568534e-04,
       1.43324539e-04, 1.47213991e-04, 1.43320352e-04, 8.73476862e-04,
       1.49354588e-04, 1.44059304e-04, 1.46204087e-04, 1.38035773e-04,
       1.36865259e-04, 1.40962172e-04, 8.36982653e-04, 1.39002027e-04,
       1.41453527e-04, 1.40663896e-04, 1.46221367e-04, 1.44029103e-04,
       1.43131074e-04, 1.48341003e-04, 8.73935251e-04, 1.44597519e-04,
       1.47615185e-04, 1.46902156e-04, 1.51435727e-04, 1.36702801e-04,
       1.23101467e-04, 7.61265980e-04, 1.10629367e-04, 9.24944615e-05,
       1.35188173e-04, 1.30954411e-04, 1.39002027e-04, 8.14656964e-04,
       1.32412494e-04, 1.39727814e-04, 1.37372044e-04, 1.41740145e-04,
       1.42729347e-04, 1.38503936e-04, 1.41092473e-04, 8.39623932e-04,
       1.35990147e-04, 1.39567886e-04, 1.57577906e-04, 1.55015153e-04,
       9.41471612e-04, 1.58384492e-04, 1.56136209e-04, 1.57336161e-04,
       1.57021691e-04, 1.62090652e-04, 1.66611326e-04, 1.67369324e-04,
       1.58384492e-04, 9.73853072e-04, 1.61930718e-04, 1.57466559e-04,
       1.53100106e-04, 1.51257570e-04, 1.56136209e-04, 9.20215410e-04,
       1.51534525e-04, 1.55473074e-04, 1.52713926e-04, 1.50199781e-04,
       1.48966277e-04, 1.51534525e-04, 9.02399507e-04, 1.49826294e-04,
       1.51173679e-04, 1.50698952e-04, 1.48795551e-04, 1.48149110e-04,
       1.49826294e-04, 8.95492010e-04, 1.48930081e-04, 1.50020374e-04,
       1.49770601e-04, 1.48132319e-04, 1.49530287e-04, 1.48930081e-04,
       8.94917298e-04, 1.50315639e-04, 1.49010691e-04, 1.48998281e-04,
       1.62374523e-04, 1.50495533e-04, 1.56407766e-04, 8.97778899e-04,
       1.41032098e-04, 1.45482829e-04, 1.41986151e-04, 1.01315555e-04,
       8.84759998e-05, 1.22660391e-04, 6.55702213e-04, 1.00166504e-04,
       1.26369047e-04, 1.16714716e-04, 1.31673149e-04, 1.26770217e-04,
       1.36881253e-04, 1.41032098e-04, 8.20098456e-04, 1.42246914e-04,
       1.41494826e-04, 1.41620839e-04, 1.43723078e-04, 1.42246914e-04,
       8.71873706e-04, 1.49409599e-04, 1.44906415e-04, 1.49966862e-04,
       1.46029431e-04, 1.39034987e-04, 1.40861270e-04, 8.75129226e-04,
       1.52557498e-04, 1.45972127e-04, 1.50673912e-04, 1.48266620e-04,
       1.49162049e-04, 1.52557498e-04, 8.92337809e-04, 1.48214588e-04,
       1.49687697e-04, 1.44449357e-04, 1.51248624e-04, 1.51957096e-04,
       1.48214588e-04, 9.07829208e-04, 1.54280803e-04, 1.48385177e-04,
       1.53742921e-04, 1.06583408e-04, 1.15362518e-04, 1.00166504e-04,
       6.62351816e-04, 1.15667406e-04, 1.08925093e-04, 1.15646886e-04,
       1.41511071e-04, 1.40094812e-04, 1.44597519e-04, 8.60102164e-04,
       1.41924820e-04, 1.46615191e-04, 1.45358750e-04, 1.35006857e-04,
       1.32594537e-04, 7.81944453e-04, 1.30048216e-04, 1.28520824e-04,
       1.27444545e-04, 1.28329473e-04, 1.48971248e-04, 1.61910501e-04,
       1.30048216e-04, 8.44448902e-04, 1.44325463e-04, 1.28927740e-04,
       1.30265734e-04, 1.27958670e-04, 1.30108038e-04, 1.28520824e-04,
       7.72851059e-04, 1.30485176e-04, 1.27981684e-04, 1.27796668e-04,
       1.38227945e-04, 1.39942580e-04, 1.30485176e-04, 9.28959649e-04,
       1.37313320e-04, 1.28388783e-04, 1.30510384e-04, 1.24091461e-04,
       1.27987468e-04, 1.32530668e-04, 1.37313320e-04, 8.23621724e-04,
       1.47338234e-04, 1.36485160e-04, 1.41966874e-04, 1.45885694e-04,
       1.45423101e-04, 1.47338234e-04, 1.02291292e-03, 1.41455416e-04,
       1.50476873e-04, 1.43660575e-04, 1.48673029e-04, 1.36970585e-04,
       1.31481863e-04, 1.40753485e-04, 7.97237452e-04, 1.28009853e-04,
       1.33638289e-04, 1.26383378e-04, 1.29002440e-04, 1.29379502e-04,
       7.97175441e-04, 1.30762276e-04, 1.37294477e-04, 1.38445943e-04,
       1.32290804e-04, 1.34977148e-04, 1.37961004e-04, 1.30762276e-04,
       8.01039673e-04, 1.35951757e-04, 1.29590932e-04, 1.31796557e-04,
       1.40138715e-04, 1.40888080e-04, 1.35951757e-04, 8.33150994e-04,
       1.39807791e-04, 1.36628991e-04, 1.39735660e-04, 1.39404967e-04,
       1.40046051e-04, 1.39807791e-04, 8.45740771e-04, 1.42629470e-04,
       1.40835949e-04, 1.43016544e-04, 1.41643345e-04, 1.45651243e-04,
       1.42629470e-04, 8.76838349e-04, 1.51142633e-04, 1.45633196e-04,
       1.50138462e-04, 1.35332715e-04, 1.35267636e-04, 8.37088011e-04,
       1.38990497e-04, 1.40327792e-04, 1.42883498e-04, 1.44285872e-04,
       1.30247054e-04, 1.29651108e-04, 1.37294477e-04, 1.38990497e-04,
       8.22380072e-04, 1.42548431e-04, 1.43648505e-04, 1.50759385e-04,
       1.48598428e-04, 1.51142633e-04, 9.06432624e-04, 1.47795622e-04,
       1.54661305e-04, 1.53475251e-04, 1.35389257e-04, 1.17098814e-04,
       1.40327792e-04, 7.82418025e-04, 1.15198296e-04, 1.40621459e-04,
       1.33782406e-04, 9.77328153e-05, 9.67628295e-05, 1.15198296e-04,
       5.24706407e-04, 9.87724359e-05, 1.16240030e-04, 1.48499490e-04,
       1.47507884e-04, 1.47688753e-04, 8.88044483e-04, 1.46650628e-04,
       1.48872123e-04, 1.48825605e-04, 1.47608269e-04, 1.46737461e-04,
       1.46650628e-04, 8.77918305e-04, 1.46134726e-04, 1.45216158e-04,
       1.45571064e-04, 1.48387888e-04, 1.22216327e-04, 1.46134726e-04,
       7.62672542e-04, 9.35573675e-05, 1.39391015e-04, 1.12985218e-04,
       1.34500200e-04, 1.38135037e-04, 1.35990147e-04, 8.34092739e-04,
       1.42179767e-04, 1.40366402e-04, 1.42921187e-04, 1.41569331e-04,
       1.44803278e-04, 1.42179767e-04, 8.74591480e-04, 1.48670495e-04,
       1.47055913e-04, 1.50312695e-04, 1.50776574e-04, 1.52140234e-04,
       1.49354588e-04, 9.03778577e-04, 1.51911269e-04, 1.49201446e-04,
       1.50394467e-04, 1.28222645e-04, 1.10629367e-04, 1.32412494e-04,
       7.42237184e-04, 1.11730494e-04, 1.31165453e-04, 1.28076730e-04,
       1.43117427e-04, 1.43088984e-04, 1.43419215e-04, 8.62952256e-04,
       1.44124221e-04, 1.44262923e-04, 1.44939486e-04, 1.42271065e-04,
       1.42139735e-04, 1.44124221e-04, 8.75148939e-04, 1.46331410e-04,
       1.48971994e-04, 1.51310513e-04, 1.42207463e-04, 1.39567886e-04,
       1.40366402e-04, 1.46331410e-04, 8.63821666e-04, 1.46011537e-04,
       1.49336968e-04, 1.44296849e-04, 1.45693043e-04, 1.44262923e-04,
       8.73720467e-04, 1.46369602e-04, 1.46193782e-04, 1.46904267e-04,
       1.48585091e-04, 1.50315639e-04, 1.46369602e-04, 8.90757986e-04,
       1.50824856e-04, 1.45969047e-04, 1.48693752e-04, 1.27898994e-04,
       1.34261674e-04, 1.26369047e-04, 7.94214210e-04, 1.38222320e-04,
       1.30985790e-04, 1.36476385e-04, 1.49538152e-04, 1.49409599e-04,
       1.54280803e-04, 9.25285011e-04, 1.55745134e-04, 1.57359220e-04,
       1.58952103e-04, 1.23595648e-04, 1.15667406e-04, 7.49734148e-04,
       1.31629728e-04, 1.20863619e-04, 1.30407806e-04, 1.27569940e-04,
       1.32837843e-04, 1.36128770e-04, 1.31629728e-04, 8.18642701e-04,
       1.41739983e-04, 1.34743525e-04, 1.41562852e-04, 1.42386724e-04,
       1.45972127e-04, 1.41739983e-04, 8.81470837e-04, 1.52938743e-04,
       1.45409952e-04, 1.53023308e-04, 1.39654036e-04, 1.32560129e-04,
       1.38222320e-04, 9.66594856e-04, 1.09838688e-04, 1.49184205e-04,
       1.33543258e-04, 1.63592220e-04, 1.23867490e-04, 1.25046911e-04,
       1.09838688e-04, 5.68257788e-04, 1.11338646e-04, 9.81660533e-05,
       1.25863424e-04, 1.32020869e-04, 1.11338646e-04, 7.51055728e-04,
       1.35514722e-04, 1.14149718e-04, 1.32168349e-04, 1.37845716e-04,
       1.41924820e-04, 1.35514722e-04, 8.39106968e-04, 1.44755300e-04,
       1.36952467e-04, 1.42113942e-04, 1.61446092e-04, 1.61930718e-04,
       1.44325463e-04, 8.97465336e-04, 1.54437821e-04, 1.32848755e-04,
       1.42476487e-04, 1.34756969e-04, 1.41455416e-04, 1.28009853e-04,
       7.81122431e-04, 1.32090578e-04, 1.23713669e-04, 1.21095947e-04,
       1.48026058e-04, 1.33638289e-04, 8.03709723e-04, 1.45042495e-04,
       1.20721453e-04, 1.23174078e-04, 1.33107351e-04, 1.46127928e-04,
       1.45482829e-04, 1.47795622e-04, 8.92670067e-04, 1.46265520e-04,
       1.53941585e-04, 1.53056582e-04, 1.24465303e-04, 9.87724359e-05,
       1.45042495e-04, 7.59496271e-04, 1.16618775e-04, 1.38664057e-04,
       1.35933205e-04, 9.09872217e-05, 9.24944615e-05, 9.35573675e-05,
       1.11730494e-04, 5.01563552e-04, 1.12794007e-04, 1.48176968e-04,
       1.48872123e-04, 1.48670495e-04, 8.99835563e-04, 1.49509647e-04,
       1.52231166e-04, 1.52375163e-04, 1.52480982e-04, 1.52308664e-04,
       9.19039881e-04, 1.53008540e-04, 1.52109326e-04, 1.54202737e-04,
       1.54929633e-04, 1.51785546e-04, 1.50951395e-04, 1.53008540e-04,
       9.13981698e-04, 1.50328856e-04, 1.54876446e-04, 1.53030914e-04,
       1.49241981e-04, 1.48282761e-04, 1.50328856e-04, 8.96329155e-04,
       1.48008212e-04, 1.50914385e-04, 1.49552960e-04, 1.48845437e-04,
       1.51081083e-04, 1.51911269e-04, 1.48008212e-04, 8.98302441e-04,
       1.50061926e-04, 1.48394515e-04, 1.49933922e-04, 1.47615185e-04,
       1.46615191e-04, 1.52109326e-04, 8.97236712e-04, 1.48893845e-04,
       1.52069243e-04, 1.42480812e-04, 1.41453527e-04, 8.66603885e-04,
       1.45854762e-04, 1.43180436e-04, 1.47640695e-04, 1.45993654e-04,
       1.43568534e-04, 1.44059304e-04, 1.45854762e-04, 8.75555198e-04,
       1.46321947e-04, 1.47887118e-04, 1.47863533e-04, 1.46204087e-04,
       1.49201446e-04, 1.46321947e-04, 8.86703228e-04, 1.48959314e-04,
       1.48188213e-04, 1.47828221e-04, 1.40663896e-04, 1.39727814e-04,
       1.43180436e-04, 8.48805264e-04, 1.39581176e-04, 1.44157560e-04,
       1.41494382e-04, 1.37372044e-04, 1.31165453e-04, 1.39581176e-04,
       8.11118027e-04, 1.30775344e-04, 1.37411626e-04, 1.34812384e-04,
       1.57336161e-04, 1.55473074e-04, 9.31713096e-04, 1.56402814e-04,
       1.54047581e-04, 1.53980659e-04, 1.54472806e-04, 1.57021691e-04,
       1.57466559e-04, 1.54437821e-04, 1.56402814e-04, 9.32898139e-04,
       1.52579696e-04, 1.54989557e-04, 1.52713926e-04, 1.51173679e-04,
       1.54047581e-04, 9.16737244e-04, 1.52195171e-04, 1.53459525e-04,
       1.53147363e-04, 1.50698952e-04, 1.50020374e-04, 1.52195171e-04,
       9.08688326e-04, 1.50328974e-04, 1.53316237e-04, 1.52128619e-04,
       1.49770601e-04, 1.49010691e-04, 1.50328974e-04, 8.98211895e-04,
       1.49437776e-04, 1.49897571e-04, 1.49766283e-04, 1.48998281e-04,
       1.50824856e-04, 1.49437776e-04, 9.02154768e-04, 1.52055683e-04,
       1.50023421e-04, 1.50814753e-04, 1.41986151e-04, 1.41494826e-04,
       1.44906415e-04, 1.46265520e-04, 8.80391623e-04, 1.52435465e-04,
       1.53303245e-04, 1.16714716e-04, 1.08925093e-04, 1.30985790e-04,
       7.27379987e-04, 1.16080015e-04, 1.30895262e-04, 1.23779109e-04,
       1.49966862e-04, 1.55745134e-04, 1.52435465e-04, 9.35227906e-04,
       1.57466853e-04, 1.60180239e-04, 1.59433354e-04, 1.50673912e-04,
       1.49687697e-04, 1.52938743e-04, 8.85283178e-04, 1.43654771e-04,
       1.48312906e-04, 1.40015149e-04, 1.44449357e-04, 1.48385177e-04,
       1.43654771e-04, 8.39074999e-04, 1.44387117e-04, 1.30701025e-04,
       1.27497551e-04, 1.53742921e-04, 1.57359220e-04, 1.44387117e-04,
       9.12753143e-04, 1.59144286e-04, 1.43489365e-04, 1.54630235e-04,
       1.15646886e-04, 1.20863619e-04, 1.16080015e-04, 7.35055694e-04,
       1.28224685e-04, 1.23656276e-04, 1.30584212e-04, 1.45358750e-04,
       1.44755300e-04, 1.48893845e-04, 8.84932067e-04, 1.46315866e-04,
       1.50791418e-04, 1.48816888e-04, 1.27444545e-04, 1.28927740e-04,
       7.66965395e-04, 1.26159824e-04, 1.28607645e-04, 1.28395134e-04,
       1.27430508e-04, 1.28329473e-04, 1.27981684e-04, 1.26159824e-04,
       7.62448341e-04, 1.25842055e-04, 1.27052642e-04, 1.27082663e-04,
       1.30265734e-04, 1.32848755e-04, 1.28607645e-04, 7.85030349e-04,
       1.33157240e-04, 1.29092200e-04, 1.31058775e-04, 1.27796668e-04,
       1.28388783e-04, 1.25842055e-04, 7.68845050e-04, 1.25135233e-04,
       1.31490624e-04, 1.30191687e-04, 1.30510384e-04, 1.36485160e-04,
       7.71930941e-04, 1.35361376e-04, 1.17943254e-04, 1.20093711e-04,
       1.31537057e-04, 1.41966874e-04, 1.50476873e-04, 1.35361376e-04,
       8.41279596e-04, 1.40475265e-04, 1.38197658e-04, 1.34801549e-04,
       1.43660575e-04, 1.32090578e-04, 7.98286499e-04, 1.36679925e-04,
       1.25738751e-04, 1.29743923e-04, 1.30372746e-04, 1.26383378e-04,
       1.23713669e-04, 1.20721453e-04, 7.19023553e-04, 1.16294346e-04,
       1.16834052e-04, 1.15076655e-04, 1.38445943e-04, 1.42548431e-04,
       8.53400763e-04, 1.39274147e-04, 1.44937625e-04, 1.44877802e-04,
       1.43316813e-04, 1.32290804e-04, 1.29590932e-04, 1.39274147e-04,
       8.05711809e-04, 1.33038451e-04, 1.37526649e-04, 1.33990825e-04,
       1.31796557e-04, 1.36628991e-04, 1.33038451e-04, 8.05826008e-04,
       1.37014199e-04, 1.34102249e-04, 1.33245561e-04, 1.39735660e-04,
       1.40835949e-04, 1.37014199e-04, 8.23246983e-04, 1.37259738e-04,
       1.34628094e-04, 1.33773344e-04, 1.43016544e-04, 1.45633196e-04,
       1.37259738e-04, 8.49218266e-04, 1.47307687e-04, 1.34284902e-04,
       1.41716199e-04, 1.50138462e-04, 1.54661305e-04, 1.47307687e-04,
       9.08268820e-04, 1.56783274e-04, 1.47212675e-04, 1.52165419e-04,
       1.42883498e-04, 1.43648505e-04, 1.44937625e-04, 8.67286510e-04,
       1.46524431e-04, 1.43470319e-04, 1.45822131e-04, 1.44285872e-04,
       1.40621459e-04, 1.46524431e-04, 8.75147963e-04, 1.44729993e-04,
       1.48558116e-04, 1.50428092e-04, 1.53475251e-04, 1.53941585e-04,
       1.56783274e-04, 9.37084331e-04, 1.57713550e-04, 1.57353364e-04,
       1.57817307e-04, 1.33782406e-04, 1.16240030e-04, 1.16618775e-04,
       1.44729993e-04, 7.98346388e-04, 1.38203171e-04, 1.48772013e-04,
       1.48825605e-04, 1.45216158e-04, 1.49509647e-04, 8.83719514e-04,
       1.42817194e-04, 1.50216958e-04, 1.47133952e-04, 1.45571064e-04,
       1.39391015e-04, 1.42817194e-04, 8.32773045e-04, 1.29034500e-04,
       1.34178313e-04, 1.41780958e-04, 1.12985218e-04, 1.28076730e-04,
       1.12794007e-04, 1.30775344e-04, 1.29034500e-04, 7.45207702e-04,
       1.31541902e-04, 1.42921187e-04, 1.47055913e-04, 1.46011537e-04,
       9.02912822e-04, 1.56488013e-04, 1.52047212e-04, 1.58388960e-04,
       1.50312695e-04, 1.52231166e-04, 1.56488013e-04, 9.28994227e-04,
       1.48812522e-04, 1.65371896e-04, 1.55777935e-04, 1.50394467e-04,
       1.50061926e-04, 1.48959314e-04, 8.96266820e-04, 1.49311205e-04,
       1.48427738e-04, 1.49112170e-04, 1.44939486e-04, 1.48971994e-04,
       1.46193782e-04, 8.96546224e-04, 1.54682185e-04, 1.48651441e-04,
       1.53107336e-04, 1.51310513e-04, 1.49336968e-04, 1.52047212e-04,
       1.54682185e-04, 9.32271433e-04, 1.62116564e-04, 1.62777992e-04,
       1.46904267e-04, 1.45969047e-04, 1.48651441e-04, 8.79767139e-04,
       1.43801975e-04, 1.49177327e-04, 1.45263082e-04, 1.48693752e-04,
       1.52055683e-04, 1.43801975e-04, 8.86886667e-04, 1.51476311e-04,
       1.43273270e-04, 1.47585676e-04, 1.36476385e-04, 1.49184205e-04,
       1.30895262e-04, 8.30747783e-04, 1.29452486e-04, 1.49445021e-04,
       1.35294424e-04, 1.58952103e-04, 1.57466853e-04, 1.59144286e-04,
       9.33410512e-04, 1.54312497e-04, 1.53440656e-04, 1.50094117e-04,
       1.30407806e-04, 1.34743525e-04, 8.06038896e-04, 1.32962659e-04,
       1.36559703e-04, 1.35313264e-04, 1.36051938e-04, 1.27569940e-04,
       1.28224685e-04, 1.32962659e-04, 8.03967825e-04, 1.35329743e-04,
       1.39084230e-04, 1.40796568e-04, 1.41562852e-04, 1.45409952e-04,
       1.36559703e-04, 8.46226832e-04, 1.44579747e-04, 1.36970817e-04,
       1.41143761e-04, 1.53023308e-04, 1.48312906e-04, 1.44579747e-04,
       9.00302603e-04, 1.08304938e-04, 1.49411580e-04, 7.88230593e-05,
       1.17847064e-04, 1.33543258e-04, 9.81660533e-05, 1.14149718e-04,
       7.56192610e-04, 1.31703906e-04, 1.48226346e-04, 1.30403329e-04,
       1.32168349e-04, 1.36952467e-04, 1.31703906e-04, 8.14182203e-04,
       1.40133684e-04, 1.34789069e-04, 1.38434729e-04, 1.42113942e-04,
       1.46315866e-04, 1.40133684e-04, 8.61293655e-04, 1.46121752e-04,
       1.42411133e-04, 1.44197278e-04, 1.42476487e-04, 1.52579696e-04,
       1.33157240e-04, 8.55552067e-04, 1.51718620e-04, 1.33580927e-04,
       1.42039098e-04, 1.24091461e-04, 1.25135233e-04, 1.17943254e-04,
       7.06983607e-04, 1.19989448e-04, 1.09410895e-04, 1.10413316e-04,
       1.48673029e-04, 1.40475265e-04, 1.36679925e-04, 8.09496512e-04,
       1.30919240e-04, 1.25155986e-04, 1.27593066e-04, 1.21095947e-04,
       1.25738751e-04, 1.16294346e-04, 7.14277560e-04, 1.25534890e-04,
       1.10507944e-04, 1.15105683e-04, 1.23174078e-04, 1.16834052e-04,
       7.16478338e-04, 1.28002461e-04, 1.11665414e-04, 1.13402656e-04,
       1.23399677e-04, 1.33107351e-04, 1.38664057e-04, 1.28002461e-04,
       8.08311016e-04, 1.37969183e-04, 1.33078975e-04, 1.37488989e-04,
       1.53056582e-04, 1.53303245e-04, 1.60180239e-04, 1.57713550e-04,
       9.49235388e-04, 1.63600898e-04, 1.61380873e-04, 1.35933205e-04,
       1.38203171e-04, 1.37969183e-04, 8.43983957e-04, 1.46251109e-04,
       1.39807665e-04, 1.45819624e-04, 1.52375163e-04, 1.50216958e-04,
       1.48812522e-04, 8.83517014e-04, 1.46410899e-04, 1.42972445e-04,
       1.42729027e-04, 1.54202737e-04, 1.52069243e-04, 1.50791418e-04,
       9.07725695e-04, 1.52250339e-04, 1.49543266e-04, 1.48868692e-04,
       1.54929633e-04, 1.54876446e-04, 1.52250339e-04, 9.10493441e-04,
       1.52088049e-04, 1.48229039e-04, 1.48119935e-04, 1.53030914e-04,
       1.50914385e-04, 1.52088049e-04, 9.01936235e-04, 1.49559053e-04,
       1.48378151e-04, 1.47965684e-04, 1.49552960e-04, 1.48394515e-04,
       1.49311205e-04, 1.49559053e-04, 8.96303717e-04, 1.50198291e-04,
       1.49287694e-04, 1.47640695e-04, 1.47887118e-04, 9.11163383e-04,
       1.51663354e-04, 1.51844233e-04, 1.56096763e-04, 1.56031220e-04,
       1.45993654e-04, 1.44157560e-04, 1.51663354e-04, 8.94440572e-04,
       1.46459258e-04, 1.54767375e-04, 1.51399372e-04, 1.47863533e-04,
       1.48188213e-04, 1.51844233e-04, 9.00248249e-04, 1.49636850e-04,
       1.52623742e-04, 1.50091678e-04, 1.47828221e-04, 1.48427738e-04,
       1.49636850e-04, 8.94199201e-04, 1.48539638e-04, 1.50731741e-04,
       1.49035012e-04, 1.41494382e-04, 1.37411626e-04, 1.46459258e-04,
       8.55454596e-04, 1.38815171e-04, 1.47417745e-04, 1.43856414e-04,
       1.34812384e-04, 1.34178313e-04, 1.31541902e-04, 1.38815171e-04,
       8.19212970e-04, 1.39249412e-04, 1.40615787e-04, 1.53980659e-04,
       1.53459525e-04, 9.15644988e-04, 1.51969262e-04, 1.53300002e-04,
       1.52393707e-04, 1.50541833e-04, 1.54472806e-04, 1.54989557e-04,
       1.51718620e-04, 1.51969262e-04, 9.11107090e-04, 1.48594726e-04,
       1.49362119e-04, 1.53147363e-04, 1.53316237e-04, 1.53300002e-04,
       9.19202286e-04, 1.52706362e-04, 1.53747418e-04, 1.52984904e-04,
       1.52128619e-04, 1.49897571e-04, 1.52706362e-04, 9.05719410e-04,
       1.48910323e-04, 1.51820763e-04, 1.50255771e-04, 1.49766283e-04,
       1.50023421e-04, 1.48910323e-04, 8.99123578e-04, 1.50919659e-04,
       1.48875847e-04, 1.50628046e-04, 1.50814753e-04, 1.51476311e-04,
       1.50919659e-04, 9.07508749e-04, 1.50625701e-04, 1.52313921e-04,
       1.51358405e-04, 1.23779109e-04, 1.23656276e-04, 1.29452486e-04,
       7.56483857e-04, 1.31897299e-04, 1.22565520e-04, 1.25133167e-04,
       1.59433354e-04, 1.54312497e-04, 1.63600898e-04, 9.44782899e-04,
       1.61043469e-04, 1.51914626e-04, 1.54478054e-04, 1.40015149e-04,
       1.30701025e-04, 1.08304938e-04, 5.57712477e-04, 9.94295053e-05,
       7.92618595e-05, 1.27497551e-04, 1.43489365e-04, 9.94295053e-05,
       7.63338576e-04, 1.45338024e-04, 1.08833713e-04, 1.38750418e-04,
       1.54630235e-04, 1.53440656e-04, 1.45338024e-04, 8.86802663e-04,
       1.47960093e-04, 1.40811004e-04, 1.44622652e-04, 1.30584212e-04,
       1.35329743e-04, 1.31897299e-04, 8.22372070e-04, 1.43376556e-04,
       1.37225266e-04, 1.43958993e-04, 1.63592220e-04, 1.49445021e-04,
       1.48226346e-04, 8.66957046e-04, 1.34940367e-04, 1.34945936e-04,
       1.35807156e-04, 1.48816888e-04, 1.46121752e-04, 1.49543266e-04,
       8.82668577e-04, 1.44753663e-04, 1.47763737e-04, 1.45669271e-04,
       1.28395134e-04, 1.29092200e-04, 7.76813492e-04, 1.28606382e-04,
       1.30349463e-04, 1.30708164e-04, 1.29662149e-04, 1.27430508e-04,
       1.27052642e-04, 1.28606382e-04, 7.75415821e-04, 1.29948533e-04,
       1.30328870e-04, 1.32048887e-04, 1.27082663e-04, 1.31490624e-04,
       1.29948533e-04, 8.22393480e-04, 1.44762417e-04, 1.39351660e-04,
       1.49757583e-04, 1.31058775e-04, 1.33580927e-04, 1.30349463e-04,
       7.96338759e-04, 1.34836817e-04, 1.32273788e-04, 1.34238990e-04,
       1.30191687e-04, 1.19989448e-04, 1.44762417e-04, 9.13226246e-04,
       1.21506506e-04, 1.35168105e-04, 1.32857432e-04, 1.28750651e-04,
       1.20093711e-04, 1.09410895e-04, 7.00937396e-04, 1.28350791e-04,
       1.06363454e-04, 1.24011533e-04, 1.12707012e-04, 1.31537057e-04,
       1.38197658e-04, 1.28350791e-04, 8.11738295e-04, 1.39750156e-04,
       1.34581935e-04, 1.39320698e-04, 1.34801549e-04, 1.30919240e-04,
       1.39750156e-04, 8.09570326e-04, 1.28185467e-04, 1.41798147e-04,
       1.34115767e-04, 1.29743923e-04, 1.25155986e-04, 7.79844013e-04,
       1.33655395e-04, 1.22805986e-04, 1.37372097e-04, 1.31110625e-04,
       1.30372746e-04, 1.25534890e-04, 1.33655395e-04, 7.88896839e-04,
       1.29618194e-04, 1.34457479e-04, 1.35258134e-04, 1.15076655e-04,
       1.10507944e-04, 1.11665414e-04, 7.05824764e-04, 1.25292305e-04,
       1.12543422e-04, 1.30739024e-04, 1.44877802e-04, 1.43470319e-04,
       8.72499618e-04, 1.46445332e-04, 1.44731445e-04, 1.46640562e-04,
       1.46334158e-04, 1.43316813e-04, 1.37526649e-04, 1.46445332e-04,
       8.53637296e-04, 1.38152196e-04, 1.45349639e-04, 1.42846667e-04,
       1.33990825e-04, 1.34102249e-04, 1.38152196e-04, 8.28046912e-04,
       1.38761087e-04, 1.41271544e-04, 1.41769011e-04, 1.33245561e-04,
       1.34628094e-04, 1.38761087e-04, 7.99381292e-04, 1.30330053e-04,
       1.36115032e-04, 1.26301465e-04, 1.33773344e-04, 1.34284902e-04,
       1.30330053e-04, 7.80026889e-04, 1.31477112e-04, 1.24762988e-04,
       1.25398489e-04, 1.41716199e-04, 1.47212675e-04, 1.31477112e-04,
       8.34774339e-04, 1.47806778e-04, 1.27864193e-04, 1.38697383e-04,
       1.52165419e-04, 1.57353364e-04, 1.47806778e-04, 9.20146697e-04,
       1.57556996e-04, 1.50350933e-04, 1.54913207e-04, 1.45822131e-04,
       1.48558116e-04, 1.44731445e-04, 8.89347043e-04, 1.49184008e-04,
       1.49667382e-04, 1.51383960e-04, 1.50428092e-04, 1.48772013e-04,
       1.46251109e-04, 1.49184008e-04, 8.91595519e-04, 1.47841738e-04,
       1.49118559e-04, 1.57817307e-04, 1.61380873e-04, 1.61043469e-04,
       1.57556996e-04, 9.53878519e-04, 1.58001391e-04, 1.58078484e-04,
       1.47133952e-04, 1.41780958e-04, 1.46410899e-04, 1.39249412e-04,
       8.57908443e-04, 1.42570857e-04, 1.40762364e-04, 1.58388960e-04,
       1.65371896e-04, 1.62116564e-04, 1.07865684e-03, 1.44594215e-04,
       1.72242712e-04, 1.24155165e-04, 1.51787328e-04, 1.55777935e-04,
       1.42972445e-04, 1.44594215e-04, 8.30244435e-04, 1.40181293e-04,
       1.19162990e-04, 1.27555557e-04, 1.49112170e-04, 1.50198291e-04,
       1.48539638e-04, 8.99822826e-04, 1.50722442e-04, 1.50076801e-04,
       1.51173484e-04, 1.53107336e-04, 1.62777992e-04, 1.49177327e-04,
       1.72242712e-04, 9.39986168e-04, 1.44643037e-04, 1.58037765e-04,
       1.45263082e-04, 1.43273270e-04, 1.44643037e-04, 8.55075867e-04,
       1.43128153e-04, 1.39146698e-04, 1.39621627e-04, 1.47585676e-04,
       1.50625701e-04, 1.43128153e-04, 8.88229071e-04, 1.51061070e-04,
       1.45467790e-04, 1.50360681e-04, 1.35294424e-04, 1.22565520e-04,
       1.34940367e-04, 7.66070359e-04, 1.18436509e-04, 1.29304246e-04,
       1.25529292e-04, 1.50094117e-04, 1.51914626e-04, 1.47960093e-04,
       8.98582771e-04, 1.52264205e-04, 1.46933063e-04, 1.49416666e-04,
       1.35313264e-04, 1.36970817e-04, 8.36355415e-04, 1.41399866e-04,
       1.38925212e-04, 1.39807025e-04, 1.43939232e-04, 1.36051938e-04,
       1.39084230e-04, 1.41399866e-04, 8.63019148e-04, 1.47365801e-04,
       1.48091835e-04, 1.51025479e-04, 1.40796568e-04, 1.43376556e-04,
       1.47365801e-04, 8.94407634e-04, 1.52395508e-04, 1.54011741e-04,
       1.56461460e-04, 1.41143761e-04, 1.49411580e-04, 1.38925212e-04,
       8.67383717e-04, 1.50445525e-04, 1.42102573e-04, 1.45355067e-04,
       7.88230593e-05, 7.92618595e-05, 1.08833713e-04, 4.64290563e-04,
       1.13471947e-04, 8.38999844e-05, 1.30403329e-04, 1.34789069e-04,
       1.34945936e-04, 8.13134332e-04, 1.37001486e-04, 1.39162351e-04,
       1.36832162e-04, 1.38434729e-04, 1.42411133e-04, 1.37001486e-04,
       8.36895346e-04, 1.41940327e-04, 1.38072617e-04, 1.39035054e-04,
       1.44197278e-04, 1.44753663e-04, 1.41940327e-04, 8.55063444e-04,
       1.43086294e-04, 1.40248151e-04, 1.40837732e-04, 1.42039098e-04,
       1.48594726e-04, 1.34836817e-04, 8.48284789e-04, 1.47176188e-04,
       1.34927109e-04, 1.40710852e-04, 1.10413316e-04, 1.21506506e-04,
       1.06363454e-04, 6.95795948e-04, 1.23904957e-04, 1.13579701e-04,
       1.20028014e-04, 1.27593066e-04, 1.28185467e-04, 1.22805986e-04,
       7.46341676e-04, 1.26870509e-04, 1.18707263e-04, 1.22179385e-04,
       1.15105683e-04, 1.29618194e-04, 1.25292305e-04, 8.20529230e-04,
       1.39340581e-04, 1.55818351e-04, 1.55354117e-04, 1.13402656e-04,
       1.12543422e-04, 6.89029292e-04, 1.20803380e-04, 1.10122571e-04,
       1.11167844e-04, 1.20989420e-04, 1.23399677e-04, 1.33078975e-04,
       1.20803380e-04, 7.68870768e-04, 1.35469277e-04, 1.25566431e-04,
       1.30553029e-04, 1.37488989e-04, 1.39807665e-04, 1.35469277e-04,
       8.30414880e-04, 1.41853459e-04, 1.35864992e-04, 1.39930498e-04,
       1.45819624e-04, 1.47841738e-04, 1.41853459e-04, 8.71959768e-04,
       1.46970827e-04, 1.43189433e-04, 1.46284687e-04, 1.42729027e-04,
       1.42570857e-04, 1.40181293e-04, 8.49355337e-04, 1.44162752e-04,
       1.37785889e-04, 1.41925518e-04, 1.48868692e-04, 1.48229039e-04,
       1.47763737e-04, 8.87003741e-04, 1.48263991e-04, 1.46369187e-04,
       1.47509095e-04, 1.48119935e-04, 1.48378151e-04, 1.48263991e-04,
       8.93102201e-04, 1.49171119e-04, 1.49260046e-04, 1.49908958e-04,
       1.47965684e-04, 1.49287694e-04, 1.50722442e-04, 1.49171119e-04,
       9.00675791e-04, 1.51879083e-04, 1.51649769e-04, 1.56096763e-04,
       1.54767375e-04, 9.61306033e-04, 1.63438380e-04, 1.55805905e-04,
       1.68750348e-04, 1.62447261e-04, 1.56031220e-04, 1.52623742e-04,
       1.63438380e-04, 1.02575743e-03, 1.73630319e-04, 1.83217175e-04,
       1.96816591e-04, 1.51399372e-04, 1.47417745e-04, 1.55805905e-04,
       9.08818825e-04, 1.46500395e-04, 1.51185762e-04, 1.56509645e-04,
       1.50091678e-04, 1.50731741e-04, 1.73630319e-04, 9.24143041e-04,
       1.45201037e-04, 1.66778805e-04, 1.37709460e-04, 1.49035012e-04,
       1.50076801e-04, 1.45201037e-04, 8.89515380e-04, 1.50093668e-04,
       1.45629010e-04, 1.49479852e-04, 1.43856414e-04, 1.40615787e-04,
       1.40762364e-04, 1.44162752e-04, 1.46500395e-04, 8.62557868e-04,
       1.46660156e-04, 1.52393707e-04, 1.53747418e-04, 9.19746981e-04,
       1.51509121e-04, 1.54034379e-04, 1.54616951e-04, 1.53445405e-04,
       1.50541833e-04, 1.49362119e-04, 1.47176188e-04, 1.51509121e-04,
       8.93917700e-04, 1.46087610e-04, 1.49240830e-04, 1.52984904e-04,
       1.51820763e-04, 1.54034379e-04, 9.16217032e-04, 1.50189589e-04,
       1.54934022e-04, 1.52253374e-04, 1.50255771e-04, 1.48875847e-04,
       1.50189589e-04, 8.94000194e-04, 1.47607401e-04, 1.49136925e-04,
       1.47934660e-04, 1.50628046e-04, 1.52313921e-04, 1.47607401e-04,
       9.00809628e-04, 1.52839993e-04, 1.46936775e-04, 1.50483492e-04,
       1.51358405e-04, 1.51061070e-04, 1.52839993e-04, 9.12931191e-04,
       1.51726664e-04, 1.53380526e-04, 1.52564532e-04, 1.25133167e-04,
       1.37225266e-04, 1.18436509e-04, 8.00581849e-04, 1.45762862e-04,
       1.29394896e-04, 1.44629149e-04, 1.54478054e-04, 1.58001391e-04,
       1.52264205e-04, 9.33774984e-04, 1.58051183e-04, 1.54358255e-04,
       1.56621896e-04, 1.38750418e-04, 1.40811004e-04, 1.13471947e-04,
       7.90901964e-04, 1.41935279e-04, 1.14765284e-04, 1.41168031e-04,
       1.44622652e-04, 1.46933063e-04, 1.41935279e-04, 8.73528645e-04,
       1.47647181e-04, 1.44494489e-04, 1.47895980e-04, 1.43958993e-04,
       1.52395508e-04, 1.45762862e-04, 9.15218162e-04, 1.58602978e-04,
       1.55047109e-04, 1.59450710e-04, 1.17847064e-04, 1.50445525e-04,
       8.38999844e-05, 1.14765284e-04, 7.56480905e-04, 1.45628103e-04,
       1.43894943e-04, 1.35807156e-04, 1.29304246e-04, 1.39162351e-04,
       7.83709201e-04, 1.18885143e-04, 1.37162302e-04, 1.23388003e-04,
       1.45669271e-04, 1.43086294e-04, 1.46369187e-04, 8.65670736e-04,
       1.41704357e-04, 1.45461740e-04, 1.43379887e-04, 1.30708164e-04,
       1.32273788e-04, 7.92380541e-04, 1.30112590e-04, 1.34122093e-04,
       1.33803893e-04, 1.31360013e-04, 1.29662149e-04, 1.30328870e-04,
       1.30112590e-04, 7.71782998e-04, 1.27917379e-04, 1.28311970e-04,
       1.25450039e-04, 1.32048887e-04, 1.39351660e-04, 1.27917379e-04,
       7.23670285e-04, 1.22339685e-04, 1.07446571e-04, 9.45661032e-05,
       1.49757583e-04, 1.35168105e-04, 1.22339685e-04, 9.22889125e-04,
       1.14267277e-04, 9.74940699e-05, 1.55189083e-04, 1.48673322e-04,
       1.34238990e-04, 1.34927109e-04, 1.34122093e-04, 8.10796567e-04,
       1.36270265e-04, 1.34889029e-04, 1.36349082e-04, 1.32857432e-04,
       1.23904957e-04, 7.27466539e-04, 1.19731408e-04, 1.20157743e-04,
       1.10718095e-04, 1.20096905e-04, 1.24011533e-04, 1.34581935e-04,
       7.67575420e-04, 1.17308068e-04, 1.37901140e-04, 1.21875038e-04,
       1.31897707e-04, 1.12707012e-04, 1.13579701e-04, 1.17308068e-04,
       6.98244164e-04, 1.16214395e-04, 1.20200676e-04, 1.18234313e-04,
       1.39320698e-04, 1.41798147e-04, 1.37901140e-04, 8.43677262e-04,
       1.42434133e-04, 1.40083800e-04, 1.42139344e-04, 1.34115767e-04,
       1.26870509e-04, 1.42434133e-04, 8.04904355e-04, 1.25657401e-04,
       1.42679009e-04, 1.33147535e-04, 1.37372097e-04, 1.34457479e-04,
       8.44906627e-04, 1.46902247e-04, 1.36329705e-04, 1.48751512e-04,
       1.41093587e-04, 1.31110625e-04, 1.18707263e-04, 1.46902247e-04,
       8.08156179e-04, 1.18532873e-04, 1.54434591e-04, 1.38468580e-04,
       1.35258134e-04, 1.39340581e-04, 1.36329705e-04, 8.54156819e-04,
       1.46506078e-04, 1.45314197e-04, 1.51408123e-04, 1.30739024e-04,
       1.55818351e-04, 1.10122571e-04, 9.66924888e-04, 1.68651262e-04,
       9.84006643e-05, 1.70555647e-04, 1.32637370e-04, 1.46640562e-04,
       1.45349639e-04, 8.78274798e-04, 1.46811125e-04, 1.44028962e-04,
       1.48467875e-04, 1.46976636e-04, 1.46334158e-04, 1.49667382e-04,
       1.46811125e-04, 8.92899690e-04, 1.50882429e-04, 1.49233257e-04,
       1.49971338e-04, 1.42846667e-04, 1.41271544e-04, 1.44028962e-04,
       8.53083095e-04, 1.39950282e-04, 1.43744598e-04, 1.41241042e-04,
       1.41769011e-04, 1.36115032e-04, 1.39950282e-04, 8.22683710e-04,
       1.33219171e-04, 1.36353673e-04, 1.35276541e-04, 1.26301465e-04,
       1.24762988e-04, 1.33219171e-04, 7.29188674e-04, 1.14957803e-04,
       1.23335069e-04, 1.06612178e-04, 1.25398489e-04, 1.27864193e-04,
       1.14957803e-04, 7.28868592e-04, 1.31647037e-04, 1.07388769e-04,
       1.21612300e-04, 1.38697383e-04, 1.50350933e-04, 1.31647037e-04,
       8.54218399e-04, 1.52799761e-04, 1.35612055e-04, 1.45111229e-04,
       1.54913207e-04, 1.58078484e-04, 1.58051183e-04, 1.52799761e-04,
       9.31274353e-04, 1.54758931e-04, 1.52672787e-04, 1.51383960e-04,
       1.49118559e-04, 1.46970827e-04, 1.50882429e-04, 8.96277496e-04,
       1.48137760e-04, 1.49783961e-04, 1.24155165e-04, 1.19162990e-04,
       6.26885028e-04, 1.26583717e-04, 1.26806164e-04, 1.30176991e-04,
       1.27555557e-04, 1.37785889e-04, 1.26583717e-04, 8.12073647e-04,
       1.41445732e-04, 1.37035534e-04, 1.41667217e-04, 1.51173484e-04,
       1.51879083e-04, 1.50093668e-04, 9.11498084e-04, 1.53018848e-04,
       1.51556710e-04, 1.53776291e-04, 1.51787328e-04, 1.58037765e-04,
       1.39146698e-04, 1.26806164e-04, 8.43749587e-04, 1.37030864e-04,
       1.30940768e-04, 1.39621627e-04, 1.45467790e-04, 1.37030864e-04,
       8.56719620e-04, 1.50000513e-04, 1.38737670e-04, 1.45861156e-04,
       1.50360681e-04, 1.51726664e-04, 1.50000513e-04, 9.08864839e-04,
       1.53584985e-04, 1.50486753e-04, 1.52705242e-04, 1.25529292e-04,
       1.29394896e-04, 1.18885143e-04, 8.07477661e-04, 1.51985146e-04,
       1.26224393e-04, 1.55458792e-04, 1.49416666e-04, 1.54358255e-04,
       1.47647181e-04, 9.10765148e-04, 1.56043980e-04, 1.49921928e-04,
       1.53377139e-04, 1.39807025e-04, 1.42102573e-04, 8.59529110e-04,
       1.44804568e-04, 1.44348706e-04, 1.42857415e-04, 1.45608825e-04,
       1.43939232e-04, 1.48091835e-04, 1.44804568e-04, 8.86925986e-04,
       1.51492318e-04, 1.48031443e-04, 1.50566591e-04, 1.51025479e-04,
       1.54011741e-04, 1.51492318e-04, 9.21738679e-04, 1.56810950e-04,
       1.53032911e-04, 1.55365281e-04, 1.56461460e-04, 1.58602978e-04,
       1.56810950e-04, 9.45275405e-04, 1.59688634e-04, 1.56487609e-04,
       1.57223775e-04, 1.45355067e-04, 1.45628103e-04, 1.44348706e-04,
       8.74687808e-04, 1.47868829e-04, 1.44120008e-04, 1.47367095e-04,
       1.36832162e-04, 1.38072617e-04, 1.37162302e-04, 8.34256785e-04,
       1.37216568e-04, 1.43079663e-04, 1.41893473e-04, 1.39035054e-04,
       1.40248151e-04, 1.37216568e-04, 8.35029161e-04, 1.40295674e-04,
       1.38183853e-04, 1.40049862e-04, 1.40837732e-04, 1.41704357e-04,
       1.40295674e-04, 8.45821713e-04, 1.42145252e-04, 1.39927872e-04,
       1.40910825e-04, 1.28750651e-04, 1.14267277e-04, 1.19731408e-04,
       5.78524280e-04, 1.10526905e-04, 1.05248038e-04, 1.40710852e-04,
       1.46087610e-04, 1.36270265e-04, 8.53274726e-04, 1.47840159e-04,
       1.38086353e-04, 1.44279489e-04, 1.20028014e-04, 1.20157743e-04,
       1.16214395e-04, 7.19330502e-04, 1.20337173e-04, 1.19300064e-04,
       1.23293114e-04, 1.22179385e-04, 1.25657401e-04, 1.18532873e-04,
       7.43174300e-04, 1.31110585e-04, 1.18297180e-04, 1.27396875e-04,
       1.55354117e-04, 1.46506078e-04, 1.68651262e-04, 9.36876384e-04,
       1.51223271e-04, 1.61860804e-04, 1.53280852e-04, 1.11167844e-04,
       9.84006643e-05, 5.11031563e-04, 1.10150631e-04, 9.39303618e-05,
       9.73820617e-05, 1.20989420e-04, 1.25566431e-04, 1.10150631e-04,
       7.26221218e-04, 1.29438959e-04, 1.12682230e-04, 1.27393547e-04,
       1.30553029e-04, 1.35864992e-04, 1.29438959e-04, 8.02426850e-04,
       1.38377884e-04, 1.32282511e-04, 1.35909473e-04, 1.39930498e-04,
       1.43189433e-04, 1.38377884e-04, 8.51204327e-04, 1.45391094e-04,
       1.40280572e-04, 1.44034847e-04, 1.46284687e-04, 1.48137760e-04,
       1.45391094e-04, 8.82787405e-04, 1.48815059e-04, 1.46293957e-04,
       1.47864849e-04, 1.41925518e-04, 1.51185762e-04, 1.46660156e-04,
       1.41445732e-04, 8.80528230e-04, 1.52158227e-04, 1.47152836e-04,
       1.47509095e-04, 1.49260046e-04, 1.45461740e-04, 8.81473783e-04,
       1.48516834e-04, 1.44710962e-04, 1.46015105e-04, 1.49908958e-04,
       1.51649769e-04, 1.53018848e-04, 1.48516834e-04, 9.05275866e-04,
       1.52471329e-04, 1.49710127e-04, 1.68750348e-04, 1.83217175e-04,
       1.03745410e-03, 1.62886436e-04, 1.85069802e-04, 1.64906900e-04,
       1.72623439e-04, 1.62447261e-04, 1.56509645e-04, 1.52158227e-04,
       1.62886436e-04, 9.45664024e-04, 1.53435930e-04, 1.58226523e-04,
       1.96816591e-04, 1.66778805e-04, 1.85069802e-04, 1.21913037e-03,
       1.36713373e-04, 1.81977403e-04, 1.63294102e-04, 1.88480298e-04,
       1.37709460e-04, 1.45629010e-04, 1.36713373e-04, 7.07501586e-04,
       1.46805051e-04, 1.40644692e-04, 1.49479852e-04, 1.51556710e-04,
       1.46805051e-04, 9.03900342e-04, 1.54622885e-04, 1.47847434e-04,
       1.53588411e-04, 1.54616951e-04, 1.54934022e-04, 9.64803095e-04,
       1.58088910e-04, 1.63762463e-04, 1.64022134e-04, 1.69378615e-04,
       1.53445405e-04, 1.49240830e-04, 1.47840159e-04, 1.58088910e-04,
       9.17481568e-04, 1.51411044e-04, 1.57455220e-04, 1.52253374e-04,
       1.49136925e-04, 1.63762463e-04, 9.45179860e-04, 1.45880498e-04,
       1.74456058e-04, 1.59690543e-04, 1.47934660e-04, 1.46936775e-04,
       1.45880498e-04, 8.75096272e-04, 1.46808844e-04, 1.42804633e-04,
       1.44730863e-04, 1.50483492e-04, 1.53380526e-04, 1.46808844e-04,
       8.99500605e-04, 1.53329016e-04, 1.45937794e-04, 1.49560933e-04,
       1.52564532e-04, 1.53584985e-04, 1.53329016e-04, 9.22382181e-04,
       1.55265329e-04, 1.53361230e-04, 1.54277089e-04, 1.44629149e-04,
       1.55047109e-04, 1.51985146e-04, 9.25661513e-04, 1.57542963e-04,
       1.60658644e-04, 1.55798501e-04, 1.56621896e-04, 1.54758931e-04,
       1.56043980e-04, 9.29402445e-04, 1.52171867e-04, 1.55907459e-04,
       1.53898311e-04, 1.41168031e-04, 1.44494489e-04, 1.43894943e-04,
       1.47868829e-04, 8.76721965e-04, 1.49324123e-04, 1.49971550e-04,
       1.47895980e-04, 1.49921928e-04, 1.49324123e-04, 9.06954260e-04,
       1.53034970e-04, 1.52546180e-04, 1.54231079e-04, 1.59450710e-04,
       1.59688634e-04, 1.57542963e-04, 9.49175788e-04, 1.57052968e-04,
       1.58084216e-04, 1.57356297e-04, 1.23388003e-04, 1.26224393e-04,
       1.43079663e-04, 8.28506255e-04, 1.43735054e-04, 1.47130070e-04,
       1.44949072e-04, 1.43379887e-04, 1.42145252e-04, 1.44710962e-04,
       8.60022829e-04, 1.41826079e-04, 1.44805448e-04, 1.43155200e-04,
       1.33803893e-04, 1.34889029e-04, 8.11564899e-04, 1.33429115e-04,
       1.36973494e-04, 1.37464306e-04, 1.35005063e-04, 1.31360013e-04,
       1.28311970e-04, 1.33429115e-04, 7.85470527e-04, 1.26909304e-04,
       1.34465946e-04, 1.30994178e-04, 1.25450039e-04, 1.07446571e-04,
       1.26909304e-04, 6.86867516e-04, 9.99799426e-05, 1.18003776e-04,
       1.09077884e-04, 9.45661032e-05, 9.74940699e-05, 9.99799426e-05,
       4.84687301e-04, 9.36173269e-05, 9.90298580e-05, 1.55189083e-04,
       1.10526905e-04, 1.02184154e-03, 1.36569783e-04, 1.79781247e-04,
       1.45378365e-04, 1.55089868e-04, 1.39306290e-04, 1.36349082e-04,
       1.38086353e-04, 1.36973494e-04, 8.33045714e-04, 1.41091680e-04,
       1.39082096e-04, 1.41463010e-04, 1.10718095e-04, 1.05248038e-04,
       1.36569783e-04, 7.90146421e-04, 1.30539684e-04, 1.59285489e-04,
       1.47785332e-04, 1.20096905e-04, 1.20337173e-04, 1.30539684e-04,
       7.66860118e-04, 1.26901131e-04, 1.36432023e-04, 1.32553202e-04,
       1.21875038e-04, 1.20200676e-04, 7.49257004e-04, 1.27486646e-04,
       1.23074300e-04, 1.26941180e-04, 1.29679164e-04, 1.31897707e-04,
       1.40083800e-04, 1.27486646e-04, 8.13227237e-04, 1.42611653e-04,
       1.32104257e-04, 1.39043172e-04, 1.18234313e-04, 1.19300064e-04,
       1.23074300e-04, 7.40525344e-04, 1.25295603e-04, 1.26732756e-04,
       1.27888308e-04, 1.42139344e-04, 1.42679009e-04, 1.42611653e-04,
       8.62547734e-04, 1.44962572e-04, 1.44171951e-04, 1.45983204e-04,
       1.33147535e-04, 1.31110585e-04, 1.44962572e-04, 8.58241309e-04,
       1.45258304e-04, 1.50714815e-04, 1.53047498e-04, 1.48751512e-04,
       1.54434591e-04, 9.25445223e-04, 1.51550789e-04, 1.56360983e-04,
       1.57610117e-04, 1.56737231e-04, 1.41093587e-04, 1.45314197e-04,
       1.51550789e-04, 9.13595938e-04, 1.55991157e-04, 1.59713224e-04,
       1.59932982e-04, 1.38468580e-04, 1.18297180e-04, 1.56360983e-04,
       8.12743811e-04, 1.09889042e-04, 1.58014283e-04, 1.31713743e-04,
       1.51408123e-04, 1.51223271e-04, 1.55991157e-04, 9.30732283e-04,
       1.53882640e-04, 1.60075378e-04, 1.58151713e-04, 1.70555647e-04,
       1.61860804e-04, 9.54929698e-04, 1.63181907e-04, 1.53610500e-04,
       1.53298702e-04, 1.52422138e-04, 1.32637370e-04, 9.39303618e-05,
       1.63181907e-04, 9.17251177e-04, 1.15253610e-04, 1.41860896e-04,
       1.37162837e-04, 1.33224197e-04, 1.48467875e-04, 1.49233257e-04,
       8.93642581e-04, 1.48020821e-04, 1.49132670e-04, 1.49567213e-04,
       1.49220746e-04, 1.46976636e-04, 1.43744598e-04, 1.48020821e-04,
       8.73042638e-04, 1.42110637e-04, 1.47434046e-04, 1.44755900e-04,
       1.49971338e-04, 1.49783961e-04, 1.48815059e-04, 1.49132670e-04,
       8.95643833e-04, 1.48905285e-04, 1.49035520e-04, 1.41241042e-04,
       1.36353673e-04, 1.42110637e-04, 8.38482963e-04, 1.33304057e-04,
       1.44696383e-04, 1.40777171e-04, 1.35276541e-04, 1.23335069e-04,
       1.33304057e-04, 7.56141264e-04, 1.09264387e-04, 1.33529704e-04,
       1.21431506e-04, 1.06612178e-04, 1.07388769e-04, 1.09264387e-04,
       5.46434268e-04, 1.11435724e-04, 1.11733210e-04, 1.21612300e-04,
       1.35612055e-04, 1.11435724e-04, 7.63720065e-04, 1.37578654e-04,
       1.22669078e-04, 1.34812254e-04, 1.45111229e-04, 1.52672787e-04,
       1.52171867e-04, 1.37578654e-04, 8.78208502e-04, 1.48852810e-04,
       1.41821155e-04, 1.30176991e-04, 1.37035534e-04, 1.30940768e-04,
       1.38737670e-04, 8.19906489e-04, 1.41038583e-04, 1.41976943e-04,
       1.41667217e-04, 1.47152836e-04, 1.53435930e-04, 1.41038583e-04,
       8.83203332e-04, 1.46498518e-04, 1.53410247e-04, 1.53776291e-04,
       1.52471329e-04, 1.54622885e-04, 9.14051283e-04, 1.50966114e-04,
       1.52283236e-04, 1.49931427e-04, 1.45861156e-04, 1.50486753e-04,
       1.41976943e-04, 1.46498518e-04, 8.92953027e-04, 1.54116839e-04,
       1.54012817e-04, 1.52705242e-04, 1.55265329e-04, 1.54116839e-04,
       9.39150426e-04, 1.58094963e-04, 1.60193045e-04, 1.58775009e-04,
       1.55458792e-04, 1.60658644e-04, 1.43735054e-04, 1.01226677e-03,
       1.53720613e-04, 1.42605857e-04, 1.30692455e-04, 1.25395355e-04,
       1.53377139e-04, 1.55907459e-04, 1.53034970e-04, 9.30817088e-04,
       1.56199222e-04, 1.55832183e-04, 1.56466115e-04, 1.42857415e-04,
       1.44120008e-04, 8.78141783e-04, 1.46183923e-04, 1.47573433e-04,
       1.48640043e-04, 1.48766961e-04, 1.45608825e-04, 1.48031443e-04,
       1.46183923e-04, 8.91192198e-04, 1.51066793e-04, 1.48920482e-04,
       1.51380733e-04, 1.50566591e-04, 1.53032911e-04, 1.51066793e-04,
       9.14874511e-04, 1.54133520e-04, 1.52737145e-04, 1.53337552e-04,
       1.55365281e-04, 1.56487609e-04, 1.54133520e-04, 9.27057988e-04,
       1.54669932e-04, 1.53493782e-04, 1.52907865e-04, 1.57223775e-04,
       1.57052968e-04, 1.54669932e-04, 9.23949713e-04, 1.52747546e-04,
       1.52003536e-04, 1.50251957e-04, 1.47367095e-04, 1.49971550e-04,
       1.52546180e-04, 1.47573433e-04, 9.01920115e-04, 1.53415074e-04,
       1.51046782e-04, 1.41893473e-04, 1.38183853e-04, 1.47130070e-04,
       8.49800776e-04, 1.37268592e-04, 1.45738323e-04, 1.39586465e-04,
       1.40049862e-04, 1.39927872e-04, 1.37268592e-04, 8.27967178e-04,
       1.37951289e-04, 1.35982438e-04, 1.36787125e-04, 1.40910825e-04,
       1.41826079e-04, 1.37951289e-04, 8.34369692e-04, 1.40259994e-04,
       1.36014027e-04, 1.37407478e-04, 1.48673322e-04, 9.36173269e-05,
       1.79781247e-04, 9.40961418e-04, 9.54742542e-05, 1.60023619e-04,
       1.16914130e-04, 1.46477519e-04, 1.44279489e-04, 1.51411044e-04,
       1.41091680e-04, 8.85063023e-04, 1.54952023e-04, 1.43300000e-04,
       1.50028788e-04, 1.23293114e-04, 1.26901131e-04, 1.25295603e-04,
       7.81027752e-04, 1.34435797e-04, 1.32784965e-04, 1.38317142e-04,
       1.27396875e-04, 1.45258304e-04, 1.09889042e-04, 8.16016354e-04,
       1.61594976e-04, 1.19016325e-04, 1.52860830e-04, 1.53280852e-04,
       1.53882640e-04, 1.53610500e-04, 9.20398387e-04, 1.55376951e-04,
       1.51541390e-04, 1.52706053e-04, 9.73820617e-05, 1.12682230e-04,
       1.15253610e-04, 7.26000113e-04, 1.30726605e-04, 1.34891390e-04,
       1.35064217e-04, 1.27393547e-04, 1.32282511e-04, 1.30726605e-04,
       8.03441298e-04, 1.37056732e-04, 1.37270370e-04, 1.38711533e-04,
       1.35909473e-04, 1.40280572e-04, 1.37056732e-04, 8.43534979e-04,
       1.44199401e-04, 1.41658615e-04, 1.44430186e-04, 1.44034847e-04,
       1.46293957e-04, 1.44199401e-04, 8.73836445e-04, 1.46807885e-04,
       1.46075491e-04, 1.46424864e-04, 1.47864849e-04, 1.48905285e-04,
       1.46807885e-04, 8.84368611e-04, 1.47764350e-04, 1.46555107e-04,
       1.46471135e-04, 1.46015105e-04, 1.49710127e-04, 1.44805448e-04,
       1.50966114e-04, 8.84638101e-04, 1.45337832e-04, 1.47803476e-04,
       1.64906900e-04, 1.58226523e-04, 1.53410247e-04, 1.54012817e-04,
       1.58094963e-04, 9.52824219e-04, 1.64172769e-04, 1.72623439e-04,
       1.81977403e-04, 1.60193045e-04, 1.64172769e-04, 1.01814197e-03,
       1.62920812e-04, 1.76254499e-04, 1.63294102e-04, 1.40644692e-04,
       1.47847434e-04, 9.25797234e-04, 1.53858168e-04, 1.68395757e-04,
       1.51757081e-04, 1.53588411e-04, 1.52283236e-04, 1.53858168e-04,
       9.07774061e-04, 1.48525259e-04, 1.51773360e-04, 1.47745626e-04,
       1.64022134e-04, 1.57455220e-04, 1.54952023e-04, 9.67767828e-04,
       1.68833884e-04, 1.57594810e-04, 1.64909758e-04, 1.69378615e-04,
       1.74456058e-04, 1.68833884e-04, 1.10687113e-03, 1.49721131e-04,
       1.73069266e-04, 1.22518130e-04, 1.48894045e-04, 1.59690543e-04,
       1.42804633e-04, 1.49721131e-04, 8.45024831e-04, 1.39694216e-04,
       1.23127657e-04, 1.29986652e-04, 1.44730863e-04, 1.45937794e-04,
       1.39694216e-04, 8.51325022e-04, 1.44938318e-04, 1.35993330e-04,
       1.40030501e-04, 1.49560933e-04, 1.53361230e-04, 1.44938318e-04,
       8.96246900e-04, 1.53994979e-04, 1.44567538e-04, 1.49823901e-04,
       1.54277089e-04, 1.58775009e-04, 1.62920812e-04, 1.53994979e-04,
       9.44551861e-04, 1.59505942e-04, 1.55078029e-04, 1.55798501e-04,
       1.58084216e-04, 1.53720613e-04, 9.41447815e-04, 1.57685205e-04,
       1.57240201e-04, 1.58919079e-04, 1.53898311e-04, 1.48852810e-04,
       1.56199222e-04, 9.12329880e-04, 1.45773340e-04, 1.56493289e-04,
       1.51112908e-04, 1.54231079e-04, 1.55832183e-04, 1.53415074e-04,
       9.34099967e-04, 1.58394363e-04, 1.54424541e-04, 1.57802726e-04,
       1.57356297e-04, 1.52747546e-04, 1.57685205e-04, 9.14951141e-04,
       1.47642655e-04, 1.52476618e-04, 1.47042819e-04, 1.44949072e-04,
       1.42605857e-04, 1.45738323e-04, 8.13630762e-04, 1.22126082e-04,
       1.39740228e-04, 1.18471202e-04, 1.43155200e-04, 1.40259994e-04,
       1.45337832e-04, 8.50929582e-04, 1.43651236e-04, 1.38658355e-04,
       1.39866965e-04, 1.37464306e-04, 1.39082096e-04, 8.37054272e-04,
       1.38537467e-04, 1.40626164e-04, 1.40436666e-04, 1.40907574e-04,
       1.35005063e-04, 1.34465946e-04, 1.38537467e-04, 8.23587893e-04,
       1.37281953e-04, 1.39506931e-04, 1.38790533e-04, 1.30994178e-04,
       1.18003776e-04, 1.37281953e-04, 7.67680370e-04, 1.16308952e-04,
       1.36537055e-04, 1.28554456e-04, 1.09077884e-04, 9.90298580e-05,
       9.54742542e-05, 1.16308952e-04, 6.49769607e-04, 1.09545994e-04,
       1.20332666e-04, 1.45378365e-04, 1.60023619e-04, 8.44511581e-04,
       1.44792818e-04, 1.30545988e-04, 1.31686184e-04, 1.32084608e-04,
       1.55089868e-04, 1.59285489e-04, 1.00257131e-03, 1.53965216e-04,
       1.43883953e-04, 1.37872083e-04, 1.28142027e-04, 1.24332674e-04,
       1.41463010e-04, 1.43300000e-04, 1.40626164e-04, 8.56210863e-04,
       1.44328903e-04, 1.42313519e-04, 1.44179267e-04, 1.47785332e-04,
       1.36432023e-04, 1.53965216e-04, 8.65027109e-04, 1.37376670e-04,
       1.47351552e-04, 1.42116315e-04, 1.32553202e-04, 1.34435797e-04,
       1.37376670e-04, 8.32441350e-04, 1.43175823e-04, 1.40491650e-04,
       1.44408208e-04, 1.26941180e-04, 1.26732756e-04, 7.85052456e-04,
       1.30732189e-04, 1.31518493e-04, 1.35061283e-04, 1.34066554e-04,
       1.29679164e-04, 1.32104257e-04, 1.30732189e-04, 7.98240490e-04,
       1.36892105e-04, 1.32548975e-04, 1.36283799e-04, 1.39043172e-04,
       1.44171951e-04, 1.36892105e-04, 8.56682923e-04, 1.47786790e-04,
       1.41511499e-04, 1.47277405e-04, 1.27888308e-04, 1.32784965e-04,
       1.31518493e-04, 8.25311462e-04, 1.43116764e-04, 1.41650659e-04,
       1.48352273e-04, 1.45983204e-04, 1.50714815e-04, 1.47786790e-04,
       9.08196952e-04, 1.56150609e-04, 1.51964662e-04, 1.55596871e-04,
       1.53047498e-04, 1.61594976e-04, 1.56150609e-04, 9.72113587e-04,
       1.72028396e-04, 1.60980900e-04, 1.68311208e-04, 1.57610117e-04,
       1.59713224e-04, 9.87200994e-04, 1.65366626e-04, 1.66167739e-04,
       1.69822641e-04, 1.68520647e-04, 1.56737231e-04, 1.58014283e-04,
       1.65366626e-04, 9.65868879e-04, 1.56713914e-04, 1.69483295e-04,
       1.59553530e-04, 1.59932982e-04, 1.60075378e-04, 1.66167739e-04,
       9.91659412e-04, 1.63560348e-04, 1.72336376e-04, 1.69586589e-04,
       1.31713743e-04, 1.19016325e-04, 1.56713914e-04, 9.19482389e-04,
       1.36552402e-04, 1.44126818e-04, 1.16993228e-04, 1.14365959e-04,
       1.58151713e-04, 1.55376951e-04, 1.63560348e-04, 9.57360191e-04,
       1.55755232e-04, 1.64773150e-04, 1.59742796e-04, 1.53298702e-04,
       1.41860896e-04, 1.43883953e-04, 1.47351552e-04, 8.65797270e-04,
       1.47153786e-04, 1.32248381e-04, 1.52422138e-04, 1.51541390e-04,
       1.42116315e-04, 1.43175823e-04, 1.47153786e-04, 8.83741999e-04,
       1.47332547e-04, 1.37162837e-04, 1.34891390e-04, 8.09457526e-04,
       1.34801595e-04, 1.35184320e-04, 1.32381606e-04, 1.35035778e-04,
       1.49567213e-04, 1.49035520e-04, 1.47764350e-04, 8.83381276e-04,
       1.46972443e-04, 1.45682675e-04, 1.44359075e-04, 1.49220746e-04,
       1.47434046e-04, 1.46972443e-04, 8.76968287e-04, 1.44626093e-04,
       1.44637305e-04, 1.44077655e-04, 1.44755900e-04, 1.44696383e-04,
       1.44626093e-04, 9.07975996e-04, 1.53694975e-04, 1.55537123e-04,
       1.64665523e-04, 1.40777171e-04, 1.33529704e-04, 1.53694975e-04,
       8.68846230e-04, 1.37867001e-04, 1.55778944e-04, 1.47198436e-04,
       1.21431506e-04, 1.11733210e-04, 1.22669078e-04, 1.37867001e-04,
       7.69618654e-04, 1.35209116e-04, 1.40708743e-04, 1.34812254e-04,
       1.41821155e-04, 1.45773340e-04, 1.35209116e-04, 8.47352277e-04,
       1.46645867e-04, 1.43090545e-04, 1.49931427e-04, 1.47803476e-04,
       1.48525259e-04, 1.43651236e-04, 8.76529629e-04, 1.44682151e-04,
       1.41936079e-04, 1.88480298e-04, 1.76254499e-04, 1.68395757e-04,
       1.59505942e-04, 1.00050071e-03, 1.52264113e-04, 1.55600098e-04,
       1.30692455e-04, 1.57240201e-04, 8.45319800e-04, 1.03865280e-04,
       1.60417357e-04, 1.61554419e-04, 1.31550087e-04, 1.56466115e-04,
       1.56493289e-04, 1.58394363e-04, 9.47042762e-04, 1.56663613e-04,
       1.60391653e-04, 1.58633729e-04, 1.48640043e-04, 1.48920482e-04,
       9.09527458e-04, 1.51229258e-04, 1.51803148e-04, 1.54320538e-04,
       1.54613989e-04, 1.48766961e-04, 1.51046782e-04, 1.54424541e-04,
       1.51229258e-04, 9.21198015e-04, 1.58322967e-04, 1.57407506e-04,
       1.51380733e-04, 1.52737145e-04, 1.51803148e-04, 9.10895623e-04,
       1.51328180e-04, 1.52738899e-04, 1.50907519e-04, 1.53337552e-04,
       1.53493782e-04, 1.51328180e-04, 9.05334630e-04, 1.50116513e-04,
       1.49213251e-04, 1.47845354e-04, 1.52907865e-04, 1.52003536e-04,
       1.50116513e-04, 8.94670528e-04, 1.48065092e-04, 1.46362307e-04,
       1.45215216e-04, 1.50251957e-04, 1.47642655e-04, 1.48065092e-04,
       8.82390742e-04, 1.47299929e-04, 1.43643485e-04, 1.45487624e-04,
       1.25395355e-04, 1.22126082e-04, 1.03865280e-04, 5.51014696e-04,
       1.00594837e-04, 9.90331416e-05, 1.39586465e-04, 1.35982438e-04,
       1.39740228e-04, 8.12239050e-04, 1.35435110e-04, 1.31097950e-04,
       1.30396860e-04, 1.36787125e-04, 1.36014027e-04, 1.35435110e-04,
       8.03956024e-04, 1.34038219e-04, 1.31152667e-04, 1.30528875e-04,
       1.37407478e-04, 1.38658355e-04, 1.34038219e-04, 8.13905344e-04,
       1.37048238e-04, 1.32496957e-04, 1.34256098e-04, 1.16914130e-04,
       1.09545994e-04, 7.12750420e-04, 1.20432485e-04, 1.24273781e-04,
       1.26396731e-04, 1.15187300e-04, 1.46477519e-04, 1.44792818e-04,
       1.20432485e-04, 7.75720346e-04, 1.29532614e-04, 1.11850041e-04,
       1.22634870e-04, 1.39306290e-04, 1.30545988e-04, 1.37872083e-04,
       7.75504210e-04, 1.24953951e-04, 1.23491864e-04, 1.19334034e-04,
       1.50028788e-04, 1.57594810e-04, 1.44328903e-04, 9.00251715e-04,
       1.54762250e-04, 1.45334820e-04, 1.48202145e-04, 1.38317142e-04,
       1.40491650e-04, 1.43116764e-04, 8.70918505e-04, 1.47590277e-04,
       1.49551833e-04, 1.51850839e-04, 1.52860830e-04, 1.72028396e-04,
       1.36552402e-04, 8.99560148e-04, 1.72339590e-04, 1.24579653e-04,
       1.41199276e-04, 1.52706053e-04, 1.44408208e-04, 1.55755232e-04,
       1.47332547e-04, 1.47590277e-04, 9.01356111e-04, 1.53563794e-04,
       1.33224197e-04, 1.28142027e-04, 1.32248381e-04, 1.34801595e-04,
       7.80141142e-04, 1.23020595e-04, 1.28704347e-04, 1.35064217e-04,
       1.37270370e-04, 1.35184320e-04, 8.23975052e-04, 1.38602540e-04,
       1.38320769e-04, 1.39532836e-04, 1.38711533e-04, 1.41658615e-04,
       1.38602540e-04, 8.49990020e-04, 1.43953681e-04, 1.42329796e-04,
       1.44733855e-04, 1.44430186e-04, 1.46075491e-04, 1.43953681e-04,
       8.75554308e-04, 1.46912293e-04, 1.46434675e-04, 1.47747982e-04,
       1.46424864e-04, 1.46555107e-04, 1.46912293e-04, 8.81241187e-04,
       1.45871918e-04, 1.48323812e-04, 1.47153194e-04, 1.46471135e-04,
       1.45682675e-04, 1.45871918e-04, 8.67338522e-04, 1.43265404e-04,
       1.43932721e-04, 1.42114668e-04, 1.51757081e-04, 1.51773360e-04,
       1.52264113e-04, 9.02893231e-04, 1.49169408e-04, 1.50520127e-04,
       1.47409142e-04, 1.47745626e-04, 1.44682151e-04, 1.49169408e-04,
       8.68607044e-04, 1.40980092e-04, 1.45577803e-04, 1.40451963e-04,
       1.64909758e-04, 1.73069266e-04, 1.54762250e-04, 9.46080992e-04,
       1.57899535e-04, 1.50231203e-04, 1.45208980e-04, 1.22518130e-04,
       1.23127657e-04, 6.19946356e-04, 1.26386365e-04, 1.22335235e-04,
       1.25578969e-04, 1.29986652e-04, 1.35993330e-04, 1.26386365e-04,
       7.93500389e-04, 1.36461200e-04, 1.30302343e-04, 1.34370499e-04,
       1.40030501e-04, 1.44567538e-04, 1.36461200e-04, 8.45508058e-04,
       1.45161720e-04, 1.41725291e-04, 1.37561808e-04, 1.49823901e-04,
       1.55078029e-04, 1.55600098e-04, 1.50520127e-04, 1.45161720e-04,
       9.01519752e-04, 1.45335877e-04, 1.58919079e-04, 1.52476618e-04,
       1.60417357e-04, 9.15378776e-04, 1.45703849e-04, 1.53066460e-04,
       1.44795413e-04, 1.51112908e-04, 1.46645867e-04, 1.56663613e-04,
       9.13611989e-04, 1.48927828e-04, 1.56765258e-04, 1.53496515e-04,
       1.57802726e-04, 1.60391653e-04, 1.58322967e-04, 9.70148874e-04,
       1.64155645e-04, 1.63116066e-04, 1.66359818e-04, 1.47042819e-04,
       1.47299929e-04, 1.45703849e-04, 8.68641705e-04, 1.46841415e-04,
       1.40436619e-04, 1.41317074e-04, 1.18471202e-04, 1.00594837e-04,
       1.31097950e-04, 7.42715325e-04, 1.26767527e-04, 1.26118933e-04,
       1.39664876e-04, 1.39866965e-04, 1.41936079e-04, 1.37048238e-04,
       1.40980092e-04, 8.31865330e-04, 1.35085607e-04, 1.36948348e-04,
       1.40436666e-04, 1.39506931e-04, 8.42290135e-04, 1.41222961e-04,
       1.39729132e-04, 1.40979270e-04, 1.40415176e-04, 1.40907574e-04,
       1.42313519e-04, 1.41222961e-04, 8.52841404e-04, 1.43062778e-04,
       1.42450351e-04, 1.42884222e-04, 1.38790533e-04, 1.36537055e-04,
       1.39729132e-04, 8.22539039e-04, 1.36871360e-04, 1.35607625e-04,
       1.35003332e-04, 1.28554456e-04, 1.20332666e-04, 1.24273781e-04,
       1.36871360e-04, 9.11508226e-04, 1.31071857e-04, 1.37186423e-04,
       1.33217684e-04, 1.31686184e-04, 1.29532614e-04, 7.84568055e-04,
       1.33323121e-04, 1.25723207e-04, 1.29428293e-04, 1.34874637e-04,
       1.32084608e-04, 1.24953951e-04, 1.33323121e-04, 7.79742974e-04,
       1.25739904e-04, 1.32046976e-04, 1.31594415e-04, 1.24332674e-04,
       1.23491864e-04, 1.23020595e-04, 7.45334284e-04, 1.26873321e-04,
       1.21461147e-04, 1.26154683e-04, 1.44179267e-04, 1.45334820e-04,
       1.43062778e-04, 8.70779520e-04, 1.46549001e-04, 1.44661492e-04,
       1.46992163e-04, 1.35061283e-04, 1.41650659e-04, 8.69623847e-04,
       1.37875258e-04, 1.52100397e-04, 1.47650243e-04, 1.55286006e-04,
       1.34066554e-04, 1.32548975e-04, 1.37875258e-04, 8.17104974e-04,
       1.34664802e-04, 1.39821131e-04, 1.38128254e-04, 1.36283799e-04,
       1.41511499e-04, 1.34664802e-04, 8.38021265e-04, 1.45644965e-04,
       1.37081869e-04, 1.42834332e-04, 1.47277405e-04, 1.51964662e-04,
       1.45644965e-04, 9.00100658e-04, 1.54411336e-04, 1.48361589e-04,
       1.52440702e-04, 1.48352273e-04, 1.49551833e-04, 1.52100397e-04,
       9.25661436e-04, 1.56460678e-04, 1.58017767e-04, 1.61178488e-04,
       1.55596871e-04, 1.60980900e-04, 1.54411336e-04, 9.51667921e-04,
       1.63674478e-04, 1.56562611e-04, 1.60441725e-04, 1.68311208e-04,
       1.72339590e-04, 1.63674478e-04, 1.00297281e-03, 1.69317790e-04,
       1.63857406e-04, 1.65472336e-04, 1.69822641e-04, 1.69483295e-04,
       1.02450515e-03, 1.66680318e-04, 1.77055742e-04, 1.76088960e-04,
       1.65374191e-04, 1.68520647e-04, 1.72336376e-04, 1.66680318e-04,
       1.03592434e-03, 1.75791619e-04, 1.73649906e-04, 1.78945478e-04,
       1.59553530e-04, 1.44126818e-04, 1.77055742e-04, 1.03290976e-03,
       1.30436647e-04, 1.83929811e-04, 1.05953811e-04, 1.31853400e-04,
       1.69586589e-04, 1.64773150e-04, 1.75791619e-04, 1.02192253e-03,
       1.65715765e-04, 1.70396498e-04, 1.75658913e-04, 1.16993228e-04,
       1.24579653e-04, 6.80088295e-04, 1.14868873e-04, 9.43878901e-05,
       1.21076604e-04, 1.08182046e-04, 1.59742796e-04, 1.51850839e-04,
       1.53563794e-04, 1.56460678e-04, 1.65715765e-04, 9.51480475e-04,
       1.64146603e-04, 1.32381606e-04, 1.28704347e-04, 1.26873321e-04,
       8.08455500e-04, 1.40283440e-04, 1.35239963e-04, 1.44972823e-04,
       1.35035778e-04, 1.38320769e-04, 1.40283440e-04, 8.38334375e-04,
       1.38622664e-04, 1.45508745e-04, 1.40562979e-04, 1.44359075e-04,
       1.44637305e-04, 1.43265404e-04, 8.42653455e-04, 1.39016388e-04,
       1.37951264e-04, 1.33424018e-04, 1.44077655e-04, 1.55537123e-04,
       1.39016388e-04, 8.02110152e-04, 1.37662468e-04, 1.19314953e-04,
       1.06501564e-04, 1.64665523e-04, 1.55778944e-04, 1.37662468e-04,
       1.03110337e-03, 1.56175785e-04, 1.63009737e-04, 1.12282719e-04,
       1.41528198e-04, 1.47198436e-04, 1.40708743e-04, 1.43090545e-04,
       1.48927828e-04, 1.56175785e-04, 8.91624714e-04, 1.55523376e-04,
       1.48894045e-04, 1.57899535e-04, 1.22335235e-04, 8.39120522e-04,
       1.42804854e-04, 1.27861362e-04, 1.39325490e-04, 1.61554419e-04,
       1.53066460e-04, 9.22036790e-04, 1.61394080e-04, 1.46243396e-04,
       1.46557855e-04, 1.53220580e-04, 1.31550087e-04, 9.90331416e-05,
       1.26767527e-04, 1.61394080e-04, 9.86644987e-04, 1.55638747e-04,
       1.56617673e-04, 1.55643731e-04, 1.58633729e-04, 1.56765258e-04,
       1.64155645e-04, 9.54623704e-04, 1.53428662e-04, 1.65249461e-04,
       1.56390949e-04, 1.54320538e-04, 1.57407506e-04, 1.63116066e-04,
       9.66592847e-04, 1.55022746e-04, 1.70866172e-04, 1.65859819e-04,
       1.54613989e-04, 1.52738899e-04, 1.55022746e-04, 9.10281960e-04,
       1.48144087e-04, 1.52382903e-04, 1.47379336e-04, 1.50907519e-04,
       1.49213251e-04, 1.48144087e-04, 8.90562678e-04, 1.47767737e-04,
       1.46606083e-04, 1.47924001e-04, 1.47845354e-04, 1.46362307e-04,
       1.47767737e-04, 8.64195023e-04, 1.39711273e-04, 1.44540885e-04,
       1.37967467e-04, 1.45215216e-04, 1.43643485e-04, 1.39711273e-04,
       7.75404295e-04, 1.10120196e-04, 1.32366736e-04, 1.04347390e-04,
       1.45487624e-04, 1.46841415e-04, 1.10120196e-04, 7.70280962e-04,
       1.59914485e-04, 7.97383721e-05, 1.28178870e-04, 1.30396860e-04,
       1.31152667e-04, 1.26118933e-04, 7.63056423e-04, 1.27447924e-04,
       1.23683427e-04, 1.24256611e-04, 1.30528875e-04, 1.32496957e-04,
       1.27447924e-04, 7.57592895e-04, 1.31709577e-04, 1.16557995e-04,
       1.18851566e-04, 1.34256098e-04, 1.35085607e-04, 1.31709577e-04,
       7.91940536e-04, 1.33474753e-04, 1.28239417e-04, 1.29175084e-04,
       1.26396731e-04, 1.31071857e-04, 8.07192219e-04, 1.29757901e-04,
       1.38060546e-04, 1.42766352e-04, 1.39138833e-04, 1.15187300e-04,
       1.11850041e-04, 1.29757901e-04, 7.46720717e-04, 1.19107471e-04,
       1.39065588e-04, 1.31752418e-04, 1.22634870e-04, 1.25723207e-04,
       1.19107471e-04, 7.37173193e-04, 1.23686884e-04, 1.22264843e-04,
       1.23755919e-04, 1.19334034e-04, 1.25739904e-04, 1.21461147e-04,
       7.50123242e-04, 1.28944673e-04, 1.26782913e-04, 1.27860570e-04,
       1.48202145e-04, 1.50231203e-04, 1.46549001e-04, 8.93144341e-04,
       1.50251026e-04, 1.48118999e-04, 1.49791967e-04, 1.41199276e-04,
       1.69317790e-04, 1.14868873e-04, 8.49741719e-04, 1.57928823e-04,
       1.25742761e-04, 1.40684196e-04, 1.14365959e-04, 1.30436647e-04,
       9.43878901e-05, 5.77538250e-04, 1.27363508e-04, 1.10984246e-04,
       1.39532836e-04, 1.42329796e-04, 1.38622664e-04, 8.47683127e-04,
       1.44334741e-04, 1.39973986e-04, 1.42889104e-04, 1.44733855e-04,
       1.46434675e-04, 1.44334741e-04, 8.81205780e-04, 1.49647418e-04,
       1.46221617e-04, 1.49833473e-04, 1.47747982e-04, 1.48323812e-04,
       1.49647418e-04, 9.03756397e-04, 1.50270584e-04, 1.53859632e-04,
       1.53906969e-04, 1.47153194e-04, 1.43932721e-04, 1.50270584e-04,
       8.84661454e-04, 1.42985629e-04, 1.52191904e-04, 1.48127421e-04,
       1.42114668e-04, 1.37951264e-04, 1.42985629e-04, 8.39975795e-04,
       1.36173983e-04, 1.41699246e-04, 1.39051005e-04, 1.47409142e-04,
       1.45577803e-04, 1.41725291e-04, 1.45335877e-04, 8.59541400e-04,
       1.40636267e-04, 1.38857020e-04, 1.40451963e-04, 1.36948348e-04,
       1.33474753e-04, 1.40636267e-04, 8.23124981e-04, 1.37635775e-04,
       1.33977875e-04, 1.45208980e-04, 1.42804854e-04, 1.50251026e-04,
       8.96114295e-04, 1.51960543e-04, 1.50887625e-04, 1.55001267e-04,
       1.25578969e-04, 1.30302343e-04, 1.27861362e-04, 7.88120773e-04,
       1.32257273e-04, 1.36224144e-04, 1.35896682e-04, 1.34370499e-04,
       1.37561808e-04, 1.38857020e-04, 1.37635775e-04, 1.32257273e-04,
       8.14909711e-04, 1.34227337e-04, 1.44795413e-04, 1.40436619e-04,
       1.46243396e-04, 8.14421245e-04, 1.31493829e-04, 1.30921381e-04,
       1.20530608e-04, 1.53496515e-04, 1.63009737e-04, 1.55523376e-04,
       1.53428662e-04, 9.25156631e-04, 1.46072065e-04, 1.53626277e-04,
       1.66359818e-04, 1.65249461e-04, 1.70866172e-04, 1.08941440e-03,
       1.41518022e-04, 1.81596883e-04, 1.15232997e-04, 1.48591045e-04,
       1.41317074e-04, 1.59914485e-04, 1.31493829e-04, 8.64035735e-04,
       1.67521850e-04, 1.23178944e-04, 1.40609553e-04, 1.39664876e-04,
       1.55638747e-04, 1.23683427e-04, 8.53362894e-04, 1.62956629e-04,
       1.24059942e-04, 1.47359272e-04, 1.40979270e-04, 1.42450351e-04,
       8.52557225e-04, 1.41084737e-04, 1.42657733e-04, 1.42641609e-04,
       1.42743525e-04, 1.40415176e-04, 1.35607625e-04, 1.41084737e-04,
       8.31080474e-04, 1.34900750e-04, 1.40224312e-04, 1.38847875e-04,
       1.42884222e-04, 1.44661492e-04, 1.42657733e-04, 8.64261824e-04,
       1.45808276e-04, 1.43438414e-04, 1.44811687e-04, 1.35003332e-04,
       1.37186423e-04, 1.34900750e-04, 8.22607186e-04, 1.37423892e-04,
       1.38876369e-04, 1.39216421e-04, 1.33217684e-04, 1.38060546e-04,
       1.37423892e-04, 8.55789794e-04, 1.43805975e-04, 1.50871237e-04,
       1.52410459e-04, 1.29428293e-04, 1.23686884e-04, 7.87426440e-04,
       1.36616111e-04, 1.26237644e-04, 1.38047284e-04, 1.33410225e-04,
       1.34874637e-04, 1.32046976e-04, 1.36616111e-04, 8.07188396e-04,
       1.32509629e-04, 1.36209931e-04, 1.34931111e-04, 1.31594415e-04,
       1.28944673e-04, 1.32509629e-04, 7.79720176e-04, 1.27421769e-04,
       1.30843904e-04, 1.28405785e-04, 1.26154683e-04, 1.35239963e-04,
       1.26782913e-04, 7.85061533e-04, 1.41142640e-04, 1.25233444e-04,
       1.30507890e-04, 1.46992163e-04, 1.48118999e-04, 1.45808276e-04,
       8.79673883e-04, 1.47119530e-04, 1.45725249e-04, 1.45909666e-04,
       1.47650243e-04, 1.39821131e-04, 8.91704408e-04, 1.56569330e-04,
       1.40740569e-04, 1.57461392e-04, 1.49461743e-04, 1.55286006e-04,
       1.58017767e-04, 1.56569330e-04, 9.58642597e-04, 1.63246459e-04,
       1.60788833e-04, 1.64734202e-04, 1.38128254e-04, 1.37081869e-04,
       1.40740569e-04, 8.38513711e-04, 1.39559420e-04, 1.41569183e-04,
       1.41434418e-04, 1.42834332e-04, 1.48361589e-04, 1.39559420e-04,
       8.69546875e-04, 1.49989257e-04, 1.41949849e-04, 1.46852429e-04,
       1.52440702e-04, 1.56562611e-04, 1.49989257e-04, 9.21876299e-04,
       1.57196817e-04, 1.51300630e-04, 1.54386281e-04, 1.61178488e-04,
       1.70396498e-04, 1.64146603e-04, 1.63246459e-04, 1.00188102e-03,
       1.73547448e-04, 1.69365525e-04, 1.60441725e-04, 1.63857406e-04,
       1.57196817e-04, 9.64919551e-04, 1.64373535e-04, 1.57644516e-04,
       1.61405552e-04, 1.65472336e-04, 1.57928823e-04, 1.64373535e-04,
       1.00562777e-03, 1.48717224e-04, 1.88624325e-04, 1.80511527e-04,
       1.76088960e-04, 1.83929811e-04, 9.59681250e-04, 1.58915339e-04,
       1.54278653e-04, 1.36995475e-04, 1.49473011e-04, 1.65374191e-04,
       1.73649906e-04, 1.58915339e-04, 1.03714068e-03, 1.80224318e-04,
       1.72971828e-04, 1.86005092e-04, 1.78945478e-04, 1.75658913e-04,
       1.73547448e-04, 1.80224318e-04, 1.06263216e-03, 1.75432848e-04,
       1.78823153e-04, 1.05953811e-04, 1.27363508e-04, 5.51205433e-04,
       1.25513063e-04, 8.66707337e-05, 1.05704317e-04, 1.21076604e-04,
       1.25742761e-04, 7.33407854e-04, 1.18849013e-04, 1.22272559e-04,
       1.22112154e-04, 1.23354764e-04, 1.08182046e-04, 1.10984246e-04,
       1.25513063e-04, 1.18849013e-04, 7.15265139e-04, 1.27799311e-04,
       1.23937461e-04, 1.44972823e-04, 1.45508745e-04, 1.41142640e-04,
       1.05489378e-03, 1.50004554e-04, 1.55729660e-04, 1.55065377e-04,
       1.62469981e-04, 1.40562979e-04, 1.39973986e-04, 1.50004554e-04,
       8.32645327e-04, 1.36398140e-04, 1.39361544e-04, 1.26344123e-04,
       1.33424018e-04, 1.19314953e-04, 1.36173983e-04, 7.68713317e-04,
       1.17373235e-04, 1.33559406e-04, 1.28867723e-04, 1.06501564e-04,
       1.12282719e-04, 1.17373235e-04, 5.76933129e-04, 1.17911149e-04,
       1.22864462e-04, 1.39325490e-04, 1.51960543e-04, 1.36224144e-04,
       8.65691903e-04, 1.44661264e-04, 1.31259928e-04, 1.62260533e-04,
       1.46557855e-04, 1.30921381e-04, 7.80750685e-04, 1.43258662e-04,
       1.05632471e-04, 1.38185017e-04, 1.16195300e-04, 1.53220580e-04,
       1.56617673e-04, 1.43258662e-04, 9.01526060e-04, 1.52677840e-04,
       1.50886695e-04, 1.44864610e-04, 1.56390949e-04, 1.46072065e-04,
       1.41518022e-04, 8.26521909e-04, 1.38851568e-04, 1.18018438e-04,
       1.25670868e-04, 1.65859819e-04, 1.52382903e-04, 1.81596883e-04,
       9.47987613e-04, 1.46764429e-04, 1.61369558e-04, 1.40014020e-04,
       1.47379336e-04, 1.46606083e-04, 1.46764429e-04, 7.92119858e-04,
       1.23738057e-04, 1.24942537e-04, 1.02689417e-04, 1.47924001e-04,
       1.44540885e-04, 1.23738057e-04, 9.46175723e-04, 1.46532198e-04,
       9.94992310e-05, 1.55059600e-04, 1.28881752e-04, 1.37967467e-04,
       1.32366736e-04, 1.46532198e-04, 8.62057137e-04, 1.45017579e-04,
       1.48043522e-04, 1.52129635e-04, 1.04347390e-04, 7.97383721e-05,
       1.45017579e-04, 6.02060270e-04, 1.16669464e-04, 1.56287465e-04,
       1.28178870e-04, 1.67521850e-04, 1.16669464e-04, 1.05916092e-03,
       1.57314295e-04, 1.60797948e-04, 1.60865384e-04, 1.67813113e-04,
       1.55643731e-04, 1.62956629e-04, 1.52677840e-04, 9.46473622e-04,
       1.57602341e-04, 1.66393493e-04, 1.51199587e-04, 1.24256611e-04,
       1.16557995e-04, 1.24059942e-04, 5.88192988e-04, 1.07050034e-04,
       1.16268406e-04, 1.18851566e-04, 1.28239417e-04, 1.44661264e-04,
       1.07050034e-04, 7.59901431e-04, 1.24787791e-04, 1.36311359e-04,
       1.29175084e-04, 1.33977875e-04, 1.35896682e-04, 1.34227337e-04,
       1.31259928e-04, 1.24787791e-04, 7.89324696e-04, 1.42766352e-04,
       1.43805975e-04, 8.81156422e-04, 1.46289260e-04, 1.47010194e-04,
       1.50482976e-04, 1.50801665e-04, 1.39138833e-04, 1.39065588e-04,
       1.46289260e-04, 8.72814453e-04, 1.43396914e-04, 1.53871480e-04,
       1.51052379e-04, 1.31752418e-04, 1.22264843e-04, 1.43396914e-04,
       8.08320211e-04, 1.24129941e-04, 1.48277746e-04, 1.38498348e-04,
       1.23755919e-04, 1.26237644e-04, 1.24129941e-04, 7.61505451e-04,
       1.28842613e-04, 1.28154194e-04, 1.30385140e-04, 1.27860570e-04,
       1.27421769e-04, 1.25233444e-04, 7.58962914e-04, 1.27753261e-04,
       1.23867626e-04, 1.26826245e-04, 1.49791967e-04, 1.50887625e-04,
       1.47119530e-04, 1.57602341e-04, 9.11218991e-04, 1.57397605e-04,
       1.48419924e-04, 1.40684196e-04, 1.48717224e-04, 1.22272559e-04,
       6.65280729e-04, 1.35492154e-04, 1.18114596e-04, 1.31853400e-04,
       1.54278653e-04, 8.66707337e-05, 7.25408731e-04, 1.28942476e-04,
       1.01908489e-04, 1.21754978e-04, 1.42889104e-04, 1.46221617e-04,
       1.36398140e-04, 8.48806722e-04, 1.47574606e-04, 1.33939651e-04,
       1.41783603e-04, 1.49833473e-04, 1.53859632e-04, 1.47574606e-04,
       9.42969557e-04, 1.68710102e-04, 1.52941203e-04, 1.70050540e-04,
       1.53906969e-04, 1.52191904e-04, 1.68710102e-04, 9.77035128e-04,
       1.48848710e-04, 1.85761885e-04, 1.67615558e-04, 1.48127421e-04,
       1.41699246e-04, 1.48848710e-04, 8.67279729e-04, 1.40648304e-04,
       1.44864139e-04, 1.43091908e-04, 1.41528198e-04, 1.53626277e-04,
       1.17911149e-04, 1.38851568e-04, 8.06550278e-04, 1.22895373e-04,
       1.31737713e-04, 1.39051005e-04, 1.33559406e-04, 1.40648304e-04,
       8.15457656e-04, 1.31099379e-04, 1.37578444e-04, 1.33521118e-04,
       1.55001267e-04, 1.47359272e-04, 1.62260533e-04, 1.66393493e-04,
       1.16268406e-04, 1.36311359e-04, 1.57397605e-04, 1.04099193e-03,
       1.20530608e-04, 1.23178944e-04, 1.05632471e-04, 5.59723662e-04,
       1.11921709e-04, 9.84599297e-05, 1.15232997e-04, 1.18018438e-04,
       5.76336681e-04, 1.18495403e-04, 1.09672416e-04, 1.14917426e-04,
       1.40609553e-04, 1.57314295e-04, 1.11921709e-04, 8.25966805e-04,
       1.21481932e-04, 1.50891917e-04, 1.43747398e-04, 1.42641609e-04,
       1.40224312e-04, 8.51479521e-04, 1.43245803e-04, 1.41365072e-04,
       1.41733079e-04, 1.42269646e-04, 1.42743525e-04, 1.43438414e-04,
       1.43245803e-04, 8.61159094e-04, 1.43676434e-04, 1.44159588e-04,
       1.43895331e-04, 1.38847875e-04, 1.38876369e-04, 1.41365072e-04,
       8.46497432e-04, 1.42490561e-04, 1.41910280e-04, 1.43007276e-04,
       1.44811687e-04, 1.45725249e-04, 1.43676434e-04, 8.69152098e-04,
       1.44198166e-04, 1.45174366e-04, 1.45566197e-04, 1.39216421e-04,
       1.50871237e-04, 1.42490561e-04, 1.05084188e-03, 1.63410145e-04,
       1.45267365e-04, 1.60011579e-04, 1.49574572e-04, 1.52410459e-04,
       1.47010194e-04, 1.63410145e-04, 8.93523830e-04, 1.44511579e-04,
       1.49839877e-04, 1.36341576e-04, 1.38047284e-04, 1.36209931e-04,
       8.45447147e-04, 1.42150430e-04, 1.39306418e-04, 1.45369872e-04,
       1.44363212e-04, 1.33410225e-04, 1.28842613e-04, 1.42150430e-04,
       8.21851720e-04, 1.32318338e-04, 1.45197297e-04, 1.39932818e-04,
       1.34931111e-04, 1.30843904e-04, 1.39306418e-04, 8.09167456e-04,
       1.30798350e-04, 1.38854266e-04, 1.34433405e-04, 1.28405785e-04,
       1.27753261e-04, 1.30798350e-04, 7.83275859e-04, 1.29409476e-04,
       1.33822668e-04, 1.33086318e-04, 1.30507890e-04, 1.55729660e-04,
       1.23867626e-04, 8.66023463e-04, 1.42393690e-04, 1.60110152e-04,
       1.53414445e-04, 1.45909666e-04, 1.50886695e-04, 1.51199587e-04,
       1.48419924e-04, 1.44198166e-04, 8.87009570e-04, 1.46395532e-04,
       1.57461392e-04, 1.60788833e-04, 9.61009115e-04, 1.56886584e-04,
       1.64312957e-04, 1.58730209e-04, 1.62829140e-04, 1.49461743e-04,
       1.41569183e-04, 1.56886584e-04, 8.91270118e-04, 1.42486499e-04,
       1.53686817e-04, 1.47179292e-04, 1.64734202e-04, 1.69365525e-04,
       1.75432848e-04, 1.64312957e-04, 1.01913058e-03, 1.75886810e-04,
       1.69398241e-04, 1.41434418e-04, 1.41949849e-04, 1.42486499e-04,
       8.56292016e-04, 1.44927845e-04, 1.41783745e-04, 1.43709660e-04,
       1.46852429e-04, 1.51300630e-04, 1.44927845e-04, 8.91002393e-04,
       1.51133287e-04, 1.47515480e-04, 1.49272721e-04, 1.54386281e-04,
       1.57644516e-04, 1.51133287e-04, 9.24606482e-04, 1.56722253e-04,
       1.51194707e-04, 1.53525438e-04, 1.61405552e-04, 1.88624325e-04,
       1.56722253e-04, 1.04753926e-03, 1.99506919e-04, 1.62857161e-04,
       1.78423053e-04, 1.80511527e-04, 1.35492154e-04, 1.99506919e-04,
       1.20548165e-03, 1.44280153e-04, 1.84439860e-04, 1.76138855e-04,
       1.85112179e-04, 1.36995475e-04, 1.28942476e-04, 7.94556640e-04,
       1.38573877e-04, 1.27672461e-04, 1.29761967e-04, 1.32610384e-04,
       1.49473011e-04, 1.72971828e-04, 1.38573877e-04, 1.00251693e-03,
       1.91068558e-04, 1.60716898e-04, 1.89712761e-04, 1.86005092e-04,
       1.78823153e-04, 1.75886810e-04, 1.91068558e-04, 1.09261188e-03,
       1.76414227e-04, 1.84414036e-04, 1.05704317e-04, 1.27799311e-04,
       1.01908489e-04, 7.17679586e-04, 1.36150942e-04, 1.16984534e-04,
       1.29131993e-04, 1.22112154e-04, 1.18114596e-04, 1.44280153e-04,
       8.19373149e-04, 1.35433778e-04, 1.56138200e-04, 1.43294268e-04,
       1.23354764e-04, 1.23937461e-04, 1.36150942e-04, 1.35433778e-04,
       8.02478806e-04, 1.41868164e-04, 1.41733698e-04, 1.55065377e-04,
       1.39361544e-04, 5.63511069e-04, 9.35227187e-05, 1.09907292e-04,
       6.56541377e-05, 1.26344123e-04, 1.33939651e-04, 9.35227187e-05,
       7.08241102e-04, 1.33045384e-04, 9.47310440e-05, 1.26658181e-04,
       1.28867723e-04, 1.22864462e-04, 1.22895373e-04, 1.31099379e-04,
       7.64359383e-04, 1.28215850e-04, 1.30416596e-04, 1.38185017e-04,
       1.44864610e-04, 1.45174366e-04, 1.46395532e-04, 8.55385681e-04,
       1.37653865e-04, 1.43112292e-04, 1.16195300e-04, 9.84599297e-05,
       1.21481932e-04, 1.37653865e-04, 7.53736805e-04, 1.40754608e-04,
       1.39191171e-04, 1.25670868e-04, 1.31737713e-04, 1.18495403e-04,
       1.28215850e-04, 7.49140279e-04, 1.24337023e-04, 1.20683422e-04,
       1.48591045e-04, 1.61369558e-04, 1.09672416e-04, 8.02305332e-04,
       1.35117554e-04, 1.17444067e-04, 1.30110692e-04, 1.40014020e-04,
       1.24942537e-04, 1.35117554e-04, 7.80262824e-04, 1.19483359e-04,
       1.30634033e-04, 1.30071321e-04, 1.02689417e-04, 9.94992310e-05,
       1.19483359e-04, 5.44732550e-04, 1.03133005e-04, 1.19927538e-04,
       1.55059600e-04, 1.48043522e-04, 9.04115589e-04, 1.47475204e-04,
       1.51696950e-04, 1.49522990e-04, 1.52317322e-04, 1.52129635e-04,
       1.56287465e-04, 1.60797948e-04, 1.47475204e-04, 9.28579873e-04,
       1.60527268e-04, 1.51362354e-04, 1.60865384e-04, 1.50891917e-04,
       9.18788450e-04, 1.59957835e-04, 1.47181809e-04, 1.51347032e-04,
       1.48544474e-04, 1.50482976e-04, 1.44511579e-04, 8.73244340e-04,
       1.53089677e-04, 1.37070368e-04, 1.49068826e-04, 1.39020914e-04,
       1.50801665e-04, 1.53871480e-04, 1.53089677e-04, 9.35712730e-04,
       1.57314047e-04, 1.59740653e-04, 1.60895208e-04, 1.51052379e-04,
       1.48277746e-04, 1.57314047e-04, 9.24632198e-04, 1.51194327e-04,
       1.60069393e-04, 1.56724306e-04, 1.38498348e-04, 1.28154194e-04,
       1.51194327e-04, 8.44803092e-04, 1.31641984e-04, 1.52261214e-04,
       1.43053026e-04, 1.30385140e-04, 1.32318338e-04, 1.31641984e-04,
       8.01503709e-04, 1.35884649e-04, 1.34482066e-04, 1.36791533e-04,
       1.26826245e-04, 1.29409476e-04, 1.42393690e-04, 8.57401694e-04,
       1.46194284e-04, 1.55680319e-04, 1.56897681e-04, 1.21754978e-04,
       1.27672461e-04, 1.16984534e-04, 7.45662276e-04, 1.27575398e-04,
       1.23500762e-04, 1.28174143e-04, 1.62469981e-04, 1.60110152e-04,
       1.09907292e-04, 9.56432823e-04, 1.40113816e-04, 1.05971612e-04,
       1.35394795e-04, 1.42465175e-04, 1.41783603e-04, 1.52941203e-04,
       1.33045384e-04, 8.66553535e-04, 1.56500960e-04, 1.34992204e-04,
       1.47290181e-04, 1.70050540e-04, 1.85761885e-04, 1.56500960e-04,
       1.10179789e-03, 1.55199297e-04, 1.57227912e-04, 1.28120000e-04,
       1.48937296e-04, 1.67615558e-04, 1.44864139e-04, 1.55199297e-04,
       8.61871084e-04, 1.44896345e-04, 1.18423640e-04, 1.30872106e-04,
       1.43091908e-04, 1.37578444e-04, 1.44896345e-04, 8.31174643e-04,
       1.32402045e-04, 1.40093368e-04, 1.33112533e-04, 1.33521118e-04,
       1.30416596e-04, 1.24337023e-04, 1.32402045e-04, 7.73381234e-04,
       1.23871976e-04, 1.28832476e-04, 1.14917426e-04, 1.20683422e-04,
       1.17444067e-04, 1.23871976e-04, 7.29792458e-04, 1.26106827e-04,
       1.26768741e-04, 1.28881752e-04, 1.03133005e-04, 1.51696950e-04,
       8.03678249e-04, 1.23782993e-04, 1.49174380e-04, 1.47009170e-04,
       1.67813113e-04, 1.60527268e-04, 1.59957835e-04, 9.34494958e-04,
       1.48438970e-04, 1.50995396e-04, 1.46762376e-04, 1.43747398e-04,
       1.40754608e-04, 1.47181809e-04, 8.65785094e-04, 1.44956129e-04,
       1.43088180e-04, 1.46056968e-04, 1.41733079e-04, 1.44159588e-04,
       1.44956129e-04, 8.63720656e-04, 1.42744610e-04, 1.45338436e-04,
       1.44788813e-04, 1.42269646e-04, 1.41910280e-04, 1.42744610e-04,
       8.56726223e-04, 1.41999169e-04, 1.44094297e-04, 1.43708221e-04,
       1.43895331e-04, 1.45566197e-04, 1.43112292e-04, 1.39191171e-04,
       1.43088180e-04, 1.45338436e-04, 8.60191607e-04, 1.43007276e-04,
       1.45267365e-04, 1.41999169e-04, 8.47702769e-04, 1.39904601e-04,
       1.40939508e-04, 1.36584851e-04, 1.60011579e-04, 1.49839877e-04,
       8.43836119e-04, 1.33516741e-04, 1.40812630e-04, 1.31265552e-04,
       1.28389739e-04, 1.36341576e-04, 1.37070368e-04, 1.33516741e-04,
       7.90877108e-04, 1.31887638e-04, 1.27209340e-04, 1.24851445e-04,
       1.45369872e-04, 1.45197297e-04, 8.88000015e-04, 1.48598762e-04,
       1.47869481e-04, 1.50203949e-04, 1.50760655e-04, 1.44363212e-04,
       1.38854266e-04, 1.48598762e-04, 8.57189387e-04, 1.37643093e-04,
       1.46588388e-04, 1.41141665e-04, 1.39932818e-04, 1.35884649e-04,
       1.47869481e-04, 8.59529923e-04, 1.39115664e-04, 1.50716480e-04,
       1.46010832e-04, 1.34433405e-04, 1.33822668e-04, 1.37643093e-04,
       8.23371843e-04, 1.36413278e-04, 1.40839239e-04, 1.40220160e-04,
       1.33086318e-04, 1.46194284e-04, 1.36413278e-04, 8.97814804e-04,
       1.61874794e-04, 1.53946289e-04, 1.66299841e-04, 1.53414445e-04,
       1.55680319e-04, 1.40113816e-04, 8.62282812e-04, 1.46454488e-04,
       1.31272473e-04, 1.35347271e-04, 1.58730209e-04, 1.53686817e-04,
       9.36248165e-04, 1.61824692e-04, 1.49705461e-04, 1.52612574e-04,
       1.59688413e-04, 1.62829140e-04, 1.69398241e-04, 1.76414227e-04,
       1.61824692e-04, 1.01443239e-03, 1.75993263e-04, 1.67972829e-04,
       1.47179292e-04, 1.41783745e-04, 1.49705461e-04, 8.66560399e-04,
       1.40315105e-04, 1.45785803e-04, 1.41790994e-04, 1.43709660e-04,
       1.47515480e-04, 1.40315105e-04, 8.62621094e-04, 1.48485791e-04,
       1.39115097e-04, 1.43479962e-04, 1.49272721e-04, 1.51194707e-04,
       1.48485791e-04, 8.97407573e-04, 1.50449316e-04, 1.48981749e-04,
       1.49023289e-04, 1.53525438e-04, 1.62857161e-04, 1.50449316e-04,
       9.39076080e-04, 1.66262453e-04, 1.49750149e-04, 1.56231563e-04,
       1.78423053e-04, 1.84439860e-04, 1.66262453e-04, 1.02437253e-03,
       1.78147441e-04, 1.55615772e-04, 1.61483953e-04, 1.76138855e-04,
       1.56138200e-04, 9.78588969e-04, 1.51322846e-04, 1.74933462e-04,
       1.58222822e-04, 1.61832784e-04, 1.29761967e-04, 1.27575398e-04,
       7.85105005e-04, 1.32796462e-04, 1.29885512e-04, 1.32180642e-04,
       1.32905023e-04, 1.32610384e-04, 1.60716898e-04, 1.32796462e-04,
       9.52787189e-04, 1.86550584e-04, 1.57232626e-04, 1.82880235e-04,
       1.89712761e-04, 1.84414036e-04, 1.75993263e-04, 1.86550584e-04,
       1.08689116e-03, 1.72480962e-04, 1.77739558e-04, 1.29131993e-04,
       1.41868164e-04, 1.23500762e-04, 8.08156105e-04, 1.46032628e-04,
       1.28913432e-04, 1.38709126e-04, 1.43294268e-04, 1.41733698e-04,
       1.51322846e-04, 1.46032628e-04, 8.83562797e-04, 1.52454503e-04,
       1.48724855e-04, 6.56541377e-05, 9.47310440e-05, 1.05971612e-04,
       6.75753226e-04, 1.33503960e-04, 1.38718515e-04, 1.37173957e-04,
       1.26658181e-04, 1.34992204e-04, 1.33503960e-04, 8.12688455e-04,
       1.39774160e-04, 1.39911887e-04, 1.37848064e-04, 1.30110692e-04,
       1.30634033e-04, 1.26106827e-04, 7.84032562e-04, 1.33035459e-04,
       1.28870130e-04, 1.35275422e-04, 1.30071321e-04, 1.19927538e-04,
       1.23782993e-04, 1.33035459e-04, 7.89248483e-04, 1.41661244e-04,
       1.40769928e-04, 1.49522990e-04, 1.49174380e-04, 9.08759673e-04,
       1.49108794e-04, 1.53925938e-04, 1.50930908e-04, 1.56096663e-04,
       1.52317322e-04, 1.51362354e-04, 1.48438970e-04, 1.49108794e-04,
       8.92785615e-04, 1.45921659e-04, 1.45636515e-04, 1.51347032e-04,
       1.50995396e-04, 8.78367565e-04, 1.47345736e-04, 1.44772106e-04,
       1.43064645e-04, 1.40842651e-04, 1.48544474e-04, 1.46056968e-04,
       1.44788813e-04, 1.44094297e-04, 1.47345736e-04, 8.74994002e-04,
       1.44163715e-04, 1.49574572e-04, 1.39904601e-04, 1.40812630e-04,
       8.34743592e-04, 1.35825746e-04, 1.31971478e-04, 1.36654564e-04,
       1.49068826e-04, 1.59740653e-04, 9.43682600e-04, 1.44592613e-04,
       1.66068830e-04, 1.56703644e-04, 1.67508034e-04, 1.39020914e-04,
       1.31887638e-04, 1.44592613e-04, 7.97943948e-04, 1.24813832e-04,
       1.35137227e-04, 1.22491724e-04, 1.60895208e-04, 1.60069393e-04,
       1.66068830e-04, 9.82555122e-04, 1.62202119e-04, 1.68180234e-04,
       1.65139339e-04, 1.56724306e-04, 1.52261214e-04, 1.62202119e-04,
       9.52754577e-04, 1.54851170e-04, 1.64801812e-04, 1.61913955e-04,
       1.43053026e-04, 1.34482066e-04, 1.54851170e-04, 8.76674502e-04,
       1.38075136e-04, 1.57209090e-04, 1.49004015e-04, 1.36791533e-04,
       1.39115664e-04, 1.38075136e-04, 8.42088828e-04, 1.42355968e-04,
       1.41896914e-04, 1.43853614e-04, 1.56897681e-04, 1.61874794e-04,
       1.46454488e-04, 9.17949851e-04, 1.59284472e-04, 1.42792772e-04,
       1.50645644e-04, 1.85112179e-04, 1.78147441e-04, 1.74933462e-04,
       1.01113365e-03, 1.54428998e-04, 1.66025648e-04, 1.52485921e-04,
       1.28174143e-04, 1.29885512e-04, 1.28913432e-04, 7.81847732e-04,
       1.31545455e-04, 1.31204268e-04, 1.32124922e-04, 1.35394795e-04,
       1.31272473e-04, 7.74885309e-04, 1.32120216e-04, 1.24049965e-04,
       1.26135221e-04, 1.25912639e-04, 1.42465175e-04, 1.38718515e-04,
       1.32120216e-04, 8.07045438e-04, 1.25237501e-04, 1.35820043e-04,
       1.32683988e-04, 1.47290181e-04, 1.57227912e-04, 1.39774160e-04,
       8.67649703e-04, 1.37626420e-04, 1.48908251e-04, 1.36822779e-04,
       1.28120000e-04, 1.18423640e-04, 6.58188055e-04, 1.30267818e-04,
       1.39127405e-04, 1.42249192e-04, 1.30872106e-04, 1.40093368e-04,
       1.30267818e-04, 8.13449101e-04, 1.35726372e-04, 1.36363365e-04,
       1.40126073e-04, 1.33112533e-04, 1.28832476e-04, 1.26768741e-04,
       1.28870130e-04, 1.35726372e-04, 7.86857954e-04, 1.33547703e-04,
       1.47009170e-04, 1.41661244e-04, 1.53925938e-04, 8.83447470e-04,
       1.44033720e-04, 1.50680846e-04, 1.46136553e-04, 1.46762376e-04,
       1.45921659e-04, 1.44772106e-04, 8.56375792e-04, 1.42303033e-04,
       1.39122486e-04, 1.37494131e-04, 1.43708221e-04, 1.40939508e-04,
       1.43064645e-04, 1.44163715e-04, 8.47900910e-04, 1.37177590e-04,
       1.38847233e-04, 1.36584851e-04, 1.35825746e-04, 1.37177590e-04,
       8.15839277e-04, 1.35872262e-04, 1.35462526e-04, 1.34916303e-04,
       1.31265552e-04, 1.27209340e-04, 1.24049965e-04, 7.54968560e-04,
       1.28780126e-04, 1.22168763e-04, 1.21494814e-04, 1.28389739e-04,
       1.31971478e-04, 1.26135221e-04, 1.25237501e-04, 1.28780126e-04,
       7.71978210e-04, 1.31464145e-04, 1.24851445e-04, 1.24813832e-04,
       1.22168763e-04, 7.39023288e-04, 1.21903833e-04, 1.22756365e-04,
       1.22529049e-04, 1.50203949e-04, 1.50716480e-04, 9.13110198e-04,
       1.51596598e-04, 1.52777323e-04, 1.54242022e-04, 1.53573827e-04,
       1.50760655e-04, 1.46588388e-04, 1.51596598e-04, 8.84339092e-04,
       1.43301961e-04, 1.48106930e-04, 1.43984560e-04, 1.41141665e-04,
       1.40839239e-04, 1.43301961e-04, 8.55628364e-04, 1.42290873e-04,
       1.44381644e-04, 1.43672983e-04, 1.46010832e-04, 1.42355968e-04,
       1.52777323e-04, 8.91544248e-04, 1.44763664e-04, 1.54997628e-04,
       1.50638833e-04, 1.40220160e-04, 1.53946289e-04, 1.42290873e-04,
       9.40245870e-04, 1.70529141e-04, 1.59373634e-04, 1.73885773e-04,
       1.66299841e-04, 1.59284472e-04, 1.70529141e-04, 9.95663321e-04,
       1.59230404e-04, 1.72301415e-04, 1.68018047e-04, 1.35347271e-04,
       1.42792772e-04, 1.25912639e-04, 1.21494814e-04, 1.21903833e-04,
       7.81218489e-04, 1.33767160e-04, 1.52612574e-04, 1.45785803e-04,
       8.91374332e-04, 1.55762967e-04, 1.41035934e-04, 1.52038658e-04,
       1.44138396e-04, 1.59688413e-04, 1.67972829e-04, 1.72480962e-04,
       1.55762967e-04, 9.85012019e-04, 1.68770213e-04, 1.60336635e-04,
       1.41790994e-04, 1.39115097e-04, 1.41035934e-04, 8.32939735e-04,
       1.37645063e-04, 1.37033810e-04, 1.36318837e-04, 1.43479962e-04,
       1.48981749e-04, 1.37645063e-04, 8.57215077e-04, 1.48622797e-04,
       1.36504780e-04, 1.41980727e-04, 1.49023289e-04, 1.49750149e-04,
       1.48622797e-04, 8.94063055e-04, 1.49771004e-04, 1.48237234e-04,
       1.48658581e-04, 1.56231563e-04, 1.55615772e-04, 1.49771004e-04,
       9.19702315e-04, 1.44040662e-04, 1.59578932e-04, 1.54464382e-04,
       1.61483953e-04, 1.54428998e-04, 1.44040662e-04, 7.21555392e-04,
       1.30192488e-04, 1.31409291e-04, 1.58222822e-04, 1.52454503e-04,
       9.28516333e-04, 1.60816152e-04, 1.50054890e-04, 1.55980454e-04,
       1.50987511e-04, 1.61832784e-04, 1.66025648e-04, 1.60816152e-04,
       9.76820502e-04, 1.65558122e-04, 1.61019346e-04, 1.61568451e-04,
       1.32180642e-04, 1.31545455e-04, 8.08596652e-04, 1.36268232e-04,
       1.33533709e-04, 1.38583974e-04, 1.36484639e-04, 1.32905023e-04,
       1.57232626e-04, 1.36268232e-04, 9.37210200e-04, 1.78813811e-04,
       1.56886266e-04, 1.75104242e-04, 1.82880235e-04, 1.77739558e-04,
       1.68770213e-04, 1.78813811e-04, 1.04383449e-03, 1.65363878e-04,
       1.70266799e-04, 1.38709126e-04, 1.48724855e-04, 1.31204268e-04,
       1.50054890e-04, 8.44645121e-04, 1.33558544e-04, 1.42393437e-04,
       1.37173957e-04, 1.39911887e-04, 1.35820043e-04, 8.15713991e-04,
       1.35317898e-04, 1.35365141e-04, 1.32125066e-04, 1.37848064e-04,
       1.37626420e-04, 1.35317898e-04, 8.14281066e-04, 1.36053558e-04,
       1.34196215e-04, 1.33238911e-04, 1.48937296e-04, 1.48908251e-04,
       1.39127405e-04, 8.75821844e-04, 1.49875265e-04, 1.42748204e-04,
       1.46225422e-04, 1.35275422e-04, 1.40769928e-04, 1.36363365e-04,
       1.33547703e-04, 1.44033720e-04, 8.31345012e-04, 1.41354875e-04,
       1.50930908e-04, 1.45636515e-04, 1.42303033e-04, 8.70413748e-04,
       1.52573495e-04, 1.41972932e-04, 1.36996863e-04, 1.56096663e-04,
       1.50680846e-04, 1.49875265e-04, 1.52573495e-04, 9.06259214e-04,
       1.48148962e-04, 1.48883982e-04, 1.40842651e-04, 1.39122486e-04,
       1.38847233e-04, 1.35872262e-04, 8.24595976e-04, 1.35583076e-04,
       1.34328269e-04, 1.36654564e-04, 1.32683988e-04, 1.35462526e-04,
       1.31464145e-04, 1.35365141e-04, 8.04583624e-04, 1.32953259e-04,
       1.56703644e-04, 1.35137227e-04, 8.42565858e-04, 1.65135311e-04,
       1.21366152e-04, 1.21010391e-04, 1.43213133e-04, 1.67508034e-04,
       1.68180234e-04, 1.65135311e-04, 1.03117042e-03, 1.67231013e-04,
       1.31465700e-04, 1.32891431e-04, 9.87586993e-05, 1.22491724e-04,
       1.22756365e-04, 1.21366152e-04, 7.34046467e-04, 1.23491808e-04,
       1.22900716e-04, 1.21039702e-04, 1.65139339e-04, 1.64801812e-04,
       1.67231013e-04, 9.93820778e-04, 1.70062597e-04, 1.61708454e-04,
       1.64877564e-04, 1.61913955e-04, 1.57209090e-04, 1.70062597e-04,
       9.90362988e-04, 1.60082861e-04, 1.73184678e-04, 1.67909808e-04,
       1.49004015e-04, 1.41896914e-04, 1.60082861e-04, 9.13223841e-04,
       1.45424766e-04, 1.62183139e-04, 1.54632146e-04, 1.43853614e-04,
       1.44763664e-04, 1.45424766e-04, 8.73880756e-04, 1.45361402e-04,
       1.47725362e-04, 1.46751948e-04, 1.50645644e-04, 1.22529049e-04,
       1.59230404e-04, 1.33767160e-04, 1.23491808e-04, 8.32978744e-04,
       1.43314679e-04, 1.52485921e-04, 1.30192488e-04, 1.65558122e-04,
       9.47029429e-04, 1.59572792e-04, 1.61456002e-04, 1.77764105e-04,
       1.32124922e-04, 1.33533709e-04, 1.33558544e-04, 8.05800707e-04,
       1.34985734e-04, 1.35789698e-04, 1.35808100e-04, 1.36822779e-04,
       1.36053558e-04, 1.42748204e-04, 1.41972932e-04, 1.48148962e-04,
       8.41549389e-04, 1.35802953e-04, 1.42249192e-04, 1.40126073e-04,
       1.46136553e-04, 1.46225422e-04, 1.41354875e-04, 1.48883982e-04,
       8.64976097e-04, 1.37494131e-04, 1.34196215e-04, 1.36996863e-04,
       1.35583076e-04, 1.35802953e-04, 8.13552308e-04, 1.33479071e-04,
       1.34916303e-04, 1.32125066e-04, 1.33238911e-04, 1.34328269e-04,
       1.32953259e-04, 1.33479071e-04, 8.01040879e-04, 1.54242022e-04,
       1.54997628e-04, 9.31239067e-04, 1.54971302e-04, 1.56111543e-04,
       1.55650603e-04, 1.55265969e-04, 1.53573827e-04, 1.48106930e-04,
       1.54971302e-04, 8.95384029e-04, 1.44106435e-04, 1.50011753e-04,
       1.44613783e-04, 1.43984560e-04, 1.44381644e-04, 1.44106435e-04,
       8.64407275e-04, 1.44410143e-04, 1.43808934e-04, 1.43715559e-04,
       1.43672983e-04, 1.59373634e-04, 1.44410143e-04, 9.50196856e-04,
       1.73536286e-04, 1.57889159e-04, 1.71314651e-04, 1.50638833e-04,
       1.45361402e-04, 1.56111543e-04, 8.99608273e-04, 1.44698599e-04,
       1.54466704e-04, 1.48331192e-04, 1.73885773e-04, 1.72301415e-04,
       1.73536286e-04, 1.04597986e-03, 1.73500519e-04, 1.75603638e-04,
       1.77152228e-04, 1.68018047e-04, 1.22900716e-04, 1.43314679e-04,
       1.73500519e-04, 8.78661797e-04, 1.22515700e-04, 1.48412135e-04,
       1.52038658e-04, 1.60336635e-04, 1.65363878e-04, 9.39614933e-04,
       1.47965816e-04, 1.61505016e-04, 1.52404930e-04, 1.44138396e-04,
       1.37033810e-04, 1.47965816e-04, 8.46454749e-04, 1.34676840e-04,
       1.44412138e-04, 1.38227748e-04, 1.36318837e-04, 1.36504780e-04,
       1.34676840e-04, 8.07402054e-04, 1.34913899e-04, 1.32468291e-04,
       1.32519408e-04, 1.41980727e-04, 1.48237234e-04, 1.34913899e-04,
       8.47097222e-04, 1.47155453e-04, 1.34412432e-04, 1.40397477e-04,
       1.48658581e-04, 1.59578932e-04, 1.47155453e-04, 9.26202410e-04,
       1.66842869e-04, 1.47599754e-04, 1.56366820e-04, 1.54464382e-04,
       1.31409291e-04, 1.59572792e-04, 1.66842869e-04, 1.12370703e-03,
       1.86942545e-04, 1.58138551e-04, 1.66336599e-04, 1.55980454e-04,
       1.61019346e-04, 9.27595769e-04, 1.49878439e-04, 1.59551947e-04,
       1.48265483e-04, 1.52900100e-04, 1.50987511e-04, 1.42393437e-04,
       1.34985734e-04, 1.49878439e-04, 8.56204695e-04, 1.35830472e-04,
       1.42129102e-04, 1.61568451e-04, 1.61456002e-04, 1.59551947e-04,
       9.62957039e-04, 1.61827320e-04, 1.58082748e-04, 1.60470571e-04,
       1.38583974e-04, 1.56886266e-04, 9.23566557e-04, 1.40941864e-04,
       1.69844439e-04, 1.53354865e-04, 1.63955149e-04, 1.36484639e-04,
       1.35789698e-04, 1.40941864e-04, 8.31621425e-04, 1.36353022e-04,
       1.42973051e-04, 1.39079151e-04, 1.75104242e-04, 1.70266799e-04,
       1.61505016e-04, 1.69844439e-04, 9.95788238e-04, 1.57782881e-04,
       1.61284861e-04, 1.21010391e-04, 1.21039702e-04, 1.22515700e-04,
       7.33703217e-04, 1.27425420e-04, 1.18386486e-04, 1.23325517e-04,
       1.43213133e-04, 1.31465700e-04, 1.27425420e-04, 7.29100119e-04,
       9.70636355e-05, 1.24273291e-04, 1.05658939e-04, 1.32891431e-04,
       1.61708454e-04, 8.15675561e-04, 1.56947824e-04, 9.93788612e-05,
       1.46750465e-04, 1.17998526e-04, 1.64877564e-04, 1.73184678e-04,
       1.56947824e-04, 1.00736120e-03, 1.80634852e-04, 1.58168183e-04,
       1.73548098e-04, 1.67909808e-04, 1.62183139e-04, 1.80634852e-04,
       1.03579927e-03, 1.63374603e-04, 1.86615225e-04, 1.75081645e-04,
       1.54632146e-04, 1.47725362e-04, 1.63374603e-04, 9.33634679e-04,
       1.48647544e-04, 1.63537650e-04, 1.55717375e-04, 1.46751948e-04,
       1.44698599e-04, 1.48647544e-04, 8.76763376e-04, 1.43156791e-04,
       1.48472950e-04, 1.45035545e-04, 1.77764105e-04, 1.86942545e-04,
       1.61827320e-04, 9.93215848e-04, 1.58638774e-04, 1.60205045e-04,
       1.47838059e-04, 1.35808100e-04, 1.35830472e-04, 1.36353022e-04,
       8.14614374e-04, 1.35538265e-04, 1.35960822e-04, 1.35123693e-04,
       9.87586993e-05, 9.70636355e-05, 9.93788612e-05, 5.09308303e-04,
       1.05897055e-04, 1.08210052e-04, 1.55650603e-04, 1.54466704e-04,
       9.24047156e-04, 1.55401921e-04, 1.52286822e-04, 1.52154953e-04,
       1.54086153e-04, 1.55265969e-04, 1.50011753e-04, 1.55401921e-04,
       8.99832005e-04, 1.44768596e-04, 1.49881438e-04, 1.44502328e-04,
       1.44613783e-04, 1.43808934e-04, 1.44768596e-04, 8.63087155e-04,
       1.43118152e-04, 1.43811643e-04, 1.42966048e-04, 1.43715559e-04,
       1.57889159e-04, 1.43118152e-04, 9.36286942e-04, 1.69613365e-04,
       1.54814548e-04, 1.67136160e-04, 1.71314651e-04, 1.75603638e-04,
       1.69613365e-04, 1.03470024e-03, 1.76400931e-04, 1.69634537e-04,
       1.72133115e-04, 1.48331192e-04, 1.43156791e-04, 1.52286822e-04,
       8.81583346e-04, 1.42489821e-04, 1.49970660e-04, 1.45348060e-04,
       1.77152228e-04, 1.48412135e-04, 1.18386486e-04, 1.76400931e-04,
       8.90062749e-04, 1.20218309e-04, 1.49492660e-04, 1.52404930e-04,
       1.44412138e-04, 1.57782881e-04, 8.96071406e-04, 1.41489924e-04,
       1.54140849e-04, 1.45840684e-04, 1.38227748e-04, 1.32468291e-04,
       1.41489924e-04, 8.17857284e-04, 1.32287717e-04, 1.38413177e-04,
       1.34970428e-04, 1.32519408e-04, 1.34412432e-04, 1.32287717e-04,
       8.05285987e-04, 1.36984435e-04, 1.33139151e-04, 1.35942845e-04,
       1.40397477e-04, 1.47599754e-04, 1.36984435e-04, 8.56385862e-04,
       1.48795444e-04, 1.39000010e-04, 1.43608742e-04, 1.56366820e-04,
       1.58138551e-04, 1.48795444e-04, 8.96942427e-04, 1.49327302e-04,
       1.42777091e-04, 1.41537219e-04, 1.66336599e-04, 1.58638774e-04,
       1.49327302e-04, 9.73625914e-04, 1.43005124e-04, 1.77471234e-04,
       1.78846881e-04, 1.48265483e-04, 1.42129102e-04, 1.35538265e-04,
       8.48158604e-04, 1.46844994e-04, 1.35105206e-04, 1.40275554e-04,
       1.52900100e-04, 1.58082748e-04, 1.46844994e-04, 9.07484276e-04,
       1.57483784e-04, 1.43358254e-04, 1.48814396e-04, 1.60470571e-04,
       1.60205045e-04, 1.57483784e-04, 9.50147843e-04, 1.63245521e-04,
       1.51357830e-04, 1.57385093e-04, 1.53354865e-04, 1.42973051e-04,
       9.06982735e-04, 1.58116970e-04, 1.45784803e-04, 1.52401347e-04,
       1.54351700e-04, 1.63955149e-04, 1.61284861e-04, 1.54140849e-04,
       1.58116970e-04, 9.35378845e-04, 1.48287591e-04, 1.49593424e-04,
       1.39079151e-04, 1.35960822e-04, 1.45784803e-04, 8.42017756e-04,
       1.34979233e-04, 1.46950494e-04, 1.39263253e-04, 1.23325517e-04,
       1.24273291e-04, 1.20218309e-04, 7.55607169e-04, 1.22605527e-04,
       1.31872540e-04, 1.33311984e-04, 1.05658939e-04, 1.05897055e-04,
       1.22605527e-04, 7.12400240e-04, 1.15785938e-04, 1.34755242e-04,
       1.27697538e-04, 1.46750465e-04, 1.58168183e-04, 8.86028651e-04,
       1.34904009e-04, 1.57811944e-04, 1.38451916e-04, 1.49942133e-04,
       1.17998526e-04, 1.08210052e-04, 1.15785938e-04, 1.34904009e-04,
       7.30480172e-04, 1.22126025e-04, 1.31455622e-04, 1.73548098e-04,
       1.86615225e-04, 1.57811944e-04, 1.04508906e-03, 1.86658742e-04,
       1.62337690e-04, 1.78117361e-04, 1.75081645e-04, 1.63537650e-04,
       1.86658742e-04, 1.04627174e-03, 1.62279087e-04, 1.85775137e-04,
       1.72939477e-04, 1.55717375e-04, 1.48472950e-04, 1.62279087e-04,
       9.25769043e-04, 1.46910337e-04, 1.60256810e-04, 1.52132484e-04,
       1.45035545e-04, 1.42489821e-04, 1.46910337e-04, 8.66125104e-04,
       1.41238527e-04, 1.46788480e-04, 1.43662394e-04, 1.47838059e-04,
       1.43005124e-04, 1.63245521e-04, 7.58960989e-04, 1.47273990e-04,
       1.57598295e-04, 1.35123693e-04, 1.35105206e-04, 1.34979233e-04,
       8.08643588e-04, 1.35611028e-04, 1.33587073e-04, 1.34237355e-04,
       1.52154953e-04, 1.49970660e-04, 8.99436345e-04, 1.52170459e-04,
       1.46949168e-04, 1.50614051e-04, 1.47577053e-04, 1.54086153e-04,
       1.49881438e-04, 1.52170459e-04, 8.90358466e-04, 1.44156617e-04,
       1.46936463e-04, 1.43127336e-04, 1.44502328e-04, 1.43811643e-04,
       1.44156617e-04, 8.59215237e-04, 1.42515726e-04, 1.42589564e-04,
       1.41639358e-04, 1.42966048e-04, 1.54814548e-04, 1.42515726e-04,
       9.23067440e-04, 1.65745886e-04, 1.52821786e-04, 1.64203446e-04,
       1.67136160e-04, 1.69634537e-04, 1.65745886e-04, 1.01347451e-03,
       1.70628033e-04, 1.68973062e-04, 1.71356833e-04, 1.72133115e-04,
       1.49492660e-04, 1.31872540e-04, 1.70628033e-04, 9.16997181e-04,
       1.38377898e-04, 1.54492935e-04, 1.45348060e-04, 1.41238527e-04,
       1.46949168e-04, 8.57226637e-04, 1.40456684e-04, 1.42808574e-04,
       1.40425623e-04, 1.45840684e-04, 1.38413177e-04, 1.48287591e-04,
       8.47171245e-04, 1.34978152e-04, 1.42766787e-04, 1.36884854e-04,
       1.34970428e-04, 1.33139151e-04, 1.34978152e-04, 8.08318543e-04,
       1.35159554e-04, 1.34029289e-04, 1.36041968e-04, 1.35942845e-04,
       1.39000010e-04, 1.35159554e-04, 8.34505088e-04, 1.42790408e-04,
       1.38519292e-04, 1.43092979e-04, 1.43608742e-04, 1.42777091e-04,
       1.42790408e-04, 8.48119391e-04, 1.36888115e-04, 1.43562839e-04,
       1.38492197e-04, 1.41537219e-04, 1.77471234e-04, 1.36888115e-04,
       8.75146078e-04, 1.66549168e-04, 1.29486652e-04, 1.23213690e-04,
       1.78846881e-04, 1.47273990e-04, 1.66549168e-04, 8.99183322e-04,
       1.51139750e-04, 1.19605030e-04, 1.35768503e-04, 1.40275554e-04,
       1.43358254e-04, 1.35611028e-04, 8.36768903e-04, 1.42373553e-04,
       1.35735345e-04, 1.39415169e-04, 1.48814396e-04, 1.51357830e-04,
       1.42373553e-04, 8.83477035e-04, 1.45442731e-04, 1.47481391e-04,
       1.48007135e-04, 1.57385093e-04, 1.57598295e-04, 1.51139750e-04,
       1.45442731e-04, 9.42433941e-04, 1.68175945e-04, 1.62692128e-04,
       1.52401347e-04, 1.46950494e-04, 8.89829373e-04, 1.49533577e-04,
       1.48817564e-04, 1.48430616e-04, 1.43695775e-04, 1.54351700e-04,
       1.49593424e-04, 1.42766787e-04, 1.49533577e-04, 8.68138658e-04,
       1.34942328e-04, 1.36950843e-04, 1.39263253e-04, 1.33587073e-04,
       1.48817564e-04, 8.09606946e-04, 1.30182860e-04, 1.35357359e-04,
       1.22398836e-04, 1.33311984e-04, 1.34755242e-04, 1.38377898e-04,
       8.30189742e-04, 1.40795267e-04, 1.40987620e-04, 1.41961731e-04,
       1.27697538e-04, 1.22126025e-04, 1.40795267e-04, 8.11735895e-04,
       1.31479463e-04, 1.46690946e-04, 1.42946655e-04, 1.38451916e-04,
       1.31455622e-04, 1.31479463e-04, 8.27601921e-04, 1.44495386e-04,
       1.37849953e-04, 1.43869582e-04, 1.49942133e-04, 1.62337690e-04,
       1.44495386e-04, 9.35741675e-04, 1.68372459e-04, 1.49556246e-04,
       1.61037761e-04, 1.78117361e-04, 1.85775137e-04, 1.68372459e-04,
       1.07584377e-03, 1.85640415e-04, 1.74164106e-04, 1.83774286e-04,
       1.72939477e-04, 1.60256810e-04, 1.85640415e-04, 1.03117006e-03,
       1.60844437e-04, 1.81801118e-04, 1.69687806e-04, 1.52132484e-04,
       1.46788480e-04, 1.60844437e-04, 9.15815199e-04, 1.46163201e-04,
       1.59611914e-04, 1.50274683e-04, 1.43662394e-04, 1.40456684e-04,
       1.46163201e-04, 8.52273420e-04, 1.38604050e-04, 1.43870266e-04,
       1.39516825e-04, 1.34237355e-04, 1.35735345e-04, 1.30182860e-04,
       7.99210044e-04, 1.36698449e-04, 1.28669218e-04, 1.33686817e-04,
       1.50614051e-04, 1.46936463e-04, 8.77861312e-04, 1.48677807e-04,
       1.42277626e-04, 1.46038979e-04, 1.43316386e-04, 1.47577053e-04,
       1.42808574e-04, 1.48677807e-04, 8.64378736e-04, 1.37929449e-04,
       1.46682866e-04, 1.40702987e-04, 1.43127336e-04, 1.42589564e-04,
       1.42277626e-04, 8.48823882e-04, 1.40880583e-04, 1.40404022e-04,
       1.39544751e-04, 1.41639358e-04, 1.52821786e-04, 1.40880583e-04,
       9.06643681e-04, 1.62392876e-04, 1.49289606e-04, 1.59619471e-04,
       1.64203446e-04, 1.68973062e-04, 1.62392876e-04, 1.01568578e-03,
       1.73881243e-04, 1.69758200e-04, 1.76476951e-04, 1.71356833e-04,
       1.54492935e-04, 1.40987620e-04, 1.73881243e-04, 9.45480202e-04,
       1.44365923e-04, 1.60395649e-04, 1.40425623e-04, 1.38604050e-04,
       1.37929449e-04, 8.20021709e-04, 1.35894092e-04, 1.33691140e-04,
       1.33477356e-04, 1.36884854e-04, 1.34029289e-04, 1.34942328e-04,
       8.02083730e-04, 1.36132721e-04, 1.28024290e-04, 1.32070248e-04,
       1.36041968e-04, 1.38519292e-04, 1.36132721e-04, 8.53727342e-04,
       1.49149874e-04, 1.41671829e-04, 1.52211659e-04, 1.43092979e-04,
       1.43562839e-04, 1.49149874e-04, 8.95914218e-04, 1.47740897e-04,
       1.57123233e-04, 1.55244396e-04, 1.38492197e-04, 1.29486652e-04,
       1.47740897e-04, 8.27607685e-04, 1.25949130e-04, 1.49362516e-04,
       1.36576294e-04, 1.23213690e-04, 1.19605030e-04, 1.25949130e-04,
       6.10253527e-04, 1.17431206e-04, 1.24054470e-04, 1.35768503e-04,
       1.68175945e-04, 1.17431206e-04, 8.52498632e-04, 1.77251548e-04,
       1.13229265e-04, 1.40642165e-04, 1.39415169e-04, 1.47481391e-04,
       1.36698449e-04, 8.52916415e-04, 1.46557254e-04, 1.40485785e-04,
       1.42278367e-04, 1.48007135e-04, 1.62692128e-04, 1.77251548e-04,
       1.46557254e-04, 9.64852605e-04, 1.73176921e-04, 1.57167620e-04,
       1.48430616e-04, 1.35357359e-04, 8.83105030e-04, 1.37047568e-04,
       1.17244237e-04, 1.75877663e-04, 1.69147588e-04, 1.43695775e-04,
       1.36950843e-04, 1.28024290e-04, 1.37047568e-04, 6.74923123e-04,
       1.29204648e-04, 1.22398836e-04, 1.28669218e-04, 1.17244237e-04,
       6.20605554e-04, 1.28892519e-04, 1.23400745e-04, 1.41961731e-04,
       1.46690946e-04, 1.44365923e-04, 8.79068199e-04, 1.51728427e-04,
       1.45843941e-04, 1.48477230e-04, 1.42946655e-04, 1.37849953e-04,
       1.51728427e-04, 8.75858401e-04, 1.41847496e-04, 1.53135050e-04,
       1.48350821e-04, 1.43869582e-04, 1.49556246e-04, 1.41847496e-04,
       8.81631076e-04, 1.52457696e-04, 1.44488260e-04, 1.49411796e-04,
       1.61037761e-04, 1.74164106e-04, 1.52457696e-04, 9.77783420e-04,
       1.73935877e-04, 1.53918071e-04, 1.62269908e-04, 1.83774286e-04,
       1.81801118e-04, 1.73935877e-04, 1.05686178e-03, 1.73076684e-04,
       1.71579917e-04, 1.72693893e-04, 1.69687806e-04, 1.59611914e-04,
       1.73076684e-04, 9.82244967e-04, 1.55706642e-04, 1.65728035e-04,
       1.58433885e-04, 1.50274683e-04, 1.43870266e-04, 1.55706642e-04,
       8.81151793e-04, 1.40855970e-04, 1.49445243e-04, 1.40998988e-04,
       1.39516825e-04, 1.35894092e-04, 1.40855970e-04, 8.23195926e-04,
       1.33718955e-04, 1.38362183e-04, 1.34847901e-04, 1.33686817e-04,
       1.40485785e-04, 1.28892519e-04, 8.56287943e-04, 1.47549713e-04,
       1.46907442e-04, 1.58765668e-04, 1.46038979e-04, 1.46682866e-04,
       8.65773846e-04, 1.43452363e-04, 1.44699323e-04, 1.42751694e-04,
       1.42148621e-04, 1.43316386e-04, 1.40404022e-04, 1.43452363e-04,
       8.39393273e-04, 1.36576008e-04, 1.39804242e-04, 1.35840252e-04,
       1.40702987e-04, 1.33691140e-04, 1.44699323e-04, 8.30936472e-04,
       1.31204801e-04, 1.43560448e-04, 1.37077773e-04, 1.39544751e-04,
       1.49289606e-04, 1.36576008e-04, 8.65117820e-04, 1.53376261e-04,
       1.39637898e-04, 1.46693296e-04, 1.59619471e-04, 1.69758200e-04,
       1.53376261e-04, 1.00999488e-03, 1.80712699e-04, 1.64665270e-04,
       1.81862979e-04, 1.76476951e-04, 1.60395649e-04, 1.45843941e-04,
       1.80712699e-04, 9.80211513e-04, 1.48997409e-04, 1.67784865e-04,
       1.33477356e-04, 1.33718955e-04, 1.31204801e-04, 7.97527525e-04,
       1.33557859e-04, 1.31728548e-04, 1.33840006e-04, 1.32070248e-04,
       1.41671829e-04, 1.75877663e-04, 1.29204648e-04, 9.44202329e-04,
       1.64153254e-04, 2.01224688e-04, 1.52211659e-04, 1.57123233e-04,
       1.64153254e-04, 1.02876070e-03, 1.69003139e-04, 1.93165554e-04,
       1.93103865e-04, 1.55244396e-04, 1.49362516e-04, 1.69003139e-04,
       9.80429429e-04, 1.58057702e-04, 1.76912616e-04, 1.71849059e-04,
       1.36576294e-04, 1.24054470e-04, 1.13229265e-04, 1.58057702e-04,
       7.77513103e-04, 1.06644379e-04, 1.38950992e-04, 1.40642165e-04,
       1.73176921e-04, 1.06644379e-04, 6.91722043e-04, 1.52231140e-04,
       1.19027440e-04, 1.42278367e-04, 1.57167620e-04, 1.47549713e-04,
       1.52231140e-04, 8.80510868e-04, 1.45745927e-04, 1.35538101e-04,
       1.69147588e-04, 1.23400745e-04, 1.46907442e-04, 2.01224688e-04,
       1.93165554e-04, 1.01844117e-03, 1.84595151e-04, 1.48477230e-04,
       1.53135050e-04, 1.48997409e-04, 9.14681160e-04, 1.56891540e-04,
       1.51971811e-04, 1.55208122e-04, 1.48350821e-04, 1.44488260e-04,
       1.56891540e-04, 9.07800258e-04, 1.47109138e-04, 1.58440171e-04,
       1.52520329e-04, 1.49411796e-04, 1.53918071e-04, 1.47109138e-04,
       9.03578199e-04, 1.54046688e-04, 1.48330615e-04, 1.50761891e-04,
       1.62269908e-04, 1.71579917e-04, 1.54046688e-04, 9.66946959e-04,
       1.67751940e-04, 1.53451632e-04, 1.57846874e-04, 1.72693893e-04,
       1.65728035e-04, 1.67751940e-04, 9.83866265e-04, 1.56170204e-04,
       1.63069036e-04, 1.58453157e-04, 1.58433885e-04, 1.49445243e-04,
       1.56170204e-04, 8.63195684e-04, 1.27643326e-04, 1.45520631e-04,
       1.25982395e-04, 1.40998988e-04, 1.38362183e-04, 1.27643326e-04,
       7.73433389e-04, 1.41044175e-04, 1.04673531e-04, 1.20711185e-04,
       1.34847901e-04, 1.33557859e-04, 1.41044175e-04, 8.36149219e-04,
       1.39506561e-04, 1.43720148e-04, 1.43472576e-04, 1.58765668e-04,
       1.93103865e-04, 1.76912616e-04, 1.45745927e-04, 1.84595151e-04,
       1.01450739e-03, 1.55384162e-04, 1.42751694e-04, 1.43560448e-04,
       8.51932116e-04, 1.41112736e-04, 1.42619906e-04, 1.41292874e-04,
       1.40594458e-04, 1.42148621e-04, 1.39804242e-04, 1.41112736e-04,
       8.30149537e-04, 1.35392640e-04, 1.37533507e-04, 1.34157790e-04,
       1.35840252e-04, 1.39637898e-04, 1.35392640e-04, 8.23976766e-04,
       1.40981977e-04, 1.35166155e-04, 1.36957845e-04, 1.37077773e-04,
       1.31728548e-04, 1.42619906e-04, 8.25558939e-04, 1.32971945e-04,
       1.42729751e-04, 1.38431016e-04, 1.46693296e-04, 1.64665270e-04,
       1.40981977e-04, 9.73917279e-04, 1.81861829e-04, 1.58403515e-04,
       1.81311392e-04, 1.81862979e-04, 1.67784865e-04, 1.51971811e-04,
       1.81861829e-04, 1.01120617e-03, 1.55956390e-04, 1.71768294e-04,
       1.33840006e-04, 1.39506561e-04, 1.32971945e-04, 8.32522215e-04,
       1.43267258e-04, 1.39153843e-04, 1.43782601e-04, 1.71849059e-04,
       1.38950992e-04, 1.19027440e-04, 1.35538101e-04, 1.55384162e-04,
       7.20749754e-04, 1.55208122e-04, 1.58440171e-04, 1.55956390e-04,
       9.47361472e-04, 1.58516736e-04, 1.59955878e-04, 1.59284175e-04,
       1.52520329e-04, 1.48330615e-04, 1.58516736e-04, 9.20997957e-04,
       1.49582233e-04, 1.58396416e-04, 1.53651627e-04, 1.50761891e-04,
       1.53451632e-04, 1.49582233e-04, 9.10482881e-04, 1.54314886e-04,
       1.50164663e-04, 1.52207575e-04, 1.57846874e-04, 1.63069036e-04,
       1.54314886e-04, 9.52092125e-04, 1.62623388e-04, 1.55575800e-04,
       1.58662140e-04, 1.58453157e-04, 1.45520631e-04, 1.62623388e-04,
       9.16348652e-04, 1.37625014e-04, 1.62096154e-04, 1.50030308e-04,
       1.25982395e-04, 1.04673531e-04, 1.37625014e-04, 5.98601825e-04,
       1.05457352e-04, 1.24863533e-04, 1.20711185e-04, 1.43720148e-04,
       1.05457352e-04, 7.97605108e-04, 1.45391443e-04, 1.32699925e-04,
       1.49625054e-04, 1.43472576e-04, 1.43267258e-04, 1.45391443e-04,
       8.84388348e-04, 1.47480853e-04, 1.51240745e-04, 1.53535472e-04,
       1.41292874e-04, 1.42729751e-04, 8.54490143e-04, 1.41870112e-04,
       1.41753150e-04, 1.44199047e-04, 1.42645208e-04, 1.40594458e-04,
       1.37533507e-04, 1.41870112e-04, 8.25560408e-04, 1.33996709e-04,
       1.38189037e-04, 1.33376585e-04, 1.34157790e-04, 1.35166155e-04,
       1.33996709e-04, 7.98206964e-04, 1.33499844e-04, 1.31445847e-04,
       1.29940619e-04, 1.36957845e-04, 1.58403515e-04, 1.33499844e-04,
       9.39659136e-04, 1.79261811e-04, 1.53609911e-04, 1.77926209e-04,
       1.38431016e-04, 1.39153843e-04, 1.41753150e-04, 8.47820213e-04,
       1.42582227e-04, 1.42896863e-04, 1.43003114e-04, 1.81311392e-04,
       1.71768294e-04, 1.59955878e-04, 1.79261811e-04, 1.02800944e-03,
       1.62974613e-04, 1.72737448e-04, 1.43782601e-04, 1.47480853e-04,
       1.42582227e-04, 9.03738498e-04, 1.57966648e-04, 1.50120000e-04,
       1.61806169e-04, 1.59284175e-04, 1.58396416e-04, 1.62974613e-04,
       9.63701875e-04, 1.56669248e-04, 1.65897515e-04, 1.60479909e-04,
       1.53651627e-04, 1.50164663e-04, 1.56669248e-04, 9.18099134e-04,
       1.49729632e-04, 1.55668308e-04, 1.52215656e-04, 1.52207575e-04,
       1.55575800e-04, 1.49729632e-04, 9.18461996e-04, 1.56701222e-04,
       1.50322201e-04, 1.53925565e-04, 1.58662140e-04, 1.62096154e-04,
       1.56701222e-04, 9.55705094e-04, 1.62766727e-04, 1.56423680e-04,
       1.59055171e-04, 1.50030308e-04, 1.24863533e-04, 1.32699925e-04,
       1.62766727e-04, 8.89144973e-04, 1.56942227e-04, 1.61842254e-04,
       1.49625054e-04, 1.51240745e-04, 1.56942227e-04, 9.41152302e-04,
       1.53762187e-04, 1.67188909e-04, 1.62393179e-04, 1.53535472e-04,
       1.57966648e-04, 1.53762187e-04, 7.78666979e-04, 1.59349079e-04,
       1.54053592e-04, 1.44199047e-04, 1.38189037e-04, 8.60760872e-04,
       1.54062759e-04, 1.34895981e-04, 1.51285407e-04, 1.38128640e-04,
       1.42645208e-04, 1.42896863e-04, 1.54062759e-04, 8.95660183e-04,
       1.45275550e-04, 1.59909333e-04, 1.50870469e-04, 1.33376585e-04,
       1.31445847e-04, 1.34895981e-04, 7.84570527e-04, 1.27810661e-04,
       1.31098018e-04, 1.25943435e-04, 1.29940619e-04, 1.53609911e-04,
       1.27810661e-04, 8.94053423e-04, 1.72593891e-04, 1.44492202e-04,
       1.65606139e-04, 1.77926209e-04, 1.72737448e-04, 1.65897515e-04,
       1.72593891e-04, 1.02334948e-03, 1.66343403e-04, 1.67851019e-04,
       1.43003114e-04, 1.50120000e-04, 1.45275550e-04, 8.76395228e-04,
       1.51754131e-04, 1.43440369e-04, 1.42802064e-04, 1.61806169e-04,
       1.59349079e-04, 1.51754131e-04, 9.52563369e-04, 1.67636404e-04,
       1.46839112e-04, 1.65178475e-04, 1.60479909e-04, 1.55668308e-04,
       1.66343403e-04, 9.57605869e-04, 1.51380964e-04, 1.66942062e-04,
       1.56791224e-04, 1.52215656e-04, 1.50322201e-04, 1.51380964e-04,
       8.96718896e-04, 1.47170310e-04, 1.48973483e-04, 1.46656284e-04,
       1.53925565e-04, 1.56423680e-04, 1.47170310e-04, 9.10212670e-04,
       1.55675618e-04, 1.45505151e-04, 1.51512346e-04, 1.59055171e-04,
       1.61842254e-04, 1.67188909e-04, 1.55675618e-04, 9.67363897e-04,
       1.66164077e-04, 1.57437869e-04, 1.62393179e-04, 1.54053592e-04,
       1.67636404e-04, 1.66164077e-04, 9.82679553e-04, 1.71122108e-04,
       1.61310194e-04, 1.51285407e-04, 1.59909333e-04, 9.63756422e-04,
       1.50066089e-04, 1.65149429e-04, 1.65443374e-04, 1.71902789e-04,
       1.38128640e-04, 1.31098018e-04, 1.50066089e-04, 7.95863839e-04,
       1.21708424e-04, 1.38094856e-04, 1.16767813e-04, 1.50870469e-04,
       1.43440369e-04, 1.65149429e-04, 8.63560995e-04, 1.39668132e-04,
       1.41241897e-04, 1.23190699e-04, 1.25943435e-04, 1.44492202e-04,
       1.21708424e-04, 7.82949847e-04, 1.47295156e-04, 1.18236332e-04,
       1.25274298e-04, 1.65606139e-04, 1.67851019e-04, 1.66942062e-04,
       1.47295156e-04, 9.54089563e-04, 1.61898607e-04, 1.44496581e-04,
       1.42802064e-04, 1.46839112e-04, 1.39668132e-04, 7.20853905e-04,
       1.49066434e-04, 1.42478164e-04, 1.65178475e-04, 1.71122108e-04,
       1.49066434e-04, 9.59098769e-04, 1.57136535e-04, 1.57234375e-04,
       1.59360844e-04, 1.56791224e-04, 1.48973483e-04, 1.61898607e-04,
       9.23162373e-04, 1.43600370e-04, 1.61189592e-04, 1.50709097e-04,
       1.46656284e-04, 1.45505151e-04, 1.43600370e-04, 8.61187356e-04,
       1.43963837e-04, 1.41488156e-04, 1.39973557e-04, 1.51512346e-04,
       1.57437869e-04, 1.61310194e-04, 1.57136535e-04, 1.43963837e-04,
       9.17076481e-04, 1.45715701e-04, 1.65443374e-04, 1.38094856e-04,
       6.70380880e-04, 1.42892897e-04, 1.12147304e-04, 1.11802448e-04,
       1.71902789e-04, 1.41241897e-04, 1.42892897e-04, 6.88541173e-04,
       1.14768036e-04, 1.17735554e-04, 1.16767813e-04, 1.18236332e-04,
       1.12147304e-04, 7.50154947e-04, 1.29648539e-04, 1.44693837e-04,
       1.28661121e-04, 1.23190699e-04, 1.42478164e-04, 1.57234375e-04,
       1.14768036e-04, 8.35614425e-04, 1.60561012e-04, 1.37382138e-04,
       1.25274298e-04, 1.44496581e-04, 1.61189592e-04, 1.29648539e-04,
       7.13398585e-04, 1.52789575e-04, 1.59360844e-04, 1.41488156e-04,
       1.45715701e-04, 1.60561012e-04, 8.85228085e-04, 1.36337329e-04,
       1.41765043e-04, 1.50709097e-04, 1.39973557e-04, 1.44693837e-04,
       1.52789575e-04, 1.36337329e-04, 8.63480526e-04, 1.38977131e-04,
       1.11802448e-04, 1.17735554e-04, 1.28661121e-04, 1.37382138e-04,
       1.41765043e-04, 1.38977131e-04, 7.76323434e-04])
        indices_f_gt = np.array([   0,    1,    2,    3,    5,    6,    9,    0,    1,    2,    3,
          4,    7,    0,    1,    2,    4,    5,    8,   10,   14,    0,
          1,    3,    6,    7,   11,   12,    1,    2,    4,    7,    8,
         13,   15,    0,    2,    5,    9,   10,   16,   17,    0,    3,
          6,    9,   11,   18,   19,    1,    3,    4,    7,   12,   13,
         20,    2,    4,    8,   14,   15,   21,   22,    0,    5,    6,
          9,   16,   19,   23,    2,    5,   10,   14,   17,   24,   25,
          3,    6,   11,   12,   18,   26,   27,    3,    7,   11,   12,
         20,   27,   28,    4,    7,   13,   15,   20,   29,   30,    2,
          8,   10,   14,   21,   24,   31,    4,    8,   13,   15,   22,
         30,   32,    5,    9,   16,   17,   23,   33,   34,    5,   10,
         16,   17,   25,   34,   35,    6,   11,   18,   19,   26,   36,
         37,    6,    9,   18,   19,   23,   37,   38,    7,   12,   13,
         20,   28,   29,   39,    8,   14,   21,   22,   31,   40,   41,
          8,   15,   21,   22,   32,   40,   42,    9,   16,   19,   23,
         33,   38,   43,   10,   14,   24,   25,   31,   44,   45,   10,
         17,   24,   25,   35,   45,   46,   11,   18,   26,   27,   36,
         47,   48,   11,   12,   26,   27,   28,   48,   49,   12,   20,
         27,   28,   39,   49,   50,   13,   20,   29,   30,   39,   51,
         52,   13,   15,   29,   30,   32,   52,   53,   14,   21,   24,
         31,   41,   44,   54,   15,   22,   30,   32,   42,   53,   55,
         16,   23,   33,   34,   43,   56,   57,   16,   17,   33,   34,
         35,   56,   58,   17,   25,   34,   35,   46,   58,   59,   18,
         26,   36,   37,   47,   60,   61,   18,   19,   36,   37,   38,
         60,   62,   19,   23,   37,   38,   43,   62,   63,   20,   28,
         29,   39,   50,   51,   64,   21,   22,   40,   41,   42,   65,
         66,   21,   31,   40,   41,   54,   65,   67,   22,   32,   40,
         42,   55,   66,   68,   23,   33,   38,   43,   57,   63,   69,
         24,   31,   44,   45,   54,   70,   71,   24,   25,   44,   45,
         46,   70,   72,   25,   35,   45,   46,   59,   72,   73,   26,
         36,   47,   48,   61,   74,   75,   26,   27,   47,   48,   49,
         74,   76,   27,   28,   48,   49,   50,   76,   77,   28,   39,
         49,   50,   64,   77,   78,   29,   39,   51,   52,   64,   79,
         80,   29,   30,   51,   52,   53,   80,   81,   30,   32,   52,
         53,   55,   81,   82,   31,   41,   44,   54,   67,   71,   83,
         32,   42,   53,   55,   68,   82,   84,   33,   34,   56,   57,
         58,   85,   86,  100,   33,   43,   56,   57,   69,   85,   87,
         34,   35,   56,   58,   59,   86,   88,   35,   46,   58,   59,
         73,   88,   89,   36,   37,   60,   61,   62,   90,   91,   36,
         47,   60,   61,   75,   91,   92,   37,   38,   60,   62,   63,
         90,   38,   43,   62,   63,   69,   90,   93,   39,   50,   51,
         64,   78,   79,   94,   95,   40,   41,   65,   66,   67,   96,
         97,  118,   40,   42,   65,   66,   68,   96,   41,   54,   65,
         67,   83,   97,   98,   42,   55,   66,   68,   84,   96,   99,
         43,   57,   63,   69,   87,   93,  101,   44,   45,   70,   71,
         72,  102,  103,   44,   54,   70,   71,   83,  103,  104,   45,
         46,   70,   72,   73,  102,  105,   46,   59,   72,   73,   89,
        105,  106,   47,   48,   74,   75,   76,  107,  108,   47,   61,
         74,   75,   92,  107,  109,   48,   49,   74,   76,   77,  108,
        110,   49,   50,   76,   77,   78,  110,  111,   50,   64,   77,
         78,   94,  111,  112,   51,   64,   79,   80,   95,  113,  114,
         51,   52,   79,   80,   81,  113,  115,   52,   53,   80,   81,
         82,  115,  116,   53,   55,   81,   82,   84,  116,  117,   54,
         67,   71,   83,   98,  104,  119,   55,   68,   82,   84,   99,
        117,  120,   56,   57,   85,   87,  100,  121,   56,   58,   86,
         88,  100,  122,  123,   57,   69,   85,   87,  101,  121,  124,
         58,   59,   86,   88,   89,  122,  125,   59,   73,   88,   89,
        106,  125,  126,   60,   62,   63,   90,   91,   93,  127,  130,
         60,   61,   90,   91,   92,  127,  128,   61,   75,   91,   92,
        109,  128,  129,   63,   69,   90,   93,  101,  130,  131,   64,
         78,   94,   95,  112,  132,   64,   79,   94,   95,  114,  132,
        133,   65,   66,   68,   96,   99,  118,  134,   65,   67,   97,
         98,  118,  135,  136,   67,   83,   97,   98,  119,  136,  137,
         68,   84,   96,   99,  120,  134,  138,   56,   85,   86,  100,
        121,  123,  139,   69,   87,   93,  101,  124,  131,  140,   70,
         72,  102,  103,  105,  141,  142,   70,   71,  102,  103,  104,
        141,  143,   71,   83,  103,  104,  119,  143,  144,   72,   73,
        102,  105,  106,  142,  145,   73,   89,  105,  106,  126,  145,
        146,   74,   75,  107,  108,  109,  147,  148,   74,   76,  107,
        108,  110,  148,   75,   92,  107,  109,  129,  147,  149,   76,
         77,  108,  110,  111,  148,  150,   77,   78,  110,  111,  112,
        150,  151,   78,   94,  111,  112,  132,  151,  152,   79,   80,
        113,  114,  115,  153,  154,   79,   95,  113,  114,  133,  153,
        155,   80,   81,  113,  115,  116,  154,  156,   81,   82,  115,
        116,  117,  156,  157,   82,   84,  116,  117,  120,  157,  158,
         65,   96,   97,  118,  134,  135,  159,   83,   98,  104,  119,
        137,  144,  160,   84,   99,  117,  120,  138,  158,  161,   85,
         87,  100,  121,  124,  139,  162,   86,   88,  122,  123,  125,
        163,  164,   86,  100,  122,  123,  139,  163,  165,   87,  101,
        121,  124,  140,  162,  166,   88,   89,  122,  125,  126,  164,
        167,   89,  106,  125,  126,  146,  167,  168,   90,   91,  127,
        128,  130,  169,  170,   91,   92,  127,  128,  129,  170,  171,
         92,  109,  128,  129,  149,  171,  172,   90,   93,  127,  130,
        131,  169,  173,   93,  101,  130,  131,  140,  173,  174,   94,
         95,  112,  132,  133,  152,  175,  176,   95,  114,  132,  133,
        155,  176,  177,   96,   99,  118,  134,  138,  159,  178,   97,
        118,  135,  136,  159,  179,  180,   97,   98,  135,  136,  137,
        179,  181,   98,  119,  136,  137,  160,  181,  182,   99,  120,
        134,  138,  161,  178,  183,  100,  121,  123,  139,  162,  165,
        184,  101,  124,  131,  140,  166,  174,  185,  102,  103,  141,
        142,  143,  186,  187,  102,  105,  141,  142,  145,  187,  188,
        103,  104,  141,  143,  144,  186,  189,  104,  119,  143,  144,
        160,  189,  190,  105,  106,  142,  145,  146,  188,  191,  106,
        126,  145,  146,  168,  191,  192,  107,  109,  147,  148,  149,
        193,  107,  108,  110,  147,  148,  150,  193,  194,  109,  129,
        147,  149,  172,  193,  195,  110,  111,  148,  150,  151,  194,
        196,  111,  112,  150,  151,  152,  196,  197,  112,  132,  151,
        152,  175,  197,  198,  113,  114,  153,  154,  155,  199,  200,
        113,  115,  153,  154,  156,  200,  201,  114,  133,  153,  155,
        177,  199,  202,  115,  116,  154,  156,  157,  201,  203,  116,
        117,  156,  157,  158,  203,  204,  117,  120,  157,  158,  161,
        204,  205,  118,  134,  135,  159,  178,  180,  206,  119,  137,
        144,  160,  182,  190,  207,  120,  138,  158,  161,  183,  205,
        208,  121,  124,  139,  162,  166,  184,  209,  122,  123,  163,
        164,  165,  210,  211,  122,  125,  163,  164,  167,  210,  212,
        123,  139,  163,  165,  184,  211,  213,  124,  140,  162,  166,
        185,  209,  214,  125,  126,  164,  167,  168,  212,  215,  126,
        146,  167,  168,  192,  215,  216,  127,  130,  169,  170,  173,
        217,  218,  127,  128,  169,  170,  171,  218,  219,  128,  129,
        170,  171,  172,  219,  220,  129,  149,  171,  172,  195,  220,
        221,  130,  131,  169,  173,  174,  217,  222,  131,  140,  173,
        174,  185,  222,  223,  132,  152,  175,  176,  198,  224,  225,
        132,  133,  175,  176,  177,  224,  226,  133,  155,  176,  177,
        202,  226,  227,  134,  138,  159,  178,  183,  206,  228,  135,
        136,  179,  180,  181,  229,  230,  135,  159,  179,  180,  206,
        229,  231,  136,  137,  179,  181,  182,  230,  232,  137,  160,
        181,  182,  207,  232,  233,  138,  161,  178,  183,  208,  228,
        234,  139,  162,  165,  184,  209,  213,  235,  140,  166,  174,
        185,  214,  223,  236,  141,  143,  186,  187,  189,  237,  238,
        141,  142,  186,  187,  188,  238,  239,  142,  145,  187,  188,
        191,  239,  240,  143,  144,  186,  189,  190,  237,  241,  144,
        160,  189,  190,  207,  241,  242,  145,  146,  188,  191,  192,
        240,  243,  146,  168,  191,  192,  216,  243,  244,  147,  148,
        149,  193,  194,  195,  245,  148,  150,  193,  194,  196,  245,
        246,  149,  172,  193,  195,  221,  245,  247,  150,  151,  194,
        196,  197,  246,  248,  151,  152,  196,  197,  198,  248,  249,
        152,  175,  197,  198,  225,  249,  250,  153,  155,  199,  200,
        202,  251,  252,  153,  154,  199,  200,  201,  251,  253,  154,
        156,  200,  201,  203,  253,  254,  155,  177,  199,  202,  227,
        252,  255,  156,  157,  201,  203,  204,  254,  256,  157,  158,
        203,  204,  205,  256,  257,  158,  161,  204,  205,  208,  257,
        258,  159,  178,  180,  206,  228,  231,  259,  160,  182,  190,
        207,  233,  242,  260,  161,  183,  205,  208,  234,  258,  261,
        162,  166,  184,  209,  214,  235,  262,  163,  164,  210,  211,
        212,  263,  264,  163,  165,  210,  211,  213,  264,  265,  164,
        167,  210,  212,  215,  263,  266,  165,  184,  211,  213,  235,
        265,  267,  166,  185,  209,  214,  236,  262,  268,  167,  168,
        212,  215,  216,  266,  269,  168,  192,  215,  216,  244,  269,
        270,  169,  173,  217,  218,  222,  271,  272,  169,  170,  217,
        218,  219,  272,  273,  170,  171,  218,  219,  220,  273,  274,
        171,  172,  219,  220,  221,  274,  275,  172,  195,  220,  221,
        247,  275,  276,  173,  174,  217,  222,  223,  271,  277,  174,
        185,  222,  223,  236,  277,  278,  175,  176,  224,  225,  226,
        279,  280,  175,  198,  224,  225,  250,  279,  281,  176,  177,
        224,  226,  227,  280,  282,  177,  202,  226,  227,  255,  282,
        283,  178,  183,  206,  228,  234,  259,  284,  179,  180,  229,
        230,  231,  285,  286,  179,  181,  229,  230,  232,  286,  287,
        180,  206,  229,  231,  259,  285,  288,  181,  182,  230,  232,
        233,  287,  289,  182,  207,  232,  233,  260,  289,  290,  183,
        208,  228,  234,  261,  284,  291,  184,  209,  213,  235,  262,
        267,  292,  185,  214,  223,  236,  268,  278,  293,  186,  189,
        237,  238,  241,  294,  295,  186,  187,  237,  238,  239,  295,
        296,  187,  188,  238,  239,  240,  296,  297,  188,  191,  239,
        240,  243,  297,  298,  189,  190,  237,  241,  242,  294,  299,
        190,  207,  241,  242,  260,  299,  300,  191,  192,  240,  243,
        244,  298,  301,  329,  192,  216,  243,  244,  270,  301,  302,
        193,  194,  195,  245,  246,  247,  303,  194,  196,  245,  246,
        248,  303,  195,  221,  245,  247,  276,  303,  304,  196,  197,
        246,  248,  249,  303,  305,  197,  198,  248,  249,  250,  305,
        306,  198,  225,  249,  250,  281,  306,  307,  199,  200,  251,
        252,  253,  308,  309,  199,  202,  251,  252,  255,  308,  310,
        200,  201,  251,  253,  254,  309,  311,  201,  203,  253,  254,
        256,  311,  312,  202,  227,  252,  255,  283,  310,  313,  203,
        204,  254,  256,  257,  312,  314,  204,  205,  256,  257,  258,
        314,  315,  205,  208,  257,  258,  261,  315,  316,  206,  228,
        231,  259,  284,  288,  317,  207,  233,  242,  260,  290,  300,
        318,  208,  234,  258,  261,  291,  316,  319,  209,  214,  235,
        262,  268,  292,  320,  210,  212,  263,  264,  266,  321,  322,
        210,  211,  263,  264,  265,  321,  323,  211,  213,  264,  265,
        267,  323,  324,  212,  215,  263,  266,  269,  322,  325,  213,
        235,  265,  267,  292,  324,  326,  214,  236,  262,  268,  293,
        320,  327,  215,  216,  266,  269,  270,  325,  328,  216,  244,
        269,  270,  302,  328,  330,  217,  222,  271,  272,  277,  331,
        332,  217,  218,  271,  272,  273,  331,  333,  218,  219,  272,
        273,  274,  333,  334,  219,  220,  273,  274,  275,  334,  335,
        220,  221,  274,  275,  276,  335,  336,  221,  247,  275,  276,
        304,  336,  337,  222,  223,  271,  277,  278,  332,  338,  223,
        236,  277,  278,  293,  338,  339,  224,  225,  279,  280,  281,
        340,  341,  224,  226,  279,  280,  282,  340,  342,  225,  250,
        279,  281,  307,  341,  343,  226,  227,  280,  282,  283,  342,
        344,  227,  255,  282,  283,  313,  344,  345,  228,  234,  259,
        284,  291,  317,  346,  229,  231,  285,  286,  288,  347,  348,
        229,  230,  285,  286,  287,  347,  349,  230,  232,  286,  287,
        289,  349,  350,  231,  259,  285,  288,  317,  348,  351,  232,
        233,  287,  289,  290,  350,  352,  233,  260,  289,  290,  318,
        352,  353,  234,  261,  284,  291,  319,  346,  354,  235,  262,
        267,  292,  320,  326,  355,  236,  268,  278,  293,  327,  339,
        356,  237,  241,  294,  295,  299,  357,  358,  237,  238,  294,
        295,  296,  358,  359,  238,  239,  295,  296,  297,  359,  360,
        239,  240,  296,  297,  298,  360,  240,  243,  297,  298,  329,
        360,  361,  241,  242,  294,  299,  300,  357,  362,  242,  260,
        299,  300,  318,  362,  363,  243,  244,  301,  302,  329,  364,
        244,  270,  301,  302,  330,  364,  365,  245,  246,  247,  248,
        303,  304,  305,  247,  276,  303,  304,  305,  337,  366,  248,
        249,  303,  304,  305,  306,  366,  249,  250,  305,  306,  307,
        366,  367,  250,  281,  306,  307,  343,  367,  368,  251,  252,
        308,  309,  310,  369,  370,  251,  253,  308,  309,  311,  370,
        371,  252,  255,  308,  310,  313,  369,  372,  253,  254,  309,
        311,  312,  371,  373,  254,  256,  311,  312,  314,  373,  374,
        255,  283,  310,  313,  345,  372,  375,  256,  257,  312,  314,
        315,  374,  376,  257,  258,  314,  315,  316,  376,  377,  258,
        261,  315,  316,  319,  377,  378,  259,  284,  288,  317,  346,
        351,  379,  260,  290,  300,  318,  353,  363,  380,  261,  291,
        316,  319,  354,  378,  381,  262,  268,  292,  320,  327,  355,
        382,  263,  264,  321,  322,  323,  383,  384,  263,  266,  321,
        322,  325,  384,  385,  264,  265,  321,  323,  324,  383,  386,
        265,  267,  323,  324,  326,  386,  387,  266,  269,  322,  325,
        328,  385,  388,  420,  267,  292,  324,  326,  355,  387,  389,
        268,  293,  320,  327,  356,  382,  390,  269,  270,  325,  328,
        330,  388,  391,  243,  298,  301,  329,  361,  364,  392,  270,
        302,  328,  330,  365,  391,  393,  271,  272,  331,  332,  333,
        394,  395,  271,  277,  331,  332,  338,  395,  396,  272,  273,
        331,  333,  334,  394,  273,  274,  333,  334,  335,  394,  397,
        274,  275,  334,  335,  336,  397,  398,  275,  276,  335,  336,
        337,  398,  399,  276,  304,  336,  337,  366,  399,  400,  277,
        278,  332,  338,  339,  396,  401,  278,  293,  338,  339,  356,
        401,  402,  279,  280,  340,  341,  342,  403,  404,  279,  281,
        340,  341,  343,  403,  405,  432,  280,  282,  340,  342,  344,
        404,  406,  281,  307,  341,  343,  368,  405,  407,  282,  283,
        342,  344,  345,  406,  408,  283,  313,  344,  345,  375,  408,
        409,  284,  291,  317,  346,  354,  379,  410,  285,  286,  347,
        348,  349,  411,  285,  288,  347,  348,  351,  411,  412,  286,
        287,  347,  349,  350,  411,  413,  446,  287,  289,  349,  350,
        352,  413,  414,  288,  317,  348,  351,  379,  412,  415,  289,
        290,  350,  352,  353,  414,  416,  290,  318,  352,  353,  380,
        416,  417,  291,  319,  346,  354,  381,  410,  418,  292,  320,
        326,  355,  382,  389,  419,  293,  327,  339,  356,  390,  402,
        421,  294,  299,  357,  358,  362,  422,  423,  294,  295,  357,
        358,  359,  422,  424,  295,  296,  358,  359,  360,  424,  425,
        296,  297,  298,  359,  360,  361,  425,  298,  329,  360,  361,
        392,  425,  426,  299,  300,  357,  362,  363,  423,  427,  300,
        318,  362,  363,  380,  427,  428,  301,  302,  329,  364,  365,
        392,  429,  302,  330,  364,  365,  393,  429,  430,  304,  305,
        306,  337,  366,  367,  400,  306,  307,  366,  367,  368,  400,
        431,  307,  343,  367,  368,  407,  431,  433,  308,  310,  369,
        370,  372,  434,  435,  308,  309,  369,  370,  371,  435,  436,
        309,  311,  370,  371,  373,  436,  437,  310,  313,  369,  372,
        375,  434,  438,  311,  312,  371,  373,  374,  437,  439,  312,
        314,  373,  374,  376,  439,  440,  313,  345,  372,  375,  409,
        438,  441,  314,  315,  374,  376,  377,  440,  442,  315,  316,
        376,  377,  378,  442,  443,  316,  319,  377,  378,  381,  443,
        444,  317,  346,  351,  379,  410,  415,  445,  318,  353,  363,
        380,  417,  428,  447,  319,  354,  378,  381,  418,  444,  448,
        320,  327,  355,  382,  390,  419,  449,  321,  323,  383,  384,
        386,  450,  451,  321,  322,  383,  384,  385,  450,  452,  322,
        325,  384,  385,  420,  452,  453,  323,  324,  383,  386,  387,
        451,  454,  324,  326,  386,  387,  389,  454,  455,  325,  328,
        388,  391,  420,  456,  457,  326,  355,  387,  389,  419,  455,
        458,  327,  356,  382,  390,  421,  449,  459,  328,  330,  388,
        391,  393,  456,  460,  329,  361,  364,  392,  426,  429,  461,
        330,  365,  391,  393,  430,  460,  462,  331,  333,  334,  394,
        395,  397,  463,  331,  332,  394,  395,  396,  463,  464,  332,
        338,  395,  396,  401,  464,  465,  334,  335,  394,  397,  398,
        463,  466,  335,  336,  397,  398,  399,  466,  467,  336,  337,
        398,  399,  400,  467,  468,  337,  366,  367,  399,  400,  431,
        468,  338,  339,  396,  401,  402,  465,  469,  339,  356,  401,
        402,  421,  469,  470,  340,  341,  403,  404,  432,  471,  472,
        340,  342,  403,  404,  406,  471,  473,  341,  343,  405,  407,
        432,  474,  475,  342,  344,  404,  406,  408,  473,  476,  343,
        368,  405,  407,  433,  474,  477,  344,  345,  406,  408,  409,
        476,  478,  345,  375,  408,  409,  441,  478,  479,  346,  354,
        379,  410,  418,  445,  480,  347,  348,  349,  411,  412,  446,
        481,  348,  351,  411,  412,  415,  481,  482,  349,  350,  413,
        414,  446,  483,  484,  350,  352,  413,  414,  416,  483,  485,
        351,  379,  412,  415,  445,  482,  486,  352,  353,  414,  416,
        417,  485,  487,  353,  380,  416,  417,  447,  487,  488,  354,
        381,  410,  418,  448,  480,  489,  355,  382,  389,  419,  449,
        458,  490,  325,  385,  388,  420,  453,  457,  356,  390,  402,
        421,  459,  470,  357,  358,  422,  423,  424,  491,  492,  357,
        362,  422,  423,  427,  491,  493,  358,  359,  422,  424,  425,
        492,  494,  359,  360,  361,  424,  425,  426,  494,  361,  392,
        425,  426,  461,  494,  495,  362,  363,  423,  427,  428,  493,
        496,  363,  380,  427,  428,  447,  496,  497,  364,  365,  392,
        429,  430,  461,  498,  365,  393,  429,  430,  462,  498,  499,
        367,  368,  400,  431,  433,  468,  500,  341,  403,  405,  432,
        472,  475,  368,  407,  431,  433,  477,  500,  501,  369,  372,
        434,  435,  438,  502,  503,  369,  370,  434,  435,  436,  502,
        504,  370,  371,  435,  436,  437,  504,  505,  371,  373,  436,
        437,  439,  505,  506,  372,  375,  434,  438,  441,  503,  507,
        373,  374,  437,  439,  440,  506,  508,  374,  376,  439,  440,
        442,  508,  509,  375,  409,  438,  441,  479,  507,  510,  376,
        377,  440,  442,  443,  509,  511,  377,  378,  442,  443,  444,
        511,  378,  381,  443,  444,  448,  511,  512,  546,  379,  410,
        415,  445,  480,  486,  349,  411,  413,  446,  481,  484,  513,
        380,  417,  428,  447,  488,  497,  514,  381,  418,  444,  448,
        489,  512,  515,  382,  390,  419,  449,  459,  490,  516,  383,
        384,  450,  451,  452,  517,  518,  383,  386,  450,  451,  454,
        518,  519,  384,  385,  450,  452,  453,  517,  520,  385,  420,
        452,  453,  457,  520,  521,  386,  387,  451,  454,  455,  519,
        522,  387,  389,  454,  455,  458,  522,  523,  388,  391,  456,
        457,  460,  524,  525,  388,  420,  453,  456,  457,  521,  525,
        389,  419,  455,  458,  490,  523,  526,  390,  421,  449,  459,
        470,  516,  527,  391,  393,  456,  460,  462,  524,  528,  392,
        426,  429,  461,  495,  498,  529,  393,  430,  460,  462,  499,
        528,  530,  394,  395,  397,  463,  464,  466,  531,  395,  396,
        463,  464,  465,  531,  532,  396,  401,  464,  465,  469,  532,
        533,  397,  398,  463,  466,  467,  531,  534,  398,  399,  466,
        467,  468,  534,  399,  400,  431,  467,  468,  500,  534,  568,
        401,  402,  465,  469,  470,  533,  535,  402,  421,  459,  469,
        470,  527,  535,  558,  403,  404,  471,  472,  473,  536,  537,
        403,  432,  471,  472,  475,  536,  538,  404,  406,  471,  473,
        476,  537,  539,  405,  407,  474,  475,  477,  540,  541,  405,
        432,  472,  474,  475,  538,  541,  406,  408,  473,  476,  478,
        539,  542,  407,  433,  474,  477,  501,  540,  543,  408,  409,
        476,  478,  479,  542,  544,  409,  441,  478,  479,  510,  544,
        545,  410,  418,  445,  480,  486,  489,  547,  411,  412,  446,
        481,  482,  513,  548,  412,  415,  481,  482,  486,  548,  549,
        413,  414,  483,  484,  485,  550,  551,  413,  446,  483,  484,
        513,  551,  552,  414,  416,  483,  485,  487,  550,  553,  415,
        445,  480,  482,  486,  547,  549,  582,  416,  417,  485,  487,
        488,  553,  554,  417,  447,  487,  488,  514,  554,  555,  418,
        448,  480,  489,  515,  547,  556,  419,  449,  458,  490,  516,
        526,  557,  422,  423,  491,  492,  493,  559,  560,  422,  424,
        491,  492,  494,  560,  561,  423,  427,  491,  493,  496,  559,
        562,  424,  425,  426,  492,  494,  495,  561,  426,  461,  494,
        495,  529,  561,  563,  427,  428,  493,  496,  497,  562,  564,
        428,  447,  496,  497,  514,  564,  565,  429,  430,  461,  498,
        499,  529,  566,  430,  462,  498,  499,  530,  566,  567,  431,
        433,  468,  500,  501,  568,  433,  477,  500,  501,  543,  568,
        569,  607,  434,  435,  502,  503,  504,  570,  571,  434,  438,
        502,  503,  507,  570,  572,  435,  436,  502,  504,  505,  571,
        573,  436,  437,  504,  505,  506,  573,  574,  437,  439,  505,
        506,  508,  574,  575,  438,  441,  503,  507,  510,  572,  576,
        439,  440,  506,  508,  509,  575,  577,  440,  442,  508,  509,
        511,  577,  578,  441,  479,  507,  510,  545,  576,  579,  442,
        443,  444,  509,  511,  546,  578,  444,  448,  512,  515,  546,
        580,  581,  446,  481,  484,  513,  548,  552,  583,  447,  488,
        497,  514,  555,  565,  584,  448,  489,  512,  515,  556,  581,
        585,  449,  459,  490,  516,  527,  557,  586,  587,  450,  452,
        517,  518,  520,  588,  589,  450,  451,  517,  518,  519,  589,
        590,  451,  454,  518,  519,  522,  590,  591,  452,  453,  517,
        520,  521,  588,  592,  453,  457,  520,  521,  525,  592,  593,
        454,  455,  519,  522,  523,  591,  594,  455,  458,  522,  523,
        526,  594,  595,  456,  460,  524,  525,  528,  596,  597,  456,
        457,  521,  524,  525,  593,  597,  458,  490,  523,  526,  557,
        595,  598,  459,  470,  516,  527,  558,  586,  599,  460,  462,
        524,  528,  530,  596,  600,  461,  495,  498,  529,  563,  566,
        601,  462,  499,  528,  530,  567,  600,  602,  463,  464,  466,
        531,  532,  534,  603,  464,  465,  531,  532,  533,  603,  604,
        465,  469,  532,  533,  535,  604,  605,  466,  467,  468,  531,
        534,  568,  603,  469,  470,  533,  535,  558,  605,  606,  471,
        472,  536,  537,  538,  608,  609,  471,  473,  536,  537,  539,
        609,  610,  472,  475,  536,  538,  541,  608,  611,  473,  476,
        537,  539,  542,  610,  612,  474,  477,  540,  541,  543,  613,
        614,  474,  475,  538,  540,  541,  611,  614,  476,  478,  539,
        542,  544,  612,  615,  477,  501,  540,  543,  569,  613,  616,
        478,  479,  542,  544,  545,  615,  617,  479,  510,  544,  545,
        579,  617,  618,  444,  511,  512,  546,  578,  580,  619,  480,
        486,  489,  547,  556,  582,  620,  481,  482,  513,  548,  549,
        583,  621,  482,  486,  548,  549,  582,  621,  622,  483,  485,
        550,  551,  553,  623,  624,  483,  484,  550,  551,  552,  623,
        625,  484,  513,  551,  552,  583,  625,  626,  485,  487,  550,
        553,  554,  624,  627,  487,  488,  553,  554,  555,  627,  488,
        514,  554,  555,  584,  627,  628,  489,  515,  547,  556,  585,
        620,  629,  490,  516,  526,  557,  587,  598,  630,  470,  527,
        535,  558,  599,  606,  631,  491,  493,  559,  560,  562,  632,
        633,  491,  492,  559,  560,  561,  632,  634,  492,  494,  495,
        560,  561,  563,  634,  493,  496,  559,  562,  564,  633,  635,
        495,  529,  561,  563,  601,  634,  636,  496,  497,  562,  564,
        565,  635,  637,  497,  514,  564,  565,  584,  637,  638,  498,
        499,  529,  566,  567,  601,  639,  499,  530,  566,  567,  602,
        639,  640,  468,  500,  501,  534,  568,  603,  607,  501,  543,
        569,  607,  616,  641,  642,  502,  503,  570,  571,  572,  643,
        644,  502,  504,  570,  571,  573,  644,  645,  503,  507,  570,
        572,  576,  643,  646,  504,  505,  571,  573,  574,  645,  647,
        505,  506,  573,  574,  575,  647,  648,  506,  508,  574,  575,
        577,  648,  649,  507,  510,  572,  576,  579,  646,  650,  508,
        509,  575,  577,  578,  649,  651,  509,  511,  546,  577,  578,
        619,  651,  510,  545,  576,  579,  618,  650,  652,  512,  546,
        580,  581,  619,  653,  654,  512,  515,  580,  581,  585,  653,
        655,  486,  547,  549,  582,  620,  622,  656,  513,  548,  552,
        583,  621,  626,  657,  514,  555,  565,  584,  628,  638,  658,
        515,  556,  581,  585,  629,  655,  659,  516,  527,  586,  587,
        599,  660,  661,  516,  557,  586,  587,  630,  661,  517,  520,
        588,  589,  592,  662,  663,  517,  518,  588,  589,  590,  662,
        664,  518,  519,  589,  590,  591,  664,  665,  519,  522,  590,
        591,  594,  665,  666,  520,  521,  588,  592,  593,  663,  667,
        521,  525,  592,  593,  597,  667,  668,  522,  523,  591,  594,
        595,  666,  669,  523,  526,  594,  595,  598,  669,  670,  524,
        528,  596,  597,  600,  671,  672,  524,  525,  593,  596,  597,
        668,  672,  526,  557,  595,  598,  630,  670,  673,  705,  527,
        558,  586,  599,  631,  660,  674,  528,  530,  596,  600,  602,
        671,  675,  529,  563,  566,  601,  636,  639,  676,  530,  567,
        600,  602,  640,  675,  677,  707,  531,  532,  534,  568,  603,
        604,  607,  532,  533,  603,  604,  605,  607,  641,  533,  535,
        604,  605,  606,  641,  678,  535,  558,  605,  606,  631,  678,
        679,  501,  568,  569,  603,  604,  607,  641,  536,  538,  608,
        609,  611,  680,  681,  536,  537,  608,  609,  610,  681,  682,
        537,  539,  609,  610,  612,  682,  683,  538,  541,  608,  611,
        614,  680,  684,  539,  542,  610,  612,  615,  683,  685,  540,
        543,  613,  614,  616,  686,  687,  540,  541,  611,  613,  614,
        684,  686,  542,  544,  612,  615,  617,  685,  688,  543,  569,
        613,  616,  642,  687,  689,  544,  545,  615,  617,  618,  688,
        690,  545,  579,  617,  618,  652,  690,  691,  546,  578,  580,
        619,  651,  654,  692,  547,  556,  582,  620,  629,  656,  693,
        548,  549,  583,  621,  622,  657,  694,  549,  582,  621,  622,
        656,  694,  695,  550,  551,  623,  624,  625,  696,  697,  550,
        553,  623,  624,  627,  697,  698,  551,  552,  623,  625,  626,
        696,  699,  552,  583,  625,  626,  657,  699,  700,  553,  554,
        555,  624,  627,  628,  698,  701,  555,  584,  627,  628,  658,
        701,  702,  556,  585,  620,  629,  659,  693,  703,  557,  587,
        598,  630,  661,  673,  704,  558,  599,  606,  631,  674,  679,
        706,  559,  560,  632,  633,  634,  708,  709,  559,  562,  632,
        633,  635,  708,  710,  560,  561,  563,  632,  634,  636,  709,
        562,  564,  633,  635,  637,  710,  711,  563,  601,  634,  636,
        676,  709,  712,  564,  565,  635,  637,  638,  711,  713,  754,
        565,  584,  637,  638,  658,  713,  714,  566,  567,  601,  639,
        640,  676,  715,  567,  602,  639,  640,  677,  715,  569,  604,
        605,  607,  641,  642,  678,  569,  616,  641,  642,  678,  689,
        716,  570,  572,  643,  644,  646,  717,  718,  570,  571,  643,
        644,  645,  718,  719,  571,  573,  644,  645,  647,  719,  720,
        572,  576,  643,  646,  650,  717,  721,  573,  574,  645,  647,
        648,  720,  722,  574,  575,  647,  648,  649,  722,  723,  575,
        577,  648,  649,  651,  723,  724,  576,  579,  646,  650,  652,
        721,  725,  577,  578,  619,  649,  651,  692,  724,  579,  618,
        650,  652,  691,  725,  726,  580,  581,  653,  654,  655,  727,
        728,  580,  619,  653,  654,  692,  727,  729,  581,  585,  653,
        655,  659,  728,  730,  582,  620,  622,  656,  693,  695,  731,
        583,  621,  626,  657,  694,  700,  732,  584,  628,  638,  658,
        702,  714,  733,  585,  629,  655,  659,  703,  730,  734,  586,
        599,  660,  661,  674,  735,  736,  586,  587,  630,  660,  661,
        704,  735,  737,  588,  589,  662,  663,  664,  738,  739,  588,
        592,  662,  663,  667,  738,  740,  589,  590,  662,  664,  665,
        739,  741,  590,  591,  664,  665,  666,  741,  742,  591,  594,
        665,  666,  669,  742,  743,  592,  593,  663,  667,  668,  740,
        744,  593,  597,  667,  668,  672,  744,  745,  594,  595,  666,
        669,  670,  743,  746,  595,  598,  669,  670,  705,  746,  747,
        596,  600,  671,  672,  675,  748,  749,  596,  597,  668,  671,
        672,  745,  748,  598,  630,  673,  704,  705,  750,  751,  599,
        631,  660,  674,  706,  736,  752,  600,  602,  671,  675,  707,
        749,  753,  601,  636,  639,  676,  712,  715,  755,  602,  640,
        677,  707,  715,  756,  757,  605,  606,  641,  642,  678,  679,
        716,  758,  606,  631,  678,  679,  706,  758,  608,  611,  680,
        681,  684,  759,  760,  608,  609,  680,  681,  682,  759,  761,
        609,  610,  681,  682,  683,  761,  762,  610,  612,  682,  683,
        685,  762,  763,  611,  614,  680,  684,  686,  760,  764,  612,
        615,  683,  685,  688,  763,  765,  613,  614,  684,  686,  687,
        764,  766,  613,  616,  686,  687,  689,  766,  767,  615,  617,
        685,  688,  690,  765,  768,  616,  642,  687,  689,  716,  767,
        769,  617,  618,  688,  690,  691,  768,  770,  618,  652,  690,
        691,  726,  770,  771,  619,  651,  654,  692,  724,  729,  772,
        620,  629,  656,  693,  703,  731,  773,  621,  622,  657,  694,
        695,  732,  774,  622,  656,  694,  695,  731,  774,  775,  623,
        625,  696,  697,  699,  776,  623,  624,  696,  697,  698,  776,
        777,  624,  627,  697,  698,  701,  777,  778,  625,  626,  696,
        699,  700,  776,  779,  626,  657,  699,  700,  732,  779,  780,
        627,  628,  698,  701,  702,  778,  781,  628,  658,  701,  702,
        733,  781,  782,  629,  659,  693,  703,  734,  773,  783,  630,
        661,  673,  704,  737,  750,  598,  670,  673,  705,  747,  751,
        631,  674,  679,  706,  752,  758,  784,  602,  675,  677,  707,
        753,  757,  785,  632,  633,  708,  709,  710,  786,  787,  632,
        634,  636,  708,  709,  712,  786,  633,  635,  708,  710,  711,
        787,  788,  635,  637,  710,  711,  754,  788,  789,  636,  676,
        709,  712,  755,  786,  790,  637,  638,  713,  714,  754,  791,
        792,  638,  658,  713,  714,  733,  791,  793,  639,  640,  676,
        677,  715,  755,  756,  642,  678,  689,  716,  758,  769,  794,
        643,  646,  717,  718,  721,  795,  796,  643,  644,  717,  718,
        719,  795,  797,  644,  645,  718,  719,  720,  797,  798,  645,
        647,  719,  720,  722,  798,  799,  646,  650,  717,  721,  725,
        796,  800,  647,  648,  720,  722,  723,  799,  801,  648,  649,
        722,  723,  724,  801,  802,  649,  651,  692,  723,  724,  772,
        802,  650,  652,  721,  725,  726,  800,  803,  652,  691,  725,
        726,  771,  803,  804,  653,  654,  727,  728,  729,  805,  806,
        653,  655,  727,  728,  730,  806,  807,  654,  692,  727,  729,
        772,  805,  808,  655,  659,  728,  730,  734,  807,  809,  656,
        693,  695,  731,  773,  775,  657,  694,  700,  732,  774,  780,
        810,  658,  702,  714,  733,  782,  793,  811,  659,  703,  730,
        734,  783,  809,  812,  660,  661,  735,  736,  737,  813,  814,
        660,  674,  735,  736,  752,  814,  815,  661,  704,  735,  737,
        750,  813,  816,  662,  663,  738,  739,  740,  817,  818,  662,
        664,  738,  739,  741,  817,  819,  663,  667,  738,  740,  744,
        818,  820,  664,  665,  739,  741,  742,  819,  821,  665,  666,
        741,  742,  743,  821,  822,  862,  666,  669,  742,  743,  746,
        822,  823,  667,  668,  740,  744,  745,  820,  824,  668,  672,
        744,  745,  748,  824,  825,  669,  670,  743,  746,  747,  823,
        826,  670,  705,  746,  747,  751,  826,  827,  671,  672,  745,
        748,  749,  825,  828,  671,  675,  748,  749,  753,  828,  829,
        673,  704,  737,  750,  751,  816,  830,  673,  705,  747,  750,
        751,  827,  830,  674,  706,  736,  752,  784,  815,  831,  675,
        707,  749,  753,  785,  829,  832,  637,  711,  713,  754,  789,
        792,  833,  676,  712,  715,  755,  756,  790,  834,  677,  715,
        755,  756,  757,  834,  835,  677,  707,  756,  757,  785,  835,
        836,  678,  679,  706,  716,  758,  784,  794,  680,  681,  759,
        760,  761,  837,  838,  680,  684,  759,  760,  764,  838,  839,
        681,  682,  759,  761,  762,  837,  840,  682,  683,  761,  762,
        763,  840,  841,  683,  685,  762,  763,  765,  841,  842,  684,
        686,  760,  764,  766,  839,  843,  685,  688,  763,  765,  768,
        842,  844,  686,  687,  764,  766,  767,  843,  845,  687,  689,
        766,  767,  769,  845,  846,  688,  690,  765,  768,  770,  844,
        847,  689,  716,  767,  769,  794,  846,  848,  690,  691,  768,
        770,  771,  847,  849,  691,  726,  770,  771,  804,  849,  850,
        692,  724,  729,  772,  802,  808,  851,  693,  703,  731,  773,
        775,  783,  852,  694,  695,  732,  774,  775,  810,  853,  695,
        731,  773,  774,  775,  852,  853,  890,  696,  697,  699,  776,
        777,  779,  854,  855,  697,  698,  776,  777,  778,  855,  856,
        698,  701,  777,  778,  781,  856,  857,  699,  700,  776,  779,
        780,  854,  858,  892,  700,  732,  779,  780,  810,  858,  701,
        702,  778,  781,  782,  857,  859,  702,  733,  781,  782,  811,
        859,  860,  703,  734,  773,  783,  812,  852,  861,  706,  752,
        758,  784,  794,  831,  863,  707,  753,  757,  785,  832,  836,
        864,  708,  709,  712,  786,  787,  790,  865,  708,  710,  786,
        787,  788,  865,  866,  710,  711,  787,  788,  789,  866,  867,
        711,  754,  788,  789,  833,  867,  868,  712,  755,  786,  790,
        834,  865,  869,  713,  714,  791,  792,  793,  870,  871,  713,
        754,  791,  792,  833,  870,  872,  714,  733,  791,  793,  811,
        871,  873,  716,  758,  769,  784,  794,  848,  863,  717,  718,
        795,  796,  797,  874,  875,  717,  721,  795,  796,  800,  875,
        876,  718,  719,  795,  797,  798,  874,  877,  719,  720,  797,
        798,  799,  877,  878,  720,  722,  798,  799,  801,  878,  879,
        721,  725,  796,  800,  803,  876,  880,  722,  723,  799,  801,
        802,  879,  881,  723,  724,  772,  801,  802,  851,  881,  725,
        726,  800,  803,  804,  880,  882,  726,  771,  803,  804,  850,
        882,  883,  727,  729,  805,  806,  808,  884,  885,  727,  728,
        805,  806,  807,  885,  886,  728,  730,  806,  807,  809,  886,
        887,  729,  772,  805,  808,  851,  884,  888,  730,  734,  807,
        809,  812,  887,  889,  732,  774,  780,  810,  853,  858,  891,
        733,  782,  793,  811,  860,  873,  893,  734,  783,  809,  812,
        861,  889,  894,  735,  737,  813,  814,  816,  895,  896,  735,
        736,  813,  814,  815,  895,  897,  736,  752,  814,  815,  831,
        897,  898,  737,  750,  813,  816,  830,  896,  899,  738,  739,
        817,  818,  819,  900,  901,  738,  740,  817,  818,  820,  900,
        902,  739,  741,  817,  819,  821,  901,  903,  740,  744,  818,
        820,  824,  902,  904,  741,  742,  819,  821,  862,  903,  905,
        742,  743,  822,  823,  862,  906,  907,  743,  746,  822,  823,
        826,  907,  908,  744,  745,  820,  824,  825,  904,  909,  745,
        748,  824,  825,  828,  909,  910,  746,  747,  823,  826,  827,
        908,  911,  747,  751,  826,  827,  830,  911,  912,  748,  749,
        825,  828,  829,  910,  913,  749,  753,  828,  829,  832,  913,
        914,  750,  751,  816,  827,  830,  899,  912,  752,  784,  815,
        831,  863,  898,  915,  753,  785,  829,  832,  864,  914,  916,
        754,  789,  792,  833,  868,  872,  917,  755,  756,  790,  834,
        835,  869,  918,  756,  757,  834,  835,  836,  918,  919,  757,
        785,  835,  836,  864,  919,  920,  759,  761,  837,  838,  840,
        921,  922,  759,  760,  837,  838,  839,  921,  923,  760,  764,
        838,  839,  843,  923,  924,  761,  762,  837,  840,  841,  922,
        925,  762,  763,  840,  841,  842,  925,  926,  763,  765,  841,
        842,  844,  926,  927,  764,  766,  839,  843,  845,  924,  928,
        765,  768,  842,  844,  847,  927,  929,  766,  767,  843,  845,
        846,  928,  930,  767,  769,  845,  846,  848,  930,  931,  768,
        770,  844,  847,  849,  929,  932,  769,  794,  846,  848,  863,
        931,  933,  770,  771,  847,  849,  850,  932,  934,  771,  804,
        849,  850,  883,  934,  935,  772,  802,  808,  851,  881,  888,
        936,  773,  775,  783,  852,  861,  890,  937,  774,  775,  810,
        853,  890,  891,  938,  776,  779,  854,  855,  892,  939,  940,
        776,  777,  854,  855,  856,  940,  777,  778,  855,  856,  857,
        940,  941,  778,  781,  856,  857,  859,  941,  942,  779,  780,
        810,  858,  891,  892,  943,  781,  782,  857,  859,  860,  942,
        944,  782,  811,  859,  860,  893,  944,  945,  783,  812,  852,
        861,  894,  937,  946,  742,  821,  822,  862,  905,  906,  784,
        794,  831,  848,  863,  915,  933,  785,  832,  836,  864,  916,
        920,  947,  786,  787,  790,  865,  866,  869,  948,  787,  788,
        865,  866,  867,  948,  949,  788,  789,  866,  867,  868,  949,
        950,  789,  833,  867,  868,  917,  950,  951,  790,  834,  865,
        869,  918,  948,  952,  791,  792,  870,  871,  872,  953,  954,
        791,  793,  870,  871,  873,  953,  955,  792,  833,  870,  872,
        917,  954,  956,  793,  811,  871,  873,  893,  955,  957,  795,
        797,  874,  875,  877,  958,  959,  795,  796,  874,  875,  876,
        959,  960,  796,  800,  875,  876,  880,  960,  797,  798,  874,
        877,  878,  958,  961,  798,  799,  877,  878,  879,  961,  962,
        799,  801,  878,  879,  881,  962,  963,  800,  803,  876,  880,
        882,  960,  964,  801,  802,  851,  879,  881,  936,  963,  803,
        804,  880,  882,  883,  964,  965,  804,  850,  882,  883,  935,
        965,  966,  805,  808,  884,  885,  888,  967,  968,  805,  806,
        884,  885,  886,  967,  969,  806,  807,  885,  886,  887,  969,
        970,  807,  809,  886,  887,  889,  970,  971,  808,  851,  884,
        888,  936,  968,  972,  809,  812,  887,  889,  894,  971,  973,
        775,  852,  853,  890,  937,  938,  974,  810,  853,  858,  891,
        938,  943,  975,  779,  854,  858,  892,  939,  943,  976,  811,
        860,  873,  893,  945,  957,  977,  812,  861,  889,  894,  946,
        973,  978,  813,  814,  895,  896,  897,  979,  980,  813,  816,
        895,  896,  899,  979,  981,  814,  815,  895,  897,  898,  980,
        982,  815,  831,  897,  898,  915,  982,  983,  816,  830,  896,
        899,  912,  981,  984,  817,  818,  900,  901,  902,  985,  986,
        817,  819,  900,  901,  903,  985,  987,  818,  820,  900,  902,
        904,  986,  988,  819,  821,  901,  903,  905,  987,  989,  820,
        824,  902,  904,  909,  988,  990,  821,  862,  903,  905,  906,
        989,  991,  822,  862,  905,  906,  907,  991,  992,  822,  823,
        906,  907,  908,  992,  993,  823,  826,  907,  908,  911,  993,
        994,  824,  825,  904,  909,  910,  990,  995,  825,  828,  909,
        910,  913,  995,  996,  826,  827,  908,  911,  912,  994,  997,
        827,  830,  899,  911,  912,  984,  997,  828,  829,  910,  913,
        914,  996,  998,  829,  832,  913,  914,  916,  998,  999,  831,
        863,  898,  915,  933,  983, 1000,  832,  864,  914,  916,  947,
        999, 1001,  833,  868,  872,  917,  951,  956, 1002,  834,  835,
        869,  918,  919,  952, 1003,  835,  836,  918,  919,  920, 1003,
       1004,  836,  864,  919,  920,  947, 1004, 1005,  837,  838,  921,
        922,  923, 1006, 1007,  837,  840,  921,  922,  925, 1007, 1008,
        838,  839,  921,  923,  924, 1006, 1009,  839,  843,  923,  924,
        928, 1009, 1010,  840,  841,  922,  925,  926, 1008, 1011,  841,
        842,  925,  926,  927, 1011,  842,  844,  926,  927,  929, 1011,
       1012, 1041,  843,  845,  924,  928,  930, 1010, 1013,  844,  847,
        927,  929,  932, 1012, 1014,  845,  846,  928,  930,  931, 1013,
       1015,  846,  848,  930,  931,  933, 1015, 1016,  847,  849,  929,
        932,  934, 1014, 1017,  848,  863,  915,  931,  933, 1000, 1016,
        849,  850,  932,  934,  935, 1017, 1018,  850,  883,  934,  935,
        966, 1018, 1019,  851,  881,  888,  936,  963,  972, 1020,  852,
        861,  890,  937,  946,  974,  853,  890,  891,  938,  974,  975,
       1021,  854,  892,  939,  940,  976, 1022, 1023,  854,  855,  856,
        939,  940,  941, 1022,  856,  857,  940,  941,  942, 1022, 1024,
        857,  859,  941,  942,  944, 1024, 1025,  858,  891,  892,  943,
        975,  976, 1026,  859,  860,  942,  944,  945, 1025, 1027,  860,
        893,  944,  945,  977, 1027, 1028,  861,  894,  937,  946,  974,
        978, 1029,  864,  916,  920,  947, 1001, 1005, 1030,  865,  866,
        869,  948,  949,  952, 1031, 1034,  866,  867,  948,  949,  950,
       1031,  867,  868,  949,  950,  951, 1031, 1032,  868,  917,  950,
        951, 1002, 1032, 1033,  869,  918,  948,  952, 1003, 1034, 1035,
        870,  871,  953,  954,  955, 1036, 1037,  870,  872,  953,  954,
        956, 1036,  871,  873,  953,  955,  957, 1037, 1038,  872,  917,
        954,  956, 1002, 1036, 1039,  873,  893,  955,  957,  977, 1038,
       1040,  874,  877,  958,  959,  961, 1042, 1043,  874,  875,  958,
        959,  960, 1042, 1044,  875,  876,  880,  959,  960,  964, 1044,
       1045,  877,  878,  958,  961,  962, 1043, 1046,  878,  879,  961,
        962,  963, 1046, 1047,  879,  881,  936,  962,  963, 1020, 1047,
        880,  882,  960,  964,  965, 1045, 1048,  882,  883,  964,  965,
        966, 1048, 1049,  883,  935,  965,  966, 1019, 1049, 1050,  884,
        885,  967,  968,  969, 1051, 1052,  884,  888,  967,  968,  972,
       1051, 1053,  885,  886,  967,  969,  970, 1052, 1054,  886,  887,
        969,  970,  971, 1054, 1055,  887,  889,  970,  971,  973, 1055,
       1056,  888,  936,  968,  972, 1020, 1053, 1057,  889,  894,  971,
        973,  978, 1056, 1058,  890,  937,  938,  946,  974, 1021, 1029,
       1060,  891,  938,  943,  975, 1021, 1026, 1059,  892,  939,  943,
        976, 1023, 1026, 1061,  893,  945,  957,  977, 1028, 1040, 1062,
        894,  946,  973,  978, 1029, 1058, 1063,  895,  896,  979,  980,
        981, 1064, 1065,  895,  897,  979,  980,  982, 1065, 1066,  896,
        899,  979,  981,  984, 1064, 1067,  897,  898,  980,  982,  983,
       1066, 1068,  898,  915,  982,  983, 1000, 1068, 1069,  899,  912,
        981,  984,  997, 1067, 1070,  900,  901,  985,  986,  987, 1071,
       1072,  900,  902,  985,  986,  988, 1071, 1073,  901,  903,  985,
        987,  989, 1072, 1074,  902,  904,  986,  988,  990, 1073, 1075,
        903,  905,  987,  989,  991, 1074, 1076,  904,  909,  988,  990,
        995, 1075, 1077,  905,  906,  989,  991,  992, 1076, 1078,  906,
        907,  991,  992,  993, 1078, 1079,  907,  908,  992,  993,  994,
       1079, 1080,  908,  911,  993,  994,  997, 1080, 1081,  909,  910,
        990,  995,  996, 1077, 1082,  910,  913,  995,  996,  998, 1082,
       1083,  911,  912,  984,  994,  997, 1070, 1081,  913,  914,  996,
        998,  999, 1083, 1084,  914,  916,  998,  999, 1001, 1084, 1085,
        915,  933,  983, 1000, 1016, 1069, 1086,  916,  947,  999, 1001,
       1030, 1085, 1087,  917,  951,  956, 1002, 1033, 1039, 1088,  918,
        919,  952, 1003, 1004, 1035,  919,  920, 1003, 1004, 1005, 1035,
       1089, 1117,  920,  947, 1004, 1005, 1030, 1089, 1090,  921,  923,
       1006, 1007, 1009, 1091, 1092,  921,  922, 1006, 1007, 1008, 1092,
       1093,  922,  925, 1007, 1008, 1011, 1093, 1094,  923,  924, 1006,
       1009, 1010, 1091, 1095,  924,  928, 1009, 1010, 1013, 1095, 1096,
       1123,  925,  926,  927, 1008, 1011, 1041, 1094,  927,  929, 1012,
       1014, 1041, 1097, 1098,  928,  930, 1010, 1013, 1015, 1096,  929,
        932, 1012, 1014, 1017, 1097, 1099,  930,  931, 1013, 1015, 1016,
       1096, 1100,  931,  933, 1000, 1015, 1016, 1086, 1100,  932,  934,
       1014, 1017, 1018, 1099, 1101,  934,  935, 1017, 1018, 1019, 1101,
        935,  966, 1018, 1019, 1050, 1101, 1102,  936,  963,  972, 1020,
       1047, 1057, 1103,  938,  974,  975, 1021, 1059, 1060, 1104,  939,
        940,  941, 1022, 1023, 1024, 1105,  939,  976, 1022, 1023, 1061,
       1105, 1106,  941,  942, 1022, 1024, 1025, 1105, 1107,  942,  944,
       1024, 1025, 1027, 1107, 1108,  943,  975,  976, 1026, 1059, 1061,
       1109,  944,  945, 1025, 1027, 1028, 1108, 1110,  945,  977, 1027,
       1028, 1062, 1110, 1111,  946,  974,  978, 1029, 1060, 1063, 1112,
        947, 1001, 1005, 1030, 1087, 1090, 1113,  948,  949,  950, 1031,
       1032, 1034,  950,  951, 1031, 1032, 1033, 1034, 1114, 1116,  951,
       1002, 1032, 1033, 1088, 1114, 1115,  948,  952, 1031, 1032, 1034,
       1035, 1116,  952, 1003, 1004, 1034, 1035, 1116, 1117,  953,  954,
        956, 1036, 1037, 1039, 1118,  953,  955, 1036, 1037, 1038, 1118,
       1119,  955,  957, 1037, 1038, 1040, 1119, 1120,  956, 1002, 1036,
       1039, 1088, 1118, 1121,  957,  977, 1038, 1040, 1062, 1120, 1122,
        927, 1011, 1012, 1041, 1094, 1098, 1124,  958,  959, 1042, 1043,
       1044, 1125, 1126,  958,  961, 1042, 1043, 1046, 1125, 1127,  959,
        960, 1042, 1044, 1045, 1126, 1128,  960,  964, 1044, 1045, 1048,
       1128, 1129,  961,  962, 1043, 1046, 1047, 1127,  962,  963, 1020,
       1046, 1047, 1103, 1127,  964,  965, 1045, 1048, 1049, 1129, 1130,
        965,  966, 1048, 1049, 1050, 1130, 1131,  966, 1019, 1049, 1050,
       1102, 1131, 1132,  967,  968, 1051, 1052, 1053, 1133, 1134,  967,
        969, 1051, 1052, 1054, 1133, 1135,  968,  972, 1051, 1053, 1057,
       1134, 1136,  969,  970, 1052, 1054, 1055, 1135, 1137,  970,  971,
       1054, 1055, 1056, 1137, 1138,  971,  973, 1055, 1056, 1058, 1138,
       1139,  972, 1020, 1053, 1057, 1103, 1136, 1140,  973,  978, 1056,
       1058, 1063, 1139, 1141,  975, 1021, 1026, 1059, 1104, 1109, 1142,
        974, 1021, 1029, 1060, 1104, 1112, 1143,  976, 1023, 1026, 1061,
       1106, 1109, 1144,  977, 1028, 1040, 1062, 1111, 1122, 1145,  978,
       1029, 1058, 1063, 1112, 1141, 1146,  979,  981, 1064, 1065, 1067,
       1147, 1148,  979,  980, 1064, 1065, 1066, 1148, 1149,  980,  982,
       1065, 1066, 1068, 1149, 1150,  981,  984, 1064, 1067, 1070, 1147,
       1151,  982,  983, 1066, 1068, 1069, 1150, 1152,  983, 1000, 1068,
       1069, 1086, 1152, 1153,  984,  997, 1067, 1070, 1081, 1151, 1154,
        985,  986, 1071, 1072, 1073, 1155, 1156,  985,  987, 1071, 1072,
       1074, 1156, 1157,  986,  988, 1071, 1073, 1075, 1155, 1158,  987,
        989, 1072, 1074, 1076, 1157, 1159,  988,  990, 1073, 1075, 1077,
       1158, 1160,  989,  991, 1074, 1076, 1078, 1159, 1161,  990,  995,
       1075, 1077, 1082, 1160, 1162,  991,  992, 1076, 1078, 1079, 1161,
       1163,  992,  993, 1078, 1079, 1080, 1163, 1164,  993,  994, 1079,
       1080, 1081, 1164, 1165,  994,  997, 1070, 1080, 1081, 1154, 1165,
        995,  996, 1077, 1082, 1083, 1162, 1166,  996,  998, 1082, 1083,
       1084, 1166, 1167,  998,  999, 1083, 1084, 1085, 1167, 1168,  999,
       1001, 1084, 1085, 1087, 1168, 1169, 1000, 1016, 1069, 1086, 1100,
       1153, 1170, 1001, 1030, 1085, 1087, 1113, 1169, 1171, 1002, 1033,
       1039, 1088, 1115, 1121, 1172, 1173, 1004, 1005, 1089, 1090, 1117,
       1174, 1175, 1005, 1030, 1089, 1090, 1113, 1175, 1176, 1208, 1006,
       1009, 1091, 1092, 1095, 1177, 1178, 1006, 1007, 1091, 1092, 1093,
       1178, 1179, 1007, 1008, 1092, 1093, 1094, 1179, 1180, 1008, 1011,
       1041, 1093, 1094, 1124, 1180, 1009, 1010, 1091, 1095, 1123, 1177,
       1181, 1010, 1013, 1015, 1096, 1100, 1123, 1182, 1012, 1014, 1097,
       1098, 1099, 1183, 1184, 1012, 1041, 1097, 1098, 1124, 1183, 1185,
       1014, 1017, 1097, 1099, 1101, 1184, 1186, 1015, 1016, 1086, 1096,
       1100, 1170, 1182, 1017, 1018, 1019, 1099, 1101, 1102, 1186, 1187,
       1019, 1050, 1101, 1102, 1132, 1187, 1188, 1020, 1047, 1057, 1103,
       1127, 1140, 1189, 1021, 1059, 1060, 1104, 1142, 1143, 1190, 1022,
       1023, 1024, 1105, 1106, 1107, 1191, 1023, 1061, 1105, 1106, 1144,
       1191, 1192, 1024, 1025, 1105, 1107, 1108, 1191, 1193, 1025, 1027,
       1107, 1108, 1110, 1193, 1194, 1026, 1059, 1061, 1109, 1142, 1144,
       1195, 1027, 1028, 1108, 1110, 1111, 1194, 1196, 1028, 1062, 1110,
       1111, 1145, 1196, 1197, 1029, 1060, 1063, 1112, 1143, 1146, 1198,
       1030, 1087, 1090, 1113, 1171, 1176, 1199, 1032, 1033, 1114, 1115,
       1116, 1200, 1201, 1033, 1088, 1114, 1115, 1172, 1200, 1202, 1032,
       1034, 1035, 1114, 1116, 1117, 1201, 1004, 1035, 1089, 1116, 1117,
       1174, 1201, 1036, 1037, 1039, 1118, 1119, 1121, 1203, 1037, 1038,
       1118, 1119, 1120, 1203, 1204, 1038, 1040, 1119, 1120, 1122, 1204,
       1205, 1039, 1088, 1118, 1121, 1173, 1203, 1206, 1040, 1062, 1120,
       1122, 1145, 1205, 1207, 1010, 1095, 1096, 1123, 1181, 1182, 1209,
       1041, 1094, 1098, 1124, 1180, 1185, 1210, 1042, 1043, 1125, 1126,
       1127, 1189, 1211, 1042, 1044, 1125, 1126, 1128, 1211, 1212, 1043,
       1046, 1047, 1103, 1125, 1127, 1189, 1044, 1045, 1126, 1128, 1129,
       1212, 1213, 1045, 1048, 1128, 1129, 1130, 1213, 1214, 1048, 1049,
       1129, 1130, 1131, 1214, 1215, 1049, 1050, 1130, 1131, 1132, 1215,
       1216, 1050, 1102, 1131, 1132, 1188, 1216, 1217, 1051, 1052, 1133,
       1134, 1135, 1218, 1219, 1051, 1053, 1133, 1134, 1136, 1218, 1220,
       1052, 1054, 1133, 1135, 1137, 1219, 1221, 1053, 1057, 1134, 1136,
       1140, 1220, 1222, 1054, 1055, 1135, 1137, 1138, 1221, 1223, 1055,
       1056, 1137, 1138, 1139, 1223, 1224, 1056, 1058, 1138, 1139, 1141,
       1224, 1225, 1057, 1103, 1136, 1140, 1189, 1222, 1226, 1058, 1063,
       1139, 1141, 1146, 1225, 1227, 1059, 1104, 1109, 1142, 1190, 1195,
       1228, 1060, 1104, 1112, 1143, 1190, 1198, 1229, 1061, 1106, 1109,
       1144, 1192, 1195, 1230, 1062, 1111, 1122, 1145, 1197, 1207, 1231,
       1063, 1112, 1141, 1146, 1198, 1227, 1232, 1064, 1067, 1147, 1148,
       1151, 1233, 1234, 1064, 1065, 1147, 1148, 1149, 1234, 1235, 1065,
       1066, 1148, 1149, 1150, 1235, 1066, 1068, 1149, 1150, 1152, 1235,
       1236, 1283, 1067, 1070, 1147, 1151, 1154, 1233, 1237, 1068, 1069,
       1150, 1152, 1153, 1236, 1238, 1069, 1086, 1152, 1153, 1170, 1238,
       1239, 1070, 1081, 1151, 1154, 1165, 1237, 1240, 1071, 1073, 1155,
       1156, 1158, 1241, 1242, 1071, 1072, 1155, 1156, 1157, 1242, 1243,
       1072, 1074, 1156, 1157, 1159, 1243, 1244, 1073, 1075, 1155, 1158,
       1160, 1241, 1245, 1074, 1076, 1157, 1159, 1161, 1244, 1246, 1075,
       1077, 1158, 1160, 1162, 1245, 1247, 1076, 1078, 1159, 1161, 1163,
       1246, 1248, 1077, 1082, 1160, 1162, 1166, 1247, 1249, 1078, 1079,
       1161, 1163, 1164, 1248, 1250, 1079, 1080, 1163, 1164, 1165, 1250,
       1251, 1080, 1081, 1154, 1164, 1165, 1240, 1251, 1082, 1083, 1162,
       1166, 1167, 1249, 1252, 1083, 1084, 1166, 1167, 1168, 1252, 1253,
       1084, 1085, 1167, 1168, 1169, 1253, 1254, 1284, 1085, 1087, 1168,
       1169, 1171, 1254, 1086, 1100, 1153, 1170, 1182, 1239, 1255, 1087,
       1113, 1169, 1171, 1199, 1254, 1256, 1088, 1115, 1172, 1173, 1202,
       1257, 1088, 1121, 1172, 1173, 1206, 1257, 1258, 1089, 1117, 1174,
       1175, 1201, 1259, 1260, 1089, 1090, 1174, 1175, 1208, 1259, 1090,
       1113, 1176, 1199, 1208, 1261, 1262, 1091, 1095, 1177, 1178, 1181,
       1263, 1264, 1091, 1092, 1177, 1178, 1179, 1263, 1265, 1092, 1093,
       1178, 1179, 1180, 1265, 1266, 1093, 1094, 1124, 1179, 1180, 1210,
       1266, 1095, 1123, 1177, 1181, 1209, 1264, 1267, 1096, 1100, 1123,
       1170, 1182, 1209, 1255, 1097, 1098, 1183, 1184, 1185, 1268, 1269,
       1097, 1099, 1183, 1184, 1186, 1269, 1270, 1098, 1124, 1183, 1185,
       1210, 1268, 1271, 1295, 1099, 1101, 1184, 1186, 1187, 1270, 1272,
       1101, 1102, 1186, 1187, 1188, 1272, 1273, 1102, 1132, 1187, 1188,
       1217, 1273, 1274, 1103, 1125, 1127, 1140, 1189, 1211, 1226, 1104,
       1142, 1143, 1190, 1228, 1229, 1275, 1105, 1106, 1107, 1191, 1192,
       1193, 1276, 1106, 1144, 1191, 1192, 1230, 1276, 1277, 1107, 1108,
       1191, 1193, 1194, 1276, 1278, 1108, 1110, 1193, 1194, 1196, 1278,
       1279, 1109, 1142, 1144, 1195, 1228, 1230, 1110, 1111, 1194, 1196,
       1197, 1279, 1280, 1111, 1145, 1196, 1197, 1231, 1280, 1281, 1112,
       1143, 1146, 1198, 1229, 1232, 1282, 1113, 1171, 1176, 1199, 1256,
       1262, 1285, 1114, 1115, 1200, 1201, 1202, 1260, 1286, 1114, 1116,
       1117, 1174, 1200, 1201, 1260, 1115, 1172, 1200, 1202, 1257, 1286,
       1287, 1118, 1119, 1121, 1203, 1204, 1206, 1288, 1119, 1120, 1203,
       1204, 1205, 1288, 1289, 1120, 1122, 1204, 1205, 1207, 1289, 1290,
       1121, 1173, 1203, 1206, 1258, 1288, 1291, 1122, 1145, 1205, 1207,
       1231, 1290, 1292, 1090, 1175, 1176, 1208, 1259, 1261, 1293, 1123,
       1181, 1182, 1209, 1255, 1267, 1294, 1124, 1180, 1185, 1210, 1266,
       1271, 1125, 1126, 1189, 1211, 1212, 1226, 1296, 1126, 1128, 1211,
       1212, 1213, 1296, 1297, 1128, 1129, 1212, 1213, 1214, 1297, 1298,
       1129, 1130, 1213, 1214, 1215, 1298, 1299, 1130, 1131, 1214, 1215,
       1216, 1299, 1300, 1131, 1132, 1215, 1216, 1217, 1300, 1301, 1132,
       1188, 1216, 1217, 1274, 1301, 1302, 1133, 1134, 1218, 1219, 1220,
       1303, 1304, 1133, 1135, 1218, 1219, 1221, 1304, 1305, 1134, 1136,
       1218, 1220, 1222, 1303, 1306, 1135, 1137, 1219, 1221, 1223, 1305,
       1307, 1136, 1140, 1220, 1222, 1226, 1306, 1308, 1137, 1138, 1221,
       1223, 1224, 1307, 1309, 1138, 1139, 1223, 1224, 1225, 1309, 1310,
       1139, 1141, 1224, 1225, 1227, 1310, 1311, 1140, 1189, 1211, 1222,
       1226, 1296, 1308, 1141, 1146, 1225, 1227, 1232, 1311, 1142, 1190,
       1195, 1228, 1230, 1275, 1312, 1143, 1190, 1198, 1229, 1275, 1282,
       1313, 1144, 1192, 1195, 1228, 1230, 1277, 1312, 1314, 1145, 1197,
       1207, 1231, 1281, 1292, 1315, 1146, 1198, 1227, 1232, 1282, 1311,
       1316, 1358, 1147, 1151, 1233, 1234, 1237, 1317, 1318, 1147, 1148,
       1233, 1234, 1235, 1317, 1319, 1148, 1149, 1150, 1234, 1235, 1283,
       1319, 1150, 1152, 1236, 1238, 1283, 1320, 1321, 1151, 1154, 1233,
       1237, 1240, 1318, 1322, 1152, 1153, 1236, 1238, 1239, 1320, 1323,
       1153, 1170, 1238, 1239, 1255, 1323, 1324, 1154, 1165, 1237, 1240,
       1251, 1322, 1325, 1155, 1158, 1241, 1242, 1245, 1326, 1327, 1155,
       1156, 1241, 1242, 1243, 1327, 1328, 1156, 1157, 1242, 1243, 1244,
       1328, 1329, 1157, 1159, 1243, 1244, 1246, 1329, 1330, 1158, 1160,
       1241, 1245, 1247, 1326, 1331, 1159, 1161, 1244, 1246, 1248, 1330,
       1332, 1160, 1162, 1245, 1247, 1249, 1331, 1333, 1161, 1163, 1246,
       1248, 1250, 1332, 1334, 1162, 1166, 1247, 1249, 1252, 1333, 1335,
       1163, 1164, 1248, 1250, 1251, 1334, 1336, 1164, 1165, 1240, 1250,
       1251, 1325, 1336, 1166, 1167, 1249, 1252, 1253, 1335, 1337, 1167,
       1168, 1252, 1253, 1284, 1337, 1168, 1169, 1171, 1254, 1256, 1284,
       1338, 1170, 1182, 1209, 1239, 1255, 1294, 1324, 1171, 1199, 1254,
       1256, 1285, 1338, 1339, 1172, 1173, 1202, 1257, 1258, 1287, 1340,
       1173, 1206, 1257, 1258, 1291, 1340, 1341, 1174, 1175, 1208, 1259,
       1260, 1293, 1342, 1174, 1200, 1201, 1259, 1260, 1286, 1342, 1370,
       1176, 1208, 1261, 1262, 1293, 1343, 1176, 1199, 1261, 1262, 1285,
       1343, 1344, 1379, 1177, 1178, 1263, 1264, 1265, 1345, 1346, 1177,
       1181, 1263, 1264, 1267, 1346, 1347, 1178, 1179, 1263, 1265, 1266,
       1345, 1348, 1179, 1180, 1210, 1265, 1266, 1271, 1348, 1181, 1209,
       1264, 1267, 1294, 1347, 1349, 1183, 1185, 1268, 1269, 1295, 1350,
       1351, 1183, 1184, 1268, 1269, 1270, 1350, 1352, 1184, 1186, 1269,
       1270, 1272, 1352, 1353, 1185, 1210, 1266, 1271, 1295, 1348, 1354,
       1186, 1187, 1270, 1272, 1273, 1353, 1355, 1187, 1188, 1272, 1273,
       1274, 1355, 1356, 1188, 1217, 1273, 1274, 1302, 1356, 1357, 1190,
       1228, 1229, 1275, 1312, 1313, 1359, 1191, 1192, 1193, 1276, 1277,
       1278, 1360, 1192, 1230, 1276, 1277, 1314, 1360, 1361, 1193, 1194,
       1276, 1278, 1279, 1360, 1362, 1194, 1196, 1278, 1279, 1280, 1362,
       1363, 1196, 1197, 1279, 1280, 1281, 1363, 1364, 1197, 1231, 1280,
       1281, 1315, 1364, 1365, 1198, 1229, 1232, 1282, 1313, 1316, 1366,
       1150, 1235, 1236, 1283, 1319, 1321, 1367, 1168, 1253, 1254, 1284,
       1337, 1338, 1368, 1199, 1256, 1262, 1285, 1339, 1344, 1369, 1200,
       1202, 1260, 1286, 1287, 1370, 1371, 1202, 1257, 1286, 1287, 1340,
       1371, 1372, 1203, 1204, 1206, 1288, 1289, 1291, 1373, 1204, 1205,
       1288, 1289, 1290, 1373, 1374, 1205, 1207, 1289, 1290, 1292, 1374,
       1375, 1206, 1258, 1288, 1291, 1341, 1373, 1376, 1427, 1207, 1231,
       1290, 1292, 1315, 1375, 1377, 1208, 1259, 1261, 1293, 1342, 1343,
       1378, 1209, 1255, 1267, 1294, 1324, 1349, 1380, 1185, 1268, 1271,
       1295, 1351, 1354, 1381, 1211, 1212, 1226, 1296, 1297, 1308, 1382,
       1212, 1213, 1296, 1297, 1298, 1382, 1383, 1213, 1214, 1297, 1298,
       1299, 1383, 1384, 1214, 1215, 1298, 1299, 1300, 1384, 1385, 1215,
       1216, 1299, 1300, 1301, 1385, 1386, 1216, 1217, 1300, 1301, 1302,
       1386, 1387, 1217, 1274, 1301, 1302, 1357, 1387, 1388, 1218, 1220,
       1303, 1304, 1306, 1389, 1390, 1218, 1219, 1303, 1304, 1305, 1390,
       1391, 1219, 1221, 1304, 1305, 1307, 1391, 1392, 1220, 1222, 1303,
       1306, 1308, 1389, 1393, 1221, 1223, 1305, 1307, 1309, 1392, 1394,
       1222, 1226, 1296, 1306, 1308, 1382, 1393, 1223, 1224, 1307, 1309,
       1310, 1394, 1395, 1224, 1225, 1309, 1310, 1311, 1395, 1396, 1225,
       1227, 1232, 1310, 1311, 1358, 1396, 1228, 1230, 1275, 1312, 1314,
       1359, 1397, 1229, 1275, 1282, 1313, 1359, 1366, 1398, 1230, 1277,
       1312, 1314, 1361, 1397, 1399, 1231, 1281, 1292, 1315, 1365, 1377,
       1400, 1232, 1282, 1316, 1358, 1366, 1401, 1402, 1233, 1234, 1317,
       1318, 1319, 1403, 1404, 1233, 1237, 1317, 1318, 1322, 1403, 1405,
       1234, 1235, 1283, 1317, 1319, 1367, 1404, 1236, 1238, 1320, 1321,
       1323, 1406, 1407, 1236, 1283, 1320, 1321, 1367, 1406, 1408, 1237,
       1240, 1318, 1322, 1325, 1405, 1409, 1238, 1239, 1320, 1323, 1324,
       1407, 1410, 1239, 1255, 1294, 1323, 1324, 1380, 1410, 1240, 1251,
       1322, 1325, 1336, 1409, 1411, 1241, 1245, 1326, 1327, 1331, 1412,
       1413, 1241, 1242, 1326, 1327, 1328, 1412, 1414, 1242, 1243, 1327,
       1328, 1329, 1414, 1415, 1243, 1244, 1328, 1329, 1330, 1415, 1416,
       1244, 1246, 1329, 1330, 1332, 1416, 1417, 1245, 1247, 1326, 1331,
       1333, 1413, 1418, 1246, 1248, 1330, 1332, 1334, 1417, 1419, 1247,
       1249, 1331, 1333, 1335, 1418, 1420, 1248, 1250, 1332, 1334, 1336,
       1419, 1421, 1249, 1252, 1333, 1335, 1337, 1420, 1422, 1250, 1251,
       1325, 1334, 1336, 1411, 1421, 1252, 1253, 1284, 1335, 1337, 1368,
       1422, 1423, 1254, 1256, 1284, 1338, 1339, 1368, 1424, 1256, 1285,
       1338, 1339, 1369, 1424, 1425, 1257, 1258, 1287, 1340, 1341, 1372,
       1426, 1258, 1291, 1340, 1341, 1376, 1426, 1428, 1259, 1260, 1293,
       1342, 1370, 1378, 1429, 1261, 1262, 1293, 1343, 1378, 1379, 1430,
       1262, 1285, 1344, 1369, 1379, 1431, 1263, 1265, 1345, 1346, 1348,
       1432, 1433, 1263, 1264, 1345, 1346, 1347, 1432, 1434, 1264, 1267,
       1346, 1347, 1349, 1434, 1435, 1265, 1266, 1271, 1345, 1348, 1354,
       1433, 1267, 1294, 1347, 1349, 1380, 1435, 1436, 1268, 1269, 1350,
       1351, 1352, 1437, 1438, 1268, 1295, 1350, 1351, 1381, 1437, 1439,
       1269, 1270, 1350, 1352, 1353, 1438, 1440, 1270, 1272, 1352, 1353,
       1355, 1440, 1441, 1271, 1295, 1348, 1354, 1381, 1433, 1442, 1272,
       1273, 1353, 1355, 1356, 1441, 1273, 1274, 1355, 1356, 1357, 1441,
       1443, 1274, 1302, 1356, 1357, 1388, 1443, 1444, 1232, 1311, 1316,
       1358, 1396, 1401, 1445, 1275, 1312, 1313, 1359, 1397, 1398, 1446,
       1276, 1277, 1278, 1360, 1361, 1362, 1447, 1277, 1314, 1360, 1361,
       1399, 1447, 1448, 1278, 1279, 1360, 1362, 1363, 1447, 1449, 1279,
       1280, 1362, 1363, 1364, 1449, 1450, 1280, 1281, 1363, 1364, 1365,
       1450, 1451, 1281, 1315, 1364, 1365, 1400, 1451, 1452, 1282, 1313,
       1316, 1366, 1398, 1402, 1453, 1283, 1319, 1321, 1367, 1404, 1408,
       1454, 1284, 1337, 1338, 1368, 1423, 1424, 1455, 1285, 1339, 1344,
       1369, 1425, 1431, 1456, 1468, 1260, 1286, 1342, 1370, 1371, 1429,
       1457, 1286, 1287, 1370, 1371, 1372, 1457, 1458, 1287, 1340, 1371,
       1372, 1426, 1458, 1459, 1288, 1289, 1291, 1373, 1374, 1427, 1460,
       1289, 1290, 1373, 1374, 1375, 1460, 1461, 1290, 1292, 1374, 1375,
       1377, 1461, 1462, 1291, 1341, 1376, 1427, 1428, 1463, 1464, 1292,
       1315, 1375, 1377, 1400, 1462, 1465, 1293, 1342, 1343, 1378, 1429,
       1430, 1466, 1262, 1343, 1344, 1379, 1430, 1431, 1467, 1294, 1324,
       1349, 1380, 1410, 1436, 1469, 1295, 1351, 1354, 1381, 1439, 1442,
       1470, 1296, 1297, 1308, 1382, 1383, 1393, 1471, 1297, 1298, 1382,
       1383, 1384, 1471, 1472, 1298, 1299, 1383, 1384, 1385, 1472, 1473,
       1299, 1300, 1384, 1385, 1386, 1473, 1474, 1300, 1301, 1385, 1386,
       1387, 1474, 1301, 1302, 1386, 1387, 1388, 1474, 1475, 1531, 1302,
       1357, 1387, 1388, 1444, 1475, 1476, 1303, 1306, 1389, 1390, 1393,
       1477, 1478, 1303, 1304, 1389, 1390, 1391, 1478, 1479, 1304, 1305,
       1390, 1391, 1392, 1479, 1480, 1305, 1307, 1391, 1392, 1394, 1480,
       1481, 1306, 1308, 1382, 1389, 1393, 1471, 1477, 1307, 1309, 1392,
       1394, 1395, 1481, 1482, 1309, 1310, 1394, 1395, 1396, 1482, 1483,
       1310, 1311, 1358, 1395, 1396, 1445, 1483, 1312, 1314, 1359, 1397,
       1399, 1446, 1484, 1313, 1359, 1366, 1398, 1446, 1453, 1485, 1314,
       1361, 1397, 1399, 1448, 1484, 1486, 1536, 1315, 1365, 1377, 1400,
       1452, 1465, 1487, 1316, 1358, 1401, 1402, 1445, 1488, 1489, 1316,
       1366, 1401, 1402, 1453, 1489, 1490, 1317, 1318, 1403, 1404, 1405,
       1491, 1492, 1317, 1319, 1367, 1403, 1404, 1454, 1492, 1318, 1322,
       1403, 1405, 1409, 1491, 1493, 1320, 1321, 1406, 1407, 1408, 1494,
       1495, 1320, 1323, 1406, 1407, 1410, 1495, 1496, 1321, 1367, 1406,
       1408, 1454, 1494, 1497, 1322, 1325, 1405, 1409, 1411, 1493, 1498,
       1323, 1324, 1380, 1407, 1410, 1469, 1496, 1325, 1336, 1409, 1411,
       1421, 1498, 1499, 1326, 1327, 1412, 1413, 1414, 1500, 1501, 1326,
       1331, 1412, 1413, 1418, 1500, 1502, 1327, 1328, 1412, 1414, 1415,
       1501, 1503, 1328, 1329, 1414, 1415, 1416, 1503, 1504, 1329, 1330,
       1415, 1416, 1417, 1504, 1505, 1330, 1332, 1416, 1417, 1419, 1505,
       1506, 1331, 1333, 1413, 1418, 1420, 1502, 1507, 1332, 1334, 1417,
       1419, 1421, 1506, 1508, 1333, 1335, 1418, 1420, 1422, 1507, 1509,
       1334, 1336, 1411, 1419, 1421, 1499, 1508, 1335, 1337, 1420, 1422,
       1423, 1509, 1510, 1337, 1368, 1422, 1423, 1455, 1510, 1511, 1338,
       1339, 1368, 1424, 1425, 1455, 1512, 1339, 1369, 1424, 1425, 1456,
       1512, 1513, 1340, 1341, 1372, 1426, 1428, 1459, 1514, 1291, 1373,
       1376, 1427, 1460, 1463, 1515, 1341, 1376, 1426, 1428, 1464, 1514,
       1516, 1550, 1342, 1370, 1378, 1429, 1457, 1466, 1517, 1343, 1378,
       1379, 1430, 1466, 1467, 1518, 1344, 1369, 1379, 1431, 1467, 1468,
       1519, 1345, 1346, 1432, 1433, 1434, 1520, 1521, 1345, 1348, 1354,
       1432, 1433, 1442, 1521, 1346, 1347, 1432, 1434, 1435, 1520, 1522,
       1347, 1349, 1434, 1435, 1436, 1522, 1523, 1349, 1380, 1435, 1436,
       1469, 1523, 1524, 1350, 1351, 1437, 1438, 1439, 1525, 1526, 1350,
       1352, 1437, 1438, 1440, 1526, 1527, 1351, 1381, 1437, 1439, 1470,
       1525, 1528, 1352, 1353, 1438, 1440, 1441, 1527, 1529, 1353, 1355,
       1356, 1440, 1441, 1443, 1529, 1354, 1381, 1433, 1442, 1470, 1521,
       1356, 1357, 1441, 1443, 1444, 1529, 1530, 1357, 1388, 1443, 1444,
       1476, 1530, 1532, 1358, 1396, 1401, 1445, 1483, 1488, 1533, 1359,
       1397, 1398, 1446, 1484, 1485, 1534, 1360, 1361, 1362, 1447, 1448,
       1449, 1535, 1361, 1399, 1447, 1448, 1486, 1535, 1537, 1362, 1363,
       1447, 1449, 1450, 1535, 1538, 1363, 1364, 1449, 1450, 1451, 1538,
       1539, 1364, 1365, 1450, 1451, 1452, 1539, 1540, 1365, 1400, 1451,
       1452, 1487, 1540, 1541, 1366, 1398, 1402, 1453, 1485, 1490, 1542,
       1367, 1404, 1408, 1454, 1492, 1497, 1543, 1368, 1423, 1424, 1455,
       1511, 1512, 1544, 1369, 1425, 1456, 1468, 1513, 1545, 1546, 1370,
       1371, 1429, 1457, 1458, 1517, 1547, 1371, 1372, 1457, 1458, 1459,
       1547, 1548, 1372, 1426, 1458, 1459, 1514, 1548, 1549, 1373, 1374,
       1427, 1460, 1461, 1515, 1551, 1374, 1375, 1460, 1461, 1462, 1551,
       1552, 1375, 1377, 1461, 1462, 1465, 1552, 1553, 1376, 1427, 1463,
       1464, 1515, 1554, 1555, 1376, 1428, 1463, 1464, 1516, 1555, 1556,
       1377, 1400, 1462, 1465, 1487, 1553, 1557, 1378, 1429, 1430, 1466,
       1517, 1518, 1558, 1379, 1430, 1431, 1467, 1518, 1519, 1559, 1369,
       1431, 1456, 1468, 1519, 1546, 1560, 1380, 1410, 1436, 1469, 1496,
       1524, 1561, 1381, 1439, 1442, 1470, 1521, 1528, 1562, 1563, 1382,
       1383, 1393, 1471, 1472, 1477, 1564, 1383, 1384, 1471, 1472, 1473,
       1564, 1565, 1384, 1385, 1472, 1473, 1474, 1565, 1566, 1385, 1386,
       1387, 1473, 1474, 1531, 1566, 1387, 1388, 1475, 1476, 1531, 1567,
       1568, 1388, 1444, 1475, 1476, 1532, 1567, 1569, 1389, 1393, 1471,
       1477, 1478, 1564, 1570, 1389, 1390, 1477, 1478, 1479, 1570, 1571,
       1390, 1391, 1478, 1479, 1480, 1571, 1572, 1391, 1392, 1479, 1480,
       1481, 1572, 1573, 1392, 1394, 1480, 1481, 1482, 1573, 1574, 1394,
       1395, 1481, 1482, 1483, 1574, 1575, 1395, 1396, 1445, 1482, 1483,
       1533, 1575, 1397, 1399, 1446, 1484, 1534, 1536, 1398, 1446, 1453,
       1485, 1534, 1542, 1576, 1399, 1448, 1486, 1536, 1537, 1577, 1578,
       1400, 1452, 1465, 1487, 1541, 1557, 1579, 1401, 1445, 1488, 1489,
       1533, 1580, 1581, 1401, 1402, 1488, 1489, 1490, 1581, 1582, 1402,
       1453, 1489, 1490, 1542, 1582, 1583, 1403, 1405, 1491, 1492, 1493,
       1584, 1585, 1403, 1404, 1454, 1491, 1492, 1543, 1585, 1405, 1409,
       1491, 1493, 1498, 1584, 1586, 1406, 1408, 1494, 1495, 1497, 1587,
       1588, 1406, 1407, 1494, 1495, 1496, 1587, 1589, 1407, 1410, 1469,
       1495, 1496, 1561, 1589, 1408, 1454, 1494, 1497, 1543, 1588, 1590,
       1409, 1411, 1493, 1498, 1499, 1586, 1591, 1411, 1421, 1498, 1499,
       1508, 1591, 1592, 1412, 1413, 1500, 1501, 1502, 1593, 1594, 1412,
       1414, 1500, 1501, 1503, 1594, 1595, 1413, 1418, 1500, 1502, 1507,
       1593, 1596, 1414, 1415, 1501, 1503, 1504, 1595, 1597, 1415, 1416,
       1503, 1504, 1505, 1597, 1598, 1416, 1417, 1504, 1505, 1506, 1598,
       1599, 1417, 1419, 1505, 1506, 1508, 1599, 1600, 1418, 1420, 1502,
       1507, 1509, 1596, 1601, 1419, 1421, 1499, 1506, 1508, 1592, 1600,
       1636, 1420, 1422, 1507, 1509, 1510, 1601, 1602, 1422, 1423, 1509,
       1510, 1511, 1602, 1603, 1423, 1455, 1510, 1511, 1544, 1603, 1604,
       1424, 1425, 1455, 1512, 1513, 1544, 1605, 1425, 1456, 1512, 1513,
       1545, 1605, 1606, 1426, 1428, 1459, 1514, 1549, 1550, 1607, 1427,
       1460, 1463, 1515, 1551, 1554, 1608, 1428, 1464, 1516, 1550, 1556,
       1609, 1610, 1429, 1457, 1466, 1517, 1547, 1558, 1611, 1430, 1466,
       1467, 1518, 1558, 1559, 1612, 1431, 1467, 1468, 1519, 1559, 1560,
       1613, 1432, 1434, 1520, 1521, 1522, 1563, 1614, 1432, 1433, 1442,
       1470, 1520, 1521, 1563, 1434, 1435, 1520, 1522, 1523, 1614, 1615,
       1435, 1436, 1522, 1523, 1524, 1615, 1616, 1436, 1469, 1523, 1524,
       1561, 1616, 1617, 1437, 1439, 1525, 1526, 1528, 1618, 1619, 1437,
       1438, 1525, 1526, 1527, 1619, 1620, 1438, 1440, 1526, 1527, 1529,
       1620, 1621, 1439, 1470, 1525, 1528, 1562, 1618, 1622, 1440, 1441,
       1443, 1527, 1529, 1530, 1621, 1443, 1444, 1529, 1530, 1532, 1621,
       1623, 1387, 1474, 1475, 1531, 1566, 1568, 1624, 1444, 1476, 1530,
       1532, 1569, 1623, 1625, 1445, 1483, 1488, 1533, 1575, 1580, 1626,
       1446, 1484, 1485, 1534, 1536, 1576, 1627, 1447, 1448, 1449, 1535,
       1537, 1538, 1628, 1399, 1484, 1486, 1534, 1536, 1577, 1627, 1448,
       1486, 1535, 1537, 1578, 1628, 1629, 1449, 1450, 1535, 1538, 1539,
       1628, 1630, 1450, 1451, 1538, 1539, 1540, 1630, 1631, 1451, 1452,
       1539, 1540, 1541, 1631, 1632, 1452, 1487, 1540, 1541, 1579, 1632,
       1633, 1453, 1485, 1490, 1542, 1576, 1583, 1634, 1454, 1492, 1497,
       1543, 1585, 1590, 1635, 1455, 1511, 1512, 1544, 1604, 1605, 1456,
       1513, 1545, 1546, 1606, 1637, 1638, 1698, 1456, 1468, 1545, 1546,
       1560, 1637, 1639, 1697, 1700, 1457, 1458, 1517, 1547, 1548, 1611,
       1640, 1458, 1459, 1547, 1548, 1549, 1640, 1641, 1459, 1514, 1548,
       1549, 1607, 1641, 1428, 1514, 1516, 1550, 1607, 1610, 1642, 1460,
       1461, 1515, 1551, 1552, 1608, 1643, 1461, 1462, 1551, 1552, 1553,
       1643, 1644, 1462, 1465, 1552, 1553, 1557, 1644, 1645, 1463, 1515,
       1554, 1555, 1608, 1646, 1647, 1703, 1463, 1464, 1554, 1555, 1556,
       1647, 1648, 1464, 1516, 1555, 1556, 1609, 1648, 1649, 1465, 1487,
       1553, 1557, 1579, 1645, 1650, 1466, 1517, 1518, 1558, 1611, 1612,
       1651, 1467, 1518, 1519, 1559, 1612, 1613, 1652, 1468, 1519, 1546,
       1560, 1613, 1639, 1469, 1496, 1524, 1561, 1589, 1617, 1653, 1470,
       1528, 1562, 1563, 1622, 1654, 1655, 1470, 1520, 1521, 1562, 1563,
       1614, 1654, 1471, 1472, 1477, 1564, 1565, 1570, 1656, 1472, 1473,
       1564, 1565, 1566, 1656, 1657, 1473, 1474, 1531, 1565, 1566, 1624,
       1657, 1475, 1476, 1567, 1568, 1569, 1658, 1659, 1475, 1531, 1567,
       1568, 1624, 1659, 1660, 1476, 1532, 1567, 1569, 1625, 1658, 1661,
       1477, 1478, 1564, 1570, 1571, 1656, 1662, 1478, 1479, 1570, 1571,
       1572, 1662, 1663, 1479, 1480, 1571, 1572, 1573, 1663, 1664, 1480,
       1481, 1572, 1573, 1574, 1664, 1665, 1481, 1482, 1573, 1574, 1575,
       1665, 1666, 1482, 1483, 1533, 1574, 1575, 1626, 1666, 1485, 1534,
       1542, 1576, 1627, 1634, 1667, 1486, 1536, 1577, 1578, 1627, 1668,
       1669, 1486, 1537, 1577, 1578, 1629, 1668, 1670, 1487, 1541, 1557,
       1579, 1633, 1650, 1671, 1488, 1533, 1580, 1581, 1626, 1672, 1673,
       1488, 1489, 1580, 1581, 1582, 1673, 1674, 1489, 1490, 1581, 1582,
       1583, 1674, 1675, 1490, 1542, 1582, 1583, 1634, 1675, 1676, 1491,
       1493, 1584, 1585, 1586, 1677, 1678, 1491, 1492, 1543, 1584, 1585,
       1635, 1678, 1493, 1498, 1584, 1586, 1591, 1677, 1679, 1494, 1495,
       1587, 1588, 1589, 1680, 1681, 1494, 1497, 1587, 1588, 1590, 1680,
       1682, 1495, 1496, 1561, 1587, 1589, 1653, 1681, 1497, 1543, 1588,
       1590, 1635, 1682, 1683, 1498, 1499, 1586, 1591, 1592, 1679, 1684,
       1729, 1499, 1508, 1591, 1592, 1636, 1684, 1685, 1500, 1502, 1593,
       1594, 1596, 1686, 1687, 1500, 1501, 1593, 1594, 1595, 1687, 1688,
       1501, 1503, 1594, 1595, 1597, 1688, 1689, 1502, 1507, 1593, 1596,
       1601, 1686, 1690, 1503, 1504, 1595, 1597, 1598, 1689, 1691, 1504,
       1505, 1597, 1598, 1599, 1691, 1692, 1505, 1506, 1598, 1599, 1600,
       1692, 1693, 1506, 1508, 1599, 1600, 1636, 1693, 1507, 1509, 1596,
       1601, 1602, 1690, 1694, 1732, 1509, 1510, 1601, 1602, 1603, 1694,
       1510, 1511, 1602, 1603, 1604, 1694, 1695, 1511, 1544, 1603, 1604,
       1605, 1695, 1696, 1733, 1512, 1513, 1544, 1604, 1605, 1606, 1696,
       1513, 1545, 1605, 1606, 1638, 1696, 1699, 1514, 1549, 1550, 1607,
       1641, 1642, 1701, 1515, 1551, 1554, 1608, 1643, 1646, 1702, 1516,
       1556, 1609, 1610, 1649, 1704, 1705, 1516, 1550, 1609, 1610, 1642,
       1705, 1706, 1517, 1547, 1558, 1611, 1640, 1651, 1518, 1558, 1559,
       1612, 1651, 1652, 1707, 1519, 1559, 1560, 1613, 1639, 1652, 1708,
       1520, 1522, 1563, 1614, 1615, 1654, 1709, 1522, 1523, 1614, 1615,
       1616, 1709, 1710, 1523, 1524, 1615, 1616, 1617, 1710, 1711, 1524,
       1561, 1616, 1617, 1653, 1711, 1712, 1525, 1528, 1618, 1619, 1622,
       1713, 1714, 1525, 1526, 1618, 1619, 1620, 1714, 1715, 1526, 1527,
       1619, 1620, 1621, 1715, 1716, 1527, 1529, 1530, 1620, 1621, 1623,
       1716, 1528, 1562, 1618, 1622, 1655, 1713, 1717, 1530, 1532, 1621,
       1623, 1625, 1716, 1718, 1531, 1566, 1568, 1624, 1657, 1660, 1719,
       1532, 1569, 1623, 1625, 1661, 1718, 1720, 1533, 1575, 1580, 1626,
       1666, 1672, 1721, 1534, 1536, 1576, 1577, 1627, 1667, 1669, 1535,
       1537, 1538, 1628, 1629, 1630, 1722, 1537, 1578, 1628, 1629, 1670,
       1722, 1723, 1538, 1539, 1628, 1630, 1631, 1722, 1724, 1539, 1540,
       1630, 1631, 1632, 1724, 1725, 1540, 1541, 1631, 1632, 1633, 1725,
       1726, 1541, 1579, 1632, 1633, 1671, 1726, 1727, 1542, 1576, 1583,
       1634, 1667, 1676, 1728, 1543, 1585, 1590, 1635, 1678, 1683, 1730,
       1508, 1592, 1600, 1636, 1685, 1693, 1731, 1545, 1546, 1637, 1697,
       1698, 1734, 1735, 1545, 1606, 1638, 1698, 1699, 1736, 1737, 1546,
       1560, 1613, 1639, 1700, 1708, 1547, 1548, 1611, 1640, 1641, 1651,
       1738, 1548, 1549, 1607, 1640, 1641, 1701, 1738, 1550, 1607, 1610,
       1642, 1701, 1706, 1739, 1551, 1552, 1608, 1643, 1644, 1702, 1740,
       1552, 1553, 1643, 1644, 1645, 1740, 1741, 1553, 1557, 1644, 1645,
       1650, 1741, 1742, 1554, 1608, 1646, 1702, 1703, 1743, 1744, 1554,
       1555, 1647, 1648, 1703, 1745, 1746, 1555, 1556, 1647, 1648, 1649,
       1746, 1747, 1556, 1609, 1648, 1649, 1704, 1747, 1748, 1557, 1579,
       1645, 1650, 1671, 1742, 1749, 1558, 1611, 1612, 1640, 1651, 1707,
       1738, 1559, 1612, 1613, 1652, 1707, 1708, 1750, 1561, 1589, 1617,
       1653, 1681, 1712, 1751, 1562, 1563, 1614, 1654, 1655, 1709, 1752,
       1562, 1622, 1654, 1655, 1717, 1752, 1753, 1564, 1565, 1570, 1656,
       1657, 1662, 1754, 1565, 1566, 1624, 1656, 1657, 1719, 1754, 1567,
       1569, 1658, 1659, 1661, 1755, 1756, 1567, 1568, 1658, 1659, 1660,
       1755, 1757, 1568, 1624, 1659, 1660, 1719, 1757, 1758, 1569, 1625,
       1658, 1661, 1720, 1756, 1759, 1570, 1571, 1656, 1662, 1663, 1754,
       1760, 1571, 1572, 1662, 1663, 1664, 1760, 1761, 1572, 1573, 1663,
       1664, 1665, 1761, 1762, 1573, 1574, 1664, 1665, 1666, 1762, 1763,
       1574, 1575, 1626, 1665, 1666, 1721, 1763, 1576, 1627, 1634, 1667,
       1669, 1728, 1764, 1577, 1578, 1668, 1669, 1670, 1765, 1766, 1577,
       1627, 1667, 1668, 1669, 1764, 1765, 1578, 1629, 1668, 1670, 1723,
       1766, 1767, 1579, 1633, 1650, 1671, 1727, 1749, 1768, 1580, 1626,
       1672, 1673, 1721, 1769, 1770, 1580, 1581, 1672, 1673, 1674, 1770,
       1771, 1581, 1582, 1673, 1674, 1675, 1771, 1772, 1582, 1583, 1674,
       1675, 1676, 1772, 1773, 1583, 1634, 1675, 1676, 1728, 1773, 1774,
       1818, 1584, 1586, 1677, 1678, 1679, 1775, 1776, 1584, 1585, 1635,
       1677, 1678, 1730, 1775, 1586, 1591, 1677, 1679, 1729, 1776, 1777,
       1587, 1588, 1680, 1681, 1682, 1778, 1779, 1587, 1589, 1653, 1680,
       1681, 1751, 1778, 1588, 1590, 1680, 1682, 1683, 1779, 1780, 1590,
       1635, 1682, 1683, 1730, 1780, 1781, 1591, 1592, 1684, 1685, 1729,
       1782, 1592, 1636, 1684, 1685, 1731, 1782, 1783, 1593, 1596, 1686,
       1687, 1690, 1784, 1785, 1593, 1594, 1686, 1687, 1688, 1784, 1786,
       1594, 1595, 1687, 1688, 1689, 1786, 1787, 1595, 1597, 1688, 1689,
       1691, 1787, 1788, 1596, 1601, 1686, 1690, 1732, 1785, 1789, 1597,
       1598, 1689, 1691, 1692, 1788, 1790, 1598, 1599, 1691, 1692, 1693,
       1790, 1791, 1599, 1600, 1636, 1692, 1693, 1731, 1791, 1601, 1602,
       1603, 1694, 1695, 1732, 1792, 1603, 1604, 1694, 1695, 1733, 1792,
       1793, 1604, 1605, 1606, 1696, 1699, 1733, 1794, 1546, 1637, 1697,
       1700, 1705, 1706, 1734, 1545, 1637, 1638, 1698, 1735, 1737, 1795,
       1606, 1638, 1696, 1699, 1736, 1794, 1796, 1546, 1639, 1697, 1700,
       1706, 1708, 1739, 1607, 1641, 1642, 1701, 1738, 1739, 1750, 1608,
       1643, 1646, 1702, 1740, 1744, 1797, 1554, 1646, 1647, 1703, 1743,
       1745, 1798, 1609, 1649, 1704, 1705, 1734, 1748, 1799, 1609, 1610,
       1697, 1704, 1705, 1706, 1734, 1610, 1642, 1697, 1700, 1705, 1706,
       1739, 1612, 1651, 1652, 1707, 1738, 1750, 1613, 1639, 1652, 1700,
       1708, 1739, 1750, 1614, 1615, 1654, 1709, 1710, 1752, 1800, 1615,
       1616, 1709, 1710, 1711, 1800, 1801, 1616, 1617, 1710, 1711, 1712,
       1801, 1802, 1617, 1653, 1711, 1712, 1751, 1802, 1618, 1622, 1713,
       1714, 1717, 1803, 1804, 1618, 1619, 1713, 1714, 1715, 1804, 1805,
       1619, 1620, 1714, 1715, 1716, 1805, 1806, 1620, 1621, 1623, 1715,
       1716, 1718, 1806, 1622, 1655, 1713, 1717, 1753, 1803, 1807, 1623,
       1625, 1716, 1718, 1720, 1806, 1808, 1624, 1657, 1660, 1719, 1754,
       1758, 1809, 1625, 1661, 1718, 1720, 1759, 1808, 1810, 1626, 1666,
       1672, 1721, 1763, 1769, 1811, 1628, 1629, 1630, 1722, 1723, 1724,
       1812, 1629, 1670, 1722, 1723, 1767, 1812, 1813, 1630, 1631, 1722,
       1724, 1725, 1812, 1814, 1631, 1632, 1724, 1725, 1726, 1814, 1815,
       1632, 1633, 1725, 1726, 1727, 1815, 1816, 1633, 1671, 1726, 1727,
       1768, 1816, 1817, 1634, 1667, 1676, 1728, 1764, 1774, 1819, 1591,
       1679, 1684, 1729, 1777, 1782, 1820, 1822, 1635, 1678, 1683, 1730,
       1775, 1781, 1821, 1636, 1685, 1693, 1731, 1783, 1791, 1823, 1601,
       1690, 1694, 1732, 1789, 1792, 1824, 1604, 1695, 1696, 1733, 1793,
       1794, 1825, 1637, 1697, 1704, 1705, 1734, 1735, 1799, 1637, 1698,
       1734, 1735, 1795, 1799, 1826, 1638, 1699, 1736, 1737, 1796, 1827,
       1828, 1638, 1698, 1736, 1737, 1795, 1827, 1829, 1640, 1641, 1651,
       1701, 1707, 1738, 1750, 1642, 1700, 1701, 1706, 1708, 1739, 1750,
       1643, 1644, 1702, 1740, 1741, 1797, 1830, 1644, 1645, 1740, 1741,
       1742, 1830, 1831, 1645, 1650, 1741, 1742, 1749, 1831, 1832, 1646,
       1703, 1743, 1744, 1798, 1833, 1834, 1646, 1702, 1743, 1744, 1797,
       1834, 1835, 1647, 1703, 1745, 1746, 1798, 1836, 1837, 1647, 1648,
       1745, 1746, 1747, 1837, 1838, 1648, 1649, 1746, 1747, 1748, 1838,
       1839, 1649, 1704, 1747, 1748, 1799, 1839, 1840, 1887, 1650, 1671,
       1742, 1749, 1768, 1832, 1841, 1652, 1701, 1707, 1708, 1738, 1739,
       1750, 1653, 1681, 1712, 1751, 1778, 1802, 1842, 1654, 1655, 1709,
       1752, 1753, 1800, 1843, 1655, 1717, 1752, 1753, 1807, 1843, 1844,
       1656, 1657, 1662, 1719, 1754, 1760, 1809, 1658, 1659, 1755, 1756,
       1757, 1845, 1846, 1658, 1661, 1755, 1756, 1759, 1845, 1847, 1659,
       1660, 1755, 1757, 1758, 1846, 1848, 1660, 1719, 1757, 1758, 1809,
       1848, 1849, 1661, 1720, 1756, 1759, 1810, 1847, 1850, 1662, 1663,
       1754, 1760, 1761, 1809, 1851, 1663, 1664, 1760, 1761, 1762, 1851,
       1852, 1664, 1665, 1761, 1762, 1763, 1852, 1853, 1665, 1666, 1721,
       1762, 1763, 1811, 1853, 1667, 1669, 1728, 1764, 1765, 1819, 1854,
       1668, 1669, 1764, 1765, 1766, 1854, 1668, 1670, 1765, 1766, 1767,
       1854, 1855, 1670, 1723, 1766, 1767, 1813, 1855, 1856, 1671, 1727,
       1749, 1768, 1817, 1841, 1857, 1672, 1721, 1769, 1770, 1811, 1858,
       1859, 1672, 1673, 1769, 1770, 1771, 1858, 1860, 1673, 1674, 1770,
       1771, 1772, 1860, 1861, 1674, 1675, 1771, 1772, 1773, 1861, 1862,
       1675, 1676, 1772, 1773, 1818, 1862, 1676, 1728, 1774, 1818, 1819,
       1863, 1864, 1677, 1678, 1730, 1775, 1776, 1821, 1865, 1677, 1679,
       1775, 1776, 1777, 1865, 1866, 1679, 1729, 1776, 1777, 1820, 1866,
       1867, 1680, 1681, 1751, 1778, 1779, 1842, 1868, 1680, 1682, 1778,
       1779, 1780, 1868, 1869, 1682, 1683, 1779, 1780, 1781, 1869, 1870,
       1683, 1730, 1780, 1781, 1821, 1870, 1871, 1684, 1685, 1729, 1782,
       1783, 1822, 1872, 1685, 1731, 1782, 1783, 1823, 1872, 1873, 1686,
       1687, 1784, 1785, 1786, 1874, 1875, 1686, 1690, 1784, 1785, 1789,
       1875, 1876, 1913, 1687, 1688, 1784, 1786, 1787, 1874, 1877, 1912,
       1688, 1689, 1786, 1787, 1788, 1877, 1689, 1691, 1787, 1788, 1790,
       1877, 1878, 1690, 1732, 1785, 1789, 1824, 1876, 1691, 1692, 1788,
       1790, 1791, 1878, 1879, 1692, 1693, 1731, 1790, 1791, 1823, 1879,
       1694, 1695, 1732, 1792, 1793, 1824, 1880, 1695, 1733, 1792, 1793,
       1825, 1880, 1881, 1696, 1699, 1733, 1794, 1796, 1825, 1882, 1698,
       1735, 1737, 1795, 1826, 1829, 1883, 1699, 1736, 1794, 1796, 1828,
       1882, 1884, 1702, 1740, 1744, 1797, 1830, 1835, 1885, 1703, 1743,
       1745, 1798, 1833, 1836, 1886, 1704, 1734, 1735, 1748, 1799, 1826,
       1840, 1709, 1710, 1752, 1800, 1801, 1843, 1888, 1710, 1711, 1800,
       1801, 1802, 1888, 1889, 1711, 1712, 1751, 1801, 1802, 1842, 1889,
       1890, 1713, 1717, 1803, 1804, 1807, 1891, 1892, 1713, 1714, 1803,
       1804, 1805, 1891, 1893, 1714, 1715, 1804, 1805, 1806, 1893, 1894,
       1715, 1716, 1718, 1805, 1806, 1808, 1894, 1717, 1753, 1803, 1807,
       1844, 1892, 1895, 1718, 1720, 1806, 1808, 1810, 1894, 1896, 1719,
       1754, 1758, 1760, 1809, 1849, 1851, 1720, 1759, 1808, 1810, 1850,
       1896, 1897, 1721, 1763, 1769, 1811, 1853, 1859, 1898, 1722, 1723,
       1724, 1812, 1813, 1814, 1899, 1723, 1767, 1812, 1813, 1856, 1899,
       1900, 1724, 1725, 1812, 1814, 1815, 1899, 1901, 1725, 1726, 1814,
       1815, 1816, 1901, 1902, 1726, 1727, 1815, 1816, 1817, 1902, 1903,
       1727, 1768, 1816, 1817, 1857, 1903, 1904, 1676, 1773, 1774, 1818,
       1862, 1864, 1905, 1728, 1764, 1774, 1819, 1854, 1863, 1906, 1729,
       1777, 1820, 1822, 1867, 1907, 1908, 1730, 1775, 1781, 1821, 1865,
       1871, 1909, 1729, 1782, 1820, 1822, 1872, 1907, 1910, 1731, 1783,
       1791, 1823, 1873, 1879, 1911, 1732, 1789, 1792, 1824, 1876, 1880,
       1914, 1733, 1793, 1794, 1825, 1881, 1882, 1915, 1735, 1795, 1799,
       1826, 1840, 1883, 1916, 1736, 1737, 1827, 1828, 1829, 1917, 1918,
       1736, 1796, 1827, 1828, 1884, 1917, 1737, 1795, 1827, 1829, 1883,
       1918, 1919, 1740, 1741, 1797, 1830, 1831, 1885, 1920, 1741, 1742,
       1830, 1831, 1832, 1920, 1921, 1742, 1749, 1831, 1832, 1841, 1921,
       1922, 1743, 1798, 1833, 1834, 1886, 1923, 1924, 1743, 1744, 1833,
       1834, 1835, 1923, 1925, 1744, 1797, 1834, 1835, 1885, 1925, 1926,
       1745, 1798, 1836, 1837, 1886, 1927, 1928, 1745, 1746, 1836, 1837,
       1838, 1928, 1929, 1746, 1747, 1837, 1838, 1839, 1887, 1929, 1747,
       1748, 1838, 1839, 1887, 1748, 1799, 1826, 1840, 1887, 1916, 1930,
       1749, 1768, 1832, 1841, 1857, 1922, 1931, 1751, 1778, 1802, 1842,
       1868, 1890, 1932, 1752, 1753, 1800, 1843, 1844, 1888, 1933, 1753,
       1807, 1843, 1844, 1895, 1933, 1934, 1755, 1756, 1845, 1846, 1847,
       1935, 1936, 1755, 1757, 1845, 1846, 1848, 1936, 1937, 1756, 1759,
       1845, 1847, 1850, 1935, 1938, 1757, 1758, 1846, 1848, 1849, 1937,
       1939, 1758, 1809, 1848, 1849, 1851, 1939, 1940, 1759, 1810, 1847,
       1850, 1897, 1938, 1941, 1760, 1761, 1809, 1849, 1851, 1852, 1940,
       1761, 1762, 1851, 1852, 1853, 1940, 1942, 1762, 1763, 1811, 1852,
       1853, 1898, 1942, 1764, 1765, 1766, 1819, 1854, 1855, 1906, 1943,
       1766, 1767, 1854, 1855, 1856, 1943, 1944, 1767, 1813, 1855, 1856,
       1900, 1944, 1945, 1768, 1817, 1841, 1857, 1904, 1931, 1946, 1769,
       1770, 1858, 1859, 1860, 1947, 1948, 1769, 1811, 1858, 1859, 1898,
       1947, 1949, 1770, 1771, 1858, 1860, 1861, 1948, 1950, 1771, 1772,
       1860, 1861, 1862, 1950, 1951, 1993, 1772, 1773, 1818, 1861, 1862,
       1905, 1951, 1774, 1819, 1863, 1864, 1906, 1952, 1953, 1774, 1818,
       1863, 1864, 1905, 1953, 1954, 1775, 1776, 1821, 1865, 1866, 1909,
       1955, 1776, 1777, 1865, 1866, 1867, 1955, 1956, 1777, 1820, 1866,
       1867, 1908, 1956, 1957, 1778, 1779, 1842, 1868, 1869, 1932, 1958,
       1779, 1780, 1868, 1869, 1870, 1958, 1959, 1780, 1781, 1869, 1870,
       1871, 1959, 1960, 1781, 1821, 1870, 1871, 1909, 1960, 1961, 1782,
       1783, 1822, 1872, 1873, 1910, 1962, 1783, 1823, 1872, 1873, 1911,
       1962, 1963, 1784, 1786, 1874, 1875, 1912, 1964, 1965, 1784, 1785,
       1874, 1875, 1913, 1964, 1966, 1785, 1789, 1824, 1876, 1913, 1914,
       1967, 1786, 1787, 1788, 1877, 1878, 1912, 1968, 1788, 1790, 1877,
       1878, 1879, 1968, 1790, 1791, 1823, 1878, 1879, 1911, 1968, 1792,
       1793, 1824, 1880, 1881, 1914, 1969, 1793, 1825, 1880, 1881, 1915,
       1969, 1970, 1794, 1796, 1825, 1882, 1884, 1915, 1971, 1795, 1826,
       1829, 1883, 1916, 1919, 1972, 1796, 1828, 1882, 1884, 1917, 1971,
       1973, 1797, 1830, 1835, 1885, 1920, 1926, 1974, 1798, 1833, 1836,
       1886, 1924, 1927, 1975, 1748, 1838, 1839, 1840, 1887, 1929, 1930,
       1800, 1801, 1843, 1888, 1889, 1933, 1976, 1801, 1802, 1888, 1889,
       1890, 1976, 1977, 1802, 1842, 1889, 1890, 1932, 1977, 1978, 1803,
       1804, 1891, 1892, 1893, 1979, 1980, 1803, 1807, 1891, 1892, 1895,
       1979, 1981, 1804, 1805, 1891, 1893, 1894, 1980, 1982, 1805, 1806,
       1808, 1893, 1894, 1896, 1982, 1807, 1844, 1892, 1895, 1934, 1981,
       1983, 1808, 1810, 1894, 1896, 1897, 1982, 1984, 1810, 1850, 1896,
       1897, 1941, 1984, 1985, 1811, 1853, 1859, 1898, 1942, 1949, 1986,
       1812, 1813, 1814, 1899, 1900, 1901, 1987, 1813, 1856, 1899, 1900,
       1945, 1987, 1988, 1814, 1815, 1899, 1901, 1902, 1987, 1989, 1815,
       1816, 1901, 1902, 1903, 1989, 1990, 1816, 1817, 1902, 1903, 1904,
       1990, 1991, 1817, 1857, 1903, 1904, 1946, 1991, 1992, 1818, 1862,
       1864, 1905, 1951, 1954, 1994, 1819, 1854, 1863, 1906, 1943, 1952,
       1995, 1820, 1822, 1907, 1908, 1910, 1996, 1997, 1820, 1867, 1907,
       1908, 1957, 1997, 1998, 1821, 1865, 1871, 1909, 1955, 1961, 1999,
       1822, 1872, 1907, 1910, 1962, 1996, 2000, 1823, 1873, 1879, 1911,
       1963, 1968, 2001, 1786, 1874, 1877, 1912, 1965, 1968, 2001, 1785,
       1875, 1876, 1913, 1966, 1967, 2002, 1824, 1876, 1880, 1914, 1967,
       1969, 2003, 1825, 1881, 1882, 1915, 1970, 1971, 2004, 1826, 1840,
       1883, 1916, 1930, 1972, 2005, 1827, 1828, 1884, 1917, 1918, 1973,
       2006, 1827, 1829, 1917, 1918, 1919, 2006, 2007, 1829, 1883, 1918,
       1919, 1972, 2007, 2008, 1830, 1831, 1885, 1920, 1921, 1974, 2009,
       1831, 1832, 1920, 1921, 1922, 2009, 2010, 1832, 1841, 1921, 1922,
       1931, 2010, 1833, 1834, 1923, 1924, 1925, 2011, 2012, 1833, 1886,
       1923, 1924, 1975, 2011, 2013, 1834, 1835, 1923, 1925, 1926, 2012,
       2014, 1835, 1885, 1925, 1926, 1974, 2014, 2015, 1836, 1886, 1927,
       1928, 1975, 2016, 2017, 1836, 1837, 1927, 1928, 1929, 2016, 2018,
       1837, 1838, 1887, 1928, 1929, 1930, 2018, 1840, 1887, 1916, 1929,
       1930, 2005, 2018, 1841, 1857, 1922, 1931, 1946, 2010, 2019, 1842,
       1868, 1890, 1932, 1958, 1978, 2020, 1843, 1844, 1888, 1933, 1934,
       1976, 2021, 1844, 1895, 1933, 1934, 1983, 2021, 2022, 1845, 1847,
       1935, 1936, 1938, 2023, 2024, 1845, 1846, 1935, 1936, 1937, 2024,
       2025, 1846, 1848, 1936, 1937, 1939, 2025, 2026, 1847, 1850, 1935,
       1938, 1941, 2023, 2027, 1848, 1849, 1937, 1939, 1940, 2026, 2028,
       1849, 1851, 1852, 1939, 1940, 1942, 2028, 1850, 1897, 1938, 1941,
       1985, 2027, 2029, 1852, 1853, 1898, 1940, 1942, 1986, 2028, 1854,
       1855, 1906, 1943, 1944, 1995, 2030, 1855, 1856, 1943, 1944, 1945,
       2030, 2031, 1856, 1900, 1944, 1945, 1988, 2031, 2032, 2073, 1857,
       1904, 1931, 1946, 1992, 2019, 2033, 1858, 1859, 1947, 1948, 1949,
       2034, 2035, 1858, 1860, 1947, 1948, 1950, 2034, 2036, 1859, 1898,
       1947, 1949, 1986, 2035, 2037, 1860, 1861, 1948, 1950, 1993, 2036,
       2038, 1861, 1862, 1905, 1951, 1993, 1994, 2039, 1863, 1906, 1952,
       1953, 1995, 2040, 2041, 1863, 1864, 1952, 1953, 1954, 2040, 2042,
       1864, 1905, 1953, 1954, 1994, 2042, 2043, 1865, 1866, 1909, 1955,
       1956, 1999, 2044, 1866, 1867, 1955, 1956, 1957, 2044, 2045, 1867,
       1908, 1956, 1957, 1998, 2045, 2046, 1868, 1869, 1932, 1958, 1959,
       2020, 2047, 1869, 1870, 1958, 1959, 1960, 2047, 2048, 1870, 1871,
       1959, 1960, 1961, 2048, 2049, 1871, 1909, 1960, 1961, 1999, 2049,
       2050, 1872, 1873, 1910, 1962, 1963, 2000, 2051, 1873, 1911, 1962,
       1963, 2001, 2051, 2052, 1874, 1875, 1964, 1965, 1966, 2053, 2054,
       1874, 1912, 1964, 1965, 2001, 2052, 2054, 1875, 1913, 1964, 1966,
       2002, 2053, 2055, 1876, 1913, 1914, 1967, 2002, 2003, 2056, 1877,
       1878, 1879, 1911, 1912, 1968, 2001, 1880, 1881, 1914, 1969, 1970,
       2003, 2057, 1881, 1915, 1969, 1970, 2004, 2057, 2058, 1882, 1884,
       1915, 1971, 1973, 2004, 1883, 1916, 1919, 1972, 2005, 2008, 2059,
       1884, 1917, 1971, 1973, 2004, 2006, 1885, 1920, 1926, 1974, 2009,
       2015, 2060, 1886, 1924, 1927, 1975, 2013, 2017, 2061, 1888, 1889,
       1933, 1976, 1977, 2021, 2062, 1889, 1890, 1976, 1977, 1978, 2062,
       1890, 1932, 1977, 1978, 2020, 2062, 2063, 1891, 1892, 1979, 1980,
       1981, 2064, 2065, 1891, 1893, 1979, 1980, 1982, 2064, 2066, 1892,
       1895, 1979, 1981, 1983, 2065, 2067, 1893, 1894, 1896, 1980, 1982,
       1984, 2066, 1895, 1934, 1981, 1983, 2022, 2067, 2068, 1896, 1897,
       1982, 1984, 1985, 2066, 2069, 1897, 1941, 1984, 1985, 2029, 2069,
       2070, 1898, 1942, 1949, 1986, 2028, 2037, 2071, 1899, 1900, 1901,
       1987, 1988, 1989, 2072, 1900, 1945, 1987, 1988, 2032, 2072, 2074,
       1901, 1902, 1987, 1989, 1990, 2072, 2075, 1902, 1903, 1989, 1990,
       1991, 2075, 2076, 1903, 1904, 1990, 1991, 1992, 2076, 2077, 1904,
       1946, 1991, 1992, 2033, 2077, 2078, 1861, 1950, 1951, 1993, 2038,
       2039, 2079, 1905, 1951, 1954, 1994, 2039, 2043, 2080, 1906, 1943,
       1952, 1995, 2030, 2041, 2081, 1907, 1910, 1996, 1997, 2000, 2082,
       2083, 1907, 1908, 1996, 1997, 1998, 2082, 2084, 1908, 1957, 1997,
       1998, 2046, 2084, 2085, 1909, 1955, 1961, 1999, 2044, 2050, 2086,
       1910, 1962, 1996, 2000, 2051, 2083, 2087, 1911, 1912, 1963, 1965,
       1968, 2001, 2052, 1913, 1966, 1967, 2002, 2055, 2056, 2088, 1914,
       1967, 1969, 2003, 2056, 2057, 2089, 1915, 1970, 1971, 1973, 2004,
       2006, 2058, 1916, 1930, 1972, 2005, 2018, 2059, 2090, 1917, 1918,
       1973, 2004, 2006, 2007, 2058, 1918, 1919, 2006, 2007, 2008, 2058,
       2091, 1919, 1972, 2007, 2008, 2059, 2091, 2092, 1920, 1921, 1974,
       2009, 2010, 2060, 2093, 1921, 1922, 1931, 2009, 2010, 2019, 2093,
       1923, 1924, 2011, 2012, 2013, 2094, 2095, 1923, 1925, 2011, 2012,
       2014, 2094, 2096, 1924, 1975, 2011, 2013, 2061, 2095, 2097, 1925,
       1926, 2012, 2014, 2015, 2096, 2098, 1926, 1974, 2014, 2015, 2060,
       2098, 2099, 1927, 1928, 2016, 2017, 2018, 2090, 2100, 1927, 1975,
       2016, 2017, 2061, 2100, 2101, 1928, 1929, 1930, 2005, 2016, 2018,
       2090, 1931, 1946, 2010, 2019, 2033, 2093, 2102, 2142, 1932, 1958,
       1978, 2020, 2047, 2063, 2103, 1933, 1934, 1976, 2021, 2022, 2062,
       2104, 1934, 1983, 2021, 2022, 2068, 2104, 2105, 1935, 1938, 2023,
       2024, 2027, 2106, 2107, 1935, 1936, 2023, 2024, 2025, 2106, 2108,
       1936, 1937, 2024, 2025, 2026, 2108, 2109, 1937, 1939, 2025, 2026,
       2028, 2071, 2109, 1938, 1941, 2023, 2027, 2029, 2107, 2110, 1939,
       1940, 1942, 1986, 2026, 2028, 2071, 1941, 1985, 2027, 2029, 2070,
       2110, 2111, 1943, 1944, 1995, 2030, 2031, 2081, 2112, 2152, 1944,
       1945, 2030, 2031, 2073, 2112, 2113, 1945, 1988, 2032, 2073, 2074,
       2114, 1946, 1992, 2019, 2033, 2078, 2102, 2115, 1947, 1948, 2034,
       2035, 2036, 2116, 2117, 1947, 1949, 2034, 2035, 2037, 2116, 2118,
       1948, 1950, 2034, 2036, 2038, 2117, 2119, 1949, 1986, 2035, 2037,
       2071, 2118, 2120, 1950, 1993, 2036, 2038, 2079, 2119, 2121, 1951,
       1993, 1994, 2039, 2079, 2080, 2122, 1952, 1953, 2040, 2041, 2042,
       2123, 2124, 1952, 1995, 2040, 2041, 2081, 2123, 2125, 1953, 1954,
       2040, 2042, 2043, 2124, 2126, 1954, 1994, 2042, 2043, 2080, 2126,
       2127, 1955, 1956, 1999, 2044, 2045, 2086, 2128, 1956, 1957, 2044,
       2045, 2046, 2128, 2129, 1957, 1998, 2045, 2046, 2085, 2129, 2130,
       1958, 1959, 2020, 2047, 2048, 2103, 2131, 1959, 1960, 2047, 2048,
       2049, 2131, 2132, 1960, 1961, 2048, 2049, 2050, 2132, 2133, 1961,
       1999, 2049, 2050, 2086, 2133, 2134, 1962, 1963, 2000, 2051, 2052,
       2087, 1963, 1965, 2001, 2051, 2052, 2054, 2087, 1964, 1966, 2053,
       2054, 2055, 2135, 2136, 1964, 1965, 2052, 2053, 2054, 2087, 2136,
       1966, 2002, 2053, 2055, 2088, 2135, 2137, 1967, 2002, 2003, 2056,
       2088, 2089, 2138, 1969, 1970, 2003, 2057, 2058, 2089, 2091, 1970,
       2004, 2006, 2007, 2057, 2058, 2091, 1972, 2005, 2008, 2059, 2090,
       2092, 2139, 2140, 1974, 2009, 2015, 2060, 2093, 2099, 2141, 1975,
       2013, 2017, 2061, 2097, 2101, 2143, 1976, 1977, 1978, 2021, 2062,
       2063, 2104, 1978, 2020, 2062, 2063, 2103, 2104, 2144, 1979, 1980,
       2064, 2065, 2066, 2145, 2146, 1979, 1981, 2064, 2065, 2067, 2145,
       2147, 1980, 1982, 1984, 2064, 2066, 2069, 2146, 1981, 1983, 2065,
       2067, 2068, 2147, 2148, 1983, 2022, 2067, 2068, 2105, 2148, 2149,
       1984, 1985, 2066, 2069, 2070, 2146, 2150, 1985, 2029, 2069, 2070,
       2111, 2150, 2151, 1986, 2026, 2028, 2037, 2071, 2109, 2120, 1987,
       1988, 1989, 2072, 2074, 2075, 2153, 1945, 2031, 2032, 2073, 2113,
       2114, 2154, 1988, 2032, 2072, 2074, 2114, 2153, 2155, 1989, 1990,
       2072, 2075, 2076, 2153, 2156, 1990, 1991, 2075, 2076, 2077, 2156,
       2157, 1991, 1992, 2076, 2077, 2078, 2157, 2158, 1992, 2033, 2077,
       2078, 2115, 2158, 2159, 1993, 2038, 2039, 2079, 2121, 2122, 2160,
       1994, 2039, 2043, 2080, 2122, 2127, 2161, 1995, 2030, 2041, 2081,
       2125, 2152, 2162, 1996, 1997, 2082, 2083, 2084, 2163, 1996, 2000,
       2082, 2083, 2087, 2163, 2164, 1997, 1998, 2082, 2084, 2085, 2163,
       2165, 1998, 2046, 2084, 2085, 2130, 2165, 2166, 1999, 2044, 2050,
       2086, 2128, 2134, 2167, 2000, 2051, 2052, 2054, 2083, 2087, 2136,
       2164, 2002, 2055, 2056, 2088, 2137, 2138, 2168, 2003, 2056, 2057,
       2089, 2091, 2092, 2138, 2005, 2016, 2018, 2059, 2090, 2100, 2139,
       2169, 2007, 2008, 2057, 2058, 2089, 2091, 2092, 2008, 2059, 2089,
       2091, 2092, 2138, 2140, 2009, 2010, 2019, 2060, 2093, 2141, 2142,
       2011, 2012, 2094, 2095, 2096, 2170, 2171, 2011, 2013, 2094, 2095,
       2097, 2171, 2172, 2012, 2014, 2094, 2096, 2098, 2170, 2173, 2013,
       2061, 2095, 2097, 2143, 2172, 2174, 2014, 2015, 2096, 2098, 2099,
       2173, 2175, 2015, 2060, 2098, 2099, 2141, 2175, 2176, 2016, 2017,
       2090, 2100, 2101, 2169, 2177, 2017, 2061, 2100, 2101, 2143, 2177,
       2178, 2019, 2033, 2102, 2115, 2142, 2179, 2020, 2047, 2063, 2103,
       2131, 2144, 2180, 2021, 2022, 2062, 2063, 2104, 2105, 2144, 2022,
       2068, 2104, 2105, 2144, 2149, 2181, 2023, 2024, 2106, 2107, 2108,
       2182, 2183, 2023, 2027, 2106, 2107, 2110, 2182, 2184, 2024, 2025,
       2106, 2108, 2109, 2183, 2185, 2025, 2026, 2071, 2108, 2109, 2120,
       2185, 2027, 2029, 2107, 2110, 2111, 2184, 2186, 2029, 2070, 2110,
       2111, 2151, 2186, 2187, 2030, 2031, 2112, 2113, 2152, 2188, 2189,
       2031, 2073, 2112, 2113, 2154, 2188, 2190, 2032, 2073, 2074, 2114,
       2154, 2155, 2191, 2192, 2033, 2078, 2102, 2115, 2159, 2179, 2193,
       2034, 2035, 2116, 2117, 2118, 2194, 2195, 2034, 2036, 2116, 2117,
       2119, 2195, 2196, 2035, 2037, 2116, 2118, 2120, 2194, 2197, 2036,
       2038, 2117, 2119, 2121, 2196, 2198, 2037, 2071, 2109, 2118, 2120,
       2185, 2197, 2038, 2079, 2119, 2121, 2160, 2198, 2199, 2039, 2079,
       2080, 2122, 2160, 2161, 2200, 2040, 2041, 2123, 2124, 2125, 2201,
       2202, 2040, 2042, 2123, 2124, 2126, 2201, 2203, 2041, 2081, 2123,
       2125, 2162, 2202, 2204, 2042, 2043, 2124, 2126, 2127, 2203, 2205,
       2043, 2080, 2126, 2127, 2161, 2205, 2206, 2044, 2045, 2086, 2128,
       2129, 2167, 2207, 2045, 2046, 2128, 2129, 2130, 2207, 2208, 2046,
       2085, 2129, 2130, 2166, 2208, 2209, 2047, 2048, 2103, 2131, 2132,
       2180, 2210, 2048, 2049, 2131, 2132, 2133, 2210, 2049, 2050, 2132,
       2133, 2134, 2210, 2211, 2050, 2086, 2133, 2134, 2167, 2211, 2212,
       2053, 2055, 2135, 2136, 2137, 2213, 2214, 2053, 2054, 2087, 2135,
       2136, 2164, 2213, 2055, 2088, 2135, 2137, 2168, 2214, 2215, 2056,
       2088, 2089, 2092, 2138, 2140, 2168, 2059, 2090, 2139, 2140, 2169,
       2216, 2059, 2092, 2138, 2139, 2140, 2168, 2216, 2060, 2093, 2099,
       2141, 2142, 2176, 2217, 2019, 2093, 2102, 2141, 2142, 2179, 2217,
       2061, 2097, 2101, 2143, 2174, 2178, 2218, 2063, 2103, 2104, 2105,
       2144, 2180, 2181, 2064, 2065, 2145, 2146, 2147, 2219, 2220, 2064,
       2066, 2069, 2145, 2146, 2150, 2220, 2065, 2067, 2145, 2147, 2148,
       2219, 2221, 2067, 2068, 2147, 2148, 2149, 2221, 2222, 2068, 2105,
       2148, 2149, 2181, 2222, 2223, 2069, 2070, 2146, 2150, 2151, 2220,
       2224, 2070, 2111, 2150, 2151, 2187, 2224, 2225, 2030, 2081, 2112,
       2152, 2162, 2189, 2072, 2074, 2075, 2153, 2155, 2156, 2226, 2073,
       2113, 2114, 2154, 2190, 2191, 2227, 2074, 2114, 2153, 2155, 2192,
       2226, 2228, 2075, 2076, 2153, 2156, 2157, 2226, 2229, 2076, 2077,
       2156, 2157, 2158, 2229, 2230, 2077, 2078, 2157, 2158, 2159, 2230,
       2231, 2078, 2115, 2158, 2159, 2193, 2231, 2232, 2079, 2121, 2122,
       2160, 2199, 2200, 2233, 2080, 2122, 2127, 2161, 2200, 2206, 2234,
       2081, 2125, 2152, 2162, 2189, 2204, 2235, 2082, 2083, 2084, 2163,
       2164, 2165, 2236, 2083, 2087, 2136, 2163, 2164, 2213, 2236, 2084,
       2085, 2163, 2165, 2166, 2236, 2237, 2085, 2130, 2165, 2166, 2209,
       2237, 2238, 2086, 2128, 2134, 2167, 2207, 2212, 2239, 2088, 2137,
       2138, 2140, 2168, 2215, 2216, 2240, 2090, 2100, 2139, 2169, 2177,
       2216, 2241, 2094, 2096, 2170, 2171, 2173, 2242, 2243, 2094, 2095,
       2170, 2171, 2172, 2243, 2244, 2095, 2097, 2171, 2172, 2174, 2244,
       2245, 2096, 2098, 2170, 2173, 2175, 2242, 2246, 2097, 2143, 2172,
       2174, 2218, 2245, 2247, 2290, 2098, 2099, 2173, 2175, 2176, 2246,
       2248, 2099, 2141, 2175, 2176, 2217, 2248, 2249, 2100, 2101, 2169,
       2177, 2178, 2241, 2250, 2101, 2143, 2177, 2178, 2218, 2250, 2251,
       2102, 2115, 2142, 2179, 2193, 2217, 2252, 2103, 2131, 2144, 2180,
       2181, 2210, 2253, 2105, 2144, 2149, 2180, 2181, 2223, 2253, 2106,
       2107, 2182, 2183, 2184, 2254, 2255, 2106, 2108, 2182, 2183, 2185,
       2255, 2256, 2107, 2110, 2182, 2184, 2186, 2254, 2257, 2108, 2109,
       2120, 2183, 2185, 2197, 2256, 2110, 2111, 2184, 2186, 2187, 2257,
       2258, 2111, 2151, 2186, 2187, 2225, 2258, 2259, 2112, 2113, 2188,
       2189, 2190, 2260, 2261, 2112, 2152, 2162, 2188, 2189, 2235, 2260,
       2113, 2154, 2188, 2190, 2227, 2261, 2262, 2114, 2154, 2191, 2192,
       2227, 2263, 2264, 2114, 2155, 2191, 2192, 2228, 2263, 2265, 2115,
       2159, 2179, 2193, 2232, 2252, 2266, 2116, 2118, 2194, 2195, 2197,
       2267, 2268, 2116, 2117, 2194, 2195, 2196, 2267, 2269, 2117, 2119,
       2195, 2196, 2198, 2269, 2270, 2118, 2120, 2185, 2194, 2197, 2256,
       2268, 2119, 2121, 2196, 2198, 2199, 2270, 2271, 2121, 2160, 2198,
       2199, 2233, 2271, 2272, 2122, 2160, 2161, 2200, 2233, 2234, 2273,
       2123, 2124, 2201, 2202, 2203, 2274, 2275, 2123, 2125, 2201, 2202,
       2204, 2275, 2276, 2124, 2126, 2201, 2203, 2205, 2274, 2277, 2307,
       2125, 2162, 2202, 2204, 2235, 2276, 2278, 2126, 2127, 2203, 2205,
       2206, 2277, 2279, 2127, 2161, 2205, 2206, 2234, 2279, 2280, 2128,
       2129, 2167, 2207, 2208, 2239, 2281, 2129, 2130, 2207, 2208, 2209,
       2281, 2282, 2130, 2166, 2208, 2209, 2238, 2282, 2283, 2131, 2132,
       2133, 2180, 2210, 2211, 2253, 2284, 2133, 2134, 2210, 2211, 2212,
       2284, 2285, 2134, 2167, 2211, 2212, 2239, 2285, 2286, 2135, 2136,
       2164, 2213, 2214, 2236, 2287, 2135, 2137, 2213, 2214, 2215, 2287,
       2288, 2137, 2168, 2214, 2215, 2240, 2288, 2289, 2139, 2140, 2168,
       2169, 2216, 2240, 2241, 2141, 2142, 2176, 2179, 2217, 2249, 2252,
       2143, 2174, 2178, 2218, 2247, 2251, 2291, 2145, 2147, 2219, 2220,
       2221, 2292, 2293, 2145, 2146, 2150, 2219, 2220, 2224, 2293, 2147,
       2148, 2219, 2221, 2222, 2292, 2294, 2148, 2149, 2221, 2222, 2223,
       2294, 2295, 2324, 2149, 2181, 2222, 2223, 2253, 2295, 2150, 2151,
       2220, 2224, 2225, 2293, 2296, 2151, 2187, 2224, 2225, 2259, 2296,
       2297, 2153, 2155, 2156, 2226, 2228, 2229, 2298, 2154, 2190, 2191,
       2227, 2262, 2264, 2299, 2155, 2192, 2226, 2228, 2265, 2298, 2300,
       2332, 2339, 2156, 2157, 2226, 2229, 2230, 2298, 2301, 2157, 2158,
       2229, 2230, 2231, 2301, 2302, 2158, 2159, 2230, 2231, 2232, 2302,
       2303, 2159, 2193, 2231, 2232, 2266, 2303, 2304, 2160, 2199, 2200,
       2233, 2272, 2273, 2305, 2161, 2200, 2206, 2234, 2273, 2280, 2306,
       2162, 2189, 2204, 2235, 2260, 2278, 2308, 2331, 2163, 2164, 2165,
       2213, 2236, 2237, 2287, 2165, 2166, 2236, 2237, 2238, 2287, 2309,
       2166, 2209, 2237, 2238, 2283, 2309, 2310, 2167, 2207, 2212, 2239,
       2281, 2286, 2311, 2168, 2215, 2216, 2240, 2241, 2289, 2312, 2169,
       2177, 2216, 2240, 2241, 2250, 2312, 2170, 2173, 2242, 2243, 2246,
       2313, 2314, 2170, 2171, 2242, 2243, 2244, 2314, 2315, 2171, 2172,
       2243, 2244, 2245, 2315, 2316, 2172, 2174, 2244, 2245, 2290, 2316,
       2173, 2175, 2242, 2246, 2248, 2313, 2317, 2174, 2218, 2247, 2290,
       2291, 2318, 2319, 2175, 2176, 2246, 2248, 2249, 2317, 2320, 2176,
       2217, 2248, 2249, 2252, 2320, 2321, 2177, 2178, 2241, 2250, 2251,
       2312, 2322, 2178, 2218, 2250, 2251, 2291, 2322, 2323, 2366, 2179,
       2193, 2217, 2249, 2252, 2266, 2321, 2180, 2181, 2210, 2223, 2253,
       2284, 2295, 2182, 2184, 2254, 2255, 2257, 2325, 2326, 2182, 2183,
       2254, 2255, 2256, 2325, 2327, 2183, 2185, 2197, 2255, 2256, 2268,
       2327, 2184, 2186, 2254, 2257, 2258, 2326, 2328, 2186, 2187, 2257,
       2258, 2259, 2328, 2329, 2187, 2225, 2258, 2259, 2297, 2329, 2330,
       2188, 2189, 2235, 2260, 2261, 2331, 2333, 2188, 2190, 2260, 2261,
       2262, 2333, 2334, 2190, 2227, 2261, 2262, 2299, 2334, 2335, 2191,
       2192, 2263, 2264, 2265, 2336, 2337, 2191, 2227, 2263, 2264, 2299,
       2336, 2338, 2192, 2228, 2263, 2265, 2300, 2337, 2193, 2232, 2252,
       2266, 2304, 2321, 2340, 2194, 2195, 2267, 2268, 2269, 2341, 2342,
       2194, 2197, 2256, 2267, 2268, 2327, 2342, 2195, 2196, 2267, 2269,
       2270, 2341, 2343, 2196, 2198, 2269, 2270, 2271, 2343, 2344, 2198,
       2199, 2270, 2271, 2272, 2344, 2345, 2199, 2233, 2271, 2272, 2305,
       2345, 2346, 2200, 2233, 2234, 2273, 2305, 2306, 2347, 2201, 2203,
       2274, 2275, 2307, 2348, 2349, 2201, 2202, 2274, 2275, 2276, 2349,
       2350, 2202, 2204, 2275, 2276, 2278, 2350, 2351, 2203, 2205, 2277,
       2279, 2307, 2352, 2353, 2204, 2235, 2276, 2278, 2308, 2351, 2354,
       2205, 2206, 2277, 2279, 2280, 2352, 2355, 2206, 2234, 2279, 2280,
       2306, 2355, 2356, 2207, 2208, 2239, 2281, 2282, 2311, 2357, 2208,
       2209, 2281, 2282, 2283, 2357, 2358, 2209, 2238, 2282, 2283, 2310,
       2358, 2359, 2210, 2211, 2253, 2284, 2285, 2295, 2360, 2211, 2212,
       2284, 2285, 2286, 2360, 2361, 2212, 2239, 2285, 2286, 2311, 2361,
       2362, 2213, 2214, 2236, 2237, 2287, 2288, 2309, 2214, 2215, 2287,
       2288, 2289, 2309, 2363, 2215, 2240, 2288, 2289, 2312, 2363, 2364,
       2174, 2245, 2247, 2290, 2316, 2318, 2365, 2218, 2247, 2251, 2291,
       2319, 2323, 2219, 2221, 2292, 2293, 2294, 2367, 2368, 2219, 2220,
       2224, 2292, 2293, 2296, 2368, 2221, 2222, 2292, 2294, 2324, 2367,
       2369, 2222, 2223, 2253, 2284, 2295, 2324, 2360, 2224, 2225, 2293,
       2296, 2297, 2368, 2370, 2225, 2259, 2296, 2297, 2330, 2370, 2371,
       2226, 2228, 2229, 2298, 2301, 2332, 2372, 2227, 2262, 2264, 2299,
       2335, 2338, 2373, 2228, 2265, 2300, 2337, 2339, 2374, 2375, 2229,
       2230, 2298, 2301, 2302, 2372, 2376, 2230, 2231, 2301, 2302, 2303,
       2376, 2377, 2231, 2232, 2302, 2303, 2304, 2377, 2378, 2232, 2266,
       2303, 2304, 2340, 2378, 2379, 2233, 2272, 2273, 2305, 2346, 2347,
       2380, 2234, 2273, 2280, 2306, 2347, 2356, 2381, 2203, 2274, 2277,
       2307, 2348, 2353, 2235, 2278, 2308, 2331, 2354, 2382, 2237, 2238,
       2287, 2288, 2309, 2310, 2363, 2238, 2283, 2309, 2310, 2359, 2363,
       2383, 2239, 2281, 2286, 2311, 2357, 2362, 2384, 2240, 2241, 2250,
       2289, 2312, 2322, 2364, 2242, 2246, 2313, 2314, 2317, 2385, 2386,
       2242, 2243, 2313, 2314, 2315, 2386, 2387, 2243, 2244, 2314, 2315,
       2316, 2387, 2388, 2244, 2245, 2290, 2315, 2316, 2365, 2388, 2246,
       2248, 2313, 2317, 2320, 2385, 2389, 2247, 2290, 2318, 2319, 2365,
       2390, 2391, 2247, 2291, 2318, 2319, 2323, 2390, 2392, 2248, 2249,
       2317, 2320, 2321, 2389, 2393, 2249, 2252, 2266, 2320, 2321, 2340,
       2393, 2250, 2251, 2312, 2322, 2364, 2366, 2394, 2251, 2291, 2319,
       2323, 2366, 2392, 2395, 2222, 2294, 2295, 2324, 2360, 2369, 2396,
       2254, 2255, 2325, 2326, 2327, 2397, 2398, 2254, 2257, 2325, 2326,
       2328, 2397, 2399, 2255, 2256, 2268, 2325, 2327, 2342, 2398, 2257,
       2258, 2326, 2328, 2329, 2399, 2400, 2258, 2259, 2328, 2329, 2330,
       2400, 2401, 2259, 2297, 2329, 2330, 2371, 2401, 2402, 2235, 2260,
       2308, 2331, 2333, 2382, 2403, 2228, 2298, 2332, 2339, 2372, 2404,
       2260, 2261, 2331, 2333, 2334, 2403, 2405, 2261, 2262, 2333, 2334,
       2335, 2405, 2262, 2299, 2334, 2335, 2373, 2405, 2406, 2263, 2264,
       2336, 2337, 2338, 2407, 2408, 2263, 2265, 2300, 2336, 2337, 2375,
       2407, 2264, 2299, 2336, 2338, 2373, 2408, 2409, 2228, 2300, 2332,
       2339, 2374, 2404, 2410, 2266, 2304, 2321, 2340, 2379, 2393, 2411,
       2267, 2269, 2341, 2342, 2343, 2412, 2413, 2267, 2268, 2327, 2341,
       2342, 2398, 2413, 2450, 2269, 2270, 2341, 2343, 2344, 2412, 2414,
       2270, 2271, 2343, 2344, 2345, 2414, 2415, 2271, 2272, 2344, 2345,
       2346, 2415, 2416, 2272, 2305, 2345, 2346, 2380, 2416, 2417, 2273,
       2305, 2306, 2347, 2380, 2381, 2418, 2274, 2307, 2348, 2349, 2353,
       2419, 2420, 2274, 2275, 2348, 2349, 2350, 2420, 2421, 2275, 2276,
       2349, 2350, 2351, 2421, 2422, 2276, 2278, 2350, 2351, 2354, 2422,
       2423, 2277, 2279, 2352, 2353, 2355, 2424, 2425, 2277, 2307, 2348,
       2352, 2353, 2419, 2425, 2278, 2308, 2351, 2354, 2382, 2423, 2426,
       2279, 2280, 2352, 2355, 2356, 2424, 2427, 2280, 2306, 2355, 2356,
       2381, 2427, 2428, 2281, 2282, 2311, 2357, 2358, 2384, 2429, 2282,
       2283, 2357, 2358, 2359, 2429, 2430, 2283, 2310, 2358, 2359, 2383,
       2430, 2431, 2284, 2285, 2295, 2324, 2360, 2361, 2396, 2285, 2286,
       2360, 2361, 2362, 2396, 2432, 2286, 2311, 2361, 2362, 2384, 2432,
       2433, 2288, 2289, 2309, 2310, 2363, 2364, 2383, 2289, 2312, 2322,
       2363, 2364, 2383, 2394, 2290, 2316, 2318, 2365, 2388, 2391, 2434,
       2251, 2322, 2323, 2366, 2394, 2395, 2435, 2292, 2294, 2367, 2368,
       2369, 2436, 2437, 2292, 2293, 2296, 2367, 2368, 2370, 2436, 2294,
       2324, 2367, 2369, 2396, 2437, 2438, 2296, 2297, 2368, 2370, 2371,
       2436, 2439, 2297, 2330, 2370, 2371, 2402, 2439, 2440, 2298, 2301,
       2332, 2372, 2376, 2404, 2441, 2299, 2335, 2338, 2373, 2406, 2409,
       2442, 2300, 2339, 2374, 2375, 2410, 2443, 2444, 2300, 2337, 2374,
       2375, 2407, 2444, 2445, 2301, 2302, 2372, 2376, 2377, 2441, 2446,
       2302, 2303, 2376, 2377, 2378, 2446, 2447, 2303, 2304, 2377, 2378,
       2379, 2447, 2448, 2304, 2340, 2378, 2379, 2411, 2448, 2449, 2305,
       2346, 2347, 2380, 2417, 2418, 2451, 2306, 2347, 2356, 2381, 2418,
       2428, 2452, 2454, 2308, 2331, 2354, 2382, 2403, 2426, 2453, 2310,
       2359, 2363, 2364, 2383, 2394, 2431, 2455, 2311, 2357, 2362, 2384,
       2429, 2433, 2456, 2313, 2317, 2385, 2386, 2389, 2457, 2458, 2313,
       2314, 2385, 2386, 2387, 2458, 2459, 2314, 2315, 2386, 2387, 2388,
       2459, 2460, 2315, 2316, 2365, 2387, 2388, 2434, 2460, 2317, 2320,
       2385, 2389, 2393, 2457, 2461, 2318, 2319, 2390, 2391, 2392, 2462,
       2463, 2318, 2365, 2390, 2391, 2434, 2463, 2464, 2319, 2323, 2390,
       2392, 2395, 2462, 2465, 2320, 2321, 2340, 2389, 2393, 2411, 2461,
       2322, 2364, 2366, 2383, 2394, 2435, 2455, 2323, 2366, 2392, 2395,
       2435, 2465, 2466, 2324, 2360, 2361, 2369, 2396, 2432, 2438, 2325,
       2326, 2397, 2398, 2399, 2467, 2468, 2325, 2327, 2342, 2397, 2398,
       2450, 2468, 2326, 2328, 2397, 2399, 2400, 2467, 2469, 2328, 2329,
       2399, 2400, 2401, 2469, 2470, 2329, 2330, 2400, 2401, 2402, 2470,
       2471, 2330, 2371, 2401, 2402, 2440, 2471, 2472, 2331, 2333, 2382,
       2403, 2405, 2453, 2473, 2332, 2339, 2372, 2404, 2410, 2441, 2474,
       2333, 2334, 2335, 2403, 2405, 2406, 2473, 2335, 2373, 2405, 2406,
       2442, 2473, 2475, 2336, 2337, 2375, 2407, 2408, 2445, 2476, 2336,
       2338, 2407, 2408, 2409, 2476, 2477, 2338, 2373, 2408, 2409, 2442,
       2477, 2478, 2339, 2374, 2404, 2410, 2443, 2474, 2479, 2340, 2379,
       2393, 2411, 2449, 2461, 2480, 2341, 2343, 2412, 2413, 2414, 2481,
       2482, 2341, 2342, 2412, 2413, 2450, 2481, 2483, 2343, 2344, 2412,
       2414, 2415, 2482, 2484, 2344, 2345, 2414, 2415, 2416, 2484, 2485,
       2519, 2345, 2346, 2415, 2416, 2417, 2485, 2486, 2346, 2380, 2416,
       2417, 2451, 2486, 2487, 2520, 2347, 2380, 2381, 2418, 2451, 2452,
       2488, 2348, 2353, 2419, 2420, 2425, 2489, 2490, 2348, 2349, 2419,
       2420, 2421, 2490, 2491, 2349, 2350, 2420, 2421, 2422, 2491, 2492,
       2350, 2351, 2421, 2422, 2423, 2492, 2493, 2351, 2354, 2422, 2423,
       2426, 2493, 2494, 2352, 2355, 2424, 2425, 2427, 2495, 2496, 2352,
       2353, 2419, 2424, 2425, 2489, 2495, 2354, 2382, 2423, 2426, 2453,
       2494, 2497, 2355, 2356, 2424, 2427, 2428, 2496, 2498, 2356, 2381,
       2427, 2428, 2454, 2498, 2357, 2358, 2384, 2429, 2430, 2456, 2499,
       2358, 2359, 2429, 2430, 2431, 2499, 2500, 2359, 2383, 2430, 2431,
       2455, 2500, 2501, 2361, 2362, 2396, 2432, 2433, 2438, 2502, 2362,
       2384, 2432, 2433, 2456, 2502, 2503, 2365, 2388, 2391, 2434, 2460,
       2464, 2504, 2366, 2394, 2395, 2435, 2455, 2466, 2501, 2367, 2368,
       2370, 2436, 2437, 2439, 2505, 2367, 2369, 2436, 2437, 2438, 2505,
       2506, 2369, 2396, 2432, 2437, 2438, 2502, 2506, 2370, 2371, 2436,
       2439, 2440, 2505, 2507, 2371, 2402, 2439, 2440, 2472, 2507, 2508,
       2372, 2376, 2404, 2441, 2446, 2474, 2509, 2373, 2406, 2409, 2442,
       2475, 2478, 2510, 2374, 2410, 2443, 2444, 2479, 2511, 2512, 2374,
       2375, 2443, 2444, 2445, 2511, 2513, 2375, 2407, 2444, 2445, 2476,
       2513, 2514, 2376, 2377, 2441, 2446, 2447, 2509, 2515, 2549, 2377,
       2378, 2446, 2447, 2448, 2515, 2378, 2379, 2447, 2448, 2449, 2515,
       2516, 2379, 2411, 2448, 2449, 2480, 2516, 2517, 2342, 2398, 2413,
       2450, 2468, 2483, 2518, 2380, 2417, 2418, 2451, 2487, 2488, 2521,
       2381, 2418, 2452, 2454, 2488, 2522, 2523, 2382, 2403, 2426, 2453,
       2473, 2497, 2524, 2381, 2428, 2452, 2454, 2498, 2523, 2525, 2383,
       2394, 2431, 2435, 2455, 2501, 2384, 2429, 2433, 2456, 2499, 2503,
       2526, 2385, 2389, 2457, 2458, 2461, 2527, 2528, 2385, 2386, 2457,
       2458, 2459, 2528, 2529, 2386, 2387, 2458, 2459, 2460, 2529, 2530,
       2387, 2388, 2434, 2459, 2460, 2504, 2530, 2389, 2393, 2411, 2457,
       2461, 2480, 2527, 2390, 2392, 2462, 2463, 2465, 2531, 2532, 2390,
       2391, 2462, 2463, 2464, 2531, 2533, 2391, 2434, 2463, 2464, 2504,
       2533, 2534, 2392, 2395, 2462, 2465, 2466, 2532, 2535, 2395, 2435,
       2465, 2466, 2501, 2535, 2536, 2397, 2399, 2467, 2468, 2469, 2537,
       2538, 2397, 2398, 2450, 2467, 2468, 2518, 2538, 2399, 2400, 2467,
       2469, 2470, 2537, 2539, 2400, 2401, 2469, 2470, 2471, 2539, 2540,
       2401, 2402, 2470, 2471, 2472, 2540, 2541, 2402, 2440, 2471, 2472,
       2508, 2541, 2542, 2403, 2405, 2406, 2453, 2473, 2475, 2524, 2404,
       2410, 2441, 2474, 2479, 2509, 2543, 2406, 2442, 2473, 2475, 2510,
       2524, 2544, 2407, 2408, 2445, 2476, 2477, 2514, 2545, 2408, 2409,
       2476, 2477, 2478, 2545, 2546, 2409, 2442, 2477, 2478, 2510, 2546,
       2547, 2410, 2443, 2474, 2479, 2512, 2543, 2548, 2411, 2449, 2461,
       2480, 2517, 2527, 2550, 2412, 2413, 2481, 2482, 2483, 2551, 2552,
       2412, 2414, 2481, 2482, 2484, 2552, 2553, 2413, 2450, 2481, 2483,
       2518, 2551, 2554, 2414, 2415, 2482, 2484, 2519, 2553, 2555, 2415,
       2416, 2485, 2486, 2519, 2556, 2557, 2416, 2417, 2485, 2486, 2520,
       2557, 2558, 2417, 2451, 2487, 2520, 2521, 2559, 2560, 2418, 2451,
       2452, 2488, 2521, 2522, 2561, 2419, 2425, 2489, 2490, 2495, 2562,
       2563, 2419, 2420, 2489, 2490, 2491, 2563, 2564, 2420, 2421, 2490,
       2491, 2492, 2564, 2565, 2421, 2422, 2491, 2492, 2493, 2565, 2566,
       2422, 2423, 2492, 2493, 2494, 2566, 2567, 2423, 2426, 2493, 2494,
       2497, 2567, 2568, 2424, 2425, 2489, 2495, 2496, 2562, 2569, 2424,
       2427, 2495, 2496, 2498, 2569, 2570, 2426, 2453, 2494, 2497, 2524,
       2568, 2571, 2427, 2428, 2454, 2496, 2498, 2525, 2570, 2429, 2430,
       2456, 2499, 2500, 2526, 2572, 2430, 2431, 2499, 2500, 2501, 2536,
       2572, 2431, 2435, 2455, 2466, 2500, 2501, 2536, 2432, 2433, 2438,
       2502, 2503, 2506, 2573, 2433, 2456, 2502, 2503, 2526, 2573, 2574,
       2434, 2460, 2464, 2504, 2530, 2534, 2575, 2436, 2437, 2439, 2505,
       2506, 2507, 2576, 2437, 2438, 2502, 2505, 2506, 2573, 2576, 2439,
       2440, 2505, 2507, 2508, 2576, 2577, 2440, 2472, 2507, 2508, 2542,
       2577, 2578, 2441, 2446, 2474, 2509, 2543, 2549, 2579, 2442, 2475,
       2478, 2510, 2544, 2547, 2580, 2443, 2444, 2511, 2512, 2513, 2581,
       2582, 2443, 2479, 2511, 2512, 2548, 2582, 2583, 2444, 2445, 2511,
       2513, 2514, 2581, 2584, 2445, 2476, 2513, 2514, 2545, 2584, 2585,
       2618, 2446, 2447, 2448, 2515, 2516, 2549, 2586, 2448, 2449, 2515,
       2516, 2517, 2586, 2587, 2449, 2480, 2516, 2517, 2550, 2587, 2588,
       2450, 2468, 2483, 2518, 2538, 2554, 2589, 2415, 2484, 2485, 2519,
       2555, 2556, 2590, 2417, 2486, 2487, 2520, 2558, 2559, 2591, 2451,
       2487, 2488, 2521, 2560, 2561, 2592, 2452, 2488, 2522, 2523, 2561,
       2593, 2594, 2452, 2454, 2522, 2523, 2525, 2594, 2595, 2453, 2473,
       2475, 2497, 2524, 2544, 2571, 2454, 2498, 2523, 2525, 2570, 2595,
       2596, 2456, 2499, 2503, 2526, 2572, 2574, 2597, 2457, 2461, 2480,
       2527, 2528, 2550, 2598, 2457, 2458, 2527, 2528, 2529, 2598, 2599,
       2458, 2459, 2528, 2529, 2530, 2599, 2600, 2459, 2460, 2504, 2529,
       2530, 2575, 2600, 2462, 2463, 2531, 2532, 2533, 2601, 2602, 2462,
       2465, 2531, 2532, 2535, 2601, 2603, 2463, 2464, 2531, 2533, 2534,
       2602, 2604, 2464, 2504, 2533, 2534, 2575, 2604, 2605, 2465, 2466,
       2532, 2535, 2536, 2603, 2606, 2466, 2500, 2501, 2535, 2536, 2572,
       2606, 2467, 2469, 2537, 2538, 2539, 2607, 2608, 2467, 2468, 2518,
       2537, 2538, 2589, 2608, 2469, 2470, 2537, 2539, 2540, 2607, 2609,
       2470, 2471, 2539, 2540, 2541, 2609, 2610, 2471, 2472, 2540, 2541,
       2542, 2610, 2611, 2472, 2508, 2541, 2542, 2578, 2611, 2612, 2474,
       2479, 2509, 2543, 2548, 2579, 2613, 2475, 2510, 2524, 2544, 2571,
       2580, 2614, 2476, 2477, 2514, 2545, 2546, 2585, 2477, 2478, 2545,
       2546, 2547, 2585, 2615, 2478, 2510, 2546, 2547, 2580, 2615, 2616,
       2479, 2512, 2543, 2548, 2583, 2613, 2617, 2446, 2509, 2515, 2549,
       2579, 2586, 2619, 2480, 2517, 2527, 2550, 2588, 2598, 2620, 2481,
       2483, 2551, 2552, 2554, 2621, 2622, 2481, 2482, 2551, 2552, 2553,
       2622, 2623, 2482, 2484, 2552, 2553, 2555, 2623, 2624, 2483, 2518,
       2551, 2554, 2589, 2621, 2625, 2484, 2519, 2553, 2555, 2590, 2624,
       2626, 2660, 2485, 2519, 2556, 2557, 2590, 2627, 2628, 2485, 2486,
       2556, 2557, 2558, 2627, 2629, 2486, 2520, 2557, 2558, 2591, 2629,
       2630, 2487, 2520, 2559, 2560, 2591, 2631, 2632, 2487, 2521, 2559,
       2560, 2592, 2631, 2633, 2488, 2521, 2522, 2561, 2592, 2593, 2634,
       2489, 2495, 2562, 2563, 2569, 2635, 2636, 2489, 2490, 2562, 2563,
       2564, 2635, 2637, 2490, 2491, 2563, 2564, 2565, 2637, 2638, 2491,
       2492, 2564, 2565, 2566, 2638, 2639, 2492, 2493, 2565, 2566, 2567,
       2639, 2640, 2493, 2494, 2566, 2567, 2568, 2640, 2641, 2494, 2497,
       2567, 2568, 2571, 2641, 2642, 2495, 2496, 2562, 2569, 2570, 2636,
       2643, 2496, 2498, 2525, 2569, 2570, 2596, 2643, 2497, 2524, 2544,
       2568, 2571, 2614, 2642, 2499, 2500, 2526, 2536, 2572, 2597, 2606,
       2502, 2503, 2506, 2573, 2574, 2576, 2644, 2647, 2503, 2526, 2573,
       2574, 2597, 2644, 2645, 2504, 2530, 2534, 2575, 2600, 2605, 2646,
       2505, 2506, 2507, 2573, 2576, 2577, 2647, 2507, 2508, 2576, 2577,
       2578, 2647, 2648, 2508, 2542, 2577, 2578, 2612, 2648, 2649, 2509,
       2543, 2549, 2579, 2613, 2619, 2650, 2510, 2544, 2547, 2580, 2614,
       2616, 2651, 2511, 2513, 2581, 2582, 2584, 2652, 2653, 2511, 2512,
       2581, 2582, 2583, 2653, 2654, 2512, 2548, 2582, 2583, 2617, 2654,
       2655, 2513, 2514, 2581, 2584, 2618, 2652, 2656, 2514, 2545, 2546,
       2585, 2615, 2618, 2515, 2516, 2549, 2586, 2587, 2619, 2657, 2516,
       2517, 2586, 2587, 2588, 2657, 2658, 2517, 2550, 2587, 2588, 2620,
       2658, 2659, 2518, 2538, 2554, 2589, 2608, 2625, 2661, 2519, 2555,
       2556, 2590, 2626, 2628, 2662, 2520, 2558, 2559, 2591, 2630, 2632,
       2663, 2521, 2560, 2561, 2592, 2633, 2634, 2664, 2522, 2561, 2593,
       2594, 2634, 2665, 2666, 2522, 2523, 2593, 2594, 2595, 2666, 2667,
       2523, 2525, 2594, 2595, 2596, 2667, 2668, 2525, 2570, 2595, 2596,
       2643, 2668, 2669, 2526, 2572, 2574, 2597, 2606, 2645, 2670, 2527,
       2528, 2550, 2598, 2599, 2620, 2671, 2528, 2529, 2598, 2599, 2600,
       2671, 2672, 2529, 2530, 2575, 2599, 2600, 2646, 2672, 2531, 2532,
       2601, 2602, 2603, 2673, 2674, 2531, 2533, 2601, 2602, 2604, 2673,
       2675, 2532, 2535, 2601, 2603, 2606, 2670, 2674, 2533, 2534, 2602,
       2604, 2605, 2675, 2676, 2534, 2575, 2604, 2605, 2646, 2676, 2677,
       2535, 2536, 2572, 2597, 2603, 2606, 2670, 2537, 2539, 2607, 2608,
       2609, 2678, 2679, 2537, 2538, 2589, 2607, 2608, 2661, 2679, 2539,
       2540, 2607, 2609, 2610, 2678, 2680, 2540, 2541, 2609, 2610, 2611,
       2680, 2681, 2541, 2542, 2610, 2611, 2612, 2681, 2682, 2542, 2578,
       2611, 2612, 2649, 2682, 2683, 2543, 2548, 2579, 2613, 2617, 2650,
       2684, 2544, 2571, 2580, 2614, 2642, 2651, 2685, 2546, 2547, 2585,
       2615, 2616, 2618, 2686, 2547, 2580, 2615, 2616, 2651, 2686, 2687,
       2548, 2583, 2613, 2617, 2655, 2684, 2688, 2514, 2584, 2585, 2615,
       2618, 2656, 2686, 2549, 2579, 2586, 2619, 2650, 2657, 2689, 2550,
       2588, 2598, 2620, 2659, 2671, 2690, 2551, 2554, 2621, 2622, 2625,
       2691, 2692, 2551, 2552, 2621, 2622, 2623, 2692, 2693, 2552, 2553,
       2622, 2623, 2624, 2693, 2694, 2553, 2555, 2623, 2624, 2660, 2694,
       2695, 2734, 2554, 2589, 2621, 2625, 2661, 2691, 2696, 2555, 2590,
       2626, 2660, 2662, 2697, 2698, 2556, 2557, 2627, 2628, 2629, 2699,
       2700, 2556, 2590, 2627, 2628, 2662, 2699, 2701, 2557, 2558, 2627,
       2629, 2630, 2700, 2702, 2558, 2591, 2629, 2630, 2663, 2702, 2703,
       2559, 2560, 2631, 2632, 2633, 2704, 2705, 2559, 2591, 2631, 2632,
       2663, 2704, 2706, 2560, 2592, 2631, 2633, 2664, 2705, 2707, 2561,
       2592, 2593, 2634, 2664, 2665, 2708, 2709, 2562, 2563, 2635, 2636,
       2637, 2710, 2711, 2562, 2569, 2635, 2636, 2643, 2710, 2712, 2563,
       2564, 2635, 2637, 2638, 2711, 2713, 2564, 2565, 2637, 2638, 2639,
       2713, 2714, 2565, 2566, 2638, 2639, 2640, 2714, 2715, 2566, 2567,
       2639, 2640, 2641, 2715, 2716, 2567, 2568, 2640, 2641, 2642, 2716,
       2717, 2568, 2571, 2614, 2641, 2642, 2685, 2717, 2569, 2570, 2596,
       2636, 2643, 2669, 2712, 2573, 2574, 2644, 2645, 2647, 2718, 2574,
       2597, 2644, 2645, 2670, 2718, 2719, 2575, 2600, 2605, 2646, 2672,
       2677, 2720, 2573, 2576, 2577, 2644, 2647, 2648, 2718, 2577, 2578,
       2647, 2648, 2649, 2718, 2721, 2578, 2612, 2648, 2649, 2683, 2721,
       2722, 2579, 2613, 2619, 2650, 2684, 2689, 2723, 2580, 2614, 2616,
       2651, 2685, 2687, 2724, 2581, 2584, 2652, 2653, 2656, 2725, 2726,
       2581, 2582, 2652, 2653, 2654, 2726, 2727, 2582, 2583, 2653, 2654,
       2655, 2727, 2728, 2583, 2617, 2654, 2655, 2688, 2728, 2729, 2584,
       2618, 2652, 2656, 2686, 2725, 2730, 2586, 2587, 2619, 2657, 2658,
       2689, 2731, 2587, 2588, 2657, 2658, 2659, 2731, 2732, 2588, 2620,
       2658, 2659, 2690, 2732, 2733, 2555, 2624, 2626, 2660, 2695, 2697,
       2589, 2608, 2625, 2661, 2679, 2696, 2735, 2590, 2626, 2628, 2662,
       2698, 2701, 2736, 2591, 2630, 2632, 2663, 2703, 2706, 2737, 2592,
       2633, 2634, 2664, 2707, 2708, 2738, 2593, 2634, 2665, 2666, 2709,
       2739, 2593, 2594, 2665, 2666, 2667, 2739, 2740, 2594, 2595, 2666,
       2667, 2668, 2740, 2741, 2595, 2596, 2667, 2668, 2669, 2741, 2742,
       2596, 2643, 2668, 2669, 2712, 2742, 2743, 2597, 2603, 2606, 2645,
       2670, 2674, 2719, 2598, 2599, 2620, 2671, 2672, 2690, 2744, 2599,
       2600, 2646, 2671, 2672, 2720, 2744, 2601, 2602, 2673, 2674, 2675,
       2745, 2746, 2601, 2603, 2670, 2673, 2674, 2719, 2745, 2602, 2604,
       2673, 2675, 2676, 2746, 2747, 2791, 2604, 2605, 2675, 2676, 2677,
       2747, 2605, 2646, 2676, 2677, 2720, 2747, 2748, 2607, 2609, 2678,
       2679, 2680, 2749, 2750, 2607, 2608, 2661, 2678, 2679, 2735, 2749,
       2609, 2610, 2678, 2680, 2681, 2750, 2751, 2610, 2611, 2680, 2681,
       2682, 2751, 2752, 2611, 2612, 2681, 2682, 2683, 2752, 2753, 2612,
       2649, 2682, 2683, 2722, 2753, 2754, 2613, 2617, 2650, 2684, 2688,
       2723, 2755, 2614, 2642, 2651, 2685, 2717, 2724, 2756, 2615, 2616,
       2618, 2656, 2686, 2687, 2730, 2616, 2651, 2686, 2687, 2724, 2730,
       2757, 2617, 2655, 2684, 2688, 2729, 2755, 2758, 2619, 2650, 2657,
       2689, 2723, 2731, 2759, 2620, 2659, 2671, 2690, 2733, 2744, 2760,
       2621, 2625, 2691, 2692, 2696, 2761, 2762, 2621, 2622, 2691, 2692,
       2693, 2762, 2763, 2622, 2623, 2692, 2693, 2694, 2763, 2764, 2623,
       2624, 2693, 2694, 2734, 2764, 2624, 2660, 2695, 2697, 2734, 2765,
       2766, 2806, 2625, 2661, 2691, 2696, 2735, 2761, 2767, 2626, 2660,
       2695, 2697, 2698, 2766, 2768, 2626, 2662, 2697, 2698, 2736, 2768,
       2769, 2627, 2628, 2699, 2700, 2701, 2770, 2771, 2627, 2629, 2699,
       2700, 2702, 2771, 2772, 2628, 2662, 2699, 2701, 2736, 2770, 2773,
       2629, 2630, 2700, 2702, 2703, 2772, 2774, 2630, 2663, 2702, 2703,
       2737, 2774, 2775, 2631, 2632, 2704, 2705, 2706, 2776, 2777, 2631,
       2633, 2704, 2705, 2707, 2776, 2778, 2632, 2663, 2704, 2706, 2737,
       2777, 2779, 2633, 2664, 2705, 2707, 2738, 2778, 2780, 2634, 2664,
       2708, 2709, 2738, 2781, 2782, 2634, 2665, 2708, 2709, 2739, 2781,
       2783, 2811, 2635, 2636, 2710, 2711, 2712, 2784, 2785, 2635, 2637,
       2710, 2711, 2713, 2785, 2786, 2636, 2643, 2669, 2710, 2712, 2743,
       2784, 2637, 2638, 2711, 2713, 2714, 2786, 2787, 2638, 2639, 2713,
       2714, 2715, 2787, 2788, 2639, 2640, 2714, 2715, 2716, 2788, 2640,
       2641, 2715, 2716, 2717, 2788, 2789, 2641, 2642, 2685, 2716, 2717,
       2756, 2789, 2644, 2645, 2647, 2648, 2718, 2719, 2721, 2645, 2670,
       2674, 2718, 2719, 2721, 2745, 2646, 2672, 2677, 2720, 2744, 2748,
       2790, 2648, 2649, 2718, 2719, 2721, 2722, 2745, 2649, 2683, 2721,
       2722, 2745, 2746, 2754, 2650, 2684, 2689, 2723, 2755, 2759, 2792,
       2800, 2651, 2685, 2687, 2724, 2756, 2757, 2793, 2652, 2656, 2725,
       2726, 2730, 2794, 2795, 2652, 2653, 2725, 2726, 2727, 2794, 2796,
       2653, 2654, 2726, 2727, 2728, 2796, 2797, 2654, 2655, 2727, 2728,
       2729, 2797, 2798, 2655, 2688, 2728, 2729, 2758, 2798, 2799, 2656,
       2686, 2687, 2725, 2730, 2757, 2795, 2657, 2658, 2689, 2731, 2732,
       2759, 2801, 2658, 2659, 2731, 2732, 2733, 2801, 2802, 2659, 2690,
       2732, 2733, 2760, 2802, 2803, 2624, 2694, 2695, 2734, 2764, 2765,
       2804, 2805, 2661, 2679, 2696, 2735, 2749, 2767, 2807, 2662, 2698,
       2701, 2736, 2769, 2773, 2808, 2663, 2703, 2706, 2737, 2775, 2779,
       2809, 2664, 2707, 2708, 2738, 2780, 2782, 2810, 2665, 2666, 2709,
       2739, 2740, 2783, 2812, 2666, 2667, 2739, 2740, 2741, 2812, 2813,
       2667, 2668, 2740, 2741, 2742, 2813, 2814, 2668, 2669, 2741, 2742,
       2743, 2814, 2815, 2669, 2712, 2742, 2743, 2784, 2815, 2816, 2671,
       2672, 2690, 2720, 2744, 2760, 2790, 2673, 2674, 2719, 2721, 2722,
       2745, 2746, 2673, 2675, 2722, 2745, 2746, 2754, 2791, 2675, 2676,
       2677, 2747, 2748, 2791, 2817, 2677, 2720, 2747, 2748, 2790, 2817,
       2818, 2678, 2679, 2735, 2749, 2750, 2807, 2819, 2678, 2680, 2749,
       2750, 2751, 2819, 2820, 2857, 2680, 2681, 2750, 2751, 2752, 2820,
       2821, 2681, 2682, 2751, 2752, 2753, 2821, 2822, 2682, 2683, 2752,
       2753, 2754, 2822, 2823, 2683, 2722, 2746, 2753, 2754, 2791, 2823,
       2684, 2688, 2723, 2755, 2758, 2792, 2824, 2685, 2717, 2724, 2756,
       2789, 2793, 2825, 2687, 2724, 2730, 2757, 2793, 2795, 2826, 2688,
       2729, 2755, 2758, 2799, 2824, 2827, 2689, 2723, 2731, 2759, 2800,
       2801, 2828, 2690, 2733, 2744, 2760, 2790, 2803, 2829, 2691, 2696,
       2761, 2762, 2767, 2830, 2831, 2691, 2692, 2761, 2762, 2763, 2830,
       2832, 2692, 2693, 2762, 2763, 2764, 2832, 2833, 2693, 2694, 2734,
       2763, 2764, 2804, 2833, 2695, 2734, 2765, 2805, 2806, 2834, 2835,
       2695, 2697, 2766, 2768, 2781, 2806, 2811, 2836, 2696, 2735, 2761,
       2767, 2807, 2831, 2837, 2697, 2698, 2766, 2768, 2769, 2781, 2782,
       2698, 2736, 2768, 2769, 2782, 2808, 2810, 2699, 2701, 2770, 2771,
       2773, 2838, 2839, 2699, 2700, 2770, 2771, 2772, 2839, 2840, 2700,
       2702, 2771, 2772, 2774, 2840, 2841, 2701, 2736, 2770, 2773, 2808,
       2838, 2842, 2702, 2703, 2772, 2774, 2775, 2841, 2843, 2703, 2737,
       2774, 2775, 2809, 2843, 2844, 2704, 2705, 2776, 2777, 2778, 2845,
       2846, 2704, 2706, 2776, 2777, 2779, 2845, 2847, 2705, 2707, 2776,
       2778, 2780, 2846, 2848, 2706, 2737, 2777, 2779, 2809, 2847, 2849,
       2876, 2707, 2738, 2778, 2780, 2810, 2848, 2850, 2708, 2709, 2766,
       2768, 2781, 2782, 2811, 2708, 2738, 2768, 2769, 2781, 2782, 2810,
       2709, 2739, 2783, 2811, 2812, 2851, 2852, 2710, 2712, 2743, 2784,
       2785, 2816, 2853, 2710, 2711, 2784, 2785, 2786, 2853, 2854, 2711,
       2713, 2785, 2786, 2787, 2854, 2855, 2713, 2714, 2786, 2787, 2788,
       2855, 2856, 2714, 2715, 2716, 2787, 2788, 2789, 2856, 2716, 2717,
       2756, 2788, 2789, 2825, 2856, 2720, 2744, 2748, 2760, 2790, 2818,
       2829, 2675, 2746, 2747, 2754, 2791, 2817, 2823, 2723, 2755, 2792,
       2800, 2824, 2858, 2859, 2724, 2756, 2757, 2793, 2825, 2826, 2860,
       2725, 2726, 2794, 2795, 2796, 2861, 2862, 2725, 2730, 2757, 2794,
       2795, 2826, 2861, 2726, 2727, 2794, 2796, 2797, 2862, 2863, 2727,
       2728, 2796, 2797, 2798, 2863, 2864, 2728, 2729, 2797, 2798, 2799,
       2864, 2865, 2729, 2758, 2798, 2799, 2827, 2865, 2866, 2723, 2759,
       2792, 2800, 2828, 2859, 2731, 2732, 2759, 2801, 2802, 2828, 2867,
       2732, 2733, 2801, 2802, 2803, 2867, 2868, 2733, 2760, 2802, 2803,
       2829, 2868, 2869, 2734, 2764, 2804, 2805, 2833, 2870, 2871, 2734,
       2765, 2804, 2805, 2834, 2871, 2872, 2695, 2765, 2766, 2806, 2835,
       2836, 2873, 2735, 2749, 2767, 2807, 2819, 2837, 2874, 2736, 2769,
       2773, 2808, 2810, 2842, 2850, 2737, 2775, 2779, 2809, 2844, 2849,
       2875, 2738, 2769, 2780, 2782, 2808, 2810, 2850, 2709, 2766, 2781,
       2783, 2811, 2836, 2851, 2739, 2740, 2783, 2812, 2813, 2852, 2877,
       2740, 2741, 2812, 2813, 2814, 2877, 2878, 2741, 2742, 2813, 2814,
       2815, 2878, 2879, 2742, 2743, 2814, 2815, 2816, 2879, 2880, 2743,
       2784, 2815, 2816, 2853, 2880, 2881, 2747, 2748, 2791, 2817, 2818,
       2823, 2882, 2748, 2790, 2817, 2818, 2829, 2882, 2883, 2749, 2750,
       2807, 2819, 2857, 2874, 2884, 2750, 2751, 2820, 2821, 2857, 2885,
       2751, 2752, 2820, 2821, 2822, 2885, 2886, 2752, 2753, 2821, 2822,
       2823, 2882, 2886, 2753, 2754, 2791, 2817, 2822, 2823, 2882, 2755,
       2758, 2792, 2824, 2827, 2858, 2887, 2756, 2789, 2793, 2825, 2856,
       2860, 2888, 2757, 2793, 2795, 2826, 2860, 2861, 2889, 2758, 2799,
       2824, 2827, 2866, 2887, 2890, 2759, 2800, 2801, 2828, 2859, 2867,
       2891, 2760, 2790, 2803, 2818, 2829, 2869, 2883, 2761, 2762, 2830,
       2831, 2832, 2892, 2893, 2761, 2767, 2830, 2831, 2837, 2892, 2894,
       2762, 2763, 2830, 2832, 2833, 2893, 2895, 2763, 2764, 2804, 2832,
       2833, 2870, 2895, 2896, 2765, 2805, 2834, 2835, 2872, 2897, 2898,
       2765, 2806, 2834, 2835, 2873, 2898, 2899, 2766, 2806, 2811, 2836,
       2851, 2873, 2900, 2767, 2807, 2831, 2837, 2874, 2894, 2901, 2770,
       2773, 2838, 2839, 2842, 2902, 2903, 2770, 2771, 2838, 2839, 2840,
       2902, 2904, 2771, 2772, 2839, 2840, 2841, 2904, 2905, 2772, 2774,
       2840, 2841, 2843, 2905, 2906, 2773, 2808, 2838, 2842, 2850, 2903,
       2907, 2774, 2775, 2841, 2843, 2844, 2906, 2908, 2775, 2809, 2843,
       2844, 2875, 2908, 2909, 2776, 2777, 2845, 2846, 2847, 2910, 2911,
       2776, 2778, 2845, 2846, 2848, 2911, 2912, 2777, 2779, 2845, 2847,
       2876, 2910, 2913, 2941, 2778, 2780, 2846, 2848, 2850, 2907, 2912,
       2779, 2809, 2849, 2875, 2876, 2914, 2915, 2780, 2808, 2810, 2842,
       2848, 2850, 2907, 2783, 2811, 2836, 2851, 2852, 2900, 2916, 2783,
       2812, 2851, 2852, 2877, 2916, 2917, 2784, 2785, 2816, 2853, 2854,
       2881, 2918, 2785, 2786, 2853, 2854, 2855, 2918, 2919, 2786, 2787,
       2854, 2855, 2856, 2888, 2919, 2946, 2787, 2788, 2789, 2825, 2855,
       2856, 2888, 2750, 2819, 2820, 2857, 2884, 2885, 2920, 2792, 2824,
       2858, 2859, 2887, 2921, 2922, 2792, 2800, 2828, 2858, 2859, 2891,
       2922, 2930, 2793, 2825, 2826, 2860, 2888, 2889, 2923, 2794, 2795,
       2826, 2861, 2862, 2889, 2924, 2794, 2796, 2861, 2862, 2863, 2924,
       2925, 2796, 2797, 2862, 2863, 2864, 2925, 2926, 2797, 2798, 2863,
       2864, 2865, 2926, 2927, 2798, 2799, 2864, 2865, 2866, 2927, 2928,
       2799, 2827, 2865, 2866, 2890, 2928, 2929, 2801, 2802, 2828, 2867,
       2868, 2891, 2931, 2802, 2803, 2867, 2868, 2869, 2931, 2932, 2803,
       2829, 2868, 2869, 2883, 2932, 2933, 2804, 2833, 2870, 2871, 2896,
       2934, 2935, 2804, 2805, 2870, 2871, 2872, 2935, 2936, 2805, 2834,
       2871, 2872, 2897, 2936, 2937, 2806, 2835, 2836, 2873, 2899, 2900,
       2938, 2807, 2819, 2837, 2874, 2884, 2901, 2939, 2809, 2844, 2849,
       2875, 2909, 2914, 2940, 2779, 2847, 2849, 2876, 2913, 2915, 2812,
       2813, 2852, 2877, 2878, 2917, 2942, 2813, 2814, 2877, 2878, 2879,
       2942, 2943, 2814, 2815, 2878, 2879, 2880, 2943, 2944, 2815, 2816,
       2879, 2880, 2881, 2944, 2945, 2816, 2853, 2880, 2881, 2918, 2945,
       2947, 2817, 2818, 2822, 2823, 2882, 2883, 2886, 2818, 2829, 2869,
       2882, 2883, 2886, 2933, 2819, 2857, 2874, 2884, 2920, 2939, 2948,
       2820, 2821, 2857, 2885, 2886, 2920, 2933, 2821, 2822, 2882, 2883,
       2885, 2886, 2933, 2824, 2827, 2858, 2887, 2890, 2921, 2949, 2825,
       2855, 2856, 2860, 2888, 2923, 2946, 2826, 2860, 2861, 2889, 2923,
       2924, 2950, 2984, 2827, 2866, 2887, 2890, 2929, 2949, 2951, 2828,
       2859, 2867, 2891, 2930, 2931, 2948, 2830, 2831, 2892, 2893, 2894,
       2952, 2953, 2830, 2832, 2892, 2893, 2895, 2952, 2954, 2831, 2837,
       2892, 2894, 2901, 2953, 2955, 2832, 2833, 2893, 2895, 2896, 2954,
       2956, 2833, 2870, 2895, 2896, 2934, 2956, 2957, 2834, 2872, 2897,
       2898, 2937, 2958, 2959, 2834, 2835, 2897, 2898, 2899, 2958, 2960,
       2835, 2873, 2898, 2899, 2938, 2960, 2961, 2836, 2851, 2873, 2900,
       2916, 2938, 2962, 2837, 2874, 2894, 2901, 2939, 2955, 2963, 2838,
       2839, 2902, 2903, 2904, 2964, 2965, 2838, 2842, 2902, 2903, 2907,
       2964, 2966, 2839, 2840, 2902, 2904, 2905, 2965, 2967, 2840, 2841,
       2904, 2905, 2906, 2967, 2968, 2841, 2843, 2905, 2906, 2908, 2968,
       2969, 2842, 2848, 2850, 2903, 2907, 2912, 2966, 2843, 2844, 2906,
       2908, 2909, 2969, 2970, 2844, 2875, 2908, 2909, 2940, 2970, 2971,
       2845, 2847, 2910, 2911, 2941, 2972, 2973, 2845, 2846, 2910, 2911,
       2912, 2973, 2974, 2846, 2848, 2907, 2911, 2912, 2966, 2974, 2847,
       2876, 2913, 2915, 2941, 2975, 2849, 2875, 2914, 2915, 2940, 2976,
       2977, 2849, 2876, 2913, 2914, 2915, 2975, 2977, 2851, 2852, 2900,
       2916, 2917, 2962, 2978, 2997, 2852, 2877, 2916, 2917, 2942, 2978,
       2979, 2853, 2854, 2881, 2918, 2919, 2947, 2980, 2854, 2855, 2918,
       2919, 2946, 2980, 2857, 2884, 2885, 2920, 2932, 2933, 2948, 2858,
       2887, 2921, 2922, 2949, 2981, 2982, 2858, 2859, 2921, 2922, 2930,
       2963, 2981, 2860, 2888, 2889, 2923, 2946, 2950, 2983, 2861, 2862,
       2889, 2924, 2925, 2984, 2985, 2862, 2863, 2924, 2925, 2926, 2985,
       2986, 2863, 2864, 2925, 2926, 2927, 2986, 2987, 3004, 2864, 2865,
       2926, 2927, 2928, 2987, 2988, 2865, 2866, 2927, 2928, 2929, 2988,
       2866, 2890, 2928, 2929, 2951, 2988, 2989, 3005, 2859, 2891, 2922,
       2930, 2939, 2948, 2963, 2867, 2868, 2891, 2931, 2932, 2948, 2868,
       2869, 2920, 2931, 2932, 2933, 2948, 2869, 2883, 2885, 2886, 2920,
       2932, 2933, 2870, 2896, 2934, 2935, 2957, 2990, 2991, 2870, 2871,
       2934, 2935, 2936, 2991, 2992, 2871, 2872, 2935, 2936, 2937, 2992,
       2993, 2872, 2897, 2936, 2937, 2959, 2993, 2994, 2873, 2899, 2900,
       2938, 2961, 2962, 2995, 2874, 2884, 2901, 2930, 2939, 2948, 2963,
       2875, 2909, 2914, 2940, 2971, 2976, 2847, 2910, 2913, 2941, 2972,
       2975, 2996, 2877, 2878, 2917, 2942, 2943, 2979, 2998, 2878, 2879,
       2942, 2943, 2944, 2998, 2999, 2879, 2880, 2943, 2944, 2945, 2999,
       3000, 2880, 2881, 2944, 2945, 2947, 3000, 3001, 2855, 2888, 2919,
       2923, 2946, 2980, 2983, 2881, 2918, 2945, 2947, 2980, 3001, 3002,
       2884, 2891, 2920, 2930, 2931, 2932, 2939, 2948, 2887, 2890, 2921,
       2949, 2951, 2982, 2889, 2923, 2950, 2983, 2984, 3003, 2890, 2929,
       2949, 2951, 2982, 2989, 3006, 2892, 2893, 2952, 2953, 2954, 3007,
       3008, 2892, 2894, 2952, 2953, 2955, 3007, 3009, 2893, 2895, 2952,
       2954, 2956, 3008, 3010, 2894, 2901, 2953, 2955, 2963, 2981, 3009,
       2895, 2896, 2954, 2956, 2957, 3010, 3011, 3040, 2896, 2934, 2956,
       2957, 2990, 3011, 3012, 2897, 2898, 2958, 2959, 2960, 3013, 3014,
       2897, 2937, 2958, 2959, 2994, 3013, 3015, 2898, 2899, 2958, 2960,
       2961, 3014, 3016, 2899, 2938, 2960, 2961, 2995, 3016, 3017, 2900,
       2916, 2938, 2962, 2995, 2997, 3018, 2901, 2922, 2930, 2939, 2955,
       2963, 2981, 2902, 2903, 2964, 2965, 2966, 3019, 3020, 2902, 2904,
       2964, 2965, 2967, 3019, 3021, 2903, 2907, 2912, 2964, 2966, 2974,
       3020, 2904, 2905, 2965, 2967, 2968, 3021, 3022, 2905, 2906, 2967,
       2968, 2969, 3022, 3023, 2906, 2908, 2968, 2969, 2970, 3023, 3024,
       2908, 2909, 2969, 2970, 2971, 3024, 3025, 2909, 2940, 2970, 2971,
       2976, 3025, 3026, 3048, 2910, 2941, 2972, 2973, 2996, 3027, 3028,
       2910, 2911, 2972, 2973, 2974, 3028, 3029, 2911, 2912, 2966, 2973,
       2974, 3020, 3029, 2913, 2915, 2941, 2975, 2977, 2996, 3030, 2914,
       2940, 2971, 2976, 2977, 3026, 3031, 2914, 2915, 2975, 2976, 2977,
       3030, 3031, 2916, 2917, 2978, 2979, 2997, 3032, 2917, 2942, 2978,
       2979, 2998, 3032, 3033, 2918, 2919, 2946, 2947, 2980, 2983, 3002,
       2921, 2922, 2955, 2963, 2981, 2982, 3009, 2921, 2949, 2951, 2981,
       2982, 3006, 3009, 2923, 2946, 2950, 2980, 2983, 3002, 3003, 2889,
       2924, 2950, 2984, 2985, 3003, 3034, 2924, 2925, 2984, 2985, 2986,
       3034, 3035, 2925, 2926, 2985, 2986, 3004, 3035, 2926, 2927, 2987,
       2988, 3004, 3036, 3037, 2927, 2928, 2929, 2987, 2988, 3005, 3037,
       2929, 2951, 2989, 3005, 3006, 3038, 3039, 2934, 2957, 2990, 2991,
       3012, 3041, 3042, 2934, 2935, 2990, 2991, 2992, 3041, 3043, 2935,
       2936, 2991, 2992, 2993, 3043, 3044, 2936, 2937, 2992, 2993, 2994,
       3044, 3045, 2937, 2959, 2993, 2994, 3015, 3045, 3046, 2938, 2961,
       2962, 2995, 3017, 3018, 3047, 2941, 2972, 2975, 2996, 3027, 3030,
       3049, 2916, 2962, 2978, 2997, 3018, 3032, 3050, 3051, 2942, 2943,
       2979, 2998, 2999, 3033, 3052, 2943, 2944, 2998, 2999, 3000, 3052,
       3053, 3085, 2944, 2945, 2999, 3000, 3001, 3053, 3054, 2945, 2947,
       3000, 3001, 3002, 3054, 3055, 2947, 2980, 2983, 3001, 3002, 3003,
       3055, 2950, 2983, 2984, 3002, 3003, 3034, 3055, 2926, 2986, 2987,
       3004, 3035, 3036, 3056, 2929, 2988, 2989, 3005, 3037, 3038, 3057,
       2951, 2982, 2989, 3006, 3007, 3009, 3039, 2952, 2953, 3006, 3007,
       3008, 3009, 3039, 2952, 2954, 3007, 3008, 3010, 3039, 3058, 2953,
       2955, 2981, 2982, 3006, 3007, 3009, 2954, 2956, 3008, 3010, 3040,
       3058, 3059, 2956, 2957, 3011, 3012, 3040, 3060, 3061, 2957, 2990,
       3011, 3012, 3042, 3060, 3062, 2958, 2959, 3013, 3014, 3015, 3063,
       3064, 2958, 2960, 3013, 3014, 3016, 3064, 3065, 2959, 2994, 3013,
       3015, 3046, 3063, 3066, 2960, 2961, 3014, 3016, 3017, 3065, 3067,
       2961, 2995, 3016, 3017, 3047, 3067, 3068, 2962, 2995, 2997, 3018,
       3047, 3050, 3069, 2964, 2965, 3019, 3020, 3021, 3070, 3071, 2964,
       2966, 2974, 3019, 3020, 3029, 3071, 2965, 2967, 3019, 3021, 3022,
       3070, 3072, 2967, 2968, 3021, 3022, 3023, 3072, 3073, 2968, 2969,
       3022, 3023, 3024, 3073, 3074, 2969, 2970, 3023, 3024, 3025, 3074,
       3075, 2970, 2971, 3024, 3025, 3048, 3075, 3076, 2971, 2976, 3026,
       3031, 3048, 3077, 3078, 2972, 2996, 3027, 3028, 3049, 3079, 3080,
       2972, 2973, 3027, 3028, 3029, 3080, 3081, 2973, 2974, 3020, 3028,
       3029, 3071, 3081, 2975, 2977, 2996, 3030, 3031, 3049, 3082, 2976,
       2977, 3026, 3030, 3031, 3077, 3082, 2978, 2979, 2997, 3032, 3033,
       3051, 3083, 2979, 2998, 3032, 3033, 3052, 3083, 3084, 2984, 2985,
       3003, 3034, 3035, 3055, 3086, 2985, 2986, 3004, 3034, 3035, 3056,
       3086, 2987, 3004, 3036, 3037, 3056, 3087, 3088, 2987, 2988, 3005,
       3036, 3037, 3057, 3087, 2989, 3005, 3038, 3039, 3057, 3058, 3089,
       2989, 3006, 3007, 3008, 3038, 3039, 3058, 2956, 3010, 3011, 3040,
       3059, 3061, 3090, 2990, 2991, 3041, 3042, 3043, 3091, 3092, 2990,
       3012, 3041, 3042, 3062, 3091, 3093, 2991, 2992, 3041, 3043, 3044,
       3092, 3094, 2992, 2993, 3043, 3044, 3045, 3094, 3095, 2993, 2994,
       3044, 3045, 3046, 3095, 3096, 2994, 3015, 3045, 3046, 3066, 3096,
       3097, 2995, 3017, 3018, 3047, 3068, 3069, 3098, 2971, 3025, 3026,
       3048, 3076, 3078, 3099, 2996, 3027, 3030, 3049, 3079, 3082, 3100,
       2997, 3018, 3050, 3051, 3060, 3061, 3069, 2997, 3032, 3050, 3051,
       3061, 3083, 3090, 2998, 2999, 3033, 3052, 3084, 3085, 3101, 2999,
       3000, 3053, 3054, 3085, 3102, 3000, 3001, 3053, 3054, 3055, 3086,
       3102, 3001, 3002, 3003, 3034, 3054, 3055, 3086, 3004, 3035, 3036,
       3056, 3086, 3088, 3102, 3005, 3037, 3038, 3057, 3087, 3089, 3103,
       3008, 3010, 3038, 3039, 3058, 3059, 3089, 3010, 3040, 3058, 3059,
       3089, 3090, 3104, 3011, 3012, 3050, 3060, 3061, 3062, 3069, 3011,
       3040, 3050, 3051, 3060, 3061, 3090, 3012, 3042, 3060, 3062, 3069,
       3093, 3098, 3013, 3015, 3063, 3064, 3066, 3105, 3106, 3013, 3014,
       3063, 3064, 3065, 3106, 3107, 3014, 3016, 3064, 3065, 3067, 3107,
       3108, 3015, 3046, 3063, 3066, 3097, 3105, 3109, 3016, 3017, 3065,
       3067, 3068, 3108, 3110, 3017, 3047, 3067, 3068, 3098, 3110, 3111,
       3018, 3047, 3050, 3060, 3062, 3069, 3098, 3019, 3021, 3070, 3071,
       3072, 3112, 3113, 3019, 3020, 3029, 3070, 3071, 3081, 3112, 3021,
       3022, 3070, 3072, 3073, 3113, 3114, 3022, 3023, 3072, 3073, 3074,
       3114, 3115, 3023, 3024, 3073, 3074, 3075, 3115, 3116, 3024, 3025,
       3074, 3075, 3076, 3116, 3117, 3025, 3048, 3075, 3076, 3099, 3117,
       3026, 3031, 3077, 3078, 3082, 3118, 3119, 3026, 3048, 3077, 3078,
       3099, 3118, 3120, 3027, 3049, 3079, 3080, 3100, 3121, 3122, 3027,
       3028, 3079, 3080, 3081, 3121, 3123, 3028, 3029, 3071, 3080, 3081,
       3112, 3123, 3030, 3031, 3049, 3077, 3082, 3100, 3119, 3032, 3033,
       3051, 3083, 3084, 3090, 3104, 3033, 3052, 3083, 3084, 3101, 3103,
       3104, 2999, 3052, 3053, 3085, 3088, 3101, 3102, 3034, 3035, 3054,
       3055, 3056, 3086, 3102, 3036, 3037, 3057, 3087, 3088, 3101, 3103,
       3036, 3056, 3085, 3087, 3088, 3101, 3102, 3038, 3057, 3058, 3059,
       3089, 3103, 3104, 3040, 3051, 3059, 3061, 3083, 3090, 3104, 3041,
       3042, 3091, 3092, 3093, 3124, 3125, 3041, 3043, 3091, 3092, 3094,
       3125, 3126, 3133, 3042, 3062, 3091, 3093, 3098, 3111, 3124, 3043,
       3044, 3092, 3094, 3095, 3126, 3127, 3044, 3045, 3094, 3095, 3096,
       3127, 3128, 3045, 3046, 3095, 3096, 3097, 3128, 3129, 3046, 3066,
       3096, 3097, 3109, 3129, 3130, 3047, 3062, 3068, 3069, 3093, 3098,
       3111, 3048, 3076, 3078, 3099, 3117, 3120, 3131, 3049, 3079, 3082,
       3100, 3119, 3122, 3132, 3052, 3084, 3085, 3087, 3088, 3101, 3103,
       3053, 3054, 3056, 3085, 3086, 3088, 3102, 3057, 3084, 3087, 3089,
       3101, 3103, 3104, 3059, 3083, 3084, 3089, 3090, 3103, 3104, 3063,
       3066, 3105, 3106, 3109, 3134, 3135, 3063, 3064, 3105, 3106, 3107,
       3135, 3136, 3064, 3065, 3106, 3107, 3108, 3136, 3137, 3065, 3067,
       3107, 3108, 3110, 3137, 3138, 3066, 3097, 3105, 3109, 3130, 3134,
       3139, 3067, 3068, 3108, 3110, 3111, 3138, 3140, 3068, 3093, 3098,
       3110, 3111, 3124, 3140, 3070, 3071, 3081, 3112, 3113, 3123, 3141,
       3070, 3072, 3112, 3113, 3114, 3141, 3142, 3072, 3073, 3113, 3114,
       3115, 3142, 3143, 3073, 3074, 3114, 3115, 3116, 3143, 3144, 3074,
       3075, 3115, 3116, 3117, 3144, 3145, 3075, 3076, 3099, 3116, 3117,
       3131, 3145, 3146, 3077, 3078, 3118, 3119, 3120, 3147, 3148, 3077,
       3082, 3100, 3118, 3119, 3132, 3147, 3078, 3099, 3118, 3120, 3131,
       3148, 3149, 3079, 3080, 3121, 3122, 3123, 3150, 3151, 3079, 3100,
       3121, 3122, 3132, 3150, 3152, 3080, 3081, 3112, 3121, 3123, 3141,
       3151, 3091, 3093, 3111, 3124, 3125, 3140, 3153, 3091, 3092, 3124,
       3125, 3133, 3153, 3154, 3092, 3094, 3126, 3127, 3133, 3155, 3156,
       3094, 3095, 3126, 3127, 3128, 3155, 3157, 3095, 3096, 3127, 3128,
       3129, 3157, 3158, 3096, 3097, 3128, 3129, 3130, 3158, 3159, 3097,
       3109, 3129, 3130, 3139, 3159, 3160, 3099, 3117, 3120, 3131, 3146,
       3149, 3161, 3100, 3119, 3122, 3132, 3147, 3152, 3162, 3092, 3125,
       3126, 3133, 3154, 3156, 3105, 3109, 3134, 3135, 3139, 3163, 3164,
       3105, 3106, 3134, 3135, 3136, 3164, 3165, 3106, 3107, 3135, 3136,
       3137, 3165, 3166, 3107, 3108, 3136, 3137, 3138, 3166, 3167, 3108,
       3110, 3137, 3138, 3140, 3167, 3168, 3109, 3130, 3134, 3139, 3160,
       3163, 3169, 3110, 3111, 3124, 3138, 3140, 3153, 3168, 3112, 3113,
       3123, 3141, 3142, 3151, 3170, 3113, 3114, 3141, 3142, 3143, 3170,
       3171, 3114, 3115, 3142, 3143, 3144, 3171, 3172, 3115, 3116, 3143,
       3144, 3145, 3172, 3173, 3116, 3117, 3144, 3145, 3146, 3173, 3174,
       3117, 3131, 3145, 3146, 3161, 3174, 3175, 3118, 3119, 3132, 3147,
       3148, 3162, 3176, 3118, 3120, 3147, 3148, 3149, 3176, 3177, 3120,
       3131, 3148, 3149, 3161, 3177, 3178, 3121, 3122, 3150, 3151, 3152,
       3179, 3180, 3121, 3123, 3141, 3150, 3151, 3170, 3180, 3122, 3132,
       3150, 3152, 3162, 3179, 3181, 3124, 3125, 3140, 3153, 3154, 3168,
       3182, 3125, 3133, 3153, 3154, 3156, 3182, 3183, 3126, 3127, 3155,
       3156, 3157, 3184, 3185, 3126, 3133, 3154, 3155, 3156, 3183, 3184,
       3127, 3128, 3155, 3157, 3158, 3185, 3186, 3128, 3129, 3157, 3158,
       3159, 3186, 3187, 3129, 3130, 3158, 3159, 3160, 3187, 3188, 3130,
       3139, 3159, 3160, 3169, 3188, 3189, 3131, 3146, 3149, 3161, 3175,
       3178, 3132, 3147, 3152, 3162, 3176, 3181, 3190, 3134, 3139, 3163,
       3164, 3169, 3191, 3192, 3134, 3135, 3163, 3164, 3165, 3191, 3193,
       3135, 3136, 3164, 3165, 3166, 3193, 3194, 3136, 3137, 3165, 3166,
       3167, 3194, 3195, 3137, 3138, 3166, 3167, 3168, 3195, 3196, 3138,
       3140, 3153, 3167, 3168, 3182, 3196, 3139, 3160, 3163, 3169, 3189,
       3192, 3197, 3141, 3142, 3151, 3170, 3171, 3180, 3198, 3142, 3143,
       3170, 3171, 3172, 3198, 3199, 3143, 3144, 3171, 3172, 3173, 3199,
       3200, 3144, 3145, 3172, 3173, 3174, 3200, 3201, 3145, 3146, 3173,
       3174, 3175, 3201, 3202, 3146, 3161, 3174, 3175, 3178, 3202, 3203,
       3147, 3148, 3162, 3176, 3177, 3190, 3204, 3148, 3149, 3176, 3177,
       3178, 3204, 3205, 3149, 3161, 3175, 3177, 3178, 3203, 3205, 3150,
       3152, 3179, 3180, 3181, 3206, 3207, 3150, 3151, 3170, 3179, 3180,
       3198, 3207, 3152, 3162, 3179, 3181, 3190, 3206, 3208, 3153, 3154,
       3168, 3182, 3183, 3196, 3209, 3154, 3156, 3182, 3183, 3184, 3209,
       3210, 3155, 3156, 3183, 3184, 3185, 3210, 3211, 3155, 3157, 3184,
       3185, 3186, 3211, 3212, 3157, 3158, 3185, 3186, 3187, 3212, 3213,
       3158, 3159, 3186, 3187, 3188, 3213, 3214, 3159, 3160, 3187, 3188,
       3189, 3214, 3215, 3160, 3169, 3188, 3189, 3197, 3215, 3216, 3162,
       3176, 3181, 3190, 3204, 3208, 3217, 3163, 3164, 3191, 3192, 3193,
       3218, 3219, 3163, 3169, 3191, 3192, 3197, 3218, 3220, 3164, 3165,
       3191, 3193, 3194, 3219, 3221, 3165, 3166, 3193, 3194, 3195, 3221,
       3222, 3166, 3167, 3194, 3195, 3196, 3222, 3223, 3167, 3168, 3182,
       3195, 3196, 3209, 3223, 3169, 3189, 3192, 3197, 3216, 3220, 3224,
       3170, 3171, 3180, 3198, 3199, 3207, 3225, 3171, 3172, 3198, 3199,
       3200, 3225, 3226, 3172, 3173, 3199, 3200, 3201, 3226, 3227, 3173,
       3174, 3200, 3201, 3202, 3227, 3228, 3174, 3175, 3201, 3202, 3203,
       3228, 3175, 3178, 3202, 3203, 3205, 3228, 3229, 3176, 3177, 3190,
       3204, 3205, 3217, 3230, 3177, 3178, 3203, 3204, 3205, 3229, 3230,
       3179, 3181, 3206, 3207, 3208, 3225, 3231, 3179, 3180, 3198, 3206,
       3207, 3225, 3181, 3190, 3206, 3208, 3217, 3231, 3182, 3183, 3196,
       3209, 3210, 3223, 3232, 3183, 3184, 3209, 3210, 3211, 3232, 3233,
       3184, 3185, 3210, 3211, 3212, 3233, 3234, 3185, 3186, 3211, 3212,
       3213, 3234, 3235, 3186, 3187, 3212, 3213, 3214, 3235, 3236, 3187,
       3188, 3213, 3214, 3215, 3236, 3237, 3188, 3189, 3214, 3215, 3216,
       3237, 3238, 3189, 3197, 3215, 3216, 3224, 3238, 3239, 3190, 3204,
       3208, 3217, 3230, 3231, 3240, 3191, 3192, 3218, 3219, 3220, 3241,
       3242, 3191, 3193, 3218, 3219, 3221, 3242, 3243, 3192, 3197, 3218,
       3220, 3224, 3241, 3244, 3193, 3194, 3219, 3221, 3222, 3243, 3245,
       3194, 3195, 3221, 3222, 3223, 3245, 3246, 3195, 3196, 3209, 3222,
       3223, 3232, 3246, 3197, 3216, 3220, 3224, 3239, 3244, 3247, 3198,
       3199, 3206, 3207, 3225, 3226, 3231, 3199, 3200, 3225, 3226, 3227,
       3231, 3240, 3200, 3201, 3226, 3227, 3228, 3240, 3248, 3201, 3202,
       3203, 3227, 3228, 3229, 3248, 3203, 3205, 3228, 3229, 3230, 3248,
       3204, 3205, 3217, 3229, 3230, 3240, 3248, 3206, 3208, 3217, 3225,
       3226, 3231, 3240, 3209, 3210, 3223, 3232, 3233, 3246, 3249, 3210,
       3211, 3232, 3233, 3234, 3249, 3250, 3211, 3212, 3233, 3234, 3235,
       3250, 3251, 3212, 3213, 3234, 3235, 3236, 3251, 3252, 3213, 3214,
       3235, 3236, 3237, 3252, 3253, 3214, 3215, 3236, 3237, 3238, 3253,
       3254, 3215, 3216, 3237, 3238, 3239, 3254, 3255, 3216, 3224, 3238,
       3239, 3247, 3255, 3256, 3217, 3226, 3227, 3230, 3231, 3240, 3248,
       3218, 3220, 3241, 3242, 3244, 3257, 3258, 3218, 3219, 3241, 3242,
       3243, 3258, 3259, 3219, 3221, 3242, 3243, 3245, 3259, 3260, 3220,
       3224, 3241, 3244, 3247, 3257, 3261, 3221, 3222, 3243, 3245, 3246,
       3260, 3262, 3222, 3223, 3232, 3245, 3246, 3249, 3262, 3224, 3239,
       3244, 3247, 3256, 3261, 3263, 3227, 3228, 3229, 3230, 3240, 3248,
       3232, 3233, 3246, 3249, 3250, 3262, 3264, 3233, 3234, 3249, 3250,
       3251, 3264, 3265, 3234, 3235, 3250, 3251, 3252, 3265, 3266, 3235,
       3236, 3251, 3252, 3253, 3266, 3267, 3236, 3237, 3252, 3253, 3254,
       3267, 3268, 3237, 3238, 3253, 3254, 3255, 3268, 3238, 3239, 3254,
       3255, 3256, 3268, 3269, 3239, 3247, 3255, 3256, 3263, 3269, 3270,
       3241, 3244, 3257, 3258, 3261, 3271, 3272, 3241, 3242, 3257, 3258,
       3259, 3271, 3273, 3242, 3243, 3258, 3259, 3260, 3273, 3274, 3243,
       3245, 3259, 3260, 3262, 3274, 3275, 3244, 3247, 3257, 3261, 3263,
       3272, 3276, 3245, 3246, 3249, 3260, 3262, 3264, 3275, 3247, 3256,
       3261, 3263, 3270, 3276, 3277, 3249, 3250, 3262, 3264, 3265, 3275,
       3278, 3250, 3251, 3264, 3265, 3266, 3278, 3279, 3251, 3252, 3265,
       3266, 3267, 3279, 3280, 3252, 3253, 3266, 3267, 3268, 3280, 3281,
       3253, 3254, 3255, 3267, 3268, 3269, 3281, 3255, 3256, 3268, 3269,
       3270, 3281, 3282, 3256, 3263, 3269, 3270, 3277, 3282, 3257, 3258,
       3271, 3272, 3273, 3283, 3284, 3257, 3261, 3271, 3272, 3276, 3283,
       3285, 3258, 3259, 3271, 3273, 3274, 3284, 3286, 3259, 3260, 3273,
       3274, 3275, 3286, 3287, 3260, 3262, 3264, 3274, 3275, 3278, 3287,
       3261, 3263, 3272, 3276, 3277, 3285, 3288, 3263, 3270, 3276, 3277,
       3282, 3288, 3289, 3264, 3265, 3275, 3278, 3279, 3287, 3290, 3265,
       3266, 3278, 3279, 3280, 3290, 3291, 3266, 3267, 3279, 3280, 3281,
       3291, 3292, 3267, 3268, 3269, 3280, 3281, 3282, 3292, 3269, 3270,
       3277, 3281, 3282, 3289, 3292, 3271, 3272, 3283, 3284, 3285, 3293,
       3294, 3271, 3273, 3283, 3284, 3286, 3293, 3295, 3272, 3276, 3283,
       3285, 3288, 3294, 3296, 3273, 3274, 3284, 3286, 3287, 3295, 3297,
       3274, 3275, 3278, 3286, 3287, 3290, 3297, 3276, 3277, 3285, 3288,
       3289, 3296, 3277, 3282, 3288, 3289, 3292, 3296, 3298, 3278, 3279,
       3287, 3290, 3291, 3297, 3299, 3279, 3280, 3290, 3291, 3292, 3298,
       3299, 3280, 3281, 3282, 3289, 3291, 3292, 3298, 3283, 3284, 3293,
       3294, 3295, 3300, 3283, 3285, 3293, 3294, 3296, 3300, 3284, 3286,
       3293, 3295, 3297, 3299, 3300, 3285, 3288, 3289, 3294, 3296, 3298,
       3300, 3286, 3287, 3290, 3295, 3297, 3299, 3289, 3291, 3292, 3296,
       3298, 3299, 3300, 3290, 3291, 3295, 3297, 3298, 3299, 3300, 3293,
       3294, 3295, 3296, 3298, 3299, 3300], dtype=np.int32)
        indptr_f_gt = np.array([    0,     7,    13,    21,    28,    35,    42,    49,    56,
          63,    70,    77,    84,    91,    98,   105,   112,   119,
         126,   133,   140,   147,   154,   161,   168,   175,   182,
         189,   196,   203,   210,   217,   224,   231,   238,   245,
         252,   259,   266,   273,   280,   287,   294,   301,   308,
         315,   322,   329,   336,   343,   350,   357,   364,   371,
         378,   385,   392,   400,   407,   414,   421,   428,   435,
         441,   448,   456,   464,   470,   477,   484,   491,   498,
         505,   512,   519,   526,   533,   540,   547,   554,   561,
         568,   575,   582,   589,   596,   602,   609,   616,   623,
         630,   638,   645,   652,   659,   665,   672,   679,   686,
         693,   700,   707,   714,   721,   728,   735,   742,   749,
         756,   762,   769,   776,   783,   790,   797,   804,   811,
         818,   825,   832,   839,   846,   853,   860,   867,   874,
         881,   888,   895,   902,   909,   916,   923,   931,   938,
         945,   952,   959,   966,   973,   980,   987,   994,  1001,
        1008,  1015,  1022,  1029,  1035,  1043,  1050,  1057,  1064,
        1071,  1078,  1085,  1092,  1099,  1106,  1113,  1120,  1127,
        1134,  1141,  1148,  1155,  1162,  1169,  1176,  1183,  1190,
        1197,  1204,  1211,  1218,  1225,  1232,  1239,  1246,  1253,
        1260,  1267,  1274,  1281,  1288,  1295,  1302,  1309,  1316,
        1323,  1330,  1337,  1344,  1351,  1358,  1365,  1372,  1379,
        1386,  1393,  1400,  1407,  1414,  1421,  1428,  1435,  1442,
        1449,  1456,  1463,  1470,  1477,  1484,  1491,  1498,  1505,
        1512,  1519,  1526,  1533,  1540,  1547,  1554,  1561,  1568,
        1575,  1582,  1589,  1596,  1603,  1610,  1617,  1624,  1631,
        1638,  1645,  1652,  1659,  1666,  1673,  1680,  1687,  1694,
        1701,  1709,  1716,  1723,  1729,  1736,  1743,  1750,  1757,
        1764,  1771,  1778,  1785,  1792,  1799,  1806,  1813,  1820,
        1827,  1834,  1841,  1848,  1855,  1862,  1869,  1876,  1883,
        1890,  1897,  1904,  1911,  1918,  1925,  1932,  1939,  1946,
        1953,  1960,  1967,  1974,  1981,  1988,  1995,  2002,  2009,
        2016,  2023,  2030,  2037,  2044,  2051,  2058,  2065,  2072,
        2079,  2085,  2092,  2099,  2106,  2112,  2119,  2126,  2133,
        2140,  2147,  2154,  2161,  2168,  2175,  2182,  2189,  2196,
        2203,  2210,  2217,  2224,  2231,  2238,  2245,  2252,  2259,
        2266,  2273,  2281,  2288,  2295,  2302,  2309,  2316,  2323,
        2330,  2336,  2343,  2350,  2357,  2364,  2371,  2378,  2385,
        2393,  2400,  2407,  2414,  2421,  2428,  2434,  2441,  2449,
        2456,  2463,  2470,  2477,  2484,  2491,  2498,  2505,  2512,
        2519,  2526,  2533,  2540,  2547,  2554,  2561,  2568,  2575,
        2582,  2589,  2596,  2603,  2610,  2617,  2624,  2631,  2638,
        2645,  2652,  2659,  2666,  2673,  2680,  2687,  2694,  2701,
        2708,  2715,  2722,  2729,  2736,  2743,  2750,  2757,  2764,
        2771,  2778,  2785,  2792,  2799,  2806,  2813,  2820,  2827,
        2834,  2841,  2848,  2855,  2862,  2869,  2876,  2883,  2890,
        2897,  2904,  2911,  2918,  2925,  2932,  2939,  2945,  2951,
        2958,  2965,  2972,  2979,  2986,  2993,  3000,  3007,  3014,
        3021,  3027,  3034,  3041,  3048,  3055,  3062,  3069,  3076,
        3083,  3090,  3097,  3103,  3111,  3117,  3124,  3131,  3138,
        3145,  3152,  3159,  3166,  3173,  3180,  3187,  3194,  3201,
        3208,  3215,  3222,  3229,  3236,  3243,  3250,  3257,  3264,
        3270,  3278,  3285,  3293,  3300,  3307,  3314,  3321,  3328,
        3335,  3342,  3349,  3356,  3363,  3370,  3377,  3384,  3391,
        3398,  3406,  3413,  3420,  3427,  3434,  3441,  3448,  3455,
        3462,  3469,  3476,  3483,  3490,  3497,  3503,  3511,  3518,
        3525,  3532,  3539,  3546,  3553,  3560,  3567,  3574,  3581,
        3588,  3595,  3602,  3609,  3617,  3624,  3631,  3638,  3645,
        3652,  3659,  3666,  3673,  3680,  3687,  3694,  3701,  3708,
        3715,  3722,  3729,  3736,  3743,  3750,  3757,  3764,  3771,
        3778,  3785,  3792,  3799,  3806,  3813,  3820,  3827,  3834,
        3841,  3848,  3855,  3862,  3869,  3876,  3882,  3889,  3896,
        3903,  3910,  3917,  3924,  3931,  3938,  3945,  3952,  3959,
        3966,  3973,  3980,  3987,  3994,  4001,  4008,  4015,  4022,
        4029,  4036,  4043,  4050,  4057,  4064,  4071,  4078,  4085,
        4092,  4099,  4106,  4112,  4119,  4126,  4133,  4140,  4147,
        4154,  4161,  4168,  4175,  4182,  4190,  4197,  4204,  4211,
        4219,  4226,  4233,  4240,  4247,  4254,  4261,  4268,  4275,
        4282,  4289,  4296,  4303,  4310,  4317,  4324,  4331,  4338,
        4345,  4352,  4359,  4366,  4373,  4380,  4387,  4395,  4402,
        4409,  4416,  4423,  4430,  4437,  4444,  4451,  4458,  4466,
        4473,  4480,  4486,  4493,  4500,  4507,  4514,  4521,  4528,
        4535,  4542,  4549,  4556,  4563,  4570,  4577,  4584,  4591,
        4598,  4605,  4612,  4619,  4626,  4634,  4641,  4648,  4655,
        4662,  4669,  4676,  4683,  4690,  4697,  4704,  4711,  4718,
        4725,  4732,  4739,  4746,  4754,  4760,  4767,  4774,  4781,
        4788,  4795,  4802,  4809,  4816,  4823,  4830,  4837,  4844,
        4851,  4858,  4865,  4872,  4878,  4885,  4892,  4899,  4906,
        4913,  4920,  4927,  4933,  4939,  4946,  4953,  4960,  4967,
        4974,  4981,  4988,  4995,  5002,  5009,  5016,  5023,  5030,
        5037,  5044,  5051,  5058,  5065,  5072,  5079,  5086,  5093,
        5100,  5107,  5114,  5120,  5127,  5134,  5141,  5148,  5155,
        5162,  5169,  5176,  5183,  5190,  5198,  5205,  5212,  5219,
        5226,  5233,  5240,  5247,  5254,  5261,  5268,  5275,  5282,
        5289,  5296,  5303,  5310,  5317,  5324,  5331,  5338,  5345,
        5352,  5359,  5366,  5373,  5380,  5387,  5394,  5401,  5408,
        5415,  5422,  5430,  5438,  5445,  5452,  5460,  5466,  5473,
        5480,  5487,  5494,  5501,  5508,  5515,  5522,  5529,  5536,
        5543,  5550,  5557,  5564,  5571,  5578,  5585,  5592,  5599,
        5606,  5613,  5620,  5627,  5634,  5641,  5648,  5655,  5662,
        5669,  5676,  5683,  5690,  5697,  5704,  5711,  5718,  5725,
        5732,  5739,  5746,  5753,  5760,  5767,  5774,  5781,  5788,
        5795,  5802,  5809,  5816,  5823,  5830,  5837,  5844,  5851,
        5858,  5865,  5872,  5879,  5886,  5893,  5900,  5907,  5914,
        5921,  5928,  5935,  5942,  5949,  5956,  5963,  5970,  5977,
        5984,  5990,  5997,  6004,  6011,  6018,  6025,  6032,  6038,
        6045,  6052,  6059,  6066,  6073,  6080,  6087,  6094,  6101,
        6108,  6115,  6122,  6129,  6135,  6142,  6149,  6156,  6163,
        6170,  6177,  6184,  6191,  6198,  6205,  6212,  6219,  6226,
        6233,  6240,  6247,  6254,  6261,  6268,  6275,  6282,  6289,
        6296,  6303,  6310,  6317,  6324,  6331,  6338,  6345,  6352,
        6359,  6366,  6373,  6380,  6387,  6394,  6401,  6408,  6415,
        6422,  6429,  6436,  6443,  6450,  6457,  6464,  6471,  6478,
        6484,  6492,  6499,  6506,  6513,  6520,  6527,  6534,  6541,
        6548,  6555,  6561,  6568,  6575,  6582,  6589,  6596,  6603,
        6610,  6617,  6624,  6631,  6639,  6645,  6652,  6659,  6666,
        6673,  6679,  6686,  6693,  6700,  6707,  6714,  6722,  6729,
        6736,  6743,  6750,  6757,  6764,  6771,  6778,  6785,  6792,
        6799,  6806,  6813,  6821,  6828,  6835,  6842,  6849,  6856,
        6863,  6870,  6877,  6884,  6891,  6898,  6905,  6912,  6919,
        6926,  6933,  6940,  6947,  6954,  6961,  6968,  6975,  6982,
        6989,  6996,  7003,  7010,  7017,  7023,  7031,  7038,  7045,
        7052,  7059,  7066,  7074,  7081,  7088,  7094,  7101,  7108,
        7115,  7122,  7128,  7135,  7142,  7149,  7156,  7163,  7170,
        7177,  7184,  7191,  7198,  7205,  7212,  7218,  7226,  7233,
        7240,  7247,  7254,  7261,  7268,  7275,  7282,  7289,  7296,
        7303,  7310,  7317,  7323,  7330,  7337,  7344,  7351,  7358,
        7365,  7372,  7379,  7386,  7393,  7400,  7407,  7414,  7421,
        7428,  7435,  7442,  7449,  7456,  7463,  7470,  7477,  7484,
        7491,  7498,  7505,  7512,  7519,  7526,  7533,  7540,  7547,
        7554,  7561,  7568,  7575,  7582,  7589,  7596,  7603,  7610,
        7618,  7625,  7633,  7640,  7647,  7654,  7661,  7668,  7675,
        7682,  7689,  7696,  7703,  7711,  7718,  7725,  7732,  7739,
        7746,  7753,  7760,  7767,  7774,  7781,  7788,  7795,  7802,
        7809,  7816,  7823,  7830,  7837,  7844,  7851,  7858,  7865,
        7872,  7879,  7886,  7893,  7900,  7907,  7914,  7921,  7928,
        7935,  7942,  7949,  7956,  7963,  7970,  7977,  7984,  7991,
        7998,  8005,  8012,  8019,  8026,  8033,  8040,  8046,  8054,
        8061,  8068,  8075,  8082,  8089,  8096,  8103,  8110,  8117,
        8124,  8131,  8138,  8145,  8152,  8159,  8166,  8173,  8181,
        8187,  8194,  8201,  8207,  8214,  8221,  8227,  8234,  8241,
        8248,  8255,  8262,  8269,  8276,  8283,  8290,  8298,  8305,
        8312,  8319,  8326,  8333,  8340,  8347,  8354,  8361,  8367,
        8374,  8381,  8388,  8395,  8402,  8409,  8416,  8423,  8430,
        8437,  8444,  8451,  8458,  8465,  8471,  8478,  8485,  8492,
        8499,  8506,  8513,  8520,  8527,  8534,  8541,  8548,  8555,
        8562,  8569,  8576,  8583,  8589,  8596,  8603,  8611,  8618,
        8626,  8633,  8640,  8647,  8654,  8661,  8668,  8675,  8682,
        8689,  8696,  8703,  8710,  8717,  8724,  8731,  8738,  8745,
        8752,  8759,  8766,  8772,  8779,  8786,  8793,  8800,  8807,
        8814,  8822,  8828,  8836,  8843,  8850,  8857,  8864,  8871,
        8878,  8885,  8892,  8899,  8906,  8913,  8920,  8927,  8934,
        8941,  8948,  8955,  8962,  8969,  8976,  8983,  8990,  8997,
        9004,  9011,  9018,  9025,  9032,  9040,  9047,  9054,  9061,
        9068,  9075,  9082,  9089,  9096,  9103,  9110,  9117,  9124,
        9131,  9138,  9145,  9152,  9159,  9166,  9173,  9180,  9187,
        9194,  9201,  9208,  9215,  9222,  9229,  9236,  9243,  9250,
        9257,  9264,  9271,  9278,  9285,  9292,  9299,  9306,  9313,
        9320,  9327,  9334,  9341,  9348,  9355,  9363,  9370,  9377,
        9384,  9391,  9398,  9405,  9411,  9418,  9425,  9432,  9439,
        9446,  9453,  9460,  9467,  9474,  9481,  9487,  9494,  9501,
        9508,  9515,  9522,  9529,  9536,  9543,  9550,  9557,  9564,
        9571,  9578,  9586,  9593,  9600,  9607,  9614,  9621,  9628,
        9635,  9642,  9649,  9656,  9663,  9670,  9677,  9684,  9691,
        9698,  9704,  9712,  9719,  9726,  9733,  9740,  9747,  9754,
        9761,  9768,  9775,  9782,  9789,  9797,  9804,  9811,  9818,
        9825,  9832,  9839,  9846,  9853,  9860,  9867,  9874,  9881,
        9888,  9895,  9902,  9909,  9916,  9923,  9930,  9937,  9944,
        9951,  9958,  9965,  9972,  9979,  9986,  9993, 10001, 10008,
       10015, 10022, 10029, 10036, 10043, 10050, 10057, 10064, 10071,
       10078, 10085, 10092, 10098, 10105, 10112, 10119, 10126, 10133,
       10140, 10147, 10154, 10161, 10168, 10175, 10182, 10189, 10196,
       10203, 10210, 10217, 10224, 10231, 10238, 10245, 10252, 10259,
       10266, 10273, 10280, 10287, 10295, 10302, 10309, 10316, 10323,
       10330, 10337, 10344, 10351, 10358, 10365, 10372, 10379, 10386,
       10392, 10399, 10406, 10413, 10420, 10427, 10434, 10441, 10448,
       10455, 10462, 10469, 10476, 10483, 10490, 10497, 10504, 10511,
       10518, 10525, 10532, 10539, 10546, 10553, 10561, 10568, 10575,
       10582, 10589, 10596, 10603, 10610, 10617, 10624, 10631, 10638,
       10645, 10652, 10659, 10666, 10673, 10680, 10687, 10694, 10701,
       10708, 10715, 10722, 10729, 10736, 10743, 10750, 10757, 10764,
       10771, 10778, 10785, 10792, 10799, 10806, 10812, 10820, 10829,
       10836, 10843, 10849, 10856, 10863, 10870, 10877, 10885, 10892,
       10899, 10906, 10913, 10920, 10926, 10933, 10940, 10947, 10954,
       10961, 10968, 10975, 10982, 10989, 10996, 11003, 11010, 11017,
       11024, 11031, 11038, 11045, 11052, 11059, 11066, 11073, 11080,
       11087, 11094, 11101, 11108, 11115, 11122, 11129, 11136, 11144,
       11151, 11158, 11165, 11172, 11179, 11186, 11193, 11200, 11206,
       11214, 11220, 11227, 11235, 11242, 11249, 11256, 11263, 11270,
       11277, 11283, 11290, 11297, 11304, 11311, 11318, 11325, 11332,
       11339, 11346, 11353, 11360, 11367, 11374, 11381, 11388, 11395,
       11402, 11409, 11416, 11423, 11430, 11437, 11444, 11451, 11458,
       11465, 11472, 11478, 11485, 11492, 11499, 11506, 11513, 11520,
       11527, 11534, 11541, 11548, 11555, 11562, 11569, 11576, 11583,
       11590, 11597, 11604, 11611, 11618, 11625, 11632, 11639, 11646,
       11653, 11660, 11667, 11674, 11681, 11688, 11695, 11702, 11709,
       11716, 11723, 11730, 11738, 11745, 11752, 11759, 11766, 11773,
       11780, 11787, 11793, 11800, 11807, 11814, 11821, 11828, 11835,
       11842, 11849, 11856, 11863, 11870, 11877, 11884, 11891, 11898,
       11905, 11912, 11919, 11926, 11933, 11940, 11947, 11953, 11960,
       11967, 11974, 11981, 11987, 11994, 12001, 12008, 12015, 12022,
       12029, 12036, 12043, 12050, 12057, 12064, 12071, 12078, 12085,
       12092, 12099, 12107, 12114, 12121, 12128, 12135, 12142, 12149,
       12156, 12163, 12170, 12177, 12184, 12191, 12198, 12205, 12212,
       12219, 12226, 12233, 12241, 12248, 12255, 12262, 12269, 12276,
       12283, 12290, 12297, 12304, 12311, 12318, 12325, 12332, 12339,
       12346, 12353, 12359, 12366, 12373, 12380, 12387, 12394, 12401,
       12408, 12414, 12421, 12428, 12435, 12442, 12449, 12456, 12463,
       12470, 12477, 12484, 12491, 12499, 12507, 12513, 12520, 12526,
       12533, 12540, 12547, 12554, 12561, 12568, 12575, 12582, 12589,
       12596, 12603, 12610, 12618, 12625, 12632, 12639, 12646, 12653,
       12660, 12667, 12674, 12681, 12688, 12695, 12702, 12709, 12716,
       12723, 12730, 12737, 12744, 12751, 12758, 12765, 12772, 12779,
       12786, 12793, 12799, 12806, 12813, 12820, 12827, 12834, 12841,
       12848, 12855, 12862, 12869, 12874, 12881, 12888, 12895, 12902,
       12909, 12916, 12923, 12930, 12937, 12944, 12951, 12958, 12965,
       12972, 12980, 12987, 12994, 13001, 13008, 13015, 13022, 13030,
       13037, 13044, 13051, 13058, 13065, 13072, 13079, 13086, 13093,
       13100, 13107, 13114, 13121, 13128, 13135, 13142, 13148, 13155,
       13162, 13169, 13176, 13183, 13190, 13197, 13204, 13211, 13218,
       13225, 13232, 13239, 13246, 13253, 13260, 13267, 13274, 13281,
       13288, 13295, 13302, 13309, 13316, 13323, 13330, 13337, 13344,
       13351, 13358, 13365, 13372, 13379, 13386, 13393, 13400, 13407,
       13414, 13421, 13428, 13435, 13442, 13449, 13455, 13462, 13469,
       13476, 13483, 13490, 13497, 13504, 13511, 13518, 13525, 13532,
       13539, 13546, 13553, 13560, 13567, 13574, 13581, 13588, 13595,
       13602, 13609, 13617, 13624, 13631, 13638, 13645, 13652, 13659,
       13666, 13673, 13680, 13687, 13694, 13701, 13708, 13715, 13722,
       13729, 13736, 13743, 13750, 13757, 13764, 13771, 13778, 13785,
       13792, 13798, 13805, 13811, 13818, 13825, 13832, 13838, 13845,
       13852, 13859, 13866, 13873, 13880, 13887, 13894, 13901, 13908,
       13915, 13922, 13929, 13936, 13943, 13950, 13957, 13964, 13971,
       13978, 13985, 13992, 13999, 14006, 14013, 14020, 14027, 14034,
       14041, 14048, 14055, 14062, 14069, 14076, 14083, 14090, 14097,
       14104, 14111, 14118, 14125, 14133, 14140, 14147, 14154, 14161,
       14168, 14175, 14182, 14189, 14196, 14203, 14211, 14218, 14224,
       14231, 14238, 14245, 14252, 14259, 14266, 14273, 14280, 14287,
       14294, 14301, 14308, 14315, 14322, 14329, 14336, 14343, 14350,
       14356, 14363, 14370, 14377, 14384, 14391, 14398, 14405, 14413,
       14420, 14427, 14434, 14441, 14448, 14455, 14462, 14469, 14476,
       14483, 14490, 14497, 14504, 14511, 14518, 14525, 14532, 14539,
       14546, 14553, 14560, 14567, 14573, 14580, 14587, 14594, 14601,
       14609, 14616, 14623, 14631, 14638, 14645, 14652, 14659, 14666,
       14673, 14680, 14687, 14694, 14701, 14708, 14714, 14721, 14728,
       14735, 14742, 14749, 14756, 14763, 14770, 14777, 14784, 14791,
       14799, 14806, 14813, 14820, 14827, 14834, 14841, 14848, 14855,
       14862, 14869, 14876, 14883, 14890, 14897, 14904, 14911, 14918,
       14924, 14931, 14938, 14945, 14952, 14959, 14966, 14972, 14979,
       14986, 14993, 15000, 15007, 15014, 15021, 15028, 15035, 15042,
       15049, 15056, 15062, 15069, 15076, 15083, 15090, 15097, 15104,
       15111, 15118, 15125, 15132, 15139, 15146, 15153, 15160, 15167,
       15175, 15182, 15189, 15196, 15203, 15210, 15218, 15225, 15232,
       15239, 15246, 15253, 15260, 15267, 15274, 15281, 15288, 15295,
       15302, 15309, 15316, 15323, 15330, 15337, 15344, 15351, 15358,
       15365, 15372, 15379, 15386, 15393, 15400, 15407, 15414, 15422,
       15429, 15436, 15443, 15450, 15457, 15464, 15472, 15479, 15486,
       15493, 15500, 15507, 15514, 15521, 15528, 15535, 15542, 15549,
       15557, 15563, 15570, 15577, 15584, 15591, 15600, 15607, 15614,
       15621, 15628, 15635, 15642, 15650, 15657, 15664, 15671, 15678,
       15685, 15692, 15699, 15706, 15713, 15719, 15726, 15733, 15740,
       15747, 15754, 15762, 15769, 15776, 15783, 15790, 15797, 15804,
       15811, 15818, 15825, 15832, 15839, 15846, 15853, 15859, 15866,
       15873, 15880, 15887, 15894, 15901, 15908, 15915, 15922, 15929,
       15936, 15943, 15950, 15957, 15964, 15971, 15978, 15985, 15992,
       15999, 16006, 16013, 16020, 16027, 16034, 16040, 16047, 16054,
       16061, 16068, 16075, 16082, 16089, 16096, 16103, 16110, 16117,
       16124, 16131, 16138, 16145, 16151, 16157, 16164, 16171, 16178,
       16185, 16192, 16199, 16206, 16213, 16220, 16227, 16234, 16241,
       16248, 16255, 16262, 16269, 16276, 16283, 16290, 16297, 16304,
       16311, 16318, 16324, 16331, 16337, 16344, 16351, 16358, 16365,
       16372, 16379, 16386, 16394, 16401, 16408, 16415, 16422, 16429,
       16436, 16443, 16450, 16457, 16464, 16471, 16478, 16485, 16492,
       16499, 16506, 16513, 16520, 16527, 16534, 16541, 16548, 16555,
       16562, 16569, 16576, 16583, 16590, 16597, 16604, 16611, 16618,
       16625, 16632, 16639, 16646, 16653, 16660, 16668, 16675, 16683,
       16690, 16697, 16704, 16711, 16718, 16725, 16732, 16739, 16746,
       16753, 16760, 16767, 16774, 16781, 16788, 16795, 16802, 16809,
       16816, 16823, 16830, 16837, 16844, 16851, 16858, 16865, 16872,
       16879, 16886, 16893, 16900, 16908, 16915, 16923, 16930, 16937,
       16944, 16951, 16958, 16965, 16972, 16979, 16986, 16993, 16999,
       17006, 17013, 17020, 17027, 17034, 17041, 17048, 17055, 17062,
       17069, 17076, 17083, 17090, 17097, 17104, 17111, 17118, 17126,
       17132, 17139, 17146, 17153, 17160, 17167, 17174, 17181, 17187,
       17194, 17201, 17208, 17215, 17222, 17229, 17236, 17243, 17250,
       17257, 17264, 17271, 17278, 17285, 17292, 17299, 17306, 17313,
       17320, 17327, 17334, 17341, 17348, 17355, 17362, 17369, 17376,
       17383, 17390, 17397, 17404, 17411, 17418, 17425, 17432, 17439,
       17446, 17453, 17460, 17467, 17474, 17481, 17488, 17495, 17502,
       17509, 17516, 17523, 17530, 17537, 17544, 17551, 17558, 17565,
       17572, 17579, 17586, 17593, 17601, 17608, 17615, 17622, 17629,
       17636, 17643, 17650, 17657, 17664, 17671, 17678, 17685, 17692,
       17699, 17706, 17713, 17720, 17727, 17734, 17741, 17748, 17755,
       17762, 17769, 17776, 17783, 17790, 17797, 17804, 17811, 17817,
       17824, 17831, 17838, 17845, 17852, 17859, 17866, 17873, 17880,
       17888, 17895, 17902, 17909, 17916, 17923, 17930, 17937, 17944,
       17951, 17958, 17965, 17972, 17979, 17986, 17993, 18000, 18007,
       18015, 18022, 18029, 18036, 18043, 18050, 18057, 18064, 18071,
       18078, 18085, 18092, 18098, 18105, 18112, 18119, 18126, 18133,
       18140, 18147, 18154, 18161, 18168, 18175, 18182, 18189, 18196,
       18203, 18210, 18217, 18224, 18231, 18238, 18245, 18252, 18259,
       18266, 18273, 18280, 18287, 18294, 18301, 18308, 18315, 18322,
       18329, 18336, 18343, 18350, 18357, 18364, 18372, 18379, 18386,
       18393, 18400, 18407, 18414, 18421, 18428, 18435, 18443, 18450,
       18457, 18464, 18471, 18478, 18485, 18492, 18499, 18506, 18512,
       18519, 18526, 18533, 18540, 18547, 18554, 18561, 18568, 18575,
       18582, 18589, 18596, 18603, 18610, 18617, 18623, 18630, 18637,
       18644, 18651, 18657, 18664, 18671, 18678, 18685, 18692, 18699,
       18706, 18713, 18720, 18728, 18734, 18741, 18748, 18755, 18762,
       18769, 18776, 18783, 18790, 18797, 18804, 18811, 18818, 18825,
       18832, 18839, 18846, 18853, 18859, 18867, 18874, 18881, 18888,
       18895, 18902, 18909, 18916, 18923, 18930, 18937, 18944, 18951,
       18958, 18966, 18973, 18980, 18987, 18994, 19001, 19007, 19014,
       19021, 19028, 19035, 19042, 19049, 19056, 19064, 19071, 19078,
       19085, 19092, 19099, 19106, 19113, 19120, 19127, 19134, 19142,
       19149, 19156, 19163, 19170, 19177, 19184, 19191, 19198, 19205,
       19212, 19219, 19226, 19233, 19240, 19247, 19255, 19262, 19269,
       19276, 19283, 19290, 19297, 19304, 19311, 19318, 19325, 19332,
       19339, 19346, 19353, 19360, 19368, 19375, 19382, 19389, 19396,
       19403, 19410, 19417, 19424, 19431, 19438, 19445, 19452, 19460,
       19467, 19474, 19481, 19488, 19495, 19502, 19509, 19516, 19523,
       19530, 19537, 19544, 19551, 19558, 19565, 19572, 19579, 19586,
       19593, 19600, 19606, 19613, 19620, 19627, 19634, 19641, 19648,
       19655, 19662, 19669, 19676, 19683, 19690, 19697, 19704, 19711,
       19718, 19725, 19732, 19739, 19745, 19752, 19759, 19766, 19773,
       19780, 19787, 19794, 19801, 19808, 19815, 19822, 19829, 19837,
       19844, 19851, 19858, 19865, 19872, 19879, 19886, 19893, 19900,
       19907, 19914, 19921, 19928, 19936, 19943, 19950, 19957, 19964,
       19971, 19978, 19985, 19993, 20000, 20007, 20014, 20022, 20029,
       20036, 20043, 20050, 20057, 20064, 20071, 20078, 20085, 20092,
       20099, 20106, 20113, 20120, 20127, 20134, 20140, 20147, 20154,
       20161, 20168, 20175, 20182, 20189, 20196, 20203, 20210, 20217,
       20224, 20232, 20239, 20246, 20253, 20260, 20267, 20274, 20281,
       20288, 20295, 20302, 20309, 20316, 20323, 20330, 20337, 20344,
       20351, 20358, 20365, 20372, 20379, 20386, 20393, 20399, 20406,
       20413, 20421, 20428, 20435, 20441, 20448, 20455, 20462, 20469,
       20476, 20483, 20491, 20498, 20504, 20512, 20519, 20525, 20532,
       20539, 20546, 20553, 20560, 20567, 20574, 20581, 20587, 20594,
       20601, 20608, 20615, 20622, 20629, 20636, 20644, 20650, 20656,
       20663, 20670, 20677, 20684, 20691, 20699, 20706, 20713, 20720,
       20727, 20734, 20741, 20748, 20755, 20762, 20769, 20776, 20783,
       20790, 20797, 20805, 20812, 20819, 20826, 20833, 20840, 20847,
       20853, 20860, 20867, 20874, 20881, 20888, 20895, 20902, 20908,
       20915, 20922, 20929, 20936, 20943, 20950, 20957, 20964, 20971,
       20978, 20986, 20993, 21001, 21008, 21015, 21022, 21029, 21036,
       21043, 21050, 21057, 21064, 21071, 21078, 21085, 21092, 21099,
       21106, 21113, 21120, 21127, 21134, 21141, 21148, 21155, 21162,
       21169, 21176, 21183, 21190, 21197, 21204, 21211, 21218, 21225,
       21232, 21239, 21246, 21253, 21260, 21267, 21274, 21281, 21288,
       21295, 21302, 21309, 21316, 21323, 21330, 21337, 21344, 21351,
       21358, 21365, 21372, 21378, 21385, 21392, 21399, 21406, 21413,
       21420, 21427, 21434, 21441, 21448, 21455, 21462, 21469, 21476,
       21483, 21490, 21497, 21504, 21511, 21518, 21525, 21532, 21538,
       21545, 21552, 21559, 21566, 21573, 21580, 21587, 21594, 21601,
       21608, 21615, 21622, 21629, 21636, 21643, 21651, 21658, 21665,
       21672, 21679, 21686, 21693, 21700, 21707, 21714, 21721, 21728,
       21735, 21742, 21749, 21756, 21763, 21770, 21777, 21784, 21791,
       21798, 21805, 21812, 21819, 21827, 21834, 21841, 21848, 21855,
       21862, 21869, 21876, 21883, 21890, 21897, 21904, 21911, 21918,
       21925, 21932, 21938, 21945, 21952, 21959, 21966, 21973, 21980,
       21987, 21994, 22001, 22008, 22015, 22022, 22029, 22036, 22043,
       22050, 22057, 22064, 22071, 22078, 22085, 22092, 22099, 22106,
       22113, 22120, 22127, 22133, 22140, 22147, 22154, 22161, 22168,
       22175, 22182, 22189, 22196, 22203, 22210, 22217, 22224, 22231,
       22238, 22245, 22252, 22259, 22266, 22273, 22280, 22287, 22294,
       22301, 22308, 22315, 22322, 22329, 22336, 22343, 22350, 22357,
       22364, 22371, 22378, 22385, 22392, 22399, 22406, 22413, 22419,
       22426, 22433, 22440, 22447, 22453, 22459, 22466, 22473, 22480,
       22487, 22494, 22501, 22508, 22515, 22522, 22529, 22536, 22543,
       22550, 22557, 22564, 22571, 22578, 22585, 22592, 22599, 22605,
       22612, 22619, 22626, 22633, 22640, 22647, 22654, 22661, 22668,
       22675, 22682, 22689, 22696, 22703, 22710, 22717, 22724, 22731,
       22737, 22744, 22751, 22758, 22765, 22772, 22778, 22785, 22792,
       22799, 22806, 22813, 22820, 22827, 22834, 22841, 22848, 22855,
       22862, 22869, 22876, 22883, 22889, 22896, 22903, 22910, 22917,
       22924, 22931, 22938, 22945, 22952, 22959, 22966, 22973, 22980,
       22987, 22994, 23001, 23008, 23014, 23021, 23028, 23035, 23042,
       23048, 23054, 23061, 23068, 23074, 23081, 23088, 23095],
      dtype=np.int32)


        M_b_gt = sp.sparse.csr_matrix((data_b_gt, indices_b_gt, indptr_b_gt))
        M_v_gt = sp.sparse.csr_matrix((data_v_gt, indices_v_gt, indptr_v_gt))
        M_f_gt = sp.sparse.csr_matrix((data_f_gt, indices_f_gt, indptr_f_gt))

        self.assertTrue(np.isclose(sp.sparse.linalg.norm(M_b-M_b_gt) / M_b.nnz, 0.))
        self.assertTrue(np.isclose(sp.sparse.linalg.norm(M_v-M_v_gt) / M_v.nnz, 0.))
        self.assertTrue(np.isclose(sp.sparse.linalg.norm(M_d-M_v), 0.))
        self.assertTrue(np.isclose(sp.sparse.linalg.norm(M_f-M_f_gt) / M_f.nnz, 0.))


if __name__ == '__main__':
    unittest.main()