from .context import gpytoolbox
from .context import numpy as np
from .context import unittest
from scipy.sparse import csr_matrix

class TestGrad(unittest.TestCase):
    def test_polyline_grad(self):
        # This is a cube, centered at the origin, with side length 1
        # v,f = gpy.read_mesh("test/unit_tests_data/cube.obj")
        #
        # Let's make up a simple polyline
        v = np.array([[0],[0.2],[0.5],[0.98],[1.0]])
        edge_centers = (v[0:4,:] + v[1:5,:])/2.0
        fun_zero_grad = 0*v + 5
        fun_constant_grad = 2*v
        fun_other_grad = v**2.0
        G = gpytoolbox.grad(v)
        # Finite elements should get exact gradients in these
        self.assertTrue(np.isclose((G @ fun_zero_grad),0.0).all())
        self.assertTrue(np.isclose((G @ fun_constant_grad) - 2.0,0.0).all())
        self.assertTrue(np.isclose((G @ fun_other_grad) - 2.0*edge_centers,0.0).all())

    def test_single_triangle_2d(self):
        v = np.array([[0.0,0.0],[1.0,0.0],[0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        G = gpytoolbox.grad(v,f)
        G_gt = np.array([[-1.0,1.0,0.0],[-1,0.0,1.0]])
        #print(G)
        #print(csr_matrix(igl.grad(np.hstack((v,np.zeros((v.shape[0],1)))),f)))
        self.assertTrue(np.isclose(G.toarray() - G_gt,0.0).all())
        # print(G-igl.grad(np.hstack((v,np.zeros((v.shape[0],1)))),f))
    
    def test_single_triangle_3d(self):
        v = np.array([[0.0,0.0,0.0],[1.0,0.0,0.0],[0.0,0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        G = gpytoolbox.grad(v,f)
        G_gt = np.array([[-1.0,1.0,0.0],[0.0,0.0,0.0],[-1,0.0,1.0]])
        self.assertTrue(np.isclose(G.toarray() - G_gt,0.0).all())
        # print(G-igl.grad(np.hstack((v,np.zeros((v.shape[0],1)))),f))

    def test_2d_grad(self):
        v,f = gpytoolbox.regular_square_mesh(400)
        barycenters = (v[f[:,0],:] + v[f[:,1],:] + v[f[:,2],:])/3.0

        fun_zero_grad = 0*v[:,0] + 5
        fun_constant_grad = v[:,0] + 2*v[:,1]
        fun_other_grad = v[:,0]**2.0 + 5*v[:,1]**2.0 + 3.0*v[:,0]
        G = gpytoolbox.grad(v,f)
        # Finite elements should get exact gradients if they are analytically piecewise linear
        self.assertTrue(np.isclose((G @ fun_zero_grad),0.0).all())
        self.assertTrue(np.isclose((G @ fun_constant_grad)[0:f.shape[0]] - 1.0,0.0).all())
        self.assertTrue(np.isclose((G @ fun_constant_grad)[f.shape[0]:(2*f.shape[0])] - 2.0,0.0).all())
        # For a function without constant gradients it shouldn't be exact but it should converge
        err = 1.0
        for i in range(2,10):
            v,f = gpytoolbox.regular_square_mesh(2**i)
            barycenters = (v[f[:,0],:] + v[f[:,1],:] + v[f[:,2],:])/3.0
            fun_other_grad = v[:,0]**2.0 + 5*v[:,1]**2.0 + 3.0*v[:,0]
            G = gpytoolbox.grad(v,f)
            self.assertTrue(np.amax(np.abs((G @ fun_other_grad)[0:f.shape[0]] - (2.0*barycenters[:,0] + 3)))<err )
            err = np.amax(np.abs((G @ fun_other_grad)[0:f.shape[0]] - (2.0*barycenters[:,0] + 3)))

    def test_3d_grad(self):
        v,f = gpytoolbox.read_mesh("test/unit_tests_data/bunny_oded.obj")
        #v = v[:,0:2]
        G = gpytoolbox.grad(v,f).toarray()
        #G_igl = csr_matrix(igl.grad(v,f)).toarray()
        I = np.array([1,2,4,5,9,12,6599,6600,6602,6603,6607,6610,13197,13198,13200,13201,13205,13208,1,2,3,6,7,6599,6600,6601,6604,6605,13197,13198,13199,13202,13203,1,3,4,8,10,15,18,6599,6601,6602,6606,6608,6613,6616,13197,13199,13200,13204,13206,13211,13214,2,5,6,11,13,21,6600,6603,6604,6609,6611,6619,13198,13201,13202,13207,13209,13217,3,7,8,14,16,25,6601,6605,6606,6612,6614,6623,13199,13203,13204,13210,13212,13221,4,9,10,17,19,29,6602,6607,6608,6615,6617,6627,13200,13205,13206,13213,13215,13225,5,11,12,20,22,33,6603,6609,6610,6618,6620,6631,13201,13207,13208,13216,13218,13229,6,7,13,14,23,24,6604,6605,6611,6612,6621,6622,13202,13203,13209,13210,13219,13220,8,15,16,26,27,39,6606,6613,6614,6624,6625,6637,13204,13211,13212,13222,13223,13235,9,12,17,22,28,35,6607,6610,6615,6620,6626,6633,13205,13208,13213,13218,13224,13231,10,18,19,30,31,45,6608,6616,6617,6628,6629,6643,13206,13214,13215,13226,13227,13241,11,20,21,32,34,48,6609,6618,6619,6630,6632,6646,13207,13216,13217,13228,13230,13244,13,21,23,34,36,50,6611,6619,6621,6632,6634,6648,13209,13217,13219,13230,13232,13246,14,24,25,37,38,54,6612,6622,6623,6635,6636,6652,13210,13220,13221,13233,13234,13250,15,18,26,30,40,44,6613,6616,6624,6628,6638,6642,13211,13214,13222,13226,13236,13240,16,25,27,38,41,55,6614,6623,6625,6636,6639,6653,13212,13221,13223,13234,13237,13251,17,28,29,42,43,59,6615,6626,6627,6640,6641,6657,13213,13224,13225,13238,13239,13255,19,29,31,43,46,61,6617,6627,6629,6641,6644,6659,13215,13225,13227,13239,13242,13257,20,32,33,47,49,65,6618,6630,6631,6645,6647,6663,13216,13228,13229,13243,13245,13261,22,33,35,49,51,68,6620,6631,6633,6647,6649,6666,13218,13229,13231,13245,13247,13264,23,24,36,37,52,53,6621,6622,6634,6635,6650,6651,13219,13220,13232,13233,13248,13249,26,39,40,56,57,75,6624,6637,6638,6654,6655,6673,13222,13235,13236,13252,13253,13271,27,39,41,56,58,76,6625,6637,6639,6654,6656,6674,13223,13235,13237,13252,13254,13272,28,35,42,51,60,70,6626,6633,6640,6649,6658,6668,13224,13231,13238,13247,13256,13266,30,44,45,62,63,82,6628,6642,6643,6660,6661,6680,13226,13240,13241,13258,13259,13278,31,45,46,63,64,84,6629,6643,6644,6661,6662,6682,13227,13241,13242,13259,13260,13280,32,47,48,66,67,87,6630,6645,6646,6664,6665,6685,13228,13243,13244,13262,13263,13283,34,48,50,67,69,89,6632,6646,6648,6665,6667,6687,13230,13244,13246,13263,13265,13285,36,50,52,69,71,91,6634,6648,6650,6667,6669,6689,13232,13246,13248,13265,13267,13287,37,53,54,72,73,95,6635,6651,6652,6670,6671,6693,13233,13249,13250,13268,13269,13291,38,54,55,73,74,96,6636,6652,6653,6671,6672,6694,13234,13250,13251,13269,13270,13292,40,44,57,62,77,83,6638,6642,6655,6660,6675,6681,13236,13240,13253,13258,13273,13279,41,55,58,74,78,97,6639,6653,6656,6672,6676,6695,13237,13251,13254,13270,13274,13293,42,59,60,79,80,102,6640,6657,6658,6677,6678,6700,13238,13255,13256,13275,13276,13298,43,59,61,79,81,103,6641,6657,6659,6677,6679,6701,13239,13255,13257,13275,13277,13299,46,61,64,81,85,105,6644,6659,6662,6679,6683,6703,13242,13257,13260,13277,13281,13301,47,65,66,86,88,111,6645,6663,6664,6684,6686,6709,13243,13261,13262,13282,13284,13307,49,65,68,86,90,110,6647,6663,6666,6684,6688,6708,13245,13261,13264,13282,13286,13306,51,68,70,90,92,115,6649,6666,6668,6688,6690,6713,13247,13264,13266,13286,13288,13311,52,53,71,72,93,94,6650,6651,6669,6670,6691,6692,13248,13249,13267,13268,13289,13290,56,75,76,98,99,123,6654,6673,6674,6696,6697,6721,13252,13271,13272,13294,13295,13319,57,75,77,98,100,124,6655,6673,6675,6696,6698,6722,13253,13271,13273,13294,13296,13320,58,76,78,99,101,125,6656,6674,6676,6697,6699,6723,13254,13272,13274,13295,13297,13321,60,70,80,92,104,117,6658,6668,6678,6690,6702,6715,13256,13266,13276,13288,13300,13313,62,82,83,106,107,133,6660,6680,6681,6704,6705,6731,13258,13278,13279,13302,13303,13329,63,82,84,106,108,132,6661,6680,6682,6704,6706,6730,13259,13278,13280,13302,13304,13328,64,84,85,108,109,135,6662,6682,6683,6706,6707,6733,13260,13280,13281,13304,13305,13331,66,87,88,112,113,139,6664,6685,6686,6710,6711,6737,13262,13283,13284,13308,13309,13335,67,87,89,112,114,140,6665,6685,6687,6710,6712,6738,13263,13283,13285,13308,13310,13336,69,89,91,114,116,142,6667,6687,6689,6712,6714,6740,13265,13285,13287,13310,13312,13338,71,91,93,116,118,144,6669,6689,6691,6714,6716,6742,13267,13287,13289,13312,13314,13340,72,94,95,119,120,147,6670,6692,6693,6717,6718,6745,13268,13290,13291,13315,13316,13343,73,95,96,120,121,149,6671,6693,6694,6718,6719,6747,13269,13291,13292,13316,13317,13345,74,96,97,121,122,150,6672,6694,6695,6719,6720,6748,13270,13292,13293,13317,13318,13346,77,83,100,107,126,134,6675,6681,6698,6705,6724,6732,13273,13279,13296,13303,13322,13330,78,97,101,122,127,151,6676,6695,6699,6720,6725,6749,13274,13293,13297,13318,13323,13347,79,102,103,128,129,158,160,6677,6700,6701,6726,6727,6756,6758,13275,13298,13299,13324,13325,13354,13356,80,102,104,128,130,157,6678,6700,6702,6726,6728,6755,13276,13298,13300,13324,13326,13353,81,103,105,129,131,159,6679,6701,6703,6727,6729,6757,13277,13299,13301,13325,13327,13355,85,105,109,131,136,162,6683,6703,6707,6729,6734,6760,13281,13301,13305,13327,13332,13358,86,110,111,137,138,168,6684,6708,6709,6735,6736,6766,13282,13306,13307,13333,13334,13364,88,111,113,138,141,169,6686,6709,6711,6736,6739,6767,13284,13307,13309,13334,13337,13365,90,110,115,137,143,6688,6708,6713,6735,6741,13286,13306,13311,13333,13339,92,115,117,143,145,174,6690,6713,6715,6741,6743,6772,13288,13311,13313,13339,13341,13370,93,94,118,119,146,148,178,6691,6692,6716,6717,6744,6746,6776,13289,13290,13314,13315,13342,13344,13374,98,123,124,152,153,184,185,6696,6721,6722,6750,6751,6782,6783,13294,13319,13320,13348,13349,13380,13381,99,123,125,152,154,6697,6721,6723,6750,6752,13295,13319,13321,13348,13350,100,124,126,153,155,186,6698,6722,6724,6751,6753,6784,13296,13320,13322,13349,13351,13382,101,125,127,154,156,187,6699,6723,6725,6752,6754,6785,13297,13321,13323,13350,13352,13383,104,117,130,145,161,176,6702,6715,6728,6743,6759,6774,13300,13313,13326,13341,13357,13372,106,132,133,163,164,196,6704,6730,6731,6761,6762,6794,13302,13328,13329,13359,13360,13392,107,133,134,164,165,198,6705,6731,6732,6762,6763,6796,13303,13329,13330,13360,13361,13394,108,132,135,163,166,197,6706,6730,6733,6761,6764,6795,13304,13328,13331,13359,13362,13393,109,135,136,166,167,200,6707,6733,6734,6764,6765,6798,13305,13331,13332,13362,13363,13396,112,139,140,170,171,205,6710,6737,6738,6768,6769,6803,13308,13335,13336,13366,13367,13401,113,139,141,170,172,204,6711,6737,6739,6768,6770,6802,13309,13335,13337,13366,13368,13400,114,140,142,171,173,206,6712,6738,6740,6769,6771,6804,13310,13336,13338,13367,13369,13402,116,142,144,173,175,208,6714,6740,6742,6771,6773,6806,13312,13338,13340,13369,13371,13404,118,144,146,175,177,210,6716,6742,6744,6773,6775,6808,13314,13340,13342,13371,13373,13406,119,147,148,179,180,214,6717,6745,6746,6777,6778,6812,13315,13343,13344,13375,13376,13410,120,147,149,179,181,215,6718,6745,6747,6777,6779,6813,13316,13343,13345,13375,13377,13411,121,149,150,181,182,217,6719,6747,6748,6779,6780,6815,13317,13345,13346,13377,13378,13413,122,150,151,182,183,218,6720,6748,6749,6780,6781,6816,13318,13346,13347,13378,13379,13414,126,134,155,165,188,199,6724,6732,6753,6763,6786,6797,13322,13330,13351,13361,13384,13395,127,151,156,183,189,219,6725,6749,6754,6781,6787,6817,13323,13347,13352,13379,13385,13415,128,157,158,190,191,6726,6755,6756,6788,6789,13324,13353,13354,13386,13387,129,159,160,192,193,228,6727,6757,6758,6790,6791,6826,13325,13355,13356,13388,13389,13424,130,157,161,190,194,226,6728,6755,6759,6788,6792,6824,13326,13353,13357,13386,13390,13422,131,159,162,192,195,229,6729,6757,6760,6790,6793,6827,13327,13355,13358,13388,13391,13425,136,162,167,195,201,232,6734,6760,6765,6793,6799,6830,13332,13358,13363,13391,13397,13428,137,143,168,174,202,209,239,6735,6741,6766,6772,6800,6807,6837,13333,13339,13364,13370,13398,13405,13435,138,168,169,202,203,240,6736,6766,6767,6800,6801,6838,13334,13364,13365,13398,13399,13436,141,169,172,203,207,241,6739,6767,6770,6801,6805,6839,13337,13365,13368,13399,13403,13437,145,174,176,209,211,247,6743,6772,6774,6807,6809,6845,13341,13370,13372,13405,13407,13443,146,177,178,212,213,6744,6775,6776,6810,6811,13342,13373,13374,13408,13409,148,178,180,213,216,251,6746,6776,6778,6811,6814,6849,13344,13374,13376,13409,13412,13447,152,154,184,187,220,223,6750,6752,6782,6785,6818,6821,13348,13350,13380,13383,13416,13419,153,185,186,221,222,259,6751,6783,6784,6819,6820,6857,13349,13381,13382,13417,13418,13455,155,186,188,222,224,261,6753,6784,6786,6820,6822,6859,13351,13382,13384,13418,13420,13457,156,187,189,223,225,262,6754,6785,6787,6821,6823,6860,13352,13383,13385,13419,13421,13458,158,160,191,193,227,230,6756,6758,6789,6791,6825,6828,13354,13356,13387,13389,13423,13426,161,176,194,211,231,249,6759,6774,6792,6809,6829,6847,13357,13372,13390,13407,13427,13445,163,196,197,233,234,273,6761,6794,6795,6831,6832,6871,13359,13392,13393,13429,13430,13469,164,196,198,233,235,272,6762,6794,6796,6831,6833,6870,13360,13392,13394,13429,13431,13468,165,198,199,235,236,275,6763,6796,6797,6833,6834,6873,13361,13394,13395,13431,13432,13471,166,197,200,234,237,274,6764,6795,6798,6832,6835,6872,13362,13393,13396,13430,13433,13470,167,200,201,237,238,277,6765,6798,6799,6835,6836,6875,13363,13396,13397,13433,13434,13473,170,204,205,242,243,283,6768,6802,6803,6840,6841,6881,13366,13400,13401,13438,13439,13479,171,205,206,243,244,6769,6803,6804,6841,6842,13367,13401,13402,13439,13440,172,204,207,242,245,282,6770,6802,6805,6840,6843,6880,13368,13400,13403,13438,13441,13478,173,206,208,244,246,284,6771,6804,6806,6842,6844,6882,13369,13402,13404,13440,13442,13480,175,208,210,246,248,286,6773,6806,6808,6844,6846,6884,13371,13404,13406,13442,13444,13482,177,210,212,248,250,288,6775,6808,6810,6846,6848,6886,13373,13406,13408,13444,13446,13484,179,214,215,252,253,293,6777,6812,6813,6850,6851,6891,13375,13410,13411,13448,13449,13489,180,214,216,252,254,292,6778,6812,6814,6850,6852,6890,13376,13410,13412,13448,13450,13488,181,215,217,253,255,294,6779,6813,6815,6851,6853,6892,13377,13411,13413,13449,13451,13490,182,217,218,255,256,296,6780,6815,6816,6853,6854,6894,13378,13413,13414,13451,13452,13492,183,218,219,256,257,297,6781,6816,6817,6854,6855,6895,13379,13414,13415,13452,13453,13493,184,185,220,221,258,260,6782,6783,6818,6819,6856,6858,13380,13381,13416,13417,13454,13456,188,199,224,236,263,276,6786,6797,6822,6834,6861,6874,13384,13395,13420,13432,13459,13472,189,219,225,257,264,298,6787,6817,6823,6855,6862,6896,13385,13415,13421,13453,13460,13494,190,191,226,227,265,266,6788,6789,6824,6825,6863,6864,13386,13387,13422,13423,13461,13462,192,228,229,267,268,308,6790,6826,6827,6865,6866,6906,13388,13424,13425,13463,13464,13504,193,228,230,267,269,309,6791,6826,6828,6865,6867,6907,13389,13424,13426,13463,13465,13505,194,226,231,265,270,306,6792,6824,6829,6863,6868,6904,13390,13422,13427,13461,13466,13502,195,229,232,268,271,310,6793,6827,6830,6866,6869,6908,13391,13425,13428,13464,13467,13506,201,232,238,271,278,313,6799,6830,6836,6869,6876,6911,13397,13428,13434,13467,13474,13509,202,239,240,279,280,322,6800,6837,6838,6877,6878,6920,13398,13435,13436,13475,13476,13518,203,240,241,280,281,323,6801,6838,6839,6878,6879,6921,13399,13436,13437,13476,13477,13519,207,241,245,281,285,324,6805,6839,6843,6879,6883,6922,13403,13437,13441,13477,13481,13520,209,239,247,279,287,321,6807,6837,6845,6877,6885,6919,13405,13435,13443,13475,13483,13517,211,247,249,287,289,330,6809,6845,6847,6885,6887,6928,13407,13443,13445,13483,13485,13526,212,213,250,251,290,291,333,6810,6811,6848,6849,6888,6889,6931,13408,13409,13446,13447,13486,13487,13529,216,251,254,291,295,335,6814,6849,6852,6889,6893,6933,13412,13447,13450,13487,13491,13531,220,223,258,262,299,303,6818,6821,6856,6860,6897,6901,13416,13419,13454,13458,13495,13499,221,259,260,300,301,344,6819,6857,6858,6898,6899,6942,13417,13455,13456,13496,13497,13540,222,259,261,300,302,345,6820,6857,6859,6898,6900,6943,13418,13455,13457,13496,13498,13541,224,261,263,302,304,347,6822,6859,6861,6900,6902,6945,13420,13457,13459,13498,13500,13543,225,262,264,303,305,348,6823,6860,6862,6901,6903,6946,13421,13458,13460,13499,13501,13544,227,230,266,269,307,311,6825,6828,6864,6867,6905,6909,13423,13426,13462,13465,13503,13507,231,249,270,289,312,332,6829,6847,6868,6887,6910,6930,13427,13445,13466,13485,13508,13528,233,272,273,314,315,360,6831,6870,6871,6912,6913,6958,13429,13468,13469,13510,13511,13556,234,273,274,315,316,361,6832,6871,6872,6913,6914,6959,13430,13469,13470,13511,13512,13557,235,272,275,314,317,359,6833,6870,6873,6912,6915,6957,13431,13468,13471,13510,13513,13555,236,275,276,317,318,363,6834,6873,6874,6915,6916,6961,13432,13471,13472,13513,13514,13559,237,274,277,316,319,362,6835,6872,6875,6914,6917,6960,13433,13470,13473,13512,13515,13558,238,277,278,319,320,365,6836,6875,6876,6917,6918,6963,13434,13473,13474,13515,13516,13561,242,282,283,325,326,6840,6880,6881,6923,6924,13438,13478,13479,13521,13522,243,244,283,284,326,327,372,6841,6842,6881,6882,6924,6925,6970,13439,13440,13479,13480,13522,13523,13568,245,282,285,325,328,371,6843,6880,6883,6923,6926,6969,13441,13478,13481,13521,13524,13567,246,284,286,327,329,373,6844,6882,6884,6925,6927,6971,13442,13480,13482,13523,13525,13569,248,286,288,329,331,375,6846,6884,6886,6927,6929,6973,13444,13482,13484,13525,13527,13571,250,288,290,331,334,377,6848,6886,6888,6929,6932,6975,13446,13484,13486,13527,13530,13573,252,292,293,336,337,382,6850,6890,6891,6934,6935,6980,13448,13488,13489,13532,13533,13578,253,293,294,337,338,384,6851,6891,6892,6935,6936,6982,13449,13489,13490,13533,13534,13580,254,292,295,336,339,383,6852,6890,6893,6934,6937,6981,13450,13488,13491,13532,13535,13579,255,294,296,338,340,385,6853,6892,6894,6936,6938,6983,13451,13490,13492,13534,13536,13581,256,296,297,340,341,387,6854,6894,6895,6938,6939,6985,13452,13492,13493,13536,13537,13583,257,297,298,341,342,388,6855,6895,6896,6939,6940,6986,13453,13493,13494,13537,13538,13584,258,260,299,301,343,346,6856,6858,6897,6899,6941,6944,13454,13456,13495,13497,13539,13542,263,276,304,318,349,364,6861,6874,6902,6916,6947,6962,13459,13472,13500,13514,13545,13560,264,298,305,342,350,389,6862,6896,6903,6940,6948,6987,13460,13494,13501,13538,13546,13585,265,266,306,307,351,352,6863,6864,6904,6905,6949,6950,13461,13462,13502,13503,13547,13548,267,308,309,353,354,401,6865,6906,6907,6951,6952,6999,13463,13504,13505,13549,13550,13597,268,308,310,353,355,400,6866,6906,6908,6951,6953,6998,13464,13504,13506,13549,13551,13596,269,309,311,354,356,402,6867,6907,6909,6952,6954,7000,13465,13505,13507,13550,13552,13598,270,306,312,351,357,398,6868,6904,6910,6949,6955,6996,13466,13502,13508,13547,13553,13594,271,310,313,355,358,403,6869,6908,6911,6953,6956,7001,13467,13506,13509,13551,13554,13599,278,313,320,358,366,406,6876,6911,6918,6956,6964,7004,13474,13509,13516,13554,13562,13602,279,321,322,367,368,416,6877,6919,6920,6965,6966,7014,13475,13517,13518,13563,13564,13612,280,322,323,368,369,417,6878,6920,6921,6966,6967,7015,13476,13518,13519,13564,13565,13613,281,323,324,369,370,418,6879,6921,6922,6967,6968,7016,13477,13519,13520,13565,13566,13614,285,324,328,370,374,419,6883,6922,6926,6968,6972,7017,13481,13520,13524,13566,13570,13615,287,321,330,367,376,415,6885,6919,6928,6965,6974,7013,13483,13517,13526,13563,13572,13611,289,330,332,376,378,425,6887,6928,6930,6974,6976,7023,13485,13526,13528,13572,13574,13621,290,333,334,379,380,428,6888,6931,6932,6977,6978,7026,13486,13529,13530,13575,13576,13624,291,333,335,379,381,429,6889,6931,6933,6977,6979,7027,13487,13529,13531,13575,13577,13625,295,335,339,381,386,431,6893,6933,6937,6979,6984,7029,13491,13531,13535,13577,13582,13627,299,303,343,348,390,395,6897,6901,6941,6946,6988,6993,13495,13499,13539,13544,13586,13591,300,344,345,391,392,442,6898,6942,6943,6989,6990,7040,13496,13540,13541,13587,13588,13638,301,344,346,391,393,441,6899,6942,6944,6989,6991,7039,13497,13540,13542,13587,13589,13637,302,345,347,392,394,443,6900,6943,6945,6990,6992,7041,13498,13541,13543,13588,13590,13639,304,347,349,394,396,445,6902,6945,6947,6992,6994,7043,13500,13543,13545,13590,13592,13641,305,348,350,395,397,446,6903,6946,6948,6993,6995,7044,13501,13544,13546,13591,13593,13642,307,311,352,356,399,404,6905,6909,6950,6954,6997,7002,13503,13507,13548,13552,13595,13600,312,332,357,378,405,427,6910,6930,6955,6976,7003,7025,13508,13528,13553,13574,13601,13623,314,359,360,407,408,459,6912,6957,6958,7005,7006,7057,13510,13555,13556,13603,13604,13655,315,360,361,408,409,460,6913,6958,6959,7006,7007,7058,13511,13556,13557,13604,13605,13656,316,361,362,409,410,461,6914,6959,6960,7007,7008,7059,13512,13557,13558,13605,13606,13657,317,359,363,407,411,458,6915,6957,6961,7005,7009,7056,13513,13555,13559,13603,13607,13654,318,363,364,411,412,463,6916,6961,6962,7009,7010,7061,13514,13559,13560,13607,13608,13659,319,362,365,410,413,462,6917,6960,6963,7008,7011,7060,13515,13558,13561,13606,13609,13658,320,365,366,413,414,465,6918,6963,6964,7011,7012,7063,13516,13561,13562,13609,13610,13661,325,326,371,372,420,421,6923,6924,6969,6970,7018,7019,13521,13522,13567,13568,13616,13617,327,372,373,421,422,473,6925,6970,6971,7019,7020,7071,13523,13568,13569,13617,13618,13669,328,371,374,420,423,472,6926,6969,6972,7018,7021,7070,13524,13567,13570,13616,13619,13668,329,373,375,422,424,474,6927,6971,6973,7020,7022,7072,13525,13569,13571,13618,13620,13670,331,375,377,424,426,476,6929,6973,6975,7022,7024,7074,13527,13571,13573,13620,13622,13672,334,377,380,426,430,478,6932,6975,6978,7024,7028,7076,13530,13573,13576,13622,13626,13674,336,382,383,432,433,484,6934,6980,6981,7030,7031,7082,13532,13578,13579,13628,13629,13680,337,382,384,432,434,485,6935,6980,6982,7030,7032,7083,13533,13578,13580,13628,13630,13681,338,384,385,434,435,487,6936,6982,6983,7032,7033,7085,13534,13580,13581,13630,13631,13683,339,383,386,433,436,486,6937,6981,6984,7031,7034,7084,13535,13579,13582,13629,13632,13682,340,385,387,435,437,488,6938,6983,6985,7033,7035,7086,13536,13581,13583,13631,13633,13684,341,387,388,437,438,490,6939,6985,6986,7035,7036,7088,13537,13583,13584,13633,13634,13686,342,388,389,438,439,491,6940,6986,6987,7036,7037,7089,13538,13584,13585,13634,13635,13687,343,346,390,393,440,444,6941,6944,6988,6991,7038,7042,13539,13542,13586,13589,13636,13640,349,364,396,412,447,464,6947,6962,6994,7010,7045,7062,13545,13560,13592,13608,13643,13660,350,389,397,439,448,492,6948,6987,6995,7037,7046,7090,13546,13585,13593,13635,13644,13688,351,352,398,399,449,450,6949,6950,6996,6997,7047,7048,13547,13548,13594,13595,13645,13646,353,400,401,451,452,504,6951,6998,6999,7049,7050,7102,13549,13596,13597,13647,13648,13700,354,401,402,452,453,506,6952,6999,7000,7050,7051,7104,13550,13597,13598,13648,13649,13702,355,400,403,451,454,505,6953,6998,7001,7049,7052,7103,13551,13596,13599,13647,13650,13701,356,402,404,453,455,507,6954,7000,7002,7051,7053,7105,13552,13598,13600,13649,13651,13703,357,398,405,449,456,502,6955,6996,7003,7047,7054,7100,13553,13594,13601,13645,13652,13698,358,403,406,454,457,508,6956,7001,7004,7052,7055,7106,13554,13599,13602,13650,13653,13704,366,406,414,457,466,511,6964,7004,7012,7055,7064,7109,13562,13602,13610,13653,13662,13707,367,415,416,467,468,521,6965,7013,7014,7065,7066,7119,13563,13611,13612,13663,13664,13717,368,416,417,468,469,523,6966,7014,7015,7066,7067,7121,13564,13612,13613,13664,13665,13719,369,417,418,469,470,524,6967,7015,7016,7067,7068,7122,13565,13613,13614,13665,13666,13720,370,418,419,470,471,525,6968,7016,7017,7068,7069,7123,13566,13614,13615,13666,13667,13721,374,419,423,471,475,526,6972,7017,7021,7069,7073,7124,13570,13615,13619,13667,13671,13722,376,415,425,467,477,522,6974,7013,7023,7065,7075,7120,13572,13611,13621,13663,13673,13718,378,425,427,477,479,532,6976,7023,7025,7075,7077,7130,13574,13621,13623,13673,13675,13728,379,428,429,480,481,535,6977,7026,7027,7078,7079,7133,13575,13624,13625,13676,13677,13731,380,428,430,480,482,536,6978,7026,7028,7078,7080,7134,13576,13624,13626,13676,13678,13732,381,429,431,481,483,537,6979,7027,7029,7079,7081,7135,13577,13625,13627,13677,13679,13733,386,431,436,483,489,539,6984,7029,7034,7081,7087,7137,13582,13627,13632,13679,13685,13735,390,395,440,446,493,499,6988,6993,7038,7044,7091,7097,13586,13591,13636,13642,13689,13695,391,441,442,494,495,550,6989,7039,7040,7092,7093,7148,13587,13637,13638,13690,13691,13746,392,442,443,495,496,552,6990,7040,7041,7093,7094,7150,13588,13638,13639,13691,13692,13748,393,441,444,494,497,551,6991,7039,7042,7092,7095,7149,13589,13637,13640,13690,13693,13747,394,443,445,496,498,553,6992,7041,7043,7094,7096,7151,13590,13639,13641,13692,13694,13749,396,445,447,498,500,555,6994,7043,7045,7096,7098,7153,13592,13641,13643,13694,13696,13751,397,446,448,499,501,556,6995,7044,7046,7097,7099,7154,13593,13642,13644,13695,13697,13752,399,404,450,455,503,509,6997,7002,7048,7053,7101,7107,13595,13600,13646,13651,13699,13705,405,427,456,479,510,534,7003,7025,7054,7077,7108,7132,13601,13623,13652,13675,13706,13730,407,458,459,512,513,570,7005,7056,7057,7110,7111,7168,13603,13654,13655,13708,13709,13766,408,459,460,513,514,571,7006,7057,7058,7111,7112,7169,13604,13655,13656,13709,13710,13767,409,460,461,514,515,572,7007,7058,7059,7112,7113,7170,13605,13656,13657,13710,13711,13768,410,461,462,515,516,573,7008,7059,7060,7113,7114,7171,13606,13657,13658,13711,13712,13769,411,458,463,512,517,569,7009,7056,7061,7110,7115,7167,13607,13654,13659,13708,13713,13765,412,463,464,517,518,575,7010,7061,7062,7115,7116,7173,13608,13659,13660,13713,13714,13771,413,462,465,516,519,574,578,7011,7060,7063,7114,7117,7172,7176,13609,13658,13661,13712,13715,13770,13774,414,465,466,519,520,577,7012,7063,7064,7117,7118,7175,13610,13661,13662,13715,13716,13773,420,421,472,473,527,528,7018,7019,7070,7071,7125,7126,13616,13617,13668,13669,13723,13724,422,473,474,528,529,7020,7071,7072,7126,7127,13618,13669,13670,13724,13725,423,472,475,527,530,586,7021,7070,7073,7125,7128,7184,13619,13668,13671,13723,13726,13782,424,474,476,529,531,587,7022,7072,7074,7127,7129,7185,13620,13670,13672,13725,13727,13783,426,476,478,531,533,589,7024,7074,7076,7129,7131,7187,13622,13672,13674,13727,13729,13785,430,478,482,533,538,591,7028,7076,7080,7131,7136,7189,13626,13674,13678,13729,13734,13787,432,484,485,540,541,599,7030,7082,7083,7138,7139,7197,13628,13680,13681,13736,13737,13795,433,484,486,540,542,598,7031,7082,7084,7138,7140,7196,13629,13680,13682,13736,13738,13794,434,485,487,541,543,600,7032,7083,7085,7139,7141,7198,13630,13681,13683,13737,13739,13796,435,487,488,543,544,602,7033,7085,7086,7141,7142,7200,13631,13683,13684,13739,13740,13798,436,486,489,542,545,601,7034,7084,7087,7140,7143,7199,13632,13682,13685,13738,13741,13797,437,488,490,544,546,603,7035,7086,7088,7142,7144,7201,13633,13684,13686,13740,13742,13799,438,490,491,546,547,605,7036,7088,7089,7144,7145,7203,13634,13686,13687,13742,13743,13801,439,491,492,547,548,606,7037,7089,7090,7145,7146,7204,13635,13687,13688,13743,13744,13802,440,444,493,497,549,554,7038,7042,7091,7095,7147,7152,13636,13640,13689,13693,13745,13750,447,464,500,518,557,576,7045,7062,7098,7116,7155,7174,13643,13660,13696,13714,13753,13772,448,492,501,548,558,607,7046,7090,7099,7146,7156,7205,13644,13688,13697,13744,13754,13803,449,450,502,503,559,560,7047,7048,7100,7101,7157,7158,13645,13646,13698,13699,13755,13756,451,504,505,561,562,621,7049,7102,7103,7159,7160,7219,13647,13700,13701,13757,13758,13817,452,504,506,561,563,620,7050,7102,7104,7159,7161,7218,13648,13700,13702,13757,13759,13816,453,506,507,563,564,623,7051,7104,7105,7161,7162,7221,13649,13702,13703,13759,13760,13819,454,505,508,562,565,622,7052,7103,7106,7160,7163,7220,13650,13701,13704,13758,13761,13818,455,507,509,564,566,624,7053,7105,7107,7162,7164,7222,13651,13703,13705,13760,13762,13820,456,502,510,559,567,618,7054,7100,7108,7157,7165,7216,13652,13698,13706,13755,13763,13814,457,508,511,565,568,625,7055,7106,7109,7163,7166,7223,13653,13704,13707,13761,13764,13821,466,511,520,568,579,628,7064,7109,7118,7166,7177,7226,13662,13707,13716,13764,13775,13824,467,521,522,580,581,641,7065,7119,7120,7178,7179,7239,13663,13717,13718,13776,13777,13837,468,521,523,580,582,640,7066,7119,7121,7178,7180,7238,13664,13717,13719,13776,13778,13836,469,523,524,582,583,643,7067,7121,7122,7180,7181,7241,13665,13719,13720,13778,13779,13839,470,524,525,583,584,644,7068,7122,7123,7181,7182,7242,13666,13720,13721,13779,13780,13840,471,525,526,584,585,645,7069,7123,7124,7182,7183,7243,13667,13721,13722,13780,13781,13841,475,526,530,585,588,646,7073,7124,7128,7183,7186,7244,13671,13722,13726,13781,13784,13842,477,522,532,581,590,642,7075,7120,7130,7179,7188,7240,13673,13718,13728,13777,13786,13838,479,532,534,590,592,650,7077,7130,7132,7188,7190,7248,13675,13728,13730,13786,13788,13846,480,535,536,593,594,653,7078,7133,7134,7191,7192,7251,13676,13731,13732,13789,13790,13849,481,535,537,593,595,654,7079,7133,7135,7191,7193,7252,13677,13731,13733,13789,13791,13850,482,536,538,594,596,655,7080,7134,7136,7192,7194,7253,13678,13732,13734,13790,13792,13851,483,537,539,595,597,656,7081,7135,7137,7193,7195,7254,13679,13733,13735,13791,13793,13852,489,539,545,597,604,658,7087,7137,7143,7195,7202,7256,13685,13735,13741,13793,13800,13854,493,499,549,556,608,615,7091,7097,7147,7154,7206,7213,13689,13695,13745,13752,13804,13811,494,550,551,609,610,671,7092,7148,7149,7207,7208,7269,13690,13746,13747,13805,13806,13867,495,550,552,609,611,670,7093,7148,7150,7207,7209,7268,13691,13746,13748,13805,13807,13866,496,552,553,611,612,673,7094,7150,7151,7209,7210,7271,13692,13748,13749,13807,13808,13869,497,551,554,610,613,672,7095,7149,7152,7208,7211,7270,13693,13747,13750,13806,13809,13868,498,553,555,612,614,674,7096,7151,7153,7210,7212,7272,13694,13749,13751,13808,13810,13870,500,555,557,614,616,676,7098,7153,7155,7212,7214,7274,13696,13751,13753,13810,13812,13872,501,556,558,615,617,677,7099,7154,7156,7213,7215,7275,13697,13752,13754,13811,13813,13873,503,509,560,566,619,626,7101,7107,7158,7164,7217,7224,13699,13705,13756,13762,13815,13822,510,534,567,592,627,652,7108,7132,7165,7190,7225,7250,13706,13730,13763,13788,13823,13848,512,569,570,629,630,691,7110,7167,7168,7227,7228,7289,13708,13765,13766,13825,13826,13887,513,570,571,630,631,693,7111,7168,7169,7228,7229,7291,13709,13766,13767,13826,13827,13889,514,571,572,631,632,694,7112,7169,7170,7229,7230,7292,13710,13767,13768,13827,13828,13890,515,572,573,632,633,7113,7170,7171,7230,7231,13711,13768,13769,13828,13829,516,573,574,633,634,695,7114,7171,7172,7231,7232,7293,13712,13769,13770,13829,13830,13891,517,569,575,629,635,692,7115,7167,7173,7227,7233,7290,13713,13765,13771,13825,13831,13888,518,575,576,635,636,697,7116,7173,7174,7233,7234,7295,13714,13771,13772,13831,13832,13893,519,577,578,637,638,7117,7175,7176,7235,7236,13715,13773,13774,13833,13834,520,577,579,637,639,699,7118,7175,7177,7235,7237,7297,13716,13773,13775,13833,13835,13895,527,528,529,586,587,647,7125,7126,7127,7184,7185,7245,13723,13724,13725,13782,13783,13843,530,586,588,647,648,709,7128,7184,7186,7245,7246,7307,13726,13782,13784,13843,13844,13905,531,587,589,647,649,709,7129,7185,7187,7245,7247,7307,13727,13783,13785,13843,13845,13905,533,589,591,649,651,711,7131,7187,7189,7247,7249,7309,13729,13785,13787,13845,13847,13907,538,591,596,651,657,713,7136,7189,7194,7249,7255,7311,13734,13787,13792,13847,13853,13909,540,598,599,659,660,722,7138,7196,7197,7257,7258,7320,13736,13794,13795,13855,13856,13918,541,599,600,660,661,723,7139,7197,7198,7258,7259,7321,13737,13795,13796,13856,13857,13919,542,598,601,659,662,721,7140,7196,7199,7257,7260,7319,13738,13794,13797,13855,13858,13917,543,600,602,661,663,724,7141,7198,7200,7259,7261,7322,13739,13796,13798,13857,13859,13920,544,602,603,663,664,726,7142,7200,7201,7261,7262,7324,13740,13798,13799,13859,13860,13922,545,601,604,662,665,725,7143,7199,7202,7260,7263,7323,13741,13797,13800,13858,13861,13921,546,603,605,664,666,727,7144,7201,7203,7262,7264,7325,13742,13799,13801,13860,13862,13923,547,605,606,666,667,729,7145,7203,7204,7264,7265,7327,13743,13801,13802,13862,13863,13925,548,606,607,667,668,730,7146,7204,7205,7265,7266,7328,13744,13802,13803,13863,13864,13926,549,554,608,613,669,675,7147,7152,7206,7211,7267,7273,13745,13750,13804,13809,13865,13871,557,576,616,636,678,698,7155,7174,7214,7234,7276,7296,13753,13772,13812,13832,13874,13894,558,607,617,668,679,731,7156,7205,7215,7266,7277,7329,13754,13803,13813,13864,13875,13927,559,560,618,619,680,681,7157,7158,7216,7217,7278,7279,13755,13756,13814,13815,13876,13877,561,620,621,682,683,745,7159,7218,7219,7280,7281,7343,13757,13816,13817,13878,13879,13941,562,621,622,683,684,747,7160,7219,7220,7281,7282,7345,13758,13817,13818,13879,13880,13943,563,620,623,682,685,746,7161,7218,7221,7280,7283,7344,13759,13816,13819,13878,13881,13942,564,623,624,685,686,749,7162,7221,7222,7283,7284,7347,13760,13819,13820,13881,13882,13945,565,622,625,684,687,748,752,7163,7220,7223,7282,7285,7346,7350,13761,13818,13821,13880,13883,13944,13948,566,624,626,686,688,750,7164,7222,7224,7284,7286,7348,13762,13820,13822,13882,13884,13946,567,618,627,680,689,743,7165,7216,7225,7278,7287,7341,13763,13814,13823,13876,13885,13939,568,625,628,687,690,751,7166,7223,7226,7285,7288,7349,13764,13821,13824,13883,13886,13947,574,578,634,638,696,700,7172,7176,7232,7236,7294,7298,13770,13774,13830,13834,13892,13896,579,628,639,690,701,755,7177,7226,7237,7288,7299,7353,13775,13824,13835,13886,13897,13951,580,640,641,702,703,767,7178,7238,7239,7300,7301,7365,13776,13836,13837,13898,13899,13963,581,641,642,703,704,768,7179,7239,7240,7301,7302,7366,13777,13837,13838,13899,13900,13964,582,640,643,702,705,7180,7238,7241,7300,7303,13778,13836,13839,13898,13901,583,643,644,705,706,770,7181,7241,7242,7303,7304,7368,13779,13839,13840,13901,13902,13966,584,644,645,706,707,771,7182,7242,7243,7304,7305,7369,13780,13840,13841,13902,13903,13967,585,645,646,707,708,772,7183,7243,7244,7305,7306,7370,13781,13841,13842,13903,13904,13968,588,646,648,708,710,773,7186,7244,7246,7306,7308,7371,13784,13842,13844,13904,13906,13969,590,642,650,704,712,769,7188,7240,7248,7302,7310,7367,13786,13838,13846,13900,13908,13965,592,650,652,712,714,775,7190,7248,7250,7310,7312,7373,13788,13846,13848,13908,13910,13971,593,653,654,715,716,778,7191,7251,7252,7313,7314,7376,13789,13849,13850,13911,13912,13974,594,653,655,715,717,779,782,7192,7251,7253,7313,7315,7377,7380,13790,13849,13851,13911,13913,13975,13978,595,654,656,716,718,780,7193,7252,7254,7314,7316,7378,13791,13850,13852,13912,13914,13976,596,655,657,717,719,781,7194,7253,7255,7315,7317,7379,13792,13851,13853,13913,13915,13977,597,656,658,718,720,783,7195,7254,7256,7316,7318,7381,13793,13852,13854,13914,13916,13979,604,658,665,720,728,785,7202,7256,7263,7318,7326,7383,13800,13854,13861,13916,13924,13981,608,615,669,677,732,740,7206,7213,7267,7275,7330,7338,13804,13811,13865,13873,13928,13936,609,670,671,733,734,7207,7268,7269,7331,7332,13805,13866,13867,13929,13930,610,671,672,734,735,799,7208,7269,7270,7332,7333,7397,13806,13867,13868,13930,13931,13995,611,670,673,733,736,798,802,7209,7268,7271,7331,7334,7396,7400,13807,13866,13869,13929,13932,13994,13998,612,673,674,736,737,801,7210,7271,7272,7334,7335,7399,13808,13869,13870,13932,13933,13997,613,672,675,735,738,800,7211,7270,7273,7333,7336,7398,13809,13868,13871,13931,13934,13996,614,674,676,737,739,803,7212,7272,7274,7335,7337,7401,13810,13870,13872,13933,13935,13999,616,676,678,739,741,805,7214,7274,7276,7337,7339,7403,13812,13872,13874,13935,13937,14001,617,677,679,740,742,806,7215,7275,7277,7338,7340,7404,13813,13873,13875,13936,13938,14002,619,626,681,688,744,753,7217,7224,7279,7286,7342,7351,13815,13822,13877,13884,13940,13949,627,652,689,714,754,777,7225,7250,7287,7312,7352,7375,13823,13848,13885,13910,13950,13973,629,691,692,756,757,822,7227,7289,7290,7354,7355,7420,13825,13887,13888,13952,13953,14018,630,691,693,756,758,823,7228,7289,7291,7354,7356,7421,13826,13887,13889,13952,13954,14019,631,693,694,758,759,825,7229,7291,7292,7356,7357,7423,13827,13889,13890,13954,13955,14021,632,633,694,695,759,760,7230,7231,7292,7293,7357,7358,13828,13829,13890,13891,13955,13956,634,695,696,760,761,826,7232,7293,7294,7358,7359,7424,13830,13891,13892,13956,13957,14022,635,692,697,757,762,824,7233,7290,7295,7355,7360,7422,13831,13888,13893,13953,13958,14020,636,697,698,762,763,828,7234,7295,7296,7360,7361,7426,13832,13893,13894,13958,13959,14024,637,638,699,700,764,765,7235,7236,7297,7298,7362,7363,13833,13834,13895,13896,13960,13961,639,699,701,764,766,830,7237,7297,7299,7362,7364,7428,13835,13895,13897,13960,13962,14026,648,649,709,710,711,774,7246,7247,7307,7308,7309,7372,13844,13845,13905,13906,13907,13970,651,711,713,774,776,840,7249,7309,7311,7372,7374,7438,13847,13907,13909,13970,13972,14036,657,713,719,776,784,842,7255,7311,7317,7374,7382,7440,13853,13909,13915,13972,13980,14038,659,721,722,786,787,852,7257,7319,7320,7384,7385,7450,13855,13917,13918,13982,13983,14048,660,722,723,787,788,854,7258,7320,7321,7385,7386,7452,13856,13918,13919,13983,13984,14050,661,723,724,788,789,855,7259,7321,7322,7386,7387,7453,13857,13919,13920,13984,13985,14051,662,721,725,786,790,853,7260,7319,7323,7384,7388,7451,13858,13917,13921,13982,13986,14049,663,724,726,789,791,856,7261,7322,7324,7387,7389,7454,13859,13920,13922,13985,13987,14052,664,726,727,791,792,858,7262,7324,7325,7389,7390,7456,13860,13922,13923,13987,13988,14054,665,725,728,790,793,857,7263,7323,7326,7388,7391,7455,13861,13921,13924,13986,13989,14053,666,727,729,792,794,859,7264,7325,7327,7390,7392,7457,13862,13923,13925,13988,13990,14055,667,729,730,794,795,861,7265,7327,7328,7392,7393,7459,13863,13925,13926,13990,13991,14057,668,730,731,795,796,862,7266,7328,7329,7393,7394,7460,13864,13926,13927,13991,13992,14058,669,675,732,738,797,804,7267,7273,7330,7336,7395,7402,13865,13871,13928,13934,13993,14000,678,698,741,763,807,829,7276,7296,7339,7361,7405,7427,13874,13894,13937,13959,14003,14025,679,731,742,796,808,863,7277,7329,7340,7394,7406,7461,13875,13927,13938,13992,14004,14059,680,681,743,744,809,810,7278,7279,7341,7342,7407,7408,13876,13877,13939,13940,14005,14006,682,745,746,811,812,879,7280,7343,7344,7409,7410,7477,13878,13941,13942,14007,14008,14075,683,745,747,811,813,878,7281,7343,7345,7409,7411,7476,13879,13941,13943,14007,14009,14074,684,747,748,813,814,881,7282,7345,7346,7411,7412,7479,13880,13943,13944,14009,14010,14077,685,746,749,812,815,880,7283,7344,7347,7410,7413,7478,13881,13942,13945,14008,14011,14076,686,749,750,815,816,883,7284,7347,7348,7413,7414,7481,13882,13945,13946,14011,14012,14079,687,751,752,817,818,886,7285,7349,7350,7415,7416,7484,13883,13947,13948,14013,14014,14082,688,750,753,816,819,884,7286,7348,7351,7414,7417,7482,13884,13946,13949,14012,14015,14080,689,743,754,809,820,876,7287,7341,7352,7407,7418,7474,13885,13939,13950,14005,14016,14072,690,751,755,817,821,885,7288,7349,7353,7415,7419,7483,13886,13947,13951,14013,14017,14081,696,700,761,765,827,831,7294,7298,7359,7363,7425,7429,13892,13896,13957,13961,14023,14027,701,755,766,821,832,889,7299,7353,7364,7419,7430,7487,13897,13951,13962,14017,14028,14085,702,705,767,770,833,836,7300,7303,7365,7368,7431,7434,13898,13901,13963,13966,14029,14032,703,767,768,833,834,901,7301,7365,7366,7431,7432,7499,13899,13963,13964,14029,14030,14097,704,768,769,834,835,902,7302,7366,7367,7432,7433,7500,13900,13964,13965,14030,14031,14098,706,770,771,836,837,904,7304,7368,7369,7434,7435,7502,13902,13966,13967,14032,14033,14100,707,771,772,837,838,905,7305,7369,7370,7435,7436,7503,13903,13967,13968,14033,14034,14101,708,772,773,838,839,906,7306,7370,7371,7436,7437,7504,13904,13968,13969,14034,14035,14102,710,773,774,839,840,907,7308,7371,7372,7437,7438,7505,13906,13969,13970,14035,14036,14103,712,769,775,835,841,903,7310,7367,7373,7433,7439,7501,13908,13965,13971,14031,14037,14099,714,775,777,841,843,908,7312,7373,7375,7439,7441,7506,13910,13971,13973,14037,14039,14104,715,778,779,844,845,910,7313,7376,7377,7442,7443,7508,13911,13974,13975,14040,14041,14106,716,778,780,844,846,911,7314,7376,7378,7442,7444,7509,13912,13974,13976,14040,14042,14107,717,781,782,847,848,915,7315,7379,7380,7445,7446,7513,13913,13977,13978,14043,14044,14111,718,780,783,846,849,913,7316,7378,7381,7444,7447,7511,13914,13976,13979,14042,14045,14109,719,781,784,847,850,914,7317,7379,7382,7445,7448,7512,13915,13977,13980,14043,14046,14110,720,783,785,849,851,916,7318,7381,7383,7447,7449,7514,13916,13979,13981,14045,14047,14112,728,785,793,851,860,918,7326,7383,7391,7449,7458,7516,13924,13981,13989,14047,14056,14114,732,740,797,806,864,873,7330,7338,7395,7404,7462,7471,13928,13936,13993,14002,14060,14069,733,734,798,799,865,866,7331,7332,7396,7397,7463,7464,13929,13930,13994,13995,14061,14062,735,799,800,866,867,933,7333,7397,7398,7464,7465,7531,13931,13995,13996,14062,14063,14129,736,801,802,868,869,936,7334,7399,7400,7466,7467,7534,13932,13997,13998,14064,14065,14132,737,801,803,868,870,935,7335,7399,7401,7466,7468,7533,13933,13997,13999,14064,14066,14131,738,800,804,867,871,934,7336,7398,7402,7465,7469,7532,13934,13996,14000,14063,14067,14130,739,803,805,870,872,938,7337,7401,7403,7468,7470,7536,13935,13999,14001,14066,14068,14134,741,805,807,872,874,939,7339,7403,7405,7470,7472,7537,13937,14001,14003,14068,14070,14135,742,806,808,873,875,940,7340,7404,7406,7471,7473,7538,13938,14002,14004,14069,14071,14136,744,753,810,819,877,887,7342,7351,7408,7417,7475,7485,13940,13949,14006,14015,14073,14083,748,752,814,818,882,7346,7350,7412,7416,7480,13944,13948,14010,14014,14078,754,777,820,843,888,7352,7375,7418,7441,7486,13950,13973,14016,14039,14084,756,822,823,890,891,958,7354,7420,7421,7488,7489,7556,13952,14018,14019,14086,14087,14154,757,822,824,890,892,957,7355,7420,7422,7488,7490,7555,13953,14018,14020,14086,14088,14153,758,823,825,891,893,959,7356,7421,7423,7489,7491,7557,13954,14019,14021,14087,14089,14155,759,760,825,826,893,894,7357,7358,7423,7424,7491,7492,13955,13956,14021,14022,14089,14090,761,826,827,894,895,961,7359,7424,7425,7492,7493,7559,13957,14022,14023,14090,14091,14157,762,824,828,892,896,960,7360,7422,7426,7490,7494,7558,13958,14020,14024,14088,14092,14156,763,828,829,896,897,963,7361,7426,7427,7494,7495,7561,13959,14024,14025,14092,14093,14159,764,765,830,831,898,899,7362,7363,7428,7429,7496,7497,13960,13961,14026,14027,14094,14095,766,830,832,898,900,965,7364,7428,7430,7496,7498,7563,13962,14026,14028,14094,14096,14161,776,840,842,907,909,974,7374,7438,7440,7505,7507,7572,13972,14036,14038,14103,14105,14170,779,782,845,848,912,7377,7380,7443,7446,7510,13975,13978,14041,14044,14108,784,842,850,909,917,976,7382,7440,7448,7507,7515,7574,13980,14038,14046,14105,14113,14172,786,852,853,919,920,986,7384,7450,7451,7517,7518,7584,13982,14048,14049,14115,14116,14182,787,852,854,919,921,987,7385,7450,7452,7517,7519,7585,13983,14048,14050,14115,14117,14183,788,854,855,921,922,989,7386,7452,7453,7519,7520,7587,13984,14050,14051,14117,14118,14185,789,855,856,922,923,990,7387,7453,7454,7520,7521,7588,13985,14051,14052,14118,14119,14186,790,853,857,920,924,988,7388,7451,7455,7518,7522,7586,13986,14049,14053,14116,14120,14184,791,856,858,923,925,991,7389,7454,7456,7521,7523,7589,13987,14052,14054,14119,14121,14187,792,858,859,925,926,993,7390,7456,7457,7523,7524,7591,13988,14054,14055,14121,14122,14189,793,857,860,924,927,992,7391,7455,7458,7522,7525,7590,13989,14053,14056,14120,14123,14188,794,859,861,926,928,994,7392,7457,7459,7524,7526,7592,13990,14055,14057,14122,14124,14190,795,861,862,928,929,7393,7459,7460,7526,7527,13991,14057,14058,14124,14125,796,862,863,929,930,996,997,7394,7460,7461,7527,7528,7594,7595,13992,14058,14059,14125,14126,14192,14193,797,804,864,871,931,7395,7402,7462,7469,7529,13993,14000,14060,14067,14127,798,802,865,869,932,937,7396,7400,7463,7467,7530,7535,13994,13998,14061,14065,14128,14133,807,829,874,897,941,964,7405,7427,7472,7495,7539,7562,14003,14025,14070,14093,14137,14160,808,863,875,930,942,998,7406,7461,7473,7528,7540,7596,14004,14059,14071,14126,14138,14194,809,810,876,877,943,944,7407,7408,7474,7475,7541,7542,14005,14006,14072,14073,14139,14140,811,878,879,945,946,1014,7409,7476,7477,7543,7544,7612,14007,14074,14075,14141,14142,14210,812,879,880,946,947,1015,7410,7477,7478,7544,7545,7613,14008,14075,14076,14142,14143,14211,813,878,881,945,948,1013,7411,7476,7479,7543,7546,7611,14009,14074,14077,14141,14144,14209,814,881,882,948,949,1017,7412,7479,7480,7546,7547,7615,14010,14077,14078,14144,14145,14213,815,880,883,947,950,1016,7413,7478,7481,7545,7548,7614,14011,14076,14079,14143,14146,14212,816,883,884,950,951,1019,7414,7481,7482,7548,7549,7617,14012,14079,14080,14146,14147,14215,817,885,886,952,953,1022,7415,7483,7484,7550,7551,7620,14013,14081,14082,14148,14149,14218,818,882,886,949,953,1018,7416,7480,7484,7547,7551,7616,14014,14078,14082,14145,14149,14214,819,884,887,951,954,1020,7417,7482,7485,7549,7552,7618,14015,14080,14083,14147,14150,14216,820,876,888,943,955,1011,7418,7474,7486,7541,7553,7609,14016,14072,14084,14139,14151,14207,821,885,889,952,956,1021,7419,7483,7487,7550,7554,7619,14017,14081,14085,14148,14152,14217,827,831,895,899,962,966,7425,7429,7493,7497,7560,7564,14023,14027,14091,14095,14158,14162,832,889,900,956,967,1025,7430,7487,7498,7554,7565,7623,14028,14085,14096,14152,14163,14221,833,836,901,904,968,971,7431,7434,7499,7502,7566,7569,14029,14032,14097,14100,14164,14167,834,901,902,968,969,1037,7432,7499,7500,7566,7567,7635,14030,14097,14098,14164,14165,14233,835,902,903,969,970,1038,7433,7500,7501,7567,7568,7636,14031,14098,14099,14165,14166,14234,837,904,905,971,972,1040,7435,7502,7503,7569,7570,7638,14033,14100,14101,14167,14168,14236,838,905,906,972,973,7436,7503,7504,7570,7571,14034,14101,14102,14168,14169,839,906,907,973,974,1041,1042,7437,7504,7505,7571,7572,7639,7640,14035,14102,14103,14169,14170,14237,14238,841,903,908,970,975,1039,7439,7501,7506,7568,7573,7637,14037,14099,14104,14166,14171,14235,843,888,908,955,975,1024,1043,7441,7486,7506,7553,7573,7622,7641,14039,14084,14104,14151,14171,14220,14239,844,910,911,977,978,1046,7442,7508,7509,7575,7576,7644,14040,14106,14107,14173,14174,14242,845,910,912,977,979,1045,7443,7508,7510,7575,7577,7643,14041,14106,14108,14173,14175,14241,846,911,913,978,980,1047,7444,7509,7511,7576,7578,7645,14042,14107,14109,14174,14176,14243,847,914,915,981,982,1051,7445,7512,7513,7579,7580,7649,14043,14110,14111,14177,14178,14247,848,912,915,979,982,1048,7446,7510,7513,7577,7580,7646,14044,14108,14111,14175,14178,14244,849,913,916,980,983,1049,7447,7511,7514,7578,7581,7647,14045,14109,14112,14176,14179,14245,850,914,917,981,984,1050,7448,7512,7515,7579,7582,7648,14046,14110,14113,14177,14180,14246,851,916,918,983,985,1052,7449,7514,7516,7581,7583,7650,14047,14112,14114,14179,14181,14248,860,918,927,985,995,1054,7458,7516,7525,7583,7593,7652,14056,14114,14123,14181,14191,14250,864,873,931,940,999,1008,7462,7471,7529,7538,7597,7606,14060,14069,14127,14136,14195,14204,865,866,932,933,1000,1001,7463,7464,7530,7531,7598,7599,14061,14062,14128,14129,14196,14197,867,933,934,1001,1002,1070,7465,7531,7532,7599,7600,7668,14063,14129,14130,14197,14198,14266,868,935,936,1003,1004,1072,7466,7533,7534,7601,7602,7670,14064,14131,14132,14199,14200,14268,869,936,937,1004,1005,1074,7467,7534,7535,7602,7603,7672,14065,14132,14133,14200,14201,14270,870,935,938,1003,1006,1073,7468,7533,7536,7601,7604,7671,14066,14131,14134,14199,14202,14269,871,931,934,999,1002,1068,1071,7469,7529,7532,7597,7600,7666,7669,14067,14127,14130,14195,14198,14264,14267,872,938,939,1006,1007,1076,7470,7536,7537,7604,7605,7674,14068,14134,14135,14202,14203,14272,874,939,941,1007,1009,1077,7472,7537,7539,7605,7607,7675,14070,14135,14137,14203,14205,14273,875,940,942,1008,1010,1078,7473,7538,7540,7606,7608,7676,14071,14136,14138,14204,14206,14274,877,887,944,954,1012,1023,7475,7485,7542,7552,7610,7621,14073,14083,14140,14150,14208,14219,890,957,958,1026,1027,1096,7488,7555,7556,7624,7625,7694,14086,14153,14154,14222,14223,14292,891,958,959,1027,1028,1098,7489,7556,7557,7625,7626,7696,14087,14154,14155,14223,14224,14294,892,957,960,1026,1029,1097,7490,7555,7558,7624,7627,7695,14088,14153,14156,14222,14225,14293,893,894,959,961,1028,1030,7491,7492,7557,7559,7626,7628,14089,14090,14155,14157,14224,14226,895,961,962,1030,1031,1100,7493,7559,7560,7628,7629,7698,14091,14157,14158,14226,14227,14296,896,960,963,1029,1032,1099,7494,7558,7561,7627,7630,7697,14092,14156,14159,14225,14228,14295,897,963,964,1032,1033,1102,7495,7561,7562,7630,7631,7700,14093,14159,14160,14228,14229,14298,898,899,965,966,1034,1035,7496,7497,7563,7564,7632,7633,14094,14095,14161,14162,14230,14231,900,965,967,1034,1036,1104,7498,7563,7565,7632,7634,7702,14096,14161,14163,14230,14232,14300,909,974,976,1042,1044,7507,7572,7574,7640,7642,14105,14170,14172,14238,14240,917,976,984,1044,1053,1113,1122,7515,7574,7582,7642,7651,7711,7720,14113,14172,14180,14240,14249,14309,14318,919,986,987,1055,1056,1126,7517,7584,7585,7653,7654,7724,14115,14182,14183,14251,14252,14322,920,986,988,1055,1057,1125,7518,7584,7586,7653,7655,7723,14116,14182,14184,14251,14253,14321,921,987,989,1056,1058,1127,7519,7585,7587,7654,7656,7725,14117,14183,14185,14252,14254,14323,922,989,990,1058,1059,1129,7520,7587,7588,7656,7657,7727,14118,14185,14186,14254,14255,14325,923,990,991,1059,1060,1130,7521,7588,7589,7657,7658,7728,14119,14186,14187,14255,14256,14326,924,988,992,1057,1061,1128,7522,7586,7590,7655,7659,7726,14120,14184,14188,14253,14257,14324,925,991,993,1060,1062,1131,7523,7589,7591,7658,7660,7729,14121,14187,14189,14256,14258,14327,926,993,994,1062,1063,1133,7524,7591,7592,7660,7661,7731,14122,14189,14190,14258,14259,14329,927,992,995,1061,1064,1132,7525,7590,7593,7659,7662,7730,14123,14188,14191,14257,14260,14328,928,929,994,996,1063,1065,7526,7527,7592,7594,7661,7663,14124,14125,14190,14192,14259,14261,930,997,998,1066,1067,1136,7528,7595,7596,7664,7665,7734,14126,14193,14194,14262,14263,14332,932,937,1000,1005,1069,1075,7530,7535,7598,7603,7667,7673,14128,14133,14196,14201,14265,14271,941,964,1009,1033,1079,1103,7539,7562,7607,7631,7677,7701,14137,14160,14205,14229,14275,14299,942,998,1010,1067,1080,1138,7540,7596,7608,7665,7678,7736,14138,14194,14206,14263,14276,14334,943,944,1011,1012,1081,1082,1153,7541,7542,7609,7610,7679,7680,7751,14139,14140,14207,14208,14277,14278,14349,945,1013,1014,1083,1084,1155,7543,7611,7612,7681,7682,7753,14141,14209,14210,14279,14280,14351,946,1014,1015,1084,1085,1157,7544,7612,7613,7682,7683,7755,14142,14210,14211,14280,14281,14353,947,1015,1016,1085,1086,1158,7545,7613,7614,7683,7684,7756,14143,14211,14212,14281,14282,14354,948,1013,1017,1083,1087,1156,7546,7611,7615,7681,7685,7754,14144,14209,14213,14279,14283,14352,949,1017,1018,1087,1088,1160,7547,7615,7616,7685,7686,7758,14145,14213,14214,14283,14284,14356,950,1016,1019,1086,1089,1159,7548,7614,7617,7684,7687,7757,14146,14212,14215,14282,14285,14355,951,1019,1020,1089,1090,1162,7549,7617,7618,7687,7688,7760,14147,14215,14216,14285,14286,14358,952,1021,1022,1091,1092,1165,7550,7619,7620,7689,7690,7763,14148,14217,14218,14287,14288,14361,953,1018,1022,1088,1092,1161,7551,7616,7620,7686,7690,7759,14149,14214,14218,14284,14288,14357,954,1020,1023,1090,1093,1163,7552,7618,7621,7688,7691,7761,14150,14216,14219,14286,14289,14359,955,1011,1024,1081,1094,1152,7553,7609,7622,7679,7692,7750,14151,14207,14220,14277,14290,14348,956,1021,1025,1091,1095,1164,7554,7619,7623,7689,7693,7762,14152,14217,14221,14287,14291,14360,962,966,1031,1035,1101,1105,7560,7564,7629,7633,7699,7703,14158,14162,14227,14231,14297,14301,967,1025,1036,1095,1106,1168,7565,7623,7634,7693,7704,7766,14163,14221,14232,14291,14302,14364,968,971,1037,1040,1107,1110,7566,7569,7635,7638,7705,7708,14164,14167,14233,14236,14303,14306,969,1037,1038,1107,1108,1180,7567,7635,7636,7705,7706,7778,14165,14233,14234,14303,14304,14376,970,1038,1039,1108,1109,1181,7568,7636,7637,7706,7707,7779,14166,14234,14235,14304,14305,14377,972,973,1040,1041,1110,1111,7570,7571,7638,7639,7708,7709,14168,14169,14236,14237,14306,14307,975,1039,1043,1109,1112,1182,7573,7637,7641,7707,7710,7780,14171,14235,14239,14305,14308,14378,977,1045,1046,1114,1115,1186,7575,7643,7644,7712,7713,7784,14173,14241,14242,14310,14311,14382,978,1046,1047,1115,1116,1187,7576,7644,7645,7713,7714,7785,14174,14242,14243,14311,14312,14383,979,1045,1048,1114,1117,1185,7577,7643,7646,7712,7715,7783,14175,14241,14244,14310,14313,14381,980,1047,1049,1116,1118,1188,7578,7645,7647,7714,7716,7786,14176,14243,14245,14312,14314,14384,981,1050,1051,1119,1120,1191,7579,7648,7649,7717,7718,7789,14177,14246,14247,14315,14316,14387,982,1048,1051,1117,1120,1189,7580,7646,7649,7715,7718,7787,14178,14244,14247,14313,14316,14385,983,1049,1052,1118,1121,1190,7581,7647,7650,7716,7719,7788,14179,14245,14248,14314,14317,14386,984,1050,1053,1119,1123,1192,7582,7648,7651,7717,7721,7790,14180,14246,14249,14315,14319,14388,985,1052,1054,1121,1124,1193,7583,7650,7652,7719,7722,7791,14181,14248,14250,14317,14320,14389,995,1054,1064,1124,1134,1196,7593,7652,7662,7722,7732,7794,14191,14250,14260,14320,14330,14392,996,997,1065,1066,1135,1137,7594,7595,7663,7664,7733,7735,14192,14193,14261,14262,14331,14333,999,1008,1068,1078,1139,1149,7597,7606,7666,7676,7737,7747,14195,14204,14264,14274,14335,14345,1000,1001,1069,1070,1140,1141,7598,7599,7667,7668,7738,7739,14196,14197,14265,14266,14336,14337,1002,1070,1071,1141,1142,1213,7600,7668,7669,7739,7740,7811,14198,14266,14267,14337,14338,14409,1003,1072,1073,1143,1144,1216,7601,7670,7671,7741,7742,7814,14199,14268,14269,14339,14340,14412,1004,1072,1074,1143,1145,1215,7602,7670,7672,7741,7743,7813,14200,14268,14270,14339,14341,14411,1005,1074,1075,1145,1146,1218,7603,7672,7673,7743,7744,7816,14201,14270,14271,14341,14342,14414,1006,1073,1076,1144,1147,1217,7604,7671,7674,7742,7745,7815,14202,14269,14272,14340,14343,14413,1007,1076,1077,1147,1148,7605,7674,7675,7745,7746,14203,14272,14273,14343,14344,1009,1077,1079,1148,1150,1220,7607,7675,7677,7746,7748,7818,14205,14273,14275,14344,14346,14416,1010,1078,1080,1149,1151,1221,7608,7676,7678,7747,7749,7819,14206,14274,14276,14345,14347,14417,1012,1023,1082,1093,1154,1166,7610,7621,7680,7691,7752,7764,14208,14219,14278,14289,14350,14362,1024,1043,1094,1112,1167,1184,7622,7641,7692,7710,7765,7782,14220,14239,14290,14308,14363,14380,1026,1096,1097,1169,1170,1241,7624,7694,7695,7767,7768,7839,14222,14292,14293,14365,14366,14437,1027,1096,1098,1169,1171,1242,7625,7694,7696,7767,7769,7840,14223,14292,14294,14365,14367,14438,1028,1030,1098,1100,1171,1173,7626,7628,7696,7698,7769,7771,14224,14226,14294,14296,14367,14369,1029,1097,1099,1170,1172,1243,7627,7695,7697,7768,7770,7841,14225,14293,14295,14366,14368,14439,1031,1100,1101,1173,1174,1245,7629,7698,7699,7771,7772,7843,14227,14296,14297,14369,14370,14441,1032,1099,1102,1172,1175,1244,7630,7697,7700,7770,7773,7842,14228,14295,14298,14368,14371,14440,1033,1102,1103,1175,1176,1247,7631,7700,7701,7773,7774,7845,14229,14298,14299,14371,14372,14443,1034,1035,1104,1105,1177,1178,7632,7633,7702,7703,7775,7776,14230,14231,14300,14301,14373,14374,1036,1104,1106,1177,1179,1249,7634,7702,7704,7775,7777,7847,14232,14300,14302,14373,14375,14445,1041,1042,1044,1111,1113,1183,7639,7640,7642,7709,7711,7781,14237,14238,14240,14307,14309,14379,1053,1122,1123,1194,1195,1266,7651,7720,7721,7792,7793,7864,14249,14318,14319,14390,14391,14462,1055,1125,1126,1197,1198,1270,7653,7723,7724,7795,7796,7868,14251,14321,14322,14393,14394,14466,1056,1126,1127,1198,1199,1271,7654,7724,7725,7796,7797,7869,14252,14322,14323,14394,14395,14467,1057,1125,1128,1197,1200,1269,7655,7723,7726,7795,7798,7867,14253,14321,14324,14393,14396,14465,1058,1127,1129,1199,1201,1272,7656,7725,7727,7797,7799,7870,14254,14323,14325,14395,14397,14468,1059,1129,1130,1201,1202,1274,7657,7727,7728,7799,7800,7872,14255,14325,14326,14397,14398,14470,1060,1130,1131,1202,1203,1275,7658,7728,7729,7800,7801,7873,14256,14326,14327,14398,14399,14471,1061,1128,1132,1200,1204,1273,7659,7726,7730,7798,7802,7871,14257,14324,14328,14396,14400,14469,1062,1131,1133,1203,1205,1276,7660,7729,7731,7801,7803,7874,14258,14327,14329,14399,14401,14472,1063,1065,1133,1135,1205,1207,7661,7663,7731,7733,7803,7805,14259,14261,14329,14331,14401,14403,1064,1132,1134,1204,1206,1277,7662,7730,7732,7802,7804,7875,14260,14328,14330,14400,14402,14473,1066,1136,1137,1208,1209,1280,7664,7734,7735,7806,7807,7878,14262,14332,14333,14404,14405,14476,1067,1136,1138,1208,1210,1281,7665,7734,7736,7806,7808,7879,14263,14332,14334,14404,14406,14477,1068,1071,1139,1142,1211,1214,7666,7669,7737,7740,7809,7812,14264,14267,14335,14338,14407,14410,1069,1075,1140,1146,1212,1219,7667,7673,7738,7744,7810,7817,14265,14271,14336,14342,14408,14415,1079,1103,1150,1176,1222,1248,7677,7701,7748,7774,7820,7846,14275,14299,14346,14372,14418,14444,1080,1138,1151,1210,1223,1283,7678,7736,7749,7808,7821,7881,14276,14334,14347,14406,14419,14479,1081,1152,1153,1224,1225,1297,7679,7750,7751,7822,7823,7895,14277,14348,14349,14420,14421,14493,1082,1153,1154,1225,1226,7680,7751,7752,7823,7824,14278,14349,14350,14421,14422,1083,1155,1156,1227,1228,1300,7681,7753,7754,7825,7826,7898,14279,14351,14352,14423,14424,14496,1084,1155,1157,1227,1229,1301,7682,7753,7755,7825,7827,7899,14280,14351,14353,14423,14425,14497,1085,1157,1158,1229,1230,1303,7683,7755,7756,7827,7828,7901,14281,14353,14354,14425,14426,14499,1086,1158,1159,1230,1231,1304,7684,7756,7757,7828,7829,7902,14282,14354,14355,14426,14427,14500,1087,1156,1160,1228,1232,1302,7685,7754,7758,7826,7830,7900,14283,14352,14356,14424,14428,14498,1088,1160,1161,1232,1233,1306,7686,7758,7759,7830,7831,7904,14284,14356,14357,14428,14429,14502,1089,1159,1162,1231,1234,1305,7687,7757,7760,7829,7832,7903,14285,14355,14358,14427,14430,14501,1090,1162,1163,1234,1235,1308,7688,7760,7761,7832,7833,7906,14286,14358,14359,14430,14431,14504,1091,1164,1165,1236,1237,1310,7689,7762,7763,7834,7835,7908,14287,14360,14361,14432,14433,14506,1092,1161,1165,1233,1237,1307,7690,7759,7763,7831,7835,7905,14288,14357,14361,14429,14433,14503,1093,1163,1166,1235,1238,1309,1313,7691,7761,7764,7833,7836,7907,7911,14289,14359,14362,14431,14434,14505,14509,1094,1152,1167,1224,1239,1298,7692,7750,7765,7822,7837,7896,14290,14348,14363,14420,14435,14494,1095,1164,1168,1236,1240,1311,7693,7762,7766,7834,7838,7909,14291,14360,14364,14432,14436,14507,1101,1105,1174,1178,1246,1250,7699,7703,7772,7776,7844,7848,14297,14301,14370,14374,14442,14446,1106,1168,1179,1240,1251,1315,1327,7704,7766,7777,7838,7849,7913,7925,14302,14364,14375,14436,14447,14511,14523,1107,1110,1111,1180,1183,1252,7705,7708,7709,7778,7781,7850,14303,14306,14307,14376,14379,14448,1108,1180,1181,1252,1253,1265,7706,7778,7779,7850,7851,7863,14304,14376,14377,14448,14449,14461,1109,1181,1182,1253,1254,1328,7707,7779,7780,7851,7852,7926,14305,14377,14378,14449,14450,14524,1112,1182,1184,1254,1255,1329,7710,7780,7782,7852,7853,7927,14308,14378,14380,14450,14451,14525,1113,1122,1183,1194,1252,1265,7711,7720,7781,7792,7850,7863,14309,14318,14379,14390,14448,14461,1114,1185,1186,1256,1257,1331,7712,7783,7784,7854,7855,7929,14310,14381,14382,14452,14453,14527,1115,1186,1187,1257,1258,1333,7713,7784,7785,7855,7856,7931,14311,14382,14383,14453,14454,14529,1116,1187,1188,1258,1259,1334,7714,7785,7786,7856,7857,7932,14312,14383,14384,14454,14455,14530,1117,1185,1189,1256,1260,1332,7715,7783,7787,7854,7858,7930,14313,14381,14385,14452,14456,14528,1118,1188,1190,1259,1261,1335,7716,7786,7788,7857,7859,7933,14314,14384,14386,14455,14457,14531,1119,1191,1192,1262,1263,1338,7717,7789,7790,7860,7861,7936,14315,14387,14388,14458,14459,14534,1120,1189,1191,1260,1262,1336,7718,7787,7789,7858,7860,7934,14316,14385,14387,14456,14458,14532,1121,1190,1193,1261,1264,1337,7719,7788,7791,7859,7862,7935,14317,14386,14389,14457,14460,14533,1123,1192,1195,1263,1267,1339,7721,7790,7793,7861,7865,7937,14319,14388,14391,14459,14463,14535,1124,1193,1196,1264,1268,1340,7722,7791,7794,7862,7866,7938,14320,14389,14392,14460,14464,14536,1134,1196,1206,1268,1278,1343,7732,7794,7804,7866,7876,7941,14330,14392,14402,14464,14474,14539,1135,1137,1207,1209,1279,1282,7733,7735,7805,7807,7877,7880,14331,14333,14403,14405,14475,14478,1139,1149,1211,1221,1284,1294,7737,7747,7809,7819,7882,7892,14335,14345,14407,14417,14480,14490,1140,1141,1212,1213,1285,1286,7738,7739,7810,7811,7883,7884,14336,14337,14408,14409,14481,14482,1142,1213,1214,1286,1287,1361,7740,7811,7812,7884,7885,7959,14338,14409,14410,14482,14483,14557,1143,1215,1216,1288,1289,1364,7741,7813,7814,7886,7887,7962,14339,14411,14412,14484,14485,14560,1144,1216,1217,1289,1290,1365,7742,7814,7815,7887,7888,7963,14340,14412,14413,14485,14486,14561,1145,1215,1218,1288,1291,1363,7743,7813,7816,7886,7889,7961,14341,14411,14414,14484,14487,14559,1146,1218,1219,1291,1292,1367,7744,7816,7817,7889,7890,7965,14342,14414,14415,14487,14488,14563,1147,1148,1217,1220,1290,1293,1366,7745,7746,7815,7818,7888,7891,7964,14343,14344,14413,14416,14486,14489,14562,1150,1220,1222,1293,1295,1369,7748,7818,7820,7891,7893,7967,14346,14416,14418,14489,14491,14565,1151,1221,1223,1294,1296,1370,7749,7819,7821,7892,7894,7968,14347,14417,14419,14490,14492,14566,1154,1166,1226,1238,1299,1312,7752,7764,7824,7836,7897,7910,14350,14362,14422,14434,14495,14508,1167,1184,1239,1255,1314,1330,7765,7782,7837,7853,7912,7928,14363,14380,14435,14451,14510,14526,1169,1241,1242,1316,1317,1392,7767,7839,7840,7914,7915,7990,14365,14437,14438,14512,14513,14588,1170,1241,1243,1316,1318,1393,7768,7839,7841,7914,7916,7991,14366,14437,14439,14512,14514,14589,1171,1173,1242,1245,1317,1320,7769,7771,7840,7843,7915,7918,14367,14369,14438,14441,14513,14516,1172,1243,1244,1318,1319,1394,7770,7841,7842,7916,7917,7992,14368,14439,14440,14514,14515,14590,1174,1245,1246,1320,1321,1396,7772,7843,7844,7918,7919,7994,14370,14441,14442,14516,14517,14592,1175,1244,1247,1319,1322,1395,1399,7773,7842,7845,7917,7920,7993,7997,14371,14440,14443,14515,14518,14591,14595,1176,1247,1248,1322,1323,1398,7774,7845,7846,7920,7921,7996,14372,14443,14444,14518,14519,14594,1177,1178,1249,1250,1324,1325,7775,7776,7847,7848,7922,7923,14373,14374,14445,14446,14520,14521,1179,1249,1251,1324,1326,7777,7847,7849,7922,7924,14375,14445,14447,14520,14522,1194,1253,1265,1266,1328,1341,7792,7851,7863,7864,7926,7939,14390,14449,14461,14462,14524,14537,1195,1266,1267,1341,1342,1416,7793,7864,7865,7939,7940,8014,14391,14462,14463,14537,14538,14612,1197,1269,1270,1344,1345,1419,7795,7867,7868,7942,7943,8017,14393,14465,14466,14540,14541,14615,1198,1270,1271,1345,1346,1421,7796,7868,7869,7943,7944,8019,14394,14466,14467,14541,14542,14617,1199,1271,1272,1346,1347,1422,7797,7869,7870,7944,7945,8020,14395,14467,14468,14542,14543,14618,1200,1269,1273,1344,1348,1420,7798,7867,7871,7942,7946,8018,14396,14465,14469,14540,14544,14616,1201,1272,1274,1347,1349,1423,7799,7870,7872,7945,7947,8021,14397,14468,14470,14543,14545,14619,1202,1274,1275,1349,1350,1425,7800,7872,7873,7947,7948,8023,14398,14470,14471,14545,14546,14621,1203,1275,1276,1350,1351,1426,7801,7873,7874,7948,7949,8024,14399,14471,14472,14546,14547,14622,1204,1273,1277,1348,1352,1424,7802,7871,7875,7946,7950,8022,14400,14469,14473,14544,14548,14620,1205,1207,1276,1279,1351,1354,7803,7805,7874,7877,7949,7952,14401,14403,14472,14475,14547,14550,1206,1277,1278,1352,1353,1427,7804,7875,7876,7950,7951,8025,14402,14473,14474,14548,14549,14623,1208,1280,1281,1355,1356,1431,7806,7878,7879,7953,7954,8029,14404,14476,14477,14551,14552,14627,1209,1280,1282,1355,1357,1430,7807,7878,7880,7953,7955,8028,14405,14476,14478,14551,14553,14626,1210,1281,1283,1356,1358,1432,7808,7879,7881,7954,7956,8030,14406,14477,14479,14552,14554,14628,1211,1214,1284,1287,1359,1362,7809,7812,7882,7885,7957,7960,14407,14410,14480,14483,14555,14558,1212,1219,1285,1292,1360,1368,7810,7817,7883,7890,7958,7966,14408,14415,14481,14488,14556,14564,1222,1248,1295,1323,1371,1400,7820,7846,7893,7921,7969,7998,14418,14444,14491,14519,14567,14596,1223,1283,1296,1358,1372,1434,7821,7881,7894,7956,7970,8032,14419,14479,14492,14554,14568,14630,1224,1297,1298,1373,1374,1450,7822,7895,7896,7971,7972,8048,14420,14493,14494,14569,14570,14646,1225,1226,1297,1299,1373,1375,1449,7823,7824,7895,7897,7971,7973,8047,14421,14422,14493,14495,14569,14571,14645,1227,1300,1301,1376,1377,1453,7825,7898,7899,7974,7975,8051,14423,14496,14497,14572,14573,14649,1228,1300,1302,1376,1378,1454,7826,7898,7900,7974,7976,8052,14424,14496,14498,14572,14574,14650,1229,1301,1303,1377,1379,1455,7827,7899,7901,7975,7977,8053,14425,14497,14499,14573,14575,14651,1230,1303,1304,1379,1380,1457,7828,7901,7902,7977,7978,8055,14426,14499,14500,14575,14576,14653,1231,1304,1305,1380,1381,1458,7829,7902,7903,7978,7979,8056,14427,14500,14501,14576,14577,14654,1232,1302,1306,1378,1382,1456,7830,7900,7904,7976,7980,8054,14428,14498,14502,14574,14578,14652,1233,1306,1307,1382,1383,1460,7831,7904,7905,7980,7981,8058,14429,14502,14503,14578,14579,14656,1234,1305,1308,1381,1384,1459,7832,7903,7906,7979,7982,8057,14430,14501,14504,14577,14580,14655,1235,1308,1309,1384,1385,1462,7833,7906,7907,7982,7983,8060,14431,14504,14505,14580,14581,14658,1236,1310,1311,1386,1387,1464,7834,7908,7909,7984,7985,8062,14432,14506,14507,14582,14583,14660,1237,1307,1310,1383,1386,1461,7835,7905,7908,7981,7984,8059,14433,14503,14506,14579,14582,14657,1238,1312,1313,1388,1389,1466,7836,7910,7911,7986,7987,8064,14434,14508,14509,14584,14585,14662,1239,1298,1314,1374,1390,1451,7837,7896,7912,7972,7988,8049,14435,14494,14510,14570,14586,14647,1240,1311,1315,1387,1391,1465,7838,7909,7913,7985,7989,8063,14436,14507,14511,14583,14587,14661,1246,1250,1321,1325,1397,1401,7844,7848,7919,7923,7995,7999,14442,14446,14517,14521,14593,14597,1251,1326,1327,1402,1403,1479,7849,7924,7925,8000,8001,8077,14447,14522,14523,14598,14599,14675,1254,1328,1329,1341,1404,1416,1481,7852,7926,7927,7939,8002,8014,8079,14450,14524,14525,14537,14600,14612,14677,1255,1329,1330,1404,1405,7853,7927,7928,8002,8003,14451,14525,14526,14600,14601,1256,1331,1332,1406,1407,1484,7854,7929,7930,8004,8005,8082,14452,14527,14528,14602,14603,14680,1257,1331,1333,1406,1408,1483,7855,7929,7931,8004,8006,8081,14453,14527,14529,14602,14604,14679,1258,1333,1334,1408,1409,1486,7856,7931,7932,8006,8007,8084,14454,14529,14530,14604,14605,14682,1259,1334,1335,1409,1410,1487,7857,7932,7933,8007,8008,8085,14455,14530,14531,14605,14606,14683,1260,1332,1336,1407,1411,1485,7858,7930,7934,8005,8009,8083,14456,14528,14532,14603,14607,14681,1261,1335,1337,1410,1412,1488,7859,7933,7935,8008,8010,8086,14457,14531,14533,14606,14608,14684,1262,1336,1338,1411,1413,1489,7860,7934,7936,8009,8011,8087,14458,14532,14534,14607,14609,14685,1263,1338,1339,1413,1414,1491,7861,7936,7937,8011,8012,8089,14459,14534,14535,14609,14610,14687,1264,1337,1340,1412,1415,1490,7862,7935,7938,8010,8013,8088,14460,14533,14536,14608,14611,14686,1267,1339,1342,1414,1417,1492,7865,7937,7940,8012,8015,8090,14463,14535,14538,14610,14613,14688,1268,1340,1343,1415,1418,1493,7866,7938,7941,8013,8016,8091,14464,14536,14539,14611,14614,14689,1278,1343,1353,1418,1428,1495,7876,7941,7951,8016,8026,8093,14474,14539,14549,14614,14624,14691,1279,1282,1354,1357,1429,1433,7877,7880,7952,7955,8027,8031,14475,14478,14550,14553,14625,14629,1284,1294,1359,1370,1435,1446,7882,7892,7957,7968,8033,8044,14480,14490,14555,14566,14631,14642,1285,1286,1360,1361,1436,1437,7883,7884,7958,7959,8034,8035,14481,14482,14556,14557,14632,14633,1287,1361,1362,1437,1438,1514,7885,7959,7960,8035,8036,8112,14483,14557,14558,14633,14634,14710,1288,1363,1364,1439,1440,7886,7961,7962,8037,8038,14484,14559,14560,14635,14636,1289,1364,1365,1440,1441,1516,7887,7962,7963,8038,8039,8114,14485,14560,14561,14636,14637,14712,1290,1365,1366,1441,1442,1517,7888,7963,7964,8039,8040,8115,14486,14561,14562,14637,14638,14713,1291,1363,1367,1439,1443,1515,7889,7961,7965,8037,8041,8113,14487,14559,14563,14635,14639,14711,1292,1367,1368,1443,1444,1519,7890,7965,7966,8041,8042,8117,14488,14563,14564,14639,14640,14715,1293,1366,1369,1442,1445,1518,7891,7964,7967,8040,8043,8116,14489,14562,14565,14638,14641,14714,1295,1369,1371,1445,1447,1521,7893,7967,7969,8043,8045,8119,14491,14565,14567,14641,14643,14717,1296,1370,1372,1446,1448,1522,7894,7968,7970,8044,8046,8120,14492,14566,14568,14642,14644,14718,1299,1312,1375,1388,1452,7897,7910,7973,7986,8050,14495,14508,14571,14584,14648,1309,1313,1385,1389,1463,7907,7911,7983,7987,8061,14505,14509,14581,14585,14659,1314,1330,1390,1405,1467,1482,7912,7928,7988,8003,8065,8080,14510,14526,14586,14601,14663,14678,1315,1327,1391,1403,1468,1480,7913,7925,7989,8001,8066,8078,14511,14523,14587,14599,14664,14676,1316,1392,1393,1469,1470,1545,7914,7990,7991,8067,8068,8143,14512,14588,14589,14665,14666,14741,1317,1320,1392,1396,1469,1473,7915,7918,7990,7994,8067,8071,14513,14516,14588,14592,14665,14669,1318,1393,1394,1470,1471,1546,7916,7991,7992,8068,8069,8144,14514,14589,14590,14666,14667,14742,1319,1394,1395,1471,1472,1547,7917,7992,7993,8069,8070,8145,14515,14590,14591,14667,14668,14743,1321,1396,1397,1473,1474,1549,7919,7994,7995,8071,8072,8147,14517,14592,14593,14669,14670,14745,1322,1398,1399,1475,1476,1551,7920,7996,7997,8073,8074,8149,14518,14594,14595,14671,14672,14747,1323,1398,1400,1475,1477,1552,7921,7996,7998,8073,8075,8150,14519,14594,14596,14671,14673,14748,1324,1325,1326,1401,1402,1478,7922,7923,7924,7999,8000,8076,14520,14521,14522,14597,14598,14674,1342,1416,1417,1481,1494,1558,7940,8014,8015,8079,8092,8156,14538,14612,14613,14677,14690,14754,1344,1419,1420,1496,1497,1574,7942,8017,8018,8094,8095,8172,14540,14615,14616,14692,14693,14770,1345,1419,1421,1496,1498,1573,7943,8017,8019,8094,8096,8171,14541,14615,14617,14692,14694,14769,1346,1421,1422,1498,1499,1576,7944,8019,8020,8096,8097,8174,14542,14617,14618,14694,14695,14772,1347,1422,1423,1499,1500,1577,7945,8020,8021,8097,8098,8175,14543,14618,14619,14695,14696,14773,1348,1420,1424,1497,1501,1575,7946,8018,8022,8095,8099,8173,14544,14616,14620,14693,14697,14771,1349,1423,1425,1500,1502,1578,7947,8021,8023,8098,8100,8176,14545,14619,14621,14696,14698,14774,1350,1425,1426,1502,1503,1580,7948,8023,8024,8100,8101,8178,14546,14621,14622,14698,14699,14776,1351,1354,1426,1429,1503,1506,7949,7952,8024,8027,8101,8104,14547,14550,14622,14625,14699,14702,1352,1424,1427,1501,1504,1579,7950,8022,8025,8099,8102,8177,14548,14620,14623,14697,14700,14775,1353,1427,1428,1504,1505,1581,7951,8025,8026,8102,8103,8179,14549,14623,14624,14700,14701,14777,1355,1430,1431,1507,1508,1585,7953,8028,8029,8105,8106,8183,14551,14626,14627,14703,14704,14781,1356,1431,1432,1508,1509,1586,7954,8029,8030,8106,8107,8184,14552,14627,14628,14704,14705,14782,1357,1430,1433,1507,1510,1584,7955,8028,8031,8105,8108,8182,14553,14626,14629,14703,14706,14780,1358,1432,1434,1509,1511,1587,7956,8030,8032,8107,8109,8185,14554,14628,14630,14705,14707,14783,1359,1362,1435,1438,1512,7957,7960,8033,8036,8110,14555,14558,14631,14634,14708,1360,1368,1436,1444,1513,1520,7958,7966,8034,8042,8111,8118,14556,14564,14632,14640,14709,14716,1371,1400,1447,1477,1523,1554,7969,7998,8045,8075,8121,8152,14567,14596,14643,14673,14719,14750,1372,1434,1448,1511,1524,1589,7970,8032,8046,8109,8122,8187,14568,14630,14644,14707,14720,14785,1373,1449,1450,1525,1526,1603,7971,8047,8048,8123,8124,8201,14569,14645,14646,14721,14722,14799,1374,1450,1451,1526,1527,1605,7972,8048,8049,8124,8125,8203,14570,14646,14647,14722,14723,14801,1375,1449,1452,1525,1528,1604,7973,8047,8050,8123,8126,8202,14571,14645,14648,14721,14724,14800,1376,1453,1454,1529,1530,1608,7974,8051,8052,8127,8128,8206,14572,14649,14650,14725,14726,14804,1377,1453,1455,1529,1531,1609,7975,8051,8053,8127,8129,8207,14573,14649,14651,14725,14727,14805,1378,1454,1456,1530,1532,1610,7976,8052,8054,8128,8130,8208,14574,14650,14652,14726,14728,14806,1379,1455,1457,1531,1533,1611,7977,8053,8055,8129,8131,8209,14575,14651,14653,14727,14729,14807,1380,1457,1458,1533,1534,1613,1614,7978,8055,8056,8131,8132,8211,8212,14576,14653,14654,14729,14730,14809,14810,1381,1458,1459,1534,1535,1615,7979,8056,8057,8132,8133,8213,14577,14654,14655,14730,14731,14811,1382,1456,1460,1532,1536,1612,7980,8054,8058,8130,8134,8210,14578,14652,14656,14728,14732,14808,1383,1460,1461,1536,1537,1617,7981,8058,8059,8134,8135,8215,14579,14656,14657,14732,14733,14813,1384,1459,1462,1535,1538,1616,7982,8057,8060,8133,8136,8214,14580,14655,14658,14731,14734,14812,1385,1462,1463,1538,1539,1619,7983,8060,8061,8136,8137,8217,14581,14658,14659,14734,14735,14815,1386,1461,1464,1537,1540,1618,7984,8059,8062,8135,8138,8216,14582,14657,14660,14733,14736,14814,1387,1464,1465,1540,1541,1621,7985,8062,8063,8138,8139,8219,14583,14660,14661,14736,14737,14817,1388,1452,1466,1528,1542,1607,7986,8050,8064,8126,8140,8205,14584,14648,14662,14724,14738,14803,1389,1463,1466,1539,1542,1620,7987,8061,8064,8137,8140,8218,14585,14659,14662,14735,14738,14816,1390,1451,1467,1527,1543,1606,7988,8049,8065,8125,8141,8204,14586,14647,14663,14723,14739,14802,1391,1465,1468,1541,1544,1622,7989,8063,8066,8139,8142,8220,14587,14661,14664,14737,14740,14818,1395,1399,1472,1476,1548,1553,7993,7997,8070,8074,8146,8151,14591,14595,14668,14672,14744,14749,1397,1401,1474,1478,1550,1555,7995,7999,8072,8076,8148,8153,14593,14597,14670,14674,14746,14751,1402,1478,1479,1555,1556,1635,8000,8076,8077,8153,8154,8233,14598,14674,14675,14751,14752,14831,1403,1479,1480,1556,1557,1636,8001,8077,8078,8154,8155,8234,14599,14675,14676,14752,14753,14832,1404,1405,1481,1482,1558,1559,8002,8003,8079,8080,8156,8157,14600,14601,14677,14678,14754,14755,1406,1483,1484,1560,1561,1639,8004,8081,8082,8158,8159,8237,14602,14679,14680,14756,14757,14835,1407,1484,1485,1561,1562,1641,8005,8082,8083,8159,8160,8239,14603,14680,14681,14757,14758,14837,1408,1483,1486,1560,1563,1640,8006,8081,8084,8158,8161,8238,14604,14679,14682,14756,14759,14836,1409,1486,1487,1563,1564,1643,8007,8084,8085,8161,8162,8241,14605,14682,14683,14759,14760,14839,1410,1487,1488,1564,1565,1644,8008,8085,8086,8162,8163,8242,14606,14683,14684,14760,14761,14840,1411,1485,1489,1562,1566,1642,8009,8083,8087,8160,8164,8240,14607,14681,14685,14758,14762,14838,1412,1488,1490,1565,1567,1645,8010,8086,8088,8163,8165,8243,14608,14684,14686,14761,14763,14841,1413,1489,1491,1566,1568,1646,8011,8087,8089,8164,8166,8244,14609,14685,14687,14762,14764,14842,1414,1491,1492,1568,1569,1648,8012,8089,8090,8166,8167,8246,14610,14687,14688,14764,14765,14844,1415,1490,1493,1567,1570,1647,8013,8088,8091,8165,8168,8245,14611,14686,14689,14763,14766,14843,1417,1492,1494,1569,1571,1649,8015,8090,8092,8167,8169,8247,14613,14688,14690,14765,14767,14845,1418,1493,1495,1570,1572,1650,8016,8091,8093,8168,8170,8248,14614,14689,14691,14766,14768,14846,1428,1495,1505,1572,1582,1652,8026,8093,8103,8170,8180,8250,14624,14691,14701,14768,14778,14848,1429,1433,1506,1510,1583,1588,8027,8031,8104,8108,8181,8186,14625,14629,14702,14706,14779,14784,1435,1446,1512,1522,1590,1600,8033,8044,8110,8120,8188,8198,14631,14642,14708,14718,14786,14796,1436,1437,1513,1514,1591,1592,8034,8035,8111,8112,8189,8190,14632,14633,14709,14710,14787,14788,1438,1512,1514,1590,1592,1670,1672,8036,8110,8112,8188,8190,8268,8270,14634,14708,14710,14786,14788,14866,14868,1439,1440,1515,1516,1593,1594,1674,8037,8038,8113,8114,8191,8192,8272,14635,14636,14711,14712,14789,14790,14870,1441,1516,1517,1594,1595,1675,8039,8114,8115,8192,8193,8273,14637,14712,14713,14790,14791,14871,1442,1517,1518,1595,1596,1676,8040,8115,8116,8193,8194,8274,14638,14713,14714,14791,14792,14872,1443,1515,1519,1593,1597,1673,1678,8041,8113,8117,8191,8195,8271,8276,14639,14711,14715,14789,14793,14869,14874,1444,1519,1520,1597,1598,8042,8117,8118,8195,8196,14640,14715,14716,14793,14794,1445,1518,1521,1596,1599,1677,8043,8116,8119,8194,8197,8275,14641,14714,14717,14792,14795,14873,1447,1521,1523,1599,1601,1680,8045,8119,8121,8197,8199,8278,14643,14717,14719,14795,14797,14876,1448,1522,1524,1600,1602,1681,8046,8120,8122,8198,8200,8279,14644,14718,14720,14796,14798,14877,1467,1482,1543,1559,1623,1638,8065,8080,8141,8157,8221,8236,14663,14678,14739,14755,14819,14834,1468,1480,1544,1557,1624,1637,8066,8078,8142,8155,8222,8235,14664,14676,14740,14753,14820,14833,1469,1473,1545,1549,1625,1629,8067,8071,8143,8147,8223,8227,14665,14669,14741,14745,14821,14825,1470,1545,1546,1625,1626,1706,8068,8143,8144,8223,8224,8304,14666,14741,14742,14821,14822,14902,1471,1546,1547,1626,1627,1707,8069,8144,8145,8224,8225,8305,14667,14742,14743,14822,14823,14903,1472,1547,1548,1627,1628,1708,8070,8145,8146,8225,8226,8306,14668,14743,14744,14823,14824,14904,1474,1549,1550,1629,1630,1710,8072,8147,8148,8227,8228,8308,14670,14745,14746,14825,14826,14906,1475,1551,1552,1631,1632,1712,8073,8149,8150,8229,8230,8310,14671,14747,14748,14827,14828,14908,1476,1551,1553,1631,1633,1713,8074,8149,8151,8229,8231,8311,14672,14747,14749,14827,14829,14909,1477,1552,1554,1632,1634,1714,8075,8150,8152,8230,8232,8312,14673,14748,14750,14828,14830,14910,1494,1558,1559,1571,1638,1651,8092,8156,8157,8169,8236,8249,14690,14754,14755,14767,14834,14847,1496,1573,1574,1653,1654,1735,8094,8171,8172,8251,8252,8333,14692,14769,14770,14849,14850,14931,1497,1574,1575,1654,1655,1736,8095,8172,8173,8252,8253,8334,14693,14770,14771,14850,14851,14932,1498,1573,1576,1653,1656,1734,8096,8171,8174,8251,8254,8332,14694,14769,14772,14849,14852,14930,1499,1576,1577,1656,1657,1738,8097,8174,8175,8254,8255,8336,14695,14772,14773,14852,14853,14934,1500,1577,1578,1657,1658,1739,8098,8175,8176,8255,8256,8337,14696,14773,14774,14853,14854,14935,1501,1575,1579,1655,1659,1737,8099,8173,8177,8253,8257,8335,14697,14771,14775,14851,14855,14933,1502,1578,1580,1658,1660,1740,8100,8176,8178,8256,8258,8338,14698,14774,14776,14854,14856,14936,1503,1506,1580,1583,1660,1663,8101,8104,8178,8181,8258,8261,14699,14702,14776,14779,14856,14859,1504,1579,1581,1659,1661,1741,8102,8177,8179,8257,8259,8339,14700,14775,14777,14855,14857,14937,1505,1581,1582,1661,1662,1742,8103,8179,8180,8259,8260,8340,14701,14777,14778,14857,14858,14938,1507,1584,1585,1664,1665,1745,8105,8182,8183,8262,8263,8343,14703,14780,14781,14860,14861,14941,1508,1585,1586,1665,1666,1747,8106,8183,8184,8263,8264,8345,14704,14781,14782,14861,14862,14943,1509,1586,1587,1666,1667,1748,8107,8184,8185,8264,8265,8346,14705,14782,14783,14862,14863,14944,1510,1584,1588,1664,1668,1746,8108,8182,8186,8262,8266,8344,14706,14780,14784,14860,14864,14942,1511,1587,1589,1667,1669,1749,8109,8185,8187,8265,8267,8347,14707,14783,14785,14863,14865,14945,1513,1520,1591,1598,1671,1679,8111,8118,8189,8196,8269,8277,14709,14716,14787,14794,14867,14875,1523,1554,1601,1634,1682,1716,8121,8152,8199,8232,8280,8314,14719,14750,14797,14830,14878,14912,1524,1589,1602,1669,1683,1751,8122,8187,8200,8267,8281,8349,14720,14785,14798,14865,14879,14947,1525,1603,1604,1684,1685,1766,8123,8201,8202,8282,8283,8364,14721,14799,14800,14880,14881,14962,1526,1603,1605,1684,1686,1767,8124,8201,8203,8282,8284,8365,14722,14799,14801,14880,14882,14963,1527,1605,1606,1686,1687,1769,8125,8203,8204,8284,8285,8367,14723,14801,14802,14882,14883,14965,1528,1604,1607,1685,1688,1768,8126,8202,8205,8283,8286,8366,14724,14800,14803,14881,14884,14964,1529,1608,1609,1689,1690,1772,8127,8206,8207,8287,8288,8370,14725,14804,14805,14885,14886,14968,1530,1608,1610,1689,1691,1773,8128,8206,8208,8287,8289,8371,14726,14804,14806,14885,14887,14969,1531,1609,1611,1690,1692,1774,8129,8207,8209,8288,8290,8372,14727,14805,14807,14886,14888,14970,1532,1610,1612,1691,1693,1775,8130,8208,8210,8289,8291,8373,14728,14806,14808,14887,14889,14971,1533,1611,1613,1692,1694,1776,8131,8209,8211,8290,8292,8374,14729,14807,14809,14888,14890,14972,1534,1614,1615,1695,1696,1779,8132,8212,8213,8293,8294,8377,14730,14810,14811,14891,14892,14975,1535,1615,1616,1696,1697,1780,8133,8213,8214,8294,8295,8378,14731,14811,14812,14892,14893,14976,1536,1612,1617,1693,1698,1777,8134,8210,8215,8291,8296,8375,14732,14808,14813,14889,14894,14973,1537,1617,1618,1698,1699,1782,8135,8215,8216,8296,8297,8380,14733,14813,14814,14894,14895,14978,1538,1616,1619,1697,1700,1781,8136,8214,8217,8295,8298,8379,14734,14812,14815,14893,14896,14977,1539,1619,1620,1700,1701,1784,8137,8217,8218,8298,8299,8382,14735,14815,14816,14896,14897,14980,1540,1618,1621,1699,1702,1783,8138,8216,8219,8297,8300,8381,14736,14814,14817,14895,14898,14979,1541,1621,1622,1702,1703,1785,8139,8219,8220,8300,8301,8383,14737,14817,14818,14898,14899,14981,1542,1607,1620,1688,1701,1771,8140,8205,8218,8286,8299,8369,14738,14803,14816,14884,14897,14967,1543,1606,1623,1687,1704,1770,8141,8204,8221,8285,8302,8368,14739,14802,14819,14883,14900,14966,1544,1622,1624,1703,1705,1786,8142,8220,8222,8301,8303,8384,14740,14818,14820,14899,14901,14982,1548,1553,1628,1633,1709,1715,8146,8151,8226,8231,8307,8313,14744,14749,14824,14829,14905,14911,1550,1555,1630,1635,1711,1717,8148,8153,8228,8233,8309,8315,14746,14751,14826,14831,14907,14913,1556,1635,1636,1717,1718,1800,8154,8233,8234,8315,8316,8398,14752,14831,14832,14913,14914,14996,1557,1636,1637,1718,1719,1801,8155,8234,8235,8316,8317,8399,14753,14832,14833,14914,14915,14997,1560,1639,1640,1720,1721,1804,8158,8237,8238,8318,8319,8402,14756,14835,14836,14916,14917,15000,1561,1639,1641,1720,1722,1803,8159,8237,8239,8318,8320,8401,14757,14835,14837,14916,14918,14999,1562,1641,1642,1722,1723,1806,8160,8239,8240,8320,8321,8404,14758,14837,14838,14918,14919,15002,1563,1640,1643,1721,1724,1805,8161,8238,8241,8319,8322,8403,14759,14836,14839,14917,14920,15001,1564,1643,1644,1724,1725,1808,8162,8241,8242,8322,8323,8406,14760,14839,14840,14920,14921,15004,1565,1644,1645,1725,1726,1809,8163,8242,8243,8323,8324,8407,14761,14840,14841,14921,14922,15005,1566,1642,1646,1723,1727,1807,8164,8240,8244,8321,8325,8405,14762,14838,14842,14919,14923,15003,1567,1645,1647,1726,1728,1810,8165,8243,8245,8324,8326,8408,14763,14841,14843,14922,14924,15006,1568,1646,1648,1727,1729,1811,8166,8244,8246,8325,8327,8409,14764,14842,14844,14923,14925,15007,1569,1648,1649,1729,1730,1813,8167,8246,8247,8327,8328,8411,14765,14844,14845,14925,14926,15009,1570,1647,1650,1728,1731,1812,8168,8245,8248,8326,8329,8410,14766,14843,14846,14924,14927,15008,1571,1649,1651,1730,1732,1814,8169,8247,8249,8328,8330,8412,14767,14845,14847,14926,14928,15010,1572,1650,1652,1731,1733,1815,8170,8248,8250,8329,8331,8413,14768,14846,14848,14927,14929,15011,1582,1652,1662,1733,1743,1816,8180,8250,8260,8331,8341,8414,14778,14848,14858,14929,14939,15012,1583,1588,1663,1668,1744,1750,8181,8186,8261,8266,8342,8348,14779,14784,14859,14864,14940,14946,1590,1600,1670,1681,1752,1763,8188,8198,8268,8279,8350,8361,14786,14796,14866,14877,14948,14959,1591,1592,1671,1672,1753,1754,8189,8190,8269,8270,8351,8352,14787,14788,14867,14868,14949,14950,1593,1673,1674,1755,1756,1838,8191,8271,8272,8353,8354,8436,14789,14869,14870,14951,14952,15034,1594,1674,1675,1756,1757,8192,8272,8273,8354,8355,14790,14870,14871,14952,14953,1595,1675,1676,1757,1758,1840,8193,8273,8274,8355,8356,8438,14791,14871,14872,14953,14954,15036,1596,1676,1677,1758,1759,1841,8194,8274,8275,8356,8357,8439,14792,14872,14873,14954,14955,15037,1597,1598,1678,1679,1760,1761,8195,8196,8276,8277,8358,8359,14793,14794,14874,14875,14956,14957,1599,1677,1680,1759,1762,1842,8197,8275,8278,8357,8360,8440,14795,14873,14876,14955,14958,15038,1601,1680,1682,1762,1764,1845,8199,8278,8280,8360,8362,8443,14797,14876,14878,14958,14960,15041,1602,1681,1683,1763,1765,1846,8200,8279,8281,8361,8363,8444,14798,14877,14879,14959,14961,15042,1613,1614,1694,1695,1778,8211,8212,8292,8293,8376,14809,14810,14890,14891,14974,1623,1638,1651,1704,1732,1787,8221,8236,8249,8302,8330,8385,14819,14834,14847,14900,14928,14983,1624,1637,1705,1719,1788,1802,8222,8235,8303,8317,8386,8400,14820,14833,14901,14915,14984,14998,1625,1629,1706,1710,1789,1793,8223,8227,8304,8308,8387,8391,14821,14825,14902,14906,14985,14989,1626,1706,1707,1789,1790,1872,8224,8304,8305,8387,8388,8470,14822,14902,14903,14985,14986,15068,1627,1707,1708,1790,1791,1873,8225,8305,8306,8388,8389,8471,14823,14903,14904,14986,14987,15069,1628,1708,1709,1791,1792,1874,8226,8306,8307,8389,8390,8472,14824,14904,14905,14987,14988,15070,1630,1710,1711,1793,1794,1876,8228,8308,8309,8391,8392,8474,14826,14906,14907,14989,14990,15072,1631,1712,1713,1795,1796,1878,8229,8310,8311,8393,8394,8476,14827,14908,14909,14991,14992,15074,1632,1712,1714,1795,1797,1879,8230,8310,8312,8393,8395,8477,14828,14908,14910,14991,14993,15075,1633,1713,1715,1796,1798,1880,8231,8311,8313,8394,8396,8478,14829,14909,14911,14992,14994,15076,1634,1714,1716,1797,1799,1881,8232,8312,8314,8395,8397,8479,14830,14910,14912,14993,14995,15077,1653,1734,1735,1817,1818,1901,8251,8332,8333,8415,8416,8499,14849,14930,14931,15013,15014,15097,1654,1735,1736,1818,1819,1903,8252,8333,8334,8416,8417,8501,14850,14931,14932,15014,15015,15099,1655,1736,1737,1819,1820,8253,8334,8335,8417,8418,14851,14932,14933,15015,15016,1656,1734,1738,1817,1821,1902,8254,8332,8336,8415,8419,8500,14852,14930,14934,15013,15017,15098,1657,1738,1739,1821,1822,1905,8255,8336,8337,8419,8420,8503,14853,14934,14935,15017,15018,15101,1658,1739,1740,1822,1823,1906,8256,8337,8338,8420,8421,8504,14854,14935,14936,15018,15019,15102,1659,1737,1741,1820,1824,1904,8257,8335,8339,8418,8422,8502,14855,14933,14937,15016,15020,15100,1660,1663,1740,1744,1823,1827,8258,8261,8338,8342,8421,8425,14856,14859,14936,14940,15019,15023,1661,1741,1742,1824,1825,1907,8259,8339,8340,8422,8423,8505,14857,14937,14938,15020,15021,15103,1662,1742,1743,1825,1826,1908,8260,8340,8341,8423,8424,8506,14858,14938,14939,15021,15022,15104,1664,1745,1746,1828,1829,1911,8262,8343,8344,8426,8427,8509,14860,14941,14942,15024,15025,15107,1665,1745,1747,1828,1830,1912,8263,8343,8345,8426,8428,8510,14861,14941,14943,15024,15026,15108,1666,1747,1748,1830,1831,1914,8264,8345,8346,8428,8429,8512,14862,14943,14944,15026,15027,15110,1667,1748,1749,1831,1832,1915,8265,8346,8347,8429,8430,8513,14863,14944,14945,15027,15028,15111,1668,1746,1750,1829,1833,1913,8266,8344,8348,8427,8431,8511,14864,14942,14946,15025,15029,15109,1669,1749,1751,1832,1834,1916,8267,8347,8349,8430,8432,8514,14865,14945,14947,15028,15030,15112,1670,1672,1752,1754,1835,1837,8268,8270,8350,8352,8433,8435,14866,14868,14948,14950,15031,15033,1671,1679,1753,1761,1836,1844,8269,8277,8351,8359,8434,8442,14867,14875,14949,14957,15032,15040,1673,1678,1755,1760,1839,1843,8271,8276,8353,8358,8437,8441,14869,14874,14951,14956,15035,15039,1682,1716,1764,1799,1847,1883,8280,8314,8362,8397,8445,8481,14878,14912,14960,14995,15043,15079,1683,1751,1765,1834,1848,1918,8281,8349,8363,8432,8446,8516,14879,14947,14961,15030,15044,15114,1684,1766,1767,1849,1850,1933,8282,8364,8365,8447,8448,8531,14880,14962,14963,15045,15046,15129,1685,1766,1768,1849,1851,1932,8283,8364,8366,8447,8449,8530,14881,14962,14964,15045,15047,15128,1686,1767,1769,1850,1852,1934,8284,8365,8367,8448,8450,8532,14882,14963,14965,15046,15048,15130,1687,1769,1770,1852,1853,1936,8285,8367,8368,8450,8451,8534,14883,14965,14966,15048,15049,15132,1688,1768,1771,1851,1854,1935,8286,8366,8369,8449,8452,8533,14884,14964,14967,15047,15050,15131,1689,1772,1773,1855,1856,1939,8287,8370,8371,8453,8454,8537,14885,14968,14969,15051,15052,15135,1690,1772,1774,1855,1857,1940,8288,8370,8372,8453,8455,8538,14886,14968,14970,15051,15053,15136,1691,1773,1775,1856,1858,1941,8289,8371,8373,8454,8456,8539,14887,14969,14971,15052,15054,15137,1692,1774,1776,1857,1859,1942,8290,8372,8374,8455,8457,8540,14888,14970,14972,15053,15055,15138,1693,1775,1777,1858,1860,1943,8291,8373,8375,8456,8458,8541,14889,14971,14973,15054,15056,15139,1694,1776,1778,1859,1861,1944,8292,8374,8376,8457,8459,8542,14890,14972,14974,15055,15057,15140,1695,1778,1779,1861,1862,1946,8293,8376,8377,8459,8460,8544,14891,14974,14975,15057,15058,15142,1696,1779,1780,1862,1863,1947,8294,8377,8378,8460,8461,8545,14892,14975,14976,15058,15059,15143,1697,1780,1781,1863,1864,1948,8295,8378,8379,8461,8462,8546,14893,14976,14977,15059,15060,15144,1698,1777,1782,1860,1865,1945,8296,8375,8380,8458,8463,8543,14894,14973,14978,15056,15061,15141,1699,1782,1783,1865,1866,1950,8297,8380,8381,8463,8464,8548,14895,14978,14979,15061,15062,15146,1700,1781,1784,1864,1867,1949,8298,8379,8382,8462,8465,8547,14896,14977,14980,15060,15063,15145,1701,1771,1784,1854,1867,1938,8299,8369,8382,8452,8465,8536,14897,14967,14980,15050,15063,15134,1702,1783,1785,1866,1868,1951,8300,8381,8383,8464,8466,8549,14898,14979,14981,15062,15064,15147,1703,1785,1786,1868,1869,1952,8301,8383,8384,8466,8467,8550,14899,14981,14982,15064,15065,15148,1704,1770,1787,1853,1870,1937,8302,8368,8385,8451,8468,8535,14900,14966,14983,15049,15066,15133,1705,1786,1788,1869,1871,1953,8303,8384,8386,8467,8469,8551,14901,14982,14984,15065,15067,15149,1709,1715,1792,1798,1875,1882,8307,8313,8390,8396,8473,8480,14905,14911,14988,14994,15071,15078,1711,1717,1794,1800,1877,1884,8309,8315,8392,8398,8475,8482,14907,14913,14990,14996,15073,15080,1718,1800,1801,1884,1885,1968,8316,8398,8399,8482,8483,8566,14914,14996,14997,15080,15081,15164,1719,1801,1802,1885,1886,1969,8317,8399,8400,8483,8484,8567,14915,14997,14998,15081,15082,15165,1720,1803,1804,1887,1888,1972,8318,8401,8402,8485,8486,8570,14916,14999,15000,15083,15084,15168,1721,1804,1805,1888,1889,1973,8319,8402,8403,8486,8487,8571,14917,15000,15001,15084,15085,15169,1722,1803,1806,1887,1890,1971,8320,8401,8404,8485,8488,8569,14918,14999,15002,15083,15086,15167,1723,1806,1807,1890,1891,1975,8321,8404,8405,8488,8489,8573,14919,15002,15003,15086,15087,15171,1724,1805,1808,1889,1892,1974,8322,8403,8406,8487,8490,8572,14920,15001,15004,15085,15088,15170,1725,1808,1809,1892,1893,8323,8406,8407,8490,8491,14921,15004,15005,15088,15089,1726,1809,1810,1893,1894,1977,1979,8324,8407,8408,8491,8492,8575,8577,14922,15005,15006,15089,15090,15173,15175,1727,1807,1811,1891,1895,1976,8325,8405,8409,8489,8493,8574,14923,15003,15007,15087,15091,15172,1728,1810,1812,1894,1896,1978,8326,8408,8410,8492,8494,8576,14924,15006,15008,15090,15092,15174,1729,1811,1813,1895,1897,1980,8327,8409,8411,8493,8495,8578,14925,15007,15009,15091,15093,15176,1730,1813,1814,1897,1898,1982,8328,8411,8412,8495,8496,8580,14926,15009,15010,15093,15094,15178,1731,1812,1815,1896,1899,1981,8329,8410,8413,8494,8497,8579,14927,15008,15011,15092,15095,15177,1732,1787,1814,1870,1898,1954,8330,8385,8412,8468,8496,8552,14928,14983,15010,15066,15094,15150,1733,1815,1816,1899,1900,1983,8331,8413,8414,8497,8498,8581,14929,15011,15012,15095,15096,15179,1743,1816,1826,1900,1909,1984,8341,8414,8424,8498,8507,8582,14939,15012,15022,15096,15105,15180,1744,1750,1827,1833,1910,1917,8342,8348,8425,8431,8508,8515,14940,14946,15023,15029,15106,15113,1752,1763,1835,1846,1919,8350,8361,8433,8444,8517,14948,14959,15031,15042,15115,1753,1754,1836,1837,1920,1921,8351,8352,8434,8435,8518,8519,14949,14950,15032,15033,15116,15117,1755,1838,1839,1922,1923,2006,8353,8436,8437,8520,8521,8604,14951,15034,15035,15118,15119,15202,1756,1757,1838,1840,1922,1924,8354,8355,8436,8438,8520,8522,14952,14953,15034,15036,15118,15120,1758,1840,1841,1924,1925,2008,8356,8438,8439,8522,8523,8606,14954,15036,15037,15120,15121,15204,1759,1841,1842,1925,1926,2009,8357,8439,8440,8523,8524,8607,14955,15037,15038,15121,15122,15205,1760,1761,1843,1844,1927,1928,8358,8359,8441,8442,8525,8526,14956,14957,15039,15040,15123,15124,1762,1842,1845,1926,1929,2010,8360,8440,8443,8524,8527,8608,14958,15038,15041,15122,15125,15206,1764,1845,1847,1929,1930,2013,8362,8443,8445,8527,8528,8611,14960,15041,15043,15125,15126,15209,1765,1846,1848,1919,1931,2003,8363,8444,8446,8517,8529,8601,14961,15042,15044,15115,15127,15199,1788,1802,1871,1886,1955,1970,8386,8400,8469,8484,8553,8568,14984,14998,15067,15082,15151,15166,1789,1793,1872,1876,1956,1960,2040,8387,8391,8470,8474,8554,8558,8638,14985,14989,15068,15072,15152,15156,15236,1790,1872,1873,1956,1957,8388,8470,8471,8554,8555,14986,15068,15069,15152,15153,1791,1873,1874,1957,1958,2041,8389,8471,8472,8555,8556,8639,14987,15069,15070,15153,15154,15237,1792,1874,1875,1958,1959,2042,8390,8472,8473,8556,8557,8640,14988,15070,15071,15154,15155,15238,1794,1876,1877,1960,1961,2044,8392,8474,8475,8558,8559,8642,14990,15072,15073,15156,15157,15240,1795,1878,1879,1962,1963,2046,8393,8476,8477,8560,8561,8644,14991,15074,15075,15158,15159,15242,1796,1878,1880,1962,1964,8394,8476,8478,8560,8562,14992,15074,15076,15158,15160,1797,1879,1881,1963,1965,2047,8395,8477,8479,8561,8563,8645,14993,15075,15077,15159,15161,15243,1798,1880,1882,1964,1966,2048,8396,8478,8480,8562,8564,8646,14994,15076,15078,15160,15162,15244,1799,1881,1883,1965,1967,2049,8397,8479,8481,8563,8565,8647,14995,15077,15079,15161,15163,15245,1817,1901,1902,1985,1986,2068,8415,8499,8500,8583,8584,8666,15013,15097,15098,15181,15182,15264,1818,1901,1903,1985,1987,2069,8416,8499,8501,8583,8585,8667,15014,15097,15099,15181,15183,15265,1819,1820,1903,1904,1987,1988,2071,8417,8418,8501,8502,8585,8586,8669,15015,15016,15099,15100,15183,15184,15267,1821,1902,1905,1986,1989,2070,8419,8500,8503,8584,8587,8668,15017,15098,15101,15182,15185,15266,1822,1905,1906,1989,1990,2073,8420,8503,8504,8587,8588,8671,15018,15101,15102,15185,15186,15269,1823,1827,1906,1910,1990,1994,8421,8425,8504,8508,8588,8592,15019,15023,15102,15106,15186,15190,1824,1904,1907,1988,1991,2072,8422,8502,8505,8586,8589,8670,15020,15100,15103,15184,15187,15268,1825,1907,1908,1991,1992,2074,8423,8505,8506,8589,8590,8672,15021,15103,15104,15187,15188,15270,1826,1908,1909,1992,1993,2075,8424,8506,8507,8590,8591,8673,15022,15104,15105,15188,15189,15271,1828,1911,1912,1995,1996,2078,8426,8509,8510,8593,8594,8676,15024,15107,15108,15191,15192,15274,1829,1911,1913,1995,1997,2079,8427,8509,8511,8593,8595,8677,15025,15107,15109,15191,15193,15275,1830,1912,1914,1996,1998,2080,8428,8510,8512,8594,8596,8678,15026,15108,15110,15192,15194,15276,1831,1914,1915,1998,1999,2082,8429,8512,8513,8596,8597,8680,15027,15110,15111,15194,15195,15278,1832,1915,1916,1999,2000,2083,8430,8513,8514,8597,8598,8681,15028,15111,15112,15195,15196,15279,1833,1913,1917,1997,2001,2081,8431,8511,8515,8595,8599,8679,15029,15109,15113,15193,15197,15277,1834,1916,1918,2000,2002,2084,8432,8514,8516,8598,8600,8682,15030,15112,15114,15196,15198,15280,1835,1837,1919,1921,2003,2005,2087,8433,8435,8517,8519,8601,8603,8685,15031,15033,15115,15117,15199,15201,15283,1836,1844,1920,1928,2004,2012,8434,8442,8518,8526,8602,8610,15032,15040,15116,15124,15200,15208,1839,1843,1923,1927,2007,2011,8437,8441,8521,8525,8605,8609,15035,15039,15119,15123,15203,15207,1847,1883,1930,1967,2014,2051,8445,8481,8528,8565,8612,8649,15043,15079,15126,15163,15210,15247,1848,1918,1931,2002,2015,2086,8446,8516,8529,8600,8613,8684,15044,15114,15127,15198,15211,15282,1849,1932,1933,2016,2017,2101,8447,8530,8531,8614,8615,8699,15045,15128,15129,15212,15213,15297,1850,1933,1934,2017,2018,2102,8448,8531,8532,8615,8616,8700,15046,15129,15130,15213,15214,15298,1851,1932,1935,2016,2019,2100,8449,8530,8533,8614,8617,8698,15047,15128,15131,15212,15215,15296,1852,1934,1936,2018,2020,2103,8450,8532,8534,8616,8618,8701,15048,15130,15132,15214,15216,15299,1853,1936,1937,2020,2021,2105,8451,8534,8535,8618,8619,8703,15049,15132,15133,15216,15217,15301,1854,1935,1938,2019,2022,2104,8452,8533,8536,8617,8620,8702,15050,15131,15134,15215,15218,15300,1855,1939,1940,2023,2024,2109,8453,8537,8538,8621,8622,8707,15051,15135,15136,15219,15220,15305,1856,1939,1941,2023,2025,2108,8454,8537,8539,8621,8623,8706,15052,15135,15137,15219,15221,15304,1857,1940,1942,2024,2026,2110,8455,8538,8540,8622,8624,8708,15053,15136,15138,15220,15222,15306,1858,1941,1943,2025,2027,2111,8456,8539,8541,8623,8625,8709,15054,15137,15139,15221,15223,15307,1859,1942,1944,2026,2028,2112,8457,8540,8542,8624,8626,8710,15055,15138,15140,15222,15224,15308,1860,1943,1945,2027,2029,2113,8458,8541,8543,8625,8627,8711,15056,15139,15141,15223,15225,15309,1861,1944,1946,2028,2030,2114,8459,8542,8544,8626,8628,8712,15057,15140,15142,15224,15226,15310,1862,1946,1947,2030,2031,2116,8460,8544,8545,8628,8629,8714,15058,15142,15143,15226,15227,15312,1863,1947,1948,2031,2032,2117,8461,8545,8546,8629,8630,8715,15059,15143,15144,15227,15228,15313,1864,1948,1949,2032,2033,2118,8462,8546,8547,8630,8631,8716,15060,15144,15145,15228,15229,15314,1865,1945,1950,2029,2034,2115,8463,8543,8548,8627,8632,8713,15061,15141,15146,15225,15230,15311,1866,1950,1951,2034,2035,2119,8464,8548,8549,8632,8633,8717,15062,15146,15147,15230,15231,15315,1867,1938,1949,2022,2033,2107,8465,8536,8547,8620,8631,8705,15063,15134,15145,15218,15229,15303,1868,1951,1952,2035,2036,2120,8466,8549,8550,8633,8634,8718,15064,15147,15148,15231,15232,15316,1869,1952,1953,2036,2037,2121,8467,8550,8551,8634,8635,8719,15065,15148,15149,15232,15233,15317,1870,1937,1954,2021,2038,2106,8468,8535,8552,8619,8636,8704,15066,15133,15150,15217,15234,15302,1871,1953,1955,2037,2039,2122,8469,8551,8553,8635,8637,8720,15067,15149,15151,15233,15235,15318,1875,1882,1959,1966,2043,2050,8473,8480,8557,8564,8641,8648,15071,15078,15155,15162,15239,15246,1877,1884,1961,1968,2045,8475,8482,8559,8566,8643,15073,15080,15157,15164,15241,1885,1968,1969,2045,2052,2129,2136,8483,8566,8567,8643,8650,8727,8734,15081,15164,15165,15241,15248,15325,15332,1886,1969,1970,2052,2053,2137,8484,8567,8568,8650,8651,8735,15082,15165,15166,15248,15249,15333,1887,1971,1972,2054,2055,2140,8485,8569,8570,8652,8653,8738,15083,15167,15168,15250,15251,15336,1888,1972,1973,2055,2056,2141,8486,8570,8571,8653,8654,8739,15084,15168,15169,15251,15252,15337,1889,1973,1974,2056,2057,2142,8487,8571,8572,8654,8655,8740,15085,15169,15170,15252,15253,15338,1890,1971,1975,2054,2058,2139,8488,8569,8573,8652,8656,8737,15086,15167,15171,15250,15254,15335,1891,1975,1976,2058,2059,2143,2144,8489,8573,8574,8656,8657,8741,8742,15087,15171,15172,15254,15255,15339,15340,1892,1893,1974,1977,2057,2060,8490,8491,8572,8575,8655,8658,15088,15089,15170,15173,15253,15256,1894,1978,1979,2061,2062,2146,8492,8576,8577,8659,8660,8744,15090,15174,15175,15257,15258,15342,1895,1976,1980,2059,2063,8493,8574,8578,8657,8661,15091,15172,15176,15255,15259,1896,1978,1981,2061,2064,2147,8494,8576,8579,8659,8662,8745,15092,15174,15177,15257,15260,15343,1897,1980,1982,2063,2065,2149,8495,8578,8580,8661,8663,8747,15093,15176,15178,15259,15261,15345,1898,1954,1982,2038,2065,2123,8496,8552,8580,8636,8663,8721,15094,15150,15178,15234,15261,15319,1899,1981,1983,2064,2066,2150,8497,8579,8581,8662,8664,8748,15095,15177,15179,15260,15262,15346,1900,1983,1984,2066,2067,8498,8581,8582,8664,8665,15096,15179,15180,15262,15263,1909,1984,1993,2067,2076,2151,8507,8582,8591,8665,8674,8749,15105,15180,15189,15263,15272,15347,1910,1917,1994,2001,2077,2085,8508,8515,8592,8599,8675,8683,15106,15113,15190,15197,15273,15281,1920,1921,2004,2005,2088,2089,8518,8519,8602,8603,8686,8687,15116,15117,15200,15201,15284,15285,1922,1924,2006,2008,2090,2092,8520,8522,8604,8606,8688,8690,15118,15120,15202,15204,15286,15288,1923,2006,2007,2090,2091,2174,8521,8604,8605,8688,8689,8772,15119,15202,15203,15286,15287,15370,1925,2008,2009,2092,2093,2176,8523,8606,8607,8690,8691,8774,15121,15204,15205,15288,15289,15372,1926,2009,2010,2093,2094,2177,8524,8607,8608,8691,8692,8775,15122,15205,15206,15289,15290,15373,1927,1928,2011,2012,2095,2096,8525,8526,8609,8610,8693,8694,15123,15124,15207,15208,15291,15292,1929,2010,2013,2094,2097,2178,8527,8608,8611,8692,8695,8776,15125,15206,15209,15290,15293,15374,1930,2013,2014,2097,2098,2181,8528,8611,8612,8695,8696,8779,15126,15209,15210,15293,15294,15377,1931,2003,2015,2087,2099,2171,8529,8601,8613,8685,8697,8769,15127,15199,15211,15283,15295,15367,1955,1970,2039,2053,2124,2138,8553,8568,8637,8651,8722,8736,15151,15166,15235,15249,15320,15334,1956,1957,2040,2041,2125,8554,8555,8638,8639,8723,15152,15153,15236,15237,15321,1958,2041,2042,2125,2126,2209,2211,8556,8639,8640,8723,8724,8807,8809,15154,15237,15238,15321,15322,15405,15407,1959,2042,2043,2126,2127,2210,8557,8640,8641,8724,8725,8808,15155,15238,15239,15322,15323,15406,1960,2040,2044,2125,2128,2209,8558,8638,8642,8723,8726,8807,15156,15236,15240,15321,15324,15405,1961,2044,2045,2128,2129,2213,8559,8642,8643,8726,8727,8811,15157,15240,15241,15324,15325,15409,1962,1964,2046,2048,2130,2132,8560,8562,8644,8646,8728,8730,15158,15160,15242,15244,15326,15328,1963,2046,2047,2130,2131,2214,8561,8644,8645,8728,8729,8812,15159,15242,15243,15326,15327,15410,1965,2047,2049,2131,2133,2215,8563,8645,8647,8729,8731,8813,15161,15243,15245,15327,15329,15411,1966,2048,2050,2132,2134,2216,8564,8646,8648,8730,8732,8814,15162,15244,15246,15328,15330,15412,1967,2049,2051,2133,2135,2217,8565,8647,8649,8731,8733,8815,15163,15245,15247,15329,15331,15413,1977,1979,2060,2062,2145,2148,8575,8577,8658,8660,8743,8746,15173,15175,15256,15258,15341,15344,1985,2068,2069,2152,2153,2237,8583,8666,8667,8750,8751,8835,15181,15264,15265,15348,15349,15433,1986,2068,2070,2152,2154,2236,8584,8666,8668,8750,8752,8834,15182,15264,15266,15348,15350,15432,1987,2069,2071,2153,2155,2238,8585,8667,8669,8751,8753,8836,15183,15265,15267,15349,15351,15434,1988,2071,2072,2155,2156,2239,8586,8669,8670,8753,8754,8837,15184,15267,15268,15351,15352,15435,1989,2070,2073,2154,2157,8587,8668,8671,8752,8755,15185,15266,15269,15350,15353,1990,1994,2073,2077,2157,2161,8588,8592,8671,8675,8755,8759,15186,15190,15269,15273,15353,15357,1991,2072,2074,2156,2158,2240,8589,8670,8672,8754,8756,8838,15187,15268,15270,15352,15354,15436,1992,2074,2075,2158,2159,2241,8590,8672,8673,8756,8757,8839,15188,15270,15271,15354,15355,15437,1993,2075,2076,2159,2160,2242,8591,8673,8674,8757,8758,8840,15189,15271,15272,15355,15356,15438,1995,2078,2079,2162,2163,2245,8593,8676,8677,8760,8761,8843,15191,15274,15275,15358,15359,15441,1996,2078,2080,2162,2164,2246,8594,8676,8678,8760,8762,8844,15192,15274,15276,15358,15360,15442,1997,2079,2081,2163,2165,2247,8595,8677,8679,8761,8763,8845,15193,15275,15277,15359,15361,15443,1998,2080,2082,2164,2166,2248,8596,8678,8680,8762,8764,8846,15194,15276,15278,15360,15362,15444,1999,2082,2083,2166,2167,2250,8597,8680,8681,8764,8765,8848,15195,15278,15279,15362,15363,15446,2000,2083,2084,2167,2168,2251,8598,8681,8682,8765,8766,8849,15196,15279,15280,15363,15364,15447,2001,2081,2085,2165,2169,2249,8599,8679,8683,8763,8767,8847,15197,15277,15281,15361,15365,15445,2002,2084,2086,2168,2170,2252,8600,8682,8684,8766,8768,8850,15198,15280,15282,15364,15366,15448,2004,2012,2088,2096,2172,2180,8602,8610,8686,8694,8770,8778,15200,15208,15284,15292,15368,15376,2005,2087,2089,2171,2173,2255,8603,8685,8687,8769,8771,8853,15201,15283,15285,15367,15369,15451,2007,2011,2091,2095,2175,2179,8605,8609,8689,8693,8773,8777,15203,15207,15287,15291,15371,15375,2014,2051,2098,2135,2182,2219,8612,8649,8696,8733,8780,8817,15210,15247,15294,15331,15378,15415,2015,2086,2099,2170,2183,2254,8613,8684,8697,8768,8781,8852,15211,15282,15295,15366,15379,15450,2016,2100,2101,2184,2185,2269,8614,8698,8699,8782,8783,8867,15212,15296,15297,15380,15381,15465,2017,2101,2102,2185,2186,2270,8615,8699,8700,8783,8784,8868,15213,15297,15298,15381,15382,15466,2018,2102,2103,2186,2187,2271,8616,8700,8701,8784,8785,8869,15214,15298,15299,15382,15383,15467,2019,2100,2104,2184,2188,2268,8617,8698,8702,8782,8786,8866,15215,15296,15300,15380,15384,15464,2020,2103,2105,2187,2189,2272,8618,8701,8703,8785,8787,8870,15216,15299,15301,15383,15385,15468,2021,2105,2106,2189,2190,2274,8619,8703,8704,8787,8788,8872,15217,15301,15302,15385,15386,15470,2022,2104,2107,2188,2191,2273,8620,8702,8705,8786,8789,8871,15218,15300,15303,15384,15387,15469,2023,2108,2109,2192,2193,2278,8621,8706,8707,8790,8791,8876,15219,15304,15305,15388,15389,15474,2024,2109,2110,2193,2194,2279,8622,8707,8708,8791,8792,8877,15220,15305,15306,15389,15390,15475,2025,2108,2111,2192,2195,2277,8623,8706,8709,8790,8793,8875,15221,15304,15307,15388,15391,15473,2026,2110,2112,2194,2196,2280,8624,8708,8710,8792,8794,8878,15222,15306,15308,15390,15392,15476,2027,2111,2113,2195,2197,2281,8625,8709,8711,8793,8795,8879,15223,15307,15309,15391,15393,15477,2028,2112,2114,2196,2198,2282,8626,8710,8712,8794,8796,8880,15224,15308,15310,15392,15394,15478,2029,2113,2115,2197,2199,2283,8627,8711,8713,8795,8797,8881,15225,15309,15311,15393,15395,15479,2030,2114,2116,2198,2200,2284,8628,8712,8714,8796,8798,8882,15226,15310,15312,15394,15396,15480,2031,2116,2117,2200,2201,2286,8629,8714,8715,8798,8799,8884,15227,15312,15313,15396,15397,15482,2032,2117,2118,2201,2202,2287,8630,8715,8716,8799,8800,8885,15228,15313,15314,15397,15398,15483,2033,2107,2118,2191,2202,2276,8631,8705,8716,8789,8800,8874,15229,15303,15314,15387,15398,15472,2034,2115,2119,2199,2203,2285,8632,8713,8717,8797,8801,8883,15230,15311,15315,15395,15399,15481,2035,2119,2120,2203,2204,2288,8633,8717,8718,8801,8802,8886,15231,15315,15316,15399,15400,15484,2036,2120,2121,2204,2205,2289,8634,8718,8719,8802,8803,8887,15232,15316,15317,15400,15401,15485,2037,2121,2122,2205,2206,2290,8635,8719,8720,8803,8804,8888,15233,15317,15318,15401,15402,15486,2038,2106,2123,2190,2207,2275,8636,8704,8721,8788,8805,8873,15234,15302,15319,15386,15403,15471,2039,2122,2124,2206,2208,2291,8637,8720,8722,8804,8806,8889,15235,15318,15320,15402,15404,15487,2043,2050,2127,2134,2212,2218,2297,8641,8648,8725,8732,8810,8816,8895,15239,15246,15323,15330,15408,15414,15493,2052,2136,2137,2220,2221,2305,8650,8734,8735,8818,8819,8903,15248,15332,15333,15416,15417,15501,2053,2137,2138,2221,2222,2307,2308,8651,8735,8736,8819,8820,8905,8906,15249,15333,15334,15417,15418,15503,15504,2054,2139,2140,2223,2224,2310,8652,8737,8738,8821,8822,8908,15250,15335,15336,15419,15420,15506,2055,2140,2141,2224,2225,2312,8653,8738,8739,8822,8823,8910,15251,15336,15337,15420,15421,15508,2056,2141,2142,2225,2226,2313,8654,8739,8740,8823,8824,8911,15252,15337,15338,15421,15422,15509,2057,2060,2142,2145,2226,2229,8655,8658,8740,8743,8824,8827,15253,15256,15338,15341,15422,15425,2058,2139,2143,2223,2227,2311,8656,8737,8741,8821,8825,8909,15254,15335,15339,15419,15423,15507,2059,2063,2144,2149,2228,2233,8657,8661,8742,8747,8826,8831,15255,15259,15340,15345,15424,15429,2061,2146,2147,2230,2231,2318,8659,8744,8745,8828,8829,8916,15257,15342,15343,15426,15427,15514,2062,2146,2148,2230,2232,2317,8660,8744,8746,8828,8830,8915,15258,15342,15344,15426,15428,15513,2064,2147,2150,2231,2234,2319,8662,8745,8748,8829,8832,8917,15260,15343,15346,15427,15430,15515,2065,2123,2149,2207,2233,2292,8663,8721,8747,8805,8831,8890,15261,15319,15345,15403,15429,15488,2066,2067,2150,2151,2234,2235,2321,8664,8665,8748,8749,8832,8833,8919,15262,15263,15346,15347,15430,15431,15517,2076,2151,2160,2235,2243,2322,8674,8749,8758,8833,8841,8920,15272,15347,15356,15431,15439,15518,2077,2085,2161,2169,2244,2253,8675,8683,8759,8767,8842,8851,15273,15281,15357,15365,15440,15449,2088,2089,2172,2173,2256,2257,8686,8687,8770,8771,8854,8855,15284,15285,15368,15369,15452,15453,2090,2092,2174,2176,2258,2260,8688,8690,8772,8774,8856,8858,15286,15288,15370,15372,15454,15456,2091,2174,2175,2258,2259,2344,8689,8772,8773,8856,8857,8942,15287,15370,15371,15454,15455,15540,2093,2176,2177,2260,2261,2346,8691,8774,8775,8858,8859,8944,15289,15372,15373,15456,15457,15542,2094,2177,2178,2261,2262,2347,8692,8775,8776,8859,8860,8945,15290,15373,15374,15457,15458,15543,2095,2096,2179,2180,2263,2264,8693,8694,8777,8778,8861,8862,15291,15292,15375,15376,15459,15460,2097,2178,2181,2262,2265,2348,8695,8776,8779,8860,8863,8946,15293,15374,15377,15458,15461,15544,2098,2181,2182,2265,2266,2351,8696,8779,8780,8863,8864,8949,15294,15377,15378,15461,15462,15547,2099,2171,2183,2255,2267,2341,8697,8769,8781,8853,8865,8939,15295,15367,15379,15451,15463,15537,2124,2138,2208,2222,2293,2309,8722,8736,8806,8820,8891,8907,15320,15334,15404,15418,15489,15505,2126,2210,2211,2294,2295,2380,8724,8808,8809,8892,8893,8978,15322,15406,15407,15490,15491,15576,2127,2210,2212,2294,2296,2381,8725,8808,8810,8892,8894,8979,15323,15406,15408,15490,15492,15577,2128,2209,2211,2213,2295,2298,8726,8807,8809,8811,8893,8896,15324,15405,15407,15409,15491,15494,2129,2136,2213,2220,2298,2306,8727,8734,8811,8818,8896,8904,15325,15332,15409,15416,15494,15502,2130,2132,2214,2216,2299,2301,8728,8730,8812,8814,8897,8899,15326,15328,15410,15412,15495,15497,2131,2214,2215,2299,2300,2384,8729,8812,8813,8897,8898,8982,15327,15410,15411,15495,15496,15580,2133,2215,2217,2300,2302,2385,8731,8813,8815,8898,8900,8983,15329,15411,15413,15496,15498,15581,2134,2216,2218,2301,2303,2386,8732,8814,8816,8899,8901,8984,15330,15412,15414,15497,15499,15582,2135,2217,2219,2302,2304,2387,8733,8815,8817,8900,8902,8985,15331,15413,15415,15498,15500,15583,2143,2144,2227,2228,2314,2315,8741,8742,8825,8826,8912,8913,15339,15340,15423,15424,15510,15511,2145,2148,2229,2232,2316,2320,8743,8746,8827,8830,8914,8918,15341,15344,15425,15428,15512,15516,2152,2236,2237,2323,2324,2408,8750,8834,8835,8921,8922,9006,15348,15432,15433,15519,15520,15604,2153,2237,2238,2324,2325,2409,8751,8835,8836,8922,8923,9007,15349,15433,15434,15520,15521,15605,2154,2157,2161,2236,2244,2323,8752,8755,8759,8834,8842,8921,15350,15353,15357,15432,15440,15519,2155,2238,2239,2325,2326,2410,8753,8836,8837,8923,8924,9008,15351,15434,15435,15521,15522,15606,2156,2239,2240,2326,2327,2411,8754,8837,8838,8924,8925,9009,15352,15435,15436,15522,15523,15607,2158,2240,2241,2327,2328,2412,8756,8838,8839,8925,8926,9010,15354,15436,15437,15523,15524,15608,2159,2241,2242,2328,2329,2413,8757,8839,8840,8926,8927,9011,15355,15437,15438,15524,15525,15609,2160,2242,2243,2329,2330,2414,8758,8840,8841,8927,8928,9012,15356,15438,15439,15525,15526,15610,2162,2245,2246,2331,2332,2417,8760,8843,8844,8929,8930,9015,15358,15441,15442,15527,15528,15613,2163,2245,2247,2331,2333,2416,8761,8843,8845,8929,8931,9014,15359,15441,15443,15527,15529,15612,2164,2246,2248,2332,2334,2418,8762,8844,8846,8930,8932,9016,15360,15442,15444,15528,15530,15614,2165,2247,2249,2333,2335,2419,8763,8845,8847,8931,8933,9017,15361,15443,15445,15529,15531,15615,2166,2248,2250,2334,2336,2420,8764,8846,8848,8932,8934,9018,15362,15444,15446,15530,15532,15616,2167,2250,2251,2336,2337,2422,8765,8848,8849,8934,8935,9020,15363,15446,15447,15532,15533,15618,2168,2251,2252,2337,2338,2423,8766,8849,8850,8935,8936,9021,15364,15447,15448,15533,15534,15619,2169,2249,2253,2335,2339,2421,8767,8847,8851,8933,8937,9019,15365,15445,15449,15531,15535,15617,2170,2252,2254,2338,2340,2424,8768,8850,8852,8936,8938,9022,15366,15448,15450,15534,15536,15620,2172,2180,2256,2264,2342,2350,8770,8778,8854,8862,8940,8948,15368,15376,15452,15460,15538,15546,2173,2255,2257,2341,2343,2427,8771,8853,8855,8939,8941,9025,15369,15451,15453,15537,15539,15623,2175,2179,2259,2263,2345,2349,8773,8777,8857,8861,8943,8947,15371,15375,15455,15459,15541,15545,2182,2219,2266,2304,2352,2389,8780,8817,8864,8902,8950,8987,15378,15415,15462,15500,15548,15585,2183,2254,2267,2340,2353,2426,8781,8852,8865,8938,8951,9024,15379,15450,15463,15536,15549,15622,2184,2268,2269,2354,2355,2439,8782,8866,8867,8952,8953,9037,15380,15464,15465,15550,15551,15635,2185,2269,2270,2355,2356,2441,8783,8867,8868,8953,8954,9039,15381,15465,15466,15551,15552,15637,2186,2270,2271,2356,2357,8784,8868,8869,8954,8955,15382,15466,15467,15552,15553,2187,2271,2272,2357,2358,2442,2444,8785,8869,8870,8955,8956,9040,9042,15383,15467,15468,15553,15554,15638,15640,2188,2268,2273,2354,2359,2440,8786,8866,8871,8952,8957,9038,15384,15464,15469,15550,15555,15636,2189,2272,2274,2358,2360,2443,8787,8870,8872,8956,8958,9041,15385,15468,15470,15554,15556,15639,2190,2274,2275,2360,2361,2446,8788,8872,8873,8958,8959,9044,15386,15470,15471,15556,15557,15642,2191,2273,2276,2359,2362,2445,8789,8871,8874,8957,8960,9043,15387,15469,15472,15555,15558,15641,2192,2277,2278,2363,2364,2450,8790,8875,8876,8961,8962,9048,15388,15473,15474,15559,15560,15646,2193,2278,2279,2364,2365,2451,8791,8876,8877,8962,8963,9049,15389,15474,15475,15560,15561,15647,2194,2279,2280,2365,2366,2452,8792,8877,8878,8963,8964,9050,15390,15475,15476,15561,15562,15648,2195,2277,2281,2363,2367,2449,8793,8875,8879,8961,8965,9047,15391,15473,15477,15559,15563,15645,2196,2280,2282,2366,2368,2453,8794,8878,8880,8964,8966,9051,15392,15476,15478,15562,15564,15649,2197,2281,2283,2367,2369,2454,8795,8879,8881,8965,8967,9052,15393,15477,15479,15563,15565,15650,2198,2282,2284,2368,2370,2455,8796,8880,8882,8966,8968,9053,15394,15478,15480,15564,15566,15651,2199,2283,2285,2369,2371,2456,8797,8881,8883,8967,8969,9054,15395,15479,15481,15565,15567,15652,2200,2284,2286,2370,2372,2457,8798,8882,8884,8968,8970,9055,15396,15480,15482,15566,15568,15653,2201,2286,2287,2372,2373,2459,8799,8884,8885,8970,8971,9057,15397,15482,15483,15568,15569,15655,2202,2276,2287,2362,2373,2448,8800,8874,8885,8960,8971,9046,15398,15472,15483,15558,15569,15644,2203,2285,2288,2371,2374,2458,8801,8883,8886,8969,8972,9056,15399,15481,15484,15567,15570,15654,2204,2288,2289,2374,2375,2460,8802,8886,8887,8972,8973,9058,15400,15484,15485,15570,15571,15656,2205,2289,2290,2375,2376,2461,2462,8803,8887,8888,8973,8974,9059,9060,15401,15485,15486,15571,15572,15657,15658,2206,2290,2291,2376,2377,8804,8888,8889,8974,8975,15402,15486,15487,15572,15573,2207,2275,2292,2361,2378,2447,8805,8873,8890,8959,8976,9045,15403,15471,15488,15557,15574,15643,2208,2291,2293,2377,2379,2463,8806,8889,8891,8975,8977,9061,15404,15487,15489,15573,15575,15659,2212,2296,2297,2382,2383,8810,8894,8895,8980,8981,15408,15492,15493,15578,15579,2218,2297,2303,2383,2388,2468,8816,8895,8901,8981,8986,9066,15414,15493,15499,15579,15584,15664,2220,2305,2306,2390,2391,2475,8818,8903,8904,8988,8989,9073,15416,15501,15502,15586,15587,15671,2221,2305,2307,2390,2392,8819,8903,8905,8988,8990,15417,15501,15503,15586,15588,2222,2308,2309,2393,2394,2478,8820,8906,8907,8991,8992,9076,15418,15504,15505,15589,15590,15674,2223,2310,2311,2395,2396,2481,8821,8908,8909,8993,8994,9079,15419,15506,15507,15591,15592,15677,2224,2310,2312,2395,2397,2480,8822,8908,8910,8993,8995,9078,15420,15506,15508,15591,15593,15676,2225,2312,2313,2397,2398,2483,8823,8910,8911,8995,8996,9081,15421,15508,15509,15593,15594,15679,2226,2229,2313,2316,2398,2401,8824,8827,8911,8914,8996,8999,15422,15425,15509,15512,15594,15597,2227,2311,2314,2396,2399,2482,8825,8909,8912,8994,8997,9080,15423,15507,15510,15592,15595,15678,2228,2233,2292,2315,2378,2400,8826,8831,8890,8913,8976,8998,15424,15429,15488,15511,15574,15596,2230,2317,2318,2402,2403,2487,8828,8915,8916,9000,9001,9085,15426,15513,15514,15598,15599,15683,2231,2318,2319,2403,2404,2489,8829,8916,8917,9001,9002,9087,15427,15514,15515,15599,15600,15685,2232,2317,2320,2402,2405,2488,2491,8830,8915,8918,9000,9003,9086,9089,15428,15513,15516,15598,15601,15684,15687,2234,2319,2321,2404,2406,2490,8832,8917,8919,9002,9004,9088,15430,15515,15517,15600,15602,15686,2235,2321,2322,2406,2407,2492,8833,8919,8920,9004,9005,9090,15431,15517,15518,15602,15603,15688,2243,2322,2330,2407,2415,2493,8841,8920,8928,9005,9013,9091,15439,15518,15526,15603,15611,15689,2244,2253,2323,2339,2408,2425,8842,8851,8921,8937,9006,9023,15440,15449,15519,15535,15604,15621,2256,2257,2342,2343,2428,2429,8854,8855,8940,8941,9026,9027,15452,15453,15538,15539,15624,15625,2258,2260,2344,2346,2430,2432,8856,8858,8942,8944,9028,9030,15454,15456,15540,15542,15626,15628,2259,2344,2345,2430,2431,2515,8857,8942,8943,9028,9029,9113,15455,15540,15541,15626,15627,15711,2261,2346,2347,2432,2433,2517,8859,8944,8945,9030,9031,9115,15457,15542,15543,15628,15629,15713,2262,2347,2348,2433,2434,2518,8860,8945,8946,9031,9032,9116,15458,15543,15544,15629,15630,15714,2263,2264,2349,2350,2435,8861,8862,8947,8948,9033,15459,15460,15545,15546,15631,2265,2348,2351,2434,2436,2519,8863,8946,8949,9032,9034,9117,15461,15544,15547,15630,15632,15715,2266,2351,2352,2436,2437,2521,8864,8949,8950,9034,9035,9119,15462,15547,15548,15632,15633,15717,2267,2341,2353,2427,2438,2512,8865,8939,8951,9025,9036,9110,15463,15537,15549,15623,15634,15708,2293,2309,2379,2394,2464,2479,8891,8907,8977,8992,9062,9077,15489,15505,15575,15590,15660,15675,2294,2380,2381,2465,2466,2550,8892,8978,8979,9063,9064,9148,15490,15576,15577,15661,15662,15746,2295,2298,2306,2380,2391,2465,8893,8896,8904,8978,8989,9063,15491,15494,15502,15576,15587,15661,2296,2381,2382,2466,2467,2551,8894,8979,8980,9064,9065,9149,15492,15577,15578,15662,15663,15747,2299,2301,2384,2386,2469,2471,8897,8899,8982,8984,9067,9069,15495,15497,15580,15582,15665,15667,2300,2384,2385,2469,2470,2554,8898,8982,8983,9067,9068,9152,15496,15580,15581,15665,15666,15750,2302,2385,2387,2470,2472,2555,8900,8983,8985,9068,9070,9153,15498,15581,15583,15666,15668,15751,2303,2386,2388,2471,2473,2556,8901,8984,8986,9069,9071,9154,15499,15582,15584,15667,15669,15752,2304,2387,2389,2472,2474,2557,8902,8985,8987,9070,9072,9155,15500,15583,15585,15668,15670,15753,2307,2308,2392,2393,2476,2477,8905,8906,8990,8991,9074,9075,15503,15504,15588,15589,15672,15673,2314,2315,2399,2400,2484,2485,8912,8913,8997,8998,9082,9083,15510,15511,15595,15596,15680,15681,2316,2320,2401,2405,2486,8914,8918,8999,9003,9084,15512,15516,15597,15601,15682,2324,2408,2409,2425,2494,2510,8922,9006,9007,9023,9092,9108,15520,15604,15605,15621,15690,15706,2325,2409,2410,2494,2495,2579,8923,9007,9008,9092,9093,9177,15521,15605,15606,15690,15691,15775,2326,2410,2411,2495,2496,2580,8924,9008,9009,9093,9094,9178,15522,15606,15607,15691,15692,15776,2327,2411,2412,2496,2497,2581,8925,9009,9010,9094,9095,9179,15523,15607,15608,15692,15693,15777,2328,2412,2413,2497,2498,2582,8926,9010,9011,9095,9096,9180,15524,15608,15609,15693,15694,15778,2329,2413,2414,2498,2499,2583,8927,9011,9012,9096,9097,9181,15525,15609,15610,15694,15695,15779,2330,2414,2415,2499,2500,2584,8928,9012,9013,9097,9098,9182,15526,15610,15611,15695,15696,15780,2331,2416,2417,2501,2502,2587,8929,9014,9015,9099,9100,9185,15527,15612,15613,15697,15698,15783,2332,2417,2418,2502,2503,2588,8930,9015,9016,9100,9101,9186,15528,15613,15614,15698,15699,15784,2333,2416,2419,2501,2504,2586,8931,9014,9017,9099,9102,9184,15529,15612,15615,15697,15700,15782,2334,2418,2420,2503,2505,2589,8932,9016,9018,9101,9103,9187,15530,15614,15616,15699,15701,15785,2335,2419,2421,2504,2506,2590,8933,9017,9019,9102,9104,9188,15531,15615,15617,15700,15702,15786,2336,2420,2422,2505,2507,2591,8934,9018,9020,9103,9105,9189,15532,15616,15618,15701,15703,15787,2337,2422,2423,2507,2508,2593,8935,9020,9021,9105,9106,9191,15533,15618,15619,15703,15704,15789,2338,2423,2424,2508,2509,2594,8936,9021,9022,9106,9107,9192,15534,15619,15620,15704,15705,15790,2339,2421,2425,2506,2510,2592,8937,9019,9023,9104,9108,9190,15535,15617,15621,15702,15706,15788,2340,2424,2426,2509,2511,8938,9022,9024,9107,9109,15536,15620,15622,15705,15707,2342,2350,2428,2435,2513,2520,8940,8948,9026,9033,9111,9118,15538,15546,15624,15631,15709,15716,2343,2427,2429,2512,2514,2596,8941,9025,9027,9110,9112,9194,15539,15623,15625,15708,15710,15792,2345,2349,2431,2435,2516,2520,2600,8943,8947,9029,9033,9114,9118,9198,15541,15545,15627,15631,15712,15716,15796,2352,2389,2437,2474,2522,2559,8950,8987,9035,9072,9120,9157,15548,15585,15633,15670,15718,15755,2353,2426,2438,2511,2523,2595,2607,8951,9024,9036,9109,9121,9193,9205,15549,15622,15634,15707,15719,15791,15803,2354,2439,2440,2524,2525,2609,8952,9037,9038,9122,9123,9207,15550,15635,15636,15720,15721,15805,2355,2439,2441,2524,2526,2610,8953,9037,9039,9122,9124,9208,15551,15635,15637,15720,15722,15806,2356,2357,2441,2442,2526,2527,8954,8955,9039,9040,9124,9125,15552,15553,15637,15638,15722,15723,2358,2443,2444,2528,2529,2613,8956,9041,9042,9126,9127,9211,15554,15639,15640,15724,15725,15809,2359,2440,2445,2525,2530,2611,8957,9038,9043,9123,9128,9209,15555,15636,15641,15721,15726,15807,2360,2443,2446,2528,2531,2614,8958,9041,9044,9126,9129,9212,15556,15639,15642,15724,15727,15810,2361,2446,2447,2531,2532,2617,8959,9044,9045,9129,9130,9215,15557,15642,15643,15727,15728,15813,2362,2445,2448,2530,2533,2616,8960,9043,9046,9128,9131,9214,15558,15641,15644,15726,15729,15812,2363,2449,2450,2534,2535,2619,8961,9047,9048,9132,9133,9217,15559,15645,15646,15730,15731,15815,2364,2450,2451,2535,2536,2621,8962,9048,9049,9133,9134,9219,15560,15646,15647,15731,15732,15817,2365,2451,2452,2536,2537,2622,8963,9049,9050,9134,9135,9220,15561,15647,15648,15732,15733,15818,2366,2452,2453,2537,2538,2623,8964,9050,9051,9135,9136,9221,15562,15648,15649,15733,15734,15819,2367,2449,2454,2534,2539,2620,8965,9047,9052,9132,9137,9218,15563,15645,15650,15730,15735,15816,2368,2453,2455,2538,2540,2624,8966,9051,9053,9136,9138,9222,15564,15649,15651,15734,15736,15820,2369,2454,2456,2539,2541,2625,8967,9052,9054,9137,9139,9223,15565,15650,15652,15735,15737,15821,2370,2455,2457,2540,2542,2626,8968,9053,9055,9138,9140,9224,15566,15651,15653,15736,15738,15822,2371,2456,2458,2541,2543,2627,8969,9054,9056,9139,9141,9225,15567,15652,15654,15737,15739,15823,2372,2457,2459,2542,2544,2628,8970,9055,9057,9140,9142,9226,15568,15653,15655,15738,15740,15824,2373,2448,2459,2533,2544,2618,8971,9046,9057,9131,9142,9216,15569,15644,15655,15729,15740,15814,2374,2458,2460,2543,2545,2629,8972,9056,9058,9141,9143,9227,15570,15654,15656,15739,15741,15825,2375,2460,2461,2545,2546,8973,9058,9059,9143,9144,15571,15656,15657,15741,15742,2376,2377,2462,2463,2547,2548,8974,8975,9060,9061,9145,9146,15572,15573,15658,15659,15743,15744,2378,2400,2447,2485,2532,2570,8976,8998,9045,9083,9130,9168,15574,15596,15643,15681,15728,15766,2379,2463,2464,2548,2549,2632,8977,9061,9062,9146,9147,9230,15575,15659,15660,15744,15745,15828,2382,2383,2467,2468,2552,2553,8980,8981,9065,9066,9150,9151,15578,15579,15663,15664,15748,15749,2388,2468,2473,2553,2558,2637,8986,9066,9071,9151,9156,9235,15584,15664,15669,15749,15754,15833,2390,2392,2475,2476,2560,2561,8988,8990,9073,9074,9158,9159,15586,15588,15671,15672,15756,15757,2391,2465,2475,2550,2560,2634,2644,8989,9063,9073,9148,9158,9232,9242,15587,15661,15671,15746,15756,15830,15840,2393,2477,2478,2562,2563,8991,9075,9076,9160,9161,15589,15673,15674,15758,15759,2394,2478,2479,2563,2564,2647,2648,8992,9076,9077,9161,9162,9245,9246,15590,15674,15675,15759,15760,15843,15844,2395,2480,2481,2565,2566,2650,8993,9078,9079,9163,9164,9248,15591,15676,15677,15761,15762,15846,2396,2481,2482,2566,2567,2652,8994,9079,9080,9164,9165,9250,15592,15677,15678,15762,15763,15848,2397,2480,2483,2565,2568,2651,8995,9078,9081,9163,9166,9249,15593,15676,15679,15761,15764,15847,2398,2401,2483,2486,2568,2571,8996,8999,9081,9084,9166,9169,15594,15597,15679,15682,15764,15767,2399,2482,2484,2567,2569,2653,8997,9080,9082,9165,9167,9251,15595,15678,15680,15763,15765,15849,2402,2487,2488,2572,2573,2657,9000,9085,9086,9170,9171,9255,15598,15683,15684,15768,15769,15853,2403,2487,2489,2572,2574,2658,9001,9085,9087,9170,9172,9256,15599,15683,15685,15768,15770,15854,2404,2489,2490,2574,2575,2660,9002,9087,9088,9172,9173,9258,15600,15685,15686,15770,15771,15856,2405,2486,2491,2571,2576,2656,9003,9084,9089,9169,9174,9254,15601,15682,15687,15767,15772,15852,2406,2490,2492,2575,2577,2661,9004,9088,9090,9173,9175,9259,15602,15686,15688,15771,15773,15857,2407,2492,2493,2577,2578,2663,9005,9090,9091,9175,9176,9261,15603,15688,15689,15773,15774,15859,2415,2493,2500,2578,2585,2664,9013,9091,9098,9176,9183,9262,15611,15689,15696,15774,15781,15860,2428,2429,2513,2514,2597,2598,9026,9027,9111,9112,9195,9196,15624,15625,15709,15710,15793,15794,2430,2432,2515,2517,2599,2602,9028,9030,9113,9115,9197,9200,15626,15628,15711,15713,15795,15798,2431,2515,2516,2599,2601,2685,9029,9113,9114,9197,9199,9283,15627,15711,15712,15795,15797,15881,2433,2517,2518,2602,2603,2688,9031,9115,9116,9200,9201,9286,15629,15713,15714,15798,15799,15884,2434,2518,2519,2603,2604,2689,9032,9116,9117,9201,9202,9287,15630,15714,15715,15799,15800,15885,2436,2519,2521,2604,2605,2690,9034,9117,9119,9202,9203,9288,15632,15715,15717,15800,15801,15886,2437,2521,2522,2605,2606,2691,9035,9119,9120,9203,9204,9289,15633,15717,15718,15801,15802,15887,2438,2512,2523,2596,2608,2682,9036,9110,9121,9194,9206,9280,15634,15708,15719,15792,15804,15878,2442,2444,2527,2529,2612,2615,9040,9042,9125,9127,9210,9213,15638,15640,15723,15725,15808,15811,2461,2462,2546,2547,2630,2631,9059,9060,9144,9145,9228,9229,15657,15658,15742,15743,15826,15827,2464,2479,2549,2564,2633,2649,9062,9077,9147,9162,9231,9247,15660,15675,15745,15760,15829,15845,2466,2550,2551,2634,2635,2720,9064,9148,9149,9232,9233,9318,15662,15746,15747,15830,15831,15916,2467,2551,2552,2635,2636,2721,9065,9149,9150,9233,9234,9319,15663,15747,15748,15831,15832,15917,2469,2471,2554,2556,2638,2640,9067,9069,9152,9154,9236,9238,15665,15667,15750,15752,15834,15836,2470,2554,2555,2638,2639,2724,9068,9152,9153,9236,9237,9322,15666,15750,15751,15834,15835,15920,2472,2555,2557,2639,2641,2725,9070,9153,9155,9237,9239,9323,15668,15751,15753,15835,15837,15921,2473,2556,2558,2640,2642,2726,2728,9071,9154,9156,9238,9240,9324,9326,15669,15752,15754,15836,15838,15922,15924,2474,2557,2559,2641,2643,2727,9072,9155,9157,9239,9241,9325,15670,15753,15755,15837,15839,15923,2476,2477,2561,2562,2645,2646,9074,9075,9159,9160,9243,9244,15672,15673,15757,15758,15841,15842,2484,2485,2569,2570,2654,2655,9082,9083,9167,9168,9252,9253,15680,15681,15765,15766,15850,15851,2488,2491,2573,2576,2659,2662,9086,9089,9171,9174,9257,9260,15684,15687,15769,15772,15855,15858,2494,2510,2579,2592,2665,2678,9092,9108,9177,9190,9263,9276,15690,15706,15775,15788,15861,15874,2495,2579,2580,2665,2666,2752,9093,9177,9178,9263,9264,9350,15691,15775,15776,15861,15862,15948,2496,2580,2581,2666,2667,2753,9094,9178,9179,9264,9265,9351,15692,15776,15777,15862,15863,15949,2497,2581,2582,2667,2668,2754,9095,9179,9180,9265,9266,9352,15693,15777,15778,15863,15864,15950,2498,2582,2583,2668,2669,2755,9096,9180,9181,9266,9267,9353,15694,15778,15779,15864,15865,15951,2499,2583,2584,2669,2670,2756,9097,9181,9182,9267,9268,9354,15695,15779,15780,15865,15866,15952,2500,2584,2585,2670,2671,2757,9098,9182,9183,9268,9269,9355,15696,15780,15781,15866,15867,15953,2501,2586,2587,2672,2673,2760,9099,9184,9185,9270,9271,9358,15697,15782,15783,15868,15869,15956,2502,2587,2588,2673,2674,2761,9100,9185,9186,9271,9272,9359,15698,15783,15784,15869,15870,15957,2503,2588,2589,2674,2675,2762,9101,9186,9187,9272,9273,9360,15699,15784,15785,15870,15871,15958,2504,2586,2590,2672,2676,2759,9102,9184,9188,9270,9274,9357,15700,15782,15786,15868,15872,15955,2505,2589,2591,2675,2677,2763,9103,9187,9189,9273,9275,9361,15701,15785,15787,15871,15873,15959,2506,2590,2592,2676,2678,2764,9104,9188,9190,9274,9276,9362,15702,15786,15788,15872,15874,15960,2507,2591,2593,2677,2679,2765,9105,9189,9191,9275,9277,9363,15703,15787,15789,15873,15875,15961,2508,2593,2594,2679,2680,2766,9106,9191,9192,9277,9278,9364,15704,15789,15790,15875,15876,15962,2509,2511,2594,2595,2680,2681,9107,9109,9192,9193,9278,9279,15705,15707,15790,15791,15876,15877,2513,2520,2597,2600,2683,2686,9111,9118,9195,9198,9281,9284,15709,15716,15793,15796,15879,15882,2514,2596,2598,2682,2684,2768,9112,9194,9196,9280,9282,9366,15710,15792,15794,15878,15880,15964,2516,2600,2601,2686,2687,2772,9114,9198,9199,9284,9285,9370,15712,15796,15797,15882,15883,15968,2522,2559,2606,2643,2692,2730,9120,9157,9204,9241,9290,9328,15718,15755,15802,15839,15888,15926,2523,2607,2608,2693,2694,2780,9121,9205,9206,9291,9292,9378,15719,15803,15804,15889,15890,15976,2524,2609,2610,2695,2696,2783,9122,9207,9208,9293,9294,9381,15720,15805,15806,15891,15892,15979,2525,2609,2611,2695,2697,2782,9123,9207,9209,9293,9295,9380,15721,15805,15807,15891,15893,15978,2526,2527,2610,2612,2696,2698,9124,9125,9208,9210,9294,9296,15722,15723,15806,15808,15892,15894,2528,2613,2614,2699,2700,2787,9126,9211,9212,9297,9298,9385,15724,15809,15810,15895,15896,15983,2529,2613,2615,2699,2701,2786,9127,9211,9213,9297,9299,9384,15725,15809,15811,15895,15897,15982,2530,2611,2616,2697,2702,2784,9128,9209,9214,9295,9300,9382,15726,15807,15812,15893,15898,15980,2531,2614,2617,2700,2703,2788,9129,9212,9215,9298,9301,9386,15727,15810,15813,15896,15899,15984,2532,2570,2617,2655,2703,2742,9130,9168,9215,9253,9301,9340,15728,15766,15813,15851,15899,15938,2533,2616,2618,2702,2704,2790,9131,9214,9216,9300,9302,9388,15729,15812,15814,15898,15900,15986,2534,2619,2620,2705,2706,2792,9132,9217,9218,9303,9304,9390,15730,15815,15816,15901,15902,15988,2535,2619,2621,2705,2707,2793,9133,9217,9219,9303,9305,9391,15731,15815,15817,15901,15903,15989,2536,2621,2622,2707,2708,2795,9134,9219,9220,9305,9306,9393,15732,15817,15818,15903,15904,15991,2537,2622,2623,2708,2709,2796,9135,9220,9221,9306,9307,9394,15733,15818,15819,15904,15905,15992,2538,2623,2624,2709,2710,2797,9136,9221,9222,9307,9308,9395,15734,15819,15820,15905,15906,15993,2539,2620,2625,2706,2711,2794,9137,9218,9223,9304,9309,9392,15735,15816,15821,15902,15907,15990,2540,2624,2626,2710,2712,2798,9138,9222,9224,9308,9310,9396,15736,15820,15822,15906,15908,15994,2541,2625,2627,2711,2713,2799,9139,9223,9225,9309,9311,9397,15737,15821,15823,15907,15909,15995,2542,2626,2628,2712,2714,2800,9140,9224,9226,9310,9312,9398,15738,15822,15824,15908,15910,15996,2543,2627,2629,2713,2715,2801,9141,9225,9227,9311,9313,9399,15739,15823,15825,15909,15911,15997,2544,2618,2628,2704,2714,2791,9142,9216,9226,9302,9312,9389,15740,15814,15824,15900,15910,15987,2545,2546,2629,2630,2715,2716,2802,9143,9144,9227,9228,9313,9314,9400,15741,15742,15825,15826,15911,15912,15998,2547,2548,2631,2632,2717,2718,9145,9146,9229,9230,9315,9316,15743,15744,15827,15828,15913,15914,2549,2632,2633,2718,2719,2805,9147,9230,9231,9316,9317,9403,15745,15828,15829,15914,15915,16001,2552,2553,2636,2637,2722,2723,9150,9151,9234,9235,9320,9321,15748,15749,15832,15833,15918,15919,2558,2637,2642,2723,2729,2810,9156,9235,9240,9321,9327,9408,15754,15833,15838,15919,15925,16006,2560,2561,2644,2645,2731,2732,9158,9159,9242,9243,9329,9330,15756,15757,15840,15841,15927,15928,2562,2563,2646,2647,2733,2734,9160,9161,9244,9245,9331,9332,15758,15759,15842,15843,15929,15930,2564,2648,2649,2735,2736,9162,9246,9247,9333,9334,15760,15844,15845,15931,15932,2565,2650,2651,2737,2738,2825,9163,9248,9249,9335,9336,9423,15761,15846,15847,15933,15934,16021,2566,2650,2652,2737,2739,2824,9164,9248,9250,9335,9337,9422,15762,15846,15848,15933,15935,16020,2567,2652,2653,2739,2740,2826,9165,9250,9251,9337,9338,9424,15763,15848,15849,15935,15936,16022,2568,2571,2651,2656,2738,2743,9166,9169,9249,9254,9336,9341,15764,15767,15847,15852,15934,15939,2569,2653,2654,2740,2741,2827,9167,9251,9252,9338,9339,9425,15765,15849,15850,15936,15937,16023,2572,2657,2658,2744,2745,2832,9170,9255,9256,9342,9343,9430,15768,15853,15854,15940,15941,16028,2573,2657,2659,2744,2746,2831,9171,9255,9257,9342,9344,9429,15769,15853,15855,15940,15942,16027,2574,2658,2660,2745,2747,2833,9172,9256,9258,9343,9345,9431,15770,15854,15856,15941,15943,16029,2575,2660,2661,2747,2748,2835,9173,9258,9259,9345,9346,9433,15771,15856,15857,15943,15944,16031,2576,2656,2662,2743,2749,2830,9174,9254,9260,9341,9347,9428,15772,15852,15858,15939,15945,16026,2577,2661,2663,2748,2750,9175,9259,9261,9346,9348,15773,15857,15859,15944,15946,2578,2663,2664,2750,2751,2837,9176,9261,9262,9348,9349,9435,15774,15859,15860,15946,15947,16033,2585,2664,2671,2751,2758,2838,9183,9262,9269,9349,9356,9436,15781,15860,15867,15947,15954,16034,2595,2607,2681,2693,2767,2779,9193,9205,9279,9291,9365,9377,15791,15803,15877,15889,15963,15975,2597,2598,2683,2684,2769,2770,9195,9196,9281,9282,9367,9368,15793,15794,15879,15880,15965,15966,2599,2602,2685,2688,2771,2774,9197,9200,9283,9286,9369,9372,15795,15798,15881,15884,15967,15970,2601,2685,2687,2771,2773,2858,9199,9283,9285,9369,9371,9456,15797,15881,15883,15967,15969,16054,2603,2688,2689,2774,2775,2861,9201,9286,9287,9372,9373,9459,15799,15884,15885,15970,15971,16057,2604,2689,2690,2775,2776,2862,9202,9287,9288,9373,9374,9460,15800,15885,15886,15971,15972,16058,2605,2690,2691,2776,2777,2863,9203,9288,9289,9374,9375,9461,15801,15886,15887,15972,15973,16059,2606,2691,2692,2777,2778,2864,9204,9289,9290,9375,9376,9462,15802,15887,15888,15973,15974,16060,2608,2682,2694,2768,2781,2855,9206,9280,9292,9366,9379,9453,15804,15878,15890,15964,15977,16051,2612,2615,2698,2701,2785,2789,9210,9213,9296,9299,9383,9387,15808,15811,15894,15897,15981,15985,2630,2631,2716,2717,2803,2804,9228,9229,9314,9315,9401,9402,15826,15827,15912,15913,15999,16000,2633,2649,2719,2736,2806,2823,2894,9231,9247,9317,9334,9404,9421,9492,15829,15845,15915,15932,16002,16019,16090,2634,2644,2720,2731,2807,2818,9232,9242,9318,9329,9405,9416,15830,15840,15916,15927,16003,16014,2635,2720,2721,2807,2808,2895,9233,9318,9319,9405,9406,9493,15831,15916,15917,16003,16004,16091,2636,2721,2722,2808,2809,2896,9234,9319,9320,9406,9407,9494,15832,15917,15918,16004,16005,16092,2638,2640,2724,2726,2811,2813,9236,9238,9322,9324,9409,9411,15834,15836,15920,15922,16007,16009,2639,2724,2725,2811,2812,2899,9237,9322,9323,9409,9410,9497,15835,15920,15921,16007,16008,16095,2641,2725,2727,2812,2814,2900,9239,9323,9325,9410,9412,9498,15837,15921,15923,16008,16010,16096,2642,2728,2729,2815,2816,2904,9240,9326,9327,9413,9414,9502,15838,15924,15925,16011,16012,16100,2643,2727,2730,2814,2817,2902,9241,9325,9328,9412,9415,9500,15839,15923,15926,16010,16013,16098,2645,2646,2732,2733,2819,2820,9243,9244,9330,9331,9417,9418,15841,15842,15928,15929,16015,16016,2647,2648,2734,2735,2821,2822,9245,9246,9332,9333,9419,9420,15843,15844,15930,15931,16017,16018,2654,2655,2741,2742,2828,2829,9252,9253,9339,9340,9426,9427,15850,15851,15937,15938,16024,16025,2659,2662,2746,2749,2834,2836,9257,9260,9344,9347,9432,9434,15855,15858,15942,15945,16030,16032,2665,2678,2752,2764,2839,2851,9263,9276,9350,9362,9437,9449,15861,15874,15948,15960,16035,16047,2666,2752,2753,2839,2840,2928,9264,9350,9351,9437,9438,9526,15862,15948,15949,16035,16036,16124,2667,2753,2754,2840,2841,2929,9265,9351,9352,9438,9439,9527,15863,15949,15950,16036,16037,16125,2668,2754,2755,2841,2842,2930,9266,9352,9353,9439,9440,9528,15864,15950,15951,16037,16038,16126,2669,2755,2756,2842,2843,9267,9353,9354,9440,9441,15865,15951,15952,16038,16039,2670,2756,2757,2843,2844,2931,2933,9268,9354,9355,9441,9442,9529,9531,15866,15952,15953,16039,16040,16127,16129,2671,2757,2758,2844,2845,2932,9269,9355,9356,9442,9443,9530,15867,15953,15954,16040,16041,16128,2672,2759,2760,2846,2847,2936,9270,9357,9358,9444,9445,9534,15868,15955,15956,16042,16043,16132,2673,2760,2761,2847,2848,2937,9271,9358,9359,9445,9446,9535,15869,15956,15957,16043,16044,16133,2674,2761,2762,2848,2849,2938,9272,9359,9360,9446,9447,9536,15870,15957,15958,16044,16045,16134,2675,2762,2763,2849,2850,2939,9273,9360,9361,9447,9448,9537,15871,15958,15959,16045,16046,16135,2676,2759,2764,2846,2851,2935,9274,9357,9362,9444,9449,9533,15872,15955,15960,16042,16047,16131,2677,2763,2765,2850,2852,2940,9275,9361,9363,9448,9450,9538,15873,15959,15961,16046,16048,16136,2679,2765,2766,2852,2853,2941,9277,9363,9364,9450,9451,9539,15875,15961,15962,16048,16049,16137,2680,2681,2766,2767,2853,2854,9278,9279,9364,9365,9451,9452,15876,15877,15962,15963,16049,16050,2683,2686,2769,2772,2856,2859,9281,9284,9367,9370,9454,9457,15879,15882,15965,15968,16052,16055,2684,2768,2770,2855,2857,2943,9282,9366,9368,9453,9455,9541,15880,15964,15966,16051,16053,16139,2687,2772,2773,2859,2860,2947,2948,9285,9370,9371,9457,9458,9545,9546,15883,15968,15969,16055,16056,16143,16144,2692,2730,2778,2817,2865,2906,9290,9328,9376,9415,9463,9504,15888,15926,15974,16013,16061,16102,2693,2779,2780,2866,2867,2956,9291,9377,9378,9464,9465,9554,15889,15975,15976,16062,16063,16152,2694,2780,2781,2867,2868,2957,9292,9378,9379,9465,9466,9555,15890,15976,15977,16063,16064,16153,2695,2782,2783,2869,2870,2960,9293,9380,9381,9467,9468,9558,15891,15978,15979,16065,16066,16156,2696,2698,2783,2785,2870,2872,9294,9296,9381,9383,9468,9470,15892,15894,15979,15981,16066,16068,2697,2782,2784,2869,2871,2959,9295,9380,9382,9467,9469,9557,15893,15978,15980,16065,16067,16155,2699,2786,2787,2873,2874,2963,9297,9384,9385,9471,9472,9561,15895,15982,15983,16069,16070,16159,2700,2787,2788,2874,2875,2965,9298,9385,9386,9472,9473,9563,15896,15983,15984,16070,16071,16161,2701,2786,2789,2873,2876,2964,9299,9384,9387,9471,9474,9562,15897,15982,15985,16069,16072,16160,2702,2784,2790,2871,2877,2961,9300,9382,9388,9469,9475,9559,15898,15980,15986,16067,16073,16157,2703,2742,2788,2829,2875,2918,9301,9340,9386,9427,9473,9516,15899,15938,15984,16025,16071,16114,2704,2790,2791,2877,2878,2967,9302,9388,9389,9475,9476,9565,15900,15986,15987,16073,16074,16163,2705,2792,2793,2879,2880,2970,9303,9390,9391,9477,9478,9568,15901,15988,15989,16075,16076,16166,2706,2792,2794,2879,2881,2969,9304,9390,9392,9477,9479,9567,15902,15988,15990,16075,16077,16165,2707,2793,2795,2880,2882,2971,9305,9391,9393,9478,9480,9569,15903,15989,15991,16076,16078,16167,2708,2795,2796,2882,2883,2973,9306,9393,9394,9480,9481,9571,15904,15991,15992,16078,16079,16169,2709,2796,2797,2883,2884,2974,9307,9394,9395,9481,9482,9572,15905,15992,15993,16079,16080,16170,2710,2797,2798,2884,2885,2975,9308,9395,9396,9482,9483,9573,15906,15993,15994,16080,16081,16171,2711,2794,2799,2881,2886,2972,9309,9392,9397,9479,9484,9570,15907,15990,15995,16077,16082,16168,2712,2798,2800,2885,2887,2976,9310,9396,9398,9483,9485,9574,15908,15994,15996,16081,16083,16172,2713,2799,2801,2886,2888,2977,9311,9397,9399,9484,9486,9575,15909,15995,15997,16082,16084,16173,2714,2791,2800,2878,2887,2968,9312,9389,9398,9476,9485,9566,15910,15987,15996,16074,16083,16164,2715,2801,2802,2888,2889,2978,9313,9399,9400,9486,9487,9576,15911,15997,15998,16084,16085,16174,2716,2802,2803,2889,2890,2979,9314,9400,9401,9487,9488,9577,15912,15998,15999,16085,16086,16175,2717,2718,2804,2805,2891,2892,9315,9316,9402,9403,9489,9490,15913,15914,16000,16001,16087,16088,2719,2805,2806,2892,2893,2982,9317,9403,9404,9490,9491,9580,15915,16001,16002,16088,16089,16178,2722,2723,2809,2810,2897,2898,9320,9321,9407,9408,9495,9496,15918,15919,16005,16006,16093,16094,2726,2728,2813,2815,2901,2903,9324,9326,9411,9413,9499,9501,15922,15924,16009,16011,16097,16099,2729,2810,2816,2898,2905,2988,2996,9327,9408,9414,9496,9503,9586,9594,15925,16006,16012,16094,16101,16184,16192,2731,2732,2818,2819,2907,2908,9329,9330,9416,9417,9505,9506,15927,15928,16014,16015,16103,16104,2733,2734,2820,2821,2909,2910,9331,9332,9418,9419,9507,9508,15929,15930,16016,16017,16105,16106,2735,2736,2822,2823,2911,2912,9333,9334,9420,9421,9509,9510,15931,15932,16018,16019,16107,16108,2737,2824,2825,2913,2914,3004,9335,9422,9423,9511,9512,9602,15933,16020,16021,16109,16110,16200,2738,2743,2825,2830,2914,2919,9336,9341,9423,9428,9512,9517,15934,15939,16021,16026,16110,16115,2739,2824,2826,2913,2915,3005,9337,9422,9424,9511,9513,9603,15935,16020,16022,16109,16111,16201,2740,2826,2827,2915,2916,3006,9338,9424,9425,9513,9514,9604,15936,16022,16023,16111,16112,16202,2741,2827,2828,2916,2917,3007,9339,9425,9426,9514,9515,9605,15937,16023,16024,16112,16113,16203,2744,2831,2832,2920,2921,3011,9342,9429,9430,9518,9519,9609,15940,16027,16028,16116,16117,16207,2745,2832,2833,2921,2922,3012,9343,9430,9431,9519,9520,9610,15941,16028,16029,16117,16118,16208,2746,2831,2834,2920,2923,3010,9344,9429,9432,9518,9521,9608,15942,16027,16030,16116,16119,16206,2747,2833,2835,2922,2924,3013,9345,9431,9433,9520,9522,9611,15943,16029,16031,16118,16120,16209,2748,2750,2835,2837,2924,2926,9346,9348,9433,9435,9522,9524,15944,15946,16031,16033,16120,16122,2749,2830,2836,2919,2925,9347,9428,9434,9517,9523,15945,16026,16032,16115,16121,2751,2837,2838,2926,2927,3016,9349,9435,9436,9524,9525,9614,15947,16033,16034,16122,16123,16212,2758,2838,2845,2927,2934,3017,9356,9436,9443,9525,9532,9615,15954,16034,16041,16123,16130,16213,2767,2779,2854,2866,2942,2955,9365,9377,9452,9464,9540,9553,15963,15975,16050,16062,16138,16151,2769,2770,2856,2857,2944,2945,9367,9368,9454,9455,9542,9543,15965,15966,16052,16053,16140,16141,2771,2774,2858,2861,2946,2950,9369,9372,9456,9459,9544,9548,15967,15970,16054,16057,16142,16146,2773,2858,2860,2946,2949,3036,9371,9456,9458,9544,9547,9634,15969,16054,16056,16142,16145,16232,2775,2861,2862,2950,2951,3039,9373,9459,9460,9548,9549,9637,15971,16057,16058,16146,16147,16235,2776,2862,2863,2951,2952,3040,9374,9460,9461,9549,9550,9638,15972,16058,16059,16147,16148,16236,2777,2863,2864,2952,2953,3041,9375,9461,9462,9550,9551,9639,15973,16059,16060,16148,16149,16237,2778,2864,2865,2953,2954,3042,9376,9462,9463,9551,9552,9640,15974,16060,16061,16149,16150,16238,2781,2855,2868,2943,2958,3033,9379,9453,9466,9541,9556,9631,15977,16051,16064,16139,16154,16229,2785,2789,2872,2876,2962,2966,9383,9387,9470,9474,9560,9564,15981,15985,16068,16072,16158,16162,2803,2804,2890,2891,2980,2981,9401,9402,9488,9489,9578,9579,15999,16000,16086,16087,16176,16177,2806,2893,2894,2983,2984,3072,9404,9491,9492,9581,9582,9670,16002,16089,16090,16179,16180,16268,2807,2818,2895,2907,2985,2998,9405,9416,9493,9505,9583,9596,16003,16014,16091,16103,16181,16194,2808,2895,2896,2985,2986,3075,9406,9493,9494,9583,9584,9673,16004,16091,16092,16181,16182,16271,2809,2896,2897,2986,2987,3076,9407,9494,9495,9584,9585,9674,16005,16092,16093,16182,16183,16272,2811,2813,2899,2901,2989,2991,9409,9411,9497,9499,9587,9589,16007,16009,16095,16097,16185,16187,2812,2899,2900,2989,2990,3079,9410,9497,9498,9587,9588,9677,16008,16095,16096,16185,16186,16275,2814,2900,2902,2990,2992,3080,9412,9498,9500,9588,9590,9678,16010,16096,16098,16186,16188,16276,2815,2903,2904,2993,2994,3084,9413,9501,9502,9591,9592,9682,16011,16099,16100,16189,16190,16280,2816,2904,2905,2994,2995,3085,9414,9502,9503,9592,9593,9683,16012,16100,16101,16190,16191,16281,2817,2902,2906,2992,2997,3082,9415,9500,9504,9590,9595,9680,16013,16098,16102,16188,16193,16278,2819,2820,2908,2909,2999,3000,9417,9418,9506,9507,9597,9598,16015,16016,16104,16105,16195,16196,2821,2822,2910,2911,3001,3002,9419,9420,9508,9509,9599,9600,16017,16018,16106,16107,16197,16198,2823,2894,2912,2984,3003,3074,9421,9492,9510,9582,9601,9672,16019,16090,16108,16180,16199,16270,2828,2829,2917,2918,3008,3009,9426,9427,9515,9516,9606,9607,16024,16025,16113,16114,16204,16205,2834,2836,2923,2925,3014,3015,3105,9432,9434,9521,9523,9612,9613,9703,16030,16032,16119,16121,16210,16211,16301,2839,2851,2928,2935,3018,3025,9437,9449,9526,9533,9616,9623,16035,16047,16124,16131,16214,16221,2840,2928,2929,3018,3019,3109,9438,9526,9527,9616,9617,9707,16036,16124,16125,16214,16215,16305,2841,2929,2930,3019,3020,3110,9439,9527,9528,9617,9618,9708,16037,16125,16126,16215,16216,16306,2842,2843,2930,2931,3020,3021,9440,9441,9528,9529,9618,9619,16038,16039,16126,16127,16216,16217,2844,2932,2933,3022,3023,3113,9442,9530,9531,9620,9621,9711,16040,16128,16129,16218,16219,16309,2845,2932,2934,3022,3024,3112,9443,9530,9532,9620,9622,9710,16041,16128,16130,16218,16220,16308,2846,2935,2936,3025,3026,3116,9444,9533,9534,9623,9624,9714,16042,16131,16132,16221,16222,16312,2847,2936,2937,3026,3027,3117,9445,9534,9535,9624,9625,9715,16043,16132,16133,16222,16223,16313,2848,2937,2938,3027,3028,3118,9446,9535,9536,9625,9626,9716,16044,16133,16134,16223,16224,16314,2849,2938,2939,3028,3029,3119,9447,9536,9537,9626,9627,9717,16045,16134,16135,16224,16225,16315,2850,2939,2940,3029,3030,3120,9448,9537,9538,9627,9628,9718,16046,16135,16136,16225,16226,16316,2852,2940,2941,3030,3031,3121,9450,9538,9539,9628,9629,9719,16048,16136,16137,16226,16227,16317,2853,2854,2941,2942,3031,3032,9451,9452,9539,9540,9629,9630,16049,16050,16137,16138,16227,16228,2856,2859,2944,2947,3034,9454,9457,9542,9545,9632,16052,16055,16140,16143,16230,2857,2943,2945,3033,3035,3123,9455,9541,9543,9631,9633,9721,16053,16139,16141,16229,16231,16319,2860,2948,2949,3037,3038,3127,9458,9546,9547,9635,9636,9725,16056,16144,16145,16233,16234,16323,2865,2906,2954,2997,3043,3088,9463,9504,9552,9595,9641,9686,16061,16102,16150,16193,16239,16284,2866,2955,2956,3044,3045,3136,9464,9553,9554,9642,9643,9734,16062,16151,16152,16240,16241,16332,2867,2956,2957,3045,3046,3137,9465,9554,9555,9643,9644,9735,16063,16152,16153,16241,16242,16333,2868,2957,2958,3046,3047,3138,9466,9555,9556,9644,9645,9736,16064,16153,16154,16242,16243,16334,2869,2959,2960,3048,3049,3141,9467,9557,9558,9646,9647,9739,16065,16155,16156,16244,16245,16337,2870,2872,2960,2962,3049,3051,9468,9470,9558,9560,9647,9649,16066,16068,16156,16158,16245,16247,2871,2959,2961,3048,3050,3140,9469,9557,9559,9646,9648,9738,16067,16155,16157,16244,16246,16336,2873,2963,2964,3052,3053,3144,9471,9561,9562,9650,9651,9742,16069,16159,16160,16248,16249,16340,2874,2963,2965,3052,3054,3145,9472,9561,9563,9650,9652,9743,16070,16159,16161,16248,16250,16341,2875,2918,2965,3009,3054,3100,9473,9516,9563,9607,9652,9698,16071,16114,16161,16205,16250,16296,2876,2964,2966,3053,3055,3146,9474,9562,9564,9651,9653,9744,16072,16160,16162,16249,16251,16342,2877,2961,2967,3050,3056,3142,9475,9559,9565,9648,9654,9740,16073,16157,16163,16246,16252,16338,2878,2967,2968,3056,3057,3148,9476,9565,9566,9654,9655,9746,16074,16163,16164,16252,16253,16344,2879,2969,2970,3058,3059,3151,9477,9567,9568,9656,9657,9749,16075,16165,16166,16254,16255,16347,2880,2970,2971,3059,3060,3152,9478,9568,9569,9657,9658,9750,16076,16166,16167,16255,16256,16348,2881,2969,2972,3058,3061,3150,9479,9567,9570,9656,9659,9748,16077,16165,16168,16254,16257,16346,2882,2971,2973,3060,3062,3153,9480,9569,9571,9658,9660,9751,16078,16167,16169,16256,16258,16349,2883,2973,2974,3062,3063,3155,9481,9571,9572,9660,9661,9753,16079,16169,16170,16258,16259,16351,2884,2974,2975,3063,3064,3156,9482,9572,9573,9661,9662,9754,16080,16170,16171,16259,16260,16352,2885,2975,2976,3064,3065,3157,9483,9573,9574,9662,9663,9755,16081,16171,16172,16260,16261,16353,2886,2972,2977,3061,3066,3154,9484,9570,9575,9659,9664,9752,16082,16168,16173,16257,16262,16350,2887,2968,2976,3057,3065,3149,3158,9485,9566,9574,9655,9663,9747,9756,16083,16164,16172,16253,16261,16345,16354,2888,2977,2978,3066,3067,3159,9486,9575,9576,9664,9665,9757,16084,16173,16174,16262,16263,16355,2889,2978,2979,3067,3068,3160,9487,9576,9577,9665,9666,9758,16085,16174,16175,16263,16264,16356,2890,2979,2980,3068,3069,3161,9488,9577,9578,9666,9667,9759,16086,16175,16176,16264,16265,16357,2891,2892,2981,2982,3070,3071,9489,9490,9579,9580,9668,9669,16087,16088,16177,16178,16266,16267,2893,2982,2983,3071,3073,3163,9491,9580,9581,9669,9671,9761,16089,16178,16179,16267,16269,16359,2897,2898,2987,2988,3077,3078,9495,9496,9585,9586,9675,9676,16093,16094,16183,16184,16273,16274,2901,2903,2991,2993,3081,3083,9499,9501,9589,9591,9679,9681,16097,16099,16187,16189,16277,16279,2905,2995,2996,3086,3087,3179,9503,9593,9594,9684,9685,9777,16101,16191,16192,16282,16283,16375,2907,2908,2998,2999,3089,3090,9505,9506,9596,9597,9687,9688,16103,16104,16194,16195,16285,16286,2909,2910,3000,3001,3091,3092,9507,9508,9598,9599,9689,9690,16105,16106,16196,16197,16287,16288,2911,2912,3002,3003,3093,3094,9509,9510,9600,9601,9691,9692,16107,16108,16198,16199,16289,16290,2913,3004,3005,3095,3096,3188,9511,9602,9603,9693,9694,9786,16109,16200,16201,16291,16292,16384,2914,2919,2925,3004,3015,3095,9512,9517,9523,9602,9613,9693,16110,16115,16121,16200,16211,16291,2915,3005,3006,3096,3097,3189,9513,9603,9604,9694,9695,9787,16111,16201,16202,16292,16293,16385,2916,3006,3007,3097,3098,3190,9514,9604,9605,9695,9696,9788,16112,16202,16203,16293,16294,16386,2917,3007,3008,3098,3099,3191,9515,9605,9606,9696,9697,9789,16113,16203,16204,16294,16295,16387,2920,3010,3011,3101,3102,3195,9518,9608,9609,9699,9700,9793,16116,16206,16207,16297,16298,16391,2921,3011,3012,3102,3103,3196,9519,9609,9610,9700,9701,9794,16117,16207,16208,16298,16299,16392,2922,3012,3013,3103,3104,3197,9520,9610,9611,9701,9702,9795,16118,16208,16209,16299,16300,16393,2923,3010,3014,3101,3106,3194,9521,9608,9612,9699,9704,9792,16119,16206,16210,16297,16302,16390,2924,2926,3013,3016,3104,3107,9522,9524,9611,9614,9702,9705,16120,16122,16209,16212,16300,16303,2927,3016,3017,3107,3108,3200,9525,9614,9615,9705,9706,9798,16123,16212,16213,16303,16304,16396,2931,2933,3021,3023,3111,3114,9529,9531,9619,9621,9709,9712,16127,16129,16217,16219,16307,16310,2934,3017,3024,3108,3115,3201,9532,9615,9622,9706,9713,9799,16130,16213,16220,16304,16311,16397,2942,2955,3032,3044,3122,3135,9540,9553,9630,9642,9720,9733,16138,16151,16228,16240,16318,16331,2944,2945,3034,3035,3124,3125,9542,9543,9632,9633,9722,9723,16140,16141,16230,16231,16320,16321,2946,2950,3036,3039,3126,3130,9544,9548,9634,9637,9724,9728,16142,16146,16232,16235,16322,16326,2947,2948,3034,3037,3124,3128,9545,9546,9632,9635,9722,9726,16143,16144,16230,16233,16320,16324,2949,3036,3038,3126,3129,3218,9547,9634,9636,9724,9727,9816,16145,16232,16234,16322,16325,16414,2951,3039,3040,3130,3131,3222,9549,9637,9638,9728,9729,9820,16147,16235,16236,16326,16327,16418,2952,3040,3041,3131,3132,3223,9550,9638,9639,9729,9730,9821,16148,16236,16237,16327,16328,16419,2953,3041,3042,3132,3133,3224,9551,9639,9640,9730,9731,9822,16149,16237,16238,16328,16329,16420,2954,3042,3043,3133,3134,3225,9552,9640,9641,9731,9732,9823,16150,16238,16239,16329,16330,16421,2958,3033,3047,3123,3139,3216,9556,9631,9645,9721,9737,9814,16154,16229,16243,16319,16335,16412,2962,2966,3051,3055,3143,3147,9560,9564,9649,9653,9741,9745,16158,16162,16247,16251,16339,16343,2980,2981,3069,3070,3162,9578,9579,9667,9668,9760,16176,16177,16265,16266,16358,2983,3072,3073,3164,3165,3257,3259,9581,9670,9671,9762,9763,9855,9857,16179,16268,16269,16360,16361,16453,16455,2984,3072,3074,3164,3166,3256,3260,3352,9582,9670,9672,9762,9764,9854,9858,9950,16180,16268,16270,16360,16362,16452,16456,16548,2985,2998,3075,3089,3167,3182,9583,9596,9673,9687,9765,9780,16181,16194,16271,16285,16363,16378,2986,3075,3076,3167,3168,3261,9584,9673,9674,9765,9766,9859,16182,16271,16272,16363,16364,16457,2987,3076,3077,3168,3169,9585,9674,9675,9766,9767,16183,16272,16273,16364,16365,2988,2996,3078,3087,3170,3180,9586,9594,9676,9685,9768,9778,16184,16192,16274,16283,16366,16376,2989,2991,3079,3081,3171,3173,9587,9589,9677,9679,9769,9771,16185,16187,16275,16277,16367,16369,2990,3079,3080,3171,3172,3264,9588,9677,9678,9769,9770,9862,16186,16275,16276,16367,16368,16460,2992,3080,3082,3172,3174,3265,9590,9678,9680,9770,9772,9863,16188,16276,16278,16368,16370,16461,2993,3083,3084,3175,3176,3268,3270,9591,9681,9682,9773,9774,9866,9868,16189,16279,16280,16371,16372,16464,16466,2994,3084,3085,3176,3177,3271,9592,9682,9683,9774,9775,9869,16190,16280,16281,16372,16373,16467,2995,3085,3086,3177,3178,3272,9593,9683,9684,9775,9776,9870,16191,16281,16282,16373,16374,16468,2997,3082,3088,3174,3181,3267,9595,9680,9686,9772,9779,9865,16193,16278,16284,16370,16377,16463,2999,3000,3090,3091,3183,3184,9597,9598,9688,9689,9781,9782,16195,16196,16286,16287,16379,16380,3001,3002,3092,3093,3185,3186,9599,9600,9690,9691,9783,9784,16197,16198,16288,16289,16381,16382,3003,3074,3094,3166,3187,9601,9672,9692,9764,9785,16199,16270,16290,16362,16383,3008,3009,3099,3100,3192,3193,9606,9607,9697,9698,9790,9791,16204,16205,16295,16296,16388,16389,3014,3105,3106,3198,3199,3292,9612,9703,9704,9796,9797,9890,16210,16301,16302,16394,16395,16488,3015,3095,3105,3188,3198,3282,9613,9693,9703,9786,9796,9880,16211,16291,16301,16384,16394,16478,3018,3025,3109,3116,3202,3209,9616,9623,9707,9714,9800,9807,16214,16221,16305,16312,16398,16405,3019,3109,3110,3202,3203,3296,9617,9707,9708,9800,9801,9894,16215,16305,16306,16398,16399,16492,3020,3021,3110,3111,3203,3204,9618,9619,9708,9709,9801,9802,16216,16217,16306,16307,16399,16400,3022,3112,3113,3205,3206,3298,9620,9710,9711,9803,9804,9896,16218,16308,16309,16401,16402,16494,3023,3113,3114,3206,3207,3300,9621,9711,9712,9804,9805,9898,16219,16309,16310,16402,16403,16496,3024,3112,3115,3205,3208,3299,9622,9710,9713,9803,9806,9897,16220,16308,16311,16401,16404,16495,3026,3116,3117,3209,3210,3303,9624,9714,9715,9807,9808,9901,16222,16312,16313,16405,16406,16499,3027,3117,3118,3210,3211,3304,9625,9715,9716,9808,9809,9902,16223,16313,16314,16406,16407,16500,3028,3118,3119,3211,3212,3305,9626,9716,9717,9809,9810,9903,16224,16314,16315,16407,16408,16501,3029,3119,3120,3212,3213,3306,9627,9717,9718,9810,9811,9904,16225,16315,16316,16408,16409,16502,3030,3120,3121,3213,3214,3307,9628,9718,9719,9811,9812,9905,16226,16316,16317,16409,16410,16503,3031,3032,3121,3122,3214,3215,9629,9630,9719,9720,9812,9813,16227,16228,16317,16318,16410,16411,3035,3123,3125,3216,3217,3309,9633,9721,9723,9814,9815,9907,16231,16319,16321,16412,16413,16505,3037,3127,3128,3219,3220,3312,9635,9725,9726,9817,9818,9910,16233,16323,16324,16415,16416,16508,3038,3127,3129,3219,3221,3313,9636,9725,9727,9817,9819,9911,16234,16323,16325,16415,16417,16509,3043,3088,3134,3181,3226,3276,9641,9686,9732,9779,9824,9874,16239,16284,16330,16377,16422,16472,3044,3135,3136,3227,3228,3321,9642,9733,9734,9825,9826,9919,16240,16331,16332,16423,16424,16517,3045,3136,3137,3228,3229,3322,9643,9734,9735,9826,9827,9920,16241,16332,16333,16424,16425,16518,3046,3137,3138,3229,3230,3323,9644,9735,9736,9827,9828,9921,16242,16333,16334,16425,16426,16519,3047,3138,3139,3230,3231,3324,9645,9736,9737,9828,9829,9922,16243,16334,16335,16426,16427,16520,3048,3140,3141,3232,3233,3326,9646,9738,9739,9830,9831,9924,16244,16336,16337,16428,16429,16522,3049,3051,3141,3143,3233,3235,9647,9649,9739,9741,9831,9833,16245,16247,16337,16339,16429,16431,3050,3140,3142,3232,3234,3327,9648,9738,9740,9830,9832,9925,16246,16336,16338,16428,16430,16523,3052,3144,3145,3236,3237,3330,9650,9742,9743,9834,9835,9928,16248,16340,16341,16432,16433,16526,3053,3144,3146,3236,3238,3331,9651,9742,9744,9834,9836,9929,16249,16340,16342,16432,16434,16527,3054,3100,3145,3193,3237,3287,9652,9698,9743,9791,9835,9885,16250,16296,16341,16389,16433,16483,3055,3146,3147,3238,3239,3332,9653,9744,9745,9836,9837,9930,16251,16342,16343,16434,16435,16528,3056,3142,3148,3234,3240,3328,3334,9654,9740,9746,9832,9838,9926,9932,16252,16338,16344,16430,16436,16524,16530,3057,3148,3149,3240,3241,3335,9655,9746,9747,9838,9839,9933,16253,16344,16345,16436,16437,16531,3058,3150,3151,3242,3243,3337,9656,9748,9749,9840,9841,9935,16254,16346,16347,16438,16439,16533,3059,3151,3152,3243,3244,3339,9657,9749,9750,9841,9842,9937,16255,16347,16348,16439,16440,16535,3060,3152,3153,3244,3245,3340,9658,9750,9751,9842,9843,9938,16256,16348,16349,16440,16441,16536,3061,3150,3154,3242,3246,3338,9659,9748,9752,9840,9844,9936,16257,16346,16350,16438,16442,16534,3062,3153,3155,3245,3247,3341,9660,9751,9753,9843,9845,9939,16258,16349,16351,16441,16443,16537,3063,3155,3156,3247,3248,3343,9661,9753,9754,9845,9846,9941,16259,16351,16352,16443,16444,16539,3064,3156,3157,3248,3249,3344,9662,9754,9755,9846,9847,9942,16260,16352,16353,16444,16445,16540,3065,3157,3158,3249,3250,9663,9755,9756,9847,9848,16261,16353,16354,16445,16446,3066,3154,3159,3246,3251,3342,3346,9664,9752,9757,9844,9849,9940,9944,16262,16350,16355,16442,16447,16538,16542,3067,3159,3160,3251,3252,9665,9757,9758,9849,9850,16263,16355,16356,16447,16448,3068,3160,3161,3252,3253,3347,9666,9758,9759,9850,9851,9945,16264,16356,16357,16448,16449,16543,3069,3161,3162,3253,3254,3348,3349,9667,9759,9760,9851,9852,9946,9947,16265,16357,16358,16449,16450,16544,16545,3070,3071,3162,3163,3254,3255,9668,9669,9760,9761,9852,9853,16266,16267,16358,16359,16450,16451,3073,3163,3165,3255,3258,3350,9671,9761,9763,9853,9856,9948,16269,16359,16361,16451,16454,16546,3077,3078,3169,3170,3262,3263,9675,9676,9767,9768,9860,9861,16273,16274,16365,16366,16458,16459,3081,3083,3173,3175,3266,3269,9679,9681,9771,9773,9864,9867,16277,16279,16369,16371,16462,16465,3086,3178,3179,3273,3274,3370,9684,9776,9777,9871,9872,9968,16282,16374,16375,16469,16470,16566,3087,3179,3180,3274,3275,3371,9685,9777,9778,9872,9873,9969,16283,16375,16376,16470,16471,16567,3089,3090,3182,3183,3277,9687,9688,9780,9781,9875,16285,16286,16378,16379,16473,3091,3092,3184,3185,3278,3279,9689,9690,9782,9783,9876,9877,16287,16288,16380,16381,16474,16475,3093,3094,3186,3187,3280,3281,9691,9692,9784,9785,9878,9879,16289,16290,16382,16383,16476,16477,3096,3188,3189,3282,3283,3378,9694,9786,9787,9880,9881,9976,16292,16384,16385,16478,16479,16574,3097,3189,3190,3283,3284,3379,9695,9787,9788,9881,9882,9977,16293,16385,16386,16479,16480,16575,3098,3190,3191,3284,3285,3380,9696,9788,9789,9882,9883,9978,16294,16386,16387,16480,16481,16576,3099,3191,3192,3285,3286,3381,9697,9789,9790,9883,9884,9979,16295,16387,16388,16481,16482,16577,3101,3194,3195,3288,3289,3385,9699,9792,9793,9886,9887,9983,16297,16390,16391,16484,16485,16581,3102,3195,3196,3289,3290,3386,9700,9793,9794,9887,9888,9984,16298,16391,16392,16485,16486,16582,3103,3196,3197,3290,3291,3387,9701,9794,9795,9888,9889,9985,16299,16392,16393,16486,16487,16583,3104,3107,3197,3200,3291,3294,9702,9705,9795,9798,9889,9892,16300,16303,16393,16396,16487,16490,3106,3194,3199,3288,3293,3384,9704,9792,9797,9886,9891,9982,16302,16390,16395,16484,16489,16580,3108,3200,3201,3294,3295,3390,9706,9798,9799,9892,9893,9988,16304,16396,16397,16490,16491,16586,3111,3114,3204,3207,3297,3301,9709,9712,9802,9805,9895,9899,16307,16310,16400,16403,16493,16497,3115,3201,3208,3295,3302,3391,9713,9799,9806,9893,9900,9989,16311,16397,16404,16491,16498,16587,3122,3135,3215,3227,3308,3320,9720,9733,9813,9825,9906,9918,16318,16331,16411,16423,16504,16516,3124,3125,3128,3217,3220,3310,9722,9723,9726,9815,9818,9908,16320,16321,16324,16413,16416,16506,3126,3130,3218,3222,3311,3315,9724,9728,9816,9820,9909,9913,16322,16326,16414,16418,16507,16511,3129,3218,3221,3311,3314,3407,9727,9816,9819,9909,9912,10005,16325,16414,16417,16507,16510,16603,3131,3222,3223,3315,3316,3411,9729,9820,9821,9913,9914,10009,16327,16418,16419,16511,16512,16607,3132,3223,3224,3316,3317,3412,9730,9821,9822,9914,9915,10010,16328,16419,16420,16512,16513,16608,3133,3224,3225,3317,3318,3413,9731,9822,9823,9915,9916,10011,16329,16420,16421,16513,16514,16609,3134,3225,3226,3318,3319,3414,9732,9823,9824,9916,9917,10012,16330,16421,16422,16514,16515,16610,3139,3216,3231,3309,3325,3405,9737,9814,9829,9907,9923,10003,16335,16412,16427,16505,16521,16601,3143,3147,3235,3239,3329,3333,9741,9745,9833,9837,9927,9931,16339,16343,16431,16435,16525,16529,3149,3158,3241,3250,3336,3345,9747,9756,9839,9848,9934,9943,16345,16354,16437,16446,16532,16541,3164,3256,3257,3351,3353,3447,9762,9854,9855,9949,9951,10045,16360,16452,16453,16547,16549,16643,3165,3258,3259,3354,3355,3451,9763,9856,9857,9952,9953,10049,16361,16454,16455,16550,16551,16647,3166,3187,3260,3281,3356,9764,9785,9858,9879,9954,16362,16383,16456,16477,16552,3167,3182,3261,3277,3357,3374,9765,9780,9859,9875,9955,9972,16363,16378,16457,16473,16553,16570,3168,3169,3261,3262,3357,3358,9766,9767,9859,9860,9955,9956,16364,16365,16457,16458,16553,16554,3170,3180,3263,3275,3359,3372,9768,9778,9861,9873,9957,9970,16366,16376,16459,16471,16555,16568,3171,3173,3264,3266,3360,3362,9769,9771,9862,9864,9958,9960,16367,16369,16460,16462,16556,16558,3172,3264,3265,3360,3361,3457,9770,9862,9863,9958,9959,10055,16368,16460,16461,16556,16557,16653,3174,3265,3267,3361,3363,3458,9772,9863,9865,9959,9961,10056,16370,16461,16463,16557,16559,16654,3175,3268,3269,3364,3365,3462,9773,9866,9867,9962,9963,10060,16371,16464,16465,16560,16561,16658,3176,3270,3271,3366,3367,3465,9774,9868,9869,9964,9965,10063,16372,16466,16467,16562,16563,16661,3177,3271,3272,3367,3368,3466,9775,9869,9870,9965,9966,10064,16373,16467,16468,16563,16564,16662,3178,3272,3273,3368,3369,3467,9776,9870,9871,9966,9967,10065,16374,16468,16469,16564,16565,16663,3181,3267,3276,3363,3373,3460,9779,9865,9874,9961,9971,10058,16377,16463,16472,16559,16569,16656,3183,3184,3277,3278,3374,3375,9781,9782,9875,9876,9972,9973,16379,16380,16473,16474,16570,16571,3185,3186,3279,3280,3376,3377,9783,9784,9877,9878,9974,9975,16381,16382,16475,16476,16572,16573,3192,3193,3286,3287,3382,3383,9790,9791,9884,9885,9980,9981,16388,16389,16482,16483,16578,16579,3198,3282,3292,3378,3388,3475,9796,9880,9890,9976,9986,10073,16394,16478,16488,16574,16584,16671,3199,3292,3293,3388,3389,3485,9797,9890,9891,9986,9987,10083,16395,16488,16489,16584,16585,16681,3202,3209,3296,3303,3392,3399,9800,9807,9894,9901,9990,9997,16398,16405,16492,16499,16588,16595,3203,3204,3296,3297,3392,3393,9801,9802,9894,9895,9990,9991,16399,16400,16492,16493,16588,16589,3205,3298,3299,3394,3395,3490,9803,9896,9897,9992,9993,10088,16401,16494,16495,16590,16591,16686,3206,3298,3300,3394,3396,3491,9804,9896,9898,9992,9994,10089,16402,16494,16496,16590,16592,16687,3207,3300,3301,3396,3397,3493,9805,9898,9899,9994,9995,10091,16403,16496,16497,16592,16593,16689,3208,3299,3302,3395,3398,3492,9806,9897,9900,9993,9996,10090,16404,16495,16498,16591,16594,16688,3210,3303,3304,3399,3400,3496,9808,9901,9902,9997,9998,10094,16406,16499,16500,16595,16596,16692,3211,3304,3305,3400,3401,3497,9809,9902,9903,9998,9999,10095,16407,16500,16501,16596,16597,16693,3212,3305,3306,3401,3402,3498,9810,9903,9904,9999,10000,10096,16408,16501,16502,16597,16598,16694,3213,3306,3307,3402,3403,3499,9811,9904,9905,10000,10001,10097,16409,16502,16503,16598,16599,16695,3214,3215,3307,3308,3403,3404,9812,9813,9905,9906,10001,10002,16410,16411,16503,16504,16599,16600,3217,3309,3310,3405,3406,3501,9815,9907,9908,10003,10004,10099,16413,16505,16506,16601,16602,16697,3219,3312,3313,3408,3409,3504,9817,9910,9911,10006,10007,10102,16415,16508,16509,16604,16605,16700,3220,3310,3312,3406,3408,3502,9818,9908,9910,10004,10006,10100,16416,16506,16508,16602,16604,16698,3221,3313,3314,3409,3410,3505,9819,9911,9912,10007,10008,10103,16417,16509,16510,16605,16606,16701,3226,3276,3319,3373,3415,3472,9824,9874,9917,9971,10013,10070,16422,16472,16515,16569,16611,16668,3227,3320,3321,3416,3417,3512,9825,9918,9919,10014,10015,10110,16423,16516,16517,16612,16613,16708,3228,3321,3322,3417,3418,3514,9826,9919,9920,10015,10016,10112,16424,16517,16518,16613,16614,16710,3229,3322,3323,3418,3419,3515,9827,9920,9921,10016,10017,10113,16425,16518,16519,16614,16615,16711,3230,3323,3324,3419,3420,3516,9828,9921,9922,10017,10018,10114,16426,16519,16520,16615,16616,16712,3231,3324,3325,3420,3421,3517,3519,9829,9922,9923,10018,10019,10115,10117,16427,16520,16521,16616,16617,16713,16715,3232,3326,3327,3422,3423,3520,9830,9924,9925,10020,10021,10118,16428,16522,16523,16618,16619,16716,3233,3235,3326,3329,3422,3425,9831,9833,9924,9927,10020,10023,16429,16431,16522,16525,16618,16621,3234,3327,3328,3423,3424,3521,9832,9925,9926,10021,10022,10119,16430,16523,16524,16619,16620,16717,3236,3330,3331,3426,3427,3524,9834,9928,9929,10024,10025,10122,16432,16526,16527,16622,16623,16720,3237,3287,3330,3383,3426,3480,9835,9885,9928,9981,10024,10078,16433,16483,16526,16579,16622,16676,3238,3331,3332,3427,3428,3525,9836,9929,9930,10025,10026,10123,16434,16527,16528,16623,16624,16721,3239,3332,3333,3428,3429,3526,9837,9930,9931,10026,10027,10124,16435,16528,16529,16624,16625,16722,3240,3334,3335,3430,3431,9838,9932,9933,10028,10029,16436,16530,16531,16626,16627,3241,3335,3336,3431,3432,3529,9839,9933,9934,10029,10030,10127,16437,16531,16532,16627,16628,16725,3242,3337,3338,3433,3434,3532,9840,9935,9936,10031,10032,10130,16438,16533,16534,16629,16630,16728,3243,3337,3339,3433,3435,3531,9841,9935,9937,10031,10033,10129,16439,16533,16535,16629,16631,16727,3244,3339,3340,3435,3436,3534,9842,9937,9938,10033,10034,10132,16440,16535,16536,16631,16632,16730,3245,3340,3341,3436,3437,3535,9843,9938,9939,10034,10035,10133,16441,16536,16537,16632,16633,16731,3246,3338,3342,3434,3438,3533,9844,9936,9940,10032,10036,10131,16442,16534,16538,16630,16634,16729,3247,3341,3343,3437,3439,3536,9845,9939,9941,10035,10037,10134,16443,16537,16539,16633,16635,16732,3248,3343,3344,3439,3440,3538,9846,9941,9942,10037,10038,10136,16444,16539,16540,16635,16636,16734,3249,3250,3344,3345,3440,3441,9847,9848,9942,9943,10038,10039,16445,16446,16540,16541,16636,16637,3251,3252,3346,3347,3442,3443,9849,9850,9944,9945,10040,10041,16447,16448,16542,16543,16638,16639,3253,3347,3348,3443,3444,3541,9851,9945,9946,10041,10042,10139,16449,16543,16544,16639,16640,16737,3254,3255,3349,3350,3445,3446,9852,9853,9947,9948,10043,10044,16450,16451,16545,16546,16641,16642,3256,3351,3352,3448,3449,3470,9854,9949,9950,10046,10047,10068,16452,16547,16548,16644,16645,16666,3257,3259,3353,3355,3450,3453,9855,9857,9951,9953,10048,10051,16453,16455,16549,16551,16646,16649,3258,3350,3354,3446,3452,3544,9856,9948,9952,10044,10050,10142,16454,16546,16550,16642,16648,16740,3260,3352,3356,3449,3454,3471,9858,9950,9954,10047,10052,10069,16456,16548,16552,16645,16650,16667,3262,3263,3358,3359,3455,3456,9860,9861,9956,9957,10053,10054,16458,16459,16554,16555,16651,16652,3266,3269,3362,3365,3459,3463,9864,9867,9960,9963,10057,10061,16462,16465,16558,16561,16655,16659,3268,3270,3364,3366,3461,3464,9866,9868,9962,9964,10059,10062,16464,16466,16560,16562,16657,16660,3273,3369,3370,3468,3469,3561,9871,9967,9968,10066,10067,10159,16469,16565,16566,16664,16665,16757,3274,3370,3371,3448,3469,3470,9872,9968,9969,10046,10067,10068,16470,16566,16567,16644,16665,16666,3275,3371,3372,3449,3470,3471,9873,9969,9970,10047,10068,10069,16471,16567,16568,16645,16666,16667,3278,3279,3375,3376,3473,9876,9877,9973,9974,10071,16474,16475,16571,16572,16669,3280,3281,3356,3377,3454,3474,9878,9879,9954,9975,10052,10072,16476,16477,16552,16573,16650,16670,3283,3378,3379,3475,3476,3564,9881,9976,9977,10073,10074,10162,16479,16574,16575,16671,16672,16760,3284,3379,3380,3476,3477,3565,9882,9977,9978,10074,10075,10163,16480,16575,16576,16672,16673,16761,3285,3380,3381,3477,3478,3566,9883,9978,9979,10075,10076,10164,16481,16576,16577,16673,16674,16762,3286,3381,3382,3478,3479,9884,9979,9980,10076,10077,16482,16577,16578,16674,16675,3288,3384,3385,3481,3482,3569,9886,9982,9983,10079,10080,10167,16484,16580,16581,16677,16678,16765,3289,3385,3386,3482,3483,3571,9887,9983,9984,10080,10081,10169,16485,16581,16582,16678,16679,16767,3290,3386,3387,3483,3484,3572,9888,9984,9985,10081,10082,10170,16486,16582,16583,16679,16680,16768,3291,3294,3387,3390,3484,3487,9889,9892,9985,9988,10082,10085,16487,16490,16583,16586,16680,16683,3293,3384,3389,3481,3486,3570,9891,9982,9987,10079,10084,10168,16489,16580,16585,16677,16682,16766,3295,3390,3391,3487,3488,3575,9893,9988,9989,10085,10086,10173,16491,16586,16587,16683,16684,16771,3297,3301,3393,3397,3489,3494,9895,9899,9991,9995,10087,10092,16493,16497,16589,16593,16685,16690,3302,3391,3398,3488,3495,3576,9900,9989,9996,10086,10093,10174,16498,16587,16594,16684,16691,16772,3308,3320,3404,3416,3500,3513,9906,9918,10002,10014,10098,10111,16504,16516,16600,16612,16696,16709,3311,3315,3407,3411,3503,3507,9909,9913,10005,10009,10101,10105,16507,16511,16603,16607,16699,16703,3314,3407,3410,3503,3506,3590,9912,10005,10008,10101,10104,10188,16510,16603,16606,16699,16702,16786,3316,3411,3412,3507,3508,3594,9914,10009,10010,10105,10106,10192,16512,16607,16608,16703,16704,16790,3317,3412,3413,3508,3509,3595,9915,10010,10011,10106,10107,10193,16513,16608,16609,16704,16705,16791,3318,3413,3414,3509,3510,3596,9916,10011,10012,10107,10108,10194,16514,16609,16610,16705,16706,16792,3319,3414,3415,3510,3511,3597,9917,10012,10013,10108,10109,10195,16515,16610,16611,16706,16707,16793,3325,3405,3421,3501,3518,3588,9923,10003,10019,10099,10116,10186,16521,16601,16617,16697,16714,16784,3328,3334,3424,3430,3522,3528,3609,9926,9932,10022,10028,10120,10126,10207,16524,16530,16620,16626,16718,16724,16805,3329,3333,3425,3429,3523,3527,9927,9931,10023,10027,10121,10125,16525,16529,16621,16625,16719,16723,3336,3345,3432,3441,3530,3539,9934,9943,10030,10039,10128,10137,16532,16541,16628,16637,16726,16735,3342,3346,3438,3442,3537,3540,9940,9944,10036,10040,10135,10138,16538,16542,16634,16638,16733,16736,3348,3349,3444,3445,3542,3543,9946,9947,10042,10043,10140,10141,16544,16545,16640,16641,16738,16739,3351,3447,3448,3469,3545,3561,9949,10045,10046,10067,10143,10159,16547,16643,16644,16665,16741,16757,3353,3447,3450,3545,3546,3633,9951,10045,10048,10143,10144,10231,16549,16643,16646,16741,16742,16829,3354,3451,3452,3547,3548,3635,9952,10049,10050,10145,10146,10233,16550,16647,16648,16743,16744,16831,3355,3451,3453,3547,3549,3636,9953,10049,10051,10145,10147,10234,16551,16647,16649,16743,16745,16832,3357,3358,3374,3375,3455,3473,9955,9956,9972,9973,10053,10071,16553,16554,16570,16571,16651,16669,3359,3372,3454,3456,3471,3474,9957,9970,10052,10054,10069,10072,16555,16568,16650,16652,16667,16670,3360,3362,3457,3459,3550,3552,9958,9960,10055,10057,10148,10150,16556,16558,16653,16655,16746,16748,3361,3457,3458,3550,3551,3639,9959,10055,10056,10148,10149,10237,16557,16653,16654,16746,16747,16835,3363,3458,3460,3551,3553,3640,9961,10056,10058,10149,10151,10238,16559,16654,16656,16747,16749,16836,3364,3461,3462,3554,3555,3643,9962,10059,10060,10152,10153,10241,16560,16657,16658,16750,16751,16839,3365,3462,3463,3555,3556,3645,9963,10060,10061,10153,10154,10243,16561,16658,16659,16751,16752,16841,3366,3464,3465,3557,3558,3648,9964,10062,10063,10155,10156,10246,16562,16660,16661,16753,16754,16844,3367,3465,3466,3558,3559,3649,9965,10063,10064,10156,10157,10247,16563,16661,16662,16754,16755,16845,3368,3466,3467,3559,3560,3650,9966,10064,10065,10157,10158,10248,16564,16662,16663,16755,16756,16846,3369,3467,3468,3560,3562,3651,3653,9967,10065,10066,10158,10160,10249,10251,16565,16663,16664,16756,16758,16847,16849,3373,3460,3472,3553,3563,3642,9971,10058,10070,10151,10161,10240,16569,16656,16668,16749,16759,16838,3376,3377,3455,3456,3473,3474,9974,9975,10053,10054,10071,10072,16572,16573,16651,16652,16669,16670,3382,3383,3479,3480,3567,3568,9980,9981,10077,10078,10165,10166,16578,16579,16675,16676,16763,16764,3388,3475,3485,3564,3573,3655,9986,10073,10083,10162,10171,10253,16584,16671,16681,16760,16769,16851,3389,3485,3486,3573,3574,3664,9987,10083,10084,10171,10172,10262,16585,16681,16682,16769,16770,16860,3392,3393,3399,3489,3496,3577,9990,9991,9997,10087,10094,10175,16588,16589,16595,16685,16692,16773,3394,3490,3491,3578,3579,3670,9992,10088,10089,10176,10177,10268,16590,16686,16687,16774,16775,16866,3395,3490,3492,3578,3580,3669,9993,10088,10090,10176,10178,10267,16591,16686,16688,16774,16776,16865,3396,3491,3493,3579,3581,3671,9994,10089,10091,10177,10179,10269,16592,16687,16689,16775,16777,16867,3397,3493,3494,3581,3582,3673,9995,10091,10092,10179,10180,10271,16593,16689,16690,16777,16778,16869,3398,3492,3495,3580,3583,3672,9996,10090,10093,10178,10181,10270,16594,16688,16691,16776,16779,16868,3400,3496,3497,3577,3584,3668,9998,10094,10095,10175,10182,10266,16596,16692,16693,16773,16780,16864,3401,3497,3498,3584,3585,3676,9999,10095,10096,10182,10183,10274,16597,16693,16694,16780,16781,16872,3402,3498,3499,3585,3586,3677,10000,10096,10097,10183,10184,10275,16598,16694,16695,16781,16782,16873,3403,3404,3499,3500,3586,3587,10001,10002,10097,10098,10184,10185,16599,16600,16695,16696,16782,16783,3406,3501,3502,3588,3589,3679,10004,10099,10100,10186,10187,10277,16602,16697,16698,16784,16785,16875,3408,3502,3504,3589,3591,10006,10100,10102,10187,10189,16604,16698,16700,16785,16787,3409,3504,3505,3591,3592,3681,10007,10102,10103,10189,10190,10279,16605,16700,16701,16787,16788,16877,3410,3505,3506,3592,3593,3682,10008,10103,10104,10190,10191,10280,16606,16701,16702,16788,16789,16878,3415,3472,3511,3563,3598,3654,10013,10070,10109,10161,10196,10252,16611,16668,16707,16759,16794,16850,3416,3512,3513,3599,3600,3689,10014,10110,10111,10197,10198,10287,16612,16708,16709,16795,16796,16885,3417,3512,3514,3599,3601,3690,10015,10110,10112,10197,10199,10288,16613,16708,16710,16795,16797,16886,3418,3514,3515,3601,3602,3692,10016,10112,10113,10199,10200,10290,16614,16710,16711,16797,16798,16888,3419,3515,3516,3602,3603,3693,10017,10113,10114,10200,10201,10291,16615,16711,16712,16798,16799,16889,3420,3516,3517,3603,3604,10018,10114,10115,10201,10202,16616,16712,16713,16799,16800,3421,3518,3519,3605,3606,3696,10019,10116,10117,10203,10204,10294,16617,16714,16715,16801,16802,16892,3422,3425,3520,3523,3607,3611,10020,10023,10118,10121,10205,10209,16618,16621,16716,16719,16803,16807,3423,3520,3521,3607,3608,3698,10021,10118,10119,10205,10206,10296,16619,16716,16717,16803,16804,16894,3424,3521,3522,3608,3610,3699,10022,10119,10120,10206,10208,10297,16620,16717,16718,16804,16806,16895,3426,3480,3524,3568,3612,3659,10024,10078,10122,10166,10210,10257,16622,16676,16720,16764,16808,16855,3427,3524,3525,3612,3613,3703,10025,10122,10123,10210,10211,10301,16623,16720,16721,16808,16809,16899,3428,3525,3526,3613,3614,3704,10026,10123,10124,10211,10212,10302,16624,16721,16722,16809,16810,16900,3429,3526,3527,3614,3615,3705,10027,10124,10125,10212,10213,10303,16625,16722,16723,16810,16811,16901,3430,3431,3528,3529,3616,3617,10028,10029,10126,10127,10214,10215,16626,16627,16724,16725,16812,16813,3432,3529,3530,3617,3618,3708,10030,10127,10128,10215,10216,10306,16628,16725,16726,16813,16814,16904,3433,3531,3532,3619,3620,3710,10031,10129,10130,10217,10218,10308,16629,16727,16728,16815,16816,16906,3434,3532,3533,3620,3621,3712,3713,10032,10130,10131,10218,10219,10310,10311,16630,16728,16729,16816,16817,16908,16909,3435,3531,3534,3619,3622,3711,3714,10033,10129,10132,10217,10220,10309,10312,16631,16727,16730,16815,16818,16907,16910,3436,3534,3535,3622,3623,10034,10132,10133,10220,10221,16632,16730,16731,16818,16819,3437,3535,3536,3623,3624,3715,10035,10133,10134,10221,10222,10313,16633,16731,16732,16819,16820,16911,3438,3533,3537,3621,3625,10036,10131,10135,10219,10223,16634,16729,16733,16817,16821,3439,3536,3538,3624,3626,3716,10037,10134,10136,10222,10224,10314,16635,16732,16734,16820,16822,16912,3440,3441,3538,3539,3626,3627,10038,10039,10136,10137,10224,10225,16636,16637,16734,16735,16822,16823,3442,3443,3540,3541,3628,3629,10040,10041,10138,10139,10226,10227,16638,16639,16736,16737,16824,16825,3444,3541,3542,3629,3630,3720,10042,10139,10140,10227,10228,10318,16640,16737,16738,16825,16826,16916,3445,3446,3543,3544,3631,3632,10043,10044,10141,10142,10229,10230,16641,16642,16739,16740,16827,16828,3450,3453,3546,3549,3634,3638,10048,10051,10144,10147,10232,10236,16646,16649,16742,16745,16830,16834,3452,3544,3548,3632,3637,3723,10050,10142,10146,10230,10235,10321,16648,16740,16744,16828,16833,16919,3459,3463,3552,3556,3641,3646,10057,10061,10150,10154,10239,10244,16655,16659,16748,16752,16837,16842,3461,3464,3554,3557,3644,3647,10059,10062,10152,10155,10242,10245,16657,16660,16750,16753,16840,16843,3468,3545,3561,3562,3633,3652,10066,10143,10159,10160,10231,10250,16664,16741,16757,16758,16829,16848,3476,3564,3565,3655,3656,3744,10074,10162,10163,10253,10254,10342,16672,16760,16761,16851,16852,16940,3477,3565,3566,3656,3657,3745,10075,10163,10164,10254,10255,10343,16673,16761,16762,16852,16853,16941,3478,3479,3566,3567,3657,3658,3746,10076,10077,10164,10165,10255,10256,10344,16674,16675,16762,16763,16853,16854,16942,3481,3569,3570,3660,3661,3749,10079,10167,10168,10258,10259,10347,16677,16765,16766,16856,16857,16945,3482,3569,3571,3660,3662,3750,10080,10167,10169,10258,10260,10348,16678,16765,16767,16856,16858,16946,3483,3571,3572,3662,3663,3752,10081,10169,10170,10260,10261,10350,16679,16767,16768,16858,16859,16948,3484,3487,3572,3575,3663,3666,10082,10085,10170,10173,10261,10264,16680,16683,16768,16771,16859,16862,3486,3570,3574,3661,3665,3751,10084,10168,10172,10259,10263,10349,16682,16766,16770,16857,16861,16947,3488,3575,3576,3666,3667,3755,10086,10173,10174,10264,10265,10353,16684,16771,16772,16862,16863,16951,3489,3494,3577,3582,3668,3674,10087,10092,10175,10180,10266,10272,16685,16690,16773,16778,16864,16870,3495,3576,3583,3667,3675,3756,10093,10174,10181,10265,10273,10354,16691,16772,16779,16863,16871,16952,3500,3513,3587,3600,3678,3691,10098,10111,10185,10198,10276,10289,16696,16709,16783,16796,16874,16887,3503,3507,3590,3594,3680,3684,10101,10105,10188,10192,10278,10282,16699,16703,16786,16790,16876,16880,3506,3590,3593,3680,3683,3768,10104,10188,10191,10278,10281,10366,16702,16786,16789,16876,16879,16964,3508,3594,3595,3684,3685,3772,10106,10192,10193,10282,10283,10370,16704,16790,16791,16880,16881,16968,3509,3595,3596,3685,3686,3773,10107,10193,10194,10283,10284,10371,16705,16791,16792,16881,16882,16969,3510,3596,3597,3686,3687,3774,10108,10194,10195,10284,10285,10372,16706,16792,16793,16882,16883,16970,3511,3597,3598,3687,3688,3775,10109,10195,10196,10285,10286,10373,16707,16793,16794,16883,16884,16971,3517,3519,3604,3606,3694,3697,10115,10117,10202,10204,10292,10295,16713,16715,16800,16802,16890,16893,3518,3588,3605,3679,3695,3767,10116,10186,10203,10277,10293,10365,16714,16784,16801,16875,16891,16963,3522,3609,3610,3700,3701,3789,10120,10207,10208,10298,10299,10387,16718,16805,16806,16896,16897,16985,3523,3527,3611,3615,3702,3706,10121,10125,10209,10213,10300,10304,16719,16723,16807,16811,16898,16902,3528,3609,3616,3700,3707,3788,10126,10207,10214,10298,10305,10386,16724,16805,16812,16896,16903,16984,3530,3539,3618,3627,3709,3718,10128,10137,10216,10225,10307,10316,16726,16735,16814,16823,16905,16914,3537,3540,3625,3628,3717,3719,10135,10138,10223,10226,10315,10317,16733,16736,16821,16824,16913,16915,3542,3543,3630,3631,3721,3722,10140,10141,10228,10229,10319,10320,16738,16739,16826,16827,16917,16918,3546,3633,3634,3652,3724,3741,10144,10231,10232,10250,10322,10339,16742,16829,16830,16848,16920,16937,3547,3635,3636,3725,3726,3814,10145,10233,10234,10323,10324,10412,16743,16831,16832,16921,16922,17010,3548,3635,3637,3725,3727,10146,10233,10235,10323,10325,16744,16831,16833,16921,16923,3549,3636,3638,3726,3728,3815,10147,10234,10236,10324,10326,10413,16745,16832,16834,16922,16924,17011,3550,3552,3639,3641,3729,3731,10148,10150,10237,10239,10327,10329,16746,16748,16835,16837,16925,16927,3551,3639,3640,3729,3730,3818,10149,10237,10238,10327,10328,10416,16747,16835,16836,16925,16926,17014,3553,3640,3642,3730,3732,3819,10151,10238,10240,10328,10330,10417,16749,16836,16838,16926,16928,17015,3554,3643,3644,3733,3734,3822,10152,10241,10242,10331,10332,10420,16750,16839,16840,16929,16930,17018,3555,3643,3645,3733,3735,3823,10153,10241,10243,10331,10333,10421,16751,16839,16841,16929,16931,17019,3556,3645,3646,3735,3736,3825,10154,10243,10244,10333,10334,10423,16752,16841,16842,16931,16932,17021,3557,3647,3648,3737,3738,3827,10155,10245,10246,10335,10336,10425,16753,16843,16844,16933,16934,17023,3558,3648,3649,3738,3739,3829,10156,10246,10247,10336,10337,10427,16754,16844,16845,16934,16935,17025,3559,3649,3650,3739,3740,3830,10157,10247,10248,10337,10338,10428,16755,16845,16846,16935,16936,17026,3560,3650,3651,3740,10158,10248,10249,10338,16756,16846,16847,16936,3562,3652,3653,3741,3742,3831,10160,10250,10251,10339,10340,10429,16758,16848,16849,16937,16938,17027,3563,3642,3654,3732,3743,3821,10161,10240,10252,10330,10341,10419,16759,16838,16850,16928,16939,17017,3567,3568,3658,3659,3747,3748,10165,10166,10256,10257,10345,10346,16763,16764,16854,16855,16943,16944,3573,3655,3664,3744,3753,3834,10171,10253,10262,10342,10351,10432,16769,16851,16860,16940,16949,17030,3574,3664,3665,3753,3754,3843,10172,10262,10263,10351,10352,10441,16770,16860,16861,16949,16950,17039,3578,3669,3670,3757,3758,3848,10176,10267,10268,10355,10356,10446,16774,16865,16866,16953,16954,17044,3579,3670,3671,3758,3759,3849,10177,10268,10269,10356,10357,10447,16775,16866,16867,16954,16955,17045,3580,3669,3672,3757,3760,3847,10178,10267,10270,10355,10358,10445,16776,16865,16868,16953,16956,17043,3581,3671,3673,3759,3761,3850,10179,10269,10271,10357,10359,10448,16777,16867,16869,16955,16957,17046,3582,3673,3674,3761,3762,3852,10180,10271,10272,10359,10360,10450,16778,16869,16870,16957,16958,17048,3583,3672,3675,3760,3763,3851,10181,10270,10273,10358,10361,10449,16779,16868,16871,16956,16959,17047,3584,3668,3674,3676,3762,3764,10182,10266,10272,10274,10360,10362,16780,16864,16870,16872,16958,16960,3585,3676,3677,3764,3765,3854,10183,10274,10275,10362,10363,10452,16781,16872,16873,16960,16961,17050,3586,3587,3677,3678,3765,3766,10184,10185,10275,10276,10363,10364,16782,16783,16873,16874,16961,16962,3589,3591,3679,3681,3767,3769,3856,10187,10189,10277,10279,10365,10367,10454,16785,16787,16875,16877,16963,16965,17052,3592,3681,3682,3769,3770,3858,10190,10279,10280,10367,10368,10456,16788,16877,16878,16965,16966,17054,3593,3682,3683,3770,3771,3859,10191,10280,10281,10368,10369,10457,16789,16878,16879,16966,16967,17055,3598,3654,3688,3743,3776,3833,10196,10252,10286,10341,10374,10431,16794,16850,16884,16939,16972,17029,3599,3689,3690,3777,3778,3866,10197,10287,10288,10375,10376,10464,16795,16885,16886,16973,16974,17062,3600,3689,3691,3777,3779,3867,10198,10287,10289,10375,10377,10465,16796,16885,16887,16973,16975,17063,3601,3690,3692,3778,3780,3868,10199,10288,10290,10376,10378,10466,16797,16886,16888,16974,16976,17064,3602,3692,3693,3780,3781,3870,3871,10200,10290,10291,10378,10379,10468,10469,16798,16888,16889,16976,16977,17066,17067,3603,3604,3693,3694,3781,3782,10201,10202,10291,10292,10379,10380,16799,16800,16889,16890,16977,16978,3605,3695,3696,3783,3784,3873,10203,10293,10294,10381,10382,10471,16801,16891,16892,16979,16980,17069,3606,3696,3697,3784,3785,3875,10204,10294,10295,10382,10383,10473,16802,16892,16893,16980,16981,17071,3607,3611,3698,3702,3786,3791,10205,10209,10296,10300,10384,10389,16803,16807,16894,16898,16982,16987,3608,3698,3699,3786,3787,3877,10206,10296,10297,10384,10385,10475,16804,16894,16895,16982,16983,17073,3610,3699,3701,3787,3790,3878,10208,10297,10299,10385,10388,10476,16806,16895,16897,16983,16986,17074,3612,3659,3703,3748,3792,3838,10210,10257,10301,10346,10390,10436,16808,16855,16899,16944,16988,17034,3613,3703,3704,3792,3793,3883,10211,10301,10302,10390,10391,10481,16809,16899,16900,16988,16989,17079,3614,3704,3705,3793,3794,3884,10212,10302,10303,10391,10392,10482,16810,16900,16901,16989,16990,17080,3615,3705,3706,3794,3795,3885,10213,10303,10304,10392,10393,10483,16811,16901,16902,16990,16991,17081,3616,3617,3707,3708,3796,3797,10214,10215,10305,10306,10394,10395,16812,16813,16903,16904,16992,16993,3618,3708,3709,3797,3798,3888,10216,10306,10307,10395,10396,10486,16814,16904,16905,16993,16994,17084,3619,3710,3711,3799,3800,3891,10217,10308,10309,10397,10398,10489,16815,16906,16907,16995,16996,17087,3620,3710,3712,3799,3801,3890,10218,10308,10310,10397,10399,10488,16816,16906,16908,16995,16997,17086,3621,3625,3713,3717,3802,3806,10219,10223,10311,10315,10400,10404,16817,16821,16909,16913,16998,17002,3622,3623,3714,3715,3803,3804,10220,10221,10312,10313,10401,10402,16818,16819,16910,16911,16999,17000,3624,3715,3716,3804,3805,10222,10313,10314,10402,10403,16820,16911,16912,17000,17001,3626,3627,3716,3718,3805,3807,10224,10225,10314,10316,10403,10405,16822,16823,16912,16914,17001,17003,3628,3629,3719,3720,3808,3809,10226,10227,10317,10318,10406,10407,16824,16825,16915,16916,17004,17005,3630,3720,3721,3809,3810,3899,10228,10318,10319,10407,10408,10497,16826,16916,16917,17005,17006,17095,3631,3632,3722,3723,3811,3812,10229,10230,10320,10321,10409,10410,16827,16828,16918,16919,17007,17008,3634,3638,3724,3728,3813,3817,10232,10236,10322,10326,10411,10415,16830,16834,16920,16924,17009,17013,3637,3723,3727,3812,3816,3902,10235,10321,10325,10410,10414,10500,16833,16919,16923,17008,17012,17098,3641,3646,3731,3736,3820,3826,10239,10244,10329,10334,10418,10424,16837,16842,16927,16932,17016,17022,3644,3647,3734,3737,3824,3828,10242,10245,10332,10335,10422,10426,16840,16843,16930,16933,17020,17024,3651,3653,3740,3742,3830,3832,10249,10251,10338,10340,10428,10430,16847,16849,16936,16938,17026,17028,3656,3744,3745,3834,3835,3923,10254,10342,10343,10432,10433,10521,16852,16940,16941,17030,17031,17119,3657,3745,3746,3835,3836,3924,10255,10343,10344,10433,10434,10522,16853,16941,16942,17031,17032,17120,3658,3746,3747,3836,3837,3925,10256,10344,10345,10434,10435,10523,16854,16942,16943,17032,17033,17121,3660,3749,3750,3839,3840,3928,10258,10347,10348,10437,10438,10526,16856,16945,16946,17035,17036,17124,3661,3749,3751,3839,3841,3929,10259,10347,10349,10437,10439,10527,16857,16945,16947,17035,17037,17125,3662,3750,3752,3840,3842,3930,10260,10348,10350,10438,10440,10528,16858,16946,16948,17036,17038,17126,3663,3666,3752,3755,3842,3845,10261,10264,10350,10353,10440,10443,16859,16862,16948,16951,17038,17041,3665,3751,3754,3841,3844,3931,10263,10349,10352,10439,10442,10529,16861,16947,16950,17037,17040,17127,3667,3755,3756,3845,3846,3934,10265,10353,10354,10443,10444,10532,16863,16951,16952,17041,17042,17130,3675,3756,3763,3846,3853,3935,10273,10354,10361,10444,10451,10533,16871,16952,16959,17042,17049,17131,3678,3691,3766,3779,3855,3869,10276,10289,10364,10377,10453,10467,16874,16887,16962,16975,17051,17065,3680,3684,3768,3772,3857,3861,10278,10282,10366,10370,10455,10459,16876,16880,16964,16968,17053,17057,3683,3768,3771,3857,3860,3946,10281,10366,10369,10455,10458,10544,16879,16964,16967,17053,17056,17142,3685,3772,3773,3861,3862,3950,10283,10370,10371,10459,10460,10548,16881,16968,16969,17057,17058,17146,3686,3773,3774,3862,3863,3951,10284,10371,10372,10460,10461,10549,16882,16969,16970,17058,17059,17147,3687,3774,3775,3863,3864,3952,10285,10372,10373,10461,10462,10550,16883,16970,16971,17059,17060,17148,3688,3775,3776,3864,3865,3953,10286,10373,10374,10462,10463,10551,16884,16971,16972,17060,17061,17149,3694,3697,3782,3785,3872,3876,10292,10295,10380,10383,10470,10474,16890,16893,16978,16981,17068,17072,3695,3767,3783,3856,3874,3945,10293,10365,10381,10454,10472,10543,16891,16963,16979,17052,17070,17141,3700,3788,3789,3879,3880,3968,10298,10386,10387,10477,10478,10566,16896,16984,16985,17075,17076,17164,3701,3789,3790,3880,3881,3970,10299,10387,10388,10478,10479,10568,16897,16985,16986,17076,17077,17166,3702,3706,3791,3795,3882,3886,10300,10304,10389,10393,10480,10484,16898,16902,16987,16991,17078,17082,3707,3788,3796,3879,3887,3969,10305,10386,10394,10477,10485,10567,16903,16984,16992,17075,17083,17165,3709,3718,3798,3807,3889,3897,10307,10316,10396,10405,10487,10495,16905,16914,16994,17003,17085,17093,3711,3714,3800,3803,3892,3895,10309,10312,10398,10401,10490,10493,16907,16910,16996,16999,17088,17091,3712,3713,3801,3802,3893,3894,10310,10311,10399,10400,10491,10492,16908,16909,16997,16998,17089,17090,3717,3719,3806,3808,3896,3898,10315,10317,10404,10406,10494,10496,16913,16915,17002,17004,17092,17094,3721,3722,3810,3811,3900,3901,10319,10320,10408,10409,10498,10499,16917,16918,17006,17007,17096,17097,3724,3741,3813,3831,3903,3920,10322,10339,10411,10429,10501,10518,16920,16937,17009,17027,17099,17116,3725,3727,3814,3816,3904,3906,10323,10325,10412,10414,10502,10504,16921,16923,17010,17012,17100,17102,3726,3814,3815,3904,3905,3991,10324,10412,10413,10502,10503,10589,16922,17010,17011,17100,17101,17187,3728,3815,3817,3905,3907,3992,10326,10413,10415,10503,10505,10590,16924,17011,17013,17101,17103,17188,3729,3731,3818,3820,3908,3910,10327,10329,10416,10418,10506,10508,16925,16927,17014,17016,17104,17106,3730,3818,3819,3908,3909,3995,10328,10416,10417,10506,10507,10593,16926,17014,17015,17104,17105,17191,3732,3819,3821,3909,3911,10330,10417,10419,10507,10509,16928,17015,17017,17105,17107,3733,3822,3823,3912,3913,3998,10331,10420,10421,10510,10511,10596,16929,17018,17019,17108,17109,17194,3734,3822,3824,3912,3914,3999,10332,10420,10422,10510,10512,10597,16930,17018,17020,17108,17110,17195,3735,3823,3825,3913,3915,4000,10333,10421,10423,10511,10513,10598,16931,17019,17021,17109,17111,17196,3736,3825,3826,3915,3916,4002,10334,10423,10424,10513,10514,10600,16932,17021,17022,17111,17112,17198,3737,3827,3828,3917,3918,4005,10335,10425,10426,10515,10516,10603,16933,17023,17024,17113,17114,17201,3738,3827,3829,3917,3919,4004,10336,10425,10427,10515,10517,10602,16934,17023,17025,17113,17115,17200,3739,3829,3830,3832,3919,3921,10337,10427,10428,10430,10517,10519,16935,17025,17026,17028,17115,17117,3742,3831,3832,3920,3921,4007,10340,10429,10430,10518,10519,10605,16938,17027,17028,17116,17117,17203,3743,3821,3833,3911,3922,3997,10341,10419,10431,10509,10520,10595,16939,17017,17029,17107,17118,17193,3747,3748,3837,3838,3926,3927,10345,10346,10435,10436,10524,10525,16943,16944,17033,17034,17122,17123,3753,3834,3843,3923,3932,4009,10351,10432,10441,10521,10530,10607,16949,17030,17039,17119,17128,17205,3754,3843,3844,3932,3933,4018,10352,10441,10442,10530,10531,10616,16950,17039,17040,17128,17129,17214,3757,3847,3848,3936,3937,4022,10355,10445,10446,10534,10535,10620,16953,17043,17044,17132,17133,17218,3758,3848,3849,3937,3938,4024,10356,10446,10447,10535,10536,10622,16954,17044,17045,17133,17134,17220,3759,3849,3850,3938,3939,4025,10357,10447,10448,10536,10537,10623,16955,17045,17046,17134,17135,17221,3760,3847,3851,3936,3940,4023,10358,10445,10449,10534,10538,10621,16956,17043,17047,17132,17136,17219,3761,3850,3852,3939,3941,4026,10359,10448,10450,10537,10539,10624,16957,17046,17048,17135,17137,17222,3762,3764,3852,3854,3941,3943,10360,10362,10450,10452,10539,10541,16958,16960,17048,17050,17137,17139,3763,3851,3853,3940,3942,4027,10361,10449,10451,10538,10540,10625,16959,17047,17049,17136,17138,17223,3765,3766,3854,3855,3943,3944,10363,10364,10452,10453,10541,10542,16961,16962,17050,17051,17139,17140,3769,3856,3858,3945,3947,4030,10367,10454,10456,10543,10545,10628,16965,17052,17054,17141,17143,17226,3770,3858,3859,3947,3948,4032,10368,10456,10457,10545,10546,10630,16966,17054,17055,17143,17144,17228,3771,3859,3860,3948,3949,4033,4034,10369,10457,10458,10546,10547,10631,10632,16967,17055,17056,17144,17145,17229,17230,3776,3833,3865,3922,3954,4008,10374,10431,10463,10520,10552,10606,16972,17029,17061,17118,17150,17204,3777,3866,3867,3955,3956,4041,10375,10464,10465,10553,10554,10639,16973,17062,17063,17151,17152,17237,3778,3866,3868,3955,3957,4042,10376,10464,10466,10553,10555,10640,16974,17062,17064,17151,17153,17238,3779,3867,3869,3956,3958,4043,10377,10465,10467,10554,10556,10641,16975,17063,17065,17152,17154,17239,3780,3868,3870,3957,3959,4044,10378,10466,10468,10555,10557,10642,16976,17064,17066,17153,17155,17240,3781,3782,3871,3872,3960,3961,10379,10380,10469,10470,10558,10559,16977,16978,17067,17068,17156,17157,3783,3873,3874,3962,3963,4049,10381,10471,10472,10560,10561,10647,16979,17069,17070,17158,17159,17245,3784,3873,3875,3962,3964,4050,10382,10471,10473,10560,10562,10648,16980,17069,17071,17158,17160,17246,3785,3875,3876,3964,3965,4052,10383,10473,10474,10562,10563,10650,16981,17071,17072,17160,17161,17248,3786,3791,3877,3882,3966,3972,10384,10389,10475,10480,10564,10570,16982,16987,17073,17078,17162,17168,3787,3877,3878,3966,3967,4054,10385,10475,10476,10564,10565,10652,16983,17073,17074,17162,17163,17250,3790,3878,3881,3967,3971,4055,10388,10476,10479,10565,10569,10653,16986,17074,17077,17163,17167,17251,3792,3838,3883,3927,3973,4013,10390,10436,10481,10525,10571,10611,16988,17034,17079,17123,17169,17209,3793,3883,3884,3973,3974,4061,10391,10481,10482,10571,10572,10659,16989,17079,17080,17169,17170,17257,3794,3884,3885,3974,3975,4062,10392,10482,10483,10572,10573,10660,16990,17080,17081,17170,17171,17258,3795,3885,3886,3975,3976,4063,10393,10483,10484,10573,10574,10661,16991,17081,17082,17171,17172,17259,3796,3797,3887,3888,3977,3978,10394,10395,10485,10486,10575,10576,16992,16993,17083,17084,17173,17174,3798,3888,3889,3978,3979,4066,10396,10486,10487,10576,10577,10664,16994,17084,17085,17174,17175,17262,3799,3890,3891,3980,3981,4068,10397,10488,10489,10578,10579,10666,16995,17086,17087,17176,17177,17264,3800,3891,3892,3981,3982,4069,10398,10489,10490,10579,10580,10667,16996,17087,17088,17177,17178,17265,3801,3890,3893,3980,3983,4067,10399,10488,10491,10578,10581,10665,16997,17086,17089,17176,17179,17263,3802,3806,3894,3896,3984,3985,10400,10404,10492,10494,10582,10583,16998,17002,17090,17092,17180,17181,3803,3804,3805,3807,3895,3897,10401,10402,10403,10405,10493,10495,16999,17000,17001,17003,17091,17093,3808,3809,3898,3899,3986,3987,10406,10407,10496,10497,10584,10585,17004,17005,17094,17095,17182,17183,3810,3899,3900,3987,3988,4074,10408,10497,10498,10585,10586,10672,17006,17095,17096,17183,17184,17270,3811,3812,3901,3902,3989,10409,10410,10499,10500,10587,17007,17008,17097,17098,17185,3813,3817,3903,3907,3990,3994,10411,10415,10501,10505,10588,10592,17009,17013,17099,17103,17186,17190,3816,3902,3906,3989,3993,10414,10500,10504,10587,10591,17012,17098,17102,17185,17189,3820,3826,3910,3916,3996,4003,10418,10424,10508,10514,10594,10601,17016,17022,17106,17112,17192,17199,3824,3828,3914,3918,4001,4006,10422,10426,10512,10516,10599,10604,17020,17024,17110,17114,17197,17202,3835,3923,3924,4009,4010,4094,10433,10521,10522,10607,10608,10692,17031,17119,17120,17205,17206,17290,3836,3924,3925,4010,4011,10434,10522,10523,10608,10609,17032,17120,17121,17206,17207,3837,3925,3926,4011,4012,4095,10435,10523,10524,10609,10610,10693,17033,17121,17122,17207,17208,17291,3839,3928,3929,4014,4015,4098,10437,10526,10527,10612,10613,10696,17035,17124,17125,17210,17211,17294,3840,3928,3930,4014,4016,4099,10438,10526,10528,10612,10614,10697,17036,17124,17126,17210,17212,17295,3841,3929,3931,4015,4017,4100,10439,10527,10529,10613,10615,10698,17037,17125,17127,17211,17213,17296,3842,3845,3930,3934,4016,4020,10440,10443,10528,10532,10614,10618,17038,17041,17126,17130,17212,17216,3844,3931,3933,4017,4019,4101,10442,10529,10531,10615,10617,10699,17040,17127,17129,17213,17215,17297,3846,3934,3935,4020,4021,4104,10444,10532,10533,10618,10619,10702,17042,17130,17131,17216,17217,17300,3853,3935,3942,4021,4028,4105,10451,10533,10540,10619,10626,10703,17049,17131,17138,17217,17224,17301,3855,3869,3944,3958,4029,4045,10453,10467,10542,10556,10627,10643,17051,17065,17140,17154,17225,17241,3857,3861,3946,3950,4031,4036,10455,10459,10544,10548,10629,10634,17053,17057,17142,17146,17227,17232,3860,3946,3949,4031,4035,4114,10458,10544,10547,10629,10633,10712,17056,17142,17145,17227,17231,17310,3862,3950,3951,4036,4037,4119,10460,10548,10549,10634,10635,10717,17058,17146,17147,17232,17233,17315,3863,3951,3952,4037,4038,4120,10461,10549,10550,10635,10636,10718,17059,17147,17148,17233,17234,17316,3864,3952,3953,4038,4039,4121,10462,10550,10551,10636,10637,10719,17060,17148,17149,17234,17235,17317,3865,3953,3954,4039,4040,4122,10463,10551,10552,10637,10638,10720,17061,17149,17150,17235,17236,17318,3870,3871,3959,3960,4046,4047,10468,10469,10557,10558,10644,10645,17066,17067,17155,17156,17242,17243,3872,3876,3961,3965,4048,4053,10470,10474,10559,10563,10646,10651,17068,17072,17157,17161,17244,17249,3874,3945,3963,4030,4051,4113,10472,10543,10561,10628,10649,10711,17070,17141,17159,17226,17247,17309,3879,3968,3969,4056,4057,4139,10477,10566,10567,10654,10655,10737,17075,17164,17165,17252,17253,17335,3880,3968,3970,4056,4058,4140,10478,10566,10568,10654,10656,10738,17076,17164,17166,17252,17254,17336,3881,3970,3971,4058,4059,4142,10479,10568,10569,10656,10657,10740,17077,17166,17167,17254,17255,17338,3882,3886,3972,3976,4060,4064,10480,10484,10570,10574,10658,10662,17078,17082,17168,17172,17256,17260,3887,3969,3977,4057,4065,4141,10485,10567,10575,10655,10663,10739,17083,17165,17173,17253,17261,17337,3889,3892,3895,3897,3979,3982,10487,10490,10493,10495,10577,10580,17085,17088,17091,17093,17175,17178,3893,3894,3983,3984,4070,4071,10491,10492,10581,10582,10668,10669,17089,17090,17179,17180,17266,17267,3896,3898,3985,3986,4072,4073,10494,10496,10583,10584,10670,10671,17092,17094,17181,17182,17268,17269,3900,3901,3988,3989,3993,4075,10498,10499,10586,10587,10591,10673,17096,17097,17184,17185,17189,17271,3903,3920,3990,4007,4076,4092,10501,10518,10588,10605,10674,10690,17099,17116,17186,17203,17272,17288,3904,3906,3991,3993,4075,4077,10502,10504,10589,10591,10673,10675,17100,17102,17187,17189,17271,17273,3905,3991,3992,4077,4078,4159,10503,10589,10590,10675,10676,10757,17101,17187,17188,17273,17274,17355,3907,3992,3994,4078,4079,4160,10505,10590,10592,10676,10677,10758,17103,17188,17190,17274,17275,17356,3908,3910,3995,3996,4080,4081,10506,10508,10593,10594,10678,10679,17104,17106,17191,17192,17276,17277,3909,3911,3995,3997,4080,4082,10507,10509,10593,10595,10678,10680,17105,17107,17191,17193,17276,17278,3912,3998,3999,4083,4084,4165,10510,10596,10597,10681,10682,10763,17108,17194,17195,17279,17280,17361,3913,3998,4000,4083,4085,4164,10511,10596,10598,10681,10683,10762,17109,17194,17196,17279,17281,17360,3914,3999,4001,4084,4086,4166,10512,10597,10599,10682,10684,10764,17110,17195,17197,17280,17282,17362,3915,4000,4002,4085,4087,4167,10513,10598,10600,10683,10685,10765,17111,17196,17198,17281,17283,17363,3916,4002,4003,4087,4088,4169,10514,10600,10601,10685,10686,10767,17112,17198,17199,17283,17284,17365,3917,4004,4005,4089,4090,4171,10515,10602,10603,10687,10688,10769,17113,17200,17201,17285,17286,17367,3918,4005,4006,4090,4091,4172,10516,10603,10604,10688,10689,10770,17114,17201,17202,17286,17287,17368,3919,3921,4004,4007,4089,4092,10517,10519,10602,10605,10687,10690,17115,17117,17200,17203,17285,17288,3922,3997,4008,4082,4093,4163,4174,10520,10595,10606,10680,10691,10761,10772,17118,17193,17204,17278,17289,17359,17370,3926,3927,4012,4013,4096,4097,10524,10525,10610,10611,10694,10695,17122,17123,17208,17209,17292,17293,3932,4009,4018,4094,4102,4176,10530,10607,10616,10692,10700,10774,17128,17205,17214,17290,17298,17372,3933,4018,4019,4102,4103,4184,10531,10616,10617,10700,10701,10782,17129,17214,17215,17298,17299,17380,3936,4022,4023,4106,4107,4188,10534,10620,10621,10704,10705,10786,17132,17218,17219,17302,17303,17384,3937,4022,4024,4106,4108,4189,10535,10620,10622,10704,10706,10787,17133,17218,17220,17302,17304,17385,3938,4024,4025,4108,4109,4191,10536,10622,10623,10706,10707,10789,17134,17220,17221,17304,17305,17387,3939,4025,4026,4109,4110,4192,10537,10623,10624,10707,10708,10790,17135,17221,17222,17305,17306,17388,3940,4023,4027,4107,4111,4190,10538,10621,10625,10705,10709,10788,17136,17219,17223,17303,17307,17386,3941,3943,3944,4026,4029,4110,10539,10541,10542,10624,10627,10708,17137,17139,17140,17222,17225,17306,3942,4027,4028,4111,4112,4193,10540,10625,10626,10709,10710,10791,17138,17223,17224,17307,17308,17389,3947,4030,4032,4113,4115,4195,4198,10545,10628,10630,10711,10713,10793,10796,17143,17226,17228,17309,17311,17391,17394,3948,4032,4033,4115,4116,4197,10546,10630,10631,10713,10714,10795,17144,17228,17229,17311,17312,17393,3949,4034,4035,4117,4118,10547,10632,10633,10715,10716,17145,17230,17231,17313,17314,3954,4008,4040,4093,4123,4175,10552,10606,10638,10691,10721,10773,17150,17204,17236,17289,17319,17371,3955,4041,4042,4124,4125,4208,10553,10639,10640,10722,10723,10806,17151,17237,17238,17320,17321,17404,3956,4041,4043,4124,4126,4207,10554,10639,10641,10722,10724,10805,17152,17237,17239,17320,17322,17403,3957,4042,4044,4125,4127,4209,10555,10640,10642,10723,10725,10807,17153,17238,17240,17321,17323,17405,3958,4043,4045,4126,4128,4210,10556,10641,10643,10724,10726,10808,17154,17239,17241,17322,17324,17406,3959,4044,4046,4127,4129,4211,10557,10642,10644,10725,10727,10809,17155,17240,17242,17323,17325,17407,3960,3961,4047,4048,4130,4131,10558,10559,10645,10646,10728,10729,17156,17157,17243,17244,17326,17327,3962,4049,4050,4132,4133,4216,10560,10647,10648,10730,10731,10814,17158,17245,17246,17328,17329,17412,3963,4049,4051,4132,4134,4217,10561,10647,10649,10730,10732,10815,17159,17245,17247,17328,17330,17413,3964,4050,4052,4133,4135,4218,10562,10648,10650,10731,10733,10816,17160,17246,17248,17329,17331,17414,3965,4052,4053,4135,4136,4220,10563,10650,10651,10733,10734,10818,17161,17248,17249,17331,17332,17416,3966,3972,4054,4060,4137,4144,10564,10570,10652,10658,10735,10742,17162,17168,17250,17256,17333,17340,3967,4054,4055,4137,4138,4222,10565,10652,10653,10735,10736,10820,17163,17250,17251,17333,17334,17418,3971,4055,4059,4138,4143,4223,10569,10653,10657,10736,10741,10821,17167,17251,17255,17334,17339,17419,3973,4013,4061,4097,4145,4179,10571,10611,10659,10695,10743,10777,17169,17209,17257,17293,17341,17375,3974,4061,4062,4145,4146,4230,10572,10659,10660,10743,10744,10828,17170,17257,17258,17341,17342,17426,3975,4062,4063,4146,4147,4231,10573,10660,10661,10744,10745,10829,17171,17258,17259,17342,17343,17427,3976,4063,4064,4147,4148,4232,10574,10661,10662,10745,10746,10830,17172,17259,17260,17343,17344,17428,3977,3978,4065,4066,4149,10575,10576,10663,10664,10747,17173,17174,17261,17262,17345,3979,3982,4066,4069,4149,4152,10577,10580,10664,10667,10747,10750,17175,17178,17262,17265,17345,17348,3980,4067,4068,4150,4151,4234,10578,10665,10666,10748,10749,10832,17176,17263,17264,17346,17347,17430,3981,4068,4069,4151,4152,4236,10579,10666,10667,10749,10750,10834,17177,17264,17265,17347,17348,17432,3983,4067,4070,4150,4153,4235,10581,10665,10668,10748,10751,10833,17179,17263,17266,17346,17349,17431,3984,3985,4071,4072,4154,4155,10582,10583,10669,10670,10752,10753,17180,17181,17267,17268,17350,17351,3986,3987,4073,4074,4156,4157,10584,10585,10671,10672,10754,10755,17182,17183,17269,17270,17352,17353,3988,4074,4075,4077,4157,4159,10586,10672,10673,10675,10755,10757,17184,17270,17271,17273,17353,17355,3990,3994,4076,4079,4158,4161,4242,10588,10592,10674,10677,10756,10759,10840,17186,17190,17272,17275,17354,17357,17438,3996,4003,4081,4088,4162,4170,10594,10601,10679,10686,10760,10768,17192,17199,17277,17284,17358,17366,4001,4006,4086,4091,4168,4173,10599,10604,10684,10689,10766,10771,17197,17202,17282,17287,17364,17369,4010,4011,4094,4095,4176,4177,10608,10609,10692,10693,10774,10775,17206,17207,17290,17291,17372,17373,4012,4095,4096,4177,4178,4257,10610,10693,10694,10775,10776,10855,17208,17291,17292,17373,17374,17453,4014,4098,4099,4180,4181,4261,10612,10696,10697,10778,10779,10859,17210,17294,17295,17376,17377,17457,4015,4098,4100,4180,4182,4260,10613,10696,10698,10778,10780,10858,17211,17294,17296,17376,17378,17456,4016,4020,4099,4104,4181,4186,10614,10618,10697,10702,10779,10784,17212,17216,17295,17300,17377,17382,4017,4100,4101,4182,4183,4262,10615,10698,10699,10780,10781,10860,17213,17296,17297,17378,17379,17458,4019,4101,4103,4183,4185,4263,10617,10699,10701,10781,10783,10861,17215,17297,17299,17379,17381,17459,4021,4104,4105,4186,4187,4266,10619,10702,10703,10784,10785,10864,17217,17300,17301,17382,17383,17462,4028,4105,4112,4187,4194,4267,10626,10703,10710,10785,10792,10865,17224,17301,17308,17383,17390,17463,4029,4045,4110,4128,4192,4212,10627,10643,10708,10726,10790,10810,17225,17241,17306,17324,17388,17408,4031,4036,4114,4119,4196,4202,10629,10634,10712,10717,10794,10800,17227,17232,17310,17315,17392,17398,4033,4034,4116,4117,4199,4200,10631,10632,10714,10715,10797,10798,17229,17230,17312,17313,17395,17396,4035,4114,4118,4196,4201,4275,10633,10712,10716,10794,10799,10873,17231,17310,17314,17392,17397,17471,4037,4119,4120,4202,4203,4281,10635,10717,10718,10800,10801,10879,17233,17315,17316,17398,17399,17477,4038,4120,4121,4203,4204,4282,10636,10718,10719,10801,10802,10880,17234,17316,17317,17399,17400,17478,4039,4121,4122,4204,4205,4283,10637,10719,10720,10802,10803,10881,17235,17317,17318,17400,17401,17479,4040,4122,4123,4205,4206,4284,10638,10720,10721,10803,10804,10882,17236,17318,17319,17401,17402,17480,4046,4047,4129,4130,4213,4214,10644,10645,10727,10728,10811,10812,17242,17243,17325,17326,17409,17410,4048,4053,4131,4136,4215,4221,10646,10651,10729,10734,10813,10819,17244,17249,17327,17332,17411,17417,4051,4113,4134,4195,4219,4274,10649,10711,10732,10793,10817,10872,17247,17309,17330,17391,17415,17470,4056,4139,4140,4224,4225,10654,10737,10738,10822,10823,17252,17335,17336,17420,17421,4057,4139,4141,4224,4226,4303,10655,10737,10739,10822,10824,10901,17253,17335,17337,17420,17422,17499,4058,4140,4142,4225,4227,4304,10656,10738,10740,10823,10825,10902,17254,17336,17338,17421,17423,17500,4059,4142,4143,4227,4228,4306,10657,10740,10741,10825,10826,10904,17255,17338,17339,17423,17424,17502,4060,4064,4144,4148,4229,4233,10658,10662,10742,10746,10827,10831,17256,17260,17340,17344,17425,17429,4065,4141,4149,4152,4226,4236,4305,10663,10739,10747,10750,10824,10834,10903,17261,17337,17345,17348,17422,17432,17501,4070,4071,4153,4154,4237,4238,10668,10669,10751,10752,10835,10836,17266,17267,17349,17350,17433,17434,4072,4073,4155,4156,4239,4240,10670,10671,10753,10754,10837,10838,17268,17269,17351,17352,17435,17436,4076,4089,4092,4158,4171,4241,4252,10674,10687,10690,10756,10769,10839,10850,17272,17285,17288,17354,17367,17437,17448,4078,4156,4157,4159,4160,4240,10676,10754,10755,10757,10758,10838,17274,17352,17353,17355,17356,17436,4079,4160,4161,4239,4240,4243,10677,10758,10759,10837,10838,10841,17275,17356,17357,17435,17436,17439,4080,4081,4082,4162,4163,4244,10678,10679,10680,10760,10761,10842,17276,17277,17278,17358,17359,17440,4083,4164,4165,4245,4246,4321,10681,10762,10763,10843,10844,10919,17279,17360,17361,17441,17442,17517,4084,4165,4166,4246,4247,4322,10682,10763,10764,10844,10845,10920,17280,17361,17362,17442,17443,17518,4085,4164,4167,4245,4248,4320,10683,10762,10765,10843,10846,10918,17281,17360,17363,17441,17444,17516,4086,4166,4168,4247,4249,4323,10684,10764,10766,10845,10847,10921,17282,17362,17364,17443,17445,17519,4087,4167,4169,4248,4250,4324,10685,10765,10767,10846,10848,10922,17283,17363,17365,17444,17446,17520,4088,4169,4170,4250,4251,4326,10686,10767,10768,10848,10849,10924,17284,17365,17366,17446,17447,17522,4090,4171,4172,4252,4253,4328,10688,10769,10770,10850,10851,10926,17286,17367,17368,17448,17449,17524,4091,4172,4173,4253,4254,4329,10689,10770,10771,10851,10852,10927,17287,17368,17369,17449,17450,17525,4093,4174,4175,4255,4256,10691,10772,10773,10853,10854,17289,17370,17371,17451,17452,4096,4097,4178,4179,4258,4259,10694,10695,10776,10777,10856,10857,17292,17293,17374,17375,17454,17455,4102,4176,4177,4184,4257,4264,10700,10774,10775,10782,10855,10862,17298,17372,17373,17380,17453,17460,4103,4184,4185,4264,4265,4339,10701,10782,10783,10862,10863,10937,17299,17380,17381,17460,17461,17535,4106,4188,4189,4268,4269,4344,10704,10786,10787,10866,10867,10942,17302,17384,17385,17464,17465,17540,4107,4188,4190,4268,4270,4343,10705,10786,10788,10866,10868,10941,17303,17384,17386,17464,17466,17539,4108,4189,4191,4269,4271,4345,10706,10787,10789,10867,10869,10943,17304,17385,17387,17465,17467,17541,4109,4191,4192,4212,4271,4291,10707,10789,10790,10810,10869,10889,17305,17387,17388,17408,17467,17487,4111,4190,4193,4270,4272,4346,10709,10788,10791,10868,10870,10944,17307,17386,17389,17466,17468,17542,4112,4193,4194,4272,4273,4347,10710,10791,10792,10870,10871,10945,17308,17389,17390,17468,17469,17543,4115,4197,4198,4276,4277,4351,10713,10795,10796,10874,10875,10949,17311,17393,17394,17472,17473,17547,4116,4197,4199,4276,4278,4352,10714,10795,10797,10874,10876,10950,17312,17393,17395,17472,17474,17548,4117,4118,4200,4201,4279,4280,4354,10715,10716,10798,10799,10877,10878,10952,17313,17314,17396,17397,17475,17476,17550,4123,4175,4206,4256,4285,4332,10721,10773,10804,10854,10883,10930,17319,17371,17402,17452,17481,17528,4124,4207,4208,4286,4287,4362,10722,10805,10806,10884,10885,10960,17320,17403,17404,17482,17483,17558,4125,4208,4209,4287,4288,4364,10723,10806,10807,10885,10886,10962,17321,17404,17405,17483,17484,17560,4126,4207,4210,4286,4289,4363,10724,10805,10808,10884,10887,10961,17322,17403,17406,17482,17485,17559,4127,4209,4211,4288,4290,4365,10725,10807,10809,10886,10888,10963,17323,17405,17407,17484,17486,17561,4128,4210,4212,4289,4291,4366,10726,10808,10810,10887,10889,10964,17324,17406,17408,17485,17487,17562,4129,4211,4213,4290,4292,4367,10727,10809,10811,10888,10890,10965,17325,17407,17409,17486,17488,17563,4130,4131,4214,4215,4293,4294,10728,10729,10812,10813,10891,10892,17326,17327,17410,17411,17489,17490,4132,4216,4217,4295,4296,4372,10730,10814,10815,10893,10894,10970,17328,17412,17413,17491,17492,17568,4133,4216,4218,4295,4297,4371,10731,10814,10816,10893,10895,10969,17329,17412,17414,17491,17493,17567,4134,4217,4219,4296,4298,4373,10732,10815,10817,10894,10896,10971,17330,17413,17415,17492,17494,17569,4135,4218,4220,4297,4299,4374,10733,10816,10818,10895,10897,10972,17331,17414,17416,17493,17495,17570,4136,4220,4221,4299,4300,4376,10734,10818,10819,10897,10898,10974,17332,17416,17417,17495,17496,17572,4137,4144,4222,4229,4301,4308,10735,10742,10820,10827,10899,10906,17333,17340,17418,17425,17497,17504,4138,4222,4223,4301,4302,4378,10736,10820,10821,10899,10900,10976,17334,17418,17419,17497,17498,17574,4143,4223,4228,4302,4307,4379,10741,10821,10826,10900,10905,10977,17339,17419,17424,17498,17503,17575,4145,4179,4230,4259,4309,4334,10743,10777,10828,10857,10907,10932,17341,17375,17426,17455,17505,17530,4146,4230,4231,4309,4310,10744,10828,10829,10907,10908,17342,17426,17427,17505,17506,4147,4231,4232,4310,4311,4386,10745,10829,10830,10908,10909,10984,17343,17427,17428,17506,17507,17582,4148,4232,4233,4311,4312,4387,10746,10830,10831,10909,10910,10985,17344,17428,17429,17507,17508,17583,4150,4234,4235,4313,4314,4389,10748,10832,10833,10911,10912,10987,17346,17430,17431,17509,17510,17585,4151,4234,4236,4305,4313,4382,10749,10832,10834,10903,10911,10980,17347,17430,17432,17501,17509,17578,4153,4235,4237,4314,4315,4390,10751,10833,10835,10912,10913,10988,17349,17431,17433,17510,17511,17586,4154,4155,4238,4239,4243,4316,10752,10753,10836,10837,10841,10914,17350,17351,17434,17435,17439,17512,4158,4241,4242,4317,4318,10756,10839,10840,10915,10916,17354,17437,17438,17513,17514,4161,4242,4243,4316,4318,4392,10759,10840,10841,10914,10916,10990,17357,17438,17439,17512,17514,17588,4162,4170,4244,4251,4319,4327,10760,10768,10842,10849,10917,10925,17358,17366,17440,17447,17515,17523,4163,4174,4244,4255,4319,4331,10761,10772,10842,10853,10917,10929,17359,17370,17440,17451,17515,17527,4168,4173,4249,4254,4325,4330,10766,10771,10847,10852,10923,10928,17364,17369,17445,17450,17521,17526,4178,4257,4258,4264,4333,4339,10776,10855,10856,10862,10931,10937,17374,17453,17454,17460,17529,17535,4180,4260,4261,4335,4336,4410,10778,10858,10859,10933,10934,11008,17376,17456,17457,17531,17532,17606,4181,4186,4261,4266,4336,4341,10779,10784,10859,10864,10934,10939,17377,17382,17457,17462,17532,17537,4182,4260,4262,4335,4337,4409,10780,10858,10860,10933,10935,11007,17378,17456,17458,17531,17533,17605,4183,4262,4263,4337,4338,4411,10781,10860,10861,10935,10936,11009,17379,17458,17459,17533,17534,17607,4185,4263,4265,4338,4340,4412,10783,10861,10863,10936,10938,11010,17381,17459,17461,17534,17536,17608,4187,4266,4267,4341,4342,4414,10785,10864,10865,10939,10940,11012,17383,17462,17463,17537,17538,17610,4194,4267,4273,4342,4348,4415,10792,10865,10871,10940,10946,11013,17390,17463,17469,17538,17544,17611,4195,4198,4274,4277,4349,10793,10796,10872,10875,10947,17391,17394,17470,17473,17545,4196,4202,4275,4281,4350,4357,10794,10800,10873,10879,10948,10955,17392,17398,17471,17477,17546,17553,4199,4200,4278,4279,4353,4355,10797,10798,10876,10877,10951,10953,17395,17396,17474,17475,17549,17551,4201,4275,4280,4350,4356,4423,10799,10873,10878,10948,10954,11021,17397,17471,17476,17546,17552,17619,4203,4281,4282,4357,4358,4430,10801,10879,10880,10955,10956,11028,17399,17477,17478,17553,17554,17626,4204,4282,4283,4358,4359,4431,10802,10880,10881,10956,10957,11029,17400,17478,17479,17554,17555,17627,4205,4283,4284,4359,4360,4432,10803,10881,10882,10957,10958,11030,17401,17479,17480,17555,17556,17628,4206,4284,4285,4360,4361,4433,10804,10882,10883,10958,10959,11031,17402,17480,17481,17556,17557,17629,4213,4214,4292,4293,4368,4369,10811,10812,10890,10891,10966,10967,17409,17410,17488,17489,17564,17565,4215,4221,4294,4300,4370,4377,10813,10819,10892,10898,10968,10975,17411,17417,17490,17496,17566,17573,4219,4274,4298,4349,4375,4422,10817,10872,10896,10947,10973,11020,17415,17470,17494,17545,17571,17618,4224,4225,4303,4304,4380,4381,10822,10823,10901,10902,10978,10979,17420,17421,17499,17500,17576,17577,4226,4303,4305,4380,4382,4453,10824,10901,10903,10978,10980,11051,17422,17499,17501,17576,17578,17649,4227,4304,4306,4381,4383,4454,10825,10902,10904,10979,10981,11052,17423,17500,17502,17577,17579,17650,4228,4306,4307,4383,4384,4455,10826,10904,10905,10981,10982,11053,17424,17502,17503,17579,17580,17651,4229,4233,4308,4312,4385,4388,10827,10831,10906,10910,10983,10986,17425,17429,17504,17508,17581,17584,4237,4238,4315,4316,4391,4392,4464,10835,10836,10913,10914,10989,10990,11062,17433,17434,17511,17512,17587,17588,17660,4241,4252,4317,4328,4393,4402,10839,10850,10915,10926,10991,11000,17437,17448,17513,17524,17589,17598,4245,4320,4321,4394,4395,4467,10843,10918,10919,10992,10993,11065,17441,17516,17517,17590,17591,17663,4246,4321,4322,4395,4396,4468,10844,10919,10920,10993,10994,11066,17442,17517,17518,17591,17592,17664,4247,4322,4323,4396,4397,4469,10845,10920,10921,10994,10995,11067,17443,17518,17519,17592,17593,17665,4248,4320,4324,4394,4398,4466,10846,10918,10922,10992,10996,11064,17444,17516,17520,17590,17594,17662,4249,4323,4325,4397,4399,4470,4472,10847,10921,10923,10995,10997,11068,11070,17445,17519,17521,17593,17595,17666,17668,4250,4324,4326,4398,4400,4471,10848,10922,10924,10996,10998,11069,17446,17520,17522,17594,17596,17667,4251,4326,4327,4400,4401,4474,10849,10924,10925,10998,10999,11072,17447,17522,17523,17596,17597,17670,4253,4328,4329,4402,4403,4476,10851,10926,10927,11000,11001,11074,17449,17524,17525,17598,17599,17672,4254,4329,4330,4403,4404,4477,10852,10927,10928,11001,11002,11075,17450,17525,17526,17599,17600,17673,4255,4256,4331,4332,4405,4406,10853,10854,10929,10930,11003,11004,17451,17452,17527,17528,17601,17602,4258,4259,4333,4334,4407,4408,10856,10857,10931,10932,11005,11006,17454,17455,17529,17530,17603,17604,4265,4333,4339,4340,4407,4413,10863,10931,10937,10938,11005,11011,17461,17529,17535,17536,17603,17609,4268,4343,4344,4416,4417,4488,10866,10941,10942,11014,11015,11086,17464,17539,17540,17612,17613,17684,4269,4344,4345,4417,4418,4490,10867,10942,10943,11015,11016,11088,17465,17540,17541,17613,17614,17686,4270,4343,4346,4416,4419,4489,10868,10941,10944,11014,11017,11087,17466,17539,17542,17612,17615,17685,4271,4291,4345,4366,4418,4439,10869,10889,10943,10964,11016,11037,17467,17487,17541,17562,17614,17635,4272,4346,4347,4419,4420,4491,10870,10944,10945,11017,11018,11089,17468,17542,17543,17615,17616,17687,4273,4347,4348,4420,4421,4492,10871,10945,10946,11018,11019,11090,17469,17543,17544,17616,17617,17688,4276,4351,4352,4424,4425,4496,10874,10949,10950,11022,11023,11094,17472,17547,17548,17620,17621,17692,4277,4349,4351,4422,4424,4494,10875,10947,10949,11020,11022,11092,17473,17545,17547,17618,17620,17690,4278,4352,4353,4425,4426,4497,10876,10950,10951,11023,11024,11095,17474,17548,17549,17621,17622,17693,4279,4354,4355,4427,4428,4499,10877,10952,10953,11025,11026,11097,17475,17550,17551,17623,17624,17695,4280,4354,4356,4427,4429,4500,10878,10952,10954,11025,11027,11098,17476,17550,17552,17623,17625,17696,4285,4332,4361,4406,4434,4479,10883,10930,10959,11004,11032,11077,17481,17528,17557,17602,17630,17675,4286,4362,4363,4435,4436,4509,10884,10960,10961,11033,11034,11107,17482,17558,17559,17631,17632,17705,4287,4362,4364,4435,4437,4508,10885,10960,10962,11033,11035,11106,17483,17558,17560,17631,17633,17704,4288,4364,4365,4437,4438,4511,10886,10962,10963,11035,11036,11109,17484,17560,17561,17633,17634,17707,4289,4363,4366,4436,4439,4510,10887,10961,10964,11034,11037,11108,17485,17559,17562,17632,17635,17706,4290,4365,4367,4438,4440,4512,10888,10963,10965,11036,11038,11110,17486,17561,17563,17634,17636,17708,4292,4367,4368,4440,4441,4513,10890,10965,10966,11038,11039,11111,17488,17563,17564,17636,17637,17709,4293,4294,4369,4370,4442,4443,10891,10892,10967,10968,11040,11041,17489,17490,17565,17566,17638,17639,4295,4371,4372,4444,4445,4518,10893,10969,10970,11042,11043,11116,17491,17567,17568,17640,17641,17714,4296,4372,4373,4445,4446,4519,10894,10970,10971,11043,11044,11117,17492,17568,17569,17641,17642,17715,4297,4371,4374,4444,4447,4517,4522,10895,10969,10972,11042,11045,11115,11120,17493,17567,17570,17640,17643,17713,17718,4298,4373,4375,4446,4448,4520,10896,10971,10973,11044,11046,11118,17494,17569,17571,17642,17644,17716,4299,4374,4376,4447,4449,4521,10897,10972,10974,11045,11047,11119,17495,17570,17572,17643,17645,17717,4300,4376,4377,4449,4450,4524,10898,10974,10975,11047,11048,11122,17496,17572,17573,17645,17646,17720,4301,4308,4378,4385,4451,4457,10899,10906,10976,10983,11049,11055,17497,17504,17574,17581,17647,17653,4302,4378,4379,4451,4452,4526,10900,10976,10977,11049,11050,11124,17498,17574,17575,17647,17648,17722,4307,4379,4384,4452,4456,4527,10905,10977,10982,11050,11054,11125,17503,17575,17580,17648,17652,17723,4309,4310,4334,4386,4408,4458,4480,10907,10908,10932,10984,11006,11056,11078,17505,17506,17530,17582,17604,17654,17676,4311,4386,4387,4458,4459,4533,10909,10984,10985,11056,11057,11131,17507,17582,17583,17654,17655,17729,4312,4387,4388,4459,4460,4534,10910,10985,10986,11057,11058,11132,17508,17583,17584,17655,17656,17730,4313,4382,4389,4453,4461,4528,10911,10980,10987,11051,11059,11126,17509,17578,17585,17649,17657,17724,4314,4389,4390,4461,4462,4536,10912,10987,10988,11059,11060,11134,17510,17585,17586,17657,17658,17732,4315,4390,4391,4462,4463,4537,10913,10988,10989,11060,11061,11135,17511,17586,17587,17658,17659,17733,4317,4318,4392,4393,4464,4465,10915,10916,10990,10991,11062,11063,17513,17514,17588,17589,17660,17661,4319,4327,4331,4401,4405,4475,10917,10925,10929,10999,11003,11073,17515,17523,17527,17597,17601,17671,4325,4330,4399,4404,4473,4478,10923,10928,10997,11002,11071,11076,17521,17526,17595,17600,17669,17674,4335,4409,4410,4481,4482,4556,10933,11007,11008,11079,11080,11154,17531,17605,17606,17677,17678,17752,4336,4341,4410,4414,4482,4486,10934,10939,11008,11012,11080,11084,17532,17537,17606,17610,17678,17682,4337,4409,4411,4481,4483,4555,10935,11007,11009,11079,11081,11153,17533,17605,17607,17677,17679,17751,4338,4411,4412,4483,4484,4557,4558,10936,11009,11010,11081,11082,11155,11156,17534,17607,17608,17679,17680,17753,17754,4340,4412,4413,4484,4485,10938,11010,11011,11082,11083,17536,17608,17609,17680,17681,4342,4414,4415,4486,4487,4559,10940,11012,11013,11084,11085,11157,17538,17610,17611,17682,17683,17755,4348,4415,4421,4487,4493,4560,10946,11013,11019,11085,11091,11158,17544,17611,17617,17683,17689,17756,4350,4357,4423,4430,4495,4503,10948,10955,11021,11028,11093,11101,17546,17553,17619,17626,17691,17699,4353,4355,4426,4428,4498,4501,10951,10953,11024,11026,11096,11099,17549,17551,17622,17624,17694,17697,4356,4423,4429,4495,4502,4568,4575,4642,10954,11021,11027,11093,11100,11166,11173,11240,17552,17619,17625,17691,17698,17764,17771,17838,4358,4430,4431,4503,4504,4577,10956,11028,11029,11101,11102,11175,17554,17626,17627,17699,17700,17773,4359,4431,4432,4504,4505,4578,10957,11029,11030,11102,11103,11176,17555,17627,17628,17700,17701,17774,4360,4432,4433,4505,4506,4579,10958,11030,11031,11103,11104,11177,17556,17628,17629,17701,17702,17775,4361,4433,4434,4506,4507,4580,10959,11031,11032,11104,11105,11178,17557,17629,17630,17702,17703,17776,4368,4369,4441,4442,4514,4515,10966,10967,11039,11040,11112,11113,17564,17565,17637,17638,17710,17711,4370,4377,4443,4450,4516,4525,10968,10975,11041,11048,11114,11123,17566,17573,17639,17646,17712,17721,4375,4422,4448,4494,4523,4567,4598,10973,11020,11046,11092,11121,11165,11196,17571,17618,17644,17690,17719,17763,17794,4380,4381,4453,4454,4528,4529,10978,10979,11051,11052,11126,11127,17576,17577,17649,17650,17724,17725,4383,4454,4455,4529,4530,4603,10981,11052,11053,11127,11128,11201,17579,17650,17651,17725,17726,17799,4384,4455,4456,4530,4531,4604,10982,11053,11054,11128,11129,11202,17580,17651,17652,17726,17727,17800,4385,4388,4457,4460,4532,4535,10983,10986,11055,11058,11130,11133,17581,17584,17653,17656,17728,17731,4391,4463,4464,4465,4538,4539,10989,11061,11062,11063,11136,11137,17587,17659,17660,17661,17734,17735,4393,4402,4465,4476,4539,4550,10991,11000,11063,11074,11137,11148,17589,17598,17661,17672,17735,17746,4394,4466,4467,4540,4541,4614,10992,11064,11065,11138,11139,11212,17590,17662,17663,17736,17737,17810,4395,4467,4468,4541,4542,4615,10993,11065,11066,11139,11140,11213,17591,17663,17664,17737,17738,17811,4396,4468,4469,4542,4543,4616,10994,11066,11067,11140,11141,11214,17592,17664,17665,17738,17739,17812,4397,4469,4470,4543,4544,10995,11067,11068,11141,11142,17593,17665,17666,17739,17740,4398,4466,4471,4540,4545,4613,10996,11064,11069,11138,11143,11211,17594,17662,17667,17736,17741,17809,4399,4472,4473,4546,4547,4619,10997,11070,11071,11144,11145,11217,17595,17668,17669,17742,17743,17815,4400,4471,4474,4545,4548,4618,10998,11069,11072,11143,11146,11216,17596,17667,17670,17741,17744,17814,4401,4474,4475,4548,4549,4622,10999,11072,11073,11146,11147,11220,17597,17670,17671,17744,17745,17818,4403,4476,4477,4550,4551,4623,11001,11074,11075,11148,11149,11221,17599,17672,17673,17746,17747,17819,4404,4477,4478,4551,4552,4624,4625,11002,11075,11076,11149,11150,11222,11223,17600,17673,17674,17747,17748,17820,17821,4405,4406,4475,4479,4549,4553,11003,11004,11073,11077,11147,11151,17601,17602,17671,17675,17745,17749,4407,4408,4413,4480,4485,4554,11005,11006,11011,11078,11083,11152,17603,17604,17609,17676,17681,17750,4416,4488,4489,4561,4562,4634,11014,11086,11087,11159,11160,11232,17612,17684,17685,17757,17758,17830,4417,4488,4490,4561,4563,4635,11015,11086,11088,11159,11161,11233,17613,17684,17686,17757,17759,17831,4418,4439,4490,4510,4563,4584,11016,11037,11088,11108,11161,11182,17614,17635,17686,17706,17759,17780,4419,4489,4491,4562,4564,4636,11017,11087,11089,11160,11162,11234,17615,17685,17687,17758,17760,17832,4420,4491,4492,4564,4565,4637,11018,11089,11090,11162,11163,11235,17616,17687,17688,17760,17761,17833,4421,4492,4493,4565,4566,4638,11019,11090,11091,11163,11164,11236,17617,17688,17689,17761,17762,17834,4424,4494,4496,4567,4569,4640,11022,11092,11094,11165,11167,11238,17620,17690,17692,17763,17765,17836,4425,4496,4497,4569,4570,4643,11023,11094,11095,11167,11168,11241,17621,17692,17693,17765,17766,17839,4426,4497,4498,4570,4571,4644,11024,11095,11096,11168,11169,11242,17622,17693,17694,17766,17767,17840,4427,4499,4500,4572,4573,4646,11025,11097,11098,11170,11171,11244,17623,17695,17696,17768,17769,17842,4428,4499,4501,4572,4574,4647,11026,11097,11099,11170,11172,11245,17624,17695,17697,17768,17770,17843,4429,4500,4502,4573,4576,11027,11098,11100,11171,11174,17625,17696,17698,17769,17772,4434,4479,4507,4553,4581,4626,11032,11077,11105,11151,11179,11224,17630,17675,17703,17749,17777,17822,4435,4508,4509,4582,4583,4657,11033,11106,11107,11180,11181,11255,17631,17704,17705,17778,17779,17853,4436,4509,4510,4583,4584,4658,11034,11107,11108,11181,11182,11256,17632,17705,17706,17779,17780,17854,4437,4508,4511,4582,4585,4656,11035,11106,11109,11180,11183,11254,17633,17704,17707,17778,17781,17852,4438,4511,4512,4585,4586,4659,11036,11109,11110,11183,11184,11257,17634,17707,17708,17781,17782,17855,4440,4512,4513,4586,4587,4660,11038,11110,11111,11184,11185,11258,17636,17708,17709,17782,17783,17856,4441,4513,4514,4587,4588,4661,11039,11111,11112,11185,11186,11259,17637,17709,17710,17783,17784,17857,4442,4443,4515,4516,4589,4590,11040,11041,11113,11114,11187,11188,17638,17639,17711,17712,17785,17786,4444,4517,4518,4591,4592,4666,11042,11115,11116,11189,11190,11264,17640,17713,17714,17787,17788,17862,4445,4518,4519,4592,4593,4667,11043,11116,11117,11190,11191,11265,17641,17714,17715,17788,17789,17863,4446,4519,4520,4593,4594,4668,11044,11117,11118,11191,11192,11266,17642,17715,17716,17789,17790,17864,4447,4521,4522,4595,4596,4671,11045,11119,11120,11193,11194,11269,17643,17717,17718,17791,17792,17867,4448,4520,4523,4594,4597,4669,11046,11118,11121,11192,11195,11267,17644,17716,17719,17790,17793,17865,4449,4521,4524,4595,4599,4670,11047,11119,11122,11193,11197,11268,17645,17717,17720,17791,17795,17866,4450,4524,4525,4599,4600,4674,11048,11122,11123,11197,11198,11272,17646,17720,17721,17795,17796,17870,4451,4457,4526,4532,4601,4606,11049,11055,11124,11130,11199,11204,17647,17653,17722,17728,17797,17802,4452,4526,4527,4601,4602,4676,11050,11124,11125,11199,11200,11274,17648,17722,17723,17797,17798,17872,4456,4527,4531,4602,4605,4677,11054,11125,11129,11200,11203,11275,17652,17723,17727,17798,17801,17873,4458,4480,4533,4554,4607,4627,11056,11078,11131,11152,11205,11225,17654,17676,17729,17750,17803,17823,4459,4533,4534,4607,4608,4681,11057,11131,11132,11205,11206,11279,17655,17729,17730,17803,17804,17877,4460,4534,4535,4608,4609,4682,11058,11132,11133,11206,11207,11280,17656,17730,17731,17804,17805,17878,4461,4528,4529,4536,4603,4610,11059,11126,11127,11134,11201,11208,17657,17724,17725,17732,17799,17806,4462,4536,4537,4610,4611,4684,11060,11134,11135,11208,11209,11282,17658,17732,17733,17806,17807,17880,4463,4537,4538,4611,4612,4685,11061,11135,11136,11209,11210,11283,17659,17733,17734,17807,17808,17881,4470,4472,4544,4546,4617,4620,11068,11070,11142,11144,11215,11218,17666,17668,17740,17742,17813,17816,4473,4478,4547,4552,4621,11071,11076,11145,11150,11219,17669,17674,17743,17748,17817,4481,4555,4556,4628,4629,4700,11079,11153,11154,11226,11227,11298,17677,17751,17752,17824,17825,17896,4482,4486,4556,4559,4629,4632,11080,11084,11154,11157,11227,11230,17678,17682,17752,17755,17825,17828,4483,4555,4557,4628,4630,4701,11081,11153,11155,11226,11228,11299,17679,17751,17753,17824,17826,17897,4484,4485,4554,4558,4627,4631,11082,11083,11152,11156,11225,11229,17680,17681,17750,17754,17823,17827,4487,4559,4560,4632,4633,4704,11085,11157,11158,11230,11231,11302,17683,17755,17756,17828,17829,17900,4493,4560,4566,4633,4639,4705,11091,11158,11164,11231,11237,11303,17689,17756,17762,17829,17835,17901,4495,4503,4568,4577,4641,4651,11093,11101,11166,11175,11239,11249,17691,17699,17764,17773,17837,17847,4498,4501,4571,4574,4645,4648,11096,11099,11169,11172,11243,11246,17694,17697,17767,17770,17841,17844,4502,4575,4576,4649,4650,4722,11100,11173,11174,11247,11248,11320,17698,17771,17772,17845,17846,17918,4504,4577,4578,4651,4652,4724,11102,11175,11176,11249,11250,11322,17700,17773,17774,17847,17848,17920,4505,4578,4579,4652,4653,4725,11103,11176,11177,11250,11251,11323,17701,17774,17775,17848,17849,17921,4506,4579,4580,4653,4654,4726,11104,11177,11178,11251,11252,11324,17702,17775,17776,17849,17850,17922,4507,4580,4581,4654,4655,4727,11105,11178,11179,11252,11253,11325,17703,17776,17777,17850,17851,17923,4514,4515,4588,4589,4662,4663,11112,11113,11186,11187,11260,11261,17710,17711,17784,17785,17858,17859,4516,4525,4590,4600,4664,4675,11114,11123,11188,11198,11262,11273,17712,17721,17786,17796,17860,17871,4517,4522,4591,4596,4665,11115,11120,11189,11194,11263,17713,17718,17787,17792,17861,4523,4597,4598,4672,4673,11121,11195,11196,11270,11271,17719,17793,17794,17868,17869,4530,4603,4604,4610,4678,4684,11128,11201,11202,11208,11276,11282,17726,17799,17800,17806,17874,17880,4531,4604,4605,4678,4679,4751,11129,11202,11203,11276,11277,11349,17727,17800,17801,17874,17875,17947,4532,4535,4606,4609,4680,4683,11130,11133,11204,11207,11278,11281,17728,17731,17802,17805,17876,17879,4538,4539,4550,4612,4623,4686,11136,11137,11148,11210,11221,11284,17734,17735,17746,17808,17819,17882,4540,4613,4614,4687,4688,4760,11138,11211,11212,11285,11286,11358,17736,17809,17810,17883,17884,17956,4541,4614,4615,4688,4689,4761,11139,11212,11213,11286,11287,11359,17737,17810,17811,17884,17885,17957,4542,4615,4616,4689,4690,4762,11140,11213,11214,11287,11288,11360,17738,17811,17812,17885,17886,17958,4543,4544,4616,4617,4690,4691,11141,11142,11214,11215,11288,11289,17739,17740,17812,17813,17886,17887,4545,4613,4618,4687,4692,4759,11143,11211,11216,11285,11290,11357,17741,17809,17814,17883,17888,17955,4546,4619,4620,4693,4694,4766,11144,11217,11218,11291,11292,11364,17742,17815,17816,17889,17890,17962,4547,4619,4621,4693,4695,4765,11145,11217,11219,11291,11293,11363,17743,17815,17817,17889,17891,17961,4548,4618,4622,4692,4696,4764,11146,11216,11220,11290,11294,11362,17744,17814,17818,17888,17892,17960,4549,4553,4622,4626,4696,4699,11147,11151,11220,11224,11294,11297,17745,17749,17818,17822,17892,17895,4551,4623,4624,4686,4697,4758,11149,11221,11222,11284,11295,11356,17747,17819,17820,17882,17893,17954,4552,4621,4625,4695,4698,4768,11150,11219,11223,11293,11296,11366,17748,17817,17821,17891,17894,17964,4557,4558,4630,4631,4702,4703,11155,11156,11228,11229,11300,11301,17753,17754,17826,17827,17898,17899,4561,4634,4635,4706,4707,4778,11159,11232,11233,11304,11305,11376,17757,17830,17831,17902,17903,17974,4562,4634,4636,4706,4708,4777,11160,11232,11234,11304,11306,11375,17758,17830,17832,17902,17904,17973,4563,4584,4635,4658,4707,4731,11161,11182,11233,11256,11305,11329,17759,17780,17831,17854,17903,17927,4564,4636,4637,4708,4709,4779,11162,11234,11235,11306,11307,11377,17760,17832,17833,17904,17905,17975,4565,4637,4638,4709,4710,4780,11163,11235,11236,11307,11308,11378,17761,17833,17834,17905,17906,17976,4566,4638,4639,4710,4711,4781,11164,11236,11237,11308,11309,11379,17762,17834,17835,17906,17907,17977,4567,4598,4640,4673,4712,4746,11165,11196,11238,11271,11310,11344,17763,17794,17836,17869,17908,17942,4568,4641,4642,4713,4714,11166,11239,11240,11311,11312,17764,17837,17838,17909,17910,4569,4640,4643,4712,4715,4783,11167,11238,11241,11310,11313,11381,17765,17836,17839,17908,17911,17979,4570,4643,4644,4715,4716,11168,11241,11242,11313,11314,17766,17839,17840,17911,17912,4571,4644,4645,4716,4717,4786,11169,11242,11243,11314,11315,11384,17767,17840,17841,17912,17913,17982,4572,4646,4647,4718,4719,4788,11170,11244,11245,11316,11317,11386,17768,17842,17843,17914,17915,17984,4573,4576,4646,4650,4718,4723,11171,11174,11244,11248,11316,11321,17769,17772,17842,17846,17914,17919,4574,4647,4648,4719,4720,4789,11172,11245,11246,11317,11318,11387,17770,17843,17844,17915,17916,17985,4575,4642,4649,4714,4721,4785,11173,11240,11247,11312,11319,11383,17771,17838,17845,17910,17917,17981,4581,4626,4655,4699,4728,4771,11179,11224,11253,11297,11326,11369,17777,17822,17851,17895,17924,17967,4582,4656,4657,4729,4730,4799,11180,11254,11255,11327,11328,11397,17778,17852,17853,17925,17926,17995,4583,4657,4658,4730,4731,4801,4802,11181,11255,11256,11328,11329,11399,11400,17779,17853,17854,17926,17927,17997,17998,4585,4656,4659,4729,4732,4800,11183,11254,11257,11327,11330,11398,17781,17852,17855,17925,17928,17996,4586,4659,4660,4732,4733,4803,11184,11257,11258,11330,11331,11401,17782,17855,17856,17928,17929,17999,4587,4660,4661,4733,4734,4804,11185,11258,11259,11331,11332,11402,17783,17856,17857,17929,17930,18000,4588,4661,4662,4734,4735,4805,11186,11259,11260,11332,11333,11403,17784,17857,17858,17930,17931,18001,4589,4590,4663,4664,4736,4737,11187,11188,11261,11262,11334,11335,17785,17786,17859,17860,17932,17933,4591,4665,4666,4738,4739,4810,11189,11263,11264,11336,11337,11408,17787,17861,17862,17934,17935,18006,4592,4666,4667,4739,4740,4811,11190,11264,11265,11337,11338,11409,17788,17862,17863,17935,17936,18007,4593,4667,4668,4740,4741,4812,11191,11265,11266,11338,11339,11410,17789,17863,17864,17936,17937,18008,4594,4668,4669,4741,4742,4813,11192,11266,11267,11339,11340,11411,17790,17864,17865,17937,17938,18009,4595,4670,4671,4743,4744,4815,11193,11268,11269,11341,11342,11413,17791,17866,17867,17939,17940,18011,4596,4665,4671,4738,4744,4809,11194,11263,11269,11336,11342,11407,17792,17861,17867,17934,17940,18005,4597,4669,4672,4742,4745,4814,11195,11267,11270,11340,11343,11412,17793,17865,17868,17938,17941,18010,4599,4670,4674,4743,4747,4816,11197,11268,11272,11341,11345,11414,17795,17866,17870,17939,17943,18012,4600,4674,4675,4747,4748,4819,11198,11272,11273,11345,11346,11417,17796,17870,17871,17943,17944,18015,4601,4606,4676,4680,4749,4753,11199,11204,11274,11278,11347,11351,17797,17802,17872,17876,17945,17949,4602,4676,4677,4749,4750,4821,11200,11274,11275,11347,11348,11419,17798,17872,17873,17945,17946,18017,4605,4677,4679,4750,4752,4822,11203,11275,11277,11348,11350,11420,17801,17873,17875,17946,17948,18018,4607,4627,4631,4681,4703,4754,11205,11225,11229,11279,11301,11352,17803,17823,17827,17877,17899,17950,4608,4681,4682,4754,4755,4825,11206,11279,11280,11352,11353,11423,17804,17877,17878,17950,17951,18021,4609,4682,4683,4755,4756,4826,11207,11280,11281,11353,11354,11424,17805,17878,17879,17951,17952,18022,4611,4678,4684,4685,4751,4757,11209,11276,11282,11283,11349,11355,17807,17874,17880,17881,17947,17953,4612,4685,4686,4757,4758,4828,11210,11283,11284,11355,11356,11426,17808,17881,17882,17953,17954,18024,4617,4620,4691,4694,4763,4767,11215,11218,11289,11292,11361,11365,17813,17816,17887,17890,17959,17963,4624,4625,4697,4698,4769,4770,11222,11223,11295,11296,11367,11368,17820,17821,17893,17894,17965,17966,4628,4700,4701,4772,4773,4842,11226,11298,11299,11370,11371,11440,17824,17896,17897,17968,17969,18038,4629,4632,4700,4704,4772,4775,11227,11230,11298,11302,11370,11373,17825,17828,17896,17900,17968,17971,4630,4701,4702,4773,4774,4843,11228,11299,11300,11371,11372,11441,17826,17897,17898,17969,17970,18039,4633,4704,4705,4775,4776,4845,11231,11302,11303,11373,11374,11443,17829,17900,17901,17971,17972,18041,4639,4705,4711,4776,4782,4846,11237,11303,11309,11374,11380,11444,17835,17901,17907,17972,17978,18042,4641,4651,4713,4724,4784,4794,11239,11249,11311,11322,11382,11392,17837,17847,17909,17920,17980,17990,4645,4648,4717,4720,4787,4790,11243,11246,11315,11318,11385,11388,17841,17844,17913,17916,17983,17986,4649,4721,4722,4791,4792,4861,11247,11319,11320,11389,11390,11459,17845,17917,17918,17987,17988,18057,4650,4722,4723,4792,4793,4863,11248,11320,11321,11390,11391,11461,17846,17918,17919,17988,17989,18059,4652,4724,4725,4794,4795,4865,11250,11322,11323,11392,11393,11463,17848,17920,17921,17990,17991,18061,4653,4725,4726,4795,4796,4866,11251,11323,11324,11393,11394,11464,17849,17921,17922,17991,17992,18062,4654,4726,4727,4796,4797,4867,11252,11324,11325,11394,11395,11465,17850,17922,17923,17992,17993,18063,4655,4727,4728,4797,4798,4868,11253,11325,11326,11395,11396,11466,17851,17923,17924,17993,17994,18064,4662,4663,4735,4736,4806,4807,11260,11261,11333,11334,11404,11405,17858,17859,17931,17932,18002,18003,4664,4675,4737,4748,4808,4820,4880,11262,11273,11335,11346,11406,11418,11478,17860,17871,17933,17944,18004,18016,18076,4672,4673,4745,4746,4817,4818,11270,11271,11343,11344,11415,11416,17868,17869,17941,17942,18013,18014,4679,4751,4752,4757,4823,4828,4896,11277,11349,11350,11355,11421,11426,11494,17875,17947,17948,17953,18019,18024,18092,4680,4683,4753,4756,4824,4827,11278,11281,11351,11354,11422,11425,17876,17879,17949,17952,18020,18023,4687,4759,4760,4829,4830,4901,11285,11357,11358,11427,11428,11499,17883,17955,17956,18025,18026,18097,4688,4760,4761,4830,4831,4902,11286,11358,11359,11428,11429,11500,17884,17956,17957,18026,18027,18098,4689,4761,4762,4831,4832,4903,11287,11359,11360,11429,11430,11501,17885,17957,17958,18027,18028,18099,4690,4691,4762,4763,4832,4833,11288,11289,11360,11361,11430,11431,17886,17887,17958,17959,18028,18029,4692,4759,4764,4829,4834,4900,11290,11357,11362,11427,11432,11498,17888,17955,17960,18025,18030,18096,4693,4765,4766,4835,4836,4905,11291,11363,11364,11433,11434,11503,17889,17961,17962,18031,18032,18101,4694,4766,4767,4836,4837,4907,11292,11364,11365,11434,11435,11505,17890,17962,17963,18032,18033,18103,4695,4765,4768,4835,4838,4906,11293,11363,11366,11433,11436,11504,17891,17961,17964,18031,18034,18102,4696,4699,4764,4771,4834,4841,11294,11297,11362,11369,11432,11439,17892,17895,17960,17967,18030,18037,4697,4758,4769,4828,4839,4896,11295,11356,11367,11426,11437,11494,17893,17954,17965,18024,18035,18092,4698,4768,4770,4838,4840,4909,11296,11366,11368,11436,11438,11507,17894,17964,17966,18034,18036,18105,4702,4703,4754,4774,4825,4844,11300,11301,11352,11372,11423,11442,17898,17899,17950,17970,18021,18040,4706,4777,4778,4847,4848,4919,11304,11375,11376,11445,11446,11517,17902,17973,17974,18043,18044,18115,4707,4731,4778,4802,4848,4873,11305,11329,11376,11400,11446,11471,17903,17927,17974,17998,18044,18069,4708,4777,4779,4847,4849,4918,11306,11375,11377,11445,11447,11516,17904,17973,17975,18043,18045,18114,4709,4779,4780,4849,4850,4920,11307,11377,11378,11447,11448,11518,17905,17975,17976,18045,18046,18116,4710,4780,4781,4850,4851,4921,11308,11378,11379,11448,11449,11519,17906,17976,17977,18046,18047,18117,4711,4781,4782,4851,4852,4922,11309,11379,11380,11449,11450,11520,17907,17977,17978,18047,18048,18118,4712,4746,4783,4818,4853,4890,11310,11344,11381,11416,11451,11488,17908,17942,17979,18014,18049,18086,4713,4714,4784,4785,4854,4855,11311,11312,11382,11383,11452,11453,17909,17910,17980,17981,18050,18051,4715,4716,4783,4786,4853,4856,11313,11314,11381,11384,11451,11454,17911,17912,17979,17982,18049,18052,4717,4786,4787,4856,4857,4926,11315,11384,11385,11454,11455,11524,17913,17982,17983,18052,18053,18122,4718,4723,4788,4793,4858,4864,11316,11321,11386,11391,11456,11462,17914,17919,17984,17989,18054,18060,4719,4788,4789,4858,4859,4928,11317,11386,11387,11456,11457,11526,17915,17984,17985,18054,18055,18124,4720,4789,4790,4859,4860,4929,11318,11387,11388,11457,11458,11527,17916,17985,17986,18055,18056,18125,4721,4785,4791,4855,4862,4925,11319,11383,11389,11453,11460,11523,17917,17981,17987,18051,18058,18121,4728,4771,4798,4841,4869,4912,11326,11369,11396,11439,11467,11510,17924,17967,17994,18037,18065,18108,4729,4799,4800,4870,4871,4941,11327,11397,11398,11468,11469,11539,17925,17995,17996,18066,18067,18137,4730,4799,4801,4870,4872,4940,11328,11397,11399,11468,11470,11538,17926,17995,17997,18066,18068,18136,4732,4800,4803,4871,4874,4942,11330,11398,11401,11469,11472,11540,17928,17996,17999,18067,18070,18138,4733,4803,4804,4874,4875,4945,4946,11331,11401,11402,11472,11473,11543,11544,17929,17999,18000,18070,18071,18141,18142,4734,4804,4805,4875,4876,4947,11332,11402,11403,11473,11474,11545,17930,18000,18001,18071,18072,18143,4735,4805,4806,4876,4877,4948,4949,11333,11403,11404,11474,11475,11546,11547,17931,18001,18002,18072,18073,18144,18145,4736,4737,4807,4808,4878,4879,11334,11335,11405,11406,11476,11477,17932,17933,18003,18004,18074,18075,4738,4809,4810,4881,4882,4955,11336,11407,11408,11479,11480,11553,17934,18005,18006,18077,18078,18151,4739,4810,4811,4882,4883,4956,11337,11408,11409,11480,11481,11554,17935,18006,18007,18078,18079,18152,4740,4811,4812,4883,4884,4957,11338,11409,11410,11481,11482,11555,17936,18007,18008,18079,18080,18153,4741,4812,4813,4884,4885,4958,11339,11410,11411,11482,11483,11556,17937,18008,18009,18080,18081,18154,4742,4813,4814,4885,4886,4959,11340,11411,11412,11483,11484,11557,17938,18009,18010,18081,18082,18155,4743,4815,4816,4887,4888,4961,11341,11413,11414,11485,11486,11559,17939,18011,18012,18083,18084,18157,4744,4809,4815,4881,4887,4954,11342,11407,11413,11479,11485,11552,17940,18005,18011,18077,18083,18150,4745,4814,4817,4886,4889,4960,11343,11412,11415,11484,11487,11558,17941,18010,18013,18082,18085,18156,4747,4816,4819,4888,4891,4962,11345,11414,11417,11486,11489,11560,17943,18012,18015,18084,18087,18158,4748,4819,4820,4891,4892,11346,11417,11418,11489,11490,17944,18015,18016,18087,18088,4749,4753,4821,4824,4893,4897,11347,11351,11419,11422,11491,11495,17945,17949,18017,18020,18089,18093,4750,4821,4822,4893,4894,4966,11348,11419,11420,11491,11492,11564,17946,18017,18018,18089,18090,18162,4752,4822,4823,4894,4895,4967,11350,11420,11421,11492,11493,11565,17948,18018,18019,18090,18091,18163,4755,4825,4826,4844,4898,4915,11353,11423,11424,11442,11496,11513,17951,18021,18022,18040,18094,18111,4756,4826,4827,4898,4899,4969,11354,11424,11425,11496,11497,11567,17952,18022,18023,18094,18095,18165,4763,4767,4833,4837,4904,4908,11361,11365,11431,11435,11502,11506,17959,17963,18029,18033,18100,18104,4769,4770,4839,4840,4910,4911,11367,11368,11437,11438,11508,11509,17965,17966,18035,18036,18106,18107,4772,4775,4842,4845,4913,4916,11370,11373,11440,11443,11511,11514,17968,17971,18038,18041,18109,18112,4773,4842,4843,4913,4914,4983,11371,11440,11441,11511,11512,11581,17969,18038,18039,18109,18110,18179,4774,4843,4844,4914,4915,4984,11372,11441,11442,11512,11513,11582,17970,18039,18040,18110,18111,18180,4776,4845,4846,4916,4917,4985,11374,11443,11444,11514,11515,11583,17972,18041,18042,18112,18113,18181,4782,4846,4852,4917,4923,4986,11380,11444,11450,11515,11521,11584,17978,18042,18048,18113,18119,18182,4784,4794,4854,4865,4924,4935,11382,11392,11452,11463,11522,11533,17980,17990,18050,18061,18120,18131,4787,4790,4857,4860,4927,4930,11385,11388,11455,11458,11525,11528,17983,17986,18053,18056,18123,18126,4791,4861,4862,4931,4932,5001,11389,11459,11460,11529,11530,11599,17987,18057,18058,18127,18128,18197,4792,4861,4863,4931,4933,5000,11390,11459,11461,11529,11531,11598,17988,18057,18059,18127,18129,18196,4793,4863,4864,4933,4934,5003,11391,11461,11462,11531,11532,11601,17989,18059,18060,18129,18130,18199,4795,4865,4866,4935,4936,5005,5006,11393,11463,11464,11533,11534,11603,11604,17991,18061,18062,18131,18132,18201,18202,4796,4866,4867,4936,4937,11394,11464,11465,11534,11535,17992,18062,18063,18132,18133,4797,4867,4868,4937,4938,5007,11395,11465,11466,11535,11536,11605,17993,18063,18064,18133,18134,18203,4798,4868,4869,4938,4939,5008,11396,11466,11467,11536,11537,11606,17994,18064,18065,18134,18135,18204,4801,4802,4872,4873,4943,4944,11399,11400,11470,11471,11541,11542,17997,17998,18068,18069,18139,18140,4806,4807,4877,4878,4950,4951,11404,11405,11475,11476,11548,11549,18002,18003,18073,18074,18146,18147,4808,4879,4880,4952,4953,5023,11406,11477,11478,11550,11551,11621,18004,18075,18076,18148,18149,18219,4817,4818,4889,4890,4963,4964,11415,11416,11487,11488,11561,11562,18013,18014,18085,18086,18159,18160,4820,4880,4892,4953,4965,5024,11418,11478,11490,11551,11563,11622,18016,18076,18088,18149,18161,18220,4823,4839,4895,4896,4910,11421,11437,11493,11494,11508,18019,18035,18091,18092,18106,4824,4827,4897,4899,4968,4970,11422,11425,11495,11497,11566,11568,18020,18023,18093,18095,18164,18166,4829,4900,4901,4971,4972,5041,11427,11498,11499,11569,11570,11639,18025,18096,18097,18167,18168,18237,4830,4901,4902,4972,4973,5042,11428,11499,11500,11570,11571,11640,18026,18097,18098,18168,18169,18238,4831,4902,4903,4973,4974,5043,11429,11500,11501,11571,11572,11641,18027,18098,18099,18169,18170,18239,4832,4833,4903,4904,4974,4975,11430,11431,11501,11502,11572,11573,18028,18029,18099,18100,18170,18171,4834,4841,4900,4912,4971,4982,11432,11439,11498,11510,11569,11580,18030,18037,18096,18108,18167,18178,4835,4905,4906,4976,4977,5045,11433,11503,11504,11574,11575,11643,18031,18101,18102,18172,18173,18241,4836,4905,4907,4976,4978,5046,11434,11503,11505,11574,11576,11644,18032,18101,18103,18172,18174,18242,4837,4907,4908,4978,4979,5048,11435,11505,11506,11576,11577,11646,18033,18103,18104,18174,18175,18244,4838,4906,4909,4977,4980,5047,11436,11504,11507,11575,11578,11645,18034,18102,18105,18173,18176,18243,4840,4909,4911,4980,4981,5050,11438,11507,11509,11578,11579,11648,18036,18105,18107,18176,18177,18246,4847,4918,4919,4987,4988,5057,11445,11516,11517,11585,11586,11655,18043,18114,18115,18183,18184,18253,4848,4873,4919,4944,4988,5014,11446,11471,11517,11542,11586,11612,18044,18069,18115,18140,18184,18210,4849,4918,4920,4987,4989,5056,11447,11516,11518,11585,11587,11654,18045,18114,18116,18183,18185,18252,4850,4920,4921,4989,4990,5058,11448,11518,11519,11587,11588,11656,18046,18116,18117,18185,18186,18254,4851,4921,4922,4990,4991,5059,11449,11519,11520,11588,11589,11657,18047,18117,18118,18186,18187,18255,4852,4922,4923,4991,4992,5060,11450,11520,11521,11589,11590,11658,18048,18118,18119,18187,18188,18256,4853,4856,4890,4926,4964,4995,11451,11454,11488,11524,11562,11593,18049,18052,18086,18122,18160,18191,4854,4855,4924,4925,4993,4994,11452,11453,11522,11523,11591,11592,18050,18051,18120,18121,18189,18190,4857,4926,4927,4995,4996,5064,11455,11524,11525,11593,11594,11662,18053,18122,18123,18191,18192,18260,4858,4864,4928,4934,4997,5004,11456,11462,11526,11532,11595,11602,18054,18060,18124,18130,18193,18200,4859,4928,4929,4997,4998,5066,11457,11526,11527,11595,11596,11664,18055,18124,18125,18193,18194,18262,4860,4929,4930,4998,4999,5067,11458,11527,11528,11596,11597,11665,18056,18125,18126,18194,18195,18263,4862,4925,4932,4994,5002,5063,11460,11523,11530,11592,11600,11661,18058,18121,18128,18190,18198,18259,4869,4912,4939,4982,5009,5051,11467,11510,11537,11580,11607,11649,18065,18108,18135,18178,18205,18247,4870,4940,4941,5010,5011,5080,11468,11538,11539,11608,11609,11678,18066,18136,18137,18206,18207,18276,4871,4941,4942,5011,5012,5081,11469,11539,11540,11609,11610,11679,18067,18137,18138,18207,18208,18277,4872,4940,4943,5010,5013,5079,11470,11538,11541,11608,11611,11677,18068,18136,18139,18206,18209,18275,4874,4942,4945,5012,5015,5082,11472,11540,11543,11610,11613,11680,18070,18138,18141,18208,18211,18278,4875,4946,4947,5016,5017,5086,11473,11544,11545,11614,11615,11684,18071,18142,18143,18212,18213,18282,4876,4947,4948,5017,5018,5088,11474,11545,11546,11615,11616,11686,18072,18143,18144,18213,18214,18284,4877,4949,4950,5019,5020,5090,11475,11547,11548,11617,11618,11688,18073,18145,18146,18215,18216,18286,4878,4879,4951,4952,5021,5022,11476,11477,11549,11550,11619,11620,18074,18075,18147,18148,18217,18218,4881,4954,4955,5025,5026,5097,11479,11552,11553,11623,11624,11695,18077,18150,18151,18221,18222,18293,4882,4955,4956,5026,5027,5099,11480,11553,11554,11624,11625,11697,18078,18151,18152,18222,18223,18295,4883,4956,4957,5027,5028,5100,11481,11554,11555,11625,11626,11698,18079,18152,18153,18223,18224,18296,4884,4957,4958,5028,5029,5101,11482,11555,11556,11626,11627,11699,18080,18153,18154,18224,18225,18297,4885,4958,4959,5029,5030,5102,11483,11556,11557,11627,11628,11700,18081,18154,18155,18225,18226,18298,4886,4959,4960,5030,5031,5103,11484,11557,11558,11628,11629,11701,18082,18155,18156,18226,18227,18299,4887,4954,4961,5025,5032,5098,11485,11552,11559,11623,11630,11696,18083,18150,18157,18221,18228,18294,4888,4961,4962,5032,5033,5105,11486,11559,11560,11630,11631,11703,18084,18157,18158,18228,18229,18301,4889,4960,4963,5031,5034,5104,11487,11558,11561,11629,11632,11702,18085,18156,18159,18227,18230,18300,4891,4892,4962,4965,5033,5035,11489,11490,11560,11563,11631,11633,18087,18088,18158,18161,18229,18231,4893,4897,4966,4968,5036,5038,11491,11495,11564,11566,11634,11636,18089,18093,18162,18164,18232,18234,4894,4966,4967,5036,5037,5108,11492,11564,11565,11634,11635,11706,18090,18162,18163,18232,18233,18304,4895,4910,4911,4967,4981,5037,11493,11508,11509,11565,11579,11635,18091,18106,18107,18163,18177,18233,4898,4915,4969,4984,5039,5053,11496,11513,11567,11582,11637,11651,18094,18111,18165,18180,18235,18249,4899,4969,4970,5039,5040,5110,11497,11567,11568,11637,11638,11708,18095,18165,18166,18235,18236,18306,4904,4908,4975,4979,5044,5049,11502,11506,11573,11577,11642,11647,18100,18104,18171,18175,18240,18245,4913,4916,4983,4985,5052,5054,11511,11514,11581,11583,11650,11652,18109,18112,18179,18181,18248,18250,4914,4983,4984,5052,5053,5123,11512,11581,11582,11650,11651,11721,18110,18179,18180,18248,18249,18319,4917,4985,4986,5054,5055,5124,11515,11583,11584,11652,11653,11722,18113,18181,18182,18250,18251,18320,4923,4986,4992,5055,5061,5125,11521,11584,11590,11653,11659,11723,18119,18182,18188,18251,18257,18321,4924,4935,4993,5005,5062,5074,11522,11533,11591,11603,11660,11672,18120,18131,18189,18201,18258,18270,4927,4930,4996,4999,5065,5068,11525,11528,11594,11597,11663,11666,18123,18126,18192,18195,18261,18264,4931,5000,5001,5069,5070,5139,11529,11598,11599,11667,11668,11737,18127,18196,18197,18265,18266,18335,4932,5001,5002,5070,5071,5140,11530,11599,11600,11668,11669,11738,18128,18197,18198,18266,18267,18336,4933,5000,5003,5069,5072,5138,11531,11598,11601,11667,11670,11736,18129,18196,18199,18265,18268,18334,4934,5003,5004,5072,5073,5142,5143,11532,11601,11602,11670,11671,11740,11741,18130,18199,18200,18268,18269,18338,18339,4936,4937,5006,5007,5075,5076,11534,11535,11604,11605,11673,11674,18132,18133,18202,18203,18271,18272,4938,5007,5008,5076,5077,5146,11536,11605,11606,11674,11675,11744,18134,18203,18204,18272,18273,18342,4939,5008,5009,5077,5078,5147,11537,11606,11607,11675,11676,11745,18135,18204,18205,18273,18274,18343,4943,4944,5013,5014,5083,5084,11541,11542,11611,11612,11681,11682,18139,18140,18209,18210,18279,18280,4945,4946,5015,5016,5085,5087,11543,11544,11613,11614,11683,11685,18141,18142,18211,18212,18281,18283,4948,4949,5018,5019,5089,5091,11546,11547,11616,11617,11687,11689,18144,18145,18214,18215,18285,18287,4950,4951,5020,5021,5092,5093,11548,11549,11618,11619,11690,11691,18146,18147,18216,18217,18288,18289,4952,5022,5023,5094,5095,5165,11550,11620,11621,11692,11693,11763,18148,18218,18219,18290,18291,18361,4953,5023,5024,5095,5096,5166,11551,11621,11622,11693,11694,11764,18149,18219,18220,18291,18292,18362,4963,4964,4995,5034,5064,5106,11561,11562,11593,11632,11662,11704,18159,18160,18191,18230,18260,18302,4965,5024,5035,5096,5107,5167,11563,11622,11633,11694,11705,11765,18161,18220,18231,18292,18303,18363,4968,4970,5038,5040,5109,5111,11566,11568,11636,11638,11707,11709,18164,18166,18234,18236,18305,18307,4971,4982,5041,5051,5112,5122,11569,11580,11639,11649,11710,11720,18167,18178,18237,18247,18308,18318,4972,5041,5042,5112,5113,5183,11570,11639,11640,11710,11711,11781,18168,18237,18238,18308,18309,18379,4973,5042,5043,5113,5114,5184,11571,11640,11641,11711,11712,11782,18169,18238,18239,18309,18310,18380,4974,4975,5043,5044,5114,5115,11572,11573,11641,11642,11712,11713,18170,18171,18239,18240,18310,18311,4976,5045,5046,5116,5117,5186,11574,11643,11644,11714,11715,11784,18172,18241,18242,18312,18313,18382,4977,5045,5047,5116,5118,5187,11575,11643,11645,11714,11716,11785,18173,18241,18243,18312,18314,18383,4978,5046,5048,5117,5119,5188,11576,11644,11646,11715,11717,11786,18174,18242,18244,18313,18315,18384,4979,5048,5049,5119,5120,5190,11577,11646,11647,11717,11718,11788,18175,18244,18245,18315,18316,18386,4980,5047,5050,5118,5121,5189,11578,11645,11648,11716,11719,11787,18176,18243,18246,18314,18317,18385,4981,5037,5050,5108,5121,5179,11579,11635,11648,11706,11719,11777,18177,18233,18246,18304,18317,18375,4987,5056,5057,5126,5127,5197,11585,11654,11655,11724,11725,11795,18183,18252,18253,18322,18323,18393,4988,5014,5057,5084,5127,5154,11586,11612,11655,11682,11725,11752,18184,18210,18253,18280,18323,18350,4989,5056,5058,5126,5128,5196,11587,11654,11656,11724,11726,11794,18185,18252,18254,18322,18324,18392,4990,5058,5059,5128,5129,5198,11588,11656,11657,11726,11727,11796,18186,18254,18255,18324,18325,18394,4991,5059,5060,5129,5130,5199,11589,11657,11658,11727,11728,11797,18187,18255,18256,18325,18326,18395,4992,5060,5061,5130,5131,5200,11590,11658,11659,11728,11729,11798,18188,18256,18257,18326,18327,18396,4993,4994,5062,5063,5132,5133,11591,11592,11660,11661,11730,11731,18189,18190,18258,18259,18328,18329,4996,5064,5065,5106,5134,5177,11594,11662,11663,11704,11732,11775,18192,18260,18261,18302,18330,18373,4997,5004,5066,5073,5135,11595,11602,11664,11671,11733,18193,18200,18262,18269,18331,4998,5066,5067,5135,5136,5205,11596,11664,11665,11733,11734,11803,18194,18262,18263,18331,18332,18401,4999,5067,5068,5136,5137,5206,11597,11665,11666,11734,11735,11804,18195,18263,18264,18332,18333,18402,5002,5063,5071,5133,5141,5203,11600,11661,11669,11731,11739,11801,18198,18259,18267,18329,18337,18399,5005,5006,5074,5075,5144,5145,11603,11604,11672,11673,11742,11743,18201,18202,18270,18271,18340,18341,5009,5051,5078,5122,5148,5192,11607,11649,11676,11720,11746,11790,18205,18247,18274,18318,18344,18388,5010,5079,5080,5149,5150,5220,11608,11677,11678,11747,11748,11818,18206,18275,18276,18345,18346,18416,5011,5080,5081,5150,5151,5221,11609,11678,11679,11748,11749,11819,18207,18276,18277,18346,18347,18417,5012,5081,5082,5151,5152,5222,11610,11679,11680,11749,11750,11820,18208,18277,18278,18347,18348,18418,5013,5079,5083,5149,5153,5219,11611,11677,11681,11747,11751,11817,18209,18275,18279,18345,18349,18415,5015,5082,5085,5152,5155,5223,5226,11613,11680,11683,11750,11753,11821,11824,18211,18278,18281,18348,18351,18419,18422,5016,5086,5087,5156,5157,5228,11614,11684,11685,11754,11755,11826,18212,18282,18283,18352,18353,18424,5017,5086,5088,5156,5158,5229,11615,11684,11686,11754,11756,11827,18213,18282,18284,18352,18354,18425,5018,5088,5089,5158,5159,5231,11616,11686,11687,11756,11757,11829,18214,18284,18285,18354,18355,18427,5019,5090,5091,5160,5161,5233,11617,11688,11689,11758,11759,11831,18215,18286,18287,18356,18357,18429,5020,5090,5092,5160,5162,5234,11618,11688,11690,11758,11760,11832,18216,18286,18288,18356,18358,18430,5021,5022,5093,5094,5163,5164,11619,11620,11691,11692,11761,11762,18217,18218,18289,18290,18359,18360,5025,5097,5098,5168,5169,5242,11623,11695,11696,11766,11767,11840,18221,18293,18294,18364,18365,18438,5026,5097,5099,5168,5170,5243,11624,11695,11697,11766,11768,11841,18222,18293,18295,18364,18366,18439,5027,5099,5100,5170,5171,5245,11625,11697,11698,11768,11769,11843,18223,18295,18296,18366,18367,18441,5028,5100,5101,5171,5172,5246,11626,11698,11699,11769,11770,11844,18224,18296,18297,18367,18368,18442,5029,5101,5102,5172,5173,5247,11627,11699,11700,11770,11771,11845,18225,18297,18298,18368,18369,18443,5030,5102,5103,5173,5174,5248,11628,11700,11701,11771,11772,11846,18226,18298,18299,18369,18370,18444,5031,5103,5104,5174,5175,5249,11629,11701,11702,11772,11773,11847,18227,18299,18300,18370,18371,18445,5032,5098,5105,5169,5176,5244,11630,11696,11703,11767,11774,11842,18228,18294,18301,18365,18372,18440,5033,5035,5105,5107,5176,5178,11631,11633,11703,11705,11774,11776,18229,18231,18301,18303,18372,18374,5034,5104,5106,5175,5177,5250,11632,11702,11704,11773,11775,11848,18230,18300,18302,18371,18373,18446,5036,5038,5108,5109,5179,5180,11634,11636,11706,11707,11777,11778,18232,18234,18304,18305,18375,18376,5039,5053,5110,5123,5181,5193,5254,11637,11651,11708,11721,11779,11791,11852,18235,18249,18306,18319,18377,18389,18450,5040,5110,5111,5181,5182,5253,11638,11708,11709,11779,11780,11851,18236,18306,18307,18377,18378,18449,5044,5049,5115,5120,5185,5191,11642,11647,11713,11718,11783,11789,18240,18245,18311,18316,18381,18387,5052,5054,5123,5124,5193,5194,11650,11652,11721,11722,11791,11792,18248,18250,18319,18320,18389,18390,5055,5124,5125,5194,5195,5266,11653,11722,11723,11792,11793,11864,18251,18320,18321,18390,18391,18462,5061,5125,5131,5195,5201,5267,11659,11723,11729,11793,11799,11865,18257,18321,18327,18391,18397,18463,5062,5074,5132,5144,5202,5214,11660,11672,11730,11742,11800,11812,18258,18270,18328,18340,18398,18410,5065,5068,5134,5137,5204,5207,11663,11666,11732,11735,11802,11805,18261,18264,18330,18333,18400,18403,5069,5138,5139,5208,5209,5280,11667,11736,11737,11806,11807,11878,18265,18334,18335,18404,18405,18476,5070,5139,5140,5209,5210,5281,11668,11737,11738,11807,11808,11879,18266,18335,18336,18405,18406,18477,5071,5140,5141,5210,5211,5282,11669,11738,11739,11808,11809,11880,18267,18336,18337,18406,18407,18478,5072,5138,5142,5208,5212,5279,11670,11736,11740,11806,11810,11877,18268,18334,18338,18404,18408,18475,5073,5135,5143,5205,5213,11671,11733,11741,11803,11811,18269,18331,18339,18401,18409,5075,5076,5145,5146,5215,5216,11673,11674,11743,11744,11813,11814,18271,18272,18341,18342,18411,18412,5077,5146,5147,5216,5217,5288,11675,11744,11745,11814,11815,11886,18273,18342,18343,18412,18413,18484,5078,5147,5148,5217,5218,5289,11676,11745,11746,11815,11816,11887,18274,18343,18344,18413,18414,18485,5083,5084,5153,5154,5224,5225,11681,11682,11751,11752,11822,11823,18279,18280,18349,18350,18420,18421,5085,5087,5155,5157,5227,5230,11683,11685,11753,11755,11825,11828,18281,18283,18351,18353,18423,18426,5089,5091,5159,5161,5232,5235,11687,11689,11757,11759,11830,11833,18285,18287,18355,18357,18428,18431,5092,5093,5162,5163,5236,5237,11690,11691,11760,11761,11834,11835,18288,18289,18358,18359,18432,18433,5094,5164,5165,5238,5239,5311,11692,11762,11763,11836,11837,11909,18290,18360,18361,18434,18435,18507,5095,5165,5166,5239,5240,5312,11693,11763,11764,11837,11838,11910,18291,18361,18362,18435,18436,18508,5096,5166,5167,5240,5241,5313,11694,11764,11765,11838,11839,11911,18292,18362,18363,18436,18437,18509,5107,5167,5178,5241,5251,5314,11705,11765,11776,11839,11849,11912,18303,18363,18374,18437,18447,18510,5109,5111,5180,5182,5252,5255,11707,11709,11778,11780,11850,11853,18305,18307,18376,18378,18448,18451,5112,5122,5183,5192,5256,5265,11710,11720,11781,11790,11854,11863,18308,18318,18379,18388,18452,18461,5113,5183,5184,5256,5257,5328,11711,11781,11782,11854,11855,11926,18309,18379,18380,18452,18453,18524,5114,5115,5184,5185,5257,5258,11712,11713,11782,11783,11855,11856,18310,18311,18380,18381,18453,18454,5116,5186,5187,5259,5260,5330,11714,11784,11785,11857,11858,11928,18312,18382,18383,18455,18456,18526,5117,5186,5188,5259,5261,5331,11715,11784,11786,11857,11859,11929,18313,18382,18384,18455,18457,18527,5118,5187,5189,5260,5262,5332,11716,11785,11787,11858,11860,11930,18314,18383,18385,18456,18458,18528,5119,5188,5190,5261,5263,5333,11717,11786,11788,11859,11861,11931,18315,18384,18386,18457,18459,18529,5120,5190,5191,5263,5264,5334,11718,11788,11789,11861,11862,11932,18316,18386,18387,18459,18460,18530,5121,5179,5180,5189,5252,5262,11719,11777,11778,11787,11850,11860,18317,18375,18376,18385,18448,18458,5126,5196,5197,5268,5269,5339,11724,11794,11795,11866,11867,11937,18322,18392,18393,18464,18465,18535,5127,5154,5197,5225,5269,5297,11725,11752,11795,11823,11867,11895,18323,18350,18393,18421,18465,18493,5128,5196,5198,5268,5270,5340,11726,11794,11796,11866,11868,11938,18324,18392,18394,18464,18466,18536,5129,5198,5199,5270,5271,5341,11727,11796,11797,11868,11869,11939,18325,18394,18395,18466,18467,18537,5130,5199,5200,5271,5272,5342,11728,11797,11798,11869,11870,11940,18326,18395,18396,18467,18468,18538,5131,5200,5201,5272,5273,5343,11729,11798,11799,11870,11871,11941,18327,18396,18397,18468,18469,18539,5132,5133,5202,5203,5274,5275,11730,11731,11800,11801,11872,11873,18328,18329,18398,18399,18470,18471,5134,5177,5204,5250,5276,5323,11732,11775,11802,11848,11874,11921,18330,18373,18400,18446,18472,18519,5136,5205,5206,5213,5277,5285,11734,11803,11804,11811,11875,11883,18332,18401,18402,18409,18473,18481,5137,5206,5207,5277,5278,5348,11735,11804,11805,11875,11876,11946,18333,18402,18403,18473,18474,18544,5141,5203,5211,5275,5283,5346,11739,11801,11809,11873,11881,11944,18337,18399,18407,18471,18479,18542,5142,5143,5212,5213,5284,5285,11740,11741,11810,11811,11882,11883,18338,18339,18408,18409,18480,18481,5144,5145,5214,5215,5286,5287,11742,11743,11812,11813,11884,11885,18340,18341,18410,18411,18482,18483,5148,5192,5218,5265,5290,5336,11746,11790,11816,11863,11888,11934,18344,18388,18414,18461,18486,18532,5149,5219,5220,5291,5292,5362,11747,11817,11818,11889,11890,11960,18345,18415,18416,18487,18488,18558,5150,5220,5221,5292,5293,5363,11748,11818,11819,11890,11891,11961,18346,18416,18417,18488,18489,18559,5151,5221,5222,5293,5294,5364,11749,11819,11820,11891,11892,11962,18347,18417,18418,18489,18490,18560,5152,5222,5223,5294,5295,5365,5366,11750,11820,11821,11892,11893,11963,11964,18348,18418,18419,18490,18491,18561,18562,5153,5219,5224,5291,5296,5361,11751,11817,11822,11889,11894,11959,18349,18415,18420,18487,18492,18557,5155,5226,5227,5298,5299,5370,11753,11824,11825,11896,11897,11968,18351,18422,18423,18494,18495,18566,5156,5228,5229,5300,5301,5373,11754,11826,11827,11898,11899,11971,18352,18424,18425,18496,18497,18569,5157,5228,5230,5300,5302,5372,11755,11826,11828,11898,11900,11970,18353,18424,18426,18496,18498,18568,5158,5229,5231,5301,5303,5374,11756,11827,11829,11899,11901,11972,18354,18425,18427,18497,18499,18570,5159,5231,5232,5303,5304,5376,11757,11829,11830,11901,11902,11974,18355,18427,18428,18499,18500,18572,5160,5233,5234,5305,5306,5378,11758,11831,11832,11903,11904,11976,18356,18429,18430,18501,18502,18574,5161,5233,5235,5305,5307,5379,11759,11831,11833,11903,11905,11977,18357,18429,18431,18501,18503,18575,5162,5234,5236,5306,5308,5380,11760,11832,11834,11904,11906,11978,18358,18430,18432,18502,18504,18576,5163,5164,5237,5238,5309,5310,5383,11761,11762,11835,11836,11907,11908,11981,18359,18360,18433,18434,18505,18506,18579,5168,5242,5243,5315,5316,5391,11766,11840,11841,11913,11914,11989,18364,18438,18439,18511,18512,18587,5169,5242,5244,5315,5317,5390,11767,11840,11842,11913,11915,11988,18365,18438,18440,18511,18513,18586,5170,5243,5245,5316,5318,5392,11768,11841,11843,11914,11916,11990,18366,18439,18441,18512,18514,18588,5171,5245,5246,5318,5319,5393,11769,11843,11844,11916,11917,11991,18367,18441,18442,18514,18515,18589,5172,5246,5247,5319,5320,5394,11770,11844,11845,11917,11918,11992,18368,18442,18443,18515,18516,18590,5173,5247,5248,5320,5321,5395,11771,11845,11846,11918,11919,11993,18369,18443,18444,18516,18517,18591,5174,5248,5249,5321,5322,5396,11772,11846,11847,11919,11920,11994,18370,18444,18445,18517,18518,18592,5175,5249,5250,5322,5323,5397,11773,11847,11848,11920,11921,11995,18371,18445,18446,18518,18519,18593,5176,5178,5244,5251,5317,5324,11774,11776,11842,11849,11915,11922,18372,18374,18440,18447,18513,18520,5181,5253,5254,5325,5326,11779,11851,11852,11923,11924,18377,18449,18450,18521,18522,5182,5253,5255,5325,5327,5399,11780,11851,11853,11923,11925,11997,18378,18449,18451,18521,18523,18595,5185,5191,5258,5264,5329,5335,11783,11789,11856,11862,11927,11933,18381,18387,18454,18460,18525,18531,5193,5194,5254,5266,5326,5337,11791,11792,11852,11864,11924,11935,18389,18390,18450,18462,18522,18533,5195,5266,5267,5337,5338,5409,11793,11864,11865,11935,11936,12007,18391,18462,18463,18533,18534,18605,5201,5267,5273,5338,5344,5410,11799,11865,11871,11936,11942,12008,18397,18463,18469,18534,18540,18606,5202,5214,5274,5286,5345,5356,11800,11812,11872,11884,11943,11954,18398,18410,18470,18482,18541,18552,5204,5207,5276,5278,5347,5349,11802,11805,11874,11876,11945,11947,18400,18403,18472,18474,18543,18545,5208,5279,5280,5350,5351,5422,11806,11877,11878,11948,11949,12020,18404,18475,18476,18546,18547,18618,5209,5280,5281,5351,5352,5424,11807,11878,11879,11949,11950,12022,18405,18476,18477,18547,18548,18620,5210,5281,5282,5352,5353,5425,11808,11879,11880,11950,11951,12023,18406,18477,18478,18548,18549,18621,5211,5282,5283,5353,5354,5426,11809,11880,11881,11951,11952,12024,18407,18478,18479,18549,18550,18622,5212,5279,5284,5350,5355,5423,11810,11877,11882,11948,11953,12021,18408,18475,18480,18546,18551,18619,5215,5216,5287,5288,5357,5358,11813,11814,11885,11886,11955,11956,18411,18412,18483,18484,18553,18554,5217,5288,5289,5358,5359,5430,11815,11886,11887,11956,11957,12028,18413,18484,18485,18554,18555,18626,5218,5289,5290,5359,5360,5431,11816,11887,11888,11957,11958,12029,18414,18485,18486,18555,18556,18627,5223,5226,5295,5298,5367,11821,11824,11893,11896,11965,18419,18422,18491,18494,18563,5224,5225,5296,5297,5368,5369,11822,11823,11894,11895,11966,11967,18420,18421,18492,18493,18564,18565,5227,5230,5299,5302,5371,5375,11825,11828,11897,11900,11969,11973,18423,18426,18495,18498,18567,18571,5232,5235,5304,5307,5377,5381,11830,11833,11902,11905,11975,11979,18428,18431,18500,18503,18573,18577,5236,5237,5308,5309,5382,5384,11834,11835,11906,11907,11980,11982,18432,18433,18504,18505,18578,18580,5238,5310,5311,5385,5386,11836,11908,11909,11983,11984,18434,18506,18507,18581,18582,5239,5311,5312,5386,5387,5458,11837,11909,11910,11984,11985,12056,18435,18507,18508,18582,18583,18654,5240,5312,5313,5387,5388,5459,11838,11910,11911,11985,11986,12057,18436,18508,18509,18583,18584,18655,5241,5313,5314,5388,5389,5460,11839,11911,11912,11986,11987,12058,18437,18509,18510,18584,18585,18656,5251,5314,5324,5389,5398,5461,11849,11912,11922,11987,11996,12059,18447,18510,18520,18585,18594,18657,5252,5255,5262,5327,5332,5400,11850,11853,11860,11925,11930,11998,18448,18451,18458,18523,18528,18596,5256,5265,5328,5336,5401,5408,11854,11863,11926,11934,11999,12006,18452,18461,18524,18532,18597,18604,5257,5258,5328,5329,5401,5402,11855,11856,11926,11927,11999,12000,18453,18454,18524,18525,18597,18598,5259,5330,5331,5403,5404,5474,11857,11928,11929,12001,12002,12072,18455,18526,18527,18599,18600,18670,5260,5330,5332,5400,5403,5472,11858,11928,11930,11998,12001,12070,18456,18526,18528,18596,18599,18668,5261,5331,5333,5404,5405,5475,5476,11859,11929,11931,12002,12003,12073,12074,18457,18527,18529,18600,18601,18671,18672,5263,5333,5334,5405,5406,11861,11931,11932,12003,12004,18459,18529,18530,18601,18602,5264,5334,5335,5406,5407,5477,11862,11932,11933,12004,12005,12075,18460,18530,18531,18602,18603,18673,5268,5339,5340,5411,5412,5481,11866,11937,11938,12009,12010,12079,18464,18535,18536,18607,18608,18677,5269,5297,5339,5369,5411,5441,11867,11895,11937,11967,12009,12039,18465,18493,18535,18565,18607,18637,5270,5340,5341,5412,5413,5482,11868,11938,11939,12010,12011,12080,18466,18536,18537,18608,18609,18678,5271,5341,5342,5413,5414,5483,11869,11939,11940,12011,12012,12081,18467,18537,18538,18609,18610,18679,5272,5342,5343,5414,5415,5484,11870,11940,11941,12012,12013,12082,18468,18538,18539,18610,18611,18680,5273,5343,5344,5415,5416,5485,11871,11941,11942,12013,12014,12083,18469,18539,18540,18611,18612,18681,5274,5275,5345,5346,5417,5418,11872,11873,11943,11944,12015,12016,18470,18471,18541,18542,18613,18614,5276,5323,5347,5397,5419,5469,11874,11921,11945,11995,12017,12067,18472,18519,18543,18593,18615,18665,5277,5284,5285,5348,5355,5420,11875,11882,11883,11946,11953,12018,18473,18480,18481,18544,18551,18616,5278,5348,5349,5420,5421,5490,11876,11946,11947,12018,12019,12088,18474,18544,18545,18616,18617,18686,5283,5346,5354,5418,5427,5488,11881,11944,11952,12016,12025,12086,18479,18542,18550,18614,18623,18684,5286,5287,5356,5357,5428,5429,11884,11885,11954,11955,12026,12027,18482,18483,18552,18553,18624,18625,5290,5336,5360,5408,5432,5479,11888,11934,11958,12006,12030,12077,18486,18532,18556,18604,18628,18675,5291,5361,5362,5433,5434,5503,11889,11959,11960,12031,12032,12101,18487,18557,18558,18629,18630,18699,5292,5362,5363,5434,5435,5505,11890,11960,11961,12032,12033,12103,18488,18558,18559,18630,18631,18701,5293,5363,5364,5435,5436,5506,11891,11961,11962,12033,12034,12104,18489,18559,18560,18631,18632,18702,5294,5364,5365,5436,5437,11892,11962,11963,12034,12035,18490,18560,18561,18632,18633,5295,5366,5367,5438,5439,5509,5510,11893,11964,11965,12036,12037,12107,12108,18491,18562,18563,18634,18635,18705,18706,5296,5361,5368,5433,5440,5504,11894,11959,11966,12031,12038,12102,18492,18557,18564,18629,18636,18700,5298,5367,5370,5439,5442,5511,11896,11965,11968,12037,12040,12109,18494,18563,18566,18635,18638,18707,5299,5370,5371,5442,5443,5514,11897,11968,11969,12040,12041,12112,18495,18566,18567,18638,18639,18710,5300,5372,5373,5444,5445,5517,11898,11970,11971,12042,12043,12115,18496,18568,18569,18640,18641,18713,5301,5373,5374,5445,5446,5518,11899,11971,11972,12043,12044,12116,18497,18569,18570,18641,18642,18714,5302,5372,5375,5444,5447,5516,11900,11970,11973,12042,12045,12114,18498,18568,18571,18640,18643,18712,5303,5374,5376,5446,5448,5519,11901,11972,11974,12044,12046,12117,18499,18570,18572,18642,18644,18715,5304,5376,5377,5448,5449,5521,11902,11974,11975,12046,12047,12119,18500,18572,18573,18644,18645,18717,5305,5378,5379,5450,5451,5523,11903,11976,11977,12048,12049,12121,18501,18574,18575,18646,18647,18719,5306,5378,5380,5450,5452,5524,11904,11976,11978,12048,12050,12122,18502,18574,18576,18646,18648,18720,5307,5379,5381,5451,5453,5525,11905,11977,11979,12049,12051,12123,18503,18575,18577,18647,18649,18721,5308,5380,5382,5452,5454,5526,11906,11978,11980,12050,12052,12124,18504,18576,18578,18648,18650,18722,5309,5383,5384,5455,5456,5529,11907,11981,11982,12053,12054,12127,18505,18579,18580,18651,18652,18725,5310,5383,5385,5455,5457,5530,5532,11908,11981,11983,12053,12055,12128,12130,18506,18579,18581,18651,18653,18726,18728,5315,5390,5391,5462,5463,5538,11913,11988,11989,12060,12061,12136,18511,18586,18587,18658,18659,18734,5316,5391,5392,5463,5464,5539,11914,11989,11990,12061,12062,12137,18512,18587,18588,18659,18660,18735,5317,5324,5390,5398,5462,5470,11915,11922,11988,11996,12060,12068,18513,18520,18586,18594,18658,18666,5318,5392,5393,5464,5465,5540,11916,11990,11991,12062,12063,12138,18514,18588,18589,18660,18661,18736,5319,5393,5394,5465,5466,5541,11917,11991,11992,12063,12064,12139,18515,18589,18590,18661,18662,18737,5320,5394,5395,5466,5467,11918,11992,11993,12064,12065,18516,18590,18591,18662,18663,5321,5395,5396,5467,5468,5542,11919,11993,11994,12065,12066,12140,18517,18591,18592,18663,18664,18738,5322,5396,5397,5468,5469,5543,11920,11994,11995,12066,12067,12141,18518,18592,18593,18664,18665,18739,5325,5326,5337,5399,5409,5471,11923,11924,11935,11997,12007,12069,18521,18522,18533,18595,18605,18667,5327,5399,5400,5471,5472,5545,11925,11997,11998,12069,12070,12143,18523,18595,18596,18667,18668,18741,5329,5335,5402,5407,5473,5478,11927,11933,12000,12005,12071,12076,18525,18531,18598,18603,18669,18674,5338,5409,5410,5471,5480,5545,11936,12007,12008,12069,12078,12143,18534,18605,18606,18667,18676,18741,5344,5410,5416,5480,5486,5547,11942,12008,12014,12078,12084,12145,18540,18606,18612,18676,18682,18743,5345,5356,5417,5428,5487,5498,5558,11943,11954,12015,12026,12085,12096,12156,18541,18552,18613,18624,18683,18694,18754,5347,5349,5419,5421,5489,5491,11945,11947,12017,12019,12087,12089,18543,18545,18615,18617,18685,18687,5350,5422,5423,5492,5493,5563,11948,12020,12021,12090,12091,12161,18546,18618,18619,18688,18689,18759,5351,5422,5424,5492,5494,5564,11949,12020,12022,12090,12092,12162,18547,18618,18620,18688,18690,18760,5352,5424,5425,5494,5495,5565,11950,12022,12023,12092,12093,12163,18548,18620,18621,18690,18691,18761,5353,5425,5426,5495,5496,5566,11951,12023,12024,12093,12094,12164,18549,18621,18622,18691,18692,18762,5354,5426,5427,5496,5497,5567,11952,12024,12025,12094,12095,12165,18550,18622,18623,18692,18693,18763,5355,5420,5423,5490,5493,5561,11953,12018,12021,12088,12091,12159,18551,18616,18619,18686,18689,18757,5357,5358,5429,5430,5499,5500,11955,11956,12027,12028,12097,12098,18553,18554,18625,18626,18695,18696,5359,5430,5431,5500,5501,5571,11957,12028,12029,12098,12099,12169,18555,18626,18627,18696,18697,18767,5360,5431,5432,5501,5502,5572,11958,12029,12030,12099,12100,12170,18556,18627,18628,18697,18698,18768,5365,5366,5437,5438,5507,5508,5579,11963,11964,12035,12036,12105,12106,12177,18561,18562,18633,18634,18703,18704,18775,5368,5369,5440,5441,5512,5513,11966,11967,12038,12039,12110,12111,18564,18565,18636,18637,18708,18709,5371,5375,5443,5447,5515,5520,11969,11973,12041,12045,12113,12118,18567,18571,18639,18643,18711,18716,5377,5381,5449,5453,5522,5527,11975,11979,12047,12051,12120,12125,18573,18577,18645,18649,18718,18723,5382,5384,5454,5456,5528,5531,11980,11982,12052,12054,12126,12129,18578,18580,18650,18652,18724,18727,5385,5386,5457,5458,5533,5534,11983,11984,12055,12056,12131,12132,18581,18582,18653,18654,18729,18730,5387,5458,5459,5534,5535,5606,11985,12056,12057,12132,12133,12204,18583,18654,18655,18730,18731,18802,5388,5459,5460,5535,5536,5607,11986,12057,12058,12133,12134,12205,18584,18655,18656,18731,18732,18803,5389,5460,5461,5536,5537,5608,11987,12058,12059,12134,12135,12206,18585,18656,18657,18732,18733,18804,5398,5461,5470,5537,5544,5609,11996,12059,12068,12135,12142,12207,18594,18657,18666,18733,18740,18805,5401,5402,5408,5473,5479,5546,11999,12000,12006,12071,12077,12144,18597,18598,18604,18669,18675,18742,5403,5472,5474,5480,5545,5547,12001,12070,12072,12078,12143,12145,18599,18668,18670,18676,18741,18743,5404,5474,5475,5486,5547,5548,12002,12072,12073,12084,12145,12146,18600,18670,18671,18682,18743,18744,5405,5406,5476,5477,5549,5550,12003,12004,12074,12075,12147,12148,18601,18602,18672,18673,18745,18746,5407,5477,5478,5550,5551,5620,12005,12075,12076,12148,12149,12218,18603,18673,18674,18746,18747,18816,5411,5441,5481,5513,5552,5585,12009,12039,12079,12111,12150,12183,18607,18637,18677,18709,18748,18781,5412,5481,5482,5552,5553,5622,5623,12010,12079,12080,12150,12151,12220,12221,18608,18677,18678,18748,18749,18818,18819,5413,5482,5483,5553,5554,5624,12011,12080,12081,12151,12152,12222,18609,18678,18679,18749,18750,18820,5414,5483,5484,5554,5555,5625,12012,12081,12082,12152,12153,12223,18610,18679,18680,18750,18751,18821,5415,5484,5485,5555,5556,5626,12013,12082,12083,12153,12154,12224,18611,18680,18681,18751,18752,18822,5416,5485,5486,5548,5556,5618,12014,12083,12084,12146,12154,12216,18612,18681,18682,18744,18752,18814,5417,5418,5487,5488,5557,5559,12015,12016,12085,12086,12155,12157,18613,18614,18683,18684,18753,18755,5419,5469,5489,5543,5560,5615,12017,12067,12087,12141,12158,12213,18615,18665,18685,18739,18756,18811,5421,5490,5491,5561,5562,5631,12019,12088,12089,12159,12160,12229,18617,18686,18687,18757,18758,18827,5427,5488,5497,5559,5568,5629,12025,12086,12095,12157,12166,12227,18623,18684,18693,18755,18764,18825,5428,5429,5498,5499,5569,5570,12026,12027,12096,12097,12167,12168,18624,18625,18694,18695,18765,18766,5432,5479,5502,5546,5573,5617,12030,12077,12100,12144,12171,12215,18628,18675,18698,18742,18769,18813,5433,5503,5504,5574,5575,5644,12031,12101,12102,12172,12173,12242,18629,18699,18700,18770,18771,18840,5434,5503,5505,5574,5576,5645,12032,12101,12103,12172,12174,12243,18630,18699,18701,18770,18772,18841,5435,5505,5506,5576,5577,5647,12033,12103,12104,12174,12175,12245,18631,18701,18702,18772,18773,18843,5436,5437,5506,5507,5577,5578,12034,12035,12104,12105,12175,12176,18632,18633,18702,18703,18773,18774,5438,5508,5509,5580,5581,5651,12036,12106,12107,12178,12179,12249,18634,18704,18705,18776,18777,18847,5439,5510,5511,5582,5583,5602,5654,12037,12108,12109,12180,12181,12200,12252,18635,18706,18707,18778,18779,18798,18850,5440,5504,5512,5575,5584,5646,12038,12102,12110,12173,12182,12244,18636,18700,18708,18771,18780,18842,5442,5511,5514,5583,5586,5601,12040,12109,12112,12181,12184,12199,18638,18707,18710,18779,18782,18797,5443,5514,5515,5586,5587,5603,12041,12112,12113,12184,12185,12201,18639,18710,18711,18782,18783,18799,5444,5516,5517,5588,5589,5658,12042,12114,12115,12186,12187,12256,18640,18712,18713,18784,18785,18854,5445,5517,5518,5589,5590,5660,12043,12115,12116,12187,12188,12258,18641,18713,18714,18785,18786,18856,5446,5518,5519,5590,5591,5661,12044,12116,12117,12188,12189,12259,18642,18714,18715,18786,18787,18857,5447,5516,5520,5588,5592,5659,12045,12114,12118,12186,12190,12257,18643,18712,18716,18784,18788,18855,5448,5519,5521,5591,5593,5662,12046,12117,12119,12189,12191,12260,18644,18715,18717,18787,18789,18858,5449,5521,5522,5593,5594,5664,12047,12119,12120,12191,12192,12262,18645,18717,18718,18789,18790,18860,5450,5523,5524,5595,5596,5667,12048,12121,12122,12193,12194,12265,18646,18719,18720,18791,18792,18863,5451,5523,5525,5595,5597,5666,12049,12121,12123,12193,12195,12264,18647,18719,18721,18791,18793,18862,5452,5524,5526,5596,5598,5668,12050,12122,12124,12194,12196,12266,18648,18720,18722,18792,18794,18864,5453,5525,5527,5597,5599,5669,5672,12051,12123,12125,12195,12197,12267,12270,18649,18721,18723,18793,18795,18865,18868,5454,5526,5528,5598,5600,5670,12052,12124,12126,12196,12198,12268,18650,18722,18724,18794,18796,18866,5455,5529,5530,5583,5601,5602,12053,12127,12128,12181,12199,12200,18651,18725,18726,18779,18797,18798,5456,5529,5531,5586,5601,5603,12054,12127,12129,12184,12199,12201,18652,18725,18727,18782,18797,18799,5457,5532,5533,5604,5605,5674,12055,12130,12131,12202,12203,12272,18653,18728,18729,18800,18801,18870,5462,5470,5538,5544,5610,5616,12060,12068,12136,12142,12208,12214,18658,18666,18734,18740,18806,18812,5463,5538,5539,5610,5611,5680,12061,12136,12137,12208,12209,12278,18659,18734,18735,18806,18807,18876,5464,5539,5540,5611,5612,5681,12062,12137,12138,12209,12210,12279,18660,18735,18736,18807,18808,18877,5465,5540,5541,5612,5613,5682,12063,12138,12139,12210,12211,12280,18661,18736,18737,18808,18809,18878,5466,5467,5541,5542,5613,5614,12064,12065,12139,12140,12211,12212,18662,18663,18737,18738,18809,18810,5468,5542,5543,5614,5615,5683,12066,12140,12141,12212,12213,12281,18664,18738,18739,18810,18811,18879,5473,5478,5546,5551,5617,5621,12071,12076,12144,12149,12215,12219,18669,18674,18742,18747,18813,18817,5475,5476,5548,5549,5618,5619,12073,12074,12146,12147,12216,12217,18671,18672,18744,18745,18814,18815,5487,5557,5558,5627,5628,5694,12085,12155,12156,12225,12226,12292,18683,18753,18754,18823,18824,18890,5489,5491,5560,5562,5630,5632,12087,12089,12158,12160,12228,12230,18685,18687,18756,18758,18826,18828,5492,5563,5564,5633,5634,5699,12090,12161,12162,12231,12232,12297,18688,18759,18760,18829,18830,18895,5493,5561,5563,5631,5633,5697,12091,12159,12161,12229,12231,12295,18689,18757,18759,18827,18829,18893,5494,5564,5565,5634,5635,5700,12092,12162,12163,12232,12233,12298,18690,18760,18761,18830,18831,18896,5495,5565,5566,5635,5636,5701,12093,12163,12164,12233,12234,12299,18691,18761,18762,18831,18832,18897,5496,5566,5567,5636,5637,5702,12094,12164,12165,12234,12235,12300,18692,18762,18763,18832,18833,18898,5497,5567,5568,5637,5638,5703,12095,12165,12166,12235,12236,12301,18693,18763,18764,18833,18834,18899,5498,5558,5569,5628,5639,12096,12156,12167,12226,12237,18694,18754,18765,18824,18835,5499,5500,5570,5571,5640,5641,12097,12098,12168,12169,12238,12239,18695,18696,18766,18767,18836,18837,5501,5571,5572,5641,5642,5707,12099,12169,12170,12239,12240,12305,18697,18767,18768,18837,18838,18903,5502,5572,5573,5642,5643,5708,12100,12170,12171,12240,12241,12306,18698,18768,18769,18838,18839,18904,5507,5578,5579,5648,5649,5715,12105,12176,12177,12246,12247,12313,18703,18774,18775,18844,18845,18911,5508,5579,5580,5649,5650,5716,12106,12177,12178,12247,12248,12314,18704,18775,18776,18845,18846,18912,5509,5510,5581,5582,5652,5653,12107,12108,12179,12180,12250,12251,18705,18706,18777,18778,18848,18849,5512,5513,5584,5585,5655,5656,12110,12111,12182,12183,12253,12254,18708,18709,18780,18781,18851,18852,5515,5520,5587,5592,5657,5663,12113,12118,12185,12190,12255,12261,18711,18716,18783,18788,18853,18859,5522,5527,5594,5599,5665,5671,12120,12125,12192,12197,12263,12269,18718,18723,18790,18795,18861,18867,5528,5531,5587,5600,5603,5657,12126,12129,12185,12198,12201,12255,18724,18727,18783,18796,18799,18853,5530,5532,5602,5604,5654,5673,12128,12130,12200,12202,12252,12271,18726,18728,18798,18800,18850,18869,5533,5534,5605,5606,5675,5676,12131,12132,12203,12204,12273,12274,18729,18730,18801,18802,18871,18872,5535,5606,5607,5676,5677,5741,12133,12204,12205,12274,12275,12339,18731,18802,18803,18872,18873,18937,5536,5607,5608,5677,5678,5742,12134,12205,12206,12275,12276,12340,18732,18803,18804,18873,18874,18938,5537,5608,5609,5678,5679,5743,12135,12206,12207,12276,12277,12341,18733,18804,18805,18874,18875,18939,5544,5609,5616,5679,5684,5744,12142,12207,12214,12277,12282,12342,18740,18805,18812,18875,18880,18940,5549,5550,5619,5620,5685,5686,12147,12148,12217,12218,12283,12284,18745,18746,18815,18816,18881,18882,5551,5620,5621,5686,5687,5750,12149,12218,12219,12284,12285,12348,18747,18816,18817,18882,18883,18946,5552,5585,5622,5656,5688,5722,12150,12183,12220,12254,12286,12320,18748,18781,18818,18852,18884,18918,5553,5623,5624,5689,5690,12151,12221,12222,12287,12288,18749,18819,18820,18885,18886,5554,5624,5625,5690,5691,5753,12152,12222,12223,12288,12289,12351,18750,18820,18821,18886,18887,18949,5555,5625,5626,5691,5692,5754,12153,12223,12224,12289,12290,12352,18751,18821,18822,18887,18888,18950,5556,5618,5619,5626,5685,5692,12154,12216,12217,12224,12283,12290,18752,18814,18815,18822,18881,18888,5557,5559,5627,5629,5693,5695,12155,12157,12225,12227,12291,12293,18753,18755,18823,18825,18889,18891,5560,5615,5630,5683,5696,5748,12158,12213,12228,12281,12294,12346,18756,18811,18826,18879,18892,18944,5562,5631,5632,5697,5698,5759,12160,12229,12230,12295,12296,12357,18758,18827,18828,18893,18894,18955,5568,5629,5638,5695,5704,5757,12166,12227,12236,12293,12302,12355,18764,18825,18834,18891,18900,18953,5569,5570,5639,5640,5705,5706,12167,12168,12237,12238,12303,12304,18765,18766,18835,18836,18901,18902,5573,5617,5621,5643,5687,5709,12171,12215,12219,12241,12285,12307,18769,18813,18817,18839,18883,18905,5574,5644,5645,5710,5711,5772,12172,12242,12243,12308,12309,12370,18770,18840,18841,18906,18907,18968,5575,5644,5646,5710,5712,5773,12173,12242,12244,12308,12310,12371,18771,18840,18842,18906,18908,18969,5576,5645,5647,5711,5713,5774,12174,12243,12245,12309,12311,12372,18772,18841,18843,18907,18909,18970,5577,5578,5647,5648,5713,5714,5776,12175,12176,12245,12246,12311,12312,12374,18773,18774,18843,18844,18909,18910,18972,5580,5650,5651,5717,5718,5780,12178,12248,12249,12315,12316,12378,18776,18846,18847,18913,18914,18976,5581,5651,5652,5718,5719,5782,12179,12249,12250,12316,12317,12380,18777,18847,18848,18914,18915,18978,5582,5653,5654,5673,5720,5738,12180,12251,12252,12271,12318,12336,18778,18849,18850,18869,18916,18934,5584,5646,5655,5712,5721,5775,12182,12244,12253,12310,12319,12373,18780,18842,18851,18908,18917,18971,5588,5658,5659,5723,5724,5787,12186,12256,12257,12321,12322,12385,18784,18854,18855,18919,18920,18983,5589,5658,5660,5723,5725,5788,12187,12256,12258,12321,12323,12386,18785,18854,18856,18919,18921,18984,5590,5660,5661,5725,5726,5790,12188,12258,12259,12323,12324,12388,18786,18856,18857,18921,18922,18986,5591,5661,5662,5726,5727,5791,12189,12259,12260,12324,12325,12389,18787,18857,18858,18922,18923,18987,5592,5659,5663,5724,5728,5789,12190,12257,12261,12322,12326,12387,18788,18855,18859,18920,18924,18985,5593,5662,5664,5727,5729,5792,12191,12260,12262,12325,12327,12390,18789,18858,18860,18923,18925,18988,5594,5664,5665,5729,5730,5793,12192,12262,12263,12327,12328,12391,18790,18860,18861,18925,18926,18989,5595,5666,5667,5731,5732,5796,12193,12264,12265,12329,12330,12394,18791,18862,18863,18927,18928,18992,5596,5667,5668,5732,5733,5797,12194,12265,12266,12330,12331,12395,18792,18863,18864,18928,18929,18993,5597,5666,5669,5731,5734,5795,5800,12195,12264,12267,12329,12332,12393,12398,18793,18862,18865,18927,18930,18991,18996,5598,5668,5670,5733,5735,5798,12196,12266,12268,12331,12333,12396,18794,18864,18866,18929,18931,18994,5599,5671,5672,5736,5737,5802,12197,12269,12270,12334,12335,12400,18795,18867,18868,18932,18933,18998,5600,5657,5663,5670,5728,5735,12198,12255,12261,12268,12326,12333,18796,18853,18859,18866,18924,18931,5604,5673,5674,5738,5739,5803,12202,12271,12272,12336,12337,12401,18800,18869,18870,18934,18935,18999,5605,5674,5675,5739,5740,5804,12203,12272,12273,12337,12338,12402,18801,18870,18871,18935,18936,19000,5610,5616,5680,5684,5745,5749,12208,12214,12278,12282,12343,12347,18806,18812,18876,18880,18941,18945,5611,5680,5681,5745,5746,5810,12209,12278,12279,12343,12344,12408,18807,18876,18877,18941,18942,19006,5612,5681,5682,5746,5747,5811,5812,12210,12279,12280,12344,12345,12409,12410,18808,18877,18878,18942,18943,19007,19008,5613,5614,5682,5683,5747,5748,12211,12212,12280,12281,12345,12346,18809,18810,18878,18879,18943,18944,5622,5623,5688,5689,5751,5752,12220,12221,12286,12287,12349,12350,18818,18819,18884,18885,18947,18948,5627,5693,5694,5755,5756,5818,12225,12291,12292,12353,12354,12416,18823,18889,18890,18951,18952,19014,5628,5639,5694,5705,5756,5767,5820,12226,12237,12292,12303,12354,12365,12418,18824,18835,18890,18901,18952,18963,19016,5630,5632,5696,5698,5758,5760,12228,12230,12294,12296,12356,12358,18826,18828,18892,18894,18954,18956,5633,5697,5699,5759,5761,5823,12231,12295,12297,12357,12359,12421,18829,18893,18895,18955,18957,19019,5634,5699,5700,5761,5762,5825,12232,12297,12298,12359,12360,12423,18830,18895,18896,18957,18958,19021,5635,5700,5701,5762,5763,5826,12233,12298,12299,12360,12361,12424,18831,18896,18897,18958,18959,19022,5636,5701,5702,5763,5764,5827,12234,12299,12300,12361,12362,12425,18832,18897,18898,18959,18960,19023,5637,5702,5703,5764,5765,5828,12235,12300,12301,12362,12363,12426,18833,18898,18899,18960,18961,19024,5638,5703,5704,5765,5766,5829,12236,12301,12302,12363,12364,12427,18834,18899,18900,18961,18962,19025,5640,5641,5706,5707,5768,5769,12238,12239,12304,12305,12366,12367,18836,18837,18902,18903,18964,18965,5642,5707,5708,5769,5770,5833,12240,12305,12306,12367,12368,12431,18838,18903,18904,18965,18966,19029,5643,5708,5709,5770,5771,5834,12241,12306,12307,12368,12369,12432,18839,18904,18905,18966,18967,19030,5648,5714,5715,5777,5778,5842,12246,12312,12313,12375,12376,12440,18844,18910,18911,18973,18974,19038,5649,5715,5716,5778,5779,5843,12247,12313,12314,12376,12377,12441,18845,18911,18912,18974,18975,19039,5650,5716,5717,5779,5781,5844,12248,12314,12315,12377,12379,12442,18846,18912,18913,18975,18977,19040,5652,5653,5719,5720,5783,5784,12250,12251,12317,12318,12381,12382,18848,18849,18915,18916,18979,18980,5655,5656,5721,5722,5785,5786,12253,12254,12319,12320,12383,12384,18851,18852,18917,18918,18981,18982,5665,5671,5730,5736,5794,5801,12263,12269,12328,12334,12392,12399,18861,18867,18926,18932,18990,18997,5669,5672,5734,5737,5799,12267,12270,12332,12335,12397,18865,18868,18930,18933,18995,5675,5676,5740,5741,5805,5806,12273,12274,12338,12339,12403,12404,18871,18872,18936,18937,19001,19002,5677,5741,5742,5806,5807,5871,12275,12339,12340,12404,12405,12469,18873,18937,18938,19002,19003,19067,5678,5742,5743,5807,5808,5872,12276,12340,12341,12405,12406,12470,18874,18938,18939,19003,19004,19068,5679,5743,5744,5808,5809,5873,12277,12341,12342,12406,12407,12471,18875,18939,18940,19004,19005,19069,5684,5744,5749,5809,5813,5874,12282,12342,12347,12407,12411,12472,18880,18940,18945,19005,19009,19070,5685,5686,5692,5750,5754,5814,12283,12284,12290,12348,12352,12412,18881,18882,18888,18946,18950,19010,5687,5709,5750,5771,5814,5835,12285,12307,12348,12369,12412,12433,18883,18905,18946,18967,19010,19031,5688,5722,5751,5786,5815,5851,12286,12320,12349,12384,12413,12449,18884,18918,18947,18982,19011,19047,5689,5690,5752,5753,5816,5817,12287,12288,12350,12351,12414,12415,18885,18886,18948,18949,19012,19013,5691,5753,5754,5814,5817,5835,12289,12351,12352,12412,12415,12433,18887,18949,18950,19010,19013,19031,5693,5695,5755,5757,5819,5821,12291,12293,12353,12355,12417,12419,18889,18891,18951,18953,19015,19017,5696,5747,5748,5758,5812,5822,12294,12345,12346,12356,12410,12420,18892,18943,18944,18954,19008,19018,5698,5759,5760,5823,5824,5885,5887,12296,12357,12358,12421,12422,12483,12485,18894,18955,18956,19019,19020,19081,19083,5704,5757,5766,5821,5830,5883,12302,12355,12364,12419,12428,12481,18900,18953,18962,19017,19026,19079,5705,5706,5767,5768,5831,5832,12303,12304,12365,12366,12429,12430,18901,18902,18963,18964,19027,19028,5710,5772,5773,5836,5837,5896,12308,12370,12371,12434,12435,12494,18906,18968,18969,19032,19033,19092,5711,5772,5774,5836,5838,5897,12309,12370,12372,12434,12436,12495,18907,18968,18970,19032,19034,19093,5712,5773,5775,5837,5839,5898,12310,12371,12373,12435,12437,12496,18908,18969,18971,19033,19035,19094,5713,5774,5776,5838,5840,5899,12311,12372,12374,12436,12438,12497,18909,18970,18972,19034,19036,19095,5714,5776,5777,5840,5841,5901,12312,12374,12375,12438,12439,12499,18910,18972,18973,19036,19037,19097,5717,5780,5781,5845,5846,5906,12315,12378,12379,12443,12444,12504,18913,18976,18977,19041,19042,19102,5718,5780,5782,5845,5847,5907,12316,12378,12380,12443,12445,12505,18914,18976,18978,19041,19043,19103,5719,5782,5783,5847,5848,5909,12317,12380,12381,12445,12446,12507,18915,18978,18979,19043,19044,19105,5720,5738,5784,5803,5849,5868,12318,12336,12382,12401,12447,12466,18916,18934,18980,18999,19045,19064,5721,5775,5785,5839,5850,5900,12319,12373,12383,12437,12448,12498,18917,18971,18981,19035,19046,19096,5723,5787,5788,5852,5853,5913,12321,12385,12386,12450,12451,12511,18919,18983,18984,19048,19049,19109,5724,5787,5789,5852,5854,5914,12322,12385,12387,12450,12452,12512,18920,18983,18985,19048,19050,19110,5725,5788,5790,5853,5855,5915,12323,12386,12388,12451,12453,12513,18921,18984,18986,19049,19051,19111,5726,5790,5791,5855,5856,5916,12324,12388,12389,12453,12454,12514,18922,18986,18987,19051,19052,19112,5727,5791,5792,5856,5857,5917,12325,12389,12390,12454,12455,12515,18923,18987,18988,19052,19053,19113,5728,5735,5789,5798,5854,5863,12326,12333,12387,12396,12452,12461,18924,18931,18985,18994,19050,19059,5729,5792,5793,5857,5858,5918,12327,12390,12391,12455,12456,12516,18925,18988,18989,19053,19054,19114,5730,5793,5794,5858,5859,5919,12328,12391,12392,12456,12457,12517,18926,18989,18990,19054,19055,19115,5731,5795,5796,5860,5861,5922,12329,12393,12394,12458,12459,12520,18927,18991,18992,19056,19057,19118,5732,5796,5797,5861,5862,5923,12330,12394,12395,12459,12460,12521,18928,18992,18993,19057,19058,19119,5733,5797,5798,5862,5863,5924,12331,12395,12396,12460,12461,12522,18929,18993,18994,19058,19059,19120,5734,5799,5800,5864,5865,12332,12397,12398,12462,12463,18930,18995,18996,19060,19061,5736,5801,5802,5866,5867,5927,12334,12399,12400,12464,12465,12525,18932,18997,18998,19062,19063,19123,5737,5799,5802,5864,5867,5925,12335,12397,12400,12462,12465,12523,18933,18995,18998,19060,19063,19121,5739,5803,5804,5868,5869,5928,5929,12337,12401,12402,12466,12467,12526,12527,18935,18999,19000,19064,19065,19124,19125,5740,5804,5805,5869,5870,5930,12338,12402,12403,12467,12468,12528,18936,19000,19001,19065,19066,19126,5745,5749,5810,5813,5875,5877,12343,12347,12408,12411,12473,12475,18941,18945,19006,19009,19071,19073,5746,5810,5811,5875,5876,12344,12408,12409,12473,12474,18942,19006,19007,19071,19072,5751,5752,5815,5816,5878,5879,12349,12350,12413,12414,12476,12477,18947,18948,19011,19012,19074,19075,5755,5818,5819,5880,5881,5938,12353,12416,12417,12478,12479,12536,18951,19014,19015,19076,19077,19134,5756,5818,5820,5880,5882,5939,12354,12416,12418,12478,12480,12537,18952,19014,19016,19076,19078,19135,5758,5760,5822,5824,5884,5886,12356,12358,12420,12422,12482,12484,18954,18956,19018,19020,19080,19082,5761,5823,5825,5885,5888,5941,12359,12421,12423,12483,12486,12539,18957,19019,19021,19081,19084,19137,5762,5825,5826,5888,5889,5944,12360,12423,12424,12486,12487,12542,18958,19021,19022,19084,19085,19140,5763,5826,5827,5889,5890,5945,5946,12361,12424,12425,12487,12488,12543,12544,18959,19022,19023,19085,19086,19141,19142,5764,5827,5828,5890,5891,5947,12362,12425,12426,12488,12489,12545,18960,19023,19024,19086,19087,19143,5765,5828,5829,5891,5892,12363,12426,12427,12489,12490,18961,19024,19025,19087,19088,5766,5829,5830,5892,5893,5948,5949,12364,12427,12428,12490,12491,12546,12547,18962,19025,19026,19088,19089,19144,19145,5767,5820,5831,5882,5894,5912,12365,12418,12429,12480,12492,12510,18963,19016,19027,19078,19090,19108,5768,5769,5832,5833,5895,12366,12367,12430,12431,12493,18964,18965,19028,19029,19091,5770,5833,5834,5878,5879,5895,12368,12431,12432,12476,12477,12493,18966,19029,19030,19074,19075,19091,5771,5816,5817,5834,5835,5879,12369,12414,12415,12432,12433,12477,18967,19012,19013,19030,19031,19075,5777,5841,5842,5902,5903,5957,12375,12439,12440,12500,12501,12555,18973,19037,19038,19098,19099,19153,5778,5842,5843,5903,5904,5959,12376,12440,12441,12501,12502,12557,18974,19038,19039,19099,19100,19155,5779,5843,5844,5904,5905,5960,12377,12441,12442,12502,12503,12558,18975,19039,19040,19100,19101,19156,5781,5844,5846,5905,5908,5961,12379,12442,12444,12503,12506,12559,18977,19040,19042,19101,19104,19157,5783,5784,5848,5849,5910,5911,12381,12382,12446,12447,12508,12509,18979,18980,19044,19045,19106,19107,5785,5786,5850,5851,5894,5912,12383,12384,12448,12449,12492,12510,18981,18982,19046,19047,19090,19108,5794,5801,5859,5866,5920,12392,12399,12457,12464,12518,18990,18997,19055,19062,19116,5795,5800,5860,5865,5921,5926,12393,12398,12458,12463,12519,12524,18991,18996,19056,19061,19117,19122,5805,5806,5870,5871,5931,5932,12403,12404,12468,12469,12529,12530,19001,19002,19066,19067,19127,19128,5807,5871,5872,5932,5933,5987,12405,12469,12470,12530,12531,12585,19003,19067,19068,19128,19129,19183,5808,5872,5873,5933,5934,5988,12406,12470,12471,12531,12532,12586,19004,19068,19069,19129,19130,19184,5809,5873,5874,5934,5935,5989,12407,12471,12472,12532,12533,12587,19005,19069,19070,19130,19131,19185,5811,5812,5822,5876,5884,5936,12409,12410,12420,12474,12482,12534,19007,19008,19018,19072,19080,19132,5813,5874,5877,5935,5937,5990,12411,12472,12475,12533,12535,12588,19009,19070,19073,19131,19133,19186,5815,5831,5832,5851,5878,5894,5895,12413,12429,12430,12449,12476,12492,12493,19011,19027,19028,19047,19074,19090,19091,5819,5821,5881,5883,5940,12417,12419,12479,12481,12538,19015,19017,19077,19079,19136,5824,5886,5887,5942,5943,12422,12484,12485,12540,12541,19020,19082,19083,19138,19139,5830,5883,5893,5940,5950,5993,12428,12481,12491,12538,12548,12591,19026,19079,19089,19136,19146,19189,5836,5896,5897,5951,5952,6005,12434,12494,12495,12549,12550,12603,19032,19092,19093,19147,19148,19201,5837,5896,5898,5951,5953,6004,12435,12494,12496,12549,12551,12602,19033,19092,19094,19147,19149,19200,5838,5897,5899,5952,5954,6006,12436,12495,12497,12550,12552,12604,19034,19093,19095,19148,19150,19202,5839,5898,5900,5953,5955,6007,12437,12496,12498,12551,12553,12605,19035,19094,19096,19149,19151,19203,5840,5899,5901,5954,5956,6008,6010,12438,12497,12499,12552,12554,12606,12608,19036,19095,19097,19150,19152,19204,19206,5841,5901,5902,5956,5958,6009,12439,12499,12500,12554,12556,12607,19037,19097,19098,19152,19154,19205,5845,5906,5907,5962,5963,6017,12443,12504,12505,12560,12561,12615,19041,19102,19103,19158,19159,19213,5846,5906,5908,5962,5964,6016,12444,12504,12506,12560,12562,12614,19042,19102,19104,19158,19160,19212,5847,5907,5909,5963,5965,6018,12445,12505,12507,12561,12563,12616,19043,19103,19105,19159,19161,19214,5848,5909,5910,5965,5966,6020,12446,12507,12508,12563,12564,12618,19044,19105,19106,19161,19162,19216,5849,5868,5911,5928,5967,5983,12447,12466,12509,12526,12565,12581,19045,19064,19107,19124,19163,19179,5850,5882,5900,5912,5939,5955,12448,12480,12498,12510,12537,12553,19046,19078,19096,19108,19135,19151,5852,5913,5914,5968,5969,6024,12450,12511,12512,12566,12567,12622,19048,19109,19110,19164,19165,19220,5853,5913,5915,5968,5970,6023,12451,12511,12513,12566,12568,12621,19049,19109,19111,19164,19166,19219,5854,5863,5914,5924,5969,5979,12452,12461,12512,12522,12567,12577,19050,19059,19110,19120,19165,19175,5855,5915,5916,5970,5971,6025,12453,12513,12514,12568,12569,12623,19051,19111,19112,19166,19167,19221,5856,5916,5917,5971,5972,6026,12454,12514,12515,12569,12570,12624,19052,19112,19113,19167,19168,19222,5857,5917,5918,5972,5973,6027,12455,12515,12516,12570,12571,12625,19053,19113,19114,19168,19169,19223,5858,5918,5919,5973,5974,6028,12456,12516,12517,12571,12572,12626,19054,19114,19115,19169,19170,19224,5859,5919,5920,5974,5975,6029,6031,12457,12517,12518,12572,12573,12627,12629,19055,19115,19116,19170,19171,19225,19227,5860,5921,5922,5976,5977,6033,12458,12519,12520,12574,12575,12631,19056,19117,19118,19172,19173,19229,5861,5922,5923,5977,5978,6034,12459,12520,12521,12575,12576,12632,19057,19118,19119,19173,19174,19230,5862,5923,5924,5978,5979,6035,12460,12521,12522,12576,12577,12633,19058,19119,19120,19174,19175,19231,5864,5865,5925,5926,5980,5981,12462,12463,12523,12524,12578,12579,19060,19061,19121,19122,19176,19177,5866,5920,5927,5975,5982,6030,12464,12518,12525,12573,12580,12628,19062,19116,19123,19171,19178,19226,5867,5925,5927,5980,5982,6036,12465,12523,12525,12578,12580,12634,19063,19121,19123,19176,19178,19232,5869,5929,5930,5984,5985,12467,12527,12528,12582,12583,19065,19125,19126,19180,19181,5870,5930,5931,5985,5986,6040,12468,12528,12529,12583,12584,12638,19066,19126,19127,19181,19182,19236,5875,5876,5877,5936,5937,5991,12473,12474,12475,12534,12535,12589,19071,19072,19073,19132,19133,19187,5880,5938,5939,5955,5992,6007,12478,12536,12537,12553,12590,12605,19076,19134,19135,19151,19188,19203,5881,5938,5940,5992,5993,6046,12479,12536,12538,12590,12591,12644,19077,19134,19136,19188,19189,19242,5884,5886,5936,5942,5991,5995,12482,12484,12534,12540,12589,12593,19080,19082,19132,19138,19187,19191,5885,5887,5941,5943,5994,5996,12483,12485,12539,12541,12592,12594,19081,19083,19137,19139,19190,19192,5888,5941,5944,5994,5997,6047,12486,12539,12542,12592,12595,12645,19084,19137,19140,19190,19193,19243,5889,5944,5945,5997,5998,12487,12542,12543,12595,12596,19085,19140,19141,19193,19194,5890,5946,5947,5999,6000,6051,12488,12544,12545,12597,12598,12649,19086,19142,19143,19195,19196,19247,5891,5892,5947,5948,6000,6001,12489,12490,12545,12546,12598,12599,19087,19088,19143,19144,19196,19197,5893,5949,5950,6002,6003,6055,12491,12547,12548,12600,12601,12653,19089,19145,19146,19198,19199,19251,5902,5957,5958,6011,6012,6063,12500,12555,12556,12609,12610,12661,19098,19153,19154,19207,19208,19259,5903,5957,5959,6011,6013,6064,12501,12555,12557,12609,12611,12662,19099,19153,19155,19207,19209,19260,5904,5959,5960,6013,6014,6066,12502,12557,12558,12611,12612,12664,19100,19155,19156,19209,19210,19262,5905,5960,5961,6014,6015,6067,12503,12558,12559,12612,12613,12665,19101,19156,19157,19210,19211,19263,5908,5961,5964,6015,6019,6068,12506,12559,12562,12613,12617,12666,19104,19157,19160,19211,19215,19264,5910,5911,5966,5967,6021,6022,12508,12509,12564,12565,12619,12620,19106,19107,19162,19163,19217,19218,5921,5926,5976,5981,6032,6037,12519,12524,12574,12579,12630,12635,19117,19122,19172,19177,19228,19233,5928,5929,5983,5984,6038,6039,6092,12526,12527,12581,12582,12636,12637,12690,19124,19125,19179,19180,19234,19235,19288,5931,5932,5986,5987,6041,6042,12529,12530,12584,12585,12639,12640,19127,19128,19182,19183,19237,19238,5933,5987,5988,6042,6043,6096,6097,12531,12585,12586,12640,12641,12694,12695,19129,19183,19184,19238,19239,19292,19293,5934,5988,5989,6043,6044,6098,12532,12586,12587,12641,12642,12696,19130,19184,19185,19239,19240,19294,5935,5989,5990,6044,6045,6099,12533,12587,12588,12642,12643,12697,19131,19185,19186,19240,19241,19295,5937,5990,5991,5995,6045,6048,12535,12588,12589,12593,12643,12646,19133,19186,19187,19191,19241,19244,5942,5943,5995,5996,6048,6049,12540,12541,12593,12594,12646,12647,19138,19139,19191,19192,19244,19245,5945,5946,5998,5999,6050,6052,12543,12544,12596,12597,12648,12650,19141,19142,19194,19195,19246,19248,5948,5949,6001,6002,6053,6054,12546,12547,12599,12600,12651,12652,19144,19145,19197,19198,19249,19250,5950,5993,6003,6046,6056,6057,12548,12591,12601,12644,12654,12655,19146,19189,19199,19242,19252,19253,5951,6004,6005,6056,6057,6058,12549,12602,12603,12654,12655,12656,19147,19200,19201,19252,19253,19254,5952,6005,6006,6058,6059,6108,12550,12603,12604,12656,12657,12706,19148,19201,19202,19254,19255,19304,5953,5992,6004,6007,6046,6057,12551,12590,12602,12605,12644,12655,19149,19188,19200,19203,19242,19253,5954,6006,6008,6059,6060,6109,12552,12604,12606,12657,12658,12707,19150,19202,19204,19255,19256,19305,5956,6009,6010,6061,6062,6111,12554,12607,12608,12659,12660,12709,19152,19205,19206,19257,19258,19307,5958,6009,6012,6061,6065,6112,12556,12607,12610,12659,12663,12710,19154,19205,19208,19257,19261,19308,5962,6016,6017,6069,6070,6121,12560,12614,12615,12667,12668,12719,19158,19212,19213,19265,19266,19317,5963,6017,6018,6070,6071,6122,12561,12615,12616,12668,12669,12720,19159,19213,19214,19266,19267,19318,5964,6016,6019,6069,6072,6120,12562,12614,12617,12667,12670,12718,19160,19212,19215,19265,19268,19316,5965,6018,6020,6071,6073,6123,12563,12616,12618,12669,12671,12721,19161,19214,19216,19267,19269,19319,5966,6020,6021,6073,6074,6125,12564,12618,12619,12671,12672,12723,19162,19216,19217,19269,19270,19321,5967,5983,6022,6038,6075,6091,12565,12581,12620,12636,12673,12689,19163,19179,19218,19234,19271,19287,5968,6023,6024,6076,6077,6128,12566,12621,12622,12674,12675,12726,19164,19219,19220,19272,19273,19324,5969,5979,6024,6035,6077,6088,12567,12577,12622,12633,12675,12686,19165,19175,19220,19231,19273,19284,5970,6023,6025,6076,6078,6129,12568,12621,12623,12674,12676,12727,19166,19219,19221,19272,19274,19325,5971,6025,6026,6078,6079,6130,12569,12623,12624,12676,12677,12728,19167,19221,19222,19274,19275,19326,5972,6026,6027,6079,6080,6131,12570,12624,12625,12677,12678,12729,19168,19222,19223,19275,19276,19327,5973,6027,6028,6080,6081,6132,12571,12625,12626,12678,12679,12730,19169,19223,19224,19276,19277,19328,5974,6028,6029,6081,6082,6133,12572,12626,12627,12679,12680,12731,19170,19224,19225,19277,19278,19329,5975,6030,6031,6083,6084,6135,12573,12628,12629,12681,12682,12733,19171,19226,19227,19279,19280,19331,5976,6032,6033,6085,6086,6138,12574,12630,12631,12683,12684,12736,19172,19228,19229,19281,19282,19334,5977,6033,6034,6086,6087,6140,12575,12631,12632,12684,12685,12738,19173,19229,19230,19282,19283,19336,5978,6034,6035,6087,6088,6141,12576,12632,12633,12685,12686,12739,19174,19230,19231,19283,19284,19337,5980,5981,6036,6037,6089,6090,12578,12579,12634,12635,12687,12688,19176,19177,19232,19233,19285,19286,5982,6030,6036,6083,6089,6136,12580,12628,12634,12681,12687,12734,19178,19226,19232,19279,19285,19332,5984,5985,6039,6040,6093,6094,12582,12583,12637,12638,12691,12692,19180,19181,19235,19236,19289,19290,5986,6040,6041,6094,6095,6146,12584,12638,12639,12692,12693,12744,19182,19236,19237,19290,19291,19342,5994,5996,6047,6049,6100,6101,12592,12594,12645,12647,12698,12699,19190,19192,19243,19245,19296,19297,5997,5998,6047,6050,6100,6102,12595,12596,12645,12648,12698,12700,19193,19194,19243,19246,19296,19298,5999,6051,6052,6103,6104,6153,12597,12649,12650,12701,12702,12751,19195,19247,19248,19299,19300,19349,6000,6001,6051,6053,6103,6105,12598,12599,12649,12651,12701,12703,19196,19197,19247,19249,19299,19301,6002,6054,6055,6106,6107,6157,12600,12652,12653,12704,12705,12755,19198,19250,19251,19302,19303,19353,6003,6055,6056,6058,6107,6108,12601,12653,12654,12656,12705,12706,19199,19251,19252,19254,19303,19304,6008,6010,6060,6062,6110,6113,12606,12608,12658,12660,12708,12711,19204,19206,19256,19258,19306,19309,6011,6063,6064,6114,6115,6164,12609,12661,12662,12712,12713,12762,19207,19259,19260,19310,19311,19360,6012,6063,6065,6114,6116,6163,12610,12661,12663,12712,12714,12761,19208,19259,19261,19310,19312,19359,6013,6064,6066,6115,6117,6165,12611,12662,12664,12713,12715,12763,19209,19260,19262,19311,19313,19361,6014,6066,6067,6117,6118,6167,12612,12664,12665,12715,12716,12765,19210,19262,19263,19313,19314,19363,6015,6067,6068,6118,6119,6168,12613,12665,12666,12716,12717,12766,19211,19263,19264,19314,19315,19364,6019,6068,6072,6119,6124,6169,12617,12666,12670,12717,12722,12767,19215,19264,19268,19315,19320,19365,6021,6022,6074,6075,6126,6127,12619,12620,12672,12673,12724,12725,19217,19218,19270,19271,19322,19323,6029,6031,6082,6084,6134,6137,12627,12629,12680,12682,12732,12735,19225,19227,19278,19280,19330,19333,6032,6037,6085,6090,6139,6142,12630,12635,12683,12688,12737,12740,19228,19233,19281,19286,19335,19338,6038,6091,6092,6143,6144,6160,12636,12689,12690,12741,12742,12758,19234,19287,19288,19339,19340,19356,6039,6092,6093,6144,6145,6162,12637,12690,12691,12742,12743,12760,19235,19288,19289,19340,19341,19358,6041,6042,6095,6096,6147,6148,12639,12640,12693,12694,12745,12746,19237,19238,19291,19292,19343,19344,6043,6097,6098,6149,6150,12641,12695,12696,12747,12748,19239,19293,19294,19345,19346,6044,6098,6099,6150,6151,6198,12642,12696,12697,12748,12749,12796,19240,19294,19295,19346,19347,19394,6045,6048,6049,6099,6101,6151,12643,12646,12647,12697,12699,12749,19241,19244,19245,19295,19297,19347,6050,6052,6102,6104,6152,6154,12648,12650,12700,12702,12750,12752,19246,19248,19298,19300,19348,19350,6053,6054,6105,6106,6155,6156,12651,12652,12703,12704,12753,12754,19249,19250,19301,19302,19351,19352,6059,6107,6108,6109,6157,6158,12657,12705,12706,12707,12755,12756,19255,19303,19304,19305,19353,19354,6060,6109,6110,6158,6159,6202,12658,12707,12708,12756,12757,12800,19256,19305,19306,19354,19355,19398,6061,6111,6112,6143,6160,6161,12659,12709,12710,12741,12758,12759,19257,19307,19308,19339,19356,19357,6062,6111,6113,6144,6160,6162,12660,12709,12711,12742,12758,12760,19258,19307,19309,19340,19356,19358,6065,6112,6116,6161,6166,6177,12663,12710,12714,12759,12764,12775,19261,19308,19312,19357,19362,19373,6069,6120,6121,6170,6171,6211,12667,12718,12719,12768,12769,12809,19265,19316,19317,19366,19367,19407,6070,6121,6122,6171,6172,6212,12668,12719,12720,12769,12770,12810,19266,19317,19318,19367,19368,19408,6071,6122,6123,6172,6173,6213,12669,12720,12721,12770,12771,12811,19267,19318,19319,19368,19369,19409,6072,6120,6124,6170,6174,6210,12670,12718,12722,12768,12772,12808,19268,19316,19320,19366,19370,19406,6073,6123,6125,6173,6175,6214,12671,12721,12723,12771,12773,12812,19269,19319,19321,19369,19371,19410,6074,6125,6126,6175,6176,6216,12672,12723,12724,12773,12774,12814,19270,19321,19322,19371,19372,19412,6075,6091,6127,6143,6161,6177,12673,12689,12725,12741,12759,12775,19271,19287,19323,19339,19357,19373,6076,6128,6129,6178,6179,6217,12674,12726,12727,12776,12777,12815,19272,19324,19325,19374,19375,19413,6077,6088,6128,6141,6178,6191,12675,12686,12726,12739,12776,12789,19273,19284,19324,19337,19374,19387,6078,6129,6130,6179,6180,6218,12676,12727,12728,12777,12778,12816,19274,19325,19326,19375,19376,19414,6079,6130,6131,6180,6181,6219,12677,12728,12729,12778,12779,12817,19275,19326,19327,19376,19377,19415,6080,6131,6132,6181,6182,6220,12678,12729,12730,12779,12780,12818,19276,19327,19328,19377,19378,19416,6081,6132,6133,6182,6183,6221,12679,12730,12731,12780,12781,12819,19277,19328,19329,19378,19379,19417,6082,6133,6134,6183,6184,12680,12731,12732,12781,12782,19278,19329,19330,19379,19380,6083,6135,6136,6185,6186,6223,12681,12733,12734,12783,12784,12821,19279,19331,19332,19381,19382,19419,6084,6135,6137,6185,6187,6224,12682,12733,12735,12783,12785,12822,19280,19331,19333,19381,19383,19420,6085,6138,6139,6188,6189,6226,12683,12736,12737,12786,12787,12824,19281,19334,19335,19384,19385,19422,6086,6138,6140,6188,6190,6227,12684,12736,12738,12786,12788,12825,19282,19334,19336,19384,19386,19423,6087,6140,6141,6190,6191,6229,12685,12738,12739,12788,12789,12827,19283,19336,19337,19386,19387,19425,6089,6090,6136,6142,6186,6192,12687,12688,12734,12740,12784,12790,19285,19286,19332,19338,19382,19388,6093,6094,6145,6146,6193,6194,12691,12692,12743,12744,12791,12792,19289,19290,19341,19342,19389,19390,6095,6146,6147,6194,6195,6231,12693,12744,12745,12792,12793,12829,19291,19342,19343,19390,19391,19427,6096,6097,6148,6149,6196,6197,12694,12695,12746,12747,12794,12795,19292,19293,19344,19345,19392,19393,6100,6101,6102,6151,6152,6198,12698,12699,12700,12749,12750,12796,19296,19297,19298,19347,19348,19394,6103,6105,6153,6155,6199,6200,12701,12703,12751,12753,12797,12798,19299,19301,19349,19351,19395,19396,6104,6153,6154,6196,6197,6199,12702,12751,12752,12794,12795,12797,19300,19349,19350,19392,19393,19395,6106,6156,6157,6158,6201,6202,12704,12754,12755,12756,12799,12800,19302,19352,19353,19354,19397,19398,6110,6113,6145,6159,6162,6193,12708,12711,12743,12757,12760,12791,19306,19309,19341,19355,19358,19389,6114,6163,6164,6203,6204,6232,12712,12761,12762,12801,12802,12830,19310,19359,19360,19399,19400,19428,6115,6164,6165,6204,6205,6234,6236,12713,12762,12763,12802,12803,12832,12834,19311,19360,19361,19400,19401,19430,19432,6116,6163,6166,6203,6206,6233,12714,12761,12764,12801,12804,12831,19312,19359,19362,19399,19402,19429,6117,6165,6167,6205,6207,6235,12715,12763,12765,12803,12805,12833,19313,19361,19363,19401,19403,19431,6118,6167,6168,6207,6208,6237,12716,12765,12766,12805,12806,12835,19314,19363,19364,19403,19404,19433,6119,6168,6169,6208,6209,6238,12717,12766,12767,12806,12807,12836,19315,19364,19365,19404,19405,19434,6124,6169,6174,6209,6215,6239,12722,12767,12772,12807,12813,12837,19320,19365,19370,19405,19411,19435,6126,6127,6166,6176,6177,6206,12724,12725,12764,12774,12775,12804,19322,19323,19362,19372,19373,19402,6134,6137,6184,6187,6222,6225,12732,12735,12782,12785,12820,12823,19330,19333,19380,19383,19418,19421,6139,6142,6189,6192,6228,6230,12737,12740,12787,12790,12826,12828,19335,19338,19385,19388,19424,19426,6147,6148,6195,6196,6199,6200,12745,12746,12793,12794,12797,12798,19343,19344,19391,19392,19395,19396,6149,6150,6152,6154,6197,6198,12747,12748,12750,12752,12795,12796,19345,19346,19348,19350,19393,19394,6155,6156,6195,6200,6201,6231,12753,12754,12793,12798,12799,12829,19351,19352,19391,19396,19397,19427,6159,6193,6194,6201,6202,6231,12757,12791,12792,12799,12800,12829,19355,19389,19390,19397,19398,19427,6170,6210,6211,6240,6241,6270,12768,12808,12809,12838,12839,12868,19366,19406,19407,19436,19437,19466,6171,6211,6212,6241,6242,6271,12769,12809,12810,12839,12840,12869,19367,19407,19408,19437,19438,19467,6172,6212,6213,6242,6243,6272,12770,12810,12811,12840,12841,12870,19368,19408,19409,19438,19439,19468,6173,6213,6214,6243,6244,6273,12771,12811,12812,12841,12842,12871,19369,19409,19410,19439,19440,19469,6174,6210,6215,6240,6245,6269,12772,12808,12813,12838,12843,12867,19370,19406,19411,19436,19441,19465,6175,6214,6216,6244,6246,6274,12773,12812,12814,12842,12844,12872,19371,19410,19412,19440,19442,19470,6176,6206,6216,6233,6246,6262,12774,12804,12814,12831,12844,12860,19372,19402,19412,19429,19442,19458,6178,6191,6217,6229,6247,6259,12776,12789,12815,12827,12845,12857,19374,19387,19413,19425,19443,19455,6179,6217,6218,6247,6248,6276,12777,12815,12816,12845,12846,12874,19375,19413,19414,19443,19444,19472,6180,6218,6219,6248,6249,6277,12778,12816,12817,12846,12847,12875,19376,19414,19415,19444,19445,19473,6181,6219,6220,6249,6250,6278,12779,12817,12818,12847,12848,12876,19377,19415,19416,19445,19446,19474,6182,6220,6221,6250,6251,6279,12780,12818,12819,12848,12849,12877,19378,19416,19417,19446,19447,19475,6183,6184,6221,6222,6251,6252,6280,12781,12782,12819,12820,12849,12850,12878,19379,19380,19417,19418,19447,19448,19476,6185,6223,6224,6253,6254,6282,12783,12821,12822,12851,12852,12880,19381,19419,19420,19449,19450,19478,6186,6192,6223,6230,6253,6260,12784,12790,12821,12828,12851,12858,19382,19388,19419,19426,19449,19456,6187,6224,6225,6254,6255,6283,12785,12822,12823,12852,12853,12881,19383,19420,19421,19450,19451,19479,6188,6226,6227,6256,6257,6286,12786,12824,12825,12854,12855,12884,19384,19422,19423,19452,19453,19482,6189,6226,6228,6256,6258,6285,12787,12824,12826,12854,12856,12883,19385,19422,19424,19452,19454,19481,6190,6227,6229,6257,6259,6287,12788,12825,12827,12855,12857,12885,19386,19423,19425,19453,19455,19483,6203,6232,6233,6261,6262,6291,12801,12830,12831,12859,12860,12889,19399,19428,19429,19457,19458,19487,6204,6232,6234,6261,6263,6290,12802,12830,12832,12859,12861,12888,19400,19428,19430,19457,19459,19486,6205,6235,6236,6264,6265,6293,12803,12833,12834,12862,12863,12891,19401,19431,19432,19460,19461,19489,6207,6235,6237,6264,6266,6294,12805,12833,12835,12862,12864,12892,19403,19431,19433,19460,19462,19490,6208,6237,6238,6266,6267,6295,12806,12835,12836,12864,12865,12893,19404,19433,19434,19462,19463,19491,6209,6238,6239,6267,6268,6296,12807,12836,12837,12865,12866,12894,19405,19434,19435,19463,19464,19492,6215,6239,6245,6268,6275,6297,12813,12837,12843,12866,12873,12895,19411,19435,19441,19464,19471,19493,6222,6225,6252,6255,6281,6284,12820,12823,12850,12853,12879,12882,19418,19421,19448,19451,19477,19480,6228,6230,6258,6260,6288,6289,12826,12828,12856,12858,12886,12887,19424,19426,19454,19456,19484,19485,6234,6236,6263,6265,6292,12832,12834,12861,12863,12890,19430,19432,19459,19461,19488,6240,6269,6270,6298,6299,6327,12838,12867,12868,12896,12897,12925,19436,19465,19466,19494,19495,19523,6241,6270,6271,6299,6300,6329,12839,12868,12869,12897,12898,12927,19437,19466,19467,19495,19496,19525,6242,6271,6272,6300,6301,6330,12840,12869,12870,12898,12899,12928,19438,19467,19468,19496,19497,19526,6243,6272,6273,6301,6302,6331,12841,12870,12871,12899,12900,12929,19439,19468,19469,19497,19498,19527,6244,6273,6274,6302,6303,6332,12842,12871,12872,12900,12901,12930,19440,19469,19470,19498,19499,19528,6245,6269,6275,6298,6304,6328,12843,12867,12873,12896,12902,12926,19441,19465,19471,19494,19500,19524,6246,6262,6274,6291,6303,6320,12844,12860,12872,12889,12901,12918,19442,19458,19470,19487,19499,19516,6247,6259,6276,6287,6305,6316,12845,12857,12874,12885,12903,12914,19443,19455,19472,19483,19501,19512,6248,6276,6277,6305,6306,6334,12846,12874,12875,12903,12904,12932,19444,19472,19473,19501,19502,19530,6249,6277,6278,6306,6307,6335,12847,12875,12876,12904,12905,12933,19445,19473,19474,19502,19503,19531,6250,6278,6279,6307,6308,6336,12848,12876,12877,12905,12906,12934,19446,19474,19475,19503,19504,19532,6251,6279,6280,6308,6309,6337,12849,12877,12878,12906,12907,12935,19447,19475,19476,19504,19505,19533,6252,6280,6281,6309,6310,6338,12850,12878,12879,12907,12908,12936,19448,19476,19477,19505,19506,19534,6253,6260,6282,6289,6311,6318,12851,12858,12880,12887,12909,12916,19449,19456,19478,19485,19507,19514,6254,6282,6283,6311,6312,6340,12852,12880,12881,12909,12910,12938,19450,19478,19479,19507,19508,19536,6255,6283,6284,6312,6313,6341,12853,12881,12882,12910,12911,12939,19451,19479,19480,19508,19509,19537,6256,6285,6286,6314,6315,6343,12854,12883,12884,12912,12913,12941,19452,19481,19482,19510,19511,19539,6257,6286,6287,6315,6316,6344,12855,12884,12885,12913,12914,12942,19453,19482,19483,19511,19512,19540,6258,6285,6288,6314,6317,6342,12856,12883,12886,12912,12915,12940,19454,19481,19484,19510,19513,19538,6261,6290,6291,6319,6320,6347,12859,12888,12889,12917,12918,12945,19457,19486,19487,19515,19516,19543,6263,6290,6292,6319,6321,6348,12861,12888,12890,12917,12919,12946,19459,19486,19488,19515,19517,19544,6264,6293,6294,6322,6323,6350,12862,12891,12892,12920,12921,12948,19460,19489,19490,19518,19519,19546,6265,6292,6293,6321,6322,6349,12863,12890,12891,12919,12920,12947,19461,19488,19489,19517,19518,19545,6266,6294,6295,6323,6324,6351,12864,12892,12893,12921,12922,12949,19462,19490,19491,19519,19520,19547,6267,6295,6296,6324,6325,6352,12865,12893,12894,12922,12923,12950,19463,19491,19492,19520,19521,19548,6268,6296,6297,6325,6326,6353,12866,12894,12895,12923,12924,12951,19464,19492,19493,19521,19522,19549,6275,6297,6304,6326,6333,6354,12873,12895,12902,12924,12931,12952,19471,19493,19500,19522,19529,19550,6281,6284,6310,6313,6339,12879,12882,12908,12911,12937,19477,19480,19506,19509,19535,6288,6289,6317,6318,6345,6346,12886,12887,12915,12916,12943,12944,19484,19485,19513,19514,19541,19542,6298,6327,6328,6355,6356,6383,12896,12925,12926,12953,12954,12981,19494,19523,19524,19551,19552,19579,6299,6327,6329,6355,6357,6384,12897,12925,12927,12953,12955,12982,19495,19523,19525,19551,19553,19580,6300,6329,6330,6357,6358,6386,12898,12927,12928,12955,12956,12984,19496,19525,19526,19553,19554,19582,6301,6330,6331,6358,6359,6387,12899,12928,12929,12956,12957,12985,19497,19526,19527,19554,19555,19583,6302,6331,6332,6359,6360,6388,12900,12929,12930,12957,12958,12986,19498,19527,19528,19555,19556,19584,6303,6320,6332,6347,6360,6375,12901,12918,12930,12945,12958,12973,19499,19516,19528,19543,19556,19571,6304,6328,6333,6356,6361,6385,12902,12926,12931,12954,12959,12983,19500,19524,19529,19552,19557,19581,6305,6316,6334,6344,6362,6372,12903,12914,12932,12942,12960,12970,19501,19512,19530,19540,19558,19568,6306,6334,6335,6362,6363,6390,12904,12932,12933,12960,12961,12988,19502,19530,19531,19558,19559,19586,6307,6335,6336,6363,6364,6391,12905,12933,12934,12961,12962,12989,19503,19531,19532,19559,19560,19587,6308,6336,6337,6364,6365,6392,12906,12934,12935,12962,12963,12990,19504,19532,19533,19560,19561,19588,6309,6337,6338,6365,6366,6393,12907,12935,12936,12963,12964,12991,19505,19533,19534,19561,19562,19589,6310,6338,6339,6366,6367,6394,12908,12936,12937,12964,12965,12992,19506,19534,19535,19562,19563,19590,6311,6318,6340,6346,6368,6374,12909,12916,12938,12944,12966,12972,19507,19514,19536,19542,19564,19570,6312,6340,6341,6368,6369,6396,12910,12938,12939,12966,12967,12994,19508,19536,19537,19564,19565,19592,6313,6339,6341,6367,6369,6395,12911,12937,12939,12965,12967,12993,19509,19535,19537,19563,19565,19591,6314,6342,6343,6370,6371,6398,12912,12940,12941,12968,12969,12996,19510,19538,19539,19566,19567,19594,6315,6343,6344,6371,6372,6399,12913,12941,12942,12969,12970,12997,19511,19539,19540,19567,19568,19595,6317,6342,6345,6370,6373,6397,12915,12940,12943,12968,12971,12995,19513,19538,19541,19566,19569,19593,6319,6347,6348,6375,6376,6402,12917,12945,12946,12973,12974,13000,19515,19543,19544,19571,19572,19598,6321,6348,6349,6376,6377,6403,12919,12946,12947,12974,12975,13001,19517,19544,19545,19572,19573,19599,6322,6349,6350,6377,6378,6404,12920,12947,12948,12975,12976,13002,19518,19545,19546,19573,19574,19600,6323,6350,6351,6378,6379,6405,12921,12948,12949,12976,12977,13003,19519,19546,19547,19574,19575,19601,6324,6351,6352,6379,6380,6406,12922,12949,12950,12977,12978,13004,19520,19547,19548,19575,19576,19602,6325,6352,6353,6380,6381,6407,12923,12950,12951,12978,12979,13005,19521,19548,19549,19576,19577,19603,6326,6353,6354,6381,6382,6408,12924,12951,12952,12979,12980,13006,19522,19549,19550,19577,19578,19604,6333,6354,6361,6382,6389,6409,12931,12952,12959,12980,12987,13007,19529,19550,19557,19578,19585,19605,6345,6346,6373,6374,6400,6401,12943,12944,12971,12972,12998,12999,19541,19542,19569,19570,19596,19597,6355,6383,6384,6410,6411,6438,12953,12981,12982,13008,13009,13036,19551,19579,19580,19606,19607,19634,6356,6383,6385,6410,6412,6437,12954,12981,12983,13008,13010,13035,19552,19579,19581,19606,19608,19633,6357,6384,6386,6411,6413,6439,12955,12982,12984,13009,13011,13037,19553,19580,19582,19607,19609,19635,6358,6386,6387,6413,6414,6441,12956,12984,12985,13011,13012,13039,19554,19582,19583,19609,19610,19637,6359,6387,6388,6414,6415,6442,12957,12985,12986,13012,13013,13040,19555,19583,19584,19610,19611,19638,6360,6375,6388,6402,6415,6429,12958,12973,12986,13000,13013,13027,19556,19571,19584,19598,19611,19625,6361,6385,6389,6412,6416,6440,12959,12983,12987,13010,13014,13038,19557,19581,19585,19608,19612,19636,6362,6372,6390,6399,6417,6426,12960,12970,12988,12997,13015,13024,19558,19568,19586,19595,19613,19622,6363,6390,6391,6417,6418,6444,12961,12988,12989,13015,13016,13042,19559,19586,19587,19613,19614,19640,6364,6391,6392,6418,6419,6445,12962,12989,12990,13016,13017,13043,19560,19587,19588,19614,19615,19641,6365,6392,6393,6419,6420,6446,12963,12990,12991,13017,13018,13044,19561,19588,19589,19615,19616,19642,6366,6393,6394,6420,6421,12964,12991,12992,13018,13019,19562,19589,19590,19616,19617,6367,6394,6395,6421,6422,6447,12965,12992,12993,13019,13020,13045,19563,19590,19591,19617,19618,19643,6368,6374,6396,6401,6423,6428,12966,12972,12994,12999,13021,13026,19564,19570,19592,19597,19619,19624,6369,6395,6396,6422,6423,6448,12967,12993,12994,13020,13021,13046,19565,19591,19592,19618,19619,19644,6370,6397,6398,6424,6425,6449,12968,12995,12996,13022,13023,13047,19566,19593,19594,19620,19621,19645,6371,6398,6399,6425,6426,12969,12996,12997,13023,13024,19567,19594,19595,19621,19622,6373,6397,6400,6424,6427,12971,12995,12998,13022,13025,19569,19593,19596,19620,19623,6376,6402,6403,6429,6430,6452,12974,13000,13001,13027,13028,13050,19572,19598,19599,19625,19626,19648,6377,6403,6404,6430,6431,6453,12975,13001,13002,13028,13029,13051,19573,19599,19600,19626,19627,19649,6378,6404,6405,6431,6432,6454,12976,13002,13003,13029,13030,13052,19574,19600,19601,19627,19628,19650,6379,6405,6406,6432,6433,6455,12977,13003,13004,13030,13031,13053,19575,19601,19602,19628,19629,19651,6380,6406,6407,6433,6434,6456,12978,13004,13005,13031,13032,13054,19576,19602,19603,19629,19630,19652,6381,6407,6408,6434,6435,6457,12979,13005,13006,13032,13033,13055,19577,19603,19604,19630,19631,19653,6382,6408,6409,6435,6436,6458,12980,13006,13007,13033,13034,13056,19578,19604,19605,19631,19632,19654,6389,6409,6416,6436,6443,6459,12987,13007,13014,13034,13041,13057,19585,19605,19612,19632,19639,19655,6400,6401,6427,6428,6450,6451,12998,12999,13025,13026,13048,13049,19596,19597,19623,19624,19646,19647,6410,6437,6438,6460,6461,6483,13008,13035,13036,13058,13059,13081,19606,19633,19634,19656,19657,19679,6411,6438,6439,6461,6462,6484,13009,13036,13037,13059,13060,13082,19607,19634,19635,19657,19658,19680,6412,6437,6440,6460,6463,6482,13010,13035,13038,13058,13061,13080,19608,19633,19636,19656,19659,19678,6413,6439,6441,6462,6464,6485,13011,13037,13039,13060,13062,13083,19609,19635,19637,19658,19660,19681,6414,6441,6442,6464,6465,6487,13012,13039,13040,13062,13063,13085,19610,19637,19638,19660,19661,19683,6415,6429,6442,6452,6465,6474,13013,13027,13040,13050,13063,13072,19611,19625,19638,19648,19661,19670,6416,6440,6443,6463,6466,6486,13014,13038,13041,13061,13064,13084,19612,19636,19639,19659,19662,19682,6417,6425,6426,6444,6449,6467,13015,13023,13024,13042,13047,13065,19613,19621,19622,19640,19645,19663,6418,6444,6445,6467,6468,6472,13016,13042,13043,13065,13066,13070,19614,19640,19641,19663,19664,19668,6419,6445,6446,6468,6469,6489,13017,13043,13044,13066,13067,13087,19615,19641,19642,19664,19665,19685,6420,6421,6446,6447,6469,6470,13018,13019,13044,13045,13067,13068,19616,19617,19642,19643,19665,19666,6422,6447,6448,6470,6471,13020,13045,13046,13068,13069,19618,19643,19644,19666,19667,6423,6428,6448,6451,6471,6473,13021,13026,13046,13049,13069,13071,19619,19624,19644,19647,19667,19669,6424,6427,6449,6450,6467,6472,13022,13025,13047,13048,13065,13070,19620,19623,19645,19646,19663,19668,6430,6452,6453,6474,6475,6490,13028,13050,13051,13072,13073,13088,19626,19648,19649,19670,19671,19686,6431,6453,6454,6475,6476,6491,13029,13051,13052,13073,13074,13089,19627,19649,19650,19671,19672,19687,6432,6454,6455,6476,6477,6492,13030,13052,13053,13074,13075,13090,19628,19650,19651,19672,19673,19688,6433,6455,6456,6477,6478,6493,13031,13053,13054,13075,13076,13091,19629,19651,19652,19673,19674,19689,6434,6456,6457,6478,6479,6494,13032,13054,13055,13076,13077,13092,19630,19652,19653,19674,19675,19690,6435,6457,6458,6479,6480,6495,13033,13055,13056,13077,13078,13093,19631,19653,19654,19675,19676,19691,6436,6458,6459,6480,6481,6496,13034,13056,13057,13078,13079,13094,19632,19654,19655,19676,19677,19692,6443,6459,6466,6481,6488,6497,13041,13057,13064,13079,13086,13095,19639,19655,19662,19677,19684,19693,6450,6451,6468,6472,6473,6489,13048,13049,13066,13070,13071,13087,19646,19647,19664,19668,19669,19685,6460,6482,6483,6498,6499,6513,13058,13080,13081,13096,13097,13111,19656,19678,19679,19694,19695,19709,6461,6483,6484,6499,6500,6515,13059,13081,13082,13097,13098,13113,19657,19679,19680,19695,19696,19711,6462,6484,6485,6500,6501,6516,13060,13082,13083,13098,13099,13114,19658,19680,19681,19696,19697,19712,6463,6482,6486,6498,6502,6514,13061,13080,13084,13096,13100,13112,19659,19678,19682,19694,19698,19710,6464,6485,6487,6501,6503,6517,13062,13083,13085,13099,13101,13115,19660,19681,19683,19697,19699,19713,6465,6474,6487,6490,6503,6505,13063,13072,13085,13088,13101,13103,19661,19670,19683,19686,19699,19701,6466,6486,6488,6502,6504,6518,13064,13084,13086,13100,13102,13116,19662,19682,19684,19698,19700,19714,6469,6470,6471,6473,6489,13067,13068,13069,13071,13087,19665,19666,19667,19669,19685,6475,6490,6491,6505,6506,6520,13073,13088,13089,13103,13104,13118,19671,19686,19687,19701,19702,19716,6476,6491,6492,6506,6507,6521,13074,13089,13090,13104,13105,13119,19672,19687,19688,19702,19703,19717,6477,6492,6493,6507,6508,6522,13075,13090,13091,13105,13106,13120,19673,19688,19689,19703,19704,19718,6478,6493,6494,6508,6509,6523,13076,13091,13092,13106,13107,13121,19674,19689,19690,19704,19705,19719,6479,6494,6495,6509,6510,6524,13077,13092,13093,13107,13108,13122,19675,19690,19691,19705,19706,19720,6480,6495,6496,6510,6511,13078,13093,13094,13108,13109,19676,19691,19692,19706,19707,6481,6496,6497,6511,6512,6525,13079,13094,13095,13109,13110,13123,19677,19692,19693,19707,19708,19721,6488,6497,6504,6512,6519,6526,13086,13095,13102,13110,13117,13124,19684,19693,19700,19708,19715,19722,6498,6513,6514,6527,6528,6541,13096,13111,13112,13125,13126,13139,19694,19709,19710,19723,19724,19737,6499,6513,6515,6527,6529,6542,13097,13111,13113,13125,13127,13140,19695,19709,19711,19723,19725,19738,6500,6515,6516,6529,6530,6544,13098,13113,13114,13127,13128,13142,19696,19711,19712,19725,19726,19740,6501,6516,6517,6530,6531,6545,13099,13114,13115,13128,13129,13143,19697,19712,19713,19726,19727,19741,6502,6514,6518,6528,6532,6543,13100,13112,13116,13126,13130,13141,19698,19710,19714,19724,19728,19739,6503,6505,6517,6520,6531,6534,13101,13103,13115,13118,13129,13132,19699,19701,19713,19716,19727,19730,6504,6518,6519,6532,6533,6546,13102,13116,13117,13130,13131,13144,19700,19714,19715,19728,19729,19742,6506,6520,6521,6534,6535,6548,13104,13118,13119,13132,13133,13146,19702,19716,19717,19730,19731,19744,6507,6521,6522,6535,6536,6549,13105,13119,13120,13133,13134,13147,19703,19717,19718,19731,19732,19745,6508,6522,6523,6536,6537,6550,13106,13120,13121,13134,13135,13148,19704,19718,19719,19732,19733,19746,6509,6523,6524,6537,6538,6551,13107,13121,13122,13135,13136,13149,19705,19719,19720,19733,19734,19747,6510,6511,6524,6525,6538,6539,13108,13109,13122,13123,13136,13137,19706,19707,19720,19721,19734,19735,6512,6525,6526,6539,6540,6552,13110,13123,13124,13137,13138,13150,19708,19721,19722,19735,19736,19748,6519,6526,6533,6540,6547,13117,13124,13131,13138,13145,19715,19722,19729,19736,19743,6527,6541,6542,6553,6554,6565,13125,13139,13140,13151,13152,13163,19723,19737,19738,19749,19750,19761,6528,6541,6543,6553,6555,6566,13126,13139,13141,13151,13153,13164,19724,19737,19739,19749,19751,19762,6529,6542,6544,6554,6556,6567,13127,13140,13142,13152,13154,13165,19725,19738,19740,19750,19752,19763,6530,6544,6545,6556,6557,6569,13128,13142,13143,13154,13155,13167,19726,19740,19741,19752,19753,19765,6531,6534,6545,6548,6557,6560,13129,13132,13143,13146,13155,13158,19727,19730,19741,19744,19753,19756,6532,6543,6546,6555,6558,6568,13130,13141,13144,13153,13156,13166,19728,19739,19742,19751,19754,19764,6533,6546,6547,6558,6559,6570,13131,13144,13145,13156,13157,13168,19729,19742,19743,19754,19755,19766,6535,6548,6549,6560,6561,6572,13133,13146,13147,13158,13159,13170,19731,19744,19745,19756,19757,19768,6536,6549,6550,6561,6562,6573,13134,13147,13148,13159,13160,13171,19732,19745,19746,19757,19758,19769,6537,6550,6551,6562,6563,6574,13135,13148,13149,13160,13161,13172,19733,19746,19747,19758,19759,19770,6538,6539,6551,6552,6563,6564,13136,13137,13149,13150,13161,13162,19734,19735,19747,19748,19759,19760,6540,6547,6552,6559,6564,6571,13138,13145,13150,13157,13162,13169,19736,19743,19748,19755,19760,19767,6553,6565,6566,6575,6576,6585,13151,13163,13164,13173,13174,13183,19749,19761,19762,19771,19772,19781,6554,6565,6567,6575,6577,6586,13152,13163,13165,13173,13175,13184,19750,19761,19763,19771,19773,19782,6555,6566,6568,6576,6578,6587,13153,13164,13166,13174,13176,13185,19751,19762,19764,19772,19774,19783,6556,6567,6569,6577,6579,6588,13154,13165,13167,13175,13177,13186,19752,19763,19765,19773,19775,19784,6557,6560,6569,6572,6579,6582,13155,13158,13167,13170,13177,13180,19753,19756,19765,19768,19775,19778,6558,6568,6570,6578,6580,13156,13166,13168,13176,13178,19754,19764,19766,19774,19776,6559,6570,6571,6580,6581,6589,13157,13168,13169,13178,13179,13187,19755,19766,19767,19776,19777,19785,6561,6572,6573,6582,6583,6590,13159,13170,13171,13180,13181,13188,19757,19768,19769,19778,19779,19786,6562,6573,6574,6583,6584,6591,13160,13171,13172,13181,13182,13189,19758,19769,19770,19779,19780,19787,6563,6564,6571,6574,6581,6584,13161,13162,13169,13172,13179,13182,19759,19760,19767,19770,19777,19780,6575,6585,6586,6592,6593,13173,13183,13184,13190,13191,19771,19781,19782,19788,19789,6576,6585,6587,6592,6594,13174,13183,13185,13190,13192,19772,19781,19783,19788,19790,6577,6586,6588,6593,6595,6598,13175,13184,13186,13191,13193,13196,19773,19782,19784,19789,19791,19794,6578,6580,6587,6589,6594,6596,13176,13178,13185,13187,13192,13194,19774,19776,19783,19785,19790,19792,6579,6582,6588,6590,6595,13177,13180,13186,13188,13193,19775,19778,19784,19786,19791,6581,6584,6589,6591,6596,6597,13179,13182,13187,13189,13194,13195,19777,19780,19785,19787,19792,19793,6583,6590,6591,6595,6597,6598,13181,13188,13189,13193,13195,13196,19779,19786,19787,19791,19793,19794,6592,6593,6594,6596,6597,6598,13190,13191,13192,13194,13195,13196,19788,19789,19790,19792,19793,19794])
        J = np.array([1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,22,22,22,22,22,22,22,22,22,22,22,22,22,22,22,22,22,22,23,23,23,23,23,23,23,23,23,23,23,23,23,23,23,23,23,23,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,25,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,27,27,27,27,27,27,27,27,27,27,27,27,27,27,27,27,27,27,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,29,29,29,29,29,29,29,29,29,29,29,29,29,29,29,29,29,29,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,31,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,34,34,34,34,34,34,34,34,34,34,34,34,34,34,34,34,34,34,35,35,35,35,35,35,35,35,35,35,35,35,35,35,35,35,35,35,36,36,36,36,36,36,36,36,36,36,36,36,36,36,36,36,36,36,37,37,37,37,37,37,37,37,37,37,37,37,37,37,37,37,37,37,38,38,38,38,38,38,38,38,38,38,38,38,38,38,38,38,38,38,39,39,39,39,39,39,39,39,39,39,39,39,39,39,39,39,39,39,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,41,41,41,41,41,41,41,41,41,41,41,41,41,41,41,41,41,41,42,42,42,42,42,42,42,42,42,42,42,42,42,42,42,42,42,42,43,43,43,43,43,43,43,43,43,43,43,43,43,43,43,43,43,43,44,44,44,44,44,44,44,44,44,44,44,44,44,44,44,44,44,44,45,45,45,45,45,45,45,45,45,45,45,45,45,45,45,45,45,45,46,46,46,46,46,46,46,46,46,46,46,46,46,46,46,46,46,46,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51,52,52,52,52,52,52,52,52,52,52,52,52,52,52,52,52,52,52,53,53,53,53,53,53,53,53,53,53,53,53,53,53,53,53,53,53,54,54,54,54,54,54,54,54,54,54,54,54,54,54,54,54,54,54,55,55,55,55,55,55,55,55,55,55,55,55,55,55,55,55,55,55,56,56,56,56,56,56,56,56,56,56,56,56,56,56,56,56,56,56,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,57,58,58,58,58,58,58,58,58,58,58,58,58,58,58,58,58,58,58,59,59,59,59,59,59,59,59,59,59,59,59,59,59,59,59,59,59,60,60,60,60,60,60,60,60,60,60,60,60,60,60,60,60,60,60,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,61,62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,63,63,63,63,63,63,63,63,63,63,63,63,63,63,63,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,65,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,66,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,68,68,68,68,68,68,68,68,68,68,68,68,68,68,68,68,68,68,69,69,69,69,69,69,69,69,69,69,69,69,69,69,69,69,69,69,70,70,70,70,70,70,70,70,70,70,70,70,70,70,70,70,70,70,71,71,71,71,71,71,71,71,71,71,71,71,71,71,71,71,71,71,72,72,72,72,72,72,72,72,72,72,72,72,72,72,72,72,72,72,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,74,74,74,74,74,74,74,74,74,74,74,74,74,74,74,74,74,74,75,75,75,75,75,75,75,75,75,75,75,75,75,75,75,75,75,75,76,76,76,76,76,76,76,76,76,76,76,76,76,76,76,76,76,76,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,79,79,79,79,79,79,79,79,79,79,79,79,79,79,79,79,79,79,80,80,80,80,80,80,80,80,80,80,80,80,80,80,80,80,80,80,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,82,82,82,82,82,82,82,82,82,82,82,82,82,82,82,82,82,82,83,83,83,83,83,83,83,83,83,83,83,83,83,83,83,83,83,83,84,84,84,84,84,84,84,84,84,84,84,84,84,84,84,84,84,84,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,86,86,86,86,86,86,86,86,86,86,86,86,86,86,86,87,87,87,87,87,87,87,87,87,87,87,87,87,87,87,87,87,87,88,88,88,88,88,88,88,88,88,88,88,88,88,88,88,88,88,88,89,89,89,89,89,89,89,89,89,89,89,89,89,89,89,89,89,89,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,91,92,92,92,92,92,92,92,92,92,92,92,92,92,92,92,92,92,92,93,93,93,93,93,93,93,93,93,93,93,93,93,93,93,93,93,93,94,94,94,94,94,94,94,94,94,94,94,94,94,94,94,94,94,94,95,95,95,95,95,95,95,95,95,95,95,95,95,95,95,96,96,96,96,96,96,96,96,96,96,96,96,96,96,96,96,96,96,97,97,97,97,97,97,97,97,97,97,97,97,97,97,97,97,97,97,98,98,98,98,98,98,98,98,98,98,98,98,98,98,98,98,98,98,99,99,99,99,99,99,99,99,99,99,99,99,99,99,99,99,99,99,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,100,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,102,102,102,102,102,102,102,102,102,102,102,102,102,102,102,102,102,102,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,103,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,104,105,105,105,105,105,105,105,105,105,105,105,105,105,105,105,105,105,105,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,106,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,107,108,108,108,108,108,108,108,108,108,108,108,108,108,108,108,108,108,108,109,109,109,109,109,109,109,109,109,109,109,109,109,109,109,110,110,110,110,110,110,110,110,110,110,110,110,110,110,110,110,110,110,111,111,111,111,111,111,111,111,111,111,111,111,111,111,111,111,111,111,112,112,112,112,112,112,112,112,112,112,112,112,112,112,112,112,112,112,113,113,113,113,113,113,113,113,113,113,113,113,113,113,113,113,113,113,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,114,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,115,116,116,116,116,116,116,116,116,116,116,116,116,116,116,116,116,116,116,117,117,117,117,117,117,117,117,117,117,117,117,117,117,117,117,117,117,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,120,120,120,120,120,120,120,120,120,120,120,120,120,120,120,120,120,120,121,121,121,121,121,121,121,121,121,121,121,121,121,121,121,121,121,121,122,122,122,122,122,122,122,122,122,122,122,122,122,122,122,122,122,122,123,123,123,123,123,123,123,123,123,123,123,123,123,123,123,123,123,123,124,124,124,124,124,124,124,124,124,124,124,124,124,124,124,124,124,124,125,125,125,125,125,125,125,125,125,125,125,125,125,125,125,125,125,125,126,126,126,126,126,126,126,126,126,126,126,126,126,126,126,126,126,126,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,129,129,129,129,129,129,129,129,129,129,129,129,129,129,129,129,129,129,130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,131,131,131,131,131,131,131,131,131,131,131,131,131,131,131,131,131,131,132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,135,135,135,135,135,135,135,135,135,135,135,135,135,135,135,135,135,135,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,137,137,137,137,137,137,137,137,137,137,137,137,137,137,137,137,137,137,138,138,138,138,138,138,138,138,138,138,138,138,138,138,138,138,138,138,139,139,139,139,139,139,139,139,139,139,139,139,139,139,139,139,139,139,140,140,140,140,140,140,140,140,140,140,140,140,140,140,140,140,140,140,141,141,141,141,141,141,141,141,141,141,141,141,141,141,141,141,141,141,142,142,142,142,142,142,142,142,142,142,142,142,142,142,142,142,142,142,143,143,143,143,143,143,143,143,143,143,143,143,143,143,143,143,143,143,144,144,144,144,144,144,144,144,144,144,144,144,144,144,144,144,144,144,145,145,145,145,145,145,145,145,145,145,145,145,145,145,145,145,145,145,146,146,146,146,146,146,146,146,146,146,146,146,146,146,146,146,146,146,147,147,147,147,147,147,147,147,147,147,147,147,147,147,147,147,147,147,148,148,148,148,148,148,148,148,148,148,148,148,148,148,148,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,149,150,150,150,150,150,150,150,150,150,150,150,150,150,150,150,150,150,150,151,151,151,151,151,151,151,151,151,151,151,151,151,151,151,151,151,151,152,152,152,152,152,152,152,152,152,152,152,152,152,152,152,152,152,152,153,153,153,153,153,153,153,153,153,153,153,153,153,153,153,153,153,153,154,154,154,154,154,154,154,154,154,154,154,154,154,154,154,154,154,154,155,155,155,155,155,155,155,155,155,155,155,155,155,155,155,155,155,155,156,156,156,156,156,156,156,156,156,156,156,156,156,156,156,156,156,156,157,157,157,157,157,157,157,157,157,157,157,157,157,157,157,157,157,157,158,158,158,158,158,158,158,158,158,158,158,158,158,158,158,158,158,158,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,159,160,160,160,160,160,160,160,160,160,160,160,160,160,160,160,160,160,160,161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,162,162,162,162,162,162,162,162,162,162,162,162,162,162,162,162,162,162,163,163,163,163,163,163,163,163,163,163,163,163,163,163,163,163,163,163,164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,165,165,165,165,165,165,165,165,165,165,165,165,165,165,165,165,165,165,166,166,166,166,166,166,166,166,166,166,166,166,166,166,166,166,166,166,167,167,167,167,167,167,167,167,167,167,167,167,167,167,167,167,167,167,168,168,168,168,168,168,168,168,168,168,168,168,168,168,168,168,168,168,169,169,169,169,169,169,169,169,169,169,169,169,169,169,169,169,169,169,170,170,170,170,170,170,170,170,170,170,170,170,170,170,170,170,170,170,171,171,171,171,171,171,171,171,171,171,171,171,171,171,171,171,171,171,172,172,172,172,172,172,172,172,172,172,172,172,172,172,172,172,172,172,173,173,173,173,173,173,173,173,173,173,173,173,173,173,173,173,173,173,174,174,174,174,174,174,174,174,174,174,174,174,174,174,174,174,174,174,175,175,175,175,175,175,175,175,175,175,175,175,175,175,175,175,175,175,176,176,176,176,176,176,176,176,176,176,176,176,176,176,176,176,176,176,177,177,177,177,177,177,177,177,177,177,177,177,177,177,177,177,177,177,178,178,178,178,178,178,178,178,178,178,178,178,178,178,178,178,178,178,179,179,179,179,179,179,179,179,179,179,179,179,179,179,179,179,179,179,180,180,180,180,180,180,180,180,180,180,180,180,180,180,180,180,180,180,181,181,181,181,181,181,181,181,181,181,181,181,181,181,181,181,181,181,182,182,182,182,182,182,182,182,182,182,182,182,182,182,182,182,182,182,183,183,183,183,183,183,183,183,183,183,183,183,183,183,183,183,183,183,184,184,184,184,184,184,184,184,184,184,184,184,184,184,184,184,184,184,185,185,185,185,185,185,185,185,185,185,185,185,185,185,185,185,185,185,186,186,186,186,186,186,186,186,186,186,186,186,186,186,186,186,186,186,187,187,187,187,187,187,187,187,187,187,187,187,187,187,187,187,187,187,188,188,188,188,188,188,188,188,188,188,188,188,188,188,188,188,188,188,189,189,189,189,189,189,189,189,189,189,189,189,189,189,189,189,189,189,190,190,190,190,190,190,190,190,190,190,190,190,190,190,190,190,190,190,191,191,191,191,191,191,191,191,191,191,191,191,191,191,191,191,191,191,192,192,192,192,192,192,192,192,192,192,192,192,192,192,192,192,192,192,193,193,193,193,193,193,193,193,193,193,193,193,193,193,193,193,193,193,194,194,194,194,194,194,194,194,194,194,194,194,194,194,194,194,194,194,195,195,195,195,195,195,195,195,195,195,195,195,195,195,195,195,195,195,196,196,196,196,196,196,196,196,196,196,196,196,196,196,196,196,196,196,197,197,197,197,197,197,197,197,197,197,197,197,197,197,197,197,197,197,198,198,198,198,198,198,198,198,198,198,198,198,198,198,198,198,198,198,199,199,199,199,199,199,199,199,199,199,199,199,199,199,199,199,199,199,200,200,200,200,200,200,200,200,200,200,200,200,200,200,200,200,200,200,201,201,201,201,201,201,201,201,201,201,201,201,201,201,201,201,201,201,202,202,202,202,202,202,202,202,202,202,202,202,202,202,202,202,202,202,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,204,204,204,204,204,204,204,204,204,204,204,204,204,204,204,204,204,204,205,205,205,205,205,205,205,205,205,205,205,205,205,205,205,205,205,205,206,206,206,206,206,206,206,206,206,206,206,206,206,206,206,206,206,206,207,207,207,207,207,207,207,207,207,207,207,207,207,207,207,207,207,207,208,208,208,208,208,208,208,208,208,208,208,208,208,208,208,208,208,208,209,209,209,209,209,209,209,209,209,209,209,209,209,209,209,209,209,209,210,210,210,210,210,210,210,210,210,210,210,210,210,210,210,210,210,210,211,211,211,211,211,211,211,211,211,211,211,211,211,211,211,211,211,211,212,212,212,212,212,212,212,212,212,212,212,212,212,212,212,212,212,212,213,213,213,213,213,213,213,213,213,213,213,213,213,213,213,213,213,213,214,214,214,214,214,214,214,214,214,214,214,214,214,214,214,214,214,214,215,215,215,215,215,215,215,215,215,215,215,215,215,215,215,215,215,215,216,216,216,216,216,216,216,216,216,216,216,216,216,216,216,216,216,216,217,217,217,217,217,217,217,217,217,217,217,217,217,217,217,217,217,217,218,218,218,218,218,218,218,218,218,218,218,218,218,218,218,218,218,218,219,219,219,219,219,219,219,219,219,219,219,219,219,219,219,219,219,219,220,220,220,220,220,220,220,220,220,220,220,220,220,220,220,220,220,220,221,221,221,221,221,221,221,221,221,221,221,221,221,221,221,221,221,221,222,222,222,222,222,222,222,222,222,222,222,222,222,222,222,222,222,222,223,223,223,223,223,223,223,223,223,223,223,223,223,223,223,223,223,223,224,224,224,224,224,224,224,224,224,224,224,224,224,224,224,224,224,224,225,225,225,225,225,225,225,225,225,225,225,225,225,225,225,225,225,225,226,226,226,226,226,226,226,226,226,226,226,226,226,226,226,226,226,226,227,227,227,227,227,227,227,227,227,227,227,227,227,227,227,227,227,227,228,228,228,228,228,228,228,228,228,228,228,228,228,228,228,228,228,228,229,229,229,229,229,229,229,229,229,229,229,229,229,229,229,229,229,229,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,231,231,231,231,231,231,231,231,231,231,231,231,231,231,231,231,231,231,232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,233,233,233,233,233,233,233,233,233,233,233,233,233,233,233,233,233,233,234,234,234,234,234,234,234,234,234,234,234,234,234,234,234,234,234,234,235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,237,237,237,237,237,237,237,237,237,237,237,237,237,237,237,237,237,237,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,239,239,239,239,239,239,239,239,239,239,239,239,239,239,239,239,239,239,240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,242,242,242,242,242,242,242,242,242,242,242,242,242,242,242,242,242,242,243,243,243,243,243,243,243,243,243,243,243,243,243,243,243,243,243,243,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,245,245,245,245,245,245,245,245,245,245,245,245,245,245,245,245,245,245,246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,247,247,247,247,247,247,247,247,247,247,247,247,247,247,247,248,248,248,248,248,248,248,248,248,248,248,248,248,248,248,248,248,248,249,249,249,249,249,249,249,249,249,249,249,249,249,249,249,249,249,249,250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,251,251,251,251,251,251,251,251,251,251,251,251,251,251,251,251,251,251,252,252,252,252,252,252,252,252,252,252,252,252,252,252,252,252,252,252,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,254,254,254,254,254,254,254,254,254,254,254,254,254,254,254,254,254,254,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,256,256,256,256,256,256,256,256,256,256,256,256,256,256,256,256,256,256,257,257,257,257,257,257,257,257,257,257,257,257,257,257,257,257,257,257,258,258,258,258,258,258,258,258,258,258,258,258,258,258,258,258,258,258,259,259,259,259,259,259,259,259,259,259,259,259,259,259,259,259,259,259,260,260,260,260,260,260,260,260,260,260,260,260,260,260,260,260,260,260,261,261,261,261,261,261,261,261,261,261,261,261,261,261,261,261,261,261,262,262,262,262,262,262,262,262,262,262,262,262,262,262,262,262,262,262,263,263,263,263,263,263,263,263,263,263,263,263,263,263,263,263,263,263,264,264,264,264,264,264,264,264,264,264,264,264,264,264,264,264,264,264,265,265,265,265,265,265,265,265,265,265,265,265,265,265,265,265,265,265,266,266,266,266,266,266,266,266,266,266,266,266,266,266,266,266,266,266,267,267,267,267,267,267,267,267,267,267,267,267,267,267,267,267,267,267,268,268,268,268,268,268,268,268,268,268,268,268,268,268,268,268,268,268,269,269,269,269,269,269,269,269,269,269,269,269,269,269,269,269,269,269,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,271,271,271,271,271,271,271,271,271,271,271,271,271,271,271,271,271,271,272,272,272,272,272,272,272,272,272,272,272,272,272,272,272,272,272,272,273,273,273,273,273,273,273,273,273,273,273,273,273,273,273,273,273,273,274,274,274,274,274,274,274,274,274,274,274,274,274,274,274,274,274,274,275,275,275,275,275,275,275,275,275,275,275,275,275,275,275,275,275,275,276,276,276,276,276,276,276,276,276,276,276,276,276,276,276,276,276,276,277,277,277,277,277,277,277,277,277,277,277,277,277,277,277,277,277,277,278,278,278,278,278,278,278,278,278,278,278,278,278,278,278,278,278,278,279,279,279,279,279,279,279,279,279,279,279,279,279,279,279,279,279,279,280,280,280,280,280,280,280,280,280,280,280,280,280,280,280,280,280,280,281,281,281,281,281,281,281,281,281,281,281,281,281,281,281,281,281,281,282,282,282,282,282,282,282,282,282,282,282,282,282,282,282,282,282,282,283,283,283,283,283,283,283,283,283,283,283,283,283,283,283,283,283,283,284,284,284,284,284,284,284,284,284,284,284,284,284,284,284,284,284,284,285,285,285,285,285,285,285,285,285,285,285,285,285,285,285,285,285,285,286,286,286,286,286,286,286,286,286,286,286,286,286,286,286,286,286,286,287,287,287,287,287,287,287,287,287,287,287,287,287,287,287,287,287,287,288,288,288,288,288,288,288,288,288,288,288,288,288,288,288,288,288,288,289,289,289,289,289,289,289,289,289,289,289,289,289,289,289,289,289,289,290,290,290,290,290,290,290,290,290,290,290,290,290,290,290,290,290,290,291,291,291,291,291,291,291,291,291,291,291,291,291,291,291,291,291,291,292,292,292,292,292,292,292,292,292,292,292,292,292,292,292,292,292,292,293,293,293,293,293,293,293,293,293,293,293,293,293,293,293,293,293,293,294,294,294,294,294,294,294,294,294,294,294,294,294,294,294,294,294,294,295,295,295,295,295,295,295,295,295,295,295,295,295,295,295,295,295,295,296,296,296,296,296,296,296,296,296,296,296,296,296,296,296,296,296,296,297,297,297,297,297,297,297,297,297,297,297,297,297,297,297,297,297,297,298,298,298,298,298,298,298,298,298,298,298,298,298,298,298,299,299,299,299,299,299,299,299,299,299,299,299,299,299,299,299,299,299,300,300,300,300,300,300,300,300,300,300,300,300,300,300,300,300,300,300,301,301,301,301,301,301,301,301,301,301,301,301,301,301,301,301,301,301,302,302,302,302,302,302,302,302,302,302,302,302,302,302,302,303,303,303,303,303,303,303,303,303,303,303,303,303,303,303,303,303,303,304,304,304,304,304,304,304,304,304,304,304,304,304,304,304,304,304,304,305,305,305,305,305,305,305,305,305,305,305,305,305,305,305,305,305,305,306,306,306,306,306,306,306,306,306,306,306,306,306,306,306,306,306,306,307,307,307,307,307,307,307,307,307,307,307,307,307,307,307,307,307,307,308,308,308,308,308,308,308,308,308,308,308,308,308,308,308,308,308,308,309,309,309,309,309,309,309,309,309,309,309,309,309,309,309,309,309,309,310,310,310,310,310,310,310,310,310,310,310,310,310,310,310,310,310,310,311,311,311,311,311,311,311,311,311,311,311,311,311,311,311,311,311,311,312,312,312,312,312,312,312,312,312,312,312,312,312,312,312,312,312,312,313,313,313,313,313,313,313,313,313,313,313,313,313,313,313,313,313,313,314,314,314,314,314,314,314,314,314,314,314,314,314,314,314,314,314,314,315,315,315,315,315,315,315,315,315,315,315,315,315,315,315,315,315,315,316,316,316,316,316,316,316,316,316,316,316,316,316,316,316,316,316,316,317,317,317,317,317,317,317,317,317,317,317,317,317,317,317,317,317,317,318,318,318,318,318,318,318,318,318,318,318,318,318,318,318,318,318,318,319,319,319,319,319,319,319,319,319,319,319,319,319,319,319,319,319,319,320,320,320,320,320,320,320,320,320,320,320,320,320,320,320,320,320,320,321,321,321,321,321,321,321,321,321,321,321,321,321,321,321,321,321,321,322,322,322,322,322,322,322,322,322,322,322,322,322,322,322,322,322,322,323,323,323,323,323,323,323,323,323,323,323,323,323,323,323,323,323,323,324,324,324,324,324,324,324,324,324,324,324,324,324,324,324,324,324,324,325,325,325,325,325,325,325,325,325,325,325,325,325,325,325,325,325,325,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,326,327,327,327,327,327,327,327,327,327,327,327,327,327,327,327,327,327,327,328,328,328,328,328,328,328,328,328,328,328,328,328,328,328,328,328,328,329,329,329,329,329,329,329,329,329,329,329,329,329,329,329,329,329,329,330,330,330,330,330,330,330,330,330,330,330,330,330,330,330,330,330,330,331,331,331,331,331,331,331,331,331,331,331,331,331,331,331,331,331,331,332,332,332,332,332,332,332,332,332,332,332,332,332,332,332,332,332,332,333,333,333,333,333,333,333,333,333,333,333,333,333,333,333,333,333,333,334,334,334,334,334,334,334,334,334,334,334,334,334,334,334,335,335,335,335,335,335,335,335,335,335,335,335,335,335,335,335,335,335,336,336,336,336,336,336,336,336,336,336,336,336,336,336,336,336,336,336,337,337,337,337,337,337,337,337,337,337,337,337,337,337,337,337,337,337,338,338,338,338,338,338,338,338,338,338,338,338,338,338,338,338,338,338,339,339,339,339,339,339,339,339,339,339,339,339,339,339,339,339,339,339,340,340,340,340,340,340,340,340,340,340,340,340,340,340,340,340,340,340,341,341,341,341,341,341,341,341,341,341,341,341,341,341,341,341,341,341,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,342,343,343,343,343,343,343,343,343,343,343,343,343,343,343,343,343,343,343,344,344,344,344,344,344,344,344,344,344,344,344,344,344,344,344,344,344,345,345,345,345,345,345,345,345,345,345,345,345,345,345,345,345,345,345,346,346,346,346,346,346,346,346,346,346,346,346,346,346,346,346,346,346,347,347,347,347,347,347,347,347,347,347,347,347,347,347,347,347,347,347,348,348,348,348,348,348,348,348,348,348,348,348,348,348,348,349,349,349,349,349,349,349,349,349,349,349,349,349,349,349,349,349,349,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,350,351,351,351,351,351,351,351,351,351,351,351,351,351,351,351,351,351,351,352,352,352,352,352,352,352,352,352,352,352,352,352,352,352,352,352,352,353,353,353,353,353,353,353,353,353,353,353,353,353,353,353,353,353,353,354,354,354,354,354,354,354,354,354,354,354,354,354,354,354,354,354,354,355,355,355,355,355,355,355,355,355,355,355,355,355,355,355,355,355,355,356,356,356,356,356,356,356,356,356,356,356,356,356,356,356,356,356,356,357,357,357,357,357,357,357,357,357,357,357,357,357,357,357,357,357,357,358,358,358,358,358,358,358,358,358,358,358,358,358,358,358,358,358,358,359,359,359,359,359,359,359,359,359,359,359,359,359,359,359,359,359,359,360,360,360,360,360,360,360,360,360,360,360,360,360,360,360,360,360,360,361,361,361,361,361,361,361,361,361,361,361,361,361,361,361,361,361,361,362,362,362,362,362,362,362,362,362,362,362,362,362,362,362,362,362,362,363,363,363,363,363,363,363,363,363,363,363,363,363,363,363,363,363,363,364,364,364,364,364,364,364,364,364,364,364,364,364,364,364,364,364,364,365,365,365,365,365,365,365,365,365,365,365,365,365,365,365,365,365,365,366,366,366,366,366,366,366,366,366,366,366,366,366,366,366,366,366,366,367,367,367,367,367,367,367,367,367,367,367,367,367,367,367,367,367,367,368,368,368,368,368,368,368,368,368,368,368,368,368,368,368,368,368,368,369,369,369,369,369,369,369,369,369,369,369,369,369,369,369,369,369,369,370,370,370,370,370,370,370,370,370,370,370,370,370,370,370,370,370,370,371,371,371,371,371,371,371,371,371,371,371,371,371,371,371,371,371,371,372,372,372,372,372,372,372,372,372,372,372,372,372,372,372,372,372,372,373,373,373,373,373,373,373,373,373,373,373,373,373,373,373,373,373,373,374,374,374,374,374,374,374,374,374,374,374,374,374,374,374,374,374,374,375,375,375,375,375,375,375,375,375,375,375,375,375,375,375,375,375,375,376,376,376,376,376,376,376,376,376,376,376,376,376,376,376,376,376,376,377,377,377,377,377,377,377,377,377,377,377,377,377,377,377,377,377,377,378,378,378,378,378,378,378,378,378,378,378,378,378,378,378,378,378,378,379,379,379,379,379,379,379,379,379,379,379,379,379,379,379,379,379,379,380,380,380,380,380,380,380,380,380,380,380,380,380,380,380,380,380,380,381,381,381,381,381,381,381,381,381,381,381,381,381,381,381,381,381,381,382,382,382,382,382,382,382,382,382,382,382,382,382,382,382,382,382,382,383,383,383,383,383,383,383,383,383,383,383,383,383,383,383,383,383,383,384,384,384,384,384,384,384,384,384,384,384,384,384,384,384,384,384,384,385,385,385,385,385,385,385,385,385,385,385,385,385,385,385,385,385,385,386,386,386,386,386,386,386,386,386,386,386,386,386,386,386,386,386,386,387,387,387,387,387,387,387,387,387,387,387,387,387,387,387,387,387,387,388,388,388,388,388,388,388,388,388,388,388,388,388,388,388,388,388,388,389,389,389,389,389,389,389,389,389,389,389,389,389,389,389,389,389,389,390,390,390,390,390,390,390,390,390,390,390,390,390,390,390,390,390,390,391,391,391,391,391,391,391,391,391,391,391,391,391,391,391,391,391,391,392,392,392,392,392,392,392,392,392,392,392,392,392,392,392,392,392,392,393,393,393,393,393,393,393,393,393,393,393,393,393,393,393,393,393,393,394,394,394,394,394,394,394,394,394,394,394,394,394,394,394,394,394,394,395,395,395,395,395,395,395,395,395,395,395,395,395,395,395,395,395,395,396,396,396,396,396,396,396,396,396,396,396,396,396,396,396,396,396,396,397,397,397,397,397,397,397,397,397,397,397,397,397,397,397,397,397,397,398,398,398,398,398,398,398,398,398,398,398,398,398,398,398,398,398,398,399,399,399,399,399,399,399,399,399,399,399,399,399,399,399,399,399,399,400,400,400,400,400,400,400,400,400,400,400,400,400,400,400,400,400,400,401,401,401,401,401,401,401,401,401,401,401,401,401,401,401,401,401,401,402,402,402,402,402,402,402,402,402,402,402,402,402,402,402,402,402,402,403,403,403,403,403,403,403,403,403,403,403,403,403,403,403,403,403,403,404,404,404,404,404,404,404,404,404,404,404,404,404,404,404,404,404,404,405,405,405,405,405,405,405,405,405,405,405,405,405,405,405,405,405,405,406,406,406,406,406,406,406,406,406,406,406,406,406,406,406,406,406,406,407,407,407,407,407,407,407,407,407,407,407,407,407,407,407,407,407,407,408,408,408,408,408,408,408,408,408,408,408,408,408,408,408,408,408,408,409,409,409,409,409,409,409,409,409,409,409,409,409,409,409,409,409,409,410,410,410,410,410,410,410,410,410,410,410,410,410,410,410,410,410,410,411,411,411,411,411,411,411,411,411,411,411,411,411,411,411,411,411,411,412,412,412,412,412,412,412,412,412,412,412,412,412,412,412,412,412,412,413,413,413,413,413,413,413,413,413,413,413,413,413,413,413,413,413,413,414,414,414,414,414,414,414,414,414,414,414,414,414,414,414,414,414,414,415,415,415,415,415,415,415,415,415,415,415,415,415,415,415,415,415,415,416,416,416,416,416,416,416,416,416,416,416,416,416,416,416,416,416,416,417,417,417,417,417,417,417,417,417,417,417,417,417,417,417,417,417,417,418,418,418,418,418,418,418,418,418,418,418,418,418,418,418,418,418,418,419,419,419,419,419,419,419,419,419,419,419,419,419,419,419,419,419,419,420,420,420,420,420,420,420,420,420,420,420,420,420,420,420,420,420,420,421,421,421,421,421,421,421,421,421,421,421,421,421,421,421,422,422,422,422,422,422,422,422,422,422,422,422,422,422,422,423,423,423,423,423,423,423,423,423,423,423,423,423,423,423,423,423,423,424,424,424,424,424,424,424,424,424,424,424,424,424,424,424,424,424,424,425,425,425,425,425,425,425,425,425,425,425,425,425,425,425,425,425,425,426,426,426,426,426,426,426,426,426,426,426,426,426,426,426,426,426,426,427,427,427,427,427,427,427,427,427,427,427,427,427,427,427,427,427,427,428,428,428,428,428,428,428,428,428,428,428,428,428,428,428,428,428,428,429,429,429,429,429,429,429,429,429,429,429,429,429,429,429,429,429,429,430,430,430,430,430,430,430,430,430,430,430,430,430,430,430,430,430,430,431,431,431,431,431,431,431,431,431,431,431,431,431,431,431,431,431,431,432,432,432,432,432,432,432,432,432,432,432,432,432,432,432,432,432,432,433,433,433,433,433,433,433,433,433,433,433,433,433,433,433,434,434,434,434,434,434,434,434,434,434,434,434,434,434,434,434,434,434,435,435,435,435,435,435,435,435,435,435,435,435,435,435,435,435,435,435,436,436,436,436,436,436,436,436,436,436,436,436,436,436,436,436,436,436,437,437,437,437,437,437,437,437,437,437,437,437,437,437,437,437,437,437,438,438,438,438,438,438,438,438,438,438,438,438,438,438,438,438,438,438,439,439,439,439,439,439,439,439,439,439,439,439,439,439,439,439,439,439,440,440,440,440,440,440,440,440,440,440,440,440,440,440,440,440,440,440,441,441,441,441,441,441,441,441,441,441,441,441,441,441,441,441,441,441,442,442,442,442,442,442,442,442,442,442,442,442,442,442,442,442,442,442,443,443,443,443,443,443,443,443,443,443,443,443,443,443,443,443,443,443,444,444,444,444,444,444,444,444,444,444,444,444,444,444,444,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,445,446,446,446,446,446,446,446,446,446,446,446,446,446,446,446,447,447,447,447,447,447,447,447,447,447,447,447,447,447,447,447,447,447,448,448,448,448,448,448,448,448,448,448,448,448,448,448,448,448,448,448,449,449,449,449,449,449,449,449,449,449,449,449,449,449,449,449,449,449,450,450,450,450,450,450,450,450,450,450,450,450,450,450,450,450,450,450,451,451,451,451,451,451,451,451,451,451,451,451,451,451,451,451,451,451,452,452,452,452,452,452,452,452,452,452,452,452,452,452,452,452,452,452,453,453,453,453,453,453,453,453,453,453,453,453,453,453,453,453,453,453,454,454,454,454,454,454,454,454,454,454,454,454,454,454,454,454,454,454,455,455,455,455,455,455,455,455,455,455,455,455,455,455,455,455,455,455,456,456,456,456,456,456,456,456,456,456,456,456,456,456,456,456,456,456,457,457,457,457,457,457,457,457,457,457,457,457,457,457,457,457,457,457,458,458,458,458,458,458,458,458,458,458,458,458,458,458,458,458,458,458,459,459,459,459,459,459,459,459,459,459,459,459,459,459,459,459,459,459,460,460,460,460,460,460,460,460,460,460,460,460,460,460,460,460,460,460,461,461,461,461,461,461,461,461,461,461,461,461,461,461,461,461,461,461,462,462,462,462,462,462,462,462,462,462,462,462,462,462,462,462,462,462,463,463,463,463,463,463,463,463,463,463,463,463,463,463,463,463,463,463,464,464,464,464,464,464,464,464,464,464,464,464,464,464,464,464,464,464,465,465,465,465,465,465,465,465,465,465,465,465,465,465,465,465,465,465,466,466,466,466,466,466,466,466,466,466,466,466,466,466,466,466,466,466,467,467,467,467,467,467,467,467,467,467,467,467,467,467,467,467,467,467,468,468,468,468,468,468,468,468,468,468,468,468,468,468,468,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,469,470,470,470,470,470,470,470,470,470,470,470,470,470,470,470,470,470,470,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,471,472,472,472,472,472,472,472,472,472,472,472,472,472,472,472,472,472,472,473,473,473,473,473,473,473,473,473,473,473,473,473,473,473,473,473,473,474,474,474,474,474,474,474,474,474,474,474,474,474,474,474,474,474,474,475,475,475,475,475,475,475,475,475,475,475,475,475,475,475,475,475,475,476,476,476,476,476,476,476,476,476,476,476,476,476,476,476,476,476,476,477,477,477,477,477,477,477,477,477,477,477,477,477,477,477,477,477,477,478,478,478,478,478,478,478,478,478,478,478,478,478,478,478,478,478,478,479,479,479,479,479,479,479,479,479,479,479,479,479,479,479,479,479,479,480,480,480,480,480,480,480,480,480,480,480,480,480,480,480,480,480,480,481,481,481,481,481,481,481,481,481,481,481,481,481,481,481,481,481,481,482,482,482,482,482,482,482,482,482,482,482,482,482,482,482,482,482,482,483,483,483,483,483,483,483,483,483,483,483,483,483,483,483,483,483,483,484,484,484,484,484,484,484,484,484,484,484,484,484,484,484,484,484,484,485,485,485,485,485,485,485,485,485,485,485,485,485,485,485,485,485,485,486,486,486,486,486,486,486,486,486,486,486,486,486,486,486,486,486,486,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,487,488,488,488,488,488,488,488,488,488,488,488,488,488,488,488,488,488,488,489,489,489,489,489,489,489,489,489,489,489,489,489,489,489,489,489,489,490,490,490,490,490,490,490,490,490,490,490,490,490,490,490,490,490,490,491,491,491,491,491,491,491,491,491,491,491,491,491,491,491,491,491,491,492,492,492,492,492,492,492,492,492,492,492,492,492,492,492,492,492,492,493,493,493,493,493,493,493,493,493,493,493,493,493,493,493,493,493,493,494,494,494,494,494,494,494,494,494,494,494,494,494,494,494,494,494,494,495,495,495,495,495,495,495,495,495,495,495,495,495,495,495,495,495,495,496,496,496,496,496,496,496,496,496,496,496,496,496,496,496,496,496,496,497,497,497,497,497,497,497,497,497,497,497,497,497,497,497,497,497,497,498,498,498,498,498,498,498,498,498,498,498,498,498,498,498,498,498,498,499,499,499,499,499,499,499,499,499,499,499,499,499,499,499,499,499,499,500,500,500,500,500,500,500,500,500,500,500,500,500,500,500,500,500,500,501,501,501,501,501,501,501,501,501,501,501,501,501,501,501,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,502,503,503,503,503,503,503,503,503,503,503,503,503,503,503,503,503,503,503,504,504,504,504,504,504,504,504,504,504,504,504,504,504,504,504,504,504,505,505,505,505,505,505,505,505,505,505,505,505,505,505,505,505,505,505,506,506,506,506,506,506,506,506,506,506,506,506,506,506,506,506,506,506,507,507,507,507,507,507,507,507,507,507,507,507,507,507,507,507,507,507,508,508,508,508,508,508,508,508,508,508,508,508,508,508,508,508,508,508,509,509,509,509,509,509,509,509,509,509,509,509,509,509,509,509,509,509,510,510,510,510,510,510,510,510,510,510,510,510,510,510,510,510,510,510,511,511,511,511,511,511,511,511,511,511,511,511,511,511,511,511,511,511,512,512,512,512,512,512,512,512,512,512,512,512,512,512,512,512,512,512,513,513,513,513,513,513,513,513,513,513,513,513,513,513,513,513,513,513,514,514,514,514,514,514,514,514,514,514,514,514,514,514,514,514,514,514,515,515,515,515,515,515,515,515,515,515,515,515,515,515,515,515,515,515,516,516,516,516,516,516,516,516,516,516,516,516,516,516,516,516,516,516,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,517,518,518,518,518,518,518,518,518,518,518,518,518,518,518,518,518,518,518,519,519,519,519,519,519,519,519,519,519,519,519,519,519,519,519,519,519,520,520,520,520,520,520,520,520,520,520,520,520,520,520,520,520,520,520,521,521,521,521,521,521,521,521,521,521,521,521,521,521,521,521,521,521,522,522,522,522,522,522,522,522,522,522,522,522,522,522,522,522,522,522,523,523,523,523,523,523,523,523,523,523,523,523,523,523,523,523,523,523,524,524,524,524,524,524,524,524,524,524,524,524,524,524,524,524,524,524,525,525,525,525,525,525,525,525,525,525,525,525,525,525,525,525,525,525,526,526,526,526,526,526,526,526,526,526,526,526,526,526,526,526,526,526,527,527,527,527,527,527,527,527,527,527,527,527,527,527,527,527,527,527,528,528,528,528,528,528,528,528,528,528,528,528,528,528,528,528,528,528,529,529,529,529,529,529,529,529,529,529,529,529,529,529,529,529,529,529,530,530,530,530,530,530,530,530,530,530,530,530,530,530,530,530,530,530,531,531,531,531,531,531,531,531,531,531,531,531,531,531,531,531,531,531,532,532,532,532,532,532,532,532,532,532,532,532,532,532,532,532,532,532,533,533,533,533,533,533,533,533,533,533,533,533,533,533,533,533,533,533,534,534,534,534,534,534,534,534,534,534,534,534,534,534,534,534,534,534,535,535,535,535,535,535,535,535,535,535,535,535,535,535,535,535,535,535,536,536,536,536,536,536,536,536,536,536,536,536,536,536,536,536,536,536,537,537,537,537,537,537,537,537,537,537,537,537,537,537,537,537,537,537,538,538,538,538,538,538,538,538,538,538,538,538,538,538,538,538,538,538,539,539,539,539,539,539,539,539,539,539,539,539,539,539,539,539,539,539,540,540,540,540,540,540,540,540,540,540,540,540,540,540,540,540,540,540,541,541,541,541,541,541,541,541,541,541,541,541,541,541,541,541,541,541,542,542,542,542,542,542,542,542,542,542,542,542,542,542,542,542,542,542,543,543,543,543,543,543,543,543,543,543,543,543,543,543,543,543,543,543,544,544,544,544,544,544,544,544,544,544,544,544,544,544,544,544,544,544,545,545,545,545,545,545,545,545,545,545,545,545,545,545,545,545,545,545,546,546,546,546,546,546,546,546,546,546,546,546,546,546,546,546,546,546,547,547,547,547,547,547,547,547,547,547,547,547,547,547,547,547,547,547,548,548,548,548,548,548,548,548,548,548,548,548,548,548,548,548,548,548,549,549,549,549,549,549,549,549,549,549,549,549,549,549,549,549,549,549,550,550,550,550,550,550,550,550,550,550,550,550,550,550,550,550,550,550,551,551,551,551,551,551,551,551,551,551,551,551,551,551,551,551,551,551,552,552,552,552,552,552,552,552,552,552,552,552,552,552,552,552,552,552,553,553,553,553,553,553,553,553,553,553,553,553,553,553,553,553,553,553,554,554,554,554,554,554,554,554,554,554,554,554,554,554,554,554,554,554,555,555,555,555,555,555,555,555,555,555,555,555,555,555,555,556,556,556,556,556,556,556,556,556,556,556,556,556,556,556,556,556,556,557,557,557,557,557,557,557,557,557,557,557,557,557,557,557,557,557,557,558,558,558,558,558,558,558,558,558,558,558,558,558,558,558,558,558,558,559,559,559,559,559,559,559,559,559,559,559,559,559,559,559,559,559,559,560,560,560,560,560,560,560,560,560,560,560,560,560,560,560,560,560,560,561,561,561,561,561,561,561,561,561,561,561,561,561,561,561,561,561,561,562,562,562,562,562,562,562,562,562,562,562,562,562,562,562,562,562,562,563,563,563,563,563,563,563,563,563,563,563,563,563,563,563,563,563,563,564,564,564,564,564,564,564,564,564,564,564,564,564,564,564,564,564,564,565,565,565,565,565,565,565,565,565,565,565,565,565,565,565,565,565,565,566,566,566,566,566,566,566,566,566,566,566,566,566,566,566,566,566,566,567,567,567,567,567,567,567,567,567,567,567,567,567,567,567,567,567,567,568,568,568,568,568,568,568,568,568,568,568,568,568,568,568,568,568,568,569,569,569,569,569,569,569,569,569,569,569,569,569,569,569,569,569,569,570,570,570,570,570,570,570,570,570,570,570,570,570,570,570,570,570,570,571,571,571,571,571,571,571,571,571,571,571,571,571,571,571,571,571,571,572,572,572,572,572,572,572,572,572,572,572,572,572,572,572,572,572,572,573,573,573,573,573,573,573,573,573,573,573,573,573,573,573,573,573,573,574,574,574,574,574,574,574,574,574,574,574,574,574,574,574,574,574,574,575,575,575,575,575,575,575,575,575,575,575,575,575,575,575,575,575,575,576,576,576,576,576,576,576,576,576,576,576,576,576,576,576,576,576,576,577,577,577,577,577,577,577,577,577,577,577,577,577,577,577,577,577,577,578,578,578,578,578,578,578,578,578,578,578,578,578,578,578,578,578,578,579,579,579,579,579,579,579,579,579,579,579,579,579,579,579,579,579,579,580,580,580,580,580,580,580,580,580,580,580,580,580,580,580,580,580,580,581,581,581,581,581,581,581,581,581,581,581,581,581,581,581,581,581,581,582,582,582,582,582,582,582,582,582,582,582,582,582,582,582,582,582,582,583,583,583,583,583,583,583,583,583,583,583,583,583,583,583,583,583,583,584,584,584,584,584,584,584,584,584,584,584,584,584,584,584,584,584,584,585,585,585,585,585,585,585,585,585,585,585,585,585,585,585,585,585,585,586,586,586,586,586,586,586,586,586,586,586,586,586,586,586,586,586,586,587,587,587,587,587,587,587,587,587,587,587,587,587,587,587,587,587,587,588,588,588,588,588,588,588,588,588,588,588,588,588,588,588,589,589,589,589,589,589,589,589,589,589,589,589,589,589,589,589,589,589,590,590,590,590,590,590,590,590,590,590,590,590,590,590,590,590,590,590,591,591,591,591,591,591,591,591,591,591,591,591,591,591,591,591,591,591,592,592,592,592,592,592,592,592,592,592,592,592,592,592,592,592,592,592,593,593,593,593,593,593,593,593,593,593,593,593,593,593,593,593,593,593,594,594,594,594,594,594,594,594,594,594,594,594,594,594,594,594,594,594,595,595,595,595,595,595,595,595,595,595,595,595,595,595,595,595,595,595,596,596,596,596,596,596,596,596,596,596,596,596,596,596,596,596,596,596,597,597,597,597,597,597,597,597,597,597,597,597,597,597,597,597,597,597,598,598,598,598,598,598,598,598,598,598,598,598,598,598,598,598,598,598,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,599,600,600,600,600,600,600,600,600,600,600,600,600,600,600,600,600,600,600,601,601,601,601,601,601,601,601,601,601,601,601,601,601,601,601,601,601,602,602,602,602,602,602,602,602,602,602,602,602,602,602,602,602,602,602,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,603,604,604,604,604,604,604,604,604,604,604,604,604,604,604,604,604,604,604,605,605,605,605,605,605,605,605,605,605,605,605,605,605,605,605,605,605,606,606,606,606,606,606,606,606,606,606,606,606,606,606,606,606,606,606,607,607,607,607,607,607,607,607,607,607,607,607,607,607,607,607,607,607,608,608,608,608,608,608,608,608,608,608,608,608,608,608,608,608,608,608,609,609,609,609,609,609,609,609,609,609,609,609,609,609,609,609,609,609,610,610,610,610,610,610,610,610,610,610,610,610,610,610,610,610,610,610,611,611,611,611,611,611,611,611,611,611,611,611,611,611,611,611,611,611,612,612,612,612,612,612,612,612,612,612,612,612,612,612,612,612,612,612,613,613,613,613,613,613,613,613,613,613,613,613,613,613,613,613,613,613,614,614,614,614,614,614,614,614,614,614,614,614,614,614,614,614,614,614,615,615,615,615,615,615,615,615,615,615,615,615,615,615,615,615,615,615,616,616,616,616,616,616,616,616,616,616,616,616,616,616,616,616,616,616,617,617,617,617,617,617,617,617,617,617,617,617,617,617,617,617,617,617,618,618,618,618,618,618,618,618,618,618,618,618,618,618,618,618,618,618,619,619,619,619,619,619,619,619,619,619,619,619,619,619,619,619,619,619,620,620,620,620,620,620,620,620,620,620,620,620,620,620,620,620,620,620,621,621,621,621,621,621,621,621,621,621,621,621,621,621,621,621,621,621,622,622,622,622,622,622,622,622,622,622,622,622,622,622,622,622,622,622,623,623,623,623,623,623,623,623,623,623,623,623,623,623,623,623,623,623,624,624,624,624,624,624,624,624,624,624,624,624,624,624,624,624,624,624,625,625,625,625,625,625,625,625,625,625,625,625,625,625,625,625,625,625,626,626,626,626,626,626,626,626,626,626,626,626,626,626,626,626,626,626,627,627,627,627,627,627,627,627,627,627,627,627,627,627,627,627,627,627,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,628,629,629,629,629,629,629,629,629,629,629,629,629,629,629,629,629,629,629,630,630,630,630,630,630,630,630,630,630,630,630,630,630,630,630,630,630,631,631,631,631,631,631,631,631,631,631,631,631,631,631,631,631,631,631,632,632,632,632,632,632,632,632,632,632,632,632,632,632,632,632,632,632,633,633,633,633,633,633,633,633,633,633,633,633,633,633,633,633,633,633,634,634,634,634,634,634,634,634,634,634,634,634,634,634,634,634,634,634,635,635,635,635,635,635,635,635,635,635,635,635,635,635,635,635,635,635,636,636,636,636,636,636,636,636,636,636,636,636,636,636,636,636,636,636,637,637,637,637,637,637,637,637,637,637,637,637,637,637,637,637,637,637,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,638,639,639,639,639,639,639,639,639,639,639,639,639,639,639,639,639,639,639,640,640,640,640,640,640,640,640,640,640,640,640,640,640,640,640,640,640,641,641,641,641,641,641,641,641,641,641,641,641,641,641,641,642,642,642,642,642,642,642,642,642,642,642,642,642,642,642,642,642,642,643,643,643,643,643,643,643,643,643,643,643,643,643,643,643,643,643,643,644,644,644,644,644,644,644,644,644,644,644,644,644,644,644,644,644,644,645,645,645,645,645,645,645,645,645,645,645,645,645,645,645,645,645,645,646,646,646,646,646,646,646,646,646,646,646,646,646,646,646,646,646,646,647,647,647,647,647,647,647,647,647,647,647,647,647,647,647,647,647,647,648,648,648,648,648,648,648,648,648,648,648,648,648,648,648,648,648,648,649,649,649,649,649,649,649,649,649,649,649,649,649,649,649,649,649,649,650,650,650,650,650,650,650,650,650,650,650,650,650,650,650,650,650,650,651,651,651,651,651,651,651,651,651,651,651,651,651,651,651,651,651,651,652,652,652,652,652,652,652,652,652,652,652,652,652,652,652,652,652,652,653,653,653,653,653,653,653,653,653,653,653,653,653,653,653,653,653,653,654,654,654,654,654,654,654,654,654,654,654,654,654,654,654,654,654,654,655,655,655,655,655,655,655,655,655,655,655,655,655,655,655,655,655,655,656,656,656,656,656,656,656,656,656,656,656,656,656,656,656,656,656,656,657,657,657,657,657,657,657,657,657,657,657,657,657,657,657,657,657,657,658,658,658,658,658,658,658,658,658,658,658,658,658,658,658,658,658,658,659,659,659,659,659,659,659,659,659,659,659,659,659,659,659,659,659,659,660,660,660,660,660,660,660,660,660,660,660,660,660,660,660,660,660,660,661,661,661,661,661,661,661,661,661,661,661,661,661,661,661,661,661,661,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,662,663,663,663,663,663,663,663,663,663,663,663,663,663,663,663,663,663,663,664,664,664,664,664,664,664,664,664,664,664,664,664,664,664,664,664,664,665,665,665,665,665,665,665,665,665,665,665,665,665,665,665,665,665,665,666,666,666,666,666,666,666,666,666,666,666,666,666,666,666,666,666,666,667,667,667,667,667,667,667,667,667,667,667,667,667,667,667,667,667,667,668,668,668,668,668,668,668,668,668,668,668,668,668,668,668,668,668,668,669,669,669,669,669,669,669,669,669,669,669,669,669,669,669,669,669,669,670,670,670,670,670,670,670,670,670,670,670,670,670,670,670,670,670,670,671,671,671,671,671,671,671,671,671,671,671,671,671,671,671,671,671,671,672,672,672,672,672,672,672,672,672,672,672,672,672,672,672,672,672,672,673,673,673,673,673,673,673,673,673,673,673,673,673,673,673,673,673,673,674,674,674,674,674,674,674,674,674,674,674,674,674,674,674,674,674,674,675,675,675,675,675,675,675,675,675,675,675,675,675,675,675,675,675,675,676,676,676,676,676,676,676,676,676,676,676,676,676,676,676,676,676,676,677,677,677,677,677,677,677,677,677,677,677,677,677,677,677,677,677,677,678,678,678,678,678,678,678,678,678,678,678,678,678,678,678,678,678,678,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,679,680,680,680,680,680,680,680,680,680,680,680,680,680,680,680,681,681,681,681,681,681,681,681,681,681,681,681,681,681,681,681,681,681,682,682,682,682,682,682,682,682,682,682,682,682,682,682,682,682,682,682,683,683,683,683,683,683,683,683,683,683,683,683,683,683,683,683,683,683,684,684,684,684,684,684,684,684,684,684,684,684,684,684,684,684,684,684,685,685,685,685,685,685,685,685,685,685,685,685,685,685,685,685,685,685,686,686,686,686,686,686,686,686,686,686,686,686,686,686,686,686,686,686,687,687,687,687,687,687,687,687,687,687,687,687,687,687,687,687,687,687,688,688,688,688,688,688,688,688,688,688,688,688,688,688,688,688,688,688,689,689,689,689,689,689,689,689,689,689,689,689,689,689,689,689,689,689,690,690,690,690,690,690,690,690,690,690,690,690,690,690,690,690,690,690,691,691,691,691,691,691,691,691,691,691,691,691,691,691,691,691,691,691,692,692,692,692,692,692,692,692,692,692,692,692,692,692,692,692,692,692,693,693,693,693,693,693,693,693,693,693,693,693,693,693,693,693,693,693,694,694,694,694,694,694,694,694,694,694,694,694,694,694,694,694,694,694,695,695,695,695,695,695,695,695,695,695,695,695,695,695,695,695,695,695,696,696,696,696,696,696,696,696,696,696,696,696,696,696,696,696,696,696,697,697,697,697,697,697,697,697,697,697,697,697,697,697,697,698,698,698,698,698,698,698,698,698,698,698,698,698,698,698,698,698,698,699,699,699,699,699,699,699,699,699,699,699,699,699,699,699,699,699,699,700,700,700,700,700,700,700,700,700,700,700,700,700,700,700,700,700,700,701,701,701,701,701,701,701,701,701,701,701,701,701,701,701,701,701,701,702,702,702,702,702,702,702,702,702,702,702,702,702,702,702,702,702,702,703,703,703,703,703,703,703,703,703,703,703,703,703,703,703,703,703,703,704,704,704,704,704,704,704,704,704,704,704,704,704,704,704,704,704,704,705,705,705,705,705,705,705,705,705,705,705,705,705,705,705,706,706,706,706,706,706,706,706,706,706,706,706,706,706,706,707,707,707,707,707,707,707,707,707,707,707,707,707,707,707,707,707,707,708,708,708,708,708,708,708,708,708,708,708,708,708,708,708,708,708,708,709,709,709,709,709,709,709,709,709,709,709,709,709,709,709,709,709,709,710,710,710,710,710,710,710,710,710,710,710,710,710,710,710,710,710,710,711,711,711,711,711,711,711,711,711,711,711,711,711,711,711,711,711,711,712,712,712,712,712,712,712,712,712,712,712,712,712,712,712,712,712,712,713,713,713,713,713,713,713,713,713,713,713,713,713,713,713,713,713,713,714,714,714,714,714,714,714,714,714,714,714,714,714,714,714,714,714,714,715,715,715,715,715,715,715,715,715,715,715,715,715,715,715,715,715,715,716,716,716,716,716,716,716,716,716,716,716,716,716,716,716,716,716,716,717,717,717,717,717,717,717,717,717,717,717,717,717,717,717,717,717,717,718,718,718,718,718,718,718,718,718,718,718,718,718,718,718,718,718,718,719,719,719,719,719,719,719,719,719,719,719,719,719,719,719,719,719,719,720,720,720,720,720,720,720,720,720,720,720,720,720,720,720,720,720,720,721,721,721,721,721,721,721,721,721,721,721,721,721,721,721,721,721,721,722,722,722,722,722,722,722,722,722,722,722,722,722,722,722,722,722,722,723,723,723,723,723,723,723,723,723,723,723,723,723,723,723,723,723,723,724,724,724,724,724,724,724,724,724,724,724,724,724,724,724,724,724,724,725,725,725,725,725,725,725,725,725,725,725,725,725,725,725,725,725,725,726,726,726,726,726,726,726,726,726,726,726,726,726,726,726,726,726,726,727,727,727,727,727,727,727,727,727,727,727,727,727,727,727,727,727,727,728,728,728,728,728,728,728,728,728,728,728,728,728,728,728,728,728,728,729,729,729,729,729,729,729,729,729,729,729,729,729,729,729,729,729,729,730,730,730,730,730,730,730,730,730,730,730,730,730,730,730,730,730,730,731,731,731,731,731,731,731,731,731,731,731,731,731,731,731,731,731,731,732,732,732,732,732,732,732,732,732,732,732,732,732,732,732,733,733,733,733,733,733,733,733,733,733,733,733,733,733,733,733,733,733,734,734,734,734,734,734,734,734,734,734,734,734,734,734,734,734,734,734,735,735,735,735,735,735,735,735,735,735,735,735,735,735,735,735,735,735,736,736,736,736,736,736,736,736,736,736,736,736,736,736,736,736,736,736,737,737,737,737,737,737,737,737,737,737,737,737,737,737,737,737,737,737,738,738,738,738,738,738,738,738,738,738,738,738,738,738,738,738,738,738,739,739,739,739,739,739,739,739,739,739,739,739,739,739,739,739,739,739,740,740,740,740,740,740,740,740,740,740,740,740,740,740,740,740,740,740,741,741,741,741,741,741,741,741,741,741,741,741,741,741,741,741,741,741,742,742,742,742,742,742,742,742,742,742,742,742,742,742,742,742,742,742,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,743,744,744,744,744,744,744,744,744,744,744,744,744,744,744,744,744,744,744,745,745,745,745,745,745,745,745,745,745,745,745,745,745,745,745,745,745,746,746,746,746,746,746,746,746,746,746,746,746,746,746,746,746,746,746,747,747,747,747,747,747,747,747,747,747,747,747,747,747,747,747,747,747,748,748,748,748,748,748,748,748,748,748,748,748,748,748,748,748,748,748,749,749,749,749,749,749,749,749,749,749,749,749,749,749,749,749,749,749,750,750,750,750,750,750,750,750,750,750,750,750,750,750,750,750,750,750,751,751,751,751,751,751,751,751,751,751,751,751,751,751,751,751,751,751,752,752,752,752,752,752,752,752,752,752,752,752,752,752,752,752,752,752,753,753,753,753,753,753,753,753,753,753,753,753,753,753,753,753,753,753,754,754,754,754,754,754,754,754,754,754,754,754,754,754,754,754,754,754,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,755,756,756,756,756,756,756,756,756,756,756,756,756,756,756,756,756,756,756,757,757,757,757,757,757,757,757,757,757,757,757,757,757,757,757,757,757,758,758,758,758,758,758,758,758,758,758,758,758,758,758,758,758,758,758,759,759,759,759,759,759,759,759,759,759,759,759,759,759,759,759,759,759,760,760,760,760,760,760,760,760,760,760,760,760,760,760,760,760,760,760,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,761,762,762,762,762,762,762,762,762,762,762,762,762,762,762,762,762,762,762,763,763,763,763,763,763,763,763,763,763,763,763,763,763,763,763,763,763,764,764,764,764,764,764,764,764,764,764,764,764,764,764,764,764,764,764,765,765,765,765,765,765,765,765,765,765,765,765,765,765,765,765,765,765,766,766,766,766,766,766,766,766,766,766,766,766,766,766,766,766,766,766,767,767,767,767,767,767,767,767,767,767,767,767,767,767,767,767,767,767,768,768,768,768,768,768,768,768,768,768,768,768,768,768,768,768,768,768,769,769,769,769,769,769,769,769,769,769,769,769,769,769,769,769,769,769,770,770,770,770,770,770,770,770,770,770,770,770,770,770,770,770,770,770,771,771,771,771,771,771,771,771,771,771,771,771,771,771,771,771,771,771,772,772,772,772,772,772,772,772,772,772,772,772,772,772,772,772,772,772,773,773,773,773,773,773,773,773,773,773,773,773,773,773,773,773,773,773,774,774,774,774,774,774,774,774,774,774,774,774,774,774,774,774,774,774,775,775,775,775,775,775,775,775,775,775,775,775,775,775,775,775,775,775,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,776,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,777,778,778,778,778,778,778,778,778,778,778,778,778,778,778,778,778,778,778,779,779,779,779,779,779,779,779,779,779,779,779,779,779,779,779,779,779,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,780,781,781,781,781,781,781,781,781,781,781,781,781,781,781,781,782,782,782,782,782,782,782,782,782,782,782,782,782,782,782,782,782,782,783,783,783,783,783,783,783,783,783,783,783,783,783,783,783,783,783,783,784,784,784,784,784,784,784,784,784,784,784,784,784,784,784,784,784,784,785,785,785,785,785,785,785,785,785,785,785,785,785,785,785,785,785,785,786,786,786,786,786,786,786,786,786,786,786,786,786,786,786,786,786,786,787,787,787,787,787,787,787,787,787,787,787,787,787,787,787,787,787,787,788,788,788,788,788,788,788,788,788,788,788,788,788,788,788,788,788,788,789,789,789,789,789,789,789,789,789,789,789,789,789,789,789,789,789,789,790,790,790,790,790,790,790,790,790,790,790,790,790,790,790,790,790,790,791,791,791,791,791,791,791,791,791,791,791,791,791,791,791,791,791,791,792,792,792,792,792,792,792,792,792,792,792,792,792,792,792,792,792,792,793,793,793,793,793,793,793,793,793,793,793,793,793,793,793,793,793,793,794,794,794,794,794,794,794,794,794,794,794,794,794,794,794,794,794,794,795,795,795,795,795,795,795,795,795,795,795,795,795,795,795,795,795,795,796,796,796,796,796,796,796,796,796,796,796,796,796,796,796,796,796,796,797,797,797,797,797,797,797,797,797,797,797,797,797,797,797,797,797,797,798,798,798,798,798,798,798,798,798,798,798,798,798,798,798,798,798,798,799,799,799,799,799,799,799,799,799,799,799,799,799,799,799,799,799,799,800,800,800,800,800,800,800,800,800,800,800,800,800,800,800,800,800,800,801,801,801,801,801,801,801,801,801,801,801,801,801,801,801,801,801,801,802,802,802,802,802,802,802,802,802,802,802,802,802,802,802,802,802,802,803,803,803,803,803,803,803,803,803,803,803,803,803,803,803,803,803,803,804,804,804,804,804,804,804,804,804,804,804,804,804,804,804,804,804,804,805,805,805,805,805,805,805,805,805,805,805,805,805,805,805,805,805,805,806,806,806,806,806,806,806,806,806,806,806,806,806,806,806,806,806,806,807,807,807,807,807,807,807,807,807,807,807,807,807,807,807,807,807,807,808,808,808,808,808,808,808,808,808,808,808,808,808,808,808,808,808,808,809,809,809,809,809,809,809,809,809,809,809,809,809,809,809,809,809,809,810,810,810,810,810,810,810,810,810,810,810,810,810,810,810,810,810,810,811,811,811,811,811,811,811,811,811,811,811,811,811,811,811,811,811,811,812,812,812,812,812,812,812,812,812,812,812,812,812,812,812,812,812,812,813,813,813,813,813,813,813,813,813,813,813,813,813,813,813,813,813,813,814,814,814,814,814,814,814,814,814,814,814,814,814,814,814,814,814,814,815,815,815,815,815,815,815,815,815,815,815,815,815,815,815,815,815,815,816,816,816,816,816,816,816,816,816,816,816,816,816,816,816,816,816,816,817,817,817,817,817,817,817,817,817,817,817,817,817,817,817,817,817,817,818,818,818,818,818,818,818,818,818,818,818,818,818,818,818,818,818,818,819,819,819,819,819,819,819,819,819,819,819,819,819,819,819,819,819,819,820,820,820,820,820,820,820,820,820,820,820,820,820,820,820,820,820,820,821,821,821,821,821,821,821,821,821,821,821,821,821,821,821,821,821,821,822,822,822,822,822,822,822,822,822,822,822,822,822,822,822,822,822,822,823,823,823,823,823,823,823,823,823,823,823,823,823,823,823,823,823,823,824,824,824,824,824,824,824,824,824,824,824,824,824,824,824,824,824,824,825,825,825,825,825,825,825,825,825,825,825,825,825,825,825,825,825,825,826,826,826,826,826,826,826,826,826,826,826,826,826,826,826,826,826,826,827,827,827,827,827,827,827,827,827,827,827,827,827,827,827,827,827,827,828,828,828,828,828,828,828,828,828,828,828,828,828,828,828,828,828,828,829,829,829,829,829,829,829,829,829,829,829,829,829,829,829,829,829,829,830,830,830,830,830,830,830,830,830,830,830,830,830,830,830,830,830,830,831,831,831,831,831,831,831,831,831,831,831,831,831,831,831,831,831,831,832,832,832,832,832,832,832,832,832,832,832,832,832,832,832,832,832,832,833,833,833,833,833,833,833,833,833,833,833,833,833,833,833,833,833,833,834,834,834,834,834,834,834,834,834,834,834,834,834,834,834,834,834,834,835,835,835,835,835,835,835,835,835,835,835,835,835,835,835,835,835,835,836,836,836,836,836,836,836,836,836,836,836,836,836,836,836,836,836,836,837,837,837,837,837,837,837,837,837,837,837,837,837,837,837,837,837,837,838,838,838,838,838,838,838,838,838,838,838,838,838,838,838,838,838,838,839,839,839,839,839,839,839,839,839,839,839,839,839,839,839,839,839,839,840,840,840,840,840,840,840,840,840,840,840,840,840,840,840,840,840,840,841,841,841,841,841,841,841,841,841,841,841,841,841,841,841,841,841,841,842,842,842,842,842,842,842,842,842,842,842,842,842,842,842,842,842,842,843,843,843,843,843,843,843,843,843,843,843,843,843,843,843,843,843,843,844,844,844,844,844,844,844,844,844,844,844,844,844,844,844,844,844,844,845,845,845,845,845,845,845,845,845,845,845,845,845,845,845,845,845,845,846,846,846,846,846,846,846,846,846,846,846,846,846,846,846,846,846,846,847,847,847,847,847,847,847,847,847,847,847,847,847,847,847,847,847,847,848,848,848,848,848,848,848,848,848,848,848,848,848,848,848,848,848,848,849,849,849,849,849,849,849,849,849,849,849,849,849,849,849,849,849,849,850,850,850,850,850,850,850,850,850,850,850,850,850,850,850,850,850,850,851,851,851,851,851,851,851,851,851,851,851,851,851,851,851,851,851,851,852,852,852,852,852,852,852,852,852,852,852,852,852,852,852,852,852,852,853,853,853,853,853,853,853,853,853,853,853,853,853,853,853,853,853,853,854,854,854,854,854,854,854,854,854,854,854,854,854,854,854,854,854,854,855,855,855,855,855,855,855,855,855,855,855,855,855,855,855,855,855,855,856,856,856,856,856,856,856,856,856,856,856,856,856,856,856,857,857,857,857,857,857,857,857,857,857,857,857,857,857,857,857,857,857,858,858,858,858,858,858,858,858,858,858,858,858,858,858,858,858,858,858,859,859,859,859,859,859,859,859,859,859,859,859,859,859,859,859,859,859,860,860,860,860,860,860,860,860,860,860,860,860,860,860,860,860,860,860,861,861,861,861,861,861,861,861,861,861,861,861,861,861,861,861,861,861,862,862,862,862,862,862,862,862,862,862,862,862,862,862,862,862,862,862,863,863,863,863,863,863,863,863,863,863,863,863,863,863,863,864,864,864,864,864,864,864,864,864,864,864,864,864,864,864,864,864,864,865,865,865,865,865,865,865,865,865,865,865,865,865,865,865,865,865,865,866,866,866,866,866,866,866,866,866,866,866,866,866,866,866,866,866,866,867,867,867,867,867,867,867,867,867,867,867,867,867,867,867,867,867,867,868,868,868,868,868,868,868,868,868,868,868,868,868,868,868,868,868,868,869,869,869,869,869,869,869,869,869,869,869,869,869,869,869,869,869,869,870,870,870,870,870,870,870,870,870,870,870,870,870,870,870,870,870,870,871,871,871,871,871,871,871,871,871,871,871,871,871,871,871,871,871,871,872,872,872,872,872,872,872,872,872,872,872,872,872,872,872,872,872,872,873,873,873,873,873,873,873,873,873,873,873,873,873,873,873,873,873,873,874,874,874,874,874,874,874,874,874,874,874,874,874,874,874,874,874,874,875,875,875,875,875,875,875,875,875,875,875,875,875,875,875,875,875,875,876,876,876,876,876,876,876,876,876,876,876,876,876,876,876,876,876,876,877,877,877,877,877,877,877,877,877,877,877,877,877,877,877,878,878,878,878,878,878,878,878,878,878,878,878,878,878,878,878,878,878,879,879,879,879,879,879,879,879,879,879,879,879,879,879,879,879,879,879,880,880,880,880,880,880,880,880,880,880,880,880,880,880,880,880,880,880,881,881,881,881,881,881,881,881,881,881,881,881,881,881,881,881,881,881,882,882,882,882,882,882,882,882,882,882,882,882,882,882,882,882,882,882,883,883,883,883,883,883,883,883,883,883,883,883,883,883,883,883,883,883,884,884,884,884,884,884,884,884,884,884,884,884,884,884,884,884,884,884,885,885,885,885,885,885,885,885,885,885,885,885,885,885,885,885,885,885,886,886,886,886,886,886,886,886,886,886,886,886,886,886,886,886,886,886,887,887,887,887,887,887,887,887,887,887,887,887,887,887,887,887,887,887,888,888,888,888,888,888,888,888,888,888,888,888,888,888,888,888,888,888,889,889,889,889,889,889,889,889,889,889,889,889,889,889,889,889,889,889,890,890,890,890,890,890,890,890,890,890,890,890,890,890,890,890,890,890,891,891,891,891,891,891,891,891,891,891,891,891,891,891,891,891,891,891,892,892,892,892,892,892,892,892,892,892,892,892,892,892,892,892,892,892,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,893,894,894,894,894,894,894,894,894,894,894,894,894,894,894,894,894,894,894,895,895,895,895,895,895,895,895,895,895,895,895,895,895,895,895,895,895,896,896,896,896,896,896,896,896,896,896,896,896,896,896,896,896,896,896,897,897,897,897,897,897,897,897,897,897,897,897,897,897,897,897,897,897,898,898,898,898,898,898,898,898,898,898,898,898,898,898,898,898,898,898,899,899,899,899,899,899,899,899,899,899,899,899,899,899,899,899,899,899,900,900,900,900,900,900,900,900,900,900,900,900,900,900,900,900,900,900,901,901,901,901,901,901,901,901,901,901,901,901,901,901,901,901,901,901,902,902,902,902,902,902,902,902,902,902,902,902,902,902,902,902,902,902,903,903,903,903,903,903,903,903,903,903,903,903,903,903,903,903,903,903,904,904,904,904,904,904,904,904,904,904,904,904,904,904,904,904,904,904,905,905,905,905,905,905,905,905,905,905,905,905,905,905,905,905,905,905,906,906,906,906,906,906,906,906,906,906,906,906,906,906,906,906,906,906,907,907,907,907,907,907,907,907,907,907,907,907,907,907,907,907,907,907,908,908,908,908,908,908,908,908,908,908,908,908,908,908,908,908,908,908,909,909,909,909,909,909,909,909,909,909,909,909,909,909,909,909,909,909,910,910,910,910,910,910,910,910,910,910,910,910,910,910,910,910,910,910,911,911,911,911,911,911,911,911,911,911,911,911,911,911,911,911,911,911,912,912,912,912,912,912,912,912,912,912,912,912,912,912,912,912,912,912,913,913,913,913,913,913,913,913,913,913,913,913,913,913,913,913,913,913,914,914,914,914,914,914,914,914,914,914,914,914,914,914,914,914,914,914,915,915,915,915,915,915,915,915,915,915,915,915,915,915,915,915,915,915,916,916,916,916,916,916,916,916,916,916,916,916,916,916,916,916,916,916,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,917,918,918,918,918,918,918,918,918,918,918,918,918,918,918,918,918,918,918,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,919,920,920,920,920,920,920,920,920,920,920,920,920,920,920,920,920,920,920,921,921,921,921,921,921,921,921,921,921,921,921,921,921,921,921,921,921,922,922,922,922,922,922,922,922,922,922,922,922,922,922,922,922,922,922,923,923,923,923,923,923,923,923,923,923,923,923,923,923,923,923,923,923,924,924,924,924,924,924,924,924,924,924,924,924,924,924,924,924,924,924,925,925,925,925,925,925,925,925,925,925,925,925,925,925,925,925,925,925,926,926,926,926,926,926,926,926,926,926,926,926,926,926,926,926,926,926,927,927,927,927,927,927,927,927,927,927,927,927,927,927,927,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,928,929,929,929,929,929,929,929,929,929,929,929,929,929,929,929,929,929,929,930,930,930,930,930,930,930,930,930,930,930,930,930,930,930,930,930,930,931,931,931,931,931,931,931,931,931,931,931,931,931,931,931,931,931,931,932,932,932,932,932,932,932,932,932,932,932,932,932,932,932,932,932,932,933,933,933,933,933,933,933,933,933,933,933,933,933,933,933,933,933,933,934,934,934,934,934,934,934,934,934,934,934,934,934,934,934,934,934,934,935,935,935,935,935,935,935,935,935,935,935,935,935,935,935,935,935,935,936,936,936,936,936,936,936,936,936,936,936,936,936,936,936,936,936,936,937,937,937,937,937,937,937,937,937,937,937,937,937,937,937,937,937,937,938,938,938,938,938,938,938,938,938,938,938,938,938,938,938,939,939,939,939,939,939,939,939,939,939,939,939,939,939,939,939,939,939,940,940,940,940,940,940,940,940,940,940,940,940,940,940,940,940,940,940,941,941,941,941,941,941,941,941,941,941,941,941,941,941,941,941,941,941,942,942,942,942,942,942,942,942,942,942,942,942,942,942,942,942,942,942,943,943,943,943,943,943,943,943,943,943,943,943,943,943,943,943,943,943,944,944,944,944,944,944,944,944,944,944,944,944,944,944,944,944,944,944,945,945,945,945,945,945,945,945,945,945,945,945,945,945,945,945,945,945,946,946,946,946,946,946,946,946,946,946,946,946,946,946,946,946,946,946,947,947,947,947,947,947,947,947,947,947,947,947,947,947,947,947,947,947,948,948,948,948,948,948,948,948,948,948,948,948,948,948,948,948,948,948,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,949,950,950,950,950,950,950,950,950,950,950,950,950,950,950,950,951,951,951,951,951,951,951,951,951,951,951,951,951,951,951,951,951,951,952,952,952,952,952,952,952,952,952,952,952,952,952,952,952,952,952,952,953,953,953,953,953,953,953,953,953,953,953,953,953,953,953,953,953,953,954,954,954,954,954,954,954,954,954,954,954,954,954,954,954,954,954,954,955,955,955,955,955,955,955,955,955,955,955,955,955,955,955,956,956,956,956,956,956,956,956,956,956,956,956,956,956,956,956,956,956,957,957,957,957,957,957,957,957,957,957,957,957,957,957,957,957,957,957,958,958,958,958,958,958,958,958,958,958,958,958,958,958,958,958,958,958,959,959,959,959,959,959,959,959,959,959,959,959,959,959,959,959,959,959,960,960,960,960,960,960,960,960,960,960,960,960,960,960,960,960,960,960,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,961,962,962,962,962,962,962,962,962,962,962,962,962,962,962,962,962,962,962,963,963,963,963,963,963,963,963,963,963,963,963,963,963,963,963,963,963,964,964,964,964,964,964,964,964,964,964,964,964,964,964,964,964,964,964,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,965,966,966,966,966,966,966,966,966,966,966,966,966,966,966,966,966,966,966,967,967,967,967,967,967,967,967,967,967,967,967,967,967,967,967,967,967,968,968,968,968,968,968,968,968,968,968,968,968,968,968,968,968,968,968,969,969,969,969,969,969,969,969,969,969,969,969,969,969,969,969,969,969,970,970,970,970,970,970,970,970,970,970,970,970,970,970,970,970,970,970,971,971,971,971,971,971,971,971,971,971,971,971,971,971,971,971,971,971,972,972,972,972,972,972,972,972,972,972,972,972,972,972,972,972,972,972,973,973,973,973,973,973,973,973,973,973,973,973,973,973,973,973,973,973,974,974,974,974,974,974,974,974,974,974,974,974,974,974,974,974,974,974,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,975,976,976,976,976,976,976,976,976,976,976,976,976,976,976,976,976,976,976,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,977,978,978,978,978,978,978,978,978,978,978,978,978,978,978,978,978,978,978,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,979,980,980,980,980,980,980,980,980,980,980,980,980,980,980,980,980,980,980,981,981,981,981,981,981,981,981,981,981,981,981,981,981,981,981,981,981,982,982,982,982,982,982,982,982,982,982,982,982,982,982,982,982,982,982,983,983,983,983,983,983,983,983,983,983,983,983,983,983,983,983,983,983,984,984,984,984,984,984,984,984,984,984,984,984,984,984,984,984,984,984,985,985,985,985,985,985,985,985,985,985,985,985,985,985,985,985,985,985,986,986,986,986,986,986,986,986,986,986,986,986,986,986,986,986,986,986,987,987,987,987,987,987,987,987,987,987,987,987,987,987,987,987,987,987,988,988,988,988,988,988,988,988,988,988,988,988,988,988,988,988,988,988,989,989,989,989,989,989,989,989,989,989,989,989,989,989,989,989,989,989,990,990,990,990,990,990,990,990,990,990,990,990,990,990,990,990,990,990,991,991,991,991,991,991,991,991,991,991,991,991,991,991,991,991,991,991,992,992,992,992,992,992,992,992,992,992,992,992,992,992,992,992,992,992,993,993,993,993,993,993,993,993,993,993,993,993,993,993,993,993,993,993,994,994,994,994,994,994,994,994,994,994,994,994,994,994,994,994,994,994,995,995,995,995,995,995,995,995,995,995,995,995,995,995,995,995,995,995,996,996,996,996,996,996,996,996,996,996,996,996,996,996,996,996,996,996,997,997,997,997,997,997,997,997,997,997,997,997,997,997,997,997,997,997,998,998,998,998,998,998,998,998,998,998,998,998,998,998,998,998,998,998,999,999,999,999,999,999,999,999,999,999,999,999,999,999,999,999,999,999,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1000,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1001,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1002,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1003,1004,1004,1004,1004,1004,1004,1004,1004,1004,1004,1004,1004,1004,1004,1004,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1005,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1006,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1007,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1008,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1009,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1010,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1011,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1012,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1013,1014,1014,1014,1014,1014,1014,1014,1014,1014,1014,1014,1014,1014,1014,1014,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1015,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1016,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1017,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1018,1019,1019,1019,1019,1019,1019,1019,1019,1019,1019,1019,1019,1019,1019,1019,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1020,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1021,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1022,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1023,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1024,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1025,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1026,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1027,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1028,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1029,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1030,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1031,1032,1032,1032,1032,1032,1032,1032,1032,1032,1032,1032,1032,1032,1032,1032,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1033,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1034,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1035,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1036,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1037,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1038,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1039,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1040,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1041,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1042,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1043,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1044,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1045,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1046,1047,1047,1047,1047,1047,1047,1047,1047,1047,1047,1047,1047,1047,1047,1047,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1048,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1049,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1050,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1051,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1052,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1053,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1054,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1055,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1056,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1057,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1058,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1059,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1060,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1061,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1062,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1063,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1064,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1065,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1066,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1067,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1068,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1069,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1070,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1071,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1072,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1073,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1074,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1075,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1076,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1077,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1078,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1079,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1080,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1081,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1082,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1083,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1084,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1085,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1086,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1087,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1088,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1089,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1090,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1091,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1092,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1093,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1094,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1095,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1096,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1097,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1098,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1099,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1100,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1101,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1102,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1103,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1104,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1105,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1106,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1107,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1108,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1109,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1110,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1111,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1112,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1113,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1114,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1115,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1116,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1117,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1118,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1119,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1120,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1121,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1122,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1123,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1124,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1125,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1126,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1127,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1128,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1129,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1130,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1131,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1132,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1133,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1134,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1135,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1136,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1137,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1138,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1139,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1140,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1141,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1142,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1143,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1144,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1145,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1146,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1147,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1148,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1149,1150,1150,1150,1150,1150,1150,1150,1150,1150,1150,1150,1150,1150,1150,1150,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1151,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1152,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1153,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1154,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1155,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1156,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1157,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1158,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1159,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1160,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1161,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1162,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1163,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1164,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1165,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1166,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1167,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1168,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1169,1170,1170,1170,1170,1170,1170,1170,1170,1170,1170,1170,1170,1170,1170,1170,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1171,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1172,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1173,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1174,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1175,1176,1176,1176,1176,1176,1176,1176,1176,1176,1176,1176,1176,1176,1176,1176,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1177,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1178,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1179,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1180,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1181,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1182,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1183,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1184,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1185,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1186,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1187,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1188,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1189,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1190,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1191,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1192,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1193,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1194,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1195,1196,1196,1196,1196,1196,1196,1196,1196,1196,1196,1196,1196,1196,1196,1196,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1197,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1198,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1199,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1200,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1201,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1202,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1203,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1204,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1205,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1206,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1207,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1208,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1209,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1210,1211,1211,1211,1211,1211,1211,1211,1211,1211,1211,1211,1211,1211,1211,1211,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1212,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1213,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1214,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1215,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1216,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1217,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1218,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1219,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1220,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1221,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1222,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1223,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1224,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1225,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1226,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1227,1228,1228,1228,1228,1228,1228,1228,1228,1228,1228,1228,1228,1228,1228,1228,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1229,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1230,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1231,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1232,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1233,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1234,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1235,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1236,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1237,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1238,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1239,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1240,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1241,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1242,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1243,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1244,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1245,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1246,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1247,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1248,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1249,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1250,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1251,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1252,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1253,1254,1254,1254,1254,1254,1254,1254,1254,1254,1254,1254,1254,1254,1254,1254,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1255,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1256,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1257,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1258,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1259,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1260,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1261,1262,1262,1262,1262,1262,1262,1262,1262,1262,1262,1262,1262,1262,1262,1262,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1263,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1264,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1265,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1266,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1267,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1268,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1269,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1270,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1271,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1272,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1273,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1274,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1275,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1276,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1277,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1278,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1279,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1280,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1281,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1282,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1283,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1284,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1285,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1286,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1287,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1288,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1289,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1290,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1291,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1292,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1293,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1294,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1295,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1296,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1297,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1298,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1299,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1300,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1301,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1302,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1303,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1304,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1305,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1306,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1307,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1308,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1309,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1310,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1311,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1312,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1313,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1314,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1315,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1316,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1317,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1318,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1319,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1320,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1321,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1322,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1323,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1324,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1325,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1326,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1327,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1328,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1329,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1330,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1331,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1332,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1333,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1334,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1335,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1336,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1337,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1338,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1339,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1340,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1341,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1342,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1343,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1344,1345,1345,1345,1345,1345,1345,1345,1345,1345,1345,1345,1345,1345,1345,1345,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1346,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1347,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1348,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1349,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1350,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1351,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1352,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1353,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1354,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1355,1356,1356,1356,1356,1356,1356,1356,1356,1356,1356,1356,1356,1356,1356,1356,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1357,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1358,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1359,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1360,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1361,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1362,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1363,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1364,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1365,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1366,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1367,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1368,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1369,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1370,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1371,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1372,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1373,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1374,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1375,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1376,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1377,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1378,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1379,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1380,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1381,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1382,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1383,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1384,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1385,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1386,1387,1387,1387,1387,1387,1387,1387,1387,1387,1387,1387,1387,1387,1387,1387,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1388,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1389,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1390,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1391,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1392,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1393,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1394,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1395,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1396,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1397,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1398,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1399,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1400,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1401,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1402,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1403,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1404,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1405,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1406,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1407,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1408,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1409,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1410,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1411,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1412,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1413,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1414,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1415,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1416,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1417,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1418,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1419,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1420,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1421,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1422,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1423,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1424,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1425,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1426,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1427,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1428,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1429,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1430,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1431,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1432,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1433,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1434,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1435,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1436,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1437,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1438,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1439,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1440,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1441,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1442,1443,1443,1443,1443,1443,1443,1443,1443,1443,1443,1443,1443,1443,1443,1443,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1444,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1445,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1446,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1447,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1448,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1449,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1450,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1451,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1452,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1453,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1454,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1455,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1456,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1457,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1458,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1459,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1460,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1461,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1462,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1463,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1464,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1465,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1466,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1467,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1468,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1469,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1470,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1471,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1472,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1473,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1474,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1475,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1476,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1477,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1478,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1479,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1480,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1481,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1482,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1483,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1484,1485,1485,1485,1485,1485,1485,1485,1485,1485,1485,1485,1485,1485,1485,1485,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1486,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1487,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1488,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1489,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1490,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1491,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1492,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1493,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1494,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1495,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1496,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1497,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1498,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1499,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1500,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1501,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1502,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1503,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1504,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1505,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1506,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1507,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1508,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1509,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1510,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1511,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1512,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1513,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1514,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1515,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1516,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1517,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1518,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1519,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1521,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1522,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1523,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1524,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1525,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1526,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1527,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1528,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1529,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1530,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1531,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1532,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1533,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1534,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1535,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1536,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1537,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1538,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1539,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1540,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1541,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1542,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1543,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1544,1545,1545,1545,1545,1545,1545,1545,1545,1545,1545,1545,1545,1545,1545,1545,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1546,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1547,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1548,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1549,1550,1550,1550,1550,1550,1550,1550,1550,1550,1550,1550,1550,1550,1550,1550,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1551,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1552,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1553,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1554,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1555,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1556,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1557,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1558,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1559,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1560,1561,1561,1561,1561,1561,1561,1561,1561,1561,1561,1561,1561,1561,1561,1561,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1562,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1563,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1564,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1565,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1566,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1567,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1568,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1569,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1570,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1571,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1572,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1573,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1574,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1575,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1576,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1577,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1578,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1579,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1580,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1581,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1582,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1583,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1584,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1585,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1586,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1587,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1588,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1589,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1590,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1591,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1592,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1593,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1594,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1595,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1596,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1597,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1598,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1599,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1600,1601,1601,1601,1601,1601,1601,1601,1601,1601,1601,1601,1601,1601,1601,1601,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1602,1603,1603,1603,1603,1603,1603,1603,1603,1603,1603,1603,1603,1603,1603,1603,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1604,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1605,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1606,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1607,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1608,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1609,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1610,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1611,1612,1612,1612,1612,1612,1612,1612,1612,1612,1612,1612,1612,1612,1612,1612,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1613,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1614,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1615,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1616,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1617,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1618,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1619,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1620,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1621,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1622,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1623,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1624,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1625,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1626,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1627,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1628,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1629,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1630,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1631,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1632,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1633,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1634,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1635,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1636,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1637,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1638,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1639,1640,1640,1640,1640,1640,1640,1640,1640,1640,1640,1640,1640,1640,1640,1640,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1641,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1642,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1643,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1644,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1645,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1646,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1647,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1648,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1649,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1650,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1651,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1652,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1653,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1654,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1655,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1656,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1657,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1658,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1659,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1660,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1661,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1662,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1663,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1664,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1665,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1666,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1667,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1668,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1669,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1670,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1671,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1672,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1673,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1674,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1675,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1676,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1677,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1678,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1679,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1680,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1681,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1682,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1683,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1684,1685,1685,1685,1685,1685,1685,1685,1685,1685,1685,1685,1685,1685,1685,1685,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1686,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1687,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1688,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1689,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1690,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1691,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1692,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1693,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1694,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1695,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1696,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1697,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1698,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1699,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1700,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1701,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1702,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1703,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1704,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1705,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1706,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1707,1708,1708,1708,1708,1708,1708,1708,1708,1708,1708,1708,1708,1708,1708,1708,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1709,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1710,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1711,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1712,1713,1713,1713,1713,1713,1713,1713,1713,1713,1713,1713,1713,1713,1713,1713,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1714,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1715,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1716,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1717,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1718,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1719,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1720,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1721,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1722,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1723,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1724,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1725,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1726,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1727,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1728,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1729,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1730,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1731,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1732,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1733,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1734,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1735,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1736,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1737,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1738,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1739,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1740,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1741,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1742,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1743,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1744,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1745,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1746,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1747,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1748,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1749,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1750,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1751,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1752,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1753,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1754,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1755,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1756,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1757,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1758,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1759,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1760,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1761,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1762,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1763,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1764,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1765,1766,1766,1766,1766,1766,1766,1766,1766,1766,1766,1766,1766,1766,1766,1766,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1767,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1768,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1769,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1770,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1771,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1772,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1773,1774,1774,1774,1774,1774,1774,1774,1774,1774,1774,1774,1774,1774,1774,1774,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1775,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1776,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1777,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1778,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1779,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1780,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1781,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1782,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1783,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1784,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1785,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1786,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1787,1788,1788,1788,1788,1788,1788,1788,1788,1788,1788,1788,1788,1788,1788,1788,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1789,1790,1790,1790,1790,1790,1790,1790,1790,1790,1790,1790,1790,1790,1790,1790,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1791,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1792,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1793,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1794,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1795,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1796,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1797,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1798,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1799,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1800,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1801,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1802,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1803,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1804,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1805,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1806,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1807,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1808,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1809,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1810,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1811,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1812,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1813,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1814,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1815,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1816,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1817,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1818,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1819,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1820,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1821,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1822,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1823,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1824,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1825,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1826,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1827,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1828,1829,1829,1829,1829,1829,1829,1829,1829,1829,1829,1829,1829,1829,1829,1829,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1830,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1831,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1832,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1833,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1834,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1835,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1836,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1837,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1838,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1839,1840,1840,1840,1840,1840,1840,1840,1840,1840,1840,1840,1840,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1841,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1842,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1843,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1844,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1845,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1846,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1847,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1848,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1849,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1850,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1851,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1852,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1853,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1854,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1855,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1856,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1857,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1858,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1859,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1860,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1861,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1862,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1863,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1864,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1865,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1866,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1867,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1868,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1869,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1870,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1871,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1872,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1873,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1874,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1875,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1876,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1877,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1878,1879,1879,1879,1879,1879,1879,1879,1879,1879,1879,1879,1879,1879,1879,1879,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1880,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1881,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1882,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1883,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1884,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1885,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1886,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1887,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1888,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1889,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1890,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1891,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1892,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1893,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1894,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1895,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1896,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1897,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1898,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1899,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1900,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1901,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1902,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1903,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1904,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1905,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1906,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1907,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1908,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1909,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1910,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1911,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1912,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1913,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1914,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1915,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1916,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1917,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1918,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1919,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1920,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1921,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1922,1923,1923,1923,1923,1923,1923,1923,1923,1923,1923,1923,1923,1923,1923,1923,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1924,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1925,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1926,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1927,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1928,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1929,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1930,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1931,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1932,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1933,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1934,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1935,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1936,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1937,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1938,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1939,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1940,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1941,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1942,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1943,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1944,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1945,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1946,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1947,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1948,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1949,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1950,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1951,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1952,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1953,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1954,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1955,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1956,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1957,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1958,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1959,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1960,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1961,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1962,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1963,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1964,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1965,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1966,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1967,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1968,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1969,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1970,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1971,1972,1972,1972,1972,1972,1972,1972,1972,1972,1972,1972,1972,1972,1972,1972,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1973,1974,1974,1974,1974,1974,1974,1974,1974,1974,1974,1974,1974,1974,1974,1974,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1975,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1976,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1977,1978,1978,1978,1978,1978,1978,1978,1978,1978,1978,1978,1978,1978,1978,1978,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1979,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1980,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1981,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1982,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1983,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1984,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1985,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1986,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1987,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1988,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1989,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1990,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1991,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1992,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1993,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1994,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1995,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1996,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1997,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1998,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,1999,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2000,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2001,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2002,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2003,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2004,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2005,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2006,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2007,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2008,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2009,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2010,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2011,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2012,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2013,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2014,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2015,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2016,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2017,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2018,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2019,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2020,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2021,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2022,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2023,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2024,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2025,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2026,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2027,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2028,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2029,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2030,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2031,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2032,2033,2033,2033,2033,2033,2033,2033,2033,2033,2033,2033,2033,2033,2033,2033,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2034,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2035,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2036,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2037,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2038,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2039,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2040,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2041,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2042,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2043,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2044,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2045,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2046,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2047,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2048,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2049,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2050,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2051,2052,2052,2052,2052,2052,2052,2052,2052,2052,2052,2052,2052,2052,2052,2052,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2053,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2054,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2055,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2056,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2057,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2058,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2059,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2060,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2061,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2062,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2063,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2064,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2065,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2066,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2067,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2068,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2069,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2070,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2071,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2072,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2073,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2074,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2075,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2076,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2077,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2078,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2079,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2080,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2081,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2082,2083,2083,2083,2083,2083,2083,2083,2083,2083,2083,2083,2083,2083,2083,2083,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2084,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2085,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2086,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2087,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2088,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2089,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2090,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2091,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2092,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2093,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2094,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2095,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2096,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2097,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2098,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2099,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2100,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2101,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2102,2103,2103,2103,2103,2103,2103,2103,2103,2103,2103,2103,2103,2103,2103,2103,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2104,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2105,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2106,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2107,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2108,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2109,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2110,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2111,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2112,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2113,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2114,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2115,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2116,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2117,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2118,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2119,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2120,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2121,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2122,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2123,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2124,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2125,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2126,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2127,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2128,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2129,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2130,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2131,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2132,2133,2133,2133,2133,2133,2133,2133,2133,2133,2133,2133,2133,2133,2133,2133,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2134,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2135,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2136,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2137,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2138,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2139,2140,2140,2140,2140,2140,2140,2140,2140,2140,2140,2140,2140,2140,2140,2140,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2141,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2142,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2143,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2144,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2145,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2146,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2147,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2148,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2149,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2150,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2151,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2152,2153,2153,2153,2153,2153,2153,2153,2153,2153,2153,2153,2153,2153,2153,2153,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2154,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2155,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2156,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2157,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2158,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2159,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2160,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2161,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2162,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2163,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2164,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2165,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2166,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2167,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2168,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2169,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2170,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2171,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2172,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2173,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2174,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2175,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2176,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2177,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2178,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2179,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2180,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2181,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2182,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2183,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2184,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2185,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2186,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2187,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2188,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2189,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2190,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2191,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2192,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2193,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2194,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2195,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2196,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2197,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2198,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2199,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2200,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2201,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2202,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2203,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2204,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2205,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2206,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2207,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2208,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2209,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2210,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2211,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2212,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2213,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2214,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2215,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2216,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2217,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2218,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2219,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2220,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2221,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2222,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2223,2224,2224,2224,2224,2224,2224,2224,2224,2224,2224,2224,2224,2224,2224,2224,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2225,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2226,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2227,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2228,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2229,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2230,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2231,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2232,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2233,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2234,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2235,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2236,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2237,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2238,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2239,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2240,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2241,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2242,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2243,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2244,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2245,2246,2246,2246,2246,2246,2246,2246,2246,2246,2246,2246,2246,2246,2246,2246,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2247,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2248,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2249,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2250,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2251,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2252,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2253,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2254,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2255,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2256,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2257,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2258,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2259,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2260,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2261,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2262,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2263,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2264,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2265,2266,2266,2266,2266,2266,2266,2266,2266,2266,2266,2266,2266,2266,2266,2266,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2267,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2268,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2269,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2270,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2271,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2272,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2273,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2274,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2275,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2276,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2277,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2278,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2279,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2280,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2281,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2282,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2283,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2284,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2285,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2286,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2287,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2288,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2289,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2290,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2291,2292,2292,2292,2292,2292,2292,2292,2292,2292,2292,2292,2292,2292,2292,2292,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2293,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2294,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2295,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2296,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2297,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2298,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2299,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2300,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2301,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2302,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2303,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2304,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2305,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2306,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2307,2308,2308,2308,2308,2308,2308,2308,2308,2308,2308,2308,2308,2308,2308,2308,2309,2309,2309,2309,2309,2309,2309,2309,2309,2309,2309,2309,2309,2309,2309,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2310,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2311,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2312,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2313,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2314,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2315,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2316,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2317,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2318,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2319,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2320,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2321,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2322,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2323,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2324,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2325,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2326,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2327,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2328,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2329,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2330,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2331,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2332,2333,2333,2333,2333,2333,2333,2333,2333,2333,2333,2333,2333,2333,2333,2333,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2334,2335,2335,2335,2335,2335,2335,2335,2335,2335,2335,2335,2335,2335,2335,2335,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2336,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2337,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2338,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2339,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2340,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2341,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2342,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2343,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2344,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2345,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2346,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2347,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2348,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2349,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2350,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2351,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2352,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2353,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2354,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2355,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2356,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2357,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2358,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2359,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2360,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2361,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2362,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2363,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2364,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2365,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2366,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2367,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2368,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2369,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2370,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2371,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2372,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2373,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2374,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2375,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2376,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2377,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2378,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2379,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2380,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2381,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2382,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2383,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2384,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2385,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2386,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2387,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2388,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2389,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2390,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2391,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2392,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2393,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2394,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2395,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2396,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2397,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2398,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2399,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2400,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2401,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2402,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2403,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2404,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2405,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2406,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2407,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2408,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2409,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2410,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2411,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2412,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2413,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2414,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2415,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2416,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2417,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2418,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2419,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2420,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2421,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2422,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2423,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2424,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2425,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2426,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2427,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2428,2429,2429,2429,2429,2429,2429,2429,2429,2429,2429,2429,2429,2429,2429,2429,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2430,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2431,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2432,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2433,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2434,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2435,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2436,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2437,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2438,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2439,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2440,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2441,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2442,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2443,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2444,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2445,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2446,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2447,2448,2448,2448,2448,2448,2448,2448,2448,2448,2448,2448,2448,2448,2448,2448,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2449,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2450,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2451,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2452,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2453,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2454,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2455,2456,2456,2456,2456,2456,2456,2456,2456,2456,2456,2456,2456,2456,2456,2456,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2457,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2458,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2459,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2460,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2461,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2462,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2463,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2464,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2465,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2466,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2467,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2468,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2469,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2470,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2471,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2472,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2473,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2474,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2475,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2476,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2477,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2478,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2479,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2481,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2482,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2483,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2484,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2485,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2486,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2487,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2488,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2489,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2490,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2491,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2492,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2493,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2494,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2495,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2496,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2497,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2498,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2499,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2500,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2501,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2502,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2503,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2504,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2505,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2506,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2507,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2508,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2509,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2510,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2511,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2512,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2513,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2514,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2515,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2516,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2517,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2518,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2519,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2520,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2521,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2522,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2523,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2524,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2525,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2526,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2527,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2528,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2529,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2530,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2531,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2532,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2533,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2534,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2535,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2536,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2537,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2538,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2539,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2540,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2541,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2542,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2543,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2544,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2545,2546,2546,2546,2546,2546,2546,2546,2546,2546,2546,2546,2546,2546,2546,2546,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2547,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2548,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2549,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2550,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2551,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2552,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2553,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2554,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2555,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2556,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2557,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2558,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2559,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2560,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2561,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2562,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2563,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2564,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2565,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2566,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2567,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2568,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2569,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2570,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2571,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2572,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2573,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2574,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2575,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2576,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2577,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2578,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2579,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2580,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2581,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2582,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2583,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2584,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2585,2586,2586,2586,2586,2586,2586,2586,2586,2586,2586,2586,2586,2586,2586,2586,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2587,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2588,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2589,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2590,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2591,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2592,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2593,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2594,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2595,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2596,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2597,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2598,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2599,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2600,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2601,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2602,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2603,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2604,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2605,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2606,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2607,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2608,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2609,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2610,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2611,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2612,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2613,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2614,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2615,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2616,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2617,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2618,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2619,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2620,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2621,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2622,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2623,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2624,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2625,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2626,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2627,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2628,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2629,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2630,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2631,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2632,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2633,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2634,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2635,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2636,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2637,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2638,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2639,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2640,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2641,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2642,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2643,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2644,2645,2645,2645,2645,2645,2645,2645,2645,2645,2645,2645,2645,2645,2645,2645,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2646,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2647,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2648,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2649,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2650,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2651,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2652,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2653,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2654,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2655,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2656,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2657,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2658,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2659,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2660,2661,2661,2661,2661,2661,2661,2661,2661,2661,2661,2661,2661,2661,2661,2661,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2662,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2663,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2664,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2665,2666,2666,2666,2666,2666,2666,2666,2666,2666,2666,2666,2666,2666,2666,2666,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2667,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2668,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2669,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2670,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2671,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2672,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2673,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2674,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2675,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2676,2677,2677,2677,2677,2677,2677,2677,2677,2677,2677,2677,2677,2677,2677,2677,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2678,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2679,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2680,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2681,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2682,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2683,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2684,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2685,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2686,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2687,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2688,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2689,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2690,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2691,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2692,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2693,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2694,2695,2695,2695,2695,2695,2695,2695,2695,2695,2695,2695,2695,2695,2695,2695,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2696,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2697,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2698,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2699,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2700,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2701,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2702,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2703,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2704,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2705,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2706,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2707,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2708,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2709,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2710,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2711,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2712,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2713,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2714,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2715,2716,2716,2716,2716,2716,2716,2716,2716,2716,2716,2716,2716,2716,2716,2716,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2717,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2718,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2719,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2720,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2721,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2722,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2723,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2724,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2725,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2726,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2727,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2728,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2729,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2730,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2731,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2732,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2733,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2734,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2735,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2736,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2737,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2738,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2739,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2740,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2741,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2742,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2743,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2744,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2745,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2746,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2747,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2748,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2749,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2750,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2751,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2752,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2753,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2754,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2755,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2756,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2757,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2758,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2759,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2760,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2761,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2762,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2763,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2764,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2765,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2766,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2767,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2768,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2769,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2770,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2771,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2772,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2773,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2774,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2775,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2776,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2777,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2778,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2779,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2780,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2781,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2782,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2783,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2784,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2785,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2786,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2787,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2788,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2789,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2790,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2791,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2792,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2793,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2794,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2795,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2796,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2797,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2798,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2799,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2800,2801,2801,2801,2801,2801,2801,2801,2801,2801,2801,2801,2801,2801,2801,2801,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2802,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2803,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2804,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2805,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2806,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2807,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2808,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2809,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2810,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2811,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2812,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2813,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2814,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2815,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2816,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2817,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2818,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2819,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2820,2821,2821,2821,2821,2821,2821,2821,2821,2821,2821,2821,2821,2821,2821,2821,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2822,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2823,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2824,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2825,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2826,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2827,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2828,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2829,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2830,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2831,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2832,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2833,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2834,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2835,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2836,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2837,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2838,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2839,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2840,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2841,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2842,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2843,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2844,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2845,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2846,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2847,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2848,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2849,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2850,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2851,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2852,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2853,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2854,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2855,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2856,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2857,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2858,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2859,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2860,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2861,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2862,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2863,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2864,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2865,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2866,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2867,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2868,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2869,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2870,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2871,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2872,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2873,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2874,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2875,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2876,2877,2877,2877,2877,2877,2877,2877,2877,2877,2877,2877,2877,2877,2877,2877,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2878,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2879,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2880,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2881,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2882,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2883,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2884,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2885,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2886,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2887,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2888,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2889,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2890,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2891,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2892,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2893,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2894,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2895,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2896,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2897,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2898,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2899,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2900,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2901,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2902,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2903,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2904,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2905,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2906,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2907,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2908,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2909,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2910,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2911,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2912,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2913,2914,2914,2914,2914,2914,2914,2914,2914,2914,2914,2914,2914,2914,2914,2914,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2915,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2916,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2917,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2918,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2919,2920,2920,2920,2920,2920,2920,2920,2920,2920,2920,2920,2920,2920,2920,2920,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2921,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2922,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2923,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2924,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2925,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2926,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2927,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2928,2929,2929,2929,2929,2929,2929,2929,2929,2929,2929,2929,2929,2929,2929,2929,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2930,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2931,2932,2932,2932,2932,2932,2932,2932,2932,2932,2932,2932,2932,2932,2932,2932,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2933,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2934,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2935,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2936,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2937,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2938,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2939,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2940,2941,2941,2941,2941,2941,2941,2941,2941,2941,2941,2941,2941,2941,2941,2941,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2942,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2943,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2944,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2945,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2946,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2947,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2948,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2949,2950,2950,2950,2950,2950,2950,2950,2950,2950,2950,2950,2950,2950,2950,2950,2951,2951,2951,2951,2951,2951,2951,2951,2951,2951,2951,2951,2951,2951,2951,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2952,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2953,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2954,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2955,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2956,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2957,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2958,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2959,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2960,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2961,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2962,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2963,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2964,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2965,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2966,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2967,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2968,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2969,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2970,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2971,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2972,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2973,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2974,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2975,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2976,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2977,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2978,2979,2979,2979,2979,2979,2979,2979,2979,2979,2979,2979,2979,2979,2979,2979,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2980,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2981,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2982,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2983,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2984,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2985,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2986,2987,2987,2987,2987,2987,2987,2987,2987,2987,2987,2987,2987,2987,2987,2987,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2988,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2989,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2990,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2991,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2992,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2993,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2994,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2995,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2996,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2997,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2998,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,2999,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3000,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3001,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3002,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3003,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3004,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3005,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3006,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3007,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3008,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3009,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3010,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3011,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3012,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3013,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3014,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3015,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3016,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3017,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3018,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3019,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3020,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3021,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3022,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3023,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3024,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3025,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3026,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3027,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3028,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3029,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3030,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3031,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3032,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3033,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3034,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3035,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3036,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3037,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3038,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3039,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3040,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3041,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3042,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3043,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3044,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3045,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3046,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3047,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3048,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3049,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3050,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3051,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3052,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3053,3054,3054,3054,3054,3054,3054,3054,3054,3054,3054,3054,3054,3054,3054,3054,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3055,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3056,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3057,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3058,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3059,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3060,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3061,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3062,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3063,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3064,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3065,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3066,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3067,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3068,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3069,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3070,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3071,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3072,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3073,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3074,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3075,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3076,3077,3077,3077,3077,3077,3077,3077,3077,3077,3077,3077,3077,3077,3077,3077,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3078,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3079,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3080,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3081,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3082,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3083,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3084,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3085,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3086,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3087,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3088,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3089,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3090,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3091,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3092,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3093,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3094,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3095,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3096,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3097,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3098,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3099,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3100,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3101,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3102,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3103,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3104,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3105,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3106,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3107,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3108,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3109,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3110,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3111,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3112,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3113,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3114,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3115,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3116,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3117,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3118,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3119,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3120,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3121,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3122,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3123,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3124,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3125,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3126,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3127,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3128,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3129,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3130,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3131,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3132,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3133,3134,3134,3134,3134,3134,3134,3134,3134,3134,3134,3134,3134,3134,3134,3134,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3135,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3136,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3137,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3138,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3139,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3140,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3141,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3142,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3143,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3144,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3145,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3146,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3147,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3148,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3149,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3150,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3151,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3152,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3153,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3154,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3155,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3156,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3157,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3158,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3159,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3160,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3161,3162,3162,3162,3162,3162,3162,3162,3162,3162,3162,3162,3162,3162,3162,3162,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3163,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3164,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3165,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3166,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3167,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3168,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3169,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3170,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3171,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3172,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3173,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3174,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3175,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3176,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3177,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3178,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3179,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3180,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3181,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3182,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3183,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3184,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3185,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3186,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3187,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3188,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3189,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3190,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3191,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3192,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3193,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3194,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3195,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3196,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3197,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3198,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3199,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3200,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3201,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3202,3203,3203,3203,3203,3203,3203,3203,3203,3203,3203,3203,3203,3203,3203,3203,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3204,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3205,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3206,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3207,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3208,3209,3209,3209,3209,3209,3209,3209,3209,3209,3209,3209,3209,3209,3209,3209,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3210,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3211,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3212,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3213,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3214,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3215,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3216,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3217,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3218,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3219,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3220,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3221,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3222,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3223,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3224,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3225,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3226,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3227,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3228,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3229,3230,3230,3230,3230,3230,3230,3230,3230,3230,3230,3230,3230,3230,3230,3230,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3231,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3232,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3233,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3234,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3235,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3236,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3237,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3238,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3239,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3240,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3241,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3242,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3243,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3244,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3245,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3246,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3247,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3248,3249,3249,3249,3249,3249,3249,3249,3249,3249,3249,3249,3249,3249,3249,3249,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3250,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3251,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3252,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3253,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3254,3255,3255,3255,3255,3255,3255,3255,3255,3255,3255,3255,3255,3255,3255,3255,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3256,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3257,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3258,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3259,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3260,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3261,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3262,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3263,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3264,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3265,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3266,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3267,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3268,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3269,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3270,3271,3271,3271,3271,3271,3271,3271,3271,3271,3271,3271,3271,3271,3271,3271,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3272,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3273,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3274,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3275,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3276,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3277,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3278,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3279,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3280,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3281,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3282,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3283,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3284,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3285,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3286,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3287,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3288,3289,3289,3289,3289,3289,3289,3289,3289,3289,3289,3289,3289,3289,3289,3289,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3290,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3291,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3292,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3293,3294,3294,3294,3294,3294,3294,3294,3294,3294,3294,3294,3294,3294,3294,3294,3295,3295,3295,3295,3295,3295,3295,3295,3295,3295,3295,3295,3295,3295,3295,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3296,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3297,3298,3298,3298,3298,3298,3298,3298,3298,3298,3298,3298,3298,3298,3298,3298,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3299,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3300,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301,3301])
        S = np.array([13.3046,3.9296,23.9666,-23.916,8.1369,-18.3967,-31.4274,-36.1467,9.5529,-10.6703,26.3879,16.7468,2.3362,2.336,-0.54524,0.65613,-0.23805,-1.6968,10.3474,-26.9051,36.9569,-22.5423,9.868,42.962,24.645,-19.8927,-14.3974,-33.0056,-8.62,-1.2545,-3.9465,6.8934,4.9004,-23.6521,-23.6521,-16.1982,-6.3562,5.337,8.8535,18.1843,-11.5347,-11.5344,16.8564,-23.046,26.5412,-10.1088,13.8169,6.2838,6.2838,4.741,-4.2966,-0.62212,-19.1772,-14.1538,22.9755,5.5267,25.4931,-20.6848,3.1054,-22.3868,11.5016,27.4426,-10.4558,15.8876,-25.7631,-12.1415,-1.0814,-2.4282,6.533,-0.27149,11.5954,5.5262,-13.3048,-32.9894,13.9776,-23.3937,7.4338,-5.2575,31.4272,22.1435,13.9699,-12.8766,-10.9997,-24.4632,-2.3373,8.2343,-16.2461,11.6836,-20.7126,-5.1193,-7.7684,-24.2983,18.0389,-16.3036,25.2737,9.395,-26.4093,-8.9399,-15.2837,17.4465,8.2962,24.6265,-4.1958,-2.5586,-8.4967,-1.7614,-3.2819,2.4621,18.3892,-1.6845,23.4503,-20.8749,8.0611,-13.781,-16.7722,-25.5927,8.3694,-9.4124,22.9196,13.357,1.7721,3.7198,-0.37989,10.2805,5.6739,13.8023,-2.9508,23.1214,-23.0173,28.4998,-20.0272,4.3143,24.8532,10.862,11.2503,-15.5631,-16.4102,-29.8073,-13.4264,-13.1347,-8.6813,-14.2589,7.3323,2.0393,-7.6214,8.9092,-13.6431,18.5521,-3.8788,12.0101,9.0761,22.7883,-14.2034,12.5864,-26.2987,-13.4745,20.5427,4.8004,16.422,-12.108,-3.9323,-17.5265,16.1614,-5.0536,23.4012,-21.653,8.636,-13.1033,-17.4481,-25.1162,6.5378,-9.4636,20.8832,12.5875,2.7967,2.0767,3.7225,8.0297,9.6407,14.7081,-23.376,-9.6612,-16.0241,11.4865,6.5165,19.6926,-11.2575,-24.8096,14.8641,-10.4993,24.9,12.2425,9.1189,-5.1988,13.3626,-19.6012,6.3002,-12.0365,22.3694,6.3011,21.0196,-14.8283,-0.081612,-20.0024,9.7051,23.2783,-12.5018,13.9974,-23.8963,-9.3048,-3.4483,2.1648,1.1282,11.9693,6.5819,11.7313,19.9119,1.3672,21.9783,-20.5382,2.2436,-20.3619,14.5129,24.6433,-4.3625,12.8081,-21.078,-12.4419,-2.9141,-6.6544,7.7445,-1.0242,14.3986,8.2705,-5.1061,-20.5911,14.3476,-16.8925,11.165,-1.7777,28.4398,18.604,15.4686,-11.2506,-16.2642,-29.708,2.5753,16.2519,-17.9418,17.6689,-22.5819,-2.7613,-17.7628,-8.5231,-6.5654,8.709,13.0656,20.4974,-12.6795,10.9927,-24.5762,22.2725,-11.9047,11.6082,14.3768,19.3526,-5.5046,8.0627,-17.396,-10.2536,6.2093,-9.0901,15.2284,-11.593,12.2495,-0.063588,25.2031,8.9946,13.7706,-13.0671,-15.5066,-28.7937,4.2906,23.061,-15.2683,19.6754,-18.6135,-1.6684,-7.0976,-23.2886,15.6159,-13.9024,23.1472,9.2716,-23.9842,-8.2608,-17.6383,12.1121,3.152,17.5002,-1.9611,4.5656,1.7412,15.7869,6.6585,13.6467,-9.2496,-25.0109,14.1588,-15.3123,22.1177,8.366,-23.1602,-6.9883,-13.1961,19.587,8.934,23.8785,-10.0807,-4.2033,-16.6963,1.5739,-8.3421,3.6544,14.5738,-6.6196,22.2497,-19.7131,10.3682,-10.4578,-13.8659,-23.1529,9.7615,-9.9306,21.4951,11.0037,-12.4452,-2.6649,-8.3158,12.9487,6.8086,18.097,13.5919,-8.4687,22.7327,-21.134,10.9063,-10.3468,-13.4561,-23.1185,9.4502,-10.5625,20.6156,10.1236,-13.7035,-5.4865,-6.3675,11.3523,10.0814,18.6874,-1.9511,16.2768,-19.8634,19.3551,-17.8994,1.4936,20.7728,11.2033,5.7763,-18.8474,-16.1956,-31.157,-15.0768,-18.2912,-8.8533,-17.5126,9.0865,2.6399,-11.9867,-18.137,6.8056,-3.7332,20.1242,15.0348,11.9898,-12.6614,23.9911,-27.1962,12.0753,-14.683,17.6126,12.5249,6.4125,-2.695,-9.6048,-13.4403,-11.3497,6.1268,-14.1108,19.235,2.4921,15.5441,12.5282,26.136,-13.3788,12.501,-30.6861,-20.4232,19.2006,5.0015,16.1441,-10.304,-1.6181,-12.3339,14.6526,-9.6295,23.6389,-21.7776,11.1157,-10.0692,-12.6224,-22.0377,6.5867,-10.5739,16.3451,8.2356,-14.2063,-8.3406,-3.3684,8.6434,13.9732,19.7407,-20.1955,-7.9074,-11.6641,13.6843,7.2366,20.8695,-11.7733,-22.8993,11.633,-11.4706,23.4897,11.7288,11.5385,-7.7099,19.4075,-17.4905,7.7203,-10.4099,-20.6753,-8.0285,-11.8924,12.4042,7.948,19.1846,-11.7039,-23.8755,12.4574,-11.0624,23.3899,11.6437,10.3961,-7.371,19.6361,-19.9995,8.9791,-12.7478,21.4479,8.8269,16.0099,-10.3334,-3.952,-17.4165,9.1555,21.7079,-13.8275,11.8939,-23.1647,-9.4597,-9.3044,4.618,-10.0887,17.4463,1.8403,15.6088,20.6199,3.9925,20.7316,-15.8053,1.4991,-18.3062,11.0883,23.1323,-9.8879,13.8731,-21.4542,-8.0204,-5.5577,-1.6426,1.8539,10.1834,10.1823,13.8916,17.6199,-0.36962,19.4138,-20.0761,3.3976,-18.1805,15.3017,22.3298,-3.5838,10.4415,-17.7827,-11.4137,-5.5453,-10.1244,9.047,-1.6933,17.3902,10.9174,-2.4626,-16.5239,13.1836,-14.9133,12.1162,-0.95261,30.098,20.3657,12.6707,-11.5048,-15.944,-29.7076,-0.15628,15.8563,-18.2718,18.6304,-20.5153,-0.62941,0.42796,-11.4059,11.5341,-12.7495,10.6046,-1.3988,29.3313,17.0372,13.8244,-12.9799,-15.3732,-28.8822,2.9065,21.033,-18.315,18.5351,-20.5291,-2.4877,-19.8712,-12.59,-6.1329,8.2852,14.8318,21.5486,-12.0864,11.2911,-24.3701,22.2804,-12.0312,11.1456,10.9835,17.9635,-5.4785,8.5843,-15.0986,-7.6445,1.8613,-11.4705,15.7305,-12.1861,18.0976,2.85,28.8855,14.9693,13.1676,-13.4939,-15.6321,-29.6454,2.4694,19.9834,-14.9043,17.94,-16.3927,-0.50335,-9.7366,16.212,-22.849,24.0908,-10.6507,9.9537,-18.6988,-12.761,-8.6278,1.5297,7.8054,13.5717,-12.4185,-14.7115,6.1356,-2.7999,21.4696,17.0841,-7.8349,-25.4837,16.5185,-14.6819,23.8982,9.8687,-22.739,-4.7392,-18.3835,14.665,0.81447,16.4985,-8.2324,1.0648,-5.8338,17.4977,0.44947,14.3704,-10.2253,-24.8845,9.6016,-15.1787,17.3962,7.2314,-21.3914,-5.495,-10.554,19.0782,10.3802,22.9503,-11.294,2.1794,-21.8484,11.5519,-13.2611,6.8286,10.8862,21.2611,-9.579,11.9503,-18.7361,-8.222,-11.7772,10.8117,-22.2739,20.772,-11.3497,10.1809,-17.5666,-11.6483,-3.8432,6.3711,13.505,19.41,10.7659,-10.8032,23.1101,-20.7113,13.7366,-8.0617,-10.9326,-21.8154,11.6591,-11.135,20.7687,8.9977,-18.161,-6.4488,-9.6053,13.6725,9.6804,20.6359,10.8714,-12.7632,23.6272,-22.4667,13.478,-8.528,-10.0418,-21.7827,9.1807,-12.3684,17.0667,6.7162,-18.7248,-9.0821,-6.5299,11.4896,14.856,21.4082,-1.5144,15.0303,-19.6456,16.4847,-16.1137,0.36197,19.7794,10.7913,3.537,-17.2231,-15.5978,-30.2755,-18.1335,-18.4963,-8.811,-16.0235,12.1368,6.1773,-15.5017,3.5325,-18.6913,18.5191,1.333,13.6308,14.6953,27.577,-13.0198,13.2065,-37.405,-28.5291,12.9989,1.6827,11.1672,-10.9378,-0.93589,-7.5465,-13.9914,-18.5674,5.7986,-4.4247,20.082,16.839,12.2947,-12.8941,23.8896,-26.3415,12.333,-11.2466,15.0833,11.7576,6.0866,0.18101,-8.0681,-12.2121,-18.2227,3.1472,-15.6614,18.5619,4.8299,19.9295,17.5185,33.443,-13.7925,13.5293,-29.8203,-17.9075,16.5224,1.1667,14.8911,-10.3967,0.049233,-10.7442,11.7333,-13.5579,23.2141,-22.8898,13.3968,-8.4938,-7.7173,-17.4163,7.6843,-10.1653,14.2251,6.2803,-20.1088,-13.2108,-5.6768,6.4532,15.6867,22.2406,-21.9695,-14.6061,-7.5913,6.8125,14.1458,21.5453,-10.8099,12.2704,-22.8798,23.1301,-11.2819,11.4165,8.9062,16.2274,-8.7224,7.4328,-16.897,-8.4655,-19.6408,-6.2634,-12.2754,15.9212,6.4612,21.3991,-12.4272,-23.9993,12.2688,-12.7193,24.2195,11.5421,12.2792,-5.8176,20.1573,-14.9779,6.9172,-9.9491,-17.5496,-6.9092,-8.2455,12.2532,9.5483,18.9883,-12.836,-23.9125,11.4198,-11.6115,22.5821,10.8637,12.8693,-7.4095,23.0399,-21.0582,8.8711,-13.9813,19.9124,11.3827,9.7475,-6.6932,-7.7287,-14.1532,10.38,-13.2886,22.4151,-23.9042,10.3032,-13.138,-13.603,-16.2366,5.2059,1.7182,19.7425,16.4187,19.7573,6.0338,16.8072,-10.2336,-1.1983,-14.6215,9.2916,22.7483,-14.4489,13.2356,-22.3898,-6.7343,-12.0237,0.62781,-8.3828,16.6965,6.1226,19.0085,18.5769,1.499,20.2277,-16.2774,4.0358,-16.1682,11.0126,22.4694,-8.7145,14.6257,-18.444,-6.0803,-8.4891,-5.5089,2.5278,8.8737,13.0814,16.2808,16.2481,-2.0472,19.3329,-20.917,5.3051,-15.8036,14.2458,20.1282,-2.2924,8.6966,-13.5071,-9.562,-8.5793,-13.4452,8.9443,-1.3232,20.7449,15.712,-1.5714,-13.2437,11.8015,-12.3067,11.0717,0.41348,28.7279,19.4348,12.8022,-12.3578,-13.7895,-27.5179,-2.6069,12.9416,-18.6792,19.8753,-19.1687,1.338,0.63334,-10.8489,11.8761,-11.6949,10.7904,-0.69222,28.9239,16.9054,13.5429,-13.0978,-13.7182,-27.5232,1.9802,19.3086,-18.1515,18.7427,-20.362,-1.6947,1.5814,-10.4773,14.2408,-11.682,15.2526,1.0721,28.867,15.3394,13.6077,-13.6212,-13.7966,-27.8702,2.5891,20.6392,-16.0761,18.2608,-18.0746,-1.7418,-20.6304,-13.9572,-4.8593,7.3146,14.6934,20.2843,-11.8583,11.7342,-24.616,23.1419,-11.4707,12.8048,9.0121,16.3669,-6.5828,8.2914,-16.2503,-8.6591,-2.4363,-17.0907,16.7824,-15.479,18.6518,0.52832,29.4246,16.0378,13.7753,-13.7974,-13.0312,-27.9808,1.5015,16.5795,-13.7909,14.8857,-17.0731,-3.289,-9.4089,-22.6576,14.5791,-12.2162,21.9493,-10.5382,12.3772,-16.1947,-4.9084,-14.2033,10.3785,-1.7976,11.5989,9.0984,-14.6977,3.9351,-17.9526,21.8167,-7.259,21.4109,8.0312,-12.5634,12.7039,-23.0218,24.7591,-8.7044,17.0866,-15.4897,-8.6632,-8.5293,8.7352,5.18,13.7693,-15.7928,-21.0192,6.9838,-0.36433,23.6214,16.1843,-8.7195,-24.4479,12.72,-13.494,20.7716,12.0975,-19.8927,-2.2952,-14.6334,16.4419,5.1392,16.0991,-12.0014,3.5822,-19.0937,19.3391,-11.142,8.8756,-9.1506,-19.9514,8.1326,-9.8907,18.5088,11.5267,-21.7999,-8.3169,-10.826,12.5119,10.2443,19.5904,-9.7788,12.2651,-23.6874,22.4348,-14.3605,9.0457,8.761,23.7099,-11.6212,18.7927,-18.0082,-4.9749,-9.6369,12.4399,-20.3015,17.9335,-11.6966,5.74,-20.0437,-11.6117,-6.9093,1.8825,14.3365,23.0178,8.9887,19.8432,-7.878,13.022,-13.1514,-3.216,-11.0654,10.1206,-23.11,19.1528,-14.4308,6.513,-18.7109,-12.5007,-4.2297,8.7935,15.2281,23.4693,8.7301,-15.6482,26.3959,-23.7086,15.2194,-8.4004,-21.4376,13.4567,-12.3833,21.0615,-21.17,-9.0242,-8.4869,21.2692,25.8461,9.4119,-17.8679,22.7346,-21.8267,15.0531,-5.4548,-6.9014,-20.1728,8.0674,-17.0762,12.6548,2.8814,-21.3091,-12.9213,-6.7101,-2.8729,16.4389,23.7789,-3.2192,12.8817,-21.334,13.4966,-15.0394,2.5753,-13.149,17.8902,10.8407,1.8251,-12.5162,-13.6414,-23.8501,-8.7287,-21.081,-19.1189,-6.6693,-14.0839,18.9301,4.6724,23.2223,-14.0945,-17.2407,1.6325,-17.2405,18.7056,2.4424,18.7664,13.1349,-13.5932,24.2731,-13.5937,12.9759,-23.1961,-10.4948,10.7568,12.9058,1.6537,12.9056,-10.2454,-0.42368,-11.4883,-19.8948,3.61,-18.3848,30.8714,4.8534,23.8757,42.1223,-13.7736,-14.9351,-34.9484,11.3326,-5.3593,10.6893,-20.4515,-2.5609,-15.2227,-18.4715,3.3927,-0.3863,18.1549,16.9157,12.283,-13.0265,25.0601,-24.9975,13.5797,-12.1181,14.651,10.5584,5.0297,-2.0653,-11.0536,-13.8097,-21.6122,-1.5447,-18.4488,18.4349,0.68116,21.2348,16.045,31.6812,-13.8928,13.6342,-26.7072,-11.7825,13.7417,0.054962,11.5234,-10.9482,-3.2291,-14.0493,9.625,-14.2408,23.4282,-21.9814,15.7267,-6.5292,-5.6958,-14.3477,7.517,-9.6372,11.1816,2.5637,-22.6705,-15.5305,-6.6308,7.5522,16.1628,24.2615,-22.7336,-17.1329,-7.5283,2.6334,14.716,19.0839,-10.4108,13.4491,-22.5069,25.7121,-11.625,12.3152,7.5452,12.8449,-8.6143,-0.19464,-16.202,-13.2066,-21.4604,-14.017,-7.4526,8.4098,11.9958,20.4244,-11.86,11.0904,-23.0242,21.9845,-9.5156,13.0403,8.6056,17.0799,-9.7365,9.5282,-19.3846,-8.2783,-18.7145,-4.2662,-10.9278,14.7298,6.9531,18.4851,-12.608,-24.9913,12.8057,-13.3402,25.0734,12.0292,14.141,-2.8958,22.2953,-15.1051,6.4525,-14.085,-17.6809,-8.0604,-6.8659,8.3266,11.5636,17.5966,-11.7561,-23.6694,9.3418,-11.9912,20.8929,12.208,14.8162,-8.314,24.4595,-23.4051,8.8702,-15.6143,16.9268,5.9626,10.6472,-8.9331,-5.7754,-9.9235,10.6686,26.389,-15.9247,9.0342,-24.3161,-14.3282,-18.4147,2.9582,-15.6536,25.3821,8.919,18.752,15.6067,8.1906,9.3353,-0.7292,-3.9748,-8.1583,12.8068,-13.251,23.3393,-23.2719,5.6953,-14.7044,-15.5128,-19.377,8.393,-6.4908,26.6423,18.5612,17.4757,3.9743,17.5283,-8.4331,1.6617,-11.4974,7.7642,22.659,-14.5582,16.3501,-20.3652,-2.3582,-14.9963,-3.355,-7.1044,15.0897,10.404,22.7412,16.8812,-1.36,21.3582,-16.668,8.1473,-13.4242,9.7474,20.6384,-6.6912,15.0667,-14.4626,-3.8573,-11.7582,-9.1764,3.1162,8.0109,16.5023,19.7571,16.0288,-5.5546,21.797,-23.4805,12.4136,-13.5527,11.6821,16.2533,-2.6199,5.6595,-9.0285,-7.767,-14.0756,-18.8282,9.1556,-0.71993,21.7888,20.384,-1.19,8.9138,-13.3967,8.4278,-11.7477,-0.81772,24.8739,13.0574,12.2977,-12.0588,-13.5334,-25.5931,-5.7914,-20.1633,14.8931,-19.3462,19.1255,0.72776,0.6232,-9.3273,10.9406,-10.0035,10.5309,-0.17453,26.8873,14.4606,13.5235,-13.4648,-12.3104,-26.5361,0.42599,18.8253,-18.4249,19.6702,-19.8827,-0.89629,0.89163,-10.2484,12.546,-10.7148,13.2004,1.1045,27.3394,13.9997,13.8669,-13.96,-12.4172,-26.7443,2.1012,20.1195,-17.1117,18.529,-19.0141,-1.5647,0.22639,-13.6181,15.4054,-12.8521,16.0387,1.5593,27.5939,14.0033,14.059,-14.2648,-12.3608,-26.9654,3.189,18.8535,-14.2748,16.8528,-17.2462,-1.8364,-18.0861,-12.8317,-3.1393,5.9337,16.0017,19.6542,-13.5893,10.2193,-25.2087,23.5369,-11.6235,13.0098,11.2207,18.3956,-5.0381,8.7767,-15.5035,-8.9425,-0.20299,-15.9338,19.177,-15.2853,18.7002,0.77909,26.924,13.9218,13.8427,-14.3997,-12.6213,-26.9448,5.5497,17.5637,-9.8937,14.3481,-15.072,-3.4158,-12.5429,-26.7031,36.2321,-8.4862,19.7686,-19.1138,-9.159,-5.1324,4.5587,11.308,-21.4524,7.8165,-20.2129,25.0325,15.681,-8.4553,9.5805,-22.9891,23.4667,-8.1644,16.436,-14.6443,-12.0847,-0.31924,2.1937,9.2162,10.9967,-12.0801,-22.65,15.6827,-11.7939,25.6134,13.2577,-14.7238,9.6164,-23.2462,25.6247,-7.065,17.4658,-12.6971,-4.6103,-8.7011,7.1215,2.5965,10.9738,-16.9906,-24.0008,8.3544,-8.8851,24.9482,16.1553,-10.8809,-21.678,7.7658,-7.3159,21.5131,15.9108,-17.651,-4.0144,-10.2143,9.466,6.8548,13.4906,-11.2929,13.7745,-24.3106,25.2075,-13.7191,12.3881,-11.6429,-19.2925,5.5535,-5.5816,19.7664,15.5476,-19.5862,-9.376,-8.3755,6.6355,10.1709,14.9364,-10.099,15.2648,-25.0584,25.7967,-14.9725,11.3436,4.9159,6.6073,-12.1396,16.7458,-15.2514,12.5534,-2.8621,-5.5503,-3.9853,-14.3368,8.782,-10.7055,10.5982,1.2485,-23.1517,-22.9733,-7.9542,-6.2947,13.0207,13.7531,24.4791,4.9862,17.1145,-11.1176,12.571,-13.1815,-0.62812,-7.4562,8.5968,-19.9568,13.8779,-14.3396,2.7858,-23.13,-15.0636,-8.6046,12.1936,14.7311,24.5647,3.816,14.3336,-6.5717,12.1509,-9.079,0.17154,-8.9085,13.4437,-22.4428,19.4761,-17.5498,3.4336,-23.621,-14.8647,-9.5987,10.3426,15.8771,25.9015,6.9283,-11.2909,21.7119,-17.4298,15.858,-4.7341,-3.0177,-11.6634,8.6869,-8.8885,10.2337,0.48322,-23.9911,-17.4842,-7.9982,11.3585,15.6576,24.8256,-6.7576,-26.8641,33.982,-12.6614,15.6259,16.2613,0.22135,-33.6617,-10.2909,-48.1686,-28.0857,-1.918,-41.6173,22.9244,3.1739,10.8214,-20.8331,12.5948,-14.984,7.0841,2.0288,11.5524,42.3903,-11.0113,43.866,-21.5789,-10.9105,-19.5655,18.395,-15.6266,20.893,3.0534,24.466,-13.6309,-23.2883,16.8607,-18.8959,18.2849,0.30723,28.5287,21.3142,13.6687,-13.7904,-10.6271,-24.6487,7.5459,13.5091,-13.0468,10.1175,-15.0365,-3.7804,-18.3193,-18.22,-0.72567,0.6704,17.5695,17.9629,12.0216,-13.7183,25.1275,-24.7594,12.8702,-11.7524,12.3106,11.0011,0.34385,-2.5234,-12.2259,-13.2457,-15.0156,-16.19,3.3799,1.8091,20.5469,19.701,11.629,-13.0094,24.6008,-25.0343,12.5505,-12.6892,16.0917,13.4659,6.1708,1.1942,-6.2931,-8.2737,-19.8582,-2.3389,-20.1074,18.1898,-3.55,16.3625,12.8645,25.5729,-13.4438,13.9644,-24.8859,-10.8648,13.1228,3.9318,8.3612,-10.929,-8.2676,-17.8135,-25.6939,10.6119,-28.7703,26.575,-7.5939,18.1672,-6.4666,-8.7792,-5.8765,1.2487,6.1257,10.0361,-1.198,-23.7139,9.1201,-12.966,25.9258,14.6835,7.5195,-15.1827,23.1599,-21.5094,16.7791,-5.4392,-2.4805,-11.2506,8.4323,-9.6213,9.8677,0.32506,-24.5172,-16.2633,-8.4793,9.4105,16.0306,25.1617,-17.3632,-1.1503,-13.806,19.2413,4.4927,20.9442,-12.3718,-25.67,13.0716,-13.2158,25.4405,12.3025,15.2997,2.2323,16.1038,-9.073,1.2422,-10.5589,-23.1258,-17.9337,-10.0347,2.3971,12.4985,18.8356,-10.3595,13.3549,-20.1461,25.6886,-7.9744,13.5727,6.6738,10.9742,-12.3835,0.034392,-19.8371,-11.5357,-17.9295,-10.3897,-3.9339,10.7793,17.0497,23.2743,-14.0213,7.1058,-24.7195,18.2163,-13.2083,8.8321,10.6079,20.6618,-6.9668,13.9053,-12.8907,-1.5184,-15.2797,-4.6791,-6.5051,13.3157,11.0564,19.8168,-13.0823,-25.1007,9.0409,-12.9903,22.074,11.2777,16.9525,-2.0188,24.2582,-16.528,7.8284,-13.2268,-17.1171,-11.0915,-4.0998,5.4254,15.8185,19.7146,-12.5174,-21.2489,4.6353,-8.8829,15.4259,10.7997,16.1882,-8.6439,25.4446,-24.204,10.0182,-14.8804,9.6623,1.5209,7.2813,-9.1333,4.3667,-9.9841,14.2378,23.5101,-28.9324,2.8376,-32.4225,-5.3689,-18.8912,8.1263,-31.3857,34.2882,-26.8906,31.2391,14.2085,2.6422,9.1541,-9.6804,-2.2728,7.966,43.2606,-17.9197,20.062,-20.3383,-24.0087,12.6336,-13.8879,43.5413,10.6824,10.5465,6.6374,7.8105,-1.2008,-1.1273,-5.8824,16.7475,-8.8057,21.973,-23.3718,2.4298,-21.6207,-17.0436,-26.6875,9.997,-8.2808,27.4341,16.7471,15.0063,2.3433,18.2182,-7.7534,5.4881,-7.4737,5.2985,20.2779,-14.1336,18.1246,-17.0585,1.3177,-18.4149,-8.8533,-6.499,13.2807,15.0375,23.176,15.3332,-4.794,23.1741,-16.1521,11.5588,-8.15,8.8031,17.9909,-2.2147,15.4344,-6.6991,1.4462,-15.7824,-13.2581,0.78696,6.1602,19.5241,22.0633,14.4504,-9.6214,24.5745,-23.9196,11.8221,-10.727,8.8071,9.9817,1.3406,4.2095,-6.2079,-3.5046,-19.8708,-21.1709,-2.1497,1.7471,22.0819,24.0368,1.5757,-7.8023,10.2779,-8.9258,9.9105,0.022039,25.5236,11.9995,13.8983,-13.7884,-11.7282,-25.8295,-0.32399,19.6829,-18.6384,20.2939,-19.6101,-0.70806,-0.84714,8.62,-8.0913,7.6176,-5.4823,0.46558,24.5446,13.5935,10.247,-11.1544,-12.7406,-24.7497,-3.4989,-20.4106,21.991,-19.7932,24.1292,0.76902,0.18395,-10.1034,11.2339,-10.4963,11.5377,0.15935,26.2704,12.6378,14.176,-14.0372,-11.6534,-26.3156,1.3538,19.5347,-17.7382,18.5176,-19.7348,-1.8352,-0.34827,-12.3384,13.377,-11.9659,14.3484,0.96037,26.682,12.5683,14.4325,-14.4439,-11.6696,-26.4141,2.1614,19.3029,-16.0091,17.2497,-18.2366,-2.1389,-0.75342,-14.9363,14.9509,-13.8653,15.2557,0.59231,26.7605,12.5329,14.5323,-14.6411,-11.2508,-26.5066,2.8981,17.8455,-14.3615,15.5765,-18.0542,-3.0997,-19.3031,-0.54643,-17.8761,17.2184,-0.44219,16.7007,9.5274,24.2131,-14.1334,13.7688,-24.6779,-10.6726,13.4705,0.48725,11.5873,-12.4823,-4.9096,-16.484,-19.3816,-15.7202,-1.362,5.912,19.2735,22.4731,-12.9773,11.7097,-25.5782,23.2419,-13.0064,10.4994,9.3327,15.9093,-3.276,10.3122,-9.5023,-2.6251,1.4073,-15.73,18.7773,-15.3403,17.5555,1.0905,26.065,12.4125,14.1074,-14.9153,-11.4797,-26.3319,6.7108,17.7772,-10.1674,13.9273,-15.7271,-2.9795,-17.1385,9.0017,-25.1801,25.9062,-6.716,18.3368,-11.6803,-5.4314,-8.0505,4.1088,2.9424,9.4387,-16.1475,-24.8011,8.8392,-11.3825,25.8537,15.846,-16.1508,-23.2858,5.5654,-5.5532,22.4981,17.7731,-11.6597,-2.8482,-6.5595,7.0829,5.216,10.1997,-13.4136,12.3529,-26.2263,26.8695,-13.1605,15.1707,-18.4106,6.8498,-25.3618,24.0917,-6.1137,18.6451,-10.465,-8.1485,-3.617,1.8996,5.9339,8.4967,-12.6474,-25.6106,10.9665,-11.5388,26.6168,15.5114,-16.0949,7.7143,-22.7265,24.178,-5.5619,17.7072,-11.0288,-2.9233,-9.6818,6.8028,0.37125,9.075,-16.4689,-24.9945,9.3194,-9.8313,25.7075,16.1636,-15.9315,-21.4763,4.198,-4.3743,21.7854,18.5759,-13.4903,-6.9312,-4.7358,4.4116,7.1444,9.8568,-12.0776,13.8382,-26.3751,27.0095,-14.2587,13.5348,-15.6666,-19.7456,3.5703,-2.5774,21.8547,19.6903,-14.8062,-10.2006,-4.2143,1.2796,6.4886,7.7243,-10.4721,15.0315,-25.4773,26.8425,-14.8141,12.5237,2.6803,16.722,-13.8806,14.3213,-13.8001,0.37055,-3.1724,10.9977,-16.3506,11.5544,-14.2763,-0.39242,-25.2143,-12.026,-10.1382,13.3597,13.9332,25.0831,1.0307,14.5087,-10.6304,14.5737,-9.8547,2.8016,-5.1365,13.5648,-20.4004,15.8728,-17.4469,1.2775,-25.0738,-14.4265,-10.1823,12.0974,14.6956,25.2535,1.2685,10.4589,-6.9279,11.3066,-6.809,2.5704,-4.4232,16.9668,-21.8607,20.1046,-20.0533,1.8381,-25.8741,-15.7192,-10.2697,10.808,14.7494,25.6269,4.8764,-13.8598,19.9061,-16.1565,15.8719,-2.258,-1.7097,-12.2461,9.8959,-11.8045,9.5125,-1.8008,-25.1116,-12.4531,-9.001,12.7335,14.7066,25.1979,5.6514,-15.172,21.9854,-19.6944,17.1153,-4.0047,-0.61243,-10.3791,9.5134,-10.8938,8.9159,-1.9391,-25.0681,-15.8246,-9.0539,11.0248,15.617,25.3962,-11.913,-0.64185,-23.8474,3.9305,-11.4801,5.1421,2.4114,8.9503,4.3026,1.4381,-11.8456,-4.8028,-21.9143,-12.2258,-20.7747,-24.067,1.7207,-23.035,24.844,-0.85167,22.4359,1.0072,-5.9593,5.073,-5.1001,1.6447,-1.4532,11.3318,22.7561,-11.5348,10.2884,-23.5236,-11.7004,-25.0444,-1.431,-22.5855,25.6547,0.5021,24.6604,-0.40883,-18.4971,16.5541,-20.0061,14.9241,-3.8456,24.7605,10.6843,14.4741,-13.2146,-10.0849,-23.9454,3.4493,14.7093,-12.7425,8.4425,-18.9178,-8.9986,-17.8888,0.96716,-18.3617,18.1218,-1.6994,16.217,10.9905,25.219,-14.134,13.7384,-24.8656,-11.1036,15.0057,3.2136,10.9681,-10.8592,-6.1021,-16.3494,-19.3786,-18.93,0.35667,-1.1985,18.7005,17.0495,12.1641,-13.4666,25.7405,-25.3492,13.3379,-12.0252,11.0317,10.032,0.74766,-4.6023,-10.0499,-14.6041,-19.1849,-20.0576,1.8214,-1.2847,20.6697,18.6665,13.0277,-13.0513,25.8206,-25.842,13.0464,-12.9683,9.5691,7.526,3.2448,-4.0403,-6.6414,-12.0047,-15.2274,3.6436,-16.6537,20.0849,1.4029,18.6502,10.7785,24.0794,-14.8803,13.1621,-26.4032,-11.5833,18.435,9.371,12.6163,-8.3854,-2.4731,-14.0176,-18.3123,7.1946,-24.9299,25.0158,-5.8014,19.0107,-10.2345,-6.4191,-5.8541,2.4214,3.4086,7.7337,-14.5433,-25.65,9.7606,-11.1598,26.2629,15.7874,5.9474,-16.5462,22.5214,-21.6138,17.4179,-4.4906,-0.18592,-9.8385,9.5201,-10.602,9.063,-1.5779,-25.35,-16.1078,-9.5262,9.9055,15.9558,25.5655,-21.6384,-2.9797,-19.7642,18.9282,1.9859,21.6599,-12.4729,-26.2245,13.6342,-13.0083,26.323,12.8729,9.0387,-4.0914,7.8143,-12.8277,-2.5623,-10.1401,-17.8083,-1.18,-11.6502,19.9815,8.2873,22.4186,-12.4502,-25.9366,12.4267,-13.5796,24.2663,11.0869,15.2859,2.7446,18.1864,-7.4291,4.1398,-8.4214,-23.2778,-15.8558,-7.0819,4.8762,16.4077,21.9409,-10.2419,12.6518,-22.597,24.9239,-11.565,13.4449,5.9318,15.6271,-13.463,7.7873,-17.6187,-6.6163,-22.9616,-16.1925,-5.037,5.4457,15.6646,20.4626,-10.0336,13.7649,-23.8014,24.217,-11.9702,13.7315,2.5784,14.9814,-11.5376,10.7945,-16.9903,-6.6191,-16.4818,-8.1666,-4.105,11.0615,15.9571,21.9744,-13.1911,-23.7043,4.8901,-12.3451,16.0229,8.2933,16.3756,-4.9596,24.5441,-18.4179,9.3085,-11.5243,-19.3887,-15.6097,-1.7724,3.7217,19.1733,22.0338,-11.2116,-15.6898,0.30916,-4.9437,5.4797,4.6634,15.4592,-9.6637,25.8883,-24.3566,11.9302,-14.3857,10.3342,2.1338,10.9502,-3.9009,-3.976,20.5342,23.2826,-18.3714,-0.093218,-27.8201,-26.0074,9.8192,-38.9304,27.9353,15.9588,5.3137,10.0262,-0.96608,10.1494,-0.93487,3.1608,-2.4311,12.3605,2.2137,23.7403,-21.4734,23.3582,-19.5675,0.54545,-16.6507,-23.9631,7.6912,-10.1511,9.8901,16.0601,23.2605,8.0552,3.7486,7.9648,-1.7817,1.7158,-3.3946,19.4309,-1.6619,21.5101,-22.7955,0.70044,-23.1403,-17.1644,-26.5663,10.5223,-10.19,25.618,14.9686,10.6641,-2.6758,19.3665,-8.1278,11.7218,-2.4003,1.6241,20.1557,-13.7085,21.672,-11.5637,4.087,-21.1978,-13.0249,-3.8364,7.6216,18.418,23.2568,12.3608,-11.2165,23.5649,-19.2779,14.1352,-7.4116,2.4896,12.2623,-4.0509,14.474,-5.687,3.7773,-21.2712,-18.2268,-2.3421,5.1416,19.8216,23.1578,12.0253,-12.8379,23.9459,-23.4326,14.201,-6.4881,4.7698,7.5554,-1.3424,5.3663,0.52856,4.0004,-23.8026,-21.6947,-1.5233,2.9558,21.8836,25.5857,1.3082,-6.2989,9.3694,-5.5289,9.6522,2.1617,24.9428,10.8119,14.1723,-13.4705,-11.5331,-25.4887,-0.50071,20.8135,-19.2573,21.6585,-19.8373,0.15663,0.58582,-9.3915,10.4657,-9.0773,11.1286,1.4953,25.7654,11.6572,14.3471,-14.075,-11.4667,-26.0303,1.0925,19.9653,-17.8533,19.3995,-19.3688,-0.90981,0.40925,5.8333,-2.6504,6.5567,-0.7132,3.0692,24.2754,13.9378,11.4293,-11.4977,-12.4358,-24.4275,-1.5437,-21.5825,23.3306,-20.7214,23.8613,0.75765,0.42825,-10.625,12.7191,-9.9476,13.7704,2.1768,26.0973,11.9685,14.5801,-14.4809,-11.2546,-26.1391,2.4851,19.6886,-16.2897,18.1348,-18.2964,-1.1343,-0.48313,-13.6794,13.4228,-12.5988,15.065,2.212,26.3106,11.834,14.916,-14.8461,-11.0825,-26.263,2.6601,18.4285,-15.2526,16.3523,-17.4235,-1.5425,0.084552,-14.0151,16.2814,-12.4025,17.4144,2.9909,26.1661,11.5906,14.8212,-15.0538,-10.8448,-26.1428,4.1269,18.3523,-12.8681,16.0697,-15.6898,-1.1472,-16.1119,1.6609,-17.6699,17.9147,-2.2022,15.7578,10.2038,24.8065,-14.6669,14.3379,-25.0078,-10.6544,17.6521,5.5159,11.4988,-11.1366,-7.1408,-17.8141,-21.0949,-17.4361,-1.4673,3.2345,17.9844,19.2486,-12.8142,13.3021,-26.1252,25.5015,-12.7795,13.3238,6.2575,14.1628,-4.1329,8.176,-13.0402,-8.5414,-0.90177,-17.3719,17.0046,-16.1891,19.2862,2.6799,26.36,11.5107,14.8961,-15.3704,-10.8867,-26.3284,3.1108,15.8476,-11.9936,12.8878,-13.1861,-1.1107,-17.462,6.593,-23.7402,24.1925,-5.406,18.5925,-9.7451,-3.5846,-7.6516,4.9202,1.0401,7.5486,-16.0224,-25.6065,9.4133,-10.4242,26.0304,16.1127,-18.5385,4.699,-23.7726,23.6593,-3.8721,19.9316,-8.9826,-5.1909,-2.683,1.9367,2.9076,3.555,-15.3307,-27.2682,10.7439,-11.2521,27.5284,17.4903,-18.1238,-22.4721,3.7746,-3.5438,23.5531,20.6404,-9.6275,-5.0088,-4.3737,0.70996,2.0059,2.6254,-13.849,12.0976,-26.7294,28.0241,-12.509,16.5279,-18.9021,5.1276,-24.567,24.2551,-4.6775,20.5756,-8.3553,-5.8137,-3.446,0.18837,2.5107,3.2372,-15.457,-26.2552,10.1218,-10.1641,26.6871,15.8439,-16.9595,6.033,-22.0701,23.4448,-3.9973,18.5729,-9.8913,-1.4234,-10.4734,6.8287,-2.4351,6.4329,-16.1814,-25.5769,9.6312,-9.7963,25.736,16.134,-19.2081,-22.3505,2.5031,-2.7127,23.1019,20.8817,-8.424,-5.4831,-2.4661,0.33225,2.0262,1.9829,-12.5838,13.1946,-26.4477,27.2898,-13.5915,14.7849,-20.0823,-22.1934,3.0578,-2.0225,22.7129,19.8571,-6.7978,-5.2582,-1.6141,-0.76713,0.77501,0.20692,-11.0741,13.924,-26.3735,26.5195,-14.3124,13.6662,1.8352,18.5225,-15.5923,16.6869,-14.356,0.52091,0.25011,12.5875,-13.9559,10.0982,-15.2923,-4.0541,-26.0932,-10.9926,-11.7462,14.4953,13.7248,25.5374,-0.77357,15.2218,-14.5009,16.455,-10.8032,4.9264,-1.5965,14.3484,-18.3471,13.0159,-17.7446,-1.4805,-26.0306,-13.3369,-10.8308,13.3712,14.3155,25.2019,-1.4519,11.6994,-10.2695,15.6299,-7.05,6.1338,-2.6578,17.0695,-21.0037,17.6067,-19.5819,0.13951,-25.5037,-14.4226,-10.6048,11.5278,13.9394,24.9928,-1.1558,7.6991,-7.4241,11.2393,-4.3124,5.7973,-1.4568,19.1656,-22.2946,20.6577,-22.1503,1.313,-25.2717,-15.0222,-10.1529,11.3543,12.6862,24.5037,3.8226,-16.2645,21.1247,-18.0686,17.8219,-1.8305,1.3813,-10.7867,11.2929,-13.4245,8.1153,-4.6724,-25.7314,-14.2053,-9.8694,11.778,15.3622,25.3631,4.4985,-17.12,21.8679,-20.3736,18.0266,-3.0474,1.6861,-9.3537,10.8425,-12.8491,8.0986,-4.7377,-25.5225,-15.5269,-9.6363,10.3202,15.6858,25.3638,-12.4658,1.8896,-24.0996,4.3363,-10.7321,3.2107,6.1452,-11.9982,6.6731,-26.1216,2.9811,-11.182,-23.3207,-28.2121,3.9587,-2.3988,26.9438,24.5232,-0.041998,-4.301,3.5501,-2.6915,2.4697,0.58869,11.6258,24.224,-12.6894,13.6439,-24.6434,-11.3869,-24.8031,5.7762,-25.9433,25.0443,-0.6391,22.7432,1.0057,-2.0969,2.9465,-2.7722,2.2898,-0.36773,12.0943,24.3898,-12.136,13.6309,-24.3401,-11.7801,-23.8327,1.2829,-23.2958,24.1377,0.13193,23.4707,2.7458,-16.2393,17.9657,-17.8814,16.963,-0.21492,24.7519,10.7834,14.5704,-15.0511,-10.9601,-26.5501,7.419,17.384,-10.9765,11.2954,-16.8026,-4.2593,-16.9232,-18.8416,2.0264,-0.98074,19.2449,17.8511,11.6108,-13.8885,25.4195,-25.7677,13.3995,-12.4701,15.4615,9.8754,5.1255,-4.8547,-8.7365,-12.9967,-16.2153,2.6246,-18.3435,19.6761,-0.90591,18.2972,10.5277,24.9921,-14.4458,13.6149,-25.8689,-12.1716,17.2387,6.4739,10.5991,-7.9282,-5.7038,-13.5301,-17.4158,-19.076,2.4506,-1.5433,20.7397,18.3105,12.5041,-13.3943,25.2194,-25.6137,12.5106,-12.528,14.0902,9.4786,6.3705,-4.4937,-5.7722,-12.0167,-19.2023,-21.1171,-0.50863,-2.6432,18.8594,18.0975,13.0788,-12.2511,26.0379,-25.0889,12.8823,-12.2665,10.7743,5.6342,1.882,-6.7081,-10.0335,-13.067,-18.4075,-0.7688,-16.0419,16.7597,2.3817,17.7996,11.5071,26.6344,-15.504,15.2998,-26.4696,-10.8701,14.4668,2.7223,13.1844,-12.3889,-1.0956,-15.1487,-18.391,5.5563,-23.7445,24.1045,-4.4478,19.4932,-8.3288,-4.2877,-5.9055,1.4136,0.060473,3.3046,-15.8387,-25.9092,9.6906,-10.4643,26.1874,16.2957,4.6522,-17.3773,22.549,-21.1567,19.1658,-2.5069,1.4104,-9.2645,9.338,-12.2296,5.5133,-5.3254,-25.587,-15.9292,-9.6499,9.6427,15.7399,25.2028,-23.8045,-5.3003,-20.8179,18.8729,2.2016,22.6728,-11.9155,-24.3631,14.3082,-11.9149,27.6419,12.9113,5.0404,-10.5741,8.27,-14.9676,-0.45184,-8.0593,-21.9674,-0.84206,-19.2264,20.6821,3.9311,22.8866,-12.7434,-27.181,15.9284,-14.666,27.2447,11.8019,9.9914,1.8701,8.6136,-6.5663,0.53135,-6.6903,-19.3488,-3.1922,-8.4323,19.4022,14.3719,24.155,-11.9213,-27.0153,10.7142,-15.9566,20.0635,9.2325,14.278,-0.19216,20.9202,-8.1572,8.2391,-4.6541,-21.8534,-16.6406,-6.0559,6.1591,15.9037,22.7726,-12.6521,10.9182,-23.6579,22.5385,-9.6782,13.1512,6.8242,17.1904,-11.8179,13.1942,-18.1523,-3.0873,-18.899,-14.4067,-3.2288,5.5251,17.605,21.3931,-13.5314,9.9264,-25.4274,22.4866,-12.333,12.5188,8.8143,18.4369,-7.3233,12.8182,-14.5558,-3.5489,-19.6788,-13.5421,-2.5632,8.8204,17.7046,22.1889,-11.0792,-19.0075,0.3168,-9.8711,3.7339,-0.05806,15.0481,-9.3959,24.5634,-21.8005,11.49,-12.9871,-22.2311,-19.4706,-2.7009,4.255,19.4462,21.9433,-3.8657,-4.9802,-1.3583,-1.5846,-1.2815,-0.54297,14.4433,-10.1777,26.4347,-25.3711,12.4448,-13.9216,5.6826,4.9108,2.9731,3.31,1.8012,2.9364,22.8887,4.4619,22.6369,-24.3225,-0.51228,-23.4802,-17.7453,-25.8489,9.9737,-8.9323,24.7849,17.358,4.967,-0.87894,11.8942,-1.9518,9.0042,1.9347,-2.1045,23.7771,-21.8525,24.419,-17.1816,0.30205,-23.6817,-14.3282,-5.9172,7.95,21.0044,25.8018,5.7082,0.42148,9.119,-0.62241,7.599,2.9414,21.5941,0.50339,21.8206,-25.7255,0.14948,-26.0917,-15.4651,-24.9423,11.0024,-8.1815,22.4304,9.252,7.556,-9.4939,20.5036,-12.0137,13.9595,-4.0759,-2.9103,17.7654,-14.537,21.8883,-9.8366,9.4568,-23.5596,-17.3396,-3.9295,6.3506,18.8367,22.3026,9.2974,-13.092,22.9202,-19.6454,17.6363,-2.8689,0.32064,10.7597,-1.7326,15.4131,2.5207,9.0073,-22.7774,-19.2282,-4.3856,3.4793,16.4948,22.2697,9.8986,-16.432,23.8275,-23.702,17.7404,-2.4319,-7.2016,-2.2677,-9.2589,3.3412,0.10489,8.0384,-25.8424,-21.2001,-2.9112,7.5181,19.5369,26.9241,-1.0278,6.1101,-4.7295,8.8667,-2.5171,3.8536,24.9682,13.9376,11.2667,-11.2816,-13.0398,-25.0137,-0.93708,-20.6569,21.8216,-20.2843,22.4484,0.61887,-0.57487,-8.2718,9.2489,-5.4132,11.817,4.8035,25.6081,11.5511,14.4601,-13.8727,-10.9896,-25.2464,0.43772,20.5003,-18.5156,20.9041,-18.8218,0.56116,-1.1809,-10.7443,10.4827,-8.0333,13.1415,4.2219,25.9476,11.5702,14.626,-14.5928,-10.9452,-25.7577,1.234,19.4254,-17.5367,19.1481,-18.3083,0.084337,-2.2333,1.6603,-2.8792,4.0784,-0.6463,2.4048,24.5718,13.1608,12.33,-11.9815,-12.1964,-24.75,-0.56553,-22.5793,22.9381,-21.9114,23.0655,0.28084,-1.1715,-12.6594,13.0436,-9.8999,14.7429,3.2182,26.1007,11.5131,15.0521,-14.9914,-10.6847,-26.052,1.9441,18.671,-15.4667,17.8736,-17.3864,-0.693,-2.6626,-15.2556,12.6641,-13.5015,14.5399,1.1384,26.1363,11.2109,15.0603,-15.3363,-10.5753,-26.1604,1.3539,17.0092,-15.8756,15.0304,-17.9087,-2.8654,-1.2253,-15.6549,17.1748,-13.6869,19.6623,3.39,26.2152,11.0826,15.4511,-15.6376,-10.3645,-26.3202,2.802,17.0228,-11.1452,14.9148,-12.4562,-0.66026,-15.7636,2.5857,-16.8673,19.2031,1.1171,19.1553,10.4374,25.1003,-15.2102,14.1745,-26.2147,-11.3424,18.1173,7.2151,12.1511,-8.6918,-2.7321,-12.9263,-17.4758,-16.0198,0.91966,3.0705,19.753,21.5516,-13.2584,12.1036,-25.6508,23.993,-13.8041,10.7488,11.1583,15.8647,0.20116,9.403,-5.8288,-1.7277,-3.2444,-19.8547,15.1841,-17.2491,17.8494,1.8851,26.3907,10.8773,15.4847,-16.2173,-10.5123,-26.4854,0.0017646,12.2559,-14.096,10.8719,-14.8071,-2.0194,-18.0388,5.1521,-22.8393,23.7306,-3.7368,19.5156,-7.8687,-1.6432,-8.4404,4.1778,-3.1501,3.1756,-16.2341,-25.8034,9.6529,-9.9112,25.8553,16.1342,-20.1156,2.8006,-23.584,23.6232,-4.413,19.9407,-2.6466,-0.84932,-1.201,-0.33125,-3.3593,-4.3061,-16.772,-27.664,10.0218,-11.0655,26.9361,16.276,-20.383,3.6524,-24.2961,22.9432,-4.5738,18.6774,-3.0959,-2.354,0.13772,-2.8666,-2.1323,-6.2832,-17.3643,-27.5121,10.2995,-10.5192,25.9997,15.2476,-20.8404,-23.441,2.1727,-3.5148,23.5351,20.481,-2.3381,-1.7761,-0.82474,-1.8284,0.74926,-0.82237,-14.7808,11.1362,-26.9706,27.0009,-11.8513,15.8908,-19.427,3.7205,-23.8972,22.9237,-4.17,18.5346,-3.9243,-3.3749,-2.1334,-5.23,-2.6121,-6.2128,-16.2228,-26.1435,9.7515,-10.8904,26.1223,16.2618,-18.5517,4.2664,-21.8943,23.1942,-2.357,19.893,-6.9029,2.0076,-10.9558,7.2653,-6.4395,2.8055,-16.0861,-25.7869,9.5598,-9.728,25.0941,15.6236,-21.0795,-23.0545,2.7431,-2.9785,21.7246,21.0894,-1.2591,-1.1581,-0.64834,0.27818,-1.228,-0.49815,-12.928,12.1858,-26.9137,26.3941,-13.8287,14.3281,-20.0119,-22.6002,2.4652,-0.72747,20.4767,18.034,0.58325,0.44142,0.23007,-0.52718,-9.4761,-8.4437,-12.1223,13.2475,-26.3422,28.3337,-14.6258,15.9498,1.3817,20.3384,-17.4038,19.2212,-16.5438,0.2563,3.3263,13.5898,-12.6405,7.6327,-17.0574,-7.6368,-26.2733,-10.3915,-13.5547,15.085,11.6913,24.8959,-2.0991,16.8829,-17.488,17.7728,-12.2833,3.5149,2.2763,16.6946,-16.7191,11.9811,-18.4132,-4.7537,-27.0961,-11.9827,-11.4335,15.2044,13.9827,25.6945,-4.8266,12.5616,-15.2401,18.2672,-8.0983,8.6115,0.13787,18.1996,-19.2209,16.316,-19.6237,-0.76305,-25.8433,-13.7684,-10.9811,13.8479,14.5177,25.7774,-4.1893,9.1063,-9.8352,16.3437,-1.9802,11.3642,-1.0758,19.0814,-22.0955,18.5548,-21.5784,0.051278,-25.2937,-14.0117,-11.1149,11.0741,12.0974,23.0008,-4.8066,4.0379,-5.139,13.0512,2.8787,12.5072,0.32964,20.7825,-23.6024,21.04,-23.519,-0.055572,-23.6887,-13.3888,-11.5592,10.5883,8.6914,21.4668,2.5517,-18.5078,20.9603,-19.4792,19.1837,-0.11293,4.7338,-8.9173,13.3651,-14.9917,6.3545,-8.2105,-25.6824,-14.9716,-9.6926,10.0607,15.6551,24.695,3.1301,-17.9129,22.0249,-19.3853,20.2191,-0.36368,4.131,-8.6274,11.0091,-15.916,4.0838,-10.1133,-25.3284,-15.6712,-9.6167,9.0114,15.2101,24.1616,-1.6448,-2.6691,1.2338,0.035345,1.8632,1.9685,12.4777,26.3819,-14.2856,16.6476,-26.6143,-10.7435,-22.6455,2.4044,-25.1076,23.739,-0.48433,24.4226,-13.0954,-0.54152,-21.1152,2.9898,-9.1012,2.3671,6.2779,-15.1999,17.8718,-27.7655,13.7394,-9.9679,-24.0326,-26.9276,4.8489,0.61817,24.0332,26.1469,0.30246,-1.8225,2.5315,-1.2808,2.1141,0.36996,11.0125,25.6725,-12.9981,15.0004,-25.071,-11.2167,-23.4986,2.3643,-24.1312,23.6785,-0.086321,23.1999,0.58935,-2.1638,2.8317,-2.5653,2.3103,-0.33938,12.0101,24.7781,-12.6516,13.5984,-24.9229,-11.7505,-23.07,0.66048,-22.9422,22.923,0.29615,22.8412,-0.095672,-16.5447,17.9965,-15.4832,20.5068,3.923,26.1703,11.2503,15.038,-15.527,-10.9944,-26.2159,4.6515,16.6482,-10.6159,13.8214,-11.158,0.94378,-18.6954,-20.0555,1.2575,-0.41281,19.5984,18.3781,12.1528,-13.7689,26.288,-26.7118,13.7021,-13.8273,12.7829,7.2006,4.0608,-5.2964,-7.9149,-11.3395,-17.7017,-19.1086,2.2528,-0.86598,20.7712,19.0897,12.2141,-13.818,25.2652,-26.2832,12.608,-12.542,13.2302,8.9359,5.8729,-2.6511,-5.9112,-10.1579,-18.2972,1.7583,-18.3917,20.3369,2.0308,20.5231,11.6944,25.9405,-14.9427,13.6725,-26.6864,-12.4156,14.3956,6.3295,9.902,-5.9196,-2.0372,-9.6726,-18.0965,-20.5632,2.3568,-2.2148,19.9867,17.8765,12.5783,-12.7372,25.261,-25.2759,13.2844,-12.4314,12.4803,6.1438,6.1379,-5.4678,-7.1784,-12.5861,-19.779,-20.4543,1.771,-2.7666,21.3087,18.0055,12.7685,-12.9945,26.1467,-25.0148,12.5446,-12.1933,9.8323,6.9291,3.6279,-6.5283,-4.9483,-12.3071,-17.5657,-2.3164,-14.7758,17.1733,3.9457,20.8298,10.985,26.3971,-16.0116,15.5748,-26.1848,-10.0022,15.1916,1.3274,14.4779,-11.8795,1.3705,-11.2454,-19.2828,4.404,-23.4851,23.799,-3.6134,19.9192,-4.2383,-1.1712,-5.3177,-1.2732,-2.3914,-1.4717,-16.2762,-26.0472,9.7834,-9.9404,26.1463,16.1656,2.7285,-19.5181,22.5579,-21.2383,20.6016,-1.5856,5.4424,-5.6837,9.9638,-12.9097,2.7027,-9.2799,-25.2997,-15.5861,-9.562,9.4224,14.679,23.994,-25.032,-7.3796,-21.0264,15.2819,2.991,21.7355,-10.6237,-20.6008,15.374,-7.0731,27.5117,14.7747,1.7733,-15.5183,9.075,-18.9295,6.7965,-3.7155,-22.8837,-1.6463,-20.4352,21.3336,2.7757,24.4194,-12.976,-28.2853,16.4686,-16.4678,28.0722,12.777,7.0181,-1.0158,6.8679,-4.0258,2.1514,-0.77631,-23.3333,-2.4514,-17.4371,22.3158,4.0313,25.1794,-11.288,-28.2705,19.1929,-15.8449,28.4626,14.5171,7.6258,-0.17757,12.255,-5.4587,5.7024,-3.8773,-23.1923,-6.7179,-4.8615,22.9163,16.7866,23.1506,-10.1923,-28.4254,5.1799,-15.5406,5.2161,-1.9553,13.5614,-7.6009,25.3974,-16.8484,12.6984,-17.1699,-21.4288,-15.3929,-4.2917,6.6244,17.5896,22.4455,-12.8083,7.4496,-23.6509,20.6932,-11.2257,12.4668,5.3341,18.6056,-11.9074,15.214,-15.079,-1.5293,-20.6755,-17.1013,-1.9959,4.3893,19.0995,21.8693,-11.66,11.1322,-25.1257,23.2609,-13.9748,11.8658,5.1528,15.4563,-4.9919,12.4314,-8.7674,-1.5338,-21.9597,-17.3274,-2.5257,5.2601,20.0275,18.4391,20.0272,-2.1494,-5.1219,-1.081,-4.9793,-1.1469,-1.7491,-1.1471,13.881,-12.4103,26.4596,-25.6925,12.1905,-15.6276,12.1912,-21.9114,-19.4176,-7.4415,17.3359,13.254,26.7427,1.0514,1.624,-5.5814,2.31,-16.677,-6.1766,13.8974,-12.538,23.2841,-35.7279,7.304,-30.502,-1.1788,-0.98458,3.9188,3.2135,8.226,5.3099,26.2378,-0.93879,25.6171,-26.4282,-1.6251,-26.1208,-16.6033,-25.308,13.0593,-6.4378,22.3411,4.669,3.0095,-5.1482,15.2349,3.1775,14.7059,-4.7066,26.1262,-23.3112,27.3409,-8.1121,-27.355,-19.3641,-3.2982,19.7606,24.7136,-2.46,-6.8602,7.8327,-2.6012,12.2526,6.7339,23.4529,0.47458,23.5269,-21.1334,-1.5823,-22.6118,-10.8711,-22.3113,13.6121,-15.207,22.2125,6.7622,5.6858,-11.159,21.4019,-11.8124,19.1851,1.6498,-5.5764,13.8544,-10.909,22.9451,-0.075582,16.7059,-22.316,-19.0044,-5.1881,-1.8118,16.9132,18.3758,6.0657,-18.533,21.6718,-20.7723,19.4767,0.69203,-5.8619,1.9017,-2.7832,12.1423,7.226,14.5738,-24.0129,-17.0816,-9.6664,4.3561,13.6459,20.8954,3.3748,-19.2399,20.3022,-21.7777,20.4185,1.3045,-17.9767,-5.2552,-23.2182,5.7957,-2.5469,15.7914,-24.3858,-17.2577,1.8216,9.8818,23.6758,25.243,-3.4535,-8.5441,6.2954,-5.1573,10.3537,3.9832,25.1543,11.2786,14.2982,-13.7804,-11.0088,-25.303,-0.61985,20.4866,-20.0372,20.9533,-19.8112,0.12879,-1.5613,4.6905,-3.2174,8.1324,0.01749,5.1225,25.0213,13.7351,12.085,-11.3638,-12.5595,-24.7127,-0.53705,-21.1055,21.9576,-20.622,22.3151,0.7434,-3.7837,-11.0988,8.1235,-7.0873,12.2346,4.3749,25.5824,10.9482,14.712,-14.421,-10.7217,-25.506,-0.32629,19.4761,-18.8929,19.6774,-18.841,-0.061548,-3.2417,-12.3454,9.6423,-8.1437,12.707,4.0057,25.9366,11.0457,15.0922,-14.9691,-10.63,-25.8081,0.43473,18.8086,-17.76,18.8598,-18.5145,-0.23898,-2.1854,0.81257,-2.6636,4.0785,0.15918,3.9229,24.848,12.665,13.0181,-12.447,-12.0594,-24.9893,-0.12333,-22.2384,22.2852,-21.8769,22.6157,0.3408,-1.2414,-12.8605,13.088,-9.7366,15.5524,4.486,26.021,10.9597,15.4568,-15.3105,-10.2206,-25.9785,2.3559,18.453,-15.0711,17.6788,-16.7688,-0.28307,-0.85308,-14.2265,15.4995,-11.6221,19.0759,5.6227,26.2129,10.7036,15.7726,-15.9412,-9.6335,-25.9213,2.9939,17.9365,-12.5824,16.2137,-12.8146,0.91864,-2.4131,-18.8895,17.1286,-14.9599,19.7421,4.1855,26.5817,10.5476,16.1116,-17.1312,-9.5597,-26.6824,1.5843,13.2427,-11.5114,13.034,-12.3354,0.52968,-19.1136,-0.76356,-16.5452,18.9209,4.5694,21.9031,11.1767,26.2851,-15.3963,14.9196,-26.2068,-10.8256,13.348,3.0243,12.8039,-8.7268,2.7018,-8.5937,-21.5239,-19.5557,-2.9901,1.1664,19.0081,22.0106,-12.3426,14.3769,-25.2336,25.3791,-13.583,10.4008,2.2009,6.7195,-6.5989,2.85,-9.1398,-4.4035,-3.0736,-19.0137,16.265,-16.0173,20.0918,2.7992,26.5239,10.3738,15.9727,-17.4765,-9.6579,-26.4612,0.32924,13.5308,-13.3317,12.869,-12.0417,1.5205,-19.4574,3.9695,-22.7833,23.3465,-2.6697,20.0771,-4.1152,2.1421,-8.8259,5.0173,-6.4073,1.0987,-16.1273,-25.9176,9.7403,-9.9574,25.3611,15.7089,-20.1083,-23.562,2.843,-3.9946,23.5902,19.6175,2.1596,-1.5371,3.4041,-2.0794,3.8151,2.0703,-15.9354,9.7216,-26.5346,26.4432,-10.2316,15.9952,-18.5302,3.6213,-22.9461,22.5482,-3.8846,18.5113,6.2259,5.8432,2.6858,4.4968,0.70799,2.9803,-16.4169,-25.9976,10.2515,-9.9907,25.8742,16.1346,-18.3499,4.2687,-22.4128,23.1867,-3.7162,19.2551,7.3623,3.5973,6.5394,1.0587,3.9162,2.2343,-15.1092,-25.4992,10.2055,-10.0021,25.8282,15.9787,-20.5566,-23.3241,0.48466,-3.8879,20.7824,21.2966,-1.0274,-2.5817,-1.8113,-2.0521,-2.0153,-0.75689,-14.5428,10.6438,-25.0878,25.9405,-11.3304,15.0185,-19.629,3.8782,-23.5024,23.5523,-3.743,19.6825,3.8853,-0.32661,3.7167,-3.6027,2.3938,-0.65403,-16.1819,-26.4673,9.9052,-9.9172,25.9175,16.4435,-20.201,2.8903,-22.1626,23.0599,-1.152,20.8275,-3.5243,6.0204,-11.1685,8.4396,-9.2053,0.61734,-15.532,-25.3639,9.4214,-9.7022,23.9472,15.0218,-20.9971,-21.5741,-4.9131,0.55108,12.1943,18.7753,1.7552,2.3094,-8.7733,2.1641,-15.9315,-6.5313,-14.505,10.7597,-25.3626,25.7464,-11.034,16.2076,-13.0352,-13.1209,5.9914,7.003,18.7847,22.0905,15.0575,17.217,4.2216,9.9172,-3.4974,-3.1153,-8.6583,9.4128,-23.8508,25.2129,-17.2732,11.8393,0.25799,-20.2331,20.7006,-18.3844,21.7175,2.4961,7.359,-9.1313,14.4112,-18.6876,3.9026,-13.2454,-25.1457,-14.5454,-9.7637,8.1192,14.8491,22.7786,-1.229,19.9768,-19.004,21.9125,-13.5487,2.3192,5.0763,16.7681,-15.9228,6.0779,-22.4759,-15.7061,-26.8958,-10.3505,-13.7904,15.7389,11.3321,24.0578,-5.9839,15.4891,-18.3147,23.0927,-7.1872,16.5488,2.0972,20.6765,-20.5749,11.691,-22.515,0.60251,-27.8305,-11.9041,-12.2858,20.461,15.6099,25.9549,-8.2454,9.7032,-13.6153,19.0818,-1.193,14.1395,1.0689,21.3379,-22.1767,19.4988,-22.3832,-0.75066,-25.5918,-13.4916,-11.1959,8.8626,12.1572,21.5746,-11.0711,2.2511,-11.0098,14.2745,1.6978,13.8111,0.53846,22.1254,-22.3228,21.7923,-22.3874,0.014826,-22.6857,-11.8049,-11.34,9.8905,10.3725,21.1647,-10.7114,-1.4974,-6.3049,11.0237,9.4034,16.5931,-0.007864,22.3783,-21.4083,22.31,-22.774,-0.80017,-22.3035,-10.1268,-15.2957,11.2127,6.2016,19.2789,0.20155,-20.5876,20.332,-19.5329,21.2319,2.0003,9.5615,-6.2007,15.6555,-16.3192,3.2627,-13.092,-24.6665,-14.9313,-8.6682,8.3129,14.9745,22.4602,1.0192,-19.9683,21.986,-18.928,21.373,1.7617,8.8259,-5.5422,12.6492,-17.6548,0.91622,-14.7757,-24.6325,-15.4934,-9.1174,6.9018,14.3609,21.1675,-3.0251,-1.4744,-3.5388,1.0907,1.1683,4.4483,11.1179,-15.1756,26.292,-26.2258,17.2372,-9.2016,-24.3571,-23.6787,-2.5954,1.4225,21.9045,25.9967,-0.58238,-0.49411,0.96965,2.0589,2.1221,3.1909,11.614,25.9191,-14.1172,15.6006,-25.7913,-10.5987,-23.1942,-0.74388,-23.6646,22.6897,0.57225,24.1059,-11.201,1.1717,-11.7203,5.9287,5.5732,12.3715,9.4788,-16.324,25.3495,-26.4238,22.6105,-7.3785,-25.8549,-23.5514,-8.4457,3.6315,15.411,25.4509,0.45121,-1.3396,2.6518,-0.061595,2.2339,1.2988,11.4726,25.3339,-13.3914,14.5911,-25.3262,-10.9315,-22.8367,0.46474,-22.916,22.7935,0.26588,23.3947,0.35328,-2.3124,3.184,-2.7255,3.3987,0.12222,11.9048,25.1419,-12.9715,13.7412,-25.041,-11.6455,-22.5813,0.074776,-22.2514,22.2158,0.40097,22.5238,-3.9615,-21.0963,17.2739,-17.2781,22.1938,5.1378,26.3908,10.6411,15.4661,-16.1287,-10.1812,-26.0911,-1.6459,10.9357,-11.9484,11.5846,-8.4767,2.3095,-19.9241,1.5353,-19.776,20.1647,4.3159,20.4956,13.0393,27.8538,-14.8825,13.9873,-29.2099,-17.1695,11.216,4.0863,7.076,-6.3961,-2.5468,-8.0464,-18.7324,-19.9134,1.2196,-0.32729,20.7036,18.383,12.5811,-14.0266,26.2632,-28.5086,13.5837,-15.6526,10.566,7.2532,3.2086,-3.115,-6.1285,-8.6951,-18.5564,-20.3093,1.393,-1.8644,21.331,20.2672,12.6679,-13.7211,25.5098,-25.7104,12.3957,-11.7934,11.379,6.9493,4.1207,-1.7711,-4.7534,-6.2985,-20.9517,-0.74712,-17.817,20.1345,4.6958,24.4369,11.7667,27.298,-15.6528,14.8156,-26.4736,-10.0469,10.7639,2.5967,10.8207,-6.1926,2.9203,-6.4469,-17.2201,-19.2694,3.6523,-0.63569,21.6025,20.1622,11.7303,-13.0785,24.4376,-25.2659,12.1113,-13.1047,13.7066,8.4655,8.2005,0.64547,-2.1967,-4.7594,-18.3185,-21.6577,4.048,-2.5239,22.9092,19.4774,12.689,-12.2444,24.2607,-25.4213,10.3699,-14.0441,11.5472,4.1066,9.505,-6.5626,1.963,-6.5154,-20.2107,-3.5518,-16.3025,18.0244,6.0332,23.3135,10.212,26.1309,-16.002,16.5102,-25.4696,-8.9135,11.9612,-0.33927,13.2723,-9.378,3.6781,-6.4178,-19.733,3.5832,-23.0384,23.8605,-2.851,19.9905,-2.6259,-2.245,-6.6817,-3.6126,-5.7432,-3.0993,-16.1889,-26.0707,9.8678,-9.637,25.6045,15.7685,1.561,-20.4003,23.0257,-20.7764,23.1186,1.2479,8.4659,-3.3693,10.3258,-15.2529,-3.2864,-16.2663,-24.1004,-14.8765,-9.3424,7.6976,13.4633,20.5953,-21.9063,-5.0009,-15.47,19.5275,5.017,24.5127,-13.6201,-22.539,7.9861,-12.4964,16.2594,5.3554,3.7155,-13.769,18.8462,-11.6398,19.9221,7.1952,-24.3247,-6.2655,-21.5557,20.9912,0.52564,25.1686,-11.0439,-22.7608,15.154,-11.2616,26.7075,11.8566,-2.7708,-15.1307,0.3379,-14.0979,-1.9162,-4.6929,-25.0915,-2.8637,-22.5524,23.5455,-0.58849,24.222,-12.2273,-27.931,14.5116,-14.113,27.2235,13.9921,3.3074,0.43841,4.7923,-2.1311,1.0819,-3.1984,-26.9475,-2.627,-9.1058,24.6578,28.1317,-12.922,-29.0287,10.9853,-13.2394,7.2915,11.146,-0.915,41.9693,-5.5092,14.5665,-22.0468,-14.0448,1.8228,-5.9185,24.1553,19.8481,-0.23682,-9.03,3.381,-10.2958,2.092,-3.5683,12.9941,-24.7994,31.4704,-30.5996,12.4564,-17.2706,-21.9789,-17.4446,-4.2206,4.9811,18.9852,23.6833,-12.0352,10.0722,-23.6833,18.7343,-13.1215,6.9547,2.6477,15.2983,-11.235,17.7964,-10.0343,6.452,-20.2659,-17.6487,-2.0675,4.6161,18.8083,22.9352,-11.4043,11.8175,-24.4551,21.4403,-14.5789,9.3609,5.9174,12.7688,-0.19306,14.6325,-4.4658,4.0442,-37.3634,-14.0563,-2.6916,20.3503,25.1193,-1.1631,-11.081,3.4586,-12.6643,0.80144,23.5374,37.1843,-47.9183,7.3533,-31.0559,-19.2454,-12.6864,2.5476,4.984,21.7569,21.7975,12.4554,17.2575,-2.2574,20.8213,-5.3918,9.3008,16.5468,-6.6823,27.8398,-15.5289,10.2947,-10.765,-5.6248,-8.4874,-2.8934,4.7985,13.7746,16.2007,22.7586,-1.2201,-14.833,21.2479,-22.5213,-3.7883,-7.1342,-24.4296,-22.9017,16.1006,2.0179,21.885,-5.9477,-11.5325,8.5125,-2.7798,20.8994,16.2679,22.9906,1.3639,21.8672,-17.9067,0.027427,-19.6481,-6.9168,-22.8628,11.4643,-18.4003,14.3227,0.068282,1.5872,-15.4244,20.2952,-13.4209,21.362,4.7513,-12.0667,5.8154,-10.2228,21.695,4.498,19.9619,-21.2694,-20.3937,-6.5931,-3.4847,11.0401,14.327,2.301,-20.9872,20.559,-19.7394,21.5846,2.8223,-13.0217,-4.351,-5.631,13.4816,9.58,18.9931,-23.5277,-14.3023,-11.0849,7.4791,9.676,18.1999,-8.6982,-21.8635,7.7752,-20.5814,16.814,1.9302,-22.8025,-10.1603,-25.4113,7.9228,1.1081,17.1061,-15.2301,-14.1581,7.4934,11.4906,23.857,22.2,-2.9751,-6.1185,5.7267,-0.42876,10.9101,7.8805,25.1442,11.7388,14.1718,-12.8021,-10.7667,-24.2923,-0.33121,21.0598,-20.1616,21.9011,-19.5477,1.3187,-3.2664,-9.71,7.6902,-4.3628,12.72,7.0983,25.4299,11.1312,14.7607,-14.11,-10.3817,-24.8611,0.1338,20.0328,-18.8801,20.6931,-18.5644,1.126,-4.0959,0.99597,-4.2541,6.739,0.87832,6.8055,25.0065,12.9739,12.5237,-11.9639,-12.2031,-24.4894,-0.43821,-21.8032,21.6729,-20.9549,22.1406,0.77669,-4.0908,-12.0651,7.8508,-6.8763,13.1942,6.3885,25.6908,10.7453,15.0584,-14.9527,-10.466,-25.301,-0.018709,18.9417,-18.7476,19.2894,-18.2575,0.8329,-2.9704,-11.8565,10.6084,-6.3972,16.6033,8.5007,25.9405,10.7497,15.5912,-14.835,-9.5604,-24.7292,0.83572,18.9866,-16.6816,19.5322,-15.6908,2.3349,-3.3432,0.33125,-3.128,5.5371,1.5667,6.5721,25.0308,12.4656,13.2166,-12.4731,-11.6492,-24.5148,-0.36439,-22.0137,21.8105,-21.5389,22.29,0.40478,-3.9303,-15.0944,12.8358,-8.7521,19.0559,8.9083,26.1618,10.3873,16.2717,-15.7104,-8.8717,-24.8993,0.5551,16.9647,-14.7715,18.0861,-13.0372,3.1919,-4.116,-18.4585,15.369,-11.6883,20.5459,8.8498,26.7647,9.6496,17.0297,-17.258,-8.744,-26.1014,-0.21949,13.8528,-12.9893,15.8338,-10.9023,3.3173,-3.7247,-19.5545,17.533,-13.2856,20.8384,0.050814,27.0363,9.6527,16.8956,-19.4432,-8.8023,-25.2435,-0.53358,12.4597,-13.3234,15.4057,-10.9408,8.0561,-21.8433,-4.086,-16.7047,18.5035,6.0711,23.3752,10.7407,26.4784,-15.9323,15.5864,-25.9915,-9.8106,9.2466,-2.227,12.8976,-10.2589,5.1105,-6.5943,-23.056,-19.9432,-3.4592,2.3046,19.2517,23.1836,-10.6777,14.0543,-25.201,24.5798,-15.4322,8.8964,-0.36517,4.5966,-7.8646,0.041103,-4.4526,-1.3655,-3.7893,-20.3322,17.4149,-17.2068,23.7824,7.8163,25.6599,9.5656,16.6946,-13.3687,-7.6669,-22.4145,-1.2306,11.8029,-10.2707,14.2,-5.2931,4.9881,-20.3903,2.9613,-22.5934,23.4057,-1.2618,21.1341,-2.0324,5.583,-10.062,6.5419,-10.7388,-2.1229,-15.6589,-25.5767,9.1824,-9.6367,23.8128,15.3336,-18.5536,-22.6358,4.0527,-3.7946,23.6292,20.0672,-2.4175,-3.9321,1.3972,-1.4724,3.4367,2.0884,-16.4525,9.9468,-26.4514,26.2809,-10.4933,16.0666,-19.7023,-23.6703,-0.43723,-3.5947,25.091,21.6921,-1.763,-3.4675,-0.52599,-1.3799,0.54123,1.223,-15.7089,10.4562,-24.0638,26.8206,-6.1777,15.6164,-19.3021,4.1244,-23.2468,22.8853,-3.9661,19.1362,-1.7667,0.95172,-0.57078,3.5189,0.44902,1.8728,-15.8721,-26.0814,10.1873,-9.7987,26.2182,16.3347,-19.8361,3.9918,-23.5752,23.4337,-3.9628,19.4299,-0.31352,-1.6635,2.5133,-0.43647,1.9418,0.31874,-15.911,-26.166,9.5486,-10.0062,25.9234,16.4807,-21.3334,-20.8594,-4.9725,-3.3855,12.1234,22.5834,23.0795,-0.14884,1.2829,-7.158,1.7651,-11.8446,-0.99207,-0.73181,-14.416,9.0452,-23.0252,21.3028,-9.3929,15.2422,14.4296,-20.1175,3.8927,-24.286,23.6289,-5.125,19.7328,1.2189,-1.8593,1.6072,-2.4434,1.473,0.55015,-16.2806,-25.9922,10.9678,-9.513,26.065,16.4418,-21.8737,1.7659,-21.7635,23.7725,-0.071824,23.1754,-1.1206,9.4447,-13.8315,8.6355,-17.1464,-5.4997,-14.6048,-24.2042,6.5268,-9.1749,18.891,13.0672,-19.1973,-13.8028,-3.7931,10.002,18.2519,23.5598,6.0143,13.6893,-3.1502,9.6001,-5.4815,0.43212,-14.1789,6.8176,-27.6606,23.6236,-16.0424,12.4853,-20.2619,-17.3356,-4.445,-3.7923,17.4501,23.1605,-1.6319,-2.3115,-1.8552,-7.0947,-0.75523,-5.4264,-15.8428,35.7271,-29.3815,43.7191,-18.0716,10.7317,-21.3323,-18.2974,-3.0287,4.9681,18.9678,23.2877,5.7548,6.2656,3.0592,4.4247,0.11117,0.99137,-10.5666,15.8213,-27.2065,27.9262,-16.8165,11.7035,-3.5281,-21.9526,19.4727,-19.3886,22.4266,0.64922,12.6097,-7.8792,18.5512,-22.4971,3.7085,-18.9228,-23.8581,-16.0362,-7.7616,5.1292,17.022,23.0849,-2.1846,-21.9688,20.3723,-19.458,22.921,2.5158,12.4166,-5.3058,15.7231,-17.9383,1.2845,-15.5288,-23.162,-15.0169,-8.4084,6.0914,14.1285,21.4119,-9.5441,19.6334,-31.523,22.9712,-3.6993,10.785,23.5853,-29.5727,4.0547,-30.0389,-31.7931,-8.0216,-13.0646,20.5224,26.6694,-11.8945,14.9742,-17.1112,21.3287,0.87527,19.0988,3.0162,28.9702,-22.4471,28.592,-24.0131,-2.0132,-24.4725,-12.8903,-9.3581,-4.5328,14.9552,20.3674,-13.0814,2.9717,-13.1728,16.9534,4.158,19.3411,0.59094,23.1978,-22.518,21.8452,-23.1525,-2.1732,-22.0477,-12.2165,-10.4022,5.9907,11.1453,19.046,-12.7215,-0.63824,-5.8562,15.9215,12.0204,21.64,0.077406,22.5032,-22.218,21.6544,-22.6171,0.97368,-21.5852,-10.7625,-14.8358,7.9645,3.2829,15.2723,-17.9159,-10.7369,-4.8851,7.9365,18.0705,22.025,0.90678,23.0182,-20.9953,22.7399,-19.8196,2.7058,-17.6659,-4.4431,-15.3573,13.6896,-2.5713,13.3457,-2.3039,-22.3726,20.0988,-19.9572,22.5862,2.9138,14.3921,-2.6312,17.4557,-16.2729,1.2705,-15.6113,-21.8763,-14.0518,-6.4881,7.1932,14.2985,20.6243,-0.59662,-21.8605,23.4726,-19.3719,25.1334,3.299,14.3367,-2.6801,12.8609,-18.4703,-4.4791,-19.2722,-22.0585,-14.6793,-8.3209,5.3338,9.9434,16.8472,-3.1496,-0.63785,-2.0449,4.8265,2.4855,6.4452,10.6251,24.4024,-14.7936,15.1264,-25.5638,-9.3754,-24.1121,-4.1903,-22.7388,21.8378,1.7231,24.3603,-7.097,-3.8105,-3.7069,1.0182,5.2357,7.1958,6.6131,9.1866,-15.2008,23.3761,-23.876,17.2966,-7.9826,-6.8181,-25.536,-21.8064,-8.6274,2.5872,13.0891,25.2451,25.401,-2.0605,-1.146,-0.84175,3.3677,1.6404,4.9248,11.2002,25.3923,-14.2473,15.3071,-25.5759,-10.0244,-23.3658,-1.3671,-22.7599,21.5319,1.4718,23.4277,-13.3484,-8.6645,-7.3941,2.2671,6.6435,10.3902,2.8009,-15.9976,21.3487,-23.9176,22.4241,-3.5602,-22.9043,-16.8235,-11.2081,11.8956,12.2935,26.7826,0.49161,-0.45702,3.3866,1.9772,5.1293,4.684,11.585,25.1788,-13.4153,14.9172,-24.9639,-10.4955,-22.4817,-0.6348,-22.1723,21.3124,0.71467,22.6363,-0.27075,-3.5088,4.4999,-1.6814,6.5789,3.6283,11.8244,25.0608,-12.8724,13.9882,-24.4029,-10.8916,-22.2115,-0.35149,-21.7286,21.6453,0.60789,22.3485,-5.4891,-23.1622,17.1409,-17.6372,22.8281,6.5816,26.1135,9.581,16.0825,-16.8874,-9.6094,-26.3833,-4.4208,7.0686,-12.2433,10.197,-7.8509,2.8171,-19.8375,2.5666,-19.6239,54.6695,13.728,14.5213,29.7972,-14.8698,-13.183,-41.9082,9.5111,3.1714,7.7031,-11.7444,-2.2285,-24.4504,-0.87167,-19.4147,20.2003,4.4982,27.8681,14.3943,32.0392,-15.9694,14.5832,-25.7357,-7.6498,8.7394,0.34336,7.7638,-7.2407,1.7522,-7.2871,-18.8392,-20.9496,1.6405,-20.8052,21.8066,2.2539,19.7296,12.1267,-14.1446,24.8625,-14.4689,13.0103,-20.0164,-8.3939,7.8995,5.5237,4.1872,4.0398,-0.082549,-0.43699,-1.7416,-20.6954,-21.9077,0.35825,-2.4,21.9374,20.3272,12.8702,-13.0692,25.1895,-24.5242,11.5174,-10.808,4.1079,2.1113,-1.4508,-2.1932,-4.9082,-3.4703,-23.1993,-5.0222,-17.9012,20.285,4.4003,24.3868,10.8872,26.0164,-16.675,15.6823,-26.6787,-9.5621,7.3386,-1.3169,11.3412,-7.6712,3.3653,-6.971,-19.0786,-20.5205,3.5626,-1.503,22.6988,19.757,13.31,-12.0848,24.7191,-24.1408,10.5989,-12.9183,8.7592,6.2102,9.5211,3.1309,4.0413,0.16261,-19.45,-23.04,3.8892,-3.2412,23.0515,19.4629,14.8311,-10.675,24.7107,-25.3708,9.3919,-14.9044,5.9015,-3.0057,9.8652,-8.4269,5.342,-3.9579,-23.448,-5.6762,-17.0184,16.6163,9.2897,23.873,8.775,25.8009,-15.6865,17.0849,-24.2799,-8.4123,6.5926,-3.7792,11.6785,-10.4979,5.5729,-5.8173,-20.5548,4.2955,-22.7802,25.4184,-2.7039,20.6746,-0.79872,1.4921,-9.253,1.4319,-10.0674,-1.9094,-15.9678,-26.7364,8.4573,-10.0938,23.8019,16.482,-1.355,-24.7204,24.3201,-22.8712,25.236,1.6921,17.1178,3.4054,13.134,-15.506,-10.7122,-20.5297,-19.9901,-12.2744,-5.425,7.222,14.8836,16.038,-24.5086,-19.5869,-4.7501,5.1031,20.1379,24.3003,-6.238,11.4089,-18.6613,17.3209,-11.9088,6.4083,-6.1566,12.412,-18.1642,19.2031,-10.8197,8.0009,-26.0081,-4.9258,-22.46,20.343,1.4106,24.0986,-4.9978,-16.7643,13.596,-10.7343,24.8497,10.9138,-5.8241,-19.6072,2.9451,-12.6388,-1.8575,-4.9168,-24.0712,-2.7086,-23.4795,22.2126,-0.14401,23.828,-12.5944,-25.4526,12.4476,-13.7022,25.9696,12.6995,4.0474,1.7478,3.0108,-3.3268,-0.66714,-4.6684,-24.0693,-22.2131,-0.74251,4.0759,22.9792,25.4844,-13.9841,3.0043,-26.4397,9.9259,-12.6913,5.5812,4.4273,16.0331,0.18766,29.6172,-3.5882,11.1951,-19.7103,-23.924,6.2092,-4.8776,24.5378,19.5781,-0.23679,-6.3576,-0.14855,-10.5189,-0.47654,-5.154,16.9251,-12.3467,30.3429,-28.6132,11.3793,-17.873,-23.6013,-18.9332,-5.3018,4.0793,20.0882,23.1105,-8.3187,11.7067,-20.2833,17.8801,-11.6301,6.3423,-4.5983,11.7123,-15.3339,18.5745,-8.9095,8.6062,-21.1129,-17.6334,-3.8131,2.961,17.411,20.6427,-10.0009,10.9223,-23.0106,17.6962,-14.5119,5.0937,4.4247,11.2896,3.1544,16.7795,4.5603,11.2302,-25.3343,-21.327,-2.6727,-5.8907,20.6279,18.239,-8.157,6.2932,-23.7736,9.4706,-14.2959,2.5442,8.1755,-12.6632,14.4859,-30.6339,1.8864,-19.7561,-18.7282,-19.1247,4.2065,0.28108,23.6122,20.2867,2.3326,14.4728,0.56661,22.1728,-0.057515,9.5278,16.9117,-3.721,27.7353,-14.6693,10.557,-11.5429,-16.0143,-1.6226,-21.0192,4.501,20.9412,23.441,20.9679,-17.9796,-0.31374,21.9258,-12.5569,4.0533,1.0346,-18.5192,-14.3953,12.6998,-7.6872,7.0893,-1.0033,-23.7635,22.4209,-20.3786,23.8469,4.8416,-17.5028,-6.4362,-7.705,16.0187,7.0297,21.2896,-21.1666,-10.5127,-10.2083,9.2645,4.8285,15.6357,-9.4199,-24.3511,5.219,-21.9402,12.2416,-2.6413,-22.4568,-9.4011,-21.3795,15.9705,-0.088116,19.6312,-12.6489,-11.9916,10.8054,13.1314,22.1189,19.5867,-6.3102,-6.2756,1.9136,1.6206,10.044,9.723,24.766,12.0494,13.2992,-11.7935,-11.2061,-23.4945,-0.94617,21.0964,-21.4695,22.0237,-20.0874,1.3245,-6.5472,-9.7942,4.6646,-1.787,12.2436,9.0087,24.8768,10.9931,14.3223,-13.1186,-10.6019,-24.0968,-1.1454,20.1508,-20.3514,21.5129,-19.0275,1.5884,-5.8437,-11.7629,7.2465,-4.2552,14.7413,9.1424,25.3344,10.5387,15.0713,-14.0683,-9.9267,-24.2344,-0.72498,19.2253,-19.0351,20.5429,-17.5425,2.0821,-6.4154,-0.52988,-5.3717,7.5842,2.3255,9.3195,24.6762,12.44,12.5459,-12.0071,-11.5823,-23.567,-0.60167,-21.8731,21.5041,-20.9199,22.0136,0.78907,-6.797,-13.635,7.898,-6.0849,16.664,9.7056,25.301,10.2297,15.3106,-14.716,-9.4237,-24.3925,-1.2747,18.2022,-18.6482,19.7319,-16.1274,2.5377,-7.8512,-16.3987,9.6086,-7.532,18.5956,10.2421,25.2709,9.4186,16.0325,-15.5461,-8.6474,-24.5911,-2.3953,16.3133,-17.4023,18.871,-14.0963,3.3911,-6.0666,-1.2004,-3.9823,6.6158,3.6923,9.3732,24.5217,11.9688,13.1177,-12.2142,-10.9693,-23.5362,-0.56141,-21.9089,21.5446,-21.3891,22.1745,0.49527,-7.3677,-18.5169,11.8338,-9.1519,21.0662,11.9233,26.1297,8.8668,17.7163,-16.795,-7.9311,-25.1805,-2.7966,14.2104,-14.8095,17.6665,-10.8716,4.2432,-7.2603,-20.6836,20.0408,-10.5874,20.4616,6.8065,28.1872,8.3851,16.114,-19.3384,-8.5051,-25.9684,-4.5034,11.4922,-19.9859,15.8959,-11.576,6.7629,-3.6317,-20.0916,16.4718,-17.6376,23.735,-0.59469,22.171,9.1295,15.2537,-23.0408,-6.7351,-30.0056,-3.2592,11.9297,-9.758,25.7551,-5.9354,16.3169,-23.212,-5.474,-17.1232,17.8739,4.0067,21.5399,9.909,26.4856,-17.7719,16.0351,-27.4875,-9.6167,7.1329,-4.7469,11.784,-13.038,2.278,-10.8161,-23.1408,-19.3704,-3.6625,5.8925,19.4991,22.8901,-9.2785,14.1141,-24.838,24.2735,-14.5863,10.8061,-5.4126,-1.7889,-10.0362,0.21011,-6.9418,-0.59702,-6.7641,-24.2881,14.9539,-16.6913,23.6758,9.8616,23.3533,7.1608,16.2653,-15.623,-7.7382,-23.2175,-6.3854,4.77,-11.4122,8.9184,-6.8978,2.0783,-22.5107,1.6461,-23.1149,23.7469,-1.5332,22.1451,2.1033,11.376,-10.8871,10.5225,-15.0362,-2.3545,-14.6379,-23.7909,6.2779,-6.8711,19.461,15.0681,-19.0907,3.5103,-22.8622,23.2442,-3.4642,20.4743,-2.0465,1.3092,-3.6991,3.7723,-3.8286,0.3656,-16.4822,-26.1564,9.8039,-10.3014,26.0226,15.9352,-20.0345,-23.5775,4.1515,-3.2049,24.2463,21.4313,-2.0568,-3.3976,0.29476,-0.80122,1.8665,2.1589,-16.3273,10.0899,-27.0405,26.4652,-12.5548,15.6061,-21.7055,-25.8435,13.6581,-2.8014,25.1069,22.2476,-2.3064,-1.5178,-4.9536,0.27985,-5.0939,-0.54861,-15.1251,11.4241,-23.4895,28.0342,-15.8428,15.2376,-19.4676,3.726,-23.3471,23.0101,-3.5813,19.4604,-0.012548,1.8264,0.71633,2.0889,1.8905,1.4365,-16.212,-26.1387,9.4527,-9.9497,26.1752,16.4828,-19.6661,3.9173,-23.4547,23.0553,-3.8209,19.4345,0.50153,-1.0351,3.3767,-0.046822,6.5385,4.8166,-16.4105,-25.9333,8.8126,-9.7344,24.6016,15.8381,-22.1253,-5.4471,-8.6385,19.3462,5.4163,23.7568,2.2445,-6.9998,15.6569,-7.1658,14.4693,0.85838,-14.2306,-28.2487,16.0147,-14.8989,25.5466,13.3282,-20.2934,3.7219,-24.5512,23.6637,-3.0874,20.4878,-2.9048,-3.9268,-6.5208,-0.89122,-8.6529,-2.5176,-15.9712,-25.2544,8.2033,-9.1794,25.2471,17.0807,-24.2482,-0.060426,-23.892,23.5378,-1.3755,23.3002,4.0124,16.3869,-10.695,12.1359,-20.3198,-7.381,-13.466,-19.3451,0.28455,-4.6321,13.4246,13.0498,-23.22,-18.1127,-4.3392,3.9342,20.0008,23.4085,1.0568,6.5676,-1.1586,5.9423,0.14199,1.7691,-11.8837,15.7634,-28.6805,27.7965,-16.4548,12.5226,-23.6593,-17.2698,-5.3105,5.0937,18.5108,24.2257,0.90378,-4.0442,1.0927,-12.5372,0.11517,-8.6777,-12.2713,19.9022,-31.2363,29.4902,-20.2061,9.5976,-23.1742,-18.9485,-4.6614,3.314,20.1312,23.3901,-0.67778,0.16726,-1.5803,1.5099,-0.81081,1.4305,-10.9188,16.9771,-27.2956,28.1556,-16.2746,11.658,-3.5827,-17.6295,19.6539,-18.7288,22.7975,2.9187,18.4424,1.4469,22.5842,-21.1569,-0.88274,-23.9853,-25.6516,-22.1367,-4.8339,-4.5549,18.4657,16.5232,-2.9686,-20.3031,20.0705,-17.9731,24.3302,4.8451,14.2298,-3.6614,16.7207,-20.6095,-1.9111,-22.4616,-23.1134,-18.251,-7.0894,-1.5598,11.7309,14.1876,-2.9638,-22.5862,20.543,-18.641,25.0405,6.1963,14.9884,-1.1919,15.6021,-18.5462,-3.0647,-21.3972,-21.3217,-14.3225,-7.5886,3.3315,9.0134,13.4353,-17.8287,-0.36999,-15.3462,17.6983,6.7173,22.6551,2.1679,23.1701,-21.4904,20.8495,-24.3015,-4.2143,-20.946,-15.8125,-8.2509,1.9282,10.0661,15.5347,-20.0795,-3.9949,-11.6995,16.5214,12.0997,27.6544,1.498,23.6636,-25.4444,21.016,-24.2246,-7.6216,-19.1098,-10.795,-10.818,5.8925,2.8505,11.7658,-19.9569,-9.9405,-2.9731,25.4488,15.4619,21.9345,-0.12277,24.4708,-21.7642,26.4655,-18.3744,27.2822,-16.9726,-4.4544,-19.0122,12.0756,-9.8241,19.1028,-22.5715,-19.0519,-3.0623,3.4354,19.7567,23.481,-2.1062,19.0584,-20.0719,24.2304,-15.1733,1.208,-10.1285,5.6665,-16.3538,15.0473,-10.768,1.2524,-3.2142,-23.4568,22.9977,-18.8657,26.3346,5.2663,17.1999,0.0091843,15.5681,-18.9695,-2.7338,-22.5353,-19.6323,-13.0357,-7.2838,3.6968,5.3286,10.8961,-2.2621,-26.2966,34.5752,-21.7641,34.4818,0.63292,19.985,3.7041,11.7333,-20.6017,11.8457,-22.9917,-17.1654,-9.5634,-4.9457,5.5103,-2.5911,9.8815,-5.8447,-4.1837,0.41058,2.9691,8.3606,10.0072,8.7496,-15.7502,25.5311,-25.2469,18.7114,-8.2955,-24.425,-21.9125,-0.43153,2.7238,20.4779,23.911,-5.8532,-2.2614,-4.3764,4.6962,0.25112,5.8064,10.2567,25.1256,-15.2791,16.1029,-25.4811,-9.0029,-23.255,-2.4477,-21.3246,20.9487,2.5942,24.1701,-7.5027,-1.5617,-4.6206,7.6422,-0.3073,11.1269,6.621,22.3932,-45.9731,19.0879,-47.6078,-5.4151,-24.9847,-13.3576,1.9101,13.0797,5.8263,25.0281,-3.6176,-0.5484,-0.83018,3.8005,4.9356,6.48,10.6587,25.3035,-14.4862,15.5969,-25.1107,-10.1335,-22.7842,-2.1031,-21.4883,20.6648,1.5999,22.8511,-11.8625,-8.8284,-4.1747,-0.10034,6.6518,7.576,-1.0445,-18.833,22.517,-21.9916,22.5796,-2.5341,-23.0989,-13.425,-10.5371,12.0779,14.013,25.4318,-3.4479,-3.8538,1.8484,1.7161,7.0205,7.315,10.9757,24.9817,-13.7429,14.8217,-24.4679,-9.8392,-22.36,-1.148,-21.6624,20.9403,0.93744,22.3665,-2.5965,-5.4766,5.1334,-0.38589,9.8893,7.3353,11.2851,24.6345,-12.66,14.2383,-23.1233,-9.6766,-22.1525,-0.68602,-21.5238,21.2498,0.47662,22.0396,-5.7048,-23.1979,16.5024,-15.6767,20.6595,7.5327,27.3813,9.2984,17.2595,-17.0637,-10.1045,-27.291,-3.9331,7.6808,-13.6806,12.5603,-11.1578,2.6068,-33.8643,-33.9283,21.3943,-19.7726,23.7267,3.4877,27.6519,27.325,14.3605,-15.3124,-7.4352,-21.91,7.7046,9.4691,-2.9326,7.7592,-4.2763,0.56489,-24.7833,-8.0955,-20.4165,19.5309,0.75394,22.6447,10.0534,22.9622,-16.7453,14.9698,-25.2947,-8.6175,5.919,-0.47202,6.7034,-9.8349,-1.1166,-10.6666,-19.4065,2.1239,-21.9046,20.7568,-0.44445,19.583,11.5139,24.5485,-14.4649,14.5395,-23.5106,-9.2398,2.2758,0.7568,0.59058,-5.4206,-4.2218,-9.0405,-20.4344,-22.451,3.0461,-1.1819,23.1777,20.9605,12.6234,-13.7405,25.0283,-25.1429,11.8322,-11.4528,1.7773,2.7135,-1.2017,-3.6694,-1.5884,-4.7691,-22.2742,-3.9703,-18.3273,20.962,-7.1999,22.7074,10.6435,26.3075,-18.093,16.1383,-23.0632,-9.6018,9.6727,0.26765,11.8747,-9.6578,6.2789,-8.7649,-19.4577,-22.8031,3.7607,-2.2064,22.8682,19.8225,14.7719,-12.11,25.416,-25.6083,10.3472,-14.3789,4.3856,1.0391,9.1437,0.68792,6.3356,1.4193,-19.389,-23.2236,3.5462,-2.9962,23.6707,20.1722,15.4461,-10.5115,25.8622,-26.6077,10.4349,-15.8001,4.6943,-5.1858,7.0204,-7.8027,1.9529,-3.0161,-24.2435,-8.1963,-15.0859,12.5503,10.6505,22.8595,8.0147,25.4761,-17.0066,16.9115,-25.0894,-9.9744,5.8393,-6.7429,10.687,-14.4288,4.0754,-8.2687,-21.043,3.8766,-22.643,24.3572,0.82849,21.7276,-0.45514,8.4302,-12.4614,6.3955,-14.2076,-4.4175,-16.9308,-24.6853,4.0125,-8.4224,20.8044,15.275,-36.2414,-14.441,-2.788,16.9413,21.0621,5.9457,-14.9251,4.9154,-16.3078,-4.3232,8.2473,-30.4443,30.7284,-10.4346,16.0313,-1.3441,-36.2673,22.623,-32.5106,21.8641,21.4072,8.7964,11.1577,-33.0569,-13.188,-15.1682,-11.0923,-1.1911,12.6818,10.9727,-25.4461,-4.2204,-21.6549,19.7446,0.98681,23.3682,-6.5865,-18.6405,13.2586,-11.5875,23.5139,10.0337,-6.5643,-18.9611,2.6775,-11.8303,-2.8658,-5.7553,-24.2172,-20.0799,-3.1701,5.4295,20.3039,23.9309,-5.9713,12.2321,-18.764,19.9178,-12.2715,8.201,-7.7548,10.9602,-18.7493,18.3657,-10.0627,8.2563,-23.6232,-2.4437,-22.2041,21.5903,0.63159,22.9486,-11.1475,-24.1724,12.3146,-13.2862,24.518,12.3077,5.1843,2.2393,3.9028,-3.9018,-0.99283,-5.8433,-22.8351,-20.6068,-1.6239,4.833,22.3781,25.959,-13.2783,4.9376,-25.0141,10.5258,-12.1823,4.7289,4.2553,17.4181,0.76561,28.5882,-4.4454,9.4015,-19.2272,-24.411,6.4149,-6.2421,25.5501,18.875,-0.61621,-5.3718,-2.3069,-10.466,-1.5665,-5.3357,19.857,-10.7152,31.7093,-27.1705,10.8999,-18.2707,-23.0492,-19.9404,-1.9059,3.4256,19.6118,22.7728,-6.0661,12.4217,-21.4349,18.8938,-15.3204,5.8735,-7.87,10.1431,-14.2159,18.7795,-4.3402,10.414,-23.3035,-18.7368,-3.3249,3.6589,19.2518,22.7827,-9.7616,16.3412,-25.5742,22.8438,-16.6303,6.6332,-4.7704,2.9857,-5.844,12.9489,-0.96082,10.4813,-20.909,-23.3327,1.9219,-5.0018,22.3285,19.4594,-7.8769,9.993,-21.8999,15.1183,-12.8283,5.8187,12.7828,-9.7341,16.7844,-27.798,4.3454,-18.3391,-18.9508,-22.2086,3.0839,-1.6024,23.7605,20.7967,1.6379,12.3721,2.5481,21.3563,1.1957,9.6242,16.7386,-5.2415,26.5162,-18.0868,9.863,-12.7044,-1.9067,-24.5983,27.4279,-18.6212,25.3574,8.0898,-23.0002,-6.1162,-20.207,20.1823,-7.0962,21.0096,-17.96,-4.8677,-4.6673,14.4094,9.2387,16.5862,-7.6064,-1.9925,1.8385,11.6276,11.533,-17.5485,52.7912,-26.5843,40.3016,-7.4241,-24.8135,-27.3111,3.2877,18.0189,23.7522,-8.0669,-24.7866,2.0103,-19.999,9.8192,5.8689,-22.4289,0.57584,-22.4142,42.3802,-2.639,35.4238,-11.5818,-14.9194,9.7212,-1.2352,21.3676,7.0263,-9.2048,-1.0401,-7.4176,8.433,2.0003,9.701,23.8005,11.9398,11.8838,-11.7061,-11.8662,-23.5861,-1.1037,-22.0641,21.2737,-20.7859,22.072,0.90474,-8.2571,-8.6828,2.4317,0.86983,10.9113,9.7592,24.3247,11.5547,13.3788,-12.2752,-10.8795,-23.7558,-1.4255,20.7396,-21.2721,22.0288,-19.8736,1.3915,-7.9884,-11.4405,5.3599,-2.9033,14.1261,9.3074,24.6702,10.7179,14.4023,-13.7402,-9.9029,-24.0997,-1.5154,19.6837,-20.1372,21.0895,-18.26,1.8061,-8.6563,-14.5024,6.7618,-5.6641,16.7905,10.242,24.6427,9.8321,15.0657,-14.7131,-8.9467,-24.0223,-2.1894,18.0551,-19.1929,19.9432,-16.3862,2.5415,-8.9413,-1.9019,-6.134,7.4404,3.5844,10.092,23.7965,11.6832,12.3236,-11.8721,-11.3152,-23.2965,-0.62455,-22.0627,21.5614,-21.2708,22.0758,0.57307,-9.132,-16.4675,8.1364,-6.2498,18.7486,11.6558,24.9698,9.3268,16.0045,-15.0988,-8.3186,-23.5536,-2.7436,16.6552,-18.087,19.5208,-14.3374,3.7863,-9.4437,-18.3785,9.1947,-7.6323,21.2766,13.0781,25.4424,8.5867,17.6858,-15.9612,-7.25,-23.7805,-3.5702,14.6959,-16.0559,18.5097,-11.5526,4.8755,-8.8257,-3.2393,-3.8978,6.3589,5.8243,10.5155,23.6293,11.2125,12.9745,-12.07,-10.286,-22.9743,-0.65068,-22.0566,21.8111,-21.5485,22.3031,0.27503,-10.4788,-21.1179,14.1824,-8.9938,21.3583,14.1964,27.2694,7.4947,17.8808,-17.5066,-8.0582,-25.3274,-5.0242,11.8127,-17.9509,16.3393,-10.6833,3.8887,-2.824,-20.9889,22.856,-11.0249,21.6668,31.5459,8.0876,22.1624,-26.1156,-37.8471,-14.179,11.188,-26.5028,21.4531,6.1303,-7.0437,-22.2613,14.5411,-22.2613,24.5938,-8.4175,12.9144,22.3581,7.8433,16.1878,7.8432,-5.0844,-15.5446,-20.8527,-2.983,10.1859,-8.5323,10.1859,-4.2045,14.4946,7.6374,-20.5091,-3.2127,-15.8613,30.7129,16.7328,10.228,27.7097,-19.8809,13.8089,-33.6931,11.4026,-1.0586,14.2901,-13.5061,2.192,-23.6482,2.175,-21.4979,21.0511,2.3391,21.283,5.6559,22.8589,-14.7423,14.9568,-22.2086,-8.5306,3.3696,1.1511,1.8795,-4.9597,-2.4961,-7.2978,-23.0452,-19.5652,-3.5047,3.3132,18.7976,22.7899,-11.2759,14.7682,-26.3315,23.4593,-17.0768,6.6246,-0.078636,6.441,-4.5718,12.5777,0.66382,11.6179,-8.5899,-24.4027,14.3826,-13.0137,24.768,11.5897,24.7448,7.0297,17.4518,-17.7065,-6.2483,-23.2721,-3.7892,6.454,-9.9018,10.7702,-5.2985,5.5045,-22.0046,0.4979,-22.4063,21.3158,-1.4995,23.3468,2.9004,14.8159,-12.202,13.3806,-17.7344,-3.8696,-14.8289,-19.0806,-0.20463,-4.0627,13.9911,16.0967,-20.0393,2.6148,-22.753,23.8312,-2.7469,21.7346,-2.9711,-0.28318,-6.1226,0.75275,-6.2244,-1.9817,-16.1638,-26.1472,9.3091,-10.7452,24.7181,14.6223,-19.5458,2.2787,-22.996,23.0605,-4.0336,19.9243,1.7398,5.757,0.92842,5.6802,2.6959,3.2061,-16.0729,-25.2443,9.6307,-9.2276,25.6578,15.0821,-21.445,-24.0461,2.6013,-2.2122,23.9127,21.5536,-2.1464,-1.8758,-3.5091,-0.83505,-5.15,-2.9451,-15.4793,10.5412,-27.7278,26.3904,-12.9218,15.6117,-22.3189,-24.8489,-2.9456,-2.3135,17.6212,19.4281,0.17848,4.0577,-9.6389,2.2597,-14.874,-7.3501,-14.8856,12.4902,-26.6386,28.1584,-10.2346,16.2589,-19.474,3.5356,-22.978,22.7157,-3.9115,18.6319,-1.8437,-2.3649,2.5649,-0.65561,7.123,4.3939,-16.4408,-26.1136,8.3961,-9.8952,23.9852,15.5572,-19.8428,3.5435,-22.7919,22.7778,-3.9889,18.7499,-5.6473,-7.3815,-5.8777,-2.7601,-5.0186,0.083995,-15.4221,-24.2342,7.9092,-8.5148,25.0189,17.1918,-23.2804,-3.3655,-19.1429,21.954,1.8553,22.8903,1.2235,-1.6745,8.5354,0.082967,6.9559,1.3934,-12.8976,-28.6416,14.0458,-15.0924,26.9302,13.5804,-22.3576,-18.1165,-4.6139,2.1071,20.3055,21.7625,1.8386,13.9621,-9.3937,8.1462,-10.2434,-4.4372,-15.112,10.6073,-27.374,27.2317,-12.5673,15.8775,-21.2699,2.3041,-23.995,21.9062,-5.7305,16.6534,2.2573,8.3953,-8.6573,8.6395,-5.5501,6.0345,-16.8246,-24.9898,7.5483,-7.4476,25.2477,19.8657,-21.2475,-0.89386,-22.0536,21.5305,-4.7554,18.3535,9.1621,19.583,-9.3866,14.2207,-18.0136,-4.8191,-12.2335,-12.8452,-1.961,3.6844,15.067,21.9586,-23.3148,-20.043,-3.2937,0.89004,21.6792,22.8884,-1.6519,-0.094551,-1.8288,1.3121,-0.29265,1.3187,-11.7008,16.119,-28.1447,27.3221,-15.6982,12.2048,-24.9257,-19.2238,-7.1286,4.0741,19.0573,24.2524,2.1917,-6.4406,3.5089,-15.873,1.0273,-9.5253,-11.5032,18.2004,-30.778,26.9613,-19.6985,8.2025,-23.2151,-20.0964,-3.616,1.1697,20.9864,23.2323,-1.7373,0.39827,-3.1058,1.7083,-1.4653,1.2999,-10.2416,16.4867,-26.0166,27.0856,-15.852,11.1192,-4.8245,-20.6169,17.64,-16.1982,24.9074,8.8648,21.4923,3.1358,19.7288,-20.247,-5.841,-25.3708,-16.9059,-18.4513,-1.2337,-5.3628,9.1021,6.6331,-5.6893,-22.4851,18.7834,-16.0791,26.6494,9.7133,20.4572,2.7327,17.6012,-20.0708,-7.6954,-26.2008,-15.0624,-12.9539,-5.2043,-3.4868,1.9487,1.3615,-6.1749,-24.9796,19.4397,-17.7956,25.9326,7.7787,22.0341,3.796,18.1159,-19.4664,-8.8975,-26.8517,-12.7102,-8.231,-5.3924,-0.85449,1.7417,1.2868,-23.2387,-6.4568,-16.1101,16.642,12.2374,26.7425,3.2856,24.4612,-17.4108,19.806,-24.1464,-5.7408,-15.9586,-10.1719,-10.0065,1.637,-2.6807,3.0703,-37.5485,-11.5443,-6.6071,14.6196,27.3227,-2.241,25.0324,-46.6079,18.5567,-3.8655,-14.9261,-1.7593,-29.8411,6.1402,-1.4241,-18.8973,-15.3274,-4.8598,-13.5121,16.8168,15.9123,23.218,-5.856,19.3257,-21.3903,22.186,-21.0919,18.6773,-4.0808,-5.2232,10.7383,-15.6619,11.0569,-13.3825,8.2506,-1.4892,-4.5705,-24.706,19.6071,-19.1106,20.6551,3.0363,23.3355,4.4193,20.6164,-18.8173,-12.1426,-26.2694,-10.8389,-5.5038,-6.4638,0.29416,6.0003,4.9449,-1.9713,0.18946,-20.24,17.6732,-19.6362,12.5886,-6.3623,21.2112,22.5746,2.3752,13.7889,-14.4096,-2.5339,-19.8546,-10.0907,-9.0117,-3.4177,7.893,2.2297,22.4596,16.2045,-7.6653,-1.2726,-6.6513,7.093,0.45501,7.9442,9.1439,25.3123,-15.7864,17.592,-25.1974,-8.4031,-23.6725,-3.3316,-20.4489,19.3362,4.3078,24.0061,-10.1991,-8.7346,-0.39929,0.84589,9.6433,10.1701,7.8729,-17.0168,27.2657,-25.6302,20.6524,-7.2928,-23.7656,-20.5794,-3.0473,4.6021,18.2662,24.1656,-4.0516,0.84492,-1.3487,7.5046,6.7301,10.6663,9.8842,24.7894,-14.8637,16.1705,-24.8416,-9.0103,-23.2589,-3.7212,-21.1785,18.8775,1.6765,22.1862,-7.5419,0.5297,-4.3874,10.7579,5.1836,14.2385,2.9037,23.3162,-23.883,20.291,-27.6574,-4.9174,-25.1576,-11.1518,-11.8208,14.1505,5.1378,22.2544,-11.3203,-11.1337,-6.7395,1.3244,9.1175,12.5883,7.3062,-19.8416,29.2981,-27.9957,22.6423,-6.7994,-23.8452,-20.7049,-13.2073,5.5232,16.9717,23.6894,-6.6517,-5.1313,-0.96805,2.7589,7.7847,10.0046,10.2891,24.9972,-14.672,15.3026,-24.2813,-9.0352,-22.5402,-1.6727,-21.0999,20.5418,1.2125,21.9991,-8.6621,-8.1057,0.51007,0.84604,10.4684,10.0485,-0.16545,-20.782,22.7435,-23.4778,20.0426,-4.095,-23.7342,-14.28,-10.2818,8.9632,14.0024,22.8855,-6.6346,-6.347,1.2266,2.2683,9.0276,9.8213,10.2296,24.5112,-13.966,15.1185,-23.6171,-8.88,-22.1873,-1.2667,-21.3955,20.5602,0.82964,21.7996,-5.9915,-8.5619,3.6675,-0.83639,9.9085,7.9663,10.1488,23.6425,-12.9993,14.1191,-23.1573,-9.4096,-22.2877,-0.64415,-21.7971,21.3009,0.54919,22.1602,-4.7982,-20.0829,6.749,-10.9808,24.9561,12.2942,29.9854,10.3795,24.3321,-14.0517,-6.294,-21.3333,-3.1323,11.822,-9.3832,17.089,-1.831,12.2848,-2.2288,-23.0186,21.4209,-20.148,24.0228,2.2969,22.1775,6.9402,14.7727,-15.211,-8.3272,-23.4825,2.3968,9.27,-2.0054,8.3685,-3.4594,0.456,-21.7159,-2.4967,-20.4269,21.562,5.6152,24.1527,9.1564,23.8285,-9.2037,15.1143,-19.455,-8.0081,10.7744,2.2981,11.5025,-4.685,3.6856,-2.542,-19.5749,2.5934,-20.8623,22.4815,0.98244,21.5885,10.6034,23.9456,-14.9117,12.3663,-25.2323,-11.7214,9.09,8.9252,5.1313,6.3738,-1.3278,0.39864,-20.6067,1.2793,-20.0495,21.6268,2.9683,22.6687,8.5538,24.1515,-15.0143,14.5354,-23.8619,-9.3496,9.1815,3.9092,7.7379,-1.3508,2.1934,-1.4071,-20.9713,-23.5539,3.5688,-0.84829,23.4482,21.3944,13.7761,-12.4927,26.2815,-25.6386,11.8745,-13.4232,0.90043,-4.1562,2.2642,-7.2739,3.604,-0.87051,-23.513,-23.4818,-2.2805,-10.8943,18.2654,8.9369,24.3705,9.2542,9.361,18.8055,-13.2121,11.13,-18.9789,-7.1818,7.2272,7.1912,-2.7375,11.2141,-7.6932,8.6662,-1.6645,-19.872,-23.3913,3.6579,-2.4704,23.3006,19.8785,16.2605,-11.9026,27.376,-27.0635,11.3785,-15.7669,1.4671,-3.6835,5.6286,-2.2797,5.5697,2.6262,-20.166,-23.8301,4.9098,-3.8353,24.1864,17.8851,15.8966,-11.576,26.0738,-28.0682,8.145,-17.4021,2.6188,-2.6125,9.8501,-1.6948,11.2215,4.2792,-25.0332,-11.8787,-13.898,12.9796,11.9603,24.8291,7.6376,24.0261,-17.9319,15.1722,-24.0268,-7.7208,5.8265,-8.8202,10.6319,-15.3059,6.1517,-4.7417,-22.1443,2.2674,-21.17,20.1321,1.3389,16.24,0.82703,13.0749,-10.8964,16.1887,-8.712,7.2632,-16.7418,-22.8233,-1.0211,-5.4581,23.9714,21.7649,-25.1741,-3.8334,-21.2355,19.7606,-0.011422,22.456,-8.3303,-22.0587,12.5967,-12.9652,21.6117,9.1226,-6.5354,-17.9535,1.9783,-11.1411,-4.6795,-7.0392,-22.5771,-2.1327,-21.4448,21.3752,-0.23988,21.9763,-10.2278,-22.6304,11.7758,-12.6966,23.2272,11.3518,6.7676,3.7769,4.9511,-4.3458,-1.2204,-7.6182,-23.7295,-20.0975,-2.883,4.9433,20.1094,24.0813,-6.6223,13.8578,-20.9611,24.0884,-14.0911,10.6308,-8.7168,9.6972,-17.4452,16.1639,-8.361,6.9115,-23.0097,-19.7169,-1.5038,6.04,21.6716,25.9442,-12.3357,5.7371,-24.0835,11.3043,-11.5471,5.8303,5.4382,17.769,0.89218,26.787,-6.058,7.4416,-18.4215,-24.9149,6.0409,-7.8298,25.3908,19.4999,-1.9424,-5.9686,-5.0314,-10.6241,-3.214,-4.8626,19.8781,-8.5163,30.1477,-25.144,10.3404,-17.1736,-23.2707,-19.8898,-3.8886,3.655,19.4402,22.6812,-7.5234,15.0876,-24.1083,22.4015,-17.1708,7.2378,-8.6086,7.0312,-10.8939,15.9094,-0.73011,10.3675,-22.5649,-18.8941,-3.5052,3.3918,18.9592,22.6076,-6.8289,17.4751,-23.3039,24.1186,-16.7939,6.7909,-11.6169,0.4126,-12.5553,10.0588,-1.0399,9.8895,-20.726,-23.5335,2.3825,-4.4356,23.4849,20.7951,-8.5279,10.0543,-20.2205,16.3412,-11.1225,6.4806,13.7414,-8.6223,19.6642,-26.1475,5.9795,-17.5901,-20.1445,-23.1792,2.096,-2.0684,24.0407,21.3014,1.9101,10.5963,3.3883,19.469,1.6397,9.4847,16.1536,-6.9598,25.3429,-20.9218,8.8003,-14.1506,-5.3584,-24.9066,6.4257,-11.9037,18.1644,-35.284,0.082277,-43.1522,38.1802,27.6947,-8.0034,-3.2037,15.0973,9.6766,29.1491,-10.3293,-12.2946,-1.909,-12.2945,9.0709,-0.72973,10.2832,-20.1045,7.7284,-22.3838,7.7289,-3.0736,23.7819,18.7727,-11.0858,-22.1235,8.9754,-22.1235,22.2865,-5.7476,14.6163,-9.3029,-8.0045,-0.40736,2.5011,9.7182,11.2697,23.9813,11.6652,12.4316,-11.5444,-11.5568,-23.0801,-1.2429,21.2297,-21.9853,22.2543,-20.5235,1.2536,-9.4407,-1.6965,-6.7067,8.9736,4.2023,12.0304,23.7382,11.921,12.0017,-11.519,-11.0199,-22.4888,-0.80122,-22.1344,21.6475,-21.0913,22.2763,0.74402,-8.008,-9.3519,4.204,0.58098,13.3468,11.062,24.6197,11.3242,14.1263,-12.2226,-9.9452,-23.1062,-1.2159,20.5938,-20.7177,22.0274,-18.9829,2.1467,-8.462,-13.5114,6.9464,-4.3079,17.1848,10.9657,24.616,9.9734,15.404,-14.2695,-8.3185,-23.3859,-1.6832,18.9116,-18.8734,20.6569,-16.2781,2.6914,-10.5407,-17.1884,7.0906,-6.7497,17.7164,10.6089,24.0454,8.6183,15.5627,-15.558,-8.2311,-24.0255,-3.1347,16.3318,-18.8803,18.9798,-15.9333,2.4258,-9.9434,-3.3854,-5.641,7.5545,5.8781,12.3646,23.3852,11.2948,12.1821,-11.6382,-10.6239,-22.1001,-0.52721,-22.2205,21.9194,-21.5892,22.1999,0.3273,-11.1163,-18.7464,8.0851,-7.9083,21.0567,12.3964,24.2797,7.9909,16.7176,-16.383,-7.0159,-23.7105,-4.1723,15.0941,-17.3159,18.0773,-12.5288,3.7798,-12.2828,-21.1631,7.4373,-8.2658,22.9729,15.1063,24.7566,7.063,18.0749,-16.5308,-6.272,-22.6528,-4.7868,12.4404,-15.4285,17.2014,-9.7046,6.0164,-9.4918,-4.8745,-3.2476,6.4686,8.4692,13.4253,23.2853,10.7921,13.1841,-11.5184,-9.4492,-21.4426,-0.50603,-22.1944,21.7404,-21.9194,22.1079,0.12592,-10.3334,0.59452,-21.6337,22.4583,-7.277,14.5505,34.1738,30.0039,7.2525,-7.0709,-15.6626,-22.3147,-10.7698,-16.3162,11.5398,-9.5793,16.6481,6.4952,-11.5801,-24.586,13.3545,-11.116,24.9655,12.9287,22.7909,5.2239,17.2711,-15.8636,-4.4936,-20.3315,-6.5657,6.4802,-11.6354,14.8547,-4.9399,8.737,-23.7599,-1.2336,-21.2393,20.7243,3.2524,23.5304,7.4359,23.5449,-14.7621,15.265,-23.2462,-8.4765,4.5016,-0.44004,3.2419,-4.7344,-0.18534,-3.4763,-23.7074,-19.2847,-5.7319,3.1651,18.1159,22.6648,-8.9971,16.6792,-25.7475,23.1809,-17.738,5.267,-10.5139,0.93746,-9.1824,12.4581,2.0517,12.1553,-10.87,-24.9442,14.2165,-12.8034,26.1569,13.3162,24.1802,6.0011,18.1534,-17.0289,-2.9882,-20.1248,-5.3333,6.1309,-9.9593,12.9245,-0.78751,10.6335,-20.031,-2.1768,-21.8632,11.7106,-7.7956,13.6874,12.4341,3.5137,14.766,-10.6227,14.9802,-13.1798,5.642,3.6128,-17.6754,-15.0756,-5.2007,-1.4714,13.8166,22.6132,23.9001,-21.619,2.2019,-23.7451,22.6075,-1.6494,20.9655,0.082298,-1.9552,-2.5519,-7.0445,-3.0813,-5.9635,-15.6452,-26.4471,10.0542,-11.2426,25.2201,15.6283,-20.3136,2.0105,-23.7109,23.8278,-3.8689,21.8406,0.54421,4.5335,-1.8787,4.1573,2.7937,4.8676,-15.4905,-24.6765,10.6258,-9.672,25.6677,14.121,-18.6821,3.7866,-22.5637,23.4096,-4.294,19.5565,-2.0403,-1.3274,2.7288,3.2847,6.5183,6.3236,-15.7625,-25.7079,8.4364,-10.8322,23.8492,14.2791,-21.5992,-23.7556,-1.5846,-2.2276,16.5471,16.9799,2.8903,4.9003,-7.1109,-0.62875,-15.5788,-11.2359,-15.2365,10.8354,-26.2262,26.9057,-9.9597,16.4382,-19.7282,-17.8435,-1.0165,0.80032,21.8774,20.5553,6.7278,14.4609,-4.4252,4.9285,-8.2317,-5.9392,-16.9971,9.9673,-27.7934,27.2539,-11.4223,16.6194,-18.8662,3.9318,-22.5405,22.5548,-4.7814,18.2395,-4.3629,-7.1226,-1.2149,-3.1806,-1.2558,-0.38751,-15.4703,-23.9936,7.8548,-8.3067,24.1289,15.7504,-17.9174,3.7906,-22.0132,22.2813,-4.254,16.9258,-3.6209,1.1309,-6.3849,1.1104,-2.5828,0.068493,-17.5713,-25.0465,4.7568,-8.1003,25.2107,18.8809,-22.844,-1.1797,-21.9763,23.0152,-0.92284,22.4093,-1.3951,-1.4054,-0.18209,-0.17686,0.84569,0.86637,-12.2298,-27.1164,14.3402,-14.2094,26.8362,13.1568,-22.1608,-20.746,-0.91399,-0.35438,23.2066,22.2388,3.2876,8.8624,-1.2113,5.709,0.0017527,1.2269,-14.363,11.9159,-27.9206,26.4063,-13.605,14.3392,-14.4016,5.3598,-18.0427,21.1078,-3.7389,16.2651,-10.6386,0.35045,-17.5494,4.446,-8.3736,3.1756,-19.7895,-24.6225,2.8659,-6.2841,25.1482,20.8377,-12.9177,3.5097,-18.8887,20.4727,-8.8916,11.0303,4.2248,15.4418,-15.7498,13.4987,-16.2774,-0.48393,-22.96,-16.7579,-5.4304,8.3668,17.1551,24.2597,-22.8489,-21.7087,-2.208,-0.58309,22.2056,22.3014,-1.4156,0.086701,-1.8173,1.1597,-0.48322,1.0092,-11.3875,14.9117,-26.3234,26.1814,-14.6371,12.0513,-25.0983,-19.8168,-6.3275,3.2286,20.1548,24.5572,4.0041,-6.8158,6.3233,-15.7948,2.4802,-8.8385,-10.4491,17.945,-29.5245,25.7747,-18.8311,7.9788,-23.0823,-21.0243,-2.9477,0.085288,21.0769,22.015,-1.923,0.51747,-3.5284,1.6728,-1.7368,1.2269,-9.4909,15.2042,-24.6689,25.3089,-15.8951,10.8512,-8.8283,-25.5069,17.9297,-15.6673,26.9008,9.3163,25.9118,5.5648,19.5697,-19.8743,-4.2948,-24.851,-5.6153,-8.2701,0.38975,-5.8673,-1.176,-4.8922,-8.8538,-27.643,19.196,-17.4565,27.5701,9.7341,27.1618,6.6312,19.0616,-20.5191,-9.7175,-25.3804,-1.0942,-1.7513,1.1961,-3.0832,2.1034,-1.0633,-6.822,-26.9747,21.4563,-19.7143,23.4852,1.5374,27.7148,7.7901,18.1956,-17.1906,-8.8106,-24.9037,-2.0358,-2.4829,3.8157,-1.588,12.7792,7.3056,-26.857,-13.8105,-11.0751,6.8001,17.1425,24.7282,5.5897,-18.3205,25.6152,-20.7095,19.9241,0.32968,-3.4595,-9.6328,2.797,-11.7321,4.2424,-2.4302,-1.0188,-24.4926,17.7813,-22.813,12.4073,-3.5799,26.5522,8.0739,17.6916,-13.8985,-2.6216,-19.2475,-8.23,-8.7606,6.6339,-1.5704,22.108,15.3424,-7.9389,0.4734,-7.1333,10.189,1.8921,10.8045,8.0382,25.0282,-16.5731,18.1165,-25.1904,-7.7846,-23.9382,-5.7426,-19.5168,17.0793,4.7188,23.1649,-7.9596,-0.81094,-4.9316,7.7718,5.9256,11.6154,9.027,24.9762,-15.8661,16.8764,-24.9458,-8.6786,-23.1852,-4.4893,-19.8783,18.2478,2.4854,22.2208,-10.9677,-10.6434,-0.73221,-0.14449,13.4996,13.3004,7.3432,-17.7354,26.42,-25.216,19.3565,-6.8087,-23.7894,-18.423,-7.4812,6.2943,15.5403,22.6304,-9.489,-5.7348,-2.1444,4.9712,9.3939,12.858,9.5389,24.8763,-15.1713,15.9255,-23.6318,-7.8927,-22.2183,-2.3079,-20.7353,19.7385,0.54504,21.2791,-11.6039,-1.5393,-8.0928,11.1517,5.1691,14.9659,3.1869,24.2787,-20.6122,19.8686,-25.0899,-5.3632,-23.1137,-7.9829,-15.5958,14.448,4.6037,20.4809,-14.3011,-11.8561,-6.1457,1.5056,9.3783,13.9496,5.0151,-19.6206,25.5296,-26.3223,20.3084,-6.6154,-22.1095,-16.2083,-6.6586,5.8279,16.8823,22.4384,-10.053,-7.8602,-2.3733,3.8876,8.0861,12.0522,9.1627,24.2065,-15.2422,15.6914,-23.9933,-7.9309,-21.7727,-1.2638,-20.5121,20.1963,1.1969,21.255,-8.5594,-8.5092,3.0147,2.958,14.083,14.4234,2.3412,-20.1838,22.7087,-24.2801,19.0242,-5.2291,-22.9777,-14.9026,-8.9363,6.3126,11.6815,19.6799,-8.1912,-7.4481,0.65327,2.9045,11.7224,13.0607,9.498,24.1222,-14.156,15.4926,-22.4676,-7.4332,-22.1305,-1.2875,-21.2658,20.1313,0.27463,20.4565,-6.6608,-8.6196,4.8036,2.1887,14.9791,14.4249,9.9732,23.5656,-12.0681,15.5754,-19.9313,-5.879,-22.2896,-0.89443,-21.9362,19.8947,-0.87656,19.7402,-14.0407,11.6716,-22.1568,24.5663,-7.4994,14.5343,22.6155,15.6288,7.1599,-4.3925,-13.3092,-18.4419,-4.9153,-14.1176,10.8544,-5.9987,18.1026,9.9244,-14.0618,-25.2738,15.5482,-13.2974,25.0025,10.6305,19.5061,6.1611,11.2403,-16.6736,-7.7922,-23.2459,-9.3831,3.0211,-12.6949,13.8645,-3.0424,10.7091,-2.7835,-23.8589,20.6155,-21.3823,24.073,4.8631,23.0894,8.3682,14.9865,-13.2684,-7.8169,-21.9088,0.21753,4.229,-3.0126,6.5043,-1.28,3.9774,-23.8806,-2.7703,-17.0045,19.2539,6.6445,24.1764,8.325,21.2766,-13.3791,14.0543,-21.2596,-7.8566,4.0076,-3.9623,10.0291,-5.9744,6.7152,-1.692,-21.6332,-22.6989,1.7889,0.081655,22.9788,21.4982,13.2723,-14.4629,26.13,-26.7771,12.3391,-13.0646,0.90012,-5.6431,7.6702,-5.4303,8.1398,1.3905,-22.6092,1.1104,-21.4179,22.4969,2.7583,22.4021,10.6969,26.1843,-15.7791,14.6276,-25.693,-10.2264,2.6786,5.2445,3.0078,5.4297,2.055,2.0057,-23.6927,-1.2508,-20.8993,20.6564,4.0191,23.3855,8.5969,25.1287,-15.2447,16.1545,-24.0019,-8.6646,2.541,-1.6008,3.6501,-2.375,2.7355,0.74067,-20.9777,-23.1834,3.1136,-1.7193,21.2996,20.1178,15.189,-12.7068,27.3588,-26.6822,12.8019,-15.0118,-1.3243,-6.7997,4.4907,-6.3079,7.3107,2.0626,-19.4653,-22.9921,8.5516,-1.7067,42.631,16.6897,-11.5919,30.5114,-28.7,-2.0769,-3.8749,-7.1169,4.6678,-3.5954,11.14,-18.4544,-26.4366,5.6409,-24.7774,24.858,16.1184,17.6025,-13.1093,25.6863,-14.8715,4.8537,-17.3807,-2.0391,-8.947,9.3816,-8.1147,14.749,7.3805,-26.1769,-11.5316,-13.3865,15.2358,13.4629,26.5156,5.8734,24.3944,-17.3036,16.7588,-18.4468,-2.8149,3.8075,-9.1228,13.0673,-13.1142,15.0781,1.8336,-13.0495,1.8027,-10.6312,15.6536,-5.081,13.0462,-6.2682,10.2862,-19.0729,18.4698,-15.5107,7.6135,-22.5,-24.6309,11.3944,-3.84,22.7278,24.2924,6.3001,-11.419,17.2524,-16.7189,8.7549,-6.4598,18.2837,2.163,17.7121,-18.0475,-0.51199,-19.0537,-17.0292,-22.8384,7.7346,-7.4881,25.7648,17.0458,-24.704,-21.6343,-4.1334,-1.2248,20.9438,20.2945,-11.1232,11.5211,-26.049,19.6133,-13.1558,8.7498,-5.0228,0.97001,-14.1815,-6.4538,-10.5796,-8.5681,-21.3638,-0.82161,-20.9075,22.1409,0.20604,21.9407,-8.9151,-20.6437,11.2003,-11.7333,22.0375,10.7966,9.0253,6.0692,6.1415,-3.283,-0.51374,-6.9118,-21.4317,-18.1144,-1.0688,5.2489,21.5189,25.0006,-11.68,4.7938,-22.5521,9.6365,-10.9882,4.7202,7.2784,17.7024,1.4767,24.5446,-5.3549,7.1451,-23.7645,-19.9479,-3.178,0.93044,23.4233,24.0882,-8.3105,15.4183,-25.4763,26.9437,-16.4089,11.4998,-7.5485,7.27,-12.5639,8.7087,-2.269,3.9954,-19.0633,-24.7488,4.973,-6.2732,25.4086,19.844,-3.1094,-4.7739,-6.8614,-9.3724,-4.0046,-4.5959,19.1841,-7.371,28.8406,-23.6465,9.8519,-16.1287,-22.832,-19.5032,-4.0923,-0.053462,18.4013,21.0857,-6.9477,18.2384,-23.8793,24.8612,-17.379,8.5543,-9.3287,2.1965,-9.4422,9.7459,1.6385,10.5454,-22.1243,-18.5154,-4.0847,1.7614,17.9611,20.9178,-6.387,17.0883,-22.9923,22.9423,-18.1301,4.3879,-11.4181,-0.44728,-11.659,9.5692,1.146,11.8702,-21.4165,-24.0236,2.6373,-3.8403,24.7029,21.618,-8.3465,9.3141,-18.3469,16.0608,-9.3619,6.8383,14.9423,-8.1846,22.1209,-24.8948,6.6873,-16.6553,-21.093,-23.9387,0.81944,-2.9486,23.555,21.6295,1.8887,8.8623,3.4823,17.1701,1.7453,9.3115,15.8686,-7.9703,23.8764,-22.8724,6.7078,-15.2955,-22.7123,-11.3144,-5.8699,-7.9031,14.957,11.292,2.0322,-34.0994,-35.4236,19.4111,-6.2052,13.8692,3.4815,-8.1874,-7.0257,11.1656,22.8921,20.2547,-12.0856,-10.269,-0.98853,1.9752,12.8474,14.8695,-19.6351,5.0831,-23.3437,22.2281,-4.4165,18.7537,-13.3501,-22.273,8.7646,-9.9016,20.4605,10.8801,-11.4747,-8.2317,-2.5572,4.7252,9.9593,13.8096,23.0634,11.5089,11.353,-10.5533,-11.6227,-21.693,-1.1629,21.8597,-22.4904,22.6893,-21.0717,1.038,-10.2992,-8.7125,0.47191,3.3758,12.5647,13.8452,23.7794,11.727,12.4729,-10.6807,-10.5537,-21.5241,-1.5039,21.2368,-22.1084,22.6048,-19.6641,2.1995,-11.7569,-3.7987,-6.6547,9.2957,6.8086,14.6247,22.6581,10.9799,11.6987,-11.1016,-9.9981,-20.9288,-0.68707,-22.6038,22.3309,-21.7626,22.6655,0.44104,-9.0389,-11.5339,5.9556,0.32574,17.0762,13.7122,24.2148,10.6334,15.1168,-11.9855,-8.2211,-21.4611,-1.674,19.9617,-19.5571,22.3573,-16.7591,3.6976,-10.4351,-16.9213,6.5755,-5.7738,17.451,11.4153,23.8765,8.2691,15.5873,-15.1162,-8.2877,-23.3235,-2.7017,16.8657,-18.9094,19.6721,-16.2742,2.9128,-9.808,-17.1844,9.2703,-6.3642,20.9266,12.2713,24.6141,8.4382,17.4066,-15.3075,-6.2676,-22.8987,-2.144,16.4835,-16.1849,19.1068,-13.1843,3.8093,-12.3467,-5.7099,-5.1657,7.7155,8.507,14.7108,22.1423,10.4014,12.0375,-11.0743,-9.3232,-20.7099,-0.2805,-22.6582,22.2532,-22.3261,22.3566,-0.020042,-12.7908,-21.6668,8.5657,-9.9505,22.3428,12.4463,23.5467,6.3039,17.4837,-17.4484,-5.6695,-23.5276,-4.6726,12.4051,-15.6595,15.4387,-11.5233,2.7025,-15.6959,7.6063,-23.6721,23.2588,-9.8339,13.9371,21.9346,15.1548,5.1691,-6.1259,-17.6987,-22.0955,-6.9435,-17.3496,9.6431,-9.4064,14.0078,5.3446,-13.2728,-8.2596,-2.2974,5.3916,11.4638,14.1902,21.5172,9.4051,13.3201,-11.9091,-7.544,-20.7188,-0.17163,-22.3791,22.0956,-22.2765,21.6555,-0.45568,-13.4503,11.9693,-24.2639,25.7465,-9.6608,14.2961,20.2562,15.5602,5.1252,-1.8715,-14.7471,-17.1692,-8.856,-15.2836,8.43,-3.578,17.9026,12.5644,-12.1621,-24.898,13.2682,-11.0996,25.2801,13.7966,21.5226,4.7713,18.2542,-13.3786,1.6518,-13.7303,-7.9845,6.5465,-10.6706,17.8789,1.3571,15.7061,-24.4851,-7.366,-13.713,16.8293,9.4666,24.4609,7.7386,20.5608,-14.2966,12.4069,-21.9182,-8.6778,4.0288,-8.3646,14.8645,-11.9255,11.5374,-2.9996,-23.8679,-2.6311,-20.3308,19.7483,4.8316,23.615,8.2597,23.7212,-14.9623,15.9109,-22.83,-7.3804,3.1979,-0.17383,5.078,-0.30711,5.1118,3.0246,-23.7568,-18.5801,-9.1535,2.5052,13.6406,18.8185,-7.9482,17.7253,-23.4419,22.3706,-18.9953,2.3657,-11.4333,-0.49633,-7.8882,11.7698,8.4618,16.4483,-12.7704,-26.5844,13.3364,-12.3146,25.5068,14.201,20.2917,1.8707,18.7756,-17.9961,2.4798,-15.6048,-12.2798,0.037101,-10.9172,14.1,3.6638,16.2888,-12.6771,-22.0152,10.8963,-11.5566,18.7277,8.9107,-0.31889,-14.0388,17.756,-13.746,20.3824,2.918,-22.1834,-6.5124,-11.2183,18.039,4.5397,24.1496,-3.0563,-23.3304,17.0874,-7.8975,12.636,13.4309,-21.3689,23.1751,-16.6666,8.2029,-34.0076,-12.6818,1.7184,17.4206,23.1737,-20.3799,-22.0086,-0.1544,-3.2591,17.4783,15.4095,7.6733,8.4574,-4.0497,-2.7692,-15.2554,-14.5089,-15.663,9.6759,-26.795,25.4925,-10.0081,14.3449,-22.1784,1.0431,-24.3271,18.096,-5.0172,14.2398,-1.076,-2.4939,-0.60095,-12.0644,-5.8095,-15.7188,-15.5481,-25.3042,10.6154,-11.1186,24.1728,11.5312,-19.5407,2.4865,-23.387,19.5264,-6.1641,13.9073,-6.0785,-4.2666,-3.0155,-9.8114,-7.6346,-16.4167,-14.8355,-24.7364,10.1796,-12.6658,22.9429,11.0309,-18.2608,3.8305,-22.5209,20.467,-7.6444,13.9328,-3.3377,-3.3081,-0.5523,-8.9042,-7.2337,-17.0061,-15.5425,-24.4587,8.2064,-12.1091,20.7552,11.7872,-17.3474,-16.8255,2.0937,0.45739,22.9847,20.2299,10.6503,15.2856,0.90091,4.4533,-3.3617,-4.0643,-17.2942,10.3568,-27.6061,26.7181,-10.9614,16.7605,-21.523,-22.649,0.94253,-2.4268,23.8281,21.6765,2.5227,5.0382,-1.5368,1.4293,-0.56401,-0.21619,-14.9839,10.9868,-27.0096,27.027,-12.24,15.5668,-17.4998,4.2814,-21.9899,22.6552,-7.7421,14.8695,0.14541,0.93981,-0.77556,-8.8939,-6.5096,-15.5493,-16.0286,-23.9568,6.5985,-8.5317,22.1792,16.3369,-16.8538,5.0641,-20.8539,24.1438,-2.9828,18.7797,-1.8632,0.70707,-3.5443,-5.7159,-0.76442,-4.8722,-18.9266,-25.4794,4.073,-4.5469,26.8275,20.2469,-22.4321,-0.099149,-22.9952,23.3693,-1.7985,22.085,-0.98283,-1.1274,-0.041517,-0.0095075,0.89321,1.056,-11.972,-25.7617,13.4828,-13.0815,25.9253,12.9104,-22.2838,-23.1813,0.58587,-2.039,23.8361,21.9685,-0.84744,0.30988,-0.82486,1.203,0.21289,1.0684,-13.2312,12.6703,-26.6395,26.3151,-12.9917,13.9631,-11.9147,4.5887,-14.3798,19.3339,-4.9849,16.1136,14.648,-10.0962,0.3687,-17.351,2.9721,-11.0902,3.3129,3.4899,-21.3081,-24.9107,-1.0656,-6.0748,17.6244,21.3867,22.2811,-8.3608,10.9849,-16.2308,20.809,-6.8045,10.5503,-1.4347,14.5228,-17.9311,14.9423,-19.0183,-2.3,-24.8897,-17.7473,-8.425,5.6124,16.7168,24.1543,-22.2909,-22.2022,-0.89315,-1.2394,22.371,21.4123,-1.1896,0.11817,-1.6225,1.1091,-0.35996,1.0598,-10.6718,13.7104,-24.7791,25.037,-13.3103,12.3596,-25.1277,-20.7169,-5.8871,3.3304,20.952,25.3033,4.3812,-7.2223,7.2065,-15.9515,2.7176,-8.8795,-10.0095,16.9159,-27.8463,24.6891,-17.4966,7.8566,-21.8963,-21.1218,-1.765,-0.89678,0.21131,18.9605,19.5009,-1.7454,0.39555,-3.2854,1.4991,-3.3156,1.1867,0.065946,-7.9813,13.9278,-23.1831,23.898,-22.5975,12.0828,-3.254,-9.4444,-26.4588,-16.8251,21.5942,9.7305,30.7932,24.8139,4.9269,-19.7408,19.1732,-25.1545,-7.802,4.2592,0.64974,-8.7354,7.1288,-0.42658,10.3204,-7.8558,-31.3283,18.4188,-22.9746,18.4999,0.05715,26.9081,6.2072,17.3795,-16.8628,-4.3163,-22.526,-0.51545,-6.0655,10.2339,-5.169,18.3465,8.1665,-0.67216,-19.9562,18.0794,-19.9963,14.6657,-2.5376,22.709,7.5242,18.0429,-17.0424,-1.8478,-20.2867,-11.2088,-17.5395,7.1769,-8.1701,22.7137,13.7446,4.3116,-14.4995,14.356,-17.7013,10.9631,-0.61716,20.6691,1.2046,19.6561,-17.2724,-3.4081,-19.3566,-14.62,-22.5194,9.2509,-8.0526,26.8668,18.6758,-14.2273,-0.014191,-21.0226,14.6457,-7.8186,15.5308,-17.5767,-23.8558,11.2853,-4.0674,24.6648,19.0506,-17.1445,7.6567,-19.8282,21.7545,-5.1514,12.5172,-10.0445,2.1048,-9.4675,13.648,2.4018,13.1167,7.0995,24.242,-17.0102,18.103,-24.8869,-7.3591,-23.3736,-8.0377,-17.7109,14.3557,5.0808,21.7219,-9.6639,-1.337,-6.147,10.2129,8.1058,15.9607,8.314,24.7948,-16.0596,17.1904,-24.2486,-7.8044,-22.9666,-5.454,-19.1142,16.4869,1.7437,19.5878,-10.8967,-5.4684,-2.9763,8.5686,11.4179,17.5913,9.0202,24.7382,-15.3101,16.6931,-22.3091,-6.2036,-22.2239,-3.1066,-20.5994,17.3835,-0.54647,18.102,-15.0052,-15.4052,-3.2847,-3.1219,12.3229,12.8125,6.9658,-17.4333,25.3925,-23.9105,18.7884,-5.5426,-21.3682,-14.5927,-5.8883,8.5724,15.3279,22.6622,-13.2815,-9.8817,-4.9234,5.2924,8.2618,14.8557,7.9404,23.2028,-16.335,16.0526,-24.0189,-6.6379,-20.7414,-0.67964,-19.64,19.992,1.206,20.5691,-14.1096,-10.3555,-2.0265,6.1496,13.6466,18.4654,4.4115,-19.4721,24.4611,-24.5321,19.0158,-5.6493,-20.7606,-15.1525,-6.7032,2.8387,12.5351,16.7629,-14.5474,-10.6649,-4.6105,4.3789,13.4546,18.4629,4.7815,-18.7771,24.8353,-25.2455,19.4851,-6.538,-21.4859,-16.5501,-5.3284,4.4016,13.3526,18.0748,-10.9906,-7.1288,-0.51653,6.0118,11.6516,14.3633,8.5007,24.2659,-14.9047,16.6964,-22.6257,-6.6887,-21.3282,-1.615,-21.0316,18.9459,-0.58148,19.9595,-13.0945,-12.3968,0.80878,2.8774,14.8572,16.0301,4.3196,-19.2319,23.3507,-24.5703,18.5806,-6.1937,-20.4461,-12.9767,-8.6825,6.0596,11.101,18.201,-13.911,-12.5442,-1.0855,-0.13916,12.1268,11.9285,6.8922,22.3379,-15.5345,14.4247,-21.903,-7.7236,-20.1693,0.5751,-20.3447,21.4966,-0.468,20.7405,-12.6818,-13.3394,0.98022,0.3157,12.2049,12.0393,6.6111,21.4135,-14.2424,14.713,-21.599,-7.0044,-21.219,0.60453,-21.6359,21.3637,-0.4864,20.9945,-15.7593,9.7295,-25.036,23.5065,-11.031,13.2797,19.4351,13.3168,3.4951,-5.6941,-17.9989,-21.1313,-8.6962,-18.3544,6.6261,-9.5152,12.6379,5.8214,-11.2895,-25.8663,15.378,-14.2956,26.4622,14.2493,22.0888,6.4871,13.4286,-13.8474,-5.5651,-18.049,-11.8221,2.4052,-15.512,17.5582,0.11137,18.5708,-3.7422,-24.117,19.3573,-19.0203,24.2611,6.3977,22.7792,7.8545,15.6442,-14.8647,-7.5568,-22.8119,-3.798,1.997,-3.3699,6.3999,1.6648,5.7148,-23.4738,-5.1561,-15.4713,18.3171,7.2229,24.7396,8.8527,22.7213,-13.8203,15.4122,-20.8808,-6.3018,5.2103,-4.7079,14.3649,-5.2354,12.9231,2.8194,-22.5785,-22.8083,1.3584,5.7245,26.0563,21.786,12.1495,-18.1061,27.3759,-31.2444,12.1659,-12.2773,0.00061191,-3.0846,8.4014,0.814,10.4597,4.0888,-21.2595,-22.8566,0.61001,-4.1714,22.2583,20.9346,14.3432,-14.3113,26.0128,-25.7802,10.3114,-13.9507,-1.8319,-9.7919,6.7578,-8.3223,7.9866,0.97417,-23.4148,0.40617,-20.5872,17.434,4.8708,22.9504,9.5385,28.3325,-16.0422,20.5122,-25.5577,-9.1956,0.32002,1.0789,3.7885,3.1631,4.184,3.5191,-23.7674,-2.7983,-19.6976,18.8183,4.4174,23.3746,8.0909,24.7068,-16.441,16.6259,-24.2236,-7.6282,-2.4283,-4.5292,1.1371,-4.2001,2.4225,-0.90385,-19.5929,-17.8536,-20.7278,6.2196,-1.2456,18.7999,19.36,15.8981,16.9484,-11.001,15.5906,-25.0492,4.3214,-15.6205,-3.7153,-3.0254,-8.8204,4.9865,-7.8945,6.6302,0.11964,-15.7045,-22.338,5.4754,0.5493,20.5793,17.442,18.5882,1.7902,22.7196,-20.2913,4.0349,-17.6592,-6.8608,-12.3671,8.536,-6.5449,15.2651,4.5667,-26.7992,-12.22,-12.1769,12.5388,15.8602,26.3168,-0.32877,16.6623,-19.9289,13.3721,-17.6847,-1.7347,-4.1608,-19.3918,13.3047,-18.2229,16.7292,0.96364,-12.0064,1.3336,-16.0405,15.8742,3.5539,10.7007,-7.6644,9.7375,-21.1998,19.6259,-9.116,0.84151,-24.4462,-23.2268,-5.0393,6.0049,29.8764,32.4762,7.4759,-7.8962,16.8182,-11.7903,13.5173,0.59742,18.443,-0.60236,17.418,-19.9262,-5.6,-23.8426,-17.3398,-26.2967,7.833,-10.8207,23.6994,14.1726,-20.9162,3.0505,-21.0564,22.5507,0.19453,21.3585,-7.88,-20.1753,10.9748,-12.1363,21.3769,10.4495,9.7368,6.9911,6.7855,-2.7264,0.46039,-6.8986,-21.8742,-23.3451,-1.4935,-2.5566,23.1261,19.4288,-13.7879,11.4256,-27.5708,20.1986,-16.3056,7.8085,1.8708,1.577,-0.79128,-7.9645,-5.9249,-11.5438,-21.725,-18.7274,-0.88433,5.0859,21.3795,25.1186,-11.0493,4.6522,-21.7713,9.0035,-10.9094,4.7695,5.8686,16.5013,0.12632,23.3572,-6.2587,6.8993,-23.3699,-22.5947,-3.1448,-3.4337,25.7128,24.5156,-8.4523,16.0711,-24.1547,24.9581,-11.8724,6.784,-7.4769,-3.2041,-6.5208,-4.9438,-6.2527,-9.1014,-19.5215,-24.93,4.8803,-6.3313,25.7231,20.0919,-3.2019,-4.4076,-7.3618,-8.4091,-4.2691,-3.5822,17.9945,-7.2285,27.3871,-22.4777,10.1088,-15.1601,-20.1627,-17.941,-3.3555,-0.9134,14.452,10.6181,17.435,-5.5633,15.6004,-21.864,22.7527,-18.694,18.2273,-0.59274,-11.2077,-4.0246,-9.1611,-1.5291,3.9752,-0.21932,11.33,-20.4663,-17.5623,-5.5382,1.6857,10.2882,14.1444,-4.2405,17.4761,-22.7211,20.9647,-20.8482,0.57821,-12.9158,-2.7091,-8.4552,11.1564,8.962,17.9126,-21.7543,-24.9484,2.2387,-3.9639,25.7989,23.2661,-7.8082,9.1133,-15.3917,15.8962,-6.7933,6.3292,16.1851,-8.0338,23.6374,-24.0862,5.9318,-15.6377,-21.79,-23.8682,31.5132,-2.9268,31.3966,1.5402,6.0802,6.6081,13.4403,8.6167,16.4752,-8.3419,31.5104,-24.2706,-14.3273,-16.6209,1.4965,-15.588,-2.8146,15.5642,14.0645,-18.1608,21.3587,3.4754,-21.9216,19.1684,-0.43785,-11.8529,-10.1764,-20.6837,9.7773,9.8343,22.2886,-13.6561,-12.0549,-0.045915,0.53483,13.3859,15.5646,-18.9342,3.1679,-23.5663,19.1537,-6.4079,18.2597,-11.778,-20.6574,7.8599,-11.3572,18.2717,9.9389,-14.0209,-8.2021,-4.6524,6.3967,9.8845,15.1044,21.6549,11.1716,10.1733,-9.328,-11.802,-20.8378,-0.92664,22.9145,-23.2465,23.3523,-22.1519,0.44324,-13.3352,-9.1572,-2.3323,4.8079,11.9076,14.8121,22.3034,11.5197,10.8206,-9.716,-11.6141,-21.2066,-1.5331,22.2085,-23.0884,23.2597,-20.8331,1.4002,-12.8904,-11.5129,1.3257,2.0674,15.0069,14.4577,22.5392,10.7036,12.3916,-10.657,-10.1188,-21.3564,-2.6933,20.8889,-22.6192,23.0406,-19.0172,3.0352,-14.5241,-6.4226,-7.1912,8.7439,7.3558,14.9936,21.0724,9.7571,11.0516,-11.4778,-9.5761,-20.5375,-0.33947,-23.3555,22.9199,-22.901,22.902,-0.17488,-11.3024,-15.0378,5.8802,-1.8754,18.0645,13.8839,23.3373,9.0695,15.1267,-12.5471,-8.0071,-21.5759,-2.913,18.9216,-19.6764,22.3901,-16.2377,4.4921,-11.0869,-17.2955,7.486,-5.3429,19.7168,12.9151,23.5952,8.1968,15.8705,-14.5123,-6.9404,-22.0302,-2.8326,16.7636,-18.3417,20.134,-14.5416,4.5625,-10.9761,-19.7573,9.9189,-7.5947,22.978,13.7017,23.716,7.0282,17.9385,-15.6558,-4.6211,-21.3515,-2.2544,14.5324,-14.3682,18.3112,-10.4209,5.5927,-13.8986,-7.5196,-4.6421,7.1698,9.7256,14.9952,21.2323,9.6584,12.0642,-11.1849,-8.5276,-20.037,-0.080108,-22.8998,22.7198,-22.9177,22.2352,-0.64987,-12.5089,11.0989,-22.3652,25.0242,-9.4838,14.1927,23.3682,18.6004,5.5891,-2.6218,-16.8976,-20.2702,-2.4844,-11.9707,11.6657,-5.8579,15.4371,7.173,-12.4441,-9.5482,-0.8801,4.7645,12.247,14.8635,21.7864,8.6546,14.213,-11.8333,-7.0118,-19.6807,-0.019623,-22.2641,21.7593,-22.7484,20.9845,-1.2088,-14.6469,-26.2779,12.2594,-11.6279,25.5527,14.1078,15.2501,-0.44328,14.8544,-19.4498,0.0080168,-18.7772,-14.3009,1.8584,-16.6629,13.1544,-2.9936,10.7005,-13.8457,11.9818,-24.9189,25.6579,-10.7218,14.2294,20.4412,17.6125,3.7369,-0.17123,-17.9413,-19.2466,-8.3875,-14.4228,7.9986,-3.9233,14.8693,9.1559,-12.9655,-26.056,11.8286,-11.4331,25.2513,13.8993,16.3443,-1.1241,18.5417,-17.4403,3.4902,-15.8095,-15.4571,0.95675,-13.4554,16.0068,1.3491,14.9947,-24.8446,-8.9895,-12.9041,18.2658,10.6149,25.6366,8.4896,22.4981,-12.2725,13.9624,-18.4019,-5.7735,3.9746,-11.3653,20.7094,-13.173,19.991,0.8284,-24.1889,-3.9174,-19.1683,18.7947,6.1477,23.3947,7.1858,23.8213,-16.5677,16.4583,-25.0071,-9.1904,-1.7419,-4.1617,3.4138,-3.1944,2.4975,-1.4825,-19.116,-13.2804,-3.9844,3.3057,12.8409,14.3445,-3.7243,20.3553,-22.7291,22.3291,-20.083,0.85939,-16.9979,-7.9931,-9.9352,9.923,7.8574,18.884,-13.33,-26.0297,10.7206,-11.3023,25.4681,15.6122,17.4491,-2.9369,19.4782,-19.4548,3.601,-14.9525,-16.9685,-2.8334,-13.9007,13.4035,2.7191,15.8213,-9.2524,10.0639,-18.3211,21.6072,-5.2028,14.6749,-1.1963,14.4244,-16.4691,13.863,-21.0991,-6.6855,-23.6513,-17.9162,-7.2448,3.7323,15.3113,20.5517,-10.8301,3.4045,-18.9747,16.6416,-7.503,16.6414,9.2274,-3.7157,12.9969,-17.3424,21.1025,-17.3567,21.1028,-1.6912,-21.9603,-18.1344,-6.2334,5.3606,16.4874,5.3595,22.1485,-14.8369,4.7586,-19.5506,22.4789,-3.1417,19.249,14.8336,2.5956,10.352,-6.836,-2.1479,-7.3327,-14.3739,-26.0364,13.0224,-11.4494,25.5614,14.2387,-17.9357,-20.1681,2.9239,-2.447,22.9882,19.7059,10.8021,11.9133,2.5754,2.8653,0.074991,-0.68558,-16.71,11.6915,-27.0966,26.178,-10.4519,15.0356,-14.5092,5.3108,-20.2129,22.0052,-4.2402,18.0836,15.6208,5.3668,8.9011,-7.2894,-5.4936,-11.3709,-11.507,-24.5536,12.0381,-11.8046,24.5468,13.5088,-14.3029,6.3056,-19.6697,21.6615,-5.3386,17.0942,16.5388,7.5156,9.8847,-6.4071,-4.8506,-12.6172,-10.8338,-23.069,8.8069,-11.0403,20.8155,13.3653,-15.0108,5.7369,-21.0256,21.4269,-8.9819,15.7864,16.1275,7.1214,10.8928,-8.2208,-1.5132,-13.3239,-12.2235,-20.5941,6.5012,-7.48,19.6971,14.6084,-20.5579,-23.1538,2.0723,-3.4361,23.8716,20.9227,1.9323,1.489,0.56753,-0.46433,0.46205,-0.13476,-16.0656,10.336,-26.8221,26.7002,-11.1644,15.6302,-21.789,-23.7489,1.8141,-2.889,24.3104,21.5916,-0.63903,-0.35134,-0.74859,-0.44668,0.0095326,0.035031,-14.0751,11.2553,-26.22,26.2383,-11.9174,14.7891,-16.4017,6.1561,-22.4223,22.3954,-5.8273,14.7415,12.2255,4.6565,11.098,-11.9377,5.4856,-8.8137,-17.6323,-22.8381,6.5431,-5.0712,25.1979,20.0958,-16.3511,3.6426,-20.2344,22.6497,-8.7358,16.436,-2.2077,-6.2258,-2.2873,-12.464,-2.0437,-6.5609,-20.7526,-26.79,12.9954,-5.5346,27.9621,21.4124,-22.1299,-23.3487,0.92295,-2.2863,23.7164,21.785,-1.0996,-0.10691,-1.1501,0.71987,0.047141,0.89466,-11.9555,12.7208,-24.8574,25.0377,-11.703,13.0098,-22.0376,-23.7826,1.2637,-2.6218,24.3243,21.9707,-1.1061,-0.31977,-0.94908,0.43353,-0.097163,0.48838,-12.9336,12.2569,-25.6312,25.7099,-12.1446,13.6532,-10.8894,-15.0784,20.7046,-4.749,26.406,12.4452,-8.5357,-20.8629,17.5703,-20.907,20.1688,-1.1852,-23.6293,-5.4367,-15.3802,21.9294,-5.3588,26.0812,-10.0137,7.7708,-13.834,20.2101,-1.7667,15.1252,1.6003,18.7691,-18.8479,14.5634,-23.8564,-6.4778,-24.5498,-16.9094,-9.6692,5.2588,12.5566,20.3803,-21.4742,-22.3352,0.83466,-1.9077,22.9423,20.9343,-1.1391,0.090296,-1.489,1.1509,-0.066534,1.1899,-10.5877,12.4978,-23.8618,24.2873,-9.7753,13.0162,-25.8323,-21.3394,-5.9104,2.7212,22.2803,25.7437,4.6442,-7.0166,7.6266,-15.4709,2.9481,-8.7496,-9.8905,16.2296,-26.3968,23.6943,-15.3184,8.5742,-31.7245,-32.0994,-0.79526,-0.469,24.3966,27.7333,-3.2925,2.7175,2.3049,11.0017,3.1803,7.179,-8.9129,-10.1722,22.0881,-24.7762,17.7986,-9.7531,3.0356,-13.0266,17.4255,-14.5993,16.7201,-2.6025,8.5945,19.1202,1.1184,17.3681,-18.7159,17.6086,-19.2218,-6.3139,-14.6611,-23.5789,7.8655,-10.9314,8.578,7.2963,14.2451,0.82717,-16.8084,13.2523,-14.4695,14.1533,23.3343,1.9885,19.422,-41.1293,-8.59,-16.0461,-26.5412,10.3227,10.1409,26.8059,-10.5261,-10.8644,3.6124,3.9433,17.453,17.9578,5.8075,-17.3454,22.417,-24.7696,16.365,-7.3696,-22.928,-16.3212,-10.1253,4.3131,9.9157,17.4803,-12.6147,-2.2523,-7.3078,12.382,8.1438,17.456,7.6965,24.7045,-16.3975,17.4439,-24.1271,-7.3002,-21.5677,-5.4006,-18.4383,14.654,0.93393,17.9937,-16.6743,-8.6529,-7.1838,7.1608,10.9072,17.4644,7.5555,24.2019,-16.8252,16.2865,-22.7897,-7.1806,-19.1272,-1.1495,-18.2828,18.6876,-0.61066,18.2097,-16.7103,-10.4076,-6.8188,8.2264,8.8668,17.5,6.2565,23.0287,-17.203,16.9349,-23.6392,-5.9022,-19.4455,0.18078,-18.8512,18.2704,1.3069,18.9,-16.7018,-16.4249,-3.1916,-1.5052,12.8808,13.3893,6.4571,-16.8744,24.674,-23.3884,18.5354,-6.4067,-19.7295,-12.5368,-6.2358,9.7419,13.8183,21.5148,-14.2736,-8.0368,-5.5265,7.7521,8.6434,15.687,7.3225,23.8409,-17.0986,17.4672,-23.802,-5.3936,-20.1519,-1.7178,-19.1068,18.2874,0.83985,19.9633,-19.6043,-15.2713,-6.9935,0.66135,9.5861,12.8886,5.0471,-18.136,24.3306,-24.1471,18.3999,-5.8164,-16.1913,-11.839,-2.0174,7.3248,16.3154,21.2418,-16.524,-11.4719,-2.635,5.5204,14.6593,17.6313,5.5544,-18.6812,24.9626,-24.623,18.4908,-6.2646,-18.5947,-14.7455,-5.7702,3.5218,12.3243,17.5432,-11.5124,-8.8368,0.70901,5.3006,13.5562,15.4744,8.201,23.7873,-14.0846,17.0841,-21.0778,-5.7469,-20.9151,-0.85264,-21.6982,19.0853,-1.5919,18.9325,-14.8113,-13.3951,0.585,3.1882,16.1696,18.0888,4.9857,-18.7689,24.7395,-24.9945,17.7245,-7.4249,-18.9609,-12.4308,-6.9369,5.0652,9.9999,15.8658,-12.4425,-12.6375,-0.50961,0.68239,11.6093,12.8911,7.19,21.8082,-14.9847,14.7317,-21.7487,-6.2918,-20.8957,0.95769,-21.2608,21.5107,-0.67992,20.4243,-11.3248,-11.5297,1.7011,0.8554,14.0982,13.5516,7.386,21.9891,-13.4445,15.1433,-19.9831,-6.1314,-21.2729,0.2663,-22.0705,21.1706,-1.7348,20.0399,-13.9932,11.6392,-25.1712,24.6372,-10.5772,13.9826,20.6207,17.3944,2.4084,-2.3473,-18.3842,-20.1781,-6.78,-13.82,6.6895,-7.4547,14.1062,7.006,-13.5582,-26.7881,14.7797,-11.4332,26.4854,19.0101,17.8376,4.6769,10.8706,-12.2941,-5.2431,-15.7724,-20.8208,-0.34789,-21.3974,20.9636,0.9257,21.1852,-5.0928,-24.7148,17.9127,-17.8399,24.7736,7.3668,24.1246,7.3997,16.4726,-16.0111,-8.3264,-23.3033,-5.0786,-0.4794,-5.8288,7.8166,0.071144,8.8801,-25.4887,-6.8997,-16.6427,17.8325,1.4033,25.3836,6.9184,22.3129,-16.4401,15.2291,-19.9476,-4.4331,0.24992,-10.636,17.223,-9.4448,18.4097,3.2781,-23.1586,-18.0368,10.5664,5.6216,60.3061,10.7322,-16.8239,51.0291,-26.2703,23.3013,-3.9771,-3.675,5.3363,-0.36768,12.9522,-21.8849,-32.3524,1.4737,-38.8229,23.105,21.4962,13.6144,-38.7518,24.6334,-35.2324,10.04,-11.9307,-2.1374,-9.4251,6.4186,-9.9176,7.8883,3.7537,-21.0127,-22.4083,-0.68384,-1.5197,20.2334,18.9211,14.7377,-10.6827,24.0275,-24.759,6.1554,-16.1131,-0.092044,-7.3928,7.8793,-6.0151,9.7906,3.0485,-23.6891,-4.9136,-18.9081,18.484,2.1551,22.7375,8.9318,26.0195,-16.5471,16.67,-23.2891,-3.5725,0.016103,0.15584,3.4242,3.4978,3.5295,3.7104,-23.2121,-4.4665,-18.7509,18.1318,5.3367,20.7319,7.7653,24.1753,-17.1433,16.9742,-26.5043,-11.0556,0.77187,-2.5203,2.3679,-2.182,2.1114,0.99078,-19.3492,-18.6761,2.1742,-1.7684,17.4769,15.951,15.9699,-8.4071,20.0226,-23.1442,3.2065,-18.1096,-0.085282,-7.999,11.1916,-5.9332,16.4804,6.9961,-16.5949,-19.6161,0.44694,-1.5572,13.5481,13.1544,18.6942,-2.3634,21.4264,-21.9377,0.83012,-19.7643,-5.3299,-15.7583,12.2688,-9.7792,20.3566,8.3772,-26.5808,-14.8836,-10.7574,8.1467,17.4782,25.7847,-1.7935,14.0289,-18.8296,12.0465,-18.2572,-4.2224,-2.8286,-21.9272,15.0021,-21.3657,14.3894,-2.3831,-20.1955,4.3777,-13.0873,15.5241,8.4014,-11.9865,20.0214,-30.219,20.4506,-3.3326,-35.2371,-27.0396,24.517,4.7087,31.3168,4.1208,-35.3526,25.3653,-14.0323,16.9468,-1.0256,-21.0601,2.0278,-14.7237,0.48754,-34.3821,-6.9009,-6.3682,26.8967,20.5098,0.31672,-13.8497,13.5734,-13.1728,14.3713,0.53761,24.4479,4.4206,18.9489,-18.7588,-7.5493,-26.5459,-14.0302,-24.4953,10.0393,-10.7979,21.7991,10.1762,-19.7951,-18.7057,-2.0397,3.1031,20.4595,22.8718,0.30235,-2.3709,1.4433,-2.0446,1.3337,-0.094226,11.779,-18.834,23.0924,-22.5006,13.5093,-8.0432,-19.9941,-20.7411,4.1707,0.44422,22.095,20.6406,-8.0623,10.7861,-21.5689,21.1061,-14.1034,10.3986,10.6909,7.9131,6.9414,1.2876,-2.2907,-6.1261,-21.574,-18.7873,-0.61744,4.8981,20.6162,25.2569,-10.4675,3.6397,-21.2356,7.9811,-10.9238,4.7749,5.7983,15.5785,-1.0145,22.1333,-7.5017,6.5985,-19.6924,-23.5995,2.594,-2.5125,22.0762,19.6364,-8.6526,13.7604,-23.1839,22.3926,-18.0016,8.2354,10.8687,4.6025,7.4761,-7.0745,-0.58621,-9.4066,-24.7994,-27.1096,4.3341,-2.2103,19.6842,18.4458,-10.8804,16.3999,-20.9375,23.2036,-4.0144,9.0589,7.7818,1.6253,18.1754,-7.962,14.179,-4.2933,-19.8128,-24.99,3.786,-6.8207,25.4295,20.3297,-3.3575,-4.3989,-7.274,-7.4165,-4.4959,-2.9401,16.2881,-6.9733,25.562,-20.7735,10.6542,-13.4903,-16.1377,-4.8895,-10.6794,8.6563,6.2803,13.8495,-2.2707,-21.6931,21.1326,-21.1616,20.6566,0.68083,-15.1316,-8.2393,-8.0106,10.237,7.855,19.3312,-13.5939,-9.2549,-2.9559,4.823,9.8394,13.3052,-1.481,21.1149,-22.0271,21.6717,-21.4521,0.63442,-18.8849,-9.6732,-10.3543,9.266,8.8724,19.5215,-22.8721,-25.9873,0.70287,-3.2646,25.5949,25.0609,-6.6471,9.1418,-11.3343,15.7524,-5.0273,8.0142,18.3388,-8.0567,24.4995,-22.9978,12.9156,-12.4207,-13.9709,-12.962,1.9751,7.2424,16.4858,16.4537,-18.3316,0.96208,-25.5004,23.9517,-8.2077,17.2534,-11.3348,-17.2353,5.9095,-4.9631,16.8589,9.6615,-15.1405,-6.2341,-8.408,10.0294,6.8979,16.2914,20.8058,9.029,11.0649,-11.966,-8.7116,-19.8428,-0.45132,-23.8885,23.9994,-23.4224,23.8353,-0.36383,-14.6924,-8.8703,-4.2352,6.4008,11.789,16.8437,21.518,11.8088,9.5005,-8.2296,-11.3595,-19.7128,-1.1079,23.4453,-23.9652,23.8855,-22.4423,0.63937,-13.975,-10.5769,-0.80772,4.9798,14.4532,16.7299,22.2711,11.7061,10.8103,-8.9549,-10.7708,-19.7837,-2.2074,22.565,-23.5765,24.0419,-20.3353,2.2245,-13.1315,-13.65,3.2409,1.1917,17.8637,16.1136,22.6659,10.546,13.1949,-10.5747,-8.6146,-19.9247,-3.3729,20.5413,-22.0851,23.6884,-17.2426,4.6723,-14.5256,-6.5857,-6.0033,9.4563,9.6032,16.6009,20.761,9.4726,11.4744,-10.9187,-8.3036,-18.9439,0.015755,-23.8246,23.711,-23.9925,22.9999,-0.94265,-12.7215,-17.1249,6.126,-2.9472,20.1717,15.2204,22.5194,8.381,14.7106,-12.7913,-6.8497,-20.1244,-3.8963,17.5931,-20.0411,22.285,-14.5797,6.4539,-12.1221,-19.041,9.1737,-5.3645,23.2799,15.2243,22.5962,7.3196,16.7464,-13.7284,-3.248,-18.6107,-3.7696,15.4786,-16.7438,20.785,-10.0737,8.492,-13.4941,10.9785,-22.8754,24.6884,-8.4367,14.8709,21.5187,17.8617,4.6051,-1.6665,-15.948,-18.68,-5.0162,-13.8625,11.1512,-6.8912,17.9385,8.5592,-14.4901,-8.9919,-3.7278,6.5231,11.6753,16.1313,20.3608,8.5626,12.2402,-10.957,-7.5879,-18.7087,0.51314,-23.0611,23.0949,-23.7726,21.7973,-1.5784,-13.9481,-11.1357,-0.94009,3.7465,14.3473,16.3256,20.4563,7.4406,13.7332,-11.6657,-5.6303,-18.0772,1.086,-21.8861,22.2004,-23.1437,20.006,-2.2026,-14.03,-25.6219,11.5978,-11.2572,25.5486,14.44,19.621,0.27439,20.3371,-19.779,0.81588,-19.8838,-9.231,4.0406,-11.5933,12.6026,-2.6832,8.1916,-14.1196,-25.7056,11.4197,-11.4383,25.5622,14.8936,17.4323,-1.5599,19.1857,-21.6032,0.39721,-21.1628,-13.0131,0.89282,-13.9017,9.8663,-3.5416,6.1024,-13.9154,11.3926,-24.8709,25.2704,-11.0257,14.0529,20.2886,18.9723,1.9542,-0.39706,-19.0167,-20.1842,-7.4146,-13.1965,7.4374,-4.9201,13.6186,7.0604,-13.9489,-25.319,9.9578,-11.1295,26.3215,15.9333,15.9646,-3.3762,19.9231,-22.5522,1.049,-22.4339,-14.7526,-1.0929,-12.6559,8.7127,-4.0966,3.9963,-25.3946,-8.9939,-10.0783,24.6415,27.4709,7.5313,22.2136,-11.4417,16.3934,-13.3041,1.4064,-18.0514,30.7554,-15.0615,34.3557,-24.0604,-4.6438,-17.5909,17.6109,9.8161,23.3455,8.5345,26.3337,-15.2447,17.6612,-23.2203,-10.3693,3.3313,-0.8854,8.4993,-1.4204,11.3307,6.2691,-13.2879,-11.3886,-0.56919,2.1284,11.8252,12.6543,-1.3434,21.1677,-21.3441,21.8023,-20.7956,0.052135,-20.1262,-8.5298,-12.2706,11.4427,7.8712,20.1858,-14.7107,-25.57,9.355,-10.0484,23.3976,13.9989,15.2286,-4.9706,16.5815,-22.4831,-1.6187,-21.7616,-17.7212,-3.1653,-17.326,5.9317,-8.4659,0.92646,-14.1043,3.8914,-20.184,15.0237,-3.4267,13.8103,3.4938,21.3096,-14.7347,18.8472,-23.2853,-8.0177,-20.2197,-13.5878,-5.3961,8.9515,13.8964,21.3425,-15.0073,5.5091,-20.2357,21.2735,-5.5495,16.8762,6.5358,21.4202,-15.0598,13.5389,-21.3232,-6.8782,-20.5702,-15.1556,-5.0368,3.7349,14.865,18.6702,-3.5541,-13.1188,7.8066,-3.0689,18.8755,12.4737,9.1161,-19.6184,22.8041,-21.8353,17.5897,-5.5188,-29.8766,-8.5606,-26.4567,11.847,5.1864,18.8449,-20.0319,-23.4106,3.4126,-3.6512,24.032,20.2301,3.9707,2.9337,1.1437,-0.65428,0.87582,0.19751,-14.7286,11.0712,-26.4796,25.8484,-10.728,15.449,-18.8635,4.1616,-22.7735,24.0174,-3.2113,20.309,9.4373,4.3989,5.2465,0.83544,-0.79311,-0.11912,-13.7568,-25.3099,10.9096,-10.4589,25.7072,15.8402,-19.5522,-23.1186,2.91,-3.8054,24.0268,20.5003,0.38934,-0.45815,0.73308,-0.66949,0.71894,0.13139,-16.2483,11.444,-26.4555,26.1993,-10.676,15.3483,-17.4212,4.6899,-21.1684,23.7695,1.3534,20.4725,11.9007,6.1244,6.8483,0.25804,-1.0948,-0.88701,-13.5066,-24.4184,6.8124,-9.9851,24.6055,16.3924,-16.0883,4.0742,-20.5081,19.1772,-5.8779,20.5323,20.1354,13.0714,5.7689,9.0669,-0.16852,0.55255,-1.8958,-2.3526,-13.3355,-20.1777,3.9473,-8.2587,17.1815,16.6564,16.9625,-13.4135,4.7217,-18.9531,25.4796,-4.7932,20.2413,13.4509,4.2571,16.3402,-2.3303,12.1384,2.1163,-14.6258,-18.5557,4.782,-0.21453,20.4162,16.1421,-20.9826,-23.8328,2.6547,-3.5231,24.442,21.1321,-0.015372,-0.59832,0.45013,-0.60834,0.55894,0.085323,-15.0739,10.8254,-26.1287,26.0019,-11.2038,15.0462,-21.6886,-24.2464,2.33,-3.245,24.5739,21.6087,-0.44307,-0.48516,-0.38967,-0.38811,0.046867,0.09906,-13.7925,11.3396,-25.5679,25.7259,-11.7444,14.333,-16.8224,4.2115,-22.7952,24.7031,-6.8783,20.1828,6.9783,-7.5265,12.6482,-12.3407,13.2887,2.7994,-19.6633,-24.8778,6.2329,-3.3422,22.4257,18.2913,-16.6294,6.3592,-25.0472,26.748,-9.3256,19.4711,0.015922,-6.0873,-1.5792,-5.7996,-0.51887,1.6378,-21.5939,-27.6453,5.2907,-6.557,24.3751,18.9912,-22.038,-24.3008,1.9319,-2.8719,24.3482,21.6616,-0.62271,-0.09871,-0.46409,0.33467,0.54979,0.80238,-12.893,11.9147,-24.8696,25.2315,-11.4831,13.4986,-21.8087,-23.7169,2.1444,-2.6383,24.0319,21.2386,-1.198,-0.43058,-1.1566,0.4989,0.048759,0.92013,-12.5844,11.8598,-24.4037,24.4475,-10.7935,13.1914,-10.775,-16.2079,7.3072,-6.4519,21.8662,15.5578,0.45644,-19.4714,20.0865,-24.8249,16.6766,-6.3135,-26.638,-4.8602,-22.675,14.1816,1.6524,18.8254,-12.3737,8.1005,-19.7523,25.4364,-6.1583,17.9291,-5.4451,1.0917,-18.9014,2.1857,-22.8746,-7.052,-21.5379,-25.8006,-3.4062,-4.7637,16.8963,16.9931,-11.8067,5.1105,-11.5649,20.1451,2.064,16.9945,4.9075,21.5376,-18.1435,15.1214,-25.0333,-8.1514,-22.5959,-15.3436,-12.0075,5.0123,7.0016,17.2358,-20.9026,-23.0787,2.6096,-2.8011,23.7236,21.0443,-1.3767,-0.033325,-1.4828,1.092,-0.034542,1.2105,-13.3171,11.3875,-23.3324,23.9963,-9.3947,13.2824,-14.9522,-6.7556,-0.89248,6.8576,22.5971,20.6416,2.7102,-20.5398,11.8288,-20.8536,6.2978,-5.901,-17.956,-3.3194,-24.1072,12.7015,-7.6511,12.7949,-26.0663,-22.4791,-5.1736,0.53456,23.3891,24.3162,4.3259,-7.0028,7.0283,-13.5016,4.6468,-5.0387,-10.2436,14.4236,-24.6277,22.8891,-12.0117,11.5932,-25.1259,-25.5955,-2.4415,-1.3403,23.0842,24.4906,-5.9744,5.4874,-9.5154,13.9554,-4.0396,8.2219,11.8606,-10.4684,23.7294,-22.0827,14.3737,-10.0333,-27.4997,-25.2918,-2.5673,1.4366,20.8308,22.6032,-1.1357,2.3364,1.5848,8.4596,2.4932,5.705,4.702,-13.9763,22.9212,-23.6424,14.6136,-10.0955,-2.2506,-0.98046,-15.8369,14.6535,-3.0833,14.2731,23.5206,27.3489,-17.6378,18.2059,-26.8876,-8.1993,-18.7189,-16.008,-9.2821,10.3712,10.7373,21.4009,-16.3252,-11.2934,-2.464,7.1409,15.2107,19.9146,7.3257,-17.3549,24.7254,-24.3916,17.541,-6.9566,-18.967,-15.611,-5.6667,1.1502,10.99,14.6186,-15.9478,-15.4938,-0.31845,3.0997,15.8151,18.9302,7.0234,-17.3558,24.4894,-24.9165,17.7139,-7.3211,-19.6576,-11.8136,-7.6919,5.7318,10.4346,16.2109,-15.3046,-6.1627,-5.4065,12.4223,12.6811,21.1995,7.8407,24.655,-15.0872,17.6042,-21.0555,-5.3501,-19.6216,-2.3827,-19.9904,14.3711,-3.6968,13.2516,-19.1336,-12.0579,-9.3955,7.9157,9.1137,18.9297,5.8548,22.2679,-18,15.874,-23.5644,-6.2267,-17.6597,1.7806,-17.0404,19.4296,0.81759,17.4008,-16.619,-8.1045,-8.1719,11.0797,9.5546,19.7678,6.172,23.9022,-18.3142,18.2789,-23.2977,-4.0135,-19.5943,-1.8596,-17.7658,16.197,1.0702,16.9081,-13.5422,-13.0709,-0.85204,3.5745,14.3734,18.2979,5.6117,-18.0828,24.6094,-25.1103,18.5106,-7.1077,-21.1431,-13.8229,-6.7286,5.1855,12.6404,17.2777,-13.9441,-7.5152,-3.2335,10.0914,13.0847,19.565,6.718,23.7078,-16.4869,19.0285,-21.533,-2.3898,-19.9251,-2.1975,-20.2688,15.6047,-2.2706,15.9458,-15.1065,-12.0366,-2.7772,3.0002,14.6482,17.3412,6.223,-18.793,24.9589,-24.9156,18.5925,-5.8216,-19.8372,-14.5132,-5.5005,5.3522,13.354,18.8138,-17.8475,-14.8542,-1.8935,2.1704,16.5759,17.8142,6.5037,-18.6943,25.241,-24.8835,17.5645,-6.8329,-17.3895,-12.0427,-5.9301,6.1118,11.3108,18.0352,-14.2386,-12.2409,-1.6888,4.3719,13.4617,17.3925,6.3461,22.2339,-15.7019,17.2623,-20.6773,-3.6205,-19.9187,1.3362,-20.961,19.3384,-1.8846,16.9221,-18.1447,-16.1953,-3.1845,1.5683,14.2451,16.935,7.7759,-17.816,25.929,-25.1949,17.3853,-8.281,-15.9094,-9.9356,-4.158,6.6005,13.0797,17.9639,-12.4647,-11.2023,0.18896,4.2252,15.262,18.0625,6.6055,21.9937,-14.1566,17.4713,-18.9022,-2.8332,-20.4907,0.53671,-21.6856,18.8538,-2.8381,16.597,-13.1582,-13.7405,2.06,2.1078,17.3148,17.7244,6.2499,20.288,-12.6731,15.8078,-17.2392,-2.8363,-20.4656,1.6458,-22.698,20.6173,-3.3567,17.5109,-14.1111,10.8883,-24.3461,24.2955,-10.8528,13.32,20.0507,18.224,2.3174,-2.1084,-18.0818,-20.9798,-7.2149,-14.4433,9.0687,-8.4767,14.9859,4.612,-16.4071,-27.1569,-1.5648,-8.621,26.5334,16.0114,16.6848,3.726,12.8203,-13.1775,0.090279,-12.0691,-31.6811,0.18042,-31.5428,20.2092,3.9884,21.7617,-7.1827,-25.1993,17.1892,-18.0821,25.4596,9.3247,23.5711,8.0742,16.2006,-11.3239,-2.1822,-15.3743,-8.5705,0.5647,-6.6722,11.8144,6.5013,16.0976,-26.0448,-25.9061,-7.3015,-9.7618,16.7578,10.6264,25.0972,3.5542,0.4838,15.7571,-6.9011,12.2428,-7.8179,0.64899,-3.3482,-2.8129,-15.0925,17.5706,-11.2906,20.2711,5.7496,-24.1056,-21.4832,-18.3336,1.8668,6.66,22.2197,24.4204,9.6003,11.9311,-17.9876,20.9504,-21.9047,10.9058,-2.5079,-3.1301,-3.0346,-2.6824,2.4046,0.41959,0.05994,-1.1966,-21.5853,-23.363,1.1151,3.3081,22.029,19.841,14.719,-9.0197,24.3189,-22.6853,7.659,-16.2747,-1.8733,-6.1583,8.207,-6.8612,9.7814,-1.9412,-18.4649,-20.0362,1.6408,-2.2487,19.1986,18.4268,16.9888,-8.2058,22.7589,-22.3327,4.044,-16.6048,-3.8574,-11.2555,10.1101,-9.983,16.582,4.4848,-20.2869,-4.4039,-18.0191,16.5715,-15.5426,22.3605,8.3267,6.3149,21.5602,-18.1068,18.278,-19.4915,-1.194,-18.5652,-1.3475,-1.028,0.99447,1.0627,0.053695,3.4903,5.013,-22.9477,-2.7128,-17.6681,49.4179,19.2817,8.8432,29.1625,-16.9752,1.9711,-33.9381,-0.69103,-1.9852,6.2424,10.5649,14.797,-15.9198,-17.5919,0.26328,-1.7219,13.8692,14.3841,18.7312,-4.6493,21.2201,-21.6652,0.89306,-19.2739,-6.7013,-17.1062,12.4713,-11.1556,21.0557,8.4578,-12.9789,-13.4177,0.13107,-0.2292,12.1027,12.2243,20.514,-1.4558,21.1191,-21.1661,0.23003,-20.5876,-8.0861,-20.8485,13.0384,-12.5226,20.9105,7.895,-26.8332,-17.1637,-9.3764,11.1245,17.1388,26.5041,1.6757,17.3999,-21.2609,16.0383,-20.3504,1.4275,2.9366,-17.826,13.7127,-18.0049,10.4131,-1.0928,-2.8064,-15.1911,12.8662,-15.3882,13.7731,-1.5263,25.6928,8.34,17.6954,-17.1886,-7.9621,-25.0721,-9.7916,-20.5474,12.5277,-10.0531,20.7569,9.5873,-23.0691,-20.3044,-2.7535,1.9822,20.8949,23.0863,0.14903,-1.4905,1.5254,-0.94204,1.9382,1.0971,9.8231,-14.8779,23.2347,-22.7659,13.8344,-7.966,-21.0605,-18.4362,-0.30028,4.3167,20.0346,23.6917,-10.1822,2.6416,-21.1864,6.5053,-11.2058,3.7233,6.2141,14.175,-1.3826,20.2461,-6.1956,7.3823,-19.5825,-20.3403,1.9476,0.85733,21.522,21.3578,-8.2893,10.7878,-22.0515,21.2885,-13.4205,9.8667,9.3652,7.5087,7.0974,2.044,0.025431,-2.872,-19.8659,-21.584,2.1812,-0.66996,20.5833,20.7676,-5.202,13.8161,-19.0028,22.9008,-9.7473,10.998,8.5482,2.3092,14.4896,-5.996,10.4663,-2.0074,-18.7917,-20.627,-2.0828,0.47858,19.1306,21.306,-7.8144,9.9439,-11.9645,20.5379,-3.7364,11.9694,9.9282,-10.1963,21.4988,-14.836,15.4807,-2.0818,-20.2558,-24.6465,0.42931,-5.072,22.1879,21.7345,-2.5323,-3.5652,-4.533,-6.3862,-1.6632,-2.9895,13.9735,-6.7558,23.0428,-19.7483,11.6562,-11.1622,-13.4793,-8.5116,-4.9975,5.4238,7.7728,13.2125,-0.51017,20.1296,-21.5635,20.8277,-20.9552,1.0498,-19.503,-10.954,-9.2352,10.1471,9.6013,20.439,-13.1379,-5.3379,2.0762,7.2452,25.196,22.1558,0.197,-20.8104,13.9825,-20.1782,10.4138,-2.2867,-20.5566,-8.3772,-21.9875,10.9054,-5.4961,13.2207,-11.9678,-8.3076,-1.7277,5.7031,10.9762,14.7022,-0.35019,20.9291,-21.3473,21.5965,-21.202,0.12715,-20.3151,-10.2863,-11.9109,10.3752,7.5744,19.5552,-13.3012,-13.3705,1.1151,3.6347,16.2233,17.6135,-17.7213,9.6342,25.3879,-26.0073,16.7736,-8.7716,-12.701,-20.3989,-11.3478,3.9798,10.3756,17.0941,-16.4302,-5.7137,-9.498,12.3169,6.7705,18.3631,20.1956,8.4378,11.5778,-10.763,-6.895,-17.9934,-0.46307,-24.2876,24.8829,-23.5803,24.534,-0.45223,-16.3542,-6.7934,-8.0762,11.2486,8.5596,18.1227,19.6302,8.265,10.976,-11.3566,-6.9095,-17.8152,0.15714,-24.519,25.8063,-24.8965,24.8594,-0.96574,-16.7688,-11.1301,-4.0691,6.2202,13.928,18.7582,20.3144,11.275,8.6322,-7.7476,-10.6051,-18.0936,-1.5996,23.6482,-25.1559,24.3222,-22.7623,0.96145,-15.6449,-12.6607,0.68108,4.99,17.3148,18.8574,21.3454,11.1515,11.0972,-8.1517,-8.5679,-17.6719,-3.3531,22.9314,-24.179,25.2788,-19.2897,3.6427,-14.9165,-16.7947,4.7635,0.27326,20.3481,17.2255,21.4059,8.8275,13.4243,-11.408,-6.6864,-18.5692,-5.0424,19.5066,-22.2368,24.5194,-15.5423,6.9215,-16.1262,-8.5247,-5.5583,9.5675,12.1873,18.3084,19.2606,7.968,10.9879,-10.9578,-7.4475,-17.06,0.77269,-24.8637,24.8115,-25.8946,23.0055,-2.0583,-14.8073,-19.9839,7.8924,-4.306,22.3843,15.2232,20.5781,6.7001,15.1986,-13.5225,-4.3383,-19.3403,-6.2053,15.7829,-19.7547,22.5189,-12.3486,8.2835,-14.8432,9.4753,-23.1167,22.6657,-8.4646,12.8631,19.196,16.3625,3.4121,-3.6426,-16.4312,-21.2201,-7.8648,-17.6279,11.2628,-11.8777,18.7172,5.956,-15.4217,-10.573,-2.0013,6.186,15.2166,18.8382,19.2536,7.7208,12.9209,-9.4529,-4.5641,-15.7298,1.3464,-23.2331,22.9562,-24.9464,20.3904,-2.8069,-16.4073,-14.3243,-0.51389,2.9915,16.7906,18.5384,18.3033,5.1563,13.9105,-12.2405,-3.2231,-16.4084,2.692,-20.7535,22.3476,-23.3626,18.971,-3.1446,-14.0132,-25.3571,10.9221,-10.9127,25.0334,14.3026,20.1761,0.16506,20.0782,-20.2162,-0.14748,-20.1894,-7.6826,4.4254,-11.9553,12.1391,-4.9694,6.9393,-14.1102,-25.3621,10.5216,-11.0664,25.0026,14.3886,20.7873,-0.19439,21.2104,-20.7156,-0.5421,-21.4976,-7.1831,3.7637,-10.6761,10.8012,-6.3711,4.1651,-14.4327,-25.4151,9.777,-11.121,24.8923,13.9529,22.155,-0.047562,22.3018,-21.5303,-0.21772,-22.368,-5.1711,4.5737,-9.259,8.8078,-7.3751,1.299,-13.2697,11.3042,-24.4406,25.0949,-9.7339,14.5044,21.1251,20.0191,1.9219,0.54669,-18.79,-18.73,-5.1419,-11.4858,8.1452,-4.1381,15.3035,9.3661,-16.2731,-25.7103,10.8375,-10.9873,25.3453,14.846,21.4341,0.13204,22.28,-21.7329,0.31298,-21.6422,-1.8351,5.2628,-7.3049,7.2059,-5.3809,1.4294,-27.0053,-5.6774,-17.2668,16.3796,9.6207,27.6678,7.0198,27.3445,-15.1097,17.4499,-17.8935,-1.0464,-4.6585,-12.5115,9.3691,-6.8309,15.6191,5.6542,-11.9563,-10.9265,-0.029883,2.8197,10.9564,12.5167,-0.32355,21.2952,-21.1153,21.4466,-20.9424,2.0293,-20.9097,-8.2749,-13.0782,12.8579,6.1469,19.9752,-14.0212,-24.8364,9.2567,-10.514,26.4968,16.5046,22.8796,-0.5184,23.2955,-22.1962,1.8369,-21.4009,-5.2468,6.3785,-9.0229,8.2916,-1.9503,4.3276,-11.9548,-12.9339,6.4618,2.3161,21.0528,20.0523,2.988,-17.1554,21.2957,-24.6941,13.6304,-10.6442,-20.7985,-11.4486,-13.2362,2.8011,1.9984,9.3528,-17.8468,-0.8764,-22.3286,18.2846,-4.6506,17.2615,9.7465,25.1731,-12.0986,15.5258,-20.0849,-7.4772,-17.6312,-9.8938,-2.0917,9.2514,15.748,17.6612,-14.5956,5.4524,-17.6387,22.379,-0.77085,18.9113,6.2018,18.9769,-16.4752,11.3906,-24.5663,-9.3654,-19.8774,-16.5785,-8.0037,0.63622,8.4028,13.6183,-12.4236,-18.9355,6.0625,-2.0842,22.8926,20.1199,7.2352,-15.7769,21.5467,-24.8945,12.8187,-11.7911,-19.3681,-5.6087,-16.3361,10.4951,-1.6884,10.7338,-20.3662,3.7608,-23.9507,24.2973,-3.848,20.3208,-0.18116,0.6816,-1.124,0.87646,-1.1283,0.0098929,-15.3895,-25.8959,10.0335,-10.2132,25.7203,15.9514,-20.2267,-23.991,3.5344,-3.9209,24.3749,20.4386,-0.20633,-0.87911,0.5651,-0.68854,0.75571,0.16681,-15.4713,10.4469,-26.0793,25.7735,-10.4743,15.5339,-20.5582,3.6417,-23.799,24.2441,-3.9597,20.3058,0.53507,1.2431,-0.58019,1.2386,-0.78531,0.31363,-15.7221,-25.8737,10.2059,-9.7969,26.1278,16.4817,-20.5037,-24.0347,3.2628,-3.9066,24.4486,20.7339,-0.11061,-0.69649,0.55544,-0.51435,0.68263,0.18982,-15.3259,10.731,-25.8867,25.8024,-10.801,15.2136,-20.5306,3.3266,-18.7313,24.3919,-5.4552,20.3634,1.2633,1.4672,-0.74365,1.1929,-0.84793,0.38999,-16.3468,-26.5983,16.6105,-9.6956,27.4409,16.8874,-19.6017,19.1459,-26.6412,27.6593,-5.0827,20.8503,1.7778,-2.2736,8.6787,-1.7477,11.4217,4.1852,-16.967,-17.2931,5.258,-10.2717,22.1374,15.768,-19.9099,6.3999,-23.6893,26.1292,-3.8534,21.8918,0.20222,-10.7949,13.3808,-8.0073,18.1338,7.0673,-17.074,-21.4001,1.3488,-6.1305,16.7256,13.7996,-21.197,-24.3949,2.9038,-3.6559,24.6493,21.2201,-0.17083,-0.64076,0.30133,-0.52081,0.56732,0.21676,-14.5221,10.8405,-25.5699,25.6079,-11.1432,14.6185,-21.7019,-24.5125,2.6932,-3.3666,24.712,21.5302,-0.38153,-0.40039,0.072167,-0.17443,0.93276,0.69873,-13.4871,11.2369,-25.1536,25.2653,-11.4314,14.0086,-19.8696,3.5065,-25.743,25.4744,-6.6783,20.5886,-7.4891,-16.1802,-1.6819,-10.1244,-0.379,3.1481e-05,-15.8687,-19.6401,4.8059,-3.639,23.3855,17.2004,-16.1108,6.2718,-23.5664,24.8916,-6.3475,19.21,-1.6668,0.044071,-11.9816,1.2463,-18.6917,-6.9979,-19.6114,-23.7971,2.2718,-5.6545,18.6868,15.5967,-21.7099,-24.3548,2.8499,-3.1196,24.8363,21.5577,-1.0487,-0.87455,-0.60712,-0.056347,0.99002,1.3515,-12.9644,11.655,-24.5428,24.8885,-10.8944,13.4633,-21.2308,-24.0885,2.8272,-3.3088,24.5186,21.5344,-1.1408,-0.31301,-1.0288,1.0071,1.6737,2.8813,-13.2028,11.3514,-24.0118,24.2529,-10.362,13.2876,-15.7079,-21.6203,5.6373,-2.7302,23.6742,19.3228,6.198,-15.2332,19.0336,-25.6171,11.996,-11.898,-18.5487,-2.4893,-19.2649,12.1819,-2.0834,12.761,-14.9302,0.64427,-13.9413,18.0315,1.5961,18.2936,7.3379,24.6266,-17.1032,16.2864,-25.2629,-9.0962,-19.5293,-9.2322,-11.7992,7.7531,5.2746,14.8922,-20.97,-23.8715,2.3399,-3.0978,23.7687,20.9188,-1.4909,-0.18171,-1.1914,1.9828,3.2232,5.0088,-13.84,10.7294,-23.0972,23.8136,-8.9594,13.4658,-20.5143,-22.7178,3.1339,-4.2664,24.1116,19.2084,5.6667,-8.0815,14.8221,-11.3108,11.5831,-1.0464,-13.8477,9.1926,-20.5652,22.0832,-2.7179,17.1528,-23.8184,-22.9759,-0.48644,-1.1238,24.793,23.3803,-0.11382,-8.9168,3.4144,-13.6903,4.5354,-3.4344,-11.0311,10.4895,-23.1752,20.7365,-9.4463,11.9468,-24.5208,-23.3668,-1.3306,0.12359,21.2091,20.6102,-4.4199,5.4684,-7.5498,14.8839,2.246,13.9597,9.2687,-10.7033,24.0675,-19.3855,15.0217,-6.9897,-22.813,-21.2726,-2.2488,-0.11494,19.3344,20.7715,-1.5512,1.8448,0.53314,10.6323,6.1141,12.8964,8.1523,-13.972,22.5748,-22.5646,14.9037,-7.7323,-19.5631,-6.4308,-10.1218,14.441,10.5833,22.0372,6.7874,24.7361,-17.5224,18.0785,-23.3081,-6.2129,-15.5213,-1.627,-16.8629,11.2673,-2.7669,11.2894,-18.3104,-13.4837,-2.642,7.6237,16.6038,21.9498,7.3755,-17.7795,24.8078,-24.5248,17.5805,-7.0115,-16.7218,-12.9916,-6.3251,0.095868,7.4723,10.1402,-19.3896,-16.2882,-4.757,4.4046,13.283,19.1455,7.3964,-17.4867,25.2542,-25.2459,18.4642,-7.7877,-15.6201,-9.8859,-3.6556,4.8061,12.3627,14.8506,-20.5968,-11.0778,-9.3033,9.8098,10.2466,20.2633,5.1815,22.8725,-17.5016,16.9634,-23.3111,-6.6787,-15.7328,3.6112,-17.8063,17.9802,-1.3715,15.4332,-20.1934,-9.6265,-10.4368,10.9308,10.1706,21.2285,5.2854,23.7283,-19.729,17.7514,-23.4998,-5.6326,-17.0146,0.40546,-16.2238,17.3608,-0.015567,15.9668,-19.646,-9.331,-8.4779,12.9452,11.5042,18.4155,4.2693,23.7425,-20.5071,19.85,-22.5826,7.2873,-16.6749,-0.68434,-16.4752,16.2058,-1.3235,16.1603,-17.3735,-13.5409,-3.4232,5.1078,14.155,18.662,6.405,-18.1465,24.948,-25.174,19.3102,-6.2005,-17.9926,-13.6221,-4.9398,2.9759,12.279,15.8006,-17.4566,-11.0871,-3.8086,4.9102,14.2869,18.4504,4.2707,22.8969,-17.4341,21.2889,-20.1066,-3.7279,-17.0679,0.52933,-19.6537,12.4928,-4.1314,10.1485,-16.8186,-13.918,-0.94962,3.9028,18.5048,20.1601,6.2909,-19.1264,24.7456,-24.9008,17.3146,-3.5996,-19.4658,-13.8741,-7.1841,3.9958,11.8859,16.8433,-18.1442,-16.8646,-1.7886,1.1346,17.1261,18.746,7.6304,-17.9127,25.622,-24.7936,16.8821,-9.169,-17.9113,-10.8511,-5.8317,6.9095,10.9988,16.9549,-17.6869,-13.5839,-3.6225,4.2776,14.9421,19.2493,3.2061,21.0546,-17.192,18.6619,-19.3899,-2.463,-16.9692,2.7316,-19.3437,17.5862,-3.6054,13.2794,-17.8798,-15.1464,-4.0242,2.9392,14.8544,18.9671,8.622,-17.341,25.7155,-25.6963,16.7055,-9.1519,-17.0596,-12.1321,-4.0455,3.9516,12.6808,15.6457,-17.3697,-14.44,-1.3089,4.4103,16.9582,19.7722,3.0945,20.0252,-15.9309,17.9689,-18.1725,-1.6824,-17.7792,2.7467,-21.0172,18.403,-4.0176,14.5764,-16.8009,-16.4155,1.1599,2.2156,18.9105,19.1296,3.3288,18.7672,-14.4375,16.6392,-16.9681,-1.3933,-18.9909,3.5064,-22.4997,20.6014,-3.6288,16.5606,-11.8128,11.1206,-20.8382,24.6521,-6.8389,15.8897,21.7244,19.0579,5.4468,-0.053889,-13.385,-16.4666,-3.1082,-12.7573,13.6347,-6.2176,21.2622,11.1912,-16.7717,-27.1359,15.6844,-9.8219,25.7452,16.7841,6.8108,-3.9692,5.9713,-20.3492,-1.1938,-16.6651,-21.559,-3.7568,-17.0312,15.3729,2.2368,17.2569,-8.1928,-26.0825,16.6789,-15.9287,24.8291,6.8181,17.2919,3.1315,14.7534,-11.7464,1.3337,-11.2412,-15.8703,-4.807,-10.0417,15.0066,6.6504,20.1644,-23.2315,-16.3501,-4.4321,10.1184,11.6499,22.6983,3.6266,-19.8039,24.1656,-20.4965,23.1847,0.1609,-1.4823,2.7613,-3.0889,9.8476,0.73327,10.1493,-25.5278,-19.9883,1.9228,11.5629,22.4811,11.7796,-21.6577,23.3249,-23.9133,2.7838,6.8012,4.2855,12.3912,9.5067,15.0403,-19.7803,-21.7638,-0.2326,-6.6186,17.5947,14.8634,14.6737,-7.0503,20.1953,-21.8673,2.323,-19.4284,0.2016,-10.4499,13.4958,-8.8982,19.3426,7.8553,-17.4767,-18.1942,-1.2391,-1.0603,13.7155,14.9644,17.6213,-3.5905,20.2182,-20.2934,0.40447,-19.0189,-5.4264,-17.9806,13.2871,-12.7567,21.9299,8.9226,-33.8753,-35.6613,15.2465,-16.6999,23.3153,7.5617,17.5204,16.4883,19.4323,-16.1418,0.50937,-16.8145,-10.6186,-7.9661,-0.38139,8.6579,8.3,13.972,-13.64,-13.145,-0.25068,1.2509,12.0687,13.2015,20.273,-0.94428,20.929,-20.0594,-0.35698,-20.2134,-8.5332,-21.7448,13.1887,-13.397,21.4641,8.1884,-12.0728,-11.9736,0.20924,-0.048579,11.7326,11.0409,20.8853,-0.34144,21.0777,-20.9693,-1.1668,-21.2891,-7.8323,-21.0836,13.6665,-13.0593,20.2399,6.6118,-26.3955,-16.6822,-9.9995,9.2606,18.957,27.2447,-2.945,18.9217,-23.3107,21.9487,-20.7497,2.4573,-1.3903,-14.2801,5.4814,-13.1915,6.0525,0.27864,-1.801,-39.2813,26.0198,-7.2042,20.5197,2.6395,4.6262,1.2185,2.1045,1.2294,-33.2669,0.33062,-10.636,27.2317,16.8897,0.1682,-14.697,-13.5893,16.5946,4.2754,19.073,25.0653,8.2985,-16.9438,16.3248,-25.4352,-9.3733,-8.9577,-19.9629,-13.0487,9.9718,2.6377,14.1576,-23.2348,-20.8375,-3.0671,2.8928,18.9358,22.2611,-0.74679,-1.6302,2.7484,2.9439,8.5058,9.7057,9.2627,-14.6088,22.5821,-22.9662,13.694,-7.8247,-20.892,-18.6197,-0.46662,0.34177,16.9832,15.4693,-10.0827,2.6629,-21.4753,5.3262,-13.8404,2.6832,4.1516,12.366,-1.1519,18.3649,-0.18492,10.7623,-20.852,-20.8912,0.16003,-0.46783,19.8434,20.4711,-9.4803,11.6086,-21.1738,20.6996,-9.9255,2.9857,5.9705,4.024,9.8069,3.9509,9.408,5.7872,-21.0619,-20.9277,-1.454,1.6397,18.2103,19.5995,-10.7906,10.1758,-18.6665,23.0089,-6.5021,14.5061,4.3696,-7.7995,16.6961,-8.3721,15.6939,2.386,-22.2645,-19.852,-4.6848,2.4463,16.0638,19.5222,-11.0858,6.6972,-13.9007,20.556,-1.8787,15.2187,5.0845,-14.6143,19.9387,-15.6971,18.5492,0.81502,-21.7015,-22.0762,-2.4063,2.8574,17.5203,23.7534,-1.7512,-2.3367,-1.7717,-6.5836,0.21146,-3.4378,11.5189,-7.2027,21.8675,-19.7659,14.2704,-5.0532,-12.669,-4.7981,1.934,9.182,23.7861,23.1662,-0.64958,-20.8272,17.8256,-19.6715,17.8185,4.0979,-21.0525,-10.5091,-18.1659,5.891,-2.2874,10.9058,-13.4759,-8.4144,-4.7295,8.9282,12.5415,20.4129,-0.64129,19.7774,-21.3828,21.2219,-20.7657,-1.5392,-19.9765,-9.9299,-11.1409,9.1577,5.8333,15.0025,-20.9296,-24.0897,5.786,-2.2354,26.865,20.8056,0.89705,-15.5388,12.5299,-16.5793,10.1264,-2.985,-16.5871,4.9452,-22.2146,19.5204,-4.4089,15.5827,-13.7959,-9.9727,-2.8922,9.1554,16.571,24.8878,-0.24463,21.2557,-22.2769,21.9538,-18.6252,1.9104,-20.4324,-8.4143,-12.3063,10.1986,2.1388,11.6394,-18.5371,-5.3812,-11.7877,13.355,7.2707,20.0079,18.5105,7.6889,10.6548,-9.1564,-7.0118,-16.5988,-0.74188,-25.2178,24.7076,-24.225,24.2695,-0.53933,-18.019,-6.5754,-10.0333,12.5276,2.0057,19.762,18.2516,7.3386,11.3812,-9.6949,-0.37197,-16.0222,0.3625,-24.2554,27.0207,-24.6736,26.379,-1.5215,-18.1272,-8.0894,-7.002,17.2645,22.8016,17.8673,6.434,11.3998,-15.6119,-14.5732,1.0353,-26.055,29.2628,-27.4135,25.8455,-18.918,-13.377,-3.5236,6.905,16.3666,20.8407,18.7568,10.4046,8.3448,-8.2141,-9.1095,-16.6792,-2.5166,24.2563,-26.3803,25.3209,-22.5236,1.169,-17.5881,-15.3338,2.271,4.9534,21.0398,20.265,19.9759,9.3271,11.789,-8.6352,-7.6478,-16.3505,-5.2297,22.7376,-25.6329,26.6693,-18.013,6.3094,-16.0422,-19.4965,5.9147,-2.1641,21.9082,16.9934,20.2089,6.7802,13.7365,-11.1739,-6.2727,-17.8746,-6.9766,18.7114,-22.9036,25.7477,-14.1461,9.5868,-18.3733,-11.3064,-4.7168,-3.6505,15.5606,19.6253,16.9003,5.6602,10.8105,-1.4288,-5.9012,-15.1013,1.9409,-27.2045,25.0164,-28.0669,21.9733,-2.5407,-13.9197,7.9751,-21.1379,22.8445,-6.1117,15.8202,20.7695,15.7733,5.6038,-4.5185,-13.1644,-18.2432,-6.3686,-19.5906,14.6201,-11.5668,22.8534,9.9658,-18.2081,-14.1213,-2.5065,3.8006,16.342,19.74,16.8045,4.9193,12.7329,-9.8444,-3.6393,-15.225,2.9722,-22.2095,23.2203,-24.8998,19.9643,-3.094,-17.9505,-16.0319,-0.52104,2.8761,18.3883,20.0085,17.6606,3.6755,15.3436,-12.7126,-1.2989,-15.5106,3.5287,-20.0757,21.8314,-23.3488,18.0993,-3.6275,-14.1823,10.5551,-25.0972,24.3378,-10.8984,13.7785,19.6695,19.6681,-0.19176,-0.54341,-19.1724,-20.0517,-8.001,-13.1576,5.946,-7.2061,14.5542,6.9029,-13.967,-24.8577,10.0078,-10.9191,24.8622,15.1972,20.8631,0.52128,20.1551,-19.9653,-0.16646,-19.866,-5.8317,6.2183,-12.0103,12.192,-5.9177,6.1126,-13.8816,-24.3964,10.2773,-9.4696,24.8194,15.2278,22.0724,1.3425,21.2933,-19.1098,0.083854,-19.5739,-2.4367,7.8451,-9.5653,13.9371,-5.8052,7.5415,-13.905,-24.2301,10.6111,-9.7781,25.1284,15.4735,21.9507,1.0747,21.9977,-20.0485,1.5488,-19.5789,0.16926,8.2663,-5.5446,11.9512,-2.4208,6.8855,-14.9182,10.5928,-24.0625,24.9562,-8.507,15.083,18.8439,18.9218,1.7171,-0.20878,-14.7585,-17.159,-9.0859,-15.3121,9.0914,-6.5432,19.8964,10.7896,-14.8313,-25.4571,9.7091,-10.0033,25.8458,16.4249,21.8832,-0.35551,22.5306,-22.158,1.9831,-20.3026,-2.9107,4.1151,-6.7987,7.3572,-1.5813,5.5927,-26.3108,-9.1684,-11.828,18.5529,-2.8294,24.9515,1.8466,11.0974,-8.4933,11.6986,-8.9093,1.6045,-3.2398,-20.7562,24.4889,-14.4798,25.2711,5.0962,-26.2996,-10.9679,-16.9874,16.9409,6.8649,23.6222,3.14,17.1882,-12.7736,16.4575,-10.9731,2.2564,-5.5774,-14.3121,12.339,-8.1366,19.1209,8.3917,-6.0104,-23.5732,13.0536,-15.8454,21.9699,8.2931,20.998,-0.86711,20.651,-19.1004,1.3334,-17.8761,-6.2515,-4.6316,-0.52523,4.1334,11.4069,14.2157,-11.1657,-9.6245,-0.70312,11.0596,17.2886,27.0154,-0.13532,20.2475,-20.3971,24.2335,-18.3646,7.0333,-19.8134,-7.6689,-13.8127,4.1591,-2.4392,3.816,-16.4974,-26.2137,8.0443,-9.5271,26.5243,17.4139,21.4738,-1.1296,23.4978,-22.4113,3.2194,-20.0995,-3.5311,2.4711,-3.7625,7.7771,1.4926,6.8738,-20.6007,-0.058756,-21.7674,20.1446,-3.5601,18.7309,9.1683,25.3815,-14.106,14.6232,-22.6219,-8.3215,-12.0525,-4.6067,-3.3035,7.295,13.2578,16.4427,-18.9686,-19.9936,2.841,-0.02999,23.4566,21.4264,11.2641,-14.7373,23.7804,-25.3705,10.9286,-13.5956,-12.4935,-4.7461,-12.7793,7.2027,-3.7129,8.4009,-17.7284,4.5059,-21.4972,22.3766,-1.9351,19.5529,8.6943,21.5833,-13.8849,13.2495,-23.7161,-9.4361,-16.3842,-14.3577,-2.2707,1.9784,11.2099,13.5008,-17.2606,2.5859,-16.9779,21.4262,1.4972,20.4379,8.2799,23.2503,-16.1565,13.7932,-25.1499,-10.0718,-16.1559,-11.3476,-9.3272,1.6836,3.9789,10.2962,-20.1625,-22.9609,1.6082,-1.9833,22.9142,21.3165,12.7984,-11.9893,27.0226,-25.2332,13.6571,-13.8253,-10.4935,2.0454,-8.8262,10.8533,1.2721,8.3737,-20.3765,-24.2298,3.8976,-4.0942,24.5006,20.4379,-0.18792,-1.0414,0.66295,-0.94362,0.9078,0.15745,-15.5603,9.7959,-25.764,25.607,-10.102,15.8943,-20.3961,3.9089,-24.226,24.3976,-4.3435,20.2474,-0.11027,1.0315,-1.0057,1.1338,-0.73499,0.27212,-15.9234,-25.7473,9.6312,-9.6588,25.7676,16.2819,-20.4682,-24.3362,3.6706,-4.0165,24.4465,20.5944,-0.24136,-0.82976,0.44504,-0.59331,0.7681,0.31476,-15.3281,10.2301,-25.7014,25.6765,-10.4532,15.5289,-20.4322,3.9202,-24.436,24.4662,-4.6763,20.0647,-0.40756,0.69203,-1.2055,0.97237,-0.9174,0.16926,-16.4322,-26.1129,9.9881,-9.438,26.3698,16.6829,-20.7926,-24.4046,3.3886,-3.8001,24.6212,21.0306,-0.16182,-0.63486,0.55605,-0.32757,0.92064,0.50642,-14.8069,10.4878,-25.4895,25.6099,-10.8573,14.9922,-20.5646,4.0727,-25.8668,24.7852,-4.6908,19.9905,-0.37054,0.81553,-0.18799,1.1241,1.4677,1.5832,-16.8049,-26.8755,9.7023,-9.7036,27.1228,17.1065,-20.455,5.3471,-26.6304,25.201,-4.031,20.5378,-0.35686,-1.0414,6.8034,0.51469,11.6953,6.2592,-16.96,-26.592,6.7923,-10.2592,23.6791,15.287,-21.0465,5.7801,-24.8615,25.1747,-2.4847,21.981,-3.4144,-10.9886,11.3284,-6.7079,19.3243,8.8964,-16.0069,-22.5603,2.2475,-8.1155,15.2129,11.5607,-21.621,2.9696,-26.0808,24.7073,-5.355,21.3996,-8.0094,-18.3957,0.31453,-10.2851,0.69441,0.37524,-13.0866,-16.0471,5.9098,-3.9586,23.2175,15.7182,-21.2827,-24.6087,3.1385,-3.5271,24.8156,21.299,-0.39289,-0.77281,0.45573,-0.4372,1.2154,0.69167,-14.1221,10.871,-25.1637,25.3983,-11.0573,14.4504,-21.5924,-24.6686,3.3014,-3.4456,24.9452,21.6791,-0.87641,-1.1545,0.52401,-0.42507,2.8731,2.2275,-13.4571,11.1551,-24.9821,24.9655,-11.4103,13.7581,-18.2133,5.4923,-23.9589,24.8171,-5.0728,20.1771,-0.86735,-0.31457,-9.5776,0.19915,-16.9242,-7.4408,-17.7309,-23.1102,3.8236,-6.9755,18.634,14.0516,-17.3266,-20.9311,4.7489,-1.1641,24.0191,20.9064,6.6957,-15.1246,16.5754,-27.5957,9.0605,-14.0844,-16.6035,-3.9347,-19.4204,4.2891,-4.3298,6.9409,-21.5275,-24.8591,2.9297,-3.345,24.5755,22.2452,-1.9971,-1.8755,-0.26746,0.14981,5.0974,5.4377,-13.3585,11.5188,-24.1446,24.9922,-10.6729,13.1018,-21.4208,-24.4641,2.8158,-2.8013,23.5715,20.7809,-3.6565,-2.6138,-0.8586,1.9309,9.4966,11.092,-13.4515,10.857,-23.8576,24.3008,-10.1387,13.1694,-18.1907,-1.3157,-15.7524,18.2191,3.2292,20.4178,8.9381,25.2528,-16.485,15.7341,-25.4735,-10.0842,-15.2464,-5.5651,-11.0867,8.589,1.4328,11.4921,-20.7016,-23.7346,4.2365,-4.0172,21.2481,16.294,-5.9716,-4.1502,-0.15759,3.893,11.4113,13.8219,-13.6226,10.3917,-23.5439,23.305,-10.5509,13.8799,-19.4269,-24.9945,4.4001,-6.4309,22.0283,17.1298,2.3176,-11.4835,15.9979,-13.2286,13.3841,-1.867,-17.2208,5.0617,-18.8744,21.0438,0.59759,18.2809,-22.3867,-23.5039,7.6089,1.1247,28.6374,20.6442,-2.7637,-11.4495,0.42331,-13.0899,0.082171,-3.0925,-12.4212,7.4387,-23.8566,21.8512,-6.7456,14.0921,-21.0941,-21.7348,-0.022508,11.1056,19.973,20.2231,-12.8783,-0.8698,-10.6005,14.2505,0.56096,13.3417,7.5429,-14.8615,22.502,-28.5165,14.7595,-22.0384,-22.2272,-20.749,-3.0451,-0.24306,17.7688,20.3723,-9.058,-2.2958,-2.5339,9.2568,9.394,14.1378,8.0626,-14.7698,23.0011,-22.6846,15.4642,-4.6494,-22.0647,-6.6106,-14.1711,14.5431,7.0719,20.7747,6.4463,25.3019,-18.7877,18.2208,-25.1839,-7.2788,-11.3632,-0.62652,-11.3443,10.6598,1.0907,12.6,-20.3931,-7.8661,-10.0623,13.7229,11.1321,22.6412,6.3447,25.0005,-17.0511,18.499,-22.9684,-6.3716,-15.2133,0.054965,-17.7212,12.6969,-3.8588,11.2383,-21.0084,-15.3392,-6.2281,6.1311,12.7274,19.3825,7.6654,-18.2904,25.4067,-25.5107,17.8252,-8.1133,-12.2784,-9.5136,-1.7714,2.0966,12.4821,13.4053,-18.3907,-12.9174,-5.123,6.2021,10.9728,17.4061,6.7098,-17.619,25.2995,-25.2479,20.2683,-10.0696,-15.3386,-13.0792,-2.7575,1.2344,9.3249,11.2689,-21.1775,-10.201,-10.736,12.1297,11.0082,23.2936,5.5597,23.7298,-18.1504,17.6522,-23.4019,-6.7403,-15.9893,2.2881,-17.403,16.5589,-2.3178,13.2585,-23.1158,-10.4925,-7.0464,13.2735,29.1913,3.6498,23.783,-30.3588,18.2012,-16.0431,-16.1903,1.4363,-17.7256,17.537,14.3601,-16.4144,-11.3691,-5.4707,-10.7716,13.9605,9.9606,20.4665,1.2936,23.0715,-18.1726,23.3077,-21.2424,12.4218,-5.1238,-11.1693,1.5653,-13.4433,2.0919,-4.4854,11.4537,7.5617,-18.0578,-13.539,-3.1583,5.5711,26.8483,27.4211,5.5905,-19.099,25.1172,-25.0585,19.9375,13.8144,-16.2748,-13.0431,-4.4937,1.2339,16.3605,17.7457,-18.5644,-12.9797,-4.7811,7.4897,15.7097,21.6327,1.4446,21.9006,-18.0275,18.3676,-19.1861,-1.9636,-13.4548,3.2948,-17.4099,13.3358,-5.9057,7.9976,-19.6394,-17.0018,-0.29928,3.6578,18.979,21.0402,7.479,-21.5177,24.8391,-25.1526,15.365,-11.5828,-18.7954,-12.3495,-7.4646,3.6352,10.9026,15.965,-20.0653,-18.4468,-3.8462,-0.42709,14.0227,15.2629,8.8142,-15.6701,25.52,-24.5226,16.2488,-9.9152,-14.9504,-9.4902,-3.1844,7.6134,14.3001,19.4395,-19.3524,-14.4682,-3.8136,6.6254,16.0065,21.8663,1.4209,20.4905,-18.3662,19.3587,-19.7703,-1.0706,-14.7976,4.1305,-18.7066,15.8636,-4.5853,11.3893,-19.1298,-3.3207,-15.1209,17.5921,1.9239,18.9419,8.7297,25.8583,-16.368,15.9401,-25.2293,-9.5583,-15.3185,-3.071,-12.4613,10.7925,4.2914,15.369,-19.1738,-15.9586,-1.0479,4.9791,19.0859,20.997,1.5333,20.0486,-17.7633,18.61,-18.6407,-0.79436,-16.5838,4.1302,-20.9242,18.97,-4.3946,14.2074,-18.3894,-18.0817,1.1414,1.5568,20.9402,20.3224,1.6245,19.1566,-15.5108,18.3591,-16.2458,0.17977,-18.2025,4.3636,-22.0745,20.3976,-3.4574,16.0098,-16.0055,8.1728,-23.1301,24.2344,-7.0563,16.4662,17.9035,14.7496,3.9837,-2.2391,-12.1183,-15.3474,-9.6955,-20.2826,12.1355,-9.1163,22.9152,12.1737,-13.9172,-26.0447,29.727,-8.9667,25.1668,9.6871,-5.2836,7.506,-23.7816,-20.4925,-26.7257,-4.0653,-22.9892,8.7608,9.6506,-7.8418,-25.371,17.7875,-19.2769,23.6917,6.2911,11.4398,-0.45735,13.5185,-10.0704,5.0377,-6.4533,-18.9894,-5.6847,-11.5931,12.9609,7.8311,19.2711,-23.172,-10.1246,-14.0053,9.7633,7.3107,18.3958,-0.15445,21.2886,-19.9219,22.0204,-17.6757,2.2907,-9.3223,-8.9902,2.7048,3.2467,15.973,16.8279,-23.2128,-15.8626,-12.5737,4.3013,9.5442,17.3381,0.72862,19.0835,-21.4495,21.4175,-20.4968,1.88,-10.24,-6.142,-1.1591,9.1016,13.2354,18.6315,-16.5344,-19.1647,-1.2853,-5.3515,11.4833,9.7852,17.9704,-1.9891,19.3945,-22.0035,0.36369,-20.3079,-6.5859,-16.9569,13.0236,-7.5854,22.5908,12.5821,-14.9664,-13.6791,-2.1657,0.67005,11.5212,12.3992,19.6549,-0.37559,19.6248,-20.9297,-1.4006,-20.4305,-8.5329,-21.9462,13.9281,-11.7751,22.4559,10.2039,-7.4699,-24.5026,14.2914,-17.6203,20.4895,5.5315,18.5911,0.35692,19.7983,-14.2796,4.8062,-11.0299,-12.4334,-5.8355,-3.1228,11.138,12.6167,20.2158,-12.0202,-11.0358,0.71883,2.2803,12.7128,13.1291,21.3263,0.5886,20.9584,-19.8604,-1.0855,-20.9165,-8.4048,-22.1165,14.1957,-13.8235,20.8263,6.2072,-11.0295,-11.7598,9.1613,-0.27831,26.4302,17.7993,21.5639,0.33069,23.9219,-20.5889,6.2765,-17.7348,-6.4273,-20.8075,5.94,-14.5531,1.2607,-4.9926,-27.0013,-18.2779,-8.6434,2.0832,18.3589,27.462,-2.7481,21.3243,-22.8796,23.1975,-19.5625,2.1871,-2.29,-9.0394,3.0935,-8.7368,4.8429,-0.096914,-23.1723,-19.216,-4.9611,3.2974,15.586,19.8233,-8.3482,-7.1717,2.5991,5.0172,12.3695,13.707,9.8499,-15.1764,23.5065,-23.2735,13.8435,-8.9149,-16.5154,-18.3267,1.0863,-5.785,2.7686,9.4823,13.9542,-6.8593,3.9004,-21.8885,5.8901,-21.6353,3.0833,-5.6042,-3.766,9.0036,-1.7821,17.797,-1.4572,14.7192,6.6076,-21.4831,-21.5573,-0.37016,17.2482,18.8822,-13.0834,18.9028,-23.0472,24.3285,-8.4991,-1.0359,-4.0051,11.7594,11.5009,14.4586,-20.6566,-19.2294,-2.7354,3.0225,13.3827,13.3487,-14.0539,8.5411,-19.4476,25.4376,-5.1018,20.9037,0.0032111,-14.1453,17.8241,-10.9282,19.5829,-2.3736,-20.4639,-16.7868,-4.6676,-0.17924,14.213,16.411,-14.1192,4.2289,-16.4321,19.7,-2.518,13.2425,0.32519,-18.6391,17.8205,-16.4063,19.702,6.9413,-25.1292,-17.9684,-8.7461,10.1581,15.9707,24.0828,-0.63476,-2.4523,-0.69194,-8.0618,-1.2797,-4.6538,9.5863,-12.7438,22.9232,-18.2892,17.0794,4.0823,-18.1103,-4.2022,-8.3647,5.6915,15.0683,25.0561,-1.5504,22.8193,-20.789,24.1216,-20.2642,5.3305,-15.0486,-9.3423,-10.1515,-6.0621,-7.5277,-5.705,-21.5507,-18.964,6.5656,22.3698,32.9851,-1.2392,-26.9171,15.4704,-21.1678,7.9521,-17.233,-1.5635,-21.9812,13.0894,-5.4923,-21.6969,-12.0482,-7.4792,8.367,13.6788,20.9603,-1.1881,22.3282,-20.3421,22.2768,-16.3341,5.1371,-16.0319,-4.851,-13.7119,0.67497,-9.7303,-6.5021,-20.4341,-27.3712,5.7765,-8.8863,25.2414,17.4217,3.1023,-12.4855,15.7434,-16.199,11.2687,-7.8222,-16.6349,6.3985,-19.7078,19.304,-2.0379,16.4503,-27.6306,-17.4086,-9.6374,8.6828,16.1606,23.8502,-5.6083,18.4317,-25.1641,24.2936,-17.7348,6.2956,-6.2979,2.0725,-6.1634,4.7023,-6.3259,-4.4859,-20.2601,-12.8636,-5.223,9.1447,14.0912,21.7345,17.3705,9.4057,8.4297,-7.8378,-7.401,-16.147,-1.0959,24.7709,-26.9567,24.2223,-25.8996,-1.3087,-19.7982,-7.1444,-13.2903,12.4227,10.7928,21.2272,16.7067,7.193,11.3931,-7.7206,-7.363,-14.8191,0.40404,-24.2316,20.651,-24.7011,18.8111,-1.895,-19.2702,-19.1511,-6.4717,-3.5949,9.9493,13.172,20.8079,15.9839,16.0019,4.629,6.7028,-7.7884,-5.9696,-13.9735,1.0346,2.2214,-19.1295,19.3964,-20.3771,17.2094,-2.5102,-21.3199,-15.6177,-1.1132,7.6224,22.6272,23.0372,17.7447,8.2495,7.9301,-10.706,-11.7866,-17.7176,-4.1458,25.7877,-30.6907,27.0662,-23.3003,0.0049751,-18.8757,-19.1518,4.0066,0.50428,23.9539,19.4278,18.8217,8.4205,10.902,-7.8068,-7.5234,-15.7823,-7.7347,24.3813,-26.9611,30.5004,-15.5821,12.4067,-15.7965,7.3099,-21,24.2832,-4.5194,17.7313,19.4371,14.2595,6.9727,-2.9335,-8.2309,-13.7816,-8.7073,-22.1013,17.3743,-10.7695,26.7767,13.2525,-19.3612,-16.0305,-4.1189,7.0624,15.5049,20.1557,15.7456,8.3984,10.7552,-9.1252,-4.6244,-14.7021,2.9265,-16.8557,23.6149,-19.8628,20.5732,-3.0641,-19.2181,-15.6211,-2.2133,4.0867,17.7675,20.5889,16.3519,4.4698,13.7673,-10.9808,-1.2426,-14.1872,3.3844,-20.5208,22.7302,-23.5748,18.9767,-3.5383,-19.5298,-17.7952,-1.2978,2.7068,19.4517,21.4476,16.8097,1.7434,16.1717,-13.6039,0.78117,-14.0244,3.9752,-19.1027,21.0682,-22.9689,17.3521,-3.9635,-13.4187,-23.6316,10.0213,-9.6235,24.8502,15.6828,20.5087,1.6353,20.6092,-16.0159,3.1415,-14.9447,-4.9859,8.72,-10.4147,17.5051,-1.6235,12.6843,-14.0578,9.8532,-23.8056,24.6846,-9.1013,15.2383,19.3812,18.4164,2.6833,-0.47134,-13.773,-15.8547,-8.011,-15.623,9.4651,-6.4503,20.0855,11.837,-15.3926,-25.2185,9.1949,-9.4079,24.7213,15.7817,19.2763,-0.7432,18.8367,-19.3683,0.34344,-17.5334,-8.0194,4.3021,-14.3477,13.6739,-5.4078,10.1613,-15.0413,-24.4227,9.4248,-8.9038,24.9023,15.8504,19.9646,0.7372,20.3843,-17.7399,2.0642,-16.6597,-6.146,6.8062,-11.1315,15.7479,-2.4902,11.3192,-15.125,-24.8983,9.379,-9.1617,24.9492,16.0087,20.6093,-0.80545,22.296,-19.4502,3.895,-16.5489,-4.9364,4.246,-6.6883,13.0071,1.2679,11.7633,-15.7274,8.7226,-23.6168,24.241,-8.4301,15.3016,16.9976,14.4757,3.7233,-2.905,-11.7068,-16.2504,-10.7801,-20.2546,11.1688,-9.0512,22.1298,11.7712,-16.3187,-25.8039,9.0828,-9.0771,25.7719,16.6061,20.4281,-1.9933,22.5554,-21.8015,3.8103,-18.0055,-6.1958,1.0956,-7.2073,9.516,1.0833,10.3993,-26.8977,-5.6746,-27.25,17.7735,-8.0227,26.2203,13.584,1.4033,8.4659,-2.705,13.248,-18.4871,0.75569,-16.2301,-2.2819,-18.0572,-0.91374,-11.4624,6.0609,1.7965,7.562,-24.6524,-6.0019,-18.5134,17.7984,7.9209,23.4749,-2.5455,12.0233,-11.4885,15.7014,-7.804,4.3826,-7.5278,-19.5297,12.2804,-9.9205,20.0705,9.7225,-7.9645,-22.5845,13.5097,-15.5144,19.1464,4.6421,18.5884,-1.9222,20.8333,-19.1098,4.9456,-13.99,-14.1116,-11.0929,-0.78631,7.1312,16.0026,20.0118,-26.4499,-17.378,-8.7876,7.826,16.0948,24.5,-5.5572,18.1308,-23.6944,25.6702,-16.7675,7.1763,-3.5007,2.3474,-7.9979,0.68696,-9.6401,-6.989,-17.8809,-26.4966,9.263,-9.103,26.1237,16.3965,19.6602,-2.4559,22.9838,-22.0614,5.1613,-18.0657,-4.5861,0.33345,-2.8904,8.7468,3.799,10.315,-20.1146,2.0081,-21.6229,23.492,-1.7374,21.5885,10.7473,23.1253,-15.0351,9.4388,-24.6103,-13.1007,-14.4977,-12.8518,-4.3714,-4.5314,9.5203,8.0153,-18.8165,2.892,-21.7805,23.2775,-2.1494,21.5676,9.3724,23.3566,-14.9952,13.4979,-24.6634,-9.1345,-15.2362,-12.0713,-3.0237,1.9646,9.4944,11.0901,-21.4733,-23.4345,1.9342,-1.9514,23.1478,21.6196,14.3046,-9.5297,25.841,-23.2228,12.0216,-13.1034,-7.1404,4.4508,-9.4656,12.7574,-1.3615,9.3562,-19.4912,2.2276,-20.4595,21.9102,0.38415,20.7546,9.9229,24.4313,-14.9482,14.739,-25.2952,-9.9562,-12.8935,-10.4771,-4.0895,2.0813,4.9279,8.7827,-19.7163,0.02162,-17.5771,20.3949,2.9719,21.157,9.4159,25.02,-15.546,14.9094,-25.3385,-10.9592,-12.5679,-6.2067,-9.7853,2.6649,0.059156,6.271,-21.7501,-23.2507,0.7446,-1.3587,22.4685,21.0071,13.9386,-12.0158,27.8216,-26.294,13.2891,-14.4786,-5.5612,1.0918,-2.4107,9.3917,3.5201,8.536,-20.3034,3.9912,-24.3583,24.4003,-4.2675,20.4262,-0.19021,0.89982,-1.0278,1.1137,-0.90499,0.15465,-15.9483,-25.567,9.3957,-9.692,25.5442,16.043,-20.484,-24.4292,3.7893,-3.9561,24.3352,20.59,-0.31449,-1.0573,0.58103,-0.84602,0.90584,0.22246,-15.5745,9.6727,-25.5828,25.5972,-10.09,15.8466,-20.1227,4.111,-24.4901,24.3814,-4.9788,20.1094,-0.23737,0.75569,-1.0367,0.87563,-1.0665,-0.13604,-16.3296,-25.6776,9.5892,-9.3849,25.8015,16.1975,-20.6464,-24.3837,3.5432,-3.7062,24.2754,20.9348,-0.44053,-0.89579,0.37267,-0.56968,0.51649,0.029047,-15.1566,10.0539,-25.5002,25.646,-10.7323,15.4274,-20.1089,4.4254,-24.8923,24.6735,-5.1677,19.568,-0.20669,0.86742,0.35406,0.83956,1.9993,1.4015,-16.6662,-26.2721,10.0641,-9.4474,26.938,16.9044,-21.0942,-24.5738,3.4948,-3.2778,24.4728,21.8265,-0.48344,-0.8791,0.63335,-0.68698,1.2457,0.35454,-14.541,10.508,-25.3843,25.6789,-11.1551,14.5909,-20.5103,4.9018,-25.2882,24.8479,-4.0842,19.7074,-1.9824,-1.9372,4.3962,-0.2831,10.3741,5.5943,-16.8636,-27.1706,9.1478,-10.1066,25.1476,15.967,-21.1438,4.7504,-24.2105,24.6077,-2.4669,21.2896,-4.9874,-10.6553,10.51,-5.0398,19.5348,9.5004,-15.5637,-24.4347,3.4043,-9.4589,14.4909,10.4236,-22.2226,2.2295,-25.3062,24.036,-4.4697,20.9595,-9.0392,-19.4064,0.65475,-10.2749,1.1988,0.70908,-11.2543,-14.9649,7.0796,-3.9716,22.3747,14.475,-19.462,3.9066,-23.9987,24.5678,-3.4136,20.633,-0.89356,-1.03,-8.9744,-0.38518,-17.5472,-8.5864,-16.242,-22.7978,4.3845,-7.6369,17.6545,12.6071,-21.37,-24.7938,3.0836,-2.675,22.6214,20.3328,-0.79029,-1.325,1.6992,-0.66729,7.8506,5.8802,-13.9082,10.9339,-24.8154,25.9375,-11.8537,15.465,-21.6003,-24.7627,1.9024,-3.4977,17.6803,14.7453,-3.0229,-3.9266,1.5248,1.5206,12.5517,13.1006,-13.582,11.0573,-24.478,24.9954,-13.8415,13.9084,-18.9463,-21.651,3.8216,-0.71602,23.7186,21.2739,7.8637,-13.7372,16.4151,-27.7135,8.0056,-14.7591,-14.3043,-4.5302,-18.4361,0.82926,-5.7435,3.8929,-21.7743,-24.1477,1.5748,-5.1438,18.8451,15.9753,-7.0283,-6.9626,0.85156,3.0912,14.1366,16.6049,-13.6279,11.3762,-24.4011,23.5826,-12.384,8.019,-19.5543,-22.3557,4.68,-2.4422,24.5427,21.7274,-13.3896,-11.9436,-2.0591,1.788,10.2697,12.7082,-13.1663,11.2316,-24.2082,24.6695,-8.3095,11.9419,-20.8213,-2.8407,-19.4827,18.704,0.91685,20.5825,9.5334,25.6302,-14.8711,15.3566,-25.1478,-10.6634,-12.2253,-1.7067,-7.923,8.6465,5.6997,11.4423,-16.2871,-20.9741,7.0676,-2.4217,26.1128,20.487,-14.0104,-11.7627,-2.9005,0.57735,4.2712,5.2766,-12.9556,10.3282,-23.9409,25.3644,-8.1412,18.4907,-17.3606,-22.9063,3.5886,-7.0212,19.894,15.5571,3.048,-13.8764,18.4263,-16.9591,15.1526,-2.7712,-18.418,1.4269,-16.3641,16.5976,5.1175,17.7002,-19.8913,-31.7498,15.7943,-0.60384,36.6856,0.60977,-11.1581,-0.4786,-11.9421,-0.29706,-16.1776,14.4244,-24.6228,37.8459,-2.3645,-19.7299,-19.6192,-2.6624,-17.2528,20.1856,12.4422,26.3693,-9.8178,-1.3996,-5.8577,-0.61737,0.99821,-3.7481,-1.533,7.9251,-15.8075,22.8038,-18.3608,17.393,-20.1634,-4.596,-21.0662,-17.71,-2.5021,6.3711,16.3305,24.1633,-14.4113,-8.2801,-4.6355,1.5696,2.5739,2.8639,7.8092,-18.1544,23.9459,-23.7999,16.1145,-8.4208,-20.6742,-6.8673,-14.4336,13.4053,7.8483,21.5599,7.2899,25.6247,-18.2289,17.6388,-25.4447,-8.4602,-12.7564,-0.55574,-10.8212,11.9521,0.067489,10.2865,-20.7948,-6.3412,-12.1137,15.4307,11.042,24.1429,6.6849,25.5077,-17.3204,18.3909,-23.1249,-6.2597,-13.7876,-1.7788,-15.2915,9.1544,-5.2942,6.8949,-23.2618,-10.5275,-12.868,13.5706,11.1232,24.4579,5.3162,23.692,-17.309,17.6626,-23.5448,-6.2692,-12.7001,4.0532,-15.8082,14.4337,-3.7845,9.8198,-18.9295,-12.5153,-5.1365,7.4159,14.1891,19.8875,7.4227,-17.5114,25.7102,-25.2469,14.829,-9.106,-13.7165,-12.8496,-2.103,-0.90302,9.4443,9.5085,-16.5439,-12.2696,-4.8624,5.6773,-4.1797,19.5295,15.2846,4.7902,-15.6405,25.4795,-25.49,25.819,-11.3693,12.2205,-10.5588,-9.1658,-1.8599,0.73433,-1.4413,8.9602,8.4976,-24.2817,-18.4196,-10.4255,12.4779,13.6801,25.4023,5.2007,-7.2646,24.0493,-22.8923,17.8257,-6.9065,-15.2192,-16.452,2.5497,-3.4814,14.5338,10.5023,-21.4502,-6.4283,-13.8396,15.3699,10.0268,23.0559,2.8748,-18.5537,21.9228,-20.2825,16.491,-2.6584,-8.8504,-14.6313,5.7885,-6.9985,13.7078,7.0277,-30.5061,-22.5587,-3.6682,22.7112,17.6446,5.215,-39.2939,25.5551,-16.5004,14.3299,-19.9957,-15.8858,-2.564,13.3432,11.7424,-22.3351,-15.2044,-7.4336,8.107,14.3015,22.424,-0.17253,20.5173,-20.5798,18.6051,-21.5043,-2.2769,-9.9578,6.6337,-14.0332,14.9045,-3.0625,9.828,-18.5519,-17.3721,-0.55339,4.291,18.0577,21.5877,9.1577,-13.9723,25.4735,-25.3781,15.1069,-11.6022,-18.5159,-13.4011,-5.6007,1.1261,10.3932,11.7688,-15.9466,0.54074,-14.7095,20.0109,3.3449,21.6024,8.9805,24.4295,-15.5583,14.6208,-25.5465,-10.9854,-18.5915,-7.446,-13.8388,5.593,1.3717,9.7777,-20.9855,-14.4327,-2.7783,8.6377,7.6998,22.6096,1.1603,21.6504,-20.7231,20.6738,-21.1224,0.64608,-14.3847,2.6438,-19.8776,14.2767,-12.7829,10.5796,-20.6427,-18.2187,-0.11546,14.1708,29.6337,0.28156,21.5174,-19.5523,20.8979,-18.2292,-16.003,5.6702,-20.8262,26.5798,0.9871,-19.6424,-20.2069,1.7247,-8.4461,22.5359,21.2776,0.07418,19.3725,-15.7294,19.5714,-14.408,1.6042,-17.6108,4.8164,-21.4559,13.9363,-3.4865,14.6922,-17.2269,7.1507,-23.9989,23.9473,-7.9069,16.4943,15.0518,11.6241,3.9184,-5.4335,-8.4775,-15.3361,-12.1457,-23.3425,13.2203,-11.2179,25.2606,13.476,-5.1783,-24.0646,18.0571,-15.0587,25.7297,12.5412,6.4507,-6.7947,9.064,-19.6452,-2.8361,-18.3714,-20.1115,-7.8087,-13.5701,7.8722,1.4169,10.7959,-19.3074,-11.9866,-10.3083,3.9848,5.7916,12.2105,-1.5236,20.1235,-21.925,21.8969,-18.9411,1.961,-16.4821,-11.0462,-2.5487,9.1002,16.236,22.2328,-20.8205,-8.0875,-14.4421,9.219,3.231,14.3079,-3.1576,19.6343,-20.9081,22.694,-15.7367,4.6542,-15.1867,-14.2792,3.3605,5.3005,19.1091,21.1666,-12.1534,-13.77,0.22553,-4.7403,10.4186,6.9432,20.566,-1.589,21.0615,-22.881,0.26129,-20.9825,-10.8157,-21.6823,12.6498,-7.7947,23.8465,13.7683,-13.8015,-12.6247,-1.0156,-1.692,11.7361,9.8012,21.261,-0.63096,19.955,-22.055,-3.7232,-21.6019,-8.6323,-22.8538,15.235,-11.4656,22.9242,11.278,-4.9751,-23.33,17.1682,-17.2997,20.7761,5.0182,14.3036,-4.6715,20.0271,-16.6959,7.7389,-10.4068,-19.7479,-10.2952,-5.4792,10.5474,14.1026,21.6693,-12.4344,-12.1135,8.6959,-0.52534,26.9771,17.3206,21.6744,0.96148,25.1667,-20.4787,6.717,-19.9397,-6.2732,-21.4423,4.1895,-14.4929,1.2268,-3.8296,-17.6425,-26.4952,9.4161,-9.0477,26.5061,17.5977,17.4179,-7.4319,25.2067,-25.0105,7.4962,-18.1845,6.7372,0.80307,4.2396,-2.2976,-2.8838,-3.586,-27.6219,-19.4394,-9.1125,13.6488,14.8941,26.1037,-3.4213,16.2999,-23.0335,19.6007,-18.5048,4.6568,-1.9525,-5.964,4.657,-4.1227,7.4618,2.2722,-22.6536,-17.3212,-4.9476,8.3022,16.2978,25.5464,-9.469,-9.0715,-1.2269,-1.4018,0.24694,-0.25283,10.0974,-15.031,25.0075,-25.4514,18.4452,-8.4298,-20.0167,-21.9047,-2.0278,-0.1552,14.9913,-2.6931,-16.9384,23.4989,-25.2367,12.8004,-10.0437,-3.5305,3.9068,21.5848,23.0899,-13.2034,-13.1935,-1.101,-11.8174,11.2795,2.9717,11.6553,-14.5983,4.3329,-19.3721,9.2757,-6.867,20.6943,14.1935,-3.1765,-19.2112,13.4144,-18.6385,20.9336,1.1022,12.8948,-17.8016,-15.31,-4.9791,2.2525,14.9377,15.8187,-15.9083,6.1296,-18.4524,23.1563,-1.839,15.3245,-3.3378,-20.3557,13.6433,-10.336,19.7791,9.779,-19.6403,-11.9264,-10.8617,-3.1739,9.4017,13.383,4.9784,-17.8947,7.1004,-22.0761,0.47423,-9.6212,3.57,-10.5144,17.6918,-4.4514,21.8943,16.8234,-31.765,-13.2211,-19.4328,16.5185,15.3372,24.8722,1.7583,-2.4466,0.91443,-11.5391,-2.3382,-9.8089,7.5435,-21.7741,20.7253,-15.7853,18.5987,1.8611,-28.0613,-24.0988,-8.7281,7.3173,13.9794,22.208,-2.9538,8.2469,-21.5345,18.1372,-16.4244,5.6362,-7.0273,-13.8117,-2.4415,-17.3043,-8.6574,-12.0359,-23.4353,-16.328,-7.5401,10.2563,15.1523,23.7769,-2.0126,16.2039,-23.6895,22.1909,-18.0426,6.2595,6.8528,8.1465,-2.3944,0.85251,-9.7444,-7.77,-22.3616,-13.4203,-8.257,8.4488,14.6824,22.6014,-7.9595,18.5524,-24.6658,23.9977,-17.957,6.9437,5.0279,8.8965,-3.087,1.8015,-9.9289,-8.3011,-18.2202,-24.739,4.1177,-7.8118,26.0289,15.5036,5.6904,-10.315,20.1849,-24.6136,10.1695,-18.7663,-14.5596,0.85397,-13.1427,6.7098,-7.3803,-1.6815,-23.9865,-15.5932,-8.4068,7.2903,16.8659,24.3717,-7.9353,18.3701,-25.234,25.1787,-18.1479,5.493,5.6389,7.5729,-1.6424,0.2984,-7.5953,-7.6436,-22.4385,-6.6269,-11.4418,14.1815,11.9188,23.9429,10.4705,-16.7989,24.0784,-21.1719,17.0271,-5.0347,-7.9723,-13.3502,3.9685,-7.0457,15.1541,8.9715,-21.5674,-9.0008,-12.3597,11.1238,11.5579,22.7931,15.5584,7.2367,8.3522,-5.4722,-7.0011,-12.6766,0.47881,-25.0336,24.0791,-26.1496,22.0112,-2.7033,-21.7136,-12.7336,-0.81182,12.4454,9.9985,26.5958,18.1071,8.9104,9.2281,-7.5788,2.5668,-11.5033,-1.1666,26.3423,-37.1241,24.9208,-36.8812,-5.1667,-20.7421,-8.8675,-8.7406,10.7396,12.9926,22.1561,15.1514,6.4669,8.0607,-5.66,-5.5746,-11.8785,1.566,-22.1841,21.5341,-23.8564,18.894,-2.6667,-20.2344,-12.0673,-4.9857,8.7933,14.6688,21.6221,15.0948,5.9128,8.5017,-7.0266,-4.5535,-12.0525,2.6534,-19.024,22.3021,-21.5609,19.212,-3.2587,-23.1315,-22.2254,5.0529,19.8888,34.8664,19.5934,8.4895,9.244,-17.0879,-11.8334,-7.2001,37.1191,-33.0167,38.1195,-13.9782,-19.4345,6.2676,-24.4807,26.215,-9.4409,17.6763,15.7543,9.8632,6.5383,-5.502,-4.4865,-13.5153,-11.1947,-26.4728,20.61,-11.5477,30.2143,19.4144,-19.5915,-15.17,-3.0447,6.6433,17.3874,21.7945,15.6053,6.2004,12.6665,-8.336,-0.88316,-11.7727,3.0017,-19.238,22.7928,-22.6637,19.1433,-3.8413,-20.4743,-17.5442,-2.3697,3.7915,18.6289,21.6743,14.8465,1.5207,14.5745,-12.5895,1.108,-12.7832,3.9922,-19.2545,22.1999,-23.0756,18.2223,-4.1252,-21.1764,-19.0779,-2.0189,2.8421,19.664,22.0347,14.9483,-0.55011,16.1571,-14.5831,2.2424,-13.2039,4.1039,-18.2365,20.6545,-22.3976,17.3252,-3.8909,-15.0612,9.341,-24.5071,25.0038,-9.3966,16.1321,16.4872,17.6275,1.1385,2.8355,-12.9052,-12.5161,-11.0548,-15.5797,7.3429,-2.7647,19.925,14.5265,-15.4424,-25.0238,8.9861,-9.1812,24.903,15.9863,16.2269,-2.6828,18.0313,-17.6436,1.5314,-15.4092,-12.0504,2.8954,-15.4224,15.8587,-4.1153,12.3813,-15.1396,9.2688,-23.915,25.3283,-8.8726,16.2116,16.6781,14.7161,3.6032,-0.18503,-9.97,-11.8884,-11.0343,-19.1799,9.5591,-5.7952,21.814,14.7178,-15.8176,-24.7678,9.1444,-8.9171,24.8807,15.9254,17.3965,-0.49784,18.4519,-16.9256,1.692,-15.8376,-10.3401,5.2611,-14.8513,16.5243,-3.8558,11.9283,-15.7406,-24.9948,9.24,-9.1708,24.9223,16.1452,17.386,-1.7922,20.1893,-18.6498,4.0979,-15.7047,-10.5169,3.5321,-11.8704,14.5844,-0.10947,12.3807,-15.8721,-25.2486,9.2476,-8.905,25.2876,16.4239,17.9065,-3.6404,21.6474,-20.4114,4.4936,-16.261,-10.784,0.10716,-9.8645,12.3618,0.81429,12.2291,-15.5173,8.6135,-22.1904,25.5167,-7.1399,17.011,17.1403,12.6472,8.3413,-2.408,-4.3791,-10.8432,-10.9119,-21.3303,11.9716,-7.5034,24.2035,15.3139,-16.6688,-25.8537,9.9625,-8.9367,25.5716,16.5504,18.2511,-3.6419,22.3944,-21.7419,5.4063,-16.8486,-9.8301,-0.53481,-8.2575,10.7059,2.4118,12.17,-25.978,-6.1752,-15.7856,16.847,10.4506,24.9056,-1.26,12.3132,-16.2189,19.6367,-15.8558,5.6171,-6.5004,-20.2698,14.2799,-8.057,19.5564,11.2005,-11.1616,-27.2327,14.6731,-11.029,25.6492,13.3271,18.8895,-3.3705,20.1891,-22.7855,3.8232,-18.5693,-9.6688,-3.4393,-6.7005,5.1108,1.071,6.8187,-4.7044,-21.8103,16.6963,-17.7422,20.5695,4.9954,15.9626,-6.0371,22.3902,-20.2434,7.0343,-13.4957,-19.3631,-14.5326,-0.026101,8.5452,17.1824,22.3707,-25.5109,-16.0931,-9.026,8.8276,17.4861,25.7839,-8.4392,18.0577,-25.5278,25.0037,-18.1124,6.8666,5.4006,8.6314,-1.6452,2.9255,-3.9867,-2.4552,-17.0112,-26.359,11.2207,-9.1849,25.2167,14.4888,17.8722,-4.3287,23.1883,-23.0299,5.9826,-17.9602,-8.456,-2.0575,-5.1884,9.6416,3.4844,11.9962,-21.5406,1.6435,-23.5217,22.7184,-2.4814,20.3017,13.784,23.6972,-9.6897,11.8963,-21.5735,-10.9854,-8.226,-11.6423,4.8195,-0.087203,14.2602,13.4874,-21.5401,1.9332,-23.5342,24.5608,-1.675,22.6735,11.1124,22.7904,-14.9615,10.0317,-24.6362,-11.4329,-11.485,-12.8348,-0.892,-2.6421,8.7926,8.1557,-19.7607,1.9666,-20.6612,25.1266,0.17716,24.1059,9.9244,24.096,-16.1182,13.6889,-26.0394,-5.203,-11.5757,-10.1981,-5.4043,-0.52162,1.4956,4.7683,-21.7891,-23.2631,2.1258,-2.5683,22.6816,21.3705,14.2725,-10.5938,26.9486,-22.0792,12.5291,-11.8662,-8.0302,2.2861,-8.8477,13.8899,-1.6972,11.0726,-20.779,-0.093422,-19.9773,19.2617,0.70154,19.2589,10.3857,26.0744,-15.144,16.8599,-25.8801,-12.0703,-7.5928,-3.3784,-4.2264,3.0167,2.2855,5.6536,-21.676,-1.1798,-19.8855,19.968,0.90193,20.3012,9.9819,26.1032,-14.7636,15.1264,-25.4987,-11.5792,-8.7056,-2.0446,-6.0561,4.5727,4.2705,8.2984,-21.7525,-23.1329,1.422,-0.95982,22.2111,21.5561,14.4244,-12.47,28.061,-28.1568,13.6422,-15.4719,-4.3494,0.3117,-1.2356,7.391,2.4887,6.5142,-20.4442,3.6773,-24.3462,23.9974,-3.8189,20.5731,-0.26765,0.78316,-1.2137,1.0024,-1.0271,0.14077,-15.9052,-25.4872,9.3638,-9.8484,25.4881,15.9239,-20.1139,3.92,-24.3696,24.1909,-4.6488,20.4025,0.029363,1.059,-0.98373,1.2447,-1.0463,0.16502,-16.1593,-25.4067,9.3265,-9.4837,25.2152,15.8135,-20.629,-24.2673,3.2528,-3.4154,23.689,20.7866,-0.33615,-1.0056,0.25011,-0.78065,0.20021,-0.16629,-15.5559,9.6406,-25.4596,25.6888,-10.4008,15.7511,-19.6947,4.2602,-24.8087,24.516,-6.4847,19.8639,0.22694,1.1198,1.0029,1.06,2.2104,1.3685,-16.354,-25.524,10.0284,-9.2039,25.8677,15.8605,-20.9976,-24.1876,2.5978,-2.8578,22.9489,21.4454,0.17049,-0.27916,1.1289,-0.32827,0.58081,-0.45934,-14.9467,10.0323,-25.3687,25.7495,-11.8245,15.0753,-19.6802,5.2407,-24.9263,25.3047,-5.3056,18.8041,-1.7162,-2.4044,4.8836,-1.0783,10.6152,5.2745,-16.8314,-26.9327,9.6659,-9.5896,25.4917,16.1869,-21.7978,-24.4243,0.57138,-1.4767,20.7471,20.5007,-0.57843,-1.4835,5.3993,-2.0392,12.3168,5.6536,-14.7825,10.7778,-24.9654,26.0543,-9.097,15.3523,-20.5235,5.2189,-23.8541,24.7826,-2.8996,20.2705,-5.3343,-10.4779,9.9722,-4.7881,19.1307,9.2253,-15.6887,-25.6329,4.2052,-9.8421,14.7571,10.5946,-21.5691,2.5645,-24.5923,23.5876,-4.2034,20.1535,-9.2599,-19.4726,0.53484,-10.346,1.2181,0.68275,-10.5193,-14.6288,7.6364,-3.9485,21.7178,13.923,-20.0982,3.6328,-23.7807,24.3615,-2.0958,20.722,-0.81361,-1.2439,-9.6413,-0.55412,-19.8395,-10.0726,-14.7379,-22.1115,4.2759,-7.7952,16.0114,11.3112,-20.305,-22.6959,3.1478,-0.27287,23.1468,21.3774,9.5417,-13.3019,18.2277,-28.8523,8.7457,-15.6625,-11.911,-2.6573,-16.8829,1.3249,-6.1585,2.9321,-19.1237,-20.9042,-0.6824,-1.3385,19.8656,19.9588,-9.3711,-11.2795,2.756,-4.6098,13.3592,7.3145,-13.1417,9.5004,-24.4718,24.8704,-9.5218,15.5722,-12.5365,-14.0629,4.8124,1.089,23.1015,19.5802,-15.6429,-15.8566,-4.6416,-7.4779,-2.7934,-3.7637,-9.7411,10.5634,-22.4608,21.9117,-13.613,12.2138,-16.4029,-20.7878,4.3819,-3.3434,20.4652,14.1837,-15.9247,-11.9632,-5.2764,-1.8805,-3.9127,-4.5675,-12.2856,14.4418,-23.781,23.0422,-14.5005,12.5934,-22.121,-26.1093,5.7766,-5.5105,27.6747,25.5102,-10.8471,-7.4319,-3.1017,-0.8226,-4.9355,-5.8333,-17.0548,11.8392,-25.7907,23.4588,-5.7646,1.7244,-20.9277,-0.69703,-19.414,20.0466,1.8958,21.0488,10.527,25.427,-14.825,14.7119,-25.7026,-11.3088,-11.2927,-5.3862,-6.9969,5.0113,2.9991,9.1349,-21.1652,-26.2636,8.9928,-7.0381,26.3933,16.7044,-3.0443,-2.1749,-1.3129,-2.4238,-1.3425,-3.5328,-16.8662,7.3,-26.1705,24.8344,-6.4556,20.4955,-14.9149,-19.6748,3.1916,-7.3443,23.714,12.3381,23.905,3.2998,-17.4137,18.1428,-20.7686,9.66,-12.924,8.2049,-18.7609,-4.5575,-13.3654,5.8389,-5.5282,1.505,-5.6756,-26.5567,-19.0081,-7.2578,6.0728,24.3859,29.6721,-2.5678,-5.9306,-1.3081,-6.3559,-0.28958,-1.8835,6.4068,-20.3613,23.0715,-25.0384,12.0893,-0.98477,-24.6327,-16.9055,-8.6515,9.6833,11.2144,12.4244,19.7376,-1.1721,-1.5558,-0.12364,-0.83339,-0.23588,-0.86796,-0.45418,9.3369,-14.6507,25.1118,-23.8764,18.5576,-22.5174,-1.3548,-20.8213,-6.5139,-13.0973,14.7446,8.3342,21.333,7.6081,25.9721,-17.1072,16.2809,-25.3326,-9.5681,-11.0491,0.49293,-12.629,10.5253,-2.86,8.5217,-23.279,-8.4627,-14.3499,13.8715,9.9667,23.216,7.0538,25.5675,-18.2753,16.948,-24.7766,-8.4616,-9.2219,2.3425,-11.4955,12.641,-4.0486,8.203,-24.6126,-9.793,-13.928,14.7782,11.104,25.0115,5.4623,24.535,-17.9543,17.9892,-23.801,-6.3544,-9.1395,4.6006,-13.7048,12.0651,-5.0092,7.2386,-24.8033,-13.9604,-10.53,10.0078,14.6297,24.3644,5.7191,-17.172,24.2235,-24.489,18.3372,-5.5849,-10.7493,-14.4708,3.885,-2.8146,12.6484,10.3683,-19.8664,-13.3736,-4.2996,5.9459,16.3385,20.1887,10.661,-16.866,26.3469,-25.6524,14.276,-10.134,-10.1787,-10.0015,-1.4442,-0.94869,8.2552,8.5489,-18.5315,-21.9356,5.2875,-3.4411,22.1353,17.7565,-9.3186,11.0482,-25.5159,25.8417,-11.9714,14.6355,-11.9019,-12.8685,-0.9622,-1.2851,8.8092,9.5227,-23.4769,-14.8325,-8.6387,8.1582,14.3121,22.8087,1.6774,21.4916,-20.1729,19.1753,-21.9922,-3.1375,-7.906,7.816,-13.0163,14.2647,-4.2273,7.146,-24.2083,-8.2235,-14.5209,14.9074,7.6009,22.3986,4.6809,-18.8333,20.7111,-21.0222,17.3297,-3.3929,-6.6621,-14.8436,9.7236,-8.592,16.4396,4.937,-22.9392,-13.7853,-11.1322,9.2822,16.3502,23.3518,0.83047,22.4498,-16.105,20.544,-15.7416,-1.7606,-11.2142,3.1882,-14.3305,12.2136,-0.56598,8.675,-21.4025,-2.1884,-18.1466,19.2588,3.6312,21.6314,10.4396,25.8104,-14.2395,14.8663,-25.7249,-11.1517,-11.7649,-2.7808,-10.4837,7.2485,0.50885,9.2317,-21.8706,-21.1877,-11.4773,0.4372,7.2961,18.7611,22.916,0.22447,-1.3422,15.4589,-14.5091,15.618,-13.4142,1.2726,-13.7969,-14.9234,3.7509,-14.3032,13.3448,-1.2243,12.1989,-20.517,-21.7148,1.9635,3.3119,22.235,20.8948,-1.749,12.9048,-16.1734,14.5144,-13.4561,2.947,-17.168,-0.38897,-20.6208,17.3044,-3.2796,15.5261,-18.3081,5.6961,-24.7279,24.6654,-9.9804,16.8555,13.9796,6.9949,8.0144,-8.7667,-3.039,-13.1399,-13.713,-25.4476,9.7584,-10.0447,25.9247,17.6062,-9.944,-27.2143,15.5083,-12.8368,25.6984,14.0121,19.055,-1.8177,19.3849,-23.2422,3.0419,-19.6224,-15.6969,-4.0955,-10.6043,6.1606,0.55194,6.109,-15.0105,-7.4702,-9.3516,5.7483,6.136,13.6593,-3.7528,20.92,-23.6048,23.0642,-19.4558,2.1376,-21.5365,-14.4382,-4.1317,9.9082,15.7529,23.0967,-19.9273,-4.9563,-14.7885,12.6062,3.7189,18.7556,-6.6535,18.9505,-26.4619,23.5837,-20.6788,5.875,-19.083,-17.0349,2.8304,5.2693,17.3277,18.9052,-8.7267,-12.6916,4.0883,-4.7594,14.3512,9.6939,21.7937,-2.0817,22.2807,-23.4618,1.1674,-21.3006,-12.3808,-23.6765,13.3702,-10.1196,24.0833,12.0623,-11.2108,-13.8896,8.6377,-3.2795,25.0785,16.2225,24.2018,-0.67883,23.8045,-22.8522,3.9647,-20.263,-8.4313,-24.6482,7.3183,-13.321,6.0275,1.5717,-3.0339,-21.8651,20.2118,-13.592,26.5965,15.6109,12.5044,-9.23,18.774,-23.6483,3.7516,-19.1673,-22.6478,-13.6123,-9.1008,6.3053,7.1083,15.8214,-17.9294,-25.9582,8.4748,-8.8972,25.6522,16.7703,18.2934,-3.8648,25.0859,-24.4732,7.1985,-19.9566,1.0708,-3.4887,-0.57488,-4.776,-5.0879,-4.7088,-17.48,-26.0725,8.5099,-8.4396,26.1006,17.3403,18.0316,-6.9014,25.2825,-25.2486,7.0671,-18.2099,4.529,4.1609,2.355,-0.36478,-1.8725,-4.1321,-26.1147,-15.0747,-10.3973,12.5138,12.9488,25.3868,-4.6835,18.1287,-23.9934,22.8547,-18.8481,5.073,-2.2734,-7.383,7.5747,-5.6073,9.9451,1.9709,-25.2906,-16.895,-9.2114,11.0938,13.5772,23.4054,1.066,0.37647,0.99942,0.4248,1.6785,1.5639,7.7253,-16.6819,25.946,-25.9133,20.3513,-5.5109,-13.5321,1.9688,-14.1143,23.7279,2.6964,24.3478,-16.2893,-25.0864,10.836,-12.6296,23.9535,12.9442,-10.5976,10.5897,-20.0836,13.2619,-11.7741,-0.95639,-18.1293,-17.7875,4.5808,1.0142,19.7338,21.9774,-16.3038,9.762,-24.7856,22.3345,-18.9226,8.2897,-6.4137,-20.3688,5.1979,-15.7986,-0.1183,-9.196,-25.9202,-16.3548,2.459,-12.375,22.8675,18.5787,11.0649,-11.073,-25.0295,13.6383,-10.6105,6.9751,-6.1089,-17.9255,7.1888,22.8248,15.17,23.2382,-27.7794,-7.3613,-12.4972,19.9858,9.7456,26.197,6.0862,7.4635,-3.8294,7.4138,-8.9809,-0.90992,1.5647,24.9573,-24.6859,19.7351,-24.2641,-4.4994,-24.2357,-14.3962,-8.6335,9.4746,15.5945,24.9619,-5.7664,18.9774,-24.1429,24.9356,-18.4177,5.8467,7.8049,5.3261,-2.1043,-1.5484,-7.772,-5.3114,-23.6011,-15.1434,-8.4348,9.6718,16.9437,25.9019,-5.9551,17.8834,-24.2435,23.7021,-18.4816,4.4752,7.9428,9.8744,-2.9469,4.2257,-6.5944,-1.5828,-21.9727,-14.1666,-6.9491,9.1971,16.6442,25.062,-7.2216,17.2998,-24.4819,22.5244,-19.3589,4.3196,9.6305,11.248,2.8069,8.1125,0.49691,4.4224,-18.6846,-24.9782,12.4919,-9.3551,26.6161,17.9025,10.5991,-6.1693,23.2869,-25.4805,8.4672,-19.8301,1.5414,3.23,-3.2099,-0.90027,-1.6571,-2.4312,-25.6935,-17.4226,-8.3497,8.9174,17.5514,25.6133,-6.8558,18.9889,-25.4104,24.7049,-18.7112,5.4722,4.6699,4.8367,-0.89741,2.3471,-1.424,1.7729,-15.2299,-20.5722,3.5253,-4.7589,20.1752,16.7109,-14.9776,13.2954,-25.9881,26.0106,-11.1872,14.156,-7.516,-7.5354,1.4661,1.6491,10.6273,10.8985,-21.9266,-9.422,-8.7842,14.3625,15.582,23.7297,7.4619,-15.6764,24.8013,-18.936,19.8003,2.9818,-12.3395,-18.6951,-0.16369,-11.9399,7.5439,4.415,-23.5692,-12.1292,-11.606,9.2339,14.5802,24.8999,13.051,9.9583,5.6265,0.84206,-8.0272,-10.1343,1.2288,-23.0121,25.2969,-27.1338,22.2613,-4.6653,-22.2976,-11.1871,-10.4118,8.846,14.059,23.1629,12.6611,7.05,6.352,-3.3341,-6.3263,-10.8702,1.8452,-22.5936,23.7753,-24.9112,20.8403,-3.2449,-29.8873,-25.4255,7.0516,-14.4666,26.4647,15.5074,14.5211,16.3198,5.5009,1.545,-9.0777,-11.2792,-1.2383,-16.2361,-29.1729,28.1789,-7.8883,22.3882,-21.7859,-11.7443,-7.4932,8.6422,14.5843,22.4167,12.6012,5.5266,6.9695,-4.9854,-5.0408,-10.8904,2.6669,-21.1086,23.0814,-23.5526,19.7436,-3.3171,-21.3121,-14.1289,-5.1881,7.4766,15.8617,22.1641,12.8895,5.083,9.2878,-6.6591,-2.5321,-11.1395,3.4517,-19.8227,23.4378,-23.0896,19.7995,-3.6193,-21.1789,-16.6064,-3.3226,5.7176,17.7551,22.5395,13.4726,2.4849,13.41,-10.0742,1.4268,-11.08,3.9323,-19.5965,22.7194,-23.4513,18.56,-3.9475,-21.471,-18.3518,-2.8343,4.7211,18.6018,22.8231,13.4751,-0.6268,14.8517,-13.6593,2.9539,-11.55,4.1753,-18.5941,22.1043,-23.1752,17.9371,-4.9047,-21.6275,-19.2004,-1.7309,4.2148,19.5636,22.9653,13.9309,-1.6478,16.6109,-14.5046,5.0389,-10.7813,3.2956,-18.2134,20.2995,-22.8215,16.4448,-5.4846,-15.8226,-24.9802,8.9423,-8.9549,24.967,16.1743,14.8081,-0.98097,17.0051,-12.4179,3.1194,-10.6161,-13.094,4.8211,-16.5176,19.8516,-2.7018,15.9161,-15.9317,8.8481,-25.1354,25.25,-9.4406,16.4598,13.0902,13.4971,1.4311,1.8603,-9.8094,-9.9011,-14.1298,-19.3476,6.8042,-3.9223,21.1415,15.8717,-15.9859,-24.9287,8.7678,-8.9576,24.9882,16.253,15.3941,-1.5959,16.4899,-15.4361,2.0564,-12.8021,-12.409,4.1364,-16.9755,17.9051,-3.5058,14.7446,-16.6442,8.9237,-24.8558,25.7774,-8.7079,16.564,12.378,10.4574,4.5164,-1.5627,-5.6788,-9.9568,-14.3107,-21.522,8.196,-6.1094,22.7409,15.9596,-15.7099,-24.6932,8.8932,-8.7438,24.4106,15.8675,16.9577,-0.65222,19.7692,-14.8653,5.3922,-11.7095,-10.7286,5.0473,-12.9415,18.2757,0.22059,15.7686,-16.0173,-25.0384,9.0417,-8.5946,24.7988,16.4228,16.3135,-4.0644,20.4209,-19.2691,5.2843,-14.0674,-12.2523,0.56075,-12.3818,14.5627,0.25566,14.4218,-16.3509,-25.4656,9.0989,-8.4829,24.63,16.755,17.2483,-4.1599,22.1024,-20.456,6.7108,-14.3238,-11.5202,0.15271,-10.1148,13.8078,2.701,15.0354,-17.5255,7.8448,-23.6566,25.9139,-8.1898,16.9941,13.1458,6.3268,9.7421,-5.0469,-1.5758,-9.8065,-14.1588,-23.51,7.6317,-7.2318,23.6534,17.5516,-16.3867,-25.6493,10.8802,-8.161,24.654,15.6934,17.6236,-5.2537,23.4702,-22.5488,6.6338,-16.0345,-12.0533,-2.0552,-9.3274,12.4959,3.2806,15.1326,-25.9589,-11.3136,-12.6458,11.0989,17.1222,26.1043,-3.5291,18.0311,-25.0478,21.7547,-21.2387,2.6212,-8.9521,-17.5058,4.1262,-9.7135,6.2546,1.6983,-12.8124,-25.8409,11.9506,-11.9672,25.6438,14.8151,19.419,-4.2854,22.6047,-23.4214,4.1125,-18.1941,-7.2316,-1.2114,-7.7482,7.1349,0.075284,8.9098,-5.7809,-25.2072,19.1433,-13.7442,25.4379,-7.2919,19.4276,-5.2783,26.6269,-31.0812,4.7301,-29.2278,-20.0128,-13.2698,-1.5273,9.3364,11.8567,13.1429,-25.996,-17.4341,-8.7247,8.3983,18.3322,26.2314,-7.1701,18.5438,-25.055,25.1919,-18.1393,6.1513,1.6317,3.3526,-2.507,3.1536,-1.9809,1.8949,-14.8195,-25.3689,12.649,-9.5552,24.9543,7.7735,18.0109,-5.5099,23.8448,-24.7236,6.2921,-20.1465,-11.0591,-2.6687,-8.7483,12.3959,2.3171,14.574,-20.1501,1.4888,-22.488,23.4178,-3.0679,20.3196,10.1829,19.6914,-11.6091,8.8404,-22.1485,-11.5443,-13.8027,-16.0247,-0.75266,-2.9151,12.0566,12.2454,-22.0794,2.1863,-24.2424,23.1838,-1.7256,20.9514,11.5418,22.5945,-13.0883,11.1998,-24.4982,-13.8624,-11.618,-12.7348,0.75679,1.1172,8.65,10.817,-23.4516,1.5689,-24.5434,27.0849,25.4488,10.9473,24.5212,-21.2499,11.5685,-32.0678,-8.271,-8.9125,-2.8241,-0.05289,6.2972,-19.4388,0.43755,-19.5011,0.66219,-0.96971,16.4406,15.1584,9.1795,26.4528,-14.106,26.4088,-25.7094,6.4282,-13.9823,-4.5122,-1.9442,-2.0962,-1.7767,4.7922,2.9291,7.0499,-21.7218,-22.8594,0.97196,-3.472,22.948,20.3954,15.6277,-7.8252,28.5029,-17.4389,12.7446,-9.4654,-5.6938,4.9521,-6.2829,16.5154,0.4825,12.6251,-20.6695,0.24218,-20.3991,19.3526,-0.647,18.0945,10.7537,26.1763,-14.2466,14.2065,-25.4899,-12.5624,-6.8582,-3.5573,-3.225,3.0453,5.7368,8.8658,-20.9485,0.097892,-19.9725,20.3507,1.0428,20.3644,10.7868,25.8258,-14.5934,14.2935,-25.8347,-11.9639,-9.2818,-5.0734,-5.2556,3.9608,2.9029,7.909,-21.9383,-22.5281,1.4877,-1.7884,23.2163,20.8343,15.2101,-13.0311,29.4263,-27.8956,13.5518,-15.2695,-3.8136,-0.2313,-2.4949,6.2154,1.7766,6.8887,-20.582,2.8246,-23.9339,23.5425,-3.3315,20.8417,-0.22174,0.38792,-1.0633,0.21537,-0.9425,-0.27592,-15.8404,-25.4513,9.3863,-9.7524,25.3317,15.6743,-20.372,3.3608,-24.124,23.7922,-4.0775,20.5305,-0.21757,0.92253,-1.2873,1.0419,-1.1734,0.027704,-16.0044,-25.3101,9.0506,-9.4971,24.5848,15.3216,-19.8672,3.7215,-24.5803,24.1023,-6.628,20.1929,-0.01367,1.1223,0.47432,1.2322,1.4196,1.0485,-16.0113,-24.8641,9.3599,-9.0713,23.7137,14.3402,-20.8312,-23.6112,1.3847,-2.7125,23.3483,21.6572,0.12805,-0.22163,0.03415,-0.39583,-0.052392,-0.13214,-15.3487,9.7002,-25.1058,25.5362,-9.6706,15.4417,-18.82,4.7164,-25.2908,25.2479,-7.3222,18.6348,-1.1321,-1.8428,6.7069,-0.85523,13.2126,6.2347,-16.2781,-25.2205,9.3235,-8.7981,23.9898,14.8381,-21.4722,-22.8301,-0.7878,-1.6783,21.4691,21.3062,1.4584,0.42519,8.6819,0.33896,13.1207,6.116,-14.2297,10.0306,-24.3036,24.8952,-8.803,15.2861,-19.477,6.4867,-24.0398,25.5177,-3.7321,18.6921,-5.8271,-11.9814,10.0239,-5.3061,19.0341,9.7638,-15.6496,-25.5104,4.5978,-9.8594,15.5454,10.9169,-19.4086,-19.7129,0.89875,1.0556,21.5266,21.8467,-7.707,-14.3355,3.7747,-10.0668,10.5025,2.9197,-15.7734,8.9513,-24.6457,23.5022,-8.9092,15.0377,-20.6881,3.7693,-24.2421,23.7937,-4.582,19.6785,-8.7846,-19.2492,0.51951,-10.4345,1.1089,0.83812,-10.8086,-15.1925,7.7587,-4.2973,21.3453,13.8486,-20.158,4.0886,-23.451,24.5191,-2.1208,20.6729,-0.66394,-1.2023,-11.2163,-0.65489,-22.418,-11.2305,-13.9226,-21.6817,3.4205,-7.5591,13.5779,10.3144,-21.051,-22.8651,2.729,-2.2282,23.3619,21.4364,11.0937,-13.7638,21.2888,-28.8186,10.4561,-15.2895,-9.8529,-0.43719,-14.7317,4.2513,-4.9013,4.9546,-20.9546,-20.8575,3.0297,-0.93643,24.9403,21.4606,-5.8812,-11.0892,-0.9387,-4.5272,-0.94281,0.91229,-12.3899,9.0734,-24.9567,24.2985,-11.8497,15.0407,-19.7581,-22.6099,5.57,-3.2725,21.2185,16.5779,4.6739,4.7024,0.050852,3.6609,-4.3325,0.090814,-9.4292,12.7429,-24.686,24.4488,-10.8974,15.5015,-14.9546,-19.7537,6.5774,-5.144,9.3113,13.026,21.1852,4.7353,4.5166,1.4624,0.66227,3.0714,-1.5022,1.7867,-8.9584,12.0926,-23.99,24.9276,-23.5787,15.7837,-2.2902,-20.6366,-32.0876,8.9738,14.3896,25.9702,7.3593,4.3708,5.2291,-1.04,3.051,-19.0698,22.2656,-24.6261,25.9362,-3.4277,-21.1546,-1.0763,-18.8352,20.3771,2.2832,21.4466,10.8363,25.9022,-14.986,14.4207,-25.8973,-11.4349,-10.2476,-3.8794,-8.1971,4.5628,0.94064,7.3784,-17.1819,-25.6782,9.5083,-9.55,24.7246,14.3545,0.34306,-1.6964,2.3881,-3.6989,3.3858,-0.56476,-19.4904,4.1306,-26.057,24.2617,-3.4938,21.1384,-28.2948,2.2382,-18.7019,24.7179,21.2242,15.1255,27.1672,-33.9516,7.9984,-19.8072,0.33027,-9.0695,13.6556,-5.2107,5.7257,-24.83,-5.2031,-8.516,3.5523,20.558,25.5396,-10.3629,25.7467,-27.4808,29.2762,-17.6641,9.7694,1.7048,-7.98,3.4941,-8.2147,6.7615,1.4082,-26.0586,-9.0748,-18.9207,19.2608,1.17,28.4091,-1.0578,3.0654,-11.8265,3.8214,-12.4358,-0.16315,5.3033,22.4555,-21.6399,21.2342,-25.1357,2.6133,-34.0692,-20.5973,11.9532,10.7166,31.1265,1.123,-1.1819,0.53023,-1.5559,-0.42748,11.7871,-21.4707,34.6043,-22.6874,-0.13567,-22.3082,-8.5656,-10.61,17.709,11.943,24.115,-0.18891,-0.83413,2.8656,-1.5356,3.4375,-1.2569,7.3557,-24.7523,26.7825,-26.5519,21.6311,-13.9673,-20.6904,-14.0272,-4.7169,4.8205,17.3344,20.4293,9.3715,-16.8342,26.6611,-26.5252,15.3959,-10.8579,-9.5766,-11.5796,0.7074,-1.0025,10.7156,11.0884,-22.2057,-7.3058,-13.1519,14.4571,9.6136,22.4109,8.3846,26.4023,-16.4346,15.6469,-24.664,-10.2417,-9.781,3.0579,-13.9988,14.2007,-6.0829,9.2305,-24.745,-10.0641,-14.3512,14.4222,11.2751,24.6759,6.7874,24.8962,-18.0186,16.0279,-24.3277,-8.4266,-8.0165,5.7958,-12.8693,14.3415,-5.8999,7.1246,-25.7338,-15.5802,-10.6603,11.0537,15.1799,25.7618,5.4639,-19.2164,24.373,-24.6688,17.9909,-5.6927,-7.6391,-10.2046,5.6307,-1.8438,12.7036,7.7582,-19.8638,-15.4718,-3.2494,3.5378,17.8968,20.9256,11.7122,-16.5271,26.5459,-26.2989,15.0015,-11.003,-9.7213,-9.2563,-0.38915,-0.42146,10.1022,9.824,-17.3763,-21.3877,-2.7963,3.9697,18.9737,21.5195,-14.0392,11.0894,26.1377,-25.9026,14.8923,-11.4339,-10.4584,-10.0315,-1.0346,-1.0386,8.3722,8.8169,-23.0656,-14.5394,-8.9479,8.6274,14.1625,22.9967,1.8469,21.9389,-19.9395,19.5739,-22.8511,-4.6546,-5.6727,10.2619,-12.3039,13.692,-5.6606,3.9875,-23.5943,-13.8608,-10.2182,9.5989,15.1336,23.664,1.4482,23.077,-19.792,20.033,-21.7075,-3.999,-7.9862,5.1579,-11.3188,12.099,-1.889,7.6759,-21.9634,-7.8593,-12.8669,13.6336,-12.3326,20.3458,10.6466,1.6063,-18.546,23.1878,-19.9866,23.436,-4.6161,11.4845,-4.4998,-15.199,9.3141,-13.1206,9.0072,-2.7205,6.4729,-23.6463,-13.1336,-5.2534,9.4951,18.4719,24.5186,0.12368,21.5526,-17.2006,19.8434,-15.3803,-0.42518,-12.7788,2.6438,-13.7508,11.0591,-2.1458,7.8887,-22.0731,-17.6626,1.0222,6.9934,23.0118,24.1991,-1.1003,15.928,-16.8098,17.7637,-12.8229,4.0235,-16.0801,1.5519,-18.0919,13.7262,-2.9873,11.1374,-20.5041,-21.917,3.217,0.61368,23.9027,20.688,-3.1548,13.8628,-16.012,17.8179,-9.9178,7.6151,-17.0199,2.5658,-21.3321,18.2231,-4.7644,14.8737,-16.4843,6.8011,-24.7412,24.9448,-10.4421,16.0938,12.1166,3.3979,10.4371,-9.6746,0.45704,-10.7035,-18.0364,-25.238,4.7457,-6.4805,25.2401,20.1933,-13.0527,-25.9627,8.9924,-11.3439,24.6128,15.696,22.0059,-2.9823,24.8452,-22.6027,5.4517,-17.6009,-4.6781,1.6392,-3.0058,10.3213,4.6416,11.7018,-18.7422,-8.8998,-11.166,6.8693,13.5815,22.206,-4.1279,21.3241,-25.9359,23.6808,-20.9471,1.9144,-21.0221,-12.9771,-4.2335,11.1974,11.8495,19.8614,-22.8622,-7.5896,-17.1176,14.8778,14.3837,26.9424,-5.9481,20.061,-21.9826,24.0817,-16.4412,6.8447,-15.8005,-14.6717,-8.3911,-1.7011,0.2853,2.5052,-11.0718,-16.5632,8.9541,-4.4068,25.0389,16.8822,21.6848,-2.3802,24.3619,-21.8944,4.7053,-18.4742,-10.7623,-23.2598,4.2034,-14.4178,4.5344,-2.1327,-16.1814,-25.1766,7.8994,-8.6605,22.9703,14.943,20.5086,-4.0989,24.7398,-24.2607,6.3281,-18.9918,-1.2515,-5.7751,-1.1066,-5.4969,-7.334,-7.0975,-12.8523,-26.7099,33.9857,-9.6427,21.3899,27.3296,-2.5874,29.795,-24.3302,-21.5724,-16.4447,-6.1079,-13.0683,6.9779,7.9218,-17.2126,-24.6697,8.6377,-8.4235,26.2415,17.1126,18.0501,-4.7832,25.0749,-24.6972,6.6056,-19.2531,5.4527,5.8154,1.9032,-1.175,0.53827,-2.8365,-17.3758,-25.9781,7.9188,-8.9136,25.4172,17.5713,17.988,-6.865,24.4085,-24.7699,6.6564,-18.0408,4.3795,2.2288,5.1405,0.66245,4.9129,2.5857,-25.5978,-13.4196,-13.9669,10.7681,14.2281,24.6384,-4.9967,18.3484,-21.418,23.0267,-14.0888,6.5332,-1.1968,-9.1058,10.0243,-7.6067,12.9656,3.1978,-23.0856,-12.7954,-10.3365,8.9501,12.6817,22.1465,-4.0667,-4.4295,-3.6414,-4.9256,-1.8405,-2.0327,5.7057,-21.2715,24.4966,-26.0604,21.1387,-1.9412,-24.7421,-24.3766,-2.1065,-1.7506,18.756,21.4069,-9.7048,13.2921,-25.5246,24.7713,-17.8255,8.6149,2.5367,-8.8761,8.5064,-10.9533,-1.2085,-8.9198,-25.5638,-28.3243,-7.2764,0.028791,21.5837,23.6896,-13.343,2.0057,12.7364,-26.2363,11.9962,-14.3881,-3.3959,1.0259,26.1393,9.8325,21.0484,7.3231,-21.972,-19.8708,-3.1307,1.7918,20.6156,23.4384,-8.2446,17.2349,-23.5981,25.1984,-16.66,7.0517,9.1878,0.68961,13.6145,-9.5775,7.2824,-7.5666,-25.2663,-15.6068,-9.1312,7.4551,16.7885,25.3473,-5.2844,19.6338,-24.3973,24.3911,-20.3914,4.1552,3.5463,6.2117,-2.2774,5.0522,-0.43407,4.1925,-26.1409,-16.7706,-8.497,8.1944,16.8023,25.1394,-4.0428,19.9221,-24.2631,23.8099,-20.2893,2.4732,-1.5181,3.8602,-5.6932,5.4228,1.2647,6.9541,-25.5616,-16.565,-8.3779,8.1701,16.1989,24.1831,-5.346,19.9436,-25.132,23.8148,-20.3886,3.8579,-2.844,1.2709,-3.4841,8.1637,2.6311,10.3613,-18.1001,-25.3577,2.8501,-8.3891,23.1655,16.3991,19.0135,-4.5611,26.1127,-23.7367,7.5034,-20.183,-1.837,-2.621,0.74468,-3.4952,12.138,8.1033,-25.9497,-17.2354,-8.2641,7.8271,17.6641,25.0747,-6.4806,19.6597,-25.3632,24.4389,-19.4407,4.5764,-1.7295,1.7112,-2.9811,8.4504,0.64213,8.6041,-24.3776,-11.172,-19.2781,12.4713,16.5001,29.3518,0.33773,1.2883,0.21673,0.69092,-0.093614,-0.036556,-12.0869,26.107,-21.3236,24.4863,-21.48,-0.82592,-16.9258,-20.6807,3.1357,-2.7737,20.937,18.5783,-15.3587,11.7466,-26.2702,26.4705,-11.3907,14.9611,-10.2381,-9.8598,-0.43014,0.039479,9.4613,9.4846,-26.6357,-10.8628,-10.8859,38.3032,17.6256,4.8685,-26.1696,25.1814,-20.8174,19.5897,-5.7002,-13.7291,4.5194,-2.728,11.2519,-23.4263,-14.4578,-9.331,7.902,15.8936,24.537,11.3613,9.6773,3.9599,0.30806,-6.9138,-9.0485,2.6499,-20.5747,23.9709,-24.82,20.5039,-4.4982,-22.7012,-13.8319,-8.4146,7.9748,15.5569,23.6939,11.3116,6.9103,5.2019,-2.582,-5.2978,-8.898,2.7123,-20.726,23.6908,-23.7295,20.1633,-3.6692,-22.0609,-14.0021,-6.9161,7.9134,15.8203,23.5477,11.7,5.6885,7.0957,-3.9502,-3.1328,-8.5352,3.346,-20.3737,23.4116,-23.5993,19.6168,-3.6886,-21.5793,-15.248,-5.0235,7.5946,17.3859,24.1658,12.6063,4.0438,11.8168,-6.2062,1.8603,-7.5096,3.6518,-19.7922,22.6688,-23.6065,18.4309,-4.1761,-22.4762,-17.516,-4.2306,6.7273,18.0972,23.8297,12.2325,-0.73677,14.5126,-11.3427,3.8531,-9.0672,4.6151,-18.7213,22.8768,-23.3549,17.8212,-5.5595,-22.8166,-18.5925,-3.582,5.5958,18.737,21.472,11.5507,-2.9626,15.5656,-14.2747,5.8283,-10.2696,4.8844,-17.972,22.4741,-23.5436,16.7784,-8.471,-22.7805,-19.3833,-3.4302,2.4879,19.5214,22.497,10.9731,-4.7843,16.7501,-14.1541,8.2858,-6.9084,4.8873,-16.9895,20.3038,-22.6559,15.0315,-5.6203,-16.2951,-25.2523,8.8663,-8.6444,25.1477,16.3872,10.5576,-1.3801,12.3619,-10.6356,2.0077,-9.042,-15.9292,4.2745,-19.8695,20.6737,-3.6508,16.7197,-16.0094,-25.0406,8.7632,-8.8627,25.0003,16.2789,12.3167,-1.7457,14.9138,-12.0558,3.1016,-9.8779,-15.2033,3.9534,-18.3015,20.0562,-2.8093,16.5097,-16.3368,8.7925,-25.1663,25.3599,-8.4858,16.8532,11.3721,11.2811,3.6326,2.1653,-4.7847,-5.9643,-15.0321,-20.1462,6.4886,-3.7259,22.7601,17.7375,-16.2445,-25.0162,8.9776,-8.8,24.6631,16.0831,12.8088,-2.1117,16.0184,-13.9362,4.8699,-10.1094,-14.7698,3.557,-17.3596,19.0079,-1.3142,16.7326,-17.206,8.6023,-24.7393,26.0382,-8.11,17.5189,10.7256,6.3242,6.8631,-1.9823,-1.8504,-6.1403,-15.5091,-22.4482,5.6434,-5.3787,22.7406,17.8935,-15.8161,-24.8451,8.5315,-8.7091,24.3661,16.1023,13.8769,-4.3089,18.9515,-15.7649,6.1817,-10.525,-14.7833,1.5909,-14.9067,18.111,0.23244,17.0493,-16.316,-24.9544,8.1127,-8.3097,24.2767,17.0725,15.1717,-4.8841,21.0557,-18.2436,7.2235,-11.9873,-14.0635,0.48486,-12.9414,16.4972,1.5283,16.5717,-16.4689,-24.8677,9.2014,-7.0627,24.0283,18.8605,15.8381,-6.732,22.8484,-21.9703,7.9324,-13.5833,-15.1968,-2.094,-12.3593,13.8398,2.7665,16.3269,-16.755,7.7452,-23.9958,25.314,-8.4342,17.0067,11.2505,2.9435,10.3954,-5.9382,1.4401,-6.7792,-17.1728,-23.195,4.6267,-5.1605,23.7026,19.1869,-15.0988,-24.8948,17.0219,-6.3527,32.2797,18.0898,-6.8139,26.2839,-25.3126,-13.3527,-15.6764,-2.7733,-12.0713,16.6934,20.1913,-26.1146,-16.4615,-8.8994,5.6176,17.6702,25.6236,-3.6065,21.709,-24.9821,23.7898,-19.98,3.7405,-3.2489,-8.6762,5.4306,-7.6486,7.9105,0.20203,-14.3,-25.5831,9.5668,-11.0793,24.667,15.3682,18.4901,-4.8326,22.9996,-21.9982,6.4056,-14.8459,-10.3966,-1.3031,-8.9929,10.3649,3.8754,14.005,-8.3202,-26.6938,10.9688,-27.0075,24.9778,-7.3822,15.8317,17.2525,-0.56718,20.4547,-2.2174,4.8181,-21.9087,-18.4253,-3.4656,-0.074627,-1.7823,-0.27316,1.704,7.3461,7.7893,-26.251,-17.9674,-7.7635,8.2413,19.3181,26.4943,-6.2692,19.2119,-24.9771,25.1839,-17.7241,6.2126,-3.1596,1.0862,-3.5816,4.9093,-1.9336,2.24,-10.9874,-24.7954,10.5487,-24.3343,23.7251,-2.6566,15.1461,15.1258,-6.1374,20.8315,-7.3243,8.1655,-16.1701,-9.9994,-12.3414,-2.5027,-9.9367,-1.953,3.5096,11.8349,14.0101,-19.9458,-22.8585,3.1769,-2.5895,23.5588,20.3744,8.5985,-9.795,16.5417,-22.1343,7.1815,-12.8738,-13.6003,0.58508,-17.1755,10.877,-4.4525,9.3339,-20.1159,2.5389,-22.2934,23.0684,-1.5243,20.6918,10.9487,21.3393,-11.2361,8.8491,-25.1215,-15.3393,-13.1738,-12.8305,-3.9468,-2.3886,4.596,5.6717,-25.3593,-26.111,1.342,-1.4245,23.1768,21.4568,12.9297,5.659,25.0986,-25.7973,10.5088,-17.1197,-8.5971,-4.5205,-6.8702,4.433,2.2344,7.5195,-18.3829,-19.5571,1.7208,-2.0691,19.6931,17.5105,11.5029,-13.1093,25.8895,-25.2147,9.3132,-13.3833,-7.8375,-2.4274,-5.6002,6.8636,3.5333,9.8387,-21.1596,-23.5723,2.5027,-3.6173,23.7605,20.4146,15.151,-7.0762,28.3826,-16.2554,12.7517,-9.5839,-6.6979,4.5504,-5.9136,16.9907,0.57975,11.8794,-19.7037,1.4626,-19.6572,20.4667,0.36591,19.3483,11.1963,25.6716,-14.3605,12.5035,-26.1161,-12.9901,-9.6977,-6.4384,-5.9507,3.3898,1.7628,7.5138,-21.4199,-0.70722,-20.339,20.2647,0.40848,20.2227,11.4141,26.3244,-14.7151,14.0105,-25.8542,-11.8366,-7.4657,-1.9584,-4.621,6.1312,3.8398,9.0522,-20.9881,-23.337,1.6615,-2.9049,23.6009,20.9833,15.2668,-13.1131,28.0991,-28.1556,12.4911,-15.422,-6.0279,-0.97514,-5.8086,5.0692,-0.47097,5.3201,-20.83,1.6896,-23.5047,23.2913,-2.4997,21.5574,0.18046,0.41878,-0.29574,0.030006,-0.68344,-0.60629,-15.7838,-25.1119,9.3977,-9.7151,25.192,15.6021,-20.4607,2.663,-23.7152,23.5342,-2.8752,21.0861,-0.099449,0.57166,-1.1078,0.18557,-1.2856,-0.68728,-15.8346,-25.072,8.9286,-9.3783,24.4679,15.282,-20.0247,3.1846,-24.0244,23.5949,-3.7997,20.8905,-0.058749,1.0801,-0.14178,0.88204,0.69392,0.58836,-15.5136,-24.2501,8.4203,-9.0327,22.7891,13.888,-18.6198,3.8315,-24.8573,23.7937,-4.4165,19.941,-0.56436,-0.90676,9.202,-0.17578,17.2537,8.4382,-14.9156,-22.7605,7.365,-8.5754,22.6252,14.1038,-21.67,-23.3468,1.3109,-1.7185,23.6067,22.7197,-0.28657,-0.28664,4.1726,-0.67256,6.2375,1.4093,-15.2246,9.6703,-24.812,25.0861,-10.4882,14.9578,-18.1955,6.2225,-25.1991,24.2078,-4.2195,20.0069,-7.9065,-15.4367,9.668,-8.5129,20.0487,9.6377,-14.1304,-22.2031,5.9706,-8.6415,18.9488,12.4773,-22.5248,-22.6171,-1.3906,-0.51621,19.0082,21.101,-3.0539,-10.2886,10.0921,-7.4786,14.7469,4.7112,-14.6992,9.5259,-23.3152,24.0189,-7.8744,15.995,-20.0616,6.5069,-24.2473,24.0358,-4.7932,19.7877,-8.5996,-19.4318,-0.0034651,-10.7724,0.57408,0.47439,-11.2481,-16.8875,7.5052,-6.0264,21.2041,13.7256,-20.5902,-20.4561,3.1246,0.88624,21.2699,20.8788,-5.9753,-13.0118,0.72803,-8.0964,3.2146,1.7112,-15.3893,8.2776,-25.1939,24.4656,-12.3703,15.8269,-19.9371,4.5688,-23.2482,24.3667,-2.5711,20.3861,-0.45404,-0.83465,-12.5395,-0.46452,-23.9817,-11.6171,-13.7862,-21.3538,1.5893,-7.4057,11.337,9.8899,-21.2411,-23.0979,2.5753,-2.5962,23.3989,21.2786,11.9618,-12.8096,23.77,-27.3817,12.2534,-14.4761,-8.6767,0.85394,-11.9037,5.9273,-2.0687,6.4163,-21.6679,-24.5851,3.2426,-0.99797,17.3489,17.4769,-2.7181,-1.6403,-8.1889,1.7707,-8.1644,0.63414,-12.5992,10.1532,-23.6572,26.8338,-10.4367,16.7984,-16.0746,-19.8205,12.5507,-0.036996,36.8336,3.6702,8.0981,5.4319,8.0314,7.0974,-14.0301,8.1557,-27.8067,26.7548,2.7664,-23.7009,-16.4201,-9.9968,10.2127,14.8466,25.2998,-2.0314,0.64798,-3.7825,4.1711,-1.4891,2.8572,-2.3575,-20.834,22.141,-24.3108,19.4814,-2.3183,-21.2568,-18.7458,-1.1076,2.1101,20.3843,21.1926,11.005,-15.0366,26.1501,-26.3023,14.6185,-11.3885,-9.3128,-8.8563,-2.7575,0.47015,5.8826,8.7004,-14.3881,-24.5672,10.1953,-10.9895,24.2965,13.6922,0.25557,-3.6063,1.633,-4.9004,0.53644,-2.0646,-21.0029,3.1725,-24.6316,22.9299,-1.0882,20.6758,-21.5872,-24.7765,2.6735,-6.7602,25.0563,20.6365,15.5997,-9.4691,22.7337,-25.0887,6.6928,-15.8527,-8.4038,2.489,-14.7919,11.7014,-4.3399,11.2287,-23.4081,-18.7794,-7.8418,4.1691,19.7103,24.7921,-8.4486,15.3193,-22.4665,23.0937,-14.849,8.5859,-7.5062,-13.1096,0.23905,-13.445,10.5034,0.40038,-29.9774,-11.8483,-19.2574,13.2842,15.5439,27.3471,-2.2655,0.21076,-6.7177,0.0094891,-4.08,-0.2944,1.4533,21.4593,-20.4278,21.3495,-18.2751,3.0419,-22.7537,-21.939,-9.1517,-3.1567,10.4772,19.386,23.9213,0.43959,-10.3832,6.8808,-23.7702,9.2497,-14.6982,4.2903,4.0873,3.6302,17.8145,4.5859,18.7031,-4.8103,3.8202,-30.1803,-13.9023,-12.494,20.9247,36.5779,0.84465,-0.095027,3.1716,0.28479,2.1209,2.0656,-22.3422,36.6689,-23.3243,8.1198,-20.8931,-11.621,-9.9699,-11.3694,11.3103,16.4479,23.8539,1.4881,-1.9147,2.8519,-1.8346,1.7161,-0.75787,0.18381,4.4293,-22.7016,24.6418,-22.8552,22.7165,-16.6546,-0.86581,-19.2775,-11.9599,-2.7586,8.3613,19.013,22.6679,10.8782,-13.7348,27.1609,-23.3992,15.7339,-10.5109,-13.1982,-17.6234,-1.1904,-8.7588,10.01,6.6678,-20.8722,-17.6707,-3.021,1.9369,19.4545,21.7328,10.903,-16.303,26.5811,-26.5795,15.1869,-10.6756,-10.2942,-9.8979,0.13709,0.12601,9.0839,8.9687,-24.0359,-10.4509,-13.6805,13.7772,10.4055,23.9109,8.636,23.9765,-16.2555,12.5173,-24.2368,-11.4489,-8.2586,8.3929,-15.4706,18.879,-8.9052,7.9757,-26.455,-14.8759,-10.9954,11.674,16.1198,26.5879,6.3368,-19.4887,24.6821,-24.7045,15.6547,-8.3692,-6.8037,-12.2776,8.346,-5.9853,15.8046,6.7044,-21.0325,-17.9046,-2.6334,2.9514,18.3812,21.2207,11.2687,-15.5781,26.641,-26.1177,14.9355,-11.4801,-9.672,-9.9611,0.52613,-1.786,11.0186,9.1332,-22.261,-10.2398,-13.1863,12.4557,12.9572,23.7702,0.41264,-20.5435,22.4208,-24.893,16.0629,-5.3307,-0.57144,-7.7948,11.3026,-4.6725,8.687,2.7498,-23.7613,-12.7569,-10.047,11.7792,13.5619,23.8232,2.8181,25.1981,-20.6746,20.3633,-24.084,-4.1134,-6.4384,3.8073,-10.6974,7.6242,-4.6826,3.719,-24.6287,-13.617,-8.3414,10.6838,17.5993,24.1289,1.864,24.6736,-20.3128,20.9395,-21.2856,-4.0151,-9.1701,3.0216,-12.4016,9.8495,-2.852,6.4419,-25.9706,-29.2996,12.9253,-11.0778,29.1131,16.5333,-2.6185,5.1148,-20.3463,25.1099,-5.4067,17.722,-6.2792,-5.2666,-14.7472,9.7538,-1.8375,14.6993,-25.4652,-16.1772,-2.0998,7.1692,31.6218,32.3641,-2.3835,20.738,-21.3179,19.9976,-16.0089,-8.348,-11.5804,4.5129,-14.8188,12.7965,1.2752,6.7403,-23.6255,-22.0993,3.0918,2.0672,26.1398,21.9993,-4.995,17.2944,-18.7922,21.1863,-9.4082,12.6815,-15.2358,3.6815,-19.5287,15.047,-4.0926,13.9912,-20.4724,-23.7798,3.017,-4.0607,24.4493,20.1506,-6.8323,11.1771,-15.4772,19.1597,-5.3686,11.4157,-15.5394,4.655,-20.9469,17.9197,-5.3224,13.2306,-15.7134,-25.2627,8.0437,-9.7882,25.293,16.417,19.5303,-5.3987,24.0772,-22.3647,5.3426,-16.1895,-10.0723,-2.7089,-7.6418,10.6652,3.056,13.8347,-28.4593,-17.7993,-13.4034,8.4494,16.8287,25.2007,-3.1347,19.9801,-24.7942,24.4202,-17.2148,6.1801,-10.1484,-5.4437,-4.2135,8.7527,-0.58344,6.46,-25.3525,-13.539,-8.6877,8.5405,14.4638,23.3788,-4.0135,17.9495,-22.7268,23.4371,-17.6863,7.2322,1.497,1.7083,4.687,4.7478,8.3861,7.8139,-16.3784,-25.3316,7.3496,-8.5537,24.6244,17.5239,19.5555,-5.946,24.9857,-24.2077,5.3113,-17.5143,0.9625,-6.62,-1.5873,-8.9304,-3.8631,-3.4451,-14.5468,-22.2926,7.9542,-7.1207,22.4842,14.5944,18.3691,-5.9939,23.0738,-23.0961,5.4501,-17.5366,8.5089,8.6848,7.4217,7.1462,7.8989,7.1748,-17.3279,-25.0668,8.0287,-7.7405,25.5268,17.6921,18.1643,-3.8207,24.6148,-25.1016,7.1659,-19.2679,-1.2007,-4.5852,0.82158,-3.416,-0.70246,-1.4302,-17.6537,-25.6,7.1154,-7.8057,24.6792,17.4953,18.3207,-6.574,24.9609,-24.3705,8.6832,-16.6637,-1.3313,-3.4073,0.031348,-4.5431,-5.1266,-7.7722,-24.7768,-13.559,-9.9351,9.5046,15.6715,24.3543,-6.7428,15.465,-20.9949,21.7271,-12.8898,7.445,-3.0289,-13.5627,12.1725,-10.3512,15.1429,3.444,-15.0162,-16.8793,1.627,-2.3513,21.581,19.0651,19.3691,-11.9072,25.6965,-25.3051,9.8365,-15.4643,-7.3621,-1.4496,-5.0593,6.7749,2.9807,9.9406,-25.5767,-11.1884,-22.7314,10.6435,15.6693,25.9104,-3.9297,1.9958,-4.6632,4.0472,-0.31371,2.3152,12.023,-19.8509,15.3663,-22.1833,18.1588,-2.2018,-22.877,-12.1767,-11.2311,13.3241,12.1563,24.087,0.2076,-0.81923,-0.33453,-1.7609,0.65008,0.063086,3.4929,-22.7006,22.9515,-25.0386,21.9053,-3.4336,-20.5478,-18.2501,-2.6512,0.62546,19.2637,21.9698,-7.3729,15.1553,-21.9371,22.7515,-16.5141,7.008,10.7859,4.3339,13.0645,-3.5335,3.6868,-7.6826,-23.2892,-20.7872,-4.2782,1.1691,19.5397,23.8827,-6.0737,14.8854,-19.4765,20.4143,-15.6492,2.6748,7.5095,-5.4978,16.6626,-14.7049,9.6996,-8.8294,-24.9829,-16.9583,-9.1046,3.1798,12.6102,19.3926,-3.4185,19.5815,-23.3708,22.5998,-21.3674,1.2937,-4.9887,-0.69723,-3.6612,6.795,8.2703,16.4778,-24.9724,-16.0348,-8.2308,6.8847,13.4209,19.8094,-3.5255,20.8976,-24.3732,23.0334,-21.2659,1.6073,-9.4284,-3.293,-7.1585,8.354,6.4835,16.3522,-24.026,-15.9524,-6.8917,5.9305,18.3017,22.9024,-4.0503,20.5153,-23.728,23.6149,-19.5939,3.4748,-11.0815,-3.2028,-11.0067,11.0997,-0.84606,13.4386,-15.3237,-19.5789,2.8405,-5.9767,14.3906,8.5478,16.0688,14.9631,-2.4169,20.3869,-19.9396,11.7245,-19.047,-1.9099,-12.3771,-14.8984,-4.3271,-7.2106,11.0618,7.6055,17.215,-25.9054,-18.183,-7.722,5.1961,18.4555,24.8125,-5.7432,19.1516,-24.672,22.9861,-19.5501,3.2451,-5.5514,2.4026,-8.1216,11.8375,-2.7771,9.4901,-29.7843,-15.4495,-12.3886,11.1763,17.147,27.6741,0.084125,0.13158,-0.17673,0.30521,0.057852,0.5207,0.13053,23.1682,-22.5156,22.4179,-19.2391,4.0284,-18.3036,-20.6884,1.696,-1.2354,20.6349,19.7984,-15.2503,11.3412,-26.4193,26.4786,-11.4763,15.018,-9.9874,-9.9547,0.10402,-1.7845,10.0728,8.0132,-7.1595,-23.5719,12.5779,-12.072,25.8575,13.3235,-17.8047,8.8618,-23.2811,19.059,-7.1396,14.7834,-8.5821,8.2743,-9.1605,17.9516,0.8296,15.9306,-23.8684,-16.1028,-8.1221,7.6429,16.6777,25.0855,9.4959,7.6084,3.1179,-0.2727,-4.6181,-6.2049,3.2256,-19.2043,23.3419,-23.4458,19.5776,-3.9978,-23.4703,-15.5718,-7.9347,8.2272,16.6483,24.9672,9.248,5.7801,4.6598,-1.8715,-2.5015,-5.4787,3.436,-19.6728,23.2362,-23.2962,19.3446,-3.9289,-23.4149,-15.613,-7.1669,8.2943,17.1686,25.1486,9.339,3.8754,8.0372,-3.2111,1.3754,-4.3256,3.9897,-19.5477,23.0546,-23.3294,18.6656,-4.1773,-24.1133,-16.9989,-5.9826,8.0272,17.697,25.4839,9.4824,-0.52757,11.9598,-7.348,3.802,-5.2654,4.9239,-18.8785,23.7402,-23.3982,17.9321,-4.7358,-23.6931,-17.8472,-2.6791,8.4224,18.1815,25.8932,10.4216,-2.8926,16.3533,-11.5744,6.9608,-6.7263,5.7224,-18.1807,25.1064,-24.7466,16.5946,-7.7259,-21.2249,-18.7929,-3.026,9.1216,18.0563,20.157,8.3258,-6.0837,15.3575,-17.411,9.9974,-13.3687,5.8776,-16.6354,20.991,-28.6239,14.9946,-18.4194,-22.5384,-19.471,-4.3088,1.2933,18.2572,22.3301,7.1914,-8.449,15.8893,-13.7523,12.2653,-0.7758,5.9154,-15.3707,19.1184,-21.2609,13.1332,-4.6273,-16.7155,-25.3012,8.7612,-8.7207,25.0773,16.4445,8.4703,0.47971,10.1206,-4.3798,3.4378,-3.0567,-16.9478,4.8179,-20.9576,22.4607,-3.2717,18.6114,-16.285,-25.1484,8.7811,-8.7108,24.8999,16.0761,10.0481,-1.0786,12.2142,-8.0296,3.7709,-5.3375,-16.4054,4.2378,-19.9444,21.9577,-2.7653,18.5037,-16.2003,-25.0599,8.6959,-8.7809,24.5208,16.08,10.8346,-2.3363,14.6426,-10.9388,5.2948,-7.0136,-16.1985,3.4347,-18.4898,20.7376,-1.4597,18.0517,-17.5524,8.448,-25.5584,25.7667,-8.3782,17.176,6.767,5.4846,2.9392,0.65592,-2.3889,-3.2263,-17.3814,-22.5554,4.5403,-4.0798,22.1164,18.138,-15.954,-24.779,8.5905,-8.4938,24.2178,16.1117,10.895,-4.5333,16.0717,-14.1463,6.7073,-8.0377,-16.7969,1.7572,-17.8523,19.2888,-0.31453,18.3183,-17.204,8.0395,-24.6496,25.839,-8.1684,17.3189,7.7886,3.2011,7.0519,-2.0125,1.232,-3.3495,-17.5801,-22.4337,4.259,-4.2114,22.6835,18.7468,-16.0564,-24.6928,7.6334,-8.3596,23.8651,16.3584,12.0619,-5.5467,17.9168,-15.2652,7.7574,-7.8564,-16.7296,0.80297,-16.6456,19.2132,0.88842,19.2455,-17.214,-24.7059,5.1047,-7.6252,23.1206,17.2761,14.7468,-5.9295,21.8451,-16.5036,9.1473,-8.5074,-15.3681,0.073884,-13.3823,18.6927,3.0497,19.5536,-17.6756,-7.9454,-23.9652,24.0238,-4.8911,20.6908,17.3802,20.677,-8.2618,7.6886,-21.6615,-12.3872,-19.4599,-18.2383,-2.9446,2.2022,14.5849,17.0115,-25.7139,-18.2414,-8.8848,9.0056,16.187,24.8476,-4.0972,18.1682,-22.343,22.2332,-17.0041,4.4965,-0.26867,-7.5481,10.294,-7.0929,13.3965,3.5399,-14.8789,-24.8728,9.2067,-8.8272,25.3319,16.57,15.959,-6.8812,21.6443,-22.3102,5.7752,-15.2924,-14.5406,-3.6538,-11.613,11.2661,2.0699,13.8027,-16.2901,-24.8373,9.0484,-8.7964,21.4978,11.971,17.9087,-3.8079,23.46,-21.7718,8.255,-18.1455,-6.391,-0.69635,-2.2148,10.0519,4.8775,12.7849,-26.4334,-18.7724,-7.2652,6.7452,18.7612,26.0025,-7.2368,18.4593,-24.9746,23.4603,-16.3891,6.0726,1.9022,5.8816,-3.1676,11.0659,-6.6062,3.9065,-13.79,-23.9111,9.2678,-6.482,24.343,15.9297,12.8294,-8.6171,20.0184,-19.6409,7.8925,-11.4057,-15.6821,-3.2993,-13.1336,14.1691,2.0842,16.5843,-20.4789,2.6303,-22.8214,23.1236,-2.6844,20.0802,13.2851,22.5346,-9.792,9.6908,-24.5296,-14.4278,-8.4884,-10.1702,-0.16425,-0.47542,6.6597,8.0701,-19.9415,-23.0047,3.7185,-3.2722,23.8008,20.1557,9.074,-9.6608,15.7789,-22.0332,5.7845,-13.1087,-12.5382,0.83629,-17.4516,10.1555,-6.3009,8.0987,-21.6526,-23.0838,2.1296,-2.3127,22.7137,19.9087,14.6127,-8.5768,25.1313,-25.0679,10.0436,-15.2984,-6.8304,-2.4602,-5.5074,6.227,1.3113,8.3478,-18.3976,2.4889,-19.5722,21.4,-0.39777,19.8478,12.7111,25.2724,-13.5368,10.7218,-26.2395,-14.2336,-10.2533,-7.2026,-5.9403,1.834,0.36986,4.8177,-17.3418,-19.9993,2.6268,-2.0242,21.6035,19.2997,15.992,-11.8892,25.1553,-25.7677,9.7761,-14.7892,-10.3081,-2.6361,-7.3853,5.1672,2.6475,7.9527,-20.8555,-24.1331,2.5838,-3.637,24.2617,20.47,15.4039,-6.1949,27.8657,-16.2703,11.8825,-10.5675,-5.649,5.5722,-5.5272,17.0707,-0.42009,10.5389,-20.6306,0.22387,-20.6969,21.0876,-0.55823,20.0445,12.1057,26.5269,-14.4087,12.6643,-26.2009,-12.7489,-7.894,-1.5734,-4.83,5.564,4.3328,8.3143,-20.7928,-19.9572,0.47419,0.5162,21.0885,20.5742,11.2357,-15.0901,26.2453,-26.2572,14.1751,-11.622,-9.7224,-6.9159,-4.2222,2.7075,5.1391,9.3054,-21.0047,-23.5672,1.4609,-3.7524,22.9314,20.8667,14.8905,-12.4437,27.0985,-27.0639,12.5075,-14.6484,-5.4563,0.20707,-4.7667,4.793,1.9566,5.9629,-21.5727,-23.2607,1.1508,-1.7429,22.8961,21.9202,0.64255,0.014418,2.1023,-0.29366,2.343,0.507,-15.3711,9.275,-24.8039,24.8535,-10.6579,15.5138,-21.0345,1.7033,-23.4996,23.3366,-2.1633,21.6133,0.49787,0.59187,-0.35114,0.28798,-0.61733,-0.069017,-15.8137,-24.8771,9.0191,-9.2149,24.5004,15.3663,-20.7197,2.4135,-23.5419,23.4775,-1.7901,21.4136,0.40352,1.0384,0.045198,0.81694,1.107,1.2497,-15.4352,-24.3011,8.4889,-9.0157,22.939,14.084,-19.9939,2.6514,-23.7867,23.0576,-3.5261,20.9404,-0.51814,-0.63356,8.9891,0.51399,18.1979,10.1926,-14.2137,-22.377,8.0576,-8.8244,21.819,13.2377,-19.7914,3.8457,-23.9737,24.2202,-3.9455,20.8427,-8.7408,-17.4273,9.6754,-8.3225,18.8823,9.3338,-13.9836,-22.1614,6.9615,-8.1832,19.9418,12.6453,-23.0905,-23.8705,-2.8148,-0.87098,16.4447,19.2598,1.2411,-3.5116,10.6693,-3.1415,16.1013,7.4028,-13.5307,9.8461,-21.5458,24.8258,-5.8765,16.3373,-19.8163,3.9668,-24.1384,24.0646,-4.8633,20.2055,-9.2763,-19.313,-0.16312,-10.0128,0.0034662,-0.0014372,-12.9223,-19.4388,7.3493,-6.4817,20.722,13.2724,-19.8944,-18.2862,-0.14368,3.0956,21.2586,20.9846,-6.6505,-15.3805,3.8508,-9.2642,6.0194,1.1567,-16.5912,5.5508,-25.0746,23.0278,-8.7263,15.7401,-19.5735,4.3507,-22.9396,24.3066,-3.0139,20.1305,-0.10956,-0.31126,-12.7702,-0.432,-24.1932,-11.6374,-13.7984,-21.0749,0.73996,-7.2273,10.1786,9.6085,-20.2719,-20.7352,-0.23391,-0.6024,20.4889,18.0309,-4.9854,-5.562,-8.495,-3.4244,-7.1224,-2.7337,-14.4635,9.2477,-24.9368,24.6558,-5.411,16.8369,-20.8278,-22.7394,2.5535,-2.2045,23.2852,20.7095,11.7283,-12.6224,24.3873,-26.5745,12.5047,-13.9994,-9.2683,-1.6496,-10.6299,4.83,-0.80116,7.0439,-17.3119,-14.1022,-17.243,8.2925,-1.6052,19.9377,16.38,0.13298,-2.4341,7.8609,0.593,9.2743,2.8684,6.6107,-16.3182,-18.1327,8.1384,-16.4894,22.6339,-2.8813,16.7991,-25.4901,-14.3102,-10.9419,11.1296,14.3601,25.438,-2.5581,2.0432,-4.176,4.9739,-0.078456,4.34,2.7019,-20.6116,21.7513,-22.9337,19.8696,-0.83012,-13.0654,-24.8218,10.7127,-11.9937,24.405,13.3593,-0.20191,-2.9093,0.81511,-3.4267,1.1602,-0.37765,-21.8633,2.2579,-22.9269,22.0914,1.6999,21.1919,-20.7781,-24.8056,4.0049,-6.4639,27.2962,20.1011,12.7837,-7.2409,17.1978,-20.1039,2.4935,-12.8805,-12.3228,2.2162,-19.0847,14.6085,-4.9786,12.8971,-22.5508,-18.3282,-5.8814,7.5064,10.4983,22.5692,-5.5379,11.518,-18.0415,18.7521,-6.7865,15.3385,-6.1763,-15.0089,15.6457,-16.4466,26.5531,0.75075,-26.0211,-14.9585,-8.0586,10.0631,17.5753,25.3452,-5.1697,0.47113,-11.3794,0.66316,-6.5444,-0.36081,-0.42797,19.4737,-20.5816,22.2495,-16.6045,5.1524,-32.101,-25.2086,-10.4909,10.6567,18.3813,27.9046,-0.59,-0.28624,-0.30186,1.128,0.36962,0.97978,0.90637,14.7354,-23.1818,22.7287,-18.9792,5.2311,-24.6344,-10.9678,-12.5822,28.2404,46.0215,0.044889,-1.5978,2.3288,-1.9936,1.7692,2.3221,-23.3737,26.5,-24.7521,18.2159,-22.1385,-5.6114,-12.7273,18.2445,9.1718,23.9008,10.8819,25.7602,-11.662,14.6726,-21.4162,-9.4733,-10.1202,7.3072,-20.2273,14.373,-13.6477,4.3644,-20.9499,-17.0565,-1.6164,4.5063,20.9978,23.3165,10.8456,-15.2493,26.3218,-25.2708,15.2642,-10.1413,-10.136,-13.9751,-1.6854,-7.589,7.2611,5.2494,-22.4059,-20.1164,-3.9596,1.199,18.6278,21.3346,10.9309,-15.6463,25.7394,-26.0721,14.0367,-11.0268,-7.2979,-7.2833,3.9116,1.174,12.4468,9.6976,-26.5253,-15.5101,-11.1836,10.902,14.7286,25.0019,8.5822,-16.7407,23.111,-23.6894,11.4876,-9.8161,-6.8994,-16.4581,12.2516,-11.9105,20.1179,4.7972,-20.0771,-17.2611,-1.1657,3.8258,20.8844,22.8884,11.4838,-14.2593,26.3506,-25.1857,14.6255,-10.9988,-11.1226,-13.0448,-1.3375,-5.074,6.5764,4.6574,-24.2349,-12.206,-11.3201,14.4293,10.2318,24.7934,4.5296,25.3293,-21.7424,19.6583,-25.0763,-3.781,-2.9517,5.0449,-7.1005,7.1801,-5.2789,1.3443,-25.5352,-11.5642,-11.7436,10.971,13.5627,24.7807,7.2182,-19.9986,23.8513,-24.6141,17.0452,-6.084,0.47352,-7.7946,11.2916,-6.5593,11.8084,1.872,-24.2457,-12.5032,-9.8063,14.6683,7.717,24.3584,3.1445,25.8559,-23.815,21.5009,-26.7149,-3.2487,-5.1669,3.3815,-8.6442,6.2827,-4.0264,3.5493,-24.7685,-14.3226,-2.7699,15.8477,5.612,25.5765,1.288,27.8302,-23.854,22.8311,-25.5136,-6.2019,-9.9445,2.2024,-14.8684,10.1766,-13.9237,2.0303,-17.0411,-27.4353,12.7551,-11.5446,26.7507,15.7636,-13.6523,5.9673,-18.3869,21.2426,-5.7087,11.6977,-16.1141,-2.7888,-18.7526,17.2507,0.94182,21.5585,-33.689,-29.5943,3.8173,20.9647,27.9162,-5.1774,32.202,-22.7254,20.2038,-9.0188,-16.3223,8.1282,-16.4981,16.9951,-0.84835,-22.0791,-25.8166,4.4838,-5.7668,26.4361,20.0809,-9.7515,10.0439,-18.3371,21.4641,-5.152,14.785,-13.8271,2.507,-18.8889,15.4983,-5.2483,12.7678,-20.1404,-24.6344,4.6027,-5.9244,24.0798,18.3789,-10.5207,6.9214,-14.673,18.1073,-0.59005,14.658,-13.7959,5.6583,-20.2145,17.3377,-6.8125,11.1558,-21.3673,8.765,-24.134,22.9217,-5.6771,16.0428,8.4816,18.6165,-8.2509,9.7619,-18.1327,-8.816,-14.0372,-10.7108,-1.7315,3.7553,15.977,18.6169,-16.4082,-25.6237,9.0042,-8.9848,26.3308,18.2934,17.0004,-5.4548,22.4062,-21.8286,4.0966,-15.8539,-13.3501,-2.2218,-10.2152,11.8992,3.685,13.7678,-25.3692,-16.647,-8.3089,7.6506,15.3238,23.568,-6.2222,18.0275,-24.7896,24.3599,-17.7082,7.3492,-4.1643,2.4704,-0.86315,7.196,5.7366,7.1967,-23.5122,-15.0699,-7.4733,8.7129,15.9493,23.6233,-5.7737,17.5575,-25.354,24.6015,-17.2072,7.6553,-6.1713,-6.9507,2.9049,-0.77573,5.4045,2.8778,-17.5038,-25.1745,9.5475,-7.8795,24.1136,15.9485,17.7848,-6.8456,25.0003,-25.4887,7.9684,-18.2305,-3.2831,-3.7509,-2.111,-3.2949,1.0985,0.43613,-14.7438,-24.1419,6.6727,-8.1547,21.1183,14.7843,19.6515,-7.4638,24.5399,-25.9976,7.6293,-17.8445,-4.4829,-5.0637,-3.3175,-2.8819,-8.7694,-7.3232,-17.7211,-24.3648,6.7839,-8.297,23.7747,16.2557,17.2046,-5.272,24.7175,-24.3791,8.4882,-16.5152,5.2456,4.7477,1.9144,-2.3975,-6.0623,-9.9329,-17.414,-24.2792,6.6369,-6.8653,24.6718,17.7636,16.2914,-8.0538,24.1938,-24.6843,8.4473,-16.4984,8.2942,5.8578,5.2244,-2.2901,-1.1887,-6.3511,-24.9393,-15.527,-8.5403,9.0872,16.4756,25.3114,-7.1287,14.8651,-20.3108,21.6372,-13.4014,6.7443,-2.0092,-14.7101,14.5657,-12.2931,15.4577,1.4305,-19.2683,-21.6919,2.786,-2.9155,22.5975,19.3048,15.2314,-9.6911,24.9477,-24.9026,9.749,-15.236,-9.2077,-2.5553,-7.0491,7.4292,1.4257,9.2664,-23.9618,-14.9685,-3.6815,11.2889,14.063,24.1542,-0.27929,1.8608,2.6116,6.1893,6.1355,8.0288,-1.6694,-19.5495,21.9812,-21.4911,19.2554,-2.7656,-22.869,-11.5048,-11.8348,-9.6677,11.3945,15.3529,22.3369,-1.4652,-2.3919,-1.2364,-2.2829,-1.1546,-1.7576,-1.3908,1.0215,-23.0664,20.8614,-23.3385,22.4441,-13.3922,3.1841,-20.0115,-15.8627,-3.5298,7.618,18.0949,23.4442,-8.0533,7.0892,-20.2976,11.524,-13.3251,3.3968,8.3438,16.7614,4.9117,20.4015,-1.8278,6.9294,-20.4327,-18.44,-4.0549,3.7018,18.0042,23.5499,-3.9002,13.2896,-16.7879,20.4128,-13.9773,6.6952,11.018,2.7709,15.1694,-4.3064,7.4791,-3.5802,-23.5446,-19.8279,-6.9463,4.6075,19.9846,25.3786,-1.5486,14.1131,-12.6556,17.1497,-10.2372,3.7161,9.3851,-6.34,19.033,-14.9637,14.0925,-3.8423,-19.4949,-13.416,-6.9362,7.7076,10.8831,17.1026,-1.6659,18.6459,-22.5127,19.9474,-21.18,0.28224,-16.6243,-9.2673,-8.825,12.1721,8.4012,19.6025,-19.3514,-12.8732,-3.5804,6.7061,19.6065,22.2951,-2.349,20.9054,-21.0364,22.3677,-17.4569,3.6823,-17.5832,-7.5271,-15.4339,11.4439,-1.9655,14.2438,-23.4979,-19.322,-5.7465,2.9742,20.5276,24.9181,-3.3922,17.5616,-21.3299,21.0025,-17.6552,3.2676,-10.9915,1.9953,-13.0316,14.9486,-3.5325,11.1383,-8.5092,-9.6743,5.8616,2.833,13.9941,10.7714,6.317,-20.2483,17.3986,-22.095,13.0816,-2.53,-26.7076,-10.6113,-16.1657,8.3533,15.126,23.1431,-25.2007,-19.066,-7.4725,4.8102,20.9161,25.8602,-3.9102,18.0848,-20.1423,21.3813,-14.188,5.6407,-8.2889,3.5415,-14.5611,12.2724,-9.3801,5.6243,-27.21,-17.1832,-8.325,9.0403,18.2312,26.1562,-0.72101,-0.21884,-1.6891,0.37627,-0.80735,0.73048,-3.0104,19.1533,-24.152,22.9704,-18.8626,5.2974,-27.1047,-12.8861,-9.4328,9.6702,20.5099,29.109,-0.37015,1.4139,-0.75768,1.3761,-0.22144,0.59729,-6.0741,24.2395,-23.9898,23.4343,-18.3698,6.3657,-19.4692,-20.3146,1.8294,0.48588,22.2344,21.835,-14.8743,11.2392,-26.1861,26.3164,-11.8752,14.2905,-8.7353,-10.7207,-1.164,-4.1403,5.4436,4.5099,-14.1139,-26.0786,12.0242,-12.4167,24.5808,0.14223,-16.7116,3.6035,-23.5723,14.5453,-9.2724,11.8884,-12.1212,2.822,-11.8626,23.5355,3.2936,23.9665,-24.9049,-16.9171,-8.2687,8.1953,17.215,25.4839,6.4896,4.9729,2.7287,0.098977,-1.3638,-2.3657,3.7186,-18.6857,23.0144,-22.5905,18.9918,-3.9562,-24.9426,-16.6986,-8.1931,8.2504,17.22,25.4486,5.7125,2.75,4.4115,-1.4657,0.89992,-1.9584,3.9848,-19.0855,23.0768,-23.0219,18.7605,-4.1722,-25.1957,-16.9556,-8.0085,8.2875,17.2109,25.7478,5.9726,-0.085865,7.9604,-3.6412,3.7457,-1.7178,4.7325,-18.8995,23.1359,-23.1795,18.1954,-4.3803,-26.1194,-17.4755,-7.769,8.966,17.3618,27.2291,7.7724,-2.695,13.0286,-7.0289,7.1254,-1.5859,6.8146,-18.4001,24.59,-23.6017,16.8117,-4.808,-27.3031,-18.1243,-2.0813,12.4571,36.693,10.4501,-6.3023,23.8727,-12.8329,0.47498,12.0294,-16.8641,32.3206,-26.6218,-4.8667,-19.3496,-18.0756,-4.1832,-17.2602,16.9201,9.2208,24.3705,3.7549,-10.504,13.4952,-10.8486,10.2151,-11.5544,-3.7203,6.2662,-13.9012,18.0159,-14.7949,15.7013,-17.6425,-2.7096,-22.8599,-18.1469,-5.6728,7.1202,17.421,24.4205,2.4077,-12.7194,14.9058,-15.7341,12.4701,-4.0215,7.0813,-13.3886,18.8642,-16.8973,14.0547,-2.3098,-17.046,-25.6326,8.7618,-8.2758,25.2643,16.7085,3.7239,-0.094998,5.0129,-2.6489,2.8349,-0.90089,-18.3809,3.8393,-22.3419,22.265,-3.4323,18.6527,-16.3665,-25.2063,9.033,-8.731,25.1656,16.1936,4.5918,-1.9562,7.3673,-4.2675,3.3228,-1.7352,-18.686,3.7305,-22.2201,22.5732,-3.1781,19.15,-16.1189,-25.1091,8.7937,-9.0756,24.5478,15.6869,7.168,-2.0298,11.098,-6.1325,5.5157,-2.5241,-17.9723,3.7164,-20.6654,22.6115,-2.1449,19.5567,-16.027,-24.8738,8.1801,-8.7422,23.5085,15.4037,8.8515,-4.0844,14.8271,-9.8305,8.2477,-3.4499,-17.8291,2.6137,-18.8052,21.5585,-0.17717,19.7798,-17.4609,8.4565,-25.5141,25.5704,-8.1707,17.2549,4.4014,3.3213,3.2505,1.0667,0.55127,-0.54198,-17.9049,-21.9772,3.8437,-3.7733,22.5577,18.6412,-15.8582,-24.2918,7.7407,-8.0685,23.5404,15.866,8.5579,-6.7894,15.5909,-13.6687,8.7844,-5.0449,-18.8987,0.48688,-18.8931,20.297,0.69794,20.2244,-16.2399,-24.0991,6.8244,-7.7875,23.6753,16.232,8.7462,-7.7345,16.1701,-14.6852,8.4742,-6.0796,-19.5811,-0.35242,-18.7718,19.9712,1.1142,20.4978,-18.2295,3.4431,-24.1005,22.7181,-7.261,16.1579,12.5142,20.6382,-7.6628,10.3223,-15.3189,-6.3234,-17.6346,-15.28,-0.78172,3.2218,19.1377,20.6011,-25.1912,-16.0512,-8.6926,9.1688,30.7905,32.1564,-5.4021,17.2753,-22.8357,21.6933,-16.9466,-13.3006,-3.1813,-13.5919,10.0593,-10.2488,15.1173,13.5293,-16.3471,-25.6572,9.1505,-8.7211,26.8827,18.1834,16.0534,-6.3448,21.664,-21.8872,5.9714,-15.601,-13.9691,-1.5096,-12.1842,12.8263,0.37845,13.1449,-14.0245,-21.1398,7.9887,-8.6387,23.73,-7.3903,16.9307,17.099,-3.5478,24.2905,-22.5789,7.1133,-23.5102,-14.1422,-7.7824,-2.5361,-6.9782,10.3906,-2.3658,9.9253,5.033,-25.3981,-18.5301,-6.7273,4.7773,19.2245,24.3845,-7.8047,14.0697,-24.1074,19.0354,-16.2393,4.3854,1.3819,10.6546,-5.2496,14.8038,-3.1101,7.6335,-16.4398,-23.5727,8.6851,-6.7513,24.4751,16.2965,9.8789,-9.6459,19.2126,-18.6882,7.4732,-10.9933,-17.9244,-2.4938,-15.1194,17.4437,0.45677,17.742,-15.8026,-24.6148,9.3311,-7.63,25.4484,16.7533,12.4183,-7.8069,19.2964,-20.4741,5.5786,-14.2842,-16.6499,-1.4649,-15.8202,15.2116,-1.6502,14.311,-19.8514,3.6206,-22.7327,23.9669,-3.2123,19.8344,12.3424,21.3885,-10.1207,8.0152,-24.096,-14.7831,-9.6801,-11.5559,-1.4977,-3.6566,6.7113,6.4997,-20.0293,-22.6948,2.6525,-3.1137,22.8614,19.5551,14.486,-9.6493,24.5485,-24.6661,10.0773,-14.7874,-7.971,-1.2987,-6.5725,7.1127,1.9043,8.794,-20.1639,-23.7763,3.6318,-4.0177,23.704,19.7756,10.4857,-8.2797,16.144,-21.2464,4.7809,-13.8419,-10.7699,3.4571,-17.1895,11.1681,-7.7726,6.5157,-19.3758,1.9983,-21.1286,22.3295,-1.2776,20.1951,15.0459,26.1001,-12.331,10.5849,-26.5501,-15.3485,-7.0011,-4.047,-3.4803,2.3359,1.1262,3.9654,-20.6898,1.2809,-21.8391,22.1228,-1.5271,20.6447,13.5751,26.5646,-13.7463,11.6317,-26.7343,-13.8621,-5.9339,-1.3374,-2.8759,3.5495,4.0265,5.9565,-18.688,-21.2979,3.3671,-2.1263,22.6553,19.6123,15.1265,-11.3109,24.7185,-25.924,9.5813,-15.4153,-10.0767,-3.9057,-8.2594,1.5777,1.4523,5.3498,-20.5093,-24.1019,1.8342,-3.6882,23.4211,20.1475,15.1813,-5.5765,26.516,-16.8991,11.3706,-10.5997,-4.3729,6.6505,-3.0128,16.442,2.2989,10.3302,-20.5303,-21.06,1.7946,-0.2462,22.6854,21.1336,12.0258,-14.6944,26.4952,-26.4171,13.1473,-12.8589,-9.4719,-5.1651,-5.4384,3.2218,1.8854,6.1685,-20.7268,-22.7008,1.8531,-2.7269,22.6738,20.2784,14.0671,-11.8676,26.4725,-25.8807,12.4266,-14.4317,-6.7867,-2.9501,-4.5592,1.9065,3.2422,5.9735,-21.5937,0.958,-23.3184,23.6939,-2.0574,21.9418,0.0056781,0.62701,-0.33591,3.6557,0.61482,3.9559,-15.6386,-24.5408,8.9607,-8.3186,24.0989,14.8535,-22.0252,-22.8782,-2.7455,-2.0173,15.6479,16.461,0.79853,-1.134,8.6695,1.4373,16.5068,11.4581,-14.168,9.0269,-22.5292,23.9003,-6.2481,16.4443,-21.3142,1.7051,-23.4936,23.3725,-1.2097,21.235,-0.19961,0.40493,0.10432,1.4873,3.3321,4.0089,-15.4848,-24.327,8.9489,-8.8971,23.5124,14.3674,-21.2675,2.08,-23.3925,22.7354,-3.6077,20.466,-1.621,-1.354,7.147,2.0618,17.7381,12.5556,-14.1146,-23.0329,8.602,-9.5541,20.3339,11.5074,-20.694,2.4522,-24.1832,24.5311,-3.017,21.0483,-9.8754,-17.3395,8.3797,-5.7157,17.6025,10.7803,-13.6358,-21.8397,7.8764,-8.5899,20.9539,12.3791,-20.1191,3.3405,-24.1388,24.4461,-3.9901,21.678,-8.8695,-17.7135,-0.28282,-8.4962,-0.55265,-0.10411,-13.4602,-20.5217,7.1703,-7.5008,20.4007,12.5973,-19.5404,-16.5144,0.055547,2.876,22.7379,20.8825,-6.8371,-16.0723,5.9734,-7.7088,7.6176,2.7696,-17.1513,6.1919,-24.9583,23.4451,-7.7835,15.8788,-19.4433,3.9333,-22.9406,23.9474,-3.3484,20.4882,0.42854,0.28426,-12.5977,-0.2633,-23.9086,-11.2241,-13.4947,-20.4427,0.29315,-7.3057,9.7002,8.9524,-20.6562,-21.0402,3.4989,-1.6855,24.1606,19.8376,-2.595,-7.1301,-1.8207,-5.7012,-1.7876,-1.215,-15.9295,9.2182,-26.07,23.8839,-7.4652,16.1865,-20.2713,-22.5626,2.8101,-2.9415,23.0352,20.012,11.6886,-12.4732,24.2351,-25.7067,12.5588,-13.2435,-9.3774,-2.4847,-9.9017,4.9433,-0.19029,7.896,-18.8838,-21.5298,4.636,-4.5848,23.82,18.5459,-2.1519,4.5544,-5.4508,5.1386,1.6917,5.136,-17.2229,9.2331,-24.9091,24.37,-5.1818,16.871,-16.2562,-21.016,11.4093,-5.7903,24.2618,15.6666,-5.4799,-1.1599,-0.32379,4.3487,4.0605,5.9446,-19.1,8.1099,-23.2936,23.7807,-5.9226,19.0961,-25.6489,-13.4443,-10.3116,12.5609,13.7625,26.5789,-6.1109,-0.91325,-4.0524,2.141,2.1306,3.5614,1.6215,-21.2612,23.8801,-22.188,21.1404,0.34649,-12.5702,-25.9202,11.8993,-13.2554,24.4548,12.1808,0.076245,-1.7634,0.41026,-0.73855,0.71674,1.3665,-22.5613,0.99606,-20.8855,21.3129,3.8113,21.7041,-20.3499,-27.6075,5.3095,-11.1207,24.0573,17.803,10.1621,-5.8716,12.4811,-12.1673,6.2485,-1.2867,-14.0545,3.5495,-18.7516,19.2821,-3.3778,18.7044,-16.2554,-6.3944,-6.2029,6.7717,7.1603,11.6461,-0.90042,22.1583,-22.0812,20.0638,-22.098,-0.68753,-19.7776,-6.6037,-11.7253,15.6873,6.4582,22.6212,-16.3599,-11.4484,-10.4438,5.8753,1.073,19.5318,15.2021,-10.6121,-3.1712,-13.8673,4.4781,-1.9872,6.3893,7.1323,-10.3874,-20.0329,11.0664,-20.1802,24.2769,-4.9097,18.4256,-25.1933,-17.0202,-6.833,7.8257,18.1412,25.3417,-4.9796,2.0499,-14.2358,2.8028,-8.7109,0.16262,-3.7969,18.9996,-19.5405,23.9545,-13.6615,6.1266,-27.4217,-18.4717,-7.9379,8.264,20.3814,27.4873,-0.7459,-0.2221,-0.34589,0.33215,0.079429,0.2469,-3.9912,18.7587,-24.1089,23.7758,-18.3733,6.5566,-37.9107,-36.3538,-8.508,9.9951,22.2223,29.7221,0.61749,0.51372,-1.1292,1.4485,-0.69479,0.28071,1.3178,5.1226,-24.4701,23.5512,-17.7421,6.2983,-22.7508,-3.7777,-16.5871,20.0196,5.3372,23.859,10.5981,25.8587,-13.986,15.2006,-23.5196,-10.026,-6.784,5.9365,-17.8007,12.0633,-12.784,4.5287,-23.9005,-13.4573,-7.3137,9.91,17.6644,24.8818,9.9285,-11.4265,23.4593,-19.5216,12.4458,-9.1154,-6.4702,-22.048,13.4363,-18.4192,19.3344,3.467,-22.1968,-19.5388,-1.6108,3.0117,20.6429,23.5006,10.8079,-15.7174,26.2021,-25.7733,15.3013,-10.2847,-8.4351,-11.1859,0.32636,-5.4896,7.9016,5.0536,-22.4536,-19.7238,-2.2835,2.7075,20.2749,22.8957,11.149,-15.1753,26.1544,-26.2074,15.0707,-11.1856,-7.3728,-10.0239,2.8269,-2.201,8.4065,6.1326,-22.7138,-20.6049,-1.1331,2.4701,21.2279,22.6177,11.5606,-15.1556,26.6982,-26.489,14.7434,-11.6192,-5.4124,-7.4843,1.5937,-1.7936,7.1252,5.7712,-25.4003,-11.1225,-14.1146,15.82,9.4285,25.4451,4.9559,23.9849,-20.5723,17.7153,-24.4601,-4.8957,-0.62081,8.8052,-6.966,10.1401,-4.6494,3.4497,-24.9002,-10.6788,-15.3656,16.3076,9.0569,25.8461,3.5754,24.3533,-21.0329,19.9218,-24.1636,-4.4343,-1.0038,5.6217,-3.1144,8.1901,-1.7848,3.0062,-25.5869,-13.6582,-16.0128,10.7581,14.9107,25.4998,6.5272,-17.9009,17.0113,-23.4504,8.2289,-7.9651,0.054193,-10.6772,12.8067,-9.7361,15.0012,2.9607,-23.5647,-8.9928,-16.4794,17.4239,7.9966,26.4167,3.8837,24.2816,-16.0244,20.9135,-21.0681,-3.1987,-6.1502,-0.43491,-12.8997,0.68185,-12.3911,-5.0916,-26.5767,-22.1494,-9.0971,6.2704,19.1071,27.6154,5.3097,-12.4453,22.2263,-20.1366,16.5624,-4.347,-3.0713,-14.6499,10.8694,-17.2479,11.2173,-3.763,-14.334,-25.6736,25.7752,-9.9474,26.7286,-8.8365,7.8238,-17.8269,18.5212,8.4105,-24.4773,-3.1392,-22.8581,20.3172,23.2949,-20.5117,-26.3513,6.2247,-8.1796,26.0937,18.7651,-12.9553,5.3516,-16.8855,18.7839,-0.9979,15.7834,-12.0895,4.4801,-18.7472,15.9718,-6.7166,10.8376,-18.407,-24.6036,7.3625,-7.1901,25.6533,17.5868,-14.3158,2.2275,-16.7648,16.1479,-2.5168,14.3919,-12.0517,7.5914,-17.6664,18.1378,-4.0535,12.599,-17.041,7.5299,-23.5401,22.9703,-7.0287,15.6653,7.8104,18.4619,-9.5409,10.1455,-15.9273,-6.6573,-19.1988,-16.1231,-1.765,2.1439,19.4009,20.3452,-17.6382,-27.444,8.6246,-8.6402,29.998,20.8148,18.7391,-5.8101,22.7352,-22.3976,11.0707,-15.3408,-13.7443,-1.5836,-10.2346,11.8671,0.521,13.116,-24.0367,-15.6885,-7.4785,7.7989,16.0526,23.612,-6.8934,18.1395,-25.6402,25.2139,-18.0583,7.0099,-4.9608,-3.9018,-1.3929,-1.8746,-1.1926,-2.6092,-23.9379,-16.1448,-7.8085,7.5601,17.6931,24.5629,-7.0833,17.9849,-24.7517,25.5995,-17.4444,6.6164,1.5738,-1.485,6.0485,-2.7138,2.0127,-3.3343,-15.9588,-23.7474,4.1218,-7.8009,21.929,16.7399,18.0292,-6.9834,25.489,-25.3871,6.6635,-18.0051,1.7834,1.4385,-5.1959,0.27917,-9.9245,-4.0521,-12.8213,-18.9061,7.5661,-3.8791,22.9839,17.6849,16.7498,-7.6444,23.1549,-23.0391,7.3914,-15.9202,11.1669,12.5191,5.2396,9.2828,-0.1588,2.2709,-16.9094,-23.8218,6.5745,-5.8186,23.9601,17.7927,16.1961,-6.6397,23.9532,-25.0599,7.6122,-17.4576,8.3524,4.6932,4.6567,-1.0654,0.43258,-1.3947,-17.9446,-24.3381,5.8281,-6.0956,24.116,18.3548,15.6601,-7.4549,22.8251,-24.6032,6.6628,-17.1851,6.4383,1.6944,8.5398,-2.7181,7.1395,0.61484,-25.8068,-16.5903,-8.6673,8.922,17.5975,26.333,-5.895,15.1429,-20.5407,21.1184,-15.5328,5.6667,0.36251,-14.2567,15.3406,-13.9348,12.726,-2.2602,-19.4838,-22.6719,3.2959,-4.1447,23.2303,19.1849,14.9171,-9.4825,24.8426,-24.0027,9.6061,-14.7855,-8.5384,-1.007,-6.8529,8.4271,0.83091,8.5719,-25.4723,-13.8425,-8.6982,14.8457,13.2178,28.7395,-5.8117,-3.9764,-0.37909,2.7585,2.5752,4.5351,4.0382,-21.1146,26.6559,-24.3736,21.3665,-3.1303,-23.2939,-12.9295,-10.282,10.8689,11.1743,20.6737,0.74435,0.88583,-0.0090412,-0.58373,-3.4147,-3.6156,-1.5587,18.8305,-24.503,22.6641,-21.3669,6.5358,-21.7968,-16.6111,-5.0906,7.4378,15.5019,21.5585,-7.0877,10.839,-18.9959,16.5717,-12.6646,6.1352,6.1342,12.6111,5.6449,17.9471,4.3919,11.1983,-22.6116,-18.4592,-5.2521,8.7757,16.2057,21.6372,-3.1724,12.3007,-14.9597,20.6894,-9.1427,10.6926,7.4846,-2.0647,16.2606,-3.6622,15.4607,3.8309,-25.2941,-20.1265,-5.526,6.5123,20.117,24.5967,-2.2438,11.2436,-9.8574,17.5444,-3.2456,7.2601,4.6591,-12.4183,20.2041,-14.2542,16.585,2.8438,-17.5892,-10.8996,-2.2783,5.4721,19.2706,19.0474,-1.1877,21.799,-20.1869,21.6711,-17.8647,0.49269,-19.8451,-7.8772,-16.5823,15.6981,-0.39662,18.5618,-22.5808,-20.0169,-3.6889,0.49673,21.6267,24.6569,-3.5456,16.5046,-19.0797,18.7463,-14.7698,3.37,-12.9831,2.3385,-16.026,17.686,-6.2806,13.1153,-25.3377,-21.2292,-4.8626,3.5881,21.4272,25.9677,-3.726,15.8121,-19.6253,19.8171,-13.6641,6.1122,-8.7399,4.8877,-14.453,14.7903,-9.0595,6.5267,-9.6047,-3.6577,-1.5784,5.4902,4.6676,5.7577,2.0312,-21.1392,22.6104,-21.881,20.5589,2.5922,-24.0405,-16.5059,-8.3609,3.0122,17.0215,23.5423,-3.5503,-9.193,13.2747,-1.6798,11.4799,2.8529,0.78573,-20.0804,17.461,-22.1975,17.7571,-0.59821,-26.1924,-14.7822,-7.3585,8.5736,17.8902,26.4644,-25.6934,-20.9976,-4.9112,4.4011,20.6096,25.5651,-4.8473,13.9846,-20.0482,19.6865,-14.5741,5.6933,-5.4237,8.8288,-13.417,14.1211,-7.0387,5.8451,-26.0569,-18.2183,-7.9295,6.8071,19.0755,25.1416,-1.9954,-0.38459,-4.4346,0.058817,-3.5594,-0.62497,-5.092,18.8115,-23.5885,23.8799,-17.209,7.3559,-28.7739,-20.601,-7.0793,7.5716,22.4269,27.5438,-0.11071,0.53194,-0.2591,0.97199,-0.44678,0.14562,-5.406,18.1044,-24.8381,24.2708,-17.7969,8.3333,-25.348,-12.0549,-7.9203,7.4819,25.5741,27.5978,0.30906,1.3998,-1.0395,-2.569,-4.0923,-3.6708,-10.1589,21.3189,-25.3877,24.207,-18.1536,-11.4147,-21.1013,-21.5888,1.4399,1.6756,22.2413,23.1303,-14.823,12.1267,-26.8741,26.7875,-12.5136,13.799,-7.0373,-7.7317,-1.4604,-4.8384,4.7899,2.2377,-8.568,-25.9175,11.5429,-25.8833,25.8781,-4.7413,15.471,-7.7389,5.9386,-12.3737,5.987,-5.6397,14.4103,9.2041,-16.1004,-1.1084,-15.4013,-1.3649,0.09784,15.3234,16.1486,-25.4654,-17.3132,-8.2518,8.3527,17.2366,25.425,2.8295,1.8145,2.1642,0.47033,1.3753,0.73975,4.03,-18.6015,22.921,-22.5609,18.672,-4.1038,-25.5075,-17.1968,-8.5551,8.2841,17.0053,25.5036,2.7413,-0.20581,4.422,-1.1548,3.4615,0.86899,4.4191,-18.7487,22.7126,-22.9869,18.2341,-4.3574,-26.177,-17.1927,-9.9652,8.6903,16.645,25.7743,3.2832,-2.7042,7.8348,-3.7322,6.4438,0.67975,5.4064,-18.3323,22.1871,-22.8576,17.4318,-4.6781,-29.8189,-19.4328,-17.2639,16.5953,9.4252,26.4585,5.7075,10.3737,-6.2489,9.4956,-7.6375,-0.27697,9.8101,19.6615,-17.379,15.7351,-22.1508,-4.5663,-24.0402,-6.9418,-16.7237,16.4231,8.721,24.8376,5.519,16.9865,-7.985,13.5019,-10.7795,-0.61229,1.196,15.7389,-16.9364,13.1539,-20.3573,-4.7663,-24.7835,-17.4787,-8.0625,8.7261,16.6878,25.7534,4.2947,-12.965,17.2346,-16.9147,14.1514,-2.9164,3.6117,-13.429,16.1406,-16.2855,12.9678,-2.6473,-17.2946,-25.6076,8.6363,-8.168,25.1359,16.7765,1.5821,0.071652,3.2752,0.15506,3.2585,1.675,-18.4917,3.9197,-22.2536,22.7707,-3.6559,18.7258,-16.5333,-25.3448,9.1163,-8.3817,25.2088,16.3259,1.4326,-2.3743,4.0609,-2.5749,3.2833,0.45328,-19.1409,3.6009,-22.6605,22.6835,-3.4062,19.081,-16.09,-25.31,9.2733,-9.0388,24.8077,15.6292,2.8097,-2.3257,6.5,-3.2306,4.889,0.33102,-19.4333,3.5105,-22.5634,22.9504,-2.8359,19.6443,-15.8055,-24.9603,8.7814,-9.2632,23.7823,15.0319,4.3148,-3.9758,10.8066,-5.624,8.1108,0.040495,-19.4136,3.0067,-21.2267,22.9326,-1.4304,20.2851,-15.44,-24.1851,7.9892,-8.5763,23.0672,14.8063,5.4211,-7.3567,13.8522,-10.1257,9.806,-1.6685,-20.1199,1.4469,-20.196,22.0743,-0.1188,20.8049,-15.7529,-23.8551,7.7982,-8.0591,23.178,15.2999,5.9009,-8.8073,14.3809,-12.9623,9.6201,-3.0412,-20.6691,-0.028355,-20.1309,21.101,0.47165,20.9773,-16.4143,7.3822,-24.0302,23.0128,-8.1025,15.3229,6.8446,15.8643,-8.3014,9.8125,-13.1035,-4.0354,-20.2519,-18.8361,-0.36692,1.1913,21.1209,21.3381,-39.4152,-23.5177,-7.9229,30.219,21.7435,-5.7886,35.8794,-23.0802,12.1772,-14.1147,-4.8827,-23.9199,10.6878,-4.8331,7.9754,-18.2425,-27.1054,8.0212,-8.5666,27.6842,19.771,16.4262,-5.5174,22.6368,-21.9371,6.4921,-17.225,-12.2455,0.78989,-11.2645,13.2789,-3.3519,8.6719,-15.9215,-24.3738,6.6748,-7.9442,25.2439,18.9949,17.6384,-8.6225,24.0093,-24.2283,6.2974,-16.7688,-3.6827,6.6785,-7.7185,8.3477,-4.2585,1.7983,-25.0527,-19.4733,-5.6041,5.02,18.9049,24.5799,-6.5858,15.6627,-23.6546,20.7717,-16.7347,5.5282,-5.4297,5.7836,-7.1791,12.1688,-0.77083,7.7362,-16.219,-23.7265,8.7418,-7.5887,25.153,16.3564,8.5427,-9.3044,16.8393,-15.8354,4.913,-9.941,-19.5875,-0.76773,-19.2921,20.66,-2.3929,18.7614,-16.8451,-25.0383,8.8125,-8.5105,25.3879,16.4863,13.0009,-5.846,19.4046,-18.0471,4.5315,-14.4858,-15.6684,1.5501,-16.7343,18.3542,-3.5348,14.7805,-16.7812,-25.5657,8.6321,-8.7215,25.5127,16.8498,14.9621,-5.1204,19.7469,-20.3637,4.0848,-16.4496,-13.6904,2.4233,-16.294,15.6699,-5.0057,11.263,-19.9492,4.2552,-23.9476,23.6323,-4.2879,19.0747,13.2312,20.8154,-8.8671,6.7449,-23.697,-15.3021,-7.5114,-12.1483,2.2599,-6.5668,8.1803,4.026,-19.6491,-22.8511,4.1132,-3.6941,23.6101,19.2161,14.0187,-10.0552,23.6502,-24.7142,8.9422,-15.6358,-8.6155,-1.9411,-8.7597,5.3114,-2.8908,4.7596,-20.0158,-24.0308,3.9737,-4.3524,23.1305,18.6152,12.1182,-6.9735,18.2416,-21.179,5.9179,-14.8795,-8.6694,5.6326,-15.068,11.4337,-8.1374,4.4941,-20.2032,-22.4188,2.0905,-3.1348,22.4672,19.5168,15.3391,-11.5877,26.1366,-26.2829,10.2597,-15.2688,-3.9137,-2.2261,0.10591,2.9329,4.8686,5.9968,-20.8452,2.2237,-22.4613,22.8654,-2.5577,19.5521,14.9184,26.9361,-13.0695,11.2145,-26.3992,-14.4936,-4.6757,-1.7393,-2.0679,3.4981,4.7608,7.4251,-21.1583,-22.8092,1.8166,-1.6368,23.1559,20.6007,13.587,-13.9286,26.9316,-26.7414,12.636,-13.3846,-5.9119,-1.3301,-3.8886,5.712,3.2939,8.1157,-18.5107,-21.7028,4.2563,-2.5096,23.5691,19.7838,14.4214,-10.7213,23.3986,-25.6536,8.6147,-15.4883,-9.8794,-5.4557,-9.7752,-1.098,-0.8489,3.5144,-20.6942,-24.1212,1.6109,-4.2849,24.0032,19.5978,14.51,-7.6419,25.6225,-19.8676,10.4004,-13.3006,-4.2055,4.7379,-0.18457,12.6037,1.9358,5.1073,-19.7323,-21.8892,2.3343,-4.3872,22.6941,19.5901,13.28,-11.1909,25.2444,-23.1861,11.6349,-13.056,-8.1854,-5.7889,-5.3955,-3.5204,0.73204,0.24339,-21.6766,0.434,-23.5979,15.1001,-5.4435,11.5625,-5.093,3.7698,-4.5037,15.8789,5.6776,16.754,-15.5817,-24.0809,9.1647,-8.2554,22.3665,11.6197,-21.3151,1.6561,-23.5509,18.4801,-4.8779,12.8689,-2.1022,0.5478,0.26293,11.8294,10.3293,17.6908,-15.2019,-24.0182,9.3902,-11.54,21.0976,9.3122,-18.5238,-16.895,2.1894,-1.2331,23.1194,18.6176,-8.7979,-15.2279,4.591,-1.9811,9.6296,9.2424,-17.197,7.6366,-25.8416,24.7315,-9.1655,16.3637,-21.5257,2.3159,-23.7844,20.7202,-5.3752,16.0895,-5.3939,-4.2718,4.6552,7.073,16.0123,17.0644,-13.9583,-23.7576,9.1528,-12.4203,19.4581,8.8807,-20.9234,3.3184,-24.5547,25.5586,-4.4952,20.5313,-12.0223,-17.2108,5.1706,-3.4895,14.0012,10.1555,-11.744,-20.6602,10.1323,-8.2109,22.8565,13.1191,-21.4291,3.5064,-24.2911,25.5115,-1.1391,21.5771,-9.1063,-15.9509,-0.81991,-6.2276,-0.29088,1.2062,-13.4531,-22.5114,7.37,-8.5763,20.8594,13.1249,-19.9572,2.6131,-23.1077,23.2412,13.6799,24.2101,0.81595,0.92401,-11.9533,1.9108,-32.3687,-26.9652,-13.095,-19.9673,1.333,-7.9198,16.6309,15.3263,-21.0524,-23.0719,4.4754,-3.5707,22.8816,16.0372,-1.9164,-7.3606,3.8166,-2.7714,3.8705,2.6094,-16.1003,9.9628,-25.18,25.4649,-9.9406,17.9387,-19.6868,-22.3463,2.6195,-3.546,22.6183,18.3202,22.6182,11.3499,-12.0009,23.1773,-24.5665,12.1186,-13.1509,12.1187,-9.51,-2.5006,-10.2854,3.176,-1.8514,4.9145,-1.8516,-19.5758,-24.313,5.9321,-5.7945,28.6306,24.1838,-3.3509,-2.6016,-3.3967,-2.5811,-0.48467,-0.080644,-16.9049,8.9935,-25.1658,24.3196,-4.3561,11.2613,-18.0297,-24.4779,8.8286,-6.9346,26.1048,17.6372,-6.0404,-1.7393,-4.0492,0.98808,-0.01226,1.0233,-18.5989,8.2948,-24.4613,24.3555,-3.2599,20.6009,-15.5636,-24.4952,18.6342,-8.7717,26.2282,13.662,-3.6814,-1.8955,-0.12991,1.578,1.3589,2.0783,-20.7333,5.3653,-28.8971,24.6728,-16.6208,21.8367,-28.6082,-13.3235,-15.0409,14.5491,13.0599,28.261,-4.8891,-2.8229,-3.1128,-0.55124,0.089756,0.34897,3.2332,-21.6594,23.9884,-21.2291,20.908,3.1989,-11.5253,-26.7299,11.6655,-16.0035,23.5643,11.0124,-1.6026,-0.81526,-2.2255,0.065132,-2.3194,-0.60869,-22.6417,-0.47498,-18.9525,18.6354,5.6915,21.705,-18.5313,-23.6783,6.8744,-4.003,20.9111,15.7036,3.6089,-3.1914,10.1721,-0.29452,11.3371,7.3375,-16.8263,1.4758,-20.4071,23.522,-6.7325,18.4294,-12.6324,-7.9884,0.95442,1.4564,20.2128,18.4553,0.4269,21.8267,-18.1753,20.745,-16.9691,-3.1332,-22.1563,-6.1154,-20.4648,16.7733,-2.8785,16.8012,-14.3476,-14.8249,4.8869,-5.2104,21.2916,13.3122,-15.4739,-16.2639,-1.3961,0.78518,8.8878,13.2329,-16.9183,8.6911,-25.232,24.95,-4.7286,19.0715,-25.579,-17.4121,-6.224,5.7676,20.6807,24.6234,-7.8608,4.2719,-22.3787,4.9555,-12.9225,-1.7501,-4.2856,17.4619,-13.3102,22.9521,-4.994,6.6442,-27.1885,-20.4079,-5.9799,4.7447,22.1062,26.0931,-0.13825,0.012201,-0.7172,-0.024286,-1.5798,-1.0394,-5.5066,18.2816,-24.622,24.283,-16.3966,7.335,-29.7939,-21.8018,-5.1426,7.2099,24.9433,29.0707,-0.2772,0.75882,-1.2008,-0.91916,-4.207,-4.6859,-6.5287,19.0895,-26.204,25.3163,-16.9422,9.7718,-23.0313,-18.9823,-3.1064,5.3265,20.5489,25.1396,10.5727,-14.2939,25.6337,-22.8674,15.2135,-8.3796,-6.5738,-14.642,3.647,-12.6909,8.9692,0.47996,-23.0016,-14.9344,-0.84533,-4.8767,25.3407,20.3453,11.0738,-9.4058,-14.3975,24.3198,-3.1222,15.8034,-6.5504,-23.7842,-21.93,10.1133,-2.0868,11.5224,-23.3504,-20.3942,-2.7992,3.1252,19.7617,23.0301,10.9062,-15.349,26.1996,-25.7003,14.9071,-10.8919,-5.7006,-8.7006,2.7984,-3.4402,9.7995,6.6587,-22.745,-20.0965,-1.4053,2.773,20.4265,22.5792,11.4183,-15.0141,27.1551,-26.4592,15.6657,-11.5728,-6.6129,-8.931,1.3042,-1.8406,8.2711,7.6496,-22.6678,-21.2124,0.066755,1.7988,22.2012,21.8794,12.1307,-15.5359,27.8079,-27.6616,15.0771,-12.3878,-5.6648,-7.0755,1.0141,0.34741,7.5109,7.9723,-26.5782,-11.9534,-15.8361,17.2158,8.6816,25.8268,5.735,21.0302,-19.3419,15.147,-24.2199,-5.4519,-0.40399,12.4059,-8.0666,12.1012,-4.1935,3.0545,-25.7362,-9.6091,-16.5074,17.2512,8.7094,26.1956,4.5383,24.2375,-19.0018,18.883,-23.4078,-5.3242,-3.5407,4.6169,-8.5873,8.8957,-5.4115,3.8489,-26.4808,-9.3387,-18.1287,17.9867,8.6393,27.5815,3.25,23.4362,-19.1456,18.6438,-22.943,-5.2728,1.103,5.5811,-4.3044,9.0773,-7.817,1.0964,-26.4536,-13.5464,-10.6292,8.971,16.3981,26.2365,4.1448,-13.0651,12.201,-17.1987,7.7124,-4.9969,0.40014,-15.3665,16.9398,-16.0683,16.2623,-1.008,-27.1037,-19.4358,-8.2881,7.5525,18.746,26.7781,4.5058,-14.4369,22.3443,-17.6238,18.4109,-1.2869,1.1737,-12.2088,9.396,-17.1373,6.1391,-6.1766,-18.9036,-26.3176,8.2672,-7.8014,26.367,17.4414,-15.15,1.8404,-16.9771,17.5399,-1.7611,15.7432,-11.4212,6.2997,-17.5973,16.8858,-4.9838,11.7634,-25.8162,-7.6468,-16.5334,16.3256,9.1218,25.0606,2.0587,11.7053,-7.7223,9.1766,-7.7871,-0.14845,1.9074,19.646,-16.877,16.3665,-21.3883,-4.9159,-17.5908,-25.854,9.3251,-8.7628,26.1634,16.8023,-14.7178,2.5852,-18.0286,16.8974,-3.4317,14.528,-12.0871,4.9982,-15.5851,16.9576,-2.8815,12.8544,-15.9841,8.0612,-23.7728,23.1021,-8.2127,14.9443,6.1148,15.9617,-8.5959,9.8131,-12.3378,-3.7715,-20.5922,-19.5775,0.020133,0.30557,22.0627,21.4837,-22.0751,-28.836,7.195,-7.7892,26.3819,18.963,12.0096,-7.296,21.0597,-24.0044,5.8785,-18.8278,-11.2088,-1.8515,-13.5522,8.1689,-10.6044,1.4325,-23.6128,-15.8111,-7.4581,6.2737,16.331,22.9274,-7.5412,18.3772,-25.54,25.0471,-19.3899,6.7606,3.9064,2.3301,3.4839,-2.0156,-3.184,-7.3995,-22.8287,7.4432,-28.9385,24.6377,-6.9332,17.1781,11.333,22.7647,-6.9449,6.3208,-22.9851,-17.9443,-5.0922,-11.7115,5.5768,-10.1551,11.0144,0.12881,-24.3678,-17.1048,-6.4565,5.1245,18.7027,24.4763,-6.5649,18.9236,-23.6495,24.6275,-17.3318,5.0909,5.7058,-0.14954,9.0429,-6.9518,4.9218,-5.3922,-18.0499,-23.0137,5.3846,-7.2834,24.2017,17.8859,16.3756,-7.0421,24.245,-24.3629,6.9481,-17.4993,0.64165,6.0677,-6.3338,0.80819,-5.0164,-3.6824,-17.1652,-23.0695,6.1489,-5.8247,23.7791,17.1663,17.6685,-8.3248,25.1768,-24.1922,7.8692,-17.3432,1.2242,4.0628,0.86895,5.2214,0.65632,2.379,-17.8645,-23.9416,5.9719,-6.4691,23.656,17.6199,16.991,-7.7192,24.191,-24.8731,7.5989,-16.8366,2.2855,0.52574,5.7991,0.46374,6.4832,2.7316,-18.5119,-24.3267,6.3777,-5.7208,24.515,18.6208,16.9918,-7.0059,24.0262,-24.4616,8.5209,-16.3772,0.03965,-6.414,6.507,-5.6977,4.2942,-1.5049,-26.2297,-17.7664,-8.5702,7.0794,18.442,25.5908,-4.2141,16.2704,-20.6655,20.462,-16.2679,5.0298,3.568,-11.0187,15.2967,-14.9635,10.1037,-6.0443,-19.5362,-23.4412,4.3339,-4.6836,23.3868,18.3459,15.1081,-8.6131,24.543,-22.9639,9.6576,-14.2474,-6.1423,1.2032,-6.7998,9.8974,0.26414,9.6564,-31.8521,-13.6986,-12.5659,21.1812,39.8903,-2.4453,-1.4223,0.077296,1.3344,2.795,7.5306,-20.8581,38.6038,-24.7008,5.3622,-22.5344,-9.64,-12.8268,-2.2399,7.9876,-2.6199,13.1174,2.8093,-3.1069,2.6144,-14.6085,0.15608,-25.5186,-13.9597,-3.7116,-25.2524,16.4305,-20.047,21.4972,-0.88414,15.3716,-18.6562,-11.0337,-6.1746,-14.2158,-0.34207,-14.9757,13.2957,-2.9794,5.9837,6.7225,6.0488,-4.3058,1.1207,-14.5947,0.67898,-15.1917,-2.8401,18.7166,-24.5208,20.9632,-28.0684,8.049,-21.4593,-7.8708,-24.2776,-15.3345,-4.6278,6.31,13.7676,18.4152,-8.0247,16.2434,-23.4484,24.5004,-12.8686,15.4343,-0.72966,2.112,9.583,6.7593,15.1286,9.8343,-22.718,-17.0094,-4.8514,4.9268,7.9453,17.2448,-8.4018,12.7559,-18.554,26.0767,-12.0046,13.7438,-1.2065,-13.414,15.7012,-5.7182,19.8988,12.1458,-26.9914,-19.7453,-4.2811,9.9047,16.3428,-6.9265,11.2939,-6.1438,27.0502,14.3106,3.8222,-18.545,25.2711,-8.2493,18.6459,-15.5288,-20.089,5.8115,-7.209,20.804,13.1882,-6.0947,-5.7362,1.9881,3.235,10.7804,12.0983,-18.6123,6.8064,-24.1057,23.1194,-5.7807,18.4741,-19.7673,-20.0018,-2.0584,0.045309,20.6433,23.5769,-0.8816,17.6826,-13.6618,19.0386,-10.1143,4.4275,-17.2894,1.903,-19.7638,20.0422,-8.0781,13.9388,-25.2148,-22.5985,-4.3128,2.867,21.0508,24.6408,-5.0472,10.2918,-18.3099,14.2283,-13.9543,2.5883,-8.5097,6.6484,-15.7108,19.7944,-7.3734,11.8678,-25.8283,-21.6549,-4.8754,3.7705,20.7749,24.9567,-6.0223,12.1977,-19.726,16.9597,-15.1611,2.5669,-5.0616,9.1841,-13.4581,17.0607,-5.7792,9.9193,-6.9466,3.8007,-6.6173,15.6401,-5.2871,17.4921,4.4193,1.136,-9.3358,18.1018,-10.5319,19.2828,-2.3445,8.5375,-19.7855,-20.7482,-5.6268,-10.5787,14.7339,9.589,21.8265,-2.9878,0.85951,1.3516,5.5028,0.4495,0.81268,1.6386,-20.694,21.8561,-18.2337,21.6707,2.2292,-25.5951,-17.9156,-8.8508,10.5509,16.8484,25.98,3.345,-4.2045,14.6686,-3.4824,8.423,-1.8724,-1.4932,-21.2579,17.9042,-21.6385,19.4933,-0.33258,-26.5813,-17.6137,-7.832,9.9211,16.3422,26.7167,-25.6296,-20.6897,-5.3115,4.5231,19.47,24.5962,-6.1976,14.0328,-21.6247,18.7857,-16.0231,3.6316,-5.1301,7.6129,-10.3829,14.7404,-2.6628,9.6746,-24.8431,-19.1617,-4.7116,4.0482,18.0312,22.7603,-1.3961,1.3422,-8.1418,0.66031,-10.7495,-4.1536,-5.7431,17.2661,-23.2456,24.0619,-16.6849,10.1875,-27.1715,-22.4013,-4.1529,3.9697,22.9304,25.0938,0.47107,1.0552,-0.43804,-0.69463,-1.9957,-2.702,-6.4861,17.8707,-24.7579,25.9443,-17.126,11.7714,-32.7841,-21.4231,-4.2735,23.4908,23.8871,5.0114,-2.3779,0.27145,-15.517,-12.9636,-7.1627,35.9356,-27.4184,15.3999,-17.3422,-22.308,-21.4935,-0.39453,2.7434,21.4715,24.7021,-15.2943,12.9425,-27.9028,27.5276,-13.2983,13.2724,-5.8041,-7.9497,0.65756,-5.2535,7.0435,3.6473,-15.2489,-24.6368,9.6736,-6.8634,25.167,17.6535,-6.5613,8.3454,-10.8572,19.0194,-3.4659,11.7078,-17.0376,-2.8388,-17.575,15.5099,-2.2817,15.955,-20.5994,-25.6718,9.1658,-6.2154,23.8966,17.6374,-11.2881,7.064,-17.5496,21.3034,-8.6856,12.174,-13.2367,1.1685,-13.3098,14.0757,1.8628,15.9645,-25.5207,-17.257,-8.4801,8.6369,16.8679,25.2609,-0.22055,-0.89481,1.6472,0.5688,3.0832,2.656,4.3149,-18.6669,22.7558,-22.863,18.3681,-4.2688,-25.6956,-17.0235,-9.0264,8.5459,16.4986,25.2132,0.27072,-2.5162,4.3736,-1.1317,5.357,2.6218,4.6236,-18.3983,22.3917,-22.8179,17.6863,-4.6573,-26.0702,-9.9251,-16.7479,16.317,8.9853,25.2533,1.1937,7.9993,-5.0534,7.9352,-4.0756,1.8714,4.719,21.4433,-17.7136,16.8279,-22.4542,-5.0928,-25.1492,-9.0584,-16.4099,15.7994,8.8369,24.8395,3.4128,16.6412,-11.3707,15.4784,-11.0334,2.4783,3.1316,16.0196,-15.3546,11.7571,-20.8295,-6.723,-25.0466,-8.4278,-16.4036,16.1876,8.6743,24.5505,1.603,11.983,-7.7641,12.3317,-6.9584,2.86,3.9908,20.1208,-16.9825,14.7698,-22.2865,-6.3511,-26.013,-16.6951,-9.3385,9.7963,15.9397,25.8329,3.8772,-13.7248,18.2899,-15.0089,15.5602,-0.12343,2.6174,-13.3724,15.248,-17.8686,11.7607,-5.4116,-16.7542,-25.4134,9.0305,-8.4773,25.0977,16.3092,-0.68358,-2.2438,2.4963,-0.19758,3.8082,2.6732,-19.0276,4.1372,-22.7862,23.0733,-3.842,18.8714,-16.17,-25.3564,9.4891,-8.8652,24.9814,15.7983,-0.052715,-2.9496,3.6394,-1.7346,4.5937,2.1347,-19.5442,3.7052,-22.9841,23.1241,-3.3366,19.348,-15.5446,-25.1183,9.5094,-9.3601,24.4681,15.199,0.73494,-3.9704,6.5971,-3.0407,6.7047,1.8117,-20.0966,3.3398,-22.8237,23.313,-2.6021,20.0446,-15.206,-24.5413,8.9152,-9.2414,23.4383,14.6114,2.0149,-6.6375,10.7581,-6.5254,9.2476,0.58511,-20.6439,2.5385,-21.9481,23.2225,-1.4285,20.7677,-15.0081,-23.7215,8.1185,-8.5748,23.0634,15.1561,3.1563,-9.0896,12.9787,-10.646,10.0914,-1.301,-20.9822,1.1432,-21.1047,22.5607,-0.52037,21.1096,-15.0756,8.4499,-23.4184,23.2622,-7.6679,15.2599,3.4834,12.6313,-9.9375,9.0105,-12.9665,-3.3192,-21.5926,-21.3582,0.12743,-0.43643,21.8371,21.1791,-19.895,-26.8504,6.7858,-6.7981,25.8976,20.0067,17.5123,-3.237,22.7396,-20.0467,6.7799,-14.5458,-4.817,6.2916,-11.3297,15.8462,-5.8155,9.8339,-16.6935,6.5615,-23.3762,24.4056,-5.288,18.363,17.9075,24.429,-4.4704,9.2718,-20.4295,-12.943,1.8074,-3.1643,12.0532,2.7948,15.5244,11.3484,-18.7874,-25.5564,6.032,-6.0097,25.3977,18.5991,17.3522,-7.6602,23.5846,-24.3317,6.383,-16.0526,-4.7844,1.366,-8.8495,7.4051,-2.017,7.8481,-25.2827,-19.2684,-5.7197,5.9387,19.1942,24.1797,-7.2915,16.0965,-24.0875,21.7922,-15.8288,6.8631,-5.7362,2.6467,-7.9307,10.5054,-2.1713,7.8766,-15.5134,-23.8075,8.8237,-8.0496,24.7595,16.2878,6.0223,-8.3868,11.701,-13.4238,2.0858,-8.9535,-20.9655,0.7457,-23.057,22.3561,-4.7439,19.453,-16.6425,-25.1801,8.525,-8.4207,24.864,16.3201,13.1341,-1.76,16.5793,-11.8456,1.9603,-11.6103,-15.9613,4.2957,-19.6864,23.5683,-5.2329,17.8844,-16.6664,-25.0113,8.4032,-8.7235,25.4087,16.651,15.8322,-2.0935,19.5588,-16.7358,3.2329,-14.9862,-12.1351,4.9058,-16.5995,19.2627,-4.9523,13.683,-16.9425,-25.253,6.7937,-8.6708,24.2348,17.1851,16.5807,-3.1092,20.4493,-18.991,5.7158,-15.1403,-10.2911,5.3365,-14.8481,16.7319,-4.4751,12.3799,-19.2799,4.5883,-23.9393,23.1882,-5.3125,18.3304,14.4342,20.9748,-8.0841,4.7772,-22.8741,-16.7749,-4.8669,-12.0914,5.7345,-9.4558,9.9767,2.1198,-19.3222,-23.5501,4.8645,-4.9189,24.7983,19.5383,14.7548,-8.9072,23.3862,-23.4177,7.5727,-14.9978,-5.2896,2.0402,-9.7605,9.1201,-5.1914,6.0753,-18.8456,-23.2034,3.6338,-5.1347,22.0096,15.5059,13.9496,-6.0953,20.1084,-21.6824,4.7103,-17.9975,-4.4664,7.5973,-11.5666,10.3639,-11.2639,0.022317,-19.7306,3.4944,-23.2803,23.6151,-4.8119,18.9087,15.0684,26.3419,-12.3232,10.6098,-25.9578,-14.7334,-6.4309,-3.3708,-2.1639,4.4107,5.7304,9.3506,-19.9576,-23.0112,3.0027,-4.6197,22.4476,18.0381,15.3939,-11.0731,25.8621,-25.4524,10.2514,-15.0765,-3.7706,-2.626,0.96118,4.6284,6.8664,9.4252,-20.5982,-23.3441,3.7282,-3.743,24.4682,19.8742,13.7633,-14.143,26.8168,-26.8677,11.5005,-14.1437,-8.0547,-2.8622,-5.2612,5.0842,3.5318,8.6676,-18.8855,-22.7865,4.951,-4,23.9927,19.0145,14.3492,-10.3739,23.385,-25.4829,7.8145,-16.0977,-9.0485,-4.4756,-9.2706,1.3999,-2.209,3.8812,-19.616,-23.2316,2.2052,-6.3169,24.0421,10.6231,24.2544,12.7857,-6.8078,22.0152,-20.0223,9.0167,-16.4818,6.6983,1.5846,6.4594,4.9371,8.2084,0.18634,-1.0796,-0.73542,-19.1481,-21.7953,3.1914,2.0958,23.5992,16.8715,12.9315,-8.9592,24.6603,-23.3687,10.172,-18.5645,-3.908,-5.1805,-1.7987,-9.7703,-0.24506,-8.0522,-13.8671,5.3132,-19.0003,24.1239,-4.3142,19.4505,-13.8979,-2.0523,-12.6019,3.8745,-1.3756,3.6323,-16.476,-24.819,12.1585,-8.7332,24.8427,13.978,-13.0365,7.4378,-20.1602,22.8842,-3.5276,17.9129,-17.507,-4.1522,-10.0625,6.5552,2.9091,6.8509,-10.8266,-23.7782,14.8845,-10.6642,26.5655,15.1564,-15.8423,7.2913,-20.9904,23.7319,-1.39,19.7201,-17.4024,-7.6283,-6.268,4.5301,6.0333,8.3571,-8.6773,-24.1967,15.187,-12.6743,26.9332,16.0949,-19.5486,-23.9307,6.666,-4.9942,24.2966,16.9697,-6.8583,-7.1901,-0.62127,-0.81826,1.4747,1.4786,-16.2994,8.4554,-26.2766,24.702,-6.6074,16.8196,-20.1834,4.901,-25.3725,24.5751,-3.5394,23.2206,-12.5228,-10.7963,1.2956,1.7027,8.8535,7.5597,-11.2472,-24.0677,11.3947,-12.2939,26.5103,14.4027,-21.0163,4.8412,-25.0669,25.2092,-1.4469,22.068,-7.7735,-11.4511,-1.4559,-4.3643,-0.40178,0.10228,-14.2802,-24.5139,8.5316,-10.4256,22.8892,13.2615,-22.1021,3.4898,-22.2614,23.3795,-8.7385,22.2025,-1.6199,0.24961,-11.129,0.98481,-23.342,-9.0024,-12.9396,-21.6565,3.3045,-9.5719,19.8077,14.3292,-36.2982,-1.9487,-8.9388,21.5804,14.5917,20.2502,38.0942,-44.4862,12.4967,-32.4792,-14.7794,-18.6308,18.4833,-2.0372,15.4424,-17.0871,-22.7032,7.0954,-7.7807,9.5907,11.7983,19.5865,-1.2894,-1.9882,0.40253,-0.46475,0.38853,0.5185,0.45779,-14.379,8.3379,-24.9171,24.9692,-23.7553,19.6482,0.24585,-21.696,-31.2792,8.7312,14.5896,29.0789,-0.50341,-0.32189,-0.62939,-0.35568,-0.038107,-19.9994,13.6558,-24.6427,35.0168,-0.8704,-17.333,-26.3684,8.0901,-9.5606,24.1772,13.2663,-1.5657,-0.39396,-1.056,0.20096,0.46214,0.39923,-21.4129,4.0418,-24.623,23.6776,1.0653,22.9925,-13.6623,-21.7521,-13.6617,-10.4018,14.0297,11.8551,25.2409,-1.4362,-1.0223,-1.4362,-0.11889,-0.43884,0.052009,-0.25112,-21.9829,2.7863,-21.9833,20.9064,-18.2471,21.8878,2.1747,-34.2411,-12.2574,-26.2286,14.7672,11.9497,26.065,-1.4242,-0.4141,-1.3589,-1.0256,0.10067,-0.9253,3.7928,-21.8343,16.6211,-18.7644,21.7711,3.409,-10.7375,-25.7796,15.8481,-14.6101,23.598,9.5745,-0.29507,1.6343,-3.2973,1.5196,-2.9761,-0.21669,-22.122,-2.9407,-15.6638,19.0088,5.1158,22.2186,-16.6299,-21.5151,-1.8051,-6.5726,13.2927,12.2851,-5.1934,-9.3257,7.9801,-0.32958,21.0849,15.4729,-18.5386,5.6763,-23.3034,24.0353,-6.184,17.4231,-20.2581,-22.256,-2.7153,2.4893,17.8593,20.1956,-2.0695,12.4689,-13.4498,10.6823,-10.2879,-0.60803,-17.1637,3.947,-21.4529,24.6684,-8.7287,12.5757,-9.4582,-15.8785,8.2111,-8.8628,21.36,11.4356,-18.6894,-17.2141,-4.9028,1.347,10.538,16.4315,-17.1179,9.1339,-23.4806,23.9529,-4.168,16.6877,-14.0826,-21.5232,8.1171,-9.9152,21.3225,11.1671,-12.1228,-8.3301,-2.5535,5.6401,11.5249,15.914,-18.3908,4.409,-23.4211,21.5263,-4.3443,17.4768,-26.9907,-19.9118,-4.167,4.6277,14.1674,-11.5779,9.8919,-28.8272,7.423,-12.2869,-1.7653,16.6014,4.6538,23.533,16.2596,-26.1544,-21.9402,-3.6894,1.8728,22.1164,24.0987,0.91944,1.4774,-3.9686,0.30142,-9.1788,-5.9514,-7.6652,17.4229,-24.3235,24.1258,-12.6167,7.3686,-28.913,-24.7972,-1.9487,-0.93613,20.3751,17.4109,4.9017,4.4145,1.7887,-2.9963,-13.0713,-17.6514,-9.0021,17.6466,-27.1554,25.9174,-16.9661,10.487,-23.6741,-18.9242,-3.7912,5.6813,19.522,24.0332,10.4867,-12.9238,25.2749,-22.2157,14.4715,-9.2008,-5.529,-14.5556,4.4818,-12.7599,10.6418,2.5996,-22.5347,-19.2389,-2.5065,3.4857,19.9125,23.306,11.5521,-14.383,26.6036,-24.9355,15.2554,-10.2459,-7.9589,-11.1406,1.6278,-5.0791,8.6803,5.4042,-22.2253,-20.0727,0.25316,3.0526,16.4016,23.0095,11.996,-15.0308,29.0536,-26.7311,20.0436,-11.6601,-8.6186,-9.2774,-0.24424,-2.0229,4.8533,5.0095,-21.8067,-22.1325,2.8607,6.8501,22.7296,22.2499,12.8256,-16.6659,29.2099,-32.5829,17.0508,-13.8656,-8.1685,-7.728,-1.3411,-0.081023,3.597,4.0077,-26.1868,-10.0863,-17.288,18.2065,8.7357,25.7314,2.0517,14.1622,-18.9252,10.8216,-24.3715,-6.5896,3.9671,17.2951,-5.8656,14.7823,-3.0257,4.2805,-25.9328,-8.5388,-17.3473,17.9475,9.1931,26.6086,5.3369,24.3771,-17.9066,18.8371,-23.1218,-4.6913,-4.7021,2.811,-9.5389,5.686,-6.1392,1.5871,-26.6961,-8.8482,-18.2749,17.2978,9.013,26.7247,4.764,23.2308,-16.3974,18.1224,-21.7287,-4.5041,-3.6658,5.6901,-10.4869,8.9572,-9.179,1.3947,-27.3853,-18.9767,-9.3067,9.1028,17.1088,26.965,4.5321,-16.253,21.6703,-18.1781,16.4916,-3.2536,1.6779,-10.7092,9.3905,-16.4954,10.6913,-3.2684,-26.0717,-16.1502,-7.6117,8.6844,17.7936,26.0935,3.1448,-9.1652,11.9123,-12.3516,9.4144,-2.776,1.3127,-16.2871,19.5611,-18.3159,16.696,-3.6792,-17.6042,-26.5443,9.4076,-9.8239,26.2714,17.1652,-15.1362,2.4349,-17.6155,16.617,-2.2896,14.5789,-11.9739,4.732,-16.336,16.9053,-4.2151,12.438,-25.4387,-8.6571,-16.4034,16.2918,8.9309,24.6905,-0.14812,7.9125,-6.5391,8.7111,-4.386,2.8154,4.5605,21.8984,-17.4425,16.6229,-22.6286,-5.6951,-16.8249,-26.2099,10.4569,-9.4029,26.2024,16.2941,-14.8582,3.0875,-17.7594,18.2691,-2.3252,16.0255,-12.3665,3.4816,-16.1841,15.6588,-4.6289,11.541,-15.0495,8.8632,-23.3667,23.8558,-7.6576,15.6309,3.3273,12.1583,-9.4299,7.6077,-12.8011,-4.6316,-21.6262,-22.2294,0.81761,-1.4309,22.669,21.1065,-19.4487,-25.7488,6.1981,-5.7782,25.5122,20.1125,17.1066,-3.9118,22.4147,-21.1129,5.7669,-15.0728,-0.41,9.8972,-12.1821,14.9518,-8.0295,8.0512,-21.4554,-15.644,-5.4851,6.5448,17.037,22.9258,-5.2377,17.6024,-21.1409,23.6385,-15.175,7.3279,10.1359,6.5913,11.7633,2.7286,9.4021,3.8453,-24.7347,-18.9912,-6.794,6.8323,17.9587,24.9543,-6.2528,16.05,-22.2311,22.8959,-16.3537,6.1337,3.9277,-6.3711,10.1589,-9.067,7.4774,-1.2942,-18.377,-24.4307,6.5375,-5.6244,24.1593,17.2917,17.2441,-6.1391,24.6999,-24.8839,6.7427,-18.4071,-0.205,0.95377,-3.0249,2.5414,-1.9384,2.3906,-17.31,-23.7039,6.5574,-6.8521,24.4768,17.6914,17.0039,-7.3567,24.9818,-24.5457,8.4582,-17.7074,-1.1201,0.64592,1.5043,1.5399,0.82718,-0.90128,-17.9352,-24.1773,5.8503,-6.611,23.8584,18.2217,16.8627,-8.1453,24.8027,-24.3536,9.1865,-15.6192,-0.7855,-4.2358,2.9345,-4.8394,-0.21787,-4.6879,-18.7953,-24.4711,5.2437,-5.6571,23.6956,19.126,15.5667,-8.4255,23.469,-24.7409,8.1495,-16.0558,3.6366,-1.4297,9.4252,-4.7211,7.5344,-1.4647,-25.2357,-18.7926,-6.6703,5.5874,18.272,23.3383,-4.1814,15.0169,-19.7998,20.0147,-13.7713,7.6672,4.7444,-9.8018,15.8676,-15.2778,11.443,-5.8616,-18.4679,-23.2968,5.089,-5.0253,24.2,18.7385,13.7601,-9.1376,21.744,-24.1205,5.8165,-17.0497,-9.3842,-0.38584,-12.5661,6.2306,-8.5263,1.4041,-21.5115,-13.6795,-5.147,3.4718,14.3935,21.1998,-11.5094,32.3675,-23.1463,45.8076,-14.7945,8.0993,-3.1158,-16.6317,-1.4051,-10.1445,-0.7956,5.3816,16.4557,17.4569,13.7582,-5.982,6.3412,-9.9225,18.9143,-1.0714,10.7575,-21.0712,-11.1555,-22.4552,-0.91625,-22.3505,15.71,-12.2147,19.1686,6.4827,-23.8357,-14.8795,-15.9467,8.2076,-4.188,14.1937,3.1413,3.4717,-10.9132,-0.19623,-23.4069,-12.5851,-5.8334,18.1992,-14.6348,23.2824,-0.9023,14.5704,-23.1487,-22.951,-25.8436,-1.5611,-2.8205,14.3963,15.9599,-26.0121,0.18103,-24.0153,12.6685,-8.5751,22.3831,-25.9777,-18.2502,-18.6944,-14.2481,-8.4708,0.4286,7.8416,13.6396,-13.2081,13.393,-23.7568,27.7325,-9.6803,14.8875,-9.4104,-14.4882,7.0687,-3.4984,19.6999,15.7074,-17.85,-14.5377,-8.774,1.9434,8.024,14.1874,-15.0456,-22.2907,10.0131,-7.9602,25.2732,15.8272,-11.6494,4.6575,-19.2145,23.3683,-5.3957,17.4189,-14.2314,-21.3052,3.6721,-9.4045,15.5827,9.065,-10.4508,-9.5448,3.6281,3.2471,20.1669,17.8691,-18.2545,4.947,-24.3219,22.1698,-6.4046,16.5389,-23.5104,-20.8615,-3.1174,3.0461,19.9967,23.0019,-4.1141,9.0224,-14.7104,12.2913,-12.3252,0.8939,-11.7163,7.514,-19.3269,22.6231,-8.1529,12.8328,-24.8214,-21.5233,-3.9463,3.4403,20.309,24.1597,-3.0054,12.1221,-17.077,14.7027,-15.146,0.79382,-9.6873,7.4591,-16.8119,19.5229,-6.456,11.7385,-25.298,-21.0104,-4.7511,4.1429,20.5626,24.9167,-3.6247,14.5101,-18.9623,16.6516,-15.578,0.66926,-8.9612,6.8925,-14.3252,17.4554,-3.752,11.1966,-18.1294,-15.5487,-2.2797,2.1217,18.9624,20.6785,-0.15046,11.3411,-9.7155,10.091,-10.05,-0.38538,-14.0897,14.6407,-25.0139,23.6941,-11.9663,10.8435,-0.21569,16.7357,-5.3212,21.8613,-6.8926,5.4707,-1.0491,-14.749,19.6993,-5.0108,21.035,8.9141,-25.2848,-15.6227,-9.5618,12.355,12.5692,24.4348,3.0329,4.5085,4.1676,3.4156,-2.8204,-3.8428,-0.03223,-21.9286,20.5439,-19.7905,21.0762,3.8644,-26.7695,-16.4182,-10.6322,14.4714,15.4401,28.589,7.4556,-2.2951,15.2661,-8.9531,8.1259,-7.6277,-2.2791,-20.2113,18.0363,-19.8041,18.8854,-5.0241,-25.4761,-16.0845,-9.1592,12.9082,11.6419,25.6045,-25.4086,-19.8451,-4.892,5.2407,18.989,23.9773,-5.769,15.3308,-22.5817,18.7418,-16.5959,3.3133,-7.8426,4.6506,-10.7651,15.0288,-0.73349,12.8011,-22.6589,-19.0709,-14.596,2.3706,-4.6358,9.2999,3.3266,8.1222,-15.4456,3.8688,-25.6661,-9.8946,-6.8481,14.136,-12.7613,23.8393,3.2816,20.0658,-24.8032,-23.145,-2.3616,-5.4464,5.7803,5.0785,1.6943,0.91331,-3.8322,-6.6913,-17.8366,-23.2259,-6.9997,15.3839,-23.8909,23.5569,-18.235,6.4896,-24.3323,-20.9591,-0.40276,4.7063,22.0704,27.8869,-15.9116,13.5952,-30.5292,27.4328,-13.7069,10.8619,-5.7024,-8.7315,-0.040806,-4.8089,3.2069,2.0311,-17.0332,-23.3187,7.2901,-5.7167,24.7596,18.4414,-10.3338,10.0417,-17.8134,22.5421,-5.6641,14.3703,-17.3728,-3.2046,-16.7244,11.4646,-1.7018,12.5324,-17.5553,-24.9437,8.0236,-6.7303,25.0333,17.8282,-8.4464,6.1056,-11.2088,18.7801,-2.3434,13.2063,-17.2794,0.76935,-20.0252,16.421,-3.9899,14.6873,-25.4138,-16.7836,-8.6632,9.0621,16.4519,25.1592,-1.9516,-2.4584,1.6711,0.51648,4.4725,3.7849,4.4498,-18.8045,22.6989,-23.185,18.0325,-4.4836,-25.4838,-8.8499,-16.55,16.1598,8.951,24.847,-1.2814,4.6678,-4.2929,6.7965,-1.4594,3.611,4.7679,22.5352,-18.0416,17.4147,-22.7781,-5.3296,-25.5956,-15.8538,-9.6908,9.3641,14.9257,23.9957,-0.46945,-14.5989,14.4226,-11.7661,16.0528,3.5353,6.1115,-13.5211,18.3845,-20.5962,11.6243,-8.1324,-25.0245,-8.9858,-16.2385,15.3173,8.8322,23.996,-1.2983,12.1207,-11.3048,14.8484,-7.8073,4.7218,6.0597,20.2441,-16.0488,13.4276,-22.8298,-7.6244,-24.9661,-8.3121,-16.2531,15.7084,8.8308,24.1159,-1.7527,8.4449,-7.9502,11.7841,-4.5789,4.9226,5.6635,22.3999,-17.4183,15.9089,-23.3692,-6.7468,-26.3965,-16.1421,-9.9583,9.1098,15.9905,24.9379,2.1992,-14.2992,18.3083,-12.5394,15.836,1.5659,4.4234,-12.9729,16.4911,-19.9879,12.1134,-6.6905,-16.2324,-25.3713,9.4841,-8.9293,24.9984,15.8602,-2.0736,-3.1896,1.9571,0.06103,4.7241,3.964,-19.2821,4.3137,-23.2195,23.4053,-4.0788,18.9718,-15.6213,-25.2823,9.8359,-9.2621,24.7425,15.34,-1.553,-4.0918,3.484,-1.2938,5.5902,3.3653,-19.9763,3.8715,-23.364,23.624,-3.5983,19.6093,-15.2267,-25.0348,9.5969,-9.5176,23.8271,14.6523,-0.17928,-5.2957,7.2654,-3.1014,8.2109,3.0379,-20.6204,3.3194,-23.1688,23.8953,-2.7122,20.3869,-14.8635,-24.2082,8.523,-9.1863,23.1755,14.577,1.3984,-7.8505,10.6064,-6.7918,9.7302,1.4794,-21.1323,2.4011,-22.5627,24.033,-1.82,21.337,-15.3955,8.1068,-23.6791,23.1658,-8.222,15.1787,2.875,12.7491,-9.3053,9.5073,-10.1455,-1.1189,-21.3167,-21.9967,1.4531,-1.0605,23.4085,21.3909,-20.1193,-25.5941,5.2529,-4.7668,23.9648,19.896,14.333,-5.4689,20.2493,-19.9417,7.1709,-12.3171,-9.1363,5.4439,-15.9029,16.5099,-7.3863,9.7507,-18.3959,-25.6102,7.0705,-4.4039,28.5115,18.7795,15.06,-10.869,23.5467,-26.6523,5.2784,-15.2558,-10.1998,0.30433,-9.921,11.0246,-0.26252,8.1979,-20.2242,-25.3654,7.2472,-4.1676,23.7577,19.262,14.6626,-5.1765,23.812,-19.9858,11.6337,-12.4399,-7.4949,7.8518,-11.6527,17.5069,-2.992,10.9239,-18.6037,-25.6696,7.0733,-10,25.5702,17.4078,15.8481,-7.4942,21.9526,-20.9008,3.9733,-17.5271,-8.1419,2.0728,-10.9077,8.9881,-5.3929,5.2052,-24.4379,-19.2877,-4.5247,5.7401,18.5315,22.3374,-7.6402,15.7185,-24.1669,21.896,-16.8305,7.1936,-7.2539,2.8885,-9.3059,13.6739,0.27961,13.2536,-15.8062,-24.211,8.2269,-8.0233,24.7193,15.838,5.8161,-6.9808,10.0304,-14.0378,-0.80555,-12.3979,-20.9252,2.1651,-23.8798,22.2001,-4.8619,18.6591,-16.3387,-24.5147,7.676,-8.7373,23.4831,14.4873,9.7598,-1.0769,9.7369,-12.0012,-3.718,-14.8487,-18.8244,4.4267,-24.4696,22.4059,-7.6056,16.5412,-16.1406,-23.9961,8.3472,-9.0218,24.7693,16.151,14.7755,1.8735,16.4323,-11.7418,1.9371,-13.7914,-14.0298,6.5851,-19.4446,22.3803,-5.753,15.6873,-16.7379,-24.9982,6.5243,-8.376,28.3947,23.3183,15.758,-1.4461,19.2429,-16.0603,1.5397,-11.1282,-11.7796,5.7616,-15.6699,19.6602,2.0133,14.5244,-17.5645,-23.7094,4.5426,-6.9189,18.9595,-2.5598,16.2396,14.084,-4.1026,19.4329,-17.7515,8.7899,-17.7439,-5.7093,-11.3924,3.29,-15.307,16.7946,-5.9852,16.8733,8.5889,-18.0535,-22.8798,5.2172,-4.9955,20.9754,16.1772,16.9051,-4.6539,20.9886,-21.8196,2.117,-18.7292,-0.90812,10.842,-12.8079,12.023,-13.3082,-0.64892,-19.4858,-24.6273,4.5493,-5.3075,22.777,17.9135,15.3014,-6.7463,21.4287,-22.7159,3.8755,-17.8144,-4.7853,6.4908,-12.9617,10.7742,-11.8224,0.50531,-15.6926,-20.7231,7.7002,-5.2814,19.7386,13.4602,15.312,-2.991,20.1266,-21.534,-0.46204,-20.2688,3.0556,12.7856,-10.314,12.2501,-13.2639,-2.0074,-18.9954,-23.8801,4.9699,-5.3161,24.0089,18.6133,14.8426,-11.3924,25.8486,-25.9129,10.0194,-16.2128,-9.0391,-3.8855,-3.8752,3.6681,4.6666,7.1932,-19.6564,-24.5805,4.9713,-6.3374,25.4657,20.4386,14.4573,-13.2891,26.1258,-26.3188,10.0935,-15.707,-9.2623,-3.8587,-5.4651,2.0341,2.8184,5.2211,-18.4476,-23.008,5.1452,-5.2338,24.3832,18.7341,15.2315,-10.7721,24.7328,-25.5475,8.548,-16.2186,-8.2663,-5.55,-6.208,2.386,-1.8822,4.2415,-18.9675,-24.1597,5.7834,-5.5742,24.5503,18.7094,16.306,-8.6351,23.6332,-24.2703,6.337,-16.7147,-4.0217,2.3268,-9.196,6.5085,-8.1907,1.2046,-26.1378,-22.078,6.2332,16.2804,25.0593,14.352,-31.3041,25.3079,-21.1759,5.0224,9.584,-9.8818,12.4113,-10.0266,2.6379,-18.4522,-23.1047,7.1851,-7.9101,24.153,15.8989,12.9743,-6.7434,23.8711,-22.9142,6.8324,-18.206,1.6502,-4.3591,3.1497,-12.6172,-0.057805,-9.9961,-19.1297,-24.7277,8.0776,-6.0208,22.7698,13.9888,-3.0563,-3.1209,-0.60038,-1.2846,-0.16273,-0.87584,-15.9688,11.0613,-25.7112,24.4156,-11.1571,14.6117,-18.57,5.2772,-23.9453,23.966,-8.2385,19.0125,-5.1796,-0.51142,-4.1434,0.67901,-1.0623,-0.20983,-14.1786,-25.0394,12.671,-11.7328,25.7946,8.3613,-20.2043,6.0324,-23.9364,22.2572,2.9028,7.466,-7.4392,-2.7074,-3.8056,1.1341,2.0105,3.2416,-13.8912,-27.8274,12.667,-14.2336,34.4824,31.8155,-23.1851,4.2163,-24.5427,23.37,0.67195,25.6704,-7.736,-4.5515,-1.863,1.272,2.6271,2.6691,-14.6393,-28.7618,12.055,-13.1663,28.0771,13.9864,-16.5159,-25.0473,10.2879,-8.5839,27.0525,23.9948,-1.01,-0.87818,0.070933,-0.62038,0.2468,0.10475,-18.3618,8.8916,-25.6418,25.198,2.2548,9.2765,-21.6698,1.3221,-25.1648,22.9061,1.9827,23.5232,-4.4892,-5.6967,-0.20092,-1.5142,-0.55647,-0.68946,-16.0848,-26.4577,10.3034,-12.7775,26.6139,14.1278,-21.9326,3.0968,-22.9866,21.5373,-2.2232,22.9501,-0.58302,0.098647,-6.982,-0.19172,-16.1843,-8.2413,-13.3173,-23.5649,6.6946,-12.4861,19.4377,8.3891,-12.8419,-18.0635,0.78418,-8.0467,20.8375,19.2367,10.8452,-13.3285,15.9845,-27.3082,7.2801,-16.7676,-17.7705,-5.2979,-21.0238,-10.0566,-8.5628,-3.5591,-24.1803,-19.5183,-9.5719,11.8174,11.738,25.6789,-0.032855,-0.16286,0.34968,0.023359,0.77229,0.5614,-11.2616,-22.8072,23.8235,-23.7469,24.0928,0.27976,-13.7754,-25.0837,14.085,-12.8509,25.6675,10.5349,-0.34325,-0.42259,-0.072177,0.061948,0.24049,0.47598,-21.9717,0.75447,-20.7433,21.9582,0.64001,22.2145,-25.9795,-11.4549,-13.2961,15.4965,11.1134,25.361,0.33817,-0.59429,0.51652,-2.4732,0.40816,-1.5579,-3.524,-22.4178,19.5808,-18.8438,21.6812,3.7072,-2.4812,-14.9879,16.8425,-22.3875,12.6273,-9.2807,15.666,6.1847,15.0375,-12.6411,-1.2674,-18.2893,14.3015,18.6789,-12.6842,5.0869,-22.0158,-15.5699,-11.1383,2.8293,-19.5749,16.5564,-12.4333,8.7795,14.7612,24.8729,-9.0321,10.2883,-20.8528,-12.7867,-14.8259,-0.73678,-10.7577,15.2932,6.6948,17.8997,-8.7185,-25.071,15.471,-15.2245,24.3145,9.6381,-0.49558,2.6899,-4.2356,3.4453,-1.5202,2.9287,-23.315,-3.3748,-17.9997,19.1824,5.2431,22.6873,12.5479,-13.8631,13.2479,-19.7062,9.5677,-6.9442,25.3331,0.15426,26.1109,-19.8317,2.241,-20.4437,-0.92383,20.555,-5.1934,8.5289,-23.9432,-16.2319,-15.2361,-15.9571,-3.684,-4.7229,8.8815,8.7372,-13.1247,-19.101,7.9219,-3.7964,24.5772,18.9813,-17.1843,6.8989,-22.7165,23.602,-4.754,16.7815,-20.9055,-17.9159,-2.733,1.7911,18.956,21.2844,-2.0034,10.3235,-12.3776,10.1616,-12.3734,-1.4428,-13.8944,12.4383,-22.8116,24.4555,-10.0887,12.4598,-1.9433,-21.155,19.1509,-17.7597,21.1349,2.4513,-8.9966,6.2115,-10.742,12.6103,-1.1595,9.1909,-24.2297,-6.2038,-13.2788,13.1716,11.3826,23.9157,-6.4034,-15.0164,10.2975,-3.3552,21.3517,15.4333,-19.3833,-18.449,-6.2921,-1.7377,11.8041,13.5765,-14.7937,10.8702,-21.1468,23.6525,-4.6885,13.7028,-11.4449,-21.7331,10.793,9.0889,-11.1241,21.6421,-16.1781,-10.1394,-5.308,18.4297,7.118,13.0208,-17.3584,4.4591,-21.6466,15.7111,20.2383,-4.0753,-11.918,-21.9601,8.9467,7.0788,-12.3614,17.0835,-14.772,-10.606,0.80596,21.099,5.2685,20.8642,-17.8255,4.1698,-23.808,13.4869,19.6453,-6.8433,-24.487,-21.7371,-13.4639,-4.6245,-7.132,5.3099,9.7836,-16.0142,-6.0047,-25.1838,-11.2227,16.5222,-17.235,24.8072,4.3319,-14.9287,-15.8499,-10.4273,-3.8585,-13.7888,-7.1657,19.7627,17.4704,-2.0956,4.3049,-21.851,-18.4833,-6.5908,15.4907,23.4436,-22.9013,7.706,-15.8653,-23.0077,-18.3165,-3.8019,6.1687,19.8723,24.7301,10.464,-13.3413,24.6984,-21.7284,15.1663,-6.8793,-5.5627,-14.0642,4.6138,-12.8318,7.6622,-1.0799,-22.965,-19.5041,-5.1522,4.7695,21.5332,24.7748,11.4757,-14.4525,24.9369,-24.3106,11.5548,-9.025,-6.6573,-10.018,-0.3091,-6.6565,4.5526,0.95593,-23.2517,-17.8573,0.75955,3.1609,20.3381,23.4261,12.5393,-13.2768,41.0863,-23.8011,28.8738,-12.9077,-4.7722,-4.7004,-1.3145,-0.9782,2.4018,3.9559,-22.3269,-23.0095,8.3133,2.67,43.3462,13.4784,-27.2208,36.3587,-43.3523,14.3953,-3.5562,-2.6932,0.39767,4.0092,8.8125,-26.8909,-7.9663,-17.8917,18.7754,9.8569,27.2771,1.53,12.2409,-18.6658,12.1391,-24.5109,-5.2726,3.5336,19.2482,-9.527,13.7633,-8.7352,-1.2924,-26.6832,-7.8396,-17.7107,17.4977,10.1166,27.0414,5.5345,25.2554,-18.8916,19.9171,-23.3115,-3.9597,-2.6603,5.2464,-6.8167,5.9731,-6.9027,-1.8286,-26.491,-8.898,-17.3701,17.031,9.7628,26.4329,4.9993,23.5829,-17.0047,19.5005,-20.1877,-1.5608,-2.8179,5.2297,-10.0039,5.0207,-11.5142,-4.2554,-26.1218,-17.1411,-9.3547,8.8221,16.4578,25.4484,5.2372,-13.3634,21.5089,-15.37,18.6701,0.59567,-1.5123,-13.6369,8.6092,-17.8892,6.7737,-7.167,-25.8172,-18.1272,-7.2166,7.1039,17.9834,25.6924,1.7943,-9.4649,11.5122,-10.1977,10.1707,-0.15537,3.3293,-15.5689,20.8514,-20.3461,16.2765,-6.0013,-16.8685,-25.9873,9.8008,-8.729,26.2316,16.672,-15.9795,0.79113,-18.3081,17.0418,-1.9894,16.3826,-11.4437,5.4512,-15.7172,16.8121,-4.2451,11.2788,-25.0907,-8.4374,-16.2406,15.9026,9.1618,24.5132,-2.4105,5.1347,-5.816,8.3898,-2.1059,4.5089,5.2139,23.1134,-17.8481,17.3443,-23.3004,-6.0151,-16.2441,-26.095,9.8459,-9.6641,26.1533,16.2835,-15.9831,2.2826,-18.4939,18.5898,-2.6188,16.0725,-11.8622,4.1762,-16.2332,15.6605,-3.5952,12.1758,-15.5083,8.5801,-23.9305,25.3762,-8.21,17.3131,3.2938,11.6124,-8.3555,4.804,-9.6396,-4.4908,-21.6085,-23.2716,2.2605,-2.0689,24.3207,20.9674,-23.8693,-17.3014,-7.0209,6.4647,17.5878,24.569,-7.721,17.556,-23.3077,24.2147,-17.4415,5.5236,-0.33514,-6.3867,6.8819,-6.8184,5.0127,-2.8204,-25.032,-17.9334,-8.2914,7.0031,17.2075,24.8798,-5.599,17.1741,-21.7144,23.1204,-17.4011,5.2422,3.4304,-5.5877,8.7235,-7.4753,4.4901,-2.8155,-17.3072,-23.7564,5.9741,-7.0252,24.2996,17.8002,17.803,-5.8075,24.289,-23.0265,6.3717,-17.2493,0.39843,4.4278,-1.9447,8.263,-0.19985,5.9932,-17.8658,-23.6655,5.44,-6.3176,23.7531,17.5012,15.8954,-6.5816,24.1172,-23.542,7.8753,-16.8728,4.0123,2.846,4.0446,4.5039,4.5149,6.2753,-18.2013,-23.6617,4.4593,-5.9102,21.8685,16.6467,15.5544,-8.498,23.3935,-24.2168,7.7617,-16.6143,4.939,0.64335,8.3313,0.86082,9.0839,5.9439,-19.171,-23.5853,3.8729,-4.4125,20.2461,16.6851,16.0174,-7.3377,23.2203,-24.591,7.3332,-17.1802,1.7715,-6.8666,11.8438,-5.9601,13.3415,4.4334,-22.8147,-18.5715,-3.3076,4.0085,19.193,22.2834,-5.6616,12.2745,-19.7891,19.1736,-9.9157,8.1667,3.864,-10.6483,16.8365,-16.9333,9.133,-7.6462,-18.3234,-2.1763,-8.5293,8.8477,5.5438,23.3172,15.6498,-3.6448,24.6058,-20.8519,26.9597,-21.6655,10.0162,-6.579,11.3936,10.6173,8.581,11.7944,-5.532,-1.8383,-14.192,-18.8925,-24.5219,4.7838,-6.0886,22.4897,17.3871,16.8994,-6.5836,21.8541,-23.5498,3.4366,-18.1039,-2.2479,7.7918,-12.8493,8.82,-13.1977,-1.7474,-21.5785,-13.1531,-7.18,0.72054,15.7603,22.9972,-9.0766,19.2089,-23.4716,24.628,-17.1151,5.7076,-2.3541,4.675,-5.4345,7.7622,-5.0067,2.3831,-22.0862,-10.0146,-10.791,13.9177,9.9409,26.0399,-0.58943,-0.80747,-0.64725,-0.3333,-0.70103,-0.22067,5.9936,-24.0694,24.782,-23.7669,26.5045,1.1073,-25.9401,-11.9448,-14.2879,15.3581,10.8563,26.1574,0.020168,-0.2654,0.29869,-0.39516,0.47817,0.072402,-1.1445,-21.7555,20.3676,-19.4059,20.9957,1.6609,20.9699,15.3989,13.2986,-10.2276,-0.053274,-18.3278,14.8865,13.9634,-5.7886,-18.9221,-5.0082,-16.6365,-6.4643,15.2353,-20.798,-15.5498,24.3498,10.1525,13.2337,-5.4764,20.7452,-20.0351,5.3334,-17.2931,20.1876,8.4917,7.632,-7.6436,-12.9895,-18.2199,-8.4109,-21.718,10.6284,-13.2371,20.2964,10.7566,-23.6786,-18.0993,-15.0424,-4.8425,7.7181,13.142,4.4318,-10.1196,6.813,-22.3356,2.6595,-10.8838,-5.2828,-13.891,17.5854,1.1966,23.3294,17.1338,-12.3684,3.9056,-20.368,16.8427,-11.9617,9.5279,13.1186,22.7047,-8.0683,10.5924,-20.3271,-11.104,-14.3909,-0.67937,-11.1744,14.656,6.059,18.2116,-15.8656,-10.5034,-9.0038,4.1641,2.5009,10.0664,-15.593,-23.7491,10.7786,25.9088,-7.2645,16.888,-14.3042,5.2976,-18.989,-2.8309,23.1643,18.811,-10.8598,-18.0117,4.2211,2.6769,-10.1393,10.3207,-16.3705,-18.6751,19.61,4.9724,-0.42054,23.3044,-17.1975,7.2691,16.2372,-23.6806,23.0752,-7.7549,-23.4369,-20.2689,-3.6171,2.4105,18.4751,22.0171,-2.3775,11.4837,-15.4145,11.3266,-15.2515,-2.3566,-11.37,9.9788,-18.9725,23.4026,-7.6213,13.1975,-24.4519,-20.5426,-4.3426,4.4098,19.4486,23.7233,-1.5057,14.6207,-16.8234,14.8644,-16.622,-1.4267,-10.9994,7.234,-17.166,19.9895,-5.8955,12.1852,-25.8032,-20.5742,-4.8193,4.8306,20.9949,24.8515,-3.1638,16.1541,-19.8903,16.4806,-16.4882,-2.0004,-11.2768,5.9693,-14.305,17.8566,-0.14988,12.6408,-21.2726,-19.4287,-1.8527,0.8662,19.4687,20.5983,0.65099,9.5994,-10.3455,7.9252,-11.325,-2.1737,-10.4152,13.2305,-23.8556,24.4342,-12.3342,12.1875,-20.7534,-18.8257,-2.4476,1.3371,18.8118,20.7792,-0.11163,10.7309,-11.4788,9.7511,-13.4941,-2.9417,-12.4892,13.012,-23.6413,23.9387,-10.7797,11.8868,-4.1015,19.4912,-22.5537,21.1643,-19.5784,1.2077,-7.5995,-11.7055,7.8239,-4.8,12.6633,4.3605,-25.5266,-12.9207,-11.919,13.9691,11.9825,26.7114,3.477,17.083,-2.7671,20.2832,-7.1094,2.8186,-1.2446,-16.7379,18.5095,-11.5346,21.8809,7.494,-27.0406,-12.5159,-15.2447,15.7246,11.0049,28.1772,11.7736,6.6099,11.1658,1.6296,-10.1624,-4.6899,-1.272,-22.3739,19.9172,-18.3625,17.4413,9.8561,-28.3484,-13.3444,-11.5652,17.6425,22.3517,31.9373,6.8905,-3.5381,19.6474,-22.0888,19.2767,-18.4011,5.2871,-0.43632,-14.8931,14.1666,-15.8711,8.9684,-20.466,-9.2701,-22.5121,-14.0393,-9.9183,11.6334,11.6266,8.8089,19.9754,-24.7291,-19.1579,-3.6383,5.7797,18.9005,22.8906,-5.3001,16.577,-23.9587,19.8759,-16.8404,6.0883,-12.9404,1.5039,-13.5018,15.1505,0.36297,15.3767,-1.1557,-1.22,-11.3824,-11.4142,-2.9344,-3.1549,23.8412,18.921,-17.3127,-23.9537,8.2958,-4.8211,-6.5722,16.4116,-18.4103,6.8991,-23.1429,23.6202,-30.3837,-21.5495,-20.3386,4.7226,21.6735,26.3363,-22.6518,15.4569,-28.8744,25.8729,-15.2651,9.3956,-3.6046,-4.0653,-2.4016,-0.91227,3.8171,2.9083,-18.0293,-24.6101,7.0807,-5.6255,25.0018,18.4501,-13.116,7.3582,-18.8201,22.5714,-5.1898,15.7183,-14.7192,0.2994,-16.4866,12.262,-2.2013,11.4673,-17.8167,-24.9089,7.5665,-6.8745,24.8576,17.5698,-9.1687,5.6137,-10.7009,19.354,-0.90193,14.6426,-16.8615,1.7994,-21.3703,16.1431,-5.7291,13.5849,-25.4029,-8.6065,-16.2299,15.9428,9.4373,24.7712,-3.0131,2.205,-3.8459,5.7899,0.19682,4.5992,4.7456,23.1777,-18.9217,17.9086,-23.5124,-5.0894,-24.6814,-9.3495,-15.3937,15.2702,10.1581,25.0238,-3.0824,12.9327,-13.954,14.4943,-10.1713,2.2411,7.1686,20.235,-15.2734,13.8038,-22.627,-6.6393,-24.0354,-14.6462,-8.3014,10.2828,16.1674,25.3492,-3.5134,-16.468,10.689,-15.115,11.6716,-1.2784,8.3636,-12.1027,21.5219,-18.4035,15.3067,-4.4662,-24.5406,-8.6023,-15.6606,14.979,9.7406,24.5547,-3.9768,9.2322,-11.2413,12.8646,-6.412,4.4604,6.9209,22.8978,-16.9802,16.0243,-23.9316,-7.1076,-24.7334,-8.4553,-15.8852,15.172,9.5443,24.4541,-3.8108,6.3187,-7.8528,10.8462,-3.0526,5.58,6.0249,23.727,-17.9262,17.2456,-24.1996,-6.6685,-25.8365,-16.6365,-9.8777,9.0561,15.6146,24.7834,2.6579,-12.2549,18.1906,-11.2508,16.477,1.5755,4.1198,-14.8314,16.491,-21.3399,12.0211,-7.0366,-15.7362,-25.2975,9.7815,-9.0483,24.6179,15.4648,-3.4303,-4.1608,1.4027,0.47075,5.3588,5.1789,-19.5452,4.5406,-23.7092,24.0585,-4.5912,19.0011,-15.225,-25.1214,9.8651,-9.3727,24.0513,14.7774,-2.4888,-4.7679,3.9607,-0.54931,7.2532,5.0671,-20.297,4.0999,-23.9382,24.3976,-4.0545,19.8564,-14.6408,-24.5174,9.3495,-9.3211,23.4819,14.4687,-1.4191,-6.9986,6.9422,-3.4143,8.6462,3.811,-21.3208,3.5513,-24.0414,24.8747,-3.1862,20.8631,-14.9535,8.7518,-23.9265,23.9221,-8.9021,15.3864,0.41532,9.4744,-8.4215,8.2625,-6.5279,1.2106,-21.5885,-23.6515,2.7044,-2.3429,24.9273,21.536,-19.7972,-23.9046,4.5048,-3.2764,21.8747,19.4758,12.8149,-6.8565,20.7854,-19.233,9.4703,-10.7617,-10.1206,7.1826,-16.7773,17.717,-7.0523,10.0549,-19.3538,-23.7668,14.8459,-2.5553,48.5131,15.0186,-8.3455,31.9026,-22.2247,6.9512,-8.0326,5.8535,-12.0502,17.6496,9.8838,-18.5115,-33.6254,7.1872,-31.1364,27.1312,17.9442,15.6224,-16.6468,19.0758,-19.3896,0.64111,-15.4469,-8.7256,3.8523,-11.6701,2.764,-3.6802,7.742,-17.2788,-24.595,7.4392,-9.3306,25.646,17.1404,17.741,-1.5487,21.9036,-16.2327,4.492,-14.6181,-3.3307,6.4648,-7.495,11.8723,-0.18814,9.8057,-22.4044,-18.699,-3.2641,4.4657,18.2164,19.9971,-6.3898,16.7651,-25.2098,24.0107,-17.4653,9.3389,-12.1234,0.24813,-10.0675,14.7447,4.4504,15.9223,-17.3529,8.8464,-25.9332,25.4239,-8.7559,16.3034,9.2338,11.0921,-1.7485,-0.82782,-5.0436,-5.8992,-20.1312,-23.9872,4.2179,-4.1498,25.3397,21.0366,-15.982,-24.6844,8.2743,-9.5896,24.0578,14.9514,12.8068,1.3058,9.7648,-9.2475,-4.3219,-11.7161,-17.544,5.3281,-23.7113,23.4783,-5.5454,18.7185,-14.4613,-22.7616,7.891,-8.7089,22.6492,11.8435,15.4598,5.084,13.7327,-6.9501,3.5885,-8.014,-14.7748,7.1701,-20.6479,23.3791,-5.0606,19.7969,-16.3933,-24.042,7.1902,-9.131,24.5692,16.5962,14.1232,0.058656,17.6042,-9.4219,11.2572,-3.7991,-13.9072,4.9606,-17.9001,21.1013,-5.0071,15.9713,-21.4759,-30.5085,30.9466,-7.3178,27.1939,16.2118,-6.476,22.1331,-16.038,-1.2411,-18.8079,3.3757,-11.9898,19.4197,11.8149,-15.6519,3.2471,-19.6549,19.6281,-4.1792,14.6863,10.9992,18.9216,-9.1816,3.4876,-24.2618,-20.4496,-10.8513,-17.7203,11.0738,-17.323,11.6856,-7.5678,-17.7815,-22.6973,4.5313,-5.1755,20.529,15.8626,17.9441,-4.0397,21.1547,-22.2114,1.5153,-19.518,-0.2006,12.344,-13.7063,12.056,-15.2685,-2.2867,-15.694,-20.7085,4.6976,-5.0757,17.6948,13.2322,19.417,-2.4255,20.6731,-21.6473,-0.63361,-21.0297,1.0581,14.3552,-14.9095,12.8035,-17.8875,-3.8321,-11.2094,-18.1578,7.1073,-5.4743,12.7658,8.4239,21.314,-0.40434,20.0274,-21.5291,-3.2126,-23.25,4.6829,16.9169,-15.8745,13.8116,-21.9644,-6.1997,-20.1496,-25.1611,5.4149,-4.7226,24.9128,19.4175,15.8194,-10.1659,26.2096,-25.7726,9.3192,-17.1136,-6.4864,-4.3088,-2.6094,-0.37139,3.4495,3.3171,-18.7751,-24.0281,5.6429,-5.6222,24.5714,18.3748,15.528,-9.9968,24.9305,-25.9396,8.5065,-17.1834,-7.0526,-4.5838,-5.5855,1.1123,-1.4805,2.8494,-18.809,-24.377,5.7281,-6.021,24.2337,18.2212,15.7223,-8.7119,23.2227,-24.6394,6.5675,-17.0052,-4.6263,1.344,-10.4662,5.3232,-9.4659,0.50201,-18.4617,-24.4375,5.4799,-5.8445,22.8421,17.6648,17.2128,-6.508,22.6886,-23.2263,4.3859,-17.6021,-0.62934,9.2616,-11.7557,10.291,-13.7555,-1.7471,-25.1281,-17.1492,-12.3638,8.07,13.2919,24.3359,-5.7838,17.8918,-18.4999,22.8668,-8.5942,7.3472,-1.7678,9.9793,-19.4057,12.7776,-20.7952,-2.9199,-16.973,-23.9688,7.2396,-10.6065,22.4547,11.7476,16.6392,-4.6607,23.3204,-16.6107,8.509,-9.4245,5.4923,-2.7816,6.0448,-17.084,-6.0355,-20.0703,-17.9452,-25.0109,8.7967,-11.0086,13.3594,-2.8055,0.60556,0.10932,0.86498,-14.0698,-12.8399,-27.7709,-12.6828,16.0105,-25.7997,22.1332,-21.2913,-3.5101,-14.1859,-22.7855,11.5883,-21.0495,17.9052,-4.182,20.3516,0.78311,0.010863,1.8115,-11.6796,1.0468,-26.3829,-14.4022,-14.0409,11.188,-26.6171,12.779,-24.2279,-0.39056,-4.1645,-14.0187,5.9984,-21.56,8.9866,-20.905,-0.86206,-7.9359,-0.071822,0.1005,1.5192,-12.794,0.3661,-26.2298,-13.1142,-11.561,-24.3718,11.8832,-23.09,14.9041,-1.7703,19.3359,-26.2728,14.094,-23.3505,26.5338,2.075,-3.2825,-4.7607,-1.3936,-1.6134,-1.0478,-21.3162,-43.6987,13.2784,9.672,27.1198,-23.578,-2.3199,-22.9174,20.6643,1.5594,22.7174,-1.1129,-1.2756,0.086709,-0.61021,-1.8154,-2.2736,-15.2996,-27.2648,13.2825,-14.7506,27.969,12.4943,-16.2614,-35.5832,15.4649,5.1614,26.2908,0.40045,-1.9162,0.42114,-3.398,-1.7474,-27.0367,17.3406,-24.3984,32.9818,4.4675,-23.52,-0.60584,-21.8304,20.285,0.94628,20.4157,0.7482,0.60275,-6.1835,-0.74147,-20.5627,-12.5927,-14.1278,-27.4103,10.1638,-14.8721,13.9947,2.3297,-18.6143,-19.9573,-1.1197,-6.2948,15.4595,12.9208,8.9042,-7.8604,14.4248,-19.4287,4.6869,-13.2391,-10.8749,-4.203,-18.5529,-7.5924,-15.2361,-9.278,-25.6557,-12.8279,-14.4148,15.3788,9.197,25.7763,-0.43899,-0.62335,-0.47941,-0.14313,-0.48979,0.017569,-0.32583,-22.238,22.7384,-21.9286,23.7472,-0.0071949,-11.3796,-25.9137,16.0804,-15.3471,25.6012,7.9218,-0.53917,-0.33286,-0.35862,-0.22754,-0.51545,-0.51272,-21.0076,-0.28589,-20.3213,21.9911,-1.9007,21.9816,-26.4715,-10.1365,-15.3578,15.9062,11.9179,25.6667,-0.012998,-1.8874,0.44951,-2.4391,-0.20187,-1.1536,-2.2753,-22.8896,19.4053,-19.217,21.1547,3.967,-8.3119,11.5885,-21.7831,20.207,-16.8794,5.0094,13.2208,20.855,-4.6374,8.2869,-16.6926,-11.7859,-17.3232,-6.7253,-8.8289,10.942,10.9234,20.0334,-9.272,-25.5444,16.9124,-15.7741,23.7601,8.3052,-5.2927,-0.48962,-9.9748,0.9795,-7.675,-1.4899,-22.8285,-3.4703,-16.2499,19.7144,4.5397,23.4629,-21.3664,-18.8368,-2.9604,1.9767,18.375,20.9218,1.0468,12.9216,-11.876,10.8603,-14.7465,-3.6105,-13.3139,11.1815,-22.874,23.8374,-9.6095,13.1439,-1.7062,-21.9425,20.1845,-20.5503,20.8039,-0.053423,-8.4399,2.5145,-10.2182,8.3613,-2.8395,5.1177,-24.6132,-10.9951,-12.542,13.0613,13.5921,26.1322,-16.2922,20.0884,2.8945,-15.3626,21.6229,4.8716,-12.429,12.6517,3.0601,-19.7422,5.2557,-12.1469,-13.7342,-11.1127,-23.8553,9.1895,9.3463,19.829,-24.6418,-19.1046,-4.3319,6.509,19.5443,25.2075,9.1443,-15.6921,24.6612,-22.9711,16.0042,-6.8013,-1.0057,-11.1822,6.7306,-12.0469,7.5357,-1.0124,-24.6941,-20.4429,-2.3421,5.7384,22.5096,25.3704,12.2463,-15.6362,24.2525,-25.131,10.609,-9.1666,-3.5744,-7.6865,1.5636,-6.7977,5.5084,1.3946,-23.008,-23.0076,-21.084,4.6003,2.8301,25.6068,23.4248,14.4785,14.4791,-11.3448,23.4132,-24.4762,8.6588,-14.7709,-6.4109,-6.4109,-5.5195,-2.387,-1.7106,2.0631,2.4945,-25.8793,-17.4917,-7.0563,12.1753,17.2744,28.1624,-1.9415,-18.7233,8.9229,-22.6644,9.4171,-5.036,6.5828,-8.9776,20.8556,-14.3522,14.8604,-3.8929,-27.3546,-9.7854,-17.5177,15.4224,10.3619,27.6561,4.5938,23.9959,-20.0059,20.288,-24.0997,-2.6984,2.7621,10.27,-5.326,6.4982,-7.1714,-2.9094,-27.1476,-9.5237,-17.0649,16.6074,9.5869,26.2493,3.8111,23.9656,-18.8428,20.1287,-20.4124,-1.6295,1.882,7.1546,-7.7463,5.4244,-12.0448,-4.1188,-26.2207,-16.7193,-9.3681,9.4987,16.6617,26.0664,1.5176,-17.6374,20.4035,-18.2366,18.8711,-0.85535,4.7405,-9.6451,12.0017,-15.4614,7.8023,-5.3965,-25.55,-18.636,-6.9493,10.0165,18.0234,26.1202,0.53024,-8.7676,11.0699,-8.2327,10.942,0.72183,5.0938,-14.8544,21.9203,-21.8916,15.7335,-8.0887,-16.5674,-26.1707,9.8307,-9.3693,26.1988,16.5866,-16.6004,1.8539,-19.1774,19.3576,-2.5896,17.5451,-11.4154,4.1826,-14.9398,14.5171,-3.2443,10.6687,-25.1046,-8.628,-15.7229,15.3803,9.4785,24.4264,-3.6841,3.3439,-5.07,7.862,-0.21044,5.76,5.3919,23.9413,-18.9691,18.0223,-24.171,-5.6484,-16.2756,-26.1142,9.844,-9.6228,25.5854,15.8317,-15.5718,3.1049,-17.234,19.8061,-1.0678,17.3516,-12.8959,2.764,-17.5834,14.5619,-5.1241,11.3217,-15.7122,8.6202,-24.7696,25.9755,-8.8825,17.1045,1.3771,6.2394,-6.539,2.7639,-5.0992,-0.80377,-21.9777,-25.1853,3.5658,-3.8848,25.8892,21.6952,-24.5909,-17.5439,-7.6467,6.5084,17.8744,24.8559,-5.6788,17.503,-22.6152,23.0967,-16.5733,5.7411,2.4626,-5.4427,6.9058,-7.6524,6.4355,-0.15614,-24.6468,-17.2331,-8.5729,6.7803,17.3685,25.047,-4.5025,17.373,-20.0891,23.0245,-15.4194,6.1321,3.0049,-4.0903,8.9069,-4.6548,8.974,2.3244,-17.982,-24.3086,6.2642,-7.2814,23.4938,16.6132,17.1703,-5.8474,23.9366,-21.9754,7.2734,-15.5531,-4.304,1.6592,-2.606,8.55,5.253,10.4518,-17.8428,-23.7654,6.1661,-6.6497,22.5157,15.5571,16.3414,-7.0638,24.4457,-23.615,7.7467,-16.4024,-5.3757,-3.6693,0.11473,4.077,7.862,10.0664,-17.456,-22.8128,3.9683,-6.7333,19.1012,13.9707,16.8292,-7.8314,24.3345,-24.2713,7.768,-16.722,-3.1238,-6.0587,7.8565,1.645,13.3365,10.3797,-16.982,-20.6534,2.0666,-3.8462,19.0123,16.3333,17.8765,-7.1543,24.7338,-25.8922,8.015,-17.7488,-3.274,-12.2898,9.7051,-5.7021,14.4462,7.7309,-28.3868,3.4153,-30.5993,17.4124,-0.23915,14.4668,-4.3892,14.8909,-12.4367,3.5508,-22.8074,-19.0495,-4.8835,-19.6627,7.1268,-17.3029,13.1791,-5.1636,-22.44,-19.0069,-6.4957,2.3362,13.8368,20.3234,-9.0059,11.0663,-24.6598,19.0558,-21.0958,2.7978,8.5873,-10.0708,8.3039,-17.2118,-7.6863,-15.0339,-12.6511,5.0583,-7.116,15.9183,3.865,11.1398,1.6381,25.3634,-21.4595,22.0316,-24.8408,-3.2153,21.4065,9.0485,15.306,-10.752,-7.9189,-23.483,-17.3142,-22.867,4.481,-5.6532,19.7845,15.3699,18.7748,-4.5847,21.5217,-22.7005,1.7942,-19.221,1.1418,13.5031,-13.8831,11.555,-16.795,-3.7739,-10.9534,-20.7081,12.5588,-8.7262,26.4139,11.4283,8.4837,-18.7844,16.7823,-22.81,9.8551,-5.0446,21.244,5.1435,20.4131,-13.8817,-4.1752,-22.3899,-22.9999,-16.7025,-9.6544,8.1386,9.8704,18.643,-6.2053,13.7211,-19.0085,22.1524,-9.6806,10.4654,-1.0381,5.2093,-12.4314,1.9184,-19.5305,-11.3615,-25.4058,-11.6252,-16.3729,15.825,7.7263,25.5358,0.27988,0.70008,-1.9103,0.74346,-3.9077,-1.5584,-2.1061,-23.8456,21.9795,-22.6714,26.0299,2.3062,-26.9367,-10.7996,-15.6618,15.4949,8.9789,25.1697,-0.11955,-0.52191,-0.28611,-1.2308,-0.54729,-1.191,-0.6744,-21.0662,21.1395,-19.3778,19.3176,-2.2282,16.4497,-4.3298,21.6461,-21.9692,3.2411,-17.4687,17.6269,12.9642,4.8951,-5.3533,-12.3267,-16.8986,-11.4675,-20.103,9.0448,-9.0626,21.1979,11.8278,-12.0003,4.1001,-19.7631,11.2244,-12.7363,3.6384,11.7432,21.9789,-8.9284,13.3899,-17.9344,-3.5756,-15.8526,-1.6184,-11.7376,18.1429,9.2597,24.3267,-24.6305,-17.2421,-15.7099,-6.9047,-3.4284,7.3153,-11.095,10.3063,-21.94,-6.6914,-7.0794,-15.5154,20.0113,6.8578,25.8224,-8.2453,10.2353,-21.5039,16.3454,-17.9452,2.4941,12.0403,20.0324,-4.7569,12.0325,-13.6103,-4.7054,-17.001,-6.474,-8.8966,13.3478,11.4518,23.0274,-22.8849,-19.0567,-4.1422,2.5715,17.5055,21.1646,0.38706,14.2326,-15.3789,11.2875,-17.5167,-4.4867,-12.3683,9.6765,-19.9624,23.1516,-7.9202,13.5933,-24.2792,-19.5812,-4.4049,4.4891,17.8042,23.1962,0.14138,16.8056,-17.8142,14.5255,-19.3473,-3.8286,-11.961,7.7772,-17.3817,21.0601,-6.4593,12.4754,-26.7746,-20.4466,-3.6222,6.6467,22.7352,26.0395,-3.3877,19.8147,-21.8351,17.8628,-15.404,-5.9242,-15.0006,4.7409,-14.9021,18.1853,4.4926,12.7197,-21.0507,-0.82952,-20.5248,20.0882,-1.276,19.4441,2.293,-8.0869,8.9488,-11.3522,5.2098,-4.7975,-11.8921,-24.4383,12.2765,-11.5257,25.3129,14.2264,-20.8058,-19.7688,-2.1188,-0.23327,18.0853,19.1989,1.5739,10.2606,-11.0484,7.2203,-15.5783,-6.0957,-11.6045,12.2509,-23.1073,24.3056,-10.3407,13.2814,-20.7885,-18.6604,-2.6264,1.1798,17.2041,18.7575,2.6338,13.9901,-11.4372,9.6283,-16.5746,-6.6493,-13.0577,11.2205,-23.2727,23.6945,-9.4362,13.7089,-0.61398,21.3987,-21.1794,20.7328,-18.8077,-0.27815,-3.5613,-7.2808,5.675,-4.4125,13.1317,6.9846,-27.0303,-12.9158,-13.9107,16.2169,14.1532,27.9102,-0.70485,19.9716,-14.9213,22.1957,-11.3103,7.0407,-1.1288,-10.0355,13.0521,-3.03,21.217,12.3796,-27.7071,-12.8007,-17.869,13.7984,10.5358,24.7673,5.4798,12.1027,-0.34253,15.618,4.2232,14.9093,-3.5184,-20.5461,16.88,-12.654,23.79,11.0149,-28.6475,-10.3082,-18.3592,17.0416,1.5853,22.4412,32.2511,5.0324,32.2291,13.6974,-1.5702,-26.7361,14.3535,-14.4135,-33.9851,-13.5781,14.4958,21.8265,-3.9141,17.0976,-20.657,21.849,-14.7384,5.3013,10.7738,17.5002,-11.1423,5.4285,-21.2255,-13.3559,-20.8161,-10.7665,-10.8306,10.3561,9.4793,21.0949,-23.3662,-19.2685,-1.4742,7.6211,21.9429,17.27,-7.1704,15.7468,-28.3109,22.1836,-17.5799,14.8398,-15.1077,-0.47451,-12.9625,19.6741,7.7263,22.4052,-26.2739,-21.6137,-2.7105,5.1453,19.818,25.4165,-8.1481,16.377,-24.7128,25.9733,-17.3316,8.4029,-1.43,-2.5369,3.2969,1.5081,3.8649,3.341,-18.1273,-24.9591,7.4178,-6.3327,25.1087,17.9038,-14.1643,7.2528,-19.1175,23.3863,-4.3694,16.7828,-13.9418,0.57969,-16.5821,11.5082,-2.9091,10.6518,-17.9083,-24.9876,7.8484,-7.4382,24.7583,17.1069,-10.1679,4.4749,-10.3566,19.1056,0.31292,15.6452,-16.1912,2.9972,-22.5408,16.5601,-7.918,12.807,-25.553,-9.7297,-15.3314,15.394,10.4116,25.5312,0.62074,15.1312,-12.3855,13.7476,-11.0879,0.59769,4.5997,19.252,-16.5878,14.9252,-22.8309,-5.9925,-25.1371,-9.6924,-15.2088,15.2566,10.2884,25.3347,-2.6933,10.1444,-11.4898,11.6122,-8.1425,2.1505,6.6027,23.2271,-17.2549,17.0542,-24.0391,-6.35,-25.2234,-15.6195,-8.7581,10.346,15.6952,25.5713,-0.42078,-13.8528,13.0357,-15.3024,12.3024,-2.6816,6.0332,-14.7857,20.9965,-19.2406,15.8787,-3.7861,-24.9126,-9.346,-15.2826,15.1079,10.1733,24.9267,-4.4342,7.0293,-9.9652,10.5035,-5.2398,3.8133,6.686,24.3625,-18.0359,17.7333,-24.5974,-6.3446,-24.9246,-9.1715,-15.2468,14.8956,9.8336,24.5099,-4.8094,4.3853,-7.1462,9.0528,-1.8325,5.2142,6.1773,24.7044,-19.0646,18.4307,-24.74,-5.8968,-25.4586,-16.0253,-9.5808,9.8677,15.8768,25.506,0.75698,-14.6112,17.0271,-15.6987,16.1401,-1.4188,5.5623,-13.9599,18.1573,-19.1307,12.8722,-5.0407,-15.2452,-24.9434,-9.1796,9.846,15.0213,24.299,-4.8095,-4.9685,1.3862,1.085,6.611,6.1253,-19.8064,5.17,24.713,-24.5307,19.2164,-5.0875,-14.7302,-24.6233,9.7589,-9.286,24.0196,14.8532,-3.8388,-6.152,3.6099,-0.78728,7.3827,5.3009,-20.8202,4.6742,-24.9499,25.3113,-4.3795,20.2569,-14.5797,9.3832,-24.2276,24.5622,-9.2104,15.6684,-2.1183,5.3284,-7.4209,6.6619,-3.4425,3.1655,-21.7411,-25.1019,4.0868,-3.7925,25.7506,21.2428,-19.3194,-17.3767,-21.8121,8.2557,-5.8805,21.5238,14.9205,12.7544,12.4384,-8.2941,14.6215,-23.5935,0.70828,-20.5028,-10.5973,-12.6478,7.1569,-9.8466,8.0741,-7.514,-1.9411,-17.8006,-26.1999,7.931,-5.3052,24.1568,14.9646,15.5916,0.82534,16.8253,-19.2534,2.95,-16.9471,-8.1921,2.1046,-10.7184,8.2717,-1.4959,6.8285,-17.0731,-25.0714,7.7237,-8.4485,25.3018,15.7938,15.5972,-2.2072,20.0378,-16.4045,4.3578,-13.4677,-8.7188,0.91279,-8.7015,10.3893,0.17726,8.8515,-20.283,-18.5229,-1.578,-3.2443,18.9173,17.671,-7.2685,18.972,-25.8722,25.9255,-17.2515,9.2868,-14.1555,-2.9598,-8.7323,11.0178,7.7532,18.2506,-15.8344,-24.7731,9.1337,-9.5212,25.3915,15.2726,10.0753,4.3986,7.5166,-2.3681,-3.1267,-7.079,-19.3285,4.8141,-24.1924,24.6523,-3.706,20.3383,-17.2195,8.4698,-26.1841,24.8856,-9.3398,16.0756,2.2796,1.5006,-1.5528,-5.5751,-1.6414,-4.6607,-21.4549,-25.8508,4.4087,-4.5249,25.8678,21.1707,-15.349,-24.0851,9.5565,-10.0216,22.7777,11.6969,11.2721,4.1996,10.624,-4.7633,0.81613,-6.8464,-17.8337,5.4739,-21.8927,23.2107,-0.3214,20.2145,-13.5182,-21.4,7.7417,-10.9927,17.475,7.2528,17.0315,5.8335,-2.61,10.9401,-10.7541,5.2212,-13.9867,-11.2513,-16.0407,2.0957,-20.0959,18.8159,-13.184,16.4045,-0.65992,-17.2514,3.4054,-24.3379,15.8738,-3.8801,16.1264,4.7808,13.6778,-7.1409,3.9476,-26.5457,-21.1904,-14.4126,-18.9417,4.1247,-20.8027,13.4595,-5.6884,-13.1324,7.1856,-18.2211,19.4743,-6.7322,11.1034,21.1722,24.7314,-3.6421,3.4156,-24.6629,-20.7919,9.019,-6.8691,19.1742,-16.8817,8.8219,-11.8737,-13.2332,3.5348,-16.4453,14.8901,-6.4844,8.2194,20.711,24.0917,-4.3501,3.7773,-24.1461,-21.456,5.6173,-11.6064,19.15,-21.4306,10.7001,-12.2578,-15.4533,-20.3436,3.7678,-4.8469,16.5485,13.0492,20.132,-2.0037,20.7088,-21.9822,-0.48792,-20.9742,2.465,16.1697,-15.2948,13.2195,-19.4993,-4.9458,-12.2205,-17,1.9148,-3.9747,9.9793,8.8771,22.1627,0.32089,20.1731,-21.3698,-3.1195,-23.0928,4.076,19.1269,-17.4429,14.7917,-23.7693,-6.7448,-5.6498,-10.3387,4.7769,-2.1344,9.9066,7.4989,24.6721,3.0769,21.3343,-20.7894,-3.2239,-23.9845,6.6583,23.6426,-16.2856,17.1422,-23.6767,-6.7602,-19.2906,-24.5628,6.149,-5.2884,24.8286,17.9141,16.6204,-8.8597,25.8616,-25.5809,8.3611,-17.8855,-4.5618,-4.8252,-3.0185,-3.0014,-0.4789,-0.34026,-18.5504,-24.5238,6.1694,-6.7741,24.0458,17.4795,16.1329,-8.6782,23.7685,-25.3098,6.6109,-17.5539,-3.8427,0.1691,-9.4196,2.557,-9.7357,-1.6236,-18.3891,-24.3906,4.7669,-6.3817,22.1149,16.9,16.6588,-6.7633,22.0589,-24.0083,4.3151,-18.0076,-0.82505,8.9175,-13.215,8.6668,-15.3929,-3.2422,-17.1889,-22.4317,4.2965,-4.9156,19.7216,15.8793,18.3146,-4.4568,21.7657,-22.2897,1.7988,-18.9171,2.2006,14.9621,-13.9682,12.8468,-18.2904,-4.6487,-25.8507,-13.7294,-15.3622,12.3704,10.7293,24.5576,-8.1881,9.2635,-13.2816,17.7525,-4.9216,8.5755,0.38213,20.0039,-18.4474,16.4487,-22.0626,-5.8062,-12.8003,-24.118,10.3492,-13.5289,20.1156,8.5639,10.4994,-8.328,21.9817,-16.245,12.364,-6.9335,18.467,3.6216,9.6444,-15.8506,-10.8715,-22.0567,2.0221,8.6774,-7.5633,2.903,-12.7912,-9.3348,26.8638,15.739,13.6731,-10.0611,-15.3358,-25.2923,0.95681,-19.8679,21.9106,-24.1977,18.3071,-4.7901,7.6901,-5.8719,22.8484,-11.9431,17.8818,-2.5102,24.5196,12.0319,11.2289,-14.7878,-16.3278,-26.2709,8.5122,23.378,-4.0255,18.854,-7.337,0.50005,-23.0666,-9.9179,-21.8721,16.4135,-4.0562,19.8971,2.3512,3.6577,-10.5988,1.5067,-23.3052,-13.2752,-8.754,-26.447,2.774,-22.0178,-3.0445,-8.5633,-5.6288,-22.7393,6.5858,-20.6697,-2.611,-12.0946,1.2473,1.658,-12.6266,0.022842,-25.2982,-12.8863,-24.5761,-12.3692,-20.4295,15.0668,-2.134,19.2348,-21.8444,-2.0477,-21.7647,10.2523,-4.9451,2.5568,2.2508,-11.0007,-12.7045,-25.4781,-13.0968,-27.5612,5.4315,-21.4207,-3.427,-16.4058,-21.0594,1.349,-9.026,16.5284,9.4828,15.8759,-8.9133,23.5934,-22.3893,11.0384,-11.8828,1.2414,7.3596,-7.7612,-8.3983,-14.4142,-19.815,-25.022,-10.4292,-16.2487,16.2667,10.2585,24.4253,-0.25368,0.20997,-2.9431,-0.39071,-5.9946,-3.2342,-1.0758,-21.9839,21.4576,-21.9975,23.612,0.85767,-9.9394,-24.1886,15.909,-15.5387,24.1693,10.0383,0.80156,0.90342,0.82407,-4.16,-7.6947,-14.3856,-19.2388,0.015834,-21.1407,24.4389,-3.78,23.6917,-27.4128,-9.8926,-15.0822,16.6084,19.6998,26.7648,1.4327,0.17408,1.421,-2.6012,-1.429,-2.9882,-1.7769,-23.6813,20.0876,-19.5547,17.0491,8.5024,-4.7667,16.4946,-21.4128,20.6502,-19.7404,0.24403,11.7975,16.5427,-3.8268,6.3917,-12.3898,-7.3836,-19.9681,-11.1367,-8.9884,10.2386,13.0032,23.3507,-8.0502,-24.9137,1.4825,-16.3552,-1.5438,-5.6348,-2.6313,4.0912,-16.7017,3.6434,-24.4542,-13.7715,-24.5511,-3.9083,-19.8073,19.7885,10.0329,25.0403,-20.9466,-18.2954,-2.9071,1.3667,17.2892,19.1752,3.4591,15.0477,-11.9154,9.8726,-16.7254,-6.3802,-13.542,10.1288,-22.8021,23.9132,-9.37,14.1124,-0.27918,-21.3453,21.2102,-21.8863,19.1279,-3.3677,-6.1092,2.1631,-10.2028,4.6932,-6.6868,0.056804,-25.8686,-13.2164,-11.3481,11.423,15.5751,26.02,-13.828,15.3699,-17.9206,21.8583,-5.447,11.7989,6.1126,20.4124,-9.3765,9.3192,-23.1169,-12.5001,-23.3047,-9.1448,-23.4118,13.394,-1.9981,19.1867,-25.2827,-18.8748,-4.3278,7.2591,19.6827,26.0375,9.1268,-16.585,26.1636,-23.7062,17.5824,-6.1713,-0.73808,-10.4958,8.0247,-11.4178,9.107,-0.069993,-25.3396,-21.0426,2.1126,6.7117,30.9981,31.9199,13.8672,-16.997,26.8046,-26.9163,8.7556,-4.1164,-3.7978,-9.4193,2.1733,-8.1683,4.2252,2.0041,-22.8964,-25.5374,5.8516,1.8022,26.3724,19.5581,16.0539,-12.0337,25.5402,-29.1463,7.4291,-24.0616,-5.36,-4.6678,-4.2799,-2.5428,-1.4527,-1.2406,-27.5977,-12.1684,-15.7399,16.1438,9.8627,26.8914,2.3764,19.1363,-19.8413,16.348,-22.5448,-2.7896,7.854,18.2255,-2.7338,12.007,-4.3752,0.29299,-27.2909,-15.994,-7.8837,10.4907,18.6006,26.8681,-1.1845,-14.1003,9.5542,-15.6446,10.9026,-1.0023,7.0311,-14.3326,23.6807,-21.2279,15.1646,-8.1044,-26.9693,-11.9162,-16.8449,16.697,9.5114,26.28,3.971,22.5397,-18.9122,19.5118,-20.9093,-1.7575,1.7469,5.6432,-8.119,3.8419,-11.5625,-3.5863,-26.2486,-16.6971,-9.4044,9.5038,16.7201,26.1986,1.5414,-18.5023,20.5417,-20.259,18.5148,-2.1521,4.2424,-9.1206,12.2378,-13.1211,8.9155,-3.0443,-25.8718,-18.2365,-6.9506,7.461,18.4752,25.3039,-0.58538,-10.2761,11.3529,-8.932,12.1611,1.8254,6.8073,-15.592,23.1696,-23.8733,14.7437,-9.3955,-16.576,-26.0904,9.9023,-9.5419,25.8817,15.9102,-17.2165,2.714,-19.7675,20.6325,-2.7131,17.857,-11.3176,2.4524,-14.6912,12.4847,-2.7035,10.7435,-16.0045,-25.7341,9.4324,-9.9522,25.2303,15.6768,-15.9593,2.4159,-16.829,19.2308,-0.16476,17.5915,-13.0333,3.3695,-18.3318,15.2441,-5.959,11.2188,-15.6797,9.0796,-25.0932,25.9114,-9.2944,16.4709,-1.5626,2.3566,-5.4675,2.159,-1.8878,1.7087,-22.0967,-26.1039,4.5141,-4.6752,26.0463,21.2602,-24.6548,-17.5745,-7.4589,7.3756,17.9533,24.9248,-6.4512,16.2343,-22.3622,22.8827,-16.277,6.1974,-1.7807,-8.3939,4.7089,-6.5388,6.6085,1.8929,-25.0923,-17.5881,-9.5064,6.9594,18.4874,24.7701,-4.6184,16.2301,-18.493,22.1287,-11.9941,6.5563,-0.27245,-7.0333,7.6777,-5.1296,11.4127,5.2069,-16.8441,-23.9887,7.2412,-7.1447,23.6206,16.9824,16.3416,-7.3297,24.172,-23.3191,7.1584,-16.8327,-9.33,-3.913,-2.8107,4.4391,6.0438,8.1332,-15.7824,-22.7982,6.8273,-6.5828,22.0865,16.2526,16.5246,-7.7696,24.8616,-24.6062,7.7621,-17.5838,-9.507,-7.2557,0.35905,1.4644,8.6343,7.7853,-15.255,-20.798,3.7353,-5.1374,19.9558,16.5996,18.1243,-8.1396,25.7708,-25.4737,7.9107,-17.5997,-7.6344,-10.7388,4.8293,-2.8113,12.9114,8.4013,-17.4342,-20.0687,-0.12734,-3.657,18.0428,16.7055,17.8572,-8.022,25.1346,-26.1606,8.6068,-18.032,-5.7139,-12.5602,8.3098,-4.2652,15.7013,9.3821,-15.6346,1.9785,-15.4172,13.5306,-2.9062,9.4836,18.8598,23.3996,-3.1876,2.5677,-24.2753,-22.609,7.6235,-13.9864,20.8749,-22.228,13.52,-9.4166,-21.0224,-14.4429,-9.1253,8.016,12.6444,22.3256,-3.6355,20.7957,-24.6011,24.1272,-21.2106,1.7519,14.5555,6.9598,4.4556,-3.9838,-11.9432,-12.5005,-7.1921,13.7741,-9.6597,24.4332,-0.69849,10.9922,0.77843,16.6855,-21.3493,14.3767,-24.222,-3.9722,24.6337,17.0801,14.4042,-9.0396,-8.9286,-22.7111,-8.6418,-1.4802,-3.1721,7.3822,5.7418,8.8374,3.5065,24.5646,-21.1871,21.6181,-24.5421,-2.7537,24.2045,9.0788,16.8769,-15.6358,-7.0041,-23.8976,-14.9376,-19.6664,3.2597,-4.6269,16.2628,13.1803,20.188,-2.5447,21.1067,-21.9533,-0.12117,-20.8573,3.5755,17.7421,-15.409,13.5747,-20.5274,-5.6912,-11.0517,-25.2024,16.1368,-16.3787,27.5335,9.9016,3.4265,-11.641,11.6311,-14.2794,7.9008,-2.608,22.6225,5.3098,19.043,-14.7231,-6.0546,-23.1639,-20.2196,-9.617,-10.5826,9.7354,8.6267,18.9244,-12.301,11.9238,-21.7943,22.927,-9.6595,11.0851,9.8862,19.7598,-10.4534,7.8016,-21.1361,-13.3131,8.4254,1.3501,4.216,-8.56,-9.3685,-15.0807,12.5567,25.7408,-10.3293,11.9458,-24.6849,-14.5432,-20.1403,1.0935,-23.0159,21.0368,-5.7396,16.9326,-18.6665,1.5205,-23.0053,22.8818,-6.6943,17.4074,15.1539,25.0009,-8.7755,9.7545,-23.39,-15.6844,4.4161,1.3905,1.4147,-5.8381,-7.771,-12.0712,-24.1398,-9.2872,-21.9804,13.429,-4.5622,17.5652,2.401,4.5015,-9.0718,1.8409,-20.6892,-11.9499,-4.012,-23.7638,4.6327,-21.9251,-3.5292,-10.4291,-24.888,-10.0875,-16.6463,16.7637,4.3765,18.2479,-0.27678,-0.23009,-12.859,-2.7327,-25.6285,-12.5982,1.8231,-17.8594,18.9355,-20.1949,9.4024,-17.5627,18.1717,-4.3803,20.6284,-21.1167,1.1955,-17.134,16.1535,11.47,5.4262,-7.3269,-11.9116,-18.9953,-12.2096,-22.1838,10.8044,-11.2888,23.4887,11.1028,-4.3197,1.946,-15.5509,-0.8274,-14.5999,-3.1857,8.5501,23.3931,-15.8973,15.8741,-17.6419,6.8294,-25.0007,-6.0151,-15.9722,22.9133,11.9017,26.5407,-3.6091,11.9125,-18.3695,12.0035,-18.2227,-2.2426,5.9019,19.4729,-11.4933,18.619,-13.3432,3.2323,-22.6074,-8.3545,-12.3549,12.561,11.4574,23.2807,-2.705,15.8754,-20.3513,16.2312,-21.6535,-3.2298,7.2186,16.1987,-4.7105,14.2173,-7.3655,1.9307,-21.6903,-10.6725,-10.3524,11.3169,11.4404,22.7957,-21.9946,-18.2575,-4.4708,2.5796,16.132,20.4094,2.9912,16.4021,-14.2138,11.4512,-18.7035,-6.5428,-13.14,9.2088,-21.1777,23.0569,-8.9903,13.4912,-24.4509,-18.7253,-6.8671,4.8283,11.0601,20.6692,1.4844,18.0424,-19.6308,13.0355,-24.6751,-9.4607,-11.726,8.7023,-16.9438,22.5072,-6.5764,14.3939,-30.3563,-19.1723,5.1725,11.5044,25.9358,-6.7797,25.555,-32.525,17.2839,-14.6387,-24.1666,4.2241,-15.4352,21.3631,12.7313,-19.855,-20.5281,-1.022,-2.0435,17.869,17.8687,4.1319,9.9635,-9.2546,4.3892,-15.6747,-7.9209,-12.7799,10.9845,-23.4385,24.9438,-10.0721,14.8948,-19.9342,1.0841,-21.8784,20.7694,-3.5071,18.1312,4.993,-5.166,7.9935,-11.0908,1.3635,-7.6404,-13.9648,-25.2109,10.3587,-10.3498,25.374,15.7106,-19.2651,-18.1769,-1.8572,0.90724,16.0251,17.825,5.9499,15.3504,-10.4388,8.9076,-18.2042,-8.5316,-13.3538,10.1571,-23.2276,23.6299,-8.9813,14.1523,-18.5708,-16.9003,-1.0699,1.2717,17.613,17.9911,6.702,17.0882,-9.0835,9.2581,-15.9169,-7.5454,-14.4769,9.5188,-24.1153,23.9455,-9.9222,14.6472,1.1535,-19.3941,22.96,-21.1973,18.9894,-2.1165,-0.28061,9.5731,-5.5749,13.1581,-6.6908,1.163,-27.6399,-16.3408,-10.4778,7.5169,17.8251,27.0488,-3.388,19.6723,-22.4361,17.2411,-15.6726,-0.15232,-10.1016,-16.5577,7.2469,-12.3389,19.2024,7.2672,-27.9516,-11.5694,-13.9826,18.2815,10.4204,26.3097,-4.3077,15.3954,-9.4623,4.7818,22.2316,15.8332,-8.5629,-19.6265,12.102,21.41,-7.5395,9.3499,-27.5775,-10.7847,-20.4431,-1.7341,13.9094,17.3457,-7.12,15.8154,-16.5807,20.4937,-3.4798,9.9085,11.9063,20.6828,-8.9099,7.6862,-23.0859,-14.8769,-22.8733,-9.806,-17.4526,13.6554,4.7854,20.0756,-18.6986,-22.4425,2.2291,-12.1045,21.5999,15.0322,-8.3456,17.6852,-24.9833,26.5988,-14.877,11.4462,-18.7442,-6.97,-11.5191,7.2728,9.4848,20.3919,-25.6697,-20.1282,-5.6197,6.7511,17.5856,25.665,-8.2086,17.178,-25.8917,26.0967,-19.2755,7.1519,0.41507,-0.33964,3.2663,2.682,2.6735,2.7252,-17.6706,-25.1629,7.9715,-6.9576,25.3289,16.4731,-14.7362,6.9234,-18.3685,23.9134,-1.7712,19.3461,-13.651,0.76603,-17.5525,11.7926,-5.2332,9.9626,-17.8077,-25.0783,6.8889,-8.2776,24.3258,16.3225,-11.6658,2.7233,-10.8551,18.8703,1.6754,17.5396,-15.2516,4.7456,-23.4514,16.5931,-9.8509,11.0105,-25.74,-9.9947,-15.3365,15.5849,10.4068,25.7282,1.5548,14.48,-12.0741,12.1966,-12.2239,-0.79764,4.3154,20.3878,-17.0024,16.3169,-22.5908,-5.2908,-25.6683,-10.1947,-15.3539,15.3711,10.3644,25.5834,-0.52428,11.4764,-10.2961,11.5104,-8.7423,1.0188,5.7768,22.9949,-18.0856,17.3348,-24.2487,-6.1496,-25.3963,-9.9808,-15.1528,15.374,10.2168,25.3396,-2.3609,8.1456,-9.4062,9.8319,-5.9263,2.8002,6.3058,24.4356,-18.7446,18.2902,-25.0575,-6.1884,-25.5629,-15.5765,-9.7269,10.1507,15.5978,25.5574,3.3963,-11.7984,18.1119,-14.0031,15.8027,-1.3145,3.252,-16.6017,17.3015,-20.7612,13.3659,-5.0236,-25.0689,-9.7739,-14.9754,15.1314,10.1954,24.895,-3.813,5.5929,-7.9074,9.1031,-2.8843,4.6522,6.1667,25.0892,-19.3105,18.858,-25.2817,-6.2133,-24.8549,-15.013,-9.5345,9.817,14.9427,24.4268,-4.7785,-5.3381,2.6932,0.92526,7.6483,6.4504,5.5236,-20.2238,25.2073,-25.423,19.3827,-5.4137,-25.3092,-15.7791,-9.5036,9.991,15.7896,25.7389,0.68891,-16.6931,17.433,-19.1883,15.7677,-3.3244,5.4596,-12.2608,17.7869,-16.0931,13.5838,-2.4895,-14.8093,9.4248,-24.6702,24.8664,-9.4447,15.4918,-3.9402,2.302,-6.2262,5.7183,-0.3753,5.206,-21.3711,-25.7569,5.1661,-4.7353,26.0438,20.6584,-16.2186,-22.9364,8.5487,-11.1276,23.773,14.3157,18.5451,-3.6244,20.8268,-22.3438,2.4806,-21.0534,-0.75775,5.9249,-4.364,-1.7278,-9.0365,-13.7214,-15.7082,-23.5133,9.469,-8.8715,21.2506,13.9461,13.4545,-3.8798,16.5081,-21.3194,-0.32166,-18.0005,-8.8934,-2.4644,-10.4852,-4.8072,-14.698,-15.926,-15.7954,-25.2628,6.1259,-8.1621,24.5095,11.5367,26.7244,14.1352,-3.0404,18.8596,-24.1994,6.6445,-25.9969,-0.7381,-7.8549,1.6337,-5.6996,0.73121,3.8005,-1.3418,3.2589,-18.7899,-19.9001,0.24344,-5.3123,18.6789,15.63,-7.8831,15.6965,-25.1901,23.3121,-16.3946,7.8361,-16.063,-6.7315,-8.3282,9.0228,10.478,19.465,-15.3644,10.0898,-23.9554,25.8573,-8.8895,16.1353,7.9431,3.1287,8.4246,-3.714,3.6146,-3.9462,-20.1274,-24.5513,4.5482,-3.5742,24.5878,20.3016,-15.37,-25.3624,11.7979,-10.3803,24.9975,14.8184,7.89,3.9502,5.4396,-2.5259,-3.3567,-6.8531,-19.5047,4.213,-23.2072,24.144,-0.51454,20.5225,-16.5716,7.8798,-25.9894,24.0824,-9.7677,15.3398,-0.5176,-3.764,-1.0482,-8.8777,0.037606,-4.2216,-21.1926,-25.719,4.6944,-4.194,25.6217,20.9438,-11.7851,-23.4948,12.6188,-11.9186,23.1402,12.1451,9.9379,1.4068,3.0315,-6.6612,-12.429,-14.1568,-18.4945,2.9927,-23.1512,22.855,-1.6066,20.8184,-13.5949,-0.70914,-13.3309,13.5991,2.0393,13.381,21.3245,24.3781,-4.5379,1.9275,-22.4878,-22.0304,-0.27546,-15.1865,21.4275,-22.5432,17.2648,-7.1594,-10.349,-18.939,8.9097,-8.3437,20.3211,10.8632,21.1855,-3.5211,24.2267,-23.9077,3.2785,-19.7936,12.4261,17.4898,-2.3226,5.5099,-16.0994,-12.7607,-8.1578,7.8356,-14.3412,18.4231,-4.0184,12.5901,20.8856,24.313,-2.5172,3.954,-23.2992,-20.3531,12.6087,-5.6161,21.1184,-17.5722,10.8571,-9.4289,-7.0462,6.1218,-10.7232,16.3764,-0.52929,13.4823,21.5784,23.9732,-1.3176,2.9224,-22.0737,-20.5578,11.5279,-8.8606,23.1809,-19.3923,15.0571,-7.2566,-12.5737,-16.3089,0.080923,-3.2784,8.9123,9.2691,21.8577,-0.13251,19.9325,-21.42,-3.1873,-22.8385,4.7076,20.3547,-17.5862,15.2169,-24.3513,-7.1646,-7.8449,-8.9581,1.7094,-0.022975,9.5234,8.8026,23.9089,3.1603,20.7077,-20.2442,-3.4102,-23.4301,6.6271,24.331,-17.2479,17.4814,-23.9702,-6.7745,-6.7345,-9.2082,1.8968,-1.4097,9.2956,8.4321,24.4109,3.2769,21.0772,-20.647,-3.8418,-24.0251,6.7997,24.0081,-16.8067,17.353,-23.5291,-6.712,-18.0544,-24.6652,6.9375,-7.5871,23.7436,16.1406,16.9487,-8.2113,24.817,-25.3209,7.1027,-18.5032,-2.0781,-2.3418,-7.502,-2.2797,-10.6976,-5.1004,-17.6641,-24.417,5.8896,-7.3436,21.4334,15.1592,17.3974,-7.2631,23.3591,-25.2388,5.8554,-18.4895,1.0689,9.1255,-11.6557,5.9191,-15.7222,-6.0819,-17.1993,-22.7896,3.173,-5.8071,18.3658,14.7309,17.9747,-5.3516,21.158,-24.3852,2.6613,-18.9022,2.5461,14.8979,-15.1266,10.559,-19.8699,-6.4689,-15.0947,-19.0522,3.0439,-2.8779,15.6554,13.4792,20.1545,-2.2409,21.1977,-21.8659,-0.12003,-20.2116,4.7157,19.7753,-15.2899,14.7751,-21.5591,-6.6692,-26.8661,-11.0287,-17.5992,14.7347,8.6528,23.6912,-6.7095,7.6696,-10.0517,19.2158,-3.0162,11.8484,3.0196,21.6568,-17.0446,13.9964,-22.6752,-8.3583,-9.5329,-23.2986,12.0797,-14.4745,20.1576,7.3896,9.4303,-12.2823,23.2111,-19.5476,13.8876,-6.7986,21.3248,8.0602,9.1882,-13.9217,-13.7451,-22.8374,9.0401,19.9851,-2.8664,14.165,-16.8731,-10.3767,24.8489,12.0106,10.0385,-13.7644,-15.9751,-26.1178,5.3437,-14.5725,24.2164,-18.4569,17.8617,-5.0952,8.5752,12.2011,-5.6604,1.6897,-18.1307,-14.6994,25.6651,15.2537,10.2665,-6.545,-16.6596,-23.0839,4.7088,-19.4263,23.2111,-24.354,15.8634,-6.2046,5.1235,-17.4749,24.2931,-23.201,17.3958,-7.4192,25.1033,14.2603,7.896,-9.8757,-17.2921,-25.069,5.9223,14.0724,-4.5921,10.0752,-12.1689,-5.7573,-18.8256,2.0834,-23.2584,22.6665,-7.1445,16.2919,13.5507,22.347,-7.328,8.6213,-21.7252,-15.112,8.8826,3.9307,3.9342,-6.1372,-8.1837,-13.0068,11.1711,1.8423,-11.5832,-19.2183,5.8108,-10.5768,13.3524,25.5909,14.4397,-11.0442,-10.104,-22.2118,-18.9028,2.1859,17.8412,12.0134,-22.3328,-8.0258,-23.6875,-8.8865,-20.6926,8.7449,-5.9517,13.2224,4.1537,7.3942,-8.3966,0.92635,-18.4445,-11.3493,-1.6869,-25.2965,7.0826,-25.6718,-2.0434,-10.9371,-7.523,-18.7833,9.2348,-23.238,8.9386,-8.9503,20.5538,13.4593,9.8755,-9.0597,-10.9013,-21.9744,-15.1554,1.9801,-23.2677,12.0238,-18.1506,-3.3134,-36.4636,-10.4095,-15.1593,7.9531,1.4894,4.1617,-0.65522,-12.9445,-15.1557,-27.9398,3.1458,-28.2909,23.7325,-21.0403,1.5431,-0.88807,20.1073,-18.16,19.301,-20.7056,-1.6872,6.9636,12.0941,-7.3051,7.287,-13.3154,-2.1754,-23.8077,-12.9982,-12.508,11.5072,11.0327,23.0563,2.3712,-2.3183,-10.6498,-9.6437,-17.3122,8.5801,28.9272,-31.3534,14.0727,-20.5613,-32.9462,-4,-17.0521,22.3399,20.5677,-19.8689,-18.1053,-3.6533,0.60409,15.2746,17.8075,5.2742,15.4638,-11.152,8.6236,-18.8627,-9.2565,-13.6868,10.0029,-22.3785,24.214,-8.4031,14.4979,2.7505,-19.5923,22.4037,-23.0502,18.0847,-5.4765,-1.3067,5.5181,-8.3925,4.631,-8.6172,-2.3549,-25.9338,-15.5421,-9.8726,9.276,16.8848,25.8586,-26.3944,-19.0799,-2.5859,9.4251,13.5723,27.8778,9.3339,-17.7422,30.8815,-23.2533,29.3604,-0.41203,-0.93876,-11.7226,8.5958,-14.03,7.7204,-4.5746,-32.8003,-29.334,6.8406,15.0471,30.6585,20.3907,-26.7652,31.6763,-31.4804,2.7372,-1.6823,-10.5999,2.2916,-11.2,-3.1256,-20.7527,-26.3987,8.3604,-11.4708,27.2167,15.0645,18.4626,-7.6147,26.7697,-25.7096,5.7762,-23.163,-1.8136,-1.051,-2.5018,-4.9068,-4.7056,-8.3109,-26.6344,-16.1163,-9.1567,8.4207,19.1351,26.6325,-0.70338,-18.5959,14.7318,-21.3288,15.1448,-1.6733,9.2208,-4.9419,23.3185,-3.8892,13.6725,0.59998,-26.5597,-10.7751,-16.4977,17.0655,9.0212,25.7401,3.033,21.3855,-19.3522,18.2276,-22.121,-3.0806,0.53328,4.6489,-7.0035,6.0825,-8.0662,0.85592,-26.0616,-17.7114,-6.6738,6.9919,20.164,26.3103,-1.9706,-13.7295,9.6959,-13.1609,12.9243,2.1524,8.7087,-15.2141,24.4451,-23.7942,13.7765,-9.9984,-26.2315,-16.6566,-9.7824,9.7166,16.1846,25.6257,2.3945,-18.4804,21.1097,-21.3102,18.1258,-3.6961,2.647,-9.4404,10.5898,-11.3266,9.8876,-0.080241,-25.3641,-18.63,-6.8018,5.4045,19.1088,24.4241,-1.306,-11.5212,12.0898,-8.6626,14.7252,4.3481,8.7076,-15.0496,23.1691,-24.5861,12.0571,-11.7196,-15.9295,-25.6268,9.6185,-9.7131,25.2266,15.6595,-16.5178,3.4532,-18.0997,21.0668,-0.55671,18.6332,-12.5406,0.58313,-16.6203,11.8445,-5.4745,9.765,-15.7267,-25.2953,9.9103,-9.4196,25.7908,16.0794,-17.2682,0.50815,-19.0942,18.3549,-2.2833,17.2417,-11.8279,5.4016,-15.9251,16.7361,-3.4347,11.9155,-15.572,9.5185,-25.382,25.8483,-9.6286,15.9116,-3.8305,-0.66059,-4.4376,0.53147,0.51614,2.3167,-21.311,-25.9546,5.3066,-4.886,26.2634,21.0586,-24.9128,-17.7801,-7.4304,6.777,16.5729,23.7617,-5.8517,17.1217,-21.2944,23.6802,-14.8951,6.7151,-1.4789,-6.332,7.6327,-2.7382,12.7656,7.1477,-24.6133,-17.3397,6.2773,7.9773,17.3469,25.9972,-6.8655,14.7381,-27.2989,21.4755,-23.6947,5.3107,-5.7131,-12.8396,10.8408,-7.4489,14.0952,6.6671,-17.0378,-23.7594,6.3569,-6.9877,22.2892,15.1606,17.4478,-6.8476,24.6448,-23.5882,6.8485,-16.7996,-7.5082,-5.395,-1.0201,4.208,9.3387,12.3458,-16.9492,-22.6095,3.0982,-7.4272,17.9259,12.3831,17.7116,-7.061,24.9924,-24.071,7.5663,-16.8999,-5.8231,-6.7652,4.7358,3.3569,15.1378,14.558,-16.2988,-19.6979,2.6166,-4.578,17.7735,13.1245,18.2498,-7.3927,25.8042,-25.5092,7.5375,-17.8482,-8.6462,-13.137,4.0601,-1.9403,15.8289,13.7363,-18.2862,-19.322,-2.6438,-5.0564,13.1302,11.4771,16.5834,-7.7722,24.0463,-25.3715,7.8128,-17.6548,-7.3731,-13.4423,8.4811,-1.6707,20.1506,15.5404,-19.8716,-3.7007,-11.1482,11.0489,6.4706,14.8789,10.9552,15.7892,-0.85686,0.42742,-19.4494,-21.5851,6.7468,-20.7675,24.1401,-23.9096,20.5868,-5.5705,-10.6929,1.2396,-11.0194,12.1957,0.8085,12.2714,22.3478,23.9266,-0.46761,0.92939,-21.0954,-21.4194,9.0233,-13.7643,23.5822,-22.8837,16.6975,-7.1546,-21.5541,-11.198,-10.8661,8.6126,10.2294,22.2875,-3.0161,20.5919,-22.4638,21.8663,-18.9983,0.63237,14.2658,14.2283,1.6485,4.0296,-15.2839,-14.6722,-8.0546,-17.207,16.4057,-8.987,25.3287,8.9422,-0.09728,-20.087,10.5552,-22.7923,9.061,-5.4797,23.7627,12.6572,15.9777,-9.4864,-5.995,-22.7534,-6.6837,6.2148,-4.3371,13.6427,3.9655,8.3617,2.6039,24.0592,-21.9772,21.9146,-24.6828,-0.58281,24.5644,10.0539,16.5257,-15.7919,-6.8673,-23.9483,-7.6386,-4.5003,-0.14701,3.609,9.0148,9.6384,3.4649,24.7309,-20.4356,22.0495,-24.1636,-3.7772,23.8109,7.3719,17.4613,-15.785,-5.4895,-22.4001,-12.9844,-16.2241,-0.77866,-3.6112,8.7565,9.0478,21.5412,-0.34035,19.9187,-21.6097,-3.0262,-22.5802,5.3104,20.9811,-17.6598,14.7418,-24.8166,-8.1751,-9.9343,-26.3072,18.2047,-16.6048,25.473,10.2501,2.1509,-7.9471,9.5561,-9.9476,7.7502,-0.93217,23.0992,7.1861,17.1029,-16.2239,-2.7685,-22.4279,-20.7064,15.029,9.2699,-8.3476,-13.1561,5.2364,-13.5516,14.4184,24.6472,11.1267,-22.259,-7.9757,11.9479,-17.4715,5.4002,21.3389,-9.408,-23.3031,-17.5988,5.8511,-23.8493,23.7833,-7.9389,15.8678,15.494,23.0123,-7.1562,6.9978,-22.9193,-16.6894,12.363,8.137,5.4272,-5.4045,-7.89,-14.6049,-18.1042,3.1274,-23.0328,21.8112,-9.0473,13.4533,12.0678,20.3465,-6.2832,7.5814,-19.0877,-13.558,9.6665,3.3465,4.3555,-8.7935,-7.851,-13.8411,-13.6114,-3.0886,-19.3788,8.1543,6.1672,-10.4231,15.753,25.5427,-12.9146,13.8671,-7.8585,-25.3525,13.8654,-6.1699,17.4334,-23.883,-23.9012,-6.3514,16.9645,-3.3596,16.6614,-15.0152,2.9076,-7.2886,19.2167,11.3092,10.433,-9.3877,-13.6038,-23.51,-10.9807,-24.7582,10.7378,-17.5208,21.5631,3.7921,2.5964,13.8355,-14.6686,11.1449,-5.6954,-19.0738,-0.97704,24.524,-18.898,28.4198,9.7749,-13.524,-24.4627,-9.4886,-11.8265,3.3335,22.0151,14.2144,1.9914,16.9112,-17.5269,13.8102,-22.4085,-7.042,-0.87407,15.6658,-12.7715,19.4984,-8.1994,7.2372,-22.7742,-11.4541,-11.4256,7.8793,11.1801,21.4972,2.3525,20.7567,-18.6179,16.3562,-22.6163,-7.0158,0.078582,10.8408,-9.5783,14.6782,-7.2287,6.0839,-22.9475,-11.3701,-11.368,11.0425,10.5911,22.7294,-20.9603,-16.7561,-7.6625,2.7207,11.3511,17.5747,5.6681,17.6948,-13.0371,9.9045,-22.2232,-11.1427,-13.5169,8.8874,-20.3587,23.1397,-6.5127,14.9955,-22.5646,-13.8313,-13.0067,6.3107,7.8219,20.7934,7.3912,-11.9601,22.4979,-23.2019,12.487,-12.2316,-14.7867,-20.0042,5.9648,-10.2729,21.0673,13.5643,-18.7259,1.0908,-21.2203,16.8239,-4.3955,13.7271,6.7016,-5.7578,9.8088,-16.7314,0.91407,-13.064,-14.5939,-24.6104,9.6163,-9.2342,24.8316,16.042,-18.7762,-18.9595,-2.5024,-3.5554,12.9267,12.2428,6.7671,13.6787,-10.4193,3.0605,-19.995,-13.5608,-13.5579,9.7157,-22.6655,24.6991,-8.4473,15.7673,-18.8966,3.0892,-22.8205,21.0229,-5.979,16.2779,7.029,-2.1684,6.8578,-11.6728,-1.52,-11.0641,-15.5014,-25.3269,8.8853,-9.1006,24.6685,16.5707,-17.2968,-15.3225,-0.094443,1.6649,17.0566,16.4604,8.9461,18.9509,-7.8974,8.5005,-16.1967,-9.5717,-14.9642,8.5132,-24.5166,24.0845,-9.7956,15.0412,-18.2171,-17.8967,-2.3832,-0.39818,14.3534,15.8563,7.2933,15.4428,-9.5181,6.9198,-19.8098,-11.4901,-14.2918,9.8694,-23.2381,24.9267,-8.2571,15.2648,3.9562,-10.299,19.8597,-7.4756,17.1513,8.0001,-0.81912,16.9087,-11.0532,24.519,-10.9867,5.2101,-25.7984,-18.8684,-9.4192,3.2855,17.4463,26.7743,4.0607,-17.7432,23.5098,-20.6474,17.7085,-6.078,2.0598,9.8902,-5.4894,10.1762,-8.3336,0.61599,-27.1011,-17.6295,-8.7688,12.3479,18.0146,27.2463,-6.5589,-12.3534,10.4513,5.1067,14.3881,12.8141,-11.6629,13.736,-24.1758,24.2646,-15.0253,10.1987,-24.3298,-22.1311,-7.4413,-2.5548,18.2148,18.6574,-16.2875,-21.3429,3.2802,-10.6484,18.93,5.6215,19.7824,-8.4351,11.7558,-22.7546,19.8743,-14.9272,9.9874,-10.4079,-18.5076,-10.119,-8.9173,6.1965,11.5471,17.1473,12.0588,-25.946,-18.0779,-10.2087,7.8216,14.4299,23.8631,-7.4942,18.169,-25.0164,25.6705,-19.5444,7.2529,-0.1717,-0.44552,-2.0285,-5.0775,-7.4855,-10.2055,-17.0513,-25.8982,8.6384,-10.9346,25.0634,14.0001,-17.0992,3.9072,-18.486,22.5517,1.4639,22.7247,-11.3598,4.0674,-17.1934,11.5571,-7.9648,6.1846,-17.5241,-24.9609,5.7148,-9.967,23.027,14.472,-13.7653,0.94641,-11.3623,19.2802,5.0816,21.1783,-13.3182,6.1828,-23.313,14.5093,-13.1781,5.8585,-25.7356,-15.433,-9.8787,10.0713,15.5637,25.592,1.8065,-11.4497,14.9823,-13.3689,13.1705,-1.9438,4.4444,-17.4528,20.5696,-22.0227,15.9719,-4.7642,-25.7779,-10.2952,-15.4674,15.6149,10.1971,25.7874,0.71349,12.2474,-9.957,10.5723,-10.7443,-1.7662,5.2559,22.7436,-18.386,17.9367,-23.7335,-4.9618,-25.7385,-10.1161,-15.436,15.6096,10.3239,25.7786,-1.0897,8.9382,-8.4296,8.6301,-7.0852,-0.10373,5.9585,24.5355,-19.2879,18.9568,-25.0842,-5.6211,-25.3482,-9.9035,-15.2282,15.6187,10.3176,25.7903,-3.1768,5.6294,-7.8239,7.3337,-4.6657,1.6779,6.1995,25.4764,-19.7556,19.6109,-25.712,-5.7989,-25.5888,-15.6787,-9.9875,9.9702,15.434,25.606,3.3856,-13.6678,19.0874,-15.6177,15.775,-2.0023,2.7272,-15.5461,16.2214,-20.0824,13.7963,-4.3211,-25.1381,-14.9821,9.8902,-9.6668,25.3055,15.5462,-4.764,-6.0751,-0.76837,3.1717,4.1533,6.6809,5.8991,-20.63,-25.9649,25.9689,-5.4468,19.9882,-25.6999,-15.7514,-9.8436,10.0469,15.2521,25.4727,3.3264,-15.763,20.2251,-19.1288,17.0822,-3.4614,2.3413,-13.732,14.6728,-16.1879,12.6902,-2.2785,-14.9015,-22.9283,10.0846,-12.4712,18.5852,11.0134,9.7443,18.8388,-0.81287,22.0315,-16.6525,9.9717,-16.9512,-14.9795,13.8438,9.6918,2.8197,-2.6273,-11.714,-16.0482,-19.1772,-13.0885,-24.0307,13.7526,-10.8683,23.7015,7.9405,24.521,-4.031,22.9219,-22.6693,5.2691,-13.2669,13.9668,13.1063,4.5921,-9.2418,-7.3171,-20.3409,-30.7868,-17.7712,8.8422,9.9222,27.915,20.6544,-29.4781,32.1591,-30.1374,7.3058,-14.6413,-4.7215,-6.5555,-2.5526,1.7657,-16.0351,-18.9101,0.79805,-2.7089,14.8102,13.2309,-7.6517,14.9185,-25.0289,23.0252,-16.9025,9.0128,-18.959,-10.5477,-6.4425,10.4535,13.776,21.8532,-15.4769,-25.5953,11.0804,-9.3985,26.3228,16.9213,6.2399,6.0513,1.6494,2.3036,-5.2458,-4.6029,-20.5698,4.3111,-24.3924,24.6311,-3.3026,19.9188,-15.1929,9.4601,-23.8024,25.4672,-8.7879,16.2547,5.2631,-2.1052,9.8372,-5.805,5.9201,-2.5944,-20.3937,-24.6127,4.3519,-4.5319,24.7292,20.1418,-13.0789,-25.8988,12.4428,-10.7327,25.69,15.5032,10.0179,5.2037,2.6491,-1.6544,-9.0799,-9.0792,-22.3405,3.8699,-24.2905,24.8538,-2.7776,21.1645,-16.0806,8.4625,-25.9026,23.8945,-10.1292,15.1671,-0.56908,-5.6156,1.1912,-9.6726,2.9461,-2.6622,-20.7357,-25.2958,4.9095,-4.3342,25.8698,21.1087,-11.9919,-24.5879,6.8171,-11.0203,14.0131,8.5451,13.2859,11.5077,-2.74,-0.40563,-21.5205,-17.1065,-22.5334,3.4721,-25.3511,25.4733,-4.8356,21.4781,-13.0882,-2.3616,-9.5555,10.967,3.3872,13.2178,22.0603,22.498,1.0469,-0.51789,-19.4349,-21.9308,6.6448,-16.4227,24.0547,-24.0705,18.5868,-6.0541,-10.0794,8.0116,-15.5593,20.6951,-1.7243,16.2717,19.9537,23.8285,-2.7419,4.8146,-21.7416,-18.1852,12.0623,-5.1459,19.6334,-14.761,12.1324,-5.1202,-9.455,-18.8749,11.0197,-5.8924,22.4334,14.1842,19.1853,-4.0349,22.5408,-23.736,4.102,-19.0666,14.4509,17.9066,-1.9547,7.7566,-14.0343,-7.8471,-12.358,2.9692,-15.1366,17.4469,-1.4014,15.8918,20.3768,23.095,-2.085,3.2452,-21.2711,-19.2004,8.5351,-10.2045,20.134,-18.0264,13.9696,-5.1854,-11.6664,1.6543,-11.5823,16.57,1.2341,16.1967,21.1443,22.6428,0.087148,1.7301,-19.1454,-19.4269,7.8266,-12.8774,22.6568,-19.6259,16.7807,-4.4308,-8.8893,-8.4905,0.055337,1.1083,9.4011,9.9179,23.4315,2.9198,20.1984,-20.1546,-3.634,-23.1336,6.87,24.8244,-17.7334,17.629,-24.5382,-7.0321,-8.1137,-8.8579,1.5371,0.3035,10.764,10.1468,24.0572,3.2317,20.7592,-20.0556,-3.8211,-23.5489,6.6172,24.508,-16.9098,17.8653,-23.5995,-6.5179,-9.1486,-9.9692,0.17294,-1.0781,11.6879,10.8186,24.2774,3.2659,20.4879,-20.2535,-4.6647,-23.5719,6.0678,23.6218,-17.269,17.2392,-22.4255,-5.2201,-16.4,-23.9622,7.2013,-8.0227,21.3754,14.8619,18.1361,-7.1672,25.223,-25.4691,7.6284,-18.4136,4.7784,10.1779,-8.6633,5.0797,-14.7683,-6.9725,-15.6264,-22.3605,5.2934,-6.5872,18.0538,13.5345,18.5298,-6.7336,24.053,-26.3535,6.7277,-19.017,5.1632,14.7451,-13.1298,7.8964,-19.6209,-8.515,-15.488,-20.0243,0.60511,-4.4728,14.6913,11.9428,19.2046,-5.1508,20.653,-26.3882,-1.1419,-20.1697,5.0949,19.5987,-16.9416,11.5735,-24.2877,-8.4591,-12.0443,-14.0843,0.6115,-1.7235,11.1549,11.7539,21.7297,-0.44145,20.6147,-19.4229,-1.7242,-21.1286,6.8173,23.6108,-17.0027,17.245,-24.7095,-7.9342,-26.8575,-9.2167,-16.906,15.7188,21.1404,-6.54,7.6992,-8.9656,23.7342,18.9078,5.5723,22.28,-21.6884,11.1214,-28.7542,-7.0015,12.8943,-23.1084,16.4523,-19.5838,2.8531,8.3715,24.6352,-16.9356,17.3774,-20.2054,-1.925,23.1531,7.4397,11.7161,-18.8259,4.6905,-24.3915,9.036,25.0569,-11.5565,17.3345,-22.6381,-7.3529,23.6401,8.2272,13.4691,-18.7224,-8.309,-24.3904,8.3817,-7.3398,19.3174,-12.994,12.8505,-5.5561,15.1834,21.9332,-1.7529,10.6398,-15.9386,-8.4,22.5201,12.6486,5.7065,-11.2286,-11.5093,-18.09,6.4923,-14.2222,25.0305,-20.5305,19.4729,-9.8718,6.4536,-17.6377,24.0728,-24.1962,16.8947,-7.112,24.4484,16.8418,6.9928,-5.8953,-18.1436,-24.3323,6.7417,13.0971,-8.0258,8.2356,-13.9507,-4.4298,-16.6389,6.7409,-23.4194,23.6795,-9.3795,12.7531,14.7274,21.3952,-5.6392,5.9464,-20.1168,-15.1688,13.5883,8.6513,5.7335,-6.5005,-10.1102,-16.7825,-15.8596,10.0156,-22.8994,19.3661,-11.4722,8.5854,10.8631,20.409,-5.5246,11.2462,-16.4908,-9.1937,10.8369,-1.0867,5.9367,-14.9655,-9.6273,-19.9972,10.4402,-10.4158,16.1186,5.2636,-22.6654,-14.3626,23.8159,10.7282,15.5777,-5.9744,-8.3896,-17.1577,0.7172,18.2789,-15.6637,-22.0937,11.8528,-9.5483,4.0442,20.3051,-7.1664,16.5744,-15.8225,-2.1004,-16.0784,2.8823,11.7537,-8.4116,13.0078,-18.6016,4.7586,-18.7703,-21.7705,-11.6883,-14.8569,12.8869,10.2835,22.8235,8.5671,-17.9953,-15.4243,-4.6894,1.5673,13.1047,15.9972,8.9582,18.7746,-9.7548,8.4718,-21.3279,-12.1268,-14.7366,8.7402,-22.0334,23.8253,-6.5632,15.4652,4.7358,-18.0333,23.6965,-23.4882,17.2453,-8.6171,1.7594,7.8443,-7.67,5.2041,-10.3238,-2.8294,-25.7701,-17.0898,-8.0012,8.1541,16.7097,23.3802,-28.6193,-19.1877,-16.9432,11.1432,11.4891,26.3351,2.1199,22.9724,-22.1396,25.8472,-21.9623,2.4109,3.4796,8.0324,-6.9826,4.2832,-13.0615,-6.7694,-17.008,-26.2077,10.0559,-13.4536,24.9488,11.8701,19.2402,-2.6842,25.3059,-22.365,4.6642,-19.8428,6.7341,4.0276,0.46802,-5.4338,-9.1572,-12.4333,-25.4862,-8.4073,-16.4688,16.707,9.1033,25.6086,3.1012,21.4688,-19.0328,20.1384,-21.7744,-0.9509,-2.1933,2.658,-7.6565,1.4278,-7.7268,-3.1775,-26.127,-18.2252,-7.052,9.0301,20.5175,26.473,-1.9839,-19.7955,13.1361,-21.3434,15.9654,-0.41206,10.1217,-3.258,23.7437,-3.8377,12.8331,-1.9696,-25.2059,-15.9127,-9.2713,9.6575,16.0489,25.1635,3.9952,-17.3707,22.1134,-19.8684,19.4711,-1.4566,-1.8215,-11.7643,6.8006,-13.7864,6.5224,-4.0714,-25.5684,-19.2583,-5.0479,4.568,21.1287,24.9912,-4.2617,-15.2885,9.6608,-12.923,15.5802,4.7789,10.8096,-13.7453,24.7095,-23.1144,11.364,-10.8929,-24.8236,-19.3762,-6.7049,2.959,17.0387,20.7692,-3.3628,-14.0089,13.5254,-9.1473,20.3606,7.993,11.2559,-12.9899,21.9448,-24.2591,6.6601,-15.1726,-15.8069,-25.3169,9.6185,-9.2728,25.1598,15.6656,-17.7982,1.2352,-18.9267,20.2821,-1.1667,19.2525,-11.2615,4.0214,-16.0441,14.0778,-5.6354,9.372,-15.9472,-25.698,9.916,-9.4687,25.2411,15.192,-17.9418,1.685,-21.2674,20.3857,-2.7082,19.1635,-11.2381,4.1286,-13.1536,14.8418,-3.0901,9.4488,-15.6769,9.9909,-25.8638,25.649,-10.129,15.5056,-4.6694,-3.5079,-2.0152,-1.6378,4.3434,3.6744,-20.8166,-25.9681,5.7238,-5.181,25.9799,20.7643,-24.5501,-16.774,-7.5558,7.1167,14.5865,22.5017,-6.5804,16.8732,-21.833,24.1153,-16.1155,6.7658,-5.3168,-11.3557,7.9361,-2.8021,15.9447,11.1353,-25.2893,-8.9532,-16.6181,14.2542,4.9232,22.6489,3.075,30.2162,-11.9543,29.7143,-6.2541,5.1084,-3.2503,1.4627,-14.4338,2.7005,-21.5327,-7.5218,-26.1892,-18.4414,-12.4929,8.6413,21.3995,25.4144,-8.4644,16.5224,-25.9425,22.9733,-17.8322,5.0027,-7.5397,-14.6031,-3.7743,-5.8657,4.5053,6.8469,-14.862,-22.2773,6.7088,-7.4752,19.4693,12.4816,17.2225,-7.3157,25.0049,-23.9962,7.7203,-17.2399,-12.6957,-9.5437,-0.56845,3.8105,14.1444,15.5513,-13.348,-19.0919,4.3392,-6.384,17.2871,12.0764,17.9429,-8.105,25.5979,-25.3008,8.0331,-17.6873,-13.1975,-13.9896,2.4679,0.62388,15.9494,14.877,-12.7171,-17.4637,3.7842,-4.0812,15.5487,12.2913,17.834,-7.7497,25.513,-25.8949,8.1547,-17.8497,-14.1583,-16.2042,2.9392,-2.285,17.7944,14.4499,-13.9283,-15.2613,0.052093,-3.1536,13.2774,11.4769,17.2161,-7.8582,25.33,-25.6184,8.9027,-17.4478,-13.708,-18.4796,6.5011,-3.343,20.532,15.276,-13.2877,-5.3234,-6.1804,9.5524,6.8215,14.4187,22.1894,20.5382,3.6393,-2.6045,-17.3754,-22.0571,4.7643,-18.4842,25.0278,-24.6234,20.2966,-4.9202,-11.7755,-0.68913,-9.7317,14.1588,3.7702,16.4665,21.6133,21.3322,2.2754,-0.94768,-17.3827,-19.9607,7.373,-15.5022,23.6517,-22.3092,18.5099,-4.4631,-21.2491,-9.8163,-9.7459,45.0314,11.5974,42.2499,-3.5425,16.0201,-22.8269,15.0889,-18.2717,19.2576,17.2386,17.2995,3.8003,-13.5508,-13.5842,-10.6857,-4.6557,11.7374,-5.1112,26.5183,18.4361,0.87767,16.89,-25.2606,3.5267,-25.2742,25.2783,13.0286,17.712,-1.8753,-2.8228,-8.724,-20.6795,13.1639,-7.498,23.9783,15.2688,0.88666,-11.4103,8.2485,-22.8114,-0.5636,-19.8737,22.2189,9.7248,17.1589,-6.2893,1.5114,-7.622,-7.5745,-3.2505,-1.2188,6.599,16.6284,23.1542,2.6333,25.8434,-20.8145,25.8348,-21.4619,3.3164,22.6523,6.2363,16.914,-2.4374,1.9944,-4.4141,-9.1877,-8.4196,1.1694,5.9547,16.446,21.5062,3.6757,24.5917,-18.2237,23.7151,-18.394,4.0427,22.7585,5.4861,18.3557,-8.0166,4.6846,-5.4897,-9.8648,-9.6593,-0.77795,0.70313,9.2817,10.6938,23.1808,1.9655,20.1165,-20.1872,-3.0088,-22.3715,7.1875,25.1778,-18.2103,17.513,-25.2115,-7.6512,-8.5669,-23.414,-1.5566,13.0855,-9.5409,19.9795,11.7519,1.2154,-7.3163,1.2976,13.4344,-18.1957,6.9469,-13.3968,24.4569,5.2691,24.0637,4.9186,-15.2395,-10.8874,-20.584,-16.1339,7.5516,-22.8484,22.9493,-7.6852,15.0263,15.9265,22.3287,-6.218,6.4643,-22.7922,-16.5941,15.9158,8.8714,9.9987,-8.6402,-4.0788,-16.1287,-14.6321,9.4461,-23.6544,21.6493,-13.3833,8.6703,13.1413,19.0826,-4.3095,7.7064,-15.5554,-8.7351,14.3515,7.9044,5.8677,-11.0339,-12.3146,-22.0946,4.2589,-6.9124,-5.5255,-3.3783,2.6033,-0.29278,1.7231,22.0154,-9.4937,13.3113,7.6562,-20.6491,0.62197,-18.297,-6.7062,-21.5003,-22.4496,24.9366,-7.9454,28.6657,16.7849,6.0522,-10.1771,14.08,21.4046,-15.0106,-22.7231,-6.4788,8.7843,17.5832,9.9206,-14.2236,-6.8568,-22.2226,20.6611,8.1444,-11.949,-11.6906,9.9502,6.0419,22.0264,-14.33,-8.9492,14.0528,-19.6864,-5.7791,8.1397,-14.9705,8.4395,15.5918,-5.8223,-23.4781,-11.0388,-12.5379,22.3737,12.1444,12.5647,-19.1729,-12.8853,-10.145,3.9088,12.8781,17.6245,9.7362,20.8975,-7.6426,10.0284,-21.8541,-12.2446,-14.5546,7.0379,-19.7608,21.5858,-4.0291,14.7157,-13.2686,3.4978,-16.2053,17.0163,-2.3341,14.3198,13.6709,-4.1401,17.2843,-16.7649,4.2935,-12.142,-15.4649,-24.9932,9.0138,-9.1043,25.0893,16.4964,-16.6274,2.4782,-21.0565,16.5309,-5.8276,12.3662,10.7588,-4.2203,11.3276,-17.3845,-1.1048,-14.7746,-15.731,-25.0558,8.2332,-8.7677,24.4641,16.9429,-14.5916,-15.7406,0.83779,-2.1655,17.2223,15.387,11.4945,17.7009,-6.3273,5.0844,-16.0636,-10.7681,-15.6372,9.2259,-24.9348,25.1418,-9.8273,16.0295,-17.7175,4.7786,-23.6313,20.679,-10.004,14.8436,9.1899,-0.26348,6.8094,-13.3595,-2.7731,-15.4853,-16.6673,-24.804,6.0831,-7.8169,19.3919,12.8024,-16.6584,-17.2982,-1.4933,-0.9595,13.7672,14.9904,9.2768,15.8991,-8.5107,6.1814,-20.3561,-12.3264,-15.1311,9.8936,-24.2126,24.9645,-8.8684,15.3557,-15.9207,-14.363,-1.872,1.1357,12.802,14.3788,11.338,20.0008,-8.734,8.0257,-21.3817,-13.2841,-15.5683,8.9478,-23.709,24.4859,-8.2256,15.7051,3.4961,-10.6034,23.0618,-9.115,14.6548,-3.9081,0.81051,15.439,-11.9447,19.1063,-14.6772,2.1577,-29.7942,-18.8289,-9.8155,11.6765,18.9841,29.1436,5.7797,-16.9838,25.0185,-24.1756,16.0911,-11.1514,3.1295,11.3287,-7.2967,9.0438,-9.7894,-1.3256,-26.1687,-17.4308,-6.4675,7.3362,18.3614,23.7668,-16.2211,-32.1538,2.927,4.0488,15.4994,-8.098,42.8334,-24.4425,21.778,-15.5757,-22.0006,12.1834,-8.8736,36.1304,14.492,-24.4858,-15.8403,-11.6294,6.2781,10.5875,19.2856,-5.7615,18.2162,-23.5977,25.0784,-18.3581,6.6257,7.0175,5.1257,-1.3609,-7.4977,-13.1917,-16.8594,-15.0964,-25.4893,-12.8815,10.506,13.1055,24.9876,-20.7442,-0.76233,19.9541,-18.7604,22.5468,2.4276,-6.5445,6.8769,12.2033,-15.5982,5.0099,-9.733,-16.3221,-24.9781,4.1023,-10.7351,18.9417,13.2856,-18.607,-2.4179,-12.8643,18.4212,8.0176,23.1456,-8.7667,9.7397,-21.7689,15.508,-16.5902,1.6638,-25.6862,-10.0513,-15.6084,15.8316,9.3608,25.2231,2.7965,14.5664,-9.4583,11.42,-13.7011,-4.5017,4.0861,21.3096,-18.5819,17.3454,-22.1365,-3.2134,-25.5339,-15.5407,-10.0629,9.2475,15.9665,25.4701,2.4473,-12.6225,16.0054,-15.9341,13.367,-3.3564,4.1105,-16.5454,19.6421,-20.126,15.7702,-3.156,-25.8067,-10.179,-15.6363,15.4001,9.5358,25.1411,2.1142,11.2244,-6.6993,7.5385,-10.0686,-4.9624,4.7767,23.5437,-19.7686,19.5763,-24.2613,-3.5688,-25.9426,-10.1423,-15.6613,9.8246,15.2215,25.1441,-0.24847,6.803,-6.0214,-7.5267,4.3451,-4.2894,5.4733,25.3897,-20.1811,-25.1692,20.5772,-4.4924,-25.4809,-15.5431,-10.1209,9.4937,15.5028,25.4004,3.3538,-14.003,18.9347,-16.649,15.8274,-2.4657,2.3917,-15.3211,16.2658,-18.9519,13.5486,-4.0238,-25.1681,-15.3518,-10.1477,9.7412,14.8716,25.0267,4.1852,-15.4732,22.6677,-18.4588,19.4088,-2.3007,0.46341,-13.9874,9.7633,-16.7134,8.2022,-3.9372,-7.7169,12.2597,-16.802,17.4553,-11.5979,3.0379,12.6976,22.4857,-11.2392,12.6801,-23.9862,-9.9253,20.9558,9.3848,13.2721,-9.7065,-9.5477,-22.5193,-7.0834,-20.2002,17.2006,-16.0402,21.6707,5.0193,6.6852,-9.2187,13.5181,-18.7533,8.0401,-6.3099,21.7508,10.9561,13.5476,-12.9829,-7.2374,-23.2677,-24.1763,-15.4222,-19.996,2.5989,8.3255,18.3897,25.8319,0.42314,18.6366,-10.656,22.9507,-6.9062,10.2234,0.11831,-0.14785,2.0087,-15.8472,6.3973,-24.3213,8.901,-6.3555,-14.8623,-16.1579,-1.0298,-3.3559,11.0697,9.0488,-8.4275,15.4297,-25.5916,24.481,-16.779,10.2575,-20.2772,-12.9797,-5.6247,11.0063,15.898,24.5296,-16.0687,-25.3047,9.2603,-9.0645,26.0801,17.7873,3.5014,6.8067,-3.1757,4.6338,-6.7584,-2.5394,-20.0992,4.6853,-24.6351,24.6511,-5.1707,19.1364,-15.5901,-26.1816,10.5558,-8.6446,26.408,18.6242,6.9002,7.7786,-0.37719,3.9078,-8.5517,-5.5668,-21.5512,4.7163,-25.2057,25.0564,-3.9083,19.7019,-15.1066,9.0501,-23.4507,25.0428,-8.7453,15.9964,3.7525,-4.2123,10.4832,-6.3675,8.1306,-0.9895,-20.395,-24.827,3.9525,-5.1597,24.2577,19.8005,-14.6697,-26.059,4.7608,-9.5514,14.3101,10.3951,9.4855,9.4564,-4.7908,0.88401,-21.7509,-15.2764,-22.6957,4.0411,-26.2198,26.2374,-4.2466,22.0708,-15.5199,8.2836,-25.3302,23.468,-10.0168,14.7332,-1.3083,-7.8209,3.8523,-10.296,6.9323,-0.45645,-20.6888,-25.0611,4.4049,-4.0623,25.1375,21.3362,-7.8327,-13.3059,6.6215,-4.2519,13.7161,8.3661,17.8813,21.8972,-5.2428,5.956,-23.1685,-16.1522,-20.1922,4.7417,-25.3963,25.4442,-4.4195,21.2816,-12.9396,-3.4861,-8.4583,12.1396,5.6034,16.4872,22.0394,19.6554,3.9404,-3.0824,-16.3646,-20.5118,6.0367,-17.5976,24.3343,-23.5435,19.8715,-4.4178,-14.8026,-19.0186,6.2286,-1.0504,20.2782,18.5583,18.9213,-3.9655,23.8126,-22.5084,7.4623,-17.1409,7.0044,16.9045,-9.5508,12.7027,-14.4481,-2.507,-15.7226,2.7469,-17.1688,19.2597,-1.2327,18.3311,18.4964,22.1507,-2.3018,5.1094,-20.1803,-17.4267,5.894,-11.7842,17.7851,-15.6461,13.6524,-2.6799,-12.6875,-20.4127,8.7948,-2.2402,18.9702,15.8721,18.7248,-4.746,23.3438,-25.5829,8.0892,-19.622,10.234,17.3979,-7.4745,11.4165,-18.2965,-6.7431,-15.1686,1.277,-15.5237,18.5885,-0.81382,17.7283,19.541,21.5022,-0.34054,2.1699,-17.7076,-18.218,5.6563,-12.5997,19.9298,-17.0198,16.0644,-2.5455,-15.3929,-0.67297,-13.179,17.4534,0.87807,16.1121,20.0931,19.7674,2.1225,-1.5454,-17.6563,-20.382,5.5285,-15.499,22.4317,-20.2117,17.493,-3.8285,-9.7046,-9.14,0.19886,1.0283,11.447,11.3507,23.6896,3.0171,20.1948,-20.3473,-3.9839,-23.0142,6.673,25.2423,-17.7759,18.1695,-24.2565,-6.7691,-9.6859,-10.3457,2.8054,0.26306,16.0148,13.8666,24.0746,3.354,20.6156,-19.4731,-2.9667,-22.1745,6.3603,24.2938,-17.1201,18.1728,-21.3249,-3.7281,-12.8573,-13.624,5.0595,-1.2917,21.7958,16.5059,22.8884,2.9563,22.8901,-18.9728,4.6503,-18.1506,4.0698,22.3402,-11.4326,18.451,-6.0207,5.4786,-14.7883,-21.14,4.7811,-8.0339,15.3888,9.6709,18.7251,-6.6648,25.9751,-25.5409,9.3482,-19.0231,6.872,14.4702,-10.9417,4.6423,-19.3426,-12.7083,-13.5811,-18.3157,6.6422,-5.1438,23.1122,19.6605,-6.9581,27.4855,-29.191,-9.1857,8.0474,19.4567,-20.1787,7.4339,-38.2521,-12.9678,-18.585,-0.48679,-14.9121,9.2979,6.7815,20.5648,-7.3158,18.9734,-11.503,-4.1868,-20.954,7.0427,28.6378,-16.3274,27.4948,-25.7384,-12.1835,-11.8581,-11.2671,-1.9633,-1.5311,7.1266,7.2226,21.9114,2.1551,19.6571,-18.874,-2.4338,-23.0079,7.1965,24.2616,-18.157,15.9127,-25.9244,-9.2392,6.8617,-18.1551,24.3351,-24.5919,18.7143,-5.5566,24.6177,19.0556,5.7615,-4.925,-20.1174,-24.9218,4.7584,11.7905,-8.1877,8.4092,-8.7649,0.92461,11.9983,25.508,-11.5796,-20.137,19.0352,-5.052,19.5377,5.3348,11.1431,-9.0002,-20.3174,-24.9481,7.68,-6.2344,20.7591,10.6844,-8.7678,-1.84,5.9536,-17.2231,22.6208,-22.8894,17.3001,-5.6802,24.3615,18.5708,6.0906,-4.6568,-18.8175,-23.2399,3.952,12.6175,-11.8051,10.7775,-11.9218,2.5161,-13.5698,10.9013,-21.0477,22.3325,8.2004,-10.1128,13.6525,19.4783,-5.7919,7.1158,-12.6068,-19.5854,18.7504,10.9148,11.2806,-10.0592,-22.9317,-9.5102,19.2007,-1.1527,10.7165,-4.4993,-5.3398,10.9454,-24.7054,27.1831,-23.8886,-0.54638,-35.2201,-15.6868,13.0059,14.379,26.6945,16.4304,5.3619,11.4859,-6.0908,-0.12042,-13.2913,10.2119,-8.4128,18.4291,10.2157,-24.4393,-24.2524,-10.2856,-21.573,10.3669,24.9297,-13.1521,15.3667,-17.0135,-14.1252,-4.0662,1.2077,12.2943,14.4292,11.2995,20.8607,-9.7355,7.8887,-22.5421,-13.9221,-15.0225,8.2438,-21.5076,24.2152,-6.5048,15.7614,-18.4996,12.3714,-13.5582,12.6678,3.2506,16.321,11.8667,-32.4255,21.9801,-35.0314,9.3505,-13.3909,-13.1182,-24.2422,6.7919,-21.7695,21.7137,14.8421,6.386,-16.4014,23.9445,-24.2364,13.4767,-10.265,3.5144,10.1261,-10.3792,7.9687,-12.2025,-1.1096,-22.7928,-16.9127,-5.5306,4.9198,16.5842,20.9641,-13.3194,-24.9755,9.5993,-14.2525,18.8907,7.2228,18.9335,-2.7041,24.5873,-20.8488,6.4156,-16.7419,10.5181,7.4234,-2.5283,-5.775,-17.5156,-17.9329,-25.737,-8.2575,-16.2717,15.1792,10.0484,24.897,1.205,22.2058,-21.2267,22.0649,-22.0996,-0.30143,2.4099,4.1796,-2.5982,1.032,-6.3753,-4.5636,-25.1522,-15.8906,-9.3369,9.2496,14.7081,24.2814,2.3033,-18.8255,22.1776,-19.8747,21.4069,-0.062378,1.2043,-10.0064,5.7757,-14.7319,2.1941,-7.0597,-25.0855,-18.2155,-5.3507,9.6135,18.7662,25.1439,-3.0424,-21.7937,13.9108,-22.0698,18.0141,0.26883,10.2813,-2.21,22.1372,-5.2649,12.0788,-3.6412,-24.0878,-19.6405,-3.5725,4.6107,13.7975,17.9165,-6.433,-18.6896,12.6759,-14.3894,20.6472,2.5084,12.895,-11.2443,23.4396,-21.514,9.485,-14.2875,-21.141,-17.1967,-4.6708,0.50562,15.7041,15.9353,-7.4964,-20.6689,15.2992,-12.8324,27.8356,21.8326,15.1089,-8.267,20.9319,-22.206,-2.3912,-10.9738,-15.691,-24.9153,9.659,-9.3273,25.014,14.6403,-19.219,0.62216,-20.4843,21.4428,-0.90583,20.6871,-9.2064,5.3599,-14.7354,13.0444,-5.4014,7.6383,-15.0934,-24.851,9.905,-9.9364,24.7621,14.431,-19.9595,1.3208,-23.2942,20.3863,-4.0025,18.4265,-6.6732,5.2866,-8.0277,14.3028,0.52087,10.9117,-56.0448,7.0577,-32.3268,13.9237,7.3284,1.8622,26.2187,-34.7635,13.7612,-13.3352,29.599,5.905,-6.2374,-12.6491,-20.1738,-23.2278,-15.0265,-6.4324,7.5038,13.8183,20.0251,-6.8578,17.2304,-24.9097,24.7059,-19.4414,7.2714,-10.079,-14.9459,3.2247,-2.0663,14.6106,13.8937,-22.1238,-2.6529,-17.0419,17.2747,8.0772,24.6484,-7.264,5.5476,-13.3888,7.7011,-6.8795,2.1464,7.9851,23.3691,-16.2786,17.5778,-23.5188,-4.3759,-23.9984,-18.982,7.7008,4.8081,18.4886,16.7147,-5.1185,19.907,-30.3714,25.6564,-24.7141,8.3365,-10.9026,-10.0716,-6.441,0.60467,2.1951,14.2037,-13.0853,-19.9853,6.1361,-7.4512,18.2711,11.189,17.5805,-7.466,25.5036,-24.9327,7.713,-17.9273,-14.7683,-13.485,0.060046,2.5284,15.0425,16.0381,-13.2058,-18.2126,3.305,-6.111,15.8599,11.2242,17.8618,-7.8163,25.9992,-25.5621,8.3633,-17.9904,-13.6644,-14.9371,3.302,0.012013,17.2112,15.0183,-12.3951,-15.5963,2.5011,-2.8166,14.5073,12.4595,17.4637,-8.1495,25.8116,-25.9512,8.7827,-17.6155,-14.4514,-17.7519,4.6149,-3.8074,18.981,13.9058,-12.2476,-13.978,-0.091226,-1.6031,11.7191,12.1749,16.689,-8.3638,25.6468,-25.7629,9.4871,-16.9145,-14.9073,-19.891,7.1336,-5.2851,22.0367,14.8896,-13.4429,-5.9604,-6.8216,9.8565,5.5913,14.4,22.6182,18.1168,5.8789,-4.7889,-17.1879,-22.9157,5.0997,-19.414,24.6858,-24.4523,20.2541,-3.9115,-15.9098,-3.2875,-11.6596,12.0934,2.5145,13.2839,20.4651,17.8382,3.4655,-4.5604,-17.7193,-22.3203,5.0337,-17.9685,23.6975,-23.6843,18.7164,-5.8236,-20.3922,-9.9232,-6.9941,2.1638,9.136,12.4186,-5.0721,15.5059,-25.4607,28.2149,-17.2326,10.7889,21.0587,16.9231,6.382,5.4778,-15.2502,-20.3386,-19.0203,-25.0351,9.4706,-7.0628,24.2252,17.625,19.2847,-0.56063,22.9911,-24.954,3.2283,-21.2592,8.1646,5.2602,6.4041,3.2796,1.9631,1.5536,-14.4374,-24.7394,8.3852,-7.137,25.2673,19.9876,18.7593,-3.1174,22.3971,-26.8902,4.49,-22.8615,13.7281,1.2179,9.8997,-3.056,-4.7772,-5.6749,-22.5831,-16.0914,-5.4856,6.8091,17.2835,23.0439,-2.2532,21.6376,-23.6525,25.0354,-18.4457,5.4717,6.0222,1.1345,9.1796,-1.0216,6.5221,1.8014,-21.5055,-16.0206,-5.0871,6.1371,16.8501,22.4827,-4.4961,19.6098,-22.1129,24.7168,-16.3512,7.3342,6.748,-3.6899,12.4292,-6.1396,8.6802,0.39392,-10.31,-8.7305,0.51933,1.0544,11.7587,10.5527,23.3561,2.7144,20.6585,-21.0029,-1.913,-22.7589,7.042,25.8082,-18.6639,17.4727,-25.2733,-7.4405,-14.9356,6.0214,-20.247,16.2036,-6.7433,12.5952,4.7731,16.7017,22.386,-4.0408,9.3697,-18.9256,-0.5104,-16.968,15.884,4.8481,14.5451,-15.4195,4.8722,-19.2062,-8.1339,1.4455,7.3919,-1.7519,10.8069,-5.1865,2.2596,24.0835,-0.80935,24.803,-19.5621,-0.078593,-19.8685,-12.9789,-28.4614,12.8652,-10.3819,28.0917,17.3072,-14.8508,2.6897,-17.5074,17.2548,-4.3009,12.6576,11.6805,-3.9462,16.1121,-16.6608,3.0292,-13.9057,-16.0376,-25.102,8.7741,-8.486,25.142,16.9653,-14.1969,3.1877,-18.1472,16.8992,-5.9333,10.993,13.091,-3.9702,15.5915,-17.4855,-0.13334,-15.5476,-16.3216,-25.2704,8.1595,-8.3181,25.4831,18.2253,-14.8514,5.781,-21.2863,18.0354,-0.7322,9.7023,14.4643,-0.81692,12.148,-16.5117,-10.0178,-18.3418,-16.6472,-25.1024,6.196,-7.7642,24.0016,20.2518,-16.2628,-18.0768,-1.3471,-4.4171,11.8397,10.6442,9.8821,14.7143,-8.141,2.7167,-21.5816,-15.5596,-15.1371,9.0726,-24.2228,24.6978,-8.5618,16.2261,-13.9405,6.4427,-22.2835,21.5089,-10.6755,21.4688,4.7033,13.1523,3.3373,12.2013,-13.095,-0.81059,-13.5755,-11.7647,-13.8614,-18.9984,4.6135,-5.9269,19.6901,-4.7972,16.0934,-14.903,-13.6433,-2.0623,-0.037492,11.1185,11.8868,12.3305,20.4675,-8.5459,6.8056,-22.3726,-15.1765,-15.6175,8.8671,-24.1625,24.8552,-8.3141,16.176,-14.0097,-12.3165,-0.83294,1.3559,13.4624,14.4119,13.493,21.83,-8.2326,7.6496,-20.7878,-13.4303,-15.9896,8.4574,-24.413,24.7543,-9.8029,16.0158,9.5208,-6.8988,26.3421,-14.1285,13.8787,-8.8689,5.6334,17.4044,-7.7115,17.6595,-10.3188,2.6932,-26.3203,-18.7617,-5.1074,8.3475,20.3725,24.7207,8.1453,-15.1908,24.9029,-25.5753,13.2323,-15.6355,1.8207,9.0371,-9.4278,6.7075,-13.4044,-2.7425,-23.2812,-18.6594,-3.974,2.4192,16.687,17.2126,-16.7166,-12.1436,-15.346,3.3365,2.4367,10.9384,13.2533,-8.9053,25.667,-26.1709,8.7611,-16.3297,-14.3608,-25.4982,8.0308,-9.1524,26.207,16.1633,-20.1869,-11.2517,-10.916,5.8921,5.3899,14.1439,-6.2293,18.9152,-24.441,25.3932,-17.7922,4.7323,15.72,12.2172,0.6334,-5.3012,-17.1832,-21.1468,-14.2708,-7.9747,-20.5084,1.3755,10.4977,13.8333,-23.3169,18.0254,-6.4036,-14.9831,25.9993,8.8511,-4.3417,16.8822,16.2691,-20.4413,-4.374,-22.7994,-25.0791,-9.507,-15.885,15.6054,6.4238,23.2803,4.5141,16.7883,-11.0611,10.7095,-19.1265,-9.5296,2.7805,18.8859,-17.3516,18.1887,-17.3543,2.6725,-24.7609,-9.3381,-15.3465,14.8551,6.9274,22.559,6.1626,15.5628,-5.8912,7.5319,-14.4655,-10.9775,2.5602,20.5651,-19.8107,20.0646,-21.4038,0.61577,-25.4602,-15.9631,-9.7078,7.9733,15.965,24.7214,3.282,-13.4319,17.4782,-19.4959,13.3104,-6.8275,3.1817,-15.7299,17.829,-16.1591,15.4905,0.77043,-24.7573,-15.1272,-9.7946,7.8328,14.3988,22.7754,5.7236,-2.6429,10.8536,-10.5664,2.5332,-10.9903,3.6841,-20.6451,23.3795,-23.9619,21.2762,-2.359,-25.2441,-15.6926,-9.9103,9.0033,15.4992,25.3226,2.6314,-15.0125,18.9555,-19.2445,16.1893,-4.2819,3.1648,-13.8052,15.8601,-15.8249,12.4183,-2.1672,-24.7766,-15.1164,-10.0619,9.6336,14.4819,24.9887,3.8854,-16.6548,22.9679,-19.4229,19.4613,-2.4022,-0.17452,-11.9229,8.3796,-15.1802,7.7879,-4.1152,-1.4151,-13.91,15.5941,-10.6241,21.2819,8.5125,6.0732,-12.6521,14.6279,-21.873,7.1573,-9.6367,22.6894,11.7573,15.1295,-7.9835,-8.039,-20.3037,-2.3258,-9.4921,10.8721,-4.7639,14.2635,8.0597,10.2249,-14.8797,22.5775,-22.7699,12.2897,-11.759,22.1968,14.696,10.762,3.0755,-11.9622,-17.6456,-4.6288,-20.6134,15.6293,-14.7835,24.0512,9.4871,5.3487,-8.318,10.0541,-16.1099,1.4278,-11.1591,23.516,8.1382,17.8994,-14.266,-4.8095,-21.3508,-25.6002,-9.6052,-16.7799,19.1504,10.4878,27.3442,-0.7949,5.6739,-7.3787,3.5395,-7.5159,-2.1583,6.7436,23.1485,-19.5613,16.7371,-23.6916,-0.79752,-26.0905,-16.2267,-7.4805,10.6406,40.7429,42.7785,20.1479,-5.7922,29.5053,-8.1065,-10.7693,-8.5161,-2.46,-16.7929,16.4744,-22.6896,11.6119,7.9417,-10.9784,-12.3853,1.2718,-5.6849,10.7799,5.7071,-8.8679,15.9134,-25.6711,25.0884,-16.6459,8.4343,-23.0316,-15.3772,-7.5243,8.8636,16.1652,25.3499,-15.9783,8.2319,-24.7152,24.4428,-8.822,16.3525,1.7338,-4.7676,7.027,-6.528,6.6184,-0.40436,-19.4914,-24.4619,4.662,-7.2279,24.0682,18.1765,-17.4355,-26.0193,7.2491,-7.9362,25.745,20.2421,2.8506,7.307,-5.5724,5.2839,-10.9428,-5.3806,-19.8857,5.3256,-24.7362,24.572,-5.2459,16.4641,-16.8566,-25.8733,3.578,-6.7295,15.7369,13.9526,7.6677,11.1392,-6.6957,5.5403,-21.3445,-12.3045,-22.3291,5.0343,-26.3232,25.2267,-3.7875,21.2475,-14.7227,8.7189,-22.566,24.0871,-8.7124,14.8828,2.1655,-6.0375,11.0228,-6.4998,10.6407,1.3269,-20.1955,-24.4625,2.6257,-5.6662,22.644,19.7761,-10.0582,-13.9731,3.7832,-2.5972,12.8113,10.733,15.7948,21.9721,-7.8057,8.4254,-23.924,-14.52,-21.1976,4.2524,-25.6709,25.1751,-4.0775,21.7648,-6.8945,-12.1493,6.3584,-3.4593,11.0815,6.6662,17.2896,23.9579,-7.0637,8.3825,-25.3099,-16.7763,-20.2664,4.3892,-24.9415,25.2375,-4.1502,21.3016,-15.4599,-4.8276,-11.2088,9.6197,1.3576,11.7581,21.1535,17.0463,4.4735,-4.998,-18.1798,-23.8597,4.5808,-19.2797,23.5442,-24.5071,19.3424,-4.9797,-18.2093,-19.0635,1.138,1.0523,19.2181,19.5347,17.399,-3.0489,22.7061,-21.1722,5.3567,-17.1511,2.9434,14.6845,-12.0146,12.6533,-15.1261,-2.4282,-18.0379,-19.6964,-2.3389,-0.85419,13.8128,15.8873,18.1206,-5.5652,23.0314,-23.7483,3.2906,-19.8854,3.0317,14.5216,-14.6174,11.1736,-21.3343,-6.6696,-17.3558,0.73242,-18.1627,18.4864,-4.4897,19.7718,19.7715,18.0104,20.4756,-0.26952,4.4402,-14.2255,-16.8135,-16.8137,3.3673,-12.0046,16.282,-13.8337,14.3842,-0.63097,-0.63053,-11.9762,-13.5332,0.19371,2.3844,14.5769,16.1695,17.3715,-3.4094,23.8548,-22.7948,4.6682,-19.6237,11.9144,21.3605,-5.646,14.7851,-21.0836,-7.2537,-16.6396,0.43434,-17.9779,22.5299,-4.291,15.8781,19.253,18.4875,1.2779,-3.8789,-19.3685,-21.0936,4.1472,-13.7365,19.6945,-16.6748,14.4874,-2.622,-12.9715,1.8658,-11.7298,15.7142,-1.2827,11.3236,22.2167,19.1041,4.1933,-3.4478,-19.1423,-23.4806,6.1913,-15.8661,24.3462,-22.1097,17.5418,-7.081,-11.71,-11.87,3.2137,-0.28166,18.4249,14.9568,23.457,2.3558,20.573,-20.921,-1.4115,-21.3168,6.0836,25.433,-17.6715,18.6165,-20.6116,-4.1212,-14.723,-17.0803,5.4318,-2.2348,21.5881,16.1314,21.9395,1.6015,23.272,-19.9364,6.1506,-17.9736,2.8739,21.3996,-10.7714,18.4899,-4.3581,6.7705,-16.7087,-21.9377,5.8742,-5.2171,22.3541,16.7919,17.4626,-5.1215,23.6236,-22.3711,7.595,-15.6129,-6.4086,5.2928,-9.7772,12.6958,-1.3983,9.155,-10.245,-8.2001,-15.563,4.1483,-8.9237,9.0832,4.0393,19.8428,20.6887,-6.37,15.3795,-23.7966,-1.0475,-19.1464,11.9088,10.7573,18.0096,-11.5076,4.3534,-21.2117,-16.1296,-7.7668,-10.9298,0.921,-2.4829,6.3192,4.1507,23.0608,5.5745,18.9985,-21.7132,-2.2782,-22.6991,9.8257,23.6911,-14.9431,7.6709,-23.7629,-11.8426,-8.181,-8.1436,2.4622,-1.4223,12.7635,8.5434,23.4368,4.0094,21.7462,-22.2756,0.95864,-23.9447,8.4516,24.1824,-17.6701,13.2979,-24.2281,-7.0123,5.5566,-20.2153,25.0955,-25.0454,20.128,-5.4718,25.2424,20.4581,4.7504,-3.9692,-20.9107,-25.7879,0.35861,6.6172,-7.1887,5.3191,-3.7092,2.818,4.1751,-19.5389,23.2124,-24.5623,18.9172,-4.8802,24.7741,20.1714,4.1988,-4.4975,-19.9896,-24.4573,-2.0125,6.2641,-11.1948,8.7779,-5.2566,6.2156,2.9469,-17.5322,20.7696,-21.9445,14.5849,-4.7522,22.8583,19.0411,2.9369,-3.7317,-19.8918,-22.19,-2.6232,8.6787,-15.9047,13.5128,-7.5193,8.3505,6.2511,5.4602,0.69602,3.057,-11.3157,-5.7315,-0.91441,24.9857,24.7628,-23.216,-2.7184,-23.38,-27.2441,-13.5424,3.8395,-13.6195,22.8932,12.8807,-15.5449,-13.5963,-0.97503,2.1085,10.2742,13.7612,13.1917,22.1547,-12.2761,8.1626,-23.9819,-14.5878,-15.2089,8.6515,-22.8729,24.8637,-10.4993,15.6193,8.8069,-14.638,20.5263,-24.1491,3.118,-5.7166,0.0011697,10.5375,-15.5439,6.8316,-19.4462,-13.9072,-21.1977,-16.99,-5.2142,-0.69571,20.8992,22.1302,-24.7927,-9.053,-15.0917,13.1437,11.1681,24.3429,0.0049196,22.6571,-22.3407,22.9774,-22.6372,-0.67789,4.2329,5.8482,-1.4401,2.2123,-6.3732,-4.5797,-24.7564,-14.9541,-9.8053,10.5523,13.3711,24.744,0.69267,-21.3804,22.6421,-22.2313,22.3833,-0.89136,4.1812,-5.9847,6.0036,-11.6328,1.6338,-6.0864,-23.3769,-16.091,-7.9463,10.6865,13.9118,23.1371,-3.6247,-22.926,19.0266,-22.5674,20.0662,-0.88565,9.4352,-2.2071,16.825,-6.4519,4.7209,-4.1986,-14.3031,-9.9702,-4.2507,9.1809,-3.1153,22.3778,15.1574,-7.8148,-21.535,17.19,-21.2507,18.0239,0.082476,18.2172,12.721,-2.5376,20.1695,-11.4911,19.8955,-9.5726,0.27334,-17.0796,-11.6846,-1.1611,21.4616,17.4883,-12.8524,-39.0226,16.7131,2.2984,25.1338,22.8325,-9.1957,20.0535,-29.1418,-3.9747,-15.0776,-25.1926,10.4798,-11.1422,24.9403,13.727,-19.4805,1.5442,-21.7176,23.0508,-1.7782,21.8519,-8.9013,3.9945,-12.1426,8.9381,-4.0027,4.0303,-14.7002,-24.9108,10.439,-10.9203,25.146,13.9074,-18.9654,3.2911,-22.4734,23.0126,-2.3434,20.9498,-8.9005,1.2309,-9.7077,8.2912,-2.5921,5.3595,-18.6264,-12.5739,-6.5196,7.7338,12.978,18.9944,-6.215,17.6613,-23.9888,25.2703,-14.2431,6.9796,-15.2153,-16.4221,-1.0984,-1.7598,15.5256,14.9036,-23.7065,-7.8685,-15.0557,15.1415,9.8729,24.2657,-3.1746,5.2323,-12.3521,5.0976,-17.4288,-8.0806,5.6193,23.9373,-17.3112,20.4574,-18.4134,0.87758,-11.0081,-10.1951,7.4762,5.6001,8.0373,18.2699,15.9339,15.2958,-4.7913,15.6522,-17.433,23.5497,-19.5346,7.8444,-13.3965,11.2685,-18.6694,-13.1053,-21.7571,0.36847,-20.244,20.1202,4.2577,22.1009,-12.1601,-18.9228,6.0395,-8.1408,17.1899,9.5146,17.8491,-7.3429,25.8101,-24.9106,7.1291,-17.9112,-15.0545,-14.2783,0.73434,1.7915,17.004,16.8835,-13.0433,-17.2638,2.409,-7.0881,14.5968,9.0199,17.5879,-7.8197,26.0621,-25.044,7.8127,-18.0137,-13.4039,-15.7526,4.6855,-0.61095,19.6327,16.3018,-12.9043,-14.8684,0.50936,-3.6442,13.1274,10.9878,16.9802,-8.4466,25.676,-25.3989,8.1031,-17.3887,-13.6959,-18.5913,6.2897,-4.6334,21.7739,15.7322,-12.9909,-12.6842,-4.2285,-0.93837,8.8184,11.8072,16.184,-8.7615,24.8262,-25.1186,8.2568,-16.9652,-14.5124,-21.1793,8.7406,-6.767,24.6917,15.7569,-11.9497,-3.1912,-7.3399,10.2293,2.9115,11.7747,24.2515,18.4421,7.1069,-5.6472,-17.5865,-24.9973,4.6874,-19.6327,24.3241,-24.0893,20.22,-3.4262,-12.1343,-1.5541,-10.1691,8.1319,-0.49724,7.7322,22.7173,18.127,5.4978,-5.0864,-17.7167,-24.6302,5.7907,-18.5226,23.6275,-25.4382,18.4264,-7.1077,-9.3297,-7.6548,-0.14767,1.4008,12.6284,9.1608,13.1187,-6.6221,11.981,-19.9035,19.7785,-15.811,10.2213,-11.0954,20.8962,17.2632,9.9265,2.5831,-8.5142,-10.731,-11.7685,-17.0882,-24.4341,5.0578,-5.9862,24.6064,19.3139,23.6618,-3.7762,26.8307,-25.9991,4.9536,-20.5636,1.0929,-0.53202,0.35583,3.2453,-2.6046,1.984,-23.4206,-17.0577,-5.6569,5.1889,18.2695,23.3658,-6.2711,20.5274,-25.1296,25.3874,-17.9514,6.4205,-0.38246,-5.0467,6.9075,-5.6996,5.4716,-2.6423,-22.7243,-16.8258,-5.5698,5.0991,18.3858,23.5811,-7.2725,17.7954,-23.9775,24.3025,-17.2317,5.5436,1.097,-7.3014,9.7578,-8.8262,5.2061,-4.7667,-11.477,-13.0149,5.5945,-1.278,21.11,15.3967,22.8339,1.0127,22.8919,-22.511,3.231,-19.9804,6.6568,25.1105,-16.4054,17.718,-19.006,-3.3376,-14.0263,-18.378,6.5426,11.7992,-4.2009,8.4513,15.9887,-1.8195,19.6864,0.23676,-22.1496,-21.3469,11.0325,18.1824,-14.7592,-24.1844,9.9618,-11.3137,2.1295,11.869,-6.0676,17.0218,-9.3241,4.6613,23.2946,2.209,23.1432,-17.5766,0.046691,-18.3504,-14.4721,-25.6548,11.3037,-5.6977,25.241,17.123,-12.8377,2.5977,-17.2367,13.7687,-5.4376,10.553,13.9441,-5.1537,16.578,-20.398,2.5778,-15.7584,-16.2118,-24.9986,8.1339,-8.7677,24.5731,16.4801,-12.5983,4.5791,-17.992,16.9185,-7.8915,9.7754,14.4562,-2.6723,16.301,-17.2927,0.58037,-15.837,-16.824,-25.0992,7.6852,-7.9198,25.2765,17.9352,-12.1021,6.999,-18.98,17.6125,-9.9578,8.5477,16.6451,-0.75343,15.464,-17.0241,-1.1498,-17.0674,-17.7189,-25.9105,6.964,-7.5721,25.5477,18.9271,-20.7768,9.2777,-22.9497,19.2821,7.372,23.1128,2.8778,2.453,-15.6921,-17.9384,-18.0747,-27.2158,28.562,-6.5262,21.223,-11.8022,-13.2418,-0.087373,-2.0782,11.8537,11.3943,14.7761,20.7133,-6.8462,5.4072,-21.645,-15.3933,-16.2934,8.7725,-24.8261,25.0455,-8.9403,16.248,-9.8508,16.3144,-21.6612,20.5468,-19.3219,2.0799,16.3545,-1.7997,13.6228,-15.1631,-4.7148,-19.226,-14.4759,-20.3789,3.0447,-3.9138,21.4936,19.1109,-12.4744,-11.7994,0.47928,0.44756,14.2428,13.5001,14.723,22.0227,-5.8743,6.8075,-19.3653,-13.8914,-16.4402,8.6501,-25.0476,24.9733,-9.2716,15.8524,-15.5709,-14.8912,-4.6362,-0.76812,6.8845,10.3439,12.6252,19.3046,-9.4892,5.4737,-23.9174,-16.2027,-15.0608,9.0318,-23.8578,25.2109,-8.624,16.4394,11.6967,-8.1529,25.4891,-20.3899,5.8475,-17.1862,3.6113,14.8834,-12.23,14.2756,-17.3661,-3.0881,-22.7917,-19.023,-1.9958,3.9764,17.345,18.2671,10.9168,-9.8537,24.3555,-24.0542,23.7456,-18.5216,1.1413,6.5728,14.9725,-8.9869,10.4423,-10.4881,-5.1421,-13.7741,-15.9912,-15.2168,0.56146,-2.3362,2.6637,14.0127,16.6598,-12.7108,-5.2535,-9.125,5.6089,3.848,10.6959,15.2208,-8.7587,24.077,-24.6442,7.9444,-16.3587,-15.7077,-25.0269,8.2385,-9.3016,25.2812,16.573,-15.8876,-5.2201,-12.6723,9.0138,3.0917,13.5512,-8.2071,19.0643,-25.5642,24.1976,-17.5474,2.6103,21.5572,16.7934,2.7459,-5.7932,-19.2819,-23.9922,-23.5787,-16.3599,-8.2653,7.8609,15.3496,24.2189,8.7864,-10.3936,20.4839,-19.2828,13.8224,-6.7111,-2.0296,-16.62,13.198,-14.8646,14.1088,0.56695,-21.2789,-6.9203,-14.6269,15.6542,8.0161,23.9604,11.5946,19.9232,-4.0817,11.6092,-13.3456,-5.5994,-2.7103,13.9475,-19.4093,15.6127,-19.641,-0.8027,-21.3263,-14.063,-7.9322,8.2322,14.5188,23.6078,11.9323,0.34957,15.0592,-7.9517,5.6745,-5.8866,0.12761,-20.285,18.7935,-23.0971,18.6603,-3.0832,-24.9683,-16.4561,-9.4332,8.6921,15.3763,24.9404,5.9342,-13.6564,20.2847,-20.7653,15.4752,-5.5241,0.33436,-13.9684,13.1646,-12.9799,11.9623,0.44715,-25.1328,-15.8894,-10.0286,9.765,15.1773,25.4529,3.2336,-16.0029,20.1251,-21.0929,17.8897,-4.0218,2.7619,-10.9974,13.4587,-12.0657,8.9928,-2.7177,-24.9209,-14.9601,-10.8579,9.6739,14.0824,24.8108,3.0121,-17.723,22.3328,-19.874,20.1451,-1.3731,1.9199,-9.3435,8.5676,-13.7645,5.5953,-5.5585,-3.6394,-9.4606,11.5921,2.221,20.3375,13.3845,9.5833,-8.0196,18.2577,-21.2181,8.1498,-11.9553,19.9457,15.0624,11.0734,-2.5044,-3.7383,-9.7582,-6.4985,-20.1045,14.4959,-8.3429,26.1514,20.3381,8.9527,-8.621,13.0506,-22.3084,-3.6738,-12.0094,22.305,9.2303,16.2962,-7.9737,-5.6022,-11.2948,-8.9954,-23.9829,13.2874,-13.3622,22.3229,5.9682,10.9243,-1.8915,7.2255,-17.1961,-11.3838,-23.1901,22.1207,5.0546,20.6387,-14.8059,3.4116,-15.2975,-29.791,-16.4828,-10.0509,13.8186,8.4811,25.6088,4.567,-4.9459,6.6316,-9.2466,1.4294,-7.3828,5.9526,-22.4968,23.8394,-22.8685,27.6028,2.4347,-25.6292,-10.8615,-15.9104,11.486,9.3341,24.6225,2.4183,7.1042,-12.2189,4.0233,-18.6192,-7.5197,3.2342,23.2944,-17.5282,22.9959,-18.4845,1.0987,-48.2191,-32.7275,10.4214,15.1005,22.8517,28.2024,1.8845,8.2992,-9.0808,-9.5643,10.1451,-31.7812,29.9083,-22.5938,5.167,-6.5514,-11.316,5.3596,-6.5997,12.3207,5.0634,-8.1804,16.2099,-23.6624,24.0337,-16.1381,7.1168,-24.9057,-16.0483,-9.9845,8.5477,16.1962,25.735,-16.5066,5.5325,-23.7604,23.9784,-8.6174,23.1937,1.2441,-7.3482,6.9188,-10.656,7.8638,-3.3741,-17.3441,-23.6676,5.5225,-9.4693,21.7373,12.5042,-15.2651,7.408,-23.115,22.7822,-9.1482,12.6025,-0.11856,-6.5144,6.6248,-5.6132,8.6052,3.6612,-18.4021,-23.699,3.321,-7.2094,21.4382,18.4676,-19.0155,-25.7746,1.089,-1.2461,14.792,17.768,5.4025,12.7289,-9.1482,7.7253,-20.1605,-9.6048,-19.9808,7.2035,-23.9215,26.3888,-4.2223,19.9978,-13.1397,-15.0416,0.73988,0.70516,11.0121,12.1446,12.9191,21.4527,-9.764,8.8518,-23.9065,-12.841,-21.3876,2.674,-24.8882,25.2527,-3.7658,22.0502,-9.352,-11.4729,1.1707,-0.84159,5.6507,5.6896,15.5415,24.284,-9.4947,9.6485,-26.9679,-16.3867,-21.16,3.1234,-25.1237,24.9824,-3.6432,21.9078,-3.7416,-7.8369,7.2872,-3.2203,9.6178,2.9377,18.203,26.2711,-8.2606,9.5473,-25.7523,-15.6954,-20.1739,3.8221,-24.568,24.4756,-4.7735,21.0864,-11.5868,-1.589,-10.2337,8.587,-1.93,6.8071,23.827,18.3619,6.7337,-6.4255,-17.9163,-25.9652,4.7468,-18.6478,22.9908,-24.2319,18.8923,-4.6057,-19.5387,-17.6771,-1.2464,-7.1269,15.4788,14.6447,16.732,-14.0348,20.7893,-26.7413,2.8058,-20.0903,1.1804,20.2947,-13.5141,16.7766,-18.2833,-8.1011,-18.3639,-18.2884,-1.6906,-0.6234,13.7354,14.0667,18.3915,-3.6382,22.2484,-23.1096,1.7273,-20.6484,3.9526,15.9423,-13.7615,10.5158,-21.0328,-8.8999,-16.1971,-14.1967,-3.1431,1.0601,13.4994,15.3588,19.5043,-2.3631,22.4923,-22.8999,2.3491,-20.982,6.5492,20.4311,-14.4192,13.0817,-21.8508,-7.9669,-18.0402,-25.6328,17.6769,-12.468,19.5571,7.3179,18.1044,0.46774,14.0348,-20.5332,3.214,-22.0819,2.2906,14.5474,-20.294,8.7757,-28.2092,-12.8795,-14.4292,-13.0264,2.7654,3.0382,16.4987,18.2816,15.2353,-2.8687,20.7433,-22.3021,6.3245,-18.9598,11.1571,21.6729,-13.6567,14.5059,-20.4438,-6.2486,-11.4232,9.7547,-15.6784,21.2477,-5.3772,10.5361,22.8163,20.6258,3.3804,-1.7571,-20.3367,-24.1786,7.6223,-11.9253,22.2074,-21.1121,14.4769,-9.0373,-6.8492,4.3547,-7.192,14.3888,-0.77566,9.5763,24.2287,20.1002,6.7039,-3.8316,-18.9218,-25.3616,7.8964,-15.1264,24.3404,-23.1101,16.7106,-7.1312,-16.1901,-20.5513,5.4199,-4.2499,21.3333,15.6946,21.348,-1.575,24.4922,-22.2616,6.774,-17.5089,2.1217,20.5266,-9.481,17.774,-2.7706,5.4169,-16.371,-21.5513,5.7114,-5.7032,23.0051,16.9059,16.2205,-6.5186,23.1925,-24.1517,6.3355,-17.269,-8.3377,2.7104,-11.5363,9.4246,-3.8933,5.7212,-16.7843,-22.5032,4.8074,-6.1519,22.9524,17.4545,16.3825,-7.5796,22.2152,-23.8638,4.4122,-18.1079,-8.3596,2.3813,-13.0044,8.8137,-8.8701,2.5366,-6.6003,-13.0531,3.0028,-9.2361,7.5868,2.4279,22.7608,-5.0511,21.4656,-23.2046,-0.62762,-20.7127,13.5409,21.9228,-12.3039,6.6196,-23.679,-15.326,-4.897,-7.1535,6.5132,-1.9352,16.1524,9.2753,24.5538,1.2335,22.5824,-23.6944,0.15955,-23.8378,10.4649,24.1465,-13.3279,11.3247,-18.8204,-6.0725,-11.4855,-15.0566,8.0997,-4.034,24.7117,15.7421,21.5523,1.3623,25.184,-22.597,5.8898,-20.8665,6.51,20.3402,-12.7538,14.9452,-14.0397,0.060109,4.4343,-19.1346,-24.5027,24.442,-5.1786,20.1279,25.4082,20.8343,-4.8237,5.1385,-24.8742,-18.9474,-5.0687,-0.21344,3.7157,-7.6291,7.3154,0.73942,3.0273,-19.5618,22.1479,-23.9799,17.9074,-4.9193,23.7212,19.3188,3.1114,-4.766,-19.6796,-23.3744,-8.2562,1.4135,-14.1049,9.3056,-4.2535,9.021,3.7931,-17.3957,17.513,-21.5146,14.5751,-3.9382,21.7113,19.0786,1.8208,-2.9424,-20.1693,-22.7193,-10.6631,5.7544,-18.8165,15.2299,-7.5556,10.5712,1.4809,-21.0177,12.2758,-16.3633,19.9728,5.11,11.1225,13.5644,2.8614,-2.3277,-15.4631,-17.6734,-23.7648,4.2855,-26.8412,23.2465,-4.9475,19.4249,18.5431,-18.6388,21.4746,-29.8393,0.45969,5.8234,22.8941,-14.8108,-4.5176,-20.2606,-23.9439,-22.6916,-2.5577,18.2799,20.086,-23.8302,-10.3562,-13.4596,13.3986,11.3515,24.8156,-0.40997,23.02,-22.3163,21.9983,-23.1158,-0.92363,4.2395,6.7416,-2.0589,2.9874,-6.1574,-3.3397,-24.5391,-13.6017,-10.8833,11.1038,13.4227,24.9088,0.25397,-22.1594,22.9942,-23.3994,22.2157,-1.1974,4.7394,-2.8518,6.6386,-7.532,2.5989,-3.9509,-23.0927,-12.7809,-10.4102,11.5533,13.2081,25.1596,1.1845,-22.1343,22.4101,-23.1929,22.0946,-0.46455,6.7701,-2.5431,7.8071,-6.3295,0.85887,-4.2083,-18.3463,-20.5799,-4.3151,6.3904,9.4994,16.9572,-20.3223,-7.5863,20.5943,-20.4356,23.5352,0.90668,9.2464,23.2566,17.3726,-16.2135,-8.564,-22.1291,-14.02,-24.8308,11.2032,-11.3861,25.2429,13.5138,-21.2343,1.5474,-23.3874,23.2356,-1.8639,21.9377,-4.2885,3.5018,-7.0605,6.8524,-2.7238,3.3828,-14.2881,-25.1106,10.9174,-11.5154,25.4215,13.696,-19.9894,2.4376,-22.1748,23.3699,-1.9837,21.1661,-5.9755,1.701,-9.0543,5.6416,-2.249,4.7559,-18.5781,-10.8536,-3.5619,8.3349,8.2265,18.2573,-9.3066,17.931,-27.6526,25.5349,-23.0608,6.5381,-15.8941,-16.6951,6.8319,-0.30057,18.6368,17.8614,-23.1603,-8.3553,-13.9698,15.2146,8.6929,24.7221,10.2033,20.2995,-8.8631,13.9553,-19.4397,-7.9513,-2.2253,16.6506,-19.7673,16.101,-17.0448,-0.83788,-18.4587,-9.6185,-6.6718,6.5848,19.1138,23.0189,11.2354,17.6647,3.4841,3.006,-6.7018,-10.8952,-9.6644,23.1685,-28.2385,27.8587,-22.4429,6.2331,-10.1018,-17.8496,7.3327,-8.5224,16.3161,7.3025,17.9149,-7.6237,25.5385,-25.5247,6.8793,-19.4145,-16.393,-16.5829,1.9527,0.56574,19.2471,17.2125,-10.9526,-16.3526,3.4686,-8.0644,13.0731,6.8175,17.5863,-7.5248,25.4927,-24.7975,7.5003,-17.5953,-14.9993,-18.2545,5.1173,-2.2138,22.6137,17.984,-12.189,-14.4564,0.17997,-5.4004,10.7336,7.5561,17.0155,-8.104,24.9311,-24.6436,6.2798,-16.6984,-15.0069,-20.8496,6.8667,-4.9227,25.1572,19.1865,-14.178,-11.9871,-7.7644,-2.2431,1.8795,4.8741,15.4056,-7.966,22.3512,-24.2974,4.5545,-18.359,-14.7072,-22.6236,9.5016,-6.8086,26.8356,18.7409,-10.1987,-1.541,-7.8204,7.5042,-0.27928,5.9747,25.8471,18.2636,8.8031,-8.3038,-17.5222,-27.2495,4.348,-19.5646,23.696,-23.9611,18.5247,-3.7603,-8.0897,-0.5402,-9.4933,5.4295,-5.2033,2.566,24.1422,17.9262,7.616,-7.0272,-16.7367,-26.1347,5.8055,-17.2326,22.4623,-24.7023,17.5061,-8.4405,-2.0948,-37.4484,21.5526,-10.9221,11.8055,30.8483,2.779,6.6017,-24.8137,-21.6741,-19.6637,20.9246,-23.9427,15.7901,-9.2022,-15.3938,1.6253,-7.6267,19.3267,18.9809,-4.9323,-23.4889,32.8131,-16.3105,13.8881,22.1708,13.147,0.2833,-5.2177,-17.5674,-23.3686,-18.1872,-5.8431,3.852,18.5851,23.5323,-6.3511,18.4537,-24.2907,23.7135,-19.1803,4.4631,3.3546,-4.6731,6.8305,-10.045,-1.1683,-10.3936,-23.1932,-17.738,-5.4619,4.8964,18.7112,23.2983,-4.9835,18.7471,-22.5281,23.5247,-18.3595,4.6667,7.7983,-2.0638,11.6525,-9.2697,0.92894,-10.2351,-16.8601,-23.4964,5.9695,-7.423,22.8561,16.1397,19.0306,-5.2036,26.0533,-23.7879,7.1839,-17.0834,1.232,16.0913,-7.7004,14.9334,-1.9798,5.1572,-13.3122,0.87284,15.7287,-16.2671,19.6044,0.74564,20.3288,21.3002,4.0747,-1.6285,-16.5356,-20.2843,8.8546,-13.9273,-21.0995,20.0749,-4.0212,14.972,-11.6905,1.5141,-14.9824,12.1686,-4.7526,9.2183,14.9909,-5.4703,19.3775,-21.2094,3.3247,-16.8766,-16.2778,-24.9197,8.253,-8.7139,24.7465,16.2862,-11.4808,4.4294,-18.0671,14.5729,-8.7479,8.4889,14.715,-3.6191,16.0704,-19.5934,0.69162,-17.0319,-16.6533,-24.7331,7.2306,-8.3408,24.4923,16.9254,-9.721,8.2917,-18.0303,18.0759,-10.2869,8.1819,16.4437,-0.23337,16.4793,-16.3375,0.15172,-16.4117,-17.7045,-25.1657,6.855,-7.0672,24.5409,18.0636,-9.3243,-19.6478,9.4826,-12.9934,17.9083,6.698,16.8419,15.077,0.58814,-0.56766,-16.9685,-17.1641,-19.0216,5.6182,-25.7821,24.417,-6.6685,18.9623,-12.3013,-12.9084,0.71522,-2.0464,12.7349,10.5386,14.8375,20.8636,-5.48,4.9024,-20.4714,-16.1263,-16.033,8.6717,-25.0538,25.1295,-9.0483,16.383,-4.1834,-21.0777,15.5617,-18.0302,21.7085,4.8451,17.4908,13.9463,3.5193,-2.1181,-13.4728,-15.6304,-19.3327,2.1165,-22.7721,20.9964,-2.0545,20.5781,-2.1526,18.9978,-20.8753,21.959,-19.28,3.3052,19.5255,5.2797,14.7312,-13.3703,-1.4757,-15.8257,-18.9359,-21.2273,1.3769,-0.59097,20.5101,20.7417,-13.4747,-14.2153,-2.9674,-2.1431,6.1105,7.8337,13.8916,19.3714,-7.3296,4.34,-23.284,-17.6635,-15.9393,9.2014,-24.403,25.2585,-8.3574,16.3413,-10.7325,-4.0962,-7.3766,6.6008,2.504,9.829,15.973,-7.675,23.5323,-23.2821,6.0686,-16.9331,-16.6571,-25.1206,7.9636,-9.2692,25.52,16.917,18.2068,-2.4181,24.5073,-18.9396,2.7142,-18.2918,6.9238,19.6237,-10.3342,16.3022,-17.2941,-4.0327,-15.0088,-14.2458,3.5735,2.284,19.058,17.1003,6.0937,20.4156,-23.214,21.7024,-21.674,1.0192,15.0057,5.5293,10.8118,-14.0825,-4.9344,-16.9134,-20.9435,-21.4629,-2.7242,0.082674,17.3614,20.291,-11.9676,-21.5478,12.109,-11.2709,24.5507,10.4533,-22.4926,2.2184,-22.8461,22.2275,0.40598,23.7723,1.7655,15.9401,-5.0686,10.715,-10.1863,-7.8727,-23.5152,-8.5511,-15.4663,15.4273,9.1417,24.4234,7.6736,19.1932,-8.9943,13.4214,-13.889,-2.406,-0.7481,13.3302,-16.6568,12.8856,-18.0747,-2.248,-24.0417,-15.6678,-9.325,9.4644,15.2238,25.3885,6.9429,-12.4821,20.7774,-18.4124,15.7068,-3.9886,-1.1289,-13.8972,10.9583,-14.5006,10.8124,-2.0257,-22.5349,-14.4596,-8.4941,8.5421,14.6674,23.155,7.6711,-2.7186,14.5937,-7.767,9.9584,-1.2915,0.98069,-18.3551,17.4595,-21.5453,16.1446,-1.3472,-25.1413,-15.6154,-10.0435,10.0963,15.0576,25.3563,5.6177,-15.2533,22.0884,-20.5048,18.1147,-3.693,0.10332,-11.4054,9.0066,-11.6527,6.3463,-2.9711,-24.8512,-15.4094,-10.1226,10.4099,14.6499,25.6811,1.9843,-18.0666,20.0412,-22.1561,17.2769,-4.6785,4.7718,-6.2888,12.693,-9.0199,9.9159,-0.58493,-24.9999,-14.6882,-11.2469,10.661,14.5905,25.8072,2.0297,-18.6681,21.1143,-21.7688,19.4066,-2.9955,3.459,-7.1345,10.026,-9.4348,7.0391,-1.8865,-11.3818,-5.492,7.4862,0.079733,20.5187,17.4761,10.9969,-21.7176,24.2152,-24.5376,12.6154,-12.0177,13.2354,11.4852,5.4202,9.6694,-1.1697,-1.6238,-14.708,10.2452,-5.5515,18.4808,16.4568,19.8082,29.9697,-45.5877,5.2929,-29.112,-26.9521,-0.85654,-25.3462,19.8652,8.1019,-11.9946,-20.8708,13.1427,-3.5631,22.9751,19.9777,14.1586,-12.2599,22.8361,-27.6284,15.9527,-13.5051,11.712,4.338,9.3713,-1.814,6.9028,-0.71553,-12.7892,-33.4808,17.5121,-3.0944,23.1915,20.8699,-10.8267,13.4951,-30.995,-13.4259,20.408,1.9235,17.3877,-8.7722,2.7548,-8.3531,-23.4803,15.1675,-19.0711,21.988,2.9681,20.2469,9.6949,9.0789,-11.7886,-12.9892,-25.0174,16.3557,-2.0902,19.3255,-13.4596,5.5122,-10.161,-25.3046,-15.2383,-12.8726,7.4585,12.5692,23.7028,5.2233,6.2476,-5.6048,-0.29121,-13.4803,-9.7226,-0.12749,21.9007,-20.8448,24.0582,-19.663,2.0874,-23.5817,-16.1799,-10.3704,4.4761,16.7032,24.3993,7.6514,6.0802,1.1353,-2.1461,-6.6622,-8.4628,-5.009,23.0715,-24.3355,27.7179,-21.7622,4.593,-24.5488,-11.7499,-16.4749,11.8587,7.7596,23.2358,4.6639,13.1245,-13.8825,5.8409,-21.6269,-10.2601,2.3835,19.7461,-12.0377,20.832,-12.9287,2.8811,-6.3154,-9.7443,11.5798,-7.0345,18.3512,10.0506,-4.2681,34.3192,-21.4535,38.2186,-18.3218,8.5772,-27.4262,3.2453,-21.8994,19.1264,6.3095,27.2776,-4.5562,-11.6642,5.4984,-8.5033,7.8301,1.064,-6.2132,16.1653,-23.4689,22.2973,-18.9072,4.7906,-25.6977,-16.4659,-8.7966,8.7174,17.1903,26.0681,-22.7323,-4.3519,-23.5734,11.5659,8.3671,18.3349,2.9307,-11.5205,10.581,-20.9055,4.6754,-9.6798,-16.9196,-23.3873,6.2898,-4.4414,24.1858,16.7844,-14.1647,0.37972,-21.1446,14.44,-9.0471,22.0671,14.1729,-2.2506,-7.2069,4.1058,-9.8601,10.0248,-1.7238,8.9989,-14.5279,-18.794,3.0777,-10.8874,19.4946,3.1658,18.0091,-15.4971,-13.4161,-2.6463,4.3204,8.9371,13.4313,11.3087,21.1253,-11.4196,10.3699,-24.4729,-12.6662,-21.0304,3.3894,-24.6449,24.8936,-2.4609,21.5535,-10.1705,-9.4983,-3.2541,2.1083,1.9194,6.556,14.2581,24.2606,-10.6371,11.0355,-26.7401,-12.1427,-21.2166,2.5947,-24.2828,25.0316,-2.7388,23.7409,-2.4304,-2.4355,4.2435,2.4253,0.34775,-1.4168,17.4206,27.0238,-10.0431,12.0786,-29.4741,-20.5293,-20.8323,2.375,-24.5203,22.3679,-1.4684,20.0406,-1.7974,-7.1812,5.6974,-9.0907,-2.2011,-12.0858,16.9492,25.7385,-10.7544,10.4493,-27.8298,-11.6721,-18.9225,3.4338,-24.4208,22.0407,-8.8946,19.5802,-5.5741,2.6862,-7.0116,7.2408,-1.0805,4.2947,26.2201,18.3492,9.7846,-8.4876,-18.1735,-27.3129,5.0688,-17.8566,22.545,-23.3709,18.648,-2.7579,-14.4257,-23.8603,1.0743,-13.5975,14.417,7.3064,20.1396,-2.5773,24.8662,-23.9149,2.1026,-20.5354,7.1661,16.8337,-10.8743,5.7691,-22.0482,-15.0068,-14.8554,-15.719,0.22913,-4.8532,11.6304,8.2165,20.3038,-4.7759,22.8601,-23.9871,1.2484,-21.9796,7.7675,18.9754,-11.8723,9.303,-22.6985,-13.0814,-14.7955,-14.2959,-0.82204,-3.1677,13.2326,10.3395,21.1727,-2.2117,23.0335,-23.6389,1.261,-22.2326,7.9511,20.7722,-12.7142,9.9585,-21.7995,-11.8368,-16.5375,-14.5367,-1.6654,-0.84905,16.1007,14.861,19.9529,-2.0515,23.1869,-23.614,3.9857,-21.6294,7.3449,20.6811,-13.0572,11.5829,-19.7522,-7.9441,-8.7798,-16.6479,12.4889,-7.4869,19.5973,8.6972,22.2903,3.7437,20.617,-21.0327,0.50953,-23.4511,12.3364,23.1762,-10.7488,12.9697,-22.0328,-10.7826,-8.635,12.0548,-19.8068,21.9989,-11.4664,8.8252,21.5997,24.2514,1.4649,2.95,-22.7591,-22.3062,12.4191,-7.6315,23.6283,-19.1217,9.8113,-11.9853,-18.124,-16.6161,-0.57138,0.66112,19.0933,18.3881,17.1644,-4.2271,24.33,-22.9804,8.4914,-17.8182,7.2968,19.3058,-10.4811,12.7305,-14.8932,-3.3401,-9.0115,6.1118,-13.2053,13.8429,-6.0894,5.3942,24.1683,20.4349,5.7665,-2.333,-21.0918,-26.356,8.6331,-14.1388,22.2307,-23.9152,13.3656,-9.042,-4.6539,3.6291,-6.7565,9.8844,4.1464,8.3388,25.949,19.5951,8.3586,-5.3036,-31.3586,-36.2451,7.9917,-15.0996,23.1197,-23.466,12.3177,-0.82111,-15.6301,-21.6641,6.1847,-6.9185,23.3087,16.5648,17.3777,-8.5443,23.8844,-25.0452,5.6248,-18.2882,-6.6541,2.2835,-10.5749,7.1942,-6.2357,3.0061,-16.8531,-23.0906,5.6177,-6.9135,23.2945,16.751,17.5283,-6.6154,22.8603,-24.1641,4.4691,-18.6076,-4.9204,4.8537,-11.3456,8.3868,-8.8458,1.388,-17.4905,-23.0722,3.9003,-6.1477,23.7218,17.9204,18.1159,-4.7525,21.8141,-22.3672,3.2552,-17.7817,-2.7824,8.809,-12.2446,11.1942,-10.2963,1.9554,-5.6516,-11.9273,-5.6863,8.9595,11.9341,20.2407,24.322,-2.8225,-23.1342,24.1334,-21.9065,0.10745,12.3544,23.89,11.4141,-10.3589,-4.3088,-14.466,-12.1184,-18.2349,9.0148,-6.8781,25.7337,15.8053,22.4375,-0.29562,24.7018,-22.4263,3.9758,-20.8422,3.8752,16.4314,-9.8262,13.6541,-7.8168,3.158,-17.2887,-24.7569,7.1654,-9.1218,25.3482,16.6505,17.8981,-3.8353,25.84,-23.0514,6.5419,-18.3202,-0.89377,9.7661,-7.7767,11.8708,-2.1402,6.1381,3.6073,-23.4815,-18.613,20.5407,-5.0953,16.9683,22.6219,-4.3444,18.9114,3.5054,-23.1925,-19.587,-10.9763,10.438,1.3725,-15.5309,9.4648,-4.0029,1.692,-16.6025,18.194,-19.9786,18.8616,-0.60891,21.7978,19.2139,3.9822,-2.4951,-16.9141,-21.9865,-12.5193,4.9597,-17.256,15.7651,-2.0269,13.8225,-6.9794,-20.6717,11.1629,-14.9514,19.5484,6.9112,16.0308,14.1541,0.49367,-0.63358,-15.6189,-14.7124,-19.4695,3.3472,-24.6002,21.991,-4.5104,19.7252,-5.9857,22.0727,-23.0182,23.4048,-19.5905,-0.14694,15.4763,2.9623,9.8666,-11.0508,-8.179,-16.66,-17.5861,-13.9356,-5.6718,3.7698,17.4051,22.3632,-24.9519,-13.4069,-12.0673,11.794,13.1393,25.4853,1.1946,-21.9846,22.7885,-22.981,22.2637,-0.27663,3.3422,-3.0649,5.184,-6.1758,1.4128,-3.5576,-24.7742,-13.5227,-11.4088,11.5109,13.4058,25.2223,0.90012,-22.0382,22.9082,-23.3571,21.932,-1.0926,3.5584,-2.9015,6.4045,-6.3233,3.3178,-2.9062,-25.3171,-13.0923,-14.2807,12.536,10.6936,25.8164,0.75146,-22.324,21.338,-22.8626,22.4893,0.95721,4.2097,-0.97568,-0.23479,-4.7724,-7.4971,-7.0039,-13.7275,-25.0247,11.4761,-11.7728,25.4043,13.3587,-21.506,1.4194,-23.7678,22.939,-2.2109,21.5527,-2.9178,2.9406,-3.4888,6.2399,0.90286,4.7578,-14.1746,-25.172,10.6026,-11.7458,25.2443,13.7388,-19.1305,2.6018,-20.7378,23.3874,-1.3812,21.1818,-7.777,-1.2671,-10.7104,2.8885,-3.4661,3.7131,-18.4717,-9.7349,-12.4121,9.4684,3.8013,15.4475,-6.9089,18.9866,-25.7225,26.3957,-19.9275,6.1501,-17.7802,-18.4272,-1.681,1.7588,18.7897,21.6383,-23.8604,-8.2473,-17.2588,16.2259,6.3065,24.1082,10.3607,21.8338,-10.7418,13.9999,-22.3257,-9.0431,-2.2807,12.8756,-15.6341,11.9942,-12.7733,-0.46123,-18.1646,-11.5876,-2.364,6.649,20.3711,23.0895,18.4475,21.5454,6.8633,3.8658,-4.0066,-10.0936,-5.9593,20.8328,-25.7337,26.9069,-19.5882,6.9408,-23.5899,-20.6549,-6.6693,2.2585,17.0342,21.378,8.8479,4.0318,3.5634,-6.8937,-7.7291,-12.8195,-5.275,19.5007,-24.8827,25.6973,-20.3919,6.0433,-8.2517,-16.7709,7.2779,-11.9072,12.9277,1.9565,17.9183,-6.9812,25.5023,-23.9885,8.2747,-18.1389,-17.0334,-18.9713,4.2381,-2.0839,22.6897,18.114,-7.6727,-14.0954,4.9771,-8.574,10.9697,5.0356,17.1432,-7.907,24.8811,-24.2905,7.2333,-14.4189,-17.691,-22.2221,5.7838,-4.7967,27.2582,21.9682,-8.4905,-12.5333,-0.47498,-6.2884,3.6399,1.9627,18.0176,-8.1827,24.2195,-24.1703,4.5558,-16.6907,-18.3486,-24.9703,7.9966,-6.0063,28.1782,21.4702,-7.3779,-4.3991,-5.0993,-0.22625,0.49756,2.1484,18.9144,-5.8605,23.532,-24.2447,3.5454,-20.1205,-18.039,-26.7374,8.4283,-7.9011,26.3785,17.9374,-5.4181,1.037,-9.4544,6.7763,-6.8439,2.0272,28.2766,17.4649,12.9764,-9.5462,-21.8538,-29.1615,5.8962,-18.7847,20.8155,-22.5479,13.4538,-3.8871,-2.0375,4.1905,-4.8998,10.6381,2.0396,10.638,8.9285,25.2243,17.7761,9.7406,-7.6228,-14.3979,-7.6223,-22.7798,5.8648,-14.6792,22.6793,-21.7387,20.6754,-21.7389,2.1174,-18.7553,-19.0606,0.013031,3.0337,20.1051,23.0237,-8.0195,10.6495,-23.0815,26.3317,-12.9693,12.8344,15.6989,7.898,11.5527,-1.4095,2.5533,-1.2074,-22.6115,-18.203,-5.2581,3.0103,18.4626,19.1043,18.463,-3.4546,18.5258,-19.8224,22.0821,-17.3538,5.6155,-17.3535,11.3157,0.77023,12.8917,-11.7622,0.8015,-13.6249,0.80111,-15.9376,-23.3051,6.6908,-8.4876,24.199,16.5873,17.8613,-8.7565,24.0782,-24.4984,5.6567,-18.187,-5.2144,2.6194,-10.2653,7.1611,-5.6569,4.4849,-10.1222,2.1968,-13.1609,9.3979,-6.6512,3.4035,16.307,-4.2396,20.5592,-21.7972,2.4525,-19.0586,-16.4155,-24.9661,8.4704,-9.0149,24.2744,15.1884,-9.8203,3.9425,-15.7518,10.1861,-9.3559,2.9333,16.2687,-3.6826,18.6718,-21.6854,1.2207,-19.4034,-16.4057,-24.7566,7.8705,-9.1562,23.5648,15.2492,-9.3281,7.2629,-18.54,14.0512,-11.2842,4.8527,15.6459,-1.6399,15.6081,-20.3776,-0.083415,-18.9292,-17.4252,-24.7959,6.067,-8.0012,22.7444,16.2283,-7.6214,-17.8609,10.3581,-10.2598,18.5355,8.9292,16.8168,16.6704,0.80369,2.7883,-16.7701,-15.265,-17.8724,5.6378,-24.1305,23.3572,-5.3989,18.1939,-10.5918,-12.7354,-1.0174,-5.5106,5.1045,2.8603,16.1314,20.366,-5.4059,2.9409,-23.0266,-19.31,-16.2103,8.5831,-24.897,24.4485,-8.8872,15.2084,-3.9288,-21.3927,16.7876,-18.9397,20.9845,2.53,15.4884,13.2116,3.336,-2.4962,-15.1397,-18.3526,-20.4054,-0.40105,-20.7441,18.1758,0.21643,18.7775,-6.7571,-21.6327,13.3935,-18.2147,17.4112,0.80456,14.1063,12.2945,1.0669,-3.4963,-18.8183,-21.1574,-19.9365,0.16598,-22.0492,17.7091,-3.5272,15.5049,-2.4225,18.0875,-21.5111,21.0034,-20.7057,1.2229,15.5582,2.6142,13.0363,-14.9839,-3.2229,-17.9959,-20.5928,-20.3407,-1.8119,0.87899,17.1097,19.197,-8.6145,-1.3258,-6.8164,8.0421,-0.42004,7.1949,17.2154,-5.3642,23.0693,-22.3583,4.3937,-18.5287,-17.1626,-25.6343,8.5557,-8.7728,25.5081,16.3904,20.304,-0.56982,23.3018,-18.4954,-6.4955,-19.5181,7.4274,20.9468,-8.5808,16.3711,-25.6815,-20.1719,-13.3862,-15.0734,6.8551,-0.19757,29.2204,22.7478,-1.7308,20.4919,-22.28,21.4227,-22.7303,-0.7537,15.9852,3.8771,11.1276,-13.8838,-5.1406,-18.161,-21.1312,-18.4791,-4.8683,2.3609,15.0104,19.4976,-10.27,-22.3873,-13.3626,15.2725,9.7968,26.2574,-21.744,-1.8658,22.3189,-20.6051,24.2105,2.5416,10.421,12.1815,0.81189,2.8314,-8.8331,-5.4766,-24.8917,-9.9287,-15.107,15.0942,9.7687,24.4848,4.991,18.498,-12.4215,15.2162,-14.018,-0.38947,1.615,13.4042,-14.6856,10.545,-17.7113,-4.833,-23.8092,-14.1079,-9.3164,9.5937,15.2447,23.9192,3.9306,-8.7334,14.8275,-9.8147,13.2537,1.4214,1.93,-18.1474,16.9336,-19.6466,13.944,-2.5125,-25.3201,-15.4597,-10.1736,10.0066,15.0032,24.8267,4.798,-14.5094,21.4535,-17.7468,17.8512,-1.8076,0.84022,-11.3785,9.5204,-14.4246,6.4112,-5.1875,-25.4676,-15.1827,-10.7324,10.3772,15.0341,25.5665,4.0414,-17.7605,20.9115,-21.4019,16.4043,-4.2537,2.6736,-6.5494,10.7989,-9.7442,9.8854,-1.4848,-25.3109,-14.9487,-10.9584,10.4799,14.6868,25.5878,4.492,-16.233,22.2788,-20.6364,19.3692,-2.9643,-0.48107,-10.214,6.6795,-11.1937,4.5602,-3.6167,-25.1931,-14.8488,-10.8382,10.7442,14.9845,25.8178,1.3312,-19.2833,20.3082,-22.1773,19.4968,-1.9692,3.6712,-4.793,10.7741,-6.9432,5.9152,-3.1454,-16.9556,-20.5099,2.7922,-2.9449,21.7628,19.2828,15.013,-14.314,26.5923,-26.1368,13.0923,-13.2667,2.9837,3.0332,5.7768,3.517,7.5325,5.5281,-6.0688,-9.4223,10.6593,5.5867,15.0882,15.6143,20.4296,-9.1066,26.6051,-32.035,6.0186,-23.0045,-18.1843,-27.2283,3.4555,-21.4573,22.4053,10.5642,-19.8807,-4.1204,-22.7699,18.406,-2.4488,20.0033,15.0423,25.2145,-13.0872,11.7907,-27.6993,-14.0166,1.8694,10.7048,-5.0613,13.9352,-2.6358,6.8195,-4.7291,-21.3741,17.4023,-19.8781,22.4154,0.77039,23.7309,13.2267,12.5628,-12.3493,-12.1777,-26.1688,10.1219,-3.7742,13.1855,-10.6261,4.7951,-5.2209,-24.1617,-17.73,-9.7044,4.0455,13.8394,20.8267,6.9534,4.8994,-1.0767,-4.6356,-12.751,-13.9464,-2.2959,20.2898,-21.7504,24.4075,-19.1971,3.2718,-24.4278,-13.9984,-13.2968,8.0111,10.4067,19.7439,7.6393,10.7993,-7.4384,0.64892,-20.5578,-16.0944,-1.169,19.6629,-18.7051,21.8683,-13.9504,3.4092,-23.9855,-9.9389,-16.8588,11.9348,5.2159,19.7287,7.627,17.6985,-13.0261,6.3074,-23.4996,-13.8258,0.93452,15.824,-10.4411,19.7219,-7.3802,8.7307,-6.7636,-15.6373,13.0791,-11.4866,19.6364,5.7813,-3.2236,23.4579,-21.8193,28.1116,-18.1558,5.2236,-27.1423,-5.8203,-20.7713,16.7402,7.4775,27.2528,-2.7308,-9.1061,2.9189,-9.7961,3.8258,-2.3672,-4.6249,17.5676,-23.6186,21.6328,-20.2108,2.8723,-25.6185,-17.2953,-8.5327,8.8835,16.3601,25.5322,-15.8862,-8.4609,-6.3688,11.4704,9.2859,20.4597,10.5356,23.7751,-12.7105,15.2068,-24.1283,-9.3039,-20.4522,6.1073,-23.9612,21.3049,-4.2446,16.0073,-22.8072,-9.8739,-9.2483,10.2344,13.3595,23.3916,5.1847,-14.0953,18.2656,-23.3112,14.7366,-7.2193,-13.2984,-22.8918,14.5385,-6.6677,19.7547,10.957,-11.0453,-7.0625,-5.7737,9.3894,6.7511,17.4596,13.4374,25.3767,-13.5468,14.0432,-25.748,-10.7634,-22.5707,2.4077,-24.0975,22.0528,-1.3627,18.7313,-4.3447,-0.78225,-5.8855,11.3757,-10.7651,13.2774,-3.6354,14.6615,25.6896,-7.5491,15.2138,-22.5512,-6.7548,-21.8326,-19.6291,0.35657,-21.3906,19.6113,-5.1146,21.4251,11.3363,8.7429,7.3023,5.0716,1.8132,-13.0932,-11.9392,19.0248,28.0785,-11.8346,5.5548,-23.654,-11.5134,-20.5723,1.35,-23.7932,25.0069,-3.0104,20.4743,11.6555,7.0142,13.3964,-4.6959,5.3366,-14.9267,-9.2439,14.8534,23.5068,-7.6079,15.6453,-23.7453,-3.2983,-22.4988,-11.9208,4.213,-16.6279,17.9765,-9.2994,15.7373,4.0148,-5.6958,0.60508,-8.8465,8.3245,-2.2938,7.9078,27.7197,17.5722,9.0534,-10.0251,-19.6523,-27.0942,3.8999,-19.9214,22.218,-22.242,20.1612,0.17894,-8.4014,-19.2048,10.2867,-8.7275,22.1799,14.9348,20.9649,-0.35738,24.0735,-22.8729,2.4606,-21.123,13.3525,19.6575,-5.6157,9.2845,-16.8818,-8.6255,-9.5638,-17.5931,3.7646,-6.7815,16.007,13.5565,21.8845,-3.5381,23.9183,-24.0791,0.66748,-22.1304,12.7452,20.6225,-9.7638,9.3853,-21.8837,-10.9597,-8.4627,-11.9811,4.4604,-1.3085,17.4086,15.4623,22.3905,-1.9387,23.7435,-22.6732,1.7915,-21.5471,12.74,22.8452,-8.8976,11.9184,-18.8007,-7.911,-12.3835,-14.7998,2.5866,-1.8786,18.7949,16.4608,22.353,-1.5109,24.1642,-23.3018,2.6142,-21.0343,10.2166,20.7344,-9.4315,10.9481,-16.2584,-5.9631,-16.7618,-17.4476,1.2738,-1.707,20.9972,18.1261,18.9947,-2.5348,24.257,-23.1898,5.6739,-19.6975,7.0218,17.3757,-9.2531,10.9918,-12.1636,-2.0419,-6.356,8.2126,-12.7068,18.863,-4.4864,12.8412,23.3656,21.1447,5.3285,0.034835,-20.761,-23.3686,10.9455,-11.0337,22.1085,-18.5893,14.7973,-4.5501,-8.1309,10.3796,-16.9098,20.4798,-5.6258,13.1684,22.2496,22.6635,2.3064,1.4868,-21.4819,-21.4784,12.2215,-7.6722,21.8163,-17.1581,12.755,-5.9507,-19.1063,-19.6619,2.9695,-2.2097,23.2387,19.2384,14.5902,-6.4387,26.1114,-22.8105,12.6852,-16.2747,3.3405,12.5931,-4.2189,10.1084,-3.9865,1.175,-3.795,7.3126,-7.262,14.7498,-0.27721,11.4522,26.3954,21.0275,8.278,-3.0349,-19.307,-26.6199,10.1004,-13.0665,22.3064,-19.892,21.8254,-1.359,-14.7845,-1.0768,-6.4915,7.4489,6.1702,38.9814,27.9671,-23.7377,-7.6326,-33.4931,9.421,-21.4853,34.0571,-22.2162,6.5238,-16.3952,-23.2557,7.0111,-7.2293,24.3918,17.3521,18.5393,-5.79,23.9895,-22.4279,6.9428,-15.9333,-2.1511,7.2591,-9.2714,11.8312,-4.1488,7.7535,-17.1468,-23.762,6.4251,-6.9528,25.5045,18.2055,17.898,-5.3819,22.3853,-21.5506,7.1253,-13.8794,-2.3484,7.8834,-11.408,13.3547,-4.4478,9.8923,-18.4637,-24.3456,19.5179,-7.0406,34.1626,19.04,16.5673,-4.6035,20.1717,-19.0195,13.9831,-11.2267,-2.5954,9.4526,-22.7488,15.0014,-15.4633,11.0639,-16.6119,-25.6021,8.4462,-10.8165,26.1243,16.0235,19.0757,-3.3683,23.8804,-22.8298,4.4425,-19.6303,-4.054,5.6751,-9.056,8.6946,-2.5073,5.34,-16.8605,-25.0966,7.6099,-9.6156,25.0428,16.2889,17.9565,-5.5602,24.0707,-22.5819,5.9833,-17.4073,-5.0209,2.9179,-7.79,8.6536,-0.36736,8.2914,-9.2886,-20.3047,7.4871,-16.9896,12.567,-0.95449,12.8306,13.6455,-3.0337,-2.8642,-20.9602,-21.6489,-18.8467,2.324,-23.3827,17.6704,-6.4056,13.3895,-3.7113,22.4269,-20.5491,22.0429,-25.3703,-4.3474,16.7599,5.5324,8.8762,-12.3069,-9.8468,-19.8245,-24.2602,-17.4949,-10.0535,5.0045,11.9376,20.5437,-25.1998,-13.4495,-12.2993,11.9072,13.0685,25.5678,1.049,-21.8465,22.4799,-23.282,22.1555,-0.89948,2.858,-3.3337,5.2288,-4.6543,2.0092,-1.7091,-25.6753,-13.186,-14.1679,12.1826,10.7036,25.2129,0.59894,-22.2033,21.1779,-22.53,22.181,0.5667,3.3597,-1.6712,1.5955,-4.8766,-4.2258,-5.7402,-25.9661,-11.6485,-15.4408,14.3675,9.6317,25.8661,-1.8843,-22.1351,20.2882,-21.0827,23.3244,2.5113,4.6657,5.4084,-3.218,-0.64385,-6.175,-4.328,-13.6585,-25.2658,11.4074,-12.4128,25.4657,13.5924,-21.1765,1.7293,-23.3443,22.8347,-1.6156,21.8375,-3.7913,-0.1035,-2.9904,2.4088,0.15881,1.8813,-14.4061,-25.1462,9.9547,-11.6741,24.6905,13.4938,-18.927,2.1625,-19.6785,22.9906,-0.10122,21.521,-7.3081,-0.72275,-11.9225,2.2297,-6.5664,1.1674,-14.4605,-3.5403,-10.385,12.1648,2.1119,13.3928,-6.6777,17.8385,-25.3341,24.5312,-19.1751,6.004,-22.2452,-19.5544,-5.1289,3.482,16.9781,22.7501,-23.7087,-7.2493,-18.9738,16.534,1.673,19.9562,9.7628,22.0692,-12.264,11.2839,-24.4042,-12.5388,-0.41227,10.9023,-8.6754,13.4077,-1.6671,9.3774,-19.7281,-3.188,-14.3186,17.7392,7.1542,21.5968,17.9536,4.5385,20.8158,-11.0414,2.9259,-14.9776,-6.1356,-25.9357,19.0081,-19.6771,26.4781,5.4255,-22.6297,-19.9015,-4.1041,0.52316,15.0798,16.3545,10.9002,5.5551,4.9556,-7.8102,-11.4263,-18.7538,-6.109,18.9949,-25.2714,25.1125,-19.6384,6.6324,-21.0797,-17.2738,-5.5908,1.194,11.9844,14.7051,12.3647,7.8639,3.818,-6.8632,-13.2625,-18.7683,-4.0156,19.2282,-23.4946,24.6061,-20.3601,5.5268,-4.3537,-14.1213,7.5536,-13.0672,8.4785,-2.9636,5.0223,16.0158,-6.3923,24.7729,-22.8652,23.6136,-17.6289,0.077922,-17.893,-21.596,5.8225,-4.427,7.1931,14.3768,19.555,-4.6813,-12.5892,4.3863,-5.8156,7.3521,16.937,-10.3541,24.7197,-40.6697,3.1431,-21.2519,-27.7907,7.4489,21.3001,35.0554,-3.4136,-6.3491,-0.48808,-4.3074,-0.097274,-1.7989,19.6889,-8.029,24.308,-24.2561,3.3731,-21.6729,-20.2771,-28.9191,8.567,-8.0915,27.2854,17.8083,-3.4165,-1.6603,-2.5242,-0.9481,-2.3341,-2.5129,20.0732,-4.1875,23.7329,-23.7921,2.6191,-20.6099,-17.8458,-26.5044,8.9754,-8.9858,25.9268,16.2269,-12.8187,-23.7666,12.0739,-9.4257,25.6202,15.8662,-16.5418,0.81582,-19.8718,12.1093,-3.636,13.5276,-17.7043,1.6375,-12.5827,18.7673,2.0061,16.1716,-6.5525,6.8193,-18.1191,11.783,-6.6611,8.8056,29.4617,20.1081,1.4826,-7.6029,-24.5796,-26.9105,3.1741,-18.3309,17.9674,-20.8453,14.1274,-1.6317,-10.364,-10.3571,-1.1877,5.3463,14.2156,21.4901,24.423,8.6005,13.2473,-10.2038,-8.9526,-17.8086,1.5666,22.64,-23.1722,26.0349,-19.8349,6.9834,-23.0746,-20.0788,-3.9081,2.7321,19.8703,22.3308,-13.1421,13.3023,-25.8196,26.7444,-12.8882,13.9296,1.6656,-2.3096,2.0069,1.3074,0.35635,3.685,-4.1465,-7.7408,1.645,6.1779,15.3311,20.1382,31.36,9.5326,25.5495,-8.3893,-7.5709,-14.5356,-12.3182,21.0548,-28.1869,24.031,-19.4109,6.7918,-37.9806,27.0446,-14.0718,1.0551,20.656,-2.8179,16.8053,-22.3088,37.5254,-8.293,21.9473,-19.1668,25.6025,-23.5489,9.7155,-16.9697,-24.1972,6.7916,-8.6976,24.4927,16.9132,16.7712,-5.8837,23.2434,-21.8711,7.5293,-15.4395,-6.1742,3.3051,-9.6827,10.1527,-0.33916,10.3638,-3.8873,4.3534,-10.0465,10.1107,-4.554,7.573,18.8563,-3.3183,21.612,-21.26,3.5442,-17.6331,-15.4336,-24.6453,9.1324,-9.4812,24.6503,15.4846,-3.5349,6.643,-10.744,12.1473,-6.6286,6.1135,19.2328,-2.5534,21.4514,-20.4721,2.8636,-17.5795,-15.1182,-24.3208,8.8787,-8.9027,24.2156,16.249,-4.6952,7.8107,-13.4152,12.1282,-9.0626,3.9475,19.1569,-2.0481,20.5536,-21.1993,1.716,-18.9246,-15.5635,-24.1279,7.5069,-8.283,23.3288,16.3045,-7.2512,-16.4164,8.5625,-11.1206,13.0469,2.324,16.8535,18.2988,-1.6244,0.63399,-21.255,-20.223,-17.3455,5.1888,-23.7352,22.1537,-7.2827,16.0397,-4.6845,2.6011,-7.2137,9.8538,-1.3869,8.5516,18.633,-3.1041,22.6283,-21.4402,4.1974,-18.3237,-16.6208,-25.2739,9.4369,-9.0048,24.9854,15.0945,-2.0637,17.8212,-20.8382,19.1661,-20.5277,-0.64471,17.4801,2.7989,14.6467,-17.5633,-3.3081,-20.6605,-19.0548,-18.4948,-1.6266,-0.78151,14.9776,15.3433,-2.7699,-20.3512,15.7952,-19.0565,16.1282,-0.93243,18.636,15.5537,1.767,-2.8316,-20.2055,-21.5306,-17.9255,-0.28275,-19.8514,15.8824,-4.4069,13.6135,-0.42164,-16.5998,16.547,-16.2275,15.755,-0.10503,21.6825,19.3904,2.0067,-0.85187,-20.4812,-21.5814,-14.1432,4.3465,-18.5296,18.6606,-4.3577,13.9202,-0.71699,19.6154,-20.9627,20.0035,-22.4003,-2.4638,17.1068,3.3492,13.2916,-16.6091,-4.9911,-21.0066,-19.4705,-17.5703,-3.6967,1.0071,12.7007,15.7776,0.68741,21.7164,-20.7605,20.2041,-24.8946,-4.2392,17.4476,4.8694,11.6627,-15.9046,-7.1772,-22.1864,-20.0149,-15.8009,-6.99,3.4084,9.5448,15.6579,-25.1008,-9.7425,-14.9037,15.0672,9.0252,23.6665,2.5307,18.9848,-14.6078,17.3128,-13.8723,1.2995,3.8796,13.5236,-12.4804,8.1145,-18.3195,-7.596,-25.0134,-14.4935,-9.5811,9.413,15.3094,23.7998,0.76434,-13.5307,14.9973,-12.1704,15.7148,2.4822,3.7673,-16.2547,17.3135,-18.7225,11.2872,-5.3199,-25.3804,-15.0843,-10.4535,9.6727,15.325,24.8166,3.5507,-17.1773,20.4399,-18.8254,17.1369,-1.6486,3.333,-8.3361,11.3786,-13.7679,8.192,-5.3759,-25.5141,-15.113,-10.6138,9.6266,14.629,24.6053,4.2321,-16.1862,21.6082,-19.1857,19.0194,-1.3083,1.3083,-9.8939,8.8866,-13.2441,4.3494,-6.5587,-25.431,-14.974,-11.0894,10.4036,14.3158,25.1339,2.8081,-18.6439,22.0151,-21.1784,20.223,-1.7112,2.3829,-5.2699,6.0495,-10.0284,2.265,-6.121,-24.9392,-14.7285,-10.1308,10.8477,14.8686,25.304,0.18172,-20.0459,19.8204,-21.8413,20.0779,-1.0698,6.0073,-2.9041,11.4298,-6.7557,3.9069,-5.2108,-19.314,-0.12525,-22.0149,19.8977,-2.3562,21.0986,14.607,26.3659,-13.4777,12.8622,-26.3036,-12.1435,-4.8967,3.8066,-6.8354,9.3069,-4.5867,2.8883,-4.7031,-4.1277,8.1414,12.1502,12.5968,17.195,19.3156,-5.107,26.1712,-23.7618,6.2188,-19.0901,-17.2764,-27.3043,7.0338,-18.7336,23.4203,10.5832,-3.4416,-20.6681,18.1762,-19.4623,20.1367,1.53,24.4417,13.3066,11.8026,-11.1932,-12.2216,-24.9932,3.8802,-4.086,11.0155,-10.4646,8.795,-4.2783,-21.8505,-15.2359,-9.6499,2.69,6.8382,13.2464,12.1021,10.1284,-3.7911,-4.7351,-21.3104,-20.8001,-2.6712,20.2229,-21.3076,23.24,-16.706,1.3303,-22.3415,-10.0939,-12.8723,9.5119,7.9426,14.987,14.2504,19.8855,-8.3348,1.5976,-25.1479,-20.31,-5.7715,17.8984,-18.9101,22.2758,-7.4933,6.4002,-21.7499,-6.8564,-17.9263,11.5397,0.24725,14.8845,12.2156,22.1606,-11.6283,7.7836,-24.0622,-13.0969,-6.0275,10.1794,-10.7985,19.7917,0.62904,16.2426,-5.3178,-17.9314,13.8429,-14.582,21.2068,4.9649,-2.6599,18.5382,-17.3767,21.7526,-14.413,4.0844,-26.4856,-8.5192,-20.2685,16.4435,7.2131,26.8065,-1.3016,-7.4866,-0.80156,-8.9251,-4.8261,-6.5537,-3.5222,18.5679,-23.0006,21.9164,-20.4536,1.9209,-25.3355,-16.5274,-9.6853,9.4776,13.928,24.1796,-21.7048,-7.8255,-13.1388,15.3615,8.8828,23.9961,8.1044,25.1706,-16.0941,18.2648,-24.9987,-7.7587,-14.6372,4.4141,-19.5686,14.6788,-4.6024,10.9121,-18.6753,-7.321,-10.474,14.2688,8.1885,22.6313,10.0851,25.398,-15.2721,16.9395,-24.9938,-8.4828,-17.8083,3.5612,-21.1695,17.68,-4.5332,13.8293,-25.4333,-15.5661,-10.284,8.8709,15.5223,24.6045,5.1352,-17.9513,23.3521,-25.2732,18.5557,-6.9565,-7.172,-15.371,8.8912,-4.4484,13.3884,8.3735,-18.1269,-6.9856,-16.9403,13.2571,-2.1513,19.0314,10.5342,26.0354,-19.3463,15.8982,-26.3298,-8.7907,-18.2486,2.4382,-14.7017,19.7092,2.2372,17.5859,8.9518,24.7294,-8.3601,16.608,-22.3342,-11.5701,16.9964,16.6146,-5.2181,-2.7567,-14.4682,-12.0932,-19.8924,8.8686,-25.7223,26.1666,-4.9615,20.2663,17.7891,20.2993,2.384,7.259,-15.0843,-14.7207,8.0087,16.7315,-20.877,8.2388,-23.7379,-6.8589,-14.966,5.248,-18.249,24.5159,-2.1033,19.6443,12.7826,-9.5993,21.4763,-18.618,13.1107,-7.7371,22.2626,24.2007,-0.80236,0.6742,-25.2344,-25.1696,-8.668,11.2421,-13.7109,16.1754,-3.5974,9.3345,-9.4891,2.4493,-12.699,18.7829,1.8979,22.2319,27.2553,18.4936,4.9361,-5.5378,-15.0052,-18.7515,0.68414,-22.8189,23.085,-19.6762,25.321,5.1542,-11.7523,-18.4032,7.6591,-1.0362,22.4172,20.115,21.3861,0.026237,22.8188,-19.7675,5.2308,-16.3473,7.8736,18.4915,-6.9818,16.914,-11.1176,2.4396,-15.3984,-22.5939,3.4772,-4.1924,19.9606,18.3155,21.6186,-1.6958,23.4916,-24.0327,2.7642,-19.6557,7.4965,15.6073,-10.367,9.907,-18.2573,-5.613,-14.6985,-17.0338,1.9637,-1.3011,21.1053,20.4797,22.0057,-1.3611,22.9951,-22.0807,1.4138,-18.5972,9.9654,21.3267,-10.6754,12.9724,-16.1849,-4.9904,-15.53,-17.4261,2.7369,0.0017265,22.8328,22.4387,21.5103,-1.4479,23.6769,-21.4443,3.1372,-18.4569,7.8526,18.5864,-9.6917,13.278,-11.936,0.1202,-17.0879,-19.1976,3.0043,0.046891,22.9836,20.4361,20.5756,-2.6426,23.9657,-22.0679,4.6028,-18.354,5.2667,15.6549,-8.7677,13.745,-7.9249,5.0748,-18.7875,-21.1305,2.8835,-2.5083,23.4043,19.518,17.1366,-4.2683,23.6955,-22.0094,7.5937,-16.3276,2.0551,10.8095,-7.183,11.9991,-3.046,7.3366,-13.2372,5.2348,-18.2352,20.9309,-0.25934,20.2031,21.4471,21.4522,2.053,3.5916,-19.2872,-16.3413,5.8343,-12.5407,17.4581,-13.9453,18.1881,4.2374,-10.2634,5.394,-13.3818,19.6638,4.2056,21.3236,23.7958,21.3156,5.3869,2.9492,-12.1097,-11.9027,5.0947,-12.908,19.0389,-13.2186,23.0312,7.528,-19.3306,-22.122,2.3656,-3.5855,21.3741,17.5884,13.1344,-7.4207,24.9179,-21.9355,12.2695,-13.5174,1.9796,6.008,1.2525,9.8137,5.7623,10.4525,-7.1717,-7.8152,1.9296,6.7802,17.3121,22.7132,26.9396,7.9435,21.233,-4.1244,0.21613,-9.3298,0.39073,21.6631,-17.6799,27.8059,-15.6735,9.0748,-17.439,-24.1437,7.1252,-7.4391,25.0358,17.8582,14.6078,-7.3101,22.2309,-22.1007,8.4765,-14.7506,-9.2059,1.9292,-12.3733,10.85,-2.8455,9.7776,-18.4639,-25.3307,10.0905,-6.9523,26.4605,19.2682,11.8942,-8.3515,20.5385,-22.4614,10.0107,-13.9713,-10.5536,2.481,-14.7292,12.2652,-6.5468,8.6425,-20.0908,4.7142,25.1079,-29.1305,20.2867,-7.0303,8.3256,18.1397,8.1619,-9.3118,-13.8143,-23.1527,-10.1392,-21.6531,-13.5537,3.6653,7.0131,15.277,-16.5087,-25.6552,8.1007,-11.0071,25.7059,14.5954,18.1394,-3.6941,23.1003,-22.3913,3.6731,-19.8483,-6.1463,0.83497,-6.3662,2.8182,-1.4518,0.49431,-16.3452,-24.3897,7.0323,-10.1017,24.8535,19.4627,15.8878,-5.6929,21.7691,-21.3723,4.4279,-17.0667,-9.7854,-1.9251,-9.5729,2.6394,-2.0887,1.801,-1.4465,-15.5925,11.6957,-16.3594,11.9725,-2.0945,20.3262,19.6422,-0.28402,-1.2724,-21.8201,-22.2134,-15.7481,5.1401,-21.8294,18.2805,-6.7211,12.7827,-25.2511,-13.155,-13.7438,11.5523,13.4366,25.7334,0.3745,-21.9352,20.749,-22.885,21.4671,-0.075325,2.8674,-0.69972,2.1692,-1.1655,-1.2332,-0.81161,-25.071,-11.4691,-14.859,11.8235,10.5764,25.8261,-1.0983,-21.3157,19.88,-20.7072,22.408,1.6617,4.8697,3.5709,1.5098,0.47492,-0.74098,-0.42921,-13.7916,-25.1448,11.9836,-12.3455,25.9219,13.9208,-21.375,1.0474,-22.8697,21.6339,-0.4167,21.5354,-2.3885,-0.71583,-3.5752,-0.35521,-3.2503,-1.8105,-14.5597,-25.4774,9.2455,-12.2614,23.5043,12.7788,-19.7192,1.3487,-19.4892,22.6417,0.94142,22.3136,-4.8634,2.4077,-11.9902,2.7016,-9.4076,-1.9535,-10.2533,-0.32565,-5.4565,12.0599,4.8246,11.0303,-6.9961,16.8612,-26.5224,24.2209,-20.9044,3.1095,-24.0119,-18.3232,-8.2228,6.9118,13.3059,22.6304,-19.8492,-2.0299,-17.8728,17.0049,-0.17729,15.9723,12.6016,24.167,-12.1193,11.1519,-25.2581,-13.4324,-9.3484,1.4211,-11.5578,13.3673,2.2297,15.622,-18.2623,-1.2289,-13.3206,11.6417,7.7328,15.6192,18.8517,7.065,16.289,-16.1473,-1.3486,-21.8029,-5.4354,-25.329,19.3241,-19.3448,25.6118,5.4781,-20.9971,-8.2762,-14.6096,9.5383,6.902,18.1555,14.4508,-1.3114,16.9969,-16.9832,0.77161,-17.7312,-6.2096,-24.7496,17.7448,-19.1833,25.6668,6.7331,-16.2738,-15.1256,-2.2374,-2.8862,7.3409,5.7376,18.2894,11.6888,5.6529,-8.8014,-16.3808,-24.7011,-4.9677,18.6966,-25.0278,24.244,-20.1026,5.6743,-14.6744,-12.4677,-4.3149,-2.2574,0.86437,2.8858,0.86438,17.9976,13.1154,1.9849,-8.1042,-14.5431,-22.6628,-14.5432,-2.8799,19.501,-24.6052,24.4448,-23.2258,6.729,-23.2258,-2.6629,-3.0447,-8.6655,2.2988,-5.5575,-0.19431,17.0561,21.113,-23.5107,24.5677,-22.6146,2.8839,-28.4932,-26.9639,-7.168,9.312,16.4248,27.1863,1.0454,-0.5,-0.44449,-2.8026,-2.9561,-3.6764,20.419,-2.8948,23.9701,-23.8082,2.9804,-21.0129,-18.2995,-27.1203,9.928,-9.8895,26.4583,15.6234,3.1356,2.9574,-1.3482,0.55686,-5.0537,-3.0923,20.3815,-3.3602,23.2875,-23.3726,2.3867,-20.4887,-16.2415,-26.1549,10.7986,-10.3947,25.2911,14.9174,-15.3362,-25.2792,9.5624,-9.312,25.0267,15.8496,-19.7161,-1.3573,-24.265,14.7129,-5.7999,14.8152,-10.8973,2.5508,-6.7302,17.5271,4.2589,14.5587,3.6629,-21.094,23.8416,-27.7684,16.2468,-8.2778,26.101,5.218,17.0159,-10.0424,-13.9747,-18.6614,-6.7234,-20.2048,10.7046,-6.1689,22.6142,18.1636,12.5427,18.8432,-4.3119,9.7954,-18.5095,-9.592,24.1753,-1.7019,21.4715,-27.1603,-1.2514,-26.4724,2.5117,-15.2569,17.9613,-9.1105,16.143,4.9742,-14.8152,3.8935,-22.9017,19.3567,-5.5968,17.0626,25.3819,21.9743,-0.16765,-4.3834,-26.3797,-24.6597,-0.41652,-21.4533,11.6502,-20.1328,14.2595,2.2005,-24.1292,-20.2306,-7.2869,2.7714,21.1254,27.0438,15.7416,2.9227,8.2207,-8.1398,-6.6553,-10.4697,-6.3587,19.1433,-25.9238,26.7218,-17.5614,11.2517,-20.3935,-14.2032,-3.0461,5.8021,20.3323,24.3068,17.3419,9.5879,9.9154,-0.92749,-1.5064,-5.0241,-4.1961,18.9403,-25.4393,24.8324,-17.1445,9.784,-22.2359,-19.9745,-1.6362,-3.0026,16.5984,16.7834,-12.0297,12.3739,23.3367,-24.815,11.5321,-12.6399,-1.6088,0.90165,7.5763,2.789,12.5978,10.5998,-22.1113,-17.0921,-6.331,3.5951,19.4975,22.8347,11.6953,4.6201,7.7858,-4.7903,-1.3658,-6.2039,-8.3951,18.6475,-26.272,26.0398,-15.8378,9.8206,-17.0536,-23.9456,6.6036,-6.0361,24.8941,18.6111,14.5714,-6.3296,21.4122,-23.7644,5.1177,-19.4445,-10.5109,-0.79086,-11.5692,6.7061,-4.0404,4.3668,-8.7237,0.37341,-11.3058,2.2265,-9.142,-5.0384,17.0626,-3.5927,20.5476,-23.0062,1.2467,-21.0838,-15.5042,-24.5721,8.8473,-10.4122,23.0807,13.3112,-7.5932,3.7327,-13.3226,5.51,-9.9552,-2.6276,16.9279,-2.9145,19.5444,-23.1032,0.80029,-21.7727,-15.7476,-24.3319,7.5937,-9.6592,22.8412,13.9376,-5.4996,7.2091,-13.1922,10.682,-11.0541,0.1896,18.3357,-1.9649,20.4059,-21.8951,0.6835,-21.2509,-15.9326,-23.8426,6.898,-7.7012,22.4367,15.0801,-3.9844,-14.0197,9.2447,-13.4734,11.321,-1.1256,19.5391,20.507,-1.4813,-0.094823,-21.8167,-21.7028,-16.0461,5.7897,-23.2025,20.4953,-7.2751,14.0679,-0.10952,-19.0237,17.9886,-20.4379,16.7379,-2.5757,20.3949,17.4295,1.864,-2.8503,-19.3798,-21.1787,-15.1009,0.27299,-17.2633,13.4956,-4.3522,11.1926,0.52422,19.6684,-19.5458,18.0681,-22.4902,-4.3698,19.9172,3.231,16.0131,-18.1431,-4.5765,-21.7899,-15.9847,-15.6163,-2.7395,-1.9387,10.4256,11.0808,0.09921,-17.0562,15.7913,-18.2194,15.757,-0.35017,21.0573,19.6666,0.94441,-1.2288,-21.0211,-21.5301,-14.2538,3.6497,-18.8278,16.4546,-4.5708,12.1193,0.60435,-15.6863,15.2329,-15.7389,13.7585,-0.43699,21.7536,20.637,0.71171,0.045113,-22.0882,-21.9281,-13.9227,4.9076,-19.2803,18.2479,-5.4494,12.8626,2.1962,22.0096,-19.8299,18.8191,-24.0442,-5.0446,20.8957,4.9935,15.3558,-17.3682,-5.8364,-22.5296,-16.109,-13.0382,-4.8383,1.2919,8.373,11.4554,4.6079,-19.4302,24.0691,-25.6916,19.4552,-5.8991,20.9916,13.142,6.8306,-7.8741,-16.4933,-23.203,-16.558,-8.7302,-10.8196,4.2872,3.977,11.4817,-24.74,-9.5369,-15.3473,15.5873,7.7556,23.3812,1.5126,18.7621,-16.0486,17.9542,-14.8244,1.4554,5.6534,14.0722,-11.1683,7.257,-18.5374,-8.9068,-24.3347,-14.4878,-8.3192,9.6389,16.5521,24.6781,-1.8425,-17.1951,14.7492,-16.7038,16.0669,-0.24312,7.0323,-12.2072,18.7643,-15.4676,10.8315,-4.3664,-24.9516,-15.2797,-9.8298,8.4278,14.9828,23.7761,2.0488,-17.1135,20.0453,-17.8216,18.9607,0.47619,5.0521,-8.6963,11.8618,-15.2137,4.7057,-8.5512,-25.0325,-14.7755,-10.4801,9.4966,14.5662,24.3991,2.1591,-18.7371,21.381,-19.7723,19.7881,-0.65757,5.679,-5.3032,9.5542,-12.363,3.0892,-7.9526,-25.1635,-14.6538,-11.1022,10.3326,13.9339,24.7558,1.6183,-19.6698,21.7963,-21.2431,20.7283,-1.453,4.4908,-3.4332,6.0483,-9.9344,0.95128,-7.1446,-24.1141,-14.2017,-9.4107,10.8377,14.0305,24.874,-0.58867,-20.7265,20.3709,-21.7947,21.1897,-0.3927,8.0832,-0.83758,10.8534,-5.9348,1.8618,-6.0853,-7.1403,-2.613,4.5537,15.2523,11.0734,19.7868,20.3036,-2.6626,27.1848,-17.0386,7.887,-13.617,-15.1974,-27.0268,11.4753,-18.2882,22.9683,9.8746,-2.2638,-18.1284,14.4667,-17.9615,14.3236,-1.8813,24.3409,13.4611,12.1408,-12.1405,-12.0442,-25.2632,2.7628,-8.3195,15.2371,-10.3877,17.3484,3.1192,-16.3501,-8.9315,-9.6188,7.2967,1.1075,19.7128,18.8152,-6.8892,-27.6746,-50.0564,-5.5698,23.2749,-24.9166,27.0546,-5.8893,-19.4823,-5.3683,-13.5482,5.3785,0.81115,7.1463,17.3643,27.1993,-11.4223,15.1406,-26.4011,-17.0294,-12.2984,18.5164,-18.2791,21.8418,1.8545,17.7495,-17.2522,-1.3363,-18.0367,10.7918,-4.5191,9.4423,12.9103,24.5192,-11.8179,11.9765,-24.431,-11.5508,-13.9964,2.0365,-12.0472,18.8348,5.919,21.6645,-6.5972,-20.2172,11.4759,-16.5551,18.6078,3.4342,-2.584,12.9542,-16.6318,15.5482,-15.9176,1.4058,-24.9579,-8.5184,-18.1453,16.244,8.7459,26.3595,-9.0963,3.6432,-8.5876,7.0557,-4.0791,0.047006,-6.592,23.4328,-24.5859,24.1828,-20.6634,2.5363,-21.2886,-12.387,-8.7155,10.3478,14.7925,26.4483,6.1743,4.3272,2.5256,-3.0925,-3.771,-7.5005,-2.8339,21.0853,-23.3611,23.0506,-21.0255,1.6385,-24.7266,-13.7673,-10.4952,10.49,14.2606,24.2408,-24.2325,-8.1302,-16.06,16.3292,8.1277,24.3762,7.0084,25.7493,-17.7233,18.7499,-25.631,-8.0039,-10.2304,2.8631,-13.7557,11.4562,-2.5192,9.5625,-23.1516,-7.936,-15.0744,15.8479,7.4595,23.2426,8.0592,25.4819,-16.6089,17.698,-25.242,-7.6809,-13.0776,2.8437,-16.768,13.915,-2.5583,12.444,-21.4456,-7.557,-15.9981,14.1342,6.314,22.631,9.0955,25.0918,-15.482,16.7416,-23.758,-6.1909,-15.176,2.9386,-16.9836,16.711,-3.4433,12.3789,-25.0847,-16.4743,-8.7675,8.1486,16.2713,24.7088,5.7093,-18.7929,25.1905,-25.8365,19.1787,-7.203,-6.6582,-11.2366,4.6766,-2.5943,9.6165,7.098,-21.6902,-3.0334,-25.8508,16.8477,-11.3664,22.3628,11.3098,9.3139,24.2727,-6.6474,17.563,-21.6262,-4.1078,-21.6577,-12.9418,-0.60226,-6.3517,9.1461,6.4794,6.6926,6.6141,11.1604,-14.6256,30.6898,-27.1321,22.9995,-8.7143,12.799,4.8513,1.9457,-7.7679,-16.4676,-18.0686,-19.9977,-25.1459,1.7576,-3.3112,18.0282,17.2794,15.0752,26.1957,-6.5771,15.8619,-22.009,-11.9452,6.2293,7.242,-8.8249,-3.1355,-15.4138,-11.4068,-19.5544,4.8796,-24.3511,22.9132,-4.2593,19.4045,19.3962,21.2978,3.9162,6.9302,-15.1973,-15.1861,2.2664,15.6838,-22.8133,9.4995,-23.9436,-6.3854,-15.858,4.7068,-17.7116,23.4543,-1.4688,19.1966,8.8226,19.4915,-12.3094,12.7991,-22.2558,-9.9498,26.4861,-0.32159,27.2299,-25.2043,1.6811,-25.6933,-7.0649,-14.8848,7.1414,-4.2697,10.7083,7.6973,9.791,-11.7544,22.9668,-18.8929,17.3712,-4.0935,25.402,25.4912,3.2929,0.51908,-21.9399,-23.6219,-8.0528,5.5502,-8.2971,15.5915,3.7483,14.0325,-21.2546,-2.0013,-22.9359,26.1525,-3.4771,28.1781,19.3886,15.8288,0.20557,2.7968,-12.377,-3.0236,-5.1882,-24.2975,17.137,-14.8848,27.9115,4.6126,-19.8947,0.59111,-20.621,20.9156,1.1064,22.1322,16.1759,19.4877,-2.8397,6.4799,-14.7788,-8.6043,-2.9687,-16.9892,14.2633,-12.0505,22.3579,9.5936,-18.2248,-20.7061,1.9838,0.94429,21.9823,22.7376,18.8019,-3.1404,22.8055,-16.5677,8.9399,-9.3497,1.2106,14.5497,-12.3565,20.113,-11.4198,8.0026,-18.6595,-20.2994,1.193,0.65109,22.6721,23.1826,19.3165,-3.1498,22.3751,-18.4964,8.08,-10.1758,5.2849,17.9694,-11.5759,18.1871,-11.2836,5.5301,-21.1071,-21.6727,0.65192,0.21421,25.8272,26.2882,20.0305,-3.7778,22.2673,-19.5657,4.2251,-10.3737,2.9068,16.5663,-11.6544,15.5685,-8.4004,2.4383,-22.8797,-23.0906,2.5294,0.43432,25.9692,23.2151,18.9306,-3.8104,23.1833,-21.0324,4.4012,-16.0152,-1.809,11.5342,-13.4323,14.1505,-4.863,8.8909,-20.4753,-22.9655,3.7777,-2.856,24.4487,18.2294,17.4066,-4.8294,23.3632,-20.7379,7.4255,-15.2142,-4.0742,8.3576,-9.8996,15.9576,1.1891,14.6322,-19.8188,-23.2957,4.2049,-5.6184,21.2262,15.2857,14.3419,-7.0356,23.802,-20.6493,10.8592,-12.356,-6.7677,2.563,-6.2425,12.0849,6.7756,15.8069,-19.4044,0.41786,-18.0497,20.4782,2.0369,21.6128,16.338,19.1811,-0.21014,6.8983,-7.6634,-4.4276,-4.9695,-18.5007,16.3537,-11.818,24.4042,10.581,-21.5177,-16.3822,-3.2739,3.247,20.2856,22.3984,11.8936,1.5441,12.1128,-4.8931,2.4238,-4.7307,-7.3577,17.1972,-23.8817,25.1678,-14.681,10.1084,-18.3715,-21.7934,1.1782,-5.5232,16.6391,13.512,12.5455,-10.2846,24.7804,-22.6506,12.3397,-11.3726,-8.5513,-4.21,-0.21218,8.3516,12.1899,16.6891,-18.0834,-24.4617,6.4234,-5.9415,25.2254,18.7954,13.9849,-6.6616,21.8453,-23.1392,7.1658,-18.1274,-9.4196,1.7916,-12.9297,9.4603,-6.2976,5.2441,-15.6042,-25.1717,5.1355,-12.4426,24.5749,8.7456,24.8029,17.6992,-2.5596,20.702,-19.875,3.6296,-18.5014,1.3957,-1.1876,0.24668,-1.9559,0.55085,-0.44855,0.80476,-0.2454,-18.8174,-24.5982,6.0483,2.8252,24.6849,17.9258,19.3364,-3.6354,23.7142,-27.716,3.5645,-25.6513,-4.6174,0.15494,-6.9039,2.3193,-4.5336,0.057526,1.5009,-13.1384,13.6863,-14.6765,11.278,-2.4797,21.9149,21.5017,0.41923,0.23048,-22.2878,-22.3836,-13.7742,6.4976,-20.2138,19.2299,-6.6752,12.6843,-25.2601,-13.3879,-13.3836,11.9562,16.1102,25.8642,-0.75997,-21.5586,18.2133,-21.6515,20.1348,-0.32288,0.7583,1.1668,-0.12163,-0.17427,0.78584,0.25113,-13.6605,-25.877,12.1099,-11.021,25.6223,13.2358,-22.225,0.1161,-22.5629,22.92,0.67163,22.9137,0.54872,1.9848,-4.2385,2.1828,-5.5383,-0.52846,-14.0936,-24.8887,9.7145,-12.2465,22.0941,10.9827,-21.3123,0.2493,-20.4703,21.9934,0.96513,22.9883,-1.4458,6.1919,-10.6116,4.963,-11.8256,-3.7499,-9.3783,-2.4427,1.4618,-0.50993,8.5864,-0.47581,-6.2804,21.4765,-30.503,26.1048,-23.7155,7.0545,-24.7811,-13.9149,-11.3714,12.0084,13.4525,26.1621,-14.2894,2.0644,-14.4498,16.0765,-1.908,12.2806,13.1173,25.2503,-13.0559,12.3347,-25.1135,-11.5008,-17.4668,-3.5748,-15.177,14.443,4.1795,19.7118,-8.7555,0.6954,-4.9564,4.017,7.8482,7.0295,24.9488,7.8043,19.9777,-18.7653,-4.0999,-25.5712,-4.8992,-24.9482,17.3996,-19.2948,24.264,5.9451,-17.2711,-10.6628,-9.4025,0.15731,5.4177,7.8554,18.3317,1.8252,17.7185,-19.1918,-3.3221,-23.319,-6.4286,-22.8777,16.9935,-16.8205,25.0989,7.4385,-18.3778,-8.753,-12.3376,3.9765,2.2405,8.2951,15.8602,0.012675,16.4574,-18.0359,-3.5987,-21.8675,-7.5215,-23.7266,15.2871,-16.7742,25.1985,8.9803,-5.0836,-6.433,0.53537,-3.0934,0.93924,-1.3547,24.4849,16.8968,7.2235,-7.2296,-17.7535,-24.7902,-4.3421,19.2739,-25.2944,24.4585,-18.1677,6.2443,-8.161,-6.486,6.4323,-4.4648,5.4891,42.2177,34.9158,-13.1315,-11.6961,-40.1074,-3.8288,-15.9525,50.2805,-21.7246,8.1478,-1.4982,2.9969,-7.9815,2.0962,-8.7706,-3.8333,-1.5681,20.9242,-21.7166,24.0178,-18.8087,2.8069,-26.7726,-17.2968,-11.7955,11.0205,15.1235,26.8278,2.3992,1.5802,-1.4534,-3.6976,-6.4445,-7.9339,20.3922,-3.0049,23.4453,-22.7883,2.3848,-19.898,-16.0636,-26.644,10.7374,-11.9047,25.7733,14.5943,2.5281,4.5457,-3.8536,-0.087063,-8.7516,-7.0562,20.9744,-2.9565,23.3396,-23.2088,1.8147,-20.7851,-14.796,-25.6548,10.2738,-11.1685,24.531,13.9475,-16.2592,-25.4885,8.6868,-9.2998,25.1544,16.0219,-19.3906,1.8886,-25.3875,19.4722,-6.0625,17.5151,-8.9355,0.90893,-3.9898,13.0112,4.5687,11.5396,9.6041,-22.412,29.1716,-30.7394,17.8618,-9.1943,20.6221,16.7157,-0.48202,-2.7077,-21.6696,-20.805,-16.2625,-19.9212,-1.4549,-2.0135,7.0272,11.1121,14.5932,21.9014,-4.5982,10.0652,-19.6108,-11.5434,24.0647,-0.75747,22.9607,-28.2805,-0.57616,-26.7996,1.5686,-12.1156,17.796,-10.2804,16.2905,1.257,-17.37,5.8733,-23.0509,20.7346,-4.8633,17.591,23.0868,24.4541,-3.0239,8.571,-20.9531,-13.8417,-5.9624,-19.3375,7.2227,-14.4452,17.822,14.6874,-28.9239,-25.6133,-4.9129,4.4757,24.1942,27.4714,5.343,-4.0968,5.217,-5.6536,-1.1487,-2.9102,-11.8369,20.0104,-26.4082,25.389,-14.3044,9.2381,-26.9276,-22.1309,-4.0736,3.6662,22.784,26.047,7.5827,5.2527,2.0286,-0.85976,-2.0979,-2.6403,-7.271,15.1565,-25.9518,25.8845,-15.1774,12.6086,-23.9275,-20.2332,-3.4299,2.9732,21.7491,23.8642,6.2967,2.9954,4.7924,-0.68761,-0.3707,-2.3219,-8.8953,16.1678,-26.1745,25.4805,-15.2371,11.5879,-22.7445,-19.4048,-2.2096,1.9376,21.1224,22.6356,6.2588,1.4115,6.6798,-2.8797,2.0261,-2.8823,-9.33,16.354,-25.3956,25.6304,-14.8092,11.1888,-18.9526,-24.6595,5.8618,-5.6414,24.3579,17.0176,18.0215,-4.2697,23.0598,-24.8718,4.2619,-21.6383,-5.4199,2.5371,-9.6196,4.8376,-6.372,0.38252,3.632,8.1309,-4.4837,3.795,-7.9819,-4.0346,21.8565,-1.9668,23.1705,-23.1423,1.086,-21.5412,-13.4214,-23.8012,9.934,-10.1946,23.788,13.5443,-0.72675,7.1113,-11.029,3.6583,-12.0009,-5.0904,21.0948,-1.3978,21.7427,-22.8328,0.45602,-21.7778,-15.14,-23.8716,7.3279,-10.4446,21.5969,12.4923,-0.26694,-12.5607,10.8394,-14.6335,9.7191,-3.6722,21.1332,21.2836,-0.4918,-0.27402,-22.2354,-22.1524,-15.1616,6.146,-22.408,19.2041,-7.9245,12.5859,2.3699,20.3236,-17.9794,16.2902,-22.6837,-7.4783,20.9934,2.5331,17.6101,-18.5762,-3.2062,-20.8512,-11.5569,-13.9091,-0.012918,-5.923,8.6229,5.8595,1.4814,-17.7479,18.4116,-21.4055,16.4494,-7.4864,20.6086,18.6455,1.3871,-2.0143,-16.3734,-17.8557,-12.1024,2.7165,-16.1994,11.1595,-8.8429,6.1502,3.6711,22.3492,-18.7419,17.0689,-23.7706,-7.0125,21.9447,4.1799,17.0926,-18.9544,-4.0616,-22.4684,-11.7175,-11.0679,-2.185,-1.6621,6.6752,6.7854,-0.018081,-18.0614,14.8208,-19.7955,11.7725,4.4077,20.976,20.1431,-0.086741,-1.1128,-24.8261,-24.6837,-13.6771,4.0801,-18.7504,13.8538,-6.9404,1.6902,0.91796,-14.3838,13.1741,-16.1098,8.391,-6.2504,21.8577,22.0148,-0.078905,0.14114,-23.4615,-23.6283,-13.7805,5.8877,-19.9128,17.3545,-8.4917,9.5994,4.589,-19.3263,-24.6572,23.7865,-6.4517,17.9041,22.3297,16.0373,-4.9324,5.4371,-23.4393,-19.231,-12.35,-5.1842,4.3755,-9.2704,7.5632,0.94281,-24.0151,-8.5881,-15.7711,16.045,5.704,22.6562,-0.13259,18.2223,-17.5841,19.0899,-16.1889,2.2612,7.9567,14.6822,-9.104,5.1574,-17.3574,-9.9366,-24.3077,-15.3784,-7.6101,9.3178,17.4693,24.9567,-1.2425,-19.2291,16.1286,-20.8807,17.7692,-1.881,7.7059,-8.6448,18.0108,-10.5331,8.3582,-4.0456,-24.4794,-15.188,-9.5138,7.2104,14.707,22.7243,0.81163,-18.6985,19.9881,-17.9378,20.1862,2.2396,7.6573,-6.131,12.1066,-14.7417,2.1184,-11.0129,-24.8988,-14.8853,-10.2966,8.5505,13.663,22.5228,1.455,-19.3306,21.4853,-19.1034,20.8369,1.0041,6.8453,-4.154,9.6772,-13.9057,0.086739,-12.5983,-24.7716,-14.4593,-11.3505,9.3962,12.9961,23.2919,1.0663,-20.0323,21.4894,-20.5653,21.1637,-0.32029,4.9835,-2.5326,4.9267,-12.4788,-1.6703,-11.5945,-23.7451,-13.5234,-10.9455,11.2947,12.4507,24.3508,-0.71937,-21.0967,21.3992,-21.4302,22.175,-0.1467,8.7498,1.1586,7.5408,-5.5498,-1.8237,-7.267,-12.5351,-3.2317,-9.7442,15.2423,1.7766,17.1097,22.616,-1.9312,27.5755,-15.0711,2.775,-15.707,-11.5968,-26.1186,11.6196,-16.7446,26.1236,11.0378,0.12627,-14.7578,12.3265,-18.026,9.8796,-5.5592,25.1001,12.9236,13.5688,-12.5097,-9.8986,-23.2779,-2.1713,-15.3091,16.1124,-10.5824,20.4402,6.6197,-11.603,-2.6815,-12.6406,3.3875,-5.0291,3.2674,14.4246,28.7256,-13.9343,13.4578,-25.4771,-9.1893,-20.6893,3.9751,-17.613,19.933,6.1311,25.2093,-11.5574,3.1983,-15.712,10.3742,-6.348,5.3867,12.0962,25.4852,-13.3775,14.925,-23.8259,-9.0405,-20.3619,-4.0515,-14.6503,17.4274,6.7233,23.2971,-6.2702,-18.6765,6.3088,-17.3442,7.1428,-3.124,-0.53984,13.6653,-17.4261,12.8912,-21.4801,-6.0049,-24.033,-9.6149,-15.6499,13.283,9.0554,23.8329,-3.4212,-7.2967,-3.6782,-8.8766,-4.8636,-5.5342,15.4393,27.6747,-7.1374,26.6495,-24.7913,-16.071,18.5654,-27.0547,24.9423,-28.0808,6.4438,-18.8766,4.589,7.9345,6.1239,7.0577,0.79584,-0.84135,-5.4414,19.1803,-26.7185,22.1518,-20.9444,0.62777,-26.8009,-14.6528,-10.7893,12.5707,15.3462,29.5186,7.6245,3.7055,2.8435,-4.3358,-5.1954,-8.5275,-2.314,21.5038,-23.4143,23.1516,-21.6705,1.2516,-24.5345,-14.0461,-10.5519,10.3826,13.4397,24.1193,-24.4778,-7.7105,-16.9519,17.8391,7.6637,25.5644,7.0865,25.9354,-17.9238,18.7155,-25.6432,-8.1027,-8.8619,0.86278,-10.6389,7.0923,-2.33,5.55,-23.9756,-7.4242,-16.5415,18.0872,8.3737,25.7208,7.933,25.9278,-17.5288,17.9339,-25.2589,-8.8105,-11.3958,1.0764,-12.8955,9.6638,-1.3407,8.1467,-21.5937,-6.7011,-12.9641,18.1974,18.378,28.4175,8.5005,25.2097,-14.4427,17.5685,-26.8391,-16.9837,-14.1527,0.45148,-18.6373,11.7014,4.8592,12.2897,-23.1617,-9.6669,-14.9898,9.411,-9.007,9.4112,14.3219,6.195,20.6337,-27.0112,13.1195,-27.9672,13.1192,-9.7794,-5.7028,6.2584,-8.6775,20.5084,-7.6691,20.5085,11.7464,-24.9581,-16.9983,-8.5265,7.545,17.4896,25.4497,6.2088,-18.7324,25.4815,-25.6587,19.4623,-6.9886,-5.6267,-7.9608,2.7723,-2.0221,6.4041,3.8814,-17.8051,15.0704,-29.2758,27.3571,-11.5186,18.7609,22.1082,22.0178,0.64863,-2.5654,-23.4017,-24.9768,-5.0246,-4.9948,3.7582,-5.2717,6.8284,-2.8345,11.2703,28.6856,-13.6801,20.8003,-25.8865,-9.1912,10.9034,-1.1696,3.3861,-17.8005,-9.0089,-18.89,-19.6019,-2.2391,-22.7857,12.876,-2.8354,16.004,7.7399,-19.9713,29.7335,-29.5134,20.9824,-9.1031,19.1753,19.2382,-1.4545,-0.80289,-21.2856,-20.628,-16.0147,-15.0403,-5.6526,3.4064,5.8733,13.6014,15.0788,25.6254,-6.1303,14.3199,-21.6625,-12.0718,5.9143,8.0207,-9.3663,-3.3782,-15.7229,-12.2178,-19.195,3.3813,-23.2754,21.8456,-3.7897,19.0931,19.7955,21.3163,4.6201,6.2258,-16.2285,-16.3991,0.98297,15.7518,-22.7249,9.5467,-23.0998,-6.4046,-16.3272,4.0788,-17.8885,23.0353,-1.2032,19.043,6.0938,-12.407,20.1388,-21.1549,13.3392,-8.6848,24.6852,24.3358,0.2444,-0.82319,-23.9696,-25.0514,-11.3219,-1.2945,-13.8348,11.5096,-5.0288,10.0193,12.1907,22.3567,-10.3595,15.7744,-21.7811,-7.0575,26.5995,1.3575,28.0265,-18.5438,1.0646,-21.6456,-0.42794,-6.4028,12.5053,5.512,14.0371,13.9738,5.6796,-16.0454,22.4303,-21.9103,17.9503,-5.6448,24.9638,23.3775,6.4692,-1.2026,-14.2891,-21.2035,-10.971,-0.1977,-3.833,10.9643,11.7349,16.252,-17.2574,-2.5648,-18.9831,27.4664,-6.0928,29.5888,10.6358,3.806,7.1204,-6.6916,8.1604,-13.4822,4.2429,-4.8468,-13.4663,-24.623,1.8987,-18.5516,18.4406,-7.9782,22.1403,-21.8599,-20.6705,-0.82732,0.87123,20.7579,22.1555,10.0878,-4.9722,17.0884,-11.7907,9.7238,-4.5143,-8.0626,13.3107,-19.7173,23.425,-10.9274,11.1506,-21.5847,-1.4617,-20.3452,20.9152,0.98709,21.7719,7.8805,13.5764,-4.4993,6.1802,-8.8609,-4.1804,-10.5399,-22.9043,13.9304,-12.5543,24.377,10.8522,-22.6334,-21.9103,-0.87733,1.5084,21.2787,23.3142,9.5565,-7.7387,17.5631,-15.5622,10.1857,-5.5652,-6.7673,11.7147,-19.02,21.3825,-11.8933,9.7588,-22.8864,-22.3053,-0.7756,2.0157,22.3418,24.4645,11.4857,-7.3873,19.3011,-15.7175,13.3881,-3.9482,-4.285,13.4899,-15.388,21.131,-9.7339,9.2963,-26.2615,-25.5126,2.9877,2.1953,27.8813,27.5224,16.8074,-8.9274,22.407,-17.7077,7.4068,-2.0818,-5.7501,12.9497,-13.9063,17.8749,-1.05,7.2259,-23.1132,-26.2028,6.448,-0.68485,25.1871,15.659,16.3367,-6.3917,23.3131,-21.9249,8.5645,-16.135,-11.0946,5.0155,-13.9855,17.9545,3.359,18.9869,-18.8303,-24.6774,6.0895,-8.6527,21.1437,13.0094,13.2238,-8.0989,22.7767,-20.6075,10.7369,-11.3763,-13.274,-0.64668,-9.3011,14.4539,8.4338,19.2566,-15.703,-21.3752,4.514,-7.4675,17.7312,11.4897,11.7914,-10.4206,23.8823,-21.5396,12.4843,-10.4005,-15.1272,-6.5058,-6.1067,9.9457,12.01,19.274,-22.3225,-20.1888,-1.2676,1.3721,20.8515,21.8574,5.2397,-1.9491,8.9269,-6.2289,4.4032,-3.5869,-9.7232,15.2872,-24.5114,25.0743,-13.6799,10.9145,-27.4001,6.7704,-22.1543,25.457,14.4586,24.0864,29.0028,-27.3257,2.1141,-25.4937,-1.8708,-4.058,2.8003,-4.4677,0.12525,-19.0435,-24.6963,7.2575,-9.9119,24.2965,15.0057,21.3073,-3.3515,25.0722,-24.6762,2.4917,-22.7292,-0.30399,4.0004,-5.3865,3.5154,-7.3118,-2.9427,3.3555,-10.6944,14.5507,-14.7359,10.8045,-4.7818,22.5619,22.4625,0.2693,-0.15099,-22.0885,-22.5237,-12.5289,7.2285,-19.2565,18.7614,-6.592,12.1091,-24.8558,-14.8432,-2.6486,12.6126,10.4828,25.2114,-1.6334,-22.5971,25.9299,-22.6456,26.4581,-0.34733,-1.5906,-2.434,-2.5549,-5.0917,-4.8194,-6.6159,-13.3757,-25.8484,10.7907,-12.3869,21.6595,10.2621,-22.665,-0.26816,-21.6578,23.5906,0.87171,23.3306,0.57536,5.6202,-8.2223,3.9211,-11.7262,-4.2428,-11.1487,-21.7734,11.5865,-10.5729,24.0784,12.1684,-22.3644,-1.3305,-21.5987,21.8546,0.10098,22.3586,4.2848,11.9722,-4.9659,8.3913,-5.8164,-0.16013,1.1578,-5.6481,10.8858,-16.1857,9.9839,-13.581,-3.86,19.6639,-21.3251,24.697,-20.3721,7.3932,-25.0721,-15.3728,-14.5809,13.0441,8.6046,27.2197,-10.4185,3.4314,-13.0431,12.9058,-3.8798,8.0684,11.5447,24.8783,-14.2159,14.416,-24.0453,-9.1011,-20.2919,-5.0615,-15.5803,16.1509,5.6952,22.064,-0.92366,1.0462,1.5274,-1.9234,5.7721,0.15955,25.9949,7.1654,20.0709,-19.0394,-6.6097,-26.0353,-5.1637,-24.6856,16.7415,-18.1106,23.7735,6.7061,-8.0055,-8.5569,-0.50971,-4.9649,4.8406,0.92622,23.2917,5.5002,19.2792,-18.4716,-5.661,-25.1259,-7.4434,-22.6865,16.151,-16.3089,24.6361,7.5907,-9.3941,-7.3456,-5.3253,-2.3427,-0.93015,-0.037829,21.3579,4.0398,17.8139,-18.0709,-5.703,-23.7177,-8.3247,-23.5895,15.6195,-16.1121,24.9467,8.5192,-8.5493,-2.9698,-7.645,1.8333,-3.6906,0.0559,21.0248,4.0536,16.9509,-17.5262,-6.3957,-23.1433,-9.5486,-24.5998,15.4224,-15.9847,24.4337,8.3198,2.7389,0.3085,1.6073,-2.323,-4.2816,-5.7418,24.8909,17.6249,7.9914,-6.2828,-18.0245,-24.5035,-6.7745,18.4414,-24.4819,24.4413,-17.6827,6.7958,1.7129,7.5309,-5.0493,3.2161,-8.9748,-5.9145,-3.3432,19.9814,-19.9074,22.924,-18.4134,2.2527,-27.6941,-14.9231,-14.5368,12.0469,15.3615,27.1896,6.5316,4.7178,-0.6396,-3.4906,-9.5987,-10.8332,20.8239,-3.026,23.3238,-22.0435,2.5924,-19.3849,-14.6047,-26.6412,11.3443,-12.7059,24.8286,12.9902,5.9081,7.6959,-2.5509,-0.85217,-10.0944,-10.1126,21.5995,-2.5387,23.4313,-22.8146,1.6202,-20.3131,-13.9791,-25.2918,11.2071,-12.0562,24.0681,12.0094,-7.3731,-26.3803,31.6842,-12.5603,13.6703,31.1189,-0.3601,23.185,-28.2496,-32.7494,1.9848,-1.6193,17.8303,-3.6162,2.492,-16.6279,-25.3398,7.8537,-9.2612,25.6535,17.2527,-19.419,3.3656,-25.0824,22.8372,-5.1612,20.5751,-7.341,-1.0065,-3.9271,5.8194,-0.21031,2.9896,11.414,-20.5392,27.3177,-30.181,17.6399,-10.0131,21.0209,22.2595,-1.5287,0.40968,-22.1794,-22.0853,-10.7853,-5.4595,-9.612,7.2462,-1.9177,10.4314,14.9907,21.9029,-4.8991,11.2883,-21.5744,-10.2069,23.3011,-1.2269,22.9954,-26.9372,-0.48219,-25.7521,1.598,-13.7623,17.3773,-11.916,15.9966,0.62749,-17.567,1.3921,-22.4029,19.921,-6.789,14.5951,14.4839,20.5333,-7.4581,8.659,-16.0613,-4.5992,-13.9891,-16.5861,3.0865,3.5488,19.7988,20.3843,-31.9421,-26.5339,-3.6252,32.6446,28.4684,-2.5068,1.1011,2.8797,9.5872,4.9104,-6.8373,36.2197,-26.4215,20.2629,-15.227,-27.8604,-23.8462,-3.0982,3.6991,23.3119,26.1401,2.0084,0.030533,0.35703,0.12287,-1.2905,0.38248,-11.5802,17.1834,-26.2493,26.6692,-13.5128,14.9089,-25.7572,-22.9488,-2.1346,2.5427,22.5638,24.3647,2.7855,2.2832,0.87772,0.61516,-0.90803,-0.64814,-10.3031,13.6407,-26.0172,25.9378,-14.0328,13.2889,-23.6867,-21.7296,-1.596,1.6763,21.7092,22.8885,3.2504,1.4442,3.2614,-0.2787,0.72,-0.9493,-10.3933,14.4292,-25.8042,25.7869,-14.2467,12.6013,-22.4945,-21.0396,-1.0358,1.0854,21.2727,21.9997,4.2029,-0.37902,7.1021,-2.7843,4.2808,-0.96579,-10.2652,14.6154,-25.0584,25.5883,-13.692,12.0116,-17.9063,-24.2752,5.8665,-9.0553,23.7812,15.4084,19.1774,-3.4339,23.2925,-23.7099,3.5958,-21.2216,-1.1912,5.004,-7.618,3.0162,-7.5372,-3.1018,4.3236,8.3704,-4.6688,4.2557,-9.7845,-4.931,21.7469,-1.6104,22.8374,-23.0922,0.55659,-21.8793,-13.3434,-23.927,10.3732,-10.4537,23.1941,13.0184,2.2817,-10.8785,9.7591,-14.5725,4.784,-6.3403,21.7794,21.8831,-1.0597,-0.20391,-22.7562,-22.2558,-13.6724,6.6706,-22.8656,19.1883,-10.3781,11.0581,5.1153,-15.1387,21.8756,-23.0101,12.4378,-10.4856,20.5904,18.6361,1.5485,-1.5877,-18.9857,-20.0462,-5.2365,4.5547,-11.2129,8.6627,-6.078,2.9766,5.6148,22.617,-16.7687,-23.8715,13.8749,-9.9015,22.1606,2.2152,18.8726,-2.4111,-20.2466,-21.8113,-6.9608,-10.4143,0.75171,4.9351,-5.8508,1.5446,3.3461,-14.3892,20.6188,-22.6219,20.4145,-11.4355,6.5203,17.4862,16.3072,0.7953,-1.0361,1.0106,-18.2301,-17.5877,-5.011,5.0627,-13.479,8.7881,-13.7409,3.0572,-5.554,4.3374,-25.0265,15.0143,-18.4726,7.7168,24.685,23.8884,0.18789,-25.3507,-23.9253,-10.4141,11.7888,-18.0011,11.8243,-9.0156,6.3449,-8.7639,15.9247,-15.8786,9.7298,-7.6163,23.6125,23.4404,0.62612,-1.0715,-22.2084,-23.1251,-10.2697,8.4016,-17.5837,17.8035,-6.8344,10.5274,-23.2553,-16.3049,-8.1354,4.2169,15.5509,20.7197,-1.1521,-18.8919,18.2444,-17.2162,20.4276,1.4826,9.5842,-6.4819,13.4147,-14.9393,2.2817,-10.2832,-23.1733,-15.6955,-6.3514,7.4899,18.1497,24.2302,-1.5803,-20.9562,16.6308,-22.8763,18.3747,-2.3937,8.9992,-1.7737,16.4185,-4.5717,6.6956,-3.8117,-23.2575,-14.5889,-8.6683,5.2261,14.1965,22.0271,-1.0829,-20.484,19.801,-18.6149,20.7299,4.6085,11.7873,-2.4018,13.237,-11.813,0.71867,-10.275,-23.0592,-13.8545,-8.8823,9.659,14.7194,23.4666,-0.2716,-20.8051,20.4185,-19.8571,20.0556,-0.52016,12.392,-0.63869,13.2644,-13.4903,1.6268,-12.1315,-24.2908,-14.4096,-11.3314,8.8297,12.351,22.4542,0.26647,-20.0982,21.3905,-20.4617,21.4083,-0.34175,7.2201,-1.6699,5.5189,-13.1108,-3.3242,-12.9595,-24.0372,-13.0194,-11.3762,10.7511,12.8429,23.4033,-0.57636,-21.2438,21.4076,-21.2291,21.7377,-0.2129,6.7896,1.7481,5.9444,-7.0919,-1.5187,-9.3429,-12.6623,0.23451,-15.621,13.1167,-2.6268,10.0466,18.5502,2.8157,18.291,-14.6223,-7.0081,-20.7819,-11.5427,-24.3208,12.2688,-14.6882,23.2271,10.2918,3.1635,-12.172,12.109,-18.5924,7.856,-7.7945,24.1145,10.7936,14.5371,-12.6485,-8.9878,-22.5081,-4.8599,-18.6297,15.8441,-11.1625,21.6666,7.791,-5.3451,3.2563,2.2667,-10.7113,-1.971,-8.202,10.5521,16.7999,25.2603,-15.3566,-6.2345,-23.401,-23.5584,18.0381,-6.3327,-17.223,24.0588,6.6799,-6.5578,5.3246,-11.6353,7.9549,-6.8983,0.70913,9.41,24.3971,-15.1606,16.4917,-23.3332,-6.7576,-22.8742,-6.0742,-16.6843,17.1877,6.6062,23.6589,0.50228,-9.9927,4.7425,-12.838,-0.39999,-7.5878,4.5292,20.6273,-16.8113,14.9443,-23.2329,-8.0759,-24.4777,-9.1447,-16.1043,13.8524,7.5082,22.6767,15.3899,5.8906,17.9061,2.7151,-2.0905,-3.7097,-3.7526,19.2797,-21.1798,20.5204,-13.2539,1.7773,-28.3903,-14.9817,-14.1959,11.6167,19.9748,27.3092,7.7464,4.2718,0.48844,-5.1185,-10.6521,-12.8789,-1.7609,21.8406,-22.7835,23.0853,-20.925,1.2436,-24.6469,-13.6656,-12.2471,10.419,10.4378,22.1454,-25.3841,-7.4105,-18.2695,18.1896,7.2956,25.7163,6.9433,25.6845,-17.7146,18.7878,-25.6383,-8.3219,-5.0702,0.84757,-6.5298,3.7142,-1.9803,2.2825,-25.7509,-7.2949,-18.5467,18.7942,6.5307,26.2923,7.7092,25.8173,-17.3593,17.7205,-25.6738,-9.0796,-7.3338,0.97989,-8.8803,5.682,-2.165,4.2064,-26.5712,-7.1741,-18.7894,20.2439,4.8348,26.8611,7.6904,26.1698,-17.4931,17.0965,-31.5472,-12.7155,-10.3608,0.73364,-10.5851,7.999,-0.57787,8.6899,-27.7889,-9.628,8.9659,19.3379,24.959,13.7196,34.4768,-39.9577,17.2032,-20.2282,-25.3676,-1.7045,-15.6492,10.7,10.2521,-22.6773,-3.8041,-22.6773,4.5668,-12.315,8.8228,6.4131,4.7822,19.7853,4.7822,5.2422,-13.9528,-18.5288,-23.2581,-10.1612,7.1734,-10.1612,26.3644,-13.5369,19.0988,-3.1501,-25.3434,-18.0392,-8.1058,7.1483,17.8018,25.4885,5.6201,-18.6959,24.9166,-25.4768,19.7219,-6.527,-2.477,-4.7289,1.0116,-1.9444,0.36472,-0.2889,-14.7968,9.007,-27.1199,25.9275,-11.5249,11.9517,30.8149,27.9672,1.3482,-2.2393,-21.2823,-22.096,8.8879,7.6691,8.134,1.8233,2.041,-6.7089,-15.7991,8.359,-27.3877,23.9574,-11.4135,13.3933,24.9303,23.6287,0.090476,-0.097844,-22.8564,-21.3655,2.7837,-5.2995,12.2186,-13.5759,6.8716,-10.1628,8.7131,27.756,-18.51,19.1078,-28.1162,-8.9832,18.6034,-0.98829,17.4728,-20.5901,-2.1881,-20.5615,-16.2824,-7.9029,-13.6157,5.0604,3.2312,13.2139,11.5666,27.7012,-13.2641,19.1673,-25.3832,-9.7433,12.3871,1.4173,4.0901,-16.0453,-8.9541,-18.6291,-19.0102,-2.3884,-21.592,12.5836,-2.3328,15.5061,9.1986,-18.6529,26.7864,-28.1087,16.9371,-9.7652,20.8759,21.6163,-0.094497,0.1837,-21.9969,-21.6494,-13.1195,-5.6985,-11.977,8.4776,-1.3523,11.1205,15.4366,25.3359,-5.4715,13.7519,-22.1648,-12.4815,6.1761,8.1277,-8.5823,-4.192,-15.1773,-12.5567,-19.2456,2.4989,-22.9337,20.8903,-3.4907,18.8674,21.1276,21.8706,6.999,6.201,-17.0547,-16.0746,0.1044,14.9869,-19.7323,8.8169,-20.0663,-5.1687,-16.1741,3.8907,-17.3749,22.1257,-0.044332,18.5178,5.3805,17.4703,-15.442,8.179,-22.755,-12.1676,19.367,0.74596,17.0927,-20.2111,-4.7051,-23.2898,-17.0216,-15.4914,-8.3124,-9.582,10.0176,7.8691,8.571,-8.7854,22.0095,-20.2165,16.7047,-6.6065,25.1722,24.3054,5.7763,-2.3225,-15.1795,-21.9159,-5.9356,5.4721,-5.1204,13.4373,8.3604,14.3623,10.4929,22.4995,-14.4025,16.6115,-22.5861,-5.3744,25.8726,4.553,27.4154,-13.0544,2.6209,-17.0408,-2.1211,-5.6614,10.7493,8.7483,14.5444,19.0962,4.4527,-16.3647,21.1345,-22.1802,14.6663,-7.4809,21.7473,15.4273,11.0212,-7.3959,-4.0842,-16.3588,-14.8214,-11.1316,0.42525,3.9077,19.2075,19.4276,-13.8282,7.8203,-20.6672,14.213,-12.6433,2.9651,4.8231,15.4838,-11.4343,15.994,-8.0454,8.9781,-21.9894,-17.473,-0.74404,6.3295,20.8959,23.4398,-3.0548,-18.4561,-3.0549,-14.2979,22.2817,1.2765,22.5615,-5.344,-10.6371,-5.344,-7.3257,0.40384,4.8122,6.9781,-28.2414,-4.6672,-28.2414,19.9497,-12.432,26.6235,11.541,-21.7864,-20.8968,-1.2752,0.96346,20.2621,21.8571,5.6104,-5.0644,12.7946,-8.2838,8.9408,-1.5401,-10.8707,13.4607,-22.6119,24.7273,-11.4395,11.9006,-22.2663,-20.8803,-1.9703,1.8428,19.9327,22.7444,5.8383,-8.2804,15.1176,-12.2331,11.1895,-2.1956,-10.4551,11.4613,-21.291,23.3195,-11.145,11.8699,-21.8386,-20.8216,-0.87507,0.76063,20.952,21.8537,4.4577,-3.5152,9.2448,-6.1803,5.9269,-1.5381,-10.6972,14.1439,-24.3129,25.1576,-13.1025,11.6678,-23.2944,-21.3439,-2.0146,2.8804,18.0235,23.1591,5.5318,-9.5524,16.7435,-13.7105,15.089,-0.5119,-9.2377,11.5323,-20.4176,22.8936,-11.3536,12.3779,-24.5371,-22.4499,0.93527,4.8906,15.5464,24.7211,4.3196,-12.7953,19.9941,-14.9286,20.7519,3.6286,-8.141,11.1213,-18.8863,23.6024,-12.9672,11.802,-27.1965,-28.4577,9.6608,9.4985,28.131,14.5181,-17.9123,26.6643,-17.7142,8.1965,-16.9046,11.6604,-15.6127,25.0573,9.7819,-16.5344,-25.3198,8.9664,-14.5713,23.0858,10.7794,12.043,-10.5293,23.143,-21.2206,10.8604,-12.6673,-17.8129,-3.3742,-11.9201,11.7008,9.9598,21.7305,-13.6762,-21.9758,7.1027,-10.7017,18.7085,10.1697,10.8027,-11.7667,23.0491,-21.5729,12.4502,-10.2598,-18.3794,-7.3366,-8.1115,10.154,12.2753,20.7379,-15.5451,-24.9414,-12.8244,8.7629,12.402,23.4021,22.5622,-0.96445,-23.2433,24.476,-23.5225,0.80121,0.95229,4.6942,2.6948,-5.1016,-6.842,-10.6708,-15.2412,-23.7686,8.4631,-10.738,24.3855,13.3353,21.2182,-1.7467,23.6326,-22.9324,1.514,-22.3932,4.2956,8.0443,-1.8333,2.8849,-5.7227,-4.8052,3.768,-11.1429,12.8223,-17.0851,4.3666,-10.2541,22.2924,21.8975,-0.46786,-0.9219,-22.5356,-22.3087,-12.5964,5.4747,-20.6496,16.4987,-11.065,7.488,-11.0866,-22.6641,11.9699,-12.6642,24.0844,10.6476,-22.7263,0.19196,-22.2323,22.1312,0.067019,22.7063,3.3348,11.0848,-5.1801,7.7859,-5.4162,-0.27198,-12.7022,-24.1383,10.3231,-12.5469,22.9832,11.7249,-21.5085,-0.12634,-21.2065,21.2153,0.43888,21.4578,-0.12796,5.3403,-7.14,5.1045,-8.9237,-2.0545,5.6371,-4.3251,14.2733,-16.5007,9.4297,-17.5538,-3.0803,2.0811,13.7866,-14.3246,17.3289,-23.2099,9.8602,-11.9627,-20.8734,-13.0239,-13.6711,5.2199,10.4554,16.7274,19.8964,-8.2292,3.567,-12.5922,8.1112,-4.9593,3.8781,9.5082,24.2617,-14.6634,16.4154,-23.6506,-7.9643,-21.5394,-5.3796,-15.5606,16.9866,6.5082,23.2228,4.2463,5.7705,0.9515,3.3938,-5.2058,-4.3172,25.3222,18.9239,6.5912,-6.7873,-18.0781,-24.6346,-6.3307,16.5619,-24.4267,23.9229,-17.5518,7.4709,-0.80711,-5.9301,4.9563,-8.3773,3.3662,-4.8012,25.0814,7.1114,18.5561,-17.513,-6.3726,-24.3658,-7.4647,-23.268,15.977,-16.1683,24.1968,7.7794,-2.4979,-5.8825,0.1675,-7.2294,-1.6794,-5.4199,23.7319,6.5698,18.0113,-17.2738,-5.3816,-23.1551,-8.524,-23.5677,16.4217,-15.7535,24.8376,8.1982,-0.90314,-0.12967,-2.6853,-1.6867,-4.2193,-3.9626,23.2292,5.7064,17.2983,-17.9154,-5.4976,-23.1496,-8.9621,-24.9408,16.7295,-16.4449,24.7164,8.0598,-1.052,2.6294,-6.3825,1.3669,-7.6729,-3.7274,23.207,5.8449,16.4127,-17.93,-6.1533,-23.2926,-8.3294,-25.0493,16.7246,-16.6753,23.7205,7.5754,6.2526,0.24716,4.7903,-7.1074,-1.1031,-7.039,24.2591,6.9093,17.9124,-16.8803,-5.813,-23.5441,-6.3761,-23.8676,17.631,-17.2569,24.2404,6.8387,6.2597,9.4051,-1.4986,1.7766,-9.2863,-9.1756,-2.107,19.7908,-21.0798,22.5566,-18.7116,2.139,-26.9782,-14.4837,-11.1193,12.4273,15.3278,26.1862,10.4509,9.0566,0.57661,-1.4223,-9.343,-10.0782,20.2223,-3.1716,22.8896,-22.5217,1.9793,-20.5081,-12.7725,-25.4175,12.8911,-12.3945,25.3659,12.7112,9.6059,9.536,-2.2744,-0.22039,-12.965,-10.8176,21.1633,-2.5765,23.1473,-23.0464,1.7274,-20.7848,-11.8209,-24.9005,12.2609,-12.699,23.0386,11.0186,-18.3772,-10.5178,-5.6522,-0.30873,16.9419,2.7656,12.3938,26.8386,-10.0059,38.112,-20.568,18.2883,-16.9801,-2.4009,-4.8593,-18.9198,0.30976,-5.5326,13.4935,12.5299,19.8907,-17.5477,-24.7427,7.2265,-7.9042,24.8187,17.4173,-19.7554,2.3012,-24.7241,22.8782,-5.1387,20.3542,-0.80127,1.5821,-1.388,2.4532,-2.1956,-0.9531,9.7478,-16.7733,23.0035,-26.5891,11.8552,-11.8124,22.0889,22.1798,1.4518,0.30092,-20.7753,-22.0536,-10.301,1.5456,-16.751,12.238,-10.1314,7.4036,14.5753,24.6094,-3.221,18.9726,-19.6583,-8.1416,20.2145,-1.6633,19.9716,-29.543,2.4974,-28.4438,1.3783,-11.3768,14.1897,-0.18829,12.8371,11.5798,-14.3455,6.0721,-21.4354,16.3307,-12.1272,5.0472,5.0401,16.0335,-11.5407,17.3321,-8.3005,9.4025,-20.224,-19.6403,0.37329,2.6768,21.497,22.199,-29.5896,-32.1675,0.6376,-2.7168,23.8213,22.9273,-4.2432,-5.0332,2.5529,-2.205,1.5701,-1.615,7.9785,-11.4422,26.8425,-27.0432,14.0931,-12.4396,-25.8546,-23.4233,-1.7597,0.85811,22.8116,23.468,0.67537,1.8225,-0.70343,2.3911,-1.0957,0.72473,-12.425,12.1343,-26.3915,25.9609,-13.0036,13.6995,-24.2401,-22.6049,-1.184,0.61202,22.2222,22.5482,1.1867,1.3516,0.4411,0.97685,0.19961,0.55025,-11.7541,1