from .context import gpytoolbox as gpy
from .context import numpy as np
from .context import unittest

class TestCotangentWeights(unittest.TestCase):

    def test_single_triangle_2d(self):
        v = np.array([[0.0,0.0],[1.0,0.0],[0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        C = gpy.cotangent_weights(v,f)
        C_gt = np.array([[0., 0.5, 0.5]])
        self.assertTrue(np.isclose(C, C_gt).all())
    
    def test_single_triangle_3d(self):
        v = np.array([[0.0,0.0,0.0],[1.0,0.0,0.0],[0.0,0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        C = gpy.cotangent_weights(v,f)
        C_gt = np.array([[0., 0.5, 0.5]])
        self.assertTrue(np.isclose(C, C_gt).all())

    def test_bunny_oded(self):
        v,f = gpy.read_mesh("test/unit_tests_data/bunny_oded.obj")

        C = gpy.cotangent_weights(v,f)
        C_gt = np.array([[ 4.30199529e-01, -3.38453100e-02,  6.67484346e-01],
       [ 1.89900408e-01,  5.69392164e-01,  1.86847865e-01],
       [ 3.38413283e-02,  3.62513391e-01,  5.99796227e-01],
       [ 2.01175188e-01,  2.97002817e-01,  3.81892417e-01],
       [ 3.55568849e-01,  2.16436206e-01,  3.02518353e-01],
       [ 2.80525580e-01,  2.77047714e-01,  3.08983646e-01],
       [ 5.18205320e-01,  9.93066640e-02,  3.21514017e-01],
       [ 3.28213553e-01,  3.01726526e-01,  2.39656548e-01],
       [ 3.77641205e-01,  2.12684380e-01,  2.87437338e-01],
       [ 3.52632825e-01,  2.68736332e-01,  2.49826929e-01],
       [ 2.72715148e-01,  3.30518350e-01,  2.65009552e-01],
       [ 3.01753939e-01,  3.00238765e-01,  2.64790202e-01],
       [ 3.75862795e-01,  2.81379997e-01,  2.19461863e-01],
       [ 1.32550277e-01,  3.87190636e-01,  3.82263102e-01],
       [ 2.29470197e-01,  3.56330063e-01,  2.87184698e-01],
       [ 2.84310046e-01,  3.30772983e-01,  2.53555879e-01],
       [ 2.63041632e-01,  2.87278350e-01,  3.16968017e-01],
       [ 3.85300797e-01,  2.39909745e-01,  2.52015239e-01],
       [ 2.69236071e-01,  2.73548756e-01,  3.24900033e-01],
       [ 3.21060149e-01,  2.94927674e-01,  2.52132382e-01],
       [ 2.80945220e-01,  2.57861440e-01,  3.29533902e-01],
       [ 2.67484612e-01,  2.67096545e-01,  3.34010621e-01],
       [ 3.70429378e-01,  2.80086130e-01,  2.24818420e-01],
       [ 4.04114643e-01,  1.99343156e-01,  2.80786018e-01],
       [ 2.97841752e-01,  3.33261736e-01,  2.38852651e-01],
       [ 2.56968894e-01,  2.82412134e-01,  3.28949030e-01],
       [ 2.50809813e-01,  3.08475141e-01,  3.08664494e-01],
       [ 2.62453187e-01,  3.15708301e-01,  2.89091117e-01],
       [ 2.54171567e-01,  3.74409900e-01,  2.46325508e-01],
       [ 2.98604477e-01,  2.60503625e-01,  3.08012799e-01],
       [ 3.08288938e-01,  2.66582607e-01,  2.91918312e-01],
       [ 2.94443062e-01,  2.51995958e-01,  3.21722154e-01],
       [ 2.51337809e-01,  3.45804487e-01,  2.73111215e-01],
       [ 2.95207206e-01,  2.68485285e-01,  3.02897789e-01],
       [ 3.54040269e-01,  2.57987530e-01,  2.59239901e-01],
       [ 2.14169071e-01,  2.96087162e-01,  3.65673706e-01],
       [ 1.98447312e-01,  2.80898536e-01,  4.05253203e-01],
       [ 3.83453753e-01,  2.35993638e-01,  2.57499436e-01],
       [ 3.26366773e-01,  2.29454917e-01,  3.15053447e-01],
       [ 3.17177202e-01,  2.74301400e-01,  2.75576916e-01],
       [ 2.94751975e-01,  2.10193589e-01,  3.72406528e-01],
       [ 2.99586826e-01,  2.90487498e-01,  2.76191941e-01],
       [ 2.87547073e-01,  2.63608189e-01,  3.16063819e-01],
       [ 2.69383282e-01,  3.15875313e-01,  2.81770269e-01],
       [ 3.10431961e-01,  2.75412482e-01,  2.80796660e-01],
       [ 2.72150718e-01,  3.16227570e-01,  2.78627617e-01],
       [ 2.33848481e-01,  3.20408172e-01,  3.15869976e-01],
       [ 3.16354692e-01,  3.11886507e-01,  2.40883980e-01],
       [ 2.96943197e-01,  2.30507898e-01,  3.44206789e-01],
       [ 2.43292325e-01,  3.48398873e-01,  2.79262609e-01],
       [ 3.41486283e-01,  2.71084019e-01,  2.56996667e-01],
       [ 1.99140747e-01,  2.53997681e-01,  4.40083427e-01],
       [ 4.20339621e-01,  2.46886470e-01,  2.19151854e-01],
       [ 3.94675145e-01,  2.53521539e-01,  2.31320760e-01],
       [ 2.26657447e-01,  3.60677065e-01,  2.86463428e-01],
       [ 3.32385245e-01,  2.15011032e-01,  3.26150383e-01],
       [ 3.37735208e-01,  2.57156760e-01,  2.74249977e-01],
       [ 4.33302921e-01,  2.42150288e-01,  2.14782565e-01],
       [ 3.38232443e-01,  2.51398276e-01,  2.79783500e-01],
       [ 2.94369397e-01,  3.09236226e-01,  2.63367856e-01],
       [ 2.85733417e-01,  2.94015815e-01,  2.86313197e-01],
       [ 2.61014140e-01,  3.09300026e-01,  2.96798379e-01],
       [ 3.00048527e-01,  2.76812130e-01,  2.89399054e-01],
       [ 2.79633103e-01,  2.71047169e-01,  3.16347340e-01],
       [ 3.59769202e-01,  2.89501989e-01,  2.24630482e-01],
       [ 2.20646353e-01,  3.50080582e-01,  3.02694662e-01],
       [ 3.08494679e-01,  2.42270639e-01,  3.18213206e-01],
       [ 2.22840789e-01,  3.98495339e-01,  2.59439258e-01],
       [ 3.15805775e-01,  2.60894727e-01,  2.90632552e-01],
       [ 2.66656990e-01,  2.45138515e-01,  3.60753660e-01],
       [ 3.01529112e-01,  3.70233308e-01,  2.05971449e-01],
       [ 2.62392434e-01,  2.55440685e-01,  3.53346069e-01],
       [ 2.59359104e-01,  2.45130748e-01,  3.69527966e-01],
       [ 3.71167695e-01,  2.80100434e-01,  2.24232941e-01],
       [ 2.11737041e-01,  3.19851630e-01,  3.42888350e-01],
       [ 4.60629240e-01,  3.41817136e-01,  1.15333603e-01],
       [ 2.76021359e-01,  2.70426461e-01,  3.20902590e-01],
       [ 2.31170844e-01,  3.91796264e-01,  2.55917407e-01],
       [ 3.52750138e-01,  3.30738481e-01,  1.95075604e-01],
       [ 2.95398930e-01,  2.90747214e-01,  2.79987485e-01],
       [ 2.10172306e-01,  3.12562683e-01,  3.52583975e-01],
       [ 3.11464232e-01,  2.92148999e-01,  2.63423710e-01],
       [ 2.68770099e-01,  2.78050565e-01,  3.20522492e-01],
       [ 2.95858932e-01,  3.02897578e-01,  2.67863553e-01],
       [ 2.92839042e-01,  3.53426148e-01,  2.26691810e-01],
       [ 2.03224529e-01,  3.56666462e-01,  3.17055694e-01],
       [ 3.06593477e-01,  3.21891919e-01,  2.40753465e-01],
       [ 2.79677881e-01,  2.44004439e-01,  3.47075600e-01],
       [ 2.91473413e-01,  3.16149117e-01,  2.59784537e-01],
       [ 2.80608560e-01,  2.17042102e-01,  3.79977648e-01],
       [ 2.52515354e-01,  2.63127029e-01,  3.55976140e-01],
       [ 2.73990940e-01,  3.44052836e-01,  2.51976715e-01],
       [ 4.64355413e-01,  1.60749820e-01,  2.80520689e-01],
       [ 1.59088118e-01,  3.93797577e-01,  3.38861154e-01],
       [ 2.07106624e-01,  3.83395693e-01,  2.88900158e-01],
       [ 2.79975196e-01,  2.24937867e-01,  3.70406295e-01],
       [ 2.65082721e-01,  2.17969189e-01,  3.97928526e-01],
       [ 2.72037421e-01,  4.17015132e-01,  1.98179774e-01],
       [ 1.00328845e-01,  5.68766021e-01,  2.88354270e-01],
       [ 2.98338551e-01,  3.19673128e-01,  2.50204304e-01],
       [ 4.13904665e-01,  2.34906457e-01,  2.35463167e-01],
       [ 3.38614144e-01,  2.17461117e-01,  3.17159569e-01],
       [ 2.06815526e-01,  3.52600734e-01,  3.16538339e-01],
       [ 2.99202888e-01,  2.77869969e-01,  2.89149803e-01],
       [ 3.29127076e-01,  2.21092319e-01,  3.22112476e-01],
       [ 2.62997236e-01,  3.00649564e-01,  3.03257280e-01],
       [ 2.65424376e-01,  3.21029452e-01,  2.80995621e-01],
       [ 2.69401361e-01,  3.12244188e-01,  2.85192574e-01],
       [ 2.50999619e-01,  3.04551382e-01,  3.12406455e-01],
       [ 4.29230736e-01,  2.56797555e-01,  2.03744653e-01],
       [ 3.39089130e-01,  2.93573073e-01,  2.37808678e-01],
       [ 3.11008341e-01,  2.00821049e-01,  3.66416979e-01],
       [ 3.37960166e-01,  2.82125868e-01,  2.49405222e-01],
       [ 2.60546036e-01,  2.94409220e-01,  3.12264535e-01],
       [ 2.36597471e-01,  1.94208913e-01,  4.73648186e-01],
       [ 2.94635663e-01,  2.80552505e-01,  2.90929536e-01],
       [ 3.08658926e-01,  2.68500541e-01,  2.89564533e-01],
       [ 2.28561973e-01,  2.78375558e-01,  3.67646744e-01],
       [ 2.70183576e-01,  2.27962309e-01,  3.78219180e-01],
       [ 2.95488397e-01,  2.59063425e-01,  3.12774672e-01],
       [ 3.33437384e-01,  2.81349001e-01,  2.54052024e-01],
       [ 3.48474491e-01,  2.69843912e-01,  2.52242661e-01],
       [ 6.73173344e-01,  4.26145126e-01, -3.35385426e-02],
       [ 3.21339720e-01,  2.86744356e-01,  2.59598393e-01],
       [ 1.69319172e-01,  4.32112532e-01,  2.94023515e-01],
       [ 2.89313584e-01,  3.12990165e-01,  2.64729704e-01],
       [ 2.81099038e-01,  3.51088057e-01,  2.39342888e-01],
       [ 1.62530291e-01,  3.79544514e-01,  3.47392127e-01],
       [ 3.78036804e-01,  7.50953746e-02,  4.89065211e-01],
       [ 2.97048818e-01,  2.87220254e-01,  2.81859114e-01],
       [ 1.77855674e-01,  3.64386432e-01,  3.41529740e-01],
       [ 3.07213675e-01,  2.96934642e-01,  2.62812314e-01],
       [ 3.13563739e-01,  2.89791156e-01,  2.63745273e-01],
       [ 2.58544538e-01,  2.85459030e-01,  3.23887815e-01],
       [ 3.07652710e-01,  3.24067259e-01,  2.37921605e-01],
       [ 3.03494333e-01,  2.35060248e-01,  3.31740836e-01],
       [ 4.97209418e-01,  4.28006617e-01,  4.01971834e-02],
       [ 2.57361352e-01,  3.04197383e-01,  3.05776296e-01],
       [ 3.11948495e-01,  3.33159594e-01,  2.26429289e-01],
       [ 2.69138904e-01,  3.39261859e-01,  2.60833721e-01],
       [ 2.48463031e-01,  3.27253643e-01,  2.93007959e-01],
       [ 2.66856657e-01,  3.27691084e-01,  2.73406897e-01],
       [ 4.27434188e-01,  5.67367516e-01,  7.52685361e-03],
       [ 2.86147230e-01,  3.63569587e-01,  2.24660292e-01],
       [ 2.74293020e-01,  2.95816930e-01,  2.96187571e-01],
       [ 3.44116217e-01,  5.01783478e-01,  9.14152922e-02],
       [ 3.17358751e-01,  3.22423845e-01,  2.30822114e-01],
       [ 3.30379538e-01,  2.86745951e-01,  2.51593895e-01],
       [ 2.87669269e-01,  2.42227418e-01,  3.40290136e-01],
       [ 2.41498974e-01,  3.53201997e-01,  2.76949405e-01],
       [ 2.69904638e-01,  2.39719029e-01,  3.63599131e-01],
       [ 3.35495703e-02,  3.59063521e-01,  6.06076512e-01],
       [ 2.93119269e-01,  2.85374427e-01,  2.87559325e-01],
       [ 2.17810679e-01,  1.73899924e-01,  5.41529226e-01],
       [ 2.63314127e-01,  2.94386909e-01,  3.09276398e-01],
       [ 3.28126345e-01,  2.66695632e-01,  2.73174401e-01],
       [ 2.19268112e-01,  3.47554691e-01,  3.06607882e-01],
       [-1.24559390e-01,  6.39625370e-01,  6.40056533e-01],
       [ 1.91505544e-01,  4.40963973e-01,  2.61756417e-01],
       [ 6.74292142e-01,  1.52865171e-01,  1.77625482e-01],
       [ 3.13019710e-01,  2.67298296e-01,  2.86619342e-01],
       [ 3.62962711e-01,  2.82152250e-01,  2.28779774e-01],
       [ 2.82031194e-01,  2.64028194e-01,  3.21459198e-01],
       [ 2.55298816e-01,  3.19932550e-01,  2.92615474e-01],
       [ 3.12219175e-01,  2.61276299e-01,  2.93680660e-01],
       [ 2.59516308e-01,  2.93275434e-01,  3.14567366e-01],
       [ 3.28940749e-01,  2.76737392e-01,  2.62465795e-01],
       [ 4.25035821e-01,  2.27767994e-01,  2.34665362e-01],
       [ 2.83959167e-01,  2.87388590e-01,  2.94730089e-01],
       [ 3.77711725e-01,  3.18350732e-01,  1.86413151e-01],
       [ 2.96215714e-01,  1.80393842e-01,  4.12422509e-01],
       [ 2.35747891e-01,  3.35418767e-01,  2.99257196e-01],
       [ 3.27067656e-01,  2.58500878e-01,  2.82550571e-01],
       [ 2.06428346e-01,  2.45421622e-01,  4.41159753e-01],
       [ 2.91733024e-01,  2.69017437e-01,  3.05873541e-01],
       [ 2.75419639e-01,  2.90462002e-01,  3.00418052e-01],
       [ 3.08877504e-01,  2.72708957e-01,  2.85024410e-01],
       [ 4.04377621e-01,  1.04771208e+00, -1.19600958e-01],
       [ 3.32126319e-01,  2.74631074e-01,  2.61698983e-01],
       [ 2.17191478e-01,  3.73537835e-01,  2.85868268e-01],
       [ 3.01489792e-01,  2.97373937e-01,  2.67748380e-01],
       [ 3.17552747e-01,  2.86135974e-01,  2.63607270e-01],
       [ 3.27989475e-01,  2.77393231e-01,  2.62673410e-01],
       [ 2.60667592e-01,  2.57338311e-01,  3.53123857e-01],
       [ 3.08761254e-01,  2.56997750e-01,  3.01628523e-01],
       [ 2.98102604e-01,  2.87896670e-01,  2.80166308e-01],
       [ 3.35627759e-01,  2.21994870e-01,  3.14715276e-01],
       [ 2.95634365e-01,  2.74685823e-01,  2.95962574e-01],
       [ 2.64570892e-01,  2.72572904e-01,  3.31168571e-01],
       [ 2.99645232e-01,  3.22207147e-01,  2.46766225e-01],
       [ 3.26902337e-01,  1.96445540e-01,  3.54986620e-01],
       [ 3.84402258e-01,  2.83310172e-01,  2.11310923e-01],
       [ 4.15508456e-01,  2.08758541e-01,  2.61521210e-01],
       [ 2.80797973e-01,  2.81066985e-01,  3.04480566e-01],
       [ 2.33392819e-01,  3.71068691e-01,  2.70315363e-01],
       [ 3.22818951e-01,  2.83425601e-01,  2.61453640e-01],
       [ 3.18380485e-01,  2.87239251e-01,  2.61795676e-01],
       [ 2.65537696e-01,  2.69350777e-01,  3.33671829e-01],
       [ 3.01181238e-01,  2.99544746e-01,  2.65982739e-01],
       [ 2.77045671e-01,  2.68612620e-01,  3.21780205e-01],
       [ 2.68037868e-01,  3.66424757e-01,  2.39232831e-01],
       [ 3.34924184e-01,  1.91316691e-01,  3.53304774e-01],
       [ 2.63996654e-01,  2.84418011e-01,  3.18945878e-01],
       [ 3.51417130e-01,  1.76028775e-01,  3.56700983e-01],
       [ 7.53490810e-01,  1.89266774e-01,  1.13909691e-01],
       [ 2.19851189e-01,  2.77210551e-01,  3.80344966e-01],
       [ 2.61226512e-01,  3.02352332e-01,  3.03449210e-01],
       [ 3.28755191e-01,  2.63719349e-01,  2.75625168e-01],
       [ 3.87025236e-01,  3.29508276e-01,  1.70922895e-01],
       [ 3.58485491e-01,  2.65851822e-01,  2.47776283e-01],
       [ 3.20000303e-01,  2.81393183e-01,  2.65972445e-01],
       [ 3.56358579e-01,  3.19406568e-01,  2.01514875e-01],
       [ 1.21785040e-01,  1.26268002e-01,  9.45855952e-01],
       [ 2.76191750e-01,  2.58604716e-01,  3.33913035e-01],
       [ 3.05448028e-01,  3.07876728e-01,  2.54285611e-01],
       [ 4.04983225e-01,  1.93254311e-01,  2.87068656e-01],
       [ 2.87915654e-01,  2.57229297e-01,  3.22739224e-01],
       [ 2.54979175e-01,  3.29814124e-01,  2.83697277e-01],
       [ 2.57082120e-01,  3.37770602e-01,  2.74295152e-01],
       [ 3.11633860e-01,  2.86388509e-01,  2.68805402e-01],
       [ 2.67867565e-01,  3.09352345e-01,  2.89550893e-01],
       [ 2.78045634e-01,  2.90604123e-01,  2.97544826e-01],
       [ 2.65717222e-01,  3.11045150e-01,  2.90154065e-01],
       [ 2.63318050e-01,  3.06349243e-01,  2.97248441e-01],
       [ 2.72600401e-01,  3.11015448e-01,  2.83092147e-01],
       [ 2.42698107e-01,  3.10554937e-01,  3.15640206e-01],
       [ 2.34908645e-01,  2.74156509e-01,  3.64586468e-01],
       [ 2.15943347e-01,  3.90565030e-01,  2.73137333e-01],
       [ 2.25232418e-01,  3.73121732e-01,  2.77362312e-01],
       [ 2.48783849e-01,  2.69367748e-01,  3.53150808e-01],
       [ 2.82458573e-01,  3.18805828e-01,  2.66023667e-01],
       [ 3.67555741e-01,  2.59783263e-01,  2.46302509e-01],
       [ 2.89742997e-01,  2.55136752e-01,  3.23146370e-01],
       [ 2.85766804e-01,  2.53660234e-01,  3.29075691e-01],
       [ 3.18109100e-01,  2.92955505e-01,  2.56614745e-01],
       [ 2.60217190e-01,  3.02002830e-01,  3.04887172e-01],
       [ 2.87819443e-01,  3.07317825e-01,  2.71446544e-01],
       [ 3.61714401e-01,  2.29441662e-01,  2.82510250e-01],
       [ 2.28865054e-01,  3.46988210e-01,  2.96232625e-01],
       [ 3.17873825e-01,  2.78998488e-01,  2.70264979e-01],
       [ 2.96903484e-01,  2.62973634e-01,  3.07071331e-01],
       [ 1.69722475e-01,  5.68580238e-01,  2.07908155e-01],
       [ 1.03093115e+00, -3.88307171e-02,  2.92341163e-01],
       [ 2.06119921e-01,  3.19127524e-01,  3.50732709e-01],
       [ 3.54412631e-01,  2.67839922e-01,  2.49214163e-01],
       [ 3.02400261e-01,  2.91711843e-01,  2.72316052e-01],
       [ 2.52567525e-01,  3.01315466e-01,  3.13960713e-01],
       [ 3.16922664e-01,  2.66016740e-01,  2.84237548e-01],
       [ 2.83229348e-01,  2.77032359e-01,  3.06171747e-01],
       [ 2.84451036e-01,  2.24157298e-01,  3.66172184e-01],
       [ 2.32158983e-01,  2.63732387e-01,  3.80672399e-01],
       [ 2.39195885e-01,  3.38647373e-01,  2.92461563e-01],
       [ 3.11263468e-01,  2.70906908e-01,  2.84584347e-01],
       [ 3.00491419e-01,  3.66871092e-01,  2.09418993e-01],
       [ 2.69414693e-01,  3.02482050e-01,  2.94645657e-01],
       [ 2.67450491e-01,  3.10568572e-01,  2.88811032e-01],
       [ 2.70131684e-01,  3.12600454e-01,  2.84104311e-01],
       [ 2.93849113e-01,  2.67776683e-01,  3.05032390e-01],
       [ 2.89535697e-01,  2.93231445e-01,  2.83301883e-01],
       [ 2.88307541e-01,  2.81871857e-01,  2.95931804e-01],
       [ 2.64850271e-01,  2.95684527e-01,  3.06292979e-01],
       [ 3.15136319e-01,  2.66487963e-01,  2.85441942e-01],
       [ 2.93316655e-01,  3.08731464e-01,  2.64835840e-01],
       [ 2.70476363e-01,  3.10547746e-01,  2.85709616e-01],
       [ 2.72738844e-01,  3.26773678e-01,  2.68344895e-01],
       [ 3.18997672e-01,  2.58480911e-01,  2.90132303e-01],
       [ 2.98075693e-01,  2.29425794e-01,  3.44290495e-01],
       [ 2.82185856e-01,  2.30694861e-01,  3.60514965e-01],
       [ 2.53171579e-01,  3.44005887e-01,  2.72795770e-01],
       [ 2.78125832e-01,  2.75255817e-01,  3.13426091e-01],
       [ 3.64987489e-01,  2.53990253e-01,  2.54123411e-01],
       [ 2.96091513e-01,  3.10016062e-01,  2.61021115e-01],
       [ 3.20031131e-01,  2.82207180e-01,  2.65152372e-01],
       [ 2.92042156e-01,  2.55321661e-01,  3.20509516e-01],
       [ 2.37293788e-01,  3.39725729e-01,  2.93551933e-01],
       [ 2.85252485e-01,  2.48011309e-01,  3.36145375e-01],
       [ 2.23389931e-01,  2.98093510e-01,  3.51706108e-01],
       [ 2.81386453e-01,  3.70751769e-01,  2.23381286e-01],
       [ 3.63552745e-01,  3.18278962e-01,  1.96953014e-01],
       [ 3.25171169e-01,  2.39743118e-01,  3.04546113e-01],
       [ 2.82713322e-01,  3.10955750e-01,  2.73028653e-01],
       [ 2.36589941e-01,  2.85230416e-01,  3.49770473e-01],
       [ 7.47295629e-01,  4.54491202e-01, -7.45883434e-02],
       [ 2.40567935e-01,  3.19814524e-01,  3.08829939e-01],
       [ 2.94323473e-01,  2.99023363e-01,  2.73011323e-01],
       [ 3.14145438e-01,  2.91833742e-01,  2.61266008e-01],
       [ 2.84620591e-01,  2.37402983e-01,  3.49467403e-01],
       [ 2.49522344e-01,  3.62580258e-01,  2.60623176e-01],
       [ 3.24328182e-01,  2.70240015e-01,  2.73061270e-01],
       [ 2.58103522e-01,  2.28551294e-01,  3.92496077e-01],
       [ 2.39526603e-01,  2.04785138e-01,  4.52269191e-01],
       [ 3.29243211e-01,  3.07607411e-01,  2.33527837e-01],
       [ 2.73185776e-01,  2.80462031e-01,  3.13162556e-01],
       [ 3.01609604e-01,  2.55788061e-01,  3.10105110e-01],
       [ 2.34456149e-01,  2.64321312e-01,  3.76978228e-01],
       [ 3.15169148e-01,  2.91801475e-01,  2.60363800e-01],
       [ 2.92970895e-01,  2.54220804e-01,  3.20766386e-01],
       [ 2.84140322e-01,  2.68825082e-01,  3.13972544e-01],
       [ 2.80192219e-01,  2.86274468e-01,  2.99731874e-01],
       [ 2.78461877e-01,  3.05093934e-01,  2.82822940e-01],
       [ 2.90073271e-01,  2.96972597e-01,  2.79119906e-01],
       [ 3.14466060e-01,  2.88454155e-01,  2.64199067e-01],
       [ 2.86171632e-01,  3.05413194e-01,  2.74853919e-01],
       [ 2.58418671e-01,  3.10659104e-01,  2.98236716e-01],
       [ 2.88666940e-01,  2.67702344e-01,  3.10447016e-01],
       [ 2.84130603e-01,  3.13950559e-01,  2.68854879e-01],
       [ 3.26635926e-01,  2.77573435e-01,  2.63707507e-01],
       [ 3.37290157e-01,  3.15569536e-01,  2.19896409e-01],
       [ 3.21605235e-01,  2.98971667e-01,  2.47913105e-01],
       [ 2.80681877e-01,  2.43683169e-01,  3.46328483e-01],
       [ 2.74522660e-01,  3.15704159e-01,  2.76727606e-01],
       [ 2.81190888e-01,  3.14193028e-01,  2.71508482e-01],
       [ 2.60206247e-01,  3.52551750e-01,  2.58281137e-01],
       [ 3.03909460e-01,  2.92873359e-01,  2.69768181e-01],
       [ 2.48903496e-01,  3.37162415e-01,  2.83379554e-01],
       [ 3.38125119e-01,  2.85571206e-01,  2.46019089e-01],
       [ 3.09947323e-01,  2.86216103e-01,  2.70543408e-01],
       [ 2.44098333e-01,  3.38419063e-01,  2.87360466e-01],
       [ 3.41956145e-01,  2.18991165e-01,  3.12176602e-01],
       [ 3.96313433e-01,  2.37128761e-01,  2.46309273e-01],
       [ 2.63655078e-01,  3.20534587e-01,  2.83280309e-01],
       [ 2.67615110e-01,  3.18667691e-01,  2.80956068e-01],
       [ 2.90550186e-01,  2.75445648e-01,  3.00301178e-01],
       [ 2.94909921e-01,  2.82245409e-01,  2.88939597e-01],
       [ 3.99961275e-01,  2.11959209e-01,  2.70009795e-01],
       [ 4.21867591e-01,  1.18342289e-01,  3.70365724e-01],
       [ 2.52106287e-01,  3.09705509e-01,  3.06012254e-01],
       [ 3.14193115e-01,  2.44109317e-01,  3.10409777e-01],
       [ 2.81373633e-01,  2.93670278e-01,  2.91054168e-01],
       [ 2.78173973e-01,  3.16348444e-01,  2.72487785e-01],
       [ 2.70394442e-01,  2.89170987e-01,  3.07041435e-01],
       [ 3.12488618e-01,  2.71809438e-01,  2.82497337e-01],
       [ 3.86003450e-01,  3.69985840e-01,  1.41780037e-01],
       [ 2.06410349e-01,  3.51080293e-01,  3.18450895e-01],
       [ 2.01715093e-01,  3.99477134e-01,  2.81805761e-01],
       [ 3.00017480e-01,  2.49343285e-01,  3.18902745e-01],
       [ 2.77615226e-01,  3.10932151e-01,  2.78109302e-01],
       [ 2.98357099e-01,  3.04634790e-01,  2.63867642e-01],
       [ 3.48783446e-01,  2.49867845e-01,  2.72028532e-01],
       [ 2.66980062e-01,  3.01808745e-01,  2.97866414e-01],
       [ 2.62200093e-01,  3.08832675e-01,  2.95997101e-01],
       [ 2.98712572e-01,  2.96480974e-01,  2.71235141e-01],
       [ 2.78918959e-01,  2.82718282e-01,  3.04724293e-01],
       [ 2.92180414e-01,  2.74769762e-01,  2.99351979e-01],
       [ 2.92591618e-01,  2.67843518e-01,  3.06246380e-01],
       [ 2.77132803e-01,  3.03874640e-01,  2.85342935e-01],
       [ 2.66517877e-01,  3.14763078e-01,  2.85765448e-01],
       [ 2.65897445e-01,  3.02350326e-01,  2.98471599e-01],
       [ 2.81587685e-01,  2.59288339e-01,  3.27223965e-01],
       [ 2.96884686e-01,  3.02985468e-01,  2.66804830e-01],
       [ 2.81771860e-01,  2.68555495e-01,  3.16772585e-01],
       [ 2.80970392e-01,  2.78880684e-01,  3.06586505e-01],
       [ 3.19619716e-01,  2.33223411e-01,  3.17372490e-01],
       [ 2.51435237e-01,  2.94610984e-01,  3.22178619e-01],
       [ 2.53878425e-01,  3.30653698e-01,  2.84080470e-01],
       [ 2.92525879e-01,  2.52743041e-01,  3.22897773e-01],
       [ 2.76660968e-01,  2.80539401e-01,  3.09378291e-01],
       [ 2.54928001e-01,  2.67341526e-01,  3.48186424e-01],
       [ 3.04072160e-01,  2.70044477e-01,  2.92426628e-01],
       [ 2.77764739e-01,  2.81988390e-01,  3.06695147e-01],
       [ 3.00110685e-01,  2.65595114e-01,  3.01025849e-01],
       [ 2.55675863e-01,  3.10121857e-01,  3.01714412e-01],
       [ 2.39879980e-01,  3.12519145e-01,  3.16859506e-01],
       [ 3.25571205e-01,  2.95981486e-01,  2.47182504e-01],
       [ 2.94871542e-01,  3.93736577e-01,  1.94447734e-01],
       [ 3.91781125e-01,  2.37507785e-01,  2.49406799e-01],
       [ 3.40301362e-01,  2.91040082e-01,  2.39107800e-01],
       [ 2.20621863e-01,  3.29999608e-01,  3.21808866e-01],
       [ 2.72232812e-01,  2.75685865e-01,  3.19297860e-01],
       [ 2.86802454e-01,  2.66091509e-01,  3.14136731e-01],
       [ 2.05299854e-01,  3.34422119e-01,  3.35993709e-01],
       [ 4.00034492e-01,  2.80821327e-01,  2.02189332e-01],
       [ 2.31995093e-01,  3.61910412e-01,  2.79570670e-01],
       [ 2.67302066e-01,  3.44457707e-01,  2.58149931e-01],
       [ 2.79358281e-01,  2.70520411e-01,  3.17211570e-01],
       [ 3.28880052e-01,  2.70378150e-01,  2.68795687e-01],
       [ 2.37294403e-01,  2.47812345e-01,  3.94130814e-01],
       [ 2.67171824e-01,  2.86082059e-01,  3.13720228e-01],
       [ 2.21261398e-01,  3.11531150e-01,  3.39851191e-01],
       [ 3.71523212e-01,  2.75259710e-01,  2.28414548e-01],
       [ 3.54184069e-01,  3.03553058e-01,  2.16631139e-01],
       [ 2.76879617e-01,  3.09295713e-01,  2.80397883e-01],
       [ 3.02009907e-01,  3.19030489e-01,  2.47406824e-01],
       [ 3.08310634e-01,  2.60167769e-01,  2.98670115e-01],
       [ 3.00240494e-01,  2.57929833e-01,  3.09151187e-01],
       [ 2.69223745e-01,  2.60660941e-01,  3.39364185e-01],
       [ 3.00080620e-01,  2.56914096e-01,  3.10424953e-01],
       [ 3.10190032e-01,  2.94116917e-01,  2.62727186e-01],
       [ 2.65333084e-01,  3.02770740e-01,  2.98651230e-01],
       [ 3.08416861e-01,  2.85965289e-01,  2.72221303e-01],
       [ 3.11811592e-01,  2.89758654e-01,  2.65388613e-01],
       [ 2.97872012e-01,  2.61213356e-01,  3.07988479e-01],
       [ 3.14950599e-01,  2.86234677e-01,  2.65891770e-01],
       [ 2.94627514e-01,  2.58928482e-01,  3.13812055e-01],
       [ 3.06805914e-01,  2.99255568e-01,  2.61007550e-01],
       [ 2.82660002e-01,  2.75547966e-01,  3.08332417e-01],
       [ 3.04980106e-01,  2.64541802e-01,  2.97302018e-01],
       [ 2.76972754e-01,  3.11777413e-01,  2.77955167e-01],
       [ 2.66952989e-01,  3.10196015e-01,  2.89686451e-01],
       [ 3.23621392e-01,  2.45292799e-01,  2.99901119e-01],
       [ 2.90587218e-01,  3.37317402e-01,  2.42042930e-01],
       [ 2.88946779e-01,  2.78739876e-01,  2.98508004e-01],
       [ 3.16338423e-01,  2.97271139e-01,  2.54170936e-01],
       [ 3.05135690e-01,  2.92934705e-01,  2.68555621e-01],
       [ 3.10998677e-01,  2.71988295e-01,  2.83731898e-01],
       [ 2.73974455e-01,  2.33917271e-01,  3.66047788e-01],
       [ 2.95279060e-01,  2.77358872e-01,  2.93556755e-01],
       [ 2.82074535e-01,  3.25811930e-01,  2.60076117e-01],
       [ 2.65627327e-01,  2.87837381e-01,  3.13556624e-01],
       [ 2.04201283e-01,  3.46769992e-01,  3.25224074e-01],
       [ 2.88919532e-01,  2.68083691e-01,  3.09774483e-01],
       [ 3.39847007e-01,  2.64534324e-01,  2.64897000e-01],
       [ 2.19536302e-01,  2.34583125e-01,  4.37110761e-01],
       [ 2.33963753e-01,  3.85997792e-01,  2.57581312e-01],
       [ 2.93526131e-01,  3.12630278e-01,  2.61046228e-01],
       [ 2.60669564e-01,  2.93651035e-01,  3.12912984e-01],
       [ 2.63138399e-01,  2.94758399e-01,  3.09085386e-01],
       [ 3.17326261e-01,  3.01116099e-01,  2.49737023e-01],
       [ 2.44412596e-01,  3.15790794e-01,  3.08489301e-01],
       [ 1.48192701e-01,  4.18349038e-01,  3.31844439e-01],
       [ 3.59344505e-01,  3.42274496e-01,  1.81017819e-01],
       [ 3.60213261e-01,  2.41549553e-01,  2.70855301e-01],
       [ 2.70598397e-01,  3.72240285e-01,  2.32208142e-01],
       [ 2.59637089e-01,  3.39820333e-01,  2.69860764e-01],
       [ 2.89974771e-01,  3.19208869e-01,  2.58440100e-01],
       [ 2.82398538e-01,  3.19567991e-01,  2.65387624e-01],
       [ 2.93244250e-01,  3.04920080e-01,  2.68461244e-01],
       [ 3.41383025e-01,  1.79850670e-01,  3.61837763e-01],
       [ 3.49781110e-01,  1.95671997e-01,  3.32856536e-01],
       [ 3.71719063e-01,  2.02302518e-01,  3.04518686e-01],
       [ 2.32559676e-01,  3.49517701e-01,  2.89851974e-01],
       [ 3.09664201e-01,  2.84668090e-01,  2.72319855e-01],
       [ 2.61873825e-01,  2.97458914e-01,  3.07693943e-01],
       [ 2.87606843e-01,  3.14106693e-01,  2.65343484e-01],
       [ 2.60374706e-01,  2.96947681e-01,  3.09842811e-01],
       [ 3.27312870e-01,  2.76071969e-01,  2.64570935e-01],
       [ 2.98184634e-01,  3.06675349e-01,  2.62133598e-01],
       [ 3.07324235e-01,  2.61395458e-01,  2.98331222e-01],
       [ 3.16098864e-01,  2.60273342e-01,  2.91006212e-01],
       [ 2.71506932e-01,  2.99897132e-01,  2.95020390e-01],
       [ 2.97174089e-01,  2.54279414e-01,  3.16318140e-01],
       [ 3.12785634e-01,  3.06539119e-01,  2.48850003e-01],
       [ 2.60307618e-01,  3.09603823e-01,  2.97252791e-01],
       [ 2.98863073e-01,  2.66327182e-01,  3.01499608e-01],
       [ 2.65784304e-01,  3.06446622e-01,  2.94551185e-01],
       [ 2.68596956e-01,  2.95854482e-01,  3.02124108e-01],
       [ 3.11808682e-01,  3.16978802e-01,  2.40404368e-01],
       [ 2.87082886e-01,  3.17562196e-01,  2.62688531e-01],
       [ 2.82363121e-01,  2.76401714e-01,  3.07740106e-01],
       [ 2.86631813e-01,  2.72447499e-01,  3.07483885e-01],
       [ 3.16981503e-01,  3.02295967e-01,  2.48963958e-01],
       [ 3.18077728e-01,  3.23491270e-01,  2.29288872e-01],
       [ 2.56819641e-01,  3.28511171e-01,  2.82971399e-01],
       [ 3.26304364e-01,  2.75321951e-01,  2.66213831e-01],
       [ 2.98910843e-01,  2.65050543e-01,  3.02810659e-01],
       [ 2.91050692e-01,  3.03796701e-01,  2.71632291e-01],
       [ 2.12270346e-01,  3.35199749e-01,  3.26679091e-01],
       [ 2.77082495e-01,  3.29757355e-01,  2.61403415e-01],
       [ 2.63687801e-01,  3.07284090e-01,  2.95939146e-01],
       [ 3.00133091e-01,  2.46618134e-01,  3.21868025e-01],
       [ 2.81778484e-01,  2.17868317e-01,  3.77485447e-01],
       [ 4.23140387e-01,  2.48744336e-01,  2.15433125e-01],
       [ 3.00069271e-01,  2.44251712e-01,  3.24638535e-01],
       [ 2.81621006e-01,  2.57636116e-01,  3.29053157e-01],
       [ 2.30805400e-01,  3.86962100e-01,  2.60109277e-01],
       [ 1.84656835e-01,  2.76992623e-01,  4.30741151e-01],
       [ 2.85448086e-01,  3.11915681e-01,  2.69457698e-01],
       [ 3.08170121e-01,  2.99472573e-01,  2.59546445e-01],
       [ 3.49913865e-01,  2.51659122e-01,  2.69195904e-01],
       [ 2.66738520e-01,  3.26185563e-01,  2.74898170e-01],
       [ 3.28541781e-01,  3.09533739e-01,  2.32425832e-01],
       [ 4.43425331e-01,  1.97693935e-01,  2.53209520e-01],
       [ 3.64298541e-01,  1.21896327e-01,  4.22862024e-01],
       [ 3.64839232e-01,  2.09724317e-01,  3.01941085e-01],
       [ 1.92634707e-01,  3.32658783e-01,  3.53932377e-01],
       [ 2.94194641e-01,  2.61081175e-01,  3.11901423e-01],
       [ 2.70209865e-01,  2.78961871e-01,  3.17972938e-01],
       [ 2.01472308e-01,  3.93594450e-01,  2.86861290e-01],
       [ 2.69055378e-01,  2.86663022e-01,  3.11078007e-01],
       [ 3.31406447e-01,  2.46487185e-01,  2.91251796e-01],
       [ 2.39362600e-01,  2.99649665e-01,  3.30744008e-01],
       [ 2.51431042e-01,  2.97167680e-01,  3.19510078e-01],
       [ 3.19634072e-01,  3.02024691e-01,  2.46859897e-01],
       [ 2.87074005e-01,  2.73351629e-01,  3.06067108e-01],
       [ 2.86455964e-01,  3.07301052e-01,  2.72791355e-01],
       [ 2.87923946e-01,  3.06145012e-01,  2.72449044e-01],
       [ 3.08108958e-01,  2.60846465e-01,  2.98144390e-01],
       [ 3.02016932e-01,  3.11036741e-01,  2.54564395e-01],
       [ 2.87809899e-01,  2.68773644e-01,  3.10185752e-01],
       [ 2.56344171e-01,  3.06125539e-01,  3.04952426e-01],
       [ 3.04683934e-01,  3.08069695e-01,  2.54810589e-01],
       [ 3.14473075e-01,  2.66276974e-01,  2.86290223e-01],
       [ 2.63899902e-01,  2.97760462e-01,  3.05204095e-01],
       [ 2.87507325e-01,  2.46387135e-01,  3.35575488e-01],
       [ 2.47432964e-01,  3.04401046e-01,  3.16546540e-01],
       [ 3.06104581e-01,  2.98078050e-01,  2.62763170e-01],
       [ 2.86464856e-01,  2.63725017e-01,  3.17076086e-01],
       [ 3.06948955e-01,  2.58297285e-01,  3.02020440e-01],
       [ 2.95545610e-01,  2.96168016e-01,  2.74573435e-01],
       [ 2.55847967e-01,  3.03155785e-01,  3.08474153e-01],
       [ 2.74071631e-01,  2.78856396e-01,  3.13916756e-01],
       [ 3.12975791e-01,  2.75225332e-01,  2.78580111e-01],
       [ 3.04563144e-01,  2.95161356e-01,  2.66963796e-01],
       [ 2.57375494e-01,  2.92857856e-01,  3.17366376e-01],
       [ 2.57986796e-01,  3.00997186e-01,  3.08321358e-01],
       [ 2.96425670e-01,  2.58408641e-01,  3.12527618e-01],
       [ 2.50350841e-01,  3.09976948e-01,  3.07671713e-01],
       [ 2.80252356e-01,  2.64108336e-01,  3.23283843e-01],
       [ 2.68691650e-01,  3.00657322e-01,  2.97209438e-01],
       [ 3.07883333e-01,  2.58202368e-01,  3.01197847e-01],
       [ 2.92655638e-01,  3.76976303e-01,  2.08585868e-01],
       [ 3.17347010e-01,  2.64517565e-01,  2.85386237e-01],
       [ 3.01723029e-01,  2.48681896e-01,  3.17887681e-01],
       [ 2.80941004e-01,  2.88510990e-01,  2.96680729e-01],
       [ 2.65911219e-01,  2.46145903e-01,  3.60403235e-01],
       [ 2.00236276e-01,  4.35408029e-01,  2.56142494e-01],
       [ 2.51412218e-01,  3.20384267e-01,  2.96349287e-01],
       [ 2.60217128e-01,  3.32628815e-01,  2.75694364e-01],
       [ 4.89590997e-02,  2.47507419e-01,  8.02391651e-01],
       [ 1.53797788e-01,  3.18337124e-01,  4.25811456e-01],
       [ 3.05084468e-01,  2.38518630e-01,  3.26031384e-01],
       [ 2.61261107e-01,  3.06193522e-01,  2.99589663e-01],
       [ 3.46232812e-01,  2.63555521e-01,  2.60333663e-01],
       [ 3.37419800e-01,  3.31733602e-01,  2.06330139e-01],
       [ 3.25500218e-01,  2.04765736e-01,  3.45767453e-01],
       [ 3.37124084e-01,  3.26765052e-01,  2.10637023e-01],
       [ 2.28029663e-01,  3.07115588e-01,  3.36298483e-01],
       [ 4.39084930e-01,  9.23535351e-02,  3.94117039e-01],
       [ 3.07150962e-01,  3.70600971e-01,  2.00913562e-01],
       [ 3.05476935e-01,  2.30175214e-01,  3.35454233e-01],
       [ 3.05110781e-01,  3.07556480e-01,  2.54887460e-01],
       [ 2.97619594e-01,  3.11484334e-01,  2.58241905e-01],
       [ 2.21189263e-01,  3.31058654e-01,  3.20097867e-01],
       [ 3.02623511e-01,  3.15820380e-01,  2.49699820e-01],
       [ 2.37405173e-01,  3.33578917e-01,  2.99144305e-01],
       [ 2.53328068e-01,  3.37185191e-01,  2.78709284e-01],
       [ 3.11498924e-01,  2.30611299e-01,  3.28650560e-01],
       [ 2.27378953e-01,  3.29710292e-01,  3.14188114e-01],
       [ 3.17083349e-01,  2.97691786e-01,  2.53111883e-01],
       [ 2.73955825e-01,  3.05585414e-01,  2.86921939e-01],
       [ 3.07158032e-01,  2.74285351e-01,  2.85068256e-01],
       [ 2.83077110e-01,  2.88008764e-01,  2.95001713e-01],
       [ 2.90304196e-01,  3.13926975e-01,  2.62922023e-01],
       [ 3.15891306e-01,  2.57638679e-01,  2.93993663e-01],
       [ 3.04882157e-01,  2.88090212e-01,  2.73480592e-01],
       [ 2.96166230e-01,  3.17689173e-01,  2.53986843e-01],
       [ 3.40780592e-01,  2.44946717e-01,  2.84307919e-01],
       [ 2.68947367e-01,  3.46881478e-01,  2.54465410e-01],
       [ 2.99809208e-01,  2.68896011e-01,  2.97838835e-01],
       [ 2.11876967e-01,  3.36031150e-01,  3.26337088e-01],
       [ 2.41473862e-01,  3.16397974e-01,  3.11179286e-01],
       [ 3.18876938e-01,  2.51526306e-01,  2.97673731e-01],
       [ 3.11424634e-01,  2.94683263e-01,  2.61056445e-01],
       [ 2.62449862e-01,  2.98593953e-01,  3.05919170e-01],
       [ 2.60587867e-01,  2.98117053e-01,  3.08417389e-01],
       [ 3.07908577e-01,  2.78161811e-01,  2.80429785e-01],
       [ 3.13378440e-01,  2.55281734e-01,  2.98948680e-01],
       [ 3.14373720e-01,  2.86527760e-01,  2.66138806e-01],
       [ 3.05191900e-01,  2.78168561e-01,  2.83024338e-01],
       [ 2.88758525e-01,  2.70445560e-01,  3.07412881e-01],
       [ 3.21397217e-01,  3.14599661e-01,  2.34102634e-01],
       [ 2.82902964e-01,  2.68644609e-01,  3.15475314e-01],
       [ 3.09428346e-01,  2.99500373e-01,  2.58365373e-01],
       [ 2.92075324e-01,  2.66012881e-01,  3.08740088e-01],
       [ 2.61623798e-01,  3.52265090e-01,  2.57113416e-01],
       [ 2.92186220e-01,  2.78404636e-01,  2.95577821e-01],
       [ 3.16389885e-01,  2.88423538e-01,  2.62470547e-01],
       [ 3.27133152e-01,  4.03120415e-01,  1.61760180e-01],
       [ 3.19363734e-01,  2.48315065e-01,  3.00693234e-01],
       [ 2.99844850e-01,  3.37325564e-01,  2.33618297e-01],
       [ 2.79831785e-01,  3.30747691e-01,  2.57863701e-01],
       [ 2.66540126e-01,  2.59885424e-01,  3.43315606e-01],
       [-6.91454600e-02,  5.56891733e-01,  5.91509461e-01],
       [ 4.41806446e-01,  1.00583770e-01,  3.78991796e-01],
       [ 3.12637822e-01,  3.19132986e-01,  2.37787116e-01],
       [ 2.87021782e-01,  2.82911168e-01,  2.96172265e-01],
       [ 8.11496348e-01,  1.47130553e-01,  1.36240798e-01],
       [ 9.00309199e-01,  3.45449366e-01, -4.89751733e-02],
       [ 3.63923072e-01,  3.09461167e-01,  2.04014192e-01],
       [ 2.98087846e-01,  2.79563759e-01,  2.88522424e-01],
       [ 2.97843093e-01,  2.88071810e-01,  2.80245135e-01],
       [ 2.69490536e-01,  4.24028059e-01,  1.95709895e-01],
       [ 2.66227844e-01,  2.81285761e-01,  3.19834789e-01],
       [ 3.27966216e-01,  3.21819279e-01,  2.22310516e-01],
       [ 2.17546566e-01,  3.34856100e-01,  3.20695783e-01],
       [ 3.44973955e-01,  2.96700835e-01,  2.30094655e-01],
       [ 2.73707849e-01,  3.32661910e-01,  2.62130197e-01],
       [ 2.96288875e-01,  3.06589669e-01,  2.64001586e-01],
       [ 2.53523584e-01,  2.66965064e-01,  3.50282491e-01],
       [ 2.94648556e-01,  2.99067011e-01,  2.72656381e-01],
       [ 4.34189431e-01,  2.71721381e-01,  1.87022844e-01],
       [ 2.89726537e-01,  3.13566401e-01,  2.63804666e-01],
       [ 3.11905229e-01,  3.01285007e-01,  2.54452243e-01],
       [ 3.06819876e-01,  2.76615470e-01,  2.83028577e-01],
       [ 3.03395495e-01,  2.55975006e-01,  3.08093359e-01],
       [ 2.66886867e-01,  2.80324915e-01,  3.20141062e-01],
       [ 2.61734650e-01,  2.99741705e-01,  3.05528823e-01],
       [ 3.02342567e-01,  2.85244099e-01,  2.78696873e-01],
       [ 2.72966464e-01,  2.87440038e-01,  3.06096573e-01],
       [ 2.63107884e-01,  2.95767383e-01,  3.08085328e-01],
       [ 2.90003501e-01,  2.85909013e-01,  2.90122859e-01],
       [ 2.57013350e-01,  2.94870511e-01,  3.15672108e-01],
       [ 3.04419462e-01,  3.13635524e-01,  2.50015364e-01],
       [ 2.74327396e-01,  2.93512824e-01,  2.98466690e-01],
       [ 2.49813231e-01,  2.99550644e-01,  3.18856579e-01],
       [ 2.81443772e-01,  3.35690085e-01,  2.52007105e-01],
       [ 2.88549537e-01,  2.39351266e-01,  3.42744891e-01],
       [ 2.89402731e-01,  3.15150239e-01,  2.62664593e-01],
       [ 3.12430567e-01,  2.08920328e-01,  3.54323557e-01],
       [ 2.39831263e-01,  2.42055702e-01,  3.98324688e-01],
       [ 2.68787528e-01,  3.48632412e-01,  2.53137202e-01],
       [ 2.77637310e-01,  2.84936160e-01,  3.03766707e-01],
       [ 3.05332626e-01,  2.89052915e-01,  2.72117511e-01],
       [ 3.13032542e-01,  2.43043751e-01,  3.12761754e-01],
       [ 2.71752356e-01,  2.94343762e-01,  3.00322479e-01],
       [ 3.12599121e-01,  2.46028539e-01,  3.09851637e-01],
       [ 2.87060558e-01,  3.01553351e-01,  2.77662359e-01],
       [ 2.73460607e-01,  2.88674335e-01,  3.04302274e-01],
       [ 2.79788189e-01,  2.95440328e-01,  2.90909232e-01],
       [ 3.06555478e-01,  2.42505205e-01,  3.19925842e-01],
       [ 2.58253044e-01,  3.20273554e-01,  2.89162815e-01],
       [ 3.65409121e-01,  3.05258867e-01,  2.06444363e-01],
       [ 3.02602265e-01,  2.50986355e-01,  3.14404874e-01],
       [ 2.66736551e-01,  2.97834952e-01,  3.02099080e-01],
       [ 2.52339342e-01,  1.97502153e-01,  4.44962145e-01],
       [ 3.25686269e-01,  2.80511514e-01,  2.61698832e-01],
       [ 3.01030480e-01,  2.74925756e-01,  2.90367492e-01],
       [ 2.02011102e-01,  3.31519606e-01,  3.43053092e-01],
       [ 2.91685453e-01,  2.95515409e-01,  2.78954723e-01],
       [ 2.84499713e-01,  2.60002383e-01,  3.23285067e-01],
       [ 2.80175013e-01,  2.77307458e-01,  3.09077663e-01],
       [ 2.66924586e-01,  3.19316648e-01,  2.81055863e-01],
       [ 2.38965844e-01,  4.21057494e-01,  2.26327817e-01],
       [ 3.63508009e-01,  2.02439322e-01,  3.11710428e-01],
       [ 2.98996280e-01,  3.11349648e-01,  2.57079807e-01],
       [ 3.10282595e-01,  2.89719922e-01,  2.66840464e-01],
       [ 2.29331600e-01,  2.92120632e-01,  3.50957378e-01],
       [ 7.19954795e-01,  2.55367835e-01,  6.78203300e-02],
       [ 2.13990687e-01,  3.44769741e-01,  3.15381113e-01],
       [ 2.07681854e-01,  3.23975438e-01,  3.43672857e-01],
       [ 2.92291609e-01,  2.55327484e-01,  3.20240694e-01],
       [ 2.66810110e-01,  3.10269639e-01,  2.89763977e-01],
       [ 4.07733997e-01,  5.38500942e-01,  3.21641667e-02],
       [ 3.58567418e-01,  1.87390161e-01,  3.34839191e-01],
       [ 2.02048783e-01,  3.44141108e-01,  3.30410194e-01],
       [ 3.22944017e-01,  2.38213183e-01,  3.08417103e-01],
       [ 3.16687773e-01,  2.39301496e-01,  3.13344433e-01],
       [ 2.74407179e-01,  3.21402884e-01,  2.71571011e-01],
       [ 2.93141374e-01,  3.19602208e-01,  2.55100787e-01],
       [ 2.94551221e-01,  2.53449449e-01,  3.19974344e-01],
       [ 3.11307497e-01,  2.21610941e-01,  3.39659580e-01],
       [ 2.71369512e-01,  2.61910200e-01,  3.35519152e-01],
       [ 2.29180629e-01,  2.78123787e-01,  3.67155123e-01],
       [ 3.30446806e-01,  2.84220861e-01,  2.53926036e-01],
       [ 1.66009821e-01,  3.69589993e-01,  3.52211533e-01],
       [ 2.49365237e-01,  3.24104005e-01,  2.95010989e-01],
       [ 3.80069978e-01,  2.10004337e-01,  2.88410547e-01],
       [ 2.99987002e-01,  2.64638471e-01,  3.02168263e-01],
       [ 2.87035249e-01,  3.01502749e-01,  2.77735820e-01],
       [ 3.11495230e-01,  2.70130588e-01,  2.85158610e-01],
       [ 2.60463701e-01,  2.90569600e-01,  3.16345975e-01],
       [ 2.85473883e-01,  2.94883850e-01,  2.85717503e-01],
       [ 2.95183362e-01,  3.17950904e-01,  2.54668825e-01],
       [ 2.46069848e-01,  2.95009183e-01,  3.27876567e-01],
       [ 2.93596481e-01,  2.96672116e-01,  2.75972856e-01],
       [ 2.86651827e-01,  3.49281297e-01,  2.35681823e-01],
       [ 4.04081701e-01,  2.10126525e-01,  2.68787863e-01],
       [ 3.81214663e-01,  2.26254836e-01,  2.69558125e-01],
       [ 2.65360027e-01,  2.94822856e-01,  3.06624505e-01],
       [ 1.68235626e-01,  3.54987040e-01,  3.63666457e-01],
       [ 4.13945906e-01,  3.31219276e-01,  1.51500821e-01],
       [ 3.04874662e-01,  2.66372764e-01,  2.95475631e-01],
       [ 2.63892525e-01,  3.24434639e-01,  2.79409373e-01],
       [ 2.97093615e-01,  3.01606819e-01,  2.67904499e-01],
       [ 3.18146772e-01,  2.63840040e-01,  2.85333171e-01],
       [ 2.97214495e-01,  3.27630991e-01,  2.44257699e-01],
       [ 3.12332063e-01,  2.72234208e-01,  2.82213560e-01],
       [ 3.04173740e-01,  3.23513995e-01,  2.41514258e-01],
       [ 3.06774193e-01,  2.71185086e-01,  2.88614475e-01],
       [ 3.10925826e-01,  2.77237574e-01,  2.78493490e-01],
       [ 2.89188840e-01,  2.97759898e-01,  2.79225681e-01],
       [ 2.47889493e-01,  3.06614831e-01,  3.13781873e-01],
       [ 3.01831821e-01,  2.57012356e-01,  3.08539482e-01],
       [ 4.38267762e-01,  2.08847469e-01,  2.44885114e-01],
       [ 2.55710946e-01,  3.09214114e-01,  3.02572108e-01],
       [ 2.99889851e-01,  2.68121141e-01,  2.98573784e-01],
       [ 5.51849199e-01,  1.65938686e-01,  2.20715440e-01],
       [ 2.91848499e-01,  3.18270003e-01,  2.57512887e-01],
       [ 2.51656673e-01,  3.63237978e-01,  2.57912048e-01],
       [ 3.65856769e-01,  1.97546225e-01,  3.15451600e-01],
       [ 2.98733990e-01,  2.80144308e-01,  2.87299375e-01],
       [ 3.14369237e-01,  2.54849411e-01,  2.98449788e-01],
       [ 3.06327948e-01,  3.38526894e-01,  2.26872378e-01],
       [ 3.18083833e-01,  2.44837224e-01,  3.05764433e-01],
       [ 3.35502657e-01,  3.78889287e-01,  1.72008711e-01],
       [ 1.49310033e-01,  3.42474578e-01,  4.04374404e-01],
       [ 2.40027327e-01,  2.81799387e-01,  3.49465526e-01],
       [ 2.94972472e-01,  3.18422682e-01,  2.54442953e-01],
       [ 2.07556051e-01,  3.73038217e-01,  2.97236246e-01],
       [ 1.21634790e-01,  4.04038918e-01,  3.82090273e-01],
       [ 3.35574409e-01,  2.14411923e-01,  3.23733219e-01],
       [ 2.59489950e-01,  2.86967095e-01,  3.21223643e-01],
       [ 3.08590146e-01,  3.02506544e-01,  2.56341532e-01],
       [ 2.75158441e-01,  2.84756918e-01,  3.06558353e-01],
       [ 2.86720462e-01,  1.18145608e-01,  5.33819090e-01],
       [ 2.78987126e-01,  3.08146405e-01,  2.79376175e-01],
       [ 2.69673155e-01,  2.57818982e-01,  3.42134279e-01],
       [ 2.40086202e-01,  3.22644096e-01,  3.06607988e-01],
       [ 3.02913316e-01,  2.80980873e-01,  2.82392178e-01],
       [ 2.95165774e-01,  2.39889524e-01,  3.34905241e-01],
       [ 2.81286867e-01,  3.18655645e-01,  2.67302864e-01],
       [ 2.66939308e-01,  3.09848819e-01,  2.90035739e-01],
       [ 2.41286183e-01,  2.68828462e-01,  3.62929017e-01],
       [ 2.43370714e-01,  3.32810007e-01,  2.93317330e-01],
       [ 2.60358353e-01,  2.41513498e-01,  3.72844070e-01],
       [ 2.67607845e-01,  3.06983852e-01,  2.92118236e-01],
       [ 2.30904025e-01,  2.25705187e-01,  4.33376636e-01],
       [ 2.91283191e-01,  3.02778284e-01,  2.72372106e-01],
       [ 2.74692294e-01,  2.54667640e-01,  3.40117848e-01],
       [ 2.73494088e-01,  2.88608529e-01,  3.04334597e-01],
       [ 2.91539044e-01,  2.95018573e-01,  2.79581345e-01],
       [ 2.75656916e-01,  2.86937919e-01,  3.03777368e-01],
       [ 3.10749778e-01,  2.65213605e-01,  2.90964558e-01],
       [ 2.94908133e-01,  3.12838456e-01,  2.59551264e-01],
       [ 2.80217245e-01,  2.84149735e-01,  3.01889286e-01],
       [ 3.14677549e-01,  2.56625371e-01,  2.96245217e-01],
       [ 2.97847747e-01,  3.23026984e-01,  2.47694314e-01],
       [ 2.76637740e-01,  2.88350872e-01,  3.01300350e-01],
       [ 2.24103450e-01,  3.06020644e-01,  3.42221228e-01],
       [ 4.35428676e-01,  3.79498928e-01,  1.04003453e-01],
       [ 2.45651849e-01,  3.16536973e-01,  3.06377680e-01],
       [ 2.84020800e-01,  3.41074189e-01,  2.44967307e-01],
       [ 4.56572551e-01,  1.05926208e+00, -1.54126305e-01],
       [ 1.44323378e-01,  1.61462149e-01,  7.41360258e-01],
       [ 3.14671544e-01,  2.89986208e-01,  2.62544541e-01],
       [ 3.66262183e-01,  2.69923644e-01,  2.37567658e-01],
       [ 2.98207713e-01,  2.56912962e-01,  3.12340327e-01],
       [ 2.79579014e-01,  2.76296845e-01,  3.10776584e-01],
       [ 2.34438112e-01,  3.05714859e-01,  3.30144969e-01],
       [ 3.05252998e-01,  2.52810356e-01,  3.09694015e-01],
       [ 2.38783678e-01,  3.10341865e-01,  3.20319152e-01],
       [ 3.34551915e-01,  2.76096813e-01,  2.58137412e-01],
       [ 2.88583911e-01,  3.30052408e-01,  2.50150501e-01],
       [ 2.72078575e-01,  2.78451565e-01,  3.16493652e-01],
       [ 2.99343766e-01,  2.95884950e-01,  2.71204464e-01],
       [ 3.06920206e-01,  3.10484525e-01,  2.50574733e-01],
       [ 2.94300077e-01,  3.19278077e-01,  2.54305725e-01],
       [ 4.77702237e-01,  2.97140157e-02,  4.64718162e-01],
       [ 3.09026778e-01,  2.61567990e-01,  2.96477459e-01],
       [ 2.84717804e-01,  3.03074105e-01,  2.78515924e-01],
       [ 1.82019441e-01,  3.14692650e-01,  3.87990997e-01],
       [ 4.56510891e-01,  5.82927944e-01, -1.55015903e-02],
       [ 2.95377381e-01,  2.89402026e-01,  2.81332046e-01],
       [ 1.52935373e-01,  3.71781151e-01,  3.68087343e-01],
       [ 3.27725488e-01,  3.54419134e-01,  1.96216187e-01],
       [ 2.97955268e-01,  2.65894253e-01,  3.02874083e-01],
       [ 2.84158165e-01,  2.83376515e-01,  2.98618314e-01],
       [ 3.34053971e-01,  2.60027936e-01,  2.74602935e-01],
       [ 2.73703598e-01,  2.85197441e-01,  3.07640212e-01],
       [ 2.10862830e-01,  3.13453496e-01,  3.50750682e-01],
       [ 3.44163341e-01,  1.73782366e-01,  3.67201577e-01],
       [ 2.87868716e-01,  3.01622129e-01,  2.76802306e-01],
       [ 2.74191741e-01,  2.38572200e-01,  3.59981383e-01],
       [ 2.93622384e-01,  2.55264091e-01,  3.18916128e-01],
       [ 3.70358718e-01,  3.77167892e-01,  1.47570108e-01],
       [ 3.33579305e-01,  2.26936198e-01,  3.10961570e-01],
       [ 2.47826348e-01,  2.67017140e-01,  3.57052427e-01],
       [ 2.98782280e-01,  2.90141995e-01,  2.77303416e-01],
       [ 3.12917334e-01,  2.78104675e-01,  2.75753228e-01],
       [ 2.98182707e-01,  2.72457402e-01,  2.95734407e-01],
       [ 3.21266363e-01,  2.82539432e-01,  2.63709930e-01],
       [ 1.96186731e-01,  3.35284097e-01,  3.46626192e-01],
       [ 3.58177712e-01,  2.24822349e-01,  2.90692329e-01],
       [ 3.48839124e-01,  2.94859352e-01,  2.28587651e-01],
       [ 3.64563838e-01,  2.64865874e-01,  2.43775401e-01],
       [ 2.16137320e-01,  4.12844358e-01,  2.55601923e-01],
       [ 5.91970518e-01,  1.41363109e-01,  2.26796101e-01],
       [ 3.40293880e-01,  2.74013535e-01,  2.55173334e-01],
       [ 1.58505195e-01,  4.05028538e-01,  3.29706744e-01],
       [ 2.90552424e-01,  2.61664376e-01,  3.15044348e-01],
       [ 3.75026063e-01,  3.17604925e-01,  1.88974905e-01],
       [-1.37003379e-01,  1.02105445e+00,  4.41024194e-01],
       [ 2.71156085e-01,  2.99517550e-01,  2.95762732e-01],
       [ 3.09916865e-01,  3.06723224e-01,  2.51266992e-01],
       [ 2.99680955e-01,  2.72479686e-01,  2.94223362e-01],
       [ 2.77092559e-01,  3.00262001e-01,  2.88903293e-01],
       [ 2.93494432e-01,  3.00797353e-01,  2.72118269e-01],
       [ 2.63008975e-01,  2.95855430e-01,  3.08102225e-01],
       [ 3.13654699e-01,  2.57343010e-01,  2.96469063e-01],
       [ 2.83173668e-01,  3.03423163e-01,  2.79712643e-01],
       [ 2.46687993e-01,  2.98826414e-01,  3.23150240e-01],
       [ 2.33310731e-01,  2.94035045e-01,  3.43983922e-01],
       [ 2.99384802e-01,  2.77426658e-01,  2.89423298e-01],
       [ 1.91020104e-01,  3.01251831e-01,  3.90952298e-01],
       [ 9.50087380e-02,  6.11420874e-01,  2.71661424e-01],
       [ 2.88980419e-01,  2.48796288e-01,  3.31183448e-01],
       [ 3.49527737e-01,  2.52852946e-01,  2.68303560e-01],
       [ 1.80562274e-01,  2.55757632e-01,  4.67133903e-01],
       [ 3.27676854e-01,  1.60782038e-01,  4.03955074e-01],
       [ 2.83665607e-01,  3.04941121e-01,  2.77772381e-01],
       [ 3.00394873e-01,  2.12814243e-01,  3.62565058e-01],
       [ 1.60274393e-01,  4.00894488e-01,  3.30999964e-01],
       [ 3.49670104e-01,  2.74937762e-01,  2.46334529e-01],
       [ 2.69528971e-01,  2.50245717e-01,  3.51212812e-01],
       [ 3.43753424e-01,  2.35037063e-01,  2.92342753e-01],
       [ 2.47595668e-01,  3.02194114e-01,  3.18627323e-01],
       [ 3.20053097e-01,  2.43397057e-01,  3.05439650e-01],
       [ 2.87888521e-01,  2.59728578e-01,  3.19981104e-01],
       [ 2.63351280e-01,  2.54603742e-01,  3.53215571e-01],
       [ 3.60936026e-01,  2.38907730e-01,  2.73020419e-01],
       [ 2.63425609e-01,  2.97065455e-01,  3.06419429e-01],
       [ 3.13170529e-01,  2.50913383e-01,  3.03893303e-01],
       [ 2.89357707e-01,  2.54703588e-01,  3.24043551e-01],
       [ 2.19987620e-01,  3.93324924e-01,  2.66541730e-01],
       [ 2.53144427e-01,  2.95739643e-01,  3.19074951e-01],
       [ 2.83060541e-01,  2.99272320e-01,  2.83837348e-01],
       [ 3.55278345e-01,  2.14207175e-01,  3.05357771e-01],
       [ 3.16474532e-01,  3.70235929e-01,  1.93429058e-01],
       [ 3.02690153e-01,  2.89337266e-01,  2.74346176e-01],
       [ 2.48133537e-01,  2.91037130e-01,  3.29735904e-01],
       [ 3.20487125e-01,  3.30508008e-01,  2.21317229e-01],
       [ 2.90760595e-01,  3.09503779e-01,  2.66563374e-01],
       [ 2.31334095e-01,  2.95128190e-01,  3.45184816e-01],
       [ 3.09793737e-01,  3.04274115e-01,  2.53616574e-01],
       [ 3.07434502e-01,  3.33892783e-01,  2.29757633e-01],
       [ 3.42322789e-01,  3.49790405e-01,  1.88204436e-01],
       [ 1.59004709e-01,  3.63080575e-01,  3.68270251e-01],
       [ 2.50859444e-01,  3.31686053e-01,  2.86318274e-01],
       [ 3.06506039e-01,  3.05371236e-01,  2.55609873e-01],
       [ 2.28663608e-01,  3.05330426e-01,  3.37423326e-01],
       [ 1.65506823e-01,  3.56965440e-01,  3.65416114e-01],
       [ 2.54991243e-01,  3.16565362e-01,  2.96171199e-01],
       [ 2.40679512e-01,  3.21867140e-01,  3.06700206e-01],
       [ 3.10929619e-01,  2.65413119e-01,  2.90582651e-01],
       [ 3.09663769e-01,  2.83260540e-01,  2.73701839e-01],
       [ 2.44252679e-01,  3.03649058e-01,  3.20920509e-01],
       [ 3.00171761e-01,  3.09508474e-01,  2.57666703e-01],
       [ 5.92181313e-01,  1.48042197e-01,  2.19301570e-01],
       [ 5.25319393e-01,  1.29669859e-01,  2.77686859e-01],
       [ 2.65869407e-01,  3.23328684e-01,  2.78406867e-01],
       [ 1.82436165e-01,  3.67615109e-01,  3.32575740e-01],
       [ 4.00294214e-01,  1.38687347e-01,  3.60836569e-01],
       [ 9.24605724e-01,  4.57710027e-01, -1.25297936e-01],
       [ 2.65757722e-01,  3.21370658e-01,  2.80336076e-01],
       [ 3.15095678e-01,  2.32622503e-01,  3.22614186e-01],
       [ 2.97666087e-01,  2.71923554e-01,  2.96805924e-01],
       [ 2.99968288e-01,  2.37760448e-01,  3.32285395e-01],
       [ 1.97819634e-01,  1.09223518e-01,  7.43848028e-01],
       [ 2.89334061e-01,  2.86747037e-01,  2.89949307e-01],
       [ 2.14750574e-01,  3.46108214e-01,  3.13221556e-01],
       [ 2.95051322e-01,  2.83029019e-01,  2.88008261e-01],
       [ 2.96656875e-01,  2.88917805e-01,  2.80562928e-01],
       [ 2.80347109e-01,  2.85704828e-01,  3.00155279e-01],
       [ 2.99353909e-01,  2.71477165e-01,  2.95590512e-01],
       [ 2.96274525e-01,  3.03996301e-01,  2.66435805e-01],
       [ 3.08396471e-01,  2.55551178e-01,  3.03554628e-01],
       [ 2.75547826e-01,  2.86685496e-01,  3.04152081e-01],
       [ 3.23266036e-01,  2.41785611e-01,  3.04111889e-01],
       [ 3.26584028e-01,  3.37831219e-01,  2.10214501e-01],
       [ 2.94972030e-01,  2.73746374e-01,  2.97603305e-01],
       [ 3.83479752e-01,  2.77413347e-01,  2.17308364e-01],
       [ 3.45135862e-01,  3.18937363e-02,  6.33882139e-01],
       [ 2.76186465e-01,  3.64171819e-01,  2.33339174e-01],
       [ 2.66611747e-01,  4.10112606e-01,  2.07852962e-01],
       [ 2.65202318e-01,  3.94622304e-01,  2.20278610e-01],
       [ 4.16185619e-01,  2.16363902e-01,  2.52869460e-01],
       [ 3.16592608e-01,  3.41371460e-01,  2.15702234e-01],
       [ 3.39365447e-01,  2.76305467e-01,  2.53758084e-01],
       [ 3.21890513e-01,  3.32618448e-01,  2.18382460e-01],
       [ 2.85394455e-01,  2.69032442e-01,  3.12430067e-01],
       [ 4.75861656e-01,  5.27350724e-01, -9.42959520e-04],
       [ 3.17566715e-01,  3.24754859e-01,  2.28652862e-01],
       [ 3.35739592e-01,  2.05031408e-01,  3.35008421e-01],
       [ 2.57256005e-01,  3.02205942e-01,  3.07895305e-01],
       [ 2.71890053e-01,  2.37943963e-01,  3.63462220e-01],
       [ 3.89855028e-01,  1.93992304e-01,  2.98658767e-01],
       [ 3.21232308e-01,  3.33577773e-01,  2.18145759e-01],
       [ 2.79987732e-01,  2.92136133e-01,  2.94001836e-01],
       [ 2.80396046e-01,  3.07745735e-01,  2.78350082e-01],
       [ 2.94494430e-01,  3.20891106e-01,  2.52686077e-01],
       [ 3.19928134e-01,  2.72905395e-01,  2.74427607e-01],
       [ 3.61332468e-01,  2.19572524e-01,  2.93785250e-01],
       [ 2.73350931e-01,  2.75703397e-01,  3.18067285e-01],
       [ 3.42618855e-01,  2.71056924e-01,  2.56048540e-01],
       [ 2.12331100e-01,  3.12799829e-01,  3.49594469e-01],
       [ 2.04438897e-01,  2.95644042e-01,  3.79054840e-01],
       [ 2.76499909e-01,  2.90527443e-01,  2.99225757e-01],
       [ 3.56871818e-01,  3.45464402e-01,  1.80417138e-01],
       [ 3.28121852e-01,  2.23126173e-01,  3.20703964e-01],
       [ 2.58499575e-01,  2.72622480e-01,  3.38014968e-01],
       [ 2.61341327e-01,  3.50064743e-01,  2.59260782e-01],
       [ 2.62265590e-01,  2.94504983e-01,  3.10292039e-01],
       [ 3.41232356e-01,  2.45858173e-01,  2.82929510e-01],
       [ 2.29795470e-01,  3.03670465e-01,  3.37824575e-01],
       [ 3.41688229e-01,  3.66795600e-01,  1.75967688e-01],
       [ 3.17496334e-01,  2.61727576e-01,  2.88148419e-01],
       [ 2.85301187e-01,  3.21246832e-01,  2.61064075e-01],
       [ 3.20229746e-01,  3.12888704e-01,  2.36612485e-01],
       [ 3.49287501e-01,  1.72748830e-01,  3.63309949e-01],
       [ 3.01168600e-01,  2.95103615e-01,  2.70218959e-01],
       [ 3.07917868e-01,  2.90628074e-01,  2.68167256e-01],
       [ 2.89732977e-01,  3.00565349e-01,  2.75989783e-01],
       [ 2.45495787e-01,  3.14032286e-01,  3.09021843e-01],
       [ 2.63022895e-01,  3.18782431e-01,  2.85581644e-01],
       [ 1.72768979e-01,  3.94813049e-01,  3.20285956e-01],
       [ 9.96253133e-01,  5.61634806e-01, -1.98685943e-01],
       [ 3.21394907e-01,  1.29893099e-01,  4.61463671e-01],
       [ 5.17928008e-01,  9.97413684e-02,  3.21112167e-01],
       [ 6.75748067e-01,  4.01671776e-01, -1.98891143e-02],
       [ 2.41302982e-01,  3.49788520e-01,  2.80151189e-01],
       [ 3.32771103e-01,  2.73252496e-01,  2.62480646e-01],
       [ 3.38361213e-01,  3.38769841e-01,  1.99922046e-01],
       [ 2.82417572e-01,  2.97450719e-01,  2.86262748e-01],
       [ 2.74603370e-01,  2.82168004e-01,  3.09850188e-01],
       [ 3.90371515e-01,  1.48996273e-01,  3.55668439e-01],
       [ 2.79986724e-01,  2.96577075e-01,  2.89581754e-01],
       [ 2.62631864e-01,  2.72705359e-01,  3.33208444e-01],
       [ 2.75886421e-01,  2.95401756e-01,  2.94952134e-01],
       [ 2.94574264e-01,  2.74042009e-01,  2.97695098e-01],
       [ 3.00098627e-01,  2.85075833e-01,  2.81025823e-01],
       [ 2.94845234e-01,  3.03405129e-01,  2.68353274e-01],
       [ 3.08031950e-01,  2.64332712e-01,  2.94527406e-01],
       [ 2.99006936e-01,  3.14473551e-01,  2.54238285e-01],
       [ 3.00752243e-01,  2.80733398e-01,  2.84734118e-01],
       [ 3.03032753e-01,  3.28404033e-01,  2.38318110e-01],
       [ 3.37189966e-01,  2.12827429e-01,  3.24056527e-01],
       [ 2.75158652e-01,  2.97623799e-01,  2.93490550e-01],
       [ 2.21736821e-01,  5.87528328e-01,  1.47940803e-01],
       [-3.18937294e-02,  4.08944186e-01,  6.97632773e-01],
       [ 2.89400146e-01,  2.46245526e-01,  3.33684221e-01],
       [ 5.13764672e-01, -1.28515237e-03,  4.89112748e-01],
       [ 2.62959777e-01,  3.93172104e-01,  2.23448296e-01],
       [ 2.51579892e-01,  2.57315635e-01,  3.64052247e-01],
       [ 5.22917650e-01,  2.28755590e-01,  1.73452582e-01],
       [ 2.91241214e-01,  2.31062903e-01,  3.49805704e-01],
       [ 2.22185638e-01,  3.49079281e-01,  3.01855745e-01],
       [ 2.14713381e-01,  3.18559827e-01,  3.40540158e-01],
       [ 3.50091555e-01,  2.57617630e-01,  2.62971578e-01],
       [ 3.36285165e-01,  2.32813748e-01,  3.01719414e-01],
       [ 2.10836149e-01,  3.66428519e-01,  2.99245098e-01],
       [ 3.60734320e-01,  2.37272838e-01,  2.74925712e-01],
       [ 3.34231986e-01,  2.89198627e-01,  2.45962527e-01],
       [ 2.57841500e-01,  4.11727237e-01,  2.14824294e-01],
       [ 1.00282307e-01,  3.53669902e-01,  4.72589762e-01],
       [ 3.01499382e-01,  2.86560443e-01,  2.78206735e-01],
       [ 2.88883400e-01,  2.92149146e-01,  2.85014949e-01],
       [ 3.06296177e-01,  2.44492845e-01,  3.17930768e-01],
       [ 3.28796581e-01,  2.84315964e-01,  2.55284099e-01],
       [ 2.20452706e-01,  2.92125998e-01,  3.62090801e-01],
       [ 3.06894114e-01,  2.67284041e-01,  2.92543525e-01],
       [ 2.86425562e-01,  3.16147472e-01,  2.64610718e-01],
       [ 2.89994944e-01,  2.58050521e-01,  3.19620661e-01],
       [ 2.99293292e-01,  3.25862084e-01,  2.43894030e-01],
       [ 2.81539882e-01,  2.93887672e-01,  2.90668910e-01],
       [ 3.32711079e-01,  2.02229347e-01,  3.41563372e-01],
       [ 3.29764646e-01,  2.78071699e-01,  2.60434878e-01],
       [ 3.16004198e-01,  3.30227725e-01,  2.25378300e-01],
       [ 2.86418964e-01,  3.65628640e-01,  2.22801254e-01],
       [ 2.24558596e-01,  3.05670830e-01,  3.42038708e-01],
       [ 3.17857933e-01,  3.08589810e-01,  2.42497930e-01],
       [ 3.34816781e-01,  2.12279784e-01,  3.27044945e-01],
       [ 3.43534379e-01,  3.45572332e-01,  1.90513344e-01],
       [ 3.06709066e-01,  2.62334978e-01,  2.97937366e-01],
       [ 2.67578224e-01,  2.79987342e-01,  3.19745241e-01],
       [ 3.23624784e-01,  2.32279000e-01,  3.14494637e-01],
       [ 3.39179997e-01,  1.90632880e-01,  3.49823774e-01],
       [ 3.18074558e-01,  2.59065740e-01,  2.90393479e-01],
       [ 3.34988210e-01,  2.49833773e-01,  2.84374795e-01],
       [ 2.81630787e-01,  3.40090822e-01,  2.48053071e-01],
       [ 2.96785996e-01,  3.23533624e-01,  2.48226473e-01],
       [ 3.35232111e-01,  2.60312288e-01,  2.73254122e-01],
       [ 3.79003449e-01,  2.58717657e-01,  2.38262643e-01],
       [ 2.35340438e-01,  2.62307655e-01,  3.78316333e-01],
       [ 1.93025221e-01,  4.03798190e-01,  2.88287561e-01],
       [ 3.48450628e-01,  1.92692740e-01,  3.37906929e-01],
       [ 7.15931150e-01,  3.85929882e-01, -2.38680050e-02],
       [ 3.09963816e-01,  2.92012850e-01,  2.64938147e-01],
       [ 2.78150843e-01,  2.85249813e-01,  3.02906151e-01],
       [ 2.53559308e-01,  2.80361507e-01,  3.35090382e-01],
       [ 2.91000046e-01,  2.82807120e-01,  2.92263891e-01],
       [ 2.48699322e-01,  3.30029763e-01,  2.90156181e-01],
       [ 3.54415300e-01,  2.35257283e-01,  2.82565655e-01],
       [ 2.85731049e-01,  2.88049088e-01,  2.92264268e-01],
       [ 3.14971020e-01,  2.22538778e-01,  3.34704101e-01],
       [ 2.94927404e-01,  2.88575089e-01,  2.82588850e-01],
       [ 2.83737179e-01,  2.98326809e-01,  2.84081469e-01],
       [ 2.75304557e-01,  2.96254357e-01,  2.94702474e-01],
       [ 2.88928563e-01,  2.99776598e-01,  2.77534476e-01],
       [ 2.72488409e-01,  2.93950470e-01,  2.99947462e-01],
       [ 2.60037352e-01,  3.01211758e-01,  3.05877887e-01],
       [ 2.55044388e-01,  3.03720365e-01,  3.08784376e-01],
       [ 2.93729523e-01,  2.98793269e-01,  2.73804819e-01],
       [ 2.27330161e-01,  3.25253369e-01,  3.18612644e-01],
       [ 1.72686955e-01,  3.70490300e-01,  3.42468608e-01],
       [ 2.97160984e-01,  2.91855650e-01,  2.77194018e-01],
       [ 3.97434066e-01,  2.82748109e-01,  2.02337366e-01],
       [ 2.69797603e-01,  3.42658071e-01,  2.57245839e-01],
       [ 2.59717621e-01,  3.41048900e-01,  2.68696049e-01],
       [ 4.70691587e-01,  2.16255684e-01,  2.15752030e-01],
       [ 2.41887841e-01,  2.61558088e-01,  3.70908311e-01],
       [ 3.69920308e-01,  2.26227611e-01,  2.78980782e-01],
       [ 1.59677150e-01,  4.20003817e-01,  3.15578737e-01],
       [ 2.77360764e-01,  3.22962480e-01,  2.67227500e-01],
       [ 2.99281809e-01,  2.62506916e-01,  3.05161794e-01],
       [ 2.85229279e-01,  3.38696016e-01,  2.45853118e-01],
       [ 2.58851090e-01,  2.91525266e-01,  3.17125483e-01],
       [ 3.34100060e-01,  2.27885086e-01,  3.09373976e-01],
       [ 3.39787609e-01,  2.95397249e-01,  2.35565557e-01],
       [ 3.55708970e-01,  2.56783020e-01,  2.59040084e-01],
       [ 2.49058294e-01,  3.53572659e-01,  2.68721339e-01],
       [ 1.39106046e-01,  3.42280408e-01,  4.20424638e-01],
       [ 5.41906915e-01,  2.30487525e-01,  1.61960276e-01],
       [ 2.90438721e-01,  2.83256233e-01,  2.92370398e-01],
       [ 3.16422518e-01,  2.67218149e-01,  2.83472639e-01],
       [ 2.75717027e-01,  2.77152626e-01,  3.13969670e-01],
       [ 2.93040181e-01,  3.12611527e-01,  2.61523677e-01],
       [ 3.41466368e-01,  2.45306786e-01,  2.83305365e-01],
       [ 3.53396920e-01,  2.40187682e-01,  2.78171658e-01],
       [ 2.68559689e-01,  2.64689148e-01,  3.35519091e-01],
       [ 3.68245144e-01,  2.87570230e-01,  2.19732054e-01],
       [ 2.96226509e-01,  2.59151453e-01,  3.11918157e-01],
       [ 2.45797112e-01,  3.02776787e-01,  3.20063241e-01],
       [ 3.51007218e-01,  2.90893765e-01,  2.30400315e-01],
       [ 2.76989658e-01,  3.07814235e-01,  2.81698946e-01],
       [ 2.94275054e-01,  2.70788405e-01,  3.01406372e-01],
       [ 2.99094319e-01,  3.57089467e-01,  2.18226315e-01],
       [ 2.55948783e-01,  2.36179973e-01,  3.85163479e-01],
       [ 2.53475786e-01,  2.62073240e-01,  3.56068522e-01],
       [ 2.42664390e-01,  3.05846562e-01,  3.20471141e-01],
       [ 3.52937545e-01,  2.42177854e-01,  2.76461243e-01],
       [ 1.97169116e-01,  3.32900943e-01,  3.47807261e-01],
       [ 2.72021335e-01,  3.04376817e-01,  2.90082491e-01],
       [ 3.11593686e-01,  2.59702266e-01,  2.95955910e-01],
       [ 2.44172202e-01,  3.27928229e-01,  2.97026593e-01],
       [ 2.11814416e-01,  3.50533628e-01,  3.12532294e-01],
       [ 3.18036652e-01,  2.93246755e-01,  2.56406082e-01],
       [ 2.77175804e-01,  2.55634021e-01,  3.36225847e-01],
       [ 2.79362822e-01,  2.70878926e-01,  3.16818016e-01],
       [ 2.52003295e-01,  3.14857988e-01,  3.01052047e-01],
       [ 2.75864137e-01,  3.40653389e-01,  2.53076255e-01],
       [ 2.28539847e-01,  2.83753262e-01,  3.61416287e-01],
       [ 7.21965853e-01,  2.58672598e-01,  6.44959585e-02],
       [ 2.70411737e-01,  3.05129006e-01,  2.91012474e-01],
       [ 2.38772792e-02,  3.38187490e-01,  6.68181561e-01],
       [ 2.75308908e-01,  2.53930383e-01,  3.40282187e-01],
       [ 2.72871653e-01,  3.24936128e-01,  2.69876284e-01],
       [ 2.89296119e-01,  3.01464765e-01,  2.75555505e-01],
       [ 2.74742454e-01,  3.00049743e-01,  2.91520306e-01],
       [ 2.97968887e-01,  2.83660864e-01,  2.84507262e-01],
       [ 2.58239394e-01,  2.98441128e-01,  3.10646299e-01],
       [ 2.85533142e-01,  2.94783575e-01,  2.85756923e-01],
       [ 2.80462038e-01,  2.97970733e-01,  2.87726646e-01],
       [ 3.17968239e-01,  3.32837871e-01,  2.21522395e-01],
       [ 2.87935572e-01,  2.78966462e-01,  2.99303269e-01],
       [ 3.04552753e-01,  2.83951988e-01,  2.77859513e-01],
       [ 2.70132921e-01,  3.04529205e-01,  2.91887405e-01],
       [ 2.85706775e-01,  3.00621526e-01,  2.79895057e-01],
       [ 2.91900832e-01,  3.02073514e-01,  2.72443568e-01],
       [ 2.61019037e-01,  2.94964607e-01,  3.11175740e-01],
       [ 2.54516694e-01,  2.96967314e-01,  3.16268212e-01],
       [ 2.93849054e-01,  2.80690167e-01,  2.91571844e-01],
       [ 3.12385328e-01,  2.34857707e-01,  3.22770482e-01],
       [ 3.76429461e-01,  3.01419995e-01,  2.01426191e-01],
       [ 2.85582973e-01,  2.85457408e-01,  2.95037323e-01],
       [ 2.17224208e-01,  3.60078504e-01,  2.97560064e-01],
       [ 2.63959912e-01,  3.30530842e-01,  2.73768947e-01],
       [ 2.71117636e-01,  3.18753136e-01,  2.77315661e-01],
       [ 3.52961887e-01,  3.38547694e-01,  1.88725609e-01],
       [ 3.54499872e-01,  2.53495623e-01,  2.63383258e-01],
       [ 4.18681871e-01,  2.22944471e-01,  2.44156422e-01],
       [ 3.09284184e-01,  2.30757523e-01,  3.30771393e-01],
       [ 3.65169253e-01,  2.71883404e-01,  2.36584117e-01],
       [ 2.61289228e-01,  2.60995565e-01,  3.48094895e-01],
       [ 3.04739509e-01,  2.32454515e-01,  3.33514368e-01],
       [ 2.41797146e-01,  2.83165086e-01,  3.45799144e-01],
       [ 2.90684221e-01,  3.41895876e-01,  2.38098329e-01],
       [ 2.46042280e-01,  4.58917902e-01,  1.94460335e-01],
       [ 3.07868934e-01,  2.16521638e-01,  3.49624352e-01],
       [ 2.66621600e-01,  3.55376633e-01,  2.49597033e-01],
       [ 3.23006905e-01,  2.85783289e-01,  2.59021952e-01],
       [ 2.28230729e-01,  2.09270610e-01,  4.62256909e-01],
       [ 4.59051336e-01,  5.52114579e-01, -3.41084971e-03],
       [ 2.82718144e-01,  3.09032927e-01,  2.74829726e-01],
       [ 2.96725859e-01,  3.22273090e-01,  2.49391765e-01],
       [ 2.74963550e-01,  3.12055802e-01,  2.79711440e-01],
       [ 3.03490312e-01,  3.01882663e-01,  2.61626374e-01],
       [ 2.33784394e-01,  2.78966552e-01,  3.60373735e-01],
       [ 2.94084687e-01,  3.08337033e-01,  2.64470544e-01],
       [ 3.03452977e-01,  2.60086452e-01,  3.03574130e-01],
       [ 2.37655658e-01,  2.51818495e-01,  3.88486111e-01],
       [ 2.90751426e-01,  3.24073665e-01,  2.53364936e-01],
       [ 2.74484420e-01,  2.76037646e-01,  3.16484984e-01],
       [ 2.51664474e-01,  2.67738494e-01,  3.51595474e-01],
       [ 2.67780615e-01,  2.85346726e-01,  3.13833119e-01],
       [ 2.99502898e-01,  3.27097403e-01,  2.42632153e-01],
       [ 4.01321408e-01,  2.21790020e-01,  2.58366176e-01],
       [ 2.00874991e-01,  3.18634181e-01,  3.58019399e-01],
       [ 2.85989522e-01,  3.72913910e-01,  2.17559239e-01],
       [ 3.39581884e-01,  3.05225004e-01,  2.26968912e-01],
       [ 2.73127950e-01,  3.00395480e-01,  2.92845226e-01],
       [ 3.31367164e-01,  3.19703755e-01,  2.21267252e-01],
       [ 2.89189129e-01,  3.06812457e-01,  2.70591852e-01],
       [ 3.03323388e-01,  2.84879528e-01,  2.78117249e-01],
       [ 3.27013352e-01,  3.05137996e-01,  2.37626325e-01],
       [ 3.15306200e-01,  2.24715972e-01,  3.31737603e-01],
       [ 3.52878821e-01,  2.05603545e-01,  3.17730468e-01],
       [ 2.97265925e-01,  2.81204703e-01,  2.87667888e-01],
       [ 3.58900421e-01,  2.84498152e-01,  2.29862949e-01],
       [ 3.51678881e-01,  2.29392288e-01,  2.91405917e-01],
       [ 2.85403887e-01,  2.98168867e-01,  2.82572216e-01],
       [ 3.65457780e-01,  3.16670412e-01,  1.96840324e-01],
       [ 2.84741813e-01,  2.86288793e-01,  2.95048301e-01],
       [ 2.33146329e-01,  3.98842547e-01,  2.48440013e-01],
       [ 2.68417531e-01,  3.08297975e-01,  2.89999864e-01],
       [ 3.02254667e-01,  2.76993762e-01,  2.87057045e-01],
       [ 2.72682930e-01,  2.94977270e-01,  2.98708160e-01],
       [ 3.07272505e-01,  3.06365050e-01,  2.53997563e-01],
       [ 2.82411235e-01,  3.01797227e-01,  2.82038148e-01],
       [ 3.08928478e-01,  2.41385624e-01,  3.18779958e-01],
       [ 2.36947327e-01,  3.23948649e-01,  3.08865175e-01],
       [ 2.93643245e-01,  2.88786805e-01,  2.83638706e-01],
       [ 3.02093581e-01,  3.02407026e-01,  2.62439403e-01],
       [ 2.40875649e-01,  3.07596063e-01,  3.20723193e-01],
       [ 2.90602414e-01,  2.85639872e-01,  2.89795400e-01],
       [ 2.99703656e-01,  2.71500508e-01,  2.95218971e-01],
       [ 2.81010185e-01,  2.83797347e-01,  3.01430214e-01],
       [ 2.93539171e-01,  2.93545191e-01,  2.79062088e-01],
       [ 2.70117736e-01,  3.03815632e-01,  2.92601926e-01],
       [ 3.01367273e-01,  2.70219270e-01,  2.94906795e-01],
       [ 3.07539396e-01,  2.57152274e-01,  3.02670562e-01],
       [ 3.15334200e-01,  3.04973828e-01,  2.47991828e-01],
       [ 3.01756295e-01,  2.86355706e-01,  2.78161919e-01],
       [ 3.02763214e-01,  2.20986447e-01,  3.49581960e-01],
       [ 2.84134634e-01,  2.69773574e-01,  3.12954352e-01],
       [ 2.38282301e-01,  3.65604416e-01,  2.69724327e-01],
       [ 3.19979299e-01,  2.61094000e-01,  2.86461838e-01],
       [ 2.55613010e-01,  2.87950352e-01,  3.24518090e-01],
       [ 3.35098279e-01,  2.60490642e-01,  2.73191842e-01],
       [ 3.53721461e-01,  2.84647342e-01,  2.33899472e-01],
       [ 2.47813578e-01,  2.67338197e-01,  3.56690928e-01],
       [ 2.77533364e-01,  2.34751180e-01,  3.60832114e-01],
       [ 3.40781408e-01,  3.03341268e-01,  2.27638214e-01],
       [ 2.66549430e-01,  2.64088837e-01,  3.38474027e-01],
       [ 2.68956675e-01,  3.37671844e-01,  2.62402605e-01],
       [ 3.13959676e-01,  2.85738537e-01,  2.67283806e-01],
       [ 3.13559456e-01,  2.73407825e-01,  2.79862944e-01],
       [ 1.95857918e-01,  3.32837905e-01,  3.49560282e-01],
       [ 5.82251220e-01, -1.50728304e-01,  7.82720284e-01],
       [ 2.94435671e-01,  2.53530734e-01,  3.20004123e-01],
       [ 3.22158506e-01,  1.90934769e-01,  3.67357651e-01],
       [ 2.59212952e-01,  3.40129597e-01,  2.70019213e-01],
       [ 2.63958792e-01,  3.28715088e-01,  2.75417506e-01],
       [ 5.24144657e-01,  3.84178013e-01,  5.35439097e-02],
       [ 2.69590214e-01,  3.04794222e-01,  2.92191867e-01],
       [ 3.08882355e-01,  3.10533319e-01,  2.48753371e-01],
       [ 2.44538043e-01,  2.86788557e-01,  3.38528671e-01],
       [ 2.72538802e-01,  2.68503242e-01,  3.26818313e-01],
       [ 2.67946160e-01,  3.00456861e-01,  2.98192887e-01],
       [ 2.96768936e-01,  2.78642241e-01,  2.90761885e-01],
       [ 3.38962807e-01,  2.51851902e-01,  2.78651783e-01],
       [ 2.70978493e-01,  2.96238042e-01,  2.99225872e-01],
       [ 2.65576306e-01,  3.74902465e-01,  2.34878649e-01],
       [ 1.95206680e-01,  4.36010084e-01,  2.61222338e-01],
       [ 2.73350653e-01,  2.87271997e-01,  3.05863511e-01],
       [ 2.79251796e-01,  3.07891789e-01,  2.79353585e-01],
       [ 1.49883732e-01,  4.33649192e-01,  3.17039594e-01],
       [ 2.90293647e-01,  2.53770705e-01,  3.24101323e-01],
       [ 2.70473775e-01,  2.91736436e-01,  3.04322016e-01],
       [ 2.05595169e-01,  4.31623254e-01,  2.53069180e-01],
       [ 2.97492103e-01,  2.34891603e-01,  3.38330420e-01],
       [ 2.38608346e-01,  2.54879846e-01,  3.83359815e-01],
       [ 3.40614529e-01,  2.22507463e-01,  3.09365870e-01],
       [ 3.16184114e-01,  2.80555145e-01,  2.70290446e-01],
       [ 2.28268899e-01,  3.05364382e-01,  3.37862752e-01],
       [ 3.33347893e-01,  2.57077277e-01,  2.78280533e-01],
       [ 2.78611330e-01,  3.16525928e-01,  2.71890707e-01],
       [ 2.61226235e-01,  3.22895807e-01,  2.83589956e-01],
       [ 2.44733311e-01,  3.19660138e-01,  3.04341796e-01],
       [ 3.70637843e-01,  3.22911508e-01,  1.87898345e-01],
       [ 3.00944438e-01,  3.96425908e-01,  1.87415235e-01],
       [ 2.62141775e-01,  2.75230916e-01,  3.30963001e-01],
       [ 2.69912575e-01,  3.42548324e-01,  2.57227686e-01],
       [ 1.80621617e-01,  3.50089053e-01,  3.51917457e-01],
       [ 2.52551478e-01,  3.14712322e-01,  3.00599011e-01],
       [ 3.37239834e-01,  2.73518750e-01,  2.58299410e-01],
       [ 2.64116353e-01,  2.82111059e-01,  3.21276179e-01],
       [ 2.98591419e-01,  3.06775631e-01,  2.61658491e-01],
       [ 2.83575976e-01,  3.10751826e-01,  2.72371992e-01],
       [ 2.74834099e-01,  2.76120363e-01,  3.16020144e-01],
       [ 2.88575561e-01,  2.73801534e-01,  3.04044332e-01],
       [ 2.62721429e-01,  3.04537825e-01,  2.99671423e-01],
       [ 2.95083939e-01,  3.16938245e-01,  2.55671474e-01],
       [ 2.80345238e-01,  2.87463329e-01,  2.98359402e-01],
       [ 2.72418992e-01,  2.36346960e-01,  3.64832982e-01],
       [ 3.02975049e-01,  3.19458028e-01,  2.46150461e-01],
       [ 2.93261425e-01,  2.97335046e-01,  2.75658777e-01],
       [ 2.73875427e-01,  2.88520541e-01,  3.04023008e-01],
       [ 3.13000900e-01,  2.76596079e-01,  2.77181166e-01],
       [ 2.92684393e-01,  2.70372661e-01,  3.03461507e-01],
       [ 2.93280747e-01,  2.75418452e-01,  2.97565164e-01],
       [ 2.95701317e-01,  3.03425954e-01,  2.67516692e-01],
       [ 3.02546885e-01,  3.04153410e-01,  2.60391061e-01],
       [ 2.97036306e-01,  2.64676637e-01,  3.05105003e-01],
       [ 2.99955534e-01,  2.86657779e-01,  2.79597155e-01],
       [ 3.36871164e-01,  2.85932024e-01,  2.46751702e-01],
       [ 2.72058339e-01,  3.09594312e-01,  2.85002202e-01],
       [ 2.77551344e-01,  2.53203712e-01,  3.38617536e-01],
       [ 2.63908419e-01,  2.88399706e-01,  3.14840361e-01],
       [ 3.15986153e-01,  2.77406030e-01,  2.73585565e-01],
       [ 2.80011126e-01,  2.79873384e-01,  3.06549539e-01],
       [ 2.76740363e-01,  3.15101830e-01,  2.75070969e-01],
       [ 3.30014889e-01,  2.73320252e-01,  2.64861495e-01],
       [ 2.55126516e-01,  3.34630920e-01,  2.79143202e-01],
       [ 3.24979988e-01,  2.85813772e-01,  2.57232890e-01],
       [ 2.58196419e-01,  2.16586523e-01,  4.08772342e-01],
       [ 3.89058760e-01,  2.44958541e-01,  2.43994499e-01],
       [ 2.36580879e-01,  3.19141592e-01,  3.14000622e-01],
       [ 2.50956041e-01,  3.15750437e-01,  3.01320925e-01],
       [ 2.94391551e-01,  3.13909913e-01,  2.59061638e-01],
       [ 1.61905679e-01,  2.08774106e-01,  5.83248116e-01],
       [ 3.18025419e-01,  3.10234167e-01,  2.40883947e-01],
       [ 3.04314243e-01,  3.03826472e-01,  2.59053987e-01],
       [ 2.50087565e-01,  2.88360642e-01,  3.30365274e-01],
       [ 3.00108740e-01,  3.01410183e-01,  2.65235495e-01],
       [ 2.01747338e-01,  3.24589213e-01,  3.50564652e-01],
       [ 4.38889476e-01,  2.12393245e-01,  2.40729310e-01],
       [ 2.61195615e-01,  2.80611160e-01,  3.26141355e-01],
       [ 3.38162363e-01,  2.58715877e-01,  2.72269981e-01],
       [ 2.74995942e-01,  2.70089740e-01,  3.22383111e-01],
       [ 2.87419889e-01,  3.12719603e-01,  2.66801583e-01],
       [ 2.70926412e-01,  2.42204182e-01,  3.59324688e-01],
       [ 2.52490781e-01,  2.97438358e-01,  3.18040170e-01],
       [ 2.71048463e-01,  2.87495422e-01,  3.08077508e-01],
       [ 2.23108229e-01,  3.06015190e-01,  3.43446323e-01],
       [ 1.94215586e-01,  2.20185818e-01,  5.00086344e-01],
       [ 3.12907004e-01,  2.39726356e-01,  3.16643831e-01],
       [ 3.18776032e-01,  2.49197312e-01,  3.00299075e-01],
       [ 1.83004464e-01,  2.56376057e-01,  4.62200819e-01],
       [ 3.16744052e-01,  2.82872594e-01,  2.67507231e-01],
       [ 3.23263602e-01,  2.24749400e-01,  3.23617685e-01],
       [ 3.02473897e-01,  1.81585653e-01,  4.02998102e-01],
       [ 3.97302065e-01,  2.23334936e-01,  2.59843626e-01],
       [ 2.71932039e-01,  2.84191835e-01,  3.10576731e-01],
       [ 3.21053402e-01,  2.69929095e-01,  2.76384405e-01],
       [ 3.06932766e-01,  2.72033146e-01,  2.87588804e-01],
       [ 3.15929794e-01,  2.60555467e-01,  2.90870862e-01],
       [ 3.12354638e-01,  2.88660197e-01,  2.65942934e-01],
       [ 2.92614781e-01,  2.70473622e-01,  3.03425571e-01],
       [ 2.64624519e-01,  2.55282533e-01,  3.50920384e-01],
       [ 3.02486669e-01,  3.11771778e-01,  2.53465287e-01],
       [ 8.94568203e-01, -1.41239210e-01,  4.99580011e-01],
       [ 4.28171806e-01,  1.42663482e-01,  3.30945763e-01],
       [ 3.89089876e-01,  2.70968465e-01,  2.19024448e-01],
       [ 3.58321414e-01,  2.32073994e-01,  2.82595218e-01],
       [ 2.67330761e-01,  2.28620580e-01,  3.80849230e-01],
       [ 2.98932299e-01,  2.56645219e-01,  3.11892488e-01],
       [ 3.04132731e-01,  2.75983030e-01,  2.86261016e-01],
       [ 2.62849940e-01,  3.03096426e-01,  3.00966899e-01],
       [ 3.25266115e-01,  2.66436152e-01,  2.76046514e-01],
       [ 2.87951573e-01,  2.63549276e-01,  3.15703179e-01],
       [ 2.77799329e-01,  3.01588337e-01,  2.86887298e-01],
       [ 3.22100123e-01,  3.01626370e-01,  2.45053098e-01],
       [ 3.04809828e-01,  2.56919971e-01,  3.05642087e-01],
       [ 2.92167592e-01,  3.01724175e-01,  2.72517625e-01],
       [ 2.15771580e-01,  3.78145121e-01,  2.83553282e-01],
       [ 3.23549625e-01,  3.25417275e-01,  2.22987278e-01],
       [ 2.41907008e-01,  3.07472681e-01,  3.19669997e-01],
       [ 2.66383990e-01,  3.07385437e-01,  2.93005227e-01],
       [ 3.15865119e-01,  2.90629666e-01,  2.60843505e-01],
       [ 2.62977895e-01,  3.03312840e-01,  3.00614891e-01],
       [ 2.94385970e-01,  2.98040343e-01,  2.73892468e-01],
       [ 2.87179554e-01,  2.87629570e-01,  2.91224793e-01],
       [ 2.87845899e-01,  2.66422583e-01,  3.12684841e-01],
       [ 2.69108096e-01,  2.97308191e-01,  3.00118416e-01],
       [ 2.64587785e-01,  3.03996208e-01,  2.98225625e-01],
       [ 2.49954457e-01,  3.27001176e-01,  2.91642180e-01],
       [ 2.86234522e-01,  2.67348023e-01,  3.13369284e-01],
       [ 3.13171867e-01,  2.95505919e-01,  2.58685076e-01],
       [ 2.71003943e-01,  3.26147843e-01,  2.70639145e-01],
       [ 3.00878240e-01,  2.96781422e-01,  2.68890371e-01],
       [ 2.91822562e-01,  2.63000443e-01,  3.12262713e-01],
       [ 3.17668081e-01,  2.56583158e-01,  2.93411157e-01],
       [ 3.09884037e-01,  2.85226930e-01,  2.71567383e-01],
       [ 3.34995731e-01,  3.45748048e-01,  1.97101881e-01],
       [ 2.99044225e-01,  2.81287451e-01,  2.85841044e-01],
       [ 3.07785063e-01,  2.46776224e-01,  3.13844419e-01],
       [ 3.14079346e-01,  2.74481681e-01,  2.78290552e-01],
       [ 3.50463066e-01,  4.00590304e-01,  1.45938888e-01],
       [ 3.20386435e-01,  1.81520032e-01,  3.82229472e-01],
       [ 3.00090678e-01,  2.72070711e-01,  2.94242357e-01],
       [ 2.59987036e-01,  3.00412786e-01,  3.06739159e-01],
       [ 2.71065064e-01,  3.03566845e-01,  2.91862723e-01],
       [ 1.44462377e-01,  3.53294249e-01,  3.99717980e-01],
       [ 2.05311199e-01,  3.63383850e-01,  3.08413492e-01],
       [ 3.07672571e-01,  2.70820337e-01,  2.88121097e-01],
       [ 3.39408128e-01,  3.15965141e-01,  2.17828328e-01],
       [ 2.85661152e-01,  2.60919401e-01,  3.21023978e-01],
       [ 3.11312551e-01,  2.78287373e-01,  2.77078815e-01],
       [ 5.89510016e-01,  4.51699229e-01, -1.56368378e-02],
       [ 3.13029996e-01,  2.86227455e-01,  2.67668296e-01],
       [ 3.36285144e-01,  2.66700898e-01,  2.65864280e-01],
       [ 2.49222931e-01,  2.84570340e-01,  3.35482958e-01],
       [ 3.17827027e-01,  2.72628913e-01,  2.76652587e-01],
       [ 3.48723260e-01,  2.30687700e-01,  2.92631387e-01],
       [ 3.23764662e-01,  3.43737542e-01,  2.07804454e-01],
       [ 2.86168088e-01,  3.03624510e-01,  2.76558837e-01],
       [ 2.71697740e-01,  2.92139017e-01,  3.02616825e-01],
       [ 3.47113966e-01,  2.68099612e-01,  2.55096581e-01],
       [ 4.06406750e-01,  5.17640639e-01,  4.28845432e-02],
       [ 3.01493102e-01,  3.04852021e-01,  2.60724812e-01],
       [ 2.65140067e-01,  2.97285710e-01,  3.04355978e-01],
       [ 4.92591400e-01,  2.14992971e-01,  2.03645413e-01],
       [ 4.44057773e-01,  5.93499166e-01, -1.30575174e-02],
       [ 2.60474624e-01,  3.67680045e-01,  2.45526597e-01],
       [ 2.50934673e-01,  3.21473213e-01,  2.95822661e-01],
       [ 2.88071663e-01,  2.01780268e-01,  3.91695552e-01],
       [ 2.36156346e-01,  4.01828009e-01,  2.43118447e-01],
       [ 3.90795431e-01,  1.72767199e-01,  3.23803173e-01],
       [ 2.37551802e-01,  1.98037458e-01,  4.65933997e-01],
       [ 2.36444774e-01,  3.19880333e-01,  3.13424587e-01],
       [ 2.63019765e-01,  3.50811120e-01,  2.56959604e-01],
       [ 2.59130925e-01,  2.66898929e-01,  3.43779411e-01],
       [ 3.35878092e-01,  2.60207862e-01,  2.72782606e-01],
       [ 2.23191257e-01,  3.13219293e-01,  3.35735739e-01],
       [ 2.96542466e-01,  2.85516221e-01,  2.84047501e-01],
       [ 2.47857082e-01,  1.75796122e-01,  4.87256285e-01],
       [ 4.72125977e-01,  3.52585616e-01,  1.01290163e-01],
       [ 2.50399870e-01,  2.67927652e-01,  3.52886820e-01],
       [ 1.50296760e-01,  3.93529320e-01,  3.50946424e-01],
       [ 3.58916506e-01,  2.73017020e-01,  2.40546955e-01],
       [ 2.81033926e-01,  3.15861195e-01,  2.70118288e-01],
       [ 3.42181472e-01,  2.65080554e-01,  2.62315671e-01],
       [ 2.99644813e-01,  2.67602490e-01,  2.99365552e-01],
       [ 2.81967791e-01,  3.08429895e-01,  2.76140485e-01],
       [ 2.77147058e-01,  2.68591056e-01,  3.21694186e-01],
       [ 3.16036514e-01,  2.72114654e-01,  2.78842995e-01],
       [ 2.98772159e-01,  2.65054867e-01,  3.02945721e-01],
       [ 2.60886416e-01,  3.06972044e-01,  2.99220978e-01],
       [ 2.85007406e-01,  3.17277074e-01,  2.64947362e-01],
       [ 2.67692126e-01,  2.97623529e-01,  3.01298086e-01],
       [ 2.39925478e-01,  4.16357341e-01,  2.28720395e-01],
       [ 2.58043945e-01,  2.57306195e-01,  3.56269806e-01],
       [ 2.96258967e-01,  2.58428915e-01,  3.12676953e-01],
       [ 3.11989341e-01,  3.02809841e-01,  2.52971314e-01],
       [ 3.14584699e-01,  2.56508717e-01,  2.96459875e-01],
       [ 2.67587326e-01,  2.79841287e-01,  3.19891971e-01],
       [ 2.83548786e-01,  2.76526899e-01,  3.06371332e-01],
       [ 3.14320636e-01,  2.89952779e-01,  2.62897315e-01],
       [ 2.68568549e-01,  3.03193059e-01,  2.94828960e-01],
       [ 2.92665013e-01,  2.71988274e-01,  3.01775536e-01],
       [ 2.68567063e-01,  3.19060588e-01,  2.79617262e-01],
       [ 3.16176012e-01,  2.83618388e-01,  2.67302711e-01],
       [ 3.13649812e-01,  2.90833355e-01,  2.62670959e-01],
       [ 2.67737164e-01,  2.85581257e-01,  3.13633665e-01],
       [ 2.79518740e-01,  2.90447272e-01,  2.96183528e-01],
       [ 3.00138328e-01,  2.90996383e-01,  2.75167114e-01],
       [ 2.83465803e-01,  2.90296310e-01,  2.92300449e-01],
       [ 2.89767882e-01,  2.73512382e-01,  3.03126005e-01],
       [ 3.49336395e-01,  2.40018866e-01,  2.81922782e-01],
       [ 3.12480070e-01,  2.80039036e-01,  2.74241591e-01],
       [ 2.81338678e-01,  2.82324972e-01,  3.02611435e-01],
       [ 2.28513771e-01,  2.69036667e-01,  3.78899108e-01],
       [ 3.12149618e-01,  2.42954570e-01,  3.13746189e-01],
       [ 1.05068669e-01,  1.47033057e-01,  9.30384078e-01],
       [ 2.71750353e-01,  3.04618596e-01,  2.90126297e-01],
       [ 3.90978979e-01,  2.54444028e-01,  2.33207884e-01],
       [ 2.67138102e-01,  3.02259053e-01,  2.97253136e-01],
       [ 3.18121176e-01,  2.47075512e-01,  3.03257346e-01],
       [ 3.59552248e-01,  2.55445608e-01,  2.57161804e-01],
       [ 3.31348008e-01,  3.50163440e-01,  1.96583700e-01],
       [ 2.67828477e-01,  2.86983328e-01,  3.12065625e-01],
       [ 2.45890464e-01,  3.00639923e-01,  3.22169662e-01],
       [ 2.99332262e-01,  3.07823709e-01,  2.59996510e-01],
       [ 2.68118781e-01,  2.81352186e-01,  3.17695029e-01],
       [ 1.56151773e-02,  4.20443609e-01,  5.58261193e-01],
       [ 3.22053555e-01,  2.79771718e-01,  2.65689280e-01],
       [ 3.11176949e-01,  2.73885482e-01,  2.81633451e-01],
       [ 3.09780673e-01,  3.17338987e-01,  2.41890861e-01],
       [ 2.97137166e-01,  2.92961753e-01,  2.76140440e-01],
       [ 2.40136668e-01,  2.66094994e-01,  3.67619903e-01],
       [ 2.91652734e-01,  1.93722035e-01,  3.98661924e-01],
       [ 3.04695959e-01,  2.81791462e-01,  2.79868373e-01],
       [ 3.05520491e-01,  3.11611099e-01,  2.50832118e-01],
       [ 2.78164783e-01,  2.94000925e-01,  2.94004506e-01],
       [ 3.07782224e-01,  3.49531649e-01,  2.16670266e-01],
       [ 2.34609956e-01,  3.26213382e-01,  3.09307906e-01],
       [ 2.97018201e-01,  2.39305866e-01,  3.33607633e-01],
       [ 2.03980295e-01,  2.86517256e-01,  3.90534316e-01],
       [ 2.05147188e-01,  2.02403115e-01,  5.11538252e-01],
       [ 3.22196864e-01,  2.49410049e-01,  2.96778889e-01],
       [ 2.66717800e-01,  2.61093405e-01,  3.41716393e-01],
       [ 2.40289958e-01,  4.05464465e-01,  2.36267775e-01],
       [ 1.59607129e-01,  3.85102458e-01,  3.46120037e-01],
       [ 1.26005437e-01,  4.42647816e-01,  3.41550791e-01],
       [ 3.50172658e-01,  1.08707687e-01,  4.61849243e-01],
       [ 2.83407962e-01,  3.45903705e-01,  2.41483106e-01],
       [ 3.15261268e-01,  2.52041771e-01,  3.00617095e-01],
       [ 2.33739314e-01,  2.97632760e-01,  3.39557782e-01],
       [ 3.35112085e-01,  3.88178755e-01,  1.65793069e-01],
       [ 2.93316574e-01,  3.29741785e-01,  2.46014305e-01],
       [ 2.70404175e-01,  3.07387317e-01,  2.88825967e-01],
       [ 3.34412271e-01,  2.52918605e-01,  2.81648593e-01],
       [ 2.72371227e-01,  4.85766642e-01,  1.55237125e-01],
       [-1.44815711e-01,  5.89528568e-01,  7.54133804e-01],
       [ 2.23040138e-01,  4.62018437e-01,  2.14509167e-01],
       [ 2.77296258e-01,  2.83868445e-01,  3.05230071e-01],
       [ 2.86824973e-01,  3.09781275e-01,  2.70106112e-01],
       [ 2.75366130e-01,  2.88353065e-01,  3.02628213e-01],
       [ 3.24183236e-01,  2.61601669e-01,  2.82003041e-01],
       [ 2.72798967e-01,  2.71600187e-01,  3.23122746e-01],
       [ 2.93098312e-01,  2.59687527e-01,  3.14563094e-01],
       [ 3.04590078e-01,  2.62511664e-01,  2.99843112e-01],
       [ 3.05864184e-01,  2.96117702e-01,  2.64838867e-01],
       [ 2.62361109e-01,  3.03437560e-01,  3.01149145e-01],
       [ 3.06418260e-01,  2.98251634e-01,  2.62308831e-01],
       [ 4.10008625e-01,  3.67903066e-01,  1.27465072e-01],
       [ 2.71713789e-01,  3.11688243e-01,  2.83355212e-01],
       [ 3.04164498e-01,  2.50061321e-01,  3.13843596e-01],
       [ 3.03011048e-01,  2.40088676e-01,  3.26368198e-01],
       [ 2.94064786e-01,  2.37812683e-01,  3.38551029e-01],
       [ 3.16544981e-01,  3.07126843e-01,  2.44969443e-01],
       [ 3.01311442e-01,  3.02119272e-01,  2.63440363e-01],
       [ 2.90046081e-01,  2.85386401e-01,  2.90607148e-01],
       [ 2.85451799e-01,  2.48062221e-01,  3.35867824e-01],
       [ 2.93818110e-01,  2.93686850e-01,  2.78652089e-01],
       [ 2.83609031e-01,  2.78328165e-01,  3.04417682e-01],
       [ 2.52380301e-01,  3.29620903e-01,  2.86614831e-01],
       [ 2.49391491e-01,  3.23101586e-01,  2.95935829e-01],
       [ 2.93195099e-01,  2.83533659e-01,  2.89337610e-01],
       [ 2.89240196e-01,  2.81420539e-01,  2.95450269e-01],
       [ 2.92820085e-01,  2.78704473e-01,  2.94632541e-01],
       [ 3.03879144e-01,  2.91548041e-01,  2.71073667e-01],
       [ 2.98471840e-01,  2.91915691e-01,  2.75871996e-01],
       [ 2.88160027e-01,  2.63706164e-01,  3.15312710e-01],
       [ 3.02930912e-01,  5.49382887e-01,  9.80565384e-02],
       [ 2.68461170e-01,  2.70589238e-01,  3.29018018e-01],
       [ 2.85441964e-01,  3.17315015e-01,  2.64492963e-01],
       [ 1.20754505e-01,  4.06863595e-01,  3.80709812e-01],
       [ 2.85994408e-01,  2.52882176e-01,  3.29717632e-01],
       [ 1.32222276e+00,  4.24655779e-01, -1.78312074e-01],
       [ 3.11619522e-01,  2.72960744e-01,  2.82151337e-01],
       [ 3.03244010e-01,  3.33738344e-01,  2.33594927e-01],
       [ 3.92126487e-01,  2.77667643e-01,  2.10690354e-01],
       [ 3.35030448e-01,  2.19409386e-01,  3.18323404e-01],
       [ 3.06618113e-01,  2.47443037e-01,  3.14278457e-01],
       [ 3.55949517e-01,  1.31508522e-01,  4.16835068e-01],
       [ 2.56729825e-01,  3.01689586e-01,  3.08992278e-01],
       [ 3.57310154e-01,  3.03557284e-01,  2.14166854e-01],
       [ 3.28889634e-01,  2.80684446e-01,  2.58681923e-01],
       [ 2.91267057e-01,  3.15577769e-01,  2.60499200e-01],
       [ 2.60344482e-01,  2.93241933e-01,  3.13692815e-01],
       [ 2.72623345e-01,  5.17493655e-01,  1.37851924e-01],
       [ 2.73270075e-01,  2.74998047e-01,  3.18915611e-01],
       [ 2.46830549e-01,  3.33344610e-01,  2.89085742e-01],
       [ 2.78208151e-01,  2.86532835e-01,  3.01526247e-01],
       [ 2.94291970e-01,  2.74806851e-01,  2.97183098e-01],
       [ 2.95219602e-01,  3.86538439e-01,  1.99317452e-01],
       [ 2.96694473e-01,  4.53260845e-01,  1.54035860e-01],
       [ 3.15586065e-01,  2.78037794e-01,  2.73329894e-01],
       [ 2.72625109e-01,  2.94786000e-01,  2.98961250e-01],
       [ 3.04217457e-01,  2.51045873e-01,  3.12693912e-01],
       [ 3.33795877e-01,  2.41817694e-01,  2.94090097e-01],
       [ 3.23668930e-01,  2.61947525e-01,  2.82122750e-01],
       [ 2.48450719e-01,  3.19743137e-01,  3.00178515e-01],
       [ 2.53996600e-01,  3.10353293e-01,  3.03307081e-01],
       [ 3.04709903e-01,  3.57564762e-01,  2.12972779e-01],
       [ 2.62053755e-01,  3.71155473e-01,  2.41211448e-01],
       [ 3.29938072e-01,  2.81808575e-01,  2.56675902e-01],
       [ 3.98256214e-01,  2.48992442e-01,  2.33043686e-01],
       [ 3.30626586e-01,  3.48174445e-01,  1.98709586e-01],
       [ 3.31129382e-01,  4.37015369e-01,  1.37072304e-01],
       [ 1.89967392e-01,  3.90844067e-01,  3.02597975e-01],
       [ 3.40322713e-01,  3.54873566e-01,  1.85887740e-01],
       [ 2.16805051e-01,  3.58803283e-01,  2.99178496e-01],
       [ 3.16039053e-01,  2.52026226e-01,  2.99877280e-01],
       [ 2.36798289e-01,  3.50689319e-01,  2.84188750e-01],
       [ 2.55961349e-01,  2.79980649e-01,  3.32752006e-01],
       [ 2.81211147e-01,  2.42843455e-01,  3.46738131e-01],
       [ 2.68273576e-01,  2.13279771e-01,  4.00335029e-01],
       [ 3.41183683e-01,  2.40822561e-01,  2.88373662e-01],
       [ 1.87891760e-01,  1.84699571e-01,  5.77835431e-01],
       [ 2.26763551e-01,  2.95470075e-01,  3.50414350e-01],
       [ 3.03509642e-01,  2.77377537e-01,  2.85448274e-01],
       [ 3.07231896e-01,  2.78965930e-01,  2.80268471e-01],
       [ 2.78656062e-01,  3.04168876e-01,  2.83518237e-01],
       [ 2.94605015e-01,  3.05104404e-01,  2.66987157e-01],
       [ 3.23184796e-01,  2.74023906e-01,  2.70323656e-01],
       [ 3.31257553e-01,  2.87839496e-01,  2.49800889e-01],
       [ 3.05419071e-01,  3.05817790e-01,  2.56197596e-01],
       [ 2.57060930e-01,  2.98563389e-01,  3.11813236e-01],
       [ 3.06259148e-01,  2.57408559e-01,  3.03665213e-01],
       [ 2.77541328e-01,  3.13989281e-01,  2.75311194e-01],
       [ 2.45787651e-01,  3.17458517e-01,  3.05324077e-01],
       [ 2.94772485e-01,  2.53060701e-01,  3.20178976e-01],
       [ 3.15892393e-01,  2.97846521e-01,  2.54037256e-01],
       [ 3.21512945e-01,  3.20553968e-01,  2.28851147e-01],
       [ 2.29275652e-01,  3.36881639e-01,  3.05147078e-01],
       [ 3.06175174e-01,  3.17019916e-01,  2.45406895e-01],
       [ 3.16638045e-01,  2.64352593e-01,  2.86228212e-01],
       [ 3.05295496e-01,  2.87083580e-01,  2.74072266e-01],
       [ 2.51858701e-01,  3.33362598e-01,  2.83721267e-01],
       [ 2.88126019e-01,  2.75252180e-01,  3.02980637e-01],
       [ 2.91647440e-01,  2.85724169e-01,  2.88668987e-01],
       [ 2.77034509e-01,  2.62237827e-01,  3.28871074e-01],
       [ 2.94012406e-01,  2.55995165e-01,  3.17694255e-01],
       [ 3.06785333e-01,  2.85112490e-01,  2.74594133e-01],
       [ 2.92543111e-01,  2.92966692e-01,  2.80600960e-01],
       [ 2.85926813e-01,  2.75666065e-01,  3.04810989e-01],
       [ 2.82235564e-01,  3.17123181e-01,  2.67780460e-01],
       [ 2.82332203e-01,  2.84166297e-01,  2.99684295e-01],
       [ 3.32662516e-01,  2.74989044e-01,  2.60875580e-01],
       [-7.71503034e-02,  4.64001482e-01,  7.38779849e-01],
       [ 2.29553787e-01,  2.59027619e-01,  3.89984610e-01],
       [ 3.59865263e-01,  2.96274194e-01,  2.18522461e-01],
       [ 1.88262136e-01,  3.16575568e-01,  3.77152509e-01],
       [ 4.29121822e-01,  2.06277582e-01,  2.54142178e-01],
       [ 2.89680593e-01,  2.59194245e-01,  3.18681865e-01],
       [ 2.83739006e-01,  2.91546025e-01,  2.90772420e-01],
       [ 1.49673313e-01,  3.30437953e-01,  4.17699544e-01],
       [ 1.55343671e-01,  3.89325726e-01,  3.47955501e-01],
       [ 2.90677266e-01,  2.74780795e-01,  3.00866645e-01],
       [ 3.71742867e-01,  3.46638692e-01,  1.68628408e-01],
       [ 3.17408164e-01,  2.66266874e-01,  2.83522011e-01],
       [ 3.54793182e-01,  2.50198989e-01,  2.66501142e-01],
       [ 3.26125362e-01,  2.55040782e-01,  2.87051705e-01],
       [ 2.75514031e-01,  3.36598402e-01,  2.56917536e-01],
       [ 2.98821673e-01,  2.58451513e-01,  3.10025838e-01],
       [ 4.04178971e-01,  3.11850120e-01,  1.73116904e-01],
       [ 2.81618359e-01,  3.07560386e-01,  2.77310324e-01],
       [ 2.78562255e-01,  2.71826849e-01,  3.16647439e-01],
       [ 2.86168992e-01,  2.87627325e-01,  2.92246522e-01],
       [ 3.08728952e-01,  2.77670933e-01,  2.80141502e-01],
       [ 2.67673189e-01,  1.83048603e-01,  4.45957574e-01],
       [ 5.70275557e-01,  1.76508190e-01,  1.99979839e-01],
       [ 3.74358610e-01,  2.88036425e-01,  2.14631869e-01],
       [ 3.10307256e-01,  2.57137250e-01,  2.99956107e-01],
       [ 3.12837375e-01,  2.37130086e-01,  3.19685906e-01],
       [ 3.00153404e-01,  3.18998781e-01,  2.49133305e-01],
       [ 3.06694035e-01,  2.82866156e-01,  2.76895625e-01],
       [ 3.36285370e-01,  3.04802730e-01,  2.30076492e-01],
       [ 3.36024041e-01,  2.98106646e-01,  2.36274640e-01],
       [ 3.83935406e-01,  2.81885995e-01,  2.12930955e-01],
       [ 2.88676841e-01,  3.08471750e-01,  2.69533165e-01],
       [ 2.52584272e-01,  3.48750826e-01,  2.69252580e-01],
       [ 2.37773668e-01,  3.85674085e-01,  2.53905572e-01],
       [ 3.31170815e-01,  2.10665492e-01,  3.32635032e-01],
       [ 4.10537933e-01,  2.08788749e-01,  2.65262750e-01],
       [ 3.00274980e-01,  2.66899003e-01,  2.99479371e-01],
       [ 4.05526783e-01,  2.07611857e-01,  2.70424698e-01],
       [ 2.58253900e-01,  2.36798705e-01,  3.81466150e-01],
       [ 2.56031103e-01,  3.23504126e-01,  2.88460258e-01],
       [ 2.96025733e-01,  3.12640276e-01,  2.58681166e-01],
       [ 3.18362087e-01,  2.66812054e-01,  2.82065023e-01],
       [ 2.52764172e-01,  2.91902029e-01,  3.23533248e-01],
       [ 2.89202639e-01,  3.15980610e-01,  2.62098421e-01],
       [ 3.44872390e-01,  2.42474032e-01,  2.83269625e-01],
       [ 2.26800537e-01,  2.90478850e-01,  3.55937710e-01],
       [ 3.28072337e-01,  3.48792529e-01,  2.00292299e-01],
       [ 3.55113856e-01,  2.39828149e-01,  2.77058435e-01],
       [ 2.81623229e-01,  3.02092673e-01,  2.82541020e-01],
       [ 2.84209236e-01,  2.92426983e-01,  2.89419126e-01],
       [ 2.90692578e-01,  2.86867689e-01,  2.88471526e-01],
       [ 3.34269521e-01,  2.64393217e-01,  2.69970713e-01],
       [ 2.68967027e-01,  3.35867408e-01,  2.63977599e-01],
       [ 2.61420608e-01,  3.23306766e-01,  2.83005304e-01],
       [ 2.62559591e-01,  3.05674997e-01,  2.98718349e-01],
       [ 2.96900696e-01,  3.19882517e-01,  2.51347078e-01],
       [ 3.21598646e-01,  2.88468592e-01,  2.57723873e-01],
       [ 2.79612625e-01,  3.04690934e-01,  2.82052994e-01],
       [ 3.10317528e-01,  2.48691875e-01,  3.09165734e-01],
       [ 2.74917724e-01,  2.91999485e-01,  2.99380867e-01],
       [ 2.49020793e-01,  3.22985679e-01,  2.96447433e-01],
       [ 3.16210733e-01,  2.22846526e-01,  3.33050958e-01],
       [ 2.09786163e-01,  3.55144589e-01,  3.10649719e-01],
       [ 3.80420015e-01,  2.87136401e-01,  2.10869917e-01],
       [ 3.00561512e-01,  2.32315008e-01,  3.38117825e-01],
       [ 3.13987182e-01,  2.89722280e-01,  2.63422934e-01],
       [ 2.98236752e-01,  2.82269375e-01,  2.85641944e-01],
       [ 3.61565973e-01,  2.79326889e-01,  2.32495804e-01],
       [ 2.76149173e-01,  2.85356351e-01,  3.04892957e-01],
       [ 3.38405274e-01,  2.81295038e-01,  2.49811518e-01],
       [ 2.87551510e-01,  2.52752277e-01,  3.28187041e-01],
       [ 3.17079729e-01,  2.74439117e-01,  2.75529546e-01],
       [ 2.92652600e-01,  2.87625847e-01,  2.85768925e-01],
       [ 2.88032109e-01,  2.94908967e-01,  2.83144824e-01],
       [ 2.95812979e-01,  3.02339980e-01,  2.68432861e-01],
       [ 3.09820732e-01,  2.41295831e-01,  3.17975472e-01],
       [ 2.90833736e-01,  2.75097322e-01,  3.00376548e-01],
       [ 2.84127878e-01,  2.86891504e-01,  2.95062008e-01],
       [ 4.93106245e-01,  2.43666773e-01,  1.76236628e-01],
       [ 3.15133262e-01,  2.42100550e-01,  3.11729224e-01],
       [ 3.48770868e-01,  2.02142936e-01,  3.25819451e-01],
       [ 3.29689034e-01,  2.41690786e-01,  2.98080528e-01],
       [ 3.51835154e-01,  1.53959445e-01,  3.87176247e-01],
       [ 2.98199118e-01,  3.01152797e-01,  2.67282872e-01],
       [ 2.65595322e-01,  2.78835642e-01,  3.23167802e-01],
       [-9.51175042e-02,  4.15624673e-01,  9.03359456e-01],
       [ 7.05132277e-01,  1.94981717e-01,  1.24997610e-01],
       [ 3.18414519e-01,  2.89702222e-01,  2.59415002e-01],
       [ 2.21650381e-01,  3.48465699e-01,  3.03030297e-01],
       [ 2.87019901e-01,  3.18180590e-01,  2.62187227e-01],
       [ 3.18969508e-01,  2.34054956e-01,  3.17063019e-01],
       [ 3.64712841e-01,  2.44875302e-01,  2.63605575e-01],
       [ 2.70180455e-01,  2.40424034e-01,  3.62398155e-01],
       [ 3.02318990e-01,  2.88962089e-01,  2.75065918e-01],
       [ 2.64190105e-01,  2.98831902e-01,  3.03809738e-01],
       [ 2.88396911e-01,  2.37036650e-01,  3.45694252e-01],
       [ 3.10265719e-01,  2.78061255e-01,  2.78292739e-01],
       [ 2.98157019e-01,  2.87722885e-01,  2.80285091e-01],
       [ 2.78167520e-01,  2.78477350e-01,  3.09958207e-01],
       [ 2.92913096e-01,  3.07689134e-01,  2.66189193e-01],
       [ 2.95522369e-01,  3.02326396e-01,  2.68723123e-01],
       [ 3.97384418e-01,  4.54823269e-01,  8.12716444e-02],
       [ 5.66920657e-01, -7.19607294e-02,  5.87514277e-01],
       [ 3.14676468e-01,  1.99429846e-01,  3.64212840e-01],
       [ 2.63326079e-01,  2.69549467e-01,  3.35951983e-01],
       [ 3.10205802e-01,  3.01760504e-01,  2.55556784e-01],
       [ 3.18892771e-01,  3.08472252e-01,  2.41694107e-01],
       [ 3.53425083e-01,  2.36252092e-01,  2.82362269e-01],
       [ 2.94360937e-01,  2.44477030e-01,  3.30406400e-01],
       [ 3.10285184e-01,  2.35834428e-01,  3.23782678e-01],
       [ 2.98837962e-01,  2.39748328e-01,  3.31152317e-01],
       [ 3.29668997e-01,  2.58743738e-01,  2.79905926e-01],
       [ 2.58827590e-01,  2.64182387e-01,  3.47263566e-01],
       [ 3.66563597e-01,  2.58952468e-01,  2.47919215e-01],
       [ 2.35581361e-01,  3.16882364e-01,  3.17393547e-01],
       [ 3.24292027e-01,  2.78228313e-01,  2.65174079e-01],
       [ 2.76504929e-01,  2.83727955e-01,  3.06208056e-01],
       [ 1.84850238e-01,  3.21132721e-01,  3.76768697e-01],
       [ 2.78782831e-01,  3.34929528e-01,  2.55213042e-01],
       [ 3.05524844e-01,  2.67905946e-01,  2.93231687e-01],
       [ 2.90130214e-01,  2.75664226e-01,  3.00500441e-01],
       [ 3.50822401e-01,  2.36011104e-01,  2.84922412e-01],
       [ 2.79110883e-01,  3.02630934e-01,  2.84546182e-01],
       [ 3.52889516e-01,  2.92568551e-01,  2.27366630e-01],
       [ 2.88224294e-01,  2.94807723e-01,  2.83053499e-01],
       [ 2.81377287e-01,  3.63347505e-01,  2.29186572e-01],
       [ 2.58245904e-01,  2.91517398e-01,  3.17803726e-01],
       [ 2.92662489e-01,  2.90206405e-01,  2.83198285e-01],
       [ 3.01967161e-01,  2.67248532e-01,  2.97426304e-01],
       [ 2.94085032e-01,  2.83436652e-01,  2.88552842e-01],
       [ 2.89938850e-01,  2.77836500e-01,  2.98435648e-01],
       [ 2.79496292e-01,  2.71079938e-01,  3.16457836e-01],
       [ 3.31651337e-01,  2.97352679e-01,  2.40670301e-01],
       [ 3.26217295e-01,  2.23432636e-01,  3.22227659e-01],
       [ 3.23661974e-01,  2.98791715e-01,  2.46271307e-01],
       [ 3.00732748e-01,  3.30579074e-01,  2.38525624e-01],
       [ 2.91599723e-01,  2.63446472e-01,  3.12008412e-01],
       [ 2.79691684e-01,  2.92900398e-01,  2.93539152e-01],
       [ 2.47335507e-01,  3.07172920e-01,  3.13836944e-01],
       [ 3.14554668e-01,  2.69973013e-01,  2.82413877e-01],
       [ 3.06491061e-01,  3.11042708e-01,  2.50461106e-01],
       [ 3.27946821e-01,  3.25192588e-01,  2.19484727e-01],
       [ 3.60611031e-01,  3.31088682e-01,  1.88818596e-01],
       [ 3.77031064e-01,  3.07157277e-01,  1.96133370e-01],
       [ 3.21213385e-01,  3.10240440e-01,  2.38095982e-01],
       [ 2.90067366e-01,  2.93441893e-01,  2.82569780e-01],
       [ 2.88079366e-01,  2.94755820e-01,  2.83248050e-01],
       [ 2.65762734e-01,  2.51973959e-01,  3.53528568e-01],
       [ 2.92430416e-01,  2.98128900e-01,  2.75701420e-01],
       [ 2.55101736e-01,  3.26309083e-01,  2.86816104e-01],
       [ 2.51040545e-01,  3.10877914e-01,  3.06017779e-01],
       [ 3.02813383e-01,  3.01578240e-01,  2.62541814e-01],
       [ 2.94107537e-01,  2.73838441e-01,  2.98377059e-01],
       [ 2.86073740e-01,  2.82658225e-01,  2.97396515e-01],
       [ 2.77042119e-01,  2.95098453e-01,  2.94062871e-01],
       [ 2.79712051e-01,  3.05990894e-01,  2.80706561e-01],
       [ 3.03176369e-01,  2.91139593e-01,  2.72133622e-01],
       [ 2.99104149e-01,  2.68784652e-01,  2.98659517e-01],
       [ 2.11457036e-01,  3.25241029e-01,  3.37667505e-01],
       [ 2.28696370e-01,  2.95401987e-01,  3.48107631e-01],
       [ 2.90189214e-01,  3.09288075e-01,  2.67312774e-01],
       [ 3.47418825e-01,  3.44804734e-01,  1.88101578e-01],
       [ 3.82420318e-01,  3.74666333e-01,  1.40961383e-01],
       [ 3.17717951e-01,  2.91877515e-01,  2.57983011e-01],
       [ 3.26874277e-01,  2.56064441e-01,  2.85277193e-01],
       [ 3.26994260e-01,  2.72007286e-01,  2.68872727e-01],
       [ 2.20678933e-01,  2.55738070e-01,  4.06291115e-01],
       [ 3.42951268e-01,  3.48285553e-01,  1.88871634e-01],
       [ 2.66552631e-01,  2.88828970e-01,  3.11518923e-01],
       [ 2.32552352e-01,  3.03463001e-01,  3.34745944e-01],
       [ 3.54922140e-01,  2.75814057e-01,  2.41158642e-01],
       [ 2.59605727e-01,  3.18567812e-01,  2.89355981e-01],
       [ 3.09330759e-01,  2.81873740e-01,  2.75383192e-01],
       [ 3.13592480e-01,  3.22052151e-01,  2.34418510e-01],
       [ 2.85153516e-01,  3.09317872e-01,  2.72169401e-01],
       [ 2.89291387e-01,  2.67793080e-01,  3.09701272e-01],
       [ 3.28553773e-01,  3.19924327e-01,  2.23427214e-01],
       [ 3.06117085e-01,  2.73656684e-01,  2.86713582e-01],
       [ 2.95453953e-01,  2.82054121e-01,  2.88595090e-01],
       [ 2.87748104e-01,  3.10981970e-01,  2.68093311e-01],
       [ 2.83165849e-01,  2.92590347e-01,  2.90311085e-01],
       [ 2.58244010e-01,  2.96977083e-01,  3.12141324e-01],
       [ 2.65483119e-01,  2.71694759e-01,  3.31118676e-01],
       [ 2.31372528e-01,  2.86934253e-01,  3.54252160e-01],
       [ 3.45447272e-01,  2.92970164e-01,  2.33067343e-01],
       [ 2.85780777e-01,  2.34220140e-01,  3.52046276e-01],
       [ 2.43495314e-01,  3.13858016e-01,  3.11430890e-01],
       [ 3.11866737e-01,  2.30688034e-01,  3.28180831e-01],
       [ 2.68511972e-01,  3.24928667e-01,  2.74252810e-01],
       [ 2.48593759e-01,  3.48061737e-01,  2.73983942e-01],
       [ 3.04003284e-01,  2.24228308e-01,  3.44231320e-01],
       [ 2.86360953e-01,  2.34207375e-01,  3.51408534e-01],
       [ 2.73223899e-01,  2.95702633e-01,  2.97414454e-01],
       [ 2.48647470e-01,  3.71469874e-01,  2.54201817e-01],
       [ 2.65057014e-01,  2.37226158e-01,  3.72541932e-01],
       [ 2.53951043e-01,  3.12862281e-01,  3.00889712e-01],
       [ 2.66221115e-01,  2.87597187e-01,  3.13163281e-01],
       [ 2.83375611e-01,  3.40889373e-01,  2.45729410e-01],
       [ 2.54411129e-01,  1.81454808e-01,  4.67657277e-01],
       [ 4.27192725e-01,  2.60227149e-01,  2.01962236e-01],
       [ 2.69026704e-01,  3.02128029e-01,  2.95400672e-01],
       [ 3.50874870e-01,  1.93005470e-01,  3.35145836e-01],
       [ 2.75512982e-01,  2.92936404e-01,  2.97814057e-01],
       [ 2.31312606e-01,  3.27704432e-01,  3.11614534e-01],
       [ 3.20758988e-01,  2.27969571e-01,  3.22339175e-01],
       [ 2.72059569e-01,  2.39757499e-01,  3.61011169e-01],
       [ 2.68228713e-01,  2.79505874e-01,  3.19549840e-01],
       [ 2.56884083e-01,  3.49372638e-01,  2.64329821e-01],
       [ 2.86481528e-01,  2.88870874e-01,  2.90680685e-01],
       [ 2.87486882e-01,  2.70894805e-01,  3.08250255e-01],
       [ 3.03378155e-01,  2.68859633e-01,  2.94342429e-01],
       [ 3.19149089e-01,  2.73350321e-01,  2.74701530e-01],
       [ 2.65951695e-01,  3.36161435e-01,  2.66722794e-01],
       [ 2.56030403e-01,  2.44992293e-01,  3.73784513e-01],
       [ 2.08337485e-01,  4.76837573e-01,  2.19880829e-01],
       [ 3.27855608e-01,  2.39203134e-01,  3.02571671e-01],
       [ 3.70040120e-01,  2.28361599e-01,  2.76565125e-01],
       [ 2.53566744e-01,  2.62437495e-01,  3.55529209e-01],
       [ 2.82132020e-01,  2.85896459e-01,  2.98117895e-01],
       [ 3.05640884e-01,  3.21035384e-01,  2.42355533e-01],
       [ 2.86685739e-01,  2.82695657e-01,  2.96734646e-01],
       [ 2.50452257e-01,  3.13617665e-01,  3.03958324e-01],
       [ 3.19859709e-01,  2.03290978e-01,  3.53579594e-01],
       [ 3.33742912e-01,  1.92456133e-01,  3.53039655e-01],
       [ 1.75478831e-01,  4.08575900e-01,  3.05285737e-01],
       [ 1.73997151e-01,  4.58515401e-01,  2.69116598e-01],
       [ 2.81753075e-01,  2.35338074e-01,  3.55242542e-01],
       [ 2.54876991e-01,  2.74043620e-01,  3.40604210e-01],
       [ 3.14254768e-01,  2.53652891e-01,  2.99852215e-01],
       [ 2.34729878e-01,  3.63973813e-01,  2.74867975e-01],
       [ 3.28627814e-01,  2.84626200e-01,  2.55137203e-01],
       [ 3.07972637e-01,  2.96555257e-01,  2.62467782e-01],
       [ 2.86048873e-01,  2.87703274e-01,  2.92291372e-01],
       [ 2.93929141e-01,  2.90893233e-01,  2.81278572e-01],
       [ 2.95637466e-01,  3.09679226e-01,  2.61759209e-01],
       [ 2.94646645e-01,  2.70335170e-01,  3.01508199e-01],
       [ 2.98765079e-01,  2.64061697e-01,  3.04014652e-01],
       [ 3.18718284e-01,  2.85573764e-01,  2.63088717e-01],
       [ 3.06918009e-01,  2.66494321e-01,  2.93345791e-01],
       [ 2.91954796e-01,  2.46592311e-01,  3.30530403e-01],
       [ 3.08200053e-01,  4.10418433e-01,  1.71870081e-01],
       [ 3.41489071e-01,  2.92646781e-01,  2.36643808e-01],
       [ 2.52831856e-01,  2.65825362e-01,  3.52430998e-01],
       [ 2.79006241e-01,  2.83156503e-01,  3.04178052e-01],
       [ 3.41158316e-01,  2.15766796e-01,  3.16720075e-01],
       [ 3.46402991e-01,  2.25472837e-01,  3.00581921e-01],
       [ 2.40600863e-01,  2.85781334e-01,  3.44314009e-01],
       [ 2.55986899e-01,  3.45609203e-01,  2.68500031e-01],
       [ 2.77797841e-01,  3.32499277e-01,  2.58287995e-01],
       [ 2.64912877e-01,  2.54709465e-01,  3.51263154e-01],
       [ 2.51536296e-01,  3.06438729e-01,  3.09905514e-01],
       [ 3.28726269e-01,  2.27807497e-01,  3.14650615e-01],
       [ 2.57158554e-01,  2.77255147e-01,  3.34387886e-01],
       [ 3.73586049e-01,  2.85228027e-01,  2.17728785e-01],
       [ 2.85934614e-01,  2.62300019e-01,  3.19205199e-01],
       [ 2.78831582e-01,  3.01316415e-01,  2.86105387e-01],
       [ 3.02125063e-01,  2.50353029e-01,  3.15599981e-01],
       [ 2.84999290e-01,  2.70937601e-01,  3.10796026e-01],
       [ 2.97250436e-01,  2.99829201e-01,  2.69437491e-01],
       [ 2.20789047e-01,  3.27628051e-01,  3.23956557e-01],
       [ 2.82565570e-01,  2.86707673e-01,  2.96845646e-01],
       [ 2.88214688e-01,  3.05534181e-01,  2.72742518e-01],
       [ 2.94437128e-01,  2.80106890e-01,  2.91581021e-01],
       [ 3.05151877e-01,  2.67288106e-01,  2.94243131e-01],
       [ 2.67058316e-01,  2.99117609e-01,  3.00468719e-01],
       [ 3.05505286e-01,  2.54440079e-01,  3.07650034e-01],
       [ 2.88286566e-01,  3.11041276e-01,  2.67517988e-01],
       [ 3.07484483e-01,  2.26395485e-01,  3.37879134e-01],
       [ 3.64482868e-01,  2.67808334e-01,  2.41009918e-01],
       [ 3.81631399e-01,  2.45396115e-01,  2.49349724e-01],
       [ 2.42730837e-01,  3.20372273e-01,  3.05868972e-01],
       [ 3.27954397e-01,  3.17224033e-01,  2.26239714e-01],
       [ 2.97972303e-01,  2.81540524e-01,  2.86635108e-01],
       [ 3.46477252e-01,  2.95871682e-01,  2.29606036e-01],
       [ 3.43881623e-01,  2.83587447e-01,  2.43007179e-01],
       [ 2.97025221e-01,  3.05127808e-01,  2.64665854e-01],
       [ 2.24331371e-01,  3.45355903e-01,  3.02843059e-01],
       [ 4.15519597e-01,  2.09947808e-01,  2.60225505e-01],
       [ 2.62140195e-01,  3.12717813e-01,  2.92287990e-01],
       [ 2.57339798e-01,  3.03186988e-01,  3.06814814e-01],
       [ 2.76558874e-01,  2.88985539e-01,  3.00734090e-01],
       [ 2.48544139e-01,  3.61254239e-01,  2.62730079e-01],
       [ 3.98866154e-01,  1.96545152e-01,  2.88212181e-01],
       [ 2.92648531e-01,  3.45521370e-01,  2.33297870e-01],
       [ 2.03487507e-01,  2.98943435e-01,  3.76506958e-01],
       [ 2.42269231e-01,  3.34042506e-01,  2.93368620e-01],
       [ 3.17199995e-01,  3.52692994e-01,  2.06190819e-01],
       [ 2.94533919e-01,  2.08079895e-01,  3.75464041e-01],
       [ 2.90919802e-01,  3.06784648e-01,  2.68946083e-01],
       [ 3.11060014e-01,  2.79430491e-01,  2.76177764e-01],
       [ 3.57485532e-01,  2.57152309e-01,  2.57178714e-01],
       [ 2.67212549e-01,  3.11469008e-01,  2.88192306e-01],
       [ 2.77504155e-01,  3.07700590e-01,  2.81289256e-01],
       [ 2.75889467e-01,  3.04020166e-01,  2.86465389e-01],
       [ 2.65959276e-01,  3.00643393e-01,  3.00106424e-01],
       [ 2.53403706e-01,  2.91307693e-01,  3.23440177e-01],
       [ 3.23783979e-01,  2.76023576e-01,  2.67799208e-01],
       [ 1.21064183e-02,  4.84028188e-01,  4.92084505e-01],
       [ 1.81399731e-01,  3.97061108e-01,  3.07666846e-01],
       [ 3.68706947e-01,  3.35958403e-01,  1.78992486e-01],
       [ 2.58007930e-01,  2.62435001e-01,  3.50258747e-01],
       [ 3.01255296e-01,  3.03856175e-01,  2.61871945e-01],
       [ 2.85307385e-01,  2.74912872e-01,  3.06246205e-01],
       [ 2.78244675e-01,  3.37407635e-01,  2.53581640e-01],
       [ 3.40282934e-01,  2.47659878e-01,  2.81873622e-01],
       [ 2.33061919e-01,  3.34929847e-01,  3.02716725e-01],
       [ 1.98480786e-01,  3.47345642e-01,  3.31714543e-01],
       [ 4.60650976e-01,  1.30557235e-02,  5.15056825e-01],
       [ 1.07770955e-02,  4.31381427e-01,  5.54893661e-01],
       [ 3.11500848e-01,  2.94720877e-01,  2.60951052e-01],
       [ 3.07625230e-01,  2.91580617e-01,  2.67524836e-01],
       [ 3.01099785e-01,  2.77161428e-01,  2.88013012e-01],
       [ 3.52964107e-01,  2.47509379e-01,  2.70849716e-01],
       [ 3.16773268e-01,  2.90688088e-01,  2.59963507e-01],
       [ 2.99412425e-01,  3.09752980e-01,  2.58150099e-01],
       [ 2.97615451e-01,  3.10452280e-01,  2.59189226e-01],
       [ 2.70107736e-01,  2.89736319e-01,  3.06763960e-01],
       [ 2.62133091e-01,  2.90309659e-01,  3.14784168e-01],
       [ 2.96872041e-01,  2.93298807e-01,  2.76068845e-01],
       [ 2.98058390e-01,  2.89785563e-01,  2.78351050e-01],
       [ 3.01718723e-01,  2.68001565e-01,  2.96880617e-01],
       [ 3.05009734e-01,  2.76127332e-01,  2.85265707e-01],
       [ 2.56443708e-01,  3.14409562e-01,  2.96699091e-01],
       [ 5.05688140e-01, -4.07345554e-03,  5.02497047e-01],
       [ 2.71320245e-01,  2.35800473e-01,  3.66821058e-01],
       [ 2.18117782e-01,  4.47961197e-01,  2.28639099e-01],
       [ 2.77850558e-01,  2.93576362e-01,  2.94752729e-01],
       [ 2.95663470e-01,  2.82147987e-01,  2.88293257e-01],
       [ 2.58984728e-01,  3.51726191e-01,  2.60202140e-01],
       [ 2.52337204e-01,  2.55705257e-01,  3.65079821e-01],
       [ 2.92743709e-01,  2.56400083e-01,  3.18569546e-01],
       [ 3.08491163e-01,  2.93781571e-01,  2.64615967e-01],
       [ 2.68584104e-01,  3.36359463e-01,  2.63924115e-01],
       [ 3.26436911e-01,  2.81422612e-01,  2.60147724e-01],
       [ 3.33731072e-01,  3.14613249e-01,  2.23652430e-01],
       [ 2.47311474e-01,  3.31553031e-01,  2.90228766e-01],
       [ 2.62948073e-01,  2.53472720e-01,  3.55039610e-01],
       [ 2.92328647e-01,  2.63619046e-01,  3.11066316e-01],
       [ 2.92656226e-01,  2.75773708e-01,  2.97825813e-01],
       [ 2.57919648e-01,  2.91468274e-01,  3.18217053e-01],
       [ 2.63692502e-01,  3.04884604e-01,  2.98295893e-01],
       [ 2.98596373e-01,  2.82583840e-01,  2.84974414e-01],
       [ 3.03415221e-01,  2.49507550e-01,  3.15225961e-01],
       [ 2.91490776e-01,  2.97261148e-01,  2.77453220e-01],
       [ 2.69530523e-01,  2.92783512e-01,  3.04253311e-01],
       [ 2.96820705e-01,  2.82427744e-01,  2.86871027e-01],
       [ 3.07584554e-01,  2.63859337e-01,  2.95463730e-01],
       [ 3.06185802e-01,  2.83553486e-01,  2.76698453e-01],
       [ 3.08995766e-01,  2.50860515e-01,  3.08088289e-01],
       [ 2.85657679e-01,  2.70597520e-01,  3.10472137e-01],
       [ 2.97117528e-01,  2.96774970e-01,  2.72478530e-01],
       [ 2.47583710e-01,  3.50537688e-01,  2.72875338e-01],
       [ 2.53602367e-01,  3.42963843e-01,  2.73269848e-01],
       [ 3.24279423e-01,  3.11038863e-01,  2.34742963e-01],
       [ 3.12005203e-01,  2.26809368e-01,  3.32645602e-01],
       [ 2.84395189e-01,  3.14415690e-01,  2.68167624e-01],
       [ 2.41007865e-01,  3.61405316e-01,  2.70409881e-01],
       [ 2.50794923e-01,  2.39200090e-01,  3.87779114e-01],
       [ 2.86905165e-01,  2.81595961e-01,  2.97639981e-01],
       [ 2.95634283e-01,  2.37549468e-01,  3.37167502e-01],
       [ 5.69243160e-02,  4.34727882e-01,  4.58156017e-01],
       [ 2.84486986e-01,  2.60183883e-01,  3.23096169e-01],
       [ 3.26093923e-01,  2.96269774e-01,  2.46460756e-01],
       [ 2.81335923e-01,  3.17877617e-01,  2.67967254e-01],
       [ 1.66382646e-01,  4.28964378e-01,  3.00039750e-01],
       [ 3.85143540e-01,  3.38883254e-01,  1.65023318e-01],
       [ 4.42762263e-01,  8.26062807e-02,  4.06238742e-01],
       [ 3.19602202e-01,  2.84367874e-01,  2.63449147e-01],
       [ 1.85735571e-01,  3.78323478e-01,  3.18640174e-01],
       [ 3.15572436e-01,  2.30526662e-01,  3.24578708e-01],
       [ 2.92704694e-01,  2.49995586e-01,  3.25824632e-01],
       [ 3.67019904e-01,  2.65189768e-01,  2.41486462e-01],
       [ 2.54008672e-01,  5.61291364e-01,  1.31763917e-01],
       [ 2.61413296e-01,  3.28877851e-01,  2.77874669e-01],
       [ 2.91979396e-01,  2.29406234e-01,  3.51022536e-01],
       [ 2.95284097e-01,  2.64273492e-01,  3.07321791e-01],
       [ 2.80029913e-01,  2.85379510e-01,  3.00817768e-01],
       [ 3.24763434e-01,  2.72371465e-01,  2.70531346e-01],
       [ 2.91764983e-01,  2.53752019e-01,  3.22563817e-01],
       [ 1.95492804e-01,  4.65213294e-01,  2.40733587e-01],
       [ 2.64601752e-01,  3.69937024e-01,  2.39723750e-01],
       [ 4.01765417e-03,  4.96348440e-01,  4.95648779e-01],
       [ 3.37866516e-01,  3.13802522e-01,  2.20935209e-01],
       [ 1.76835073e-01,  5.84247750e-01,  1.92731332e-01],
       [ 2.79916046e-01,  3.26737441e-01,  2.61336910e-01],
       [ 2.44702073e-01,  2.89871003e-01,  3.34973782e-01],
       [ 2.91300657e-01,  2.96930573e-01,  2.77958600e-01],
       [ 3.18211953e-01,  2.48391775e-01,  3.01725456e-01],
       [ 3.21724336e-01,  2.63126265e-01,  2.82714726e-01],
       [ 3.35636059e-01,  1.96000391e-01,  3.46505964e-01],
       [ 1.96709343e-01,  3.80893186e-01,  3.03105930e-01],
       [ 2.56854362e-01,  4.77647458e-01,  1.73334039e-01],
       [ 2.35583028e-01,  4.24807308e-01,  2.27021203e-01],
       [ 2.22781223e-01,  4.25247640e-01,  2.39592431e-01],
       [ 2.33441384e-01,  2.85567839e-01,  3.53243527e-01],
       [ 2.75056023e-01,  2.70561800e-01,  3.21800974e-01],
       [ 2.87863452e-01,  2.65466063e-01,  3.13704615e-01],
       [ 2.96756292e-01,  2.87669554e-01,  2.81699126e-01],
       [ 2.72951296e-01,  2.79114167e-01,  3.14845681e-01],
       [ 3.00268983e-01,  2.64161743e-01,  3.02393924e-01],
       [ 2.77745079e-01,  3.03797932e-01,  2.84796853e-01],
       [ 2.91130981e-01,  3.12964971e-01,  2.63014841e-01],
       [ 2.72517327e-01,  2.96092176e-01,  2.97761031e-01],
       [ 2.66976360e-01,  3.04484821e-01,  2.95225217e-01],
       [ 2.89718033e-01,  2.53299866e-01,  3.25246112e-01],
       [ 2.87391425e-01,  3.21967152e-01,  2.58418290e-01],
       [ 3.38449994e-01,  2.84738823e-01,  2.46522632e-01],
       [ 3.77213389e-01,  8.29374528e-02,  4.75311273e-01],
       [ 4.04308036e-01,  2.50147126e-01,  2.27461735e-01],
       [ 1.96942611e-01,  4.33439633e-01,  2.61170533e-01],
       [ 3.35433642e-01,  2.84829994e-01,  2.49020625e-01],
       [ 2.55883583e-01,  3.04042575e-01,  3.07541439e-01],
       [ 2.71178678e-01,  2.53360259e-01,  3.45625629e-01],
       [ 3.43611401e-01,  2.84104847e-01,  2.42750345e-01],
       [ 3.42652159e-01,  2.99452053e-01,  2.29545461e-01],
       [ 2.95273175e-01,  2.60726007e-01,  3.11177803e-01],
       [ 2.53242622e-01,  3.55511969e-01,  2.62781126e-01],
       [ 2.85786925e-01,  3.21707787e-01,  2.60183534e-01],
       [ 2.87741363e-01,  2.57251702e-01,  3.22899603e-01],
       [ 3.66443928e-01,  2.39247153e-01,  2.68006808e-01],
       [ 2.70871233e-01,  3.16515423e-01,  2.79654083e-01],
       [ 2.79014055e-01,  2.89808173e-01,  2.97350276e-01],
       [ 3.20544339e-01,  2.74735161e-01,  2.72032213e-01],
       [ 2.93473106e-01,  3.17128013e-01,  2.57011414e-01],
       [ 2.80625098e-01,  2.75501036e-01,  3.10518575e-01],
       [ 2.77344310e-01,  2.92549873e-01,  2.96305810e-01],
       [ 2.36009537e-01,  3.14613250e-01,  3.19180892e-01],
       [ 2.94745660e-01,  2.76759633e-01,  2.94706456e-01],
       [ 2.89866638e-01,  2.91297086e-01,  2.84881327e-01],
       [ 2.95470614e-01,  2.98714060e-01,  2.72202870e-01],
       [ 2.90403884e-01,  2.75633111e-01,  3.00254357e-01],
       [ 2.60799279e-01,  3.04978676e-01,  3.01287421e-01],
       [ 2.94105396e-01,  2.58337613e-01,  3.15003559e-01],
       [ 3.13415427e-01,  3.08377816e-01,  2.46624802e-01],
       [ 3.11894529e-01,  2.73689944e-01,  2.81150904e-01],
       [ 2.67464294e-01,  2.56866744e-01,  3.45768808e-01],
       [ 3.91050533e-01,  2.37288936e-01,  2.50196022e-01],
       [ 2.83033999e-01,  2.69076446e-01,  3.14868554e-01],
       [ 2.30854102e-01,  3.35441005e-01,  3.04721100e-01],
       [ 3.60919568e-01,  2.83854555e-01,  2.28841902e-01],
       [ 2.33282159e-01,  3.75302790e-01,  2.66928315e-01],
       [ 3.21915816e-01,  2.88907710e-01,  2.57023563e-01],
       [ 2.92237152e-01,  3.01725984e-01,  2.72448655e-01],
       [ 2.99178773e-01,  3.38981361e-01,  2.32831799e-01],
       [-1.77403591e-02,  5.02613288e-01,  5.33988442e-01],
       [ 2.55406760e-01,  2.52570838e-01,  3.65157246e-01],
       [ 4.66217388e-01,  2.56202351e-01,  1.80717666e-01],
       [ 3.06393636e-01,  2.91539708e-01,  2.68715719e-01],
       [ 1.70960555e-01,  4.40537024e-01,  2.85668418e-01],
       [ 5.18051876e-01,  1.26209191e-01,  2.86556338e-01],
       [ 4.57896767e-01,  3.22343225e-01,  1.31241772e-01],
       [ 2.21570435e-01,  3.41258488e-01,  3.09840524e-01],
       [ 4.34748416e-01,  1.23787747e-01,  3.51245786e-01],
       [ 3.48594733e-01,  2.15347778e-01,  3.10192787e-01],
       [ 1.89174360e-01,  3.61880659e-01,  3.29443434e-01],
       [ 2.44932477e-01,  2.88752394e-01,  3.35919511e-01],
       [ 3.45800215e-01,  4.07063058e-02,  6.10400441e-01],
       [ 2.91441372e-01,  3.60210212e-01,  2.22541992e-01],
       [ 2.54782632e-01,  3.09689574e-01,  3.03108768e-01],
       [ 3.16340304e-01,  3.09953129e-01,  2.42616839e-01],
       [ 3.07824290e-01,  2.95318408e-01,  2.63774761e-01],
       [ 2.91530904e-01,  2.90644765e-01,  2.83880069e-01],
       [ 2.96390705e-01,  3.13582784e-01,  2.57481646e-01],
       [ 3.88493355e-01,  1.66425801e-01,  3.34003035e-01],
       [ 4.77067937e-01, -1.02750106e-01,  7.98836594e-01],
       [ 5.17173460e-01,  1.94233311e-01,  2.10214033e-01],
       [ 3.02515856e-01,  2.10175159e-01,  3.63608249e-01],
       [ 1.56678354e-01,  3.68356059e-01,  3.66236144e-01],
       [ 3.32260874e-01,  3.35077157e-01,  2.07791201e-01],
       [ 2.64926357e-01,  3.31114298e-01,  2.72261789e-01],
       [ 2.88462052e-01,  2.64154367e-01,  3.14506560e-01],
       [ 2.43207338e-01,  2.47178813e-01,  3.87213828e-01],
       [ 2.58372986e-01,  3.63056664e-01,  2.51349394e-01],
       [ 3.26081897e-01,  3.07379478e-01,  2.36429753e-01],
       [ 2.70888966e-01,  2.58443701e-01,  3.40032769e-01],
       [ 3.53626493e-01,  3.03684205e-01,  2.16958616e-01],
       [ 2.03097557e-01,  3.48395514e-01,  3.25011739e-01],
       [ 3.24895108e-01,  2.12820385e-01,  3.36340872e-01],
       [ 2.44454711e-01,  3.31741942e-01,  2.93136239e-01],
       [ 2.70507940e-01,  3.19318280e-01,  2.77406912e-01],
       [ 2.58969267e-01,  3.13399285e-01,  2.94983392e-01],
       [ 2.74142631e-01,  2.92485162e-01,  2.99698232e-01],
       [ 3.52783081e-01,  2.32514304e-01,  2.86987250e-01],
       [ 3.08307583e-01,  2.77135063e-01,  2.81081605e-01],
       [ 2.81481707e-01,  2.99973887e-01,  2.84738577e-01],
       [ 2.75952877e-01,  2.74177192e-01,  3.16906901e-01],
       [ 2.94887940e-01,  2.77722195e-01,  2.93573347e-01],
       [ 2.71005071e-01,  2.97334144e-01,  2.98098275e-01],
       [ 2.62598729e-01,  2.98226096e-01,  3.06131610e-01],
       [ 3.22448799e-01,  2.85577366e-01,  2.59718957e-01],
       [ 2.63711509e-01,  3.16171054e-01,  2.87337583e-01],
       [ 4.80776514e-01,  1.75386177e-01,  2.52495985e-01],
       [ 3.13019774e-01,  1.91862680e-01,  3.76212693e-01],
       [ 2.65621803e-01,  2.50682632e-01,  3.55242401e-01],
       [ 2.61594057e-01,  3.18349731e-01,  2.87478900e-01],
       [ 2.84246210e-01,  2.93002979e-01,  2.88809438e-01],
       [ 2.65577724e-01,  2.61739594e-01,  3.42275492e-01],
       [ 2.70393865e-01,  3.33831361e-01,  2.64361767e-01],
       [ 2.71114340e-01,  2.86784781e-01,  3.08744945e-01],
       [ 2.61284968e-01,  3.50247189e-01,  2.59161636e-01],
       [ 2.56103658e-01,  2.77860590e-01,  3.34926706e-01],
       [ 3.36225096e-01,  2.79168054e-01,  2.53718609e-01],
       [ 3.52321146e-01,  2.63111515e-01,  2.55592951e-01],
       [ 2.88974236e-01,  3.27333983e-01,  2.52161026e-01],
       [ 3.17989172e-01,  2.81160022e-01,  2.68037007e-01],
       [ 3.01834702e-01,  2.85658630e-01,  2.78774760e-01],
       [ 2.67506833e-01,  2.66288028e-01,  3.34896691e-01],
       [ 3.31562132e-01,  2.53426232e-01,  2.83720957e-01],
       [ 2.63176659e-01,  3.24149973e-01,  2.80408352e-01],
       [ 2.94826825e-01,  2.81410968e-01,  2.89867307e-01],
       [ 3.11190037e-01,  2.50672058e-01,  3.06113109e-01],
       [ 2.93950445e-01,  2.94438424e-01,  2.77791956e-01],
       [ 2.80319038e-01,  2.90398368e-01,  2.95410315e-01],
       [ 2.74503827e-01,  2.94141671e-01,  2.97649391e-01],
       [ 2.80663136e-01,  3.11384095e-01,  2.74650324e-01],
       [ 3.02102284e-01,  2.67759961e-01,  2.96754358e-01],
       [ 2.57148444e-01,  3.24020399e-01,  2.86799027e-01],
       [ 2.68113192e-01,  3.04896201e-01,  2.93630974e-01],
       [ 3.11765611e-01,  2.92130553e-01,  2.63164016e-01],
       [ 2.82025373e-01,  2.48768144e-01,  3.38815501e-01],
       [ 2.72263555e-01,  2.33289256e-01,  3.68871132e-01],
       [ 3.01410585e-01,  2.88910395e-01,  2.75984344e-01],
       [ 3.07658413e-01,  3.19144908e-01,  2.42200988e-01],
       [ 3.79677797e-01,  2.14219627e-01,  2.83997803e-01],
       [ 3.31996899e-01,  2.83479405e-01,  2.53276553e-01],
       [ 2.92155574e-01,  3.88695995e-01,  2.00396540e-01],
       [ 2.76751816e-01,  3.03627557e-01,  2.85969023e-01],
       [ 2.52509252e-01,  3.45221175e-01,  2.72411194e-01],
       [ 1.81783395e-01,  5.92595315e-01,  1.83729240e-01],
       [ 4.92855573e-01,  3.38917531e-02,  4.42899753e-01],
       [ 1.84729901e-01,  3.16932770e-01,  3.81637007e-01],
       [ 3.00589346e-01,  3.33573387e-01,  2.36108788e-01],
       [ 2.59935868e-01,  3.97350508e-01,  2.23212220e-01],
       [ 3.20502674e-02,  4.12164269e-01,  5.33053301e-01],
       [ 2.16234723e-01,  3.05311528e-01,  3.52760749e-01],
       [ 3.61716869e-01,  2.65716339e-01,  2.45262629e-01],
       [ 3.26495450e-01,  2.86737873e-01,  2.55011238e-01],
       [ 3.08913952e-01,  3.00592615e-01,  2.57819614e-01],
       [ 2.09983635e-01,  3.48508101e-01,  3.16600928e-01],
       [ 2.80176321e-01,  3.14532679e-01,  2.72192604e-01],
       [ 2.74179119e-01,  2.38884811e-01,  3.59609714e-01],
       [ 2.51566415e-01,  2.30454976e-01,  3.98374992e-01],
       [ 2.47918663e-01,  3.19279841e-01,  3.01207721e-01],
       [ 2.93379320e-01,  3.07424589e-01,  2.65990252e-01],
       [ 2.87405707e-01,  3.03634471e-01,  2.75334447e-01],
       [ 2.57637070e-01,  2.95440253e-01,  3.14393000e-01],
       [ 2.14021998e-01,  2.82703575e-01,  3.81488747e-01],
       [ 4.43360684e-01,  3.49579265e-01,  1.19820294e-01],
       [ 3.23842848e-01,  3.02583709e-01,  2.42662812e-01],
       [ 2.47059800e-01,  3.18523174e-01,  3.02883461e-01],
       [ 3.61266574e-01,  2.34643241e-01,  2.77275582e-01],
       [ 2.29268595e-01,  3.24627317e-01,  3.16978600e-01],
       [ 2.65489479e-01,  2.90620003e-01,  3.10808307e-01],
       [ 2.65353761e-01,  3.02138199e-01,  2.99257971e-01],
       [ 5.02372946e-01,  5.02458596e-01, -2.40996145e-03],
       [ 4.69293767e-01,  4.19799291e-02,  4.50441872e-01],
       [ 3.28353336e-01,  2.01811234e-01,  3.46561461e-01],
       [ 2.03339990e-01,  3.11328388e-01,  3.62747153e-01],
       [ 5.84182310e-01,  3.01857710e-01,  8.31340167e-02],
       [ 3.40193604e-01,  2.79099565e-01,  2.50369486e-01],
       [ 3.14189204e-01,  2.79539573e-01,  2.73141021e-01],
       [ 4.35769441e-01,  1.89308165e-01,  2.67975185e-01],
       [ 3.16258948e-01,  2.78994406e-01,  2.71759112e-01],
       [ 3.06300946e-01,  2.85934123e-01,  2.74246016e-01],
       [ 2.76719522e-01,  2.83906561e-01,  3.05796532e-01],
       [ 2.96157074e-01,  2.37923892e-01,  3.36160933e-01],
       [ 2.85839345e-01,  2.84875602e-01,  2.95368722e-01],
       [ 2.83721367e-01,  3.12281720e-01,  2.70802295e-01],
       [ 2.78516850e-01,  2.92773330e-01,  2.94872378e-01],
       [ 2.70178411e-01,  2.81296101e-01,  3.15517512e-01],
       [ 2.94864400e-01,  2.80167160e-01,  2.91094768e-01],
       [ 3.04039626e-01,  2.89076336e-01,  2.73318118e-01],
       [ 2.87767558e-01,  2.65563202e-01,  3.13699397e-01],
       [ 2.90788530e-01,  3.04215252e-01,  2.71490197e-01],
       [ 3.11469342e-01,  2.70942326e-01,  2.84351741e-01],
       [ 4.07674184e-01,  3.28421040e-01,  1.57739402e-01],
       [ 3.11048279e-01,  2.81035858e-01,  2.74596581e-01],
       [ 3.49807357e-01,  3.01471473e-01,  2.21937294e-01],
       [ 3.19282076e-01,  2.53664560e-01,  2.94982887e-01],
       [ 3.81923390e-01,  2.57431565e-01,  2.37240461e-01],
       [ 2.45331823e-01,  3.29888073e-01,  2.93918828e-01],
       [ 2.80967887e-01,  2.57433019e-01,  3.29994965e-01],
       [ 2.19181895e-01,  3.46933504e-01,  3.07284446e-01],
       [ 3.25391304e-01,  2.29638805e-01,  3.15798237e-01],
       [ 2.46654495e-01,  3.68581992e-01,  2.58579584e-01],
       [ 3.36640841e-01,  2.91687172e-01,  2.41603401e-01],
       [ 3.44021196e-01,  2.75645245e-01,  2.50412452e-01],
       [ 2.74699661e-01,  2.58727967e-01,  3.35429786e-01],
       [ 2.62835069e-01,  3.19427117e-01,  2.85169389e-01],
       [ 2.75989092e-01,  3.02522405e-01,  2.87819891e-01],
       [ 3.09760051e-01,  3.23503147e-01,  2.36539324e-01],
       [ 3.45711042e-01,  2.51636552e-01,  2.72883773e-01],
       [ 3.03597837e-01,  3.29469230e-01,  2.36900421e-01],
       [ 2.53447386e-01,  3.24956479e-01,  2.89833178e-01],
       [ 3.01538680e-01,  2.78991011e-01,  2.85727709e-01],
       [ 3.10757671e-01,  2.37318288e-01,  3.21582289e-01],
       [ 2.87677242e-01,  2.81041589e-01,  2.97424177e-01],
       [ 2.87178747e-01,  2.80949115e-01,  2.98026899e-01],
       [ 2.94181636e-01,  2.81263937e-01,  2.90657054e-01],
       [ 2.78346717e-01,  3.01431168e-01,  2.86484926e-01],
       [ 3.25652928e-01,  2.88100103e-01,  2.54466130e-01],
       [ 3.17533747e-01,  2.85625016e-01,  2.64116893e-01],
       [ 2.44000236e-01,  3.38143754e-01,  2.87717210e-01],
       [ 2.97811382e-01,  2.99305244e-01,  2.69400456e-01],
       [ 2.53583271e-01,  3.31020428e-01,  2.84053552e-01],
       [ 3.91353328e-01,  2.50427939e-01,  2.36831769e-01],
       [ 2.68946850e-01,  3.12743279e-01,  2.85183936e-01],
       [ 3.11385902e-01,  2.65668864e-01,  2.89876232e-01],
       [ 3.82371518e-01,  3.39938634e-01,  1.66157361e-01],
       [ 2.78561398e-01,  1.99768430e-01,  4.06314252e-01],
       [ 2.57560507e-01,  2.89766575e-01,  3.20407267e-01],
       [ 2.80562372e-01,  2.76895370e-01,  3.09106081e-01],
       [ 3.63331917e-01,  2.83648267e-01,  2.27119061e-01],
       [ 4.02455523e-01,  4.54361761e-01,  7.83592969e-02],
       [ 2.50605165e-01,  3.33958823e-01,  2.84499554e-01],
       [ 3.62325038e-01,  2.46400224e-01,  2.64032133e-01],
       [ 3.22104336e-01,  3.97582589e-01,  1.69430512e-01],
       [ 1.58357718e-01,  3.71193642e-01,  3.61095517e-01],
       [ 2.19449320e-01,  3.55261965e-01,  2.99346834e-01],
       [ 2.85570823e-01,  2.90427538e-01,  2.90039660e-01],
       [ 3.46042369e-01,  2.43790885e-01,  2.80821781e-01],
       [ 3.01496060e-01,  2.56520090e-01,  3.09417932e-01],
       [ 4.01808019e-01,  7.61548050e-02,  4.59032330e-01],
       [ 2.68551547e-01,  3.15735199e-01,  2.82752989e-01],
       [ 3.05984643e-01,  2.45870796e-01,  3.16690386e-01],
       [ 3.73751218e-01,  2.13483452e-01,  2.89850564e-01],
       [ 2.55931602e-01,  3.80431004e-01,  2.39856463e-01],
       [ 3.37825067e-01,  3.43046622e-01,  1.96968466e-01],
       [ 2.33243657e-01,  3.10465255e-01,  3.26619529e-01],
       [ 3.01481716e-01,  2.61951035e-01,  3.03543861e-01],
       [ 2.92141855e-01,  2.75089553e-01,  2.99057537e-01],
       [ 3.09023997e-01,  3.84932149e-01,  1.88840072e-01],
       [ 2.15014612e-01,  2.38088554e-01,  4.38768688e-01],
       [ 2.86644265e-01,  3.09459159e-01,  2.70582755e-01],
       [ 3.53357946e-01,  3.04962609e-01,  2.16063493e-01],
       [ 2.44618117e-01,  3.32971156e-01,  2.91815015e-01],
       [ 3.01554723e-01,  2.59944708e-01,  3.05632448e-01],
       [ 2.94859186e-01,  2.46107407e-01,  3.27992472e-01],
       [ 1.97638787e-01,  2.37037763e-01,  4.67363938e-01],
       [ 1.99959932e-01,  4.67610336e-01,  2.34427260e-01],
       [ 3.42109185e-01,  2.65220755e-01,  2.62238914e-01],
       [ 3.06539566e-01,  3.01278361e-01,  2.59364285e-01],
       [ 4.98347973e-01,  8.60076564e-02,  3.54472941e-01],
       [ 2.94254063e-01,  2.46693196e-01,  3.27960853e-01],
       [ 2.78793179e-01,  3.04114126e-01,  2.83432812e-01],
       [ 3.33312093e-01,  4.06923600e-01,  1.54501389e-01],
       [ 2.61701192e-01,  3.13061257e-01,  2.92419061e-01],
       [ 2.95192759e-01,  2.59807907e-01,  3.12263746e-01],
       [ 2.99162179e-01,  2.82991209e-01,  2.84014036e-01],
       [ 2.40961569e-01,  3.49347696e-01,  2.80904673e-01],
       [ 2.90605872e-01,  2.90413445e-01,  2.85023481e-01],
       [ 2.87860007e-01,  2.67167813e-01,  3.11863776e-01],
       [ 2.91042490e-01,  2.90684670e-01,  2.84322998e-01],
       [ 2.49679231e-01,  3.17465787e-01,  3.01043615e-01],
       [ 2.87796976e-01,  2.94834934e-01,  2.83450657e-01],
       [ 3.02287966e-01,  2.71442383e-01,  2.92726774e-01],
       [ 2.95546882e-01,  3.01428567e-01,  2.69548316e-01],
       [ 3.32864231e-01,  2.25961020e-01,  3.12773373e-01],
       [ 3.03554790e-01,  2.74596302e-01,  2.88237762e-01],
       [ 3.34795718e-01,  2.85300028e-01,  2.49127289e-01],
       [ 2.91134022e-01,  3.02237557e-01,  2.73030206e-01],
       [ 3.57566233e-01,  2.27765198e-01,  2.87971647e-01],
       [ 2.84898435e-01,  2.85354678e-01,  2.95838629e-01],
       [ 3.37553568e-01,  3.07261793e-01,  2.26859802e-01],
       [ 3.10841434e-01,  2.58130785e-01,  2.98366512e-01],
       [ 2.31657785e-01,  3.20202266e-01,  3.18600073e-01],
       [ 2.95158083e-01,  2.90131791e-01,  2.80827098e-01],
       [ 2.56307988e-01,  2.59555309e-01,  3.55663801e-01],
       [ 2.81107675e-01,  2.85919486e-01,  2.99149406e-01],
       [ 2.52334280e-01,  3.28041362e-01,  2.88130492e-01],
       [ 2.63780095e-01,  3.30154572e-01,  2.74292451e-01],
       [ 3.00306607e-01,  3.14835523e-01,  2.52710397e-01],
       [ 3.09197829e-01,  2.84309191e-01,  2.73109179e-01],
       [ 2.55357471e-01,  2.91705645e-01,  3.20823647e-01],
       [ 2.35195691e-01,  3.37566385e-01,  2.97865114e-01],
       [ 4.17823862e-01,  2.41545745e-01,  2.26089317e-01],
       [ 2.21013426e-01,  2.97865419e-01,  3.54933999e-01],
       [ 3.33199323e-01,  2.80059343e-01,  2.55494827e-01],
       [ 2.72911272e-01,  2.97640020e-01,  2.95802824e-01],
       [ 3.07678685e-01,  3.39924717e-01,  2.24539293e-01],
       [ 2.89841613e-01,  2.87785236e-01,  2.88400484e-01],
       [ 2.88562450e-01,  2.83846249e-01,  2.93658415e-01],
       [ 3.01343313e-01,  2.70265394e-01,  2.94882371e-01],
       [ 2.87611590e-01,  2.89142685e-01,  2.89272607e-01],
       [ 2.53629607e-01,  2.68244987e-01,  3.48675969e-01],
       [ 2.93651948e-01,  2.77893566e-01,  2.94632727e-01],
       [ 3.45667977e-01,  2.85223008e-01,  2.39990019e-01],
       [ 2.70512667e-01,  2.87915524e-01,  3.08213673e-01],
       [ 2.97939124e-01,  2.45603450e-01,  3.25319729e-01],
       [ 2.73976737e-01,  2.19731223e-01,  3.84435277e-01],
       [ 2.90093447e-01,  2.82228601e-01,  2.93763507e-01],
       [ 3.59800389e-01,  2.73908750e-01,  2.38985863e-01],
       [ 1.60692745e-01,  3.42615716e-01,  3.87325378e-01],
       [ 5.41589873e-01,  1.58532052e-01,  2.34445802e-01],
       [ 2.43713893e-01,  2.58822044e-01,  3.71956427e-01],
       [ 3.04555675e-01,  2.79478023e-01,  2.82318268e-01],
       [ 3.28967482e-01,  2.77164207e-01,  2.62025549e-01],
       [ 4.37952778e-01,  1.48970014e-01,  3.14791266e-01],
       [ 4.07141028e-01,  2.05153944e-01,  2.71884337e-01],
       [ 4.18213302e-01,  2.17227260e-01,  2.50460358e-01],
       [ 1.21777960e-01,  4.02043678e-01,  3.83794648e-01],
       [ 2.94677201e-01,  3.40408224e-01,  2.35699721e-01],
       [ 2.85160981e-01,  2.82483941e-01,  2.98508092e-01],
       [ 2.64664806e-01,  3.00407041e-01,  3.01718852e-01],
       [ 2.79888684e-01,  3.47595931e-01,  2.43371436e-01],
       [ 2.78271212e-01,  3.05632679e-01,  2.82497216e-01],
       [ 6.15305099e-01,  1.87652207e-01,  1.67551424e-01],
       [ 3.21184960e-01,  2.73822130e-01,  2.72353813e-01],
       [ 3.24743385e-01,  3.27913654e-01,  2.19889162e-01],
       [ 4.92068818e-01,  4.37501524e-01,  3.73496664e-02],
       [ 4.41232172e-01,  3.59067018e-01,  1.14417309e-01],
       [ 3.12259945e-01,  3.25467415e-01,  2.32653595e-01],
       [ 3.17971216e-01,  2.49187047e-01,  3.01090018e-01],
       [ 2.75639358e-01,  2.81277244e-01,  3.09685364e-01],
       [ 3.00747234e-01,  2.97066667e-01,  2.68742532e-01],
       [ 2.24090235e-01,  2.84182823e-01,  3.66569507e-01],
       [ 2.58065901e-01,  3.46297890e-01,  2.65787471e-01],
       [ 3.06064922e-01,  2.73665475e-01,  2.86755013e-01],
       [ 3.01158967e-01,  2.10741711e-01,  3.64393430e-01],
       [ 3.24026552e-01,  2.93583693e-01,  2.50758613e-01],
       [ 2.04260714e-01,  3.98086715e-01,  2.80048549e-01],
       [ 3.23094951e-01,  2.74774575e-01,  2.69660377e-01],
       [ 4.00211163e-01,  3.58010435e-01,  1.40750709e-01],
       [ 3.64200906e-01,  1.52439199e-01,  3.76435170e-01],
       [ 3.62718944e-01,  2.10597339e-01,  3.02821254e-01],
       [ 2.26536226e-01,  3.21508443e-01,  3.23270531e-01],
       [ 2.38347938e-01,  3.04588642e-01,  3.26745207e-01],
       [ 2.96118027e-01,  2.78538760e-01,  2.91512526e-01],
       [ 2.79808225e-01,  3.10063514e-01,  2.76740972e-01],
       [ 3.14386077e-01,  2.86577089e-01,  2.66079790e-01],
       [ 2.85001651e-01,  2.73333602e-01,  3.08236800e-01],
       [ 2.95497275e-01,  2.99439090e-01,  2.71485447e-01],
       [ 2.49119061e-01,  3.23256989e-01,  2.96082484e-01],
       [ 2.96935957e-01,  2.90706927e-01,  2.78534234e-01],
       [ 2.85824766e-01,  2.89712523e-01,  2.90498964e-01],
       [ 2.88047233e-01,  2.63230770e-01,  3.15951487e-01],
       [ 2.82418751e-01,  2.91239812e-01,  2.92418569e-01],
       [ 3.11349194e-01,  3.00232251e-01,  2.55931458e-01],
       [ 2.76836979e-01,  2.97532213e-01,  2.91854234e-01],
       [ 3.01880071e-01,  2.89979332e-01,  2.74492587e-01],
       [ 2.75250646e-01,  3.02285231e-01,  2.88805946e-01],
       [ 3.13136763e-01,  2.98747703e-01,  2.55687339e-01],
       [ 3.12597094e-01,  2.47006594e-01,  3.08765758e-01],
       [ 2.42645756e-01,  2.62897525e-01,  3.68334499e-01],
       [ 3.23645395e-01,  2.19940017e-01,  3.28959208e-01],
       [ 2.78567104e-01,  3.36223357e-01,  2.54296778e-01],
       [ 2.34676495e-01,  3.49394860e-01,  2.87645058e-01],
       [ 3.87887917e-01,  1.59935141e-01,  3.43109128e-01],
       [ 2.87662822e-01,  3.09773644e-01,  2.69299998e-01],
       [ 3.09098543e-01,  2.48583902e-01,  3.10504803e-01],
       [ 2.95136433e-01,  2.92602362e-01,  2.78427056e-01],
       [ 4.66023585e-01,  2.34492733e-01,  2.00881624e-01],
       [ 2.21888501e-01,  3.50901810e-01,  3.00526947e-01],
       [ 2.81077510e-01,  2.58806500e-01,  3.28321102e-01],
       [ 3.32637941e-01,  2.78184651e-01,  2.57792087e-01],
       [ 2.07144063e-01,  3.09773393e-01,  3.59500880e-01],
       [ 2.35726734e-01,  3.34257077e-01,  3.00371112e-01],
       [ 2.89493683e-01,  2.57668502e-01,  3.20575145e-01],
       [ 3.17302426e-01,  2.94638286e-01,  2.55760981e-01],
       [ 3.79125534e-01,  4.34953415e-01,  1.04532925e-01],
       [ 2.07802506e-01,  3.50716308e-01,  3.17124988e-01],
       [ 3.53908668e-01,  2.19200414e-01,  3.00855594e-01],
       [ 3.04055659e-01,  2.47367664e-01,  3.16973285e-01],
       [ 2.94142906e-01,  3.07182402e-01,  2.65487704e-01],
       [ 2.09032076e-01,  3.33491146e-01,  3.32316933e-01],
       [ 2.73413876e-01,  2.98322600e-01,  2.94601566e-01],
       [ 2.98426091e-01,  2.77469481e-01,  2.90323239e-01],
       [ 2.79608665e-01,  3.08007481e-01,  2.78886890e-01],
       [ 2.67439407e-01,  2.93687944e-01,  3.05556787e-01],
       [ 2.53332680e-01,  3.11767225e-01,  3.02635290e-01],
       [ 2.25134195e-01,  3.75324773e-01,  2.75625094e-01],
       [ 2.90469535e-01,  2.96933934e-01,  2.78768763e-01],
       [ 3.12999926e-01,  3.19606516e-01,  2.37056050e-01],
       [ 3.15845860e-01,  2.52089174e-01,  2.99996775e-01],
       [ 3.84311459e-01,  2.70871211e-01,  2.22687651e-01],
       [ 2.77801146e-01,  2.86164996e-01,  3.02328497e-01],
       [ 3.64563925e-01,  2.25638983e-01,  2.84207625e-01],
       [ 1.70414861e-01,  4.68504895e-01,  2.66324216e-01],
       [-9.17067491e-02,  6.26714121e-01,  5.74709678e-01],
       [ 2.78467525e-01,  2.80718271e-01,  3.07284411e-01],
       [ 2.84206413e-01,  3.10829752e-01,  2.71681287e-01],
       [ 3.57654429e-01,  1.83827150e-01,  3.40276406e-01],
       [ 3.46663085e-01,  2.09037886e-01,  3.19478085e-01],
       [ 2.50185862e-01,  3.54955847e-01,  2.66375731e-01],
       [ 2.21102972e-01,  3.71983816e-01,  2.82847765e-01],
       [-9.45732968e-02,  6.33874686e-01,  5.74720601e-01],
       [ 3.40154689e-01,  2.22133957e-01,  3.10232287e-01],
       [ 3.08086730e-01,  2.81690978e-01,  2.76739432e-01],
       [ 2.93282163e-01,  3.05564132e-01,  2.67820795e-01],
       [ 2.55440339e-01,  2.98673404e-01,  3.13485754e-01],
       [ 2.75120742e-01,  2.79628177e-01,  3.11976252e-01],
       [ 2.82124533e-01,  3.50139970e-01,  2.39167190e-01],
       [ 2.74835552e-01,  2.72126175e-01,  3.20333296e-01],
       [ 4.43507551e-01,  2.24212633e-01,  2.25483680e-01],
       [ 2.18988349e-01,  3.96183714e-01,  2.65357275e-01],
       [ 4.51119791e-01,  5.07102680e-01,  2.21618107e-02],
       [ 1.34971073e-01,  2.56118555e-01,  5.50849187e-01],
       [ 3.38974904e-01,  3.86768711e-01,  1.63825228e-01],
       [ 3.35773946e-01,  2.07959512e-01,  3.31362014e-01],
       [ 2.04897607e-01,  3.42284637e-01,  3.28714425e-01],
       [ 3.00722995e-01,  3.33352967e-01,  2.36175326e-01],
       [ 3.06347200e-01,  2.71635964e-01,  2.88563913e-01],
       [ 3.04880558e-01,  3.20564537e-01,  2.43452393e-01],
       [ 2.55693110e-01,  2.77888496e-01,  3.35367307e-01],
       [ 3.19854445e-01,  2.86309664e-01,  2.61352626e-01],
       [ 3.15385284e-01,  1.69815716e-01,  4.04868544e-01],
       [ 2.05093585e-01,  3.57160357e-01,  3.14357426e-01],
       [ 2.94588204e-01,  2.76505884e-01,  2.95125850e-01],
       [ 4.54038133e-01,  1.48872028e-01,  3.02543288e-01],
       [ 2.02460964e-01,  3.65446291e-01,  3.09929817e-01],
       [ 3.31503815e-01,  2.71591170e-01,  2.65242615e-01],
       [ 2.82253051e-01,  3.38422336e-01,  2.48888978e-01],
       [ 2.49276883e-01,  2.99472912e-01,  3.19541488e-01],
       [ 2.96022434e-01,  2.77320732e-01,  2.92855749e-01],
       [ 2.87446085e-01,  2.86171678e-01,  2.92426565e-01],
       [ 2.95381381e-01,  2.65388753e-01,  3.06023971e-01],
       [ 3.04903511e-01,  3.16871991e-01,  2.46688100e-01],
       [ 3.08531574e-01,  3.09672224e-01,  2.49846963e-01],
       [ 2.96874103e-01,  2.60538420e-01,  3.09739884e-01],
       [ 2.72625730e-01,  2.95849765e-01,  2.97891015e-01],
       [ 2.89037513e-01,  2.89069550e-01,  2.87919084e-01],
       [ 3.16842816e-01,  3.07855504e-01,  2.44050913e-01],
       [ 2.82732194e-01,  2.95975672e-01,  2.87395694e-01],
       [ 3.10935635e-01,  3.33640566e-01,  2.26907631e-01],
       [ 2.80472057e-01,  3.09214967e-01,  2.76882203e-01],
       [ 3.14547831e-01,  2.70559513e-01,  2.81821949e-01],
       [ 2.58281754e-01,  2.75967362e-01,  3.34530578e-01],
       [ 2.85469384e-01,  3.46486081e-01,  2.39081455e-01],
       [ 2.71008683e-01,  2.29292901e-01,  3.75492780e-01],
       [ 2.89186185e-01,  3.53618730e-01,  2.29833882e-01],
       [ 3.09033417e-01,  3.10337635e-01,  2.48793191e-01],
       [ 2.44058775e-01,  3.03989480e-01,  3.20790547e-01],
       [ 2.35399450e-01,  2.91217506e-01,  3.44553203e-01],
       [ 1.97146138e-01,  6.06286855e-01,  1.62394237e-01],
       [ 3.02418519e-01,  2.34509630e-01,  3.33526833e-01],
       [ 2.95143449e-01,  2.82379404e-01,  2.88573115e-01],
       [ 2.71177253e-01,  2.93236111e-01,  3.02049966e-01],
       [-1.23388425e-01,  7.57322209e-01,  5.41767616e-01],
       [ 2.35703999e-01,  3.00838773e-01,  3.33787178e-01],
       [ 2.62225271e-01,  3.26881476e-01,  2.78868673e-01],
       [ 2.99598015e-01,  2.55233287e-01,  3.12766444e-01],
       [ 4.25732690e-01,  1.92110024e-01,  2.72257775e-01],
       [ 2.61731892e-01,  2.81752592e-01,  3.24307990e-01],
       [ 2.93317786e-01,  3.09591551e-01,  2.64038525e-01],
       [ 2.38894625e-01,  2.51003192e-01,  3.87910866e-01],
       [ 5.97780593e-01,  6.21390220e-01, -9.96209992e-02],
       [ 3.59067024e-01,  3.06631327e-01,  2.10153445e-01],
       [ 3.07202210e-01,  3.32866271e-01,  2.30823342e-01],
       [ 3.20008830e-01,  2.96101144e-01,  2.51976149e-01],
       [ 2.98759739e-01,  2.69713533e-01,  2.98027126e-01],
       [ 3.31268465e-01,  3.19158605e-01,  2.21812445e-01],
       [ 2.92642168e-01,  2.72253902e-01,  3.01519230e-01],
       [ 2.91144223e-01,  2.94352885e-01,  2.80617710e-01],
       [ 3.07322799e-01,  2.90027513e-01,  2.69302501e-01],
       [ 3.48412014e-01,  2.13459377e-01,  3.12577205e-01],
       [ 3.17394142e-01,  2.47709780e-01,  3.03268762e-01],
       [ 3.99967391e-01,  2.51434158e-01,  2.29404637e-01],
       [ 3.26185663e-01,  2.61696531e-01,  2.80053292e-01],
       [ 3.21733553e-01,  2.39492538e-01,  3.08159612e-01],
       [ 2.90116944e-01,  3.26675355e-01,  2.51666476e-01],
       [ 2.80388431e-01,  2.07845201e-01,  3.92686202e-01],
       [ 2.79055984e-01,  2.97626317e-01,  2.89493184e-01],
       [ 3.36043996e-01,  3.24137254e-01,  2.13692258e-01],
       [ 2.73615051e-01,  2.11025649e-01,  3.96706686e-01],
       [ 1.46222951e-01,  4.10278489e-01,  3.41432843e-01],
       [ 2.90508490e-01,  2.92155128e-01,  2.83399289e-01],
       [ 2.98247049e-01,  2.76556083e-01,  2.91435372e-01],
       [ 2.44141275e-01,  3.06573843e-01,  3.18045873e-01],
       [ 2.52344648e-01,  2.99190925e-01,  3.16390782e-01],
       [ 2.53538069e-01,  3.31251913e-01,  2.83888292e-01],
       [ 3.22307574e-01,  2.61815030e-01,  2.83527862e-01],
       [ 2.68346554e-01,  2.25470825e-01,  3.83736355e-01],
       [ 2.85888119e-01,  2.86217126e-01,  2.93956271e-01],
       [ 2.75537283e-01,  2.98535154e-01,  2.92197332e-01],
       [ 2.63077359e-01,  2.99670497e-01,  3.04156605e-01],
       [ 2.94894602e-01,  2.97083395e-01,  2.74320517e-01],
       [ 2.37349200e-01,  3.10126143e-01,  3.22191329e-01],
       [ 2.76558563e-01,  3.06002174e-01,  2.83871652e-01],
       [ 2.61539022e-01,  2.09768415e-01,  4.14034150e-01],
       [ 3.14887789e-01,  1.06189661e-01,  5.14304843e-01],
       [ 2.78956431e-01,  1.71914974e-01,  4.48117158e-01],
       [ 3.85689165e-01,  2.81044785e-01,  2.12384673e-01],
       [ 1.44832569e-01,  3.01899076e-01,  4.61742935e-01],
       [ 3.04870437e-01,  3.12210687e-01,  2.50884666e-01],
       [ 2.99231947e-01,  3.06240634e-01,  2.61552750e-01],
       [ 2.80257055e-01,  3.20948967e-01,  2.66217872e-01],
       [ 2.87629090e-01,  2.62605912e-01,  3.17077067e-01],
       [ 2.63117501e-01,  2.95242081e-01,  3.08612313e-01],
       [ 2.94996518e-01,  3.08301379e-01,  2.63637860e-01],
       [ 2.79109075e-01,  2.91840194e-01,  2.95200927e-01],
       [ 4.25855315e-01,  2.85866356e-01,  1.80214118e-01],
       [ 3.16943111e-01,  2.16986541e-01,  3.39422263e-01],
       [ 2.58898417e-01,  2.67291480e-01,  3.43599640e-01],
       [ 6.38214962e-01, -1.39905335e-01,  6.80881244e-01],
       [ 3.78417648e-01,  3.51114024e-01,  1.60558152e-01],
       [ 3.07297938e-01,  2.16717581e-01,  3.49995616e-01],
       [ 2.84384586e-01,  2.98822381e-01,  2.82951903e-01],
       [ 2.90206566e-01,  2.82706311e-01,  2.93162851e-01],
       [ 2.76039398e-01,  3.00221475e-01,  2.90019768e-01],
       [ 2.97758951e-01,  2.86768669e-01,  2.81615541e-01],
       [ 2.91543497e-01,  2.73277392e-01,  3.01560648e-01],
       [ 3.10070052e-01,  2.48769963e-01,  3.09326246e-01],
       [ 2.45989434e-01,  3.16809938e-01,  3.05736132e-01],
       [ 2.83878367e-01,  2.86098574e-01,  2.96122162e-01],
       [ 3.06670680e-01,  2.63156215e-01,  2.97103745e-01],
       [ 2.77313652e-01,  2.93799406e-01,  2.95082404e-01],
       [ 2.86821350e-01,  2.84539349e-01,  2.94714075e-01],
       [ 2.50123502e-01,  3.04348007e-01,  3.13587638e-01],
       [ 2.88323930e-01,  2.88471641e-01,  2.89230242e-01],
       [ 3.39430391e-01,  2.65789524e-01,  2.64008097e-01],
       [ 2.74100956e-01,  2.92892171e-01,  2.99329865e-01],
       [ 3.03615700e-01,  2.66650950e-01,  2.96423761e-01],
       [ 2.59271683e-01,  3.27895758e-01,  2.80986143e-01],
       [ 3.02945516e-01,  2.41866887e-01,  3.24382320e-01],
       [ 7.99472363e-02,  4.34020757e-01,  4.18900093e-01],
       [ 2.60941756e-01,  2.82057918e-01,  3.24860806e-01],
       [ 3.29928219e-01,  2.70344709e-01,  2.67887563e-01],
       [ 2.85930242e-01,  2.78987987e-01,  3.01333695e-01],
       [ 2.25804231e-01,  3.37187430e-01,  3.08817454e-01],
       [ 1.88059979e-01,  3.81142323e-01,  3.13284718e-01],
       [ 3.18246214e-01,  3.47367019e-01,  2.09508697e-01],
       [ 3.06144472e-01,  2.69838700e-01,  2.90616778e-01],
       [ 3.07547859e-01,  2.52691791e-01,  3.07520507e-01],
       [ 4.00652565e-01,  3.41622475e-01,  1.52407225e-01],
       [ 2.83346677e-01,  2.50311848e-01,  3.35560965e-01],
       [ 2.65869898e-01,  3.20936270e-01,  2.80625385e-01],
       [ 2.74472714e-01,  2.51758146e-01,  3.43764063e-01],
       [ 3.14849966e-01,  3.08015638e-01,  2.45673041e-01],
       [ 2.86758164e-01,  3.31093504e-01,  2.50960292e-01],
       [ 2.98086451e-01,  2.39820838e-01,  3.31865102e-01],
       [ 1.65553481e-01,  2.02742808e-01,  5.87666042e-01],
       [ 2.02210613e-01,  3.35406083e-01,  3.38860998e-01],
       [ 1.13129271e-01,  3.73945177e-01,  4.26415008e-01],
       [ 2.23929159e-01,  3.25013770e-01,  3.22838550e-01],
       [ 3.07839930e-01,  2.45998409e-01,  3.14661977e-01],
       [ 3.04627952e-01,  2.99043546e-01,  2.63227562e-01],
       [ 2.29219720e-01,  3.13291553e-01,  3.28449208e-01],
       [ 2.47710961e-01,  3.08084261e-01,  3.12496662e-01],
       [ 3.03399265e-01,  2.64321680e-01,  2.99099404e-01],
       [ 2.67857664e-01,  2.79640162e-01,  3.19811753e-01],
       [ 3.53806098e-01,  2.75235307e-01,  2.42623250e-01],
       [ 2.43470176e-01,  3.77861856e-01,  2.54295448e-01],
       [ 2.59598952e-01,  2.73795046e-01,  3.35442644e-01],
       [ 2.69497641e-01,  3.91692881e-01,  2.18453967e-01],
       [ 2.54703360e-01,  3.02618041e-01,  3.10273692e-01],
       [ 3.85594849e-01,  2.80032999e-01,  2.13363547e-01],
       [ 3.19636084e-01,  2.83473861e-01,  2.64282701e-01],
       [ 2.81387842e-01,  2.70434293e-01,  3.15143353e-01],
       [ 3.37411164e-01,  3.27175318e-01,  2.10066559e-01],
       [ 2.49126024e-01,  1.20790540e-01,  5.94479821e-01],
       [ 2.98137499e-01,  3.71186965e-01,  2.08172947e-01],
       [ 2.99096740e-01,  3.22273543e-01,  2.47210139e-01],
       [ 3.01182146e-01,  3.72441691e-01,  2.04605604e-01],
       [ 2.79981584e-01,  3.89409000e-01,  2.10598498e-01],
       [ 3.53482737e-01,  2.85160188e-01,  2.33621622e-01],
       [ 2.54217299e-01,  2.64684985e-01,  3.52113497e-01],
       [ 3.06943103e-01,  2.99834769e-01,  2.60338738e-01],
       [ 3.00251171e-01,  2.68422514e-01,  2.97896369e-01],
       [ 2.74127610e-01,  2.98121281e-01,  2.94062127e-01],
       [ 2.83105567e-01,  3.05237487e-01,  2.78044530e-01],
       [ 2.84012581e-01,  3.09570135e-01,  2.73050718e-01],
       [ 3.17827385e-01,  3.59558268e-01,  2.00362283e-01],
       [ 2.74609215e-01,  3.10472661e-01,  2.81569731e-01],
       [ 2.07068471e-01,  5.30623915e-01,  1.89949794e-01],
       [ 3.08080128e-01,  3.87580558e-01,  1.87726768e-01],
       [ 2.26935172e-01,  2.91934754e-01,  3.54134489e-01],
       [ 4.86635420e-01, -2.75467243e-02,  5.73756693e-01],
       [ 4.15398552e-01,  2.11264380e-01,  2.58896888e-01],
       [ 3.34999504e-01,  2.41513915e-01,  2.93302727e-01],
       [ 2.45232098e-01,  3.19153058e-01,  3.04284095e-01],
       [ 2.56866438e-01,  3.08256482e-01,  3.02269204e-01],
       [ 3.28611516e-01,  2.43277862e-01,  2.97358370e-01],
       [ 3.08649109e-01,  3.00393513e-01,  2.58247624e-01],
       [ 2.98755980e-01,  2.66400897e-01,  3.01528913e-01],
       [ 2.38903262e-01,  3.02055491e-01,  3.28745875e-01],
       [ 3.58848495e-01,  1.55113278e-01,  3.78117291e-01],
       [ 4.55544194e-02,  4.56093600e-01,  4.56939750e-01],
       [ 3.41912254e-01,  3.35233376e-01,  1.99926862e-01],
       [ 3.19148124e-01,  2.53446666e-01,  2.95344937e-01],
       [ 6.91545856e-01,  1.08947412e-01,  2.18187805e-01],
       [ 2.97367217e-01,  3.34746022e-01,  2.38023028e-01],
       [ 2.70434734e-01,  2.96069648e-01,  2.99966759e-01],
       [ 3.25067391e-01,  2.93389866e-01,  2.50022973e-01],
       [ 2.68258455e-01,  3.04950599e-01,  2.93425972e-01],
       [ 2.93131897e-01,  2.85541352e-01,  2.87379315e-01],
       [ 2.61830077e-01,  2.81838539e-01,  3.24105877e-01],
       [ 2.32010983e-01,  3.02160110e-01,  3.36775123e-01],
       [ 2.74277792e-01,  2.26193755e-01,  3.75565958e-01],
       [ 2.58903486e-01,  3.41272631e-01,  2.69326489e-01],
       [ 2.67881757e-01,  2.88982743e-01,  3.09926015e-01],
       [ 2.98648606e-01,  2.77234134e-01,  2.90344546e-01],
       [ 2.94363263e-01,  2.85718082e-01,  2.85985913e-01],
       [ 3.31015350e-01,  2.45024898e-01,  2.93196522e-01],
       [ 3.00734745e-01,  2.81412066e-01,  2.84068573e-01],
       [ 2.44743326e-01,  2.77704811e-01,  3.48424251e-01],
       [ 2.71467466e-01,  2.79436860e-01,  3.16102044e-01],
       [ 2.74976217e-01,  3.40868408e-01,  2.53747923e-01],
       [ 2.11528800e-01,  2.34547113e-01,  4.49220694e-01],
       [ 3.23530798e-01,  2.64274795e-01,  2.79852670e-01],
       [ 4.99955531e-01,  6.19375673e-01, -5.32999467e-02],
       [ 3.36827903e-01,  2.42735815e-01,  2.90286985e-01],
       [ 2.44677051e-01,  3.02091940e-01,  3.22046492e-01],
       [ 2.69981090e-01,  3.07997553e-01,  2.88672404e-01],
       [ 2.03894306e-01,  2.32326221e-01,  4.64512772e-01],
       [ 3.03817900e-01,  2.70773116e-01,  2.91919427e-01],
       [ 2.52185506e-01,  2.71682842e-01,  3.46433461e-01],
       [ 3.18955711e-01,  2.95450831e-01,  2.53519876e-01],
       [ 2.98618153e-01,  2.87091168e-01,  2.80462270e-01],
       [ 2.80965113e-01,  2.37605385e-01,  3.53358274e-01],
       [ 3.17617164e-01,  2.91775623e-01,  2.58170194e-01],
       [ 2.49869408e-01,  3.02010094e-01,  3.16259103e-01],
       [ 2.90939014e-01,  3.28011466e-01,  2.49727034e-01],
       [ 3.05622812e-01,  2.57216431e-01,  3.04507536e-01],
       [ 3.31079333e-01,  2.63498754e-01,  2.73742023e-01],
       [ 3.11754402e-01,  3.10096456e-01,  2.46564048e-01],
       [ 2.04176186e-01,  3.68467764e-01,  3.05194278e-01],
       [ 3.26804462e-01,  2.33755705e-01,  3.09703763e-01],
       [ 4.17104352e-01,  2.20217970e-01,  2.48141515e-01],
       [ 3.09656959e-01,  2.33065787e-01,  3.27661883e-01],
       [ 2.62059113e-01,  3.06910625e-01,  2.98032150e-01],
       [ 2.91319364e-01,  2.74407035e-01,  3.00604529e-01],
       [ 3.04899627e-01,  2.42917038e-01,  3.21156140e-01],
       [ 2.49860950e-01,  3.07270850e-01,  3.10922861e-01],
       [ 2.59120340e-01,  3.03818197e-01,  3.04250846e-01],
       [ 2.62602119e-01,  2.88525244e-01,  3.16138649e-01],
       [ 2.80404676e-01,  2.43483313e-01,  3.46879761e-01],
       [ 3.52582026e-01,  2.98025462e-01,  2.22748095e-01],
       [ 2.62434589e-01,  2.72236608e-01,  3.33954211e-01],
       [ 3.48993496e-01,  2.80777304e-01,  2.41374397e-01],
       [ 2.53537235e-01,  3.17521590e-01,  2.96810848e-01],
       [ 2.06527222e-01,  3.89220458e-01,  2.84710098e-01],
       [ 3.30298396e-01,  2.35237932e-01,  3.04668825e-01],
       [ 2.71872443e-01,  3.16291072e-01,  2.78849622e-01],
       [ 3.43700200e-01,  4.32685205e-01,  1.30458414e-01],
       [ 4.58006037e-01,  6.46081929e-01, -4.15813098e-02],
       [ 2.95963652e-01,  3.13980890e-01,  2.57520247e-01],
       [ 3.05523672e-01,  2.87980633e-01,  2.72980496e-01],
       [ 2.69230345e-01,  2.73147271e-01,  3.25346144e-01],
       [ 3.03320174e-01,  2.89295283e-01,  2.73787837e-01],
       [ 2.95233460e-01,  2.97493675e-01,  2.73599610e-01],
       [ 2.54393405e-01,  3.13277472e-01,  3.00005310e-01],
       [ 3.40010725e-01,  2.36990835e-01,  2.93622386e-01],
       [ 3.19285517e-01,  2.81777107e-01,  2.66249545e-01],
       [ 2.79594253e-01,  2.84513692e-01,  3.02161330e-01],
       [ 3.59400817e-01,  3.19074308e-01,  1.99453787e-01],
       [ 3.01869888e-01,  2.79009847e-01,  2.85386663e-01],
       [ 4.42560472e-01,  3.02749120e-01,  1.55660423e-01],
       [ 2.95079682e-01,  2.89937226e-01,  2.81095151e-01],
       [ 3.57495447e-01,  2.06262954e-01,  3.12655089e-01],
       [ 2.18996967e-01,  3.52180632e-01,  3.02661571e-01],
       [ 1.97928381e-01,  2.31970101e-01,  4.74731924e-01],
       [ 4.05899444e-01,  5.38461395e-01,  3.32911085e-02],
       [ 1.92186117e-01,  3.28930143e-01,  3.58430944e-01],
       [ 2.95704472e-01,  2.81899834e-01,  2.88503664e-01],
       [ 2.75391105e-01,  2.74568688e-01,  3.17089045e-01],
       [ 2.98848434e-01,  2.84664479e-01,  2.82647501e-01],
       [ 2.96338535e-01,  2.59619694e-01,  3.11290797e-01],
       [ 3.07493340e-01,  2.87405657e-01,  2.71684228e-01],
       [ 3.09019512e-01,  2.92560339e-01,  2.65290046e-01],
       [ 2.69395822e-01,  2.88018771e-01,  3.09301099e-01],
       [ 1.87626732e-01,  3.44240790e-01,  3.48604150e-01],
       [ 4.44159011e-01,  1.89704048e-01,  2.61477988e-01],
       [ 3.23279346e-01,  2.07456530e-01,  3.44679146e-01],
       [ 2.32142601e-01,  2.79487226e-01,  3.61822355e-01],
       [ 2.79360958e-01,  2.44540997e-01,  3.46791592e-01],
       [ 5.64386602e-01,  2.39052244e-01,  1.43236933e-01],
       [ 2.64174469e-01,  2.75494006e-01,  3.28389605e-01],
       [ 2.81560089e-01,  2.73624319e-01,  3.11533086e-01],
       [ 2.83747678e-01,  2.83796743e-01,  2.98608050e-01],
       [ 2.91213822e-01,  2.82027573e-01,  2.92842899e-01],
       [ 3.19648491e-01,  2.48706057e-01,  2.99991414e-01],
       [ 3.75848361e-01,  3.52781116e-01,  1.61135117e-01],
       [ 2.59837583e-01,  2.60846792e-01,  3.49966714e-01],
       [ 2.53975781e-01,  3.14864854e-01,  2.98909646e-01],
       [ 2.90632962e-01,  3.12472406e-01,  2.63942633e-01],
       [ 3.02372692e-01,  2.72126347e-01,  2.91935081e-01],
       [ 2.88436081e-01,  2.98508815e-01,  2.79241353e-01],
       [ 2.85012000e-01,  2.93123567e-01,  2.87919090e-01],
       [ 2.69364405e-01,  3.47696690e-01,  2.53366950e-01],
       [ 2.97368355e-01,  2.79985247e-01,  2.88802645e-01],
       [ 2.53866427e-01,  3.37090363e-01,  2.78233666e-01],
       [ 3.01435644e-01,  3.10760865e-01,  2.55351992e-01],
       [ 3.42064453e-01,  2.15071788e-01,  3.16675840e-01],
       [ 2.53890187e-01,  1.66932919e-01,  4.93360195e-01],
       [ 2.86726771e-01,  2.78240504e-01,  3.01293556e-01],
       [ 3.07091534e-01,  2.58444146e-01,  3.01721000e-01],
       [ 2.70502879e-01,  2.99284210e-01,  2.96677062e-01],
       [ 3.46093571e-01,  2.87352209e-01,  2.37667141e-01],
       [ 2.92623226e-01,  2.75439231e-01,  2.98206793e-01],
       [ 3.12863315e-01,  2.70948881e-01,  2.83019156e-01],
       [ 2.67310110e-01,  3.57620352e-01,  2.47074633e-01],
       [ 2.92490028e-01,  2.79581892e-01,  2.94062825e-01],
       [ 3.29294213e-01,  3.05743824e-01,  2.35136038e-01],
       [ 3.18923462e-01,  2.86285042e-01,  2.62218694e-01],
       [ 2.82181479e-01,  2.61541751e-01,  3.24057815e-01],
       [ 3.27850611e-01,  2.07016285e-01,  3.40513847e-01],
       [ 2.73191883e-01,  3.04550765e-01,  2.88708482e-01],
       [ 2.60412264e-01,  3.01032514e-01,  3.05653286e-01],
       [ 2.83367955e-01,  2.57779521e-01,  3.26996895e-01],
       [ 2.64937852e-01,  2.62978580e-01,  3.41582509e-01],
       [ 3.56069103e-01,  2.64803076e-01,  2.50795271e-01],
       [ 3.17410268e-01,  3.50231422e-01,  2.07945298e-01],
       [ 3.23566373e-01,  3.21696072e-01,  2.26124997e-01],
       [ 2.50361293e-01,  3.66221419e-01,  2.56757656e-01],
       [ 3.18870771e-01,  3.15861108e-01,  2.35187691e-01],
       [ 2.53554973e-01,  3.19493045e-01,  2.94898410e-01],
       [ 2.97770595e-01,  3.17525405e-01,  2.52643071e-01],
       [ 3.12377048e-01,  3.16787747e-01,  2.40068708e-01],
       [ 2.91549651e-01,  3.41093931e-01,  2.37976307e-01],
       [ 3.25063401e-01,  2.51231249e-01,  2.92096960e-01],
       [ 3.70528719e-01,  2.50140866e-01,  2.53461148e-01],
       [ 2.26889333e-01,  3.39891061e-01,  3.05025272e-01],
       [ 2.44440151e-01,  3.40538114e-01,  2.85068389e-01],
       [ 3.23641962e-01,  3.38074037e-01,  2.12455276e-01],
       [ 2.13196659e-01,  3.95146382e-01,  2.72471452e-01],
       [ 2.75380887e-01,  2.32822040e-01,  3.65769755e-01],
       [ 3.19387062e-01,  2.84318801e-01,  2.63691746e-01],
       [ 4.51921640e-01,  1.68432763e-01,  2.80293956e-01],
       [ 2.07342550e-01,  4.95747124e-01,  2.09376603e-01],
       [ 3.11552703e-01,  2.45915855e-01,  3.11020681e-01],
       [ 2.58957084e-01,  3.08855987e-01,  2.99428743e-01],
       [ 2.62044109e-01,  3.61049541e-01,  2.49383210e-01],
       [ 3.20767456e-01,  3.16183728e-01,  2.33265207e-01],
       [ 2.50457454e-01,  2.72200050e-01,  3.47886459e-01],
       [ 3.19649078e-01,  2.55076924e-01,  2.93122107e-01],
       [ 2.77551556e-01,  3.06830480e-01,  2.82073563e-01],
       [ 2.71112730e-01,  3.02674401e-01,  2.92688887e-01],
       [ 3.04525044e-01,  2.69820190e-01,  2.92216223e-01],
       [ 2.26047536e-01,  2.61971954e-01,  3.90930873e-01],
       [ 2.92401653e-01,  3.01496187e-01,  2.72508175e-01],
       [ 4.65952640e-01,  1.19048700e-01,  3.32527348e-01],
       [ 2.91549120e-01,  2.82395795e-01,  2.92132136e-01],
       [ 2.85774252e-01,  3.07271400e-01,  2.73486105e-01],
       [ 2.21999440e-01,  2.96560027e-01,  3.55145074e-01],
       [ 3.20222510e-01,  3.32416723e-01,  2.19957176e-01],
       [ 3.11950514e-01,  1.84892593e-01,  3.87089320e-01],
       [ 2.88715256e-01,  3.76164637e-01,  2.12663268e-01],
       [ 3.94782291e-01,  1.65635028e-01,  3.29415630e-01],
       [ 3.24422833e-01,  2.83795935e-01,  2.59660055e-01],
       [ 2.85387393e-01,  3.30509991e-01,  2.52763885e-01],
       [ 2.77645576e-01,  2.81191261e-01,  3.07654182e-01],
       [ 2.74538810e-01,  2.98404522e-01,  2.93356024e-01],
       [ 2.71942226e-01,  3.04791192e-01,  2.89760225e-01],
       [ 2.72273285e-01,  3.09024995e-01,  2.85328127e-01],
       [ 2.96651240e-01,  2.91350376e-01,  2.78180443e-01],
       [ 3.57020826e-01,  3.62817776e-01,  1.67352095e-01],
       [ 3.32447013e-01,  1.75415031e-01,  3.77432807e-01],
       [ 2.97890791e-01,  3.74329872e-01,  2.06019519e-01],
       [ 4.11099952e-01,  3.36514241e-01,  1.49353788e-01],
       [-4.81452267e-03,  4.19021375e-01,  6.08433652e-01],
       [ 3.60146800e-01,  1.98664880e-01,  3.19340998e-01],
       [ 2.64089189e-01,  2.90872613e-01,  3.12064159e-01],
       [ 3.02712057e-01,  2.82492982e-01,  2.81074081e-01],
       [ 2.67202476e-01,  3.27785876e-01,  2.72971398e-01],
       [ 2.71096044e-01,  2.82651545e-01,  3.13092622e-01],
       [ 2.67346150e-01,  3.08729894e-01,  2.90694697e-01],
       [ 2.40733020e-01,  2.62620147e-01,  3.71068806e-01],
       [ 4.98895462e-01,  1.44168030e-01,  2.76917018e-01],
       [ 4.50456979e-01,  2.18384539e-01,  2.26701178e-01],
       [ 3.34002335e-01,  2.62171187e-01,  2.72461298e-01],
       [ 3.26738963e-01,  2.53329716e-01,  2.88288813e-01],
       [ 2.76034338e-01,  2.88066947e-01,  3.02221667e-01],
       [ 2.88785544e-01,  2.95514787e-01,  2.81806449e-01],
       [ 2.87498384e-01,  2.99071402e-01,  2.79621383e-01],
       [ 2.38883925e-01,  2.79828850e-01,  3.53092107e-01],
       [ 2.80066369e-01,  3.15407619e-01,  2.71489496e-01],
       [ 3.43652511e-01,  2.52452315e-01,  2.73851377e-01],
       [ 3.30179384e-01,  2.50619365e-01,  2.87966619e-01],
       [ 2.95845194e-01,  3.60074024e-01,  2.18737043e-01],
       [ 1.82340699e-01,  3.42788961e-01,  3.57046336e-01],
       [ 2.90592142e-01,  2.70095893e-01,  3.05896051e-01],
       [ 3.08443370e-01,  2.61423839e-01,  2.97201783e-01],
       [ 3.05410417e-01,  2.71769646e-01,  2.89335564e-01],
       [ 2.74365252e-01,  3.43175438e-01,  2.52363265e-01],
       [ 2.39005012e-01,  3.14213483e-01,  3.16152487e-01],
       [ 3.41096361e-01,  3.45429835e-01,  1.92527454e-01],
       [ 2.64553818e-01,  2.72301885e-01,  3.31488509e-01],
       [ 3.62937507e-01,  2.30208723e-01,  2.80619873e-01],
       [ 2.93183170e-01,  3.05180069e-01,  2.68275738e-01],
       [ 2.92242516e-01,  2.67881830e-01,  3.06563250e-01],
       [ 2.99284900e-01,  3.05733696e-01,  2.61972975e-01],
       [ 3.03422324e-01,  2.55133752e-01,  3.08987282e-01],
       [ 2.87278974e-01,  2.73275351e-01,  3.05935982e-01],
       [ 2.83249936e-01,  3.10652033e-01,  2.72785475e-01],
       [ 2.66070666e-01,  3.16011651e-01,  2.85043136e-01],
       [ 2.65111883e-01,  3.31390926e-01,  2.71824918e-01],
       [ 3.36311999e-01,  2.75176458e-01,  2.57494403e-01],
       [ 3.48702463e-01,  2.41284822e-01,  2.81130445e-01],
       [ 2.61465696e-01,  2.97724562e-01,  3.07865236e-01],
       [ 2.54926501e-01,  2.58749129e-01,  3.58276661e-01],
       [ 2.52552099e-01,  3.32057647e-01,  2.84186408e-01],
       [ 3.07696993e-01,  2.93850137e-01,  2.65287935e-01],
       [ 3.09355571e-01,  2.98943302e-01,  2.58952024e-01],
       [ 3.18991269e-01,  2.39171356e-01,  3.11211138e-01],
       [ 3.11528806e-01,  2.58759749e-01,  2.97023118e-01],
       [ 2.99575810e-01,  3.24583839e-01,  2.44749134e-01],
       [ 2.79529320e-01,  3.56494288e-01,  2.36389644e-01],
       [ 3.20517897e-01,  3.11920940e-01,  2.37214648e-01],
       [ 3.33230914e-01,  2.87472658e-01,  2.48436179e-01],
       [ 3.02602147e-01,  2.13779084e-01,  3.58862753e-01],
       [ 2.85918571e-01,  1.97469193e-01,  4.00382271e-01],
       [ 3.46005491e-01,  2.94129195e-01,  2.31560149e-01],
       [ 3.25158646e-01,  2.70754642e-01,  2.71787507e-01],
       [ 2.77667071e-01,  2.20843089e-01,  3.78486060e-01],
       [ 3.25161556e-01,  1.47559663e-01,  4.27353928e-01],
       [ 3.02978604e-01,  3.26988836e-01,  2.39582824e-01],
       [ 3.01579950e-01,  2.72170113e-01,  2.92669164e-01],
       [ 3.03676511e-01,  3.25417361e-01,  2.40311340e-01],
       [ 2.14294643e-01,  2.88865116e-01,  3.73833062e-01],
       [ 2.65101686e-01,  3.65236029e-01,  2.43005153e-01],
       [ 2.69679777e-01,  3.92811277e-01,  2.17462170e-01],
       [ 2.37026562e-01,  2.94139554e-01,  3.39406274e-01],
       [ 2.79175443e-01,  2.81814609e-01,  3.05396292e-01],
       [ 2.93136785e-01,  2.73603117e-01,  2.99602801e-01],
       [ 3.84091483e-01,  2.72290082e-01,  2.21541412e-01],
       [ 2.80100915e-01,  3.17928423e-01,  2.69130539e-01],
       [ 3.14050626e-01,  2.96919612e-01,  2.56562760e-01],
       [ 4.59492247e-01,  1.06514714e-01,  3.55220569e-01],
       [ 2.99387291e-01,  2.53232539e-01,  3.15199322e-01],
       [ 2.54878411e-01,  2.97150380e-01,  3.15676984e-01],
       [ 2.92489377e-01,  2.80604317e-01,  2.93017041e-01],
       [ 3.60739318e-01,  2.26967543e-01,  2.86067586e-01],
       [ 2.22084974e-01,  3.13821683e-01,  3.36448367e-01],
       [ 1.71462116e-01,  1.85398961e-01,  6.11473527e-01],
       [ 4.84385749e-01,  8.74928443e-01, -1.27860704e-01],
       [ 2.88627294e-01,  3.21419668e-01,  2.57733455e-01],
       [ 2.90738278e-01,  3.04451567e-01,  2.71315576e-01],
       [ 3.00597615e-01,  3.02182223e-01,  2.64051209e-01],
       [ 2.90984035e-01,  2.77401139e-01,  2.97827433e-01],
       [ 2.95978274e-01,  2.83842166e-01,  2.86276361e-01],
       [ 2.84205030e-01,  2.63974643e-01,  3.19196583e-01],
       [ 2.36978079e-01,  3.35807047e-01,  2.97530580e-01],
       [ 3.31307160e-01,  2.28519617e-01,  3.11328115e-01],
       [ 2.43025289e-01,  3.48234899e-01,  2.79690932e-01],
       [ 1.92549521e-01,  3.10193026e-01,  3.78469025e-01],
       [ 4.11717702e-01,  2.03160013e-01,  2.70550424e-01],
       [ 4.11797648e-01,  1.56977507e-01,  3.25888060e-01],
       [ 2.68810134e-01,  2.43522951e-01,  3.60192556e-01],
       [ 2.60751934e-01,  3.50059035e-01,  2.59853601e-01],
       [ 3.32724501e-01,  2.60548239e-01,  2.75268366e-01],
       [ 2.81054140e-01,  3.21777848e-01,  2.64689013e-01],
       [ 2.88469367e-01,  3.11405010e-01,  2.67004560e-01],
       [ 2.96197185e-01,  2.64503020e-01,  3.06143905e-01],
       [ 2.75295399e-01,  3.70247607e-01,  2.29376720e-01],
       [ 6.27082693e-01, -2.10632141e-02,  4.34323295e-01],
       [ 3.41804622e-01,  3.45438540e-01,  1.91966275e-01],
       [ 3.18092035e-01,  3.03323800e-01,  2.47040880e-01],
       [ 2.45604818e-01,  2.80356100e-01,  3.44404279e-01],
       [ 2.89580006e-01,  3.12718685e-01,  2.64724004e-01],
       [ 2.94333471e-01,  2.77265914e-01,  2.94596819e-01],
       [ 2.81644694e-01,  2.89467662e-01,  2.94990937e-01],
       [ 2.76017298e-01,  2.93338213e-01,  2.96885822e-01],
       [ 2.68425146e-01,  3.45702919e-01,  2.55980230e-01],
       [ 2.60522681e-01,  2.95003283e-01,  3.11677518e-01],
       [ 3.27841251e-01,  2.32031434e-01,  3.10660851e-01],
       [ 3.10748212e-01,  2.33904885e-01,  3.25554884e-01],
       [ 2.87585149e-01,  2.88442773e-01,  2.90000078e-01],
       [ 2.19159572e-01,  3.36624330e-01,  3.17075675e-01],
       [ 2.63429378e-01,  2.74070682e-01,  3.30793881e-01],
       [ 2.78799096e-01,  3.41167850e-01,  2.49824144e-01],
       [ 2.22255340e-01,  4.17049903e-01,  2.46061539e-01],
       [ 3.41955415e-01,  2.29697626e-01,  2.99926076e-01],
       [ 3.50434070e-01,  2.33644376e-01,  2.87843271e-01],
       [ 3.60023862e-01,  2.11609549e-01,  3.04068148e-01],
       [ 3.25555450e-01,  2.30538554e-01,  3.14599539e-01],
       [ 2.68836246e-01,  3.18577638e-01,  2.79793835e-01],
       [ 2.83999550e-01,  2.59458945e-01,  3.24429148e-01],
       [ 2.95034454e-01,  2.97258662e-01,  2.74017118e-01],
       [ 2.86986329e-01,  2.95730450e-01,  2.83378495e-01],
       [ 2.80867918e-01,  2.88146566e-01,  2.97125782e-01],
       [ 3.13744034e-01,  2.77518387e-01,  2.75563364e-01],
       [ 2.86718046e-01,  2.65704381e-01,  3.14646456e-01],
       [ 3.39227998e-01,  2.71758278e-01,  2.58290554e-01],
       [ 2.67464100e-01,  3.27348132e-01,  2.73104886e-01],
       [ 2.44817888e-01,  2.86814878e-01,  3.38170630e-01],
       [ 3.14041347e-01,  3.22257160e-01,  2.33849249e-01],
       [ 2.43804695e-01,  3.30077618e-01,  2.95401205e-01],
       [ 3.39747419e-01,  2.64810186e-01,  2.64708312e-01],
       [ 3.08897011e-01,  2.66377063e-01,  2.91542638e-01],
       [ 3.05200616e-01,  2.59803640e-01,  3.02135368e-01],
       [ 2.20595200e-01,  3.24721598e-01,  3.27090556e-01],
       [ 2.33558006e-01,  3.05201944e-01,  3.31720356e-01],
       [ 2.20019864e-01,  3.36174276e-01,  3.16499166e-01],
       [ 2.43648193e-01,  2.68877898e-01,  3.59959014e-01],
       [ 2.42525996e-01,  2.95347074e-01,  3.31622211e-01],
       [ 2.63834840e-01,  3.16100819e-01,  2.87275991e-01],
       [ 4.21835540e-01,  2.59617811e-01,  2.06153481e-01],
       [ 2.47500879e-01,  3.47774948e-01,  2.75377206e-01],
       [ 2.76527849e-01,  2.69435913e-01,  3.21437939e-01],
       [ 2.27365698e-01,  3.00747090e-01,  3.43904621e-01],
       [ 3.74218066e-01,  2.13203118e-01,  2.89767455e-01],
       [ 3.36392375e-01,  2.54897776e-01,  2.77789730e-01],
       [ 2.99049753e-01,  2.66597766e-01,  3.01024928e-01],
       [ 2.87268876e-01,  3.11131483e-01,  2.68418303e-01],
       [ 3.29943611e-01,  2.97925178e-01,  2.41613365e-01],
       [ 2.26539023e-01,  2.89321835e-01,  3.57571836e-01],
       [ 3.14335230e-01,  2.75282635e-01,  2.77245619e-01],
       [ 3.83979601e-01,  2.63365606e-01,  2.29974638e-01],
       [ 3.85948490e-01,  2.93755559e-01,  2.01007314e-01],
       [ 2.61831765e-01,  3.19424240e-01,  2.86215689e-01],
       [ 2.82657550e-01,  2.99168602e-01,  2.84342213e-01],
       [ 2.90401874e-01,  2.78865598e-01,  2.96902452e-01],
       [ 3.49680118e-01,  2.65979502e-01,  2.54998462e-01],
       [ 2.46197499e-01,  2.64402371e-01,  3.62132480e-01],
       [ 4.57273904e-01,  2.76797074e-01,  1.68141671e-01],
       [ 2.51853285e-01,  3.56282977e-01,  2.63541202e-01],
       [ 3.33324634e-01,  2.57032850e-01,  2.78347652e-01],
       [ 2.81821654e-01,  3.20853511e-01,  2.64780336e-01],
       [ 2.62221449e-01,  2.84997871e-01,  3.20287386e-01],
       [ 2.09017851e-01,  2.86437907e-01,  3.83746402e-01],
       [ 3.67435580e-01,  2.03670516e-01,  3.06710446e-01],
       [ 2.54923904e-01,  1.65315372e-01,  4.94616214e-01],
       [ 3.22702832e-01,  2.58115504e-01,  2.87018136e-01],
       [ 2.61428532e-01,  2.63930791e-01,  3.44527932e-01],
       [ 2.66821494e-01,  2.92280606e-01,  3.07659821e-01],
       [ 2.83947427e-01,  3.09613552e-01,  2.73073929e-01],
       [ 2.75947172e-01,  2.74203443e-01,  3.16884742e-01],
       [ 2.77062089e-01,  3.06513626e-01,  2.82871083e-01],
       [ 3.40171520e-01,  2.53254001e-01,  2.76109125e-01],
       [ 2.12657578e-01,  3.16554339e-01,  3.45196915e-01],
       [ 3.31992508e-01,  2.97764605e-01,  2.40004248e-01],
       [ 2.78563241e-01,  3.26167004e-01,  2.63161738e-01],
       [ 2.05485896e-01,  5.21449310e-01,  1.96509290e-01],
       [ 2.89139399e-01,  2.87619000e-01,  2.89268462e-01],
       [-2.83814945e-02,  5.31844466e-01,  5.26542280e-01],
       [ 2.99915292e-01,  2.99292861e-01,  2.67415410e-01],
       [ 2.85560256e-01,  2.87076684e-01,  2.93418564e-01],
       [ 3.28666987e-01,  2.63501091e-01,  2.75928231e-01],
       [ 3.15357777e-01,  2.63275346e-01,  2.88566391e-01],
       [ 2.54921484e-01,  3.25889283e-01,  2.87397945e-01],
       [ 2.17808519e-01,  4.76399013e-01,  2.10652045e-01],
       [ 5.42647250e-01,  3.82427944e-01,  4.59168380e-02],
       [ 2.63551188e-01,  3.14125461e-01,  2.89455809e-01],
       [ 2.82773399e-01,  2.90914694e-01,  2.92383728e-01],
       [ 3.42361681e-01,  2.51883039e-01,  2.75584946e-01],
       [ 2.82528052e-01,  3.16947450e-01,  2.67656400e-01],
       [ 2.87788541e-01,  2.99622646e-01,  2.78803058e-01],
       [ 2.90868411e-01,  2.91065673e-01,  2.84118416e-01],
       [ 2.78830912e-01,  3.17107571e-01,  2.71136386e-01],
       [ 2.74114973e-01,  3.41221043e-01,  2.54277661e-01],
       [ 2.56796326e-01,  2.86950874e-01,  3.24253751e-01],
       [ 3.04866489e-01,  3.22189567e-01,  2.42044067e-01],
       [ 2.49091523e-01,  3.04130696e-01,  3.14961359e-01],
       [ 3.02700752e-01,  2.86649321e-01,  2.76967871e-01],
       [ 6.12213508e-01,  1.41271901e-01,  2.17006769e-01],
       [ 3.08754999e-01,  2.76596141e-01,  2.81197981e-01],
       [ 2.24546484e-01,  3.06349743e-01,  3.41328932e-01],
       [ 4.41520682e-01, -8.41753977e-02,  8.03607021e-01],
       [ 3.06044673e-01,  3.43682185e-01,  2.22890429e-01],
       [ 2.29746324e-01,  3.07887999e-01,  3.33430840e-01],
       [ 3.34414267e-01,  3.17473957e-01,  2.20639328e-01],
       [ 2.97123138e-01,  2.85448142e-01,  2.83547710e-01],
       [ 2.89197146e-01,  2.78245895e-01,  2.98764932e-01],
       [ 3.01786611e-01,  3.15075417e-01,  2.51133075e-01],
       [ 3.31331439e-01,  2.59841791e-01,  2.77255865e-01],
       [ 2.71002790e-01,  3.16166301e-01,  2.79847922e-01],
       [ 3.06055458e-01,  2.96646470e-01,  2.64159979e-01],
       [ 2.63719679e-01,  3.19038244e-01,  2.84617901e-01],
       [ 2.73781788e-01,  2.99959729e-01,  2.92599513e-01],
       [ 2.58530912e-01,  3.30019462e-01,  2.79805731e-01],
       [ 3.18023905e-01,  2.76047523e-01,  2.73048460e-01],
       [ 2.82519610e-01,  3.20407285e-01,  2.64507455e-01],
       [ 2.60812893e-01,  2.85560884e-01,  3.21249018e-01],
       [ 2.94215924e-01,  2.59184840e-01,  3.13956366e-01],
       [ 2.67023442e-01,  2.81569096e-01,  3.18659914e-01],
       [ 2.82433682e-01,  3.03360849e-01,  2.80509068e-01],
       [ 2.67180909e-01,  3.20108184e-01,  2.80054927e-01],
       [ 3.00026229e-01,  3.59499092e-01,  2.15519918e-01],
       [ 2.39197960e-01,  2.96082015e-01,  3.34736576e-01],
       [ 2.11946693e-01,  3.60117560e-01,  3.03592251e-01],
       [ 3.43206110e-01,  2.29274022e-01,  2.99244891e-01],
       [ 3.00731247e-01,  2.74464972e-01,  2.91135097e-01],
       [ 2.98711178e-01,  2.96401256e-01,  2.71312818e-01],
       [ 2.65872302e-01,  2.04438962e-01,  4.15991191e-01],
       [ 3.28642597e-01,  2.39050530e-01,  3.01990292e-01],
       [ 2.72325845e-01,  3.34077370e-01,  2.62238217e-01],
       [ 2.99548614e-01,  3.00727215e-01,  2.66406828e-01],
       [ 3.11034759e-01,  2.29935704e-01,  3.29929296e-01],
       [ 2.22889753e-01,  3.06086042e-01,  3.43638706e-01],
       [ 2.54921142e-01,  3.92494052e-01,  2.31605574e-01],
       [ 2.45267314e-01,  3.22202563e-01,  3.01292192e-01],
       [ 3.61474361e-01,  2.38626764e-01,  2.72858250e-01],
       [ 1.97061657e-01,  3.86150377e-01,  2.98184120e-01],
       [ 4.01163334e-01,  1.76004946e-01,  3.10816231e-01],
       [ 2.48197037e-01,  2.35935593e-01,  3.95431898e-01],
       [ 2.63727658e-01,  2.53731985e-01,  3.53812825e-01],
       [ 4.05486144e-01,  2.89811053e-01,  1.90545329e-01],
       [ 2.93447949e-01,  2.71916669e-01,  3.01056355e-01],
       [ 2.74013595e-01,  3.48216147e-01,  2.48435636e-01],
       [ 3.60803147e-01,  2.15033954e-01,  2.99416401e-01],
       [ 2.53028585e-01,  2.68034109e-01,  3.49631072e-01],
       [ 2.95800576e-01,  3.58497174e-01,  2.20016543e-01],
       [ 2.39489150e-01,  2.82625878e-01,  3.49183914e-01],
       [ 3.18462657e-01,  3.74559972e-01,  1.88618136e-01],
       [ 3.24476994e-01,  2.65285016e-01,  2.77944514e-01],
       [ 2.97443632e-01,  2.24715428e-01,  3.50774009e-01],
       [ 2.61033336e-01,  3.08653257e-01,  2.97411266e-01],
       [ 2.45194477e-01,  2.58735037e-01,  3.70209707e-01],
       [ 2.80903893e-01,  3.46576401e-01,  2.43267145e-01],
       [ 2.85004818e-01,  4.04955345e-01,  1.95063110e-01],
       [ 2.21721400e-01,  2.80456534e-01,  3.74004454e-01],
       [ 3.18951094e-01,  2.45223040e-01,  3.04490817e-01],
       [ 3.40626257e-01,  2.28917093e-01,  3.02039920e-01],
       [ 2.94578775e-01,  2.96479509e-01,  2.75207088e-01],
       [ 2.84534343e-01,  2.82794582e-01,  2.98830576e-01],
       [ 2.77712734e-01,  3.04738225e-01,  2.83921440e-01],
       [ 3.33436865e-01,  2.79081761e-01,  2.56227070e-01],
       [ 2.62855256e-01,  3.63170495e-01,  2.46856821e-01],
       [ 3.33947767e-01,  2.58959715e-01,  2.75795105e-01],
       [ 2.73770513e-01,  2.90889645e-01,  3.01708895e-01],
       [ 2.61742007e-01,  3.20853155e-01,  2.84965036e-01],
       [ 1.62768235e-01,  3.67363823e-01,  3.58787657e-01],
       [ 2.59927245e-01,  3.42691193e-01,  2.67043310e-01],
       [ 2.46529856e-02,  4.82712400e-01,  4.69286445e-01],
       [ 2.42321703e-01,  2.89994865e-01,  3.37633584e-01],
       [ 2.63109969e-01,  2.71173498e-01,  3.34375964e-01],
       [ 2.61395838e-01,  2.80969918e-01,  3.25528725e-01],
       [ 2.94079122e-01,  3.21362309e-01,  2.52654511e-01],
       [ 3.19021299e-01,  2.22368771e-01,  3.30740506e-01],
       [ 3.64597589e-01,  2.25476353e-01,  2.84357356e-01],
       [ 2.90108400e-01,  2.56381932e-01,  3.21362406e-01],
       [ 2.83156961e-01,  2.68745655e-01,  3.15096888e-01],
       [ 2.77148903e-01,  3.20695877e-01,  2.69500537e-01],
       [ 2.53777215e-01,  2.77255086e-01,  3.38282579e-01],
       [ 2.54344134e-01,  2.85052316e-01,  3.29068935e-01],
       [ 3.08383175e-01,  2.67188960e-01,  2.91194466e-01],
       [ 2.91125466e-01,  2.81643964e-01,  2.93322690e-01],
       [ 2.90105552e-01,  2.86464366e-01,  2.89462027e-01],
       [ 3.05201708e-01,  2.70852822e-01,  2.90485098e-01],
       [ 3.17759771e-01,  2.54715022e-01,  2.95317480e-01],
       [ 2.85317794e-01,  3.30116257e-01,  2.53174093e-01],
       [ 3.07766510e-01,  2.72051609e-01,  2.86765143e-01],
       [ 1.58946201e-01,  3.48994964e-01,  3.82974624e-01],
       [ 2.69974339e-01,  3.68427573e-01,  2.35798181e-01],
       [ 3.44738547e-01,  2.23282029e-01,  3.04612342e-01],
       [ 4.13384810e-01,  3.90891373e-01,  1.09926722e-01],
       [ 2.35120364e-01,  2.33068848e-01,  4.16927095e-01],
       [ 2.78290367e-01,  3.20563869e-01,  2.68496328e-01],
       [ 3.12145269e-01,  3.30671005e-01,  2.28343021e-01],
       [ 2.74544117e-01,  2.82263062e-01,  3.09813420e-01],
       [ 3.22461945e-01,  2.98662352e-01,  2.47442835e-01],
       [ 3.05897603e-01,  2.72242241e-01,  2.88376165e-01],
       [ 3.05700229e-01,  2.95859450e-01,  2.65236690e-01],
       [ 2.73840210e-01,  2.74357970e-01,  3.18990034e-01],
       [ 2.87278639e-01,  2.61667442e-01,  3.18480337e-01],
       [ 3.00164377e-01,  2.63409305e-01,  3.03303571e-01],
       [ 2.90272500e-01,  2.94753785e-01,  2.81083237e-01],
       [ 2.91115233e-01,  2.60964685e-01,  3.15224660e-01],
       [ 2.90279025e-01,  2.89212421e-01,  2.86540382e-01],
       [ 3.32061972e-01,  2.85993986e-01,  2.50838571e-01],
       [ 2.81245919e-01,  2.85729536e-01,  2.99201195e-01],
       [ 3.20712665e-01,  2.42381196e-01,  3.05926405e-01],
       [ 2.57172206e-01,  2.89686409e-01,  3.20925194e-01],
       [ 2.46599468e-01,  3.21780040e-01,  3.00238152e-01],
       [ 2.78774408e-01,  3.09960625e-01,  2.77868482e-01],
       [ 2.84796958e-01,  3.00133600e-01,  2.81269052e-01],
       [ 3.02079818e-01,  2.91988585e-01,  2.72352713e-01],
       [ 3.26449359e-01,  3.08841733e-01,  2.34819622e-01],
       [ 3.55912673e-01,  2.33136205e-01,  2.83548406e-01],
       [ 2.82174977e-01,  3.32222625e-01,  2.54322426e-01],
       [ 2.87105348e-01,  2.85245872e-01,  2.93708441e-01],
       [ 2.64980848e-01,  2.85610161e-01,  3.16603022e-01],
       [ 2.62397106e-01,  3.10082858e-01,  2.94569532e-01],
       [ 4.64986799e-01,  2.26665408e-01,  2.09069784e-01],
       [ 2.64885582e-01,  2.90422243e-01,  3.11667380e-01],
       [ 2.98344365e-01,  2.79534088e-01,  2.88300384e-01],
       [ 2.86730021e-01,  2.78920043e-01,  3.00583808e-01],
       [ 2.99164260e-01,  3.10648843e-01,  2.57562469e-01],
       [ 5.16480158e-01,  2.36609552e-01,  1.69695402e-01],
       [ 2.22896035e-01,  2.57468814e-01,  4.00968603e-01],
       [ 2.47111526e-01,  2.92939493e-01,  3.28878696e-01],
       [ 2.09376156e-01,  3.17515149e-01,  3.48306941e-01],
       [ 2.47561388e-01,  1.86155543e-01,  4.70157057e-01],
       [ 4.23415297e-01,  3.12631937e-01,  1.59808841e-01],
       [ 2.76291524e-01,  2.79207256e-01,  3.11175124e-01],
       [ 3.93626808e-01,  2.00789018e-01,  2.87616938e-01],
       [ 4.02236780e-01,  2.10407334e-01,  2.69922502e-01],
       [ 2.81493149e-01,  2.81313563e-01,  3.03500572e-01],
       [ 3.48744327e-01,  2.57377018e-01,  2.64371526e-01],
       [ 2.68619325e-01,  3.02481257e-01,  2.95478054e-01],
       [ 2.55847844e-01,  2.56763120e-01,  3.59546952e-01],
       [ 1.32685587e-01,  4.27955603e-01,  3.44634791e-01],
       [ 3.69654250e-01,  2.22894541e-01,  2.82856178e-01],
       [ 2.20653724e-01,  2.61265998e-01,  3.99134288e-01],
       [ 2.61248494e-01,  2.20344779e-01,  3.99580454e-01],
       [ 2.76657730e-01,  3.47553431e-01,  2.46465726e-01],
       [ 2.83367163e-01,  4.02345156e-01,  1.98317269e-01],
       [ 2.75455331e-01,  2.56719444e-01,  3.36891692e-01],
       [ 2.87421581e-01,  3.08759649e-01,  2.70481198e-01],
       [ 3.92721477e-01,  2.58882204e-01,  2.27640516e-01],
       [ 3.44226386e-01,  2.60665929e-01,  2.64959407e-01],
       [ 4.62742635e-01,  1.72317762e-01,  2.68102412e-01],
       [ 2.49651738e-01,  2.98500463e-01,  3.20127950e-01],
       [ 3.02970486e-01,  3.00024084e-01,  2.63852388e-01],
       [ 2.98981632e-01,  2.80721046e-01,  2.86473686e-01],
       [ 3.21961441e-01,  2.72604474e-01,  2.72857671e-01],
       [ 2.68302084e-01,  2.61187166e-01,  3.39804706e-01],
       [ 3.14383296e-01,  2.84264410e-01,  2.68324786e-01],
       [ 3.71228745e-01,  2.44478429e-01,  2.58633626e-01],
       [ 2.95505515e-01,  2.89963922e-01,  2.80653526e-01],
       [ 2.83679673e-01,  2.74826425e-01,  3.08031246e-01],
       [ 2.62355246e-01,  2.79077731e-01,  3.26508545e-01],
       [ 3.47088064e-01,  1.99574585e-01,  3.30606169e-01],
       [ 2.59519066e-01,  1.90999638e-01,  4.44891523e-01],
       [ 2.78221581e-01,  3.25288380e-01,  2.64283546e-01],
       [ 2.72167781e-01,  2.93332323e-01,  3.00909568e-01],
       [ 2.81173530e-01,  3.25821176e-01,  2.60937547e-01],
       [ 2.53890180e-01,  2.97181519e-01,  3.16743775e-01],
       [ 2.46329200e-01,  3.17898278e-01,  3.04296366e-01],
       [ 3.18091881e-01,  2.30483503e-01,  3.22079834e-01],
       [ 2.96083001e-01,  2.72024530e-01,  2.98285362e-01],
       [ 2.93601354e-01,  3.04647143e-01,  2.68375411e-01],
       [ 2.99324951e-01,  2.61465608e-01,  3.06240570e-01],
       [ 3.32372128e-01,  2.60392891e-01,  2.75746131e-01],
       [ 3.24788188e-01,  2.65694735e-01,  2.77240005e-01],
       [ 2.85798435e-01,  3.07993839e-01,  2.72781998e-01],
       [ 2.88263025e-01,  2.90686171e-01,  2.87082056e-01],
       [ 2.79412499e-01,  3.05896375e-01,  2.81097274e-01],
       [ 2.73647128e-01,  3.17356101e-01,  2.76066875e-01],
       [ 2.80363147e-01,  3.26319979e-01,  2.61276269e-01],
       [ 2.64532646e-01,  2.80980490e-01,  3.22029802e-01],
       [ 3.04273520e-01,  2.80909896e-01,  2.81153827e-01],
       [ 1.88146654e-01,  3.25088966e-01,  3.67931593e-01],
       [ 3.08558058e-01,  3.25641466e-01,  2.35762873e-01],
       [ 3.11210112e-01,  2.51882221e-01,  3.04766546e-01],
       [ 3.65515023e-01,  2.66738305e-01,  2.41205756e-01],
       [ 2.95426820e-01,  2.90530360e-01,  2.80173270e-01],
       [ 3.02803620e-01,  2.57378299e-01,  3.07159002e-01],
       [ 3.31089444e-01,  2.50559151e-01,  2.87188023e-01],
       [ 2.97862829e-01,  3.06643161e-01,  2.62466218e-01],
       [ 2.90599190e-01,  2.66373399e-01,  3.09875404e-01],
       [ 2.45978282e-01,  3.20878292e-01,  3.01788701e-01],
       [ 3.43276052e-01,  2.71082500e-01,  2.55459730e-01],
       [ 3.20612048e-01,  2.86185315e-01,  2.60788114e-01],
       [ 3.06441119e-01,  2.56127149e-01,  3.04873416e-01],
       [ 3.14349300e-01,  2.59180430e-01,  2.93841460e-01],
       [ 3.06045675e-01,  3.00866328e-01,  2.60204380e-01],
       [ 2.81048371e-01,  2.80055756e-01,  3.05274508e-01],
       [ 2.46841965e-01,  3.09678477e-01,  3.11863399e-01],
       [ 2.98541056e-01,  2.97012258e-01,  2.70890352e-01],
       [ 2.60549262e-01,  3.04598687e-01,  3.01933392e-01],
       [ 3.02906734e-01,  2.74567736e-01,  2.88898631e-01],
       [ 3.09600000e-01,  2.86514904e-01,  2.70577006e-01],
       [ 2.85347075e-01,  2.97965387e-01,  2.82826888e-01],
       [ 3.41914414e-01,  2.09028766e-01,  3.24044400e-01],
       [ 3.46967494e-01,  2.49939830e-01,  2.73541632e-01],
       [ 2.81034470e-01,  2.42982556e-01,  3.46770270e-01],
       [ 2.65077786e-01,  3.16498680e-01,  2.85608584e-01],
       [ 3.24305924e-01,  2.73788840e-01,  2.69536982e-01],
       [ 2.72505628e-01,  3.07878024e-01,  2.86192599e-01],
       [ 2.93411567e-01,  2.58382040e-01,  3.15675497e-01],
       [ 2.67594743e-01,  2.86845320e-01,  3.12462450e-01],
       [ 2.55894609e-01,  4.34017987e-01,  2.01383680e-01],
       [ 2.75850568e-01,  5.71772319e-02,  7.03327555e-01],
       [ 3.80615457e-01,  2.44963107e-01,  2.50589237e-01],
       [ 2.28512189e-01,  2.52496725e-01,  3.99787644e-01],
       [ 2.50809042e-01,  3.06989325e-01,  3.10155626e-01],
       [ 6.02664892e-01,  4.20180927e-01, -3.15618703e-03],
       [ 1.56423535e-01,  2.40154978e-01,  5.35667219e-01],
       [ 3.39695880e-01,  1.82336501e-01,  3.60247849e-01],
       [ 1.34745781e-01,  4.01028777e-01,  3.65756569e-01],
       [ 2.43931545e-01,  4.07315391e-01,  2.31314605e-01],
       [-5.61381775e-02,  6.38433773e-01,  4.90885575e-01],
       [ 2.88079190e-01,  2.59546138e-01,  3.19981838e-01],
       [ 3.77643766e-01,  3.35049181e-01,  1.73245387e-01],
       [ 2.92861086e-01,  2.50623083e-01,  3.24944611e-01],
       [ 3.18867988e-01,  2.64217069e-01,  2.84263218e-01],
       [ 3.76961719e-01,  1.95253646e-01,  3.08270034e-01],
       [ 2.77283043e-01,  3.63397411e-01,  2.32933687e-01],
       [ 3.06019844e-01,  2.58747354e-01,  3.02457677e-01],
       [ 3.53093325e-01,  2.42988936e-01,  2.75469074e-01],
       [ 2.69945831e-01,  2.37796702e-01,  3.65948802e-01],
       [ 1.30560588e-01,  2.95114664e-01,  4.96786353e-01],
       [ 2.71129034e-01,  4.01353452e-01,  2.09940674e-01],
       [ 3.45939094e-01,  2.27208798e-01,  2.99049506e-01],
       [ 3.04965927e-01,  3.45159260e-01,  2.22631255e-01],
       [ 2.96684212e-01,  3.26969124e-01,  2.45318055e-01],
       [ 3.25712268e-01,  1.79144832e-01,  3.79613024e-01],
       [ 3.38053408e-01,  3.17059050e-01,  2.18003956e-01],
       [ 3.78741402e-01,  2.55379627e-01,  2.41715942e-01],
       [ 2.76500267e-01,  2.34823628e-01,  3.61945150e-01],
       [ 2.42725577e-01,  4.06658617e-01,  2.32980036e-01],
       [ 2.45927160e-01,  2.56871476e-01,  3.71576839e-01],
       [ 3.10939086e-01,  3.12539537e-01,  2.45107428e-01],
       [ 2.65690042e-01,  3.08307469e-01,  2.92833631e-01],
       [ 2.91595210e-01,  2.88194321e-01,  2.86248557e-01],
       [ 3.08641001e-01,  2.93811216e-01,  2.64448878e-01],
       [ 2.70090275e-01,  3.25626535e-01,  2.72027643e-01],
       [ 2.78555773e-01,  2.47344482e-01,  3.44363336e-01],
       [ 3.20556145e-01,  2.59133214e-01,  2.87970192e-01],
       [ 2.79962709e-01,  2.71682713e-01,  3.15309373e-01],
       [ 2.65387147e-01,  2.61599109e-01,  3.42655917e-01],
       [ 2.47925636e-01,  2.84294141e-01,  3.37297112e-01],
       [ 3.17312118e-01,  3.32204502e-01,  2.22607831e-01],
       [ 2.05251569e-01,  3.01109896e-01,  3.71664779e-01],
       [ 2.71643577e-01,  3.40669304e-01,  2.57155086e-01],
       [ 3.19006120e-01,  2.57130431e-01,  2.91552096e-01],
       [ 2.87382164e-01,  3.25986734e-01,  2.54850267e-01],
       [ 3.04516384e-01,  3.20731422e-01,  2.43634644e-01],
       [ 3.02978018e-01,  3.13200967e-01,  2.51723923e-01],
       [ 2.60530088e-01,  2.89889788e-01,  3.16985262e-01],
       [ 2.99524589e-01,  2.80743288e-01,  2.85920502e-01],
       [ 2.67565479e-01,  2.99215738e-01,  2.99833856e-01],
       [ 2.60856934e-01,  3.02006352e-01,  3.04193849e-01],
       [ 2.87994364e-01,  3.22033882e-01,  2.57784878e-01],
       [ 2.60855844e-01,  2.92344267e-01,  3.14064090e-01],
       [ 2.96314085e-01,  2.72584017e-01,  2.97468942e-01],
       [ 2.82922504e-01,  2.84786665e-01,  2.98440562e-01],
       [ 3.00027783e-01,  2.92831021e-01,  2.73492705e-01],
       [ 2.77406701e-01,  2.80524439e-01,  3.08605540e-01],
       [ 2.66421855e-01,  2.65414987e-01,  3.37110242e-01],
       [ 3.27882840e-01,  2.66844605e-01,  2.73244886e-01],
       [ 3.23065188e-01,  2.08051695e-01,  3.44153511e-01],
       [ 3.18357184e-01,  2.50973425e-01,  2.98773339e-01],
       [ 2.53889676e-01,  3.35718310e-01,  2.79447685e-01],
       [ 3.06364281e-01,  2.51203826e-01,  3.10347953e-01],
       [ 2.56100214e-01,  3.37597759e-01,  2.75461850e-01],
       [ 2.88357510e-01,  2.92321710e-01,  2.85367263e-01],
       [ 2.67247162e-01,  2.86824688e-01,  3.12859994e-01],
       [ 2.89149554e-01,  2.77804985e-01,  2.99270577e-01],
       [ 3.22704617e-01,  2.57368583e-01,  2.87801556e-01],
       [ 2.74589833e-01,  3.22010148e-01,  2.70833544e-01],
       [ 2.74658271e-01,  2.60413462e-01,  3.33553948e-01],
       [ 2.60099517e-01,  3.30021435e-01,  2.78182945e-01],
       [ 2.80034984e-01,  2.57052589e-01,  3.31447404e-01],
       [ 3.24558675e-01,  2.86263676e-01,  2.57178933e-01],
       [ 3.05179657e-01,  3.00605583e-01,  2.61249830e-01],
       [ 2.92704701e-01,  2.99372225e-01,  2.74241969e-01],
       [ 3.09238344e-01,  2.66432469e-01,  2.91154008e-01],
       [ 2.75451396e-01,  3.21354880e-01,  2.70577499e-01],
       [ 2.51086679e-01,  3.26258259e-01,  2.91127343e-01],
       [ 2.72916587e-01,  3.06676358e-01,  2.86930572e-01],
       [ 3.11075724e-01,  3.03350173e-01,  2.53301702e-01],
       [ 2.41646735e-01,  3.35155613e-01,  2.93013267e-01],
       [ 2.97257155e-01,  2.69168659e-01,  3.00105832e-01],
       [ 3.00055656e-01,  2.72093193e-01,  2.94253669e-01],
       [ 3.27522953e-01,  2.85030685e-01,  2.55726028e-01],
       [ 2.71616994e-01,  2.55171029e-01,  3.43005543e-01],
       [ 2.73373039e-01,  2.76710149e-01,  3.16960979e-01],
       [ 2.72153214e-01,  3.55390964e-01,  2.44252456e-01],
       [ 3.78690850e-01,  2.34437751e-01,  2.62947395e-01],
       [ 2.61506866e-01,  3.38238677e-01,  2.69361341e-01],
       [ 3.36038566e-01,  2.01678041e-01,  3.38893011e-01],
       [ 2.79149519e-01,  3.24894178e-01,  2.63732487e-01],
       [ 2.65216590e-01,  2.02983064e-01,  4.18978361e-01],
       [ 2.27959552e-01,  7.01168551e-01,  9.70392900e-02],
       [-5.71809842e-02,  3.90216254e-01,  8.17670002e-01],
       [ 4.41446806e-01,  4.11917552e-01,  7.98724622e-02],
       [ 2.88082629e-01,  3.10759456e-01,  2.67976488e-01],
       [ 2.31611985e-01,  2.36513154e-01,  4.17026779e-01],
       [ 3.16936621e-03,  4.13848581e-01,  5.96349303e-01],
       [ 3.30072221e-01,  3.09696342e-01,  2.30986405e-01],
       [ 3.03660106e-01,  2.67559268e-01,  2.95425771e-01],
       [ 2.23298778e-01,  3.34226819e-01,  3.14546204e-01],
       [ 2.13430842e-01,  2.94639289e-01,  3.68285551e-01],
       [ 2.08057416e-01,  3.20768692e-01,  3.46544340e-01],
       [ 3.99076973e-01,  1.57946712e-01,  3.35653779e-01],
       [ 3.82242625e-01,  2.84655806e-01,  2.11715025e-01],
       [ 2.84617431e-01,  3.20815187e-01,  2.62110764e-01],
       [ 3.27853617e-01,  3.19806880e-01,  2.24114576e-01],
       [ 3.55953491e-01,  2.46281891e-01,  2.69554241e-01],
       [ 1.99482964e-01,  3.73174976e-01,  3.06566866e-01],
       [ 2.72099196e-01,  3.62232134e-01,  2.38734744e-01],
       [ 3.49704392e-01,  4.12188216e-01,  1.38938178e-01],
       [ 3.98293711e-01,  2.53066762e-01,  2.29066863e-01],
       [ 3.83809816e-01,  2.34745579e-01,  2.58509332e-01],
       [ 2.63081361e-01,  2.69991100e-01,  3.35733671e-01],
       [ 2.19451241e-01,  2.86593170e-01,  3.69743780e-01],
       [ 2.48224477e-01,  2.86367565e-01,  3.34678683e-01],
       [ 3.31676439e-01,  2.27440797e-01,  3.12212242e-01],
       [ 3.96598766e-01,  1.65135605e-01,  3.28460270e-01],
       [ 2.19594182e-01,  3.00349723e-01,  3.53970777e-01],
       [ 2.80265178e-01,  2.89041636e-01,  2.96837646e-01],
       [ 3.85249346e-01,  3.15894830e-01,  1.82989073e-01],
       [ 3.03869375e-01,  2.80330375e-01,  2.82122996e-01],
       [ 4.20094107e-01,  2.99947873e-01,  1.72203385e-01],
       [ 2.95059135e-01,  2.66926740e-01,  3.04706639e-01],
       [ 2.78816174e-01,  3.02141569e-01,  2.85318589e-01],
       [ 2.84316817e-01,  2.96019825e-01,  2.85759289e-01],
       [ 2.50105370e-01,  3.59546203e-01,  2.62568935e-01],
       [ 3.80049661e-01,  2.42763385e-01,  2.53266784e-01],
       [ 2.83524598e-01,  3.30667075e-01,  2.54395749e-01],
       [ 3.19369641e-01,  3.27225412e-01,  2.25015853e-01],
       [ 3.01982995e-01,  2.76050099e-01,  2.88283779e-01],
       [ 3.28436821e-01,  2.62855840e-01,  2.76797726e-01],
       [ 3.48570323e-01,  2.30493682e-01,  2.92984439e-01],
       [ 2.60641729e-01,  3.03455005e-01,  3.02974565e-01],
       [ 3.39496901e-01,  2.77514288e-01,  2.52482876e-01],
       [ 3.03338956e-01,  2.64276241e-01,  2.99207494e-01],
       [ 3.05205157e-01,  2.99183018e-01,  2.62559405e-01],
       [ 3.23213112e-01,  2.54991980e-01,  2.89833575e-01],
       [ 3.15275057e-01,  2.48849130e-01,  3.04089206e-01],
       [ 2.55439203e-01,  3.02316836e-01,  3.09770611e-01],
       [ 2.54559938e-01,  3.12335563e-01,  3.00746571e-01],
       [ 2.69888376e-01,  2.98948462e-01,  2.97655274e-01],
       [ 2.99707472e-01,  2.64875045e-01,  3.02196694e-01],
       [ 2.55117222e-01,  3.28823705e-01,  2.84466120e-01],
       [ 2.64518512e-01,  2.84553773e-01,  3.18228154e-01],
       [ 3.08833842e-01,  2.70068123e-01,  2.87775537e-01],
       [ 2.80746928e-01,  2.90393751e-01,  2.94976093e-01],
       [ 2.89733733e-01,  2.80800478e-01,  2.95587234e-01],
       [ 2.64411736e-01,  3.15465667e-01,  2.87279991e-01],
       [ 2.68343079e-01,  3.27775578e-01,  2.71831271e-01],
       [ 3.27161437e-01,  2.78793806e-01,  2.62048096e-01],
       [ 2.72408487e-01,  3.07202917e-01,  2.86942799e-01],
       [ 3.06737485e-01,  2.99464047e-01,  2.60875539e-01],
       [ 2.74486458e-01,  3.13556347e-01,  2.78777373e-01],
       [ 2.81199231e-01,  2.98307138e-01,  2.86651315e-01],
       [ 3.04580354e-01,  2.53489571e-01,  3.09624384e-01],
       [ 2.90741559e-01,  3.08074620e-01,  2.67911773e-01],
       [ 2.78121399e-01,  3.06293911e-01,  2.82013674e-01],
       [ 2.98483930e-01,  3.12593084e-01,  2.56425924e-01],
       [ 2.68944380e-01,  2.69415701e-01,  3.29783295e-01],
       [ 3.17385492e-01,  2.81044255e-01,  2.68704274e-01],
       [ 3.26510024e-01,  2.88769160e-01,  2.53078568e-01],
       [ 3.47850030e-01,  2.70988203e-01,  2.51659864e-01],
       [ 2.82994531e-01,  2.41879675e-01,  3.45891211e-01],
       [ 2.96467885e-01,  2.52849363e-01,  3.18646984e-01],
       [ 3.06268992e-01,  2.65471141e-01,  2.95054364e-01],
       [ 3.02265182e-01,  3.05233025e-01,  2.59652922e-01],
       [ 2.97877029e-01,  2.63143218e-01,  3.05899262e-01],
       [ 2.56899743e-01,  2.99047658e-01,  3.11494817e-01],
       [ 3.31186053e-01,  3.50693752e-01,  1.96303102e-01],
       [ 3.07399547e-01,  2.63446523e-01,  2.96080971e-01],
       [ 3.04218494e-01,  2.81437828e-01,  2.80678960e-01],
       [ 2.68712162e-01,  2.79684238e-01,  3.18830401e-01],
       [ 3.01157601e-01,  2.87496852e-01,  2.77613015e-01],
       [ 2.86185972e-01,  2.89821268e-01,  2.90026248e-01],
       [ 1.28245245e-01,  3.82664182e-01,  3.93269585e-01],
       [ 3.06658604e-01,  3.59818007e-01,  2.09547807e-01],
       [ 3.03367758e-01,  2.64970957e-01,  2.98442373e-01],
       [ 2.43958468e-01,  2.83284029e-01,  3.43087789e-01],
       [ 2.75248754e-01,  3.32491784e-01,  2.60772552e-01],
       [ 3.03683274e-01,  2.99409724e-01,  2.63764088e-01],
       [ 2.52970607e-01,  3.18822658e-01,  2.96168649e-01],
       [ 3.03895483e-01,  2.92777379e-01,  2.69873639e-01],
       [ 2.42787574e-01,  2.61512035e-01,  3.69835956e-01],
       [ 4.31787948e-01,  4.08809424e-01,  8.74152361e-02],
       [ 8.94489227e-01, -4.91126227e-02,  3.47692035e-01],
       [ 5.06998098e-01,  5.47340394e-01, -2.60832163e-02],
       [ 4.02601409e-01,  2.19508671e-01,  2.59802091e-01],
       [ 3.05842630e-01,  3.45134645e-01,  2.21886567e-01],
       [ 2.86800278e-01,  2.19303663e-01,  3.69694115e-01],
       [ 2.37642333e-01,  3.42666231e-01,  2.90479941e-01],
       [ 4.50560981e-01,  8.08312677e-02,  4.01926421e-01],
       [ 2.23685344e-01,  3.28982853e-01,  3.19199401e-01],
       [ 3.15917842e-01,  2.62036144e-01,  2.89327716e-01],
       [ 3.06456107e-01,  2.91531571e-01,  2.68665017e-01],
       [ 7.33223371e-01, -1.48888944e-01,  6.14663174e-01],
       [ 2.80249488e-01,  3.62021325e-01,  2.31278934e-01],
       [ 2.87718823e-01,  2.52328991e-01,  3.28489802e-01],
       [ 2.59356509e-01,  3.25746832e-01,  2.82882402e-01],
       [ 3.55084306e-01,  2.20934180e-01,  2.97819852e-01],
       [ 2.42982952e-01,  2.67502680e-01,  3.62402774e-01],
       [ 2.62224787e-01,  3.91743204e-01,  2.25202494e-01],
       [ 3.46335376e-01,  2.35129022e-01,  2.89899953e-01],
       [ 1.72231428e-01,  4.64446570e-01,  2.67023055e-01],
       [ 3.32220798e-01,  1.98266055e-01,  3.47099822e-01],
       [ 3.79895548e-01,  1.70025200e-01,  3.37154371e-01],
       [ 2.72025046e-01,  4.29231632e-01,  1.89999254e-01],
       [ 2.28957776e-01,  3.77769202e-01,  2.69489918e-01],
       [ 2.79634806e-01,  2.90258791e-01,  2.96254494e-01],
       [ 3.18972571e-01,  3.27195145e-01,  2.25380687e-01],
       [ 3.06999537e-01,  2.40554529e-01,  3.21703156e-01],
       [ 2.56783603e-01,  3.15208971e-01,  2.95562412e-01],
       [ 3.41770728e-01,  3.40373365e-01,  1.95956175e-01],
       [ 3.78776684e-01,  2.39615341e-01,  2.57505416e-01],
       [ 3.20945665e-01,  1.77075092e-01,  3.87872420e-01],
       [ 3.47428380e-01,  1.69282066e-01,  3.70007241e-01],
       [ 2.39641064e-01,  3.33886042e-01,  2.96389118e-01],
       [ 3.15177309e-01,  2.65211599e-01,  2.86723815e-01],
       [ 3.04277975e-01,  2.95696523e-01,  2.66721437e-01],
       [ 2.94751389e-01,  2.75785858e-01,  2.95706785e-01],
       [ 4.10859602e-01,  1.10917910e-01,  3.91792108e-01],
       [ 2.08761770e-01,  3.80823320e-01,  2.89184127e-01],
       [ 3.55488781e-01,  2.58828800e-01,  2.57178486e-01],
       [ 3.74748484e-01,  2.19349341e-01,  2.82443664e-01],
       [ 2.92008594e-01,  3.16217202e-01,  2.59216002e-01],
       [ 2.82619080e-01,  3.24825787e-01,  2.60431923e-01],
       [ 2.86899938e-01,  3.22285178e-01,  2.58601858e-01],
       [ 3.07104215e-01,  2.98586770e-01,  2.61358924e-01],
       [ 2.61440740e-01,  3.44137199e-01,  2.64257506e-01],
       [ 2.39271065e-01,  3.15054951e-01,  3.15006803e-01],
       [ 2.64105422e-01,  2.99893745e-01,  3.02831008e-01],
       [ 2.96014545e-01,  3.18923430e-01,  2.53023934e-01],
       [ 3.05188076e-01,  3.04773194e-01,  2.57371841e-01],
       [ 2.92111626e-01,  3.00817223e-01,  2.73435493e-01],
       [ 3.03942519e-01,  2.53969932e-01,  3.09739885e-01],
       [ 2.87289482e-01,  2.93433074e-01,  2.85333783e-01],
       [ 2.77435160e-01,  2.94839206e-01,  2.93917127e-01],
       [ 3.16702054e-01,  2.78118172e-01,  2.72215699e-01],
       [ 2.60234025e-01,  2.99240172e-01,  3.07659454e-01],
       [ 3.02945684e-01,  2.98966873e-01,  2.64871158e-01],
       [ 2.76741171e-01,  3.03067385e-01,  2.86523500e-01],
       [ 2.69937053e-01,  2.94653963e-01,  3.01921166e-01],
       [ 2.85913256e-01,  2.71161733e-01,  3.09601524e-01],
       [ 2.54845435e-01,  3.45259747e-01,  2.69972888e-01],
       [ 2.73623285e-01,  2.78644498e-01,  3.14623416e-01],
       [ 2.67969050e-01,  2.96864411e-01,  3.01769526e-01],
       [ 2.89864117e-01,  3.08677273e-01,  2.68194544e-01],
       [ 3.61431104e-01,  2.63149494e-01,  2.47989752e-01],
       [ 2.69405200e-01,  3.14411789e-01,  2.83129530e-01],
       [ 2.76225033e-01,  2.97236171e-01,  2.92776438e-01],
       [ 2.75442025e-01,  2.67257632e-01,  3.25015898e-01],
       [ 2.72389684e-01,  2.89637204e-01,  3.04444176e-01],
       [ 2.61122734e-01,  3.36622208e-01,  2.71186383e-01],
       [ 3.26970574e-01,  2.64710807e-01,  2.76242181e-01],
       [ 2.68898301e-01,  2.74484109e-01,  3.24249894e-01],
       [ 2.80835067e-01,  2.48732802e-01,  3.40177562e-01],
       [ 3.51302009e-01,  2.65929042e-01,  2.53679061e-01],
       [ 3.53725580e-01,  2.71976995e-01,  2.45795344e-01],
       [ 2.64874962e-01,  3.13188513e-01,  2.88972080e-01],
       [ 2.74881103e-01,  3.36797825e-01,  2.57358290e-01],
       [ 3.37216751e-01,  2.55703655e-01,  2.76213203e-01],
       [ 2.72252939e-01,  2.85523722e-01,  3.08842839e-01],
       [ 2.57961074e-01,  3.23063009e-01,  2.86842360e-01],
       [ 2.95632371e-01,  1.81845705e-01,  4.10993789e-01],
       [ 2.66800143e-01,  2.94454354e-01,  3.05457751e-01],
       [ 2.62689457e-01,  3.24327718e-01,  2.80745666e-01],
       [ 2.88272431e-01,  2.61158501e-01,  3.17993024e-01],
       [ 2.73074435e-01,  3.09010471e-01,  2.84524026e-01],
       [ 3.15999940e-01,  2.63269731e-01,  2.87960494e-01],
       [ 5.51702371e-01,  6.18412209e-01, -7.79235501e-02],
       [ 1.90579455e-01,  3.06204559e-01,  3.85768657e-01],
       [ 4.17893595e-01,  2.78042430e-01,  1.92270330e-01],
       [ 2.19469042e-01,  2.97380140e-01,  3.57423929e-01],
       [ 2.41408285e-01,  3.75391680e-01,  2.58393883e-01],
       [ 3.94875872e-01,  2.70051676e-01,  2.15607113e-01],
       [ 3.12519017e-01,  2.24366396e-01,  3.35045860e-01],
       [ 3.10960893e-01,  2.58240251e-01,  2.98132536e-01],
       [ 3.77250447e-01,  3.78686860e-01,  1.41731612e-01],
       [ 3.73114221e-01,  2.85813242e-01,  2.17564183e-01],
       [ 3.55912372e-01,  2.92537093e-01,  2.24971161e-01],
       [ 8.99939694e-02,  4.09017306e-01,  4.27226637e-01],
       [ 2.15713175e-01,  1.93361859e-01,  5.09171379e-01],
       [ 2.67829738e-01,  3.31618812e-01,  2.68884829e-01],
       [ 2.58972963e-01,  3.00044614e-01,  3.08213130e-01],
       [ 2.80113568e-01,  2.55417320e-01,  3.33228478e-01],
       [ 2.55259057e-01,  2.98627978e-01,  3.13732752e-01],
       [ 2.40424228e-01,  3.06743648e-01,  3.22115759e-01],
       [ 1.48804585e-01,  5.95943720e-01,  2.16611224e-01],
       [ 3.46655453e-01,  2.26802668e-01,  2.98849405e-01],
       [ 2.79356005e-01,  2.80561740e-01,  3.06515368e-01],
       [ 2.94257656e-01,  2.45931696e-01,  3.28834167e-01],
       [ 4.08125966e-01,  2.49407850e-01,  2.25403130e-01],
       [ 2.56653577e-01,  3.95825348e-01,  2.27455635e-01],
       [ 2.75278653e-01,  2.30226178e-01,  3.69182718e-01],
       [ 2.58806968e-01,  2.66650057e-01,  3.44441312e-01],
       [ 2.46756509e-01,  3.06699161e-01,  3.14966483e-01],
       [ 3.50198924e-01,  2.44110950e-01,  2.76812847e-01],
       [ 2.81760018e-01,  2.70139303e-01,  3.15067511e-01],
       [ 3.06334608e-01,  3.69442374e-01,  2.02473625e-01],
       [ 3.52942407e-01,  2.63366983e-01,  2.54817832e-01],
       [ 3.32036421e-01,  3.16326672e-01,  2.23590802e-01],
       [ 3.13992281e-01,  2.47028123e-01,  3.07359724e-01],
       [ 2.24337085e-01,  3.39212199e-01,  3.08583702e-01],
       [ 1.51682970e-01,  3.91494876e-01,  3.50929067e-01],
       [ 2.08740058e-01,  3.31257844e-01,  3.34914669e-01],
       [ 3.15360682e-01,  2.36894428e-01,  3.17412747e-01],
       [ 2.46572188e-01,  3.10410813e-01,  3.11430199e-01],
       [ 2.95971613e-01,  3.19275801e-01,  2.52749419e-01],
       [ 3.03736714e-01,  2.15821344e-01,  3.55007744e-01],
       [ 3.61214098e-01,  2.39050800e-01,  2.72632102e-01],
       [ 2.70596323e-01,  2.37565531e-01,  3.65465529e-01],
       [ 3.27823969e-01,  2.65135084e-01,  2.75031410e-01],
       [ 3.28233582e-01,  2.80365044e-01,  2.59571367e-01],
       [ 2.98971368e-01,  3.01721045e-01,  2.66016422e-01],
       [ 2.83351635e-01,  2.72765290e-01,  3.10566899e-01],
       [ 2.24951122e-01,  3.88947790e-01,  2.64710940e-01],
       [ 7.75648584e-02,  6.04377982e-01,  2.97857086e-01],
       [ 1.69580700e-01,  2.00968022e-01,  5.82702596e-01],
       [ 3.27870728e-01,  3.40617429e-01,  2.06916927e-01],
       [ 2.42646334e-01,  2.79826239e-01,  3.48537317e-01],
       [ 2.71062155e-01,  2.96522306e-01,  2.98852482e-01],
       [ 2.65818182e-01,  2.78565248e-01,  3.23213901e-01],
       [ 2.45422401e-01,  2.83996715e-01,  3.40563533e-01],
       [ 3.13477601e-01,  2.57286639e-01,  2.96701142e-01],
       [ 2.58339886e-01,  3.17342843e-01,  2.91858132e-01],
       [ 2.93859182e-01,  3.17964443e-01,  2.55896017e-01],
       [ 2.68095907e-01,  3.09012231e-01,  2.89642573e-01],
       [ 2.47614314e-01,  3.03618717e-01,  3.17142932e-01],
       [ 3.02012048e-01,  2.62881268e-01,  3.02015769e-01],
       [ 2.90228415e-01,  2.72746634e-01,  3.03461365e-01],
       [ 3.09590894e-01,  2.87733765e-01,  2.69401646e-01],
       [ 2.84164703e-01,  2.79057475e-01,  3.03080600e-01],
       [ 2.87810348e-01,  2.72113605e-01,  3.06618224e-01],
       [ 2.71959079e-01,  2.81211345e-01,  3.13686370e-01],
       [ 2.95609293e-01,  3.03406063e-01,  2.67623103e-01],
       [ 2.92936912e-01,  2.76668482e-01,  2.96615150e-01],
       [ 2.78208180e-01,  2.91171176e-01,  2.96803520e-01],
       [ 3.16034680e-01,  2.91627832e-01,  2.59742025e-01],
       [ 3.28555626e-01,  2.82029152e-01,  2.57683685e-01],
       [ 2.57126259e-01,  3.22900608e-01,  2.87872483e-01],
       [ 3.23218063e-01,  2.47973690e-01,  2.97361479e-01],
       [ 2.58912136e-01,  2.95088644e-01,  3.13353294e-01],
       [ 2.90617564e-01,  2.97613393e-01,  2.77966194e-01],
       [ 3.52640132e-01,  2.05629057e-01,  3.17923585e-01],
       [ 3.31001141e-01,  3.68644895e-01,  1.82918380e-01],
       [ 3.16621604e-01,  2.42841019e-01,  3.09424222e-01],
       [ 2.62764077e-01,  3.13769813e-01,  2.90620495e-01],
       [ 2.84457357e-01,  2.82531878e-01,  2.99179804e-01],
       [ 2.76584813e-01,  3.08800102e-01,  2.81166421e-01],
       [ 2.91772001e-01,  2.73080129e-01,  3.01535314e-01],
       [ 3.26891929e-01,  2.68460434e-01,  2.72515003e-01],
       [ 2.48762602e-01,  2.78093491e-01,  3.43207077e-01],
       [ 2.62859901e-01,  2.66893812e-01,  3.39486661e-01],
       [ 2.46237436e-01,  3.30148561e-01,  2.92694593e-01],
       [ 3.10717862e-01,  3.01087167e-01,  2.55713556e-01],
       [ 1.35959915e-01,  4.45641846e-01,  3.25670562e-01],
       [ 3.67880359e-01,  1.47320077e-01,  3.80053710e-01],
       [ 2.94909190e-01,  3.01451697e-01,  2.70136972e-01],
       [ 2.95767118e-01,  2.77308650e-01,  2.93122183e-01],
       [ 2.90597005e-01,  2.70100662e-01,  3.05885982e-01],
       [ 2.70411424e-01,  2.86339715e-01,  3.09959797e-01],
       [ 2.61031064e-01,  2.97098192e-01,  3.08975280e-01],
       [ 2.80601089e-01,  3.16858240e-01,  2.69623763e-01],
       [ 2.72006647e-01,  2.79693558e-01,  3.15246381e-01],
       [ 2.91693503e-01,  3.19180095e-01,  2.56840762e-01],
       [ 1.95808085e-01,  2.90631178e-01,  3.96950000e-01],
       [ 3.27713921e-01,  2.56743520e-01,  2.83787258e-01],
       [ 2.76809000e-01,  3.11240450e-01,  2.78625960e-01],
       [ 4.18898393e-01,  1.96419316e-01,  2.72575064e-01],
       [ 2.43036812e-01,  1.74012604e-01,  4.98042974e-01],
       [ 6.03241270e-01,  1.99699590e-01,  1.61323171e-01],
       [ 2.26156724e-01,  4.82191025e-01,  1.98983138e-01],
       [ 3.42422504e-01,  2.61900460e-01,  2.65287600e-01],
       [ 1.69946300e-01,  3.56003159e-01,  3.60298080e-01],
       [ 3.53610207e-01,  2.76086403e-01,  2.41978482e-01],
       [ 2.28982726e-01,  3.29223741e-01,  3.12811586e-01],
       [ 2.35501851e-01,  4.17695485e-01,  2.32138027e-01],
       [ 2.84090502e-01,  3.33171518e-01,  2.51674542e-01],
       [ 2.84936882e-01,  2.95932589e-01,  2.85223960e-01],
       [ 2.87239280e-01,  3.32315482e-01,  2.49446780e-01],
       [ 3.13006937e-01,  3.30196218e-01,  2.27993741e-01],
       [ 3.09121706e-01,  2.92138670e-01,  2.65598071e-01],
       [ 2.55939853e-01,  3.39472944e-01,  2.73953374e-01],
       [ 3.09159869e-01,  2.71046161e-01,  2.86455837e-01],
       [ 3.58108066e-01,  1.89426870e-01,  3.32699702e-01],
       [ 2.49781290e-01,  2.81705816e-01,  3.37985918e-01],
       [ 2.93231364e-01,  3.34466452e-01,  2.42033575e-01],
       [ 2.81224868e-01,  3.18078445e-01,  2.67891779e-01],
       [ 3.20729218e-01,  2.68622093e-01,  2.78009131e-01],
       [ 2.56041838e-01,  3.07313095e-01,  3.04097790e-01],
       [ 3.71493485e-01,  2.78502514e-01,  2.25444665e-01],
       [ 2.37177549e-01,  2.65519980e-01,  3.72042056e-01],
       [ 3.44301260e-01,  2.48304003e-01,  2.77602359e-01],
       [ 2.41368507e-01,  2.64050130e-01,  3.68539268e-01],
       [ 2.97093249e-01,  3.15670296e-01,  2.54937630e-01],
       [ 2.23813556e-01,  3.15161060e-01,  3.32970566e-01],
       [ 2.80284517e-01,  2.48014533e-01,  3.41634849e-01],
       [ 2.74554141e-01,  2.36437392e-01,  3.62207831e-01],
       [ 2.93496663e-01,  2.49177544e-01,  3.25917871e-01],
       [ 2.79786437e-01,  3.27525464e-01,  2.60760273e-01],
       [ 2.78269584e-01,  2.96169061e-01,  2.91737264e-01],
       [ 4.06394937e-01,  2.91324164e-01,  1.88625099e-01],
       [ 2.56407852e-01,  4.79023540e-01,  1.72925176e-01],
       [ 3.03273327e-01,  3.35755282e-01,  2.31874405e-01],
       [ 2.27213314e-01,  3.37392227e-01,  3.07010792e-01],
       [ 2.57723500e-01,  3.46919295e-01,  2.65596062e-01],
       [ 2.77046991e-01,  3.53823231e-01,  2.40896357e-01],
       [ 2.76307113e-01,  3.54675754e-01,  2.40894919e-01],
       [ 2.44172458e-01,  3.52062310e-01,  2.75119951e-01],
       [ 2.93731663e-01,  3.41697847e-01,  2.35482489e-01],
       [ 2.68309404e-01,  2.92813980e-01,  3.05521495e-01],
       [ 2.83875194e-01,  3.05666752e-01,  2.76873923e-01],
       [ 3.63816815e-01,  1.89204689e-01,  3.27589708e-01],
       [ 6.02004307e-01,  3.62278318e-01,  3.30887348e-02],
       [-3.30751556e-02,  4.28431791e-01,  6.68182660e-01],
       [ 3.06511063e-01,  2.55180693e-01,  3.05833747e-01],
       [ 2.39535230e-01,  2.63926311e-01,  3.70992291e-01],
       [ 2.78426243e-01,  3.07540863e-01,  2.80514983e-01],
       [ 3.04623199e-01,  3.06332763e-01,  2.56457001e-01],
       [ 3.20003240e-01,  2.53720277e-01,  2.94233519e-01],
       [ 3.09855438e-01,  2.64995923e-01,  2.92057362e-01],
       [ 3.22108676e-01,  2.78218588e-01,  2.67159913e-01],
       [ 2.83638627e-01,  3.12966052e-01,  2.70247191e-01],
       [ 2.94900064e-01,  2.73314063e-01,  2.98126458e-01],
       [ 3.15594553e-01,  2.94846671e-01,  2.57105828e-01],
       [ 2.85081829e-01,  3.04841270e-01,  2.76468735e-01],
       [ 2.99137227e-01,  2.91600232e-01,  2.75539518e-01],
       [ 2.75524020e-01,  2.95113450e-01,  2.95615281e-01],
       [ 2.97949024e-01,  2.90426932e-01,  2.77828448e-01],
       [ 2.82641760e-01,  3.02220893e-01,  2.81398981e-01],
       [ 2.79485424e-01,  3.04582647e-01,  2.82284887e-01],
       [ 2.95285225e-01,  2.63259548e-01,  3.08414568e-01],
       [ 3.08910170e-01,  2.83634799e-01,  2.74042199e-01],
       [ 3.09153576e-01,  2.68612972e-01,  2.88970242e-01],
       [ 2.96050957e-01,  2.58830669e-01,  3.12450304e-01],
       [ 2.71994449e-01,  2.55633994e-01,  3.42037991e-01],
       [ 2.80575930e-01,  2.80925872e-01,  3.04859149e-01],
       [ 3.01251625e-01,  3.21327002e-01,  2.46072885e-01],
       [ 2.86300530e-01,  2.90231178e-01,  2.89501266e-01],
       [ 1.70512880e-01,  5.98623036e-01,  1.92328896e-01],
       [ 3.47892386e-01,  2.55677824e-01,  2.66831644e-01],
       [ 2.85793646e-01,  3.11859389e-01,  2.69173850e-01],
       [ 2.89403104e-01,  2.85741559e-01,  2.90892912e-01],
       [ 2.85334765e-01,  2.80193589e-01,  3.00694080e-01],
       [ 2.93928359e-01,  2.71392139e-01,  3.01121496e-01],
       [ 3.15425723e-01,  2.58144680e-01,  2.93903812e-01],
       [ 3.34703291e-01,  2.22751990e-01,  3.14723318e-01],
       [ 3.57839818e-01,  2.66728987e-01,  2.47456721e-01],
       [ 2.78043475e-01,  3.20737974e-01,  2.68580297e-01],
       [ 2.46904361e-01,  3.59480569e-01,  2.65908126e-01],
       [ 2.56221851e-01,  2.73773343e-01,  3.39348902e-01],
       [ 2.00602871e-01,  2.74072684e-01,  4.10849538e-01],
       [ 4.72119332e-01,  6.42143786e-01, -4.77162839e-02],
       [ 2.42885606e-01,  2.65215983e-01,  3.65247540e-01],
       [ 1.96856621e-01,  2.27567321e-01,  4.83483247e-01],
       [ 3.10353694e-01,  2.85036832e-01,  2.71313968e-01],
       [ 2.49053119e-01,  3.18844211e-01,  3.00390310e-01],
       [ 3.75479779e-01,  2.90115711e-01,  2.11941667e-01],
       [ 2.92975691e-01,  3.03232675e-01,  2.70308515e-01],
       [ 3.16623699e-01,  2.85603678e-01,  2.64968205e-01],
       [ 3.15898076e-01,  2.81064581e-01,  2.70054144e-01],
       [ 2.50350921e-01,  2.87594226e-01,  3.30889723e-01],
       [ 3.30139178e-01,  2.50902338e-01,  2.87702865e-01],
       [ 3.55334109e-01,  1.71600074e-01,  3.58725333e-01],
       [ 3.55885801e-01,  2.29777096e-01,  2.87239631e-01],
       [ 3.11937226e-01,  3.09776650e-01,  2.46687644e-01],
       [ 8.73881409e-02,  3.32658525e-01,  5.25964394e-01],
       [ 1.58463315e-01,  4.99113452e-01,  2.59907035e-01],
       [-2.09567145e-02,  4.63570161e-01,  5.86775910e-01],
       [ 8.34756855e-01, -1.42513606e-01,  5.32997917e-01],
       [ 3.11397361e-01,  2.28341786e-01,  3.31447092e-01],
       [ 2.75072389e-01,  3.15745371e-01,  2.76137885e-01],
       [ 2.36407435e-01,  2.61552637e-01,  3.77875702e-01],
       [ 2.40236219e-01,  2.84711790e-01,  3.45942670e-01],
       [ 2.27991197e-01,  3.32784085e-01,  3.10513254e-01],
       [ 2.87757162e-01,  3.08220995e-01,  2.70659586e-01],
       [ 3.42379787e-01,  2.83876268e-01,  2.44000042e-01],
       [ 3.75569866e-01,  2.25893870e-01,  2.74598550e-01],
       [ 2.28009078e-01,  2.97714487e-01,  3.46414744e-01],
       [ 3.07348992e-01,  2.63200882e-01,  2.96390346e-01],
       [ 3.14964356e-01,  2.53145460e-01,  2.99710018e-01],
       [ 2.16710036e-01,  3.44676601e-01,  3.12271632e-01],
       [ 3.13763338e-01,  3.23628262e-01,  2.32913826e-01],
       [ 2.96278902e-01,  3.30409278e-01,  2.42715127e-01],
       [ 3.54206849e-01,  2.60471215e-01,  2.56621020e-01],
       [ 2.64714368e-01,  3.04586227e-01,  2.97508294e-01],
       [ 2.81009968e-01,  3.45582112e-01,  2.43999225e-01],
       [ 2.82273588e-01,  2.89686335e-01,  2.94127599e-01],
       [ 1.59947720e-01,  3.44474818e-01,  3.86386460e-01],
       [ 2.70793558e-01,  3.34854271e-01,  2.63063142e-01],
       [ 2.56396636e-01,  3.47967361e-01,  2.66035601e-01],
       [ 2.54311188e-01,  3.25904743e-01,  2.88028592e-01],
       [ 3.10286519e-01,  3.13034719e-01,  2.45250021e-01],
       [ 2.57058266e-01,  3.37823575e-01,  2.74272379e-01],
       [ 3.52423431e-01,  1.78888244e-01,  3.51875556e-01],
       [-1.19855265e-01,  6.43316151e-01,  6.24888767e-01],
       [ 2.19626402e-01,  3.15885209e-01,  3.37291040e-01],
       [ 3.44441573e-01,  2.07635730e-01,  3.23290999e-01],
       [ 2.41942078e-01,  3.83971357e-01,  2.50995047e-01],
       [ 3.19074410e-01,  2.55733169e-01,  2.92971241e-01],
       [ 2.68297583e-01,  3.09142236e-01,  2.89307872e-01],
       [ 2.12227762e-01,  3.27522062e-01,  3.34397007e-01],
       [ 2.63796423e-01,  2.28348629e-01,  3.85582355e-01],
       [ 2.49889322e-01,  3.31082140e-01,  2.87907616e-01],
       [ 2.77868889e-01,  2.80535934e-01,  3.08105848e-01],
       [ 3.55585203e-01,  2.34917019e-01,  2.81907465e-01],
       [ 2.87614105e-01,  3.02747119e-01,  2.75976219e-01],
       [ 2.65786918e-01,  2.76650645e-01,  3.25327539e-01],
       [ 2.58892373e-01,  2.93509176e-01,  3.15011268e-01],
       [ 2.68630141e-01,  2.99050164e-01,  2.98876165e-01],
       [ 2.67304125e-01,  2.94715982e-01,  3.04653161e-01],
       [ 3.00924655e-01,  2.87555500e-01,  2.77779053e-01],
       [ 2.80609915e-01,  3.05372049e-01,  2.80400397e-01],
       [ 2.77981518e-01,  2.76571462e-01,  3.12176205e-01],
       [ 2.58743428e-01,  2.89162189e-01,  3.19728760e-01],
       [ 3.10297696e-01,  2.82054826e-01,  2.74294498e-01],
       [ 2.64379798e-01,  2.84553172e-01,  3.18381114e-01],
       [ 2.73000344e-01,  3.24718872e-01,  2.69945540e-01],
       [ 2.69999193e-01,  2.89889413e-01,  3.06721891e-01],
       [ 2.94513567e-01,  3.01487379e-01,  2.70482585e-01],
       [ 2.54502331e-01,  3.14152198e-01,  2.99034870e-01],
       [ 3.34758712e-01,  2.89859868e-01,  2.44896468e-01],
       [ 3.19461008e-01,  2.92411134e-01,  2.55913015e-01],
       [ 3.01242726e-01,  3.34719716e-01,  2.34554921e-01],
       [ 1.92199086e-01,  3.56442715e-01,  3.30802421e-01],
       [ 2.99537283e-01,  2.65740633e-01,  3.01446011e-01],
       [ 5.04220264e-01, -1.51523119e-01,  9.25442779e-01],
       [ 2.93346224e-01,  4.22769303e-01,  1.75924437e-01],
       [ 2.94603364e-01,  3.45985156e-01,  2.31149333e-01],
       [ 2.91577403e-01,  2.97486866e-01,  2.77150662e-01],
       [ 2.95007730e-01,  2.86552731e-01,  2.84518533e-01],
       [ 2.84945830e-01,  3.13158999e-01,  2.68793432e-01],
       [ 2.53500064e-01,  2.90998064e-01,  3.23659464e-01],
       [ 2.60812528e-01,  2.99749537e-01,  3.06516577e-01],
       [ 2.70815895e-01,  3.23573198e-01,  2.73173308e-01],
       [ 2.67458835e-01,  2.81481275e-01,  3.18277609e-01],
       [ 3.16758217e-01,  2.70786841e-01,  2.79512253e-01],
       [ 3.70506726e-01,  2.17637692e-01,  2.87962898e-01],
       [ 2.67369705e-01,  2.06738082e-01,  4.10717784e-01],
       [ 1.08177380e-01,  2.98840875e-01,  5.34797088e-01],
       [ 2.92170541e-01,  2.78114618e-01,  2.95892501e-01],
       [ 3.33877830e-01,  3.16004011e-01,  2.22337750e-01],
       [ 2.87140965e-01,  2.85877024e-01,  2.93032678e-01],
       [ 3.10741904e-01,  2.66181360e-01,  2.89962821e-01],
       [ 4.71916027e-01,  1.90765005e-01,  2.41405637e-01],
       [ 2.47833843e-01,  2.98173837e-01,  3.22527391e-01],
       [ 3.12022041e-01,  2.75900578e-01,  2.78800191e-01],
       [ 3.12880737e-01,  2.71308495e-01,  2.82635812e-01],
       [ 3.21945708e-01,  2.69830476e-01,  2.75660361e-01],
       [ 2.83322679e-01,  3.37815625e-01,  2.48397452e-01],
       [ 2.48879027e-01,  2.90836853e-01,  3.29093165e-01],
       [ 3.08806015e-01,  2.16105739e-01,  3.49135348e-01],
       [ 2.84754918e-01,  3.01757616e-01,  2.79743442e-01],
       [ 2.91828445e-01,  2.86128916e-01,  2.88082572e-01],
       [ 2.60779379e-01,  3.22566777e-01,  2.84361583e-01],
       [ 3.07860089e-01,  2.59122587e-01,  3.00232272e-01],
       [ 1.37773275e-01,  4.93040979e-01,  2.88630336e-01],
       [ 3.24886995e-01,  3.43307272e-01,  2.07221071e-01],
       [ 3.24507415e-01,  2.43526469e-01,  3.00992353e-01],
       [ 3.22623770e-01,  3.41182307e-01,  2.10794210e-01],
       [ 3.24395050e-01,  2.93472551e-01,  2.50537425e-01],
       [ 3.47209735e-01,  2.11929452e-01,  3.15513623e-01],
       [ 3.98874496e-01,  2.31545086e-01,  2.50059763e-01],
       [ 2.18596763e-01,  3.18010919e-01,  3.36342264e-01],
       [ 3.49133439e-01,  2.85337267e-01,  2.37015228e-01],
       [ 4.30837692e-01,  2.45472078e-01,  2.13277085e-01],
       [ 3.50295962e-01,  2.01033183e-01,  3.25719925e-01],
       [ 3.05717315e-01,  2.87919488e-01,  2.72857118e-01],
       [ 3.45415150e-01,  3.37711397e-01,  1.95204487e-01],
       [ 3.34587743e-01,  2.17146914e-01,  3.21432235e-01],
       [ 3.72844992e-01,  2.13730273e-01,  2.90349335e-01],
       [ 2.47783784e-01,  3.11385009e-01,  3.09108531e-01],
       [ 3.29111068e-01,  2.37237907e-01,  3.03562621e-01],
       [ 3.25135811e-01,  3.24030925e-01,  2.22817859e-01],
       [ 3.26726429e-01,  2.75987778e-01,  2.65179578e-01],
       [ 2.28379576e-01,  3.79535757e-01,  2.68658768e-01],
       [ 2.91558648e-01,  2.71516864e-01,  3.03399645e-01],
       [ 3.65903357e-01,  2.89839079e-01,  2.19517451e-01],
       [ 3.37083447e-01,  2.73880381e-01,  2.58083130e-01],
       [ 2.49899625e-01,  3.20765228e-01,  2.97619328e-01],
       [ 3.40378218e-01,  2.60256470e-01,  2.68739668e-01],
       [ 3.17277457e-01,  2.81800049e-01,  2.68064141e-01],
       [ 2.85997795e-01,  2.59086092e-01,  3.22706198e-01],
       [ 3.11984144e-01,  2.52892050e-01,  3.02901224e-01],
       [ 2.06403613e-01,  3.20121377e-01,  3.49320154e-01],
       [ 2.86354162e-01,  1.39456723e-01,  4.93331651e-01],
       [ 4.75123792e-02,  4.74956063e-01,  4.35306115e-01],
       [ 2.07342976e-01,  3.20569005e-01,  3.47656949e-01],
       [ 2.68325438e-01,  4.11665437e-01,  2.05208182e-01],
       [ 1.75408581e-01,  3.44054487e-01,  3.65088304e-01],
       [ 3.11445880e-01,  2.76060385e-01,  2.79183628e-01],
       [ 2.77762560e-01,  3.00676838e-01,  2.87814475e-01],
       [ 3.40005369e-01,  3.73244503e-01,  1.72583087e-01],
       [ 3.69001534e-01,  2.22670341e-01,  2.83661113e-01],
       [ 2.81579071e-01,  2.83025755e-01,  3.01637292e-01],
       [ 3.67783529e-01,  2.38831803e-01,  2.67321955e-01],
       [ 3.57651326e-01,  3.23878511e-01,  1.96857737e-01],
       [ 2.82855469e-01,  2.77668982e-01,  3.05891759e-01],
       [ 3.16171509e-01,  2.69219886e-01,  2.81658295e-01],
       [ 2.80606084e-01,  3.16085608e-01,  2.70331324e-01],
       [ 3.03861199e-01,  2.71374162e-01,  2.91254560e-01],
       [ 3.07407239e-01,  2.77462291e-01,  2.81611667e-01],
       [ 2.91157164e-01,  2.85646762e-01,  2.89235026e-01],
       [ 2.75055446e-01,  2.75529497e-01,  3.16416410e-01],
       [ 2.94719705e-01,  3.03050365e-01,  2.68807513e-01],
       [ 3.03057086e-01,  3.12827382e-01,  2.51987919e-01],
       [ 3.01738831e-01,  3.12074809e-01,  2.53879845e-01],
       [ 2.95421205e-01,  2.76274753e-01,  2.94531695e-01],
       [ 2.62014336e-01,  2.95965985e-01,  3.09065862e-01],
       [ 3.07593092e-01,  2.64433297e-01,  2.94850285e-01],
       [ 2.96621501e-01,  2.97704566e-01,  2.72063491e-01],
       [ 2.61559423e-01,  2.90871044e-01,  3.14826839e-01],
       [ 3.09513496e-01,  2.48805605e-01,  3.09843075e-01],
       [ 2.80320174e-01,  2.49436778e-01,  3.39925391e-01],
       [ 2.46552701e-01,  3.11137712e-01,  3.10724648e-01],
       [ 2.82696674e-01,  3.17475410e-01,  2.67008017e-01],
       [ 1.67404056e-01,  2.44616115e-01,  5.07378727e-01],
       [ 4.03373598e-01,  2.23339083e-01,  2.55158250e-01],
       [ 3.53924265e-01,  2.43866531e-01,  2.73824419e-01],
       [ 2.73820452e-01,  2.81042723e-01,  3.11869236e-01],
       [ 2.86100556e-01,  2.81178211e-01,  2.98891422e-01],
       [ 2.78614646e-01,  2.86327519e-01,  3.01313958e-01],
       [ 3.29707353e-01,  3.14950467e-01,  2.26722628e-01],
       [ 3.10590901e-01,  2.31444684e-01,  3.28604601e-01],
       [ 2.66596255e-01,  3.36392157e-01,  2.65874281e-01],
       [ 2.70295115e-01,  2.58499800e-01,  3.40640126e-01],
       [ 3.19629677e-01,  2.75941430e-01,  2.71673572e-01],
       [ 4.10896619e-01,  2.81911658e-01,  1.93651632e-01],
       [ 3.03023866e-01,  2.28098801e-01,  3.40562794e-01],
       [ 2.80642828e-01,  2.75249941e-01,  3.10766910e-01],
       [ 2.72616704e-01,  3.08934577e-01,  2.85063896e-01],
       [ 2.74942237e-01,  2.86882582e-01,  3.04585798e-01],
       [ 3.50886774e-01,  2.21548732e-01,  3.00927316e-01],
       [ 3.08870567e-01,  2.88645886e-01,  2.69190548e-01],
       [ 2.93140644e-01,  2.59514307e-01,  3.14709583e-01],
       [ 2.75538536e-01,  3.74427862e-01,  2.25905056e-01],
       [ 2.96470262e-01,  3.14454956e-01,  2.56616444e-01],
       [ 2.80635285e-01,  2.76525212e-01,  3.09421197e-01],
       [ 2.90187319e-01,  2.56912107e-01,  3.20686435e-01],
       [ 2.77834173e-01,  3.37934760e-01,  2.53520708e-01],
       [ 3.41314852e-01,  2.38000309e-01,  2.91321497e-01],
       [ 3.34271210e-01,  2.58313602e-01,  2.76168400e-01],
       [ 2.78718984e-01,  3.63332890e-01,  2.31651417e-01],
       [ 2.37574512e-01,  3.30964666e-01,  3.01423784e-01],
       [ 2.72116202e-01,  2.48974042e-01,  3.49747728e-01],
       [ 3.04154035e-01,  3.10389661e-01,  2.53185791e-01],
       [ 2.82406282e-01,  2.54995734e-01,  3.31200103e-01],
       [ 3.43919989e-01,  2.85078979e-01,  2.41583291e-01],
       [ 2.82330815e-01,  2.73361392e-01,  3.11002482e-01],
       [ 3.09893406e-01,  2.57919213e-01,  2.99522291e-01],
       [ 2.33181140e-01,  3.09025349e-01,  3.28179615e-01],
       [ 2.73461351e-01,  2.78703987e-01,  3.14734047e-01],
       [ 3.74356143e-01,  2.48630312e-01,  2.51889450e-01],
       [ 2.85897898e-01,  3.22145847e-01,  2.59683914e-01],
       [ 3.03456273e-01,  3.13049607e-01,  2.51421987e-01],
       [ 3.33014047e-01,  1.80123493e-01,  3.70302953e-01],
       [ 2.67368644e-01,  1.89072024e-01,  4.36963844e-01],
       [ 4.50593060e-01,  2.18850251e-01,  2.26139530e-01],
       [ 3.11035506e-01,  2.83330545e-01,  2.72347555e-01],
       [ 1.96318370e-01,  3.68198230e-01,  3.14810802e-01],
       [ 2.98278101e-01,  2.23661031e-01,  3.51165110e-01],
       [ 2.92142425e-01,  2.71451004e-01,  3.02873732e-01],
       [ 2.80109653e-01,  2.36270717e-01,  3.55974592e-01],
       [ 2.32062310e-01,  2.98437247e-01,  3.40705210e-01],
       [ 2.85260628e-01,  2.86407331e-01,  2.94400381e-01],
       [ 3.10948796e-01,  2.76047163e-01,  2.79667116e-01],
       [ 3.03694151e-01,  2.74114425e-01,  2.88595669e-01],
       [ 4.09296340e-01,  4.71976164e-02,  5.05334594e-01],
       [ 3.38700029e-01,  2.55009830e-01,  2.75602907e-01],
       [ 2.66107321e-01,  2.77769824e-01,  3.23755865e-01],
       [ 3.38888279e-01,  2.55614722e-01,  2.74809659e-01],
       [ 2.74698443e-01,  2.64413518e-01,  3.28996630e-01],
       [ 2.38979368e-01,  2.82824394e-01,  3.49577405e-01],
       [ 3.52458416e-01,  3.26463642e-01,  1.98749091e-01],
       [ 3.31508844e-01,  3.35718895e-01,  2.07884369e-01],
       [ 2.50785722e-01,  4.63923828e-01,  1.87005375e-01],
       [ 3.07529003e-01,  2.30636774e-01,  3.32745987e-01],
       [ 2.74370732e-01,  2.84974099e-01,  3.07165524e-01],
       [ 3.16100733e-01,  2.59146436e-01,  2.92193740e-01],
       [ 2.85033283e-01,  3.14770687e-01,  2.67220435e-01],
       [ 2.93023856e-01,  3.21511484e-01,  2.53507741e-01],
       [ 2.55235737e-01,  3.23407745e-01,  2.89391986e-01],
       [ 2.77921684e-01,  2.95430291e-01,  2.92828007e-01],
       [ 2.93840659e-01,  2.71414408e-01,  3.01186882e-01],
       [ 2.42427598e-01,  2.52017184e-01,  3.82052932e-01],
       [ 2.27102247e-01,  3.35842208e-01,  3.08608564e-01],
       [ 2.61873687e-01,  3.16034137e-01,  2.89386938e-01],
       [ 2.75124241e-01,  2.76212828e-01,  3.15609389e-01],
       [ 3.15385124e-01,  2.66966496e-01,  2.84712419e-01],
       [ 2.93887338e-01,  2.70333335e-01,  3.02279700e-01],
       [ 2.81321509e-01,  3.01343500e-01,  2.83568756e-01],
       [ 2.95550606e-01,  2.74783120e-01,  2.95945470e-01],
       [ 2.63486344e-01,  2.87093857e-01,  3.16674463e-01],
       [ 2.96736280e-01,  3.24862976e-01,  2.47106746e-01],
       [ 3.26774546e-01,  2.46262766e-01,  2.95840416e-01],
       [ 2.62292285e-01,  2.91464986e-01,  3.13406237e-01],
       [ 3.26086184e-01,  2.48125481e-01,  2.94472786e-01],
       [ 2.56483205e-01,  2.91690440e-01,  3.19581766e-01],
       [ 2.95165191e-01,  2.69918492e-01,  3.01423421e-01],
       [ 3.12760666e-01,  3.05079874e-01,  2.50198887e-01],
       [ 3.31270328e-01,  2.97397028e-01,  2.40956028e-01],
       [ 3.17441253e-01,  2.95717578e-01,  2.54627405e-01],
       [ 3.04012584e-01,  2.66950968e-01,  2.95716856e-01],
       [ 2.90913085e-01,  3.32131500e-01,  2.46175963e-01],
       [ 3.62948316e-01,  3.04727773e-01,  2.08783226e-01],
       [ 3.03674856e-01,  3.69890562e-01,  2.04395198e-01],
       [ 1.82202008e-01,  3.39606749e-01,  3.60520911e-01],
       [ 2.97927486e-01,  2.76212131e-01,  2.92104585e-01],
       [ 2.86198540e-01,  3.04882492e-01,  2.75331244e-01],
       [ 3.24968636e-01,  2.48165254e-01,  2.95487806e-01],
       [ 3.16829394e-01,  3.02091979e-01,  2.49285916e-01],
       [ 2.85315098e-01,  3.08199516e-01,  2.73061558e-01],
       [ 2.79462522e-01,  2.79689680e-01,  3.07317429e-01],
       [ 2.95444408e-01,  2.89943593e-01,  2.80733098e-01],
       [ 2.63521249e-01,  3.12140442e-01,  2.91393997e-01],
       [ 2.61544075e-01,  3.34471614e-01,  2.72678948e-01],
       [ 4.65125250e-01,  3.45316677e-01,  1.10291054e-01],
       [ 2.97174601e-01,  2.32155439e-01,  3.41959244e-01],
       [ 3.07259624e-01,  2.78777361e-01,  2.80431060e-01],
       [ 2.89887812e-01,  3.17732895e-01,  2.59854716e-01],
       [ 2.48380124e-01,  2.95192336e-01,  3.25035030e-01],
       [ 2.97369411e-01,  3.00577208e-01,  2.68615170e-01],
       [ 3.03161069e-01,  3.05493393e-01,  2.58580699e-01],
       [ 3.54567013e-01,  2.43240587e-01,  2.73925777e-01],
       [ 3.09077922e-01,  2.51478109e-01,  3.07326402e-01],
       [ 2.99721052e-01,  3.14895943e-01,  2.53196801e-01],
       [ 2.79756820e-01,  2.68320022e-01,  3.19180871e-01],
       [ 2.65702581e-01,  3.31330100e-01,  2.71282866e-01],
       [ 2.53975374e-01,  2.74975871e-01,  3.40603982e-01],
       [ 3.40914715e-01,  2.39950786e-01,  2.89563154e-01],
       [ 3.17602663e-01,  2.58590387e-01,  2.91345070e-01],
       [ 3.67231751e-01,  2.46059711e-01,  2.60298849e-01],
       [ 2.68587335e-01,  2.78831923e-01,  3.19881470e-01],
       [ 2.65394895e-01,  3.18735775e-01,  2.83171491e-01],
       [ 2.85904817e-01,  2.77031754e-01,  3.03400589e-01],
       [ 2.57106781e-01,  3.38409093e-01,  2.73700054e-01],
       [ 3.23866061e-01,  2.49330457e-01,  2.95274520e-01],
       [ 2.51837188e-01,  3.31178870e-01,  2.85750011e-01],
       [ 2.85491723e-01,  2.93426041e-01,  2.87138009e-01],
       [ 2.59673782e-01,  2.95731844e-01,  3.11855309e-01],
       [ 2.57748526e-01,  2.79448914e-01,  3.31298032e-01],
       [ 2.87912495e-01,  3.55407177e-01,  2.29549693e-01],
       [ 2.58358049e-01,  3.09778754e-01,  2.99164153e-01],
       [ 2.75734032e-01,  2.45034734e-01,  3.50319752e-01],
       [ 2.07353486e-01,  2.72446861e-01,  4.03307740e-01],
       [ 3.01685114e-01,  3.42789360e-01,  2.27449742e-01],
       [ 4.87071475e-01,  4.98348005e-01,  7.37645484e-03],
       [ 2.07233274e-01,  2.89709884e-01,  3.82261974e-01],
       [ 3.47866651e-01,  3.07065435e-01,  2.18621409e-01],
       [ 3.23285774e-01,  3.47360572e-01,  2.05329186e-01],
       [ 3.32111724e-01,  2.65190728e-01,  2.71097247e-01],
       [ 1.08819377e-01,  4.30129961e-01,  3.77017860e-01],
       [ 1.99848240e-01,  3.65808513e-01,  3.12722887e-01],
       [ 3.21685223e-01,  2.90856578e-01,  2.55387855e-01],
       [ 2.97085777e-01,  1.87061193e-01,  4.01586279e-01],
       [ 2.70632204e-01,  3.23677750e-01,  2.73262084e-01],
       [ 4.12282020e-01,  1.56900461e-01,  3.25577064e-01],
       [ 2.84344879e-01,  2.64239189e-01,  3.18756869e-01],
       [ 2.52730725e-01,  2.83247138e-01,  3.32877079e-01],
       [ 2.71167145e-01,  2.96188295e-01,  2.99077887e-01],
       [ 2.73918759e-01,  3.35932790e-01,  2.59049447e-01],
       [ 2.72545210e-01,  3.24114524e-01,  2.70948632e-01],
       [ 2.82650957e-01,  2.58472075e-01,  3.26991479e-01],
       [ 3.01614346e-01,  2.91142436e-01,  2.73615199e-01],
       [ 2.53483468e-01,  3.02273441e-01,  3.11968555e-01],
       [ 1.61539065e-01,  3.28198116e-01,  4.02222234e-01],
       [ 3.39923435e-01,  3.19116513e-01,  2.14743916e-01],
       [ 2.39004635e-01,  2.34413094e-01,  4.09731557e-01],
       [ 2.61512101e-01,  2.85560590e-01,  3.20473774e-01],
       [ 2.87813165e-01,  2.64032781e-01,  3.15319684e-01],
       [ 3.03885526e-01,  2.62884437e-01,  3.00145096e-01],
       [ 4.55057502e-01,  1.13087103e-01,  3.49451290e-01],
       [ 3.15432831e-01,  2.67287132e-01,  2.84337029e-01],
       [ 2.48606634e-01,  3.27103036e-01,  2.92994931e-01],
       [ 2.61809601e-01,  3.56063261e-01,  2.53740259e-01],
       [ 2.51920470e-01,  2.86980617e-01,  3.29751994e-01],
       [ 3.00626937e-01,  2.57603541e-01,  3.09115040e-01],
       [ 3.11277031e-01,  2.78125426e-01,  2.77273943e-01],
       [ 3.14876849e-01,  2.79202542e-01,  2.72834886e-01],
       [ 2.87875103e-01,  2.83338839e-01,  2.94869908e-01],
       [ 2.76795980e-01,  3.06334704e-01,  2.83311768e-01],
       [ 2.79194284e-01,  2.93189008e-01,  2.93759981e-01],
       [ 3.13277216e-01,  2.56135929e-01,  2.98128785e-01],
       [ 2.43076573e-01,  3.05544783e-01,  3.20310574e-01],
       [ 3.20315760e-01,  2.43029105e-01,  3.05592468e-01],
       [ 2.43583565e-01,  3.09835339e-01,  3.15365453e-01],
       [ 3.01172187e-01,  3.01182559e-01,  2.64449132e-01],
       [ 3.08499858e-01,  3.11039528e-01,  2.48643352e-01],
       [ 3.03302537e-01,  2.92143190e-01,  2.71044399e-01],
       [ 3.17238515e-01,  2.46564030e-01,  3.04681834e-01],
       [ 2.94689644e-01,  2.48669096e-01,  3.25235944e-01],
       [ 3.39101453e-01,  2.72236300e-01,  2.57933154e-01],
       [ 2.52660771e-01,  2.77201844e-01,  3.39638925e-01],
       [ 2.66611956e-01,  2.83591777e-01,  3.16957213e-01],
       [ 3.72262182e-01,  1.79400172e-01,  3.32116375e-01],
       [ 4.56040400e-01,  1.85375567e-01,  2.57962477e-01],
       [ 2.82675512e-01,  3.14025553e-01,  2.70206767e-01],
       [ 2.87170085e-01,  2.91827166e-01,  2.87041030e-01],
       [ 2.99893161e-01,  2.56694071e-01,  3.10857300e-01],
       [ 2.51932059e-01,  3.28085925e-01,  2.88516291e-01],
       [ 3.10700810e-01,  2.95537560e-01,  2.60914268e-01],
       [ 2.73573494e-01,  3.04906423e-01,  2.87971422e-01],
       [ 3.25912339e-01,  2.74757896e-01,  2.67123294e-01],
       [ 3.46047597e-01,  3.18452297e-01,  2.10384305e-01],
       [ 2.61461859e-01,  3.13821175e-01,  2.91939310e-01],
       [ 3.70003202e-01,  1.84807755e-01,  3.27355717e-01],
       [ 4.78878667e-01,  1.94833085e-01,  2.32589964e-01],
       [ 3.12745598e-01,  2.44499760e-01,  3.11413588e-01],
       [ 2.79704101e-01,  2.90240386e-01,  2.96201784e-01],
       [ 2.46516680e-01,  3.13220976e-01,  3.08690692e-01],
       [ 2.86609891e-01,  2.75899148e-01,  3.03861029e-01],
       [ 2.39522095e-01,  3.29818011e-01,  3.00349996e-01],
       [ 2.88837573e-01,  2.68209047e-01,  3.09724794e-01],
       [ 2.58665591e-01,  3.02084193e-01,  3.06484672e-01],
       [ 3.79758286e-01,  1.74087304e-01,  3.32021970e-01],
       [ 3.14451885e-01,  3.24498987e-01,  2.31568167e-01],
       [ 2.50345241e-01,  2.89270592e-01,  3.29090573e-01],
       [ 2.73173581e-01,  3.07281948e-01,  2.86083397e-01],
       [ 3.52054557e-01,  2.54453356e-01,  2.64495373e-01],
       [ 2.34779221e-01,  2.79329952e-01,  3.58715894e-01],
       [ 2.34427909e-01,  2.95250701e-01,  3.41310735e-01],
       [ 2.77757473e-01,  3.21403087e-01,  2.68255125e-01],
       [ 3.17853165e-01,  3.69491895e-01,  1.92851945e-01],
       [ 2.38273205e-01,  3.80469245e-01,  2.57529402e-01],
       [ 3.71986639e-01,  2.56936722e-01,  2.45535437e-01],
       [ 3.07616078e-01,  2.99545255e-01,  2.59988663e-01],
       [ 3.20013554e-01,  2.70039697e-01,  2.77235379e-01],
       [ 3.20557993e-01,  2.69765596e-01,  2.77008212e-01],
       [ 3.09349031e-01,  2.66334125e-01,  2.91149385e-01],
       [ 2.44890331e-01,  3.20858764e-01,  3.03004977e-01],
       [ 3.39552633e-01,  2.63565403e-01,  2.66126469e-01],
       [ 3.88885962e-01,  2.76653336e-01,  2.13981956e-01],
       [ 2.89303008e-01,  2.79315493e-01,  2.97551324e-01],
       [ 2.40590336e-01,  2.53367779e-01,  3.82708888e-01],
       [ 4.47895805e-01,  2.81334800e-01,  1.70030334e-01],
       [ 2.74628668e-01,  2.41481374e-01,  3.55897536e-01],
       [ 3.08152654e-01,  1.93233939e-01,  3.79855487e-01],
       [ 3.45918424e-01,  2.65555891e-01,  2.58619734e-01],
       [ 2.73774325e-01,  2.01072595e-01,  4.10556493e-01],
       [ 3.19931453e-01,  2.65806699e-01,  2.81627679e-01],
       [ 3.52902448e-01,  2.35886598e-01,  2.83216957e-01],
       [ 3.90763405e-01,  7.93082625e-02,  4.65906049e-01],
       [ 3.14429894e-01,  3.40374579e-01,  2.18349237e-01],
       [ 2.81061448e-01,  2.73499198e-01,  3.12192942e-01],
       [ 3.12846918e-01,  3.18538014e-01,  2.38121559e-01],
       [ 2.97313982e-01,  2.69362752e-01,  2.99844121e-01],
       [ 2.88169571e-01,  4.01507545e-01,  1.94725532e-01],
       [ 2.82621854e-01,  3.16186918e-01,  2.68263718e-01],
       [ 3.32747085e-01,  2.55146080e-01,  2.80834845e-01],
       [ 2.98753959e-01,  2.78534428e-01,  2.88914070e-01],
       [ 2.67162150e-01,  2.74474912e-01,  3.26179106e-01],
       [ 2.67197050e-01,  2.68074452e-01,  3.33235184e-01],
       [ 3.16780662e-01,  2.34146755e-01,  3.19147006e-01],
       [ 2.49387102e-01,  2.75778361e-01,  3.45080639e-01],
       [ 2.69657030e-01,  2.90118068e-01,  3.06851134e-01],
       [ 2.53569032e-01,  3.42417457e-01,  2.73787309e-01],
       [ 1.64180035e-01,  3.94016413e-01,  3.31980564e-01],
       [ 2.28821089e-01,  3.31839904e-01,  3.10469310e-01],
       [ 3.25766811e-01,  2.49811218e-01,  2.92957320e-01],
       [ 4.13045548e-01,  9.44210966e-02,  4.15790454e-01],
       [ 2.69812964e-01,  4.01500509e-01,  2.11033985e-01],
       [ 2.95420660e-01,  2.45575321e-01,  3.28009787e-01],
       [ 3.18824586e-01,  2.68627161e-01,  2.79776267e-01],
       [ 3.10636232e-01,  3.41792737e-01,  2.20448201e-01],
       [ 2.48758411e-01,  2.89906381e-01,  3.30230139e-01],
       [ 2.80168230e-01,  3.15822013e-01,  2.71005621e-01],
       [ 3.03483981e-01,  2.79732588e-01,  2.83094907e-01],
       [ 2.84513338e-01,  3.07680325e-01,  2.74337356e-01],
       [ 2.94508263e-01,  3.09112318e-01,  2.63350643e-01],
       [ 2.89495956e-01,  2.96170393e-01,  2.80466632e-01],
       [ 2.89729289e-01,  2.76809097e-01,  2.99715432e-01],
       [ 2.94058171e-01,  3.17030833e-01,  2.56549360e-01],
       [ 3.22935597e-01,  2.42300584e-01,  3.03859735e-01],
       [ 2.41804528e-01,  3.00898513e-01,  3.26589984e-01],
       [ 3.04095430e-01,  3.23399039e-01,  2.41684728e-01],
       [ 3.03380497e-01,  3.05076054e-01,  2.58762724e-01],
       [ 2.97873786e-01,  3.24330068e-01,  2.46527844e-01],
       [ 2.68448221e-01,  3.01327738e-01,  2.96799298e-01],
       [ 3.06495098e-01,  2.40420338e-01,  3.22375880e-01],
       [ 2.46764503e-01,  3.11525742e-01,  3.10101254e-01],
       [ 3.35437835e-01,  3.08287405e-01,  2.27719423e-01],
       [ 3.60740524e-01,  2.55393402e-01,  2.56225543e-01],
       [ 3.85585969e-01,  2.99453479e-01,  1.96390062e-01],
       [ 3.74375061e-01,  1.38646477e-01,  3.86132360e-01],
       [ 5.93833381e-01, -8.86115399e-03,  4.36366100e-01],
       [ 3.04045437e-01,  2.79944908e-01,  2.82340333e-01],
       [ 3.10034986e-01,  2.85145633e-01,  2.71505611e-01],
       [ 2.76293226e-01,  3.13531736e-01,  2.76986081e-01],
       [ 2.66061307e-01,  2.68726571e-01,  3.33781419e-01],
       [ 2.54768301e-01,  3.14020091e-01,  2.98876766e-01],
       [ 2.90972728e-01,  2.85662229e-01,  2.89403340e-01],
       [ 2.76750172e-01,  2.93912712e-01,  2.95550335e-01],
       [ 2.36666193e-01,  2.87515883e-01,  3.47121198e-01],
       [ 3.04384355e-01,  3.07159413e-01,  2.55918690e-01],
       [ 3.18026900e-01,  3.29946479e-01,  2.23879791e-01],
       [ 3.22295963e-01,  3.15832852e-01,  2.32254591e-01],
       [ 3.27399570e-01,  2.97125202e-01,  2.44540078e-01],
       [ 3.01845514e-01,  2.78832275e-01,  2.85589929e-01],
       [ 2.81948373e-01,  3.17864603e-01,  2.67380998e-01],
       [ 3.02926328e-01,  2.71182300e-01,  2.92369306e-01],
       [ 2.92654186e-01,  3.17265518e-01,  2.57658372e-01],
       [ 2.68510674e-01,  3.02095729e-01,  2.95972971e-01],
       [ 3.06632770e-01,  2.63161534e-01,  2.97135385e-01],
       [ 2.01192930e-01,  3.12360016e-01,  3.64432479e-01],
       [ 3.42084777e-01,  3.37400822e-01,  1.98062057e-01],
       [ 2.33478568e-01,  3.06380619e-01,  3.30580078e-01],
       [ 2.77416187e-01,  3.23239463e-01,  2.66921888e-01],
       [ 3.13696893e-01,  2.81775503e-01,  2.71394445e-01],
       [ 2.61524965e-01,  3.49833475e-01,  2.59274759e-01],
       [ 2.56188249e-01,  3.60843843e-01,  2.55344980e-01],
       [ 2.79905478e-01,  3.38510193e-01,  2.51043352e-01],
       [ 2.72515191e-01,  2.83031031e-01,  3.11170767e-01],
       [ 2.15726742e-01,  2.82253432e-01,  3.79754851e-01],
       [ 3.64007515e-01,  2.60010575e-01,  2.48957842e-01],
       [ 2.47124031e-01,  2.71725837e-01,  3.52414113e-01],
       [ 2.85182093e-01,  2.79964222e-01,  3.01088785e-01],
       [ 2.76990972e-01,  3.85815772e-01,  2.15949092e-01],
       [ 2.29480151e-01,  2.51799043e-01,  3.99387965e-01],
       [ 3.27418458e-01,  2.97921112e-01,  2.43795749e-01],
       [ 2.83785888e-01,  3.32697520e-01,  2.52375225e-01],
       [ 3.09769071e-01,  3.20145335e-01,  2.39443447e-01],
       [ 4.37060467e-01,  3.06528453e-01,  1.56038543e-01],
       [ 9.48974908e-02,  3.65130360e-01,  4.68123930e-01],
       [ 4.36056572e-01,  1.57210306e-01,  3.05844502e-01],
       [ 3.00582291e-01,  2.97063823e-01,  2.68901399e-01],
       [ 3.76421820e-01,  3.79675990e-01,  1.41624101e-01],
       [ 2.29957045e-01,  3.70045017e-01,  2.74841625e-01],
       [ 2.94582678e-01,  3.59272700e-01,  2.20483750e-01],
       [ 2.87275165e-01,  4.66136035e-01,  1.54086764e-01],
       [ 2.04995487e-01,  3.21689362e-01,  3.49459707e-01],
       [ 2.74180426e-01,  3.30798107e-01,  2.63317830e-01],
       [ 2.88573197e-01,  3.06337556e-01,  2.71636026e-01],
       [ 2.44939692e-01,  3.06841222e-01,  3.16869253e-01],
       [ 2.77236030e-01,  2.78108603e-01,  3.11335097e-01],
       [ 2.77217023e-01,  2.96762847e-01,  2.92226777e-01],
       [ 2.89402809e-01,  3.24155286e-01,  2.54561957e-01],
       [ 2.75753381e-01,  2.78505786e-01,  3.12491156e-01],
       [ 2.66508769e-01,  3.28421584e-01,  2.73095442e-01],
       [ 2.63586366e-01,  2.91190924e-01,  3.12280343e-01],
       [ 2.87357421e-01,  2.90476598e-01,  2.88195898e-01],
       [ 3.61301258e-01,  2.49680675e-01,  2.61529923e-01],
       [ 2.90676872e-01,  3.06851202e-01,  2.69117819e-01],
       [ 3.19878627e-01,  3.20601485e-01,  2.30212358e-01],
       [ 2.69268938e-01,  2.63294844e-01,  3.36303149e-01],
       [ 2.99996358e-01,  2.37625760e-01,  3.32414035e-01],
       [ 1.63828989e-01,  4.41434635e-01,  2.93558382e-01],
       [ 2.61088420e-01,  3.50618616e-01,  2.59041551e-01],
       [ 2.97917364e-01,  3.20717326e-01,  2.49667118e-01],
       [ 2.53335198e-01,  3.50202453e-01,  2.67226729e-01],
       [ 3.66860733e-01,  2.51651984e-01,  2.54932137e-01],
       [ 2.86894542e-01,  2.91701447e-01,  2.87441410e-01],
       [ 3.09815569e-01,  2.93313522e-01,  2.63835565e-01],
       [ 3.23713158e-01,  2.62283926e-01,  2.81733897e-01],
       [ 2.84342226e-01,  2.70374661e-01,  3.12089051e-01],
       [ 3.07664794e-01,  2.68859992e-01,  2.90154474e-01],
       [ 2.97528214e-01,  2.78010255e-01,  2.90656688e-01],
       [ 2.84774999e-01,  2.74481501e-01,  3.07255313e-01],
       [ 3.21223820e-01,  2.40395108e-01,  3.07645196e-01],
       [ 2.51841546e-01,  3.05596659e-01,  3.10416587e-01],
       [ 3.09834989e-01,  3.23517004e-01,  2.36461105e-01],
       [ 3.17940102e-01,  2.39061718e-01,  3.12373653e-01],
       [ 2.37768685e-01,  3.17262175e-01,  3.14514025e-01],
       [ 2.64208484e-01,  3.03548240e-01,  2.99071719e-01],
       [ 3.05118867e-01,  2.88382823e-01,  2.72971354e-01],
       [ 3.18765455e-01,  3.07867597e-01,  2.42346689e-01],
       [ 1.55339433e-01,  3.91543765e-01,  3.45919959e-01],
       [ 2.70390621e-01,  3.42329451e-01,  2.56948212e-01],
       [ 2.42169088e-01,  3.87963037e-01,  2.47642263e-01],
       [ 3.50627272e-01,  2.31088500e-01,  2.90475310e-01],
       [ 2.89250385e-01,  1.58531278e-01,  4.55902470e-01],
       [ 4.08991022e-01,  2.65082109e-01,  2.10042191e-01],
       [ 2.88188656e-01,  2.58171856e-01,  3.21395482e-01],
       [ 2.94238429e-01,  3.07466712e-01,  2.65132316e-01],
       [ 3.29046847e-01,  2.70023067e-01,  2.68999924e-01],
       [ 2.70035840e-01,  2.77202232e-01,  3.20053504e-01],
       [ 2.74741214e-01,  2.83215745e-01,  3.08606174e-01],
       [ 2.98050052e-01,  2.88443119e-01,  2.79678470e-01],
       [ 2.77775235e-01,  2.84050377e-01,  3.04539052e-01],
       [ 3.08281804e-01,  3.00486254e-01,  2.58498378e-01],
       [ 2.97663365e-01,  2.73159097e-01,  2.95521910e-01],
       [ 3.10520474e-01,  2.43356035e-01,  3.14931156e-01],
       [ 2.13106142e-01,  3.28448457e-01,  3.32386830e-01],
       [ 2.97000005e-01,  2.38365419e-01,  3.34734859e-01],
       [ 3.17967454e-01,  2.64448242e-01,  2.84872244e-01],
       [ 2.73031734e-01,  2.90626602e-01,  3.02753821e-01],
       [ 2.89972610e-01,  3.00000854e-01,  2.76297120e-01],
       [ 2.37627102e-01,  2.89260753e-01,  3.44027299e-01],
       [ 3.06593232e-01,  2.90627268e-01,  2.69407441e-01],
       [ 2.64819704e-01,  3.06664779e-01,  2.95352068e-01],
       [ 3.07592868e-01,  2.93288544e-01,  2.65920251e-01],
       [ 3.10193105e-01,  2.67524541e-01,  2.89095777e-01],
       [ 3.46698943e-01,  3.69220163e-01,  1.70398804e-01],
       [ 2.00395252e-01,  3.09053834e-01,  3.69157751e-01],
       [ 2.86519618e-01,  3.69531747e-01,  2.19680825e-01],
       [ 3.41761716e-01,  2.54219084e-01,  2.73696148e-01],
       [ 3.01734939e-01,  2.89090300e-01,  2.75498312e-01],
       [ 3.76085515e-01,  2.49713974e-01,  2.49418855e-01],
       [ 3.87690875e-01,  2.32146142e-01,  2.58130855e-01],
       [ 3.45491025e-01,  2.45097660e-01,  2.79925713e-01],
       [ 2.64723029e-01,  2.78459144e-01,  3.24542042e-01],
       [ 2.17031114e-01,  3.45416402e-01,  3.11200403e-01],
       [ 2.30418362e-01,  4.52647736e-01,  2.13305346e-01],
       [ 3.10221962e-01,  2.71688677e-01,  2.84779473e-01],
       [ 2.45566200e-01,  2.66049654e-01,  3.60948934e-01],
       [ 3.90819167e-01,  1.94057936e-01,  2.97769288e-01],
       [ 3.33723359e-01,  2.37507065e-01,  2.98895748e-01],
       [ 2.20780568e-01,  3.35143481e-01,  3.16602300e-01],
       [ 3.63377479e-01,  2.78040047e-01,  2.32245773e-01],
       [ 2.87187666e-01,  2.51904622e-01,  3.29546728e-01],
       [ 3.35768655e-01,  3.51162167e-01,  1.92291200e-01],
       [ 3.77258753e-01,  2.67225952e-01,  2.31482096e-01],
       [ 1.38998868e-01,  2.71558974e-01,  5.16988322e-01],
       [ 1.98356896e-01,  3.03747920e-01,  3.77908157e-01],
       [ 3.38613906e-01,  2.25018876e-01,  3.08366520e-01],
       [ 3.12652323e-01,  3.02516187e-01,  2.52642340e-01],
       [ 2.93743257e-01,  2.82232370e-01,  2.90109748e-01],
       [ 2.70703284e-01,  2.75804223e-01,  3.20835284e-01],
       [ 1.58844140e-01,  3.68198351e-01,  3.63374591e-01],
       [ 2.84725912e-01,  3.12300851e-01,  2.69803409e-01],
       [ 2.94250789e-01,  2.75155816e-01,  2.96862001e-01],
       [ 2.79454772e-01,  2.63937181e-01,  3.24335859e-01],
       [ 2.82696927e-01,  2.75609361e-01,  3.08228806e-01],
       [ 2.78366844e-01,  2.60038912e-01,  3.29888354e-01],
       [ 3.06735751e-01,  2.68858399e-01,  2.91058408e-01],
       [ 2.16675839e-01,  2.99902049e-01,  3.58161810e-01],
       [ 3.30031445e-01,  3.58238198e-01,  1.91451317e-01],
       [ 3.29764376e-01,  2.60723495e-01,  2.77774850e-01],
       [ 3.19577990e-01,  2.34378276e-01,  3.16086075e-01],
       [ 2.20475586e-01,  2.99432040e-01,  3.53875451e-01],
       [ 2.86851292e-01,  2.66704901e-01,  3.13419590e-01],
       [ 3.44873307e-01,  2.70694927e-01,  2.54471456e-01],
       [ 2.96330716e-01,  2.46184393e-01,  3.26346676e-01],
       [ 3.60015788e-01,  2.61448201e-01,  2.50818266e-01],
       [ 2.29299304e-01,  4.03317335e-01,  2.48996953e-01],
       [ 2.79872167e-01,  2.62991516e-01,  3.24936076e-01],
       [ 2.93126446e-01,  2.85268786e-01,  2.87658275e-01],
       [ 2.89063845e-01,  2.97679382e-01,  2.79426580e-01],
       [ 2.81271322e-01,  2.97510219e-01,  2.87360421e-01],
       [ 2.72410662e-01,  3.05172410e-01,  2.88906982e-01],
       [ 3.02623116e-01,  2.77696501e-01,  2.85984851e-01],
       [ 2.78173306e-01,  3.01325903e-01,  2.86763424e-01],
       [ 2.43651220e-01,  3.09713127e-01,  3.15412476e-01],
       [ 3.11503095e-01,  2.48570582e-01,  3.08119274e-01],
       [ 2.37485613e-01,  3.09714903e-01,  3.22454312e-01],
       [ 2.39967218e-01,  3.06401083e-01,  3.22994187e-01],
       [ 3.22568035e-01,  2.40073785e-01,  3.06695779e-01],
       [ 3.00385233e-01,  2.57275410e-01,  3.09719304e-01],
       [ 3.19194985e-01,  2.90299110e-01,  2.58145207e-01],
       [ 2.49532223e-01,  4.61728904e-01,  1.89499686e-01],
       [ 2.10635727e-01,  3.97002657e-01,  2.73809327e-01],
       [ 2.84738098e-01,  2.33535299e-01,  3.54067185e-01],
       [ 2.75698287e-01,  2.60899191e-01,  3.31851244e-01],
       [ 1.73447458e-01,  5.23232012e-01,  2.28579633e-01],
       [ 2.27680019e-01,  2.33619714e-01,  4.26641173e-01],
       [ 2.95525531e-01,  2.74498665e-01,  2.96265733e-01],
       [ 2.87334607e-01,  2.69805159e-01,  3.09573380e-01],
       [ 2.73587774e-01,  3.19458255e-01,  2.74177921e-01],
       [ 3.14013447e-01,  2.71633870e-01,  2.81232932e-01],
       [ 2.97070054e-01,  2.81097251e-01,  2.87969284e-01],
       [ 2.80195592e-01,  3.03857391e-01,  2.82269766e-01],
       [ 2.96039406e-01,  2.77483655e-01,  2.92671586e-01],
       [ 2.59421700e-01,  2.93451992e-01,  3.14488078e-01],
       [ 2.96349710e-01,  3.11439990e-01,  2.59472725e-01],
       [ 3.10715567e-01,  3.24380984e-01,  2.34940307e-01],
       [ 3.17393983e-01,  2.31597102e-01,  3.21485135e-01],
       [ 2.93104162e-01,  3.05104438e-01,  2.68422452e-01],
       [ 3.11908666e-01,  2.84104787e-01,  2.70774517e-01],
       [ 2.82689319e-01,  2.72865810e-01,  3.11154988e-01],
       [ 2.78671327e-01,  2.93196619e-01,  2.94289110e-01],
       [ 3.11400723e-01,  2.34706357e-01,  3.23951615e-01],
       [ 2.90936692e-01,  2.79281048e-01,  2.95934139e-01],
       [ 2.66839465e-01,  2.90725344e-01,  3.09242983e-01],
       [ 2.92521617e-01,  2.78031622e-01,  2.95624893e-01],
       [ 2.84411209e-01,  2.80060165e-01,  3.01782797e-01],
       [ 2.85964166e-01,  2.63082722e-01,  3.18311191e-01],
       [ 1.46925899e-01,  3.46772471e-01,  4.03181690e-01],
       [ 2.52646244e-01,  3.44601584e-01,  2.72814226e-01],
       [ 2.66920267e-01,  3.49740072e-01,  2.54025234e-01],
       [ 2.96342471e-01,  2.66264454e-01,  3.04110252e-01],
       [ 2.72600006e-01,  3.16085992e-01,  2.78306189e-01],
       [ 3.73547151e-01,  2.78494575e-01,  2.23864730e-01],
       [ 3.80216591e-01,  2.85866978e-01,  2.12148503e-01],
       [ 3.47439360e-01,  2.10470704e-01,  3.17029939e-01],
       [ 3.18388738e-01,  2.64968244e-01,  2.83937795e-01],
       [-8.62751882e-02,  5.15796297e-01,  6.85648311e-01],
       [ 3.32705002e-01,  2.17476123e-01,  3.22883862e-01],
       [ 2.94437000e-01,  2.19025287e-01,  3.61294015e-01],
       [ 1.16638890e-01,  5.35675815e-01,  2.87467639e-01],
       [ 3.28485667e-01,  3.41853059e-01,  2.05427741e-01],
       [ 2.99672365e-01,  3.42572957e-01,  2.29414441e-01],
       [ 3.05089394e-01,  2.53133004e-01,  3.09503535e-01],
       [ 2.69230453e-01,  2.27021464e-01,  3.80610940e-01],
       [ 3.35703095e-01,  3.50504088e-01,  1.92849472e-01],
       [ 6.35081251e-01,  6.42662612e-01, -1.23767353e-01],
       [ 2.15051060e-01,  3.16324616e-01,  3.42458386e-01],
       [ 3.36781880e-01,  3.00296052e-01,  2.33669577e-01],
       [ 3.53471955e-01,  2.38315525e-01,  2.80104177e-01],
       [ 2.79216530e-01,  2.54559984e-01,  3.35201418e-01],
       [ 2.21993564e-01,  2.17306564e-01,  4.59274488e-01],
       [ 3.31086120e-01,  2.72216852e-01,  2.64995842e-01],
       [ 2.75400615e-01,  3.07413718e-01,  2.83689099e-01],
       [ 2.95059139e-01,  2.96250442e-01,  2.74963581e-01],
       [ 2.63154288e-01,  3.30455720e-01,  2.74657027e-01],
       [ 2.81691367e-01,  3.45152603e-01,  2.43718532e-01],
       [ 2.61133678e-01,  2.83649068e-01,  3.22935476e-01],
       [ 3.06969495e-01,  3.06762813e-01,  2.53910658e-01],
       [ 1.76360400e-01,  2.68382052e-01,  4.55697523e-01],
       [ 3.16835133e-01,  2.78471570e-01,  2.71742990e-01],
       [ 3.24488810e-01,  2.99653688e-01,  2.44760983e-01],
       [ 4.51502816e-01,  1.00542797e+00, -1.39988504e-01],
       [ 2.98013621e-01,  2.57176936e-01,  3.12249134e-01],
       [ 3.25748106e-01,  2.84477822e-01,  2.57825636e-01],
       [ 2.96762422e-01,  2.71072392e-01,  2.98600749e-01],
       [ 2.85629379e-01,  3.25438728e-01,  2.57001038e-01],
       [ 2.50341223e-01,  3.49674192e-01,  2.70763268e-01],
       [ 3.68440862e-01,  2.60508687e-01,  2.44881255e-01],
       [ 3.14606317e-01,  2.94264819e-01,  2.58548024e-01],
       [ 2.35998460e-01,  3.25294585e-01,  3.08628407e-01],
       [ 3.52464352e-01,  2.70979034e-01,  2.47800448e-01],
       [ 2.90308378e-01,  2.74816162e-01,  3.01205405e-01],
       [ 2.59560334e-01,  3.35206147e-01,  2.74046682e-01],
       [ 2.79982837e-01,  3.16813743e-01,  2.70272309e-01],
       [ 2.86532685e-01,  2.65662560e-01,  3.14886799e-01],
       [ 3.11275028e-01,  3.16871308e-01,  2.40972312e-01],
       [ 3.10573225e-01,  3.22163728e-01,  2.36977738e-01],
       [ 2.51782431e-01,  3.05578084e-01,  3.10500660e-01],
       [ 3.11909927e-01,  3.24720055e-01,  2.33599729e-01],
       [ 3.11694846e-01,  3.21858188e-01,  2.36252458e-01],
       [ 2.57077086e-01,  3.07126507e-01,  3.03161509e-01],
       [ 2.96050862e-01,  2.60803605e-01,  3.10294481e-01],
       [-1.92329409e-01,  1.21740195e+00,  4.72300425e-01],
       [ 3.29351331e-01,  2.85920139e-01,  2.53273277e-01],
       [ 3.39978717e-01,  3.05497036e-01,  2.26402788e-01],
       [ 2.14480440e-01,  3.75260776e-01,  2.87437776e-01],
       [ 3.60644150e-01,  2.29131691e-01,  2.83777301e-01],
       [ 3.40920502e-01,  3.51535398e-01,  1.87960526e-01],
       [ 2.10386394e-01,  3.97355117e-01,  2.73803726e-01],
       [ 3.26089935e-01,  3.00974173e-01,  2.42168782e-01],
       [ 2.81793622e-01,  3.18360987e-01,  2.67077687e-01],
       [ 3.29235171e-01,  2.74803364e-01,  2.64097500e-01],
       [ 2.79456851e-01,  2.70688104e-01,  3.16924390e-01],
       [ 2.84289213e-01,  3.06953103e-01,  2.75245090e-01],
       [ 3.08469174e-01,  2.75977710e-01,  2.82094725e-01],
       [ 3.05780906e-01,  3.07016293e-01,  2.54766634e-01],
       [ 2.75741240e-01,  2.86356357e-01,  3.04288337e-01],
       [ 3.14010070e-01,  3.08060112e-01,  2.46380597e-01],
       [ 2.27858258e-01,  3.00960989e-01,  3.43072902e-01],
       [ 3.15056897e-01,  2.32727167e-01,  3.22531655e-01],
       [ 2.99449669e-01,  2.66215329e-01,  3.01029600e-01],
       [ 2.66484690e-01,  3.16297707e-01,  2.84345417e-01],
       [ 3.14262643e-01,  2.70501445e-01,  2.82150536e-01],
       [ 2.68551035e-01,  3.05982572e-01,  2.92111830e-01],
       [ 3.42407224e-01,  2.82785978e-01,  2.44999526e-01],
       [ 2.71340182e-01,  2.94632660e-01,  3.00463746e-01],
       [ 3.63008604e-01,  3.42856009e-01,  1.77853254e-01],
       [ 2.99199760e-01,  2.93023513e-01,  2.74098372e-01],
       [ 2.75777192e-01,  3.08007952e-01,  2.82738322e-01],
       [ 2.90599341e-01,  2.78102669e-01,  2.97490680e-01],
       [ 2.94119811e-01,  2.99424374e-01,  2.72824439e-01],
       [ 2.19412707e-01,  2.85983796e-01,  3.70504188e-01],
       [ 3.98778195e-01,  3.51492352e-01,  1.46390545e-01],
       [ 2.13643308e-01,  3.65108054e-01,  2.97186528e-01],
       [ 3.82954500e-01,  2.83346649e-01,  2.12353117e-01],
       [ 2.74981770e-01,  3.72569986e-01,  2.27858305e-01],
       [ 2.84863609e-01,  2.73418942e-01,  3.08289939e-01],
       [ 2.24720015e-01,  3.84486443e-01,  2.68543116e-01],
       [ 1.87777184e-01,  1.95363931e-01,  5.56753381e-01],
       [ 2.41471908e-01,  3.22881336e-01,  3.04832531e-01],
       [ 2.84691934e-01,  3.22388034e-01,  2.60622547e-01],
       [ 3.10600603e-01,  3.40472033e-01,  2.21556203e-01],
       [ 3.11741291e-01,  3.36148613e-01,  2.24125420e-01],
       [ 3.83265535e-01,  1.87640124e-01,  3.11932461e-01],
       [ 3.63479713e-01,  2.26945148e-01,  2.83711026e-01],
       [ 1.84752370e-01,  3.28999718e-01,  3.68303170e-01],
       [ 2.64818820e-01,  2.66058324e-01,  3.38200186e-01],
       [ 2.59677159e-01,  3.27920900e-01,  2.80542847e-01],
       [ 2.55535646e-01,  2.80958342e-01,  3.32166124e-01],
       [ 4.67099190e-01,  3.69515089e-01,  9.25155156e-02],
       [ 3.46775006e-01,  1.87284540e-01,  3.46505187e-01],
       [ 2.39133734e-01,  3.62888113e-01,  2.71121740e-01],
       [ 2.71731792e-01,  3.15269293e-01,  2.79950607e-01],
       [ 2.85125246e-01,  2.84242355e-01,  2.96742084e-01],
       [ 3.20015757e-01,  2.56712716e-01,  2.91034504e-01],
       [ 2.74542467e-01,  2.64393483e-01,  3.29190808e-01],
       [ 3.99127872e-01,  3.22433742e-01,  1.68118293e-01],
       [ 2.80374401e-01,  3.22045487e-01,  2.65108594e-01],
       [ 2.61663306e-01,  2.95649488e-01,  3.09771065e-01],
       [ 3.89822339e-01,  1.46605597e-01,  3.59507494e-01],
       [ 3.30671729e-01,  2.59681537e-01,  2.78021087e-01],
       [ 3.31502093e-01,  2.61085757e-01,  2.75823281e-01],
       [ 1.42085873e-01,  1.73327558e-01,  7.14530772e-01],
       [ 3.48337495e-01,  3.18770118e-01,  2.08302848e-01],
       [ 2.79002183e-01,  2.92315208e-01,  2.94833348e-01],
       [ 3.32556136e-01,  2.72099203e-01,  2.63806057e-01],
       [ 2.81962641e-01,  2.49776311e-01,  3.37707837e-01],
       [ 2.88763207e-01,  1.98801572e-01,  3.95010937e-01],
       [ 2.90238132e-01,  2.42169133e-01,  3.37548518e-01],
       [ 2.29732222e-01,  3.40945912e-01,  3.00824103e-01],
       [ 3.05523052e-01,  2.54528872e-01,  3.07534989e-01],
       [ 2.65814267e-01,  3.10456506e-01,  2.90620728e-01],
       [ 2.51751899e-01,  3.09773732e-01,  3.06333078e-01],
       [ 3.17700167e-01,  2.72030748e-01,  2.77373597e-01],
       [ 2.54197457e-01,  3.23801717e-01,  2.90122260e-01],
       [ 2.47820015e-01,  3.20679376e-01,  2.99963790e-01],
       [ 3.25136066e-01,  3.08908910e-01,  2.35886377e-01],
       [ 2.97662141e-01,  2.60781890e-01,  3.08670332e-01],
       [ 2.27600923e-01,  3.27871714e-01,  3.15724094e-01],
       [ 2.94827582e-01,  2.60413951e-01,  3.11977354e-01],
       [ 3.21807792e-01,  2.90432412e-01,  2.55678386e-01],
       [ 1.30778043e-01,  1.40464983e-01,  8.53958416e-01],
       [ 3.16543127e-01,  2.07267552e-01,  3.52018178e-01],
       [ 1.58198151e-01,  3.63416389e-01,  3.69062180e-01],
       [ 3.94274115e-01,  3.39821540e-01,  1.58040934e-01],
       [ 3.41674015e-01,  3.26757500e-01,  2.06985233e-01],
       [ 2.81165705e-01,  2.75272692e-01,  3.10192037e-01],
       [ 2.12776948e-01,  5.73470021e-01,  1.62771755e-01],
       [ 2.65448370e-01,  2.86134438e-01,  3.15539348e-01],
       [ 2.73025420e-01,  2.65954238e-01,  3.29117675e-01],
       [ 3.25928939e-01,  2.79350024e-01,  2.62609066e-01],
       [ 2.72508116e-01,  3.00586029e-01,  2.93298874e-01],
       [ 3.23645422e-01,  2.80705503e-01,  2.63341946e-01],
       [ 3.25848421e-01,  2.63780358e-01,  2.78221811e-01],
       [ 3.03542386e-01,  2.95377609e-01,  2.67715851e-01],
       [ 3.04979945e-01,  3.01108224e-01,  2.60965382e-01],
       [ 3.31285927e-01,  2.44319597e-01,  2.93708710e-01],
       [ 2.36314574e-01,  3.14722173e-01,  3.18720238e-01],
       [ 3.25282511e-01,  2.27786377e-01,  3.18052740e-01],
       [ 2.68169240e-01,  3.20956773e-01,  2.78258408e-01],
       [ 2.79340604e-01,  2.87953848e-01,  2.98897332e-01],
       [ 2.82897078e-01,  2.80799116e-01,  3.02579212e-01],
       [ 2.73006428e-01,  2.97169971e-01,  2.96172707e-01],
       [ 3.28690769e-01,  2.24726030e-01,  3.18267586e-01],
       [ 2.77003791e-01,  3.15706207e-01,  2.74245727e-01],
       [ 3.83001637e-01,  2.56224399e-01,  2.37577363e-01],
       [ 2.78668337e-01,  3.61381276e-01,  2.33254545e-01],
       [ 2.51112465e-01,  2.74766073e-01,  3.44191293e-01],
       [ 2.99545853e-01,  2.87690710e-01,  2.78973503e-01],
       [ 2.80133542e-01,  2.94742572e-01,  2.91250089e-01],
       [ 2.86433707e-01,  2.62730278e-01,  3.18202208e-01],
       [ 2.70666878e-01,  2.30782718e-01,  3.73985270e-01],
       [ 3.51497066e-01,  2.91409349e-01,  2.29536469e-01],
       [ 2.34346003e-01,  2.52496316e-01,  3.91971877e-01],
       [ 2.75888463e-01,  3.64104599e-01,  2.33670879e-01],
       [ 2.75880489e-01,  3.30177163e-01,  2.62203707e-01],
       [ 3.43583491e-01,  1.58855694e-01,  3.88942209e-01],
       [ 4.92042388e-01,  8.02841990e-01, -1.12004047e-01],
       [ 3.56357288e-01,  2.91382549e-01,  2.25651870e-01],
       [ 3.16012144e-01,  2.83039841e-01,  2.68016762e-01],
       [ 2.03672236e-01,  3.23908308e-01,  3.48816636e-01],
       [ 3.54545892e-01,  1.92186796e-01,  3.32632317e-01],
       [ 3.13918223e-01,  4.03937285e-01,  1.71617719e-01],
       [ 1.83659780e-01,  3.43629860e-01,  3.54433316e-01],
       [ 2.81516221e-01,  2.91696716e-01,  2.92880067e-01],
       [ 2.98894130e-01,  3.20230591e-01,  2.49198507e-01],
       [ 2.83844004e-01,  2.34146470e-01,  3.54328772e-01],
       [ 3.07022855e-01,  2.92394805e-01,  2.67306291e-01],
       [ 2.70523795e-01,  3.06317772e-01,  2.89739441e-01],
       [ 2.84827000e-01,  3.01855983e-01,  2.79577337e-01],
       [ 2.29304769e-01,  2.13211163e-01,  4.54468753e-01],
       [ 2.59584033e-01,  2.85247627e-01,  3.22951626e-01],
       [ 2.26256619e-01,  3.62089976e-01,  2.85673016e-01],
       [ 3.18601599e-01,  2.57667406e-01,  2.91368704e-01],
       [ 2.37619424e-01,  3.10103330e-01,  3.21902685e-01],
       [ 3.14696009e-01,  3.92157782e-01,  1.79088678e-01],
       [ 3.30776347e-01,  2.33679558e-01,  3.05966024e-01],
       [ 2.02775527e-01,  3.69327491e-01,  3.06080230e-01],
       [ 3.69571193e-01,  3.76449954e-01,  1.48621714e-01],
       [ 2.50348764e-01,  3.65403679e-01,  2.57443786e-01],
       [ 2.83583720e-01,  2.73565936e-01,  3.09470090e-01],
       [ 2.32670625e-01,  3.14426269e-01,  3.23237520e-01],
       [ 2.31866468e-01,  3.30678725e-01,  3.08111586e-01],
       [ 3.03260755e-01,  2.59861316e-01,  3.04009148e-01],
       [ 3.15757629e-01,  3.50182561e-01,  2.09368930e-01],
       [ 4.72966688e-01,  1.91197396e-01,  2.40256895e-01],
       [ 3.26044854e-01,  2.66210399e-01,  2.75562722e-01],
       [ 4.62383358e-01,  2.35469792e-01,  2.02224045e-01],
       [ 2.67722687e-01,  3.32677087e-01,  2.68046065e-01],
       [ 3.15335087e-01,  2.89781141e-01,  2.62134498e-01],
       [ 3.08965380e-01,  2.98372709e-01,  2.59844007e-01],
       [ 2.44744707e-01,  3.15671725e-01,  3.08236887e-01],
       [ 3.18784180e-01,  2.37621713e-01,  3.13170581e-01],
       [ 3.01745959e-01,  3.06180089e-01,  2.59260804e-01],
       [ 3.06569501e-01,  2.53731154e-01,  3.07359566e-01],
       [ 2.94597724e-01,  2.56116314e-01,  3.16949823e-01],
       [ 3.23845078e-01,  1.50834882e-01,  4.23765237e-01],
       [ 3.55909230e-01,  2.16990183e-01,  3.01573343e-01],
       [ 2.98932754e-01,  1.74771610e-01,  4.17465103e-01],
       [ 3.01282929e-01,  2.26281083e-01,  3.44650825e-01],
       [ 3.26387650e-01,  2.70733980e-01,  2.70691538e-01],
       [ 2.80466500e-01,  3.53419442e-01,  2.38020243e-01],
       [ 2.96824028e-01,  2.48157954e-01,  3.23571718e-01],
       [ 2.71492299e-01,  9.33833592e-02,  6.15681348e-01],
       [ 2.47892019e-01,  2.95023760e-01,  3.25770537e-01],
       [ 2.67521871e-01,  2.69543796e-01,  3.31227912e-01],
       [ 2.55635332e-01,  3.24444635e-01,  2.87995617e-01],
       [ 3.00945447e-01,  3.05711384e-01,  2.60439745e-01],
       [ 2.70583982e-01,  3.30657355e-01,  2.66996638e-01],
       [ 3.08257005e-01,  2.51492971e-01,  3.08129589e-01],
       [ 2.87913375e-01,  3.15974734e-01,  2.63337935e-01],
       [ 3.39272236e-01,  2.45146937e-01,  2.85460263e-01],
       [ 3.37839249e-01,  2.36490601e-01,  2.96178569e-01],
       [ 3.30681569e-01,  3.01178745e-01,  2.38036377e-01],
       [ 2.98212566e-01,  2.65259868e-01,  3.03291099e-01],
       [ 3.05875146e-01,  2.74006482e-01,  2.86589917e-01],
       [ 2.82625465e-01,  3.11014299e-01,  2.73060278e-01],
       [ 3.09337307e-01,  2.69843882e-01,  2.87521804e-01],
       [ 3.24253379e-01,  3.00632694e-01,  2.44074624e-01],
       [ 3.02555857e-01,  2.89384192e-01,  2.74428328e-01],
       [ 3.25827428e-01,  2.99647261e-01,  2.43601710e-01],
       [ 3.51679682e-01,  2.88520047e-01,  2.32010972e-01],
       [ 2.82041548e-01,  3.14565527e-01,  2.70327756e-01],
       [ 2.56562333e-01,  2.76351467e-01,  3.36074282e-01],
       [ 2.90514652e-01,  3.18598516e-01,  2.58478180e-01],
       [ 3.15923222e-01,  2.73006645e-01,  2.78048152e-01],
       [ 2.26628578e-01,  2.78730980e-01,  3.69700328e-01],
       [ 4.09083247e-01,  2.65081177e-01,  2.09978051e-01],
       [ 2.21796026e-01,  3.52506620e-01,  2.99172281e-01],
       [ 3.60484401e-01,  1.84727549e-01,  3.36398717e-01],
       [ 2.63758154e-01,  3.46965296e-01,  2.59503829e-01],
       [ 3.43928600e-01,  2.35166047e-01,  2.92041502e-01],
       [ 2.69134075e-01,  2.66312307e-01,  3.33042280e-01],
       [ 2.47035102e-01,  1.63739436e-01,  5.10135347e-01],
       [ 2.64860886e-01,  3.39218664e-01,  2.65121115e-01],
       [ 2.65424854e-01,  3.46838094e-01,  2.57961960e-01],
       [ 2.11847393e-01,  3.11061052e-01,  3.52074114e-01],
       [ 2.80248710e-01,  3.70217715e-01,  2.24833991e-01],
       [ 2.90126628e-01,  1.63859536e-01,  4.45960695e-01],
       [ 6.80046260e-02,  3.94429848e-01,  4.82613123e-01],
       [ 2.98452961e-01,  3.15606317e-01,  2.53731791e-01],
       [ 2.87433136e-01,  3.00887703e-01,  2.77934918e-01],
       [ 2.85615288e-01,  2.86247928e-01,  2.94201849e-01],
       [ 2.96688754e-01,  2.52569587e-01,  3.18730606e-01],
       [ 4.83340251e-01,  4.77593932e-01,  1.99385445e-02],
       [ 2.81455640e-01,  3.17614069e-01,  2.68091888e-01],
       [ 3.80124045e-01,  3.08538378e-01,  1.92717272e-01],
       [ 3.12279833e-01,  2.68596437e-01,  2.85986463e-01],
       [ 2.77884371e-01,  3.29792608e-01,  2.60592048e-01],
       [ 2.01512541e-01,  2.89886998e-01,  3.89874469e-01],
       [ 2.81584782e-01,  3.11154634e-01,  2.73954432e-01],
       [ 2.21082306e-01,  2.24164035e-01,  4.50180674e-01],
       [ 2.45482844e-01,  2.62490282e-01,  3.65301095e-01],
       [ 2.69596999e-01,  3.37775691e-01,  2.61679015e-01],
       [ 3.72013169e-01,  2.59411687e-01,  2.43093751e-01],
       [ 2.88198764e-01,  2.92311616e-01,  2.85535210e-01],
       [ 3.41624863e-01,  3.28015050e-01,  2.05994148e-01],
       [ 3.12752238e-01,  3.16269293e-01,  2.40192208e-01],
       [ 3.25946031e-01,  1.91068952e-01,  3.63087801e-01],
       [ 6.02181600e-01,  5.18238730e-01, -5.54022681e-02],
       [ 2.71551119e-01,  2.47136143e-01,  3.52601495e-01],
       [ 2.87406217e-01,  2.54021553e-01,  3.26899794e-01],
       [ 3.26802510e-01,  2.97704431e-01,  2.44528018e-01],
       [ 3.23167282e-01,  2.78936411e-01,  2.65497256e-01],
       [ 2.73501585e-01,  2.48673388e-01,  3.48518109e-01],
       [ 2.41230774e-01,  3.45208552e-01,  2.84300637e-01],
       [ 2.93591494e-01,  2.68149465e-01,  3.04897828e-01],
       [ 2.61721220e-01,  3.05575536e-01,  2.99709802e-01],
       [ 3.15441267e-01,  3.02092978e-01,  2.50524419e-01],
       [ 3.73536103e-01,  3.22227918e-01,  1.86322137e-01],
       [ 2.32876954e-01,  3.03487816e-01,  3.34333445e-01],
       [ 3.45971872e-01,  1.83960690e-01,  3.51657531e-01],
       [ 2.93555561e-01,  2.36092770e-01,  3.41157791e-01],
       [ 2.70669171e-01,  3.37026898e-01,  2.61277334e-01],
       [ 3.79339900e-01,  1.58753131e-01,  3.52687346e-01],
       [ 3.03961551e-01,  3.04451610e-01,  2.58801791e-01],
       [-9.67490844e-02,  8.67992868e-01,  4.33037546e-01],
       [ 2.65855611e-01,  3.39514367e-01,  2.63868720e-01],
       [ 2.53607676e-01,  3.37523084e-01,  2.78113687e-01],
       [ 2.97069818e-01,  2.55440441e-01,  3.15136872e-01],
       [ 3.09608095e-01,  2.73054139e-01,  2.83972804e-01],
       [ 2.63040527e-01,  3.25605356e-01,  2.79204527e-01],
       [ 3.61173290e-01,  2.54572873e-01,  2.56688694e-01],
       [ 3.54685361e-01,  2.37717194e-01,  2.79683451e-01],
       [ 2.60051786e-01,  2.75122603e-01,  3.33449954e-01],
       [ 3.13277423e-01,  3.08546687e-01,  2.46595921e-01],
       [ 2.45447242e-01,  2.95917843e-01,  3.27630627e-01],
       [ 3.12480228e-01,  2.86312689e-01,  2.68094263e-01],
       [ 2.60026914e-01,  2.82053011e-01,  3.25890367e-01],
       [ 2.87993631e-01,  2.81789321e-01,  2.96334717e-01],
       [ 3.02424550e-01,  2.98755553e-01,  2.65559331e-01],
       [ 3.43356085e-01,  2.38137407e-01,  2.89313767e-01],
       [ 2.71624726e-01,  3.25853531e-01,  2.70286194e-01],
       [ 2.49130023e-01,  3.21649008e-01,  2.97606545e-01],
       [ 1.40236831e-01,  2.79127391e-01,  5.02799353e-01],
       [ 1.93006551e-01,  3.08594074e-01,  3.79663247e-01],
       [ 3.61583341e-01,  2.39369001e-01,  2.71981895e-01],
       [ 2.95189606e-01,  3.12036457e-01,  2.60018617e-01],
       [ 3.05898268e-01,  2.71788779e-01,  2.88842002e-01],
       [ 3.71031573e-01,  2.10354713e-01,  2.95761638e-01],
       [ 3.29070997e-01,  1.72954227e-01,  3.84613703e-01],
       [ 2.51417275e-01,  2.37984047e-01,  3.88570057e-01],
       [ 2.80685177e-01,  3.65840337e-01,  2.27854952e-01],
       [ 2.22126874e-01,  2.97548948e-01,  3.53886740e-01],
       [ 2.48290379e-01,  2.64295598e-01,  3.59701501e-01],
       [ 3.63943534e-01,  1.59420293e-01,  3.66819419e-01],
       [ 2.89257514e-01,  2.36395188e-01,  3.45515043e-01],
       [ 2.81056867e-01,  2.24487810e-01,  3.69712447e-01],
       [ 3.37053939e-01,  2.08416323e-01,  3.29536676e-01],
       [ 3.41570491e-01,  2.21190942e-01,  3.09985175e-01],
       [ 2.65821517e-01,  3.30570945e-01,  2.71846377e-01],
       [ 2.11336512e-01,  5.06774882e-01,  1.98994146e-01],
       [ 3.07275223e-01,  2.88134540e-01,  2.71180295e-01],
       [ 2.63805385e-01,  2.84271638e-01,  3.19312074e-01],
       [ 2.77801850e-01,  2.92710904e-01,  2.95671514e-01],
       [ 2.83601255e-01,  3.22473296e-01,  2.61595159e-01],
       [ 2.46890963e-01,  2.42634918e-01,  3.88326009e-01],
       [ 5.04521402e-01,  3.82449084e-03,  4.87995410e-01],
       [ 2.59704744e-01,  2.90860802e-01,  3.16877929e-01],
       [ 4.69599513e-01,  2.37632969e-01,  1.95703219e-01],
       [ 1.94951203e-01,  2.86628766e-01,  4.03092714e-01],
       [ 2.87717010e-01,  3.03042127e-01,  2.75593918e-01],
       [ 3.22979721e-01,  2.51496898e-01,  2.93783240e-01],
       [ 2.97503442e-01,  3.13924351e-01,  2.56132330e-01],
       [ 3.81031043e-01,  3.76731432e-01,  1.40484166e-01],
       [ 5.13870835e-01,  4.83955663e-01,  1.31215123e-03],
       [ 2.74415851e-01,  2.57188986e-01,  3.37512478e-01],
       [ 2.86274866e-01,  2.65821253e-01,  3.14985109e-01],
       [ 2.42852665e-01,  2.78906306e-01,  3.49331531e-01],
       [ 2.70937534e-01,  2.75024169e-01,  3.21424797e-01],
       [ 2.47168451e-01,  3.28539883e-01,  2.93195871e-01],
       [ 4.60446202e-01,  8.11867427e-02,  3.92549742e-01],
       [ 2.07196234e-01,  2.53006854e-01,  4.29327699e-01],
       [ 3.22803923e-01,  2.94211542e-01,  2.51253605e-01],
       [ 2.97057914e-01,  2.52708887e-01,  3.18190958e-01],
       [ 2.81367884e-01,  2.92617479e-01,  2.92109956e-01],
       [ 2.84762198e-01,  2.88041595e-01,  2.93253369e-01],
       [ 2.85954505e-01,  3.06667136e-01,  2.73879892e-01],
       [ 3.12122087e-01,  2.73132288e-01,  2.81500467e-01],
       [ 3.13746174e-01,  2.59271679e-01,  2.94326786e-01],
       [ 2.60869101e-01,  3.00856436e-01,  3.05337466e-01],
       [ 3.63788198e-01,  1.81250122e-01,  3.37707163e-01],
       [ 3.04052293e-01,  3.28307770e-01,  2.37486962e-01],
       [ 2.75635763e-01,  3.02281638e-01,  2.88415559e-01],
       [ 2.83981034e-01,  3.21659986e-01,  2.61961557e-01],
       [ 3.19388531e-01,  2.38571323e-01,  3.11497421e-01],
       [ 2.91248831e-01,  3.23374487e-01,  2.53517161e-01],
       [ 1.29605542e-01,  1.81555989e-01,  7.27819204e-01],
       [ 2.47795192e-01,  2.88266567e-01,  3.33112608e-01],
       [ 5.09679016e-01,  2.89312992e-01,  1.28340757e-01],
       [ 3.30597662e-01,  2.56999479e-01,  2.80866876e-01],
       [ 2.68812630e-01,  2.80840953e-01,  3.17484339e-01],
       [ 2.93093365e-01,  2.97726879e-01,  2.75444568e-01],
       [ 3.06694123e-01,  2.94609126e-01,  2.65498503e-01],
       [ 3.26897528e-01,  2.99001623e-01,  2.43261408e-01],
       [ 3.72424445e-01,  1.88049520e-01,  3.21095667e-01],
       [ 2.65705740e-01,  2.80347049e-01,  3.21416141e-01],
       [ 3.20512150e-01,  2.97147652e-01,  2.50559559e-01],
       [ 3.08930319e-01,  2.24566807e-01,  3.38566969e-01],
       [ 2.49929230e-01,  3.15191145e-01,  3.02987695e-01],
       [ 2.71045994e-01,  3.14550065e-01,  2.81324409e-01],
       [ 3.05335558e-01,  3.06635154e-01,  2.55524294e-01],
       [ 2.48724255e-01,  2.93937397e-01,  3.25968565e-01],
       [ 2.86621026e-01,  3.10723159e-01,  2.69426259e-01],
       [ 8.12118542e-02,  3.52051608e-01,  5.11026790e-01],
       [ 3.18297803e-01,  2.45323309e-01,  3.05017194e-01],
       [ 3.26944924e-01,  2.68260293e-01,  2.72668408e-01],
       [ 2.81744955e-01,  2.61172458e-01,  3.24940725e-01],
       [ 3.25332646e-01,  3.57667350e-01,  1.95664913e-01],
       [-8.12001447e-02,  6.73428320e-01,  5.14468054e-01],
       [ 2.93823158e-01,  1.70194821e-01,  4.31002308e-01],
       [ 3.37691013e-01,  2.57281479e-01,  2.74161173e-01],
       [ 3.38101700e-01,  2.77548367e-01,  2.53651276e-01],
       [ 2.00974224e-01,  2.86255888e-01,  3.95028838e-01],
       [ 3.44845391e-01,  2.20487893e-01,  3.07722492e-01],
       [ 3.53753784e-01,  2.99773093e-01,  2.20272707e-01],
       [ 2.20973274e-01,  3.53473669e-01,  2.99230013e-01],
       [ 2.59064590e-01,  2.66579642e-01,  3.44222657e-01],
       [ 2.84764106e-01,  3.18642486e-01,  2.63938213e-01],
       [ 3.81478618e-01,  2.52128535e-01,  2.42766127e-01],
       [ 2.62487475e-01,  2.95714772e-01,  3.08810251e-01],
       [ 2.84229335e-01,  3.28936513e-01,  2.55243494e-01],
       [ 2.51618439e-01,  2.33335456e-01,  3.94446768e-01],
       [ 2.41949787e-01,  3.04727258e-01,  3.22441388e-01],
       [ 2.57252733e-01,  3.57863514e-01,  2.56762252e-01],
       [ 2.69483464e-01,  2.73774011e-01,  3.24381052e-01],
       [ 3.12910373e-01,  3.07188483e-01,  2.48150011e-01],
       [ 2.85284114e-01,  3.15635421e-01,  2.66182441e-01],
       [ 2.99811598e-01,  2.79600734e-01,  2.86794823e-01],
       [ 2.69783788e-01,  2.89498828e-01,  3.07354286e-01],
       [ 2.75798003e-01,  3.33903986e-01,  2.58995296e-01],
       [ 3.97433943e-01,  2.21811017e-01,  2.61358240e-01],
       [ 2.82423326e-01,  3.18792644e-01,  2.66069980e-01],
       [ 3.14013634e-01,  3.37869908e-01,  2.20751458e-01],
       [ 3.05366935e-01,  3.49171698e-01,  2.19046671e-01],
       [ 3.10339015e-01,  2.79162412e-01,  2.77124032e-01],
       [ 2.63192072e-01,  3.21577911e-01,  2.82783399e-01],
       [ 3.15458462e-01,  2.51256087e-01,  3.01278909e-01],
       [ 2.96958149e-01,  2.46220588e-01,  3.25643804e-01],
       [ 4.59995280e-01,  4.97982666e-02,  4.45460783e-01],
       [ 3.13563297e-01,  2.81910900e-01,  2.71385544e-01],
       [ 2.74911175e-01,  2.57738311e-01,  3.36327854e-01],
       [ 2.65463655e-01,  2.44403778e-01,  3.63074140e-01],
       [ 3.32161966e-01,  2.71544123e-01,  2.64703923e-01],
       [ 3.31472675e-01,  2.77223430e-01,  2.59748677e-01],
       [ 3.86106956e-01,  2.46489046e-01,  2.44751570e-01],
       [ 3.31450572e-01,  2.88062149e-01,  2.49424476e-01],
       [ 2.48174727e-01,  3.19402218e-01,  3.00809332e-01],
       [ 2.66545603e-01,  3.05211821e-01,  2.94962730e-01],
       [ 3.04407021e-01,  2.59892936e-01,  3.02829663e-01],
       [ 3.16392837e-01,  2.93503701e-01,  2.57646209e-01],
       [ 2.63037225e-01,  2.82597583e-01,  3.21948515e-01],
       [ 2.65129401e-01,  2.95064204e-01,  3.06625786e-01],
       [ 3.07643349e-01,  2.57044297e-01,  3.02684206e-01],
       [ 2.00688909e-01,  4.37307931e-01,  2.54291461e-01],
       [ 9.04436795e-02,  4.10253280e-01,  4.25197676e-01],
       [ 3.38815683e-01,  3.52514026e-01,  1.88857383e-01],
       [ 1.49235223e-01,  3.85680362e-01,  3.59763130e-01],
       [ 3.06350447e-01,  3.27481546e-01,  2.36144410e-01],
       [ 3.29006005e-01,  2.21768776e-01,  3.21432185e-01],
       [ 3.88861799e-01,  2.50281906e-01,  2.38874179e-01],
       [ 3.00457289e-01,  3.46690193e-01,  2.25349566e-01],
       [ 2.49694448e-01,  5.57298627e-01,  1.37356355e-01],
       [ 2.50909383e-01,  2.33499818e-01,  3.95146716e-01],
       [ 3.21190640e-01,  2.83743229e-01,  2.62614376e-01],
       [ 2.63258013e-01,  3.07945047e-01,  2.95745963e-01],
       [ 2.72058124e-01,  3.04542782e-01,  2.89882760e-01],
       [ 3.04943313e-01,  2.62729702e-01,  2.99260894e-01],
       [ 2.56722219e-01,  2.95070589e-01,  3.15786688e-01],
       [ 2.67383536e-01,  3.07270241e-01,  2.92073251e-01],
       [ 4.34449703e-01,  2.24602282e-01,  2.31274024e-01],
       [ 2.63232929e-01,  2.78758841e-01,  3.25874863e-01],
       [ 2.21214317e-01,  2.96745601e-01,  3.55926430e-01],
       [ 2.65391422e-01,  2.96341623e-01,  3.05044320e-01],
       [ 2.85565030e-01,  3.19060244e-01,  2.62786818e-01],
       [ 2.97622114e-01,  2.33490596e-01,  3.39867663e-01],
       [ 3.10942458e-01,  2.96445908e-01,  2.59837675e-01],
       [ 3.17573310e-01,  2.42334242e-01,  3.09053007e-01],
       [ 2.04184820e-01,  4.73576996e-01,  2.26189736e-01],
       [ 2.41871769e-01,  2.85609239e-01,  3.42987113e-01],
       [ 2.81535113e-01,  3.34352058e-01,  2.53079075e-01],
       [ 2.77213376e-01,  3.19862896e-01,  2.70199528e-01],
       [ 3.76268432e-01,  1.85721651e-01,  3.20501751e-01],
       [ 2.84662427e-01,  1.55408979e-01,  4.67562538e-01],
       [ 2.73647425e-01,  3.17387512e-01,  2.76037361e-01],
       [-3.22586684e-02,  4.46753504e-01,  6.37913070e-01],
       [ 2.70467427e-01,  3.41537814e-01,  2.57555223e-01],
       [ 2.37069880e-01,  3.38058354e-01,  2.95336824e-01],
       [ 3.84543307e-01,  1.92909507e-01,  3.04471527e-01],
       [ 2.37668217e-01,  3.01845433e-01,  3.30410425e-01],
       [ 3.49785857e-01,  2.57835858e-01,  2.63013416e-01],
       [ 3.25464858e-01,  2.70915954e-01,  2.71347392e-01],
       [ 2.65281158e-01,  2.93238957e-01,  3.08331294e-01],
       [ 1.66134952e-01,  3.44813605e-01,  3.77169885e-01],
       [ 3.55939605e-01,  2.93387948e-01,  2.24188253e-01],
       [ 3.00606546e-01,  3.43699642e-01,  2.27658278e-01],
       [ 4.08123078e-01,  2.37869402e-01,  2.36721033e-01],
       [ 3.09737744e-01,  3.14863314e-01,  2.44115737e-01],
       [ 3.28164204e-01,  2.60542400e-01,  2.79424946e-01],
       [ 2.80299068e-01,  3.11387118e-01,  2.75008078e-01],
       [ 3.09176526e-01,  2.68221774e-01,  2.89353335e-01],
       [ 2.91947815e-01,  3.03676001e-01,  2.70879791e-01],
       [ 3.16861255e-01,  2.60540865e-01,  2.89996675e-01],
       [ 2.50656969e-01,  2.77217980e-01,  3.41962394e-01],
       [ 2.39491589e-01,  3.05820300e-01,  3.24142227e-01],
       [ 2.77207905e-01,  2.80552355e-01,  3.08786197e-01],
       [ 3.68914158e-01,  2.14187155e-01,  2.93230906e-01],
       [ 3.48078359e-01,  2.38124832e-01,  2.85078454e-01],
       [ 3.95907021e-01,  2.07990413e-01,  2.77621871e-01],
       [ 2.87413176e-01,  3.03476277e-01,  2.75477788e-01],
       [ 2.86600779e-01,  3.27327583e-01,  2.54406979e-01],
       [ 3.46335501e-01,  3.28741749e-01,  2.01673307e-01],
       [ 4.32145053e-01,  2.42030735e-01,  2.15682049e-01],
       [ 3.09663323e-01,  3.07117971e-01,  2.51137364e-01],
       [ 2.76871787e-01,  3.16095034e-01,  2.74015673e-01],
       [ 3.21803702e-01,  2.89838550e-01,  2.56242732e-01],
       [ 3.25909892e-01,  2.79917598e-01,  2.62074747e-01],
       [ 2.96500472e-01,  2.69951534e-01,  3.00041735e-01],
       [ 2.57228208e-01,  3.20205961e-01,  2.90308409e-01],
       [ 2.55561045e-01,  3.23089794e-01,  2.89346914e-01],
       [ 3.10310807e-01,  3.00537690e-01,  2.56593751e-01],
       [ 2.73138966e-01,  3.08357056e-01,  2.85084793e-01],
       [ 2.89016622e-01,  2.96349294e-01,  2.80764771e-01],
       [ 2.83418627e-01,  2.70630108e-01,  3.12785456e-01],
       [ 3.29393025e-01,  2.70035520e-01,  2.68676200e-01],
       [ 2.84079396e-01,  3.13939372e-01,  2.68915126e-01],
       [ 2.85874746e-01,  3.10577521e-01,  2.70287732e-01],
       [ 4.56220149e-01,  1.85952661e-01,  2.57196578e-01],
       [ 2.15981022e-01,  4.15170989e-01,  2.54029050e-01],
       [-1.55144326e-01,  7.98630838e-01,  5.81058088e-01],
       [ 7.79624733e-02,  4.62564522e-01,  3.95793971e-01],
       [ 2.58858298e-01,  3.68811307e-01,  2.46196616e-01],
       [ 2.56238922e-01,  3.03918374e-01,  3.07278482e-01],
       [ 2.66403876e-01,  3.06245081e-01,  2.94098368e-01],
       [ 3.18456617e-01,  2.90289837e-01,  2.58819217e-01],
       [ 2.80927836e-01,  2.50062322e-01,  3.38519519e-01],
       [ 4.10253141e-01,  1.57823621e-01,  3.26104457e-01],
       [ 2.91367449e-01,  3.88287469e-01,  2.01375238e-01],
       [ 3.00331915e-01,  3.23368943e-01,  2.45120692e-01],
       [ 2.87596368e-01,  2.91572416e-01,  2.86867727e-01],
       [ 3.08449025e-01,  3.12681842e-01,  2.47216165e-01],
       [ 2.36962433e-01,  2.80625312e-01,  3.54533787e-01],
       [ 3.34242483e-01,  2.64421273e-01,  2.69966531e-01],
       [ 3.54316845e-01,  2.15558790e-01,  3.04669789e-01],
       [ 2.03821349e-01,  2.05375694e-01,  5.08654823e-01],
       [ 3.51809510e-01,  3.07807416e-01,  2.14837459e-01],
       [ 2.88381244e-01,  3.34606718e-01,  2.46402351e-01],
       [ 3.13874438e-01,  2.91792009e-01,  2.61553117e-01],
       [ 2.84766703e-01,  2.80151033e-01,  3.01322304e-01],
       [ 3.72366159e-01,  2.99110891e-01,  2.06442240e-01],
       [ 2.15709456e-01,  3.26440191e-01,  3.31243902e-01],
       [ 3.24089705e-01,  2.73596211e-01,  2.69924856e-01],
       [ 3.56227368e-01,  2.83382255e-01,  2.33035088e-01],
       [ 3.35526296e-01,  2.89952022e-01,  2.44154702e-01],
       [ 2.94006986e-01,  3.13740447e-01,  2.59578417e-01],
       [ 2.49677080e-01,  2.88806865e-01,  3.30356267e-01],
       [ 2.65033755e-01,  2.82983767e-01,  3.19332399e-01],
       [ 3.67700117e-01,  2.33531353e-01,  2.72990525e-01],
       [ 3.54455403e-01,  3.62334504e-01,  1.69601408e-01],
       [ 2.97720004e-01,  2.67556064e-01,  3.01345161e-01],
       [ 1.59838178e-01,  4.36432433e-01,  3.02281266e-01],
       [ 3.56640480e-01,  2.93205812e-01,  2.23792827e-01],
       [ 2.96814518e-01,  1.97121600e-01,  3.87684642e-01],
       [ 3.31540930e-01,  2.23506885e-01,  3.16906607e-01],
       [ 3.00080586e-01,  3.05829168e-01,  2.61138896e-01],
       [ 2.94451445e-01,  3.07845970e-01,  2.64577442e-01],
       [ 2.64483755e-01,  2.96685449e-01,  3.05668089e-01],
       [ 3.23778758e-01,  3.44266726e-01,  2.07371729e-01],
       [ 3.63899617e-01,  2.21983737e-01,  2.88829171e-01],
       [ 2.44856363e-01,  3.27851457e-01,  2.96352658e-01],
       [ 1.89065594e-01,  3.67088456e-01,  3.24723343e-01],
       [ 2.66800120e-01,  2.87416311e-01,  3.12724929e-01],
       [ 2.77764527e-01,  3.05646236e-01,  2.82994981e-01],
       [ 2.89305238e-01,  2.95182695e-01,  2.81617619e-01],
       [ 2.74131859e-01,  2.86556570e-01,  3.05777158e-01],
       [ 2.72345546e-01,  2.90022086e-01,  3.04096052e-01],
       [ 2.66822511e-01,  3.03542328e-01,  2.96315731e-01],
       [ 3.06841315e-01,  2.59663819e-01,  3.00658198e-01],
       [ 2.78699185e-01,  3.38133117e-01,  2.52520135e-01],
       [ 2.81554414e-01,  3.04756088e-01,  2.80047138e-01],
       [ 3.14467945e-01,  3.23070018e-01,  2.32778037e-01],
       [ 2.55750392e-01,  3.12333382e-01,  2.99463957e-01],
       [ 3.23270817e-01,  3.25448595e-01,  2.23196599e-01],
       [ 3.06401828e-01,  2.67536262e-01,  2.92760497e-01],
       [ 2.78987530e-01,  2.88018963e-01,  2.99196398e-01],
       [ 3.25085526e-01,  2.36365844e-01,  3.08416534e-01],
       [ 2.71469212e-01,  2.87159695e-01,  3.07977230e-01],
       [ 2.83074430e-01,  2.62642930e-01,  3.21874500e-01],
       [ 2.71073646e-01,  3.06261396e-01,  2.89226523e-01],
       [ 2.86718788e-01,  3.09996517e-01,  2.70008449e-01],
       [ 3.10123905e-01,  2.38770921e-01,  3.20555817e-01],
       [ 2.90589263e-01,  2.54066225e-01,  3.23454160e-01],
       [ 2.99097414e-01,  2.55409413e-01,  3.13084883e-01],
       [ 3.25643502e-01,  2.74894087e-01,  2.67231444e-01],
       [ 2.73945859e-01,  2.90757811e-01,  3.01659278e-01],
       [ 2.66403645e-01,  3.29264435e-01,  2.72438224e-01],
       [ 2.61730145e-01,  2.69350169e-01,  3.37995848e-01],
       [ 3.23254139e-01,  2.65277508e-01,  2.79080910e-01],
       [ 2.87629685e-01,  2.97327220e-01,  2.81182876e-01],
       [ 3.98422135e-01,  1.53850938e-01,  3.41683110e-01],
       [ 3.07220489e-01,  2.92075831e-01,  2.67427507e-01],
       [ 3.88130074e-01,  5.34648621e-02,  5.19137925e-01],
       [ 1.47579523e-01,  3.04811707e-01,  4.53183042e-01],
       [ 2.89262826e-01,  2.56046657e-01,  3.22633708e-01],
       [ 2.83234154e-01,  2.78618845e-01,  3.04502473e-01],
       [ 3.38955353e-01,  3.36498757e-01,  2.01260075e-01],
       [ 2.35565658e-01,  3.38407184e-01,  2.96674122e-01],
       [ 3.26618467e-01,  2.98545890e-01,  2.43918895e-01],
       [ 1.65634184e-01,  3.33604158e-01,  3.90081713e-01],
       [ 3.11571416e-01,  3.33104950e-01,  2.26802202e-01],
       [ 3.03043500e-01,  2.52153571e-01,  3.12657448e-01],
       [ 2.96499572e-01,  2.46910216e-01,  3.25336846e-01],
       [ 2.59897865e-01,  3.20689612e-01,  2.87042799e-01],
       [ 3.43202574e-01,  2.09533421e-01,  3.22192859e-01],
       [ 2.88724424e-01,  3.38749295e-01,  2.42551680e-01],
       [ 2.62966460e-01,  3.16690483e-01,  2.87620164e-01],
       [ 2.59689304e-01,  3.88522589e-01,  2.30024843e-01],
       [ 2.34929469e-01,  4.86679719e-01,  1.88002861e-01],
       [ 2.95933289e-01,  3.19073660e-01,  2.52965406e-01],
       [ 3.40383371e-01,  2.42017727e-01,  2.87810568e-01],
       [ 3.30651777e-01,  3.01940666e-01,  2.37376820e-01],
       [ 2.49942822e-01,  3.83812305e-01,  2.43104730e-01],
       [ 4.66551658e-01,  2.95653446e-01,  1.47023936e-01],
       [ 2.07997170e-01,  3.19918837e-01,  3.47513212e-01],
       [ 2.88711838e-01,  3.48802340e-01,  2.34185592e-01],
       [ 2.31720432e-01,  3.58667157e-01,  2.82677845e-01],
       [ 2.66578800e-01,  3.05358796e-01,  2.94783591e-01],
       [ 2.47093856e-01,  3.03928563e-01,  3.17411984e-01],
       [ 2.96487917e-01,  3.30072179e-01,  2.42814039e-01],
       [ 3.03189039e-01,  2.81357367e-01,  2.81749282e-01],
       [ 2.95999368e-01,  3.28066065e-01,  2.44994586e-01],
       [ 3.13167900e-01,  2.20074767e-01,  3.39582069e-01],
       [ 3.33865606e-01,  2.63845849e-01,  2.70884796e-01],
       [ 2.60828345e-01,  4.42879298e-01,  1.91108519e-01],
       [ 3.24070241e-01,  2.12972806e-01,  3.36996918e-01],
       [-8.92810300e-03,  3.82737984e-01,  6.77930512e-01],
       [ 3.14306761e-01,  3.17293534e-01,  2.37923411e-01],
       [ 2.89362924e-01,  2.69057733e-01,  3.08270594e-01],
       [ 2.99603131e-01,  2.80776072e-01,  2.85810740e-01],
       [ 2.43774542e-01,  3.28995041e-01,  2.96453216e-01],
       [ 3.17966272e-01,  2.27213692e-01,  3.26045931e-01],
       [ 2.90407022e-01,  3.40404469e-01,  2.39602725e-01],
       [ 2.71824420e-01,  2.58600779e-01,  3.38796108e-01],
       [ 2.82660185e-01,  2.54129855e-01,  3.31913029e-01],
       [ 2.63766509e-01,  2.93323226e-01,  3.09880340e-01],
       [ 2.96339405e-01,  2.75938075e-01,  2.93963472e-01],
       [ 2.88075767e-01,  2.93966387e-01,  2.84026521e-01],
       [ 2.94361863e-01,  3.00558581e-01,  2.71510280e-01],
       [ 2.97996692e-01,  2.89164271e-01,  2.79020600e-01],
       [ 2.86007299e-01,  3.19732966e-01,  2.61752516e-01],
       [ 2.80512133e-01,  2.69477357e-01,  3.17112118e-01],
       [ 2.94985673e-01,  2.84770463e-01,  2.86321753e-01],
       [ 2.42004714e-01,  2.89157863e-01,  3.38921531e-01],
       [ 2.51063519e-01,  2.78452807e-01,  3.40104072e-01],
       [ 2.78357318e-01,  3.22929041e-01,  2.66279678e-01],
       [ 3.46599993e-01,  2.30581381e-01,  2.94674260e-01],
       [ 2.03865317e-01,  3.71559183e-01,  3.02823341e-01],
       [ 2.25547481e-01,  2.73571463e-01,  3.77258062e-01],
       [ 3.20720737e-01,  2.67805652e-01,  2.78847605e-01],
       [ 2.61987477e-01,  3.14386503e-01,  2.90843582e-01],
       [ 2.98109739e-01,  2.70707502e-01,  2.97634188e-01],
       [ 3.01307386e-01,  2.72688854e-01,  2.92400582e-01],
       [ 2.62579910e-01,  3.12343326e-01,  2.92186691e-01],
       [ 3.02167343e-01,  2.63988164e-01,  3.00679576e-01],
       [ 3.29060663e-01,  2.90440927e-01,  2.49276707e-01],
       [ 2.81283260e-01,  2.90107553e-01,  2.94715277e-01],
       [ 2.72948924e-01,  2.62926283e-01,  3.32604592e-01],
       [ 2.59013959e-01,  3.39990302e-01,  2.70344931e-01],
       [ 2.80804728e-01,  2.65275258e-01,  3.21398802e-01],
       [ 2.82310721e-01,  2.77588230e-01,  3.06544362e-01],
       [ 2.35064779e-01,  2.63285178e-01,  3.77467531e-01],
       [ 3.02317711e-01,  2.51432208e-01,  3.14198854e-01],
       [ 4.64735395e-01,  6.63212526e-02,  4.12720880e-01],
       [ 2.45139470e-01,  3.00862177e-01,  3.22795373e-01],
       [ 2.57086433e-01,  3.15262923e-01,  2.95187158e-01],
       [ 2.74579185e-01,  2.39739201e-01,  3.58090651e-01],
       [ 2.62345188e-01,  3.35957690e-01,  2.70537085e-01],
       [ 2.57902139e-01,  2.62276345e-01,  3.50568843e-01],
       [ 3.56968084e-01,  3.80178604e-01,  1.55041560e-01],
       [ 2.24987003e-01,  3.05820090e-01,  3.41356506e-01],
       [ 3.41664292e-01,  2.09192418e-01,  3.24088674e-01],
       [ 2.74294336e-01,  2.99372669e-01,  2.92650215e-01],
       [ 2.83491606e-01,  2.28580541e-01,  3.61666489e-01],
       [ 1.90583931e-01,  3.07857439e-01,  3.83851202e-01],
       [ 2.55518944e-01,  3.78092454e-01,  2.42088789e-01],
       [ 2.77091094e-01,  2.56684623e-01,  3.35112617e-01],
       [ 3.09508723e-01,  2.23610130e-01,  3.39118966e-01],
       [ 2.55424911e-01,  2.77616347e-01,  3.35977500e-01],
       [ 4.17211408e-01,  1.79323848e-01,  2.93669223e-01],
       [ 2.65846530e-01,  2.97389676e-01,  3.03496090e-01],
       [ 3.50864076e-01,  2.77324964e-01,  2.43074334e-01],
       [ 4.52386537e-01,  2.14475049e-01,  2.29394493e-01],
       [ 3.35640146e-01,  2.95804991e-01,  2.38684188e-01],
       [ 4.76862493e-01,  1.27834321e-01,  3.12620478e-01],
       [ 3.80931155e-01,  2.16990723e-01,  2.79871801e-01],
       [ 2.93854949e-01,  2.57730959e-01,  3.15933528e-01],
       [ 2.88507556e-01,  3.19804272e-01,  2.59298018e-01],
       [ 2.68207029e-01,  3.05869654e-01,  2.92580441e-01],
       [ 2.96722370e-01,  3.07080801e-01,  2.63135844e-01],
       [ 3.08523921e-01,  3.18316397e-01,  2.42153820e-01],
       [ 2.30607840e-01,  2.92249149e-01,  3.49244743e-01],
       [ 3.40620890e-01,  2.68390608e-01,  2.60390079e-01],
       [ 3.54554974e-01,  2.48183456e-01,  2.68782133e-01],
       [ 3.21924602e-01,  2.18957872e-01,  3.31887392e-01],
       [ 2.53198357e-01,  2.53632143e-01,  3.66554022e-01],
       [ 2.53567032e-01,  2.60503211e-01,  3.57820699e-01],
       [ 1.42086651e-01,  2.24391967e-01,  5.95169503e-01],
       [ 6.60112970e-01,  3.64874644e-01,  8.91865900e-03],
       [ 2.49769500e-01,  3.21010512e-01,  2.97525073e-01],
       [ 3.15878797e-01,  2.62478575e-01,  2.88901969e-01],
       [ 4.33286627e-03,  3.46511655e-01,  7.08286994e-01],
       [ 2.49942343e-01,  3.28153540e-01,  2.90575561e-01],
       [ 2.55043716e-01,  3.08200826e-01,  3.04300004e-01],
       [ 3.15267253e-01,  2.99360009e-01,  2.53196696e-01],
       [ 3.07742493e-01,  2.48279469e-01,  3.12206835e-01],
       [ 3.01204463e-01,  3.27266063e-01,  2.40943679e-01],
       [ 2.75462042e-01,  2.82761730e-01,  3.08316995e-01],
       [ 3.11783964e-01,  2.69184205e-01,  2.85855044e-01],
       [ 2.65888945e-01,  2.94639773e-01,  3.06243974e-01],
       [ 2.89344761e-01,  2.95751650e-01,  2.81023446e-01],
       [ 3.04414385e-01,  2.75252159e-01,  2.86732579e-01],
       [ 2.71163618e-01,  2.93156579e-01,  3.02145134e-01],
       [ 2.98801880e-01,  2.74216729e-01,  2.93295058e-01],
       [ 2.87184321e-01,  2.97557013e-01,  2.81400137e-01],
       [ 3.25411310e-01,  2.70879736e-01,  2.71432334e-01],
       [ 2.51646706e-01,  3.01831638e-01,  3.14456861e-01],
       [ 6.58374259e-01,  2.18973307e-01,  1.20629059e-01],
       [ 3.30208131e-01,  2.43933408e-01,  2.95138382e-01],
       [ 2.80063248e-01,  2.98491114e-01,  2.87619661e-01],
       [ 2.97075015e-01,  2.90702778e-01,  2.78403624e-01],
       [ 3.05947027e-01,  3.08626287e-01,  2.53145884e-01],
       [ 2.67777059e-01,  2.77781090e-01,  3.21902618e-01],
       [ 3.14518094e-01,  2.93617706e-01,  2.59238017e-01],
       [ 2.65697152e-01,  3.06555301e-01,  2.94536316e-01],
       [ 2.81930880e-01,  2.55237423e-01,  3.31443028e-01],
       [ 3.01739870e-01,  2.98808558e-01,  2.66152631e-01],
       [ 3.37772157e-01,  2.68905950e-01,  2.62364927e-01],
       [ 2.63237821e-01,  3.33413641e-01,  2.71905677e-01],
       [ 2.72979741e-01,  3.16736000e-01,  2.77315793e-01],
       [ 2.90960515e-01,  3.01001550e-01,  2.74376423e-01],
       [ 2.87963515e-01,  3.64378886e-01,  2.22386549e-01],
       [ 2.65459033e-01,  2.73300678e-01,  3.29367365e-01],
       [ 1.46878573e-01,  4.05046164e-01,  3.45169160e-01],
       [ 2.55955977e-01,  2.98279571e-01,  3.13320864e-01],
       [ 3.28590133e-01,  2.47345394e-01,  2.92957694e-01],
       [ 2.93692900e-01,  2.08880729e-01,  3.75374277e-01],
       [ 2.42151155e-01,  2.50772752e-01,  3.83984396e-01],
       [ 2.57674183e-01,  2.80586445e-01,  3.30137683e-01],
       [ 3.23075709e-01,  2.39461649e-01,  3.06887632e-01],
       [ 1.79085768e-01,  3.64237519e-01,  3.40074222e-01],
       [ 2.55108792e-01,  2.96986368e-01,  3.15590824e-01],
       [ 3.46189715e-01,  3.09907775e-01,  2.17518155e-01],
       [ 2.92247076e-01,  3.29542044e-01,  2.47177534e-01],
       [ 3.75294964e-01,  3.83485799e-01,  1.39802847e-01],
       [ 1.81853511e-01,  3.29158758e-01,  3.72087590e-01],
       [ 3.23300258e-01,  2.75578155e-01,  2.68678095e-01],
       [ 2.97265999e-01,  2.67080691e-01,  3.02307065e-01],
       [ 3.36821417e-01,  2.91857827e-01,  2.41293210e-01],
       [ 2.69430894e-01,  2.66240402e-01,  3.32791045e-01],
       [ 2.98698986e-01,  2.50584468e-01,  3.18871199e-01],
       [ 3.09429233e-01,  3.16309343e-01,  2.43112137e-01],
       [ 2.86945205e-01,  3.14237825e-01,  2.65860736e-01],
       [ 3.24732711e-01,  2.28927008e-01,  3.17270529e-01],
       [ 3.41573545e-01,  2.05752820e-01,  3.28360357e-01],
       [ 2.97742909e-01,  2.51227278e-01,  3.19140937e-01],
       [ 2.58711930e-01,  3.88171835e-01,  2.31224408e-01],
       [ 2.83510076e-01,  2.30162088e-01,  3.59658828e-01],
       [ 2.20661246e-01,  3.30002271e-01,  3.21759627e-01],
       [ 2.64025969e-01,  3.13345592e-01,  2.89707076e-01],
       [ 3.18786655e-01,  2.65678295e-01,  2.82831853e-01],
       [ 2.86879033e-01,  2.35980843e-01,  3.48663289e-01],
       [ 3.21260727e-01,  2.33222759e-01,  3.15743735e-01],
       [ 3.02973361e-01,  3.34797045e-01,  2.32944980e-01],
       [ 2.93999315e-01,  2.99798094e-01,  2.72583819e-01],
       [ 2.66750305e-01,  2.94241338e-01,  3.05728322e-01],
       [ 2.83431457e-01,  3.24990338e-01,  2.59503384e-01],
       [ 3.82900164e-01,  2.60896633e-01,  2.33152195e-01],
       [ 3.72711815e-01,  2.45988158e-01,  2.55887044e-01],
       [ 2.68326938e-01,  2.11548242e-01,  4.02679521e-01],
       [ 6.22524159e-01,  2.54288295e-01,  1.04582677e-01],
       [ 2.81485267e-01,  2.54147312e-01,  3.33178531e-01],
       [ 3.41198683e-01,  2.27470844e-01,  3.03141349e-01],
       [ 6.85532773e-01,  9.37874652e-02,  2.38291796e-01],
       [ 3.55169727e-01, -4.32067110e-03,  7.16931020e-01],
       [ 3.16085721e-01,  2.93677072e-01,  2.57760678e-01],
       [ 2.60519691e-01,  2.87264624e-01,  3.19764375e-01],
       [ 2.78318380e-01,  3.15705629e-01,  2.72940686e-01],
       [ 2.71703561e-01,  2.86076985e-01,  3.08852407e-01],
       [ 2.80624610e-01,  3.14082382e-01,  2.72168910e-01],
       [ 2.75650102e-01,  2.82036846e-01,  3.08876719e-01],
       [ 2.90992187e-01,  2.78912694e-01,  2.96257483e-01],
       [ 2.89410905e-01,  3.10612910e-01,  2.66831469e-01],
       [ 2.69407216e-01,  3.03431116e-01,  2.93718940e-01],
       [ 2.80080296e-01,  3.16003318e-01,  2.70924570e-01],
       [ 3.07799974e-01,  2.71513226e-01,  2.87285420e-01],
       [ 2.82472918e-01,  2.76622631e-01,  3.07392177e-01],
       [ 3.44608523e-01,  2.48443977e-01,  2.77183028e-01],
       [ 2.04617494e-01,  2.53542981e-01,  4.32426369e-01],
       [ 2.51857685e-01,  4.17634381e-01,  2.16306628e-01],
       [ 3.07525459e-01,  2.83091819e-01,  2.75884341e-01],
       [ 2.87176895e-01,  2.74900642e-01,  3.04326140e-01],
       [ 2.79451176e-01,  3.05549986e-01,  2.81390376e-01],
       [ 2.90457631e-01,  2.88814241e-01,  2.86759479e-01],
       [ 2.74146874e-01,  2.94885767e-01,  2.97272910e-01],
       [ 2.93215915e-01,  2.78276783e-01,  2.94675364e-01],
       [ 2.69759647e-01,  3.20091790e-01,  2.77446389e-01],
       [ 3.06788677e-01,  2.82933812e-01,  2.76738827e-01],
       [ 3.36111743e-01,  2.74293118e-01,  2.58528187e-01],
       [ 2.77195649e-01,  2.60982211e-01,  3.30108092e-01],
       [ 2.94655294e-01,  2.62584175e-01,  3.09791737e-01],
       [ 2.97870469e-01,  2.96378230e-01,  2.72138043e-01],
       [ 3.46269411e-01,  2.34714931e-01,  2.90412988e-01],
       [ 3.85604815e-01,  1.63283296e-01,  3.40756470e-01],
       [ 3.00664377e-01,  3.24383188e-01,  2.43932685e-01],
       [ 2.61196889e-01,  2.98145122e-01,  3.07728399e-01],
       [ 1.75953305e-01,  3.98919303e-01,  3.12780306e-01],
       [ 3.23721909e-01,  2.65137929e-01,  2.78791374e-01],
       [ 3.17335477e-01,  2.49844314e-01,  3.00990159e-01],
       [ 2.28717430e-01,  3.31553903e-01,  3.10863385e-01],
       [ 3.10179464e-01,  3.64303262e-01,  2.03119523e-01],
       [ 3.33293311e-01,  2.30230498e-01,  3.07468314e-01],
       [ 1.82623178e-01,  3.39204832e-01,  3.60374169e-01],
       [ 2.69600045e-01,  2.52902182e-01,  3.47974710e-01],
       [-1.02869110e-01,  7.54439365e-01,  5.02798438e-01],
       [ 5.32568563e-01,  1.65946175e-01,  2.31379919e-01],
       [ 2.60195751e-01,  2.62597417e-01,  3.47505054e-01],
       [ 2.68961508e-01,  3.25928582e-01,  2.72886974e-01],
       [ 2.45958485e-01,  3.13232969e-01,  3.09299600e-01],
       [ 3.28375767e-01,  2.90350930e-01,  2.49958166e-01],
       [ 2.69316843e-01,  3.13508911e-01,  2.84075934e-01],
       [ 2.77340292e-01,  3.23009313e-01,  2.67205145e-01],
       [ 2.42637862e-01,  3.40887548e-01,  2.86684644e-01],
       [ 3.56974534e-01,  2.07188372e-01,  3.12035807e-01],
       [ 4.46229305e-01,  2.55838425e-01,  1.93481899e-01],
       [ 2.92779897e-01,  3.12932387e-01,  2.61476765e-01],
       [ 4.68792726e-01,  2.04781894e-01,  2.28630553e-01],
       [ 3.47799069e-01,  3.62846288e-01,  1.74211226e-01],
       [ 3.95557680e-01,  2.36133457e-01,  2.47898995e-01],
       [ 1.99977667e-01,  3.69783910e-01,  3.08991487e-01],
       [ 2.94136820e-01,  3.22393968e-01,  2.51685830e-01],
       [ 3.08084187e-01,  2.59604137e-01,  2.99495452e-01],
       [ 2.84538740e-01,  3.06482897e-01,  2.75444641e-01],
       [ 2.51797764e-01,  2.62706666e-01,  3.57336183e-01],
       [ 2.45697547e-01,  3.41861842e-01,  2.82533796e-01],
       [ 2.24641159e-01,  3.25461941e-01,  3.21554000e-01],
       [ 2.28924283e-01,  3.03514100e-01,  3.39040643e-01],
       [ 3.55470091e-01,  2.31714970e-01,  2.85484542e-01],
       [ 3.19506849e-01,  2.66835586e-01,  2.80969264e-01],
       [ 2.40528106e-01,  3.10658092e-01,  3.18001427e-01],
       [ 2.62926744e-01,  4.23173858e-01,  2.02209814e-01],
       [ 2.72767375e-01,  2.59793210e-01,  3.36369031e-01],
       [ 2.50757044e-01,  3.86867274e-01,  2.39938004e-01],
       [ 3.21514975e-01,  2.10265409e-01,  3.42992197e-01],
       [ 2.57017918e-01,  2.73626465e-01,  3.38594173e-01],
       [ 1.56916429e-01,  3.19061029e-01,  4.20049478e-01],
       [ 2.77550751e-01,  2.73235749e-01,  3.16208209e-01],
       [ 2.87644042e-01,  2.93200849e-01,  2.85209572e-01],
       [ 2.77982849e-01,  2.94519912e-01,  2.93672848e-01],
       [ 2.77944491e-01,  3.11042858e-01,  2.77675491e-01],
       [ 3.20637503e-01,  2.71044243e-01,  2.75643202e-01],
       [ 2.91417395e-01,  3.07851878e-01,  2.67470092e-01],
       [ 3.10012480e-01,  2.87168591e-01,  2.69556690e-01],
       [ 3.17593180e-01,  2.65397868e-01,  2.84243553e-01],
       [ 2.92521728e-01,  2.63605685e-01,  3.10881294e-01],
       [ 2.75697256e-01,  2.93454886e-01,  2.97100337e-01],
       [ 2.54630182e-01,  3.21982108e-01,  2.91380603e-01],
       [ 3.33103323e-01,  2.76303653e-01,  2.59206640e-01],
       [ 2.88125459e-01,  2.77233941e-01,  3.00909198e-01],
       [ 2.74250933e-01,  3.42604117e-01,  2.52961374e-01],
       [ 2.86738301e-01,  2.74831064e-01,  3.04851757e-01],
       [ 3.18583899e-01,  2.95216304e-01,  2.54071012e-01],
       [ 3.15657319e-01,  2.47893537e-01,  3.04765025e-01],
       [ 3.05313204e-01,  2.77604123e-01,  2.83477241e-01],
       [ 2.92464680e-01,  3.01304306e-01,  2.72629822e-01],
       [ 3.19473417e-01,  2.77958688e-01,  2.69820766e-01],
       [ 2.99668071e-01,  2.66622183e-01,  3.00379256e-01],
       [ 2.81606549e-01,  2.66750811e-01,  3.18918350e-01],
       [ 2.66676226e-01,  3.29106335e-01,  2.72305998e-01],
       [ 2.72979243e-01,  3.06408525e-01,  2.87125208e-01],
       [ 3.09743803e-01,  2.74907402e-01,  2.81961509e-01],
       [ 2.90511134e-01,  3.30591268e-01,  2.47881115e-01],
       [ 1.98830894e-01,  3.49187410e-01,  3.29497672e-01],
       [ 2.90340731e-01,  2.81866556e-01,  2.93884161e-01],
       [ 3.00664951e-01,  2.70194134e-01,  2.95628287e-01],
       [ 3.74536039e-01,  2.91149042e-01,  2.11743053e-01],
       [ 2.71458748e-01,  4.06907035e-01,  2.05702483e-01],
       [ 2.42609600e-01,  3.19232748e-01,  3.07116528e-01],
       [ 2.86764853e-01,  3.11725510e-01,  2.68355332e-01],
       [ 3.05683023e-01,  2.30317646e-01,  3.35066384e-01],
       [ 3.38417439e-01,  3.09557691e-01,  2.24144836e-01],
       [ 2.84150945e-01,  3.40961909e-01,  2.44940334e-01],
       [ 3.95755107e-01,  4.10712392e-01,  1.08446370e-01],
       [ 3.44341757e-01,  3.94707596e-01,  1.54368166e-01],
       [ 1.88388095e-01,  2.25536807e-01,  5.01326568e-01],
       [ 2.52989884e-01,  3.26603602e-01,  2.88775834e-01],
       [ 3.03228703e-01,  2.77484070e-01,  2.85612566e-01],
       [ 2.80401205e-01,  2.83216843e-01,  3.02661805e-01],
       [ 3.02089734e-01,  2.99198871e-01,  2.65455043e-01],
       [ 2.61708280e-01,  3.20626676e-01,  2.85212733e-01],
       [ 3.31173347e-01,  2.85391083e-01,  2.52181398e-01],
       [ 2.63381786e-01,  2.94157101e-01,  3.09439182e-01],
       [ 2.58750451e-01,  2.99842269e-01,  3.08660803e-01],
       [ 1.68167549e-01,  3.67479114e-01,  3.51354637e-01],
       [ 4.09730192e-01,  2.16998139e-01,  2.57032119e-01],
       [ 2.72429901e-01,  2.63208933e-01,  3.32862378e-01],
       [ 2.51137464e-01,  3.78842667e-01,  2.45814758e-01],
       [ 2.63298998e-01,  4.30131627e-01,  1.97203251e-01],
       [ 2.46254857e-01,  2.58197548e-01,  3.69544476e-01],
       [ 2.79877694e-01,  2.69351134e-01,  3.17926912e-01],
       [ 2.81579090e-01,  3.23648507e-01,  2.62491910e-01],
       [ 2.37627381e-01,  2.64584430e-01,  3.72606719e-01],
       [ 3.16395924e-01,  3.14938468e-01,  2.38153907e-01],
       [ 2.86135402e-01,  2.75997663e-01,  3.04246998e-01],
       [ 2.81069145e-01,  2.75808324e-01,  3.09724849e-01],
       [ 3.36479674e-01,  2.70191650e-01,  2.62227660e-01],
       [ 3.12565132e-01,  1.96012023e-01,  3.71100971e-01],
       [ 3.48367531e-01,  3.50159220e-01,  1.83265561e-01],
       [ 3.14620406e-01,  3.34338563e-01,  2.23142405e-01],
       [ 3.13525056e-01,  2.90671428e-01,  2.62939670e-01],
       [ 2.71892704e-01,  2.40682786e-01,  3.60064245e-01],
       [ 3.72286354e-01,  2.67975596e-01,  2.34648244e-01],
       [ 2.64145917e-01,  3.32604943e-01,  2.71710982e-01],
       [ 1.99032395e-01,  3.92862436e-01,  2.90267188e-01],
       [ 3.43679245e-01,  2.61562486e-01,  2.64532985e-01],
       [ 3.98388177e-01,  2.31611031e-01,  2.50363654e-01],
       [ 2.81491863e-01,  2.90565377e-01,  2.94040873e-01],
       [ 3.31735108e-01,  2.17056383e-01,  3.24339536e-01],
       [ 2.30166316e-01,  3.22187617e-01,  3.18352513e-01],
       [ 2.91041963e-01,  2.97048757e-01,  2.78097139e-01],
       [ 2.88845352e-01,  2.82145071e-01,  2.95107765e-01],
       [ 2.84124607e-01,  2.85528704e-01,  2.96450957e-01],
       [ 2.72825069e-01,  3.29281404e-01,  2.66005740e-01],
       [ 2.80639102e-01,  3.02283970e-01,  2.83343217e-01],
       [ 3.15828335e-01,  2.94047098e-01,  2.57645391e-01],
       [ 2.69728394e-01,  3.12882654e-01,  2.84249096e-01],
       [ 2.51873051e-01,  3.32799442e-01,  2.84222007e-01],
       [ 2.77136331e-01,  2.62892503e-01,  3.28024959e-01],
       [ 2.68768473e-01,  2.58793853e-01,  3.42034225e-01],
       [ 2.79212834e-01,  2.88379083e-01,  2.98596322e-01],
       [ 2.68677006e-01,  2.65228940e-01,  3.34776348e-01],
       [ 2.70612627e-01,  2.94534936e-01,  3.01328605e-01],
       [ 3.08990204e-01,  2.70970185e-01,  2.86696937e-01],
       [ 2.95214008e-01,  2.84657714e-01,  2.86209948e-01],
       [ 3.14815932e-01,  2.79019998e-01,  2.73072159e-01],
       [ 2.94189573e-01,  2.95721823e-01,  2.76315603e-01],
       [ 3.17931082e-01,  2.79668530e-01,  2.69552855e-01],
       [ 3.19903181e-01,  2.79888000e-01,  2.67531339e-01],
       [ 2.98037252e-01,  2.67889335e-01,  3.00673272e-01],
       [ 2.86576964e-01,  2.74747967e-01,  3.05105925e-01],
       [ 3.87222188e-01,  1.89556733e-01,  3.06182180e-01],
       [ 3.74749011e-01,  2.13509071e-01,  2.88967873e-01],
       [ 3.18720622e-01,  3.07225112e-01,  2.42961991e-01],
       [ 2.84930501e-01,  3.34101513e-01,  2.50074770e-01],
       [ 4.26906147e-01,  1.17041633e-01,  3.67745241e-01],
       [ 2.79255294e-01,  2.71318158e-01,  3.16457263e-01],
       [ 3.23152302e-01,  2.61145192e-01,  2.83434948e-01],
       [ 2.50984160e-01,  3.28679954e-01,  2.88971722e-01],
       [ 2.85901569e-01,  3.19870534e-01,  2.61729637e-01],
       [ 2.55678070e-01,  2.81065728e-01,  3.31885823e-01],
       [ 2.49633603e-01,  2.99046386e-01,  3.19581500e-01],
       [ 6.29715348e-01,  2.51653204e-01,  1.03849990e-01],
       [ 2.32614483e-01,  3.26235187e-01,  3.11556005e-01],
       [ 3.02213320e-01,  3.17624457e-01,  2.48467686e-01],
       [ 3.29023870e-01,  2.68933970e-01,  2.70109853e-01],
       [ 2.92762402e-01,  3.13891120e-01,  2.60617430e-01],
       [ 2.84566513e-01,  2.89454166e-01,  2.92029614e-01],
       [ 2.99281896e-01,  2.78178118e-01,  2.88758219e-01],
       [ 3.08355544e-01,  2.81480493e-01,  2.76693723e-01],
       [ 2.71221215e-01,  2.64387263e-01,  3.32878535e-01],
       [ 2.31884454e-01,  3.84457867e-01,  2.60975422e-01],
       [ 2.81132940e-01,  2.70477309e-01,  3.15367452e-01],
       [ 3.23236283e-01,  2.85025064e-01,  2.59542315e-01],
       [ 1.97654594e-01,  3.50315651e-01,  3.29869193e-01],
       [ 2.19530764e-01,  3.00256737e-01,  3.54153205e-01],
       [ 3.04775343e-01,  4.30897631e-01,  1.61312201e-01],
       [ 3.27188967e-01,  2.73530422e-01,  2.67186089e-01],
       [ 2.59183048e-01,  4.08292609e-01,  2.16004098e-01],
       [ 3.99295475e-01,  2.25445125e-01,  2.56075532e-01],
       [ 4.22811646e-01,  1.86561095e-01,  2.80813015e-01],
       [ 2.02995922e-01,  4.43068072e-01,  2.47744789e-01],
       [ 3.48433363e-01,  2.94574179e-01,  2.29173873e-01],
       [ 2.84329001e-01,  2.24923438e-01,  3.65335008e-01],
       [ 2.35499502e-01,  3.76704405e-01,  2.63451929e-01],
       [ 2.81363111e-01,  2.84916070e-01,  2.99914130e-01],
       [ 2.86038609e-01,  3.02051653e-01,  2.78191251e-01],
       [ 3.26178689e-01,  2.78170242e-01,  2.63534503e-01],
       [ 3.27841991e-01,  2.73426555e-01,  2.66701618e-01],
       [ 2.41573025e-01,  3.64787185e-01,  2.66965499e-01],
       [ 3.64311791e-01,  3.24606638e-01,  1.91230150e-01],
       [ 3.63411536e-01,  2.32125208e-01,  2.78140725e-01],
       [ 2.46697380e-01,  2.85114398e-01,  3.37831978e-01],
       [ 2.94782757e-01,  2.74008652e-01,  2.97520272e-01],
       [ 3.47929692e-01,  2.95124614e-01,  2.29090113e-01],
       [ 2.56355217e-01,  2.93688884e-01,  3.17631480e-01],
       [ 3.02408782e-01,  2.92888488e-01,  2.71171996e-01],
       [ 2.92563979e-01,  3.29980203e-01,  2.46504076e-01],
       [ 3.04714902e-01,  2.59450517e-01,  3.02999006e-01],
       [ 4.22461058e-01,  2.04812728e-01,  2.60611238e-01],
       [ 3.25990803e-01,  3.14065683e-01,  2.30631950e-01],
       [ 2.81315427e-01,  3.16573038e-01,  2.69185860e-01],
       [ 3.14093882e-01,  2.66701154e-01,  2.86212500e-01],
       [ 2.68328120e-01,  2.85576635e-01,  3.12999224e-01],
       [ 2.80055121e-01,  3.27608679e-01,  2.60426096e-01],
       [ 2.96824578e-01,  3.23213721e-01,  2.48472109e-01],
       [ 2.36252012e-01,  3.02459278e-01,  3.31426852e-01],
       [ 2.94443581e-01,  2.97233818e-01,  2.74611487e-01],
       [ 3.44305441e-01,  3.02161924e-01,  2.25787122e-01],
       [ 2.43026515e-01,  3.51137679e-01,  2.77135909e-01],
       [ 2.94930153e-01,  2.81262558e-01,  2.89915488e-01],
       [ 2.83241006e-01,  2.96186975e-01,  2.86675322e-01],
       [ 3.59817660e-01,  1.98405251e-01,  3.19962301e-01],
       [ 2.78363938e-01,  3.22111743e-01,  2.67014488e-01],
       [ 2.89144227e-01,  2.81606345e-01,  2.95356954e-01],
       [ 2.94235124e-01,  2.75824144e-01,  2.96184727e-01],
       [ 2.79609838e-01,  2.81726505e-01,  3.05033693e-01],
       [ 2.88006607e-01,  2.65577207e-01,  3.13434037e-01],
       [ 2.68841940e-01,  3.18613347e-01,  2.79754687e-01],
       [ 2.80525027e-01,  2.94151192e-01,  2.91439290e-01],
       [ 2.86790994e-01,  2.79177688e-01,  3.00253988e-01],
       [ 3.36469120e-01,  1.57152183e-01,  3.99340834e-01],
       [ 3.06395096e-01,  3.24109092e-01,  2.39006444e-01],
       [ 2.86235694e-01,  3.09230657e-01,  2.71194414e-01],
       [ 7.94291605e-01,  4.21087544e-01, -6.94979019e-02],
       [ 2.86440733e-01,  3.58830411e-01,  2.28146501e-01],
       [ 2.91977464e-01,  3.21228418e-01,  2.54740774e-01],
       [ 2.97612279e-01,  2.78159266e-01,  2.90421416e-01],
       [ 2.53832453e-01,  2.72668541e-01,  3.43376133e-01],
       [ 3.18042415e-01,  2.75302490e-01,  2.73773534e-01],
       [ 2.65879710e-01,  2.83092415e-01,  3.18288421e-01],
       [ 3.38752660e-01,  3.12832998e-01,  2.21040761e-01],
       [-1.03848923e-01,  4.59350831e-01,  8.37416291e-01],
       [ 3.60672221e-01,  2.34197254e-01,  2.78265339e-01],
       [ 2.62837087e-01,  3.14544175e-01,  2.89801793e-01],
       [ 2.77579129e-01,  2.95753786e-01,  2.92857636e-01],
       [ 2.90334961e-01,  2.72639924e-01,  3.03464866e-01],
       [ 2.91942705e-01,  2.87175113e-01,  2.86921444e-01],
       [ 2.87689326e-01,  2.97638358e-01,  2.80821539e-01],
       [ 2.83936577e-01,  3.05952024e-01,  2.76542095e-01],
       [ 3.15170877e-01,  2.84004487e-01,  2.67851561e-01],
       [ 3.04881198e-01,  3.12923277e-01,  2.50233825e-01],
       [ 2.74475501e-01,  3.08900665e-01,  2.83203780e-01],
       [ 2.59647641e-01,  3.00720861e-01,  3.06795506e-01],
       [ 2.33478644e-01,  3.95892671e-01,  2.50357002e-01],
       [ 2.39936245e-01,  3.79628309e-01,  2.56492093e-01],
       [ 1.67990912e-01,  2.78125704e-01,  4.55659803e-01],
       [ 3.36897763e-01,  2.88263629e-01,  2.44552255e-01],
       [ 2.26039203e-01,  2.70150747e-01,  3.80772203e-01],
       [ 2.24029853e-01,  2.58276670e-01,  3.98373870e-01],
       [ 1.68368344e-01,  2.77944246e-01,  4.55292976e-01],
       [ 2.49607297e-01,  3.98523752e-01,  2.32245568e-01],
       [ 1.12988848e-01,  3.13824382e-01,  5.02658609e-01],
       [ 3.21275131e-01,  4.34557890e-01,  1.46047280e-01],
       [ 3.46292742e-01,  1.88736193e-01,  3.45106618e-01],
       [ 2.89643707e-01,  3.01651074e-01,  2.75038221e-01],
       [ 3.04555492e-01,  2.50054131e-01,  3.13454066e-01],
       [ 3.11337364e-01,  2.93580636e-01,  2.62179962e-01],
       [ 2.33105876e-01,  3.33139282e-01,  3.04361589e-01],
       [ 2.70427270e-01,  3.08392036e-01,  2.87831421e-01],
       [ 2.56520258e-01,  2.84707577e-01,  3.26972722e-01],
       [ 3.33752865e-01,  1.46968401e-01,  4.18015364e-01],
       [ 1.83616173e-01,  3.58389248e-01,  3.39837815e-01],
       [ 3.34686416e-01,  2.90159226e-01,  2.44680667e-01],
       [ 2.89374041e-01,  2.88359744e-01,  2.88292255e-01],
       [ 2.69609341e-01,  2.92754964e-01,  3.04198765e-01],
       [ 2.86508810e-01,  2.56150893e-01,  3.25453524e-01],
       [ 2.54642267e-01,  2.98212505e-01,  3.14843067e-01],
       [ 2.63699301e-01,  2.82229635e-01,  3.21610068e-01],
       [ 2.57994033e-01,  3.07743264e-01,  3.01560591e-01],
       [ 5.39987952e-01,  4.21635395e-01,  2.32127963e-02],
       [ 3.22589500e-01,  2.05209700e-01,  3.48241349e-01],
       [ 2.67579816e-01,  2.84935537e-01,  3.14483572e-01],
       [ 2.89888875e-01,  3.15035682e-01,  2.62304875e-01],
       [ 3.43987524e-01,  2.58969506e-01,  2.66880910e-01],
       [ 3.27662105e-01,  2.44099332e-01,  2.97358108e-01],
       [ 3.00805510e-01,  2.85500326e-01,  2.79922046e-01],
       [ 3.37074741e-01,  3.30638606e-01,  2.07499338e-01],
       [ 2.74265678e-01,  2.89075425e-01,  3.03042920e-01],
       [ 2.92815790e-01,  2.21085955e-01,  3.60501872e-01],
       [ 2.61416825e-01,  2.58570964e-01,  3.50787467e-01],
       [ 2.98774505e-01,  2.75178907e-01,  2.92329577e-01],
       [ 1.98277477e-01,  4.57261233e-01,  2.43060240e-01],
       [ 3.18463955e-01,  3.03630983e-01,  2.46432605e-01],
       [ 3.06933625e-01,  2.82023463e-01,  2.77503267e-01],
       [ 2.96186849e-01,  2.92978527e-01,  2.77042101e-01],
       [ 2.83890247e-01,  2.96856520e-01,  2.85365909e-01],
       [ 3.10315845e-01,  2.88409894e-01,  2.68072390e-01],
       [ 3.16885155e-01,  2.77184846e-01,  2.72971597e-01],
       [ 2.92595235e-01,  3.05726759e-01,  2.68326768e-01],
       [ 2.89472828e-01,  2.94622147e-01,  2.82000190e-01],
       [ 1.71734102e-01,  4.23378068e-01,  2.97912825e-01],
       [ 2.50047632e-01,  2.76237919e-01,  3.43781739e-01],
       [ 3.23897094e-01,  2.71029018e-01,  2.72663253e-01],
       [ 3.39698122e-01,  1.80734012e-01,  3.62400751e-01],
       [ 2.76379591e-01,  2.89843659e-01,  3.00046189e-01],
       [ 2.88622112e-01,  3.02589581e-01,  2.75139951e-01],
       [ 3.23334997e-01,  2.87694153e-01,  2.56908221e-01],
       [ 2.87081340e-01,  2.96453120e-01,  2.82577726e-01],
       [ 3.08953107e-01,  3.00977040e-01,  2.57426541e-01],
       [ 2.54432282e-01,  2.98228364e-01,  3.15059663e-01],
       [ 3.08233868e-01,  2.73278985e-01,  2.85060185e-01],
       [ 1.91525304e-01,  1.23406682e-01,  7.18772649e-01],
       [ 2.02896434e-01,  3.06640852e-01,  3.68537631e-01],
       [ 3.01923523e-01,  2.34747716e-01,  3.33768850e-01],
       [ 3.09000521e-01,  2.53018854e-01,  3.05713736e-01],
       [ 2.69325925e-01,  2.87701756e-01,  3.09705360e-01],
       [ 3.00958571e-01,  2.89104689e-01,  2.76227104e-01],
       [ 2.91578165e-01,  2.99293605e-01,  2.75410889e-01],
       [ 2.97831845e-01,  2.92548204e-01,  2.75872853e-01],
       [ 2.67975976e-01,  2.73460676e-01,  3.26389260e-01],
       [ 2.55508202e-01,  2.37595718e-01,  3.83879214e-01],
       [ 5.29095912e-01,  1.97847704e-01,  1.99904897e-01],
       [ 2.90212169e-01,  2.72114050e-01,  3.04145860e-01],
       [ 3.13466370e-01,  2.95399844e-01,  2.58516698e-01],
       [ 2.52149739e-01,  3.32611056e-01,  2.84102851e-01],
       [ 2.63398149e-01,  2.38871572e-01,  3.72472522e-01],
       [ 3.93385565e-01,  1.87361161e-01,  3.03565764e-01],
       [ 2.74605854e-01,  2.62509949e-01,  3.31238124e-01],
       [ 4.35168294e-01,  2.17456268e-01,  2.38069384e-01],
       [ 2.96483939e-01,  2.51849414e-01,  3.19752013e-01],
       [ 3.76389870e-01,  2.16645832e-01,  2.84058283e-01],
       [ 1.99473186e-01,  4.32474363e-01,  2.59092643e-01],
       [ 2.28716803e-01,  2.54913531e-01,  3.96370902e-01],
       [ 3.01722793e-01,  4.28710955e-01,  1.65173273e-01],
       [ 4.01484283e-01,  2.68834410e-01,  2.11939800e-01],
       [ 2.14533030e-01,  5.05953215e-01,  1.96334524e-01],
       [ 4.75841371e-01,  1.86636901e-01,  2.43314306e-01],
       [ 3.75418787e-01,  2.96687192e-01,  2.06244340e-01],
       [ 2.71086750e-01,  2.52420507e-01,  3.46837876e-01],
       [ 3.02050503e-01,  2.71368156e-01,  2.93037398e-01],
       [ 2.97678420e-01,  2.49690367e-01,  3.20940051e-01],
       [ 2.83766827e-01,  2.87429796e-01,  2.94884374e-01],
       [ 3.23800737e-01,  2.47880720e-01,  2.96906674e-01],
       [ 3.06934803e-01,  2.54222185e-01,  3.06456781e-01],
       [ 2.70089443e-01,  4.19572116e-01,  1.98181266e-01],
       [ 3.09555669e-01,  2.18231859e-01,  3.45679049e-01],
       [ 2.61779758e-01,  3.44096607e-01,  2.63952323e-01],
       [ 2.82048089e-01,  2.76363620e-01,  3.08109888e-01],
       [ 2.98993235e-01,  3.01844976e-01,  2.65879219e-01],
       [ 2.76969261e-01,  3.12672746e-01,  2.77116044e-01],
       [ 2.65867962e-01,  2.98749862e-01,  3.02101662e-01],
       [ 2.52874386e-01,  1.99065106e-01,  4.41788198e-01],
       [ 5.05407292e-01,  2.22112905e-02,  4.52550876e-01],
       [ 3.50371018e-01,  2.80320284e-01,  2.40662741e-01],
       [ 2.92659689e-01,  3.04293364e-01,  2.69611819e-01],
       [ 3.18092830e-01,  2.67892587e-01,  2.81210563e-01],
       [ 2.58547442e-01,  2.77798417e-01,  3.32203423e-01],
       [ 3.88108186e-01,  2.15358850e-01,  2.75768944e-01],
       [ 2.97873618e-01,  3.24661381e-01,  2.46238268e-01],
       [ 5.12292901e-01,  1.84841147e-01,  2.22779525e-01],
       [ 3.03286093e-01,  2.53255193e-01,  3.11192048e-01],
       [ 2.64301113e-01,  2.08270708e-01,  4.12537971e-01],
       [ 2.48405905e-01,  2.57577106e-01,  3.67633541e-01],
       [ 2.99277409e-01,  2.82939660e-01,  2.83953462e-01],
       [ 2.87743979e-01,  2.89956228e-01,  2.88327473e-01],
       [ 3.12173821e-01,  2.72798891e-01,  2.81789774e-01],
       [ 2.80333390e-01,  2.66936773e-01,  3.20077214e-01],
       [ 2.39629898e-01,  3.26150815e-01,  3.03729890e-01],
       [ 3.02918251e-01,  2.58584010e-01,  3.05733736e-01],
       [ 2.81061118e-01,  2.64862126e-01,  3.21579190e-01],
       [ 2.74238877e-01,  2.95123916e-01,  2.96938175e-01],
       [ 2.73414820e-01,  2.95914712e-01,  2.97002918e-01],
       [ 3.01017885e-01,  3.46874519e-01,  2.24704851e-01],
       [ 2.64973381e-01,  2.81226374e-01,  3.21278975e-01],
       [ 2.89713641e-01,  2.88073991e-01,  2.88239184e-01],
       [ 2.76962167e-01,  2.62458208e-01,  3.28702834e-01],
       [ 2.99870894e-01,  2.82442795e-01,  2.83873845e-01],
       [ 2.89236824e-01,  2.93256730e-01,  2.83572846e-01],
       [ 3.43571094e-01,  2.40841180e-01,  2.86191683e-01],
       [ 2.95433761e-01,  3.00900994e-01,  2.70156462e-01],
       [ 3.13556893e-01,  2.75335150e-01,  2.77923209e-01],
       [ 2.69000470e-01,  2.93594222e-01,  3.03989744e-01],
       [ 2.66898889e-01,  3.21290747e-01,  2.79243031e-01],
       [ 3.24140192e-01,  2.52121295e-01,  2.92015620e-01],
       [ 4.61850009e-01,  2.23655421e-01,  2.14009599e-01],
       [ 1.62892062e-01,  3.63915193e-01,  3.62032037e-01],
       [ 3.10897758e-01,  3.03346370e-01,  2.53466474e-01],
       [ 2.98780337e-01,  2.71586401e-01,  2.96046933e-01],
       [ 2.93585554e-01,  2.87905671e-01,  2.84570165e-01],
       [ 2.88221255e-01,  2.89116501e-01,  2.88687995e-01],
       [ 2.66043181e-01,  2.92481253e-01,  3.08289748e-01],
       [ 2.27084423e-01,  2.97924668e-01,  3.47319602e-01],
       [ 5.86238918e-01,  3.52013169e-01,  4.65079494e-02],
       [-9.84365982e-02,  5.23110141e-01,  7.09941055e-01],
       [ 2.87027499e-01,  3.08547480e-01,  2.71063081e-01],
       [ 3.01994022e-01,  2.61631763e-01,  3.03372870e-01],
       [ 1.51199964e-01,  4.29945667e-01,  3.18323362e-01],
       [ 2.45424782e-01,  2.72421232e-01,  3.53659338e-01],
       [ 2.31701045e-01,  2.54344529e-01,  3.93107390e-01],
       [ 2.41258510e-01,  3.32877087e-01,  2.95558350e-01],
       [ 4.09873571e-01,  2.48986052e-01,  2.24550409e-01],
       [ 2.45685578e-01,  3.17283146e-01,  3.05608642e-01],
       [ 2.40431766e-01,  2.82010265e-01,  3.48738736e-01],
       [ 2.34037563e-01,  3.37829646e-01,  2.98907107e-01],
       [ 3.79244457e-01,  2.33849375e-01,  2.63114572e-01],
       [ 1.67591377e-01,  3.20847338e-01,  4.01746927e-01],
       [ 2.33161740e-01,  3.63881446e-01,  2.76624493e-01],
       [ 1.75890130e-01,  3.17545356e-01,  3.93459554e-01],
       [-6.32649856e-02,  5.34127055e-01,  6.02706310e-01],
       [ 2.52352553e-01,  2.63404706e-01,  3.55844046e-01],
       [ 2.20824202e-01,  3.33219205e-01,  3.18417534e-01],
       [ 2.96141735e-01,  2.73828588e-01,  2.96345125e-01],
       [ 3.01548403e-01,  2.72534555e-01,  2.92322978e-01],
       [ 2.53718746e-01,  3.02993264e-01,  3.10977536e-01],
       [ 3.04099386e-01,  3.09052919e-01,  2.54450968e-01],
       [ 3.16928924e-01,  2.60636919e-01,  2.89831235e-01],
       [ 3.10958175e-01,  2.78002269e-01,  2.77697634e-01],
       [ 3.00794303e-01,  3.41598044e-01,  2.29220125e-01],
       [ 3.98914217e-01,  3.80135384e-01,  1.26254594e-01],
       [ 2.77008838e-01,  2.98846181e-01,  2.90380324e-01],
       [ 3.00249201e-01,  2.86679847e-01,  2.79292030e-01],
       [ 2.86497874e-01,  2.88467075e-01,  2.91069562e-01],
       [ 2.88426271e-01,  3.20898550e-01,  2.58391620e-01],
       [ 3.23322129e-01,  2.85240326e-01,  2.59259323e-01],
       [ 2.32952853e-01,  3.93944300e-01,  2.52401132e-01],
       [ 3.01844058e-01,  2.60568343e-01,  3.04667880e-01],
       [ 2.76166426e-01,  2.80927163e-01,  3.09494406e-01],
       [ 3.37555601e-01,  2.55857773e-01,  2.75750064e-01],
       [ 2.75403190e-01,  2.47832116e-01,  3.47350882e-01],
       [ 3.04690553e-01,  1.78239997e-01,  4.05217596e-01],
       [ 2.78843468e-01,  3.11448489e-01,  2.76396487e-01],
       [ 5.50644233e-01,  5.31114793e-01, -3.92465390e-02],
       [ 2.45266906e-01,  3.78577438e-01,  2.51901752e-01],
       [ 3.47362348e-01,  2.41718842e-01,  2.81855843e-01],
       [ 2.93598736e-01,  2.97599729e-01,  2.75076992e-01],
       [ 3.10262153e-01,  2.83669656e-01,  2.72738452e-01],
       [ 2.78850023e-01,  2.70435355e-01,  3.17847702e-01],
       [ 2.50442979e-01,  3.23205910e-01,  2.94701780e-01],
       [ 3.27545490e-01,  2.68095130e-01,  2.72289438e-01],
       [ 3.06505771e-01,  2.91071184e-01,  2.69061584e-01],
       [ 3.24600463e-01,  2.73869769e-01,  2.69189239e-01],
       [ 3.00067336e-01,  2.73073439e-01,  2.93225658e-01],
       [ 2.94835115e-01,  2.85180991e-01,  2.86058660e-01],
       [ 1.65622334e-01,  4.22756137e-01,  3.05895186e-01],
       [ 2.94891527e-01,  2.83288606e-01,  2.87904376e-01],
       [ 2.82409728e-01,  2.97966033e-01,  2.85765713e-01],
       [ 2.94463127e-01,  2.88885591e-01,  2.82736278e-01],
       [ 2.90025955e-01,  2.94067095e-01,  2.81997723e-01],
       [ 2.71425872e-01,  2.75796768e-01,  3.20055512e-01],
       [ 2.60459726e-01,  3.07913849e-01,  2.98748659e-01],
       [ 2.80120149e-01,  3.26310190e-01,  2.61520459e-01],
       [ 2.50510836e-01,  2.83529151e-01,  3.35130102e-01],
       [ 2.88994846e-01,  3.31461218e-01,  2.48542041e-01],
       [ 2.71410890e-01,  2.82421776e-01,  3.12996442e-01],
       [ 6.80921199e-01,  1.67888046e-01,  1.59849190e-01],
       [ 1.98005650e-01,  3.00538069e-01,  3.82096408e-01],
       [ 3.31102886e-01,  2.33135359e-01,  3.06268161e-01],
       [ 2.94337336e-01,  3.06698944e-01,  2.65752759e-01],
       [ 2.95811931e-01,  2.83480737e-01,  2.86803243e-01],
       [ 2.85843243e-01,  2.97488939e-01,  2.82797353e-01],
       [ 2.88685948e-01,  2.65184937e-01,  3.13150302e-01],
       [ 3.13333266e-01,  3.03621291e-01,  2.51015812e-01],
       [ 2.01755524e-01,  2.22220467e-01,  4.83908989e-01],
       [-4.64996483e-02,  4.45030243e-01,  6.79229534e-01],
       [ 1.96304596e-01,  3.29779834e-01,  3.52153937e-01],
       [ 5.13067873e-01,  3.26237019e-01,  9.84365360e-02],
       [ 2.81634807e-01,  2.91067106e-01,  2.93390625e-01],
       [ 3.17849759e-01,  2.19458828e-01,  3.35459117e-01],
       [ 3.04876664e-01,  4.12154236e-01,  1.73414830e-01],
       [ 3.72100402e-01,  2.85968890e-01,  2.18200215e-01],
       [ 2.59918295e-01,  3.63153975e-01,  2.49745729e-01],
       [ 3.43939878e-01,  2.41342885e-01,  2.85319453e-01],
       [ 3.04456395e-01,  3.05030070e-01,  2.57809900e-01],
       [ 2.28667159e-01,  3.51408714e-01,  2.92452031e-01],
       [ 2.90212322e-01,  3.27216766e-01,  2.51101973e-01],
       [ 2.68007331e-01,  2.52418445e-01,  3.50386193e-01],
       [ 4.14322816e-01,  2.82390492e-01,  1.90895128e-01],
       [ 3.14723659e-01,  2.08181372e-01,  3.52799048e-01],
       [ 3.02835909e-01,  1.96486661e-01,  3.81510460e-01],
       [ 3.62163004e-01,  3.82912403e-01,  1.49412117e-01],
       [ 3.51580688e-01,  2.85754107e-01,  2.34624526e-01],
       [ 2.55033736e-01,  3.89355209e-01,  2.33866964e-01],
       [ 3.56067100e-01,  2.71769587e-01,  2.44063137e-01],
       [ 4.07584413e-01,  6.32633376e-02,  4.76193949e-01],
       [ 4.63584481e-01,  2.56986487e-01,  1.81612996e-01],
       [ 2.28221993e-01,  3.40514936e-01,  3.02929165e-01],
       [ 2.67144886e-01,  3.16071692e-01,  2.83879214e-01],
       [ 2.98441879e-01,  2.88696558e-01,  2.79049789e-01],
       [ 2.58671048e-01,  2.89089976e-01,  3.19885470e-01],
       [ 3.01204129e-01,  2.64678648e-01,  3.00906310e-01],
       [ 2.87534416e-01,  2.96083842e-01,  2.82488944e-01],
       [ 3.13265546e-01,  2.41853580e-01,  3.13870660e-01],
       [ 4.29522340e-01,  9.88994365e-02,  3.92717507e-01],
       [ 3.78869337e-01,  3.11246621e-01,  1.91385517e-01],
       [ 2.42406341e-01,  3.10891254e-01,  3.15631209e-01],
       [ 3.05470593e-01,  2.72604742e-01,  2.88417889e-01],
       [ 2.69882058e-01,  2.81755969e-01,  3.15350122e-01],
       [ 3.42091303e-01,  2.84342082e-01,  2.43807323e-01],
       [ 2.41402503e-01,  3.29097877e-01,  2.98956767e-01],
       [ 2.64914026e-01,  2.94602523e-01,  3.07328997e-01],
       [ 2.73501425e-01,  3.27741153e-01,  2.66718182e-01],
       [ 2.86335576e-01,  3.23633381e-01,  2.57934847e-01],
       [ 2.48992994e-01,  3.31071325e-01,  2.88874102e-01],
       [ 4.64023550e-01,  3.43802852e-02,  4.69592570e-01],
       [ 3.90614833e-01,  2.20570985e-01,  2.68071832e-01],
       [ 2.83696446e-01,  3.48165149e-01,  2.39335301e-01],
       [ 2.99356361e-01,  2.81320073e-01,  2.85503315e-01],
       [ 3.32052518e-01,  3.05779709e-01,  2.32765093e-01],
       [ 2.68114997e-01,  3.16069626e-01,  2.82884874e-01],
       [ 3.59968802e-01,  3.28810175e-01,  1.91118776e-01],
       [ 1.93401904e-01,  3.49156911e-01,  3.36318172e-01],
       [ 2.97059778e-01,  2.42897986e-01,  3.29367943e-01],
       [ 2.80712490e-01,  2.64850792e-01,  3.21966452e-01],
       [ 2.93360140e-01,  2.98296798e-01,  2.74638222e-01],
       [ 2.96038421e-01,  2.70072632e-01,  3.00379446e-01],
       [ 3.16786886e-01,  2.36442241e-01,  3.16502133e-01],
       [ 2.66370094e-01,  3.21616895e-01,  2.79480466e-01],
       [ 2.85386797e-01,  2.88559699e-01,  2.92098432e-01],
       [ 3.05628899e-01,  2.67431346e-01,  2.93625764e-01],
       [ 3.03516744e-01,  2.75910302e-01,  2.86932763e-01],
       [ 3.28258069e-01,  2.45746114e-01,  2.95001082e-01],
       [ 2.92145829e-01,  3.04110994e-01,  2.70278235e-01],
       [ 3.03381419e-01,  2.52943342e-01,  3.11439832e-01],
       [ 3.20305097e-01,  2.64572285e-01,  2.82548366e-01],
       [ 2.44157356e-01,  3.03690218e-01,  3.20986727e-01],
       [ 3.19891499e-01,  2.36056405e-01,  3.13855959e-01],
       [ 4.19146033e-01,  7.80678096e-02,  4.36991429e-01],
       [ 3.76185244e-01,  2.06676828e-01,  2.95526566e-01],
       [ 2.47965703e-01,  3.12043259e-01,  3.08252163e-01],
       [ 2.87854327e-01,  2.65071399e-01,  3.14143551e-01],
       [ 3.05231885e-01,  2.78860287e-01,  2.82289264e-01],
       [ 2.94716713e-01,  2.88297001e-01,  2.83070624e-01],
       [ 3.08304716e-01,  2.96364966e-01,  2.62340395e-01],
       [ 2.45311381e-01,  2.89518836e-01,  3.34644022e-01],
       [ 3.66253000e-01,  2.13821407e-01,  2.95974562e-01],
       [ 1.22979623e-01,  3.09995691e-01,  4.89350870e-01],
       [ 4.67943519e-01,  2.44448772e-01,  1.90361102e-01],
       [ 3.58894585e-01,  3.05127674e-01,  2.11576236e-01],
       [ 2.96890925e-01,  3.07985219e-01,  2.62139589e-01],
       [ 3.24257058e-01,  3.10008761e-01,  2.35670072e-01],
       [ 3.82428818e-01,  2.52754092e-01,  2.41410071e-01],
       [ 2.28684955e-01,  3.17262364e-01,  3.25025629e-01],
       [ 2.70052579e-01,  2.34935853e-01,  3.69424238e-01],
       [ 2.65421113e-01,  2.47937697e-01,  3.58797972e-01],
       [ 2.42660318e-01,  3.09095283e-01,  3.17160062e-01],
       [ 2.89688034e-01,  2.25995110e-01,  3.57839738e-01],
       [ 3.06572259e-01,  3.38965592e-01,  2.26295875e-01],
       [ 2.39756781e-01,  3.14475274e-01,  3.15034865e-01],
       [ 3.30601049e-01,  2.35152570e-01,  3.04475849e-01],
       [ 2.76586879e-01,  3.87037281e-01,  2.15408925e-01],
       [ 3.37520368e-01,  3.20889954e-01,  2.15204865e-01],
       [ 3.18084692e-01,  4.52781135e-01,  1.37478208e-01],
       [ 3.09753263e-01,  2.42114005e-01,  3.17113203e-01],
       [ 4.29694990e-01,  1.80548434e-01,  2.82541746e-01],
       [ 2.61421196e-01,  3.41393578e-01,  2.66669779e-01],
       [ 2.72656359e-01,  3.35144627e-01,  2.60974711e-01],
       [ 3.54942218e-01,  2.00642444e-01,  3.21793487e-01],
       [ 3.04221661e-01,  3.16803656e-01,  2.47367477e-01],
       [ 3.05760581e-01,  2.79687328e-01,  2.80951794e-01],
       [ 2.67293863e-01,  2.89802112e-01,  3.09709066e-01],
       [ 2.92174860e-01,  3.14270830e-01,  2.60827914e-01],
       [ 3.00302081e-01,  2.50448535e-01,  3.17366480e-01],
       [ 2.52498361e-01,  3.15421859e-01,  2.99965544e-01],
       [ 3.21891559e-01,  3.26987021e-01,  2.23070452e-01],
       [ 3.22422265e-01,  3.60713919e-01,  1.95711784e-01],
       [ 3.29092597e-01,  2.24101864e-01,  3.18603580e-01],
       [ 2.49656910e-01,  3.13481756e-01,  3.04964166e-01],
       [ 3.00304389e-01,  2.95637887e-01,  2.70527283e-01],
       [ 3.13858821e-01,  2.65560148e-01,  2.87618483e-01],
       [ 2.57700960e-01,  2.99405129e-01,  3.10251520e-01],
       [ 3.25824476e-01,  2.66503265e-01,  2.75466944e-01],
       [ 3.26883994e-01,  2.55959191e-01,  2.85378712e-01],
       [ 3.39516611e-01,  2.98548783e-01,  2.32950620e-01],
       [ 3.43100992e-01,  3.40549132e-01,  1.94773980e-01],
       [ 4.70799178e-01,  2.16438834e-01,  2.15501431e-01],
       [ 2.26730445e-01,  3.01161937e-01,  3.44231942e-01],
       [ 2.64859809e-01,  3.29205447e-01,  2.74055260e-01],
       [ 2.90862172e-01,  2.95142298e-01,  2.80124604e-01],
       [ 2.89259296e-01,  3.11296345e-01,  2.66344411e-01],
       [ 3.37269021e-01,  2.08931066e-01,  3.28696440e-01],
       [ 2.38601348e-01,  2.22092124e-01,  4.27634711e-01],
       [ 3.31051185e-01,  2.68638550e-01,  2.68583703e-01],
       [ 2.80040738e-01,  3.19012620e-01,  2.68195593e-01],
       [ 2.91799038e-01,  2.91869546e-01,  2.82408119e-01],
       [ 2.96778219e-01,  2.74613311e-01,  2.94895430e-01],
       [ 2.02154165e-01,  3.58264388e-01,  3.16862033e-01],
       [ 2.59555009e-01,  2.76774219e-01,  3.32187500e-01],
       [ 2.95203392e-01,  2.91311824e-01,  2.79624052e-01],
       [ 2.75878675e-01,  2.82713513e-01,  3.07926559e-01],
       [ 2.98017769e-01,  2.84747884e-01,  2.83373034e-01],
       [ 2.97499770e-01,  3.04996052e-01,  2.64339999e-01],
       [ 2.81117988e-01,  2.75168969e-01,  3.10352510e-01],
       [ 3.15132662e-01,  2.79561269e-01,  2.72242753e-01],
       [ 2.64706982e-01,  2.87156085e-01,  3.15273461e-01],
       [ 3.12974165e-01,  2.59971686e-01,  2.94330744e-01],
       [ 2.72552489e-01,  2.90920218e-01,  3.02958720e-01],
       [ 2.94545104e-01,  3.77902350e-01,  2.06247659e-01],
       [ 2.97830843e-01,  3.83560674e-01,  1.99244924e-01],
       [ 3.34683047e-01,  2.44626241e-01,  2.90221040e-01],
       [ 2.66814955e-01,  2.98582400e-01,  3.01263791e-01],
       [ 2.81287130e-01,  2.93745080e-01,  2.91067677e-01],
       [ 3.24468753e-01,  2.79884232e-01,  2.63399564e-01],
       [ 2.85142581e-01,  2.84003730e-01,  2.96968355e-01],
       [ 3.12199646e-01,  2.26574823e-01,  3.32724415e-01],
       [ 3.71710300e-01,  2.60094163e-01,  2.42670525e-01],
       [ 4.20725474e-01,  1.22210118e-01,  3.65758099e-01],
       [ 4.66251802e-01,  1.92400966e-01,  2.43364502e-01],
       [ 3.39622907e-01,  2.51599046e-01,  2.78323902e-01],
       [ 1.58727893e-01,  3.65423360e-01,  3.66300984e-01],
       [ 2.37357379e-01,  3.01392945e-01,  3.31252071e-01],
       [ 3.08397950e-01,  2.66174323e-01,  2.92238927e-01],
       [ 2.02860313e-01,  3.47738170e-01,  3.25932112e-01],
       [ 3.54378378e-01,  2.81175911e-01,  2.36576701e-01],
       [ 3.19300553e-01,  2.13548657e-01,  3.41210595e-01],
       [ 3.34048714e-01,  2.94168866e-01,  2.41529804e-01],
       [ 3.39346145e-01,  2.67727490e-01,  2.62155526e-01],
       [ 3.13762458e-01,  2.82853753e-01,  2.70276115e-01],
       [ 2.52608984e-01,  2.84911727e-01,  3.31203495e-01],
       [ 3.02335587e-01,  3.45872959e-01,  2.24357264e-01],
       [ 3.26558993e-01,  2.56788165e-01,  2.84810705e-01],
       [ 3.13192898e-01,  2.92175899e-01,  2.61811617e-01],
       [ 1.78981699e-01,  3.13527211e-01,  3.93666720e-01],
       [ 3.53406300e-01,  3.51387909e-01,  1.78516364e-01],
       [ 2.82270880e-01,  3.71065705e-01,  2.22334032e-01],
       [ 1.99471261e-01,  3.45905848e-01,  3.31883612e-01],
       [ 1.93781447e-01,  1.61020684e-01,  6.16673802e-01],
       [ 2.36680167e-01,  3.20965131e-01,  3.12087128e-01],
       [ 2.88554927e-01,  2.45427884e-01,  3.35554948e-01],
       [ 3.09615277e-01,  3.00921044e-01,  2.56872920e-01],
       [ 2.98734257e-01,  3.23342427e-01,  2.46603906e-01],
       [ 2.75204472e-01,  2.95824238e-01,  2.95235326e-01],
       [ 2.53758575e-01,  2.93387942e-01,  3.20846955e-01],
       [ 3.08801474e-01,  3.06674921e-01,  2.52322158e-01],
       [ 2.87337040e-01,  2.70094028e-01,  3.09261524e-01],
       [ 2.94060700e-01,  2.83367833e-01,  2.88646382e-01],
       [ 2.83528546e-01,  3.03419417e-01,  2.79363664e-01],
       [ 2.61564410e-01,  3.00647687e-01,  3.04798253e-01],
       [ 3.13915365e-01,  2.81542099e-01,  2.71420914e-01],
       [ 2.60141216e-01,  2.74863405e-01,  3.33635810e-01],
       [ 2.60005591e-01,  2.98395907e-01,  3.08765998e-01],
       [ 2.72733248e-01,  2.81479248e-01,  3.12571895e-01],
       [ 2.85285265e-01,  3.06039379e-01,  2.75130550e-01],
       [ 2.76465698e-01,  2.91002561e-01,  2.98779308e-01],
       [ 3.03573736e-01,  2.37536137e-01,  3.28750734e-01],
       [ 2.96336677e-01,  2.69378945e-01,  3.00810392e-01],
       [ 2.89278927e-01,  2.94517427e-01,  2.82294184e-01],
       [ 2.51412549e-01,  3.34887291e-01,  2.82799212e-01],
       [ 3.24500273e-01,  3.18188415e-01,  2.28334146e-01],
       [ 2.33748685e-01,  2.97083458e-01,  3.40139787e-01],
       [ 2.59899904e-01,  2.81048308e-01,  3.27121096e-01],
       [ 1.55290928e-01,  3.69439428e-01,  3.67101705e-01],
       [ 3.22123559e-02,  4.89163078e-01,  4.49278754e-01],
       [ 2.86129062e-01,  2.40423713e-01,  3.44139841e-01],
       [ 3.08792903e-01,  2.83647283e-01,  2.74140303e-01],
       [ 3.07324283e-01,  2.89085308e-01,  2.70212061e-01],
       [ 2.34634735e-01,  3.49916502e-01,  2.87224495e-01],
       [ 6.25224476e-01,  1.03852102e-01,  2.53840446e-01],
       [ 2.47111508e-01,  2.93961547e-01,  3.27790706e-01],
       [ 2.75901994e-01,  2.83079850e-01,  3.07519300e-01],
       [ 2.91649999e-01,  3.06340093e-01,  2.68659823e-01],
       [ 2.57975110e-01,  3.12208002e-01,  2.97199449e-01],
       [ 2.97720915e-01,  3.09823435e-01,  2.59666777e-01],
       [ 2.60874115e-01,  2.99004858e-01,  3.07204558e-01],
       [ 2.89797149e-01,  3.04102100e-01,  2.72558146e-01],
       [ 2.54931082e-01,  3.04482835e-01,  3.08140102e-01],
       [ 3.14376243e-01,  3.05899626e-01,  2.48006464e-01],
       [ 3.00031791e-01,  2.53712357e-01,  3.14004631e-01],
       [ 1.37223612e-01,  5.69168910e-01,  2.43344289e-01],
       [ 3.34151903e-01,  2.91508827e-01,  2.43889001e-01],
       [ 3.03415153e-01,  3.07805623e-01,  2.56220201e-01],
       [ 3.20540552e-01,  2.79945679e-01,  2.66893809e-01],
       [ 2.79928819e-01,  3.09073503e-01,  2.77556494e-01],
       [ 2.77124444e-01,  2.76596452e-01,  3.13060900e-01],
       [ 2.92450230e-01,  2.85567018e-01,  2.88029224e-01],
       [ 3.37976640e-01,  2.93320626e-01,  2.38975342e-01],
       [ 3.04277700e-01,  3.25365807e-01,  2.39815768e-01],
       [ 4.98483623e-01,  2.61618081e-01,  1.57331145e-01],
       [ 4.77444841e-01,  1.22766165e-01,  3.18864409e-01],
       [ 2.63191078e-01,  3.27421781e-01,  2.77382224e-01],
       [ 2.67920627e-01,  3.54773305e-01,  2.48836395e-01],
       [ 3.17033386e-01,  2.63817843e-01,  2.86408856e-01],
       [ 4.09415969e-01,  2.33831426e-01,  2.39823249e-01],
       [ 3.12563761e-01,  2.66939090e-01,  2.87426565e-01],
       [ 3.13031243e-01,  3.49269701e-01,  2.12392376e-01],
       [ 3.13100381e-01,  2.37459676e-01,  3.19041098e-01],
       [ 3.43946431e-01,  2.28107104e-01,  2.99872941e-01],
       [ 3.49304438e-01,  2.83422773e-01,  2.38648133e-01],
       [ 3.43922116e-01,  2.67516362e-01,  2.58399189e-01],
       [ 3.30579236e-01,  2.95289823e-01,  2.43474755e-01],
       [ 3.06247015e-01,  3.34915994e-01,  2.29946167e-01],
       [ 3.31486261e-01,  3.32459405e-01,  2.10550776e-01],
       [ 2.18170677e-01,  3.00809032e-01,  3.55259148e-01],
       [ 3.00320444e-01,  3.56853209e-01,  2.17339337e-01],
       [ 2.44601285e-01,  3.91814765e-01,  2.42233999e-01],
       [ 3.47943282e-01,  2.13275091e-01,  3.13233446e-01],
       [ 4.19962070e-01,  2.12178933e-01,  2.54520582e-01],
       [ 2.16209908e-01,  3.05628319e-01,  3.52446640e-01],
       [ 7.24209602e-01,  5.43884578e-02,  2.70500747e-01],
       [ 2.92727100e-01,  3.39040418e-01,  2.38622083e-01],
       [ 3.69681947e-01,  2.57014833e-01,  2.47306610e-01],
       [ 1.67441296e-01,  3.69449985e-01,  3.50422558e-01],
       [ 2.30986077e-01,  3.33044831e-01,  3.06847158e-01],
       [ 2.86807625e-01,  2.70165095e-01,  3.09736158e-01],
       [ 2.71881118e-01,  3.31899257e-01,  2.64604259e-01],
       [ 2.57223558e-01,  3.01074671e-01,  3.09075854e-01],
       [ 2.77452801e-01,  2.86946531e-01,  3.01888878e-01],
       [ 2.94783246e-01,  2.82159373e-01,  2.89152055e-01],
       [ 2.97660279e-01,  2.81104552e-01,  2.87381561e-01],
       [ 3.12536454e-01,  2.75187974e-01,  2.79031667e-01],
       [ 2.67091157e-01,  3.38941391e-01,  2.63140573e-01],
       [ 3.27335779e-01,  1.76949863e-01,  3.80891231e-01],
       [ 3.37667125e-01,  3.10213512e-01,  2.24194222e-01],
       [ 3.13951945e-01,  2.54319077e-01,  2.99427605e-01],
       [ 3.00599335e-01,  2.63323259e-01,  3.02958607e-01],
       [ 2.89405352e-01,  2.87401455e-01,  2.89220722e-01],
       [ 2.77014871e-01,  2.89124446e-01,  3.00117345e-01],
       [ 2.90687204e-01,  2.72122390e-01,  3.03650658e-01],
       [ 2.93938508e-01,  2.54752695e-01,  3.19156516e-01],
       [ 2.71160299e-01,  3.34761298e-01,  2.62783217e-01],
       [ 3.05937645e-01,  2.98380636e-01,  2.62633344e-01],
       [ 1.95576198e-01,  2.19805673e-01,  4.98363690e-01],
       [ 4.19045613e-03,  4.75461378e-01,  5.17057545e-01],
       [ 2.38525070e-01,  2.38106460e-01,  4.05356397e-01],
       [ 2.78073555e-01,  3.41484602e-01,  2.50246342e-01],
       [ 3.11106842e-01,  2.69007883e-01,  2.86684340e-01],
       [ 3.23420650e-01,  2.72893917e-01,  2.71233474e-01],
       [-3.83438942e-02,  4.30933469e-01,  6.78886258e-01],
       [ 2.64595985e-01,  3.13559150e-01,  2.88907765e-01],
       [ 2.74465967e-01,  2.77515228e-01,  3.14922893e-01],
       [ 2.92629357e-01,  2.77760820e-01,  2.95795819e-01],
       [ 2.62436060e-01,  3.31602279e-01,  2.74351996e-01],
       [ 3.02531455e-01,  2.79348164e-01,  2.84403145e-01],
       [ 2.80552670e-01,  2.86853028e-01,  2.98767915e-01],
       [ 2.85057645e-01,  2.91258104e-01,  2.89727725e-01],
       [ 2.87065172e-01,  2.87896666e-01,  2.91071308e-01],
       [ 2.81265924e-01,  2.90369781e-01,  2.94468792e-01],
       [ 2.71934877e-01,  3.30174536e-01,  2.66087898e-01],
       [ 3.54240795e-01,  3.83389095e-02,  6.02218586e-01],
       [ 3.18548998e-01,  2.92944124e-01,  2.56230098e-01],
       [ 2.63073826e-01,  2.85973322e-01,  3.18311287e-01],
       [ 3.04395973e-01,  3.00757254e-01,  2.61835674e-01],
       [ 2.87943786e-01,  2.81007569e-01,  2.97188178e-01],
       [ 2.66892096e-01,  3.15359923e-01,  2.84812974e-01],
       [ 2.73660283e-01,  2.58307806e-01,  3.37071746e-01],
       [ 3.02186513e-01,  2.81309368e-01,  2.82764811e-01],
       [ 2.81967423e-01,  2.15367649e-01,  3.80575088e-01],
       [ 4.22966772e-01,  2.02484660e-01,  2.62779344e-01],
       [ 3.41245897e-01,  1.82064519e-01,  3.59005333e-01],
       [ 3.70861842e-01,  2.88174802e-01,  2.17176030e-01],
       [ 3.01843604e-01,  2.80985898e-01,  2.83421143e-01],
       [ 2.28568458e-01,  3.32099213e-01,  3.10509423e-01],
       [ 3.01363287e-01,  2.42500639e-01,  3.25300505e-01],
       [ 2.80940772e-01,  3.02715543e-01,  2.82623279e-01],
       [ 2.95259620e-01,  3.00772855e-01,  2.70444863e-01],
       [ 2.38177435e-01,  3.16371513e-01,  3.14936031e-01],
       [ 2.19965219e-01,  3.27896734e-01,  3.24669603e-01],
       [ 2.90967722e-01,  2.52630381e-01,  3.24674999e-01],
       [ 3.51161466e-01,  2.24420989e-01,  2.97423931e-01],
       [ 2.73092042e-01,  2.68477714e-01,  3.26238441e-01],
       [ 2.65430201e-01,  2.92663259e-01,  3.08762501e-01],
       [ 3.38158956e-01,  2.43269745e-01,  2.88489641e-01],
       [ 2.25775795e-01,  2.95390612e-01,  3.51726334e-01],
       [ 2.41189932e-01,  3.00633875e-01,  3.27579065e-01],
       [ 3.09346096e-01,  2.17845871e-01,  3.46382801e-01],
       [ 4.63995904e-01,  2.05386754e-01,  2.31110540e-01],
       [ 2.17971881e-01,  2.93886634e-01,  3.63266356e-01],
       [ 3.60395204e-01,  1.84540220e-01,  3.36723548e-01],
       [ 2.56244632e-01,  1.77757151e-01,  4.71082592e-01],
       [ 3.25659289e-01,  3.46773473e-01,  2.03841938e-01],
       [ 5.46989272e-01,  7.28611818e-02,  3.39026476e-01],
       [ 8.05477582e-01,  6.57651985e-01, -1.91181927e-01],
       [ 2.57584234e-01,  2.83246244e-01,  3.27348846e-01],
       [ 2.91881602e-01,  2.61785919e-01,  3.13526620e-01],
       [ 3.32730240e-01,  2.74763683e-01,  2.61036050e-01],
       [ 3.33120332e-01,  2.68413038e-01,  2.66961349e-01],
       [ 2.81446949e-01,  3.10829670e-01,  2.74395329e-01],
       [ 2.95818142e-01,  2.95398478e-01,  2.75052773e-01],
       [ 2.85796994e-01,  2.92042079e-01,  2.88203515e-01],
       [ 2.81900928e-01,  3.24228250e-01,  2.61659991e-01],
       [ 4.49067063e-01,  2.23153057e-01,  2.22827773e-01],
       [ 2.55140927e-01,  3.33918031e-01,  2.79774786e-01],
       [ 2.61923871e-01,  3.10977596e-01,  2.94199883e-01],
       [ 2.71667080e-01,  2.93101463e-01,  3.01670452e-01],
       [ 2.91757774e-01,  2.77216699e-01,  2.97236311e-01],
       [ 3.15034687e-01,  2.64928273e-01,  2.87153518e-01],
       [ 2.80952164e-01,  3.00684018e-01,  2.84580258e-01],
       [ 2.71327164e-01,  3.53453959e-01,  2.46643718e-01],
       [ 3.99248458e-01,  2.43848305e-01,  2.37357656e-01],
       [ 2.53092120e-01,  3.49551258e-01,  2.68038009e-01],
       [ 3.18978738e-01,  2.66749460e-01,  2.81551400e-01],
       [ 2.84777053e-01,  2.88507373e-01,  2.92768673e-01],
       [ 3.20511877e-01,  2.26545455e-01,  3.24261245e-01],
       [ 2.05973199e-01,  3.48419607e-01,  3.21495693e-01],
       [ 3.02242126e-01,  2.62771430e-01,  3.01903206e-01],
       [ 2.80790969e-01,  3.22574519e-01,  2.64224572e-01],
       [ 2.41718881e-01,  3.12929347e-01,  3.14359732e-01],
       [ 3.37838677e-01,  3.00358979e-01,  2.32729027e-01],
       [ 3.06305660e-01,  2.71604126e-01,  2.88636952e-01],
       [ 2.85967702e-01,  2.87715818e-01,  2.92360793e-01],
       [ 2.77865896e-01,  3.01692509e-01,  2.86718197e-01],
       [ 2.77614831e-01,  2.78446194e-01,  3.10575996e-01],
       [ 3.55980601e-01,  2.36749278e-01,  2.79590849e-01],
       [ 3.35299359e-01,  2.82287267e-01,  2.51542462e-01],
       [ 2.66921364e-01,  3.46141216e-01,  2.57082262e-01],
       [ 3.36984565e-01,  3.11596192e-01,  2.23560276e-01],
       [ 2.44450621e-01,  3.35888738e-01,  2.89299332e-01],
       [ 2.77341856e-01,  2.65842674e-01,  3.24513475e-01],
       [ 2.78044056e-01,  3.28368295e-01,  2.61701707e-01],
       [ 2.70839692e-01,  3.04497380e-01,  2.91185865e-01],
       [ 2.64615294e-01,  3.22389876e-01,  2.80560916e-01],
       [ 2.88979262e-01,  2.79071277e-01,  2.98132256e-01],
       [ 4.08547096e-01,  2.29806184e-01,  2.44556354e-01],
       [ 2.66729196e-01,  2.49930176e-01,  3.54849511e-01],
       [ 3.77855122e-01,  2.42796554e-01,  2.54986952e-01],
       [ 1.53636359e-01,  4.04012915e-01,  3.37002011e-01],
       [ 3.37417583e-01,  3.38344309e-01,  2.01012638e-01],
       [ 2.95092599e-01,  2.97217488e-01,  2.74000600e-01],
       [ 2.83559257e-01,  3.32070433e-01,  2.53136516e-01],
       [ 2.47807712e-01,  2.95705882e-01,  3.25146977e-01],
       [ 2.74855109e-01,  2.85216667e-01,  3.06401339e-01],
       [ 2.35698696e-01,  3.07275525e-01,  3.27042340e-01],
       [ 2.50197219e-01,  3.32969026e-01,  2.85839720e-01],
       [ 3.13975033e-01,  3.47766991e-01,  2.12786618e-01],
       [ 3.33259506e-01,  2.68292400e-01,  2.66957856e-01],
       [ 3.12556804e-01,  2.51359618e-01,  3.04009308e-01],
       [ 2.30708561e-01,  2.95608170e-01,  3.45420645e-01],
       [ 3.43879694e-01,  2.05018711e-01,  3.27015395e-01],
       [ 3.53458985e-01,  2.20883251e-01,  2.99345616e-01],
       [ 2.20796887e-01,  3.24790701e-01,  3.26780206e-01],
       [ 1.90584297e-01,  4.15132572e-01,  2.82115719e-01],
       [ 2.24539481e-01,  2.97435638e-01,  3.51001321e-01],
       [ 2.89226922e-01,  3.87628344e-01,  2.03717772e-01],
       [ 4.04596789e-01,  1.98762095e-01,  2.81062398e-01],
       [ 3.66817743e-01,  2.20057958e-01,  2.88440698e-01],
       [ 1.73340933e-01,  3.19574634e-01,  3.94803182e-01],
       [ 2.67355304e-01,  2.02174733e-01,  4.17326896e-01],
       [ 3.12120511e-01,  2.85896372e-01,  2.68831672e-01],
       [ 3.00069809e-01,  2.44854834e-01,  3.23946547e-01],
       [ 2.53201294e-01,  2.57521403e-01,  3.61830890e-01],
       [ 2.89721204e-01,  2.63534847e-01,  3.13866187e-01],
       [ 3.01026311e-01,  2.90307532e-01,  2.74988142e-01],
       [ 3.05346096e-01,  2.85486502e-01,  2.75590432e-01],
       [ 2.86255973e-01,  2.78520261e-01,  3.01485618e-01],
       [ 3.42223049e-01,  2.05179647e-01,  3.28428773e-01],
       [ 6.09388976e-01,  4.57920946e-01, -2.72198128e-02],
       [ 3.52538485e-01,  3.24782579e-01,  2.00055260e-01],
       [ 2.69086306e-01,  3.02211669e-01,  2.95255691e-01],
       [ 3.32864435e-01,  2.77948222e-01,  2.57821970e-01],
       [ 3.02509878e-01,  2.54445069e-01,  3.10667594e-01],
       [ 2.67019983e-01,  2.63784175e-01,  3.38287392e-01],
       [ 2.89778624e-01,  2.78215166e-01,  2.98206063e-01],
       [ 2.46679896e-01,  2.64297503e-01,  3.61665937e-01],
       [ 2.55071084e-01,  2.90011332e-01,  3.22935560e-01],
       [ 3.11077559e-01,  3.09635483e-01,  2.47585179e-01],
       [ 2.93187700e-01,  2.75950886e-01,  2.97106186e-01],
       [ 3.25576552e-01,  2.86242553e-01,  2.56294934e-01],
       [ 3.41028222e-01,  2.37613253e-01,  2.92006678e-01],
       [ 2.57203098e-01,  3.28037208e-01,  2.83008215e-01],
       [ 2.23990011e-01,  3.93071131e-01,  2.62463445e-01],
       [ 2.74968896e-01,  2.63247832e-01,  3.30006528e-01],
       [ 2.76705162e-01,  3.24784313e-01,  2.66223284e-01],
       [ 2.46735130e-01,  3.17055010e-01,  3.04672570e-01],
       [ 2.73826408e-01,  2.46611185e-01,  3.50611383e-01],
       [ 7.14455642e-01,  1.04214003e-01,  2.14425585e-01],
       [ 4.60426617e-01,  2.27353447e-01,  2.11289086e-01],
       [ 2.74474001e-01,  2.58213377e-01,  3.36270669e-01],
       [ 3.29103649e-01,  3.33424575e-01,  2.11717404e-01],
       [ 3.25032640e-01,  2.26806970e-01,  3.19441244e-01],
       [ 2.55998990e-01,  3.14849211e-01,  2.96749503e-01],
       [ 3.19338709e-01,  2.89014648e-01,  2.59234922e-01],
       [ 3.20782800e-01,  2.85902119e-01,  2.60905641e-01],
       [ 2.91859959e-01,  2.64690982e-01,  3.10389019e-01],
       [ 3.67317695e-01,  2.41629359e-01,  2.64793564e-01],
       [ 2.68825139e-01,  3.14730002e-01,  2.83422511e-01],
       [ 2.99334114e-01,  3.75352545e-01,  2.04012006e-01],
       [ 3.12516884e-01,  3.19902112e-01,  2.37224356e-01],
       [ 2.87018227e-01,  2.75539153e-01,  3.03818324e-01],
       [ 3.20160453e-01,  3.51966868e-01,  2.04297792e-01],
       [ 3.17564007e-01,  2.68191035e-01,  2.81401215e-01],
       [ 2.21348920e-01,  2.99473128e-01,  3.52734580e-01],
       [ 3.31440923e-01,  2.71731632e-01,  2.65159640e-01],
       [ 2.31726752e-01,  2.80301421e-01,  3.61399375e-01],
       [ 2.81947909e-01,  3.77762890e-01,  2.17505221e-01],
       [ 2.83451715e-01,  3.01519494e-01,  2.81268172e-01],
       [ 2.92549246e-01,  3.28611067e-01,  2.47705909e-01],
       [ 3.56328964e-01,  2.22010993e-01,  2.95485468e-01],
       [ 2.57518353e-01,  3.19908467e-01,  2.90283882e-01],
       [ 2.12802339e-01,  2.88549668e-01,  3.76174730e-01],
       [ 2.78161256e-01,  2.79125563e-01,  3.09280746e-01],
       [ 2.82527123e-01,  3.22498724e-01,  2.62609216e-01],
       [ 3.33548712e-01,  2.40500764e-01,  2.95760708e-01],
       [ 3.23466125e-01,  3.21569033e-01,  2.26318378e-01],
       [ 4.94663252e-01,  2.28068760e-01,  1.89811386e-01],
       [ 1.84814334e-01,  2.18063585e-01,  5.20501905e-01],
       [ 3.18235654e-01,  2.74391555e-01,  2.74504481e-01],
       [ 2.96284917e-01,  3.96019352e-01,  1.91628804e-01],
       [ 1.99146444e-01,  3.09335078e-01,  3.70509076e-01],
       [ 2.26710956e-01,  2.76962892e-01,  3.71687906e-01],
       [ 5.81315516e-01,  8.99769612e-01, -1.84358098e-01],
       [ 3.15606594e-01,  4.63175019e-01,  1.33309400e-01],
       [ 4.10861384e-01,  1.87210819e-01,  2.89400348e-01],
       [ 2.42818303e-01,  2.80594424e-01,  3.47462966e-01],
       [ 2.55854108e-01,  3.86885476e-01,  2.34953262e-01],
       [ 2.69339271e-01,  2.59091509e-01,  3.41040849e-01],
       [ 3.47016845e-01,  2.62762932e-01,  2.60449497e-01],
       [ 2.58347579e-01,  2.85768715e-01,  3.23777005e-01],
       [ 2.80989236e-01,  3.12369559e-01,  2.73405429e-01],
       [ 2.87418582e-01,  3.01953653e-01,  2.76926702e-01],
       [ 2.45928237e-01,  1.83664605e-01,  4.76804239e-01],
       [ 4.65753932e-01,  5.81165212e-02,  4.25547959e-01],
       [ 4.16123512e-01,  2.22938923e-01,  2.46032099e-01],
       [ 3.04745493e-01,  2.67552937e-01,  2.94364687e-01],
       [ 2.93911646e-01,  2.76730372e-01,  2.95571857e-01],
       [ 3.33846217e-01,  1.79388811e-01,  3.70418451e-01],
       [ 2.39372017e-01,  3.23027552e-01,  3.07034808e-01],
       [ 3.22153058e-01,  2.80031442e-01,  2.65345612e-01],
       [ 2.63444193e-01,  3.70820107e-01,  2.40135849e-01],
       [ 2.62520450e-01,  4.35037731e-01,  1.94670069e-01],
       [ 2.49540887e-01,  2.84580991e-01,  3.35102183e-01],
       [ 2.79187879e-01,  2.65182454e-01,  3.23243687e-01],
       [ 2.83513008e-01,  2.59421473e-01,  3.24994349e-01],
       [ 3.43938866e-01,  3.28094843e-01,  2.04090107e-01],
       [ 3.81294551e-01,  9.03303635e-02,  4.57052878e-01],
       [ 3.07998309e-01,  2.66064688e-01,  2.92742307e-01],
       [ 3.36563421e-01,  2.84724758e-01,  2.48149034e-01],
       [ 2.89159107e-01,  3.06600882e-01,  2.70819736e-01],
       [ 4.08067684e-01,  3.64240582e-01,  1.31249650e-01],
       [ 1.00915146e+00,  4.50564002e-01, -1.40224123e-01],
       [ 4.02190568e-01,  1.41616806e-01,  3.54984256e-01],
       [ 2.51579670e-01,  3.82940657e-01,  2.42167050e-01],
       [ 8.69187729e-02,  4.40812328e-01,  4.01123097e-01],
       [ 2.00125343e-01,  2.46914525e-01,  4.48698562e-01],
       [ 2.39886217e-01,  3.68612803e-01,  2.65530204e-01],
       [ 2.60778547e-01,  2.74160504e-01,  3.33691889e-01],
       [ 3.07544306e-01,  2.67558012e-01,  2.91624727e-01],
       [ 3.01779032e-01,  3.04619780e-01,  2.60673561e-01],
       [ 2.58429298e-01,  2.96131836e-01,  3.12807455e-01],
       [ 3.01162436e-01,  2.61482179e-01,  3.04368664e-01],
       [ 3.15662301e-01,  3.46782122e-01,  2.12144525e-01],
       [ 3.11536778e-01,  2.30892157e-01,  3.28280056e-01],
       [ 2.94524729e-01,  2.70426476e-01,  3.01535272e-01],
       [ 2.96842123e-01,  2.79902383e-01,  2.89405761e-01],
       [ 2.03455336e-01,  3.16152674e-01,  3.57340625e-01],
       [ 2.77489601e-01,  2.75804648e-01,  3.13516865e-01],
       [ 3.01269831e-01,  3.36909996e-01,  2.32691753e-01],
       [ 2.73240760e-01,  3.27397651e-01,  2.67284965e-01],
       [ 2.91687940e-01,  3.60099917e-01,  2.22408558e-01],
       [ 1.80063188e-01,  3.13273548e-01,  3.92411413e-01],
       [ 2.87034966e-01,  2.88795809e-01,  2.90198982e-01],
       [ 2.49738117e-01,  3.29356324e-01,  2.89671529e-01],
       [ 2.96344332e-01,  3.56092671e-01,  2.21437402e-01],
       [ 3.19654571e-01,  2.94400558e-01,  2.53875440e-01],
       [ 2.97584457e-01,  3.69804105e-01,  2.09700996e-01],
       [ 2.62237295e-01,  3.16780191e-01,  2.88295990e-01],
       [ 3.27038749e-01,  2.56307739e-01,  2.84869183e-01],
       [ 2.41701096e-01,  2.90059173e-01,  3.38296015e-01],
       [ 3.20383275e-01,  3.24774799e-01,  2.26219887e-01],
       [ 1.60078362e-02,  4.07724313e-01,  5.74592266e-01],
       [ 1.89330786e-01,  3.26686668e-01,  3.64615877e-01],
       [ 2.46358168e-01,  3.41560792e-01,  2.82103011e-01],
       [ 1.99151176e-01,  2.74793382e-01,  4.12019869e-01],
       [ 2.08082837e-01,  2.71265102e-01,  4.03786837e-01],
       [ 3.38569540e-01,  2.02677147e-01,  3.35114646e-01],
       [ 1.90209654e-01,  1.92348823e-01,  5.57858236e-01],
       [ 1.66671783e-01,  2.32896062e-01,  5.28528010e-01],
       [ 2.44171236e-01,  2.44433972e-01,  3.89509264e-01],
       [ 2.77422522e-01,  4.89294779e-01,  1.49023125e-01],
       [ 2.97199605e-01,  2.97326391e-01,  2.71871567e-01],
       [ 4.05143111e-01,  2.59275905e-01,  2.18169784e-01],
       [ 2.70398924e-01,  3.15653496e-01,  2.80943527e-01],
       [ 2.83340787e-01,  2.49920452e-01,  3.36021690e-01],
       [ 3.29983114e-01,  2.85082769e-01,  2.53513493e-01],
       [ 2.65887917e-01,  3.07193246e-01,  2.93712371e-01],
       [ 3.32170821e-01,  2.69442494e-01,  2.66781106e-01],
       [ 2.58987663e-01,  3.93363629e-01,  2.27061209e-01],
       [ 2.56600527e-01,  3.39403489e-01,  2.73335215e-01],
       [ 5.15706842e-01,  1.59607087e-01,  2.48313453e-01],
       [ 3.37851122e-01,  1.76809564e-01,  3.69689576e-01],
       [ 3.41875309e-01,  1.94069186e-01,  3.42670815e-01],
       [ 5.26800607e-01,  3.46686447e-02,  4.12732373e-01],
       [ 2.84259682e-01,  3.20369014e-01,  2.62858854e-01],
       [ 3.35381770e-01,  3.15140052e-01,  2.21833867e-01],
       [ 2.95223890e-01,  3.19820949e-01,  2.52959143e-01],
       [ 2.52937638e-01,  2.10247176e-01,  4.24928818e-01],
       [ 2.48543663e-01,  3.90676111e-01,  2.39197745e-01],
       [ 4.89031638e-01,  6.56755160e-01, -6.21180585e-02],
       [ 2.45799092e-01,  2.87676723e-01,  3.36077696e-01],
       [ 5.13327293e-01,  3.56257989e-01,  7.71897274e-02],
       [ 2.27863082e-01,  2.92863866e-01,  3.51944791e-01],
       [ 2.50307838e-01,  2.47994965e-01,  3.77129960e-01],
       [ 2.32641093e-01,  2.27548212e-01,  4.28221458e-01],
       [ 2.58632633e-01,  3.08292114e-01,  3.00331922e-01],
       [ 3.38375122e-01,  2.17422891e-01,  3.17434569e-01],
       [-8.12781854e-02,  6.54294311e-01,  5.29094803e-01],
       [ 2.60453171e-01,  2.97481565e-01,  3.09211763e-01],
       [ 2.63743707e-01,  3.09489066e-01,  2.93727460e-01],
       [ 3.01002461e-01,  2.54364762e-01,  3.12289910e-01],
       [ 3.36016042e-01,  2.57082401e-01,  2.75866833e-01],
       [ 3.10785455e-01,  2.96295835e-01,  2.60122601e-01],
       [ 4.16538405e-01,  3.23123634e-01,  1.56026119e-01],
       [ 3.40078823e-01,  2.60406025e-01,  2.68851788e-01],
       [ 3.16810781e-01,  2.36820387e-01,  3.16045697e-01],
       [ 2.45077365e-01,  3.20085220e-01,  3.03548682e-01],
       [ 3.76380359e-01,  2.21437306e-01,  2.78772871e-01],
       [ 3.32122900e-01,  2.91959383e-01,  2.45213823e-01],
       [ 2.82423280e-01,  2.46355451e-01,  3.41207909e-01],
       [ 2.47148639e-01,  2.69603555e-01,  3.54846773e-01],
       [ 2.46795326e-01,  2.68638405e-01,  3.56401388e-01],
       [ 3.58360177e-01,  2.14831159e-01,  3.01841736e-01],
       [ 3.65185105e-01,  1.81530459e-01,  3.36020762e-01],
       [ 3.13640309e-01,  2.66840249e-01,  2.86500796e-01],
       [ 3.64258893e-01,  2.28145156e-01,  2.81726802e-01],
       [ 3.87343018e-01,  2.13257643e-01,  2.78714547e-01],
       [ 2.96451453e-01,  2.60755837e-01,  3.09935918e-01],
       [ 2.74232966e-01,  2.92310886e-01,  2.99780357e-01],
       [ 2.81380096e-01,  3.21812050e-01,  2.64341132e-01],
       [ 2.49701649e-01,  3.14122796e-01,  3.04284465e-01],
       [ 3.84943868e-01,  2.95058306e-01,  2.00615704e-01],
       [ 9.41827695e-02,  6.13204559e-01,  2.71770059e-01],
       [ 2.91413874e-01,  3.47868052e-01,  2.32489638e-01],
       [ 2.45367990e-01,  3.22725576e-01,  3.00678417e-01],
       [ 3.15869543e-01,  4.18096249e-01,  1.60683413e-01],
       [ 2.78736023e-01,  3.96489623e-01,  2.06573995e-01],
       [ 1.67072520e-01,  3.85279322e-01,  3.36072768e-01],
       [ 3.93209259e-01,  1.78282759e-01,  3.14785793e-01],
       [ 1.76858654e-01,  4.78042776e-01,  2.52639543e-01],
       [ 2.30758813e-01,  2.53640315e-01,  3.95273754e-01],
       [ 4.35773930e-01,  2.89162126e-01,  1.71036718e-01],
       [ 1.59933882e-01,  3.50929575e-01,  3.79503502e-01],
       [ 2.75079638e-01,  2.63272157e-01,  3.29857153e-01],
       [ 2.67249872e-01,  3.69124730e-01,  2.37833915e-01],
       [ 3.45540007e-01,  3.61072056e-01,  1.77233260e-01],
       [ 3.33873258e-01,  2.18448835e-01,  3.20584272e-01],
       [ 2.67013896e-01,  2.77035003e-01,  3.23551440e-01],
       [ 3.07048127e-01,  2.62315982e-01,  2.97623904e-01],
       [ 2.93315878e-01,  3.05819118e-01,  2.67549714e-01],
       [ 3.17285522e-01,  3.30272705e-01,  2.24241229e-01],
       [ 3.61114500e-01,  2.68405615e-01,  2.43161159e-01],
       [ 5.13667619e-01,  2.88885273e-01,  1.26607219e-01],
       [ 2.39626306e-01,  3.76404907e-01,  2.59408094e-01],
       [ 2.68039198e-01,  3.59106105e-01,  2.45151302e-01],
       [ 2.05621192e-01,  2.27507392e-01,  4.69190135e-01],
       [ 5.12150015e-01,  1.93902046e-01,  2.13430528e-01],
       [-7.72118769e-02,  5.10660236e-01,  6.67735912e-01],
       [ 1.62236617e-01,  4.56454032e-01,  2.84385164e-01],
       [ 2.95797975e-01,  3.14648445e-01,  2.57070272e-01],
       [ 3.99208119e-01,  1.97153773e-01,  2.87232661e-01],
       [ 3.43274621e-01,  3.53339761e-01,  1.84761370e-01],
       [ 2.30817717e-01,  2.73779870e-01,  3.70209371e-01],
       [ 3.30733897e-01,  2.61026671e-01,  2.76580835e-01],
       [ 2.89668568e-01,  2.89455852e-01,  2.86905080e-01],
       [ 2.94224573e-01,  3.27945913e-01,  2.46733423e-01],
       [ 2.20254873e-01,  3.11747931e-01,  3.40855345e-01],
       [ 3.02176166e-01,  2.42099849e-01,  3.24914549e-01],
       [ 1.89200624e-01,  2.25170656e-01,  5.00511453e-01],
       [ 3.07614843e-01,  2.61986023e-01,  2.97417404e-01],
       [ 2.09289243e-01,  3.08263803e-01,  3.58385877e-01],
       [ 2.74533446e-01,  3.40613501e-01,  2.54394829e-01],
       [ 2.62143158e-01,  2.88473800e-01,  3.16696688e-01],
       [ 2.87971433e-01,  3.41940945e-01,  2.40558499e-01],
       [ 2.99376799e-01,  2.60847565e-01,  3.06856150e-01],
       [ 3.04549357e-01,  2.93706673e-01,  2.68366408e-01],
       [ 3.25176143e-01,  2.64521733e-01,  2.78081115e-01],
       [ 2.88325481e-01,  3.26822537e-01,  2.53221550e-01],
       [ 3.18461111e-01,  3.17936487e-01,  2.33736887e-01],
       [ 3.03256908e-01,  2.58876524e-01,  3.05076866e-01],
       [ 2.68920608e-01,  3.55359226e-01,  2.47383581e-01],
       [ 2.31774272e-01,  2.70326159e-01,  3.73123264e-01],
       [ 2.58439165e-01,  3.10316412e-01,  2.98550190e-01],
       [ 3.05139491e-01,  3.11002405e-01,  2.51729164e-01],
       [ 2.68332857e-01,  2.71984647e-01,  3.27617708e-01],
       [ 3.09411906e-01,  2.60204251e-01,  2.97550735e-01],
       [ 3.14919312e-01,  3.57890416e-01,  2.04059767e-01],
       [ 2.19257423e-01,  2.68764614e-01,  3.91521998e-01],
       [ 2.37046710e-01,  2.88134227e-01,  3.45973582e-01],
       [ 3.47959217e-01,  2.43196372e-01,  2.79753053e-01],
       [ 4.59750273e-01,  1.68850656e-01,  2.74213506e-01],
       [ 4.16903792e-01,  2.04942946e-01,  2.64628725e-01],
       [ 1.95282212e-01,  3.29154533e-01,  3.54136083e-01],
       [ 2.62165925e-01,  2.18917803e-01,  4.00361103e-01],
       [ 3.38576464e-01,  1.91907463e-01,  3.48784648e-01],
       [ 3.33010935e-01,  3.13389880e-01,  2.25305631e-01],
       [ 1.23184460e+00,  4.28087826e-01, -1.67077691e-01],
       [ 8.71806623e-01,  1.41688521e-01,  1.24790937e-01],
       [ 2.32377532e-01,  3.82593371e-01,  2.61953689e-01],
       [ 2.83104142e-01,  3.08282586e-01,  2.75156535e-01],
       [ 4.98904772e-01,  2.48130725e-01,  1.68942972e-01],
       [ 3.36249941e-01,  2.34781915e-01,  2.99553507e-01],
       [ 3.01339042e-01,  2.97414084e-01,  2.67852506e-01],
       [ 2.72702709e-01,  3.07004603e-01,  2.86832700e-01],
       [ 1.81578447e-01,  3.29336158e-01,  3.72272880e-01],
       [ 2.89572047e-01,  3.46018963e-01,  2.35690213e-01],
       [ 2.60860699e-01,  2.44899403e-01,  3.67991404e-01],
       [ 2.39671464e-01,  2.75833716e-01,  3.56719073e-01],
       [ 3.61436341e-01,  2.21647496e-01,  2.91362115e-01],
       [ 1.90397618e-01,  3.19213308e-01,  3.71307868e-01],
       [ 9.13849635e-02,  3.12554201e-01,  5.48194542e-01],
       [ 4.04338045e-01,  3.22483571e-01,  1.64562557e-01],
       [ 3.23814138e-01,  3.18278170e-01,  2.28840973e-01],
       [ 3.31763304e-01,  2.33805014e-01,  3.04882841e-01],
       [ 3.25743982e-01,  2.58825968e-01,  2.83437421e-01],
       [ 3.23282037e-01,  2.73269487e-01,  2.70985619e-01],
       [ 2.45599050e-01,  2.86031339e-01,  3.38112679e-01],
       [ 3.18513258e-01,  2.74083196e-01,  2.74555926e-01],
       [ 2.87392411e-01,  3.08475410e-01,  2.70775670e-01],
       [ 3.20224378e-01,  3.30580097e-01,  2.21480029e-01],
       [ 2.18128892e-01,  3.99679669e-01,  2.63541696e-01],
       [ 2.95220010e-01,  3.69194158e-01,  2.12226807e-01],
       [ 3.19913498e-01,  4.13903058e-01,  1.60240081e-01],
       [ 2.18172086e-01,  3.26423185e-01,  3.28287045e-01],
       [ 2.89422717e-01,  3.47035109e-01,  2.34988321e-01],
       [ 2.84874087e-01,  3.43085798e-01,  2.42473684e-01],
       [ 2.94319735e-01,  3.19518426e-01,  2.54072541e-01],
       [ 2.52950842e-01,  2.61128211e-01,  3.57819284e-01],
       [ 2.34311296e-01,  3.17369431e-01,  3.18366672e-01],
       [ 3.24694797e-01,  3.11430066e-01,  2.34042540e-01],
       [ 2.67600585e-01,  2.65348562e-01,  3.35852999e-01],
       [ 2.77179640e-01,  2.48126724e-01,  3.44987870e-01],
       [ 3.25764014e-01,  2.57194102e-01,  2.85124115e-01],
       [ 2.64838972e-01,  3.47370815e-01,  2.58085829e-01],
       [ 2.89478523e-01,  2.37065464e-01,  3.44462465e-01],
       [ 3.37215558e-01,  2.31593902e-01,  3.02215319e-01],
       [ 2.26408994e-01,  2.69484259e-01,  3.81102866e-01],
       [ 2.90405158e-01,  2.53075307e-01,  3.24769033e-01],
       [ 3.22985915e-01,  2.61107154e-01,  2.83629571e-01],
       [ 2.77006917e-01,  2.88184665e-01,  3.01085260e-01],
       [ 2.89886999e-01,  2.58106895e-01,  3.19671749e-01],
       [ 2.88803984e-01,  2.37166751e-01,  3.45086305e-01],
       [ 3.86489933e-01,  3.41530694e-01,  1.62085826e-01],
       [ 2.47918369e-01,  2.91797475e-01,  3.29169596e-01],
       [ 2.46240044e-01,  3.47052440e-01,  2.77336721e-01],
       [ 2.58481584e-01,  2.68161779e-01,  3.43088191e-01],
       [ 1.81084692e-01,  2.62404300e-01,  4.56567359e-01],
       [ 2.04121399e-01,  2.72116803e-01,  4.08314866e-01],
       [ 3.60680425e-01,  1.73251772e-01,  3.51189680e-01],
       [ 3.31297400e-01,  2.78908886e-01,  2.58270383e-01],
       [ 2.73981486e-01,  2.59304749e-01,  3.35570820e-01],
       [ 1.48764966e-01,  3.04187240e-01,  4.52029139e-01],
       [ 3.84728405e-01,  1.26832441e-01,  3.93313833e-01],
       [ 2.77039691e-01,  3.05136978e-01,  2.84217755e-01],
       [ 3.25997451e-01,  2.54658149e-01,  2.87575100e-01],
       [ 7.24257676e-01, -7.61192374e-02,  4.70778963e-01],
       [ 2.31443869e-01,  2.86867036e-01,  3.54239862e-01],
       [ 3.30103109e-01,  2.50972477e-01,  2.87661725e-01],
       [ 3.74837913e-01,  2.38610526e-01,  2.61733045e-01],
       [ 3.23417194e-01,  3.40864041e-01,  2.10390872e-01],
       [ 3.17124068e-01,  2.46853990e-01,  3.04474006e-01],
       [ 2.45318714e-01,  2.66889524e-01,  3.60257785e-01],
       [ 4.16010693e-01,  3.45288377e-01,  1.39703760e-01],
       [ 3.37652610e-01,  1.93496854e-01,  3.47671032e-01],
       [ 1.82521552e-01,  3.56338553e-01,  3.43244068e-01],
       [ 2.56075877e-01,  2.75707752e-01,  3.37351294e-01],
       [ 3.07309475e-01,  2.69198042e-01,  2.90148673e-01],
       [ 2.94416192e-01,  3.30997904e-01,  2.43916564e-01],
       [ 2.68204813e-01,  3.39850291e-01,  2.61243619e-01],
       [ 2.47816674e-01,  2.77454920e-01,  3.45044443e-01],
       [ 2.68867368e-01,  3.04179542e-01,  2.93546731e-01],
       [ 2.66681042e-01,  3.06945079e-01,  2.93123972e-01],
       [ 2.76853398e-01,  2.34562843e-01,  3.61858825e-01],
       [ 4.00049856e-01,  2.06156240e-01,  2.76353581e-01],
       [ 2.47106057e-01,  3.41098205e-01,  2.81726228e-01],
       [ 2.91681006e-01,  2.66936422e-01,  3.08153143e-01],
       [ 2.45610596e-01,  2.66732130e-01,  3.60086624e-01],
       [ 3.49168151e-01,  1.98449070e-01,  3.29989448e-01],
       [ 2.55269181e-01,  2.60257728e-01,  3.56071079e-01],
       [ 2.57364109e-01,  2.83838027e-01,  3.26957836e-01],
       [ 3.37108472e-01,  2.64506846e-01,  2.67334452e-01],
       [ 3.22499363e-01,  2.47201849e-01,  2.98889238e-01],
       [ 2.49662655e-01,  2.98126529e-01,  3.20504575e-01],
       [ 3.09791934e-01,  2.75235758e-01,  2.81583564e-01],
       [ 2.55163061e-01,  3.25719457e-01,  2.87301513e-01],
       [ 3.01450370e-01,  2.95422211e-01,  2.69646939e-01],
       [ 2.75272480e-01,  2.21767821e-01,  3.80157145e-01],
       [ 2.87940047e-01,  2.93187653e-01,  2.84927965e-01],
       [ 2.63115712e-01,  2.82299006e-01,  3.22181801e-01],
       [ 3.67752332e-01,  2.40558243e-01,  2.65545515e-01],
       [ 2.67048692e-01,  3.17981802e-01,  2.82179095e-01],
       [ 2.58059472e-01,  2.82663301e-01,  3.27443317e-01],
       [ 2.72625043e-01,  3.12135276e-01,  2.82002902e-01],
       [ 2.86689789e-01,  2.55273114e-01,  3.26250753e-01],
       [ 2.51585843e-01,  3.08497387e-01,  3.07787157e-01],
       [ 4.02849100e-01,  1.96634083e-01,  2.84888954e-01],
       [ 2.86439446e-01,  3.76300829e-01,  2.14582701e-01],
       [ 2.62867711e-01,  3.16015593e-01,  2.88365035e-01],
       [ 3.34091088e-01,  2.61593347e-01,  2.72970023e-01],
       [ 4.63688506e-01,  1.70674880e-01,  2.69340922e-01],
       [ 2.75906653e-01,  4.18318892e-01,  1.93860678e-01],
       [ 1.92047472e-01,  2.98149935e-01,  3.93190695e-01],
       [ 3.37672780e-01,  3.10107212e-01,  2.24281759e-01],
       [ 3.34241675e-01,  2.50126786e-01,  2.84747065e-01],
       [ 3.07799282e-01,  3.20103222e-01,  2.41235633e-01],
       [ 3.85528730e-01,  1.76299511e-01,  3.23998440e-01],
       [ 3.58388855e-01,  3.21220913e-01,  1.98463899e-01],
       [ 2.63978609e-01,  2.74384298e-01,  3.29830329e-01],
       [ 3.40807955e-01,  2.43774043e-01,  2.85537132e-01],
       [ 4.08435978e-01,  3.28628949e-01,  1.57077226e-01],
       [ 5.22859241e-01,  1.96691211e-01,  2.04514058e-01],
       [ 3.12205324e-01,  3.19150647e-01,  2.38153238e-01],
       [ 2.15899641e-01,  3.72822544e-01,  2.87924849e-01],
       [ 3.33956110e-01,  2.47083960e-01,  2.88250003e-01],
       [ 2.26785668e-01,  3.22201844e-01,  3.22282813e-01],
       [ 2.28011029e-01,  3.05024622e-01,  3.38534621e-01],
       [ 3.26465490e-01,  1.94497721e-01,  3.57996884e-01],
       [ 2.84971132e-01,  3.46889151e-01,  2.39208904e-01],
       [ 3.01175421e-01,  2.58944371e-01,  3.07099164e-01],
       [ 2.96914466e-01,  2.93913076e-01,  2.75432244e-01],
       [ 2.84361360e-01,  2.86715488e-01,  2.95002669e-01],
       [ 2.40623959e-01,  3.48241121e-01,  2.82246050e-01],
       [ 3.69673339e-01,  2.67989524e-01,  2.36694696e-01],
       [ 2.58239721e-01,  2.55821132e-01,  3.57811377e-01],
       [ 2.22012202e-01,  3.00913852e-01,  3.50323820e-01],
       [ 3.07140252e-01,  2.92896681e-01,  2.66716315e-01],
       [ 2.50860074e-01,  3.14139399e-01,  3.03000224e-01],
       [ 2.84060478e-01,  3.35918606e-01,  2.49329057e-01],
       [ 3.21564166e-01,  2.12619562e-01,  3.40012543e-01],
       [ 2.66648791e-01,  3.64256099e-01,  2.42305225e-01],
       [ 3.01672552e-01,  3.27833345e-01,  2.40032189e-01],
       [ 3.60209397e-01,  2.69512435e-01,  2.42835932e-01],
       [ 3.10833734e-01,  3.35818704e-01,  2.25184674e-01],
       [ 3.27981985e-01,  2.41503880e-01,  2.99903981e-01],
       [ 3.21649212e-01,  2.88632380e-01,  2.57523125e-01],
       [ 2.84079981e-01,  2.73484651e-01,  3.09037690e-01],
       [ 2.88413953e-01,  2.37464009e-01,  3.45160055e-01],
       [ 3.89956893e-01,  2.41835047e-01,  2.46433591e-01],
       [ 2.60983639e-01,  3.47966917e-01,  2.61410925e-01],
       [ 2.23438780e-01,  3.24503355e-01,  3.23927209e-01],
       [ 2.85702120e-01,  3.63904100e-01,  2.24800537e-01],
       [ 3.18188741e-01,  3.01243252e-01,  2.48853451e-01],
       [ 3.15322978e-01,  2.88837395e-01,  2.63047594e-01],
       [ 3.17390036e-01,  2.62719659e-01,  2.87213607e-01],
       [ 2.87340799e-01,  2.91805530e-01,  2.86891857e-01],
       [ 2.53032278e-01,  3.92965384e-01,  2.33076809e-01],
       [ 2.01662976e-01,  5.03505612e-01,  2.10533428e-01],
       [ 2.10557642e-01,  2.96792489e-01,  3.69583177e-01],
       [ 3.76916333e-01,  2.03964571e-01,  2.98034279e-01],
       [ 2.97906227e-01,  3.00769332e-01,  2.67922986e-01],
       [ 2.91683674e-01,  2.47275611e-01,  3.30031871e-01],
       [ 3.20689400e-01,  2.34370159e-01,  3.14993177e-01],
       [ 1.99375211e-01,  2.48164501e-01,  4.48054429e-01],
       [ 4.06907946e-01,  2.17794644e-01,  2.58327132e-01],
       [ 3.28675318e-01,  2.40745232e-01,  3.00082223e-01],
       [ 3.14103030e-01,  2.35524146e-01,  3.20255548e-01],
       [ 2.57142102e-01,  2.71545340e-01,  3.40795423e-01],
       [ 3.35355955e-01,  3.33659829e-01,  2.06430390e-01],
       [ 2.52265929e-01,  3.25735512e-01,  2.90359187e-01],
       [ 2.73265670e-01,  3.37705035e-01,  2.58141358e-01],
       [ 3.37960813e-01,  2.85181912e-01,  2.46524083e-01],
       [ 4.14469609e-01,  2.40657241e-01,  2.29352357e-01],
       [ 2.27498296e-01,  3.69553666e-01,  2.77910770e-01],
       [ 3.52273468e-01,  2.38809250e-01,  2.80627114e-01],
       [ 3.30609320e-01,  2.74746619e-01,  2.62929951e-01],
       [ 2.79151814e-01,  3.92586589e-01,  2.09022948e-01],
       [ 2.84836293e-01,  2.66624332e-01,  3.15626729e-01],
       [ 2.42985356e-01,  3.14746352e-01,  3.11119564e-01],
       [ 2.92238919e-01,  3.07912436e-01,  2.66626079e-01],
       [ 2.78520281e-01,  2.99794865e-01,  2.87907123e-01],
       [ 2.75492548e-01,  2.49501291e-01,  3.45269297e-01],
       [ 2.33523496e-01,  3.65673943e-01,  2.74711692e-01],
       [ 2.83921041e-01,  3.43473889e-01,  2.43037565e-01],
       [ 2.21753545e-01,  3.29303232e-01,  3.21157181e-01],
       [ 2.68432897e-01,  3.43840868e-01,  2.57567462e-01],
       [ 2.00231276e-01,  3.47019998e-01,  3.29858972e-01],
       [ 3.25864976e-01,  2.16595573e-01,  3.30750484e-01],
       [ 3.81365384e-01,  2.50872457e-01,  2.44094736e-01],
       [ 2.19586925e-01,  3.46113268e-01,  3.07579976e-01],
       [ 2.56610373e-01,  2.53578711e-01,  3.62471637e-01],
       [ 3.30519125e-01,  2.41217804e-01,  2.97816870e-01],
       [ 2.80984734e-01,  2.50634350e-01,  3.37789932e-01],
       [ 3.33100515e-01,  2.31717459e-01,  3.05965466e-01],
       [ 2.89417507e-01,  3.32320705e-01,  2.47404080e-01],
       [ 2.97044775e-01,  2.63373207e-01,  3.06496884e-01],
       [ 2.76345782e-01,  2.95435961e-01,  2.94443676e-01],
       [ 3.40288054e-01,  2.91559191e-01,  2.38642950e-01],
       [ 2.16820021e-01,  4.10857078e-01,  2.56370609e-01],
       [ 3.91154422e-01,  3.03940641e-01,  1.88624954e-01],
       [ 3.37920558e-01,  3.19320338e-01,  2.16199409e-01],
       [ 2.67573076e-01,  3.50264138e-01,  2.52944859e-01],
       [ 2.87306192e-01,  2.92287573e-01,  2.86448855e-01],
       [ 2.82297042e-01,  3.06408493e-01,  2.77731021e-01],
       [ 2.73865019e-01,  2.78636679e-01,  3.14372175e-01],
       [ 2.95550502e-01,  2.76618836e-01,  2.94047851e-01],
       [ 3.79210910e-01,  1.24336386e-01,  4.02842568e-01],
       [ 4.77218215e-01, -2.81312512e-02,  5.86578473e-01],
       [ 2.16726611e-01,  2.59433837e-01,  4.06950607e-01],
       [ 2.72519725e-01,  3.95500853e-01,  2.12894963e-01],
       [ 3.70452478e-01,  2.48500635e-01,  2.55176557e-01],
       [ 3.11477331e-01,  2.75577234e-01,  2.79640000e-01],
       [ 2.54849751e-01,  2.97798054e-01,  3.15040137e-01],
       [ 4.40908772e-01,  1.81866065e-01,  2.72672637e-01],
       [ 2.02778098e-01,  2.77365879e-01,  4.03537863e-01],
       [ 2.43645056e-01,  3.02906061e-01,  3.22382355e-01],
       [ 2.15222188e-01,  3.63364398e-01,  2.96923439e-01],
       [ 2.62781328e-01,  2.40098017e-01,  3.71673098e-01],
       [ 2.70032342e-01,  3.62397665e-01,  2.40565609e-01],
       [ 2.77012394e-01,  2.60592519e-01,  3.30749660e-01],
       [ 2.70058466e-01,  3.32731335e-01,  2.65669868e-01],
       [ 2.40797754e-01,  2.86423621e-01,  3.43365888e-01],
       [ 2.31573171e-01,  2.77990662e-01,  3.64281781e-01],
       [ 3.17514720e-01,  2.41765667e-01,  3.09747751e-01],
       [ 3.52674175e-01,  2.55042696e-01,  2.63367755e-01],
       [ 3.57807016e-01,  2.72413168e-01,  2.42024392e-01],
       [ 3.12189207e-01,  2.15209847e-01,  3.46632795e-01],
       [ 2.21987141e-01,  4.20400176e-01,  2.43897354e-01],
       [ 2.36811781e-01,  2.93923646e-01,  3.39897072e-01],
       [ 3.07038969e-01,  2.22040546e-01,  3.43662710e-01],
       [ 3.50813227e-01,  2.44717133e-01,  2.75636649e-01],
       [ 2.28237559e-01,  3.18456451e-01,  3.24342820e-01],
       [ 2.35988434e-01,  3.57377137e-01,  2.79192351e-01],
       [ 2.19363509e-01,  3.17512358e-01,  3.35923788e-01],
       [ 2.33488702e-01,  3.00281728e-01,  3.37013066e-01],
       [ 2.62790117e-01,  3.05638774e-01,  2.98509019e-01],
       [ 2.99228302e-01,  2.63728087e-01,  3.03904700e-01],
       [ 4.23748451e-01,  2.26947739e-01,  2.36410247e-01],
       [ 3.43228637e-01,  2.85785017e-01,  2.41505725e-01],
       [ 2.41251942e-01,  2.28672055e-01,  4.14604115e-01],
       [ 3.37650816e-01,  2.37508840e-01,  2.95230976e-01],
       [ 3.12250323e-01,  2.75279100e-01,  2.79209868e-01],
       [ 3.16164619e-01,  2.69107792e-01,  2.81779278e-01],
       [ 2.92097623e-01,  2.95746744e-01,  2.78327204e-01],
       [ 3.79901083e-01,  2.53169402e-01,  2.42975583e-01],
       [ 1.95498650e-01,  4.85972070e-01,  2.27439143e-01],
       [ 2.66661488e-01,  3.67160941e-01,  2.39960452e-01],
       [ 3.67781244e-01,  2.54315710e-01,  2.51516184e-01],
       [ 2.60951166e-01,  2.50070861e-01,  3.61518109e-01],
       [ 3.00272069e-01,  3.19839280e-01,  2.48279922e-01],
       [ 2.28957768e-01,  2.48356530e-01,  4.04632428e-01],
       [ 3.19758361e-01,  2.51705834e-01,  2.96632679e-01],
       [ 3.88613880e-01,  2.28368698e-01,  2.61357063e-01],
       [ 2.94389401e-01,  3.21446466e-01,  2.52290547e-01],
       [ 2.58390428e-01,  2.60257990e-01,  3.52361677e-01],
       [ 3.54968689e-01,  2.70681352e-01,  2.46010685e-01],
       [ 2.26576548e-01,  3.37523661e-01,  3.07613881e-01],
       [ 4.32015776e-01, -2.31479592e-02,  6.35903031e-01],
       [ 2.99438414e-01,  2.77361527e-01,  2.89437103e-01],
       [ 2.31493276e-02,  3.85737047e-01,  5.89578087e-01],
       [ 2.70076638e-01,  3.57644009e-01,  2.44390095e-01],
       [ 3.29308986e-01,  2.24024535e-01,  3.18481894e-01],
       [ 2.34623823e-01,  3.42271329e-01,  2.94152224e-01],
       [ 3.10671448e-01,  3.44482413e-01,  2.18237514e-01],
       [ 2.34854262e-01,  2.96683849e-01,  3.39246669e-01],
       [ 2.96345248e-01,  2.64649941e-01,  3.05835506e-01],
       [ 3.06882112e-01,  2.98388703e-01,  2.61750345e-01],
       [ 2.56487764e-01,  4.23209740e-01,  2.08110048e-01],
       [ 3.04528416e-01,  2.32405197e-01,  3.33795481e-01],
       [ 4.30130917e-01,  2.50826113e-01,  2.08694422e-01],
       [ 2.65757860e-01,  2.75044045e-01,  3.27115865e-01],
       [ 2.99355278e-01,  3.01201446e-01,  2.66142649e-01],
       [ 3.03617039e-01,  2.69749859e-01,  2.93179371e-01],
       [ 2.78499622e-01,  3.00276862e-01,  2.87456405e-01],
       [ 3.20593017e-01,  3.31970417e-01,  2.20013250e-01],
       [ 3.73024349e-01,  1.29900783e-01,  4.00743236e-01],
       [ 3.70403508e-01,  2.30246323e-01,  2.74229586e-01],
       [ 2.56703372e-01,  3.75708417e-01,  2.42807590e-01],
       [ 2.66529903e-01,  2.48269076e-01,  3.57088640e-01],
       [ 2.89594907e-01,  2.79845304e-01,  2.96709333e-01],
       [ 2.00341708e-01,  2.84841064e-01,  3.97653145e-01],
       [ 2.57666664e-01,  3.05153755e-01,  3.04487975e-01],
       [ 2.85930033e-01,  3.83790426e-01,  2.09434830e-01],
       [ 2.92006686e-01,  3.14605406e-01,  2.60682436e-01],
       [ 3.13361332e-01,  2.96468886e-01,  2.57609595e-01],
       [ 3.10575783e-01,  2.20870580e-01,  3.41338203e-01],
       [ 1.30554384e-01,  5.00185788e-01,  2.92828269e-01],
       [ 2.63210922e-01,  3.21530522e-01,  2.82808166e-01],
       [ 5.18157129e-01,  2.06294381e-01,  1.97538543e-01],
       [ 3.31697672e-01,  2.21564632e-01,  3.19030460e-01],
       [ 2.36789281e-01,  3.28801610e-01,  3.04360106e-01],
       [ 3.48725475e-01,  2.80787592e-01,  2.41587064e-01],
       [ 2.97599923e-01,  3.39456582e-01,  2.33853271e-01],
       [ 3.40463298e-01,  2.32687312e-01,  2.97964456e-01],
       [ 3.08209736e-01,  2.97949419e-01,  2.60936566e-01],
       [ 2.91727578e-01,  2.71844833e-01,  3.02880982e-01],
       [ 2.24601071e-01,  2.53019692e-01,  4.04445370e-01],
       [ 2.00163089e-01,  2.40725543e-01,  4.57747415e-01],
       [ 2.50714238e-01,  3.28075769e-01,  2.89823133e-01],
       [ 3.07957482e-01,  2.54417015e-01,  3.05224326e-01],
       [ 2.92199083e-01,  3.13086446e-01,  2.61887027e-01],
       [ 3.01932453e-01,  3.06064860e-01,  2.59193721e-01],
       [ 2.66155039e-01,  3.04761161e-01,  2.95816237e-01],
       [ 2.96750410e-01,  2.77853601e-01,  2.91586600e-01],
       [ 1.95560868e-01,  4.08215361e-01,  2.81841253e-01],
       [ 2.67825472e-01,  2.65320749e-01,  3.35630899e-01],
       [ 2.52668881e-01,  3.40588519e-01,  2.76345276e-01],
       [ 2.28841069e-01,  3.43964858e-01,  2.99030974e-01],
       [ 2.82501447e-01,  3.20375304e-01,  2.64554095e-01],
       [ 3.55205689e-01,  2.58582601e-01,  2.57662115e-01],
       [ 3.56943578e-01,  2.46177790e-01,  2.68815578e-01],
       [ 3.04876998e-01,  2.66808173e-01,  2.95016092e-01],
       [ 3.07893718e-01,  2.59594200e-01,  2.99693740e-01],
       [ 3.63013922e-01,  1.84406667e-01,  3.34400672e-01],
       [ 2.11836665e-01,  3.03817872e-01,  3.60008924e-01],
       [ 2.28284848e-01,  3.86431168e-01,  2.63184324e-01],
       [ 1.82235049e-01,  3.49447394e-01,  3.50431803e-01],
       [ 3.11314653e-01,  2.30115126e-01,  3.29427372e-01],
       [ 2.90762053e-01,  3.71542318e-01,  2.14357022e-01],
       [ 2.25415702e-01,  3.19937770e-01,  3.26175613e-01],
       [ 3.10304349e-01,  2.31841751e-01,  3.28432679e-01],
       [ 2.28423429e-01,  3.01859405e-01,  3.41418254e-01],
       [ 2.33257192e-01,  2.91218431e-01,  3.47149035e-01],
       [ 3.04910941e-01,  2.59980527e-01,  3.02233443e-01],
       [ 2.97716387e-01,  2.98209007e-01,  2.70534354e-01],
       [ 2.11301612e-01,  2.61999602e-01,  4.11237190e-01],
       [ 1.70214456e-01,  2.80523338e-01,  4.48710704e-01],
       [ 3.20366657e-01,  2.99166299e-01,  2.48827913e-01],
       [ 2.84007224e-01,  2.90713077e-01,  2.91333690e-01],
       [ 3.19722050e-01,  2.53775341e-01,  2.94443236e-01],
       [ 3.19353998e-01,  2.63438566e-01,  2.84612143e-01],
       [ 2.87168128e-01,  2.87558041e-01,  2.91308287e-01],
       [ 2.91166135e-01,  2.66138312e-01,  3.09542723e-01],
       [-5.60301774e-02,  5.00658471e-01,  6.25358275e-01],
       [ 2.78301359e-01,  3.31513144e-01,  2.58667907e-01],
       [ 2.73984180e-01,  2.36386442e-01,  3.62939885e-01],
       [ 3.55437692e-01,  3.47306814e-01,  1.80085460e-01],
       [ 2.74827111e-01,  3.01348157e-01,  2.90157120e-01],
       [ 2.87702509e-01,  3.54545956e-01,  2.30434555e-01],
       [ 3.02628267e-01,  3.42396680e-01,  2.26938643e-01],
       [ 3.19302129e-01,  2.79159509e-01,  2.68795465e-01],
       [ 2.84875205e-01,  2.95874601e-01,  2.85342604e-01],
       [ 1.77685232e-01,  3.95752282e-01,  3.13339569e-01],
       [ 4.24666162e-01,  2.74440694e-01,  1.90892598e-01],
       [ 2.34703439e-01,  2.62206498e-01,  3.79262356e-01],
       [ 2.95253602e-01,  3.28688404e-01,  2.45140675e-01],
       [ 2.99333082e-01,  2.80888021e-01,  2.85961545e-01],
       [ 3.12943828e-01,  2.62448727e-01,  2.91745678e-01],
       [ 2.86910305e-01,  2.61987894e-01,  3.18516208e-01],
       [ 3.22041961e-01,  2.86928944e-01,  2.58792069e-01],
       [ 2.79853559e-01,  3.44252665e-01,  2.46207553e-01],
       [ 3.21868297e-01,  2.60078746e-01,  2.85745755e-01],
       [ 3.00788446e-01,  3.02200053e-01,  2.63855307e-01],
       [ 2.97618385e-01,  2.71075673e-01,  2.97739872e-01],
       [ 3.91219069e-01,  2.34807753e-01,  2.52606955e-01],
       [ 4.28485887e-01,  1.93807562e-01,  2.68291744e-01],
       [ 3.10334862e-01,  2.74905831e-01,  2.81400694e-01],
       [ 2.71124723e-01,  2.73051135e-01,  3.23368052e-01],
       [ 2.41498106e-01,  2.97308504e-01,  3.30731947e-01],
       [ 2.58309971e-01,  2.88292024e-01,  3.21131823e-01],
       [ 3.39579314e-01,  2.81646987e-01,  2.48473880e-01],
       [ 5.35629035e-01,  2.59912588e-01,  1.39255154e-01],
       [ 4.64837224e-01,  1.10821613e-01,  3.44797953e-01],
       [ 2.57029964e-01,  3.00058115e-01,  3.10320898e-01],
       [ 2.01420068e-01,  2.75404816e-01,  4.07965168e-01],
       [ 2.91958241e-01,  2.62320156e-01,  3.12863481e-01],
       [ 2.89091211e-01,  2.96374267e-01,  2.80666940e-01],
       [ 2.82720526e-01,  2.69934126e-01,  3.14272361e-01],
       [ 2.54935107e-01,  3.05913404e-01,  3.06699457e-01],
       [ 3.00611842e-01,  2.73533998e-01,  2.92212237e-01],
       [ 3.31604552e-01,  2.27994556e-01,  3.11644470e-01],
       [ 3.46618843e-01,  2.19729324e-01,  3.06944890e-01],
       [ 3.22678686e-01,  2.24922981e-01,  3.23998554e-01],
       [ 3.34526146e-01,  3.06772307e-01,  2.29809759e-01],
       [ 2.47084135e-01,  3.26955220e-01,  2.94778661e-01],
       [ 2.50042116e-01,  2.65420762e-01,  3.56249962e-01],
       [ 2.45191554e-01,  2.93654821e-01,  3.30332218e-01],
       [ 3.03349498e-01,  2.61056683e-01,  3.02634153e-01],
       [ 3.38545497e-01,  2.41138345e-01,  2.90440559e-01],
       [ 3.08834944e-01,  3.10849988e-01,  2.48511225e-01],
       [ 3.03958511e-01,  2.62314737e-01,  3.00680288e-01],
       [ 2.95449748e-01,  3.03734978e-01,  2.67466060e-01],
       [ 2.51441390e-01,  3.99598242e-01,  2.29670292e-01],
       [ 2.43935892e-01,  4.61370908e-01,  1.94886928e-01],
       [ 2.80236493e-01,  2.83379517e-01,  3.02664783e-01],
       [ 3.06484819e-01,  3.07349684e-01,  2.53817579e-01],
       [ 3.00365053e-01,  3.48813438e-01,  2.23711407e-01],
       [ 3.49964389e-01,  2.42028315e-01,  2.79224233e-01],
       [ 2.77688221e-01,  3.31751266e-01,  2.59052103e-01],
       [ 5.44316241e-01,  6.01170478e-01, -6.74183768e-02],
       [ 8.09549540e-02,  5.18118995e-01,  3.47295523e-01],
       [ 2.80655033e-01,  3.21449017e-01,  2.65375586e-01],
       [ 2.05090032e-01,  4.02469926e-01,  2.75622888e-01],
       [ 1.80914472e-01,  3.47873030e-01,  3.53761641e-01],
       [ 2.91050532e-01,  3.16126285e-01,  2.60206372e-01],
       [ 2.86997924e-01,  2.66170041e-01,  3.13846357e-01],
       [ 3.20337175e-01,  2.57497286e-01,  2.89899717e-01],
       [ 3.27497169e-01,  2.97445363e-01,  2.44162428e-01],
       [ 2.10696498e-01,  3.23178917e-01,  3.40729932e-01],
       [ 1.89198174e-01,  3.19380292e-01,  3.72752377e-01],
       [ 2.83129951e-01,  2.46304486e-01,  3.40483750e-01],
       [ 3.13916531e-01,  2.95451946e-01,  2.58058558e-01],
       [ 2.44197415e-01,  2.98313032e-01,  3.26542520e-01],
       [ 2.86669980e-01,  3.03905813e-01,  2.75797499e-01],
       [ 2.59700295e-01,  2.94699234e-01,  3.12890817e-01],
       [ 3.02709273e-01,  2.94473534e-01,  2.69364973e-01],
       [ 2.61111494e-01,  2.80792301e-01,  3.26039243e-01],
       [ 2.72777083e-01,  2.93480121e-01,  3.00120418e-01],
       [ 2.77665273e-01,  3.03216348e-01,  2.85440861e-01],
       [ 2.91478959e-01,  2.92157460e-01,  2.82439961e-01],
       [ 2.44832586e-01,  2.44012255e-01,  3.89198848e-01],
       [ 2.10853049e-01,  2.47189897e-01,  4.32010269e-01],
       [ 3.16137081e-01,  2.85923037e-01,  2.65104964e-01],
       [ 3.37710355e-01,  2.14963981e-01,  3.20992720e-01],
       [ 3.13242797e-01,  3.43692620e-01,  2.16673906e-01],
       [ 2.80577715e-01,  2.54822196e-01,  3.33400449e-01],
       [ 3.17082710e-01,  3.36697566e-01,  2.19093522e-01],
       [ 4.00696774e-01,  1.08246376e-01,  4.05990388e-01],
       [ 3.37825967e-01,  4.09125553e-01,  1.49657323e-01],
       [ 3.49552056e-01,  3.00235240e-01,  2.23230209e-01],
       [ 4.70015129e-01,  2.02397391e-01,  2.30320167e-01],
       [ 4.40474858e-01,  3.18412205e-01,  1.44616287e-01],
       [ 3.50337924e-01,  2.67763387e-01,  2.52696974e-01],
       [ 3.41678461e-01,  3.31820498e-01,  2.02857155e-01],
       [ 2.86166300e-01,  2.42377267e-01,  3.41768977e-01],
       [ 2.77939020e-01,  3.71145509e-01,  2.26232755e-01],
       [ 2.87080249e-01,  3.56370093e-01,  2.29533151e-01],
       [ 2.78124881e-01,  3.10182440e-01,  2.78307851e-01],
       [ 2.68922691e-01,  2.86567192e-01,  3.11320844e-01],
       [ 2.80158916e-01,  3.27347640e-01,  2.60557912e-01],
       [ 3.03051185e-01,  2.99645454e-01,  2.64133031e-01],
       [ 2.94855887e-01,  3.08983150e-01,  2.63140488e-01],
       [ 2.84451240e-01,  2.89723591e-01,  2.91875847e-01],
       [ 2.97724497e-01,  3.17097243e-01,  2.53069097e-01],
       [ 2.89445411e-01,  3.04300565e-01,  2.72711908e-01],
       [ 3.20519045e-01,  2.60705157e-01,  2.86359431e-01],
       [ 2.96117260e-01,  2.79694206e-01,  2.90334125e-01],
       [ 2.35909267e-01,  2.51908787e-01,  3.90662833e-01],
       [ 4.56792975e-01,  2.24411673e-01,  2.16513973e-01],
       [ 3.11977052e-01,  3.01894559e-01,  2.53824777e-01],
       [ 3.17817306e-01,  3.47589966e-01,  2.09690966e-01],
       [ 2.04205637e-01,  2.64488894e-01,  4.18161219e-01],
       [ 2.20523843e-01,  3.91030209e-01,  2.67790746e-01],
       [ 3.41483118e-01,  2.97981207e-01,  2.31825987e-01],
       [ 1.90096891e-01,  3.09344440e-01,  3.82816904e-01],
       [ 3.07990753e-01,  4.02395648e-01,  1.77460972e-01],
       [ 2.41968458e-01,  3.11554428e-01,  3.15458783e-01],
       [ 6.72656022e-01,  5.01277690e-01, -7.42694891e-02],
       [ 6.83578592e-02,  4.33862868e-01,  4.38735503e-01],
       [ 1.67710938e-01,  3.57587224e-01,  3.61753999e-01],
       [ 3.39266638e-01,  2.09141686e-01,  3.26481557e-01],
       [ 3.02028578e-01,  2.44671764e-01,  3.22118209e-01],
       [ 2.38838940e-01,  2.88027260e-01,  3.43935281e-01],
       [ 2.89773816e-01,  3.77796814e-01,  2.10501136e-01],
       [ 3.25036249e-01,  2.31372681e-01,  3.14149328e-01],
       [ 2.36195411e-01,  3.08826787e-01,  3.24861136e-01],
       [ 2.93866096e-01,  3.36724107e-01,  2.39534646e-01],
       [ 2.67013589e-01,  3.12938259e-01,  2.86991468e-01],
       [ 3.19604789e-01,  2.98552698e-01,  2.50067550e-01],
       [ 2.71546508e-01,  2.73984305e-01,  3.21889276e-01],
       [ 2.77481998e-01,  2.88087829e-01,  3.00689333e-01],
       [ 2.65318733e-01,  2.98189809e-01,  3.03251229e-01],
       [ 2.95216971e-01,  2.82003206e-01,  2.88880871e-01],
       [ 3.05470429e-01,  2.38571971e-01,  3.25568959e-01],
       [ 3.67119636e-01,  2.71996122e-01,  2.34925959e-01],
       [ 2.30719151e-01,  4.56880003e-01,  2.10280994e-01],
       [ 2.90343550e-01,  2.42759130e-01,  3.36738979e-01],
       [ 3.69403001e-01,  3.70073575e-01,  1.53207978e-01],
       [ 3.64439443e-01,  2.67270696e-01,  2.41560183e-01],
       [ 1.78006337e-01,  3.34766184e-01,  3.71333272e-01],
       [ 2.86401246e-01,  5.00588878e-01,  1.35491816e-01],
       [ 4.39830564e-01,  1.78709831e-01,  2.77100665e-01],
       [ 3.50390465e-01,  2.42255459e-01,  2.78608171e-01],
       [ 2.95348253e-01,  2.24138566e-01,  3.53812762e-01],
       [ 2.47097072e-01,  3.61337319e-01,  2.64144513e-01],
       [ 2.37073023e-01,  2.83426379e-01,  3.51215103e-01],
       [ 2.23667899e-01,  2.76054608e-01,  3.76719966e-01],
       [ 3.10347520e-01,  3.05918934e-01,  2.51610025e-01],
       [ 2.52512241e-01,  3.22780476e-01,  2.92883908e-01],
       [ 2.82175840e-01,  2.60279064e-01,  3.25475049e-01],
       [ 3.20192468e-01,  2.67579150e-01,  2.79569763e-01],
       [ 2.67542504e-01,  3.10517729e-01,  2.88764561e-01],
       [ 2.64919760e-01,  3.15366819e-01,  2.86846369e-01],
       [ 2.99874046e-01,  2.96776173e-01,  2.69847429e-01],
       [ 3.11746243e-01,  2.97460799e-01,  2.58151503e-01],
       [ 2.79781973e-01,  2.78884397e-01,  3.07828040e-01],
       [ 2.27119439e-01,  3.23581140e-01,  3.20516156e-01],
       [ 2.31973500e-01,  2.92595772e-01,  3.47190627e-01],
       [ 4.44053975e-01,  2.23791395e-01,  2.25538048e-01],
       [ 2.16531768e-01,  3.52493105e-01,  3.05213450e-01],
       [ 4.92099589e-01,  1.94726026e-01,  2.24475325e-01],
       [ 2.10819729e-01,  4.64699057e-01,  2.25060019e-01],
       [ 4.35294753e-01,  2.90435174e-01,  1.70276970e-01],
       [ 6.74383482e-01, -7.57667401e-02,  5.02985996e-01],
       [ 1.09167402e-01,  4.36145435e-01,  3.71139504e-01],
       [ 2.67636801e-01,  2.57138898e-01,  3.45252588e-01],
       [ 2.08509513e-01,  3.57855769e-01,  3.09665288e-01],
       [ 3.31223609e-01,  2.30347674e-01,  3.09316764e-01],
       [ 2.93509671e-01,  3.77591252e-01,  2.07380606e-01],
       [ 2.19059787e-01,  3.15296176e-01,  3.38596928e-01],
       [ 3.35530150e-01,  2.24882148e-01,  3.11458653e-01],
       [ 2.52425269e-01,  2.79531568e-01,  3.37319039e-01],
       [ 2.87797202e-01,  2.51998976e-01,  3.28782246e-01],
       [ 1.82792873e-01,  5.23986515e-01,  2.18199628e-01],
       [ 3.25583101e-01,  3.18846359e-01,  2.26850297e-01],
       [ 2.98159897e-01,  2.78736407e-01,  2.89292860e-01],
       [ 2.69604875e-01,  3.08046469e-01,  2.89013732e-01],
       [ 2.74014991e-01,  3.18565444e-01,  2.74575877e-01],
       [ 3.21503134e-01,  2.73847716e-01,  2.72036398e-01],
       [ 3.32710816e-01,  2.06313447e-01,  3.36454771e-01],
       [ 2.25910527e-01,  4.71061702e-01,  2.06008498e-01],
       [ 3.33499854e-01,  3.43731714e-01,  1.99880705e-01],
       [ 1.66685405e-01,  4.39432698e-01,  2.91614756e-01],
       [ 2.27554685e-01,  2.97961755e-01,  3.46701631e-01],
       [ 3.81956487e-01,  2.04305081e-01,  2.93323592e-01],
       [ 2.28030367e-01,  3.16866019e-01,  3.26199493e-01],
       [ 3.16461719e-01,  3.18027307e-01,  2.35396556e-01],
       [ 3.01808907e-01,  2.67129643e-01,  2.97708591e-01],
       [ 2.69028238e-01,  2.80945652e-01,  3.17138122e-01],
       [ 4.43941811e-01,  2.12985698e-02,  5.17033096e-01],
       [ 3.55679556e-01,  1.96066724e-01,  3.26713711e-01],
       [ 2.83364272e-01,  3.06745347e-01,  2.76354295e-01],
       [ 2.64983587e-01,  3.22850560e-01,  2.79755611e-01],
       [ 3.42969699e-01,  2.65109544e-01,  2.61603173e-01],
       [ 3.39273261e-01,  2.86168291e-01,  2.44484477e-01],
       [ 2.16567392e-01,  3.36634829e-01,  3.20128637e-01],
       [ 4.19965358e-01,  2.00224143e-01,  2.67519518e-01],
       [ 3.47889971e-01,  2.01443981e-01,  3.27523283e-01],
       [ 3.35160297e-01,  3.43532602e-01,  1.98707709e-01],
       [ 1.99349944e-01,  2.98423787e-01,  3.82722556e-01],
       [ 3.45927988e-01,  2.08203415e-01,  3.21181241e-01],
       [ 2.10950563e-01,  3.16839976e-01,  3.47036210e-01],
       [ 2.52787391e-01,  2.80467790e-01,  3.35864114e-01],
       [ 3.63768137e-01,  3.12445728e-01,  2.01625856e-01],
       [ 1.73371565e-01,  4.38629280e-01,  2.84238424e-01],
       [ 3.68394180e-01,  2.48381035e-01,  2.56978342e-01],
       [ 3.23887531e-01,  2.72746299e-01,  2.70954925e-01],
       [ 2.57990112e-01,  2.94926681e-01,  3.14535270e-01],
       [ 3.47071321e-01,  2.56873644e-01,  2.66326461e-01],
       [ 2.77297917e-01,  2.55953168e-01,  3.35723124e-01],
       [ 1.80083307e-01,  2.66372946e-01,  4.52520662e-01],
       [ 2.95440032e-01,  3.68189143e-01,  2.12802862e-01],
       [ 2.64494056e-01,  2.30026844e-01,  3.82510157e-01],
       [ 3.84404538e-01,  1.98220074e-01,  2.98310954e-01],
       [ 1.93888487e-01,  3.08265541e-01,  3.78829702e-01],
       [ 3.25987916e-01,  3.36083557e-01,  2.12123354e-01],
       [ 3.17375975e-01,  3.30526603e-01,  2.23951567e-01],
       [ 3.08295697e-01,  2.67247361e-01,  2.91218504e-01],
       [ 3.23599573e-01,  2.87149650e-01,  2.57189842e-01],
       [ 2.63950677e-01,  2.78078719e-01,  3.25814311e-01],
       [ 2.71090630e-01,  2.79897873e-01,  3.16018045e-01],
       [ 3.24477928e-01,  2.63181810e-01,  2.80099692e-01],
       [ 2.44477450e-01,  2.63784097e-01,  3.64990697e-01],
       [ 3.95652312e-01,  2.46741867e-01,  2.37200172e-01],
       [ 3.02887173e-01,  4.18037615e-01,  1.71143745e-01],
       [ 2.44707131e-01,  2.70525492e-01,  3.56733008e-01],
       [ 2.69508105e-01,  3.61521144e-01,  2.41775040e-01],
       [ 1.85833500e-01,  3.23914902e-01,  3.72351849e-01],
       [ 2.96323244e-01,  3.68789937e-01,  2.11571464e-01],
       [ 3.17212326e-01,  3.36301077e-01,  2.19308667e-01],
       [ 3.12555655e-01,  3.38283076e-01,  2.21664300e-01],
       [ 2.53324833e-01,  2.82107960e-01,  3.33440630e-01],
       [ 2.57512167e-01,  2.83622913e-01,  3.27023058e-01],
       [ 3.61057521e-01,  3.32835351e-01,  1.87099909e-01],
       [ 2.55573803e-01,  3.26205730e-01,  2.86414959e-01],
       [ 3.65452411e-01,  2.34778039e-01,  2.73561262e-01],
       [ 3.49722967e-01,  2.68277803e-01,  2.52713424e-01],
       [ 2.31228708e-01,  2.55969707e-01,  3.91652455e-01],
       [ 1.68099407e-01,  2.98465584e-01,  4.28296413e-01],
       [ 4.43088731e-01,  2.49652725e-01,  2.01203046e-01],
       [ 3.51428997e-01,  3.77794013e-01,  1.60763206e-01],
       [ 1.86253395e-01,  2.91847931e-01,  4.09206839e-01],
       [ 2.11328717e-01,  3.01139691e-01,  3.63652730e-01],
       [ 2.30826710e-01,  3.17505742e-01,  3.22270173e-01],
       [ 3.21580214e-01,  2.24855805e-01,  3.25181385e-01],
       [ 3.01768134e-01,  2.50803749e-01,  3.15461980e-01],
       [ 3.20580292e-01,  2.55305841e-01,  2.91991714e-01],
       [ 2.86572093e-01,  3.58402438e-01,  2.28368496e-01],
       [ 2.45571503e-01,  2.98911611e-01,  3.24336644e-01],
       [ 4.09463097e-01,  2.30153631e-01,  2.43521745e-01],
       [ 1.38027687e-01,  3.48661519e-01,  4.14792550e-01],
       [ 4.02105983e-01,  1.30059030e-01,  3.71505983e-01],
       [ 2.63881883e-01,  2.44628767e-01,  3.64686365e-01],
       [ 3.93360838e-01,  1.94881284e-01,  2.94676849e-01],
       [ 2.62538595e-01,  3.54338562e-01,  2.54463064e-01],
       [ 3.13655275e-01,  2.76324395e-01,  2.76839023e-01],
       [ 2.57296758e-01,  3.12222698e-01,  2.97911002e-01],
       [ 2.80635248e-01,  3.26038617e-01,  2.61264051e-01],
       [ 2.07912975e-01,  2.94484320e-01,  3.75744239e-01],
       [ 3.73426194e-01,  2.61216724e-01,  2.40221436e-01],
       [ 4.12731997e-01,  1.88094311e-01,  2.86884009e-01],
       [ 2.97981067e-01,  3.71999938e-01,  2.07694040e-01],
       [ 3.74113417e-01,  1.97294929e-01,  3.08342223e-01],
       [ 3.47313790e-01,  3.82694172e-01,  1.60388712e-01],
       [ 2.29481682e-01,  3.06600217e-01,  3.35099668e-01],
       [ 2.00244484e-01,  2.56309227e-01,  4.35163457e-01],
       [ 3.48423846e-01,  2.69339738e-01,  2.52775684e-01],
       [ 2.91591448e-01,  2.74395262e-01,  3.00340423e-01],
       [ 2.15374430e-01,  2.93755565e-01,  3.66767947e-01],
       [ 2.38316540e-01,  3.18154316e-01,  3.13005726e-01],
       [ 2.35612691e-01,  2.40030422e-01,  4.06703642e-01],
       [ 4.58663522e-01,  2.03825898e-01,  2.36249049e-01],
       [ 4.50349086e-01,  3.93472799e-01,  8.62739942e-02],
       [ 2.98715740e-01,  2.53381947e-01,  3.15724605e-01],
       [ 2.78636600e-01,  2.00181925e-01,  4.05627557e-01],
       [ 4.43912821e-01,  2.28889467e-01,  2.20559640e-01],
       [ 1.94618726e-01,  3.42545998e-01,  3.41299654e-01],
       [ 3.47705264e-01,  2.57461003e-01,  2.65182418e-01],
       [ 3.12226777e-01,  2.47605258e-01,  3.08469322e-01],
       [ 3.72572416e-01,  4.32659408e-01,  1.10282575e-01],
       [ 4.50518393e-01,  5.50072982e-01,  2.18071425e-03],
       [ 3.39436199e-01,  2.30936903e-01,  3.00876137e-01],
       [ 3.34034482e-01,  1.70896909e-01,  3.82060896e-01],
       [ 2.69135104e-01,  1.73307424e-01,  4.59623285e-01],
       [ 2.87817232e-01,  3.65112278e-01,  2.21944931e-01],
       [ 5.27713858e-01,  1.14092018e-01,  2.95715680e-01],
       [ 6.33056213e-01, -5.26317406e-02,  4.88123544e-01],
       [ 5.31340818e-02,  3.58134601e-01,  5.61605725e-01],
       [ 4.52489100e-01,  4.04088116e-01,  7.83986903e-02],
       [ 5.50465432e-01,  2.21884986e-01,  1.65546600e-01],
       [ 4.31616691e-01,  1.68672290e-01,  2.95188201e-01],
       [ 1.02379440e-01,  3.98398181e-01,  4.17774695e-01]])
        self.assertTrue(np.isclose(C, C_gt).all())

        


if __name__ == '__main__':
    unittest.main()