# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flask_sitemapper']

package_data = \
{'': ['*']}

install_requires = \
['flask>=2.2.2,<3.0.0', 'jinja2>=3.1.2,<4.0.0']

setup_kwargs = {
    'name': 'flask-sitemapper',
    'version': '1.5.0',
    'description': 'Flask extension for generating XML sitemaps',
    'long_description': '# Flask Sitemapper\n![GitHub Workflow Status](https://img.shields.io/github/workflow/status/h-janes/flask-sitemapper/Run%20Tests)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/flask-sitemapper?color=%23529AC3)\n\nFlask Sitemapper is a small Python 3 package that generates XML sitemaps for Flask applications. This allows you to create a fully functional sitemap for your project with minimal code, as demonstrated below. It is compatible with Flask blueprints.\n\nFor more information about sitemaps and the XML schema, visit [sitemaps.org](https://www.sitemaps.org).\n\n## Installation\nFlask Sitemapper requires Python 3.7 or newer. The Python packages `flask` and `jinja2` are also required, and will be installed automatically.\n\nThe latest version can be installed from PyPi with pip as shown below.\n```terminal\npip install flask-sitemapper\n```\n\n## Usage\n### Initializing Flask Sitemapper\nThe sitemapper must be initialized with the app instance as shown below.\n\nBy default, HTTPS will be used for all URLs in the sitemap. To change this, specify `https=False` when creating the `Sitemapper` instance.\n\n#### Recommended Method\n```python\nimport flask\nfrom flask_sitemapper import Sitemapper\n\n# create sitemapper instance\nsitemapper = Sitemapper()\n\n# create app\napp = flask.Flask("test_app")\n\n# initialize with app\nsitemapper.init_app(app)\n```\n\n#### Alternative Method\n```python\nimport flask\nfrom flask_sitemapper import Sitemapper\n\n# create app\napp = flask.Flask("test_app")\n\n# create instance and initialize with app\nsitemapper = Sitemapper(app)\n```\n\n### Adding URLs to the Sitemap\nDecorators are added to route functions to include their URLs in the sitemap. These must be included above the Flask decorators.\n```python\n# Define the homepage route and include it in the sitemap\n@sitemapper.include()\n@app.route("/")\ndef r_home():\n    return flask.render_template("index.html")\n```\n#### Additional Arguments\nYou can pass optional arguments to the decorator to include additional information in the sitemap. Valid arguments are listed below, as defined by the [sitemap protocol](https://www.sitemaps.org/protocol.html): \n\n* `lastmod` - The date on which this page was last modified as a string in [W3C datetime](https://www.w3.org/TR/NOTE-datetime) format. (e.g. YYYY-MM-DD)\n\n* `changefreq` - How frequently the page is likely to change. Acceptable values are:\n\n    * always\n    * hourly\n    * daily\n    * weekly\n    * monthly\n    * yearly\n    * never\n\n* `priority` - The priority of this URL relative to other URLs on your site as a string, float, or integer. Valid values range from 0.0 to 1.0\n\n```python\n@sitemapper.include(\n    lastmod = "2022-02-08",\n    changefreq = "monthly",\n    priority = 1.0,\n)\n@app.route("/about")\ndef r_about():\n    return flask.render_template("about.html")\n```\n\nThis example would appear in the sitemap as:\n```xml\n<url>\n  <loc>https://example.com/about</loc>\n  <lastmod>2022-02-08</lastmod>\n  <changefreq>monthly</changefreq>\n  <priority>1.0</priority>\n</url>\n```\n\n#### Notes\nFor routes where multiple URL paths are defined, the sitemap will only include the last path.\n```python\n@sitemapper.include()\n@app.route("/shop")  # This won\'t be included\n@app.route("/buy")  # This won\'t be included\n@app.route("/store")  # This will be included\ndef r_store():\n    return "<h1>Store Page</h1>"\n```\n\n### Using With Flask blueprints\n* Create your `Sitemapper` instance(s) in a separate file or otherwise avoiding circular imports.\n\n* Import and use the instance(s) in your blueprints.\n\n* Import the instance(s) when creating your flask app and initialize with `sitemapper.init_app(app)` ***after*** registering your blueprints.\n\nYou can also add Flask endpoints to the sitemap by using their endpoint name as shown below. Arguments can still be given after the endpoint name.\n```python\nsitemapper.add_endpoint("r_contact", lastmod="2022-02-09")\n```\n\n### Generating and Serving the Sitemap\nTo serve your sitemap, you must define a route function that returns `sitemapper.generate()`. Your sitemap will then be available at the URL(s) you specify.\n\nThis route should be defined after all routes that are included in the sitemap.\n```python\n@app.route("/sitemap.xml")\ndef r_sitemap():\n    return sitemapper.generate()\n```\n\n#### Using GZIP\nYou can serve your sitemap with GZIP compression by specifying this when calling the `generate` method. By default, GZIP will not be used.\n```python\n@app.route("/sitemap.xml")\ndef r_sitemap():\n    return sitemapper.generate(gzip=True)\n```\n\nThe sitemap will only be gzipped if the client accepts GZIP encoding. GZIP will also not be used if the response already has a `Content-Encoding` header, to avoid conflicts with other compression/encoding that you may be using.\n\n### Sitemap Indexes\nSitemap indexes are sitemaps that list other sitemaps. These are used if a single sitemap would be too large, or sometimes for organizational purposes. You can create a master sitemapper, which generates a sitemap index, by specifying `master=True` when creating the instance.\n\nNote that sitemap indexes have a different syntax to regular sitemaps, so it is important to provide this argument.\n```python\nmaster_sitemapper = Sitemapper(app, master=True)\n```\n\nYou can then decorate your sitemap route functions to add them to the sitemap index.\n```python\n@master_sitemapper.include()\n@app.route("/some_sitemap.xml")\ndef r_some_sitemap():\n    return some_sitemapper.generate()\n```\n\nOr add them with `add_endpoint`\n```python\n@master_sitemapper.add_endpoint("r_some_sitemap")\n```\n\nThen create the route for the sitemap index.\n```python\n@app.route("/sitemap.xml")\ndef r_sitemap_index():\n    return master_sitemapper.generate()\n```\n\nFor this example, the sitemap index would look like this:\n```xml\n<?xml version="1.0" encoding="utf-8"?>\n<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">\n  <sitemap>\n    <loc>https://example.com/some_sitemap.xml</loc>\n  </sitemap>\n</sitemapindex>\n```\n\n### Example App\n```python\nimport flask\nfrom flask_sitemapper import Sitemapper\n\nsitemapper = Sitemapper()\n\napp = flask.Flask("test_app")\n\nsitemapper.init_app(app)\n\n@sitemapper.include()\n@app.route("/")\ndef r_home():\n    return flask.render_template("index.html")\n\n@sitemapper.include(\n    lastmod = "2022-02-08",\n    changefreq = "monthly",\n    priority = 1.0,\n)\n@app.route("/about")\ndef r_about():\n    return flask.render_template("about.html")\n\n@app.route("/sitemap.xml")\ndef r_sitemap():\n    return sitemapper.generate()\n\napp.run()\n```\n\n## Development & Contributing\nThank you for contributing to the project! All issues and pull requests are highly appreciated.\n\n### Development Requirements\n* [Python](https://www.python.org) 3.7 or later\n* [Poetry](https://python-poetry.org) for package management and building\n\nThe following Python libraries will be installed automatically to a virtual environment by poetry during setup.\n* flask\n* jinja2\n* black\n* isort\n* pytest\n\n### Project Setup\nInstall `poetry` if not already installed.\n```terminal\npip install poetry\n```\n\nClone the repository and enter the repository\'s directory.\n```terminal\ngit clone https://github.com/h-janes/flask-sitemapper\ncd flask-sitemapper\n```\n\nCreate the poetry virtual environment and install dependencies.\n```terminal\npoetry install\n```\nYou may want to configure your editor to use the virtual environment for linting and debugging. You can find the path of the virtual environment with the command:\n```terminal\npoetry env info --path\n```\n\n### Code Style\nFlask Sitemapper uses `black` for code formatting and `isort` for import ordering.\n\nIt is recommended that you configure your editor to run `black` and `isort` on save, making sure that they access the configurations defined in `pyproject.toml`\n\nOtherwise, you should ensure that your code conforms to these style standards before submitting a pull request.\n\n### Testing\nTests are stored in the `tests` directory. You should ensure that all tests are passing before creating a pull request. If you create a pull request with new features, creating new tests for your additions is highly appreciated. You can run all tests with the command: \n```terminal\npoetry run pytest\n```\n\n### Building\nThe project can be built with the command:\n```terminal\npoetry build\n```\nThis should generate a `dist` directory containing a `.tar.gz` file and a `.whl` file.\n',
    'author': 'h-janes',
    'author_email': 'dev@hjanes.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
