import pytest

from estnltk_core import Span, Layer, ElementaryBaseSpan
from estnltk_core.converters import text_to_json, json_to_text, dict_to_layer
from estnltk_core.converters import annotation_to_json, json_to_annotation
from estnltk_core.tests import new_text

from estnltk_core.common import load_text_class

T_1 = "Tere, maailm!"
T_2 = '''Mis aias sa-das 2te sorti s-saia? Teine lause.

Teine lõik.'''


def test_json_export_import():
    # Load Text or BaseText class (depending on the available packages)
    Text = load_text_class()

    text = Text('')
    json_text = text_to_json(text)
    text_import = json_to_text(json_text)
    assert text_import == text
    assert json_text == text_to_json(text_import)

    text = Text(T_2)
    # tag_layer(['morph_analysis', 'paragraphs'])
    text.add_layer(dict_to_layer(
        {'name': 'words',
         'attributes': ('normalized_form',),
         'parent': None,
         'enveloping': None,
         'ambiguous': True,
         'serialisation_module': None,
         'meta': {},
         'spans': [{'base_span': (0, 3), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (4, 8), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (9, 15), 'annotations': [{'normalized_form': 'sadas'}]},
                   {'base_span': (16, 19), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (20, 25), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (26, 32), 'annotations': [{'normalized_form': 'saia'}]},
                   {'base_span': (32, 33), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (34, 39), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (40, 45), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (45, 46), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (48, 53), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (54, 58), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (58, 59), 'annotations': [{'normalized_form': None}]}]}
    ))
    text.add_layer(dict_to_layer({'name': 'sentences',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': 'words',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': ((0, 3),
                                                           (4, 8),
                                                           (9, 15),
                                                           (16, 19),
                                                           (20, 25),
                                                           (26, 32),
                                                           (32, 33)),
                                             'annotations': [{}]},
                                            {'base_span': ((34, 39), (40, 45), (45, 46)), 'annotations': [{}]},
                                            {'base_span': ((48, 53), (54, 58), (58, 59)), 'annotations': [{}]}]}))
    text.add_layer(dict_to_layer({'name': 'paragraphs',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': 'sentences',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (((0, 3),
                                                            (4, 8),
                                                            (9, 15),
                                                            (16, 19),
                                                            (20, 25),
                                                            (26, 32),
                                                            (32, 33)),
                                                           ((34, 39), (40, 45), (45, 46))),
                                             'annotations': [{}]},
                                            {'base_span': (((48, 53), (54, 58), (58, 59)),), 'annotations': [{}]}]}))

    text.add_layer(dict_to_layer({'name': 'tokens',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': None,
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3), 'annotations': [{}]},
                                            {'base_span': (4, 8), 'annotations': [{}]},
                                            {'base_span': (9, 11), 'annotations': [{}]},
                                            {'base_span': (11, 12), 'annotations': [{}]},
                                            {'base_span': (12, 15), 'annotations': [{}]},
                                            {'base_span': (16, 19), 'annotations': [{}]},
                                            {'base_span': (20, 25), 'annotations': [{}]},
                                            {'base_span': (26, 27), 'annotations': [{}]},
                                            {'base_span': (27, 28), 'annotations': [{}]},
                                            {'base_span': (28, 32), 'annotations': [{}]},
                                            {'base_span': (32, 33), 'annotations': [{}]},
                                            {'base_span': (34, 39), 'annotations': [{}]},
                                            {'base_span': (40, 45), 'annotations': [{}]},
                                            {'base_span': (45, 46), 'annotations': [{}]},
                                            {'base_span': (48, 53), 'annotations': [{}]},
                                            {'base_span': (54, 58), 'annotations': [{}]},
                                            {'base_span': (58, 59), 'annotations': [{}]}]}))
    text.add_layer(dict_to_layer({'name': 'compound_tokens',
                                  'attributes': ('type', 'normalized'),
                                  'parent': None,
                                  'enveloping': 'tokens',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': ((9, 11), (11, 12), (12, 15)),
                                             'annotations': [{'type': ['hyphenation'], 'normalized': 'sadas'}]},
                                            {'base_span': ((26, 27), (27, 28), (28, 32)),
                                             'annotations': [{'type': ['hyphenation'], 'normalized': 'saia'}]}]}))

    text.add_layer(dict_to_layer({'name': 'morph_analysis',
                                  'attributes': ('normalized_text',
                                                 'lemma',
                                                 'root',
                                                 'root_tokens',
                                                 'ending',
                                                 'clitic',
                                                 'form',
                                                 'partofspeech'),
                                  'parent': 'words',
                                  'enveloping': None,
                                  'ambiguous': True,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3),
                                             'annotations': [{'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'pl n',
                                                              'partofspeech': 'P'}]},
                                            {'base_span': (4, 8),
                                             'annotations': [{'normalized_text': 'aias',
                                                              'lemma': 'aed',
                                                              'root': 'aed',
                                                              'root_tokens': ['aed'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'sg in',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (9, 15),
                                             'annotations': [{'normalized_text': 'sadas',
                                                              'lemma': 'sadama',
                                                              'root': 'sada',
                                                              'root_tokens': ['sada'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 's',
                                                              'partofspeech': 'V'}]},
                                            {'base_span': (16, 19),
                                             'annotations': [{'normalized_text': '2te',
                                                              'lemma': '2',
                                                              'root': '2',
                                                              'root_tokens': ['2'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'N'}]},
                                            {'base_span': (20, 25),
                                             'annotations': [{'normalized_text': 'sorti',
                                                              'lemma': 'sort',
                                                              'root': 'sort',
                                                              'root_tokens': ['sort'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (26, 32),
                                             'annotations': [{'normalized_text': 'saia',
                                                              'lemma': 'sai',
                                                              'root': 'sai',
                                                              'root_tokens': ['sai'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (32, 33),
                                             'annotations': [{'normalized_text': '?',
                                                              'lemma': '?',
                                                              'root': '?',
                                                              'root_tokens': ['?'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]},
                                            {'base_span': (34, 39),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'O'}]},
                                            {'base_span': (40, 45),
                                             'annotations': [{'normalized_text': 'lause',
                                                              'lemma': 'lause',
                                                              'root': 'lause',
                                                              'root_tokens': ['lause'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (45, 46),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]},
                                            {'base_span': (48, 53),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'O'}]},
                                            {'base_span': (54, 58),
                                             'annotations': [{'normalized_text': 'lõik',
                                                              'lemma': 'lõik',
                                                              'root': 'lõik',
                                                              'root_tokens': ['lõik'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (58, 59),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]}]}))
    text.meta['year'] = 2017
    json_text = text_to_json(text)
    text_import = json_to_text(json_text)
    assert text_import == text, text.diff(text_import)
    # the following line randomly breaks in Python 3.5
    # assert json_text == text_to_json(text_import)

    text = Text(T_2)

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer({'name': 'tokens',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': None,
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3), 'annotations': [{}]},
                                            {'base_span': (4, 8), 'annotations': [{}]},
                                            {'base_span': (9, 11), 'annotations': [{}]},
                                            {'base_span': (11, 12), 'annotations': [{}]},
                                            {'base_span': (12, 15), 'annotations': [{}]},
                                            {'base_span': (16, 19), 'annotations': [{}]},
                                            {'base_span': (20, 25), 'annotations': [{}]},
                                            {'base_span': (26, 27), 'annotations': [{}]},
                                            {'base_span': (27, 28), 'annotations': [{}]},
                                            {'base_span': (28, 32), 'annotations': [{}]},
                                            {'base_span': (32, 33), 'annotations': [{}]},
                                            {'base_span': (34, 39), 'annotations': [{}]},
                                            {'base_span': (40, 45), 'annotations': [{}]},
                                            {'base_span': (45, 46), 'annotations': [{}]},
                                            {'base_span': (48, 53), 'annotations': [{}]},
                                            {'base_span': (54, 58), 'annotations': [{}]},
                                            {'base_span': (58, 59), 'annotations': [{}]}]}))

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer({'name': 'compound_tokens',
                                  'attributes': ('type', 'normalized'),
                                  'parent': None,
                                  'enveloping': 'tokens',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': ((9, 11), (11, 12), (12, 15)),
                                             'annotations': [{'type': ['hyphenation'], 'normalized': 'sadas'}]},
                                            {'base_span': ((26, 27), (27, 28), (28, 32)),
                                             'annotations': [{'type': ['hyphenation'], 'normalized': 'saia'}]}]}))

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer(
        {'name': 'words',
         'attributes': ('normalized_form',),
         'parent': None,
         'enveloping': None,
         'ambiguous': True,
         'serialisation_module': None,
         'meta': {},
         'spans': [{'base_span': (0, 3), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (4, 8), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (9, 15), 'annotations': [{'normalized_form': 'sadas'}]},
                   {'base_span': (16, 19), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (20, 25), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (26, 32), 'annotations': [{'normalized_form': 'saia'}]},
                   {'base_span': (32, 33), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (34, 39), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (40, 45), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (45, 46), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (48, 53), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (54, 58), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (58, 59), 'annotations': [{'normalized_form': None}]}]}
    ))

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer({'name': 'morph_analysis',
                                  'attributes': ('normalized_text',
                                                 'lemma',
                                                 'root',
                                                 'root_tokens',
                                                 'ending',
                                                 'clitic',
                                                 'form',
                                                 'partofspeech'),
                                  'parent': 'words',
                                  'enveloping': None,
                                  'ambiguous': True,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3),
                                             'annotations': [{'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'pl n',
                                                              'partofspeech': 'P'}]},
                                            {'base_span': (4, 8),
                                             'annotations': [{'normalized_text': 'aias',
                                                              'lemma': 'aed',
                                                              'root': 'aed',
                                                              'root_tokens': ['aed'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'sg in',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (9, 15),
                                             'annotations': [{'normalized_text': 'sadas',
                                                              'lemma': 'sadama',
                                                              'root': 'sada',
                                                              'root_tokens': ['sada'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 's',
                                                              'partofspeech': 'V'}]},
                                            {'base_span': (16, 19),
                                             'annotations': [{'normalized_text': '2te',
                                                              'lemma': '2',
                                                              'root': '2',
                                                              'root_tokens': ['2'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'N'}]},
                                            {'base_span': (20, 25),
                                             'annotations': [{'normalized_text': 'sorti',
                                                              'lemma': 'sort',
                                                              'root': 'sort',
                                                              'root_tokens': ['sort'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (26, 32),
                                             'annotations': [{'normalized_text': 'saia',
                                                              'lemma': 'sai',
                                                              'root': 'sai',
                                                              'root_tokens': ['sai'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (32, 33),
                                             'annotations': [{'normalized_text': '?',
                                                              'lemma': '?',
                                                              'root': '?',
                                                              'root_tokens': ['?'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]},
                                            {'base_span': (34, 39),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'O'}]},
                                            {'base_span': (40, 45),
                                             'annotations': [{'normalized_text': 'lause',
                                                              'lemma': 'lause',
                                                              'root': 'lause',
                                                              'root_tokens': ['lause'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (45, 46),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]},
                                            {'base_span': (48, 53),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'O'}]},
                                            {'base_span': (54, 58),
                                             'annotations': [{'normalized_text': 'lõik',
                                                              'lemma': 'lõik',
                                                              'root': 'lõik',
                                                              'root_tokens': ['lõik'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (58, 59),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]}]}))

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer({'name': 'sentences',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': 'words',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': ((0, 3),
                                                           (4, 8),
                                                           (9, 15),
                                                           (16, 19),
                                                           (20, 25),
                                                           (26, 32),
                                                           (32, 33)),
                                             'annotations': [{}]},
                                            {'base_span': ((34, 39), (40, 45), (45, 46)), 'annotations': [{}]},
                                            {'base_span': ((48, 53), (54, 58), (58, 59)), 'annotations': [{}]}]}))

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer({'name': 'paragraphs',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': 'sentences',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (((0, 3),
                                                            (4, 8),
                                                            (9, 15),
                                                            (16, 19),
                                                            (20, 25),
                                                            (26, 32),
                                                            (32, 33)),
                                                           ((34, 39), (40, 45), (45, 46))),
                                             'annotations': [{}]},
                                            {'base_span': (((48, 53), (54, 58), (58, 59)),), 'annotations': [{}]}]}))

    text = json_to_text(text_to_json(text))
    text.add_layer(dict_to_layer({'name': 'morph_extended',
                                  'attributes': ('normalized_text',
                                                 'lemma',
                                                 'root',
                                                 'root_tokens',
                                                 'ending',
                                                 'clitic',
                                                 'form',
                                                 'partofspeech',
                                                 'punctuation_type',
                                                 'pronoun_type',
                                                 'letter_case',
                                                 'fin',
                                                 'verb_extension_suffix',
                                                 'subcat'),
                                  'parent': 'morph_analysis',
                                  'enveloping': None,
                                  'ambiguous': True,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3),
                                             'annotations': [{'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg nom',
                                                              'partofspeech': 'P',
                                                              'punctuation_type': None,
                                                              'pronoun_type': ['inter_rel'],
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None},
                                                             {'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'pl nom',
                                                              'partofspeech': 'P',
                                                              'punctuation_type': None,
                                                              'pronoun_type': ['inter_rel'],
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (4, 8),
                                             'annotations': [{'normalized_text': 'aias',
                                                              'lemma': 'aed',
                                                              'root': 'aed',
                                                              'root_tokens': ['aed'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'com sg in',
                                                              'partofspeech': 'S',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (9, 15),
                                             'annotations': [{'normalized_text': 'sadas',
                                                              'lemma': 'sadama',
                                                              'root': 'sada',
                                                              'root_tokens': ['sada'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'mod indic impf ps3 sg ps af',
                                                              'partofspeech': 'V',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': True,
                                                              'verb_extension_suffix': [],
                                                              'subcat': ['Intr']},
                                                             {'normalized_text': 'sadas',
                                                              'lemma': 'sadama',
                                                              'root': 'sada',
                                                              'root_tokens': ['sada'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'aux indic impf ps3 sg ps af',
                                                              'partofspeech': 'V',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': True,
                                                              'verb_extension_suffix': [],
                                                              'subcat': ['Intr']},
                                                             {'normalized_text': 'sadas',
                                                              'lemma': 'sadama',
                                                              'root': 'sada',
                                                              'root_tokens': ['sada'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'main indic impf ps3 sg ps af',
                                                              'partofspeech': 'V',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': True,
                                                              'verb_extension_suffix': [],
                                                              'subcat': ['Intr']}]},
                                            {'base_span': (16, 19),
                                             'annotations': [{'normalized_text': '2te',
                                                              'lemma': '2',
                                                              'root': '2',
                                                              'root_tokens': ['2'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'card sg part l',
                                                              'partofspeech': 'N',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (20, 25),
                                             'annotations': [{'normalized_text': 'sorti',
                                                              'lemma': 'sort',
                                                              'root': 'sort',
                                                              'root_tokens': ['sort'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'com sg part',
                                                              'partofspeech': 'S',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (26, 32),
                                             'annotations': [{'normalized_text': 'saia',
                                                              'lemma': 'sai',
                                                              'root': 'sai',
                                                              'root_tokens': ['sai'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'com sg part',
                                                              'partofspeech': 'S',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (32, 33),
                                             'annotations': [{'normalized_text': '?',
                                                              'lemma': '?',
                                                              'root': '?',
                                                              'root_tokens': ['?'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z',
                                                              'punctuation_type': 'Int',
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (34, 39),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg nom',
                                                              'partofspeech': 'P',
                                                              'punctuation_type': None,
                                                              'pronoun_type': ['dem'],
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'ord sg nom roman',
                                                              'partofspeech': 'N',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'ord sg nom l',
                                                              'partofspeech': 'N',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (40, 45),
                                             'annotations': [{'normalized_text': 'lause',
                                                              'lemma': 'lause',
                                                              'root': 'lause',
                                                              'root_tokens': ['lause'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'com sg nom',
                                                              'partofspeech': 'S',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (45, 46),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z',
                                                              'punctuation_type': 'Fst',
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (48, 53),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg nom',
                                                              'partofspeech': 'P',
                                                              'punctuation_type': None,
                                                              'pronoun_type': ['dem'],
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'ord sg nom roman',
                                                              'partofspeech': 'N',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'ord sg nom l',
                                                              'partofspeech': 'N',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': 'cap',
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (54, 58),
                                             'annotations': [{'normalized_text': 'lõik',
                                                              'lemma': 'lõik',
                                                              'root': 'lõik',
                                                              'root_tokens': ['lõik'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'com sg nom',
                                                              'partofspeech': 'S',
                                                              'punctuation_type': None,
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]},
                                            {'base_span': (58, 59),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z',
                                                              'punctuation_type': 'Fst',
                                                              'pronoun_type': None,
                                                              'letter_case': None,
                                                              'fin': None,
                                                              'verb_extension_suffix': [],
                                                              'subcat': None}]}]}))

    text = json_to_text(text_to_json(text))
    text_2 = Text(T_2)
    text_2.add_layer(dict_to_layer(
        {'name': 'words',
         'attributes': ('normalized_form',),
         'parent': None,
         'enveloping': None,
         'ambiguous': True,
         'serialisation_module': None,
         'meta': {},
         'spans': [{'base_span': (0, 3), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (4, 8), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (9, 15), 'annotations': [{'normalized_form': 'sadas'}]},
                   {'base_span': (16, 19), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (20, 25), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (26, 32), 'annotations': [{'normalized_form': 'saia'}]},
                   {'base_span': (32, 33), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (34, 39), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (40, 45), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (45, 46), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (48, 53), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (54, 58), 'annotations': [{'normalized_form': None}]},
                   {'base_span': (58, 59), 'annotations': [{'normalized_form': None}]}]}
    ))
    text_2.add_layer(dict_to_layer({'name': 'sentences',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': 'words',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': ((0, 3),
                                                           (4, 8),
                                                           (9, 15),
                                                           (16, 19),
                                                           (20, 25),
                                                           (26, 32),
                                                           (32, 33)),
                                             'annotations': [{}]},
                                            {'base_span': ((34, 39), (40, 45), (45, 46)), 'annotations': [{}]},
                                            {'base_span': ((48, 53), (54, 58), (58, 59)), 'annotations': [{}]}]}))
    text_2.add_layer(dict_to_layer({'name': 'paragraphs',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': 'sentences',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (((0, 3),
                                                            (4, 8),
                                                            (9, 15),
                                                            (16, 19),
                                                            (20, 25),
                                                            (26, 32),
                                                            (32, 33)),
                                                           ((34, 39), (40, 45), (45, 46))),
                                             'annotations': [{}]},
                                            {'base_span': (((48, 53), (54, 58), (58, 59)),), 'annotations': [{}]}]}))

    text_2.add_layer(dict_to_layer({'name': 'tokens',
                                  'attributes': (),
                                  'parent': None,
                                  'enveloping': None,
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3), 'annotations': [{}]},
                                            {'base_span': (4, 8), 'annotations': [{}]},
                                            {'base_span': (9, 11), 'annotations': [{}]},
                                            {'base_span': (11, 12), 'annotations': [{}]},
                                            {'base_span': (12, 15), 'annotations': [{}]},
                                            {'base_span': (16, 19), 'annotations': [{}]},
                                            {'base_span': (20, 25), 'annotations': [{}]},
                                            {'base_span': (26, 27), 'annotations': [{}]},
                                            {'base_span': (27, 28), 'annotations': [{}]},
                                            {'base_span': (28, 32), 'annotations': [{}]},
                                            {'base_span': (32, 33), 'annotations': [{}]},
                                            {'base_span': (34, 39), 'annotations': [{}]},
                                            {'base_span': (40, 45), 'annotations': [{}]},
                                            {'base_span': (45, 46), 'annotations': [{}]},
                                            {'base_span': (48, 53), 'annotations': [{}]},
                                            {'base_span': (54, 58), 'annotations': [{}]},
                                            {'base_span': (58, 59), 'annotations': [{}]}]}))
    text_2.add_layer(dict_to_layer({'name': 'compound_tokens',
                                  'attributes': ('type', 'normalized'),
                                  'parent': None,
                                  'enveloping': 'tokens',
                                  'ambiguous': False,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': ((9, 11), (11, 12), (12, 15)),
                                             'annotations': [{'type': ['hyphenation'], 'normalized': 'sadas'}]},
                                            {'base_span': ((26, 27), (27, 28), (28, 32)),
                                             'annotations': [{'type': ['hyphenation'], 'normalized': 'saia'}]}]}))

    text_2.add_layer(dict_to_layer({'name': 'morph_analysis',
                                  'attributes': ('normalized_text',
                                                 'lemma',
                                                 'root',
                                                 'root_tokens',
                                                 'ending',
                                                 'clitic',
                                                 'form',
                                                 'partofspeech'),
                                  'parent': 'words',
                                  'enveloping': None,
                                  'ambiguous': True,
                                  'serialisation_module': None,
                                  'meta': {},
                                  'spans': [{'base_span': (0, 3),
                                             'annotations': [{'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Mis',
                                                              'lemma': 'mis',
                                                              'root': 'mis',
                                                              'root_tokens': ['mis'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'pl n',
                                                              'partofspeech': 'P'}]},
                                            {'base_span': (4, 8),
                                             'annotations': [{'normalized_text': 'aias',
                                                              'lemma': 'aed',
                                                              'root': 'aed',
                                                              'root_tokens': ['aed'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 'sg in',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (9, 15),
                                             'annotations': [{'normalized_text': 'sadas',
                                                              'lemma': 'sadama',
                                                              'root': 'sada',
                                                              'root_tokens': ['sada'],
                                                              'ending': 's',
                                                              'clitic': '',
                                                              'form': 's',
                                                              'partofspeech': 'V'}]},
                                            {'base_span': (16, 19),
                                             'annotations': [{'normalized_text': '2te',
                                                              'lemma': '2',
                                                              'root': '2',
                                                              'root_tokens': ['2'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'N'}]},
                                            {'base_span': (20, 25),
                                             'annotations': [{'normalized_text': 'sorti',
                                                              'lemma': 'sort',
                                                              'root': 'sort',
                                                              'root_tokens': ['sort'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (26, 32),
                                             'annotations': [{'normalized_text': 'saia',
                                                              'lemma': 'sai',
                                                              'root': 'sai',
                                                              'root_tokens': ['sai'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg p',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (32, 33),
                                             'annotations': [{'normalized_text': '?',
                                                              'lemma': '?',
                                                              'root': '?',
                                                              'root_tokens': ['?'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]},
                                            {'base_span': (34, 39),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'O'}]},
                                            {'base_span': (40, 45),
                                             'annotations': [{'normalized_text': 'lause',
                                                              'lemma': 'lause',
                                                              'root': 'lause',
                                                              'root_tokens': ['lause'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (45, 46),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]},
                                            {'base_span': (48, 53),
                                             'annotations': [{'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'P'},
                                                             {'normalized_text': 'Teine',
                                                              'lemma': 'teine',
                                                              'root': 'teine',
                                                              'root_tokens': ['teine'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'O'}]},
                                            {'base_span': (54, 58),
                                             'annotations': [{'normalized_text': 'lõik',
                                                              'lemma': 'lõik',
                                                              'root': 'lõik',
                                                              'root_tokens': ['lõik'],
                                                              'ending': '0',
                                                              'clitic': '',
                                                              'form': 'sg n',
                                                              'partofspeech': 'S'}]},
                                            {'base_span': (58, 59),
                                             'annotations': [{'normalized_text': '.',
                                                              'lemma': '.',
                                                              'root': '.',
                                                              'root_tokens': ['.'],
                                                              'ending': '',
                                                              'clitic': '',
                                                              'form': '',
                                                              'partofspeech': 'Z'}]}]}))

    text_2.add_layer(dict_to_layer({'name': 'morph_extended',
                                    'attributes': ('normalized_text',
                                                   'lemma',
                                                   'root',
                                                   'root_tokens',
                                                   'ending',
                                                   'clitic',
                                                   'form',
                                                   'partofspeech',
                                                   'punctuation_type',
                                                   'pronoun_type',
                                                   'letter_case',
                                                   'fin',
                                                   'verb_extension_suffix',
                                                   'subcat'),
                                    'parent': 'morph_analysis',
                                    'enveloping': None,
                                    'ambiguous': True,
                                    'serialisation_module': None,
                                    'meta': {},
                                    'spans': [{'base_span': (0, 3),
                                               'annotations': [{'normalized_text': 'Mis',
                                                                'lemma': 'mis',
                                                                'root': 'mis',
                                                                'root_tokens': ['mis'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'sg nom',
                                                                'partofspeech': 'P',
                                                                'punctuation_type': None,
                                                                'pronoun_type': ['inter_rel'],
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None},
                                                               {'normalized_text': 'Mis',
                                                                'lemma': 'mis',
                                                                'root': 'mis',
                                                                'root_tokens': ['mis'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'pl nom',
                                                                'partofspeech': 'P',
                                                                'punctuation_type': None,
                                                                'pronoun_type': ['inter_rel'],
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (4, 8),
                                               'annotations': [{'normalized_text': 'aias',
                                                                'lemma': 'aed',
                                                                'root': 'aed',
                                                                'root_tokens': ['aed'],
                                                                'ending': 's',
                                                                'clitic': '',
                                                                'form': 'com sg in',
                                                                'partofspeech': 'S',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (9, 15),
                                               'annotations': [{'normalized_text': 'sadas',
                                                                'lemma': 'sadama',
                                                                'root': 'sada',
                                                                'root_tokens': ['sada'],
                                                                'ending': 's',
                                                                'clitic': '',
                                                                'form': 'mod indic impf ps3 sg ps af',
                                                                'partofspeech': 'V',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': True,
                                                                'verb_extension_suffix': [],
                                                                'subcat': ['Intr']},
                                                               {'normalized_text': 'sadas',
                                                                'lemma': 'sadama',
                                                                'root': 'sada',
                                                                'root_tokens': ['sada'],
                                                                'ending': 's',
                                                                'clitic': '',
                                                                'form': 'aux indic impf ps3 sg ps af',
                                                                'partofspeech': 'V',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': True,
                                                                'verb_extension_suffix': [],
                                                                'subcat': ['Intr']},
                                                               {'normalized_text': 'sadas',
                                                                'lemma': 'sadama',
                                                                'root': 'sada',
                                                                'root_tokens': ['sada'],
                                                                'ending': 's',
                                                                'clitic': '',
                                                                'form': 'main indic impf ps3 sg ps af',
                                                                'partofspeech': 'V',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': True,
                                                                'verb_extension_suffix': [],
                                                                'subcat': ['Intr']}]},
                                              {'base_span': (16, 19),
                                               'annotations': [{'normalized_text': '2te',
                                                                'lemma': '2',
                                                                'root': '2',
                                                                'root_tokens': ['2'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'card sg part l',
                                                                'partofspeech': 'N',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (20, 25),
                                               'annotations': [{'normalized_text': 'sorti',
                                                                'lemma': 'sort',
                                                                'root': 'sort',
                                                                'root_tokens': ['sort'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'com sg part',
                                                                'partofspeech': 'S',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (26, 32),
                                               'annotations': [{'normalized_text': 'saia',
                                                                'lemma': 'sai',
                                                                'root': 'sai',
                                                                'root_tokens': ['sai'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'com sg part',
                                                                'partofspeech': 'S',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (32, 33),
                                               'annotations': [{'normalized_text': '?',
                                                                'lemma': '?',
                                                                'root': '?',
                                                                'root_tokens': ['?'],
                                                                'ending': '',
                                                                'clitic': '',
                                                                'form': '',
                                                                'partofspeech': 'Z',
                                                                'punctuation_type': 'Int',
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (34, 39),
                                               'annotations': [{'normalized_text': 'Teine',
                                                                'lemma': 'teine',
                                                                'root': 'teine',
                                                                'root_tokens': ['teine'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'sg nom',
                                                                'partofspeech': 'P',
                                                                'punctuation_type': None,
                                                                'pronoun_type': ['dem'],
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None},
                                                               {'normalized_text': 'Teine',
                                                                'lemma': 'teine',
                                                                'root': 'teine',
                                                                'root_tokens': ['teine'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'ord sg nom roman',
                                                                'partofspeech': 'N',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None},
                                                               {'normalized_text': 'Teine',
                                                                'lemma': 'teine',
                                                                'root': 'teine',
                                                                'root_tokens': ['teine'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'ord sg nom l',
                                                                'partofspeech': 'N',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (40, 45),
                                               'annotations': [{'normalized_text': 'lause',
                                                                'lemma': 'lause',
                                                                'root': 'lause',
                                                                'root_tokens': ['lause'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'com sg nom',
                                                                'partofspeech': 'S',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (45, 46),
                                               'annotations': [{'normalized_text': '.',
                                                                'lemma': '.',
                                                                'root': '.',
                                                                'root_tokens': ['.'],
                                                                'ending': '',
                                                                'clitic': '',
                                                                'form': '',
                                                                'partofspeech': 'Z',
                                                                'punctuation_type': 'Fst',
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (48, 53),
                                               'annotations': [{'normalized_text': 'Teine',
                                                                'lemma': 'teine',
                                                                'root': 'teine',
                                                                'root_tokens': ['teine'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'sg nom',
                                                                'partofspeech': 'P',
                                                                'punctuation_type': None,
                                                                'pronoun_type': ['dem'],
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None},
                                                               {'normalized_text': 'Teine',
                                                                'lemma': 'teine',
                                                                'root': 'teine',
                                                                'root_tokens': ['teine'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'ord sg nom roman',
                                                                'partofspeech': 'N',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None},
                                                               {'normalized_text': 'Teine',
                                                                'lemma': 'teine',
                                                                'root': 'teine',
                                                                'root_tokens': ['teine'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'ord sg nom l',
                                                                'partofspeech': 'N',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': 'cap',
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (54, 58),
                                               'annotations': [{'normalized_text': 'lõik',
                                                                'lemma': 'lõik',
                                                                'root': 'lõik',
                                                                'root_tokens': ['lõik'],
                                                                'ending': '0',
                                                                'clitic': '',
                                                                'form': 'com sg nom',
                                                                'partofspeech': 'S',
                                                                'punctuation_type': None,
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]},
                                              {'base_span': (58, 59),
                                               'annotations': [{'normalized_text': '.',
                                                                'lemma': '.',
                                                                'root': '.',
                                                                'root_tokens': ['.'],
                                                                'ending': '',
                                                                'clitic': '',
                                                                'form': '',
                                                                'partofspeech': 'Z',
                                                                'punctuation_type': 'Fst',
                                                                'pronoun_type': None,
                                                                'letter_case': None,
                                                                'fin': None,
                                                                'verb_extension_suffix': [],
                                                                'subcat': None}]}]}))
    assert text == text_2


def test_annotation_json_export_import():
    layer = Layer('my_layer', attributes=['attr', 'attr_0'])
    span = Span(base_span=ElementaryBaseSpan(0, 1), layer=layer)

    annotation = new_text(5)['layer_0'][0].annotations[0]

    a = json_to_annotation(span, annotation_to_json(annotation))
    assert a == annotation
