'''ddlsplit

Splits a DDL file generated by pgModeler
into single files by database objects (schema, extension, table etc.).
'''

import pathlib
import re
import sys


regex = {
    # A single Database Object.
    'object': r'(?s)-- object: .*?\-- ddl-end --\n{2}',
    # Name and Type of Database Object.
    'info': r'-- object: (\S+) \| type: (\S+) --',
    # Comment On Table.
    'tbl_comment': '(?<=comment on table)\s*(\w*)\.(\w*)\s*IS.*\'(.*)\';',
}

# Compile Regex.
obj = re.compile(regex['object'], re.IGNORECASE | re.MULTILINE)
obj_info = re.compile(regex['info'], re.IGNORECASE)
obj_comment = re.compile(regex['tbl_comment'], re.IGNORECASE)


class Ddl(object):
    '''Accept Database Object as DDL.'''

    def __init__(self, db_obj):
        self.ddl = db_obj
        self.name = self.extract_name(self.ddl)
        self.type = self.extract_type(self.ddl)
        self.top_block_comment = self.extract_tbl_comment(self.ddl)

    def extract_name(self, ddl):

        return obj_info.match(ddl).group(1).lower().replace('"', '')

    def extract_type(self, ddl):

        return obj_info.match(ddl).group(2).lower()

    def extract_tbl_comment(self, ddl):
        '''Extract Comment On Table from DDL code.
        Parse escape sequences in string as multiline string.
        '''

        raw_comment = obj_comment.search(ddl)
        if raw_comment:
            return raw_comment.group(3).\
                        replace('\\n', '\n').\
                        replace('\\t', '    ')
        else:
            return None


def read_file(ddl_file):
    '''Return individual DDL objects as list.'''

    with open(ddl_file) as f:
        ddl_code = f.read()

        return obj.findall(ddl_code)

def write_file(db_obj, outdir):
    '''Write individual Database Object as individual file.'''

    filename = f'''{db_obj.type}_{db_obj.name}.sql'''
    filepath = outdir.joinpath(filename).resolve()
    with filepath.open('w', encoding ='utf-8') as sql:
        if db_obj.top_block_comment:
            # Prepend block comment ahead DDL.
            sql.write('/*\n')
            sql.write(db_obj.top_block_comment)
            sql.write('\n*/\n')
        # Write actual DDL command.
        sql.write(db_obj.ddl)

        return filepath

def run(ddl_file, outdir):
    db_objs = read_file(ddl_file)
    for db_obj in db_objs:
        file = write_file(Ddl(db_obj), outdir)
        print(file)

def main():
    ddl_file = pathlib.Path(sys.argv[1]).resolve()
    outdir = pathlib.Path(sys.argv[2]).resolve()
    outdir.mkdir(parents=True, exist_ok=True)
    run(ddl_file, outdir)


if __name__ == '__main__':
    main()
