"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms.
 *
 * @stability stable
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
    ComparisonOperator["GREATER_THAN_UPPER_THRESHOLD"] = "GreaterThanUpperThreshold";
    ComparisonOperator["LESS_THAN_LOWER_THRESHOLD"] = "LessThanLowerThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '<=',
};
/**
 * Specify how missing data points are treated during alarm evaluation.
 *
 * @stability stable
 */
var TreatMissingData;
(function (TreatMissingData) {
    TreatMissingData["BREACHING"] = "breaching";
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    TreatMissingData["IGNORE"] = "ignore";
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric.
 *
 * @stability stable
 */
class Alarm extends alarm_base_1.AlarmBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.list({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.list({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.list({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // eslint-disable-next-line max-len
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo).
     * @stability stable
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Turn this alarm into a horizontal annotation.
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *    you have both a "small margin/long period" alarm as well as a
     *    "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     *
     * @stability stable
     */
    toAnnotation() {
        return this.annotation;
    }
    /**
     * Trigger this action if the alarm fires.
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     *
     * @stability stable
     */
    addAlarmAction(...actions) {
        if (this.alarmActionArns === undefined) {
            this.alarmActionArns = [];
        }
        this.alarmActionArns.push(...actions.map(a => this.validateActionArn(a.bind(this, this).alarmActionArn)));
    }
    validateActionArn(actionArn) {
        var _b, _c, _d;
        const ec2ActionsRegexp = /arn:aws:automate:[a-z|\d|-]+:ec2:[a-z]+/;
        if (ec2ActionsRegexp.test(actionArn)) {
            // Check per-instance metric
            const metricConfig = this.metric.toMetricConfig();
            if (((_c = (_b = metricConfig.metricStat) === null || _b === void 0 ? void 0 : _b.dimensions) === null || _c === void 0 ? void 0 : _c.length) != 1 || ((_d = metricConfig.metricStat) === null || _d === void 0 ? void 0 : _d.dimensions[0].name) != 'InstanceId') {
                throw new Error(`EC2 alarm actions requires an EC2 Per-Instance Metric. (${JSON.stringify(metricConfig)} does not have an 'InstanceId' dimension)`);
            }
        }
        return actionArn;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(stat, conf) {
                var _b, _c, _d;
                self.validateMetricStat(stat, metric);
                if (((_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label) == undefined) {
                    return object_1.dropUndefined({
                        dimensions: stat.dimensions,
                        namespace: stat.namespace,
                        metricName: stat.metricName,
                        period: (_c = stat.period) === null || _c === void 0 ? void 0 : _c.toSeconds(),
                        statistic: renderIfSimpleStatistic(stat.statistic),
                        extendedStatistic: renderIfExtendedStatistic(stat.statistic),
                        unit: stat.unitFilter,
                    });
                }
                return {
                    metrics: [
                        {
                            metricStat: {
                                metric: {
                                    metricName: stat.metricName,
                                    namespace: stat.namespace,
                                    dimensions: stat.dimensions,
                                },
                                period: stat.period.toSeconds(),
                                stat: stat.statistic,
                                unit: stat.unitFilter,
                            },
                            id: 'm1',
                            label: (_d = conf.renderingProperties) === null || _d === void 0 ? void 0 : _d.label,
                            returnData: true,
                        },
                    ],
                };
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _b;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _b;
                            const hasSubmetrics = mathExprHasSubmetrics(expr);
                            if (hasSubmetrics) {
                                assertSubmetricsCount(expr);
                            }
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label,
                                period: hasSubmetrics ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
_a = JSII_RTTI_SYMBOL_1;
Alarm[_a] = { fqn: "@aws-cdk/aws-cloudwatch.Alarm", version: "1.108.1" };
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
function assertSubmetricsCount(expr) {
    if (Object.keys(expr.usingMetrics).length > 10) {
        // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html#alarms-on-metric-math-expressions
        throw new Error('Alarms on math expressions cannot contain more than 10 individual metrics');
    }
    ;
}
//# sourceMappingURL=data:application/json;base64,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