from starkcore.utils.api import from_api_json
from starkcore.utils.resource import Resource
from starkcore.utils.checks import check_datetime, check_date
from ...utils import rest
from ..__issuinginvoice import _resource as _issuing_invoice_resource


class Log(Resource):
    """# issuinginvoice.Log object
    Every time an IssuingInvoice entity is updated, a corresponding issuinginvoice.Log
    is generated for the entity. This log is never generated by the
    user, but it can be retrieved to check additional information
    on the IssuingInvoice.
    ## Attributes:
    - id [string]: unique id returned when the log is created. ex: "5656565656565656"
    - invoice [IssuingInvoice]: IssuingInvoice entity to which the log refers to.
    - errors [list of strings]: list of errors linked to this IssuingInvoice event
    - type [string]: type of the IssuingInvoice event which triggered the log creation. ex: "created" or "paid"
    - created [datetime.datetime]: creation datetime for the log. ex: datetime.datetime(2020, 3, 10, 10, 30, 0, 0)
    """

    def __init__(self, id, created, type, invoice):
        Resource.__init__(self, id=id)

        self.type = type
        self.invoice = from_api_json(_issuing_invoice_resource, invoice)
        self.created = check_datetime(created)


_resource = {"class": Log, "name": "IssuingInvoiceLog"}


def query(ids=None, types=None, after=None, before=None, limit=None, user=None):
    """# Retrieve issuinginvoice.Log
    Receive a generator of issuinginvoice.Log objects previously created in the Stark Infra API
    ## Parameters (optional):
    - limit [integer, default 100]: maximum number of objects to be retrieved. Unlimited if None. ex: 35
    - after [datetime.date or string, default None] date filter for objects created only after specified date. ex: datetime.date(2020, 3, 10)
    - before [datetime.date or string, default None] date filter for objects created only before specified date. ex: datetime.date(2020, 3, 10)
    - types [list of strings, default None]: filter for log event types. ex: ["created", "paid"]
    - ids [list of strings, default None]: list of IssuingInvoice ids to filter logs. ex: ["5656565656565656", "4545454545454545"]
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call
    ## Return:
    - generator of issuinginvoice.Log objects with updated attributes
    """
    return rest.get_stream(
        resource=_resource,
        ids=ids,
        limit=limit,
        after=check_date(after),
        before=check_date(before),
        types=types,
        user=user,
    )


def page(ids=None, types=None, after=None, before=None, cursor=None, limit=None, user=None):
    """# Retrieve paged issuinginvoice.Log
    Receive a list of up to 100 issuinginvoice.Log objects previously created in the Stark Infra API and the cursor to the next page.
    Use this function instead of query if you want to manually page your requests.
    ## Parameters (optional):
    - cursor [string, default None]: cursor returned on the previous page function call
    - limit [integer, default 100]: maximum number of objects to be retrieved. It must be an integer between 1 and 100. ex: 50
    - after [datetime.date or string, default None] date filter for objects created only after specified date. ex: datetime.date(2020, 3, 10)
    - before [datetime.date or string, default None] date filter for objects created only before specified date. ex: datetime.date(2020, 3, 10)
    - types [list of strings, default None]: filter for log event types. ex: ["created", "paid"]
    - ids [list of strings, default None]: list of IssuingInvoice ids to filter logs. ex: ["5656565656565656", "4545454545454545"]
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call
    ## Return:
    - list of issuinginvoice.Log objects with updated attributes
    - cursor to retrieve the next page of issuinginvoice.Log objects
    """
    return rest.get_page(
        resource=_resource,
        ids=ids,
        limit=limit,
        after=check_date(after),
        before=check_date(before),
        types=types,
        cursor=cursor,
        user=user,
    )
