"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const assertions_1 = require("aws-cdk-lib/assertions");
const test_helper_1 = require("./test-helper");
const ec2 = require("aws-cdk-lib/aws-ec2");
test('New service/new bucket, public API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const familyName = "family-name";
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const construct = new lib_1.FargateToKinesisFirehose(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.kinesisFirehose).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: familyName
    });
    template.hasResourceProperties("AWS::ECS::Cluster", {
        ClusterName: clusterName
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    // Confirm we created a Public/Private VPC
    template.hasResourceProperties('AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('New service/new bucket, private API, new VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        existingKinesisFirehose: destination.kinesisFirehose
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
    // Confirm we created an Isolated VPC
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::S3::Bucket', 2);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const construct = new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.kinesisFirehose).toBeDefined();
});
test('Construct deploys Fargate Service in isolated subnets when publicApi is set to false', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    const template = assertions_1.Template.fromStack(stack);
    // The default isolated VPC should have two subnets, 2 route tables, and no nat/internet gateway, or routes
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::EC2::Subnet', 2);
    template.resourceCountIs('AWS::EC2::RouteTable', 2);
    template.resourceCountIs('AWS::EC2::Route', 0);
    template.resourceCountIs('AWS::EC2::NatGateway', 0);
    template.resourceCountIs('AWS::EC2::InternetGateway', 0);
    const subnet1Resources = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            Tags: assertions_1.Match.arrayWith([
                {
                    Key: "aws-cdk:subnet-type",
                    Value: "Isolated"
                }
            ])
        }
    });
    const [subnet1, subnet2] = Object.keys(subnet1Resources);
    // Verify the Fargate Service is deployed into the two isolated subnets
    template.hasResourceProperties('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: subnet1
                    },
                    {
                        Ref: subnet2
                    }
                ]
            },
        }
    });
});
test('Construct deploys Fargate Service in private subnets when publicApi is set to true', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose
    });
    // The default public/private VPC should have 4 subnets (two public, two private)
    // 4 route tables, 4 routes, 2 NAT Gateways and 1 Internet Gateway
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::EC2::Subnet', 4);
    template.resourceCountIs('AWS::EC2::RouteTable', 4);
    template.resourceCountIs('AWS::EC2::Route', 4);
    template.resourceCountIs('AWS::EC2::NatGateway', 2);
    template.resourceCountIs('AWS::EC2::InternetGateway', 1);
    const subnet1Resources = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            Tags: assertions_1.Match.arrayWith([
                {
                    Key: "aws-cdk:subnet-type",
                    Value: "Private"
                }
            ])
        }
    });
    const [privateSubnet1, privateSubnet2] = Object.keys(subnet1Resources);
    // Verify the Fargate Service is deployed into the two private subnets
    template.hasResourceProperties('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: privateSubnet1
                    },
                    {
                        Ref: privateSubnet2
                    }
                ]
            },
        }
    });
});
test('Construct uses vpcProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose,
        vpcProps: {
            vpcName: 'my-vpc'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.hasResourceProperties('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct uses existingVpc when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const existingVpc = defaults.getTestVpc(stack, false, {
        vpcName: 'my-vpc'
    });
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingKinesisFirehose: destination.kinesisFirehose,
        existingVpc
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.hasResourceProperties('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct creates VPC Interface Endpoints for ECR and Kinesis Streams', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-firehose'
                ]
            ]
        },
    });
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.ecr.api'
                ]
            ]
        },
    });
});
test('Container has default stream name environment variable', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'FIREHOSE_DELIVERYSTREAM_NAME',
                        Value: destination.kinesisFirehose.deliveryStreamName
                    }
                ],
            }
        ]
    });
});
test('Container stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        firehoseEnvironmentVariableName: 'my-stream-name'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'my-stream-name',
                        Value: destination.kinesisFirehose.deliveryStreamName
                    }
                ],
            }
        ]
    });
});
test('Construct grants PutRecord permission for the Fargate Service to write to the Kinesis Stream', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "firehose:DeleteDeliveryStream",
                        "firehose:PutRecord",
                        "firehose:PutRecordBatch",
                        "firehose:UpdateDestination"
                    ],
                    Effect: 'Allow',
                    Resource: {
                        "Fn::GetAtt": [
                            "testdestinationKinesisFirehose5F7F1053",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Construct defaults to the latest version of the ECR image', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:latest'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses ecrImageVersion when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        ecrImageVersion: 'my-version'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:my-version'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses clusterProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: {
            clusterName: 'my-cluster'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::Cluster', 1);
    template.hasResourceProperties("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
});
test('Construct uses containerDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        containerDefinitionProps: {
            containerName: 'my-container'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::TaskDefinition', 1);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
});
test('Construct uses fargateTaskDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateTaskDefinitionProps: {
            family: 'my-family'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::TaskDefinition', 1);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: 'my-family'
    });
});
test('Construct uses fargateServiceProps when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingKinesisFirehose: destination.kinesisFirehose,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateServiceProps: {
            serviceName: 'my-service',
            desiredCount: 7
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::Service', 1);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: 'my-service',
        DesiredCount: 7
    });
});
test('Construct uses existingFargateServiceObject when provided', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test-existing-fargate-service', {
        constructVpc: existingVpc,
        clientClusterProps: { clusterName: 'my-cluster' },
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clientFargateTaskDefinitionProps: {
            family: 'my-family'
        },
        clientContainerDefinitionProps: {
            containerName: 'my-container'
        },
        clientFargateServiceProps: {
            serviceName: 'my-service'
        }
    });
    new lib_1.FargateToKinesisFirehose(stack, 'test-fargate-kinesisfirehose', {
        publicApi: false,
        existingVpc,
        existingKinesisFirehose: destination.kinesisFirehose,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::Cluster', 1);
    template.hasResourceProperties("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
    template.resourceCountIs('AWS::ECS::TaskDefinition', 1);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: 'my-family',
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
    template.resourceCountIs('AWS::ECS::Service', 1);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: 'my-service',
    });
});
test('Test fail if existingFirehose does not have a stream name', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const mutatedFirehose = defaults.overrideProps(destination.kinesisFirehose, {});
    mutatedFirehose.deliveryStreamName = undefined;
    const app = () => {
        new lib_1.FargateToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: mutatedFirehose,
            publicApi: false,
            ecrRepositoryArn: defaults.fakeEcrRepoArn,
        });
    };
    expect(app).toThrowError(/existingKinesisFirehose must have a defined deliveryStreamName/);
});
test('Confirm that CheckVpcProps was called', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const familyName = "custom-family-name";
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingKinesisFirehose: destination.kinesisFirehose,
        existingVpc: defaults.getTestVpc(stack),
        vpcProps: {},
    };
    const app = () => {
        new lib_1.FargateToKinesisFirehose(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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