# -*- coding: utf-8 -*-
"""Command line interface for Axonius API Client."""
import tabulate

from ... import constants
from ..context import click
from ..options import (AUTH, EXPORT, FIELDS_SELECT, PAGING, QUERY, SQ_NAME,
                       get_option_fields_default, get_option_help)

FIELDS_GET = [
    *AUTH,
    click.option(
        "--use-cursor/--no-use-cursor",
        "use_cursor",
        default=True,
        help="Use cursor for pagination",
        hidden=True,
    ),
    click.option(
        "--adapter-re",
        "-ar",
        "adapter_re",
        default=".*",
        help="Only fetch fields for adapters matching this regex.",
        metavar="REGEX",
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--field-re",
        "-fr",
        "field_re",
        default=".*",
        help="Only fetch fields matching this regex.",
        metavar="REGEX",
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--field-key",
        "-fk",
        "field_key",
        default="name_qual",
        help="Which field key to match against for --field-re",
        type=click.Choice(
            ["name_qual", "name", "name_base", "column_title", "column_name"]
        ),
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--schema-type",
        "-st",
        "schema_type",
        type=click.Choice(["full", "basic"]),
        help="Control how much schema information to return",
        default="basic",
        show_envvar=True,
        show_default=True,
    ),
]

GET_EXPORT = [
    click.option(
        "--export-format",
        "-xt",
        "export",
        default="json",
        help="Formatter to use when exporting asset data",
        type=click.Choice(["csv", "json", "table"]),
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--table-format",
        "-tf",
        "table_format",
        default=constants.TABLE_FORMAT,
        help="""Table format to use for export-format=table""",
        type=click.Choice(tabulate.tabulate_formats),
        show_envvar=True,
        show_default=True,
        hidden=False,
    ),
    click.option(
        "--table-max-rows",
        "-tmr",
        "table_max_rows",
        default=constants.TABLE_MAX_ROWS,
        help="Only return this many rows for export-format=table",
        show_envvar=True,
        show_default=True,
        type=click.INT,
        hidden=False,
    ),
    click.option(
        "--include-schema/--no-include-schema",
        "-is/-nis",
        "export_schema",
        default=False,
        help="Add schema information to the export",
        is_flag=True,
        show_envvar=True,
        show_default=True,
        hidden=False,
    ),
    click.option(
        "--use-titles/--no-use-titles",
        "-ut/-nut",
        "field_titles",
        default=False,
        help="Rename fields from internal field names to their column titles",
        is_flag=True,
        show_envvar=True,
        show_default=True,
        hidden=False,
    ),
    # click.option(
    #     "--export-first-page/--no-export-first-page",
    #     "-xfp/-nxfp",
    #     "first_page",
    #     default=True,
    #     help="Log first page to STDERR",
    #     is_flag=True,
    #     show_envvar=True,
    #     show_default=True,
    #     hidden=False,
    # ),
    click.option(
        "--join-multivalue/--no-join-multivalue",
        "-jm/-njm",
        "field_join",
        default=False,
        help="Join multivalue fields using --export-join-value",
        is_flag=True,
        show_envvar=True,
        show_default=True,
        hidden=False,
    ),
    click.option(
        "--join-value",
        "-jv",
        "field_join_value",
        default=constants.FIELD_JOINER,
        help="Value to use for joining multivalue fields, default: \\n",
        show_envvar=True,
        show_default=True,
        hidden=False,
    ),
    click.option(
        "--join-trim",
        "-jt",
        "field_join_trim",
        default=constants.FIELD_TRIM_LEN,
        help="Character length to trim joined multivalue fields",
        show_envvar=True,
        show_default=True,
        type=click.INT,
        hidden=False,
    ),
    click.option(
        "--explode",
        "-x",
        "field_explode",
        default="",
        help="Flatten and explode a fields values into multiple rows",
        show_envvar=True,
        show_default=True,
        hidden=False,
        metavar="FIELD",
    ),
    click.option(
        "--flatten/--no-flatten",
        "-fl/-nfl",
        "field_flatten",
        default=False,
        help="Remove complex fields and re-add their sub-field values to the row",
        show_envvar=True,
        show_default=True,
        is_flag=True,
        hidden=False,
    ),
    click.option(
        "--field-exclude",
        "-fx",
        "field_excludes",
        help="Fields to exclude from each row",
        multiple=True,
        default=[],
        show_envvar=True,
        show_default=True,
        hidden=False,
        metavar="FIELD",
    ),
    click.option(
        "--field-null/--no-field-null",
        "-fn/-nfn",
        "field_null",
        default=False,
        help="Add missing fields with --export-field-null-value",
        show_envvar=True,
        show_default=True,
        is_flag=True,
        hidden=False,
    ),
    click.option(
        "--field-null-value",
        "-fnv",
        "field_null_value",
        default=None,
        help="Value to use for fields that are not returned",
        show_envvar=True,
        show_default=True,
        hidden=False,
    ),
    click.option(
        "--report-adapters-missing/--no-report-adapters-missing",
        "-ram/-nram",
        "report_adapters_missing",
        default=False,
        help="Add a column showing adapters missing from each asset",
        show_envvar=True,
        show_default=True,
        is_flag=True,
        hidden=False,
    ),
]

# XXX BAROQUE
GET_BUILDERS = [
    *AUTH,
    *PAGING,
    *EXPORT,
    *GET_EXPORT,
    *FIELDS_SELECT,
    get_option_fields_default(default=True),
    click.option(
        "--value",
        "-v",
        "values",
        help="Values to search for",
        required=True,
        multiple=True,
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--query-pre",
        "-qpre",
        "query_pre",
        help="Query to prefix the query built to search for --value",
        default="",
        metavar="QUERY",
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--query-post",
        "-qpost",
        "query_post",
        help="Query to postfix the query built to search for --value",
        default="",
        metavar="QUERY",
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--value-regex/--no-value-regex",
        "-vx/-nvx",
        "value_regex",
        help="Build the query using a regular expression instead of an exact match",
        is_flag=True,
        default=False,
        show_envvar=True,
        show_default=True,
    ),
    click.option(
        "--value-not/--no-value-not",
        "-vn/-nvn",
        "value_not",
        help="Build the query as a NOT query",
        is_flag=True,
        default=False,
        show_envvar=True,
        show_default=True,
    ),
    get_option_help(choices=["auth", "assetexport", "selectfields"]),
]

GET = [
    *AUTH,
    *PAGING,
    *EXPORT,
    *GET_EXPORT,
    *FIELDS_SELECT,
    get_option_fields_default(default=True),
    *QUERY,
    get_option_help(choices=["auth", "query", "assetexport", "selectfields"]),
]

GET_SQ = [
    *AUTH,
    *PAGING,
    *EXPORT,
    *GET_EXPORT,
    *FIELDS_SELECT,
    SQ_NAME,
    get_option_help(choices=["auth", "assetexport", "selectfields"]),
]

COUNT_QUERY = [
    *AUTH,
    *QUERY,
    get_option_help(choices=["auth", "query"]),
]

COUNT_SQ = [
    *AUTH,
    SQ_NAME,
    get_option_help(choices=["auth"]),
]
