"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms.
 *
 * @stability stable
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
    ComparisonOperator["GREATER_THAN_UPPER_THRESHOLD"] = "GreaterThanUpperThreshold";
    ComparisonOperator["LESS_THAN_LOWER_THRESHOLD"] = "LessThanLowerThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '<=',
};
/**
 * Specify how missing data points are treated during alarm evaluation.
 *
 * @stability stable
 */
var TreatMissingData;
(function (TreatMissingData) {
    TreatMissingData["BREACHING"] = "breaching";
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    TreatMissingData["IGNORE"] = "ignore";
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric.
 *
 * @stability stable
 */
class Alarm extends alarm_base_1.AlarmBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.list({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.list({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.list({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // eslint-disable-next-line max-len
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo).
     * @stability stable
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Turn this alarm into a horizontal annotation.
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *    you have both a "small margin/long period" alarm as well as a
     *    "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     *
     * @stability stable
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _b;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_b = st.period) === null || _b === void 0 ? void 0 : _b.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _b;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _b;
                            const hasSubmetrics = mathExprHasSubmetrics(expr);
                            if (hasSubmetrics) {
                                assertSubmetricsCount(expr);
                            }
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_b = conf.renderingProperties) === null || _b === void 0 ? void 0 : _b.label,
                                period: hasSubmetrics ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
_a = JSII_RTTI_SYMBOL_1;
Alarm[_a] = { fqn: "@aws-cdk/aws-cloudwatch.Alarm", version: "1.90.0" };
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
function assertSubmetricsCount(expr) {
    if (Object.keys(expr.usingMetrics).length > 10) {
        // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html#alarms-on-metric-math-expressions
        throw new Error('Alarms on math expressions cannot contain more than 10 individual metrics');
    }
    ;
}
//# sourceMappingURL=data:application/json;base64,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