import requests
import json
import logging
import jsonpickle
from dotmap import DotMap
from insightlab import InsightObjects


class API:
    """
    This is the base class that controls API interactions.

    Args:
        token (str): Token generated by Jira
        objectSchemaId (str): The Object Schema to interact with.

    Attributes:
        urlBase (str): The base url for the api
        request (Request): A request object from the 'Requests' library
        endpoint (str): The API endpoint to query
        params (DotMap): DotMap to store parameters
        debug (bool): Setting to true enables more verbosity
    """

    def __init__(self, token, objectSchemaId, debug=False):
        self.token = token
        self.urlBase = "https://insight-api.riada.io"
        self.request = requests.Request()
        self.objectSchemaId = objectSchemaId
        self.page = "1"
        self.resultsPerPage = "3000"
        self.includeAttributes = True
        self.endpoint = ""
        self.params = DotMap()
        self.debug = debug
        if self.debug:
            self.set_debug_urllib3()

    def set_debug_urllib(self):
        """
        Sets the debugging for the urllib3 (requests) library.
        """
        from http.client import HTTPConnection

        log = logging.getLogger("urllib3")
        log.setLevel(logging.DEBUG)
        ch = logging.StreamHandler()
        ch.setLevel(logging.DEBUG)
        log.addHandler(ch)
        HTTPConnection.debuglevel = 1

    def prepare_headers(self):
        """
        Prepare headers for authentication
        """
        headers = {
            "Authorization": f"Bearer {self.token}",
            "Content-type": "application/json",
        }
        self.request.headers = headers

    def prepare_params(
        self,
        page,
        resultsPerPage,
        includeAttributes,
        includeTypeAttributes,
        includeExtendedInfo,
    ):
        """
        Sets the parameters for an IQL request

        Args:
            page (str): The page to display for IQL requests. Default: 1
            resultsPerPage (str): Defines how many results are returned in an IQL query. Default: 3000
            includeAttributes (bool): Include the objects attributes in the result. Default: True
            includeTypeAttributes (bool): Include the object type attribute definition for every object attribute. Default: False
            includeExtendedInfo (bool): Include information if attachments and open JIRA issues exist for objects. Default: False
        """
        self.params.page = page
        self.params.resultPerPage = resultsPerPage
        self.params.includeAttributes = includeAttributes
        self.params.includeTypeAttributes = includeTypeAttributes
        self.params.includeExtendedInfo = includeExtendedInfo
        self.params.objectSchemaId = self.objectSchemaId

    def prepare_and_send(self):
        """
        Prepare and send the cooked request

        """
        self.prepare_headers()
        self.request.params = self.params.toDict()
        if self.debug:
            print(self.request.params)
        req = self.request.prepare()
        session = requests.Session()
        r = session.send(req)
        if not r.ok:
            print(f"HTTP {r.status_code}, {r.text}")

        # Reset some attributes for next query
        self.params = DotMap()
        self.endpoint = ""
        self.request = requests.Request()
        return r

    def iql_query(
        self,
        query,
        page=1,
        resultsPerPage=3000,
        includeAttributes=True,
        includeTypeAttributes=False,
        includeExtendedInfo=False,
    ):
        """
        Make a query using the IQL language: https://documentation.mindville.com/insight/latest/iql-insight-query-language-33467338.html

        Args:
            query (str): The query to make in IQL language.
            page (str): The page to display for IQL requests. Default: 1
            resultsPerPage (str): Defines how many results are returned in an IQL query. Default: 3000
            includeAttributes (bool): Include the objects attributes in the result. Default: True
            includeTypeAttributes (bool): Include the object type attribute definition for every object attribute. Default: False
            includeExtendedInfo (bool): Include information if attachments and open JIRA issues exist for objects. Default: False
        Returns:
            InsightObjects.IQLResponse
        """
        endpoint = "/rest/insight/1.0/iql/objects"
        self.prepare_params(
            page,
            resultsPerPage,
            includeAttributes,
            includeTypeAttributes,
            includeExtendedInfo,
        )
        self.params.iql = query
        self.request.url = self.urlBase + endpoint
        self.request.method = "GET"
        r = self.prepare_and_send()
        return InsightObjects.IQLResponse(json.loads(r.text))

    def load(self, id):
        """
        Load an object from Insight

        Args:
            id (str): The id of the object to load.

        Returns:
            InsightObjects.Object
        """
        endpoint = f"/rest/insight/1.0/object/{id}"
        self.request.url = self.urlBase + endpoint
        self.request.method = "GET"
        r = self.prepare_and_send()
        if not r.ok:
            raise Exception(f"Unexpected status: {r.status_code}, {r.reason}")
        return InsightObjects.Object(json.loads(r.text))

    def update_attribute(self, object_id, attribute):
        """
        Updates an attribute for a given object.

        Args:
            object_id (str): The id of the object to update
            attribute (Attributes): An attribute object
        """
        endpoint = f"/rest/insight/1.0/object/{object_id}"
        self.request.url = self.urlBase + endpoint
        self.request.method = "PUT"        
        new_attr = DotMap()
        new_attr.attributes = []
        sub_attr = DotMap()
        sub_attr.objectTypeAttributeId = attribute.objectTypeAttributeId
        sub_attr.objectAttributeValues = []
        for attr in attribute.objectAttributeValues:
            sub_attr.objectAttributeValues.append(attr)
        new_attr.attributes.append(sub_attr)
        self.request.data = json.dumps(new_attr.toDict())              
        if self.debug:
            print(self.request.data)
        r = self.prepare_and_send()
        if not r.ok:
            raise Exception(f"Unexpected status: {r.status_code}, {r.reason}")

    def find_object(self, object_name, object_type, resultsPerPage=3000):
        """
        Find an Insight object by name.

        Args:
            object_name (str): The name of the object to find
            object_type (str): The name of the object's type
            resultsPerPage (str): Override the number of results returned by the IQL query. It should be more than the item count for the searched category. Default: 3000.

        Returns:
            InsightObjects.Object

        Examples:
            >>> i = Insight.API("12345abcd", "42")
            >>> myserver = i.find_object("New Server", "Server")
            >>> print(myserver.name)
            New Server
        """
        returned_objects = self.iql_query(
            f'objectType IN objectTypeAndChildren("{object_type}")',
            resultsPerPage=resultsPerPage,
        )
        for obj in returned_objects.objects:
            if obj.name == object_name:
                return self.load(obj.id)
        raise Exception(f"No object with name '{object_name}' found.")

    def find_object_type_id(self, name):
        """
        Find the object type id on an object by name

        Args:
            name (str): The name of the object to find the type.

        Returns:
            str: The id of the object's type

        Examples:
            >>> i = Insight.API("12345abcd", "42")
            >>> server_type = i.find_object_type_id("Server")
            >>> print(server_type)
            245
        """
        endpoint = (
            f"/rest/insight/1.0/objectschema/{self.objectSchemaId}/objecttypes/flat"
        )
        self.request.url = self.urlBase + endpoint
        self.request.method = "GET"
        r = self.prepare_and_send()
        if not r.ok:
            raise Exception(f"Unexpected status: {r.status_code}, {r.reason}")
        for t in json.loads(r.text):
            if t["name"] == name:
                return t["id"]
        raise Exception(f"No object type with name {name} found.")

    def find_object_type_attribute_id(self, object_type_name, attr_name):
        """
        Find the id of an attribute for a specific object type.

        Args:
            object_type_name (str): The object type that has the attribute
            attr_name (str): The name of the attribute to find

        Returns:
            id (str): The id of the attribute.

        Examples:
            >>> i = Insight.API("12345abcd", "42")
            >>> attrid = i.find_object_type_attribute_id("Server", "Name")
            >>> print(attrid)
            123
        """
        obj_id = self.find_object_type_id(object_type_name)
        endpoint = f"/rest/insight/1.0/objecttype/{obj_id}/attributes"
        self.request.url = self.urlBase + endpoint
        self.request.method = "GET"
        r = self.prepare_and_send()
        if not r.ok:
            raise Exception(f"Unexpected status: {r.status_code}, {r.reason}")
        if self.debug:
            print(r.text)
        for t in json.loads(r.text):
            if t["name"] == attr_name:
                return t["id"]
        raise Exception(f"No object type with name {attr_name} found.")

    def delete(self, object_id):
        """
        Delete an object

        Args:
            object_id (str): The id of the object to delete
        """
        endpoint = f"/rest/insight/1.0/object/{object_id}"
        self.request.url = self.urlBase + endpoint
        self.request.method = "DELETE"
        r = self.prepare_and_send()
        if not r.ok:
            raise Exception(f"Unexpected status: {r.status_code}, {r.reason}")

    def create(self, new_object):
        """
        Create an object

        Args:
            new_object (InsightObjects.NewObject): The new object to be created.

        Examples:
            >>> i = Insight.API("12345abcd", "42")
            >>> obj_type = i.find_object_type_id("Server")
            >>> name_attr = i.find_object_type_attribute_id("Server", "Name")
            >>> newobj = InsightObjects.NewObject(obj_type)
            >>> newobj.add_attribute(name_attr, "New Server")
            >>> i.create(newobj)
        """
        endpoint = f"/rest/insight/1.0/object/create"
        self.request.url = self.urlBase + endpoint
        self.request.method = "POST"
        self.request.data = jsonpickle.encode(
            new_object, unpicklable=False, make_refs=False
        )
        r = self.prepare_and_send()
        if not r.ok:
            raise Exception(f"Unexpected status: {r.status_code}, {r.reason}")
