import numpy as np
import pytest

from pyextremes.extremes import get_extremes
from pyextremes.extremes.block_maxima import NoDataBlockWarning


def test_invalid_arguments(battery_wl):
    # Test wrong extremes_type value
    with pytest.raises(ValueError, match=r"invalid value.*extremes_type"):
        get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type="BAD EXTREMES_TYPE VALUE",
            block_size="365.2425D",
            errors="coerce",
        )

    # Test wrong block_size type
    with pytest.raises(TypeError, match=r"invalid type.*block_size"):
        get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type="high",
            block_size=1,
            errors="coerce",
        )

    # Test wrong errors value
    with pytest.raises(ValueError, match=r"invalid value.*errors.*argument"):
        get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type="high",
            block_size="365.2425D",
            errors="BAD ERRORS VALUE",
        )


@pytest.mark.parametrize("extremes_type", ["high", "low"])
def test_extreme_value_extraction(battery_wl, extremes_type):
    # Test errors=raise
    with pytest.raises(ValueError):
        get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type=extremes_type,
            block_size="365.2425D",
            errors="raise",
        )

    # Test errors=ignore
    with pytest.warns(NoDataBlockWarning, match=r"blocks contained no data"):
        extremes_ignored = get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type=extremes_type,
            block_size="365.2425D",
            errors="ignore",
        )
        assert len(extremes_ignored) == 96

    # Test errors=coerce
    with pytest.warns(NoDataBlockWarning, match=r"blocks contained no data"):
        extremes_coerced = get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type=extremes_type,
            block_size="365.2425D",
            errors="coerce",
        )
        assert len(extremes_coerced) == 100

    if extremes_type == "high":
        assert np.isclose(extremes_ignored.max(), extremes_coerced.max())
    else:
        assert np.isclose(extremes_ignored.min(), extremes_coerced.min())
    assert np.isclose(extremes_ignored.mean(), extremes_coerced.mean())


@pytest.mark.parametrize("extremes_type", ["high", "low"])
def test_min_last_block(battery_wl, extremes_type):
    with pytest.warns(NoDataBlockWarning, match=r"blocks contained no data"):
        extremes_full = get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type=extremes_type,
            block_size="365.2425D",
            errors="coerce",
            min_last_block=None,
        )
    with pytest.warns(NoDataBlockWarning, match=r"blocks contained no data"):
        extremes_trimmed = get_extremes(
            ts=battery_wl,
            method="BM",
            extremes_type=extremes_type,
            block_size="365.2425D",
            errors="coerce",
            min_last_block=0.9,
        )
    assert len(extremes_full) - len(extremes_trimmed) == 1
    assert np.allclose(
        extremes_full.values[:-1], extremes_trimmed.values, atol=0.01, rtol=0
    )
