from unittest import TestCase
from epytope.Core import Allele
from epytope.IO import FileReader
from epytope.IO.MartsAdapter import MartsAdapter
from epytope.IO.EnsemblAdapter import EnsemblDB
from epytope.IO.RefSeqAdapter import RefSeqAdapter
from epytope.IO.UniProtAdapter import UniProtDB
from epytope.IO.ADBAdapter import ADBAdapter, EAdapterFields, EIdentifierTypes
import warnings
import os
import inspect
import epytope

__author__ = 'walzer'


class TestIO(TestCase):
    def assertWarnings(self, warning, call, *args, **kwds):
        with warnings.catch_warnings(record=True) as warning_list:
            warnings.simplefilter('always')
            result = call(*args, **kwds)
            self.assertTrue(any(item.category == warning for item in warning_list))

    def setUp(self):
        self.ale_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/alleles.txt")
        self.ale_zonk_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/alleles_defect.txt")
        self.ale_no_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/magic.txt")
        self.fa_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/testSequences.fasta")
        self.fa_unconventional_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/testSequences_d2s.fasta")
        self.edb_cds_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/Homo_sapiens.GRCh38.cds.test_stub.fa")
        self.edb_pep_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/Homo_sapiens.GRCh38.pep.test_stub.fa")
        self.ano_path = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/test_annovar.out")
        self.NP_001005353 = "MASKLLRAVILGPPGSGKGTVCQRIAQNFGLQHLSSGHFLRENIKASTEVGEMAKQYIEKSLLVPDHVITRLMMSELENRRGQHWLLDGFPRTLGQAEALDKICEVDLVISLNIPFETLKDRLSRRWIHPPSGRVYNLDFNPPHVHGIDDVTGEPLVQQEDDKPEAVAARLRQYKDVAKPVIELYKSRGVLHQFSGTETNKIWPYVYTLFSNKITPIQSKEAY"
        self.ENSP00000369497 = "MPIGSKERPTFFEIFKTRCNKADLGPISLNWFEELSSEAPPYNSEPAEESEHKNNNYEPNLFKTPQRKPSYNQLASTPIIFKEQGLTLPLYQSPVKELDKFKLDLGRNVPNSRHKSLRTVKTKMDQADDVSCPLLNSCLSESPVVLQCTHVTPQRDKSVVCGSLFHTPKFVKGRQTPKHISESLGAEVDPDMSWSSSLATPPTLSSTVLIVRNEEASETVFPHDTTANVKSYFSNHDESLKKNDRFIASVTDSENTNQREAASHGFGKTSGNSFKVNSCKDHIGKSMPNVLEDEVYETVVDTSEEDSFSLCFSKCRTKNLQKVRTSKTRKKIFHEANADECEKSKNQVKEKYSFVSEVEPNDTDPLDSNVANQKPFESGSDKISKEVVPSLACEWSQLTLSGLNGAQMEKIPLLHISSCDQNISEKDLLDTENKRKKDFLTSENSLPRISSLPKSEKPLNEETVVNKRDEEQHLESHTDCILAVKQAISGTSPVASSFQGIKKSIFRIRESPKETFNASFSGHMTDPNFKKETEASESGLEIHTVCSQKEDSLCPNLIDNGSWPATTTQNSVALKNAGLISTLKKKTNKFIYAIHDETSYKGKKIPKDQKSELINCSAQFEANAFEAPLTFANADSGLLHSSVKRSCSQNDSEEPTLSLTSSFGTILRKCSRNETCSNNTVISQDLDYKEAKCNKEKLQLFITPEADSLSCLQEGQCENDPKSKKVSDIKEEVLAAACHPVQHSKVEYSDTDFQSQKSLLYDHENASTLILTPTSKDVLSNLVMISRGKESYKMSDKLKGNNYESDVELTKNIPMEKNQDVCALNENYKNVELLPPEKYMRVASPSRKVQFNQNTNLRVIQKNQEETTSISKITVNPDSEELFSDNENNFVFQVANERNNLALGNTKELHETDLTCVNEPIFKNSTMVLYGDTGDKQATQVSIKKDLVYVLAEENKNSVKQHIKMTLGQDLKSDISLNIDKIPEKNNDYMNKWAGLLGPISNHSFGGSFRTASNKEIKLSEHNIKKSKMFFKDIEEQYPTSLACVEIVNTLALDNQKKLSKPQSINTVSAHLQSSVVVSDCKNSHITPQMLFSKQDFNSNHNLTPSQKAEITELSTILEESGSQFEFTQFRKPSYILQKSTFEVPENQMTILKTTSEECRDADLHVIMNAPSIGQVDSSKQFEGTVEIKRKFAGLLKNDCNKSASGYLTDENEVGFRGFYSAHGTKLNVSTEALQKAVKLFSDIENISEETSAEVHPISLSSSKCHDSVVSMFKIENHNDKTVSEKNNKCQLILQNNIEMTTGTFVEEITENYKRNTENEDNKYTAASRNSHNLEFDGSDSSKNDTVCIHKDETDLLFTDQHNICLKLSGQFMKEGNTQIKEDLSDLTFLEVAKAQEACHGNTSNKEQLTATKTEQNIKDFETSDTFFQTASGKNISVAKESFNKIVNFFDQKPEELHNFSLNSELHSDIRKNKMDILSYEETDIVKHKILKESVPVGTGNQLVTFQGQPERDEKIKEPTLLGFHTASGKKVKIAKESLDKVKNLFDEKEQGTSEITSFSHQWAKTLKYREACKDLELACETIEITAAPKCKEMQNSLNNDKNLVSIETVVPPKLLSDNLCRQTENLKTSKSIFLKVKVHENVEKETAKSPATCYTNQSPYSVIENSALAFYTSCSRKTSVSQTSLLEAKKWLREGIFDGQPERINTADYVGNYLYENNSNSTIAENDKNHLSEKQDTYLSNSSMSNSYSYHSDEVYNDSGYLSKNKLDSGIEPVLKNVEDQKNTSFSKVISNVKDANAYPQTVNEDICVEELVTSSSPCKNKNAAIKLSISNSNNFEVGPPAFRIASGKIVCVSHETIKKVKDIFTDSFSKVIKENNENKSKICQTKIMAGCYEALDDSEDILHNSLDNDECSTHSHKVFADIQSEEILQHNQNMSGLEKVSKISPCDVSLETSDICKCSIGKLHKSVSSANTCGIFSTASGKSVQVSDASLQNARQVFSEIEDSTKQVFSKVLFKSNEHSDQLTREENTAIRTPEHLISQKGFSYNVVNSSAFSGFSTASGKQVSILESSLHKVKGVLEEFDLIRTEHSLHYSPTSRQNVSKILPRVDKRNPEHCVNSEMEKTCSKEFKLSNNLNVEGGSSENNHSIKVSPYLSQFQQDKQQLVLGTKVSLVENIHVLGKEQASPKNVKMEIGKTETFSDVPVKTNIEVCSTYSKDSENYFETEAVEIAKAFMEDDELTDSKLPSHATHSLFTCPENEEMVLSNSRIGKRRGEPLILVGEPSIKRNLLNEFDRIIENQEKSLKASKSTPDGTIKDRRLFMHHVSLEPITCVPFRTTKERQEIQNPNFTAPGQEFLSKSHLYEHLTLEKSSSNLAVSGHPFYQVSATRNEKMRHLITTGRPTKVFVPPFKTKSHFHRVEQCVRNINLEENRQKQNIDGHGSDDSKNKINDNEIHQFNKNNSNQAVAVTFTKCEEEPLDLITSLQNARDIQDMRIKKKQRQRVFPQPGSLYLAKTSTLPRISLKAAVGGQVPSACSHKQLYTYGVSKHCIKINSKNAESFQFHTEDYFGKESLWTGKGIQLADGGWLIPSNDGKAGKEEFYRALCDTPGVDPKLISRIWVYNHYRWIIWKLAAMECAFPKEFANRCLSPERVLLQLKYRYDTEIDRSRRSAIKKIMERDDTAAKTLVLCVSDIISLSANISETSSNKTSSADTQKVAIIELTDGWYAVKAQLDPPLLAVLKNGRLTVGQKIILHGAELVGSPDACTPLEAPESLMLKISANSTRPARWYTKLGFFPDPRPFPLPLSSLFSDGGNVGCVDVIIQRAYPIQWMEKTSSGLYIFRNEREEEKEAAKYVEAQQKRLEALFTKIQEEFEEHEENTTKPYLPSRALTRQQVRALQDGAELYEAVKNAADPAYLEGYFSEEQLRALNNHRQMLNDKKQAQIQLEIRKAMESAEQKEQGLSRDVTTVWKLRIVSYSKKEKDSVILSIWRPSSDLYSLLTEGKRYRIYHLATSKSKSKSERANIQLAATKKTQYQQLPVSDEILFQIYQPREPLHFSKFLDPDFQPSCSEVDLIGFVVSVVKKTGLAPFVYLSDECYNLLAIKFWIDLNEDIIKPHMLIAASNLQWRPESKSGLLTLFAGDFSVFSASPKEGHFQETFNKMKNTVENIDILCNEAENKLMHILHANDPKWSTPTKDCTSGPYTAQIIPGTGNKLLMSSPNCEIYYQSPLSLCMAKRKSVSTPVSAQMTSKSCKGEKEIDDQKNCKKRRALDFLSRLPLPPPVSPICTFVSPAAQKAFQPPRSCGTKYETPIKKKELNSPQMTPFKKFNEISLLESNSIADEELALINTQALLSGSTGEKQFISVSESTRTAPTSSEDYLRLKRRCTTSLIKEQESSQASTEECEKNKQDTITTKKYI"
        self.ENST00000361221 = {0: 'ARHGEF10L', 1: '+', 2: 'ATGGCTTCCTCCAACCCTCCTCCACAGCCTGCCATAGGAGATCAGCTGGTTCCAGGAGTCCCAGGCCCCTCCTCTGAGGCAGAGGACGACCCAGGAGAGGCGTTTGAGTTTGATGACAGTGATGATGAAGAGGACACCAGCGCAGCCCTGGGCGTCCCCAGCCTTGCTCCTGAGAGGGACACAGACCCCCCACTGATCCACTTGGACTCCATCCCTGTCACTGACCCAGACCCAGCAGCTGCTCCACCCGGCACAGGGGTGCCAGCCTGGGTGAGCAATGGGGATGCAGCGGACGCAGCCTTCTCCGGGGCCCGGCACTCCAGCTGGAAGCGGAAGAGTTCCCGTCGCATTGACCGGTTCACTTTCCCCGCCCTGGAAGAGGATGTGATTTATGACGACGTCCCCTGCGAGAGCCCAGATGCGCATCAGCCCGGGGCAGAGAGGAACCTGCTCTACGAGGATGCGCACCGGGCTGGGGCCCCTCGGCAGGCGGAGGACCTAGGCTGGAGCTCCAGTGAGTTCGAGAGCTACAGCGAGGACTCGGGGGAGGAGGCCAAGCCGGAGGTCGAGGTCGAGCCCGCCAAGCACCGAGTGTCCTTCCAGCCCAAGCTTTCTCCAGACCTGACTAGGCTAAAGGAGAGATACGCCAGGACTAAGAGAGACATCTTGGCTTTGAGAGTTGGGGGGAGAGACATGCAGGAGCTGAAGCACAAGTACGATTGTAAGATGACCCAGCTCATGAAGGCCGCCAAGAGCGGGACCAAGGATGGGCTGGAGAAGACACGGATGGCCGTGATGCGCAAAGTCTCCTTCCTGCACAGGAAGGACGTCCTCGGTGACTCGGAGGAGGAGGACATGGGGCTCCTGGAGGTCAGCGTTTCGGACATCAAGCCCCCAGCCCCAGAGCTGGGCCCCATGCCAGAGGGCCTGAGCCCTCAGCAGGTGGTCCGGAGGCATATCCTGGGCTCCATCGTGCAGAGCGAAGGCAGCTACGTGGAGTCTCTGAAGCGGATACTCCAGGACTACCGCAACCCCCTGATGGAGATGGAGCCCAAGGCGCTGAGCGCCCGCAAGTGCCAGGTGGTGTTCTTCCGCGTGAAGGAGATCCTGCACTGCCACTCCATGTTCCAGATCGCCCTGTCCTCCCGCGTGGCTGAGTGGGATTCCACCGAGAAGATCGGGGACCTCTTCGTGGCCTCGTTTTCCAAGTCCATGGTGCTAGATGTGTACAGTGACTACGTGAACAACTTCACCAGTGCCATGTCCATCATCAAGAAGGCCTGCCTCACCAAGCCTGCCTTCCTCGAGTTCCTCAAGCGACGGCAGGTGTGCAGCCCAGACCGTGTCACCCTCTACGGGCTGATGGTCAAGCCCATCCAGAGGTTCCCACAGTTCATACTCCTGCTTCAGGACATGCTGAAGAACACCCCCAGGGGCCATCCGGACAGGCTGTCGCTGCAGCTGGCCCTCACAGAGCTGGAGACGCTGGCTGAGAAGCTGAACGAGCAGAAGCGGCTGGCTGACCAGGTGGCTGAGATCCAGCAGCTGACCAAGAGCGTCAGTGACCGCAGCAGCCTCAACAAGCTGTTGACCTCAGGCCAGCGGCAGCTGCTCCTGTGTGAGACGTTGACGGAGACCGTGTACGGTGACCGCGGGCAGCTAATTAAGTCCAAGGAGCGTCGGGTCTTCCTGCTCAACGACATGCTTGTCTGTGCCAACATCAACTTCAAGCCTGCCAACCACAGGGGCCAGCTGGAGATCAGCAGCCTGGTGCCCCTGGGGCCCAAGTATGTGGTGAAGTGGAACACGGCGCTGCCCCAGGTGCAGGTGGTGGAGGTGGGCCAGGACGGTGGCACCTATGACAAGGACAATGTGCTCATCCAGCACTCAGGCGCCAAGAAGGCCTCTGCCTCAGGGCAGGCTCAGAATAAGGTGTACCTCGGCCCCCCACGCCTCTTCCAGGAGCTGCAGGACCTGCAGAAGGACCTGGCCGTGGTGGAGCAGATCACGCTTCTCATCAGCACGCTGCACGGCACCTACCAGAACCTGAACATGACTGTGGCTCAAGACTGGTGCCTGGCCCTGCAGAGGCTGATGCGGGTGAAGGAGGAAGAGATCCACTCGGCCAACAAGTGCCGTCTCAGGCTCCTGCTTCCTGGGAAACCCGACAAGTCCGGCCGCCCCATTAGCTTCATGGTGGTTTTCATCACCCCCAACCCCCTGAGCAAGATTTCCTGGGTCAACAGGTTACATTTGGCCAAAATCGGACTCCGGGAGGAGAACCAGCCAGGCTGGCTATGCCCGGATGAGGACAAGAAGAGCAAAGCCCCATTCTGGTGCCCGATCCTGGCCTGCTGCATCCCTGCCTTCTCCTCCCGGGCACTCAGCCTGCAGCTTGGGGCCCTGGTCCACAGTCCTGTCAACTGTCCCCTGCTGGGTTTCTCAGCAGTCAGCACCTCCCTTCCACAGGGCTACCTCTGGGTCGGGGGCGGACAGGAAGGCGCAGGGGGCCAGGTGGAAATCTTTTCCTTGAACCGGCCCTCGCCCCGCACCGTCAAGTCCTTCCCACTGGCAGCCCCTGTGCTCTGCATGGAGTATATCCCGGAGCTGGAGGAGGAGGCGGAGAGCAGAGACGAGAGCCCGACAGTTGCTGACCCCTCGGCCACGGTGCATCCAACCATCTGCCTCGGGCTCCAGGATGGCAGCATCCTCCTCTACAGCAGTGTGGACACTGGCACCCAGTGCCTGGTGAGCTGCAGGAGCCCAGGTCTGCAGCCTGTGCTCTGCCTGCGACACAGCCCCTTCCACCTGCTCGCTGGCCTGCAGGATGGGACCCTTGCTGCTTACCCTCGGACCAGCGGAGGTGTCCTGTGGGACCTGGAGAGCCCTCCCGTGTGCCTGACTGTGGGGCCCGGGCCTGTCCGCACCCTGTTGAGCCTGGAGGATGCCGTGTGGGCCAGCTGTGGGCCCTGGGTCACTGTCCTGGAAGCCACCACCCTGCAGCCTCAGCAAAGCTTCGAGGCGCACCAGGACGAGGCAGTGAGCGTGACACACATGGTGAAGGCGGGCAGCGGCGTCTGGATGGCCTTCTCCTCCGGCACCTCCATCCGCCTCTTCCACACTGAGACCCTGGAGCATCTGCAAGAGATCAACATCGCCACCAGGACCACCTTCCTCCTGCCAGGCCAGAAGCACTTGTGTGTCACCAGCCTCCTGATCTGCCAGGGTCTGCTCTGGGTGGGCACTGACCAGGGTGTCATCGTCCTGCTGCCCGTGCCTCGGCTGGAAGGCATCCCCAAGATCACAGGGAAAGGCATGGTCTCACTCAACGGGCACTGTGGGCCTGTGGCCTTCCTGGCTGTGGCTACCAGCATCCTGGCCCCTGACATCCTGCGGAGTGACCAGGAGGAGGCTGAGGGGCCCCGGGCTGAGGAGGACAAGCCAGACGGGCAGGCACACGAGCCCATGCCCGATAGCCACGTGGGCCGAGAGCTGACCCGCAAGAAGGGCATCCTCTTGCAGTACCGCCTGCGCTCCACCGCACACCTCCCGGGCCCGCTGCTCTCCATGCGGGAGCCGGCGCCTGCTGATGGCGCAGCTTTGGAGCACAGCGAGGAGGACGGCTCCATTTACGAGATGGCCGACGACCCCGACATCTGGGTGCGCAGCCGGCCCTGCGCCCGCGACGCCCACCGCAAGGAGATTTGCTCTGTGGCCATCATCTCCGGCGGGCAGGGCTACCGCAACTTTGGCAGCGCTCTGGGCAGCAGTGGGAGGCAGGCCCCGTGTGGGGAGACGGACAGCACCCTCCTCATCTGGCAGGTGCCCTTGATGCTATAG'}
        self.ENST00000361221_grch38 = {0: 'ARHGEF10L', 1: '+', 2:
        "ATGGCTTCCTCCAACCCTCCTCCACAGCCTGCCATAGGAGATCAGCTGGTTCCAGGAGTCCCAGGCCCCTCCTCTGAGGCAGAGGACGACCCAGGAGAGGCGTTTGAGTTTGATGACAGTGATGATGAAGAGGACACCAGCGCAGCCCTGGGCGTCCCCAGCCTTGCTCCTGAGAGGGACACAGACCCCCCACTGATCCACTTGGACTCCATCCCTGTCACTGACCCAGACCCAGCAGCTGCTCCACCCGGCACAGGGGTGCCAGCCTGGGTGAGCAATGGGGATGCAGCGGACGCAGCCTTCTCCGGGGCCCGGCACTCCAGCTGGAAGCGGAAGAGTTCCCGTCGCATTGACCGGTTCACTTTCCCCGCCCTGGAAGAGGATGTGATTTATGACGACGTCCCCTGCGAGAGCCCAGATGCGCATCAGCCCGGGGCAGAGAGGAACCTGCTCTACGAGGATGCGCACCGGGCTGGGGCCCCTCGGCAGGCGGAGGACCTAGGCTGGAGCTCCAGTGAGTTCGAGAGCTACAGCGAGGACTCGGGGGAGGAGGCCAAGCCGGAGGTCGAGGTCGAGCCCGCCAAGCACCGAGTGTCCTTCCAGCCCAAGCTTTCTCCAGACCTGACTAGGCTAAAGGAGAGATACGCCAGGACTAAGAGAGACATCTTGGCTTTGAGAGTTGGGGGGAGAGACATGCAGGAGCTGAAGCACAAGTACGATTGTAAGATGACCCAGCTCATGAAGGCCGCCAAGAGCGGGACCAAGGATGGGCTGGAGAAGACACGGATGGCCGTGATGCGCAAAGTCTCCTTCCTGCACAGGAAGGACGTCCTCGGTGACTCGGAGGAGGAGGACATGGGGCTCCTGGAGGTCAGCGTTTCGGACATCAAGCCCCCAGCCCCAGAGCTGGGCCCCATGCCAGAGGGCCTGAGCCCTCAGCAGGTGGTCCGGAGGCATATCCTGGGCTCCATCGTGCAGAGCGAAGGCAGCTACGTGGAGTCTCTGAAGCGGATACTCCAGGACTACCGCAACCCCCTGATGGAGATGGAGCCCAAGGCGCTGAGCGCCCGCAAGTGCCAGGTGGTGTTCTTCCGCGTGAAGGAGATCCTGCACTGCCACTCCATGTTCCAGATCGCCCTGTCCTCCCGCGTGGCTGAGTGGGATTCCACCGAGAAGATCGGGGACCTCTTCGTGGCCTCGTTTTCCAAGTCCATGGTGCTAGATGTGTACAGTGACTACGTGAACAACTTCACCAGTGCCATGTCCATCATCAAGAAGGCCTGCCTCACCAAGCCTGCCTTCCTCGAGTTCCTCAAGCGACGGCAGGTGTGCAGCCCAGACCGTGTCACCCTCTACGGGCTGATGGTCAAGCCCATCCAGAGGTTCCCACAGTTCATACTCCTGCTTCAGGACATGCTGAAGAACACCCCCAGGGGCCATCCGGACAGGCTGTCGCTGCAGCTGGCCCTCACAGAGCTGGAGACGCTGGCTGAGAAGCTGAACGAGCAGAAGCGGCTGGCTGACCAGGTGGCTGAGATCCAGCAGCTGACCAAGAGCGTCAGTGACCGCAGCAGCCTCAACAAGCTGTTGACCTCAGGCCAGCGGCAGCTGCTCCTGTGTGAGACGTTGACGGAGACCGTGTACGGTGACCGCGGGCAGCTAATTAAGTCCAAGGAGCGTCGGGTCTTCCTGCTCAACGACATGCTTGTCTGTGCCAACATCAACTTCAAGCCTGCCAACCACAGGGGCCAGCTGGAGATCAGCAGCCTGGTGCCCCTGGGGCCCAAGTATGTGGTGAAGTGGAACACGGCGCTGCCCCAGGTGCAGGTGGTGGAGGTGGGCCAGGACGGTGGCACCTATGACAAGGACAATGTGCTCATCCAGCACTCAGGCGCCAAGAAGGCCTCTGCCTCAGGGCAGGCTCAGAATAAGGTGTACCTCGGCCCCCCACGCCTCTTCCAGGAGCTGCAGGACCTGCAGAAGGACCTGGCCGTGGTGGAGCAGATCACGCTTCTCATCAGCACGCTGCACGGCACCTACCAGAACCTGAACATGACTGTGGCTCAAGACTGGTGCCTGGCCCTGCAGAGGCTGATGCGGGTGAAGGAGGAAGAGATCCACTCGGCCAACAAGTGCCGTCTCAGGCTCCTGCTTCCTGGGAAACCCGACAAGTCCGGCCGCCCCATTAGCTTCATGGTGGTTTTCATCACCCCCAACCCCCTGAGCAAGATTTCCTGGGTCAACAGGTTACATTTGGCCAAAATCGGACTCCGGGAGGAGAACCAGCCAGGCTGGCTATGCCCGGATGAGGACAAGAAGAGCAAAGCCCCATTCTGGTGCCCGATCCTGGCCTGCTGCATCCCTGCCTTCTCCTCCCGGGCACTCAGCCTGCAGCTTGGGGCCCTGGTCCACAGTCCTGTCAACTGTCCCCTGCTGGGTTTCTCAGCAGTCAGCACCTCCCTTCCACAGGGCTACCTCTGGGTCGGGGGCGGACAGGAAGGCGCAGGGGGCCAGGTGGAAATCTTTTCCTTGAACCGGCCCTCGCCCCGCACCGTCAAGTCCTTCCCACTGGCAGCCCCTGTGCTCTGCATGGAGTATATCCCGGAGCTGGAGGAGGAGGCGGAGAGCAGAGACGAGAGCCCGACAGTTGCTGACCCCTCGGCCACGGTGCATCCAACCATCTGCCTCGGGCTCCAGGATGGCAGCATCCTCCTCTACAGCAGTGTGGACACTGGCACCCAGTGCCTGGTGAGCTGCAGGAGCCCAGGTCTGCAGCCTGTGCTCTGCCTGCGACACAGCCCCTTCCACCTGCTCGCTGGCCTGCAGGATGGGACCCTTGCTGCTTACCCTCGGACCAGCGGAGGTGTCCTGTGGGACCTGGAGAGCCCTCCCGTGTGCCTGACTGTGGGGCCCGGGCCTGTCCGCACCCTGTTGAGCCTGGAGGATGCCGTGTGGGCCAGCTGTGGGCCCCGGGTCACTGTCCTGGAAGCCACCACCCTGCAGCCTCAGCAAAGCTTCGAGGCGCACCAGGACGAGGCAGTGAGCGTGACACACATGGTGAAGGCGGGCAGCGGCGTCTGGATGGCCTTCTCCTCCGGCACCTCCATCCGCCTCTTCCACACTGAGACCCTGGAGCATCTGCAAGAGATCAACATCGCCACCAGGACCACCTTCCTCCTGCCAGGCCAGAAGCACTTGTGTGTCACCAGCCTCCTGATCTGCCAGGGTCTGCTCTGGGTGGGCACTGACCAGGGTGTCATCGTCCTGCTGCCCGTGCCTCGGCTGGAAGGCATCCCCAAGATCACAGGGAAAGGCATGGTCTCACTCAACGGGCACTGTGGGCCTGTGGCCTTCCTGGCTGTGGCTACCAGCATCCTGGCCCCTGACATCCTGCGGAGTGACCAGGAGGAGGCTGAGGGGCCCCGGGCTGAGGAGGACAAGCCAGACGGGCAGGCACACGAGCCCATGCCCGATAGCCACGTGGGCCGAGAGCTGACCCGCAAGAAGGGCATCCTCTTGCAGTACCGCCTGCGCTCCACCGCACACCTCCCGGGCCCGCTGCTCTCCATGCGGGAGCCGGCGCCTGCTGATGGCGCAGCTTTGGAGCACAGCGAGGAGGACGGCTCCATTTACGAGATGGCCGACGACCCCGACATCTGGGTGCGCAGCCGGCCCTGCGCCCGCGACGCCCACCGCAAGGAGATTTGCTCTGTGGCCATCATCTCCGGCGGGCAGGGCTACCGCAACTTTGGCAGCGCTCTGGGCAGCAGTGGGAGGCAGGCCCCGTGTGGGGAGACGGACAGCACCCTCCTCATCTGGCAGGTGCCCTTGATGCTATAG"}
        self.ENST00000000233_5 = {0: '', 1: '+', 2: 'ATGGGCCTCACCGTGTCCGCGCTCTTTTCGCGGATCTTCGGGAAGAAGCAGATGCGGATTCTCATGGTTGGCTTGGATGCGGCTGGCAAGACCACAATCCTGTACAAACTGAAGTTGGGGGAGATTGTCACCACCATCCCAACCATAGGCTTCAATGTAGAAACAGTGGAATATAAGAACATCTGTTTCACAGTCTGGGACGTGGGAGGCCAGGACAAGATTCGGCCTCTGTGGCGGCACTACTTCCAGAACACTCAGGGCCTCATCTTTGTGGTGGACAGTAATGACCGGGAGCGGGTCCAAGAATCTGCTGATGAACTCCAGAAGATGCTGCAGGAGGACGAGCTGCGGGATGCAGTGCTGCTGGTATTTGCCAACAAGCAGGACATGCCCAACGCCATGCCCGTGAGCGAGCTGACTGACAAGCTGGGGCTACAGCACTTACGCAGCCGCACGTGGTATGTCCAGGCCACCTGTGCCACCCAAGGCACAGGTCTGTACGATGGTCTGGACTGGCTGTCCCACGAGCTGTCAAAGCGCTAA'}
        self.ENSEMBL_ensg = "ENSG00000138674"
        self.tp53_ensembl_biomart = [{0: 'ENSG00000141510', 1: '-', 3: 'ENST00000413465', 4: 'ENSP00000410739'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000359597', 4: 'ENSP00000352610'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000504290', 4: ''}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000510385', 4: ''}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000504937', 4: ''}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000269305', 4: 'ENSP00000269305'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000455263', 4: 'ENSP00000398846'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000420246', 4: 'ENSP00000391127'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000445888', 4: 'ENSP00000391478'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000576024', 4: 'ENSP00000458393'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000509690', 4: 'ENSP00000425104'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000514944', 4: 'ENSP00000423862'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000574684', 4: ''}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000505014', 4: ''}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000508793', 4: 'ENSP00000424104'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000604348', 4: 'ENSP00000473895'}, {0: 'ENSG00000141510', 1: '-', 3: 'ENST00000503591', 4: 'ENSP00000426252'}]
        self.vcf_path1 = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/vcftestfile1.vcf")  # general
        self.vcf_path2 = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/vcftestfile2.vcf")  # no annot
        self.vcf_path3 = os.path.join(os.path.dirname(inspect.getfile(epytope)), "Data/examples/vcftestfile3.vcf")  # checkallvatiationtypesindetail
        self.expected_biomart_mart_header = ["database","default","displayName","host", "includeDatasets","martUser","name","path","port","serverVirtualSchema","visible"]

    def test_read_lines(self):
        alleles = FileReader.read_lines(self.ale_path, in_type=Allele)
        self.assertEqual(len(alleles), 2)
        self.assertRaises(IOError, FileReader.read_lines, self.ale_no_path, in_type=Allele)
        self.assertRaises(ValueError, FileReader.read_lines, self.ale_zonk_path, in_type=Allele)

    def test_read_fasta(self):
        seqs = FileReader.read_fasta(self.fa_path)
        self.assertEqual(len(seqs), 2)
        seqs = FileReader.read_fasta(self.fa_unconventional_path)  # no "|"
        self.assertEqual(len(seqs), 174)

    def test_read_annovar_exonic(self):
        ano = FileReader.read_annovar_exonic(self.ano_path)
        self.assertEqual(len(ano), 5)

    def test_read_vcf(self):
        #general
        vcfvars, accessions = FileReader.read_vcf(self.vcf_path1)
        self.assertEqual(len(vcfvars), 17)
        #no annotations
        self.assertWarnings(UserWarning, FileReader.read_vcf, self.vcf_path2)
        #variation types
        vcfvars, accessions = FileReader.read_vcf(self.vcf_path3)
        self.assertEqual(len(vcfvars), 5) #stopgaininsertion will yield no variant
        self.assertEqual(vcfvars[0].coding['NM_014675'].cdsMutationSyntax, 'c.6026C>G')
        self.assertEqual(vcfvars[1].coding['NM_015237'].cdsMutationSyntax, 'c.1225C>T')
        self.assertEqual(vcfvars[2].coding['NM_014826'].cdsMutationSyntax, 'c.2091delG')
        self.assertEqual(vcfvars[2].coding['NM_003607'].cdsMutationSyntax, 'c.2334delG')
        self.assertEqual(vcfvars[3].coding['NM_002318'].cdsMutationSyntax, 'c.142_162delCAGGCCCCCGCCAACGTGGCC')
        self.assertEqual(vcfvars[4].coding['NM_152888'].cdsMutationSyntax, 'c.2086G>T')

    def test_EnsemblAdapter(self):
        ed = EnsemblDB()
        ed.read_seqs(self.edb_cds_path)
        self.assertEqual(len(ed.collection), 30)
        self.assertEqual(ed.get_transcript_sequence("ENST00000348405", type=EIdentifierTypes.ENSEMBL),
                         ed.get_transcript_information("ENST00000348405", type=EIdentifierTypes.ENSEMBL)[2])

        ed.read_seqs(self.edb_pep_path)
        self.assertEqual(len(ed.collection), 60)
        self.assertEqual(str(ed.get_product_sequence("ENSP00000337602", type=EIdentifierTypes.ENSEMBL).seq),
                         str(ed.get_transcript_information("ENSP00000337602", type=EIdentifierTypes.ENSEMBL)[2]))
        self.assertEqual(ed.get_transcript_information("ENSP00000337602", type=EIdentifierTypes.ENSEMBL)[1], '-')
        self.assertEqual(ed.get_transcript_information("ENSP00000337602", type=EIdentifierTypes.ENSEMBL)[0],
                         self.ENSEMBL_ensg)

    def test_MartsAdapter(self):
        # test most recent and stable GRCh37 Mart version
        urls = ["https://www.ensembl.org", "https://grch37.ensembl.org"]
        self.maxDiff = None
        for url in urls:
            ma = MartsAdapter(biomart=url)
            if "grch37" in url:
                self.assertEqual((1515, 1529), ma.get_transcript_position(
                    'ENST00000361221', 17953929, 17953943, type=EIdentifierTypes.ENSEMBL))
                self.assertEqual("TP53", ma.get_gene_by_position(17, 7566927, 7566927))
                self.assertEqual(self.ENST00000361221[2], ma.get_transcript_sequence(
                'ENST00000361221', type=EIdentifierTypes.ENSEMBL))
                self.assertIsNone(ma.get_transcript_sequence(
                "ENST00000614237", type=EIdentifierTypes.ENSEMBL))
                self.assertEqual(self.ENST00000000233_5[2], ma.get_transcript_sequence(
                "ENST00000000233.5", type=EIdentifierTypes.ENSEMBL))
                self.assertDictEqual(self.ENST00000361221, ma.get_transcript_information(
                'ENST00000361221', type=EIdentifierTypes.ENSEMBL))
                self.assertIsNone(ma.get_transcript_information(
                "ENST00000614237", type=EIdentifierTypes.ENSEMBL))
                self.assertEqual(self.ENST00000361221, ma.get_transcript_information_from_protein_id('ENSP00000355060'))
                self.assertEqual(ma.get_ensembl_ids_from_gene("TP53")[0][EAdapterFields.PROTID], ma.get_ensembl_ids_from_gene("ENSG00000141510.11", type=EIdentifierTypes.ENSEMBL)[0][EAdapterFields.PROTID])
            else:
                self.assertEqual((1054.0, 1054.0),ma.get_transcript_position(
                    "ENST00000614237", 7566927, 7566927, type=EIdentifierTypes.ENSEMBL))
                self.assertEqual("SENP3", ma.get_gene_by_position(17, 7566927, 7566927))
                self.assertEqual(self.ENST00000361221_grch38[2], ma.get_transcript_sequence(
                'ENST00000361221', type=EIdentifierTypes.ENSEMBL))
                self.assertIsNotNone(ma.get_transcript_sequence(
                "ENST00000614237", type=EIdentifierTypes.ENSEMBL))
                self.assertDictEqual(self.ENST00000361221_grch38, ma.get_transcript_information(
                'ENST00000361221', type=EIdentifierTypes.ENSEMBL))
                self.assertIsNotNone(ma.get_transcript_information(
                "ENST00000614237", type=EIdentifierTypes.ENSEMBL))
                self.assertEqual(self.ENST00000361221_grch38, ma.get_transcript_information_from_protein_id('ENSP00000355060'))
            self.assertIsNone(ma.get_product_sequence(
                "Q15942", type=EIdentifierTypes.UNIPROT))
            self.assertEqual(self.NP_001005353, ma.get_product_sequence(
                "NP_001005353", type=EIdentifierTypes.REFSEQ))
            self.assertEqual(self.ENSP00000369497, ma.get_product_sequence(
                "ENSP00000369497", type=EIdentifierTypes.ENSEMBL))
            self.assertIsNotNone(ma.get_ensembl_ids_from_gene("TP53"))
            self.assertIsNotNone(ma.get_genes_from_location("1","100000","200000"))
            self.assertIsNotNone(ma.get_protein_ids_from_transcripts(["ENST00000361221"]))
            self.assertEqual(ma.get_protein_ids_from_transcripts(["ENST00000361221"])["uniprot_id"][0], "Q9HCE6")
            self.assertIsNotNone(ma.get_variants_from_transcript_id('ENST00000361221'))
            self.assertTrue(all(ma.get_marts().columns == self.expected_biomart_mart_header))
            self.assertIsNotNone(ma.get_datasets("ENSEMBL_MART_ENSEMBL"))
            self.assertEqual(ma.get_gene_names_from_ids(["ENSG00000000003"])["gene_name"][0],"TSPAN6")

    def test_UniProtAdapter(self):
        up_adapter = UniProtDB("uniprot_adapter")
        up_adapter.read_seqs(self.fa_path)
        self.assertTrue(up_adapter.exists("QLHQEI"))
        self.assertFalse(up_adapter.exists("RANDOM"))

    def test_RefSeqAdapter(self):
        self.assertWarnings(DeprecationWarning, RefSeqAdapter,"1","2","3","4")