"""Autogenerated API for ReaExtensions by Julian Sader.

Repo on GitHub: https://github.com/juliansader/ReaExtensions
Theme on ReaperForums: https://forum.cockos.com/showthread.php?t=212174
"""

import typing as ty
import reapy_boost
import ctypes as ct
from reapy_boost.core import ReapyObject
if reapy_boost.is_inside_reaper():
    from reapy_boost.additional_api import packp, unpackp, packs_l, unpacks_l
    from reaper_python import _ft
else:
    from reapy_boost.additional_api import packp, unpackp

MAX_STRBUF = 4 * 1024 * 1024

__all__: ty.List[str] = [
    "Pointer",
    "VoidPtr",
    "LICE_IBitmap",
    "AudioWriter",
    "PCM_source",
    "ReaScriptAPI_Version",
    "Localize",
    "Mem_Alloc",
    "Mem_Free",
    "Mem_FromString",
    "String",
    "Int",
    "Byte",
    "Double",
    "Dialog_BrowseForSaveFile",
    "Dialog_BrowseForOpenFiles",
    "Dialog_BrowseForFolder",
    "Window_GetRect",
    "Window_ScreenToClient",
    "Window_ClientToScreen",
    "Window_GetClientRect",
    "Window_GetClientSize",
    "Window_MonitorFromRect",
    "Window_GetViewportFromRect",
    "Window_Update",
    "Window_InvalidateRect",
    "Window_FromPoint",
    "Window_GetParent",
    "Window_SetParent",
    "Window_IsChild",
    "Window_GetRelated",
    "Window_FindChildByID",
    "Window_SetFocus",
    "Window_GetFocus",
    "Window_SetForeground",
    "Window_GetForeground",
    "Window_Create",
    "Window_EnableMetal",
    "Window_Enable",
    "Window_Destroy",
    "Window_Show",
    "Window_IsVisible",
    "Window_IsWindow",
    "Window_FindEx",
    "Window_Find",
    "Window_FindTop",
    "Window_FindChild",
    "Window_ArrayAllChild",
    "Window_ArrayAllTop",
    "Window_ArrayFind",
    "Window_ListAllChild",
    "Window_ListAllTop",
    "Window_ListFind",
    "MIDIEditor_ListAll",
    "MIDIEditor_ArrayAll",
    "Window_SetPosition",
    "Window_Resize",
    "Window_Move",
    "Window_SetZOrder",
    "Window_GetLongPtr",
    "Window_GetLong",
    "Window_SetLong",
    "Window_SetStyle",
    "Window_SetOpacity",
    "Window_GetTitle",
    "Window_SetTitle",
    "Window_GetClassName",
    "Window_HandleFromAddress",
    "Window_AddressFromHandle",
    "WindowMessage_Post",
    "WindowMessage_Send",
    "WindowMessage_Peek",
    "WindowMessage_Intercept",
    "WindowMessage_InterceptList",
    "WindowMessage_PassThrough",
    "WindowMessage_ListIntercepts",
    "WindowMessage_Release",
    "WindowMessage_ReleaseWindow",
    "WindowMessage_ReleaseAll",
    "Window_OnCommand",
    "VKeys_GetState",
    "VKeys_GetDown",
    "VKeys_GetUp",
    "VKeys_Intercept",
    "Mouse_GetState",
    "Mouse_GetCursor",
    "Mouse_SetPosition",
    "Mouse_LoadCursor",
    "Mouse_LoadCursorFromFile",
    "Mouse_SetCursor",
    "Window_GetScrollInfo",
    "Window_SetScrollPos",
    "GDI_GetClientDC",
    "GDI_GetWindowDC",
    "GDI_GetScreenDC",
    "GDI_ReleaseDC",
    "GDI_CreateFillBrush",
    "GDI_CreatePen",
    "GDI_CreateFont",
    "GDI_SelectObject",
    "GDI_DeleteObject",
    "GDI_FillRect",
    "GDI_FillRoundRect",
    "GDI_FillPolygon",
    "GDI_FillEllipse",
    "GDI_GetSysColor",
    "GDI_SetTextBkMode",
    "GDI_SetTextBkColor",
    "GDI_SetTextColor",
    "GDI_GetTextColor",
    "GDI_DrawText",
    "GDI_SetPixel",
    "GDI_Line",
    "GDI_Polyline",
    "GDI_Blit",
    "GDI_StretchBlit",
    "Composite",
    "Composite_Unlink",
    "Composite_ListBitmaps",
    "Composite_Delay",
    "LICE_CreateBitmap",
    "LICE_ListAllBitmaps",
    "LICE_ArrayAllBitmaps",
    "LICE_GetHeight",
    "LICE_GetWidth",
    "LICE_GetDC",
    "LICE_DestroyBitmap",
    "LICE_LoadPNG",
    "LICE_Blit",
    "LICE_RotatedBlit",
    "LICE_ScaledBlit",
    "LICE_IsFlipped",
    "LICE_Resize",
    "LICE_Clear",
    "LICE_CreateFont",
    "LICE_DestroyFont",
    "LICE_SetFontFromGDI",
    "LICE_SetFontColor",
    "LICE_SetFontBkColor",
    "LICE_DrawText",
    "LICE_DrawChar",
    "LICE_MeasureText",
    "LICE_GradRect",
    "LICE_FillRect",
    "LICE_FillTriangle",
    "LICE_FillPolygon",
    "LICE_FillCircle",
    "LICE_Line",
    "LICE_Bezier",
    "LICE_Arc",
    "LICE_Circle",
    "LICE_RoundRect",
    "LICE_GetPixel",
    "LICE_PutPixel",
    "LICE_WritePNG",
    "LICE_WriteJPG",
    "LICE_LoadJPG",
    "LICE_SetAlphaFromColorMask",
    "LICE_AlterBitmapHSV",
    "LICE_AlterRectHSV",
    "LICE_ProcessRect",
    "Window_AttachTopmostPin",
    "Window_AttachResizeGrip",
    "ListView_GetItemCount",
    "ListView_GetSelectedCount",
    "ListView_EnsureVisible",
    "ListView_GetFocusedItem",
    "ListView_EnumSelItems",
    "ListView_GetItem",
    "ListView_GetItemText",
    "ListView_GetItemState",
    "ListView_ListAllSelItems",
    "Xen_AudioWriter_Create",
    "Xen_AudioWriter_Destroy",
    "Xen_AudioWriter_Write",
    "Xen_GetMediaSourceSamples",
    "Xen_StartSourcePreview",
    "Xen_StopSourcePreview"
]


class Pointer(ReapyObject):

    def __init__(
        self, ptr: ty.Union[str, int], ptr_str: str = "void*"
    ) -> None:
        self.__args = ptr, ptr_str
        if isinstance(ptr, str):
            self._str = ptr
            self._int = packp(ptr_str, ptr)
            return
        if isinstance(ptr, int):
            self._str = unpackp(ptr_str, ptr)
            self._int = ptr
            return
        raise TypeError("expect int or str, passed %s" % ptr)

    @property
    def _args(self) -> ty.Tuple[ty.Union[str, int], str]:
        return self.__args

    def __str__(self) -> str:
        return self._str

    def __int__(self) -> int:
        return self._int

    def __eq__(self, other: object) -> bool:
        if isinstance(other, (str, Pointer)):
            return self._str == other
        if isinstance(other, int):
            return self._int == other
        return False

    def __hash__(self) -> int:
        return int(self)


class VoidPtr(Pointer):
    pass


class LICE_IBitmap(Pointer):
    pass


class AudioWriter(Pointer):
    pass


class PCM_source(Pointer):
    pass



@reapy_boost.inside_reaper()
def ReaScriptAPI_Version() -> float:
    """
    Returns the version of the js_ReaScriptAPI extension.

    Returns
    -------
    version: float
    """
    a = _ft['JS_ReaScriptAPI_Version']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    prot_version = ct.c_double(1.0)
    ret = f(ct.byref(prot_version))
    return float(prot_version.value)


@reapy_boost.inside_reaper()
def Localize(USEnglish: str, LangPackSection: str, translationOut_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    Returns the translation of the given US English text, according to the
    currently loaded Language Pack.
    
    Parameters:
     * LangPackSection:
    Language Packs are divided into sections such as 'common' or
    'DLG_102'.
     * In Lua, by default, text of up to 1024 chars can be
    returned. To increase (or reduce) the default buffer size, a string
    and size can be included as optional 3rd and 4th arguments.
    
    Example:
    reaper.JS_Localize('Actions', 'common', '', 20)

    Returns
    -------
    translation: str
    """
    a = _ft['JS_Localize']
    f = ct.CFUNCTYPE(None, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_int)(a)
    prot_translation = packs_l("", size=translationOut_sz, encoding=encoding)
    prot_USEnglish = packs_l(USEnglish, size=len(USEnglish.encode(encoding)), encoding=encoding)
    prot_LangPackSection = packs_l(LangPackSection, size=len(LangPackSection.encode(encoding)), encoding=encoding)
    ret = f(prot_USEnglish, prot_LangPackSection, prot_translation, ct.c_int(translationOut_sz))
    return unpacks_l(prot_translation, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Mem_Alloc(sizeBytes: int) -> VoidPtr:
    """
    Allocates memory for general use by functions that require memory
    buffers.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Mem_Alloc']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_int)(a)
    
    ret = f(ct.c_int(sizeBytes))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Mem_Free(mallocPointer: VoidPtr) -> bool:
    """
    Frees memory that was previously allocated by JS_Mem_Alloc.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Mem_Free']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(mallocPointer)))
    return bool(ret)


@reapy_boost.inside_reaper()
def Mem_FromString(mallocPointer: VoidPtr, offset: int, packedString: str, stringLength: int, encoding: str="utf-8") -> bool:
    """
    Copies a packed string into a memory buffer.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Mem_FromString']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_int, ct.c_char_p, ct.c_int)(a)
    prot_packedString = packs_l(packedString, size=len(packedString.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(mallocPointer)), ct.c_int(offset), prot_packedString, ct.c_int(stringLength))
    return bool(ret)


@reapy_boost.inside_reaper()
def String(pointer: VoidPtr, offset: int, lengthChars: int, bufOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[bool, str]:
    """
    Returns the memory contents starting at address[offset] as a packed
    string. Offset is added as steps of 1 byte (char) each.

    Returns
    -------
    ret_value: bool
    bufNeedBig: str
    """
    a = _ft['JS_String']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_int, ct.c_int, ct.c_char_p, ct.c_int)(a)
    prot_bufNeedBig = packs_l("", size=bufOutNeedBig_sz, encoding=encoding)
    ret = f(packp("void*", str(pointer)), ct.c_int(offset), ct.c_int(lengthChars), prot_bufNeedBig, ct.c_int(bufOutNeedBig_sz))
    return bool(ret), unpacks_l(prot_bufNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Int(pointer: VoidPtr, offset: int) -> int:
    """
    Returns the 4-byte signed integer at address[offset]. Offset is added
    as steps of 4 bytes each.

    Returns
    -------
    int: int
    """
    a = _ft['JS_Int']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_void_p)(a)
    prot_int = ct.c_int(1)
    ret = f(packp("void*", str(pointer)), ct.c_int(offset), ct.byref(prot_int))
    return prot_int.value


@reapy_boost.inside_reaper()
def Byte(pointer: VoidPtr, offset: int) -> int:
    """
    Returns the unsigned byte at address[offset]. Offset is added as steps
    of 1 byte each.

    Returns
    -------
    byte: int
    """
    a = _ft['JS_Byte']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_void_p)(a)
    prot_byte = ct.c_int(1)
    ret = f(packp("void*", str(pointer)), ct.c_int(offset), ct.byref(prot_byte))
    return prot_byte.value


@reapy_boost.inside_reaper()
def Double(pointer: VoidPtr, offset: int) -> float:
    """
    Returns the 8-byte floating point value at address[offset]. Offset is
    added as steps of 8 bytes each.

    Returns
    -------
    double: float
    """
    a = _ft['JS_Double']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_void_p)(a)
    prot_double = ct.c_double(1.0)
    ret = f(packp("void*", str(pointer)), ct.c_int(offset), ct.byref(prot_double))
    return float(prot_double.value)


@reapy_boost.inside_reaper()
def Dialog_BrowseForSaveFile(windowTitle: str, initialFolder: str, initialFile: str, extensionList: str, fileNameOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    retval is 1 if a file was selected, 0 if the user cancelled the
    dialog, or negative if an error occurred.
    
    extensionList is as
    described for JS_Dialog_BrowseForOpenFiles.

    Returns
    -------
    ret_value: int
    fileNameNeedBig: str
    """
    a = _ft['JS_Dialog_BrowseForSaveFile']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_int)(a)
    prot_fileNameNeedBig = packs_l("", size=fileNameOutNeedBig_sz, encoding=encoding)
    prot_windowTitle = packs_l(windowTitle, size=len(windowTitle.encode(encoding)), encoding=encoding)
    prot_initialFolder = packs_l(initialFolder, size=len(initialFolder.encode(encoding)), encoding=encoding)
    prot_initialFile = packs_l(initialFile, size=len(initialFile.encode(encoding)), encoding=encoding)
    prot_extensionList = packs_l(extensionList, size=len(extensionList.encode(encoding)), encoding=encoding)
    ret = f(prot_windowTitle, prot_initialFolder, prot_initialFile, prot_extensionList, prot_fileNameNeedBig, ct.c_int(fileNameOutNeedBig_sz))
    return int(ret), unpacks_l(prot_fileNameNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Dialog_BrowseForOpenFiles(windowTitle: str, initialFolder: str, initialFile: str, extensionList: str, allowMultiple: bool, fileNamesOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    If allowMultiple is true, multiple files may be selected. The returned
    string is \\0-separated, with the first substring containing the
    folder path and subsequent substrings containing the file names.
     * On
    macOS, the first substring may be empty, and each file name will then
    contain its entire path.
     * This function only allows selection of
    existing files, and does not allow creation of new files.
    extensionList is a string containing pairs of \\0-terminated
    substrings. The last substring must be terminated by two \\0
    characters. Each pair defines one filter pattern:
     * The first
    substring in each pair describes the filter in user-readable form (for
    example, 'Lua script files (*.lua)') and will be displayed in the
    dialog box.
     * The second substring specifies the filter that the
    operating system must use to search for the files (for example,
    '*.txt'; the wildcard should not be omitted). To specify multiple
    extensions for a single display string, use a semicolon to separate
    the patterns (for example, '*.lua;*.eel').
    
    An example of an
    extensionList string:
    'ReaScript files\\0*.lua;*.eel\\0Lua files
    (.lua)\\0*.lua\\0EEL files (.eel)\\0*.eel\\0\\0'.
    
    On macOS, file
    dialogs do not accept empty extensionLists, nor wildcard extensions
    (such as 'All files\\0*.*\\0\\0'), so each acceptable extension must
    be listed explicitly. On Linux and Windows, wildcard extensions are
    acceptable, and if the extensionList string is empty, the dialog will
    display a default 'All files (*.*)' filter.
    
    retval is 1 if one or
    more files were selected, 0 if the user cancelled the dialog, or
    negative if an error occurred.
    
    Displaying \\0-separated strings:
     *
    REAPER's IDE and ShowConsoleMsg only display strings up to the first
    \\0 byte. If multiple files were selected, only the first substring
    containing the path will be displayed. This is not a problem for Lua
    or EEL, which can access the full string beyond the first \\0 byte as
    usual.

    Returns
    -------
    ret_value: int
    fileNamesNeedBig: str
    """
    a = _ft['JS_Dialog_BrowseForOpenFiles']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_byte, ct.c_char_p, ct.c_int)(a)
    prot_fileNamesNeedBig = packs_l("", size=fileNamesOutNeedBig_sz, encoding=encoding)
    prot_windowTitle = packs_l(windowTitle, size=len(windowTitle.encode(encoding)), encoding=encoding)
    prot_initialFolder = packs_l(initialFolder, size=len(initialFolder.encode(encoding)), encoding=encoding)
    prot_initialFile = packs_l(initialFile, size=len(initialFile.encode(encoding)), encoding=encoding)
    prot_extensionList = packs_l(extensionList, size=len(extensionList.encode(encoding)), encoding=encoding)
    ret = f(prot_windowTitle, prot_initialFolder, prot_initialFile, prot_extensionList, ct.c_byte(allowMultiple), prot_fileNamesNeedBig, ct.c_int(fileNamesOutNeedBig_sz))
    return int(ret), unpacks_l(prot_fileNamesNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Dialog_BrowseForFolder(caption: str, initialFolder: str, folderOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    retval is 1 if a file was selected, 0 if the user cancelled the
    dialog, and -1 if an error occurred.

    Returns
    -------
    ret_value: int
    folderNeedBig: str
    """
    a = _ft['JS_Dialog_BrowseForFolder']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_char_p, ct.c_char_p, ct.c_int)(a)
    prot_folderNeedBig = packs_l("", size=folderOutNeedBig_sz, encoding=encoding)
    prot_caption = packs_l(caption, size=len(caption.encode(encoding)), encoding=encoding)
    prot_initialFolder = packs_l(initialFolder, size=len(initialFolder.encode(encoding)), encoding=encoding)
    ret = f(prot_caption, prot_initialFolder, prot_folderNeedBig, ct.c_int(folderOutNeedBig_sz))
    return int(ret), unpacks_l(prot_folderNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Window_GetRect(windowHWND: VoidPtr) -> ty.Tuple[bool, int, int, int, int]:
    """
    Retrieves the screen coordinates of the bounding rectangle of the
    specified window.
    
    NOTES:
     * On Windows and Linux, coordinates are
    relative to *upper* left corner of the primary display, and the
    positive Y-axis points downward.
     * On macOS, coordinates are relative
    to the *bottom* left corner of the primary display, and the positive
    Y-axis points upward.
     * The pixel at (right, bottom) lies immediately
    outside the rectangle.

    Returns
    -------
    ret_value: bool
    left: int
    top: int
    right: int
    bottom: int
    """
    a = _ft['JS_Window_GetRect']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_left = ct.c_int(1)
    prot_top = ct.c_int(1)
    prot_right = ct.c_int(1)
    prot_bottom = ct.c_int(1)
    ret = f(packp("void*", str(windowHWND)), ct.byref(prot_left), ct.byref(prot_top), ct.byref(prot_right), ct.byref(prot_bottom))
    return bool(ret), prot_left.value, prot_top.value, prot_right.value, prot_bottom.value


@reapy_boost.inside_reaper()
def Window_ScreenToClient(windowHWND: VoidPtr, x: int, y: int) -> ty.Tuple[int, int]:
    """
    Converts the screen coordinates of a specified point on the screen to
    client-area coordinates.
    
    NOTES:
     * On Windows and Linux, screen
    coordinates are relative to *upper* left corner of the primary
    display, and the positive Y-axis points downward.
     * On macOS, screen
    coordinates are relative to the *bottom* left corner of the primary
    display, and the positive Y-axis points upward.
     * On all platforms,
    client coordinates are relative to the upper left corner of the client
    area.

    Returns
    -------
    x: int
    y: int
    """
    a = _ft['JS_Window_ScreenToClient']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_void_p, ct.c_void_p)(a)
    prot_x = ct.c_int(1)
    prot_y = ct.c_int(1)
    ret = f(packp("void*", str(windowHWND)), ct.c_int(x), ct.c_int(y), ct.byref(prot_x), ct.byref(prot_y))
    return prot_x.value, prot_y.value


@reapy_boost.inside_reaper()
def Window_ClientToScreen(windowHWND: VoidPtr, x: int, y: int) -> ty.Tuple[int, int]:
    """
    Converts the client-area coordinates of a specified point to screen
    coordinates.
    
    NOTES:
     * On Windows and Linux, screen coordinates are
    relative to *upper* left corner of the primary display, and the
    positive Y-axis points downward.
     * On macOS, screen coordinates are
    relative to the *bottom* left corner of the primary display, and the
    positive Y-axis points upward.
     * On all platforms, client coordinates
    are relative to the upper left corner of the client area.

    Returns
    -------
    x: int
    y: int
    """
    a = _ft['JS_Window_ClientToScreen']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_void_p, ct.c_void_p)(a)
    prot_x = ct.c_int(1)
    prot_y = ct.c_int(1)
    ret = f(packp("void*", str(windowHWND)), ct.c_int(x), ct.c_int(y), ct.byref(prot_x), ct.byref(prot_y))
    return prot_x.value, prot_y.value


@reapy_boost.inside_reaper()
def Window_GetClientRect(windowHWND: VoidPtr) -> ty.Tuple[bool, int, int, int, int]:
    """
    Retrieves the screen coordinates of the client area rectangle of the
    specified window.
    
    NOTES:
     * Unlike the C++ function GetClientRect,
    this function returns the screen coordinates, not the width and
    height. To get the client size, use GetClientSize.
     * The pixel at
    (right, bottom) lies immediately outside the rectangle.
     * On Windows
    and Linux, screen coordinates are relative to *upper* left corner of
    the primary display, and the positive Y-axis points downward.
     * On
    macOS, screen coordinates are relative to the *bottom* left corner of
    the primary display, and the positive Y-axis points upward.

    Returns
    -------
    ret_value: bool
    left: int
    top: int
    right: int
    bottom: int
    """
    a = _ft['JS_Window_GetClientRect']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_left = ct.c_int(1)
    prot_top = ct.c_int(1)
    prot_right = ct.c_int(1)
    prot_bottom = ct.c_int(1)
    ret = f(packp("void*", str(windowHWND)), ct.byref(prot_left), ct.byref(prot_top), ct.byref(prot_right), ct.byref(prot_bottom))
    return bool(ret), prot_left.value, prot_top.value, prot_right.value, prot_bottom.value


@reapy_boost.inside_reaper()
def Window_GetClientSize(windowHWND: VoidPtr) -> ty.Tuple[bool, int, int]:
    """
    

    Returns
    -------
    ret_value: bool
    width: int
    height: int
    """
    a = _ft['JS_Window_GetClientSize']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_width = ct.c_int(1)
    prot_height = ct.c_int(1)
    ret = f(packp("void*", str(windowHWND)), ct.byref(prot_width), ct.byref(prot_height))
    return bool(ret), prot_width.value, prot_height.value


@reapy_boost.inside_reaper()
def Window_MonitorFromRect(x1: int, y1: int, x2: int, y2: int, wantWork: bool) -> ty.Tuple[int, int, int, int]:
    """
    Deprecated - use GetViewportFromRect instead.

    Returns
    -------
    left: int
    top: int
    right: int
    bottom: int
    """
    a = _ft['JS_Window_MonitorFromRect']
    f = ct.CFUNCTYPE(None, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_byte, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_left = ct.c_int(1)
    prot_top = ct.c_int(1)
    prot_right = ct.c_int(1)
    prot_bottom = ct.c_int(1)
    ret = f(ct.c_int(x1), ct.c_int(y1), ct.c_int(x2), ct.c_int(y2), ct.c_byte(wantWork), ct.byref(prot_left), ct.byref(prot_top), ct.byref(prot_right), ct.byref(prot_bottom))
    return prot_left.value, prot_top.value, prot_right.value, prot_bottom.value


@reapy_boost.inside_reaper()
def Window_GetViewportFromRect(x1: int, y1: int, x2: int, y2: int, wantWork: bool) -> ty.Tuple[int, int, int, int]:
    """
    Retrieves the dimensions of the display monitor that has the largest
    area of intersection with the specified rectangle.
    
    If the monitor is
    not the primary display, some of the rectangle's coordinates may be
    negative.
    
    wantWork: Returns the work area of the display, which
    excludes the system taskbar or application desktop toolbars.

    Returns
    -------
    left: int
    top: int
    right: int
    bottom: int
    """
    a = _ft['JS_Window_GetViewportFromRect']
    f = ct.CFUNCTYPE(None, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_byte, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_left = ct.c_int(1)
    prot_top = ct.c_int(1)
    prot_right = ct.c_int(1)
    prot_bottom = ct.c_int(1)
    ret = f(ct.c_int(x1), ct.c_int(y1), ct.c_int(x2), ct.c_int(y2), ct.c_byte(wantWork), ct.byref(prot_left), ct.byref(prot_top), ct.byref(prot_right), ct.byref(prot_bottom))
    return prot_left.value, prot_top.value, prot_right.value, prot_bottom.value


@reapy_boost.inside_reaper()
def Window_Update(windowHWND: VoidPtr) -> None:
    """
    Similar to the Win32 function UpdateWindow.

    Returns
    -------
    
    """
    a = _ft['JS_Window_Update']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def Window_InvalidateRect(windowHWND: VoidPtr, left: int, top: int, right: int, bottom: int, eraseBackground: bool) -> bool:
    """
    Similar to the Win32 function InvalidateRect.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_InvalidateRect']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_byte)(a)
    
    ret = f(packp("void*", str(windowHWND)), ct.c_int(left), ct.c_int(top), ct.c_int(right), ct.c_int(bottom), ct.c_byte(eraseBackground))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_FromPoint(x: int, y: int) -> VoidPtr:
    """
    Retrieves a HWND to the window that contains the specified point.
    NOTES:
     * On Windows and Linux, screen coordinates are relative to
    *upper* left corner of the primary display, and the positive Y-axis
    points downward.
     * On macOS, screen coordinates are relative to the
    *bottom* left corner of the primary display, and the positive Y-axis
    points upward.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_FromPoint']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_int(x), ct.c_int(y))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_GetParent(windowHWND: VoidPtr) -> VoidPtr:
    """
    Retrieves a HWND to the specified window's parent or owner.
    Returns
    NULL if the window is unowned or if the function otherwise fails.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_GetParent']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_SetParent(childHWND: VoidPtr, parentHWND: VoidPtr) -> VoidPtr:
    """
    If successful, returns a handle to the previous parent window.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_SetParent']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(childHWND)), packp("void*", str(parentHWND)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_IsChild(parentHWND: VoidPtr, childHWND: VoidPtr) -> bool:
    """
    Determines whether a window is a child window or descendant window of
    a specified parent window.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_IsChild']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(parentHWND)), packp("void*", str(childHWND)))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_GetRelated(windowHWND: VoidPtr, relation: str, encoding: str="utf-8") -> VoidPtr:
    """
    Retrieves a handle to a window that has the specified relationship
    (Z-Order or owner) to the specified window.
    relation: 'LAST', 'NEXT',
    'PREV', 'OWNER' or 'CHILD'.
    (Refer to documentation for Win32 C++
    function GetWindow.)

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_GetRelated']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_char_p)(a)
    prot_relation = packs_l(relation, size=len(relation.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_relation)
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_FindChildByID(parentHWND: VoidPtr, ID: int) -> VoidPtr:
    """
    Similar to the C++ WIN32 function GetDlgItem, this function finds
    child windows by ID.
    
    (The ID of a window may be retrieved by
    JS_Window_GetLongPtr.)

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_FindChildByID']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(parentHWND)), ct.c_int(ID))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_SetFocus(windowHWND: VoidPtr) -> None:
    """
    Sets the keyboard focus to the specified window.

    Returns
    -------
    
    """
    a = _ft['JS_Window_SetFocus']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def Window_GetFocus() -> VoidPtr:
    """
    Retrieves a HWND to the window that has the keyboard focus, if the
    window is attached to the calling thread's message queue.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_GetFocus']
    f = ct.CFUNCTYPE(ct.c_void_p, )(a)
    
    ret = f()
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_SetForeground(windowHWND: VoidPtr) -> None:
    """
    Brings the specified window into the foreground, activates the window,
    and directs keyboard input to it.

    Returns
    -------
    
    """
    a = _ft['JS_Window_SetForeground']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def Window_GetForeground() -> VoidPtr:
    """
    Retrieves a HWND to the top-level foreground window (the window with
    which the user is currently working).

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_GetForeground']
    f = ct.CFUNCTYPE(ct.c_void_p, )(a)
    
    ret = f()
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_Create(title: str, className: str, x: int, y: int, w: int, h: int, styleOptional:  ty.Optional[str]=None, ownerHWNDOptional:  ty.Optional[VoidPtr]=None, encoding: str="utf-8") -> VoidPtr:
    """
    Creates a modeless window with WS_OVERLAPPEDWINDOW style and only
    rudimentary features. Scripts can paint into the window using GDI or
    LICE/Composite functions (and JS_Window_InvalidateRect to trigger re-
    painting).
    
    style: An optional parameter that overrides the default
    style. The string may include any combination of standard window
    styles, such as 'POPUP' for a frameless window, or
    'CAPTION,SIZEBOX,SYSMENU' for a standard framed window.
    
    On Linux and
    macOS, 'MAXIMIZE' has not yet been implemented, and the remaining
    styles may appear slightly different from their WindowsOS
    counterparts.
    
    className: On Windows, only standard ANSI characters
    are supported.
    
    ownerHWND: Optional parameter, only available on
    WindowsOS.  Usually either the REAPER main window or another script
    window, and useful for ensuring that the created window automatically
    closes when the owner is closed.
    
    NOTE: On Linux and macOS, the window
    contents are only updated *between* defer cycles, so the window cannot
    be animated by for/while loops within a single defer cycle.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_Create']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_char_p, ct.c_char_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_char_p, ct.c_void_p)(a)
    prot_styleOptional = packs_l(styleOptional, size=len(styleOptional.encode(encoding)), encoding=encoding) if styleOptional is not None else None
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    prot_className = packs_l(className, size=len(className.encode(encoding)), encoding=encoding)
    ret = f(prot_title, prot_className, ct.c_int(x), ct.c_int(y), ct.c_int(w), ct.c_int(h), None if styleOptional is None else prot_styleOptional, None if ownerHWNDOptional is None else packp("void*", str(ownerHWNDOptional)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_EnableMetal(windowHWND: VoidPtr) -> int:
    """
    On macOS, returns the Metal graphics setting:
    2 = Metal enabled and
    support GetDC()/ReleaseDC() for drawing (more overhead).
    1 = Metal
    enabled.
    0 = N/A (Windows and Linux).
    -1 = non-metal async layered
    mode.
    -2 = non-metal non-async layered mode.
    
    WARNING: If using mode
    -1, any BitBlt()/StretchBlt() MUST have the source bitmap persist. If
    it is resized after Blit it could cause crashes.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_Window_EnableMetal']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return int(ret)


@reapy_boost.inside_reaper()
def Window_Enable(windowHWND: VoidPtr, enable: bool) -> None:
    """
    Enables or disables mouse and keyboard input to the specified window
    or control.

    Returns
    -------
    
    """
    a = _ft['JS_Window_Enable']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_byte)(a)
    
    ret = f(packp("void*", str(windowHWND)), ct.c_byte(enable))
    return 


@reapy_boost.inside_reaper()
def Window_Destroy(windowHWND: VoidPtr) -> None:
    """
    Destroys the specified window.

    Returns
    -------
    
    """
    a = _ft['JS_Window_Destroy']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def Window_Show(windowHWND: VoidPtr, state: str, encoding: str="utf-8") -> None:
    """
    Sets the specified window's show state.
    
    Parameters:
     * state: One of
    the following options: 'SHOW', 'SHOWNA' (or 'SHOWNOACTIVATE'),
    'SHOWMINIMIZED', 'HIDE', 'NORMAL', 'SHOWNORMAL', 'SHOWMAXIMIZED',
    'SHOWDEFAULT' or 'RESTORE'. On Linux and macOS, only the first four
    options are fully implemented.

    Returns
    -------
    
    """
    a = _ft['JS_Window_Show']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p)(a)
    prot_state = packs_l(state, size=len(state.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_state)
    return 


@reapy_boost.inside_reaper()
def Window_IsVisible(windowHWND: VoidPtr) -> bool:
    """
    Determines the visibility state of the window.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_IsVisible']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_IsWindow(windowHWND: VoidPtr) -> bool:
    """
    Determines whether the specified window handle identifies an existing
    window.
    
    On macOS and Linux, only windows that were created by
    WDL/swell will be identified (and only such windows should be acted on
    by scripts).
    
    NOTE: Since REAPER v5.974, windows can be checked using
    the native function ValidatePtr(windowHWND, 'HWND').

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_IsWindow']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_FindEx(parentHWND: VoidPtr, childHWND: VoidPtr, className: str, title: str, encoding: str="utf-8") -> VoidPtr:
    """
    Returns a handle to a child window whose class and title match the
    specified strings.
    
    Parameters: * childWindow: The function searches
    child windows, beginning with the window *after* the specified child
    window. If childHWND is equal to parentHWND, the search begins with
    the first child window of parentHWND.
     * title: An empty string, '',
    will match all windows. (Search is not case sensitive.)

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_FindEx']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_char_p, ct.c_char_p)(a)
    prot_className = packs_l(className, size=len(className.encode(encoding)), encoding=encoding)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(parentHWND)), packp("void*", str(childHWND)), prot_className, prot_title)
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_Find(title: str, exact: bool, encoding: str="utf-8") -> VoidPtr:
    """
    Returns a HWND to a window whose title matches the specified string.
    * Unlike the Win32 function FindWindow, this function searches top-
    level as well as child windows, so that the target window can be found
    irrespective of docked state.
     * In addition, the function can
    optionally match substrings of the title.
     * Matching is not case
    sensitive.
    
    Parameters:
     * exact: Match entire title, or match
    substring of title.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_Find']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_char_p, ct.c_byte)(a)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(prot_title, ct.c_byte(exact))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_FindTop(title: str, exact: bool, encoding: str="utf-8") -> VoidPtr:
    """
    Returns a HWND to a top-level window whose title matches the specified
    string.
    
    Parameters:
     * exact: Match entire title length, or match
    substring of title. In both cases, matching is not case sensitive.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_FindTop']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_char_p, ct.c_byte)(a)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(prot_title, ct.c_byte(exact))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_FindChild(parentHWND: VoidPtr, title: str, exact: bool, encoding: str="utf-8") -> VoidPtr:
    """
    Returns a HWND to a child window whose title matches the specified
    string.
    
    Parameters:
     * exact: Match entire title length, or match
    substring of title. In both cases, matching is not case sensitive.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_FindChild']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_char_p, ct.c_byte)(a)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(parentHWND)), prot_title, ct.c_byte(exact))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_ArrayAllChild(parentHWND: VoidPtr, reaperarray: VoidPtr) -> int:
    """
    Finds all child windows of the specified parent.
    
    Returns:
     * retval:
    The number of windows found; negative if an error occurred.
     * The
    addresses are stored in the provided reaper.array, and can be
    converted to REAPER objects (HWNDs) by the function
    JS_Window_HandleFromAddress.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_Window_ArrayAllChild']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(parentHWND)), packp("void*", str(reaperarray)))
    return int(ret)


@reapy_boost.inside_reaper()
def Window_ArrayAllTop(reaperarray: VoidPtr) -> int:
    """
    Finds all top-level windows.
    
    Returns:
     * retval: The number of
    windows found; negative if an error occurred.
     * The addresses are
    stored in the provided reaper.array, and can be converted to REAPER
    objects (HWNDs) by the function JS_Window_HandleFromAddress.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_Window_ArrayAllTop']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(reaperarray)))
    return int(ret)


@reapy_boost.inside_reaper()
def Window_ArrayFind(title: str, exact: bool, reaperarray: VoidPtr, encoding: str="utf-8") -> int:
    """
    Finds all windows, whether top-level or child, whose titles match the
    specified string.
    
    Returns:
     * retval: The number of windows found;
    negative if an error occurred.
     * The addresses are stored in the
    provided reaper.array, and can be converted to REAPER objects (HWNDs)
    by the function JS_Window_HandleFromAddress.
    
    Parameters:
     * exact:
    Match entire title exactly, or match substring of title.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_Window_ArrayFind']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_byte, ct.c_void_p)(a)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(prot_title, ct.c_byte(exact), packp("void*", str(reaperarray)))
    return int(ret)


@reapy_boost.inside_reaper()
def Window_ListAllChild(parentHWND: VoidPtr, listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Finds all child windows of the specified parent.
    
    Returns:
     * retval:
    The number of windows found; negative if an error occurred.
     * list: A
    comma-separated string of hexadecimal values.
    Each value is an address
    that can be converted to a HWND by the function
    Window_HandleFromAddress.

    Returns
    -------
    ret_value: int
    listNeedBig: str
    """
    a = _ft['JS_Window_ListAllChild']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    ret = f(packp("void*", str(parentHWND)), prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return int(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Window_ListAllTop(listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Finds all top-level windows.
    
    Returns:
     * retval: The number of
    windows found; negative if an error occurred.
     * list: A comma-
    separated string of hexadecimal values. Each value is an address that
    can be converted to a HWND by the function Window_HandleFromAddress.

    Returns
    -------
    ret_value: int
    listNeedBig: str
    """
    a = _ft['JS_Window_ListAllTop']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    ret = f(prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return int(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Window_ListFind(title: str, exact: bool, listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Finds all windows (whether top-level or child) whose titles match the
    specified string.
    
    Returns:
     * retval: The number of windows found;
    negative if an error occurred.
     * list: A comma-separated string of
    hexadecimal values. Each value is an address that can be converted to
    a HWND by the function Window_HandleFromAddress.
    
    Parameters:
     *
    exact: Match entire title exactly, or match substring of title.

    Returns
    -------
    ret_value: int
    listNeedBig: str
    """
    a = _ft['JS_Window_ListFind']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_byte, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(prot_title, ct.c_byte(exact), prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return int(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def MIDIEditor_ListAll(listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Finds all open MIDI windows (whether docked or not).
    
     * retval: The
    number of MIDI editor windows found; negative if an error occurred.
    * list: Comma-separated string of hexadecimal values. Each value is an
    address that can be converted to a HWND by the function
    Window_HandleFromAddress.

    Returns
    -------
    ret_value: int
    listNeedBig: str
    """
    a = _ft['JS_MIDIEditor_ListAll']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    ret = f(prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return int(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def MIDIEditor_ArrayAll(reaperarray: VoidPtr) -> int:
    """
    Finds all open MIDI windows (whether docked or not).
    
     * retval: The
    number of MIDI editor windows found; negative if an error occurred.
    * The address of each MIDI editor window is stored in the provided
    reaper.array. Each address can be converted to a REAPER object (HWND)
    by the function JS_Window_HandleFromAddress.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_MIDIEditor_ArrayAll']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(reaperarray)))
    return int(ret)


@reapy_boost.inside_reaper()
def Window_SetPosition(windowHWND: VoidPtr, left: int, top: int, width: int, height: int, ZOrderOptional:  ty.Optional[str]=None, flagsOptional:  ty.Optional[str]=None, encoding: str="utf-8") -> bool:
    """
    Interface to the Win32/swell function SetWindowPos, with which window
    position, size, Z-order and visibility can be set, and new frame
    styles can be applied.
    
    ZOrder and flags are optional parameters. If
    no arguments are supplied, the window will simply be moved and
    resized, as if the NOACTIVATE, NOZORDER, NOOWNERZORDER flags were set.
    * ZOrder: 'BOTTOM', 'TOPMOST', 'NOTOPMOST', 'TOP' or a window HWND
    converted to a string, for example by the Lua function tostring.
     *
    flags: Any combination of the standard flags, of which 'NOMOVE',
    'NOSIZE', 'NOZORDER', 'NOACTIVATE', 'SHOWWINDOW', 'FRAMECHANGED' and
    'NOCOPYBITS' should be valid cross-platform.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_SetPosition']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_char_p, ct.c_char_p)(a)
    prot_ZOrderOptional = packs_l(ZOrderOptional, size=len(ZOrderOptional.encode(encoding)), encoding=encoding) if ZOrderOptional is not None else None
    prot_flagsOptional = packs_l(flagsOptional, size=len(flagsOptional.encode(encoding)), encoding=encoding) if flagsOptional is not None else None
    ret = f(packp("void*", str(windowHWND)), ct.c_int(left), ct.c_int(top), ct.c_int(width), ct.c_int(height), None if ZOrderOptional is None else prot_ZOrderOptional, None if flagsOptional is None else prot_flagsOptional)
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_Resize(windowHWND: VoidPtr, width: int, height: int) -> None:
    """
    Changes the dimensions of the specified window, keeping the top left
    corner position constant.
     * If resizing script GUIs, call
    gfx.update() after resizing.
    * Equivalent to calling
    JS_Window_SetPosition with NOMOVE, NOZORDER, NOACTIVATE and
    NOOWNERZORDER flags set.

    Returns
    -------
    
    """
    a = _ft['JS_Window_Resize']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(windowHWND)), ct.c_int(width), ct.c_int(height))
    return 


@reapy_boost.inside_reaper()
def Window_Move(windowHWND: VoidPtr, left: int, top: int) -> None:
    """
    Changes the position of the specified window, keeping its size
    constant.
    
    NOTES:
     * For top-level windows, position is relative to
    the primary display.
     * On Windows and Linux, position is calculated
    as the coordinates of the upper left corner of the window, relative to
    upper left corner of the primary display, and the positive Y-axis
    points downward.
     * On macOS, position is calculated as the
    coordinates of the bottom left corner of the window, relative to
    bottom left corner of the display, and the positive Y-axis points
    upward.
     * For a child window, on all platforms, position is relative
    to the upper-left corner of the parent window's client area.
     *
    Equivalent to calling JS_Window_SetPosition with NOSIZE, NOZORDER,
    NOACTIVATE and NOOWNERZORDER flags set.

    Returns
    -------
    
    """
    a = _ft['JS_Window_Move']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(windowHWND)), ct.c_int(left), ct.c_int(top))
    return 


@reapy_boost.inside_reaper()
def Window_SetZOrder(windowHWND: VoidPtr, ZOrder: str, insertAfterHWNDOptional:  ty.Optional[VoidPtr]=None, encoding: str="utf-8") -> bool:
    """
    Sets the window Z order.
     * Equivalent to calling JS_Window_SetPos
    with flags NOMOVE | NOSIZE.
     * Not all the Z orders have been
    implemented in Linux yet.
    
    Parameters:
     * ZOrder: 'BOTTOM', 'TOPMOST',
    'NOTOPMOST', 'TOP', or a window HWND converted to a string, for
    example by the Lua function tostring.
    
    * InsertAfterHWND: For
    compatibility with older versions, this parameter is still available,
    and is optional. If ZOrder is 'INSERTAFTER', insertAfterHWND must be a
    handle to the window behind which windowHWND will be placed in the Z
    order, equivalent to setting ZOrder to this HWND; otherwise,
    insertAfterHWND is ignored and can be left out (or it can simply be
    set to the same value as windowHWND).

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_SetZOrder']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_void_p)(a)
    prot_ZOrder = packs_l(ZOrder, size=len(ZOrder.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_ZOrder, None if insertAfterHWNDOptional is None else packp("void*", str(insertAfterHWNDOptional)))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_GetLongPtr(windowHWND: VoidPtr, info: str, encoding: str="utf-8") -> VoidPtr:
    """
    Returns information about the specified window.
    
    info: 'USERDATA',
    'WNDPROC', 'DLGPROC', 'ID', 'EXSTYLE' or 'STYLE'.
    
    For documentation
    about the types of information returned, refer to the Win32 function
    GetWindowLongPtr.
    
    The values returned by 'DLGPROC' and 'WNDPROC' are
    typically used as-is, as pointers, whereas the others should first be
    converted to integers.
    
    If the function fails, a null pointer is
    returned.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_GetLongPtr']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_char_p)(a)
    prot_info = packs_l(info, size=len(info.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_info)
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_GetLong(windowHWND: VoidPtr, info: str, encoding: str="utf-8") -> float:
    """
    Similar to JS_Window_GetLongPtr, but returns the information as a
    number instead of a pointer. 
    
    In the case of 'DLGPROC' and 'WNDPROC',
    the return values can be converted to pointers by
    JS_Window_HandleFromAddress.
    
    If the function fails, the return value
    is 0.

    Returns
    -------
    retval: float
    """
    a = _ft['JS_Window_GetLong']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_void_p)(a)
    prot_retval = ct.c_double(1.0)
    prot_info = packs_l(info, size=len(info.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_info, ct.byref(prot_retval))
    return float(prot_retval.value)


@reapy_boost.inside_reaper()
def Window_SetLong(windowHWND: VoidPtr, info: str, value: float, encoding: str="utf-8") -> float:
    """
    Similar to the Win32 function SetWindowLongPtr. 
    
    info: 'USERDATA',
    'WNDPROC', 'DLGPROC', 'ID', 'EXSTYLE' or 'STYLE', and only on
    WindowOS, 'INSTANCE' and 'PARENT'.

    Returns
    -------
    retval: float
    """
    a = _ft['JS_Window_SetLong']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_double, ct.c_void_p)(a)
    prot_retval = ct.c_double(1.0)
    prot_info = packs_l(info, size=len(info.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_info, ct.c_double(value), ct.byref(prot_retval))
    return float(prot_retval.value)


@reapy_boost.inside_reaper()
def Window_SetStyle(windowHWND: VoidPtr, style: str, encoding: str="utf-8") -> bool:
    """
    Sets and applies a window style.
    
    style may include any combination of
    standard window styles, such as 'POPUP' for a frameless window, or
    'CAPTION,SIZEBOX,SYSMENU' for a standard framed window.
    
    On Linux and
    macOS, 'MAXIMIZE' has not yet been implmented, and the remaining
    styles may appear slightly different from their WindowsOS
    counterparts.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_SetStyle']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p)(a)
    prot_style = packs_l(style, size=len(style.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_style)
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_SetOpacity(windowHWND: VoidPtr, mode: str, value: float, encoding: str="utf-8") -> bool:
    """
    Sets the window opacity.
    
    Parameters:
    mode: either 'ALPHA' or 'COLOR'.
    value: If ALPHA, the specified value may range from zero to one, and
    will apply to the entire window, frame included. 
    If COLOR, value
    specifies a 0xRRGGBB color, and all pixels of this color will be made
    transparent. (All mouse clicks over transparent pixels will pass
    through, too).  WARNING:
    COLOR mode is only available in Windows, not
    Linux or macOS.
    
    Transparency can only be applied to top-level
    windows. If windowHWND refers to a child window, the entire top-level
    window that contains windowHWND will be made transparent.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_SetOpacity']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_double)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_mode, ct.c_double(value))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_GetTitle(windowHWND: VoidPtr, titleOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    Returns the title (if any) of the specified window.

    Returns
    -------
    titleNeedBig: str
    """
    a = _ft['JS_Window_GetTitle']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_titleNeedBig = packs_l("", size=titleOutNeedBig_sz, encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_titleNeedBig, ct.c_int(titleOutNeedBig_sz))
    return unpacks_l(prot_titleNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Window_SetTitle(windowHWND: VoidPtr, title: str, encoding: str="utf-8") -> bool:
    """
    Changes the title of the specified window. Returns true if successful.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_SetTitle']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p)(a)
    prot_title = packs_l(title, size=len(title.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_title)
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_GetClassName(windowHWND: VoidPtr, classOut_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    WARNING: May not be fully implemented on macOS and Linux.

    Returns
    -------
    class: str
    """
    a = _ft['JS_Window_GetClassName']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_class = packs_l("", size=classOut_sz, encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_class, ct.c_int(classOut_sz))
    return unpacks_l(prot_class, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Window_HandleFromAddress(address: float) -> VoidPtr:
    """
    Converts an address to a handle (such as a HWND) that can be utilized
    by REAPER and other API functions.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Window_HandleFromAddress']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_double)(a)
    
    ret = f(ct.c_double(address))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Window_AddressFromHandle(handle: VoidPtr) -> float:
    """
    

    Returns
    -------
    address: float
    """
    a = _ft['JS_Window_AddressFromHandle']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_void_p)(a)
    prot_address = ct.c_double(1.0)
    ret = f(packp("void*", str(handle)), ct.byref(prot_address))
    return float(prot_address.value)


@reapy_boost.inside_reaper()
def WindowMessage_Post(windowHWND: VoidPtr, message: str, wParam: float, wParamHighWord: int, lParam: float, lParamHighWord: int, encoding: str="utf-8") -> bool:
    """
    If the specified window and message type are not currently being
    intercepted by a script, this function will post the message in the
    message queue of the specified window, and return without waiting.
    
    If
    the window and message type are currently being intercepted, the
    message will be sent directly to the original window process, similar
    to WindowMessage_Send, thereby skipping any intercepts.
    
    Parameters:
    * message: String containing a single message name, such as
    'WM_SETCURSOR', or in hexadecimal format, '0x0020'.
     (For a list of
    WM_ and CB_ message types that are valid cross-platform, refer to
    swell-types.h. Only these will be recognized by WM_ or CB_ name.)
     *
    wParam, wParamHigh, lParam and lParamHigh: Low and high 16-bit WORDs
    of the WPARAM and LPARAM parameters.
    (Most window messages encode
    separate information into the two WORDs. However, for those rare cases
    in which the entire WPARAM and LPARAM must be used to post a large
    pointer, the script can store this address in wParam or lParam, and
    keep wParamHigh and lParamHigh zero.)
    
    Notes:
     * For more information
    about parameter values, refer to documentation for the Win32 C++
    function PostMessage.
     * Messages should only be sent to windows that
    were created from the main thread.
     * Useful for simulating mouse
    clicks and calling mouse modifier actions from scripts.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_WindowMessage_Post']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_double, ct.c_int, ct.c_double, ct.c_int)(a)
    prot_message = packs_l(message, size=len(message.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_message, ct.c_double(wParam), ct.c_int(wParamHighWord), ct.c_double(lParam), ct.c_int(lParamHighWord))
    return bool(ret)


@reapy_boost.inside_reaper()
def WindowMessage_Send(windowHWND: VoidPtr, message: str, wParam: float, wParamHighWord: int, lParam: float, lParamHighWord: int, encoding: str="utf-8") -> int:
    """
    Sends a message to the specified window by calling the window process
    directly, and only returns after the message has been processed. Any
    intercepts of the message by scripts will be skipped, and the message
    can therefore not be blocked.
    
    Parameters:
     * message: String
    containing a single message name, such as 'WM_SETCURSOR', or in
    hexadecimal format, '0x0020'.
     (For a list of WM_ and CB_ message
    types that are valid cross-platform, refer to swell-types.h. Only
    these will be recognized by WM_ or CB_ name.)
     * wParam, wParamHigh,
    lParam and lParamHigh: Low and high 16-bit WORDs of the WPARAM and
    LPARAM parameters.
    (Most window messages encode separate information
    into the two WORDs. However, for those rare cases in which the entire
    WPARAM and LPARAM must be used to post a large pointer, the script can
    store this address in wParam or lParam, and keep wParamHigh and
    lParamHigh zero.)
    
    Notes:
     * For more information about parameter and
    return values, refer to documentation for the Win32 C++ function
    SendMessage.
     * Messages should only be sent to windows that were
    created from the main thread.
     * Useful for simulating mouse clicks
    and calling mouse modifier actions from scripts.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_WindowMessage_Send']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_double, ct.c_int, ct.c_double, ct.c_int)(a)
    prot_message = packs_l(message, size=len(message.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_message, ct.c_double(wParam), ct.c_int(wParamHighWord), ct.c_double(lParam), ct.c_int(lParamHighWord))
    return int(ret)


@reapy_boost.inside_reaper()
def WindowMessage_Peek(windowHWND: VoidPtr, message: str, encoding: str="utf-8") -> ty.Tuple[bool, bool, float, int, int, int, int]:
    """
    Polls the state of an intercepted message.
    
    Parameters:
     * message:
    String containing a single message name, such as 'WM_SETCURSOR', or in
    hexadecimal format, '0x0020'.
     (For a list of WM_ and CB_ message
    types that are valid cross-platform, refer to swell-types.h. Only
    these will be recognized by WM_ or CB_ name.)
    
    Returns:
     * A retval of
    false indicates that the message type is not being intercepted in the
    specified window.
     * All messages are timestamped. A time of 0
    indicates that no message if this type has been intercepted yet.
     *
    For more information about wParam and lParam for different message
    types, refer to Win32 C++ documentation.
     * For example, in the case
    of mousewheel, returns mousewheel delta, modifier keys, x position and
    y position.
     * wParamHigh, lParamLow and lParamHigh are signed,
    whereas wParamLow is unsigned.

    Returns
    -------
    ret_value: bool
    passedThrough: bool
    time: float
    wParamLow: int
    wParamHigh: int
    lParamLow: int
    lParamHigh: int
    """
    a = _ft['JS_WindowMessage_Peek']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_passedThrough = ct.c_byte(False)
    prot_time = ct.c_double(1.0)
    prot_wParamLow = ct.c_int(1)
    prot_wParamHigh = ct.c_int(1)
    prot_lParamLow = ct.c_int(1)
    prot_lParamHigh = ct.c_int(1)
    prot_message = packs_l(message, size=len(message.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_message, ct.byref(prot_passedThrough), ct.byref(prot_time), ct.byref(prot_wParamLow), ct.byref(prot_wParamHigh), ct.byref(prot_lParamLow), ct.byref(prot_lParamHigh))
    return bool(ret), bool(prot_passedThrough.value), float(prot_time.value), prot_wParamLow.value, prot_wParamHigh.value, prot_lParamLow.value, prot_lParamHigh.value


@reapy_boost.inside_reaper()
def WindowMessage_Intercept(windowHWND: VoidPtr, message: str, passThrough: bool, encoding: str="utf-8") -> int:
    """
    Begins intercepting a window message type to specified window.
    Parameters:
     * message: a single message type to be intercepted,
    either in WM_ or hexadecimal format. For example 'WM_SETCURSOR' or
    '0x0020'.
     * passThrough: Whether message should be blocked (false) or
    passed through (true) to the window.
        For more information on
    message codes, refer to the Win32 C++ API documentation.
        All WM_
    and CB_ message types listed in swell-types.h should be valid cross-
    platform, and the function can recognize all of these by name. Other
    messages can be specified by their hex code.
    
    Returns:
     * 1: Success.
    * 0: The message type is already being intercepted by another script.
    * -2: message string could not be parsed.
     * -3: Failure getting
    original window process / window not valid.
     * -6: Could not obtain
    the window client HDC.
    
    Notes:
     * Intercepted messages can be polled
    using JS_WindowMessage_Peek.
     * Intercepted messages can be edited, if
    necessary, and then forwarded to their original destination using
    JS_WindowMessage_Post or JS_WindowMessage_Send.
     * To check whether a
    message type is being blocked or passed through, Peek the message
    type, or retrieve the entire List of intercepts.
     * Mouse events are
    typically received by the child window under the mouse, not the parent
    window.
    Keyboard events are usually *not* received by any individual
    window. To intercept keyboard events, use the VKey functions.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_WindowMessage_Intercept']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_byte)(a)
    prot_message = packs_l(message, size=len(message.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_message, ct.c_byte(passThrough))
    return int(ret)


@reapy_boost.inside_reaper()
def WindowMessage_InterceptList(windowHWND: VoidPtr, messages: str, encoding: str="utf-8") -> int:
    """
    Begins intercepting window messages to specified window.
    
    Parameters:
    * messages: comma-separated string of message types to be intercepted
    (either in WM_ or hexadecimal format), each with a 'block' or
    'passthrough' modifier to specify whether the message should be
    blocked or passed through to the window. For example
    'WM_SETCURSOR:block, 0x0201:passthrough'.
        For more information on
    message codes, refer to the Win32 C++ API documentation.
        All WM_
    and CB_ message types listed in swell-types.h should be valid cross-
    platform, and the function can recognize all of these by name. Other
    messages can be specified by their hex code.
    
    Returns:
     * 1: Success.
    * 0: The message type is already being intercepted by another script.
    * -1: windowHWND is not a valid window.
     * -2: message string could
    not be parsed.
     * -3: Failure getting original window process.
     * -6:
    COuld not obtain the window client HDC.
    
    Notes:
     * Intercepted
    messages can be polled using JS_WindowMessage_Peek.
     * Intercepted
    messages can be edited, if necessary, and then forwarded to their
    original destination using JS_WindowMessage_Post or
    JS_WindowMessage_Send.
     * To check whether a message type is being
    blocked or passed through, Peek the message type, or retrieve the
    entire List of intercepts.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_WindowMessage_InterceptList']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p)(a)
    prot_messages = packs_l(messages, size=len(messages.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_messages)
    return int(ret)


@reapy_boost.inside_reaper()
def WindowMessage_PassThrough(windowHWND: VoidPtr, message: str, passThrough: bool, encoding: str="utf-8") -> int:
    """
    Changes the passthrough setting of a message type that is already
    being intercepted.
    
    Returns 1 if successful, 0 if the message type is
    not yet being intercepted, or -2 if the argument could not be parsed.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_WindowMessage_PassThrough']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_byte)(a)
    prot_message = packs_l(message, size=len(message.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_message, ct.c_byte(passThrough))
    return int(ret)


@reapy_boost.inside_reaper()
def WindowMessage_ListIntercepts(windowHWND: VoidPtr, listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[bool, str]:
    """
    Returns a string with a list of all message types currently being
    intercepted for the specified window.

    Returns
    -------
    ret_value: bool
    listNeedBig: str
    """
    a = _ft['JS_WindowMessage_ListIntercepts']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return bool(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def WindowMessage_Release(windowHWND: VoidPtr, messages: str, encoding: str="utf-8") -> int:
    """
    Release intercepts of specified message types.
    
    Parameters:
     *
    messages: 'WM_SETCURSOR,WM_MOUSEHWHEEL' or '0x0020,0x020E', for
    example.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_WindowMessage_Release']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p)(a)
    prot_messages = packs_l(messages, size=len(messages.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_messages)
    return int(ret)


@reapy_boost.inside_reaper()
def WindowMessage_ReleaseWindow(windowHWND: VoidPtr) -> None:
    """
    Release script intercepts of window messages for specified window.

    Returns
    -------
    
    """
    a = _ft['JS_WindowMessage_ReleaseWindow']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def WindowMessage_ReleaseAll() -> None:
    """
    Release script intercepts of window messages for all windows.

    Returns
    -------
    
    """
    a = _ft['JS_WindowMessage_ReleaseAll']
    f = ct.CFUNCTYPE(None, )(a)
    
    ret = f()
    return 


@reapy_boost.inside_reaper()
def Window_OnCommand(windowHWND: VoidPtr, commandID: int) -> bool:
    """
    Sends a 'WM_COMMAND' message to the specified window, which simulates
    a user selecting a command in the window menu.
    
    This function is
    similar to Main_OnCommand and MIDIEditor_OnCommand, but can send
    commands to any window that has a menu.
    
    In the case of windows that
    are listed among the Action list's contexts (such as the Media
    Explorer), the commandIDs of the actions in the Actions list may be
    used.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_OnCommand']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(windowHWND)), ct.c_int(commandID))
    return bool(ret)


@reapy_boost.inside_reaper()
def VKeys_GetState(cutoffTime: float, stateOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    Retrieves the current states (0 or 1) of all virtual keys, from 0x01
    to 0xFF, in a 255-byte array.
    
    cutoffTime: A key is only regarded as
    down if it sent a KEYDOWN message after the cut-off time, not followed
    by KEYUP. (This is useful for excluding old KEYDOWN messages that
    weren't properly followed by KEYUP.) 
    If cutoffTime is positive, is it
    interpreted as absolute time in similar format as time_precise().
    If
    cutoffTime is negative, it is relative to the current time.
    
    Notes:
     *
    Mouse buttons and modifier keys are not (currently) reliably detected,
    and JS_Mouse_GetState can be used instead.
     * Auto-repeated KEYDOWN
    messages are ignored.

    Returns
    -------
    stateNeedBig: str
    """
    a = _ft['JS_VKeys_GetState']
    f = ct.CFUNCTYPE(None, ct.c_double, ct.c_char_p, ct.c_int)(a)
    prot_stateNeedBig = packs_l("", size=stateOutNeedBig_sz, encoding=encoding)
    ret = f(ct.c_double(cutoffTime), prot_stateNeedBig, ct.c_int(stateOutNeedBig_sz))
    return unpacks_l(prot_stateNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def VKeys_GetDown(cutoffTime: float, stateOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    Returns a 255-byte array that specifies which virtual keys, from 0x01
    to 0xFF, have sent KEYDOWN messages since cutoffTime.
    
    Notes:
     * Mouse
    buttons and modifier keys are not (currently) reliably detected, and
    JS_Mouse_GetState can be used instead.
     * Auto-repeated KEYDOWN
    messages are ignored.

    Returns
    -------
    stateNeedBig: str
    """
    a = _ft['JS_VKeys_GetDown']
    f = ct.CFUNCTYPE(None, ct.c_double, ct.c_char_p, ct.c_int)(a)
    prot_stateNeedBig = packs_l("", size=stateOutNeedBig_sz, encoding=encoding)
    ret = f(ct.c_double(cutoffTime), prot_stateNeedBig, ct.c_int(stateOutNeedBig_sz))
    return unpacks_l(prot_stateNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def VKeys_GetUp(cutoffTime: float, stateOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    Return a 255-byte array that specifies which virtual keys, from 0x01
    to 0xFF, have sent KEYUP messages since cutoffTime.
    
    Note: Mouse
    buttons and modifier keys are not (currently) reliably detected, and
    JS_Mouse_GetState can be used instead.

    Returns
    -------
    stateNeedBig: str
    """
    a = _ft['JS_VKeys_GetUp']
    f = ct.CFUNCTYPE(None, ct.c_double, ct.c_char_p, ct.c_int)(a)
    prot_stateNeedBig = packs_l("", size=stateOutNeedBig_sz, encoding=encoding)
    ret = f(ct.c_double(cutoffTime), prot_stateNeedBig, ct.c_int(stateOutNeedBig_sz))
    return unpacks_l(prot_stateNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def VKeys_Intercept(keyCode: int, intercept: int) -> int:
    """
    Intercepting (blocking) virtual keys work similar to the native
    function PreventUIRefresh:  Each key has a (non-negative) intercept
    state, and the key is passed through as usual if the state equals 0,
    or blocked if the state is greater than 0.
    
    keyCode: The virtual key
    code of the key, or -1 to change the state of all keys.
    
    intercept: A
    script can increase the intercept state by passing +1, or lower the
    state by passing -1.  Multiple scripts can block the same key, and the
    intercept state may reach up to 255. If zero is passed, the intercept
    state is not changed, but the current state is returned.
    
    Returns: If
    keyCode refers to a single key, the intercept state of that key is
    returned.  If keyCode = -1, the state of the key that is most strongly
    blocked (highest intercept state) is returned.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_VKeys_Intercept']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_int(keyCode), ct.c_int(intercept))
    return int(ret)


@reapy_boost.inside_reaper()
def Mouse_GetState(flags: int) -> int:
    """
    Retrieves the states of mouse buttons and modifiers keys.
    
    Parameters:
    * flags, state: The parameter and the return value both use the same
    format as gfx.mouse_cap. For example, to get the states of the left
    mouse button and the ctrl key, use flags = 0b00000101.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_Mouse_GetState']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_int(flags))
    return int(ret)


@reapy_boost.inside_reaper()
def Mouse_GetCursor() -> VoidPtr:
    """
    On Windows, retrieves a handle to the current mouse cursor.
    On Linux
    and macOS, retrieves a handle to the last cursor set by REAPER or its
    extensions via SWELL.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Mouse_GetCursor']
    f = ct.CFUNCTYPE(ct.c_void_p, )(a)
    
    ret = f()
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Mouse_SetPosition(x: int, y: int) -> bool:
    """
    Moves the mouse cursor to the specified screen coordinates.
    
    NOTES:
     *
    On Windows and Linux, screen coordinates are relative to *upper* left
    corner of the primary display, and the positive Y-axis points
    downward.
     * On macOS, screen coordinates are relative to the *bottom*
    left corner of the primary display, and the positive Y-axis points
    upward.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Mouse_SetPosition']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_int(x), ct.c_int(y))
    return bool(ret)


@reapy_boost.inside_reaper()
def Mouse_LoadCursor(cursorNumber: int) -> VoidPtr:
    """
    Loads a cursor by number.
    
    cursorNumber: Same as used for
    gfx.setcursor, and includes some of Windows' predefined cursors (with
    numbers > 32000; refer to documentation for the Win32 C++ function
    LoadCursor), and REAPER's own cursors (with numbers < 2000).
    
    If
    successful, returns a handle to the cursor, which can be used in
    JS_Mouse_SetCursor.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Mouse_LoadCursor']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_int)(a)
    
    ret = f(ct.c_int(cursorNumber))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Mouse_LoadCursorFromFile(pathAndFileName: str, forceNewLoadOptional:  ty.Optional[bool]=None, encoding: str="utf-8") -> VoidPtr:
    """
    Loads a cursor from a .cur file.
    
    forceNewLoad is an optional boolean
    parameter:
     * If omitted or false, and if the .cur file has already
    been loaded previously during the REAPER session, the file will not be
    re-loaded, and the previous handle will be returned, thereby
    (slightly) improving speed and (slighty) lowering memory usage.
     * If
    true, the file will be re-loaded and a new handle will be returned.
    If successful, returns a handle to the cursor, which can be used in
    JS_Mouse_SetCursor.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_Mouse_LoadCursorFromFile']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_char_p, ct.c_void_p)(a)
    prot_forceNewLoadOptional = ct.c_byte(forceNewLoadOptional) if forceNewLoadOptional is not None else None
    prot_pathAndFileName = packs_l(pathAndFileName, size=len(pathAndFileName.encode(encoding)), encoding=encoding)
    ret = f(prot_pathAndFileName, None if forceNewLoadOptional is None else ct.byref(prot_forceNewLoadOptional))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def Mouse_SetCursor(cursorHandle: VoidPtr) -> None:
    """
    Sets the mouse cursor.  (Only lasts while script is running, and for a
    single 'defer' cycle.)

    Returns
    -------
    
    """
    a = _ft['JS_Mouse_SetCursor']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(cursorHandle)))
    return 


@reapy_boost.inside_reaper()
def Window_GetScrollInfo(windowHWND: VoidPtr, scrollbar: str, encoding: str="utf-8") -> ty.Tuple[bool, int, int, int, int, int]:
    """
    Retrieves the scroll information of a window.
    
    Parameters:
     *
    windowHWND: The window that contains the scrollbar. This is usually a
    child window, not a top-level, framed window.
     * scrollbar: 'v' (or
    'SB_VERT', or 'VERT') for vertical scroll, 'h' (or 'SB_HORZ' or
    'HORZ') for horizontal.
    
    Returns:
     * Leftmost or topmost visible pixel
    position, as well as the visible page size, the range minimum and
    maximum, and scroll box tracking position.

    Returns
    -------
    ret_value: bool
    position: int
    pageSize: int
    min: int
    max: int
    trackPos: int
    """
    a = _ft['JS_Window_GetScrollInfo']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_position = ct.c_int(1)
    prot_pageSize = ct.c_int(1)
    prot_min = ct.c_int(1)
    prot_max = ct.c_int(1)
    prot_trackPos = ct.c_int(1)
    prot_scrollbar = packs_l(scrollbar, size=len(scrollbar.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_scrollbar, ct.byref(prot_position), ct.byref(prot_pageSize), ct.byref(prot_min), ct.byref(prot_max), ct.byref(prot_trackPos))
    return bool(ret), prot_position.value, prot_pageSize.value, prot_min.value, prot_max.value, prot_trackPos.value


@reapy_boost.inside_reaper()
def Window_SetScrollPos(windowHWND: VoidPtr, scrollbar: str, position: int, encoding: str="utf-8") -> bool:
    """
    Parameters:
     * scrollbar: 'v' (or 'SB_VERT', or 'VERT') for vertical
    scroll, 'h' (or 'SB_HORZ' or 'HORZ') for horizontal.
    
    NOTE: API
    functions can scroll REAPER's windows, but cannot zoom them.  Instead,
    use actions such as 'View: Zoom to one loop iteration'.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_Window_SetScrollPos']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_scrollbar = packs_l(scrollbar, size=len(scrollbar.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_scrollbar, ct.c_int(position))
    return bool(ret)


@reapy_boost.inside_reaper()
def GDI_GetClientDC(windowHWND: VoidPtr) -> VoidPtr:
    """
    Returns the device context for the client area of the specified
    window.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_GetClientDC']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_GetWindowDC(windowHWND: VoidPtr) -> VoidPtr:
    """
    Returns the device context for the entire window, including title bar
    and frame.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_GetWindowDC']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_GetScreenDC() -> VoidPtr:
    """
    Returns a device context for the entire screen.
    
    WARNING: Only
    available on Windows, not Linux or macOS.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_GetScreenDC']
    f = ct.CFUNCTYPE(ct.c_void_p, )(a)
    
    ret = f()
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_ReleaseDC(deviceHDC: VoidPtr,  windowHWNDOptional:  ty.Optional[VoidPtr]=None) -> int:
    """
    To release a window HDC, both arguments must be supplied: the HWND as
    well as the HDC.  To release a screen DC, only the HDC needs to be
    supplied.  
    
    For compatibility with previous versions, the HWND and
    HDC can be supplied in any order.
    
    NOTE: Any GDI HDC should be
    released immediately after drawing, and deferred scripts should get
    and release new DCs in each cycle.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_GDI_ReleaseDC']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(deviceHDC)), None if  windowHWNDOptional is None else packp("void*", str( windowHWNDOptional)))
    return int(ret)


@reapy_boost.inside_reaper()
def GDI_CreateFillBrush(color: int) -> VoidPtr:
    """
    

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_CreateFillBrush']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_int)(a)
    
    ret = f(ct.c_int(color))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_CreatePen(width: int, color: int) -> VoidPtr:
    """
    

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_CreatePen']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_int(width), ct.c_int(color))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_CreateFont(height: int, weight: int, angle: int, italic: bool, underline: bool, strikeOut: bool, fontName: str, encoding: str="utf-8") -> VoidPtr:
    """
    Parameters:
     * weight: 0 - 1000, with 0 = auto, 400 = normal and 700 =
    bold.
     * angle: the angle, in tenths of degrees, between the text and
    the x-axis of the device.
     * fontName: If empty string '', uses first
    font that matches the other specified attributes.
    
    Note: Text color
    must be set separately.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_CreateFont']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_byte, ct.c_byte, ct.c_byte, ct.c_char_p)(a)
    prot_fontName = packs_l(fontName, size=len(fontName.encode(encoding)), encoding=encoding)
    ret = f(ct.c_int(height), ct.c_int(weight), ct.c_int(angle), ct.c_byte(italic), ct.c_byte(underline), ct.c_byte(strikeOut), prot_fontName)
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_SelectObject(deviceHDC: VoidPtr, GDIObject: VoidPtr) -> VoidPtr:
    """
    Activates a font, pen, or fill brush for subsequent drawing in the
    specified device context.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_GDI_SelectObject']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(deviceHDC)), packp("void*", str(GDIObject)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def GDI_DeleteObject(GDIObject: VoidPtr) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_DeleteObject']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(GDIObject)))
    return 


@reapy_boost.inside_reaper()
def GDI_FillRect(deviceHDC: VoidPtr, left: int, top: int, right: int, bottom: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_FillRect']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(left), ct.c_int(top), ct.c_int(right), ct.c_int(bottom))
    return 


@reapy_boost.inside_reaper()
def GDI_FillRoundRect(deviceHDC: VoidPtr, left: int, top: int, right: int, bottom: int, xrnd: int, yrnd: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_FillRoundRect']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(left), ct.c_int(top), ct.c_int(right), ct.c_int(bottom), ct.c_int(xrnd), ct.c_int(yrnd))
    return 


@reapy_boost.inside_reaper()
def GDI_FillPolygon(deviceHDC: VoidPtr, packedX: str, packedY: str, numPoints: int, encoding: str="utf-8") -> None:
    """
    packedX and packedY are strings of points, each packed as '<i4'.

    Returns
    -------
    
    """
    a = _ft['JS_GDI_FillPolygon']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_char_p, ct.c_int)(a)
    prot_packedX = packs_l(packedX, size=len(packedX.encode(encoding)), encoding=encoding)
    prot_packedY = packs_l(packedY, size=len(packedY.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(deviceHDC)), prot_packedX, prot_packedY, ct.c_int(numPoints))
    return 


@reapy_boost.inside_reaper()
def GDI_FillEllipse(deviceHDC: VoidPtr, left: int, top: int, right: int, bottom: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_FillEllipse']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(left), ct.c_int(top), ct.c_int(right), ct.c_int(bottom))
    return 


@reapy_boost.inside_reaper()
def GDI_GetSysColor(GUIElement: str, encoding: str="utf-8") -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_GDI_GetSysColor']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p)(a)
    prot_GUIElement = packs_l(GUIElement, size=len(GUIElement.encode(encoding)), encoding=encoding)
    ret = f(prot_GUIElement)
    return int(ret)


@reapy_boost.inside_reaper()
def GDI_SetTextBkMode(deviceHDC: VoidPtr, mode: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_SetTextBkMode']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(mode))
    return 


@reapy_boost.inside_reaper()
def GDI_SetTextBkColor(deviceHDC: VoidPtr, color: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_SetTextBkColor']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(color))
    return 


@reapy_boost.inside_reaper()
def GDI_SetTextColor(deviceHDC: VoidPtr, color: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_SetTextColor']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(color))
    return 


@reapy_boost.inside_reaper()
def GDI_GetTextColor(deviceHDC: VoidPtr) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_GDI_GetTextColor']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(deviceHDC)))
    return int(ret)


@reapy_boost.inside_reaper()
def GDI_DrawText(deviceHDC: VoidPtr, text: str, len_: int, left: int, top: int, right: int, bottom: int, align: str, encoding: str="utf-8") -> int:
    """
    Parameters:
     * align: Combination of: 'TOP', 'VCENTER', 'LEFT',
    'HCENTER', 'RIGHT', 'BOTTOM', 'WORDBREAK', 'SINGLELINE', 'NOCLIP',
    'CALCRECT', 'NOPREFIX' or 'ELLIPSIS'

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_GDI_DrawText']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_char_p)(a)
    prot_text = packs_l(text, size=len(text.encode(encoding)), encoding=encoding)
    prot_align = packs_l(align, size=len(align.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(deviceHDC)), prot_text, ct.c_int(len_), ct.c_int(left), ct.c_int(top), ct.c_int(right), ct.c_int(bottom), prot_align)
    return int(ret)


@reapy_boost.inside_reaper()
def GDI_SetPixel(deviceHDC: VoidPtr, x: int, y: int, color: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_SetPixel']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(x), ct.c_int(y), ct.c_int(color))
    return 


@reapy_boost.inside_reaper()
def GDI_Line(deviceHDC: VoidPtr, x1: int, y1: int, x2: int, y2: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_GDI_Line']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(deviceHDC)), ct.c_int(x1), ct.c_int(y1), ct.c_int(x2), ct.c_int(y2))
    return 


@reapy_boost.inside_reaper()
def GDI_Polyline(deviceHDC: VoidPtr, packedX: str, packedY: str, numPoints: int, encoding: str="utf-8") -> None:
    """
    packedX and packedY are strings of points, each packed as '<i4'.

    Returns
    -------
    
    """
    a = _ft['JS_GDI_Polyline']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_char_p, ct.c_int)(a)
    prot_packedX = packs_l(packedX, size=len(packedX.encode(encoding)), encoding=encoding)
    prot_packedY = packs_l(packedY, size=len(packedY.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(deviceHDC)), prot_packedX, prot_packedY, ct.c_int(numPoints))
    return 


@reapy_boost.inside_reaper()
def GDI_Blit(destHDC: VoidPtr, dstx: int, dsty: int, sourceHDC: VoidPtr, srcx: int, srxy: int, width: int, height: int, modeOptional:  ty.Optional[str]=None, encoding: str="utf-8") -> None:
    """
    Blits between two device contexts, which may include LICE 'system
    bitmaps'.
    
    mode: Optional parameter. 'SRCCOPY' by default, or specify
    'ALPHA' to enable per-pixel alpha blending.
    
    WARNING: On WindowsOS,
    GDI_Blit does not perform alpha multiplication of the source bitmap.
    For proper color rendering, a separate pre-multiplication step is
    therefore required, using either LICE_Blit or LICE_ProcessRect.

    Returns
    -------
    
    """
    a = _ft['JS_GDI_Blit']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_char_p)(a)
    prot_modeOptional = packs_l(modeOptional, size=len(modeOptional.encode(encoding)), encoding=encoding) if modeOptional is not None else None
    ret = f(packp("void*", str(destHDC)), ct.c_int(dstx), ct.c_int(dsty), packp("void*", str(sourceHDC)), ct.c_int(srcx), ct.c_int(srxy), ct.c_int(width), ct.c_int(height), None if modeOptional is None else prot_modeOptional)
    return 


@reapy_boost.inside_reaper()
def GDI_StretchBlit(destHDC: VoidPtr, dstx: int, dsty: int, dstw: int, dsth: int, sourceHDC: VoidPtr, srcx: int, srxy: int, srcw: int, srch: int, modeOptional:  ty.Optional[str]=None, encoding: str="utf-8") -> None:
    """
    Blits between two device contexts, which may include LICE 'system
    bitmaps'.
    
    modeOptional: 'SRCCOPY' by default, or specify 'ALPHA' to
    enable per-pixel alpha blending.
    
    WARNING: On WindowsOS, GDI_Blit does
    not perform alpha multiplication of the source bitmap. For proper
    color rendering, a separate pre-multiplication step is therefore
    required, using either LICE_Blit or LICE_ProcessRect.

    Returns
    -------
    
    """
    a = _ft['JS_GDI_StretchBlit']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_char_p)(a)
    prot_modeOptional = packs_l(modeOptional, size=len(modeOptional.encode(encoding)), encoding=encoding) if modeOptional is not None else None
    ret = f(packp("void*", str(destHDC)), ct.c_int(dstx), ct.c_int(dsty), ct.c_int(dstw), ct.c_int(dsth), packp("void*", str(sourceHDC)), ct.c_int(srcx), ct.c_int(srxy), ct.c_int(srcw), ct.c_int(srch), None if modeOptional is None else prot_modeOptional)
    return 


@reapy_boost.inside_reaper()
def Composite(windowHWND: VoidPtr, dstx: int, dsty: int, dstw: int, dsth: int, sysBitmap: VoidPtr, srcx: int, srcy: int, srcw: int, srch: int, autoUpdateOptional:  ty.Optional[bool]=None) -> int:
    """
    Composites a LICE bitmap with a REAPER window.  Each time that the
    window is re-drawn, the bitmap will be blitted over the window's
    client area (with per-pixel alpha blending).
    
     * If dstw or dsth is
    -1, the bitmap will be stretched to fill the width or height of the
    window, respectively.
    
     * autoUpdate is an optional parameter that is
    false by default. If true, JS_Composite will automatically invalidate
    and re-draw the part of the window that covers the current position of
    the bitmap, and if the bitmap is being moved, also the previous
    position. (If only one or a handful of bitmaps are being moved across
    the screen, autoUpdate should result in smoother animation on
    WindowsOS; if numerous bitmaps are spread over the entire window, it
    may be faster to disable autoUpdate and instead call
    JS_Window_InvalidateRect explicitly once all bitmaps have been moved.)
    * InvalidateRect should also be called whenever the contents of the
    bitmap contents have been changed, but not the position, to trigger a
    window update.
    
     * On WindowsOS, the key to reducing flickering is to
    slow down the frequency at which the window is re-drawn.
    InvalidateRect should only be called when absolutely necessary,
    preferably not more than 20 times per second.  (Also refer to the
    JS_Composite_Delay function.)
    
     * On WindowsOS, flickering can further
    be reduced by keeping the invalidated area as small as possible,
    covering only the bitmaps that have been edited or moved.  However, if
    numerous bitmaps are spread over the entire window, it may be faster
    to simply invalidate the entire client area. 
    
     * This function should
    not be applied directly to top-level windows, but rather to child
    windows.
    
     * Some classes of UI elements, particularly buttons, do not
    take kindly to being composited, and may crash REAPER.
    
     * On
    WindowsOS, GDI blitting does not perform alpha multiplication of the
    source bitmap. For proper color rendering, a separate pre-
    multiplication step is therefore required, using either LICE_Blit or
    LICE_ProcessRect.
    
    Returns:
    1 if successful, otherwise -1 = windowHWND
    is not a window, -3 = Could not obtain the original window process, -4
    = sysBitmap is not a LICE bitmap, -5 = sysBitmap is not a system
    bitmap, -6 = Could not obtain the window HDC, -7 = Error when
    subclassing to new window process.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_Composite']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_void_p)(a)
    prot_autoUpdateOptional = ct.c_byte(autoUpdateOptional) if autoUpdateOptional is not None else None
    ret = f(packp("void*", str(windowHWND)), ct.c_int(dstx), ct.c_int(dsty), ct.c_int(dstw), ct.c_int(dsth), packp("void*", str(sysBitmap)), ct.c_int(srcx), ct.c_int(srcy), ct.c_int(srcw), ct.c_int(srch), None if autoUpdateOptional is None else ct.byref(prot_autoUpdateOptional))
    return int(ret)


@reapy_boost.inside_reaper()
def Composite_Unlink(windowHWND: VoidPtr, bitmapOptional:  ty.Optional[VoidPtr]=None, autoUpdateOptional:  ty.Optional[bool]=None) -> None:
    """
    Unlinks the window and bitmap.
    
     * autoUpdate is an optional
    parameter. If unlinking a single bitmap and autoUpdate is true, the
    function will automatically re-draw the window to remove the blitted
    image.
    
    If no bitmap is specified, all bitmaps composited to the
    window will be unlinked -- even those by other scripts.

    Returns
    -------
    
    """
    a = _ft['JS_Composite_Unlink']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_autoUpdateOptional = ct.c_byte(autoUpdateOptional) if autoUpdateOptional is not None else None
    ret = f(packp("void*", str(windowHWND)), None if bitmapOptional is None else packp("void*", str(bitmapOptional)), None if autoUpdateOptional is None else ct.byref(prot_autoUpdateOptional))
    return 


@reapy_boost.inside_reaper()
def Composite_ListBitmaps(windowHWND: VoidPtr, listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Returns all bitmaps composited to the given window.
    
    The list is
    formatted as a comma-separated string of hexadecimal values, each
    representing a LICE_IBitmap* pointer.
    
    retval is the number of linked
    bitmaps found, or negative if an error occured.

    Returns
    -------
    ret_value: int
    listNeedBig: str
    """
    a = _ft['JS_Composite_ListBitmaps']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    ret = f(packp("void*", str(windowHWND)), prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return int(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Composite_Delay(windowHWND: VoidPtr, minTime: float, maxTime: float, numBitmapsWhenMax: int) -> ty.Tuple[int, float, float, int]:
    """
    On WindowsOS, flickering of composited images can be improved
    considerably by slowing the refresh rate of the window.  The optimal
    refresh rate may depend on the number of composited bitmaps.
    
    minTime
    is the minimum refresh delay, in seconds, when only one bitmap is
    composited onto the window.  The delay time will increase linearly
    with the number of bitmaps, up to a maximum of maxTime when
    numBitmapsWhenMax is reached.
    
    If both minTime and maxTime are 0, all
    delay settings for the window are cleared.
    
    Returns:
     * retval = 1 if
    successful, 0 if arguments are invalid (i.e. if maxTime < minTime, or
    maxBitmaps < 1).
     * If delay times have not previously been set for
    this window, prev time values are 0.

    Returns
    -------
    ret_value: int
    prevMinTime: float
    prevMaxTime: float
    prevBitmaps: int
    """
    a = _ft['JS_Composite_Delay']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_double, ct.c_double, ct.c_int, ct.c_void_p, ct.c_void_p, ct.c_void_p)(a)
    prot_prevMinTime = ct.c_double(1.0)
    prot_prevMaxTime = ct.c_double(1.0)
    prot_prevBitmaps = ct.c_int(1)
    ret = f(packp("void*", str(windowHWND)), ct.c_double(minTime), ct.c_double(maxTime), ct.c_int(numBitmapsWhenMax), ct.byref(prot_prevMinTime), ct.byref(prot_prevMaxTime), ct.byref(prot_prevBitmaps))
    return int(ret), float(prot_prevMinTime.value), float(prot_prevMaxTime.value), prot_prevBitmaps.value


@reapy_boost.inside_reaper()
def LICE_CreateBitmap(isSysBitmap: bool, width: int, height: int) -> VoidPtr:
    """
    

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_LICE_CreateBitmap']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_byte, ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_byte(isSysBitmap), ct.c_int(width), ct.c_int(height))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def LICE_ListAllBitmaps(listOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    

    Returns
    -------
    ret_value: int
    listNeedBig: str
    """
    a = _ft['JS_LICE_ListAllBitmaps']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_char_p, ct.c_int)(a)
    prot_listNeedBig = packs_l("", size=listOutNeedBig_sz, encoding=encoding)
    ret = f(prot_listNeedBig, ct.c_int(listOutNeedBig_sz))
    return int(ret), unpacks_l(prot_listNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def LICE_ArrayAllBitmaps(reaperarray: VoidPtr) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_LICE_ArrayAllBitmaps']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(reaperarray)))
    return int(ret)


@reapy_boost.inside_reaper()
def LICE_GetHeight(bitmap: VoidPtr) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_LICE_GetHeight']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(bitmap)))
    return int(ret)


@reapy_boost.inside_reaper()
def LICE_GetWidth(bitmap: VoidPtr) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_LICE_GetWidth']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(bitmap)))
    return int(ret)


@reapy_boost.inside_reaper()
def LICE_GetDC(bitmap: VoidPtr) -> VoidPtr:
    """
    

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_LICE_GetDC']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(bitmap)))
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def LICE_DestroyBitmap(bitmap: VoidPtr) -> None:
    """
    Deletes the bitmap, and also unlinks bitmap from any composited
    window.

    Returns
    -------
    
    """
    a = _ft['JS_LICE_DestroyBitmap']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(bitmap)))
    return 


@reapy_boost.inside_reaper()
def LICE_LoadPNG(filename: str, encoding: str="utf-8") -> VoidPtr:
    """
    Returns a system LICE bitmap containing the PNG.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_LICE_LoadPNG']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_char_p)(a)
    prot_filename = packs_l(filename, size=len(filename.encode(encoding)), encoding=encoding)
    ret = f(prot_filename)
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def LICE_Blit(destBitmap: VoidPtr, dstx: int, dsty: int, sourceBitmap: VoidPtr, srcx: int, srcy: int, width: int, height: int, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    Standard LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD',
    'DODGE', 'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined
    with 'ALPHA' to enable per-pixel alpha blending.
    
    In addition to the
    standard LICE modes, LICE_Blit also offers:
     * 'CHANCOPY_XTOY', with X
    and Y any of the four channels, A, R, G or B. (CHANCOPY_ATOA is
    similar to MASK mode.)
     * 'BLUR'
     * 'ALPHAMUL', which overwrites the
    destination with a per-pixel alpha-multiplied copy of the source.
    (Similar to first clearing the destination with 0x00000000 and then
    blitting with 'COPY,ALPHA'.)

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Blit']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(destBitmap)), ct.c_int(dstx), ct.c_int(dsty), packp("void*", str(sourceBitmap)), ct.c_int(srcx), ct.c_int(srcy), ct.c_int(width), ct.c_int(height), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_RotatedBlit(destBitmap: VoidPtr, dstx: int, dsty: int, dstw: int, dsth: int, sourceBitmap: VoidPtr, srcx: float, srcy: float, srcw: float, srch: float, angle: float, rotxcent: float, rotycent: float, cliptosourcerect: bool, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA' to enable per-pixel alpha blending.

    Returns
    -------
    
    """
    a = _ft['JS_LICE_RotatedBlit']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_byte, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(destBitmap)), ct.c_int(dstx), ct.c_int(dsty), ct.c_int(dstw), ct.c_int(dsth), packp("void*", str(sourceBitmap)), ct.c_double(srcx), ct.c_double(srcy), ct.c_double(srcw), ct.c_double(srch), ct.c_double(angle), ct.c_double(rotxcent), ct.c_double(rotycent), ct.c_byte(cliptosourcerect), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_ScaledBlit(destBitmap: VoidPtr, dstx: int, dsty: int, dstw: int, dsth: int, srcBitmap: VoidPtr, srcx: float, srcy: float, srcw: float, srch: float, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA' to enable per-pixel alpha blending.

    Returns
    -------
    
    """
    a = _ft['JS_LICE_ScaledBlit']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(destBitmap)), ct.c_int(dstx), ct.c_int(dsty), ct.c_int(dstw), ct.c_int(dsth), packp("void*", str(srcBitmap)), ct.c_double(srcx), ct.c_double(srcy), ct.c_double(srcw), ct.c_double(srch), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_IsFlipped(bitmap: VoidPtr) -> bool:
    """
    

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_LICE_IsFlipped']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(bitmap)))
    return bool(ret)


@reapy_boost.inside_reaper()
def LICE_Resize(bitmap: VoidPtr, width: int, height: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Resize']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int)(a)
    
    ret = f(packp("void*", str(bitmap)), ct.c_int(width), ct.c_int(height))
    return 


@reapy_boost.inside_reaper()
def LICE_Clear(bitmap: VoidPtr, color: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Clear']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(bitmap)), ct.c_int(color))
    return 


@reapy_boost.inside_reaper()
def LICE_CreateFont() -> VoidPtr:
    """
    

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_LICE_CreateFont']
    f = ct.CFUNCTYPE(ct.c_void_p, )(a)
    
    ret = f()
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def LICE_DestroyFont(LICEFont: VoidPtr) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_DestroyFont']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(LICEFont)))
    return 


@reapy_boost.inside_reaper()
def LICE_SetFontFromGDI(LICEFont: VoidPtr, GDIFont: VoidPtr, moreFormats: str, encoding: str="utf-8") -> None:
    """
    Converts a GDI font into a LICE font.
    
    The font can be modified by the
    following flags, in a comma-separated list:
    'VERTICAL', 'BOTTOMUP',
    'NATIVE', 'BLUR', 'INVERT', 'MONO', 'SHADOW' or 'OUTLINE'.

    Returns
    -------
    
    """
    a = _ft['JS_LICE_SetFontFromGDI']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_void_p, ct.c_char_p)(a)
    prot_moreFormats = packs_l(moreFormats, size=len(moreFormats.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(LICEFont)), packp("void*", str(GDIFont)), prot_moreFormats)
    return 


@reapy_boost.inside_reaper()
def LICE_SetFontColor(LICEFont: VoidPtr, color: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_SetFontColor']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(LICEFont)), ct.c_int(color))
    return 


@reapy_boost.inside_reaper()
def LICE_SetFontBkColor(LICEFont: VoidPtr, color: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_SetFontBkColor']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(LICEFont)), ct.c_int(color))
    return 


@reapy_boost.inside_reaper()
def LICE_DrawText(bitmap: VoidPtr, LICEFont: VoidPtr, text: str, textLen: int, x1: int, y1: int, x2: int, y2: int, encoding: str="utf-8") -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_LICE_DrawText']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_void_p, ct.c_char_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int)(a)
    prot_text = packs_l(text, size=len(text.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), packp("void*", str(LICEFont)), prot_text, ct.c_int(textLen), ct.c_int(x1), ct.c_int(y1), ct.c_int(x2), ct.c_int(y2))
    return int(ret)


@reapy_boost.inside_reaper()
def LICE_DrawChar(bitmap: VoidPtr, x: int, y: int, c: bytes, color: int, alpha: float, mode: int) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_DrawChar']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_char, ct.c_int, ct.c_double, ct.c_int)(a)
    
    ret = f(packp("void*", str(bitmap)), ct.c_int(x), ct.c_int(y), ct.c_char(c), ct.c_int(color), ct.c_double(alpha), ct.c_int(mode))
    return 


@reapy_boost.inside_reaper()
def LICE_MeasureText(text: str, encoding: str="utf-8") -> ty.Tuple[int, int]:
    """
    

    Returns
    -------
    width: int
    Height: int
    """
    a = _ft['JS_LICE_MeasureText']
    f = ct.CFUNCTYPE(None, ct.c_char_p, ct.c_void_p, ct.c_void_p)(a)
    prot_width = ct.c_int(1)
    prot_Height = ct.c_int(1)
    prot_text = packs_l(text, size=len(text.encode(encoding)), encoding=encoding)
    ret = f(prot_text, ct.byref(prot_width), ct.byref(prot_Height))
    return prot_width.value, prot_Height.value


@reapy_boost.inside_reaper()
def LICE_GradRect(bitmap: VoidPtr, dstx: int, dsty: int, dstw: int, dsth: int, ir: float, ig: float, ib: float, ia: float, drdx: float, dgdx: float, dbdx: float, dadx: float, drdy: float, dgdy: float, dbdy: float, dady: float, mode: str, encoding: str="utf-8") -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_LICE_GradRect']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_int(dstx), ct.c_int(dsty), ct.c_int(dstw), ct.c_int(dsth), ct.c_double(ir), ct.c_double(ig), ct.c_double(ib), ct.c_double(ia), ct.c_double(drdx), ct.c_double(dgdx), ct.c_double(dbdx), ct.c_double(dadx), ct.c_double(drdy), ct.c_double(dgdy), ct.c_double(dbdy), ct.c_double(dady), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_FillRect(bitmap: VoidPtr, x: int, y: int, w: int, h: int, color: int, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_FillRect']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_int(x), ct.c_int(y), ct.c_int(w), ct.c_int(h), ct.c_int(color), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_FillTriangle(bitmap: VoidPtr, x1: int, y1: int, x2: int, y2: int, x3: int, y3: int, color: int, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_FillTriangle']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_int(x1), ct.c_int(y1), ct.c_int(x2), ct.c_int(y2), ct.c_int(x3), ct.c_int(y3), ct.c_int(color), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_FillPolygon(bitmap: VoidPtr, packedX: str, packedY: str, numPoints: int, color: int, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    packedX and packedY are two strings of coordinates, each packed as
    '<i4'.
    
    LICE modes : 'COPY' (default if empty string), 'MASK', 'ADD',
    'DODGE', 'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined
    with 'ALPHA' to enable per-pixel alpha blending.
    
    LICE color format:
    0xAARRGGBB (AA is only used in ALPHA mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_FillPolygon']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_char_p, ct.c_char_p, ct.c_int, ct.c_int, ct.c_double, ct.c_char_p)(a)
    prot_packedX = packs_l(packedX, size=len(packedX.encode(encoding)), encoding=encoding)
    prot_packedY = packs_l(packedY, size=len(packedY.encode(encoding)), encoding=encoding)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), prot_packedX, prot_packedY, ct.c_int(numPoints), ct.c_int(color), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_FillCircle(bitmap: VoidPtr, cx: float, cy: float, r: float, color: int, alpha: float, mode: str, antialias: bool, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_FillCircle']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_int, ct.c_double, ct.c_char_p, ct.c_byte)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_double(cx), ct.c_double(cy), ct.c_double(r), ct.c_int(color), ct.c_double(alpha), prot_mode, ct.c_byte(antialias))
    return 


@reapy_boost.inside_reaper()
def LICE_Line(bitmap: VoidPtr, x1: float, y1: float, x2: float, y2: float, color: int, alpha: float, mode: str, antialias: bool, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Line']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_int, ct.c_double, ct.c_char_p, ct.c_byte)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_double(x1), ct.c_double(y1), ct.c_double(x2), ct.c_double(y2), ct.c_int(color), ct.c_double(alpha), prot_mode, ct.c_byte(antialias))
    return 


@reapy_boost.inside_reaper()
def LICE_Bezier(bitmap: VoidPtr, xstart: float, ystart: float, xctl1: float, yctl1: float, xctl2: float, yctl2: float, xend: float, yend: float, tol: float, color: int, alpha: float, mode: str, antialias: bool, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA' to enable per-pixel alpha blending.
    
    LICE color format:
    0xAARRGGBB (AA is only used in ALPHA mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Bezier']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_int, ct.c_double, ct.c_char_p, ct.c_byte)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_double(xstart), ct.c_double(ystart), ct.c_double(xctl1), ct.c_double(yctl1), ct.c_double(xctl2), ct.c_double(yctl2), ct.c_double(xend), ct.c_double(yend), ct.c_double(tol), ct.c_int(color), ct.c_double(alpha), prot_mode, ct.c_byte(antialias))
    return 


@reapy_boost.inside_reaper()
def LICE_Arc(bitmap: VoidPtr, cx: float, cy: float, r: float, minAngle: float, maxAngle: float, color: int, alpha: float, mode: str, antialias: bool, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Arc']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_int, ct.c_double, ct.c_char_p, ct.c_byte)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_double(cx), ct.c_double(cy), ct.c_double(r), ct.c_double(minAngle), ct.c_double(maxAngle), ct.c_int(color), ct.c_double(alpha), prot_mode, ct.c_byte(antialias))
    return 


@reapy_boost.inside_reaper()
def LICE_Circle(bitmap: VoidPtr, cx: float, cy: float, r: float, color: int, alpha: float, mode: str, antialias: bool, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_Circle']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_int, ct.c_double, ct.c_char_p, ct.c_byte)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_double(cx), ct.c_double(cy), ct.c_double(r), ct.c_int(color), ct.c_double(alpha), prot_mode, ct.c_byte(antialias))
    return 


@reapy_boost.inside_reaper()
def LICE_RoundRect(bitmap: VoidPtr, x: float, y: float, w: float, h: float, cornerradius: int, color: int, alpha: float, mode: str, antialias: bool, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_RoundRect']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double, ct.c_double, ct.c_int, ct.c_int, ct.c_double, ct.c_char_p, ct.c_byte)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_double(x), ct.c_double(y), ct.c_double(w), ct.c_double(h), ct.c_int(cornerradius), ct.c_int(color), ct.c_double(alpha), prot_mode, ct.c_byte(antialias))
    return 


@reapy_boost.inside_reaper()
def LICE_GetPixel(bitmap: VoidPtr, x: int, y: int) -> float:
    """
    Returns the color of the specified pixel.

    Returns
    -------
    color: float
    """
    a = _ft['JS_LICE_GetPixel']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_void_p)(a)
    prot_color = ct.c_double(1.0)
    ret = f(packp("void*", str(bitmap)), ct.c_int(x), ct.c_int(y), ct.byref(prot_color))
    return float(prot_color.value)


@reapy_boost.inside_reaper()
def LICE_PutPixel(bitmap: VoidPtr, x: int, y: int, color: float, alpha: float, mode: str, encoding: str="utf-8") -> None:
    """
    LICE modes: 'COPY' (default if empty string), 'MASK', 'ADD', 'DODGE',
    'MUL', 'OVERLAY' or 'HSVADJ', any of which may be combined with
    'ALPHA'.
    
    LICE color format: 0xAARRGGBB (AA is only used in ALPHA
    mode).

    Returns
    -------
    
    """
    a = _ft['JS_LICE_PutPixel']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_double, ct.c_double, ct.c_char_p)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_int(x), ct.c_int(y), ct.c_double(color), ct.c_double(alpha), prot_mode)
    return 


@reapy_boost.inside_reaper()
def LICE_WritePNG(filename: str, bitmap: LICE_IBitmap, wantAlpha: bool, encoding: str="utf-8") -> bool:
    """
    

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_LICE_WritePNG']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_char_p, ct.c_uint64, ct.c_byte)(a)
    prot_filename = packs_l(filename, size=len(filename.encode(encoding)), encoding=encoding)
    ret = f(prot_filename, packp("LICE_IBitmap*", str(bitmap)), ct.c_byte(wantAlpha))
    return bool(ret)


@reapy_boost.inside_reaper()
def LICE_WriteJPG(filename: str, bitmap: LICE_IBitmap, quality: int, forceBaselineOptional:  ty.Optional[bool]=None, encoding: str="utf-8") -> bool:
    """
    Parameters:
    
     * quality is an integer in the range 1..100.
     *
    forceBaseline is an optional boolean parameter that ensures
    compatibility with all JPEG viewers by preventing too low quality,
    'cubist' settings.

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_LICE_WriteJPG']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_char_p, ct.c_uint64, ct.c_int, ct.c_void_p)(a)
    prot_forceBaselineOptional = ct.c_byte(forceBaselineOptional) if forceBaselineOptional is not None else None
    prot_filename = packs_l(filename, size=len(filename.encode(encoding)), encoding=encoding)
    ret = f(prot_filename, packp("LICE_IBitmap*", str(bitmap)), ct.c_int(quality), None if forceBaselineOptional is None else ct.byref(prot_forceBaselineOptional))
    return bool(ret)


@reapy_boost.inside_reaper()
def LICE_LoadJPG(filename: str, encoding: str="utf-8") -> VoidPtr:
    """
    Returns a system LICE bitmap containing the JPEG.

    Returns
    -------
    ret_value: VoidPtr
    """
    a = _ft['JS_LICE_LoadJPG']
    f = ct.CFUNCTYPE(ct.c_void_p, ct.c_char_p)(a)
    prot_filename = packs_l(filename, size=len(filename.encode(encoding)), encoding=encoding)
    ret = f(prot_filename)
    return VoidPtr(unpackp("void*", ret))


@reapy_boost.inside_reaper()
def LICE_SetAlphaFromColorMask(bitmap: VoidPtr, colorRGB: int) -> None:
    """
    Sets all pixels that match the given color's RGB values to fully
    transparent, and all other pixels to fully opaque.  (All pixels' RGB
    values remain unchanged.)

    Returns
    -------
    
    """
    a = _ft['JS_LICE_SetAlphaFromColorMask']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(bitmap)), ct.c_int(colorRGB))
    return 


@reapy_boost.inside_reaper()
def LICE_AlterBitmapHSV(bitmap: VoidPtr, hue: float, saturation: float, value: float) -> None:
    """
    Hue is rolled over, saturation and value are clamped, all 0..1. (Alpha
    remains unchanged.)

    Returns
    -------
    
    """
    a = _ft['JS_LICE_AlterBitmapHSV']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_double, ct.c_double, ct.c_double)(a)
    
    ret = f(packp("void*", str(bitmap)), ct.c_double(hue), ct.c_double(saturation), ct.c_double(value))
    return 


@reapy_boost.inside_reaper()
def LICE_AlterRectHSV(bitmap: VoidPtr, x: int, y: int, w: int, h: int, hue: float, saturation: float, value: float) -> None:
    """
    Hue is rolled over, saturation and value are clamped, all 0..1. (Alpha
    remains unchanged.)

    Returns
    -------
    
    """
    a = _ft['JS_LICE_AlterRectHSV']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_double, ct.c_double, ct.c_double)(a)
    
    ret = f(packp("void*", str(bitmap)), ct.c_int(x), ct.c_int(y), ct.c_int(w), ct.c_int(h), ct.c_double(hue), ct.c_double(saturation), ct.c_double(value))
    return 


@reapy_boost.inside_reaper()
def LICE_ProcessRect(bitmap: VoidPtr, x: int, y: int, w: int, h: int, mode: str, operand: float, encoding: str="utf-8") -> bool:
    """
    Applies bitwise operations to each pixel in the target rectangle.
    operand: a color in 0xAARRGGBB format.
    
    modes:
     * 'XOR', 'OR' or
    'AND'.
     * 'SET_XYZ', with XYZ any combination of A, R, G, and B:
    copies the specified channels from operand to the bitmap. (Useful for
    setting the alpha values of a bitmap.)
     * 'ALPHAMUL': Performs alpha
    pre-multiplication on each pixel in the rect. operand is ignored in
    this mode. (On WindowsOS, GDI_Blit does not perform alpha
    multiplication on the fly, and a separate alpha pre-multiplication
    step is therefore required.)
    
    NOTE:
    LICE_Blit and LICE_ScaledBlit are
    also useful for processing bitmap colors. For example, to multiply all
    channel values by 1.5:
    reaper.JS_LICE_Blit(bitmap, x, y, bitmap, x, y,
    w, h, 0.5, 'ADD').

    Returns
    -------
    ret_value: bool
    """
    a = _ft['JS_LICE_ProcessRect']
    f = ct.CFUNCTYPE(ct.c_byte, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_int, ct.c_char_p, ct.c_double)(a)
    prot_mode = packs_l(mode, size=len(mode.encode(encoding)), encoding=encoding)
    ret = f(packp("void*", str(bitmap)), ct.c_int(x), ct.c_int(y), ct.c_int(w), ct.c_int(h), prot_mode, ct.c_double(operand))
    return bool(ret)


@reapy_boost.inside_reaper()
def Window_AttachTopmostPin(windowHWND: VoidPtr) -> None:
    """
    Attaches a 'pin on top' button to the window frame. The button should
    remember its state when closing and re-opening the window.
    
    WARNING:
    This function does not yet work on Linux.

    Returns
    -------
    
    """
    a = _ft['JS_Window_AttachTopmostPin']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def Window_AttachResizeGrip(windowHWND: VoidPtr) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_Window_AttachResizeGrip']
    f = ct.CFUNCTYPE(None, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(windowHWND)))
    return 


@reapy_boost.inside_reaper()
def ListView_GetItemCount(listviewHWND: VoidPtr) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_ListView_GetItemCount']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(listviewHWND)))
    return int(ret)


@reapy_boost.inside_reaper()
def ListView_GetSelectedCount(listviewHWND: VoidPtr) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_ListView_GetSelectedCount']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p)(a)
    
    ret = f(packp("void*", str(listviewHWND)))
    return int(ret)


@reapy_boost.inside_reaper()
def ListView_EnsureVisible(listviewHWND: VoidPtr, index: int, partialOK: bool) -> None:
    """
    

    Returns
    -------
    
    """
    a = _ft['JS_ListView_EnsureVisible']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_byte)(a)
    
    ret = f(packp("void*", str(listviewHWND)), ct.c_int(index), ct.c_byte(partialOK))
    return 


@reapy_boost.inside_reaper()
def ListView_GetFocusedItem(listviewHWND: VoidPtr, textOut_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Returns the index and text of the focused item, if any.

    Returns
    -------
    ret_value: int
    text: str
    """
    a = _ft['JS_ListView_GetFocusedItem']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_text = packs_l("", size=textOut_sz, encoding=encoding)
    ret = f(packp("void*", str(listviewHWND)), prot_text, ct.c_int(textOut_sz))
    return int(ret), unpacks_l(prot_text, want_raw=want_raw)


@reapy_boost.inside_reaper()
def ListView_EnumSelItems(listviewHWND: VoidPtr, index: int) -> int:
    """
    Returns the index of the next selected list item with index greater
    that the specified number. Returns -1 if no selected items left.

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_ListView_EnumSelItems']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(listviewHWND)), ct.c_int(index))
    return int(ret)


@reapy_boost.inside_reaper()
def ListView_GetItem(listviewHWND: VoidPtr, index: int, subItem: int, textOut_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[str, int]:
    """
    Returns the text and state of specified item.

    Returns
    -------
    text: str
    state: int
    """
    a = _ft['JS_ListView_GetItem']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_char_p, ct.c_int, ct.c_void_p)(a)
    prot_text = packs_l("", size=textOut_sz, encoding=encoding)
    prot_state = ct.c_int(1)
    ret = f(packp("void*", str(listviewHWND)), ct.c_int(index), ct.c_int(subItem), prot_text, ct.c_int(textOut_sz), ct.byref(prot_state))
    return unpacks_l(prot_text, want_raw=want_raw), prot_state.value


@reapy_boost.inside_reaper()
def ListView_GetItemText(listviewHWND: VoidPtr, index: int, subItem: int, textOut_sz: int, want_raw: bool=False, encoding: str="utf-8") -> str:
    """
    

    Returns
    -------
    text: str
    """
    a = _ft['JS_ListView_GetItemText']
    f = ct.CFUNCTYPE(None, ct.c_void_p, ct.c_int, ct.c_int, ct.c_char_p, ct.c_int)(a)
    prot_text = packs_l("", size=textOut_sz, encoding=encoding)
    ret = f(packp("void*", str(listviewHWND)), ct.c_int(index), ct.c_int(subItem), prot_text, ct.c_int(textOut_sz))
    return unpacks_l(prot_text, want_raw=want_raw)


@reapy_boost.inside_reaper()
def ListView_GetItemState(listviewHWND: VoidPtr, index: int) -> int:
    """
    

    Returns
    -------
    ret_value: int
    """
    a = _ft['JS_ListView_GetItemState']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("void*", str(listviewHWND)), ct.c_int(index))
    return int(ret)


@reapy_boost.inside_reaper()
def ListView_ListAllSelItems(listviewHWND: VoidPtr, itemsOutNeedBig_sz: int, want_raw: bool=False, encoding: str="utf-8") -> ty.Tuple[int, str]:
    """
    Returns the indices of all selected items as a comma-separated list.
    * retval: Number of selected items found; negative or zero if an error
    occured.

    Returns
    -------
    ret_value: int
    itemsNeedBig: str
    """
    a = _ft['JS_ListView_ListAllSelItems']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_void_p, ct.c_char_p, ct.c_int)(a)
    prot_itemsNeedBig = packs_l("", size=itemsOutNeedBig_sz, encoding=encoding)
    ret = f(packp("void*", str(listviewHWND)), prot_itemsNeedBig, ct.c_int(itemsOutNeedBig_sz))
    return int(ret), unpacks_l(prot_itemsNeedBig, want_raw=want_raw)


@reapy_boost.inside_reaper()
def Xen_AudioWriter_Create(filename: str, numchans: int, samplerate: int, encoding: str="utf-8") -> AudioWriter:
    """
    Creates writer for 32 bit floating point WAV

    Returns
    -------
    ret_value: {type_}
    """
    a = _ft['Xen_AudioWriter_Create']
    f = ct.CFUNCTYPE(ct.c_uint64, ct.c_char_p, ct.c_int, ct.c_int)(a)
    prot_filename = packs_l(filename, size=len(filename.encode(encoding)), encoding=encoding)
    ret = f(prot_filename, ct.c_int(numchans), ct.c_int(samplerate))
    return Pointer(unpackp("AudioWriter*", ret), "AudioWriter*")


@reapy_boost.inside_reaper()
def Xen_AudioWriter_Destroy(writer: AudioWriter) -> None:
    """
    Destroys writer

    Returns
    -------
    
    """
    a = _ft['Xen_AudioWriter_Destroy']
    f = ct.CFUNCTYPE(None, ct.c_uint64)(a)
    
    ret = f(packp("AudioWriter*", str(writer)))
    return 


@reapy_boost.inside_reaper()
def Xen_AudioWriter_Write(writer: AudioWriter, numframes: int, data: VoidPtr, offset: int) -> int:
    """
    Write interleaved audio data to disk

    Returns
    -------
    ret_value: int
    """
    a = _ft['Xen_AudioWriter_Write']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_uint64, ct.c_int, ct.c_void_p, ct.c_int)(a)
    
    ret = f(packp("AudioWriter*", str(writer)), ct.c_int(numframes), packp("void*", str(data)), ct.c_int(offset))
    return int(ret)


@reapy_boost.inside_reaper()
def Xen_GetMediaSourceSamples(src: PCM_source, destbuf: VoidPtr, destbufoffset: int, numframes: int, numchans: int, samplerate: float, sourceposition: float) -> int:
    """
    Get interleaved audio data from media source

    Returns
    -------
    ret_value: int
    """
    a = _ft['Xen_GetMediaSourceSamples']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_uint64, ct.c_void_p, ct.c_int, ct.c_int, ct.c_int, ct.c_double, ct.c_double)(a)
    
    ret = f(packp("PCM_source*", str(src)), packp("void*", str(destbuf)), ct.c_int(destbufoffset), ct.c_int(numframes), ct.c_int(numchans), ct.c_double(samplerate), ct.c_double(sourceposition))
    return int(ret)


@reapy_boost.inside_reaper()
def Xen_StartSourcePreview(source: PCM_source, gain: float, loop: bool, outputchanindex:  ty.Optional[int]=None) -> int:
    """
    Start audio preview of a PCM_source. Returns id of a preview handle
    that can be provided to Xen_StopSourcePreview.
    If the given PCM_source
    does not belong to an existing MediaItem/Take, it will be deleted by
    the preview system when the preview is stopped.

    Returns
    -------
    ret_value: int
    """
    a = _ft['Xen_StartSourcePreview']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_uint64, ct.c_double, ct.c_byte, ct.c_void_p)(a)
    prot_outputchanindex = ct.c_int(outputchanindex) if outputchanindex is not None else None
    ret = f(packp("PCM_source*", str(source)), ct.c_double(gain), ct.c_byte(loop), None if outputchanindex is None else ct.byref(prot_outputchanindex))
    return int(ret)


@reapy_boost.inside_reaper()
def Xen_StopSourcePreview(preview_id: int) -> int:
    """
    Stop audio preview. id -1 stops all.

    Returns
    -------
    ret_value: int
    """
    a = _ft['Xen_StopSourcePreview']
    f = ct.CFUNCTYPE(ct.c_int, ct.c_int)(a)
    
    ret = f(ct.c_int(preview_id))
    return int(ret)

