"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const fs = require("fs");
const os = require("os");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const ec2 = require("../lib");
const cfn_init_internal_1 = require("../lib/private/cfn-init-internal");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
describe('InitCommand', () => {
    test('throws error on empty argv command', () => {
        expect(() => { ec2.InitCommand.argvCommand([]); }).toThrow();
    });
    test('auto-generates an indexed command key if none is provided', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh');
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered['000']).toBeDefined();
    });
    test('renders a minimalist template when no options are defined', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh');
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '000': { command: ['/bin/sh'] },
        });
    });
    test('creates a shell command with all provided options', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh', {
            key: 'command_0',
            env: { SECRETS_FILE: '/tmp/secrets' },
            cwd: '/home/myUser',
            testCmd: 'test -d /home/myUser',
            ignoreErrors: false,
            waitAfterCompletion: ec2.InitCommandWaitDuration.of(core_1.Duration.hours(2)),
        });
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            command_0: {
                command: ['/bin/sh'],
                env: { SECRETS_FILE: '/tmp/secrets' },
                cwd: '/home/myUser',
                test: 'test -d /home/myUser',
                ignoreErrors: false,
                waitAfterCompletion: 7200,
            },
        });
    });
    test('creates an argv command with all provided options', () => {
        // GIVEN
        const command = ec2.InitCommand.argvCommand(['/bin/sh', '-c', 'doStuff'], {
            key: 'command_0',
            env: { SECRETS_FILE: '/tmp/secrets' },
            cwd: '/home/myUser',
            testCmd: 'test -d /home/myUser',
            ignoreErrors: false,
            waitAfterCompletion: ec2.InitCommandWaitDuration.of(core_1.Duration.hours(2)),
        });
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            command_0: {
                command: ['/bin/sh', '-c', 'doStuff'],
                env: { SECRETS_FILE: '/tmp/secrets' },
                cwd: '/home/myUser',
                test: 'test -d /home/myUser',
                ignoreErrors: false,
                waitAfterCompletion: 7200,
            },
        });
    });
    test('errors when waitAfterCompletion is specified for Linux systems', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh', {
            key: 'command_0',
            env: { SECRETS_FILE: '/tmp/secrets' },
            cwd: '/home/myUser',
            testCmd: 'test -d /home/myUser',
            ignoreErrors: false,
            waitAfterCompletion: ec2.InitCommandWaitDuration.of(core_1.Duration.hours(2)),
        });
        // THEN
        expect(() => {
            command._bind(defaultOptions(cfn_init_internal_1.InitPlatform.LINUX));
        }).toThrow(/'waitAfterCompletion' is only valid for Windows/);
    });
});
describe('InitFile', () => {
    test('fromString creates inline content', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', 'My content');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'My content',
                encoding: 'plain',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromString creates inline content from base64-encoded content', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', Buffer.from('Hello').toString('base64'), {
            base64Encoded: true,
        });
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'SGVsbG8=',
                encoding: 'base64',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('mode, user, group settings not allowed for Windows', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', 'My content', {
            group: 'root',
            owner: 'root',
            mode: '000644',
        });
        // WHEN
        expect(() => {
            file._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Owner, group, and mode options not supported for Windows.');
    });
    test('symlink throws an error if mode is set incorrectly', () => {
        expect(() => {
            ec2.InitFile.symlink('/tmp/foo', '/tmp/bar', {
                mode: '000644',
            });
        }).toThrow('File mode for symlinks must begin with 120XXX');
    });
    test('symlink sets mode is not set', () => {
        // GIVEN
        const file = ec2.InitFile.symlink('/tmp/foo', '/tmp/bar');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: '/tmp/bar',
                encoding: 'plain',
                owner: 'root',
                group: 'root',
                mode: '120644',
            },
        });
    });
    test('fromFileInline renders the file contents inline', () => {
        // GIVEN
        const tmpFilePath = createTmpFileWithContent('Hello World!');
        const file = ec2.InitFile.fromFileInline('/tmp/foo', tmpFilePath);
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'Hello World!',
                encoding: 'plain',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromObject renders the contents inline as an object', () => {
        // GIVEN
        const content = {
            version: '1234',
            secretsFile: '/tmp/secrets',
        };
        const file = ec2.InitFile.fromObject('/tmp/foo', content);
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: {
                    version: '1234',
                    secretsFile: '/tmp/secrets',
                },
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromFileInline respects the base64 flag', () => {
        // GIVEN
        const tmpFilePath = createTmpFileWithContent('Hello');
        const file = ec2.InitFile.fromFileInline('/tmp/foo', tmpFilePath, {
            base64Encoded: true,
        });
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'SGVsbG8=',
                encoding: 'base64',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromUrl uses the provided URL as a source', () => {
        // GIVEN
        const file = ec2.InitFile.fromUrl('/tmp/foo', 'https://aws.amazon.com/');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                source: 'https://aws.amazon.com/',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
});
describe('InitGroup', () => {
    test('renders without a group id', () => {
        // GIVEN
        const group = ec2.InitGroup.fromName('amazon');
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({ amazon: {} });
    });
    test('renders with a group id', () => {
        // GIVEN
        const group = ec2.InitGroup.fromName('amazon', 42);
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({ amazon: { gid: 42 } });
    });
    test('groups are not supported for Windows', () => {
        // GIVEN
        const group = ec2.InitGroup.fromName('amazon');
        // WHEN
        expect(() => {
            group._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Init groups are not supported on Windows');
    });
});
describe('InitUser', () => {
    test('fromName accepts just a name to create a user', () => {
        // GIVEN
        const group = ec2.InitUser.fromName('sysuser1');
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({ sysuser1: {} });
    });
    test('renders with all options present', () => {
        // GIVEN
        const group = ec2.InitUser.fromName('sysuser1', {
            userId: 42,
            homeDir: '/home/sysuser1',
            groups: ['amazon'],
        });
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            sysuser1: {
                uid: 42,
                homeDir: '/home/sysuser1',
                groups: ['amazon'],
            },
        });
    });
    test('users are not supported for Windows', () => {
        // GIVEN
        const group = ec2.InitUser.fromName('sysuser1');
        // WHEN
        expect(() => {
            group._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Init users are not supported on Windows');
    });
});
describe('InitPackage', () => {
    test('rpm auto-generates a name if none is provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.rpm('https://example.com/rpm/mypkg.rpm');
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            rpm: {
                '000': ['https://example.com/rpm/mypkg.rpm'],
            },
        });
    });
    test('rpm uses name if provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.rpm('https://example.com/rpm/mypkg.rpm', { key: 'myPkg' });
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            rpm: {
                myPkg: ['https://example.com/rpm/mypkg.rpm'],
            },
        });
    });
    test('rpm is not supported for Windows', () => {
        // GIVEN
        const pkg = ec2.InitPackage.rpm('https://example.com/rpm/mypkg.rpm');
        // THEN
        expect(() => {
            pkg._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Windows only supports the MSI package type');
    });
    test.each([
        ['yum', ec2.InitPackage.yum],
        ['rubygems', ec2.InitPackage.rubyGem],
        ['python', ec2.InitPackage.python],
        ['apt', ec2.InitPackage.apt],
    ])('%s accepts a package without versions', (pkgType, fn) => {
        // GIVEN
        const pkg = fn('httpd');
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            [pkgType]: { httpd: [] },
        });
    });
    test.each([
        ['yum', ec2.InitPackage.yum],
        ['rubygems', ec2.InitPackage.rubyGem],
        ['python', ec2.InitPackage.python],
        ['apt', ec2.InitPackage.apt],
    ])('%s accepts a package with versions', (pkgType, fn) => {
        // GIVEN
        const pkg = fn('httpd', { version: ['1.0', '2.0'] });
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            [pkgType]: { httpd: ['1.0', '2.0'] },
        });
    });
    test.each([
        ['yum', ec2.InitPackage.yum],
        ['rubygems', ec2.InitPackage.rubyGem],
        ['python', ec2.InitPackage.python],
        ['apt', ec2.InitPackage.apt],
    ])('%s is not supported on Windows', (_pkgType, fn) => {
        // GIVEN
        const pkg = fn('httpd');
        expect(() => {
            pkg._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Windows only supports the MSI package type');
    });
    test('msi auto-generates a name if none is provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.msi('https://example.com/rpm/mypkg.msi');
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            msi: {
                '000': 'https://example.com/rpm/mypkg.msi',
            },
        });
    });
    test('msi uses name if provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.msi('https://example.com/rpm/mypkg.msi', { key: 'myPkg' });
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            msi: {
                myPkg: 'https://example.com/rpm/mypkg.msi',
            },
        });
    });
    test('msi is not supported for Linux', () => {
        // GIVEN
        const pkg = ec2.InitPackage.msi('https://example.com/rpm/mypkg.msi');
        // THEN
        expect(() => {
            pkg._bind(defaultOptions(cfn_init_internal_1.InitPlatform.LINUX));
        }).toThrow('MSI installers are only supported on Windows systems.');
    });
});
describe('InitService', () => {
    test.each([
        ['Linux', 'sysvinit', cfn_init_internal_1.InitPlatform.LINUX],
        ['Windows', 'windows', cfn_init_internal_1.InitPlatform.WINDOWS],
    ])('enable always sets enabled and running to true for %s', (_platform, key, platform) => {
        // GIVEN
        const service = ec2.InitService.enable('httpd');
        // WHEN
        const rendered = service._bind(defaultOptions(platform)).config;
        // THEN
        expect(rendered[key]).toBeDefined();
        expect(rendered[key]).toEqual({
            httpd: {
                enabled: true,
                ensureRunning: true,
            },
        });
    });
    test.each([
        ['Linux', 'sysvinit', cfn_init_internal_1.InitPlatform.LINUX],
        ['Windows', 'windows', cfn_init_internal_1.InitPlatform.WINDOWS],
    ])('disable returns a minimalist disabled service for %s', (_platform, key, platform) => {
        // GIVEN
        const service = ec2.InitService.disable('httpd');
        // WHEN
        const rendered = service._bind(defaultOptions(platform)).config;
        // THEN
        expect(rendered[key]).toBeDefined();
        expect(rendered[key]).toEqual({
            httpd: {
                enabled: false,
                ensureRunning: false,
            },
        });
    });
    test('fromOptions renders all options', () => {
        // GIVEN
        const restartHandle = new ec2.InitServiceRestartHandle();
        const serviceRestartHandles = [restartHandle];
        const initFile = ec2.InitFile.fromString('/etc/my.cnf', '[mysql]', { serviceRestartHandles });
        const initSource = ec2.InitSource.fromUrl('/tmp/foo', 'https://example.com/archive.zip', { serviceRestartHandles });
        const initPackage = ec2.InitPackage.yum('httpd', { serviceRestartHandles });
        const initCommand = ec2.InitCommand.shellCommand('/bin/true', { serviceRestartHandles, key: 'myCmd' });
        const service = ec2.InitService.enable('httpd', { serviceRestartHandle: restartHandle });
        // WHEN
        const bindOptions = defaultOptions(cfn_init_internal_1.InitPlatform.LINUX);
        initFile._bind(bindOptions);
        initSource._bind(bindOptions);
        initPackage._bind(bindOptions);
        initCommand._bind(bindOptions);
        const rendered = service._bind(bindOptions).config;
        // THEN
        expect(rendered.sysvinit).toEqual({
            httpd: {
                enabled: true,
                ensureRunning: true,
                files: ['/etc/my.cnf'],
                sources: ['/tmp/foo'],
                packages: { yum: ['httpd'] },
                commands: ['myCmd'],
            },
        });
    });
});
describe('InitSource', () => {
    test('fromUrl renders correctly', () => {
        // GIVEN
        const source = ec2.InitSource.fromUrl('/tmp/foo', 'https://example.com/archive.zip');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': 'https://example.com/archive.zip',
        });
    });
    test('fromGitHub builds a path to the tarball', () => {
        // GIVEN
        const source = ec2.InitSource.fromGitHub('/tmp/foo', 'aws', 'aws-cdk', 'master');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': 'https://github.com/aws/aws-cdk/tarball/master',
        });
    });
    test('fromGitHub defaults to master if refspec is omitted', () => {
        // GIVEN
        const source = ec2.InitSource.fromGitHub('/tmp/foo', 'aws', 'aws-cdk');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': 'https://github.com/aws/aws-cdk/tarball/master',
        });
    });
    test('fromS3Object uses object URL', () => {
        // GIVEN
        const bucket = s3.Bucket.fromBucketName(stack, 'bucket', 'MyBucket');
        const source = ec2.InitSource.fromS3Object('/tmp/foo', bucket, 'myKey');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': expect.stringContaining('/MyBucket/myKey'),
        });
    });
});
function getElementConfig(element, platform) {
    return element._bind(defaultOptions(platform)).config;
}
function defaultOptions(platform) {
    return {
        scope: stack,
        index: 0,
        platform,
        instanceRole: new iam.Role(stack, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        }),
    };
}
function createTmpFileWithContent(content) {
    const suffix = crypto.randomBytes(4).toString('hex');
    const fileName = path.join(os.tmpdir(), `cfn-init-element-test-${suffix}`);
    fs.writeFileSync(fileName, content);
    return fileName;
}
//# sourceMappingURL=data:application/json;base64,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