"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * @internal
     */
    _attach(attachedResource, attachOptions) {
        var _a, _b, _c;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata('AWS::CloudFormation::Init', bindResult.configData);
        const fingerprint = contentHash(JSON.stringify(bindResult.configData)).substr(0, 16);
        attachOptions.instanceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_a = attachOptions.configSets) !== null && _a !== void 0 ? _a : ['default']).join(',');
        const printLog = (_b = attachOptions.printLog) !== null && _b !== void 0 ? _b : true;
        if ((_c = attachOptions.embedFingerprint) !== null && _c !== void 0 ? _c : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
/**
 * A collection of configuration elements
 */
class InitConfig {
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const authentication = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig]
            .map(c => c === null || c === void 0 ? void 0 : c.authentication)
            .reduce(deepMerge, undefined);
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
        };
    }
    bindForType(elementType, renderOptions) {
        var _a;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_a = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _a !== void 0 ? _a : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _a, _b;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_a = target[key]) !== null && _a !== void 0 ? _a : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_b = target[key]) !== null && _b !== void 0 ? _b : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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