# Loads SOC and O*NET-SOC Objects into database
# Requires a final_onet_taxonomy_2019.csv file located in the same directory where python is being run.
# That file is generated by https://github.com/CareerVillage/socbot/blob/master/onet_scraper/onet_soc_2019.py

import pandas as pd
from django_occupations.models import ONetAlternateTitle,
                                      ONetOccupation,
                                      SOCDetailedOccupation,
                                      SOCBroadOccupation,
                                      SOCMinorGroup,
                                      SOCMajorGroup,
                                      SOCIntermediateAggregationGroup,
                                      SOCHighLevelAggregationGroup

# Optional, rich improves console readability
from rich import pretty # Optional
pretty.install()
from rich.progress import track # Optional
from rich.traceback import install
install()


# Load dataframe from csv
df = pd.read_csv("final_onet_taxonomy_2019.csv")

# Print column names
df.columns

def create_code(full_code, code_level):
    if code_level == 'major_group':
        return full_code[0:3] + '0000'
    elif code_level == 'minor_group':
        return full_code[0:5] + '00'
    elif code_level == 'broad_occupation':
        return full_code[0:6] + '0'
    elif code_level == 'detailed_occupation':
        return full_code[0:7]
    elif code_level == 'onet_occupation':
        return full_code
    else:
        raise ValueError(f'No matching code_level found. Provided code_level is {code_level}.')


def main():
    # Iterate over each row
    for row in track(df.itertuples(index=False, name='AlternateTitle'), total=len(df.index)):
        # Each row has a major_title, minor_title, broad_title, soc_onet_code, detailed_group, and alternate_title
        major_group, created = SOCMajorGroup.objects.get_or_create(
                                    name=row.major_title,
                                    soc_code=create_code(row.soc_onet_code, 'major_group'),
                                    )
        minor_group, created = SOCMinorGroup.objects.get_or_create(
                                    name=row.minor_title,
                                    soc_code=create_code(row.soc_onet_code, 'minor_group'),
                                    )
        minor_group.major_group = major_group # Establish the relationship
        minor_group.save()
        broad_occupation, created = SOCBroadOccupation.objects.get_or_create(
                                    name=row.broad_title,
                                    soc_code=create_code(row.soc_onet_code, 'broad_occupation'),
                                    )
        broad_occupation.minor_group = minor_group # Establish the relationship
        broad_occupation.save()
        detailed_occupation, created = SOCDetailedOccupation.objects.get_or_create(
                                    name=row.detailed_group,
                                    soc_code=create_code(row.soc_onet_code, 'detailed_occupation'),
                                    )
        detailed_occupation.broad_occupation = broad_occupation # Establish the relationship
        detailed_occupation.save()
        onet_occupation, created = ONetOccupation.objects.get_or_create(
                                    title=row.detailed_group,
                                    onet_soc_code=create_code(row.soc_onet_code, 'onet_occupation'),
                                    )
        onet_occupation.soc_occupation = detailed_occupation # Establish the relationship
        onet_occupation.save()
        alternate_title, created = ONetAlternateTitle.objects.get_or_create(
                                    alternate_title=row.alternate_title,
                                    title=row.detailed_group,
                                    onet_soc_code=create_code(row.soc_onet_code, 'onet_occupation'),
                                    )
        alternate_title.onet_soc_occupation.add(onet_occupation)
        alternate_title.soc_occupation = detailed_occupation
        alternate_title.save()


#### PROBLEM:
#### PROBLEM:
#### PROBLEM:
#### PROBLEM:
#### PROBLEM:
# ONetAlternateTitle uses a foreignkey relationship for SOC occupations. This is a problem, because a single altenrnate titles text can be used multiple times. For example the alternate_title associate_rofessor is applied to many many ONet occupations and SOC occupations.
# A possible solution is to change the ONetAlternateTitlet o use M2M isntead of ForeignKey relationships and then go back over the entire df a second time at the end and add many-to-many relationships for EVERY matching occupation rather than specifying only the most recent. 
# DOES THIS AFFECT OTHER RELATIONSHIPS? I shouldn't overwrite relationships on each row if they're overwriting previously set relationships that matter. 
# How many relations are manytomany? Looks like only one: AlternateTitle.

main()
