#!/usr/bin/env python

import codecs
import io
import json
import os
import sys
import unittest
from unittest.mock import Mock, patch

from herepy import MapTileApi, MapTileApiType, UnauthorizedError


class MapTileApiTest(unittest.TestCase):
    def setUp(self):
        api = MapTileApi(api_key="api_key")
        self._api = api

    def test_initiation(self):
        self.assertIsInstance(self._api, MapTileApi)
        self.assertEqual(self._api._api_key, "api_key")
        self.assertIsNone(self._api._base_url)

    @patch("herepy.map_tile_api.requests.get")
    def test_get_map_tile_with_parameters_succeed(self, mock_get):
        with open("testdata/tiles/berlin.png", "rb") as tile:
            mock_get.return_value = Mock(ok=True)
            mock_get.return_value.content = tile.read
        map_tile = self._api.get_maptile(
            latitude=52.525439, longitude=13.38727, zoom=12
        )
        self.assertIsNotNone(map_tile)

    @patch("herepy.map_tile_api.requests.get")
    def test_get_map_tile_with_parameters_fails(self, mock_get):
        mock_get.return_value = Mock(ok=True)
        mock_get.return_value.content = None
        map_tile = self._api.get_maptile(
            latitude=52.525439, longitude=13.38727, zoom=12
        )
        self.assertIsNone(map_tile)

    @patch("herepy.map_tile_api.requests.get")
    def test_get_map_tile_with_query_parameters_succeed(self, mock_get):
        mock_get.return_value = Mock(ok=True)
        mock_get.return_value.content = b'"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"\n'
        map_tile = self._api.get_maptile(
            latitude=52.525439,
            longitude=13.38727,
            zoom=12,
            query_parameters={"output": "base64"},
        )
        self.assertIsNotNone(map_tile)

    @patch("herepy.map_tile_api.requests.get")
    def test_get_map_tile_with_query_parameters_fails(self, mock_get):
        mock_get.return_value = Mock(ok=True)
        mock_get.return_value.content = None
        map_tile = self._api.get_maptile(
            latitude=52.525439,
            longitude=13.38727,
            zoom=12,
            query_parameters={"output": "base64"},
        )
        self.assertIsNone(map_tile)

    @patch("herepy.map_tile_api.requests.get")
    def test_get_map_tile_with_default_parameters_unauthorized(self, mock_get):
        with open("testdata/models/unauthorized_error.json", "rb") as f:
            mock_get.return_value = Mock(ok=True)
            mock_get.return_value.content = f.read()
        with self.assertRaises(UnauthorizedError):
            map_tile = self._api.get_maptile(
                latitude=52.525439, longitude=13.38727, zoom=12
            )
