// Copyright (c) Mito
import React from 'react';
import { getLastModifiedString } from '../../../utils/time';
import CSVFileIcon from '../../icons/CSVFileIcon';
import DirectoryIcon from '../../icons/DirectoryIcon';
import FileIcon from '../../icons/FileIcon';
import Col from '../../spacing/Col';
import Row from '../../spacing/Row';
/*
    Helper function that gets an ending of a file, or
    undefined if no such file ending exists
*/
export const getFileEnding = (elementName) => {
    try {
        // Take just the file ending
        const nameSplit = elementName.split('.');
        return nameSplit[nameSplit.length - 1];
    }
    catch (_a) {
        return undefined;
    }
};
/*
    Helper function that, for a given file, returns if there is an
    error in importing the file.

    Helpful in displaying in-place errors that tells users they cannot
    import xlsx files.
*/
export const getInvalidFileError = (selectedElement, excelImportEnabled) => {
    // We do not display an error on directories, as you cannot
    // import them but we don't want to overload you
    if (selectedElement.isDirectory) {
        return undefined;
    }
    const VALID_FILE_ENDINGS = [
        'csv',
        'tsv',
        'txt',
        'tab',
    ];
    // If excel import is enabled, then add it as a valid ending
    if (excelImportEnabled) {
        VALID_FILE_ENDINGS.push('xlsx');
    }
    // Check if the file ending is a type that we support out of the box
    for (const ending of VALID_FILE_ENDINGS) {
        if (selectedElement.name.toLowerCase().endsWith(ending)) {
            return undefined;
        }
    }
    // We try and get the ending from the file
    const fileEnding = getFileEnding(selectedElement.name);
    if (fileEnding === undefined) {
        return 'Sorry, we don\'t support that file type.';
    }
    else if (fileEnding == 'xlsx') {
        return 'Upgrade to pandas>=0.25.0 and Python>3.6 to import Excel files.';
    }
    else {
        return `Sorry, we don't support ${fileEnding} files.`;
    }
};
/*
    An file or folder that is displayed by the file browser
*/
function FileBrowserElement(props) {
    // Check if this element being displayed is the selected element in the
    // file browser!
    const isSelected = props.selectedElement !== undefined
        && props.element.isDirectory === props.selectedElement.isDirectory
        && props.element.name === props.selectedElement.name;
    const invalidFileError = getInvalidFileError(props.element, props.excelImportEnabled);
    return (React.createElement("div", { 
        // NOTE: we make this text unselectable, as we want users
        // to be able to double click
        className: 'file-browser-element p-5px text-unselectable', title: props.element.name, style: { background: isSelected ? '#D5C0FF' : '' }, onClick: (e) => {
            // If the user is double clicking, then we don't select or
            // unselect the element
            if (e.detail > 1) {
                return;
            }
            /*
                If the element is selected, we unselect it. Otherwise,
                we select it.
            */
            if (isSelected) {
                props.setSelectedElement(undefined);
            }
            else {
                props.setSelectedElement(props.element);
            }
        }, onDoubleClick: () => {
            if (props.element.isDirectory) {
                const newPathParts = props.pathParts ? props.pathParts : [];
                newPathParts.push(props.element.name);
                props.setCurrPathParts(newPathParts);
            }
            else {
                void props.importElement(props.element);
            }
        } },
        React.createElement(Row, { suppressTopBottomMargin: true, justify: 'space-between' },
            React.createElement(Col, { span: 17, offsetRight: 1 },
                React.createElement("div", { className: 'flexbox-row' },
                    React.createElement("div", { className: 'mr-5px mt-2px' },
                        props.element.isDirectory && React.createElement(DirectoryIcon, null),
                        !props.element.isDirectory && invalidFileError === undefined && React.createElement(CSVFileIcon, null),
                        !props.element.isDirectory && invalidFileError !== undefined && React.createElement(FileIcon, null)),
                    React.createElement("div", null, props.element.name))),
            React.createElement(Col, { span: 6 },
                React.createElement("p", { className: 'text-align-right' }, getLastModifiedString(props.element.lastModified)))),
        isSelected && invalidFileError !== undefined &&
            React.createElement("div", { className: 'pl-5px pr-5px' },
                React.createElement("span", null,
                    " ",
                    invalidFileError,
                    " "))));
}
export default FileBrowserElement;
//# sourceMappingURL=FileBrowserElement.js.map