// Copyright (c) Mito
import React from 'react';
import { isFilterGroup } from './filterTypes';
import { Filter } from './Filter';
import FilterGroup from './FilterGroup';
import DropdownButton from '../../../../elements/DropdownButton';
import Select from '../../../../elements/Select';
import Row from '../../../../spacing/Row';
import Col from '../../../../spacing/Col';
import '../../../../../../css/taskpanes/ControlPanel/FilterCard.css';
import DropdownItem from '../../../../elements/DropdownItem';
import { getEmptyFilterData } from './utils';
export const ADD_FILTER_SELECT_TITLE = '+ Add Filter';
/*
    Component that contains all that one needs to filter!
*/
function FilterCard(props) {
    /*
        Adds a new, blank filter to the end of the filters list
    */
    const addFilter = () => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            newFilters.push(getEmptyFilterData(props.columnDtype));
            return newFilters;
        });
    };
    /*
        Creates a new filter group (at the bottom) with a single empty
        filter
    */
    const addFilterGroup = () => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            newFilters.push({
                filters: [
                    getEmptyFilterData(props.columnDtype)
                ],
                operator: 'And'
            });
            return newFilters;
        });
    };
    /*
        Adds a blank new filter to the end of a specific group
    */
    const addFilterToGroup = (groupIndex) => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            const filterGroup = newFilters[groupIndex];
            if (isFilterGroup(filterGroup)) {
                // If we do have a filter group at that groupIndex, then we add a new filter to it
                filterGroup.filters.push(getEmptyFilterData(props.columnDtype));
                return newFilters;
            }
            else {
                // We make no changes if this was not a filter group, which should never occur
                return prevFilters;
            }
        });
    };
    /*
        Deletes a filter that is at the given index in the main
        filter list.
    */
    const deleteFilter = (filterIndex) => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            newFilters.splice(filterIndex, 1);
            return newFilters;
        });
    };
    /*
        Deletes a filter that is at a given index in filter list
        of a specific filter group
    */
    const deleteFilterFromGroup = (groupIndex, filterIndex) => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            const filterGroup = newFilters[groupIndex];
            if (isFilterGroup(filterGroup)) {
                // If we do have a filter group at that groupIndex, then we delete the filter
                // at the passed filterIndex
                filterGroup.filters.splice(filterIndex, 1);
                // If there are no filters left in this group, then we remove the entire group
                if (filterGroup.filters.length === 0) {
                    newFilters.splice(groupIndex, 1);
                }
                return newFilters;
            }
            else {
                // We make no changes if this was not a filter group, which should never occur
                return prevFilters;
            }
        });
    };
    /*
        Sets a filter at the given index to the new filter value
    */
    const setFilter = (filterIndex, filter) => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            newFilters[filterIndex] = filter;
            return newFilters;
        });
    };
    /*
        Sets a filter at the given filterIndex in the specific group at the given
        groupIndex to the new filter value
    */
    const setFilterInGroup = (groupIndex, filterIndex, filter) => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            const filterGroup = newFilters[groupIndex];
            if (isFilterGroup(filterGroup)) {
                filterGroup.filters[filterIndex] = filter;
                return newFilters;
            }
            else {
                // We make no changes if this was not a filter group, which should never occur
                return prevFilters;
            }
        });
    };
    /*
        Sets the operator that combines a specific filter group
    */
    const setOperatorInGroup = (groupIndex, operator) => {
        props.setFilters((prevFilters) => {
            const newFilters = [...prevFilters];
            const filterGroup = newFilters[groupIndex];
            if (isFilterGroup(filterGroup)) {
                filterGroup.operator = operator;
                return newFilters;
            }
            else {
                // We make no changes if this was not a filter group, which should never occur
                return prevFilters;
            }
        });
    };
    return (React.createElement("div", null,
        React.createElement("div", { className: 'text-header-3 mt-15px' },
            React.createElement("p", null, " Filter ")),
        props.filters.map((filterOrGroup, index) => {
            if (isFilterGroup(filterOrGroup)) {
                return (
                /*
                    If the FilterGroup is the first Filter or FilterGroup
                    in the FilterCard, add a 'Where'
                */
                React.createElement(Row, { justify: 'space-between', align: 'top' },
                    React.createElement(Col, { span: 4 },
                        index === 0 &&
                            React.createElement("p", { className: 'text-body-1' }, "Where"),
                        index !== 0 &&
                            React.createElement(Select, { value: props.operator, onChange: (newOperator) => props.setOperator(newOperator), dropdownWidth: 'small' },
                                React.createElement(DropdownItem, { title: 'And' }),
                                React.createElement(DropdownItem, { title: 'Or' }))),
                    React.createElement(Col, { span: 19 },
                        React.createElement(FilterGroup, { key: index, mainOperator: props.operator, filters: filterOrGroup.filters, groupOperator: filterOrGroup.operator, setFilter: (filterIndex, newFilter) => {
                                setFilterInGroup(index, filterIndex, newFilter);
                            }, setOperator: (newOperator) => {
                                setOperatorInGroup(index, newOperator);
                            }, deleteFilter: (filterIndex) => {
                                deleteFilterFromGroup(index, filterIndex);
                            }, addFilter: () => addFilterToGroup(index), columnDtype: props.columnDtype }))));
            }
            else {
                return (React.createElement(Filter, { first: index === 0, key: index, filter: filterOrGroup, operator: props.operator, displayOperator: true, setFilter: (newFilter) => {
                        setFilter(index, newFilter);
                    }, setOperator: props.setOperator, deleteFilter: () => { deleteFilter(index); }, columnDtype: props.columnDtype }));
            }
        }),
        React.createElement(DropdownButton, { text: ADD_FILTER_SELECT_TITLE, width: 'medium', dropdownWidth: 'medium' },
            React.createElement(DropdownItem, { title: 'Add a Filter', onClick: addFilter }),
            React.createElement(DropdownItem, { title: 'Add a Group of Filters', onClick: addFilterGroup }))));
}
export default FilterCard;
//# sourceMappingURL=FilterCard.js.map