# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ptimeit']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pretty-timeit',
    'version': '0.1.0',
    'description': 'timeit, but with the interface it should have had.',
    'long_description': '# ptimeit\n[![codecov](https://codecov.io/gh/AbhinavOmprakash/pretty-timeit/branch/main/graph/badge.svg?token=QCRpIcv84o)](https://codecov.io/gh/AbhinavOmprakash/pretty-timeit)\n\n***timeit, but with the interface it should have had.***   \n\n## Why did I write this?\ntimeit is a great module, but whenever I wanted to use it, I had to look up the syntax, and even after that it was tricky to get things working. I wanted something that had the similar functionality of timeit but an easier, more intuitive syntax.   \nUnder the hood, I take the same approach as timeit does-\n- Everything is imported once.\n- The garbage collector is disabled when the function is run.\n- By default the function runs 1 milllion times.\n\n## Installation\n\nWith pip.\n\n```Bash\n$ pip install pretty-timeit\n```\n\nWith poetry.\n\n```Bash\n$ poetry add pretty-timeit\n```\n\n## Usage.\n\n### A simple example.\n\n```Python\nfrom ptimeit import timethis, Timer\n\n# although this is a decorator, your original function will not be modified.\n@timethis() \ndef function_to_be_timed():\n    lst = [i for i in range(10)]\n\nTimer.run() # Call this at the end of the file.\n```\n\noutput\n```Console\nname                 | Execution time\nfunction_to_be_timed | 0.5608107\n```\nNote: By default `Timer.run()` Prints the results to the console. \nIf you want it to return a string, do this.\n\n```Python\nTimer.run(print_results=False) # returns a string\n```\n\n\n### Mixing positional arguments with keyword arguments.\nPositional Arguments must be passed inside a list even if there is one argument.  \nKeyword arguments must be passed inside a dictionary.   \nNote the order follows the common idiom of `function(*args, **kwargs)`.\n\n```Python\n@timethis( [10], {"second_count_up_to":10} )\ndef function_to_be_timed(count_up_to, second_count_up_to=100):\n    lst = [i for i in range(count_up_to)]\n    lst2 = [i for i in range(second_count_up_to)]\n\nTimer.run()\n```\n\n### Adding custom descriptive names to your functions\nIf you want to see a different name other than the function name in the final report you can pass that to the decorator using `name=""` argument.\n```Python\n@timethis([10], name="A list comprehension that counts up to 10")\ndef function_to_be_timed(count_up_to):\n    lst = [i for i in range(count_up_to)]\nTimer.run()\n```\n\n```Console\nname                                      | Execution time\nA list comprehension that counts up to 10 | 0.5393135\n``` \n\n### Changing the number of times the function is repeated.\nBy default, like timeit, The function is repeated 1 million times,\nbut this can be changed by passing a `repeat=` to `Timer.run()` like this.\n\n```Python\nTimer.run(repeat=100) #The function to be timed will loop for hundred times.\n```\n\n### Comparing multiple functions.\nA common use case that I have found for the timeit module, is to compare\nthe runtime speed of two different algorithms, this is very easy to do in p-timeit.\n\n```Python\n@timethis(100, name="using a for loop")\ndef my_func(count_up_to):\n    lst=[]\n    for i in range( count_up_to): \n        lst.append(i)\n\n@timethis(100, name="using a List comprehension")\ndef my_func_2(count_up_to):\n    lst = [i for i in range(count_up_to)]\n\nTimer.run(compare=True)\n```\nOutput\n```console\nrank | name                       | Execution time\n1    | using a List comprehension | 2.3460704000000003\n2    | using a for loop           | 4.37266\n```\nAs you can see the list comprehension is faster than using a for loop.\n\nNote: you can compare as many functions as you like. Not just two.\nThe `compare=True` flag formats the output, And orders the results from The fastest functions to the slowest.\nYou can still run multiple functions with `Timer.run()`.\n\nHere\'s an example without compare\n\n```python\n@timethis([10], name="using a for loop")\ndef my_func(count_up_to):\n    lst=[]\n    for i in range( count_up_to): \n        lst.append(i)\n\n@timethis([10], name="using a List comprehension")\ndef my_func_2(count_up_to):\n    lst = [i for i in range(count_up_to)]\n\n@timethis([10], name="using two lists")\ndef my_func_3(count_up_to):\n    lst = [i for i in range(count_up_to)]\n    lst2 = [i for i in range(count_up_to)]\n\nTimer.run()\n```\noutput\n```Console\nname                       | Execution time\nusing a for loop           | 0.7021932000000001\nusing a List comprehension | 0.6186622999999999\nusing two lists            | 1.0627897000000002\n```\nNotice that the functions are ordered as they were defined\nand not by execution time.\n',
    'author': 'Abhinav Omprakash',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/AbhinavOmprakash/pretty-timeit',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
